use yaxpeax_arch::{Arch, Decoder};
use yaxpeax_arm::armv8::a64::{ARMv8, Instruction, Operand, Opcode, SizeCode, ShiftStyle};
use yaxpeax_arm::armv8::a64::DecodeError;

use std::fmt;

fn test_decode(data: [u8; 4], expected: Instruction) {
    let mut reader = yaxpeax_arch::U8Reader::new(&data[..]);
    let instr = <ARMv8 as Arch>::Decoder::default().decode(&mut reader).unwrap();
    assert!(
        instr == expected,
        "decode error for {:02x}{:02x}{:02x}{:02x}:\n  decoded: {:?}\n expected: {:?}\n",
        data[0], data[1], data[2], data[3],
        instr, expected
    );
}

fn test_err(data: [u8; 4], err: DecodeError) {
    let mut reader = yaxpeax_arch::U8Reader::new(&data[..]);
    let result = <ARMv8 as Arch>::Decoder::default().decode(&mut reader);
    assert_eq!(
        result, Err(err),
        "expected IncompleteDecoder error for {:02x}{:02x}{:02x}{:02x}:\n  decoded: {:?}\n",
        data[0], data[1], data[2], data[3],
        result,
    );
}

fn test_display(data: [u8; 4], expected: &'static str) {
    let mut reader = yaxpeax_arch::U8Reader::new(&data[..]);
    let instr = <ARMv8 as Arch>::Decoder::default().decode(&mut reader).unwrap();
    let text = format!("{}", instr);
    assert!(
        text == expected,
        "display error for {:02x}{:02x}{:02x}{:02x}:\n  decoded: {:?}\n displayed: {}\n expected: {}\n",
        data[0], data[1], data[2], data[3],
        instr,
        text, expected
    );
}

#[test]
fn test_neon() {
    test_display([0x00, 0x01, 0x27, 0x1e], "fmov s0, w8");
    test_display([0xe0, 0x03, 0x13, 0xaa], "mov x0, x19");
}

#[test]
fn test_sve() {
    test_err([0x00, 0x00, 0x00, 0x04], DecodeError::IncompleteDecoder);
}

#[test]
fn test_sme() {
    // SME outer product
    test_err([0x00, 0x00, 0xc0, 0x80], DecodeError::IncompleteDecoder);
}

#[test]
fn test_unpredictable() {
    // could be stx/ldx but Lo1 is `x1` and invalid.
    test_err([0x00, 0x00, 0x20, 0x08], DecodeError::InvalidOperand);
    test_err([0x00, 0xfc, 0x00, 0x12], DecodeError::InvalidOperand);
}

#[test]
fn test_barrier() {
    test_display([0xbf, 0x30, 0x03, 0xd5], "dmb #0");
    test_display([0xbf, 0x31, 0x03, 0xd5], "dmb oshld");
    test_display([0xbf, 0x32, 0x03, 0xd5], "dmb oshst");
    test_display([0xbf, 0x33, 0x03, 0xd5], "dmb osh");
    test_display([0xbf, 0x34, 0x03, 0xd5], "dmb #4");
    test_display([0xbf, 0x35, 0x03, 0xd5], "dmb nshld");
    test_display([0xbf, 0x36, 0x03, 0xd5], "dmb nshst");
    test_display([0xbf, 0x37, 0x03, 0xd5], "dmb nsh");
    test_display([0xbf, 0x38, 0x03, 0xd5], "dmb #8");
    test_display([0xbf, 0x39, 0x03, 0xd5], "dmb ishld");
    test_display([0xbf, 0x3a, 0x03, 0xd5], "dmb ishst");
    test_display([0xbf, 0x3b, 0x03, 0xd5], "dmb ish");
    test_display([0xbf, 0x3c, 0x03, 0xd5], "dmb #12");
    test_display([0xbf, 0x3d, 0x03, 0xd5], "dmb ld");
    test_display([0xbf, 0x3e, 0x03, 0xd5], "dmb st");
    test_display([0xbf, 0x3f, 0x03, 0xd5], "dmb sy");
    // only with FEAT_SB
    test_display([0xff, 0x30, 0x03, 0xd5], "sb");
}

#[test]
fn test_display_misc() {
    test_display(
        [0xc0, 0x03, 0x5f, 0xd6],
        "ret"
    );
    test_display(
        [0x1f, 0x20, 0x03, 0xd5],
        "nop"
    );
}

#[test]
fn test_decode_str_ldr() {
    test_decode(
        [0x31, 0x03, 0x40, 0xf9],
        Instruction {
            opcode: Opcode::LDR,
            operands: [
                Operand::Register(SizeCode::X, 17),
                Operand::RegPreIndex(25, 0, false),
                Operand::Nothing,
                Operand::Nothing,
            ]
        }
    );
    test_decode(
        [0xf5, 0x5b, 0x00, 0xf9],
        Instruction {
            opcode: Opcode::STR,
            operands: [
                Operand::Register(SizeCode::X, 21),
                Operand::RegPreIndex(31, 0xb0, false),
                Operand::Nothing,
                Operand::Nothing,
            ]
        }
    );
    test_decode(
        [0x60, 0x02, 0x0a, 0x39],
        Instruction {
            opcode: Opcode::STRB,
            operands: [
                Operand::Register(SizeCode::W, 0),
                Operand::RegPreIndex(19, 0x280, false),
                Operand::Nothing,
                Operand::Nothing,
            ]
        }
    );
    test_decode(
        [0xfd, 0x7b, 0xbe, 0xa9],
        Instruction {
            opcode: Opcode::STP,
            operands: [
                Operand::Register(SizeCode::X, 29),
                Operand::Register(SizeCode::X, 30),
                Operand::RegPreIndex(31, -0x20, true),
                Operand::Nothing,
            ]
        }
    );
}

#[test]
fn test_decode_misc() {
    test_decode(
        [0x22, 0x39, 0x04, 0xb0],
        Instruction {
            opcode: Opcode::ADRP,
            operands: [
                Operand::Register(SizeCode::X, 2),
                Operand::PCOffset(0x8725000),
                Operand::Nothing,
                Operand::Nothing
            ]
        }
    );
    test_display(
        [0x1d, 0x00, 0x80, 0xd2],
        "mov x29, #0x0"
    );
    test_display(
        [0x13, 0x00, 0x80, 0xd2],
        "mov x19, #0x0"
    );
    test_display(
        [0x1d, 0xff, 0xff, 0xd2],
        "mov x29, #0xfff8000000000000"
    );
    test_display(
        [0x22, 0x39, 0x04, 0xb0],
        "adrp x2, $+0x8725000"
    );
}

#[test]
fn test_display_ldr() {
    test_display(
        [0xff, 0xff, 0x00, 0x1c],
        "ldr s31, $+0x1ffc"
    );
    test_display(
        [0xff, 0xff, 0x00, 0x5c],
        "ldr d31, $+0x1ffc"
    );
    test_display(
        [0xff, 0xff, 0x00, 0x9c],
        "ldr q31, $+0x1ffc"
    );
    test_err([0xff, 0xff, 0x00, 0xdc], DecodeError::InvalidOpcode);
    test_display(
        [0x88, 0xff, 0x00, 0x18],
        "ldr w8, $+0x1ff0"
    );
    test_display(
        [0x88, 0xff, 0x00, 0x58],
        "ldr x8, $+0x1ff0"
    );
    test_display(
        [0x88, 0xff, 0x00, 0x98],
        "ldrsw x8, $+0x1ff0"
    );
    test_display(
        [0x88, 0xff, 0x00, 0xd8],
        "prfm plil1keep, $+0x1ff0"
    );
}

#[test]
fn test_decode_mov() {
    test_decode(
        [0x20, 0x00, 0x80, 0x52],
        Instruction {
            opcode: Opcode::MOVZ,
            operands: [
                Operand::Register(SizeCode::W, 0),
                Operand::ImmShift(1, 0),
                Operand::Nothing,
                Operand::Nothing
            ]
        }
    );
}

#[test]
fn test_decode_branch() {
    test_decode(
        [0x41, 0x00, 0x00, 0xb4],
        Instruction {
            opcode: Opcode::CBZ,
            operands: [
                Operand::Register(SizeCode::X, 1),
                Operand::PCOffset(8),
                Operand::Nothing,
                Operand::Nothing
            ]
        }
    );
    test_decode(
        [0x62, 0x00, 0x00, 0xb4],
        Instruction {
            opcode: Opcode::CBZ,
            operands: [
                Operand::Register(SizeCode::X, 2),
                Operand::PCOffset(12),
                Operand::Nothing,
                Operand::Nothing
            ]
        }
    );
    test_decode(
        [0x40, 0x00, 0x1f, 0xd6],
        Instruction {
            opcode: Opcode::BR,
            operands: [
                Operand::Register(SizeCode::X, 2),
                Operand::Nothing,
                Operand::Nothing,
                Operand::Nothing
            ]
        }
    );
}

#[test]
fn test_decode_arithmetic() {
    test_decode(
        [0x21, 0xfc, 0x41, 0x8b],
        Instruction {
            opcode: Opcode::ADD,
            operands: [
                Operand::Register(SizeCode::X, 1),
                Operand::Register(SizeCode::X, 1),
                Operand::RegShift(ShiftStyle::LSR, 63, SizeCode::X, 1),
                Operand::Nothing
            ]
        }
    );
    test_decode(
        [0x21, 0xfc, 0x43, 0x93],
        Instruction {
            opcode: Opcode::SBFM,
            operands: [
                Operand::Register(SizeCode::X, 1),
                Operand::Register(SizeCode::X, 1),
                Operand::Immediate(3),
                Operand::Immediate(63)
            ]
        }
    );
    test_decode(
        [0x3f, 0x38, 0x00, 0xf1],
        Instruction {
            opcode: Opcode::SUBS,
            operands: [
                Operand::Register(SizeCode::X, 31),
                Operand::RegisterOrSP(SizeCode::X, 1),
                Operand::Immediate(0xe),
                Operand::Nothing,
            ]
        }
    );
}

#[test]
fn test_decode_udf() {
    test_decode(
        [0x00, 0x00, 0x00, 0x00],
        Instruction {
            opcode: Opcode::UDF,
            operands: [
                Operand::Imm16(0),
                Operand::Nothing,
                Operand::Nothing,
                Operand::Nothing,
            ]
        }
    );
    test_decode(
        [0xfe, 0xca, 0x00, 0x00],
        Instruction {
            opcode: Opcode::UDF,
            operands: [
                Operand::Imm16(0xcafe),
                Operand::Nothing,
                Operand::Nothing,
                Operand::Nothing,
            ]
        }
    );
}

#[test]
fn test_display_udf() {
    test_display([0x00, 0x00, 0x00, 0x00], "udf #0x0");
    test_display([0xfe, 0xca, 0x00, 0x00], "udf #0xcafe");
}

#[test]
fn test_decode_mul() {
}

#[test]
fn test_decode_ccm() {
    test_display([0x00, 0xa8, 0x42, 0x3a], "ccmn w0, #0x2, #0x0, ge");
    test_display([0x00, 0xa8, 0x42, 0xfa], "ccmp x0, #0x2, #0x0, ge");
    test_display([0x00, 0xa0, 0x42, 0xfa], "ccmp x0, x2, #0x0, ge");
    test_display([0x85, 0x80, 0x42, 0xfa], "ccmp x4, x2, #0x5, hi");
}

#[test]
fn test_decode_data_processing_one_source() {
    const TESTS: &[([u8; 4], &'static str)] = &[
        ([0x14, 0x02, 0xc0, 0x5a], "rbit w20, w16"),
        ([0x14, 0x06, 0xc0, 0x5a], "rev16 w20, w16"),
        ([0x14, 0x12, 0xc0, 0x5a], "clz w20, w16"),
        ([0x14, 0x12, 0xc0, 0x5a], "clz w20, w16"),
        ([0x14, 0x16, 0xc0, 0x5a], "cls w20, w16"),
        ([0x14, 0x16, 0xc0, 0xda], "cls x20, x16"),
        ([0x14, 0x0a, 0xc0, 0xda], "rev32 x20, x16"),
    ];

    for (bytes, instr) in TESTS {
        test_display(*bytes, instr);
    }

    test_err([0x14, 0x0e, 0xc0, 0x5a], DecodeError::InvalidOpcode);
}

#[test]
fn test_decode_data_processing_three_source() {
    const TESTS: &[([u8; 4], &'static str)] = &[
        ([0x11, 0x01, 0x0f, 0x1b], "madd w17, w8, w15, w0"),
        ([0x11, 0x81, 0x0f, 0x1b], "msub w17, w8, w15, w0"),
        ([0x11, 0x81, 0x0f, 0x9b], "msub x17, x8, x15, x0"),
        ([0x11, 0x89, 0x0f, 0x9b], "msub x17, x8, x15, x2"),

        ([0x11, 0x09, 0x2f, 0x9b], "smaddl x17, w8, w15, x2"),
        ([0x11, 0x89, 0x2f, 0x9b], "smsubl x17, w8, w15, x2"),
        ([0x11, 0x09, 0x4f, 0x9b], "smulh x17, x8, x15"),
        ([0x11, 0x09, 0xaf, 0x9b], "umaddl x17, w8, w15, x2"),
        ([0x11, 0x89, 0xaf, 0x9b], "umsubl x17, w8, w15, x2"),
        ([0x11, 0x09, 0xcf, 0x9b], "umulh x17, x8, x15"),
    ];

    for (bytes, instr) in TESTS {
        test_display(*bytes, instr);
    }

    test_err([0x11, 0x81, 0x0f, 0x3b], DecodeError::InvalidOpcode);
}

#[test]
fn test_decode_chrome_entrypoint() {
    // 1400 instructions from the entrypoint of a chrome binary, sorted by
    // instruction word for no good reason.

    test_display(
        [0x00, 0x00, 0x20, 0xd4],
        "brk #0x0"
    );
    test_display(
        [0x00, 0x00, 0x80, 0x12],
        "mov w0, #0xffffffff"
    );
    test_display(
        [0x00, 0x01, 0x3f, 0xd6],
        "blr x8"
    );
    test_display(
        [0x00, 0x02, 0x00, 0x36],
        "tbz w0, #0x0, $+0x40"
    );
    test_display(
        [0x00, 0x03, 0x00, 0x35],
        "cbnz w0, $+0x60"
    );
    test_display(
        [0x00, 0x04, 0x00, 0x36],
        "tbz w0, #0x0, $+0x80"
    );
    test_display(
        [0x00, 0x04, 0x40, 0xf9],
        "ldr x0, [x0, #0x8]"
    );
    test_display(
        [0x00, 0x07, 0x00, 0x34],
        "cbz w0, $+0xe0"
    );
    test_display(
        [0x00, 0x08, 0x47, 0xf9],
        "ldr x0, [x0, #0xe10]"
    );
    test_display(
        [0x00, 0x0b, 0x80, 0x52],
        "mov w0, #0x58"
    );
    test_display(
        [0x00, 0x14, 0x42, 0xf9],
        "ldr x0, [x0, #0x428]"
    );
    test_display(
        [0x00, 0x1b, 0x80, 0x52],
        "mov w0, #0xd8"
    );
    test_display(
        [0x00, 0x20, 0x40, 0xf9],
        "ldr x0, [x0, #0x40]"
    );
    test_display(
        [0x00, 0x24, 0x47, 0xf9],
        "ldr x0, [x0, #0xe48]"
    );
    test_display(
        [0x00, 0x2b, 0x03, 0x90],
        "adrp x0, $+0x6560000"
    );
    test_display(
        [0x00, 0x34, 0x42, 0xf9],
        "ldr x0, [x0, #0x468]"
    );
    test_display(
        [0x00, 0x39, 0x04, 0xb0],
        "adrp x0, $+0x8721000"
    );
    test_display(
        [0x00, 0x3b, 0x04, 0xb0],
        "adrp x0, $+0x8761000"
    );
    test_display(
        [0x00, 0x3c, 0x43, 0xf9],
        "ldr x0, [x0, #0x678]"
    );
    test_display(
        [0x00, 0x44, 0x44, 0xf9],
        "ldr x0, [x0, #0x888]"
    );
    test_display(
        [0x00, 0x50, 0x14, 0x91],
        "add x0, x0, #0x514"
    );
    test_display(
        [0x00, 0x54, 0x44, 0xf9],
        "ldr x0, [x0, #0x8a8]"
    );
    test_display(
        [0x00, 0x58, 0x42, 0xf9],
        "ldr x0, [x0, #0x4b0]"
    );
    test_display(
        [0x00, 0x5c, 0x44, 0xf9],
        "ldr x0, [x0, #0x8b8]"
    );
    test_display(
        [0x00, 0x60, 0x1e, 0x91],
        "add x0, x0, #0x798"
    );
    test_display(
        [0x00, 0x70, 0x47, 0xf9],
        "ldr x0, [x0, #0xee0]"
    );
    test_display(
        [0x00, 0x80, 0x1e, 0x91],
        "add x0, x0, #0x7a0"
    );
    test_display(
        [0x00, 0x80, 0x44, 0xf9],
        "ldr x0, [x0, #0x900]"
    );
    test_display(
        [0x00, 0x84, 0x47, 0xf9],
        "ldr x0, [x0, #0xf08]"
    );
    test_display(
        [0x00, 0xac, 0x40, 0xf9],
        "ldr x0, [x0, #0x158]"
    );
    test_display(
        [0x00, 0xc0, 0x09, 0x91],
        "add x0, x0, #0x270"
    );
    test_display(
        [0x00, 0xc4, 0x45, 0xf9],
        "ldr x0, [x0, #0xb88]"
    );
    test_display(
        [0x00, 0xcc, 0x41, 0xf9],
        "ldr x0, [x0, #0x398]"
    );
    test_display(
        [0x00, 0xdc, 0x35, 0x91],
        "add x0, x0, #0xd77"
    );
    test_display(
        [0x00, 0xf4, 0x47, 0xf9],
        "ldr x0, [x0, #0xfe8]"
    );
    test_display(
        [0x01, 0x00, 0x00, 0x14],
        "b $+0x4"
    );
    test_display(
        [0x01, 0x00, 0x40, 0xf9],
        "ldr x1, [x0]"
    );
    test_display(
        [0x01, 0x05, 0x40, 0xf9],
        "ldr x1, [x8, #0x8]"
    );
    test_display(
        [0x01, 0xfb, 0x4b, 0x95],
        "bl $+0x52fec04"
    );
    test_display(
        [0x02, 0x00, 0x00, 0x14],
        "b $+0x8"
    );
    test_display(
        [0x02, 0x00, 0x00, 0x90],
        "adrp x2, $+0x0"
    );
    test_display(
        [0x02, 0x00, 0x80, 0x92],
        "mov x2, #0xffffffffffffffff"
    );
    test_display(
        [0x02, 0x04, 0xf8, 0x97],
        "bl $-0x1feff8"
    );
    test_display(
        [0x02, 0x2c, 0x80, 0x52],
        "mov w2, #0x160"
    );
    test_display(
        [0x08, 0x00, 0x40, 0xf9],
        "ldr x8, [x0]"
    );
    test_display(
        [0x08, 0x01, 0x40, 0xf9],
        "ldr x8, [x8]"
    );
    test_display(
        [0x08, 0x05, 0x40, 0xf9],
        "ldr x8, [x8, #0x8]"
    );
    test_display(
        [0x08, 0x06, 0xf8, 0x97],
        "bl $-0x1fe7e0"
    );
    test_display(
        [0x08, 0x09, 0x40, 0xf9],
        "ldr x8, [x8, #0x10]"
    );
    test_display(
        [0x08, 0x1d, 0x40, 0x92],
        "and x8, x8, #0xff"
    );
    test_display(
        [0x08, 0x1f, 0x00, 0x13],
        "sxtb w8, w24"
    );
    test_display(
        [0x08, 0x21, 0x0a, 0x91],
        "add x8, x8, #0x288"
    );
    test_display(
        [0x08, 0x41, 0x00, 0x91],
        "add x8, x8, #0x10"
    );
    test_display(
        [0x08, 0x41, 0x40, 0xf9],
        "ldr x8, [x8, #0x80]"
    );
    test_display(
        [0x08, 0x81, 0x0a, 0x91],
        "add x8, x8, #0x2a0"
    );
    test_display(
        [0x08, 0xa1, 0x11, 0x91],
        "add x8, x8, #0x468"
    );
    test_display(
        [0x08, 0xc1, 0x1e, 0x91],
        "add x8, x8, #0x7b0"
    );
    test_display(
        [0x08, 0xdd, 0x46, 0xf9],
        "ldr x8, [x8, #0xdb8]"
    );
    test_display(
        [0x08, 0xe1, 0x0e, 0x91],
        "add x8, x8, #0x3b8"
    );
    test_display(
        [0x08, 0xf2, 0xff, 0x36],
        "tbz w8, #0x1f, $-0x1c0"
    );
    test_display(
        [0x09, 0x1f, 0x00, 0x13],
        "sxtb w9, w24"
    );
    test_display(
        [0x09, 0x5d, 0xc0, 0x39],
        "ldrsb w9, [x8, #0x17]"
    );
    test_display(
        [0x0a, 0x2d, 0x40, 0xa9],
        "ldp x10, x11, [x8]"
    );
    test_display(
        [0x0a, 0xf0, 0x8e, 0x94],
        "bl $+0x23bc028"
    );
    test_display(
        [0x13, 0x3b, 0x04, 0xb0],
        "adrp x19, $+0x8761000"
    );
    test_display(
        [0x13, 0xfd, 0xdf, 0xc8],
        "ldar x19, [x8]"
    );
    test_display(
        [0x14, 0x05, 0x00, 0xb4],
        "cbz x20, $+0xa0"
    );
    test_display(
        [0x15, 0x05, 0x88, 0x1a],
        "cinc w21, w8, ne"
    );
    test_display(
        [0x17, 0xed, 0x7c, 0x92],
        "and x23, x8, #0xfffffffffffffff0"
    );
    test_display(
        [0x1d, 0x00, 0x80, 0xd2],
        "mov x29, #0x0"
    );
    test_display(
        [0x1e, 0x00, 0x80, 0xd2],
        "mov x30, #0x0"
    );
    test_display(
        [0x1f, 0x00, 0x00, 0x71],
        "cmp w0, #0x0"
    );
    test_display(
        [0x1f, 0x00, 0x14, 0xeb],
        "cmp x0, x20"
    );
    test_display(
        [0x1f, 0x01, 0x00, 0x71],
        "cmp w8, #0x0"
    );
    test_display(
        [0x1f, 0x01, 0x00, 0xf1],
        "cmp x8, #0x0"
    );
    test_display(
        [0x1f, 0x01, 0x09, 0xeb],
        "cmp x8, x9"
    );
    test_display(
        [0x1f, 0x01, 0x0c, 0xeb],
        "cmp x8, x12"
    );
    test_display(
        [0x1f, 0x03, 0x00, 0x71],
        "cmp w24, #0x0"
    );
    test_display(
        [0x1f, 0x0d, 0x00, 0xf1],
        "cmp x8, #0x3"
    );
    test_display(
        [0x1f, 0x20, 0x03, 0xd5],
        "nop"
    );
    test_display(
        [0x20, 0x00, 0x1f, 0xd6],
        "br x1"
    );
    test_display(
        [0x20, 0x00, 0x3f, 0xd6],
        "blr x1"
    );
    test_display(
        [0x20, 0x00, 0x80, 0x52],
        "mov w0, #0x1"
    );
    test_display(
        [0x20, 0xb1, 0x8a, 0x9a],
        "csel x0, x9, x10, lt"
    );
    test_display(
        [0x20, 0xb1, 0x94, 0x9a],
        "csel x0, x9, x20, lt"
    );
    test_display(
        [0x20, 0xb1, 0x95, 0x9a],
        "csel x0, x9, x21, lt"
    );
    test_display(
        [0x21, 0x00, 0x00, 0xcb],
        "sub x1, x1, x0"
    );
    test_display(
        [0x21, 0x01, 0x00, 0x54],
        "b.ne $+0x24"
    );
    test_display(
        [0x21, 0x01, 0x00, 0x54],
        "b.ne $+0x24"
    );
    test_display(
        [0x21, 0x04, 0x36, 0x91],
        "add x1, x1, #0xd81"
    );
    test_display(
        [0x21, 0x10, 0x34, 0x91],
        "add x1, x1, #0xd04"
    );
    test_display(
        [0x21, 0x1c, 0x00, 0x91],
        "add x1, x1, #0x7"
    );
    test_display(
        [0x21, 0x1c, 0x45, 0xf9],
        "ldr x1, [x1, #0xa38]"
    );
    test_display(
        [0x21, 0x1c, 0x46, 0xf9],
        "ldr x1, [x1, #0xc38]"
    );
    test_display(
        [0x21, 0x34, 0x42, 0xf9],
        "ldr x1, [x1, #0x468]"
    );
    test_display(
        [0x21, 0x3c, 0x36, 0x91],
        "add x1, x1, #0xd8f"
    );
    test_display(
        [0x21, 0x90, 0x36, 0x91],
        "add x1, x1, #0xda4"
    );
    test_display(
        [0x21, 0x98, 0x41, 0xf9],
        "ldr x1, [x1, #0x330]"
    );
    test_display(
        [0x21, 0xb4, 0x34, 0x91],
        "add x1, x1, #0xd2d"
    );
    test_display(
        [0x21, 0xc4, 0x40, 0xf9],
        "ldr x1, [x1, #0x188]"
    );
    test_display(
        [0x21, 0xc4, 0x45, 0xf9],
        "ldr x1, [x1, #0xb88]"
    );
    test_display(
        [0x21, 0xd8, 0x40, 0xf9],
        "ldr x1, [x1, #0x1b0]"
    );
    test_display(
        [0x21, 0xd8, 0x47, 0xf9],
        "ldr x1, [x1, #0xfb0]"
    );
    test_display(
        [0x21, 0xe4, 0x40, 0xf9],
        "ldr x1, [x1, #0x1c8]"
    );
    test_display(
        [0x21, 0xf4, 0x36, 0x91],
        "add x1, x1, #0xdbd"
    );
    test_display(
        [0x21, 0xfc, 0x41, 0x8b],
        "add x1, x1, x1, lsr #63"
    );
    test_display(
        [0x21, 0xfc, 0x41, 0x93],
        "asr x1, x1, #0x1"
    );
    test_display(
        [0x21, 0xfc, 0x43, 0x93],
        "asr x1, x1, #0x3"
    );
    test_display(
        [0x21, 0xfc, 0x44, 0xf9],
        "ldr x1, [x1, #0x9f8]"
    );
    test_display(
        [0x22, 0x09, 0x80, 0x52],
        "mov w2, #0x49"
    );
    test_display(
        [0x22, 0xb1, 0x96, 0x9a],
        "csel x2, x9, x22, lt"
    );
    test_display(
        [0x23, 0xb1, 0x96, 0x9a],
        "csel x3, x9, x22, lt"
    );
    test_display(
        [0x24, 0x01, 0x80, 0x52],
        "mov w4, #0x9"
    );
    test_display(
        [0x26, 0x00, 0x00, 0x14],
        "b $+0x98"
    );
    test_display(
        [0x28, 0x02, 0x00, 0x54],
        "b.hi $+0x44"
    );
    test_display(
        [0x28, 0x11, 0x08, 0x8b],
        "add x8, x9, x8, lsl #4"
    );
    test_display(
        [0x28, 0xb1, 0x88, 0x9a],
        "csel x8, x9, x8, lt"
    );
    test_display(
        [0x29, 0x01, 0x40, 0xf9],
        "ldr x9, [x9]"
    );
    test_display(
        [0x29, 0x1d, 0x40, 0x92],
        "and x9, x9, #0xff"
    );
    test_display(
        [0x29, 0xa1, 0x21, 0x91],
        "add x9, x9, #0x868"
    );
    test_display(
        [0x29, 0xc5, 0x46, 0xf9],
        "ldr x9, [x9, #0xd88]"
    );
    test_display(
        [0x29, 0xdd, 0x46, 0xf9],
        "ldr x9, [x9, #0xdb8]"
    );
    test_display(
        [0x2b, 0x1d, 0x00, 0x13],
        "sxtb w11, w9"
    );
    test_display(
        [0x2b, 0xb1, 0x88, 0x9a],
        "csel x11, x9, x8, lt"
    );
    test_display(
        [0x34, 0xb1, 0x94, 0x9a],
        "csel x20, x9, x20, lt"
    );
    test_display(
        [0x35, 0x01, 0x40, 0xb9],
        "ldr w21, [x9]"
    );
    test_display(
        [0x35, 0x03, 0x00, 0xb5],
        "cbnz x21, $+0x64"
    );
    test_display(
        [0x35, 0xb1, 0x95, 0x9a],
        "csel x21, x9, x21, lt"
    );
    test_display(
        [0x3f, 0x01, 0x00, 0x71],
        "cmp w9, #0x0"
    );
    test_display(
        [0x3f, 0x01, 0x08, 0xeb],
        "cmp x9, x8"
    );
    test_display(
        [0x3f, 0x38, 0x00, 0xf1],
        "cmp x1, #0xe"
    );
    test_display(
        [0x40, 0x00, 0x1f, 0xd6],
        "br x2"
    );
    test_display(
        [0x40, 0x21, 0x00, 0x91],
        "add x0, x10, #0x8"
    );
    test_display(
        [0x40, 0x7d, 0x80, 0x52],
        "mov w0, #0x3ea"
    );
    test_display(
        [0x41, 0x01, 0x00, 0x54],
        "b.ne $+0x28"
    );
    test_display(
        [0x41, 0xb1, 0x88, 0x9a],
        "csel x1, x10, x8, lt"
    );
    test_display(
        [0x42, 0x00, 0x1b, 0x91],
        "add x2, x2, #0x6c0"
    );
    test_display(
        [0x42, 0x40, 0x1a, 0x91],
        "add x2, x2, #0x690"
    );
    test_display(
        [0x42, 0x50, 0x41, 0xf9],
        "ldr x2, [x2, #0x2a0]"
    );
    test_display(
        [0x42, 0x7d, 0x80, 0x52],
        "mov w2, #0x3ea"
    );
    test_display(
        [0x42, 0xc0, 0x1b, 0x91],
        "add x2, x2, #0x6f0"
    );
    test_display(
        [0x42, 0xe4, 0x44, 0xf9],
        "ldr x2, [x2, #0x9c8]"
    );
    test_display(
        [0x48, 0xb1, 0x89, 0x9a],
        "csel x8, x10, x9, lt"
    );
    test_display(
        [0x49, 0xb1, 0x89, 0x9a],
        "csel x9, x10, x9, lt"
    );
    test_display(
        [0x4a, 0x1d, 0x00, 0x13],
        "sxtb w10, w10"
    );
    test_display(
        [0x4c, 0xb1, 0x89, 0x9a],
        "csel x12, x10, x9, lt"
    );
    test_display(
        [0x5f, 0x01, 0x00, 0x71],
        "cmp w10, #0x0"
    );
    test_display(
        [0x60, 0x02, 0x00, 0xb9],
        "str w0, [x19]"
    );
    test_display(
        [0x60, 0x02, 0x0a, 0x39],
        "strb w0, [x19, #0x280]"
    );
    test_display(
        [0x60, 0x02, 0x4a, 0x39],
        "ldrb w0, [x19, #0x280]"
    );
    test_display(
        [0x60, 0x22, 0x00, 0x91],
        "add x0, x19, #0x8"
    );
    test_display(
        [0x60, 0x36, 0x40, 0xf9],
        "ldr x0, [x19, #0x68]"
    );
    test_display(
        [0x60, 0x3e, 0x40, 0xf9],
        "ldr x0, [x19, #0x78]"
    );
    test_display(
        [0x61, 0x02, 0x00, 0x12],
        "and w1, w19, #0x1"
    );
    test_display(
        [0x61, 0xb1, 0x88, 0x9a],
        "csel x1, x11, x8, lt"
    );
    test_display(
        [0x62, 0xb1, 0x89, 0x9a],
        "csel x2, x11, x9, lt"
    );
    test_display(
        [0x63, 0x14, 0x42, 0xf9],
        "ldr x3, [x3, #0x428]"
    );
    test_display(
        [0x63, 0x18, 0x16, 0x91],
        "add x3, x3, #0x586"
    );
    test_display(
        [0x63, 0x24, 0x47, 0xf9],
        "ldr x3, [x3, #0xe48]"
    );
    test_display(
        [0x63, 0x44, 0x44, 0xf9],
        "ldr x3, [x3, #0x888]"
    );
    test_display(
        [0x63, 0x5c, 0x21, 0x91],
        "add x3, x3, #0x857"
    );
    test_display(
        [0x63, 0x5c, 0x44, 0xf9],
        "ldr x3, [x3, #0x8b8]"
    );
    test_display(
        [0x63, 0x80, 0x44, 0xf9],
        "ldr x3, [x3, #0x900]"
    );
    test_display(
        [0x63, 0x84, 0x47, 0xf9],
        "ldr x3, [x3, #0xf08]"
    );
    test_display(
        [0x63, 0x88, 0x09, 0x91],
        "add x3, x3, #0x262"
    );
    test_display(
        [0x63, 0xbc, 0x44, 0xf9],
        "ldr x3, [x3, #0x978]"
    );
    test_display(
        [0x63, 0xc4, 0x45, 0xf9],
        "ldr x3, [x3, #0xb88]"
    );
    test_display(
        [0x63, 0xcc, 0x41, 0xf9],
        "ldr x3, [x3, #0x398]"
    );
    test_display(
        [0x63, 0xf4, 0x47, 0xf9],
        "ldr x3, [x3, #0xfe8]"
    );
    test_display(
        [0x68, 0x00, 0xf8, 0x36],
        "tbz w8, #0x1f, $+0xc"
    );
    test_display(
        [0x68, 0x01, 0xf8, 0x37],
        "tbnz w8, #0x1f, $+0x2c"
    );
    test_display(
        [0x68, 0x02, 0x00, 0xf9],
        "str x8, [x19]"
    );
    test_display(
        [0x68, 0x1d, 0x00, 0x13],
        "sxtb w8, w11"
    );
    test_display(
        [0x74, 0x3a, 0x00, 0xf9],
        "str x20, [x19, #0x70]"
    );
    test_display(
        [0x74, 0x3a, 0x40, 0xf9],
        "ldr x20, [x19, #0x70]"
    );
    test_display(
        [0x74, 0x5e, 0x40, 0x39],
        "ldrb w20, [x19, #0x17]"
    );
    test_display(
        [0x75, 0x06, 0x40, 0xf9],
        "ldr x21, [x19, #0x8]"
    );
    test_display(
        [0x75, 0x36, 0x00, 0xf9],
        "str x21, [x19, #0x68]"
    );
    test_display(
        [0x75, 0x36, 0x40, 0xf9],
        "ldr x21, [x19, #0x68]"
    );
    test_display(
        [0x75, 0x3a, 0x40, 0xf9],
        "ldr x21, [x19, #0x70]"
    );
    test_display(
        [0x76, 0x5e, 0x40, 0x39],
        "ldrb w22, [x19, #0x17]"
    );
    test_display(
        [0x7f, 0x01, 0x00, 0x71],
        "cmp w11, #0x0"
    );
    test_display(
        [0x7f, 0x06, 0x00, 0xf1],
        "cmp x19, #0x1"
    );
    test_display(
        [0x7f, 0x1d, 0x00, 0xf1],
        "cmp x11, #0x7"
    );
    test_display(
        [0x80, 0x22, 0x00, 0x91],
        "add x0, x20, #0x8"
    );
    test_display(
        [0x80, 0x3a, 0x40, 0xf9],
        "ldr x0, [x20, #0x70]"
    );
    test_display(
        [0x81, 0x01, 0x00, 0x54],
        "b.ne $+0x30"
    );
    test_display(
        [0x82, 0x02, 0x80, 0x52],
        "mov w2, #0x14"
    );
    test_display(
        [0x84, 0x48, 0x46, 0xf9],
        "ldr x4, [x4, #0xc90]"
    );
    test_display(
        [0x88, 0x02, 0x00, 0xf9],
        "str x8, [x20]"
    );
    test_display(
        [0x88, 0x02, 0x40, 0xf9],
        "ldr x8, [x20]"
    );
    test_display(
        [0x89, 0x5e, 0x40, 0x39],
        "ldrb w9, [x20, #0x17]"
    );
    test_display(
        [0x8a, 0x06, 0x40, 0xf9],
        "ldr x10, [x20, #0x8]"
    );
    test_display(
        [0x93, 0x22, 0x00, 0x91],
        "add x19, x20, #0x8"
    );
    test_display(
        [0x93, 0xfe, 0xdf, 0xc8],
        "ldar x19, [x20]"
    );
    test_display(
        [0x94, 0x00, 0xf8, 0x36],
        "tbz w20, #0x1f, $+0x10"
    );
    test_display(
        [0x94, 0x22, 0x0a, 0x91],
        "add x20, x20, #0x288"
    );
    test_display(
        [0x94, 0x82, 0x0a, 0x91],
        "add x20, x20, #0x2a0"
    );
    test_display(
        [0x94, 0xa2, 0x11, 0x91],
        "add x20, x20, #0x468"
    );
    test_display(
        [0x96, 0x1e, 0x00, 0x13],
        "sxtb w22, w20"
    );
    test_display(
        [0xa0, 0x03, 0x19, 0xf8],
        "stur x0, [x29, #-0x70]"
    );
    test_display(
        [0xa0, 0x03, 0x1a, 0xf8],
        "stur x0, [x29, #-0x60]"
    );
    test_display(
        [0xa0, 0x03, 0x58, 0xf8],
        "ldur x0, [x29, #-0x80]"
    );
    test_display(
        [0xa0, 0x03, 0x5a, 0xf8],
        "ldur x0, [x29, #-0x60]"
    );
    test_display(
        [0xa0, 0x09, 0x40, 0xd4],
        "hlt #0x4d"
    );
    test_display(
        [0xa0, 0x22, 0x00, 0x91],
        "add x0, x21, #0x8"
    );
    test_display(
        [0xa0, 0x23, 0x01, 0xd1],
        "sub x0, x29, #0x48"
    );
    test_display(
        [0xa0, 0x3e, 0x40, 0xf9],
        "ldr x0, [x21, #0x78]"
    );
    test_display(
        [0xa0, 0x83, 0x1a, 0xf8],
        "stur x0, [x29, #-0x58]"
    );
    test_display(
        [0xa0, 0x83, 0x58, 0xf8],
        "ldur x0, [x29, #-0x78]"
    );
    test_display(
        [0xa0, 0x83, 0x5b, 0xf8],
        "ldur x0, [x29, #-0x48]"
    );
    test_display(
        [0xa0, 0xe3, 0x01, 0xd1],
        "sub x0, x29, #0x78"
    );
    test_display(
        [0xa1, 0x23, 0x01, 0xd1],
        "sub x1, x29, #0x48"
    );
    test_display(
        [0xa1, 0x7e, 0x80, 0x52],
        "mov w1, #0x3f5"
    );
    test_display(
        [0xa1, 0x83, 0x01, 0xd1],
        "sub x1, x29, #0x60"
    );
    test_display(
        [0xa1, 0xe3, 0x01, 0xd1],
        "sub x1, x29, #0x78"
    );
    test_display(
        [0xa2, 0x00, 0x00, 0x54],
        "b.hs $+0x14"
    );
    test_display(
        [0xa2, 0x01, 0x80, 0x52],
        "mov w2, #0xd"
    );
    test_display(
        [0xa2, 0x04, 0x80, 0x52],
        "mov w2, #0x25"
    );
    test_display(
        [0xa2, 0x23, 0x01, 0xd1],
        "sub x2, x29, #0x48"
    );
    test_display(
        [0xa2, 0x23, 0x80, 0x52],
        "mov w2, #0x11d"
    );
    test_display(
        [0xa3, 0xe3, 0x01, 0xd1],
        "sub x3, x29, #0x78"
    );
    test_display(
        [0xa8, 0x03, 0x02, 0xd1],
        "sub x8, x29, #0x80"
    );
    test_display(
        [0xa8, 0x23, 0x01, 0xd1],
        "sub x8, x29, #0x48"
    );
    test_display(
        [0xa8, 0x3e, 0x00, 0xf9],
        "str x8, [x21, #0x78]"
    );
    test_display(
        [0xa8, 0x42, 0x00, 0x91],
        "add x8, x21, #0x10"
    );
    test_display(
        [0xa8, 0x73, 0x5b, 0x38],
        "ldurb w8, [x29, #-0x49]"
    );
    test_display(
        [0xa8, 0x73, 0xdb, 0x38],
        "ldursb w8, [x29, #-0x49]"
    );
    test_display(
        [0xa8, 0x83, 0x01, 0xd1],
        "sub x8, x29, #0x60"
    );
    test_display(
        [0xa8, 0x83, 0x19, 0xf8],
        "stur x8, [x29, #-0x68]"
    );
    test_display(
        [0xa8, 0x83, 0x1b, 0xf8],
        "stur x8, [x29, #-0x48]"
    );
    test_display(
        [0xa8, 0x83, 0x1c, 0xf8],
        "stur x8, [x29, #-0x38]"
    );
    test_display(
        [0xa8, 0x83, 0x1d, 0xf8],
        "stur x8, [x29, #-0x28]"
    );
    test_display(
        [0xa8, 0x83, 0x5b, 0xf8],
        "ldur x8, [x29, #-0x48]"
    );
    test_display(
        [0xa8, 0x83, 0x5c, 0xf8],
        "ldur x8, [x29, #-0x38]"
    );
    test_display(
        [0xa8, 0x83, 0x5d, 0xf8],
        "ldur x8, [x29, #-0x28]"
    );
    test_display(
        [0xa8, 0xf3, 0x5c, 0x38],
        "ldurb w8, [x29, #-0x31]"
    );
    test_display(
        [0xa8, 0xf3, 0xd9, 0x38],
        "ldursb w8, [x29, #-0x61]"
    );
    test_display(
        [0xa8, 0xf3, 0xdc, 0x38],
        "ldursb w8, [x29, #-0x31]"
    );
    test_display(
        [0xa9, 0x03, 0x5c, 0xf8],
        "ldur x9, [x29, #-0x40]"
    );
    test_display(
        [0xa9, 0x83, 0x5a, 0xf8],
        "ldur x9, [x29, #-0x58]"
    );
    test_display(
        [0xa9, 0xab, 0x78, 0xa9],
        "ldp x9, x10, [x29, #-0x78]"
    );
    test_display(
        [0xa9, 0xaf, 0x78, 0xa9],
        "ldp x9, x11, [x29, #-0x78]"
    );
    test_display(
        [0xa9, 0x00, 0x00, 0x54],
        "b.ls $+0x14"
    );
    test_display(
        [0xaa, 0xe3, 0x01, 0xd1],
        "sub x10, x29, #0x78"
    );
    test_display(
        [0xa9, 0xb2, 0x94, 0x9a],
        "csel x9, x21, x20, lt"
    );
    test_display(
        [0xb4, 0x22, 0x03, 0x51],
        "sub w20, w21, #0xc8"
    );
    test_display(
        [0xb4, 0x92, 0x01, 0x51],
        "sub w20, w21, #0x64"
    );
    test_display(
        [0xb5, 0x56, 0x44, 0xf9],
        "ldr x21, [x21, #0x8a8]"
    );
    test_display(
        [0xb5, 0x83, 0x18, 0xf8],
        "stur x21, [x29, #-0x78]"
    );
    test_display(
        [0xb5, 0xda, 0x47, 0xf9],
        "ldr x21, [x21, #0xfb0]"
    );
    test_display(
        [0xb5, 0xe3, 0x01, 0xd1],
        "sub x21, x29, #0x78"
    );
    test_display(
        [0xb5, 0xf3, 0x19, 0x38],
        "sturb w21, [x29, #-0x61]"
    );
    test_display(
        [0xb6, 0xd7, 0x38, 0xa9],
        "stp x22, x21, [x29, #-0x78]"
    );
    test_display(
        [0xb6, 0xe3, 0x01, 0xd1],
        "sub x22, x29, #0x78"
    );
    test_display(
        [0xbf, 0x5e, 0x00, 0xf1],
        "cmp x21, #0x17"
    );
    test_display(
        [0xc0, 0x03, 0x5f, 0xd6],
        "ret"
    );
    test_display(
        [0xc0, 0x09, 0x40, 0xd4],
        "hlt #0x4e"
    );
    test_display(
        [0xc0, 0x42, 0x00, 0x91],
        "add x0, x22, #0x10"
    );
    test_display(
        [0xc0, 0x7e, 0x80, 0x52],
        "mov w0, #0x3f6"
    );
    test_display(
        [0xc0, 0x80, 0x80, 0x52],
        "mov w0, #0x406"
    );
    test_display(
        [0xc2, 0x47, 0x80, 0x52],
        "mov w2, #0x23e"
    );
    test_display(
        [0xc9, 0x1e, 0x00, 0x13],
        "sxtb w9, w22"
    );
    test_display(
        [0xd6, 0x16, 0x43, 0xf9],
        "ldr x22, [x22, #0x628]"
    );
    test_display(
        [0xdf, 0x6a, 0x35, 0x38],
        "strb wzr, [x22, x21]"
    );
    test_display(
        [0xe0, 0x03, 0x00, 0x32],
        "mov w0, #0x1"
    );
    test_display(
        [0xe0, 0x03, 0x00, 0x91],
        "mov x0, sp"
    );
    test_display(
        [0xe0, 0x03, 0x1f, 0x32],
        "mov w0, #0x2"
    );
    test_display(
        [0xe0, 0x07, 0x00, 0x32],
        "mov w0, #0x3"
    );
    test_display(
        [0xe0, 0x07, 0x00, 0xf9],
        "str x0, [sp, #0x8]"
    );
    test_display(
        [0xe0, 0x07, 0x40, 0xf9],
        "ldr x0, [sp, #0x8]"
    );
    test_display(
        [0xe0, 0x09, 0x40, 0xd4],
        "hlt #0x4f"
    );
    test_display(
        [0xe0, 0x0b, 0x00, 0xf9],
        "str x0, [sp, #0x10]"
    );
    test_display(
        [0xe0, 0x0f, 0x00, 0x32],
        "mov w0, #0xf"
    );
    test_display(
        [0xe0, 0x0f, 0x40, 0xf9],
        "ldr x0, [sp, #0x18]"
    );
    test_display(
        [0xe0, 0x13, 0x40, 0xf9],
        "ldr x0, [sp, #0x20]"
    );
    test_display(
        [0xe0, 0x17, 0x00, 0xf9],
        "str x0, [sp, #0x28]"
    );
    test_display(
        [0xe0, 0x17, 0x9f, 0x1a],
        "cset w0, eq"
    );
    test_display(
        [0xe0, 0x1b, 0x40, 0xf9],
        "ldr x0, [sp, #0x30]"
    );
    test_display(
        [0xe0, 0x1f, 0x40, 0xf9],
        "ldr x0, [sp, #0x38]"
    );
    test_display(
        [0xe0, 0x22, 0x00, 0x91],
        "add x0, x23, #0x8"
    );
    test_display(
        [0xe0, 0x23, 0x00, 0x91],
        "add x0, sp, #0x8"
    );
    test_display(
        [0xe0, 0x23, 0x00, 0xf9],
        "str x0, [sp, #0x40]"
    );
    test_display(
        [0xe0, 0x27, 0x40, 0xf9],
        "ldr x0, [sp, #0x48]"
    );
    test_display(
        [0xe0, 0x33, 0x40, 0xf9],
        "ldr x0, [sp, #0x60]"
    );
    test_display(
        [0xe0, 0x63, 0x00, 0x91],
        "add x0, sp, #0x18"
    );
    test_display(
        [0xe0, 0x83, 0x00, 0x91],
        "add x0, sp, #0x20"
    );
    test_display(
        [0xe0, 0x83, 0x01, 0x91],
        "add x0, sp, #0x60"
    );
    test_display(
        [0xe0, 0xa3, 0x00, 0x91],
        "add x0, sp, #0x28"
    );
    test_display(
        [0xe0, 0xe3, 0x00, 0x91],
        "add x0, sp, #0x38"
    );
    test_display(
        [0xe0, 0xe3, 0x01, 0x91],
        "add x0, sp, #0x78"
    );
    test_display(
        [0xe1, 0x03, 0x1f, 0x32],
        "mov w1, #0x2"
    );
    test_display(
        [0xe1, 0x03, 0x40, 0xf9],
        "ldr x1, [sp]"
    );
    test_display(
        [0xe1, 0x23, 0x00, 0x91],
        "add x1, sp, #0x8"
    );
    test_display(
        [0xe1, 0x63, 0x00, 0x91],
        "add x1, sp, #0x18"
    );
    test_display(
        [0xe1, 0x83, 0x00, 0x91],
        "add x1, sp, #0x20"
    );
    test_display(
        [0xe1, 0xe3, 0x00, 0x91],
        "add x1, sp, #0x38"
    );
    test_display(
        [0xe1, 0xe3, 0x01, 0x91],
        "add x1, sp, #0x78"
    );
    test_display(
        [0xe2, 0x07, 0x1d, 0x32],
        "mov w2, #0x18"
    );
    test_display(
        [0xe2, 0x23, 0x00, 0x91],
        "add x2, sp, #0x8"
    );
    test_display(
        [0xe2, 0xe3, 0x00, 0x91],
        "add x2, sp, #0x38"
    );
    test_display(
        [0xe3, 0x03, 0x00, 0x32],
        "mov w3, #0x1"
    );
    test_display(
        [0xe3, 0x03, 0x1f, 0x32],
        "mov w3, #0x2"
    );
    test_display(
        [0xe4, 0x07, 0x00, 0x32],
        "mov w4, #0x3"
    );
    test_display(
        [0xe4, 0x0b, 0x00, 0x32],
        "mov w4, #0x7"
    );
    test_display(
        [0xe6, 0x03, 0x00, 0x91],
        "mov x6, sp"
    );
    test_display(
        [0xe8, 0x01, 0xf8, 0x37],
        "tbnz w8, #0x1f, $+0x3c"
    );
    test_display(
        [0xe8, 0x02, 0x41, 0xb2],
        "orr x8, x23, #0x8000000000000000"
    );
    test_display(
        [0xe8, 0x03, 0x00, 0x32],
        "mov w8, #0x1"
    );
    test_display(
        [0xe8, 0x0f, 0x00, 0xf9],
        "str x8, [sp, #0x18]"
    );
    test_display(
        [0xe8, 0x1f, 0x40, 0xf9],
        "ldr x8, [sp, #0x38]"
    );
    test_display(
        [0xe8, 0x1f, 0xc1, 0x39],
        "ldrsb w8, [sp, #0x47]"
    );
    test_display(
        [0xe8, 0x23, 0x00, 0x91],
        "add x8, sp, #0x8"
    );
    test_display(
        [0xe8, 0x3f, 0x41, 0x39],
        "ldrb w8, [sp, #0x4f]"
    );
    test_display(
        [0xe8, 0x3f, 0xc1, 0x39],
        "ldrsb w8, [sp, #0x4f]"
    );
    test_display(
        [0xe8, 0x63, 0x00, 0x91],
        "add x8, sp, #0x18"
    );
    test_display(
        [0xe8, 0x7f, 0xc0, 0x39],
        "ldrsb w8, [sp, #0x1f]"
    );
    test_display(
        [0xe8, 0x7f, 0xc1, 0x39],
        "ldrsb w8, [sp, #0x5f]"
    );
    test_display(
        [0xe8, 0x83, 0x00, 0x91],
        "add x8, sp, #0x20"
    );
    test_display(
        [0xe8, 0x83, 0x01, 0x91],
        "add x8, sp, #0x60"
    );
    test_display(
        [0xe8, 0xbf, 0xc0, 0x39],
        "ldrsb w8, [sp, #0x2f]"
    );
    test_display(
        [0xe8, 0xdf, 0x41, 0x39],
        "ldrb w8, [sp, #0x77]"
    );
    test_display(
        [0xe8, 0xdf, 0xc0, 0x39],
        "ldrsb w8, [sp, #0x37]"
    );
    test_display(
        [0xe8, 0xdf, 0xc1, 0x39],
        "ldrsb w8, [sp, #0x77]"
    );
    test_display(
        [0xe8, 0xe3, 0x00, 0x91],
        "add x8, sp, #0x38"
    );
    test_display(
        [0xe8, 0xe3, 0x01, 0x91],
        "add x8, sp, #0x78"
    );
    test_display(
        [0xe9, 0x03, 0x01, 0x32],
        "mov w9, #0x80000000"
    );
    test_display(
        [0xe9, 0x07, 0x40, 0xf9],
        "ldr x9, [sp, #0x8]"
    );
    test_display(
        [0xe9, 0x13, 0x40, 0xf9],
        "ldr x9, [sp, #0x20]"
    );
    test_display(
        [0xe9, 0x1f, 0x40, 0xf9],
        "ldr x9, [sp, #0x38]"
    );
    test_display(
        [0xe9, 0x23, 0x40, 0xf9],
        "ldr x9, [sp, #0x40]"
    );
    test_display(
        [0xe9, 0x37, 0x40, 0xf9],
        "ldr x9, [sp, #0x68]"
    );
    test_display(
        [0xe9, 0xa3, 0x00, 0xb9],
        "str w9, [sp, #0xa0]"
    );
    test_display(
        [0xe9, 0xb2, 0x96, 0x9a],
        "csel x9, x23, x22, lt"
    );
    test_display(
        [0xe9, 0xdf, 0x41, 0x39],
        "ldrb w9, [sp, #0x77]"
    );
    test_display(
        [0xea, 0x37, 0x40, 0xf9],
        "ldr x10, [sp, #0x68]"
    );
    test_display(
        [0xea, 0x63, 0x00, 0x91],
        "add x10, sp, #0x18"
    );
    test_display(
        [0xf3, 0x03, 0x00, 0x91],
        "mov x19, sp"
    );
    test_display(
        [0xf3, 0x0b, 0x40, 0xf9],
        "ldr x19, [sp, #0x10]"
    );
    test_display(
        [0xf3, 0x1b, 0x00, 0xf9],
        "str x19, [sp, #0x30]"
    );
    test_display(
        [0xf3, 0x5b, 0x00, 0xf9],
        "str x19, [sp, #0xb0]"
    );
    test_display(
        [0xf3, 0x5b, 0x40, 0xf9],
        "ldr x19, [sp, #0xb0]"
    );
    test_display(
        [0xf4, 0x07, 0x9f, 0x1a],
        "cset w20, ne"
    );
    test_display(
        [0xf4, 0x0b, 0x00, 0xb9],
        "str w20, [sp, #0x8]"
    );
    test_display(
        [0xf4, 0x4f, 0x01, 0xa9],
        "stp x20, x19, [sp, #0x10]"
    );
    test_display(
        [0xf4, 0x4f, 0x02, 0xa9],
        "stp x20, x19, [sp, #0x20]"
    );
    test_display(
        [0xf4, 0x4f, 0x0c, 0xa9],
        "stp x20, x19, [sp, #0xc0]"
    );
    test_display(
        [0xf4, 0x4f, 0x10, 0xa9],
        "stp x20, x19, [sp, #0x100]"
    );
    test_display(
        [0xf4, 0x4f, 0x16, 0xa9],
        "stp x20, x19, [sp, #0x160]"
    );
    test_display(
        [0xf4, 0x4f, 0x19, 0xa9],
        "stp x20, x19, [sp, #0x190]"
    );
    test_display(
        [0xf4, 0x4f, 0x41, 0xa9],
        "ldp x20, x19, [sp, #0x10]"
    );
    test_display(
        [0xf4, 0x4f, 0x42, 0xa9],
        "ldp x20, x19, [sp, #0x20]"
    );
    test_display(
        [0xf4, 0x4f, 0x4c, 0xa9],
        "ldp x20, x19, [sp, #0xc0]"
    );
    test_display(
        [0xf4, 0x4f, 0x50, 0xa9],
        "ldp x20, x19, [sp, #0x100]"
    );
    test_display(
        [0xf4, 0x4f, 0x56, 0xa9],
        "ldp x20, x19, [sp, #0x160]"
    );
    test_display(
        [0xf4, 0x4f, 0x59, 0xa9],
        "ldp x20, x19, [sp, #0x190]"
    );
    test_display(
        [0xf4, 0x4f, 0xbe, 0xa9],
        "stp x20, x19, [sp, #-0x20]!"
    );
    test_display(
        [0xf4, 0x4f, 0xc2, 0xa8],
        "ldp x20, x19, [sp], #0x20"
    );
    test_display(
        [0xf4, 0xb2, 0x96, 0x9a],
        "csel x20, x23, x22, lt"
    );
    test_display(
        [0xf4, 0xe3, 0x00, 0x91],
        "add x20, sp, #0x38"
    );
    test_display(
        [0xf5, 0x03, 0x00, 0x32],
        "mov w21, #0x1"
    );
    test_display(
        [0xf5, 0x03, 0x00, 0xf9],
        "str x21, [sp]"
    );
    test_display(
        [0xf5, 0x03, 0x1f, 0x32],
        "mov w21, #0x2"
    );
    test_display(
        [0xf5, 0x07, 0x00, 0xf9],
        "str x21, [sp, #0x8]"
    );
    test_display(
        [0xf5, 0x07, 0x43, 0xf8],
        "ldr x21, [sp], #0x30"
    );
    test_display(
        [0xf5, 0x0f, 0x1d, 0xf8],
        "str x21, [sp, #-0x30]!"
    );
    test_display(
        [0xf5, 0x13, 0x00, 0xf9],
        "str x21, [sp, #0x20]"
    );
    test_display(
        [0xf5, 0x5b, 0x00, 0xf9],
        "str x21, [sp, #0xb0]"
    );
    test_display(
        [0xf5, 0x5b, 0x40, 0xf9],
        "ldr x21, [sp, #0xb0]"
    );
    test_display(
        [0xf5, 0x83, 0x00, 0x91],
        "add x21, sp, #0x20"
    );
    test_display(
        [0xf5, 0xa3, 0x00, 0x91],
        "add x21, sp, #0x28"
    );
    test_display(
        [0xf6, 0x1f, 0x00, 0xf9],
        "str x22, [sp, #0x38]"
    );
    test_display(
        [0xf6, 0x23, 0x00, 0x91],
        "add x22, sp, #0x8"
    );
    test_display(
        [0xf6, 0x57, 0x0f, 0xa9],
        "stp x22, x21, [sp, #0xf0]"
    );
    test_display(
        [0xf6, 0x57, 0x15, 0xa9],
        "stp x22, x21, [sp, #0x150]"
    );
    test_display(
        [0xf6, 0x57, 0x18, 0xa9],
        "stp x22, x21, [sp, #0x180]"
    );
    test_display(
        [0xf6, 0x57, 0x4f, 0xa9],
        "ldp x22, x21, [sp, #0xf0]"
    );
    test_display(
        [0xf6, 0x57, 0x55, 0xa9],
        "ldp x22, x21, [sp, #0x150]"
    );
    test_display(
        [0xf6, 0x57, 0x58, 0xa9],
        "ldp x22, x21, [sp, #0x180]"
    );
    test_display(
        [0xf6, 0x57, 0xbd, 0xa9],
        "stp x22, x21, [sp, #-0x30]!"
    );
    test_display(
        [0xf6, 0x57, 0xc3, 0xa8],
        "ldp x22, x21, [sp], #0x30"
    );
    test_display(
        [0xf6, 0xe3, 0x00, 0x91],
        "add x22, sp, #0x38"
    );
    test_display(
        [0xf7, 0xe3, 0x00, 0x91],
        "add x23, sp, #0x38"
    );
    test_display(
        [0xf8, 0x5f, 0x14, 0xa9],
        "stp x24, x23, [sp, #0x140]"
    );
    test_display(
        [0xf8, 0x5f, 0x54, 0xa9],
        "ldp x24, x23, [sp, #0x140]"
    );
    test_display(
        [0xfc, 0x5f, 0x0e, 0xa9],
        "stp x28, x23, [sp, #0xe0]"
    );
    test_display(
        [0xfc, 0x5f, 0x17, 0xa9],
        "stp x28, x23, [sp, #0x170]"
    );
    test_display(
        [0xfc, 0x5f, 0x4e, 0xa9],
        "ldp x28, x23, [sp, #0xe0]"
    );
    test_display(
        [0xfc, 0x5f, 0x57, 0xa9],
        "ldp x28, x23, [sp, #0x170]"
    );
    test_display(
        [0xfc, 0x9b, 0x00, 0xf9],
        "str x28, [sp, #0x130]"
    );
    test_display(
        [0xfd, 0x03, 0x00, 0x91],
        "mov x29, sp"
    );
    test_display(
        [0xfd, 0x03, 0x03, 0x91],
        "add x29, sp, #0xc0"
    );
    test_display(
        [0xfd, 0x43, 0x00, 0x91],
        "add x29, sp, #0x10"
    );
    test_display(
        [0xfd, 0x43, 0x03, 0x91],
        "add x29, sp, #0xd0"
    );
    test_display(
        [0xfd, 0x43, 0x04, 0x91],
        "add x29, sp, #0x110"
    );
    test_display(
        [0xfd, 0x7b, 0x01, 0xa9],
        "stp x29, x30, [sp, #0x10]"
    );
    test_display(
        [0xfd, 0x7b, 0x02, 0xa9],
        "stp x29, x30, [sp, #0x20]"
    );
    test_display(
        [0xfd, 0x7b, 0x03, 0xa9],
        "stp x29, x30, [sp, #0x30]"
    );
    test_display(
        [0xfd, 0x7b, 0x0c, 0xa9],
        "stp x29, x30, [sp, #0xc0]"
    );
    test_display(
        [0xfd, 0x7b, 0x0d, 0xa9],
        "stp x29, x30, [sp, #0xd0]"
    );
    test_display(
        [0xfd, 0x7b, 0x11, 0xa9],
        "stp x29, x30, [sp, #0x110]"
    );
    test_display(
        [0xfd, 0x7b, 0x17, 0xa9],
        "stp x29, x30, [sp, #0x170]"
    );
    test_display(
        [0xfd, 0x7b, 0x1a, 0xa9],
        "stp x29, x30, [sp, #0x1a0]"
    );
    test_display(
        [0xfd, 0x7b, 0x41, 0xa9],
        "ldp x29, x30, [sp, #0x10]"
    );
    test_display(
        [0xfd, 0x7b, 0x42, 0xa9],
        "ldp x29, x30, [sp, #0x20]"
    );
    test_display(
        [0xfd, 0x7b, 0x43, 0xa9],
        "ldp x29, x30, [sp, #0x30]"
    );
    test_display(
        [0xfd, 0x7b, 0x4c, 0xa9],
        "ldp x29, x30, [sp, #0xc0]"
    );
    test_display(
        [0xfd, 0x7b, 0x4d, 0xa9],
        "ldp x29, x30, [sp, #0xd0]"
    );
    test_display(
        [0xfd, 0x7b, 0x51, 0xa9],
        "ldp x29, x30, [sp, #0x110]"
    );
    test_display(
        [0xfd, 0x7b, 0x57, 0xa9],
        "ldp x29, x30, [sp, #0x170]"
    );
    test_display(
        [0xfd, 0x7b, 0x5a, 0xa9],
        "ldp x29, x30, [sp, #0x1a0]"
    );
    test_display(
        [0xfd, 0x7b, 0xbe, 0xa9],
        "stp x29, x30, [sp, #-0x20]!"
    );
    test_display(
        [0xfd, 0x7b, 0xbf, 0xa9],
        "stp x29, x30, [sp, #-0x10]!"
    );
    test_display(
        [0xfd, 0x7b, 0xc1, 0xa8],
        "ldp x29, x30, [sp], #0x10"
    );
    test_display(
        [0xfd, 0x7b, 0xc2, 0xa8],
        "ldp x29, x30, [sp], #0x20"
    );
    test_display(
        [0xfd, 0x83, 0x00, 0x91],
        "add x29, sp, #0x20"
    );
    test_display(
        [0xfd, 0x83, 0x06, 0x91],
        "add x29, sp, #0x1a0"
    );
    test_display(
        [0xfd, 0xc3, 0x00, 0x91],
        "add x29, sp, #0x30"
    );
    test_display(
        [0xfd, 0xc3, 0x05, 0x91],
        "add x29, sp, #0x170"
    );
    test_display(
        [0xff, 0x1f, 0x00, 0xf9],
        "str xzr, [sp, #0x38]"
    );
    test_display(
        [0xe3, 0x07, 0x00, 0x32],
        "mov w3, #0x3"
    );
    test_display(
        [0xff, 0xff, 0x01, 0xa9],
        "stp xzr, xzr, [sp, #0x18]"
    );
    test_display(
        [0x7f, 0x02, 0x00, 0xb9],
        "str wzr, [x19]"
    );
    test_display(
        [0x7f, 0x36, 0x00, 0xf9],
        "str xzr, [x19, #0x68]"
    );
    test_display(
        [0x7f, 0x3a, 0x00, 0xf9],
        "str xzr, [x19, #0x70]"
    );
    test_display(
        [0x7f, 0x3e, 0x00, 0xf9],
        "str xzr, [x19, #0x78]"
    );
    test_display(
        [0x9f, 0x3e, 0x00, 0xf9],
        "str xzr, [x20, #0x78]"
    );
    test_display(
        [0x9f, 0xfe, 0x06, 0xa9],
        "stp xzr, xzr, [x20, #0x68]"
    );
    test_display(
        [0xbf, 0x03, 0x18, 0xf8],
        "stur xzr, [x29, #-0x80]"
    );
    test_display(
        [0xbf, 0x42, 0x00, 0xb1],
        "cmn x21, #0x10"
    );
    test_display(
        [0xbf, 0x83, 0x19, 0xf8],
        "stur xzr, [x29, #-0x68]"
    );
    test_display(
        [0xbf, 0xff, 0x38, 0xa9],
        "stp xzr, xzr, [x29, #-0x78]"
    );
    test_display(
        [0xff, 0x03, 0x01, 0x91],
        "add sp, sp, #0x40"
    );
    test_display(
        [0xff, 0x03, 0x01, 0xd1],
        "sub sp, sp, #0x40"
    );
    test_display(
        [0xff, 0x03, 0x06, 0x91],
        "add sp, sp, #0x180"
    );
    test_display(
        [0xff, 0x03, 0x06, 0xd1],
        "sub sp, sp, #0x180"
    );
    test_display(
        [0xff, 0x43, 0x01, 0xd1],
        "sub sp, sp, #0x50"
    );
    test_display(
        [0xff, 0x43, 0x03, 0x91],
        "add sp, sp, #0xd0"
    );
    test_display(
        [0xff, 0x43, 0x03, 0xd1],
        "sub sp, sp, #0xd0"
    );
    test_display(
        [0xff, 0x83, 0x03, 0x91],
        "add sp, sp, #0xe0"
    );
    test_display(
        [0xff, 0x83, 0x03, 0xd1],
        "sub sp, sp, #0xe0"
    );
    test_display(
        [0xff, 0x83, 0x04, 0x91],
        "add sp, sp, #0x120"
    );
    test_display(
        [0xff, 0x83, 0x04, 0xd1],
        "sub sp, sp, #0x120"
    );
    test_display(
        [0xff, 0xc3, 0x00, 0x91],
        "add sp, sp, #0x30"
    );
    test_display(
        [0xff, 0xc3, 0x00, 0xd1],
        "sub sp, sp, #0x30"
    );
    test_display(
        [0xff, 0xc3, 0x06, 0x91],
        "add sp, sp, #0x1b0"
    );
    test_display(
        [0xff, 0xc3, 0x06, 0xd1],
        "sub sp, sp, #0x1b0"
    );
    test_display(
        [0xe0, 0x03, 0x01, 0xaa],
        "mov x0, x1"
    );
    test_display(
        [0xf4, 0x03, 0x00, 0x2a],
        "mov w20, w0"
    );
    test_display(
        [0xe1, 0x03, 0x1f, 0xaa],
        "mov x1, xzr"
    );
    test_display(
        [0xe2, 0x03, 0x1f, 0xaa],
        "mov x2, xzr"
    );
    test_display(
        [0xe3, 0x03, 0x1f, 0xaa],
        "mov x3, xzr"
    );
    test_display(
        [0xe0, 0x03, 0x13, 0x2a],
        "mov w0, w19"
    );
    test_display(
        [0xe0, 0x03, 0x13, 0xaa],
        "mov x0, x19"
    );
    test_display(
        [0xe0, 0x03, 0x14, 0x2a],
        "mov w0, w20"
    );
    test_display(
        [0xe0, 0x03, 0x14, 0xaa],
        "mov x0, x20"
    );
    test_display(
        [0xe0, 0x03, 0x15, 0xaa],
        "mov x0, x21"
    );
    test_display(
        [0xe0, 0x03, 0x16, 0xaa],
        "mov x0, x22"
    );
    test_display(
        [0xe0, 0x03, 0x17, 0xaa],
        "mov x0, x23"
    );
    test_display(
        [0xe0, 0x03, 0x1f, 0x2a],
        "mov w0, wzr"
    );
    test_display(
        [0xe1, 0x03, 0x00, 0xaa],
        "mov x1, x0"
    );
    test_display(
        [0xe1, 0x03, 0x13, 0xaa],
        "mov x1, x19"
    );
    test_display(
        [0xe1, 0x03, 0x14, 0x2a],
        "mov w1, w20"
    );
    test_display(
        [0xe1, 0x03, 0x14, 0xaa],
        "mov x1, x20"
    );
    test_display(
        [0xe1, 0x03, 0x15, 0x2a],
        "mov w1, w21"
    );
    test_display(
        [0xe1, 0x03, 0x15, 0xaa],
        "mov x1, x21"
    );
    test_display(
        [0xe1, 0x03, 0x16, 0xaa],
        "mov x1, x22"
    );
    test_display(
        [0xe2, 0x03, 0x1f, 0x2a],
        "mov w2, wzr"
    );
    test_display(
        [0xe4, 0x03, 0x00, 0x2a],
        "mov w4, w0"
    );
    test_display(
        [0xe8, 0x03, 0x1f, 0xaa],
        "mov x8, xzr"
    );
    test_display(
        [0xea, 0x03, 0x08, 0x2a],
        "mov w10, w8"
    );
    test_display(
        [0xeb, 0x03, 0x09, 0x2a],
        "mov w11, w9"
    );
    test_display(
        [0xf3, 0x03, 0x00, 0x2a],
        "mov w19, w0"
    );
    test_display(
        [0xf4, 0x03, 0x15, 0x2a],
        "mov w20, w21"
    );
    test_display(
        [0xf4, 0x03, 0x1f, 0x2a],
        "mov w20, wzr"
    );
    test_display(
        [0xf5, 0x03, 0x1f, 0x2a],
        "mov w21, wzr"
    );
    test_display(
        [0xf6, 0x03, 0x14, 0x2a],
        "mov w22, w20"
    );
    test_display(
        [0xf8, 0x03, 0x16, 0x2a],
        "mov w24, w22"
    );
    test_display(
        [0xe2, 0x03, 0x15, 0xaa],
        "mov x2, x21"
    );
    test_display(
        [0xe3, 0x03, 0x14, 0xaa],
        "mov x3, x20"
    );
    test_display(
        [0xe4, 0x03, 0x08, 0xaa],
        "mov x4, x8"
    );
    test_display(
        [0xe4, 0x03, 0x14, 0xaa],
        "mov x4, x20"
    );
    test_display(
        [0xe5, 0x03, 0x00, 0xaa],
        "mov x5, x0"
    );
    test_display(
        [0xe8, 0x03, 0x00, 0xaa],
        "mov x8, x0"
    );
    test_display(
        [0xf3, 0x03, 0x00, 0xaa],
        "mov x19, x0"
    );
    test_display(
        [0xf3, 0x03, 0x01, 0x2a],
        "mov w19, w1"
    );
    test_display(
        [0xf3, 0x03, 0x01, 0xaa],
        "mov x19, x1"
    );
    test_display(
        [0xf3, 0x03, 0x02, 0xaa],
        "mov x19, x2"
    );
    test_display(
        [0xf3, 0x0b, 0x00, 0xf9],
        "str x19, [sp, #0x10]"
    );
    test_display(
        [0xf4, 0x03, 0x00, 0xaa],
        "mov x20, x0"
    );
    test_display(
        [0xf4, 0x03, 0x01, 0xaa],
        "mov x20, x1"
    );
    test_display(
        [0xf5, 0x03, 0x00, 0xaa],
        "mov x21, x0"
    );
    test_display(
        [0xf6, 0x03, 0x00, 0xaa],
        "mov x22, x0"
    );
    test_display(
        [0xf7, 0x03, 0x00, 0xaa],
        "mov x23, x0"
    );
}

static INSTRUCTION_BYTES: [u8; 4 * 61] = [
    0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x20, 0xd4,
    0x00, 0x00, 0x80, 0x12,
    0x00, 0x01, 0x3f, 0xd6,
    0x00, 0x02, 0x00, 0x36,
    0x00, 0x03, 0x00, 0x35,
    0x00, 0x04, 0x00, 0x36,
    0x00, 0x04, 0x40, 0xf9,
    0x00, 0x07, 0x00, 0x34,
    0x00, 0x08, 0x47, 0xf9,
    0x00, 0x0b, 0x80, 0x52,
    0x00, 0x14, 0x42, 0xf9,
    0x00, 0x1b, 0x80, 0x52,
    0x00, 0x20, 0x40, 0xf9,
    0x00, 0x24, 0x47, 0xf9,
    0x00, 0x2b, 0x03, 0x90,
    0x00, 0x34, 0x42, 0xf9,
    0x00, 0x39, 0x04, 0xb0,
    0x00, 0x3b, 0x04, 0xb0,
    0x00, 0x3c, 0x43, 0xf9,
    0x00, 0x44, 0x44, 0xf9,
    0x00, 0x50, 0x14, 0x91,
    0x00, 0x54, 0x44, 0xf9,
    0x00, 0x58, 0x42, 0xf9,
    0x00, 0x5c, 0x44, 0xf9,
    0x00, 0x60, 0x1e, 0x91,
    0x00, 0x70, 0x47, 0xf9,
    0x00, 0x80, 0x1e, 0x91,
    0x00, 0x80, 0x44, 0xf9,
    0x00, 0x84, 0x47, 0xf9,
    0x00, 0xac, 0x40, 0xf9,
    0x00, 0xc0, 0x09, 0x91,
    0x00, 0xc4, 0x45, 0xf9,
    0x00, 0xcc, 0x41, 0xf9,
    0x00, 0xdc, 0x35, 0x91,
    0x00, 0xf4, 0x47, 0xf9,
    0x01, 0x00, 0x00, 0x14,
    0x01, 0x00, 0x40, 0xf9,
    0x01, 0x05, 0x40, 0xf9,
    0x01, 0xfb, 0x4b, 0x95,
    0x02, 0x00, 0x00, 0x14,
    0x02, 0x00, 0x00, 0x90,
    0x02, 0x00, 0x80, 0x92,
    0x02, 0x04, 0xf8, 0x97,
    0x02, 0x2c, 0x80, 0x52,
    0x08, 0x00, 0x40, 0xf9,
    0x08, 0x01, 0x40, 0xf9,
    0x08, 0x05, 0x40, 0xf9,
    0x08, 0x06, 0xf8, 0x97,
    0x08, 0x09, 0x40, 0xf9,
    0x08, 0x1d, 0x40, 0x92,
    0x08, 0x1f, 0x00, 0x13,
    0x08, 0x21, 0x0a, 0x91,
    0x08, 0x41, 0x00, 0x91,
    0x08, 0x41, 0x40, 0xf9,
    0x08, 0x81, 0x0a, 0x91,
    0x08, 0xa1, 0x11, 0x91,
    0x08, 0xc1, 0x1e, 0x91,
    0x08, 0xdd, 0x46, 0xf9,
    0x08, 0xe1, 0x0e, 0x91,
    0x08, 0xf2, 0xff, 0x36,
];

#[test]
fn test_decode_span() {
    let mut i = 0u64;
    while i < INSTRUCTION_BYTES.len() as u64 {
        let mut reader = yaxpeax_arch::U8Reader::new(&INSTRUCTION_BYTES[i as usize..]);
        let res = <ARMv8 as Arch>::Decoder::default().decode(&mut reader);
        if let Err(DecodeError::IncompleteDecoder) = res {
            i += 4;
            continue;
        }
        let instr = res.unwrap();
        println!(
            "Decoded {:02x}{:02x}{:02x}{:02x}: {}", //{:?}\n  {}",
            INSTRUCTION_BYTES[i as usize],
            INSTRUCTION_BYTES[i as usize + 1],
            INSTRUCTION_BYTES[i as usize + 2],
            INSTRUCTION_BYTES[i as usize + 3],
//            instr,
            instr);
        i += 4;
    }
}

/*
use test::Bencher;
#[bench]
pub fn bench_60000_instrs(b: &mut Bencher) {
    b.iter(|| {
        for i in (0..1000) {
            let mut iter = INSTRUCTION_BYTES.iter().map(|x| *x);
            let decoder = <ARMv8 as Arch>::Decoder::default();
            let mut result = Instruction::default();
            loop {
                match decoder.decode_into(&mut result, &mut iter) {
                    Ok(result) => {
                        test::black_box(&result);
                    },
                    Err(_) => {
                        break;
                    }
                }
            }
        }
    });
}
*/

enum ErrorDesc {
    IncorrectDecode([u8; 4], String, String),
    FailedDecode([u8; 4], String, DecodeError),
}

impl fmt::Display for ErrorDesc {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        match self {
            ErrorDesc::IncorrectDecode(bytes, expected, actual) => {
                writeln!(f, "  decoded incorrect instruction from {:02x}{:02x}{:02x}{:02x}", bytes[0], bytes[1], bytes[2], bytes[3])?;
                writeln!(f, "    expected \"{}\"", expected)?;
                writeln!(f, "    got      \"{}\"", actual)?;
            },
            ErrorDesc::FailedDecode(bytes, expected, err) => {
                writeln!(f, "  decode error for {:02x}{:02x}{:02x}{:02x}", bytes[0], bytes[1], bytes[2], bytes[3])?;
                writeln!(f, "    expected \"{}\"", expected)?;
                writeln!(f, "    got      \"{:?}\"", err)?;
            },
        }

        Ok(())
    }
}

fn run_tests(cases: &[([u8; 4], &'static str)]) -> Vec<ErrorDesc> {
    let decoder = <ARMv8 as Arch>::Decoder::default();

    let mut errors = Vec::new();

    for (bytes, text) in cases {
        let mut reader = yaxpeax_arch::U8Reader::new(&bytes[..]);
        match decoder.decode(&mut reader) {
            Ok(result) => {
                if &result.to_string() != text {
                    errors.push(ErrorDesc::IncorrectDecode(*bytes, text.to_string(), result.to_string()));
                }
            },
            Err(err) => {
                errors.push(ErrorDesc::FailedDecode(*bytes, text.to_string(), err));
            }
        }
    }

    errors
}

#[test]
fn test_openblas_arithmetic() {
    const TESTS: &[([u8; 4], &'static str)] = &[
        ([0x61, 0x06, 0x13, 0x0b], "add w1, w19, w19, lsl #1"),
        ([0x05, 0x07, 0x13, 0x0b], "add w5, w24, w19, lsl #1"),
        ([0x08, 0x7d, 0x48, 0x0b], "add w8, w8, w8, lsr #31"),
        ([0x00, 0x04, 0x00, 0x11], "add w0, w0, #0x1"),
        ([0x00, 0x04, 0x01, 0x11], "add w0, w0, #0x41"),
        ([0x5b, 0xff, 0x01, 0x11], "add w27, w26, #0x7f"),
        ([0x02, 0xfc, 0x3f, 0x11], "add w2, w0, #0xfff"),
        ([0x00, 0x00, 0x60, 0x11], "add w0, w0, #0x800, lsl #12"),
        ([0x0a, 0x0c, 0x00, 0x31], "adds w10, w0, #0x3"),
        ([0x82, 0x00, 0x00, 0x8b], "add x2, x4, x0"),
        ([0x00, 0x35, 0x00, 0x8b], "add x0, x8, x0, lsl #13"),
        ([0x00, 0x00, 0x01, 0x8b], "add x0, x0, x1"),
        ([0xde, 0x03, 0x12, 0x8b], "add x30, x30, x18"),
        ([0x03, 0x04, 0x12, 0x8b], "add x3, x0, x18, lsl #1"),
        ([0x56, 0x06, 0x12, 0x8b], "add x22, x18, x18, lsl #1"),
        ([0x12, 0x08, 0x12, 0x8b], "add x18, x0, x18, lsl #2"),
        ([0x32, 0x08, 0x12, 0x8b], "add x18, x1, x18, lsl #2"),
        ([0x92, 0x13, 0x12, 0x8b], "add x18, x28, x18, lsl #4"),
        ([0x4f, 0x14, 0x12, 0x8b], "add x15, x2, x18, lsl #5"),
        ([0x42, 0x18, 0x12, 0x8b], "add x2, x2, x18, lsl #6"),
        ([0xc6, 0x1c, 0x12, 0x8b], "add x6, x6, x18, lsl #7"),
        ([0x62, 0xce, 0x21, 0x8b], "add x2, x19, w1, sxtw #3"),
        ([0x87, 0xd3, 0x21, 0x8b], "add x7, x28, w1, sxtw #4"),
        ([0x2a, 0x41, 0x22, 0x8b], "add x10, x9, w2"),
        ([0x18, 0x4f, 0x22, 0x8b], "add x24, x24, w2, lsl #3"),
        ([0x82, 0x4f, 0x22, 0x8b], "add x2, x28, w2, lsl #3"),
        ([0xe2, 0x50, 0x22, 0x8b], "add x2, x7, w2, lsl #4"),
        ([0x20, 0x04, 0x84, 0x8b], "add x0, x1, x4, asr #1"),
        ([0x14, 0x04, 0x94, 0x8b], "add x20, x0, x20, asr #1"),
        ([0xe6, 0x62, 0x00, 0x91], "add x6, x23, #0x18"),
        ([0x00, 0x80, 0x04, 0x91], "add x0, x0, #0x120"),
        ([0xf8, 0x63, 0x06, 0x91], "add x24, sp, #0x198"),
        ([0xb7, 0xfc, 0x3f, 0x91], "add x23, x5, #0xfff"),
        ([0x5a, 0xff, 0x3f, 0x91], "add x26, x26, #0xfff"),
        ([0x7b, 0xff, 0x3f, 0x91], "add x27, x27, #0xfff"),
        ([0x9c, 0xff, 0x3f, 0x91], "add x28, x28, #0xfff"),
        ([0xc6, 0x08, 0x40, 0x91], "add x6, x6, #0x2, lsl #12"),
        ([0xe8, 0x13, 0x40, 0x91], "add x8, sp, #0x4, lsl #12"),
        ([0xe2, 0x43, 0x40, 0x91], "add x2, sp, #0x10, lsl #12"),
        ([0xff, 0x0f, 0x42, 0x91], "add sp, sp, #0x83, lsl #12"),
        ([0xe1, 0x13, 0x42, 0x91], "add x1, sp, #0x84, lsl #12"),
        ([0xff, 0x13, 0x42, 0x91], "add sp, sp, #0x84, lsl #12"),
        ([0x63, 0x02, 0x60, 0x91], "add x3, x19, #0x800, lsl #12"),
        ([0x03, 0x03, 0x60, 0x91], "add x3, x24, #0x800, lsl #12"),
        ([0x14, 0x00, 0x13, 0xab], "adds x20, x0, x19"),
        ([0x37, 0x01, 0x1c, 0xab], "adds x23, x9, x28"),
        ([0x0b, 0x08, 0x00, 0x4b], "sub w11, w0, w0, lsl #2"),
        ([0x00, 0x00, 0x01, 0x4b], "sub w0, w0, w1"),
        ([0xfc, 0x03, 0x08, 0x4b], "neg w28, w8"),
        ([0x00, 0x04, 0x08, 0x4b], "sub w0, w0, w8, lsl #1"),
        ([0x81, 0x09, 0x0c, 0x4b], "sub w1, w12, w12, lsl #2"),
        ([0xe0, 0x02, 0x0d, 0x4b], "sub w0, w23, w13"),
        ([0xfb, 0x03, 0x15, 0x4b], "neg w27, w21"),
        ([0x63, 0x04, 0x15, 0x4b], "sub w3, w3, w21, lsl #1"),
        ([0xfc, 0x03, 0x1c, 0x4b], "neg w28, w28"),
        ([0x00, 0x04, 0x1c, 0x4b], "sub w0, w0, w28, lsl #1"),
        ([0x00, 0x7c, 0x81, 0x4b], "sub w0, w0, w1, asr #31"),
        ([0x6e, 0x7c, 0x8e, 0x4b], "sub w14, w3, w14, asr #31"),
        ([0x4c, 0x04, 0x00, 0x51], "sub w12, w2, #0x1"),
        ([0x67, 0x0f, 0x00, 0x51], "sub w7, w27, #0x3"),
        ([0x04, 0x11, 0x00, 0x51], "sub w4, w8, #0x4"),
        ([0x42, 0x1c, 0x00, 0x51], "sub w2, w2, #0x7"),
        ([0xc5, 0x20, 0x00, 0x51], "sub w5, w6, #0x8"),
        ([0x08, 0x25, 0x00, 0x51], "sub w8, w8, #0x9"),
        ([0x00, 0x2c, 0x00, 0x51], "sub w0, w0, #0xb"),
        ([0x00, 0x30, 0x00, 0x51], "sub w0, w0, #0xc"),
        ([0x2d, 0x30, 0x00, 0x51], "sub w13, w1, #0xc"),
        ([0x21, 0x34, 0x00, 0x51], "sub w1, w1, #0xd"),
        ([0xb8, 0x46, 0x00, 0x51], "sub w24, w21, #0x11"),
        ([0xe0, 0x84, 0x01, 0x51], "sub w0, w7, #0x61"),
        ([0x0c, 0x01, 0x00, 0x6b], "subs w12, w8, w0"),
        ([0xe2, 0x03, 0x00, 0x6b], "negs w2, w0"),
        ([0x21, 0x04, 0x00, 0x71], "subs w1, w1, #0x1"),
        ([0x25, 0x04, 0x00, 0x71], "subs w5, w1, #0x1"),
        ([0x2c, 0x04, 0x00, 0x71], "subs w12, w1, #0x1"),
        ([0x3a, 0x04, 0x00, 0x71], "subs w26, w1, #0x1"),
        ([0xe3, 0x13, 0x02, 0xcb], "neg x3, x2, lsl #4"),
        ([0xeb, 0x0f, 0x03, 0xcb], "neg x11, x3, lsl #3"),
        ([0x75, 0x08, 0x02, 0xcb], "sub x21, x3, x2, lsl #2"),
        ([0xa3, 0x00, 0x03, 0xcb], "sub x3, x5, x3"),
        ([0x81, 0x08, 0x03, 0xcb], "sub x1, x4, x3, lsl #2"),
        ([0x70, 0x12, 0x15, 0xcb], "sub x16, x19, x21, lsl #4"),
        ([0xc6, 0x40, 0x20, 0xcb], "sub x6, x6, w0"),
        ([0x24, 0x50, 0x20, 0xcb], "sub x4, x1, w0, lsl #4"),
        ([0x8a, 0xc1, 0x20, 0xcb], "sub x10, x12, w0, sxtw"),
        ([0xbb, 0xcc, 0x20, 0xcb], "sub x27, x5, w0, sxtw #3"),
        ([0xd6, 0x04, 0x00, 0xd1], "sub x22, x6, #0x1"),
        ([0xc7, 0x40, 0x10, 0xd1], "sub x7, x6, #0x410"),
        ([0x7b, 0x43, 0x10, 0xd1], "sub x27, x27, #0x410"),
        ([0xff, 0xc3, 0x27, 0xd1], "sub sp, sp, #0x9f0"),
        ([0xec, 0x03, 0x42, 0xd1], "sub x12, sp, #0x80, lsl #12"),
        ([0x97, 0x02, 0x1b, 0xeb], "subs x23, x20, x27"),
        ([0x98, 0x02, 0x1b, 0xeb], "subs x24, x20, x27"),
        ([0x6b, 0x05, 0x00, 0xf1], "subs x11, x11, #0x1"),
        ([0x5a, 0x07, 0x00, 0xf1], "subs x26, x26, #0x1"),
        ([0x08, 0x09, 0x00, 0xf1], "subs x8, x8, #0x2"),
    ];

    let errs = run_tests(TESTS);

    for err in errs.iter() {
        println!("{}", err);
    }

    assert!(errs.is_empty());
}

#[test]
fn test_openblas_bitwise() {
    const TESTS: &[([u8; 4], &'static str)] = &[
        ([0x94, 0x7c, 0x01, 0x13], "asr w20, w4, #0x1"),
        ([0x0a, 0x7d, 0x02, 0x13], "asr w10, w8, #0x2"),
        ([0x00, 0x7c, 0x0c, 0x13], "asr w0, w0, #0xc"),
        ([0x20, 0x04, 0x81, 0x13], "ror w0, w1, #0x1"),
        ([0x61, 0x21, 0xc0, 0x1a], "lsl w1, w11, w0"),
        ([0x00, 0x20, 0xc1, 0x1a], "lsl w0, w0, w1"),
        ([0x94, 0x22, 0xc1, 0x1a], "lsl w20, w20, w1"),
        ([0x63, 0x20, 0xc2, 0x1a], "lsl w3, w3, w2"),
        ([0x03, 0x7c, 0x01, 0x53], "lsr w3, w0, #0x1"),
        ([0x23, 0x7c, 0x02, 0x53], "lsr w3, w1, #0x2"),
        ([0x00, 0x7c, 0x06, 0x53], "lsr w0, w0, #0x6"),
        ([0x42, 0x7c, 0x07, 0x53], "lsr w2, w2, #0x7"),
        ([0x00, 0x4c, 0x14, 0x53], "lsl w0, w0, #0xc"),
        ([0xe0, 0x56, 0x16, 0x53], "lsl w0, w23, #0xa"),
        ([0x20, 0x7c, 0x18, 0x53], "lsr w0, w1, #0x18"),
        ([0x35, 0x6c, 0x1c, 0x53], "lsl w21, w1, #0x4"),
        ([0x2c, 0x70, 0x1d, 0x53], "lsl w12, w1, #0x3"),
        ([0xa8, 0x74, 0x1e, 0x53], "lsl w8, w5, #0x2"),
        ([0x23, 0x7b, 0x1f, 0x53], "lsl w3, w25, #0x1"),
        ([0x04, 0x7f, 0x1f, 0x53], "lsr w4, w24, #0x1f"),
        ([0x05, 0xfe, 0x41, 0xd3], "lsr x5, x16, #0x1"),
        ([0x22, 0x3c, 0x44, 0xd3], "ubfx x2, x1, #0x4, #0xc"),
        ([0xb5, 0xfe, 0x46, 0xd3], "lsr x21, x21, #0x6"),
        ([0xe7, 0xfc, 0x60, 0xd3], "lsr x7, x7, #0x20"),
        ([0x21, 0xfc, 0x61, 0xd3], "lsr x1, x1, #0x21"),
        ([0x00, 0xfc, 0x63, 0xd3], "lsr x0, x0, #0x23"),
        ([0xa6, 0xd4, 0x76, 0xd3], "lsl x6, x5, #0xa"),
        ([0x03, 0x08, 0x7a, 0xd3], "ubfiz x3, x0, #0x6, #0x3"),
        ([0xa1, 0x7e, 0x79, 0x93], "sbfiz x1, x21, #0x7, #0x20"),
        ([0x01, 0x7c, 0x7a, 0x93], "sbfiz x1, x0, #0x6, #0x20"),
        ([0x9a, 0x7c, 0x7d, 0x93], "sbfiz x26, x4, #0x3, #0x20"),
        ([0xc3, 0x0f, 0x7a, 0xd3], "ubfiz x3, x30, #0x6, #0x4"),
        ([0x01, 0xe4, 0x7a, 0xd3], "lsl x1, x0, #0x6"),
        ([0x65, 0xe8, 0x7b, 0xd3], "lsl x5, x3, #0x5"),
        ([0x79, 0xf3, 0x7d, 0xd3], "lsl x25, x27, #0x3"),
        ([0x81, 0x7d, 0x7f, 0xd3], "ubfiz x1, x12, #0x1, #0x20"),
        ([0x0e, 0xf8, 0x7f, 0xd3], "lsl x14, x0, #0x1"),
        ([0xe9, 0x22, 0xce, 0x9a], "lsl x9, x23, x14"),
        ([0xca, 0xff, 0x7f, 0xd3], "lsr x10, x30, #0x3f"),
        ([0xd3, 0x7f, 0x40, 0x93], "sxtw x19, w30"),
        ([0x00, 0xfc, 0x62, 0x93], "asr x0, x0, #0x22"),
        ([0x20, 0x00, 0x00, 0x0a], "and w0, w1, w0"),
        ([0x28, 0x00, 0x00, 0x0a], "and w8, w1, w0"),
        ([0x29, 0x00, 0x00, 0x0a], "and w9, w1, w0"),
        ([0xb5, 0x02, 0x22, 0x0a], "bic w21, w21, w2"),
        ([0x00, 0x2c, 0x00, 0x12], "and w0, w0, #0xfff"),
        ([0x42, 0x2c, 0x00, 0x12], "and w2, w2, #0xfff"),
        ([0x00, 0x00, 0x01, 0x12], "and w0, w0, #0x80000000"),
        ([0x21, 0x00, 0x01, 0x12], "and w1, w1, #0x80000000"),
        ([0x10, 0x00, 0x18, 0x12], "and w16, w0, #0x100"),
        ([0x11, 0x00, 0x18, 0x12], "and w17, w0, #0x100"),
        ([0x06, 0x79, 0x1a, 0x12], "and w6, w8, #0xffffffdf"),
        ([0x08, 0x79, 0x1a, 0x12], "and w8, w8, #0xffffffdf"),
        ([0x02, 0x70, 0x1d, 0x12], "and w2, w0, #0xfffffff8"),
        ([0x00, 0x74, 0x1e, 0x12], "and w0, w0, #0xfffffffc"),
        ([0x02, 0x74, 0x1e, 0x12], "and w2, w0, #0xfffffffc"),
        ([0x00, 0x78, 0x1e, 0x12], "and w0, w0, #0xfffffffd"),
        ([0x02, 0x78, 0x1e, 0x12], "and w2, w0, #0xfffffffd"),
        ([0xc6, 0x78, 0x1e, 0x12], "and w6, w6, #0xfffffffd"),
        ([0x10, 0x7a, 0x1e, 0x12], "and w16, w16, #0xfffffffd"),
        ([0x39, 0x7b, 0x1f, 0x12], "and w25, w25, #0xfffffffe"),
        ([0x63, 0x00, 0x05, 0x6a], "ands w3, w3, w5"),
        ([0x00, 0x04, 0x00, 0x72], "ands w0, w0, #0x3"),
        ([0xc6, 0x04, 0x00, 0x72], "ands w6, w6, #0x3"),
        ([0x00, 0x78, 0x1e, 0x72], "ands w0, w0, #0xfffffffd"),
        ([0x21, 0x78, 0x1e, 0x72], "ands w1, w1, #0xfffffffd"),
        ([0x40, 0x00, 0x00, 0x8a], "and x0, x2, x0"),
        ([0x84, 0x00, 0x1c, 0x8a], "and x4, x4, x28"),
        ([0x01, 0x00, 0x40, 0x92], "and x1, x0, #0x1"),
        ([0x02, 0x00, 0x40, 0x92], "and x2, x0, #0x1"),
        ([0xa5, 0xf0, 0x7d, 0x92], "and x5, x5, #0xfffffffffffffff8"),
        ([0x00, 0x03, 0x7e, 0x92], "and x0, x24, #0x4"),
        ([0xc1, 0x06, 0x7e, 0x92], "and x1, x22, #0xc"),
        ([0x42, 0xf4, 0x7e, 0x92], "and x2, x2, #0xfffffffffffffffc"),
        ([0x01, 0x00, 0x7f, 0x92], "and x1, x0, #0x2"),
        ([0xe0, 0x01, 0x7f, 0x92], "and x0, x15, #0x2"),
        ([0x75, 0x02, 0x7f, 0x92], "and x21, x19, #0x2"),
        ([0x76, 0x02, 0x7f, 0x92], "and x22, x19, #0x2"),
        ([0x8e, 0x02, 0x7f, 0x92], "and x14, x20, #0x2"),
        ([0x9a, 0x02, 0x7f, 0x92], "and x26, x20, #0x2"),
        ([0x9b, 0x02, 0x7f, 0x92], "and x27, x20, #0x2"),
        ([0xc1, 0x06, 0x7f, 0x92], "and x1, x22, #0x6"),
        ([0x00, 0x07, 0x7f, 0x92], "and x0, x24, #0x6"),
        ([0x01, 0x08, 0x7f, 0x92], "and x1, x0, #0xe"),
        ([0xc1, 0x0a, 0x7f, 0x92], "and x1, x22, #0xe"),
        ([0x6b, 0xfa, 0x7f, 0x92], "and x11, x19, #0xfffffffffffffffe"),
        ([0x80, 0xfb, 0x7f, 0x92], "and x0, x28, #0xfffffffffffffffe"),
        ([0x0c, 0x04, 0x40, 0xf2], "ands x12, x0, #0x3"),
        ([0x88, 0x0a, 0x40, 0xf2], "ands x8, x20, #0x7"),
        ([0x01, 0x10, 0x40, 0xf2], "ands x1, x0, #0x1f"),
        ([0x05, 0x10, 0x40, 0xf2], "ands x5, x0, #0x1f"),
        ([0x0c, 0x10, 0x40, 0xf2], "ands x12, x0, #0x1f"),
        ([0x48, 0x10, 0x40, 0xf2], "ands x8, x2, #0x1f"),
        ([0x05, 0x14, 0x40, 0xf2], "ands x5, x0, #0x3f"),
        ([0x48, 0x18, 0x40, 0xf2], "ands x8, x2, #0x7f"),
        ([0x02, 0xe8, 0x7b, 0xf2], "ands x2, x0, #0xffffffffffffffe0"),
        ([0x07, 0xf4, 0x7e, 0xf2], "ands x7, x0, #0xfffffffffffffffc"),
        ([0x20, 0x00, 0x00, 0x4a], "eor w0, w1, w0"),
        ([0x99, 0x02, 0x04, 0x4a], "eor w25, w20, w4"),
        ([0xa6, 0x00, 0x00, 0x52], "eor w6, w5, #0x1"),
        ([0x48, 0x01, 0x08, 0xca], "eor x8, x10, x8"),
        ([0xe7, 0xf0, 0x7d, 0xd2], "eor x7, x7, #0xfffffffffffffff8"),
        ([0x81, 0x12, 0x17, 0x2a], "orr w1, w20, w23, lsl #4"),
        ([0x29, 0x03, 0x1c, 0x2a], "orr w9, w25, w28"),
        ([0x73, 0x7d, 0x4d, 0x2a], "orr w19, w11, w13, lsr #31"),
        ([0x00, 0x7c, 0x54, 0x2a], "orr w0, w0, w20, lsr #31"),
        ([0x20, 0x00, 0x00, 0x32], "orr w0, w1, #0x1"),
        ([0x80, 0x02, 0x00, 0x32], "orr w0, w20, #0x1"),
        ([0x12, 0x00, 0x11, 0x32], "orr w18, w0, #0x8000"),
        ([0x1e, 0x00, 0x11, 0x32], "orr w30, w0, #0x8000"),
        ([0x08, 0x01, 0x1c, 0x32], "orr w8, w8, #0x10"),
        ([0x81, 0x02, 0x1c, 0x32], "orr w1, w20, #0x10"),
        ([0x82, 0x02, 0x1c, 0x32], "orr w2, w20, #0x10"),
        ([0x94, 0x02, 0x1c, 0x32], "orr w20, w20, #0x10"),
        ([0x20, 0x00, 0x1e, 0x32], "orr w0, w1, #0x4"),
        ([0x73, 0x02, 0x1e, 0x32], "orr w19, w19, #0x4"),
        ([0x80, 0x02, 0x1e, 0x32], "orr w0, w20, #0x4"),
        ([0x94, 0x02, 0x1e, 0x32], "orr w20, w20, #0x4"),
    ];

    let errs = run_tests(TESTS);

    for err in errs.iter() {
        println!("{}", err);
    }

    assert!(errs.is_empty());
}

#[test]
fn test_openblas_simd_loadstore() {
    const TESTS: &[([u8; 4], &'static str)] = &[
        ([0x00, 0xd8, 0x21, 0x5e], "scvtf s0, s0"),
        ([0x82, 0xd8, 0x61, 0x5e], "scvtf d2, d4"),
        ([0x01, 0x00, 0x62, 0x9e], "scvtf d1, x0"),
        ([0x03, 0x00, 0x62, 0x9e], "scvtf d3, x0"),
        ([0x69, 0x03, 0x62, 0x9e], "scvtf d9, x27"),
        ([0x88, 0x03, 0x62, 0x9e], "scvtf d8, x28"),
        ([0x22, 0x00, 0x22, 0x1e], "scvtf s2, w1"),
        ([0xac, 0x02, 0x22, 0x1e], "scvtf s12, w21"),
        ([0x00, 0x00, 0x62, 0x1e], "scvtf d0, w0"),
        ([0x8a, 0x03, 0x62, 0x1e], "scvtf d10, w28"),
        ([0x03, 0xe4, 0x00, 0x2f], "movi d3, #0x0"),
        ([0x88, 0x28, 0x00, 0x0c], "st1 {v8.2s, v9.2s, v10.2s, v11.2s}, [x4]"),
        ([0x80, 0x2c, 0x00, 0x0c], "st1 {v0.1d, v1.1d, v2.1d, v3.1d}, [x4]"),
        ([0x84, 0x2c, 0x00, 0x0c], "st1 {v4.1d, v5.1d, v6.1d, v7.1d}, [x4]"),
        ([0x80, 0x2d, 0x00, 0x0c], "st1 {v0.1d, v1.1d, v2.1d, v3.1d}, [x12]"),
        ([0x20, 0x2e, 0x00, 0x0c], "st1 {v0.1d, v1.1d, v2.1d, v3.1d}, [x17]"),
        ([0x24, 0x2e, 0x00, 0x0c], "st1 {v4.1d, v5.1d, v6.1d, v7.1d}, [x17]"),
        ([0x60, 0x78, 0x00, 0x0c], "st1 {v0.2s}, [x3]"),
        ([0x62, 0x78, 0x00, 0x0c], "st1 {v2.2s}, [x3]"),
        ([0x88, 0x79, 0x00, 0x0c], "st1 {v8.2s}, [x12]"),
        ([0xac, 0x79, 0x00, 0x0c], "st1 {v12.2s}, [x13]"),
        ([0xc8, 0x79, 0x00, 0x0c], "st1 {v8.2s}, [x14]"),
        ([0xa0, 0x89, 0x00, 0x0c], "st2 {v0.2s, v1.2s}, [x13]"),
        ([0xa4, 0x89, 0x00, 0x0c], "st2 {v4.2s, v5.2s}, [x13]"),
        ([0x88, 0xa9, 0x00, 0x0c], "st1 {v8.2s, v9.2s}, [x12]"),
        ([0xa0, 0xa9, 0x00, 0x0c], "st1 {v0.2s, v1.2s}, [x13]"),
        ([0xa2, 0xa9, 0x00, 0x0c], "st1 {v2.2s, v3.2s}, [x13]"),
        ([0xac, 0xa9, 0x00, 0x0c], "st1 {v12.2s, v13.2s}, [x13]"),
        ([0xc4, 0xa9, 0x00, 0x0c], "st1 {v4.2s, v5.2s}, [x14]"),
        ([0xc6, 0xa9, 0x00, 0x0c], "st1 {v6.2s, v7.2s}, [x14]"),
        ([0xc8, 0xa9, 0x00, 0x0c], "st1 {v8.2s, v9.2s}, [x14]"),
        ([0x88, 0x79, 0x40, 0x0c], "ld1 {v8.2s}, [x12]"),
        ([0xac, 0x79, 0x40, 0x0c], "ld1 {v12.2s}, [x13]"),
        ([0x68, 0x89, 0x40, 0x0c], "ld2 {v8.2s, v9.2s}, [x11]"),
        ([0x6a, 0x89, 0x40, 0x0c], "ld2 {v10.2s, v11.2s}, [x11]"),
        ([0xa4, 0x89, 0x40, 0x0c], "ld2 {v4.2s, v5.2s}, [x13]"),
        ([0xa2, 0xa9, 0x40, 0x0c], "ld1 {v2.2s, v3.2s}, [x13]"),
        ([0xac, 0xa9, 0x40, 0x0c], "ld1 {v12.2s, v13.2s}, [x13]"),
        ([0xa5, 0x79, 0x9f, 0x0c], "st1 {v5.2s}, [x13], #0x8"),
        ([0x45, 0x79, 0xc2, 0x0c], "ld1 {v5.2s}, [x10], x2"),
        ([0x20, 0x78, 0xdf, 0x0c], "ld1 {v0.2s}, [x1], #0x8"),
        ([0xcc, 0x85, 0x00, 0x0d], "st1 {v12.d}[0], [x14]"),
        ([0xa8, 0x91, 0x00, 0x0d], "st1 {v8.s}[1], [x13]"),
        ([0xa0, 0x81, 0x20, 0x0d], "st2 {v0.s, v1.s}[0], [x13]"),
        ([0x88, 0x81, 0x40, 0x0d], "ld1 {v8.s}[0], [x12]"),
        ([0xa8, 0x91, 0x40, 0x0d], "ld1 {v8.s}[1], [x13]"),
        ([0xa0, 0x85, 0x60, 0x0d], "ld2 {v0.d, v1.d}[0], [x13]"),
        ([0x43, 0x81, 0x82, 0x0d], "st1 {v3.s}[0], [x10], x2"),
        ([0x64, 0x90, 0x9f, 0x0d], "st1 {v4.s}[1], [x3], #0x4"),
        ([0x22, 0x84, 0xc2, 0x0d], "ld1 {v2.d}[0], [x1], x2"),
        ([0x61, 0x80, 0xc4, 0x0d], "ld1 {v1.s}[0], [x3], x4"),
        ([0x24, 0xc9, 0xdf, 0x0d], "ld1r {v4.2s}, [x9], #0x4"), // TODO: could use a test for "ld1r {v4.2s}, [x9]"
        ([0x88, 0x28, 0x00, 0x4c], "st1 {v8.4s, v9.4s, v10.4s, v11.4s}, [x4]"),
        ([0x60, 0x2d, 0x00, 0x4c], "st1 {v0.2d, v1.2d, v2.2d, v3.2d}, [x11]"),
        ([0x9c, 0x2e, 0x00, 0x4c], "st1 {v28.2d, v29.2d, v30.2d, v31.2d}, [x20]"),
        ([0x60, 0x7c, 0x00, 0x4c], "st1 {v0.2d}, [x3]"),
        ([0x62, 0x7c, 0x00, 0x4c], "st1 {v2.2d}, [x3]"),
        ([0x88, 0x7d, 0x00, 0x4c], "st1 {v8.2d}, [x12]"),
        ([0xac, 0x7d, 0x00, 0x4c], "st1 {v12.2d}, [x13]"),
        ([0xc8, 0x7d, 0x00, 0x4c], "st1 {v8.2d}, [x14]"),
        ([0xe6, 0x8d, 0x00, 0x4c], "st2 {v6.2d, v7.2d}, [x15]"),
        ([0x00, 0xa3, 0x00, 0x4c], "st1 {v0.16b, v1.16b}, [x24]"),
        ([0x60, 0xa3, 0x00, 0x4c], "st1 {v0.16b, v1.16b}, [x27]"),
        ([0x00, 0xa9, 0x00, 0x4c], "st1 {v0.4s, v1.4s}, [x8]"),
        ([0x44, 0xad, 0x00, 0x4c], "st1 {v4.2d, v5.2d}, [x10]"),
        ([0x66, 0xad, 0x00, 0x4c], "st1 {v6.2d, v7.2d}, [x11]"),
        ([0xc8, 0xad, 0x00, 0x4c], "st1 {v8.2d, v9.2d}, [x14]"),
        ([0xec, 0xad, 0x00, 0x4c], "st1 {v12.2d, v13.2d}, [x15]"),
        ([0x21, 0x28, 0x40, 0x4c], "ld1 {v1.4s, v2.4s, v3.4s, v4.4s}, [x1]"),
        ([0x22, 0x2c, 0x40, 0x4c], "ld1 {v2.2d, v3.2d, v4.2d, v5.2d}, [x1]"),
        ([0x61, 0x2c, 0x40, 0x4c], "ld1 {v1.2d, v2.2d, v3.2d, v4.2d}, [x3]"),
        ([0xb0, 0x2c, 0x40, 0x4c], "ld1 {v16.2d, v17.2d, v18.2d, v19.2d}, [x5]"),
        ([0x80, 0x2d, 0x40, 0x4c], "ld1 {v0.2d, v1.2d, v2.2d, v3.2d}, [x12]"),
        ([0xa4, 0x2d, 0x40, 0x4c], "ld1 {v4.2d, v5.2d, v6.2d, v7.2d}, [x13]"),
        ([0x68, 0x79, 0x40, 0x4c], "ld1 {v8.4s}, [x11]"),
        ([0x6c, 0x79, 0x40, 0x4c], "ld1 {v12.4s}, [x11]"),
        ([0x00, 0x7e, 0x40, 0x4c], "ld1 {v0.2d}, [x16]"),
        ([0x30, 0x88, 0x40, 0x4c], "ld2 {v16.4s, v17.4s}, [x1]"),
        ([0x32, 0x88, 0x40, 0x4c], "ld2 {v18.4s, v19.4s}, [x1]"),
        ([0x7e, 0x8c, 0x40, 0x4c], "ld2 {v30.2d, v31.2d}, [x3]"),
        ([0x68, 0x8d, 0x40, 0x4c], "ld2 {v8.2d, v9.2d}, [x11]"),
        ([0x6a, 0x8d, 0x40, 0x4c], "ld2 {v10.2d, v11.2d}, [x11]"),
        ([0x6c, 0x8d, 0x40, 0x4c], "ld2 {v12.2d, v13.2d}, [x11]"),
        ([0x6e, 0x8d, 0x40, 0x4c], "ld2 {v14.2d, v15.2d}, [x11]"),
        ([0x80, 0x8d, 0x40, 0x4c], "ld2 {v0.2d, v1.2d}, [x12]"),
        ([0x82, 0x8d, 0x40, 0x4c], "ld2 {v2.2d, v3.2d}, [x12]"),
        ([0xa0, 0x8d, 0x40, 0x4c], "ld2 {v0.2d, v1.2d}, [x13]"),
        ([0xa4, 0x8d, 0x40, 0x4c], "ld2 {v4.2d, v5.2d}, [x13]"),
        ([0xa6, 0x8d, 0x40, 0x4c], "ld2 {v6.2d, v7.2d}, [x13]"),
        ([0xa3, 0x7c, 0x86, 0x4c], "st1 {v3.2d}, [x5], x6"),
        ([0x61, 0x2c, 0x9f, 0x4c], "st1 {v1.2d, v2.2d, v3.2d, v4.2d}, [x3], #0x40"),
        ([0xb0, 0x2c, 0x9f, 0x4c], "st1 {v16.2d, v17.2d, v18.2d, v19.2d}, [x5], #0x40"),
        ([0x24, 0x78, 0x9f, 0x4c], "st1 {v4.4s}, [x1], #0x10"),
        ([0xa5, 0x7d, 0x9f, 0x4c], "st1 {v5.2d}, [x13], #0x10"),
        ([0xa4, 0x88, 0x9f, 0x4c], "st2 {v4.4s, v5.4s}, [x5], #0x20"),
        ([0xc4, 0x88, 0x9f, 0x4c], "st2 {v4.4s, v5.4s}, [x6], #0x20"),
        ([0xb0, 0xad, 0x9f, 0x4c], "st1 {v16.2d, v17.2d}, [x13], #0x20"),
        ([0x20, 0x7c, 0xc2, 0x4c], "ld1 {v0.2d}, [x1], x2"),
        ([0x46, 0x7d, 0xc6, 0x4c], "ld1 {v6.2d}, [x10], x6"),
        ([0x20, 0x0c, 0xdf, 0x4c], "ld4 {v0.2d, v1.2d, v2.2d, v3.2d}, [x1], #0x40"),
        ([0x51, 0x2d, 0xdf, 0x4c], "ld1 {v17.2d, v18.2d, v19.2d, v20.2d}, [x10], #0x40"),
        ([0x20, 0x78, 0xdf, 0x4c], "ld1 {v0.4s}, [x1], #0x10"),
        ([0x21, 0x78, 0xdf, 0x4c], "ld1 {v1.4s}, [x1], #0x10"),
        ([0x46, 0x7d, 0xdf, 0x4c], "ld1 {v6.2d}, [x10], #0x10"),
        ([0x20, 0x88, 0xdf, 0x4c], "ld2 {v0.4s, v1.4s}, [x1], #0x20"),
        ([0x50, 0xad, 0xdf, 0x4c], "ld1 {v16.2d, v17.2d}, [x10], #0x20"),
        ([0xa8, 0x85, 0x00, 0x4d], "st1 {v8.d}[1], [x13]"),
        ([0xac, 0x85, 0x00, 0x4d], "st1 {v12.d}[1], [x13]"),
        ([0xec, 0x85, 0x00, 0x4d], "st1 {v12.d}[1], [x15]"),
        ([0x62, 0x84, 0x40, 0x4d], "ld1 {v2.d}[1], [x3]"),
        ([0xa8, 0x85, 0x40, 0x4d], "ld1 {v8.d}[1], [x13]"),
        ([0xec, 0x85, 0x40, 0x4d], "ld1 {v12.d}[1], [x15]"),
        ([0x64, 0x84, 0x84, 0x4d], "st1 {v4.d}[1], [x3], x4"),
        ([0x64, 0x84, 0x9f, 0x4d], "st1 {v4.d}[1], [x3], #0x8"),
        ([0x24, 0xcd, 0xdf, 0x4d], "ld1r {v4.2d}, [x9], #0x8"),
        ([0x60, 0x04, 0x81, 0x3c], "str q0, [x3], #0x10"),
        ([0x61, 0x00, 0x9f, 0x3c], "stur q1, [x3, #-0x10]"),
        ([0xa0, 0x00, 0x9f, 0x3c], "stur q0, [x5, #-0x10]"),
        ([0x30, 0x00, 0xc0, 0x3c], "ldur q16, [x1]"),
        ([0x68, 0x01, 0xc0, 0x3c], "ldur q8, [x11]"),
        ([0x6c, 0x01, 0xc0, 0x3c], "ldur q12, [x11]"),
        ([0x04, 0x02, 0xc0, 0x3c], "ldur q4, [x16]"),
        ([0x32, 0x00, 0xc1, 0x3c], "ldur q18, [x1, #0x10]"),
        ([0x6c, 0x01, 0xc1, 0x3c], "ldur q12, [x11, #0x10]"),
        ([0x05, 0x02, 0xc1, 0x3c], "ldur q5, [x16, #0x10]"),
        ([0x20, 0x04, 0xc1, 0x3c], "ldr q0, [x1], #0x10"),
        ([0x04, 0x06, 0xc1, 0x3c], "ldr q4, [x16], #0x10"),
        ([0x05, 0x06, 0xc1, 0x3c], "ldr q5, [x16], #0x10"),
        ([0x34, 0x00, 0xc2, 0x3c], "ldur q20, [x1, #0x20]"),
        ([0x3c, 0x00, 0xc6, 0x3c], "ldur q28, [x1, #0x60]"),
        ([0x06, 0x02, 0xc6, 0x3c], "ldur q6, [x16, #0x60]"),
        ([0x3e, 0x00, 0xc7, 0x3c], "ldur q30, [x1, #0x70]"),
        ([0x07, 0x02, 0xc7, 0x3c], "ldur q7, [x16, #0x70]"),
        ([0x44, 0x00, 0xdf, 0x3c], "ldur q4, [x2, #-0x10]"),
        ([0x60, 0x00, 0x80, 0x3d], "str q0, [x3]"),
        ([0x61, 0x00, 0x80, 0x3d], "str q1, [x3]"),
        ([0xa0, 0x00, 0x80, 0x3d], "str q0, [x5]"),
        ([0x60, 0x01, 0x80, 0x3d], "str q0, [x11]"),
        ([0x80, 0x01, 0x80, 0x3d], "str q0, [x12]"),
        ([0xa1, 0x01, 0x80, 0x3d], "str q1, [x13]"),
        ([0xc2, 0x01, 0x80, 0x3d], "str q2, [x14]"),
        ([0xe0, 0x01, 0x80, 0x3d], "str q0, [x15]"),
        ([0xe3, 0x01, 0x80, 0x3d], "str q3, [x15]"),
        ([0x00, 0x02, 0x80, 0x3d], "str q0, [x16]"),
        ([0xe0, 0x27, 0x80, 0x3d], "str q0, [sp, #0x90]"),
        ([0xe1, 0x2b, 0x80, 0x3d], "str q1, [sp, #0xa0]"),
        ([0xe2, 0x2f, 0x80, 0x3d], "str q2, [sp, #0xb0]"),
        ([0xe3, 0x33, 0x80, 0x3d], "str q3, [sp, #0xc0]"),
        ([0xe4, 0x37, 0x80, 0x3d], "str q4, [sp, #0xd0]"),
        ([0xe5, 0x3b, 0x80, 0x3d], "str q5, [sp, #0xe0]"),
        ([0xe6, 0x3f, 0x80, 0x3d], "str q6, [sp, #0xf0]"),
        ([0xe7, 0x43, 0x80, 0x3d], "str q7, [sp, #0x100]"),
        ([0x20, 0x00, 0xc0, 0x3d], "ldr q0, [x1]"),
        ([0x21, 0x00, 0xc0, 0x3d], "ldr q1, [x1]"),
        ([0x24, 0x00, 0xc0, 0x3d], "ldr q4, [x1]"),
        ([0x30, 0x00, 0xc0, 0x3d], "ldr q16, [x1]"),
        ([0x60, 0x00, 0xc0, 0x3d], "ldr q0, [x3]"),
        ([0x78, 0x00, 0xc0, 0x3d], "ldr q24, [x3]"),
        ([0xa1, 0x00, 0xc0, 0x3d], "ldr q1, [x5]"),
        ([0xc0, 0x00, 0xc0, 0x3d], "ldr q0, [x6]"),
        ([0xe1, 0x00, 0xc0, 0x3d], "ldr q1, [x7]"),
        ([0xe4, 0x00, 0xc0, 0x3d], "ldr q4, [x7]"),
        ([0x02, 0x01, 0xc0, 0x3d], "ldr q2, [x8]"),
        ([0x23, 0x01, 0xc0, 0x3d], "ldr q3, [x9]"),
        ([0x44, 0x01, 0xc0, 0x3d], "ldr q4, [x10]"),
        ([0xc1, 0x05, 0xc0, 0x3d], "ldr q1, [x14, #0x10]"),
        ([0xe5, 0x05, 0xc0, 0x3d], "ldr q5, [x15, #0x10]"),
        ([0xc2, 0x09, 0xc0, 0x3d], "ldr q2, [x14, #0x20]"),
        ([0xe6, 0x09, 0xc0, 0x3d], "ldr q6, [x15, #0x20]"),
        ([0x33, 0x0c, 0xc0, 0x3d], "ldr q19, [x1, #0x30]"),
        ([0x85, 0x0c, 0xc0, 0x3d], "ldr q5, [x4, #0x30]"),
        ([0x83, 0x0d, 0xc0, 0x3d], "ldr q3, [x12, #0x30]"),
        ([0xa7, 0x0d, 0xc0, 0x3d], "ldr q7, [x13, #0x30]"),
        ([0xc3, 0x0d, 0xc0, 0x3d], "ldr q3, [x14, #0x30]"),
        ([0xe7, 0x0d, 0xc0, 0x3d], "ldr q7, [x15, #0x30]"),
        ([0x00, 0x06, 0xc1, 0xac], "ldp q0, q1, [x16], #0x20"),
        ([0xbe, 0x7d, 0xc1, 0xac], "ldp q30, q31, [x13], #0x20"),
        ([0xb0, 0x44, 0x00, 0xad], "stp q16, q17, [x5]"),
        ([0x82, 0x0d, 0x01, 0xad], "stp q2, q3, [x12, #0x20]"),
        ([0xc2, 0x0d, 0x01, 0xad], "stp q2, q3, [x14, #0x20]"),
        ([0xb8, 0x64, 0x02, 0xad], "stp q24, q25, [x5, #0x40]"),
        ([0xba, 0x6c, 0x03, 0xad], "stp q26, q27, [x5, #0x60]"),
        ([0xc0, 0x04, 0x40, 0xad], "ldp q0, q1, [x6]"),
        ([0xb0, 0x44, 0x40, 0xad], "ldp q16, q17, [x5]"),
        ([0x02, 0x0e, 0x41, 0xad], "ldp q2, q3, [x16, #0x20]"),
        ([0x68, 0x25, 0x41, 0xad], "ldp q8, q9, [x11, #0x20]"),
        ([0x6c, 0x35, 0x41, 0xad], "ldp q12, q13, [x11, #0x20]"),
        ([0x7c, 0x74, 0x42, 0xad], "ldp q28, q29, [x3, #0x40]"),
        ([0x02, 0x0e, 0x43, 0xad], "ldp q2, q3, [x16, #0x60]"),
        ([0x06, 0x1e, 0x43, 0xad], "ldp q6, q7, [x16, #0x60]"),
        ([0x30, 0x44, 0x43, 0xad], "ldp q16, q17, [x1, #0x60]"),
        ([0x3e, 0x7c, 0x47, 0xad], "ldp q30, q31, [x1, #0xe0]"),
        ([0x6e, 0x3d, 0xc1, 0x6c], "ldp d14, d15, [x11], #0x10"),
        ([0xe8, 0x27, 0xc4, 0x6c], "ldp d8, d9, [sp], #0x40"),
        ([0x81, 0x01, 0x00, 0x6d], "stp d1, d0, [x12]"),
        ([0x39, 0x60, 0x00, 0x6d], "stp d25, d24, [x1]"),
        ([0xdd, 0x71, 0x01, 0x6d], "stp d29, d28, [x14, #0x10]"),
        ([0xfd, 0x70, 0x31, 0x6d], "stp d29, d28, [x7, #-0xf0]"),
        ([0x86, 0xd8, 0x3f, 0x6d], "stp d6, d22, [x4, #-0x8]"),
        ([0x24, 0x00, 0x40, 0x6d], "ldp d4, d0, [x1]"),
        ([0xe1, 0x02, 0x40, 0x6d], "ldp d1, d0, [x23]"),
        ([0xb2, 0x45, 0x7f, 0x6d], "ldp d18, d17, [x13, #-0x10]"),
        ([0x32, 0xc4, 0x7f, 0x6d], "ldp d18, d17, [x1, #-0x8]"),
        ([0x33, 0xd0, 0x7f, 0x6d], "ldp d19, d20, [x1, #-0x8]"),
        ([0x00, 0x84, 0x00, 0xfc], "str d0, [x0], #0x8"),
        ([0xc2, 0x87, 0x00, 0xfc], "str d2, [x30], #0x8"),
        ([0x40, 0x8f, 0x00, 0xfc], "str d0, [x26, #0x8]!"),
        ([0x15, 0x04, 0x01, 0xfc], "str d21, [x0], #0x10"),
        ([0xc2, 0x07, 0x01, 0xfc], "str d2, [x30], #0x10"),
        ([0x20, 0x84, 0x02, 0xfc], "str d0, [x1], #0x28"),
        ([0x66, 0x00, 0x15, 0xfc], "stur d6, [x3, #-0xb0]"),
        ([0x29, 0x03, 0x1f, 0xfc], "stur d9, [x25, #-0x10]"),
        ([0xe8, 0x0f, 0x1f, 0xfc], "str d8, [sp, #-0x10]!"),
        ([0x6d, 0x83, 0x1f, 0xfc], "stur d13, [x27, #-0x8]"),
        ([0xc3, 0x83, 0x1f, 0xfc], "stur d3, [x30, #-0x8]"),
        ([0x0d, 0x84, 0x1f, 0xfc], "str d13, [x0], #-0x8"),
        ([0x61, 0x8e, 0x1f, 0xfc], "str d1, [x19, #-0x8]!"),
        ([0xa3, 0x68, 0x20, 0xfc], "str d3, [x5, x0]"),
        ([0xd2, 0x68, 0x20, 0xfc], "str d18, [x6, x0]"),
        ([0x94, 0x6b, 0x20, 0xfc], "str d20, [x28, x0]"),
        ([0x48, 0x78, 0x20, 0xfc], "str d8, [x2, x0, lsl #3]"),
        ([0x83, 0x7b, 0x77, 0xfc], "ldr d3, [x28, x23, lsl #3]"),
        ([0x80, 0xda, 0x77, 0xfc], "ldr d0, [x20, w23, sxtw #3]"),
        ([0x22, 0x6b, 0x78, 0xfc], "ldr d2, [x25, x24]"),
        ([0x08, 0x78, 0x78, 0xfc], "ldr d8, [x0, x24, lsl #3]"),
        ([0xf3, 0x9b, 0x41, 0xfd], "ldr d19, [sp, #0x330]"),
        ([0x4a, 0x44, 0x47, 0xfd], "ldr d10, [x2, #0xe88]"),
        ([0x01, 0xd0, 0x47, 0xfd], "ldr d1, [x0, #0xfa0]"),
        ([0xe8, 0x27, 0xbc, 0x6d], "stp d8, d9, [sp, #-0x40]!"),
        ([0xe8, 0x27, 0xbf, 0x6d], "stp d8, d9, [sp, #-0x10]!"),
        ([0xa2, 0x0f, 0x32, 0x3d], "str b2, [x29, #0xc83]"),
        ([0x97, 0xff, 0xff, 0x1c], "ldr s23, $-0x10"),
        ([0x6e, 0x3d, 0xc1, 0x2c], "ldp s14, s15, [x11], #0x8"),
        ([0xef, 0x7e, 0x40, 0x2d], "ldp s15, s31, [x23]"),
        ([0x41, 0x80, 0x40, 0x2d], "ldp s1, s0, [x2, #0x4]"),
        ([0x33, 0xd0, 0x7f, 0x2d], "ldp s19, s20, [x1, #-0x4]"),
        ([0x60, 0x47, 0x00, 0xbc], "str s0, [x27], #0x4"),
        ([0x81, 0x47, 0x00, 0xbc], "str s1, [x28], #0x4"),
        ([0xc2, 0x47, 0x00, 0xbc], "str s2, [x30], #0x4"),
        ([0x00, 0x4d, 0x00, 0xbc], "str s0, [x8, #0x4]!"),
        ([0x40, 0x4f, 0x00, 0xbc], "str s0, [x26, #0x4]!"),
        ([0x15, 0x84, 0x00, 0xbc], "str s21, [x0], #0x8"),
        ([0x80, 0x87, 0x00, 0xbc], "str s0, [x28], #0x8"),
        ([0xf1, 0x00, 0x1c, 0xbc], "stur s17, [x7, #-0x40]"),
        ([0x01, 0x04, 0x1f, 0xbc], "str s1, [x0], #-0x10"),
        ([0x8d, 0xc3, 0x1f, 0xbc], "stur s13, [x28, #-0x4]"),
        ([0x0d, 0xc4, 0x1f, 0xbc], "str s13, [x0], #-0x4"),
        ([0x61, 0xce, 0x1f, 0xbc], "str s1, [x19, #-0x4]!"),
        ([0x82, 0x6b, 0x20, 0xbc], "str s2, [x28, x0]"),
        ([0xad, 0x7a, 0x20, 0xbc], "str s13, [x21, x0, lsl #2]"),
        ([0x88, 0x7b, 0x21, 0xbc], "str s8, [x28, x1, lsl #2]"),
        ([0x60, 0xda, 0x21, 0xbc], "str s0, [x19, w1, sxtw #2]"),
        ([0xc0, 0x5a, 0x32, 0xbc], "str s0, [x22, w18, uxtw #2]"),
        ([0xb3, 0x85, 0x40, 0xbc], "ldr s19, [x13], #0x8"),
        ([0x02, 0x8f, 0x40, 0xbc], "ldr s2, [x24, #0x8]!"),
        ([0x71, 0x05, 0x41, 0xbc], "ldr s17, [x11], #0x10"),
        ([0xe0, 0x42, 0x5d, 0xbc], "ldur s0, [x23, #-0x2c]"),
        ([0x27, 0x80, 0x5d, 0xbc], "ldur s7, [x1, #-0x28]"),
        ([0x21, 0x84, 0x5f, 0xbc], "ldr s1, [x1], #-0x8"),
        ([0x22, 0x87, 0x5f, 0xbc], "ldr s2, [x25], #-0x8"),
        ([0xe0, 0x8e, 0x5f, 0xbc], "ldr s0, [x23, #-0x8]!"),
        ([0x79, 0x6a, 0x60, 0xbc], "ldr s25, [x19, x0]"),
        ([0x6a, 0x78, 0x60, 0xbc], "ldr s10, [x3, x0, lsl #2]"),
        ([0xeb, 0x7a, 0x60, 0xbc], "ldr s11, [x23, x0, lsl #2]"),
        ([0x96, 0x7b, 0x61, 0xbc], "ldr s22, [x28, x1, lsl #2]"),
        ([0x68, 0xdb, 0x61, 0xbc], "ldr s8, [x27, w1, sxtw #2]"),
        ([0x49, 0x03, 0x00, 0xbd], "str s9, [x26]"),
        ([0x62, 0x78, 0x00, 0xbd], "str s2, [x3, #0x78]"),
        ([0x6c, 0x78, 0x00, 0xbd], "str s12, [x3, #0x78]"),
        ([0xea, 0x7f, 0x00, 0xbd], "str s10, [sp, #0x7c]"),
        ([0xe9, 0x87, 0x04, 0xbd], "str s9, [sp, #0x484]"),
        ([0xc0, 0x8a, 0x0a, 0xbd], "str s0, [x22, #0xa88]"),
        ([0x41, 0x58, 0x4f, 0xbd], "ldr s1, [x2, #0xf58]"),
        ([0x88, 0x7c, 0x00, 0x2d], "stp s8, s31, [x4]"),
        ([0x03, 0x84, 0x00, 0x2d], "stp s3, s1, [x0, #0x4]"),
        ([0x13, 0xdc, 0x3f, 0x2d], "stp s19, s23, [x0, #-0x4]"),
    ];

    let errs = run_tests(TESTS);

    for err in errs.iter() {
        println!("{}", err);
    }

    assert!(errs.is_empty());
}

#[test]
fn test_openblas_loadstore() {
    const TESTS: &[([u8; 4], &'static str)] = &[
        ([0x83, 0x68, 0x60, 0x38], "ldrb w3, [x4, x0]"),
        ([0x63, 0x03, 0x40, 0x29], "ldp w3, w0, [x27]"),
        ([0x49, 0x00, 0x40, 0x39], "ldrb w9, [x2]"),
        ([0x4a, 0x00, 0x40, 0x39], "ldrb w10, [x2]"),
        ([0x82, 0x08, 0x40, 0x39], "ldrb w2, [x4, #0x2]"),
        ([0x82, 0x0a, 0x40, 0x39], "ldrb w2, [x20, #0x2]"),
        ([0x41, 0x0b, 0x40, 0x39], "ldrb w1, [x26, #0x2]"),
        ([0x88, 0x10, 0x40, 0x39], "ldrb w8, [x4, #0x4]"),
        ([0xc1, 0x16, 0x40, 0x39], "ldrb w1, [x22, #0x5]"),
        ([0xc2, 0x16, 0x40, 0x39], "ldrb w2, [x22, #0x5]"),
        ([0x00, 0x40, 0x40, 0x39], "ldrb w0, [x0, #0x10]"),
        ([0xe1, 0x43, 0x41, 0x39], "ldrb w1, [sp, #0x50]"),
        ([0xe0, 0x83, 0x41, 0x39], "ldrb w0, [sp, #0x60]"),
        ([0xe1, 0x8b, 0x41, 0x39], "ldrb w1, [sp, #0x62]"),
        ([0xe2, 0x8b, 0x41, 0x39], "ldrb w2, [sp, #0x62]"),
        ([0xe2, 0xa3, 0x41, 0x39], "ldrb w2, [sp, #0x68]"),
        ([0xe1, 0xb7, 0x41, 0x39], "ldrb w1, [sp, #0x6d]"),
        ([0xe2, 0xb7, 0x41, 0x39], "ldrb w2, [sp, #0x6d]"),
        ([0xf3, 0xcb, 0x41, 0x39], "ldrb w19, [sp, #0x72]"),
        ([0xe2, 0xeb, 0x41, 0x39], "ldrb w2, [sp, #0x7a]"),
        ([0xe1, 0x13, 0x42, 0x39], "ldrb w1, [sp, #0x84]"),
        ([0xe3, 0x13, 0x42, 0x39], "ldrb w3, [sp, #0x84]"),
        ([0xe7, 0x23, 0x42, 0x39], "ldrb w7, [sp, #0x88]"),
        ([0xe3, 0x37, 0x42, 0x39], "ldrb w3, [sp, #0x8d]"),
        ([0xe1, 0x4f, 0x42, 0x39], "ldrb w1, [sp, #0x93]"),
        ([0x60, 0x02, 0x72, 0x39], "ldrb w0, [x19, #0xc80]"),
        ([0x22, 0x04, 0x40, 0x69], "ldpsw x2, x1, [x1]"),
        ([0x38, 0x15, 0x40, 0x69], "ldpsw x24, x5, [x9]"),
        ([0x38, 0x1d, 0x40, 0x69], "ldpsw x24, x7, [x9]"),
        ([0x43, 0x80, 0x40, 0x69], "ldpsw x3, x0, [x2, #0x4]"),
        ([0x05, 0x90, 0x40, 0x69], "ldpsw x5, x4, [x0, #0x4]"),
        ([0x39, 0x90, 0x40, 0x69], "ldpsw x25, x4, [x1, #0x4]"),
        ([0x05, 0x98, 0x40, 0x69], "ldpsw x5, x6, [x0, #0x4]"),
        ([0xe1, 0x13, 0x5f, 0x69], "ldpsw x1, x4, [sp, #0xf8]"),
        ([0xe3, 0x33, 0x45, 0x78], "ldurh w3, [sp, #0x53]"),
        ([0xe1, 0x13, 0x46, 0x78], "ldurh w1, [sp, #0x61]"),
        ([0xe2, 0x93, 0x46, 0x78], "ldurh w2, [sp, #0x69]"),
        ([0xe2, 0xb3, 0x46, 0x78], "ldurh w2, [sp, #0x6b]"),
        ([0xe5, 0xb3, 0x48, 0x78], "ldurh w5, [sp, #0x8b]"),
        ([0xe5, 0x8b, 0x00, 0x79], "strh w5, [sp, #0x44]"),
        ([0xe2, 0xa3, 0x00, 0x79], "strh w2, [sp, #0x50]"),
        ([0xe0, 0xab, 0x00, 0x79], "strh w0, [sp, #0x54]"),
        ([0xe1, 0xb3, 0x00, 0x79], "strh w1, [sp, #0x58]"),
        ([0xe2, 0xc3, 0x00, 0x79], "strh w2, [sp, #0x60]"),
        ([0xe5, 0xf3, 0x00, 0x79], "strh w5, [sp, #0x78]"),
        ([0x01, 0x00, 0x40, 0x79], "ldrh w1, [x0]"),
        ([0x02, 0x00, 0x40, 0x79], "ldrh w2, [x0]"),
        ([0x22, 0x00, 0x40, 0x79], "ldrh w2, [x1]"),
        ([0x83, 0x02, 0x40, 0x79], "ldrh w3, [x20]"),
        ([0x42, 0x03, 0x40, 0x79], "ldrh w2, [x26]"),
        ([0x20, 0x08, 0x40, 0x79], "ldrh w0, [x1, #0x4]"),
        ([0xe0, 0xa3, 0x40, 0x79], "ldrh w0, [sp, #0x50]"),
        ([0xe1, 0xa3, 0x40, 0x79], "ldrh w1, [sp, #0x50]"),
        ([0xe0, 0xb3, 0x40, 0x79], "ldrh w0, [sp, #0x58]"),
        ([0xe0, 0xc3, 0x40, 0x79], "ldrh w0, [sp, #0x60]"),
        ([0xe2, 0xc3, 0x40, 0x79], "ldrh w2, [sp, #0x60]"),
        ([0xe3, 0xf3, 0x40, 0x79], "ldrh w3, [sp, #0x78]"),
        ([0xfd, 0x7b, 0xc1, 0xa8], "ldp x29, x30, [sp], #0x10"),
        ([0xfd, 0x7b, 0xd9, 0xa8], "ldp x29, x30, [sp], #0x190"),
        ([0xfd, 0x7b, 0xda, 0xa8], "ldp x29, x30, [sp], #0x1a0"),
        ([0x3c, 0x61, 0x00, 0xa9], "stp x28, x24, [x9]"),
        ([0xba, 0x61, 0x00, 0xa9], "stp x26, x24, [x13]"),
        ([0xed, 0x6f, 0x00, 0xa9], "stp x13, x27, [sp]"),
        ([0xea, 0x9f, 0x1c, 0xa9], "stp x10, x7, [sp, #0x1c8]"),
        ([0xff, 0x7c, 0x3e, 0xa9], "stp xzr, xzr, [x7, #-0x20]"),
        ([0x03, 0x00, 0x40, 0xa9], "ldp x3, x0, [x0]"),
        ([0x2a, 0x00, 0x40, 0xa9], "ldp x10, x0, [x1]"),
        ([0x5a, 0x28, 0x40, 0xa9], "ldp x26, x10, [x2]"),
        ([0x16, 0x87, 0x7f, 0xa9], "ldp x22, x1, [x24, #-0x8]"),
        ([0xc1, 0x46, 0x00, 0xb8], "str w1, [x22], #0x4"),
        ([0x3f, 0x47, 0x00, 0xb8], "str wzr, [x25], #0x4"),
        ([0x7f, 0x4c, 0x00, 0xb8], "str wzr, [x3, #0x4]!"),
        ([0x1f, 0x4d, 0x00, 0xb8], "str wzr, [x8, #0x4]!"),
        ([0x3f, 0x84, 0x00, 0xb8], "str wzr, [x1], #0x8"),
        ([0x7f, 0x0e, 0x04, 0xb8], "str wzr, [x19, #0x40]!"),
        ([0x9f, 0x0e, 0x04, 0xb8], "str wzr, [x20, #0x40]!"),
        ([0xe9, 0x93, 0x08, 0xb8], "stur w9, [sp, #0x89]"),
        ([0x1f, 0x40, 0x10, 0xb8], "stur wzr, [x0, #-0xfc]"),
        ([0x1f, 0xc0, 0x11, 0xb8], "stur wzr, [x0, #-0xe4]"),
        ([0xff, 0x82, 0x14, 0xb8], "stur wzr, [x23, #-0xb8]"),
        ([0x0b, 0x83, 0x14, 0xb8], "stur w11, [x24, #-0xb8]"),
        ([0xff, 0x69, 0x20, 0xb8], "str wzr, [x15, x0]"),
        ([0x57, 0x78, 0x20, 0xb8], "str w23, [x2, x0, lsl #2]"),
        ([0x7f, 0x68, 0x21, 0xb8], "str wzr, [x3, x1]"),
        ([0xa8, 0x68, 0x21, 0xb8], "str w8, [x5, x1]"),
        ([0xe0, 0xda, 0x34, 0xb8], "str w0, [x23, w20, sxtw #2]"),
        ([0x48, 0x7b, 0x35, 0xb8], "str w8, [x26, x21, lsl #2]"),
        ([0x23, 0x44, 0x40, 0xb8], "ldr w3, [x1], #0x4"),
        ([0xe1, 0x13, 0x45, 0xb8], "ldur w1, [sp, #0x51]"),
        ([0xe2, 0xf3, 0x48, 0xb8], "ldur w2, [sp, #0x8f]"),
        ([0x27, 0x80, 0x5f, 0xb8], "ldur w7, [x1, #-0x8]"),
        ([0xe3, 0x68, 0x61, 0xb8], "ldr w3, [x7, x1]"),
        ([0x0e, 0x79, 0x61, 0xb8], "ldr w14, [x8, x1, lsl #2]"),
        ([0xf2, 0x6a, 0x62, 0xb8], "ldr w18, [x23, x2]"),
        ([0x63, 0x7b, 0x7c, 0xb8], "ldr w3, [x27, x28, lsl #2]"),
        ([0xe5, 0x44, 0x80, 0xb8], "ldrsw x5, [x7], #0x4"),
        ([0xa3, 0xc0, 0x9f, 0xb8], "ldursw x3, [x5, #-0x4]"),
        ([0x04, 0x78, 0xa2, 0xb8], "ldrsw x4, [x0, x2, lsl #2]"),
        ([0xe4, 0x68, 0xa3, 0xb8], "ldrsw x4, [x7, x3]"),
        ([0xe5, 0x6a, 0xb3, 0xb8], "ldrsw x5, [x23, x19]"),
        ([0x65, 0x02, 0x00, 0xb9], "str w5, [x19]"),
        ([0xff, 0x03, 0x00, 0xb9], "str wzr, [sp]"),
        ([0xe9, 0x23, 0x00, 0xb9], "str w9, [sp, #0x20]"),
        ([0x01, 0x3c, 0x0c, 0xb9], "str w1, [x0, #0xc3c]"),
        ([0x60, 0x4a, 0x0c, 0xb9], "str w0, [x19, #0xc48]"),
        ([0x7f, 0x4a, 0x0c, 0xb9], "str wzr, [x19, #0xc48]"),
        ([0x40, 0xf0, 0x0d, 0xb9], "str w0, [x2, #0xdf0]"),
        ([0x89, 0x00, 0x40, 0xb9], "ldr w9, [x4]"),
        ([0xa0, 0x6f, 0x40, 0xb9], "ldr w0, [x29, #0x6c]"),
        ([0xe4, 0x6f, 0x40, 0xb9], "ldr w4, [sp, #0x6c]"),
        ([0xea, 0xc3, 0x6e, 0xb9], "ldr w10, [sp, #0x2ec0]"),
        ([0xeb, 0xc3, 0x6e, 0xb9], "ldr w11, [sp, #0x2ec0]"),
        ([0xea, 0xcb, 0x6e, 0xb9], "ldr w10, [sp, #0x2ec8]"),
        ([0xf9, 0x03, 0x6f, 0xb9], "ldr w25, [sp, #0x2f00]"),
        ([0xf9, 0x0b, 0x6f, 0xb9], "ldr w25, [sp, #0x2f08]"),
        ([0xf7, 0x33, 0x6f, 0xb9], "ldr w23, [sp, #0x2f30]"),
        ([0xf7, 0x3b, 0x6f, 0xb9], "ldr w23, [sp, #0x2f38]"),
        ([0xec, 0x1b, 0x71, 0xb9], "ldr w12, [sp, #0x3118]"),
        ([0xeb, 0x23, 0x71, 0xb9], "ldr w11, [sp, #0x3120]"),
        ([0xe0, 0x73, 0x71, 0xb9], "ldr w0, [sp, #0x3170]"),
        ([0xe0, 0x7b, 0x71, 0xb9], "ldr w0, [sp, #0x3178]"),
        ([0xe0, 0x83, 0x71, 0xb9], "ldr w0, [sp, #0x3180]"),
        ([0xe0, 0x8b, 0x71, 0xb9], "ldr w0, [sp, #0x3188]"),
        ([0xe9, 0x1b, 0x72, 0xb9], "ldr w9, [sp, #0x3218]"),
        ([0xeb, 0x23, 0x72, 0xb9], "ldr w11, [sp, #0x3220]"),
        ([0xea, 0x2b, 0x72, 0xb9], "ldr w10, [sp, #0x3228]"),
        ([0xeb, 0x33, 0x72, 0xb9], "ldr w11, [sp, #0x3230]"),
        ([0x24, 0x03, 0x80, 0xb9], "ldrsw x4, [x25]"),
        ([0x34, 0x10, 0x80, 0xb9], "ldrsw x20, [x1, #0x10]"),
        ([0xc8, 0x7c, 0x89, 0xb9], "ldrsw x8, [x6, #0x97c]"),
        ([0x7f, 0xfe, 0x01, 0xc8], "stlxr w1, xzr, [x19]"),
        ([0xbf, 0xfe, 0x01, 0xc8], "stlxr w1, xzr, [x21]"),
        ([0x64, 0x7e, 0x03, 0xc8], "stxr w3, x4, [x19]"),
        ([0x60, 0xfe, 0x5f, 0xc8], "ldaxr x0, [x19]"),
        ([0x62, 0xfe, 0x5f, 0xc8], "ldaxr x2, [x19]"),
        ([0x3f, 0xfc, 0x9f, 0xc8], "stlr xzr, [x1]"),
        ([0x81, 0xfd, 0x9f, 0xc8], "stlr x1, [x12]"),
        ([0x77, 0xfe, 0xdf, 0xc8], "ldar x23, [x19]"),
        ([0xff, 0x42, 0x00, 0xf8], "stur xzr, [x23, #0x4]"),
        ([0x3f, 0x84, 0x00, 0xf8], "str xzr, [x1], #0x8"),
        ([0x3f, 0x87, 0x00, 0xf8], "str xzr, [x25], #0x8"),
        ([0x1f, 0x8c, 0x00, 0xf8], "str xzr, [x0, #0x8]!"),
        ([0x02, 0x8f, 0x00, 0xf8], "str x2, [x24, #0x8]!"),
        ([0x20, 0xc0, 0x00, 0xf8], "stur x0, [x1, #0xc]"),
        ([0x3f, 0x04, 0x01, 0xf8], "str xzr, [x1], #0x10"),
        ([0xe1, 0xc3, 0x07, 0xf8], "stur x1, [sp, #0x7c]"),
        ([0x7f, 0x80, 0x10, 0xf8], "stur xzr, [x3, #-0xf8]"),
        ([0xc3, 0x00, 0x1c, 0xf8], "stur x3, [x6, #-0x40]"),
        ([0x02, 0x87, 0x1f, 0xf8], "str x2, [x24], #-0x8"),
        ([0x1f, 0xc3, 0x1f, 0xf8], "stur xzr, [x24, #-0x4]"),
        ([0xbf, 0x68, 0x20, 0xf8], "str xzr, [x5, x0]"),
        ([0xff, 0x6a, 0x38, 0xf8], "str xzr, [x23, x24]"),
        ([0x7f, 0x7b, 0x3c, 0xf8], "str xzr, [x27, x28, lsl #3]"),
        ([0xc7, 0x40, 0x40, 0xf8], "ldur x7, [x6, #0x4]"),
        ([0x01, 0x87, 0x40, 0xf8], "ldr x1, [x24], #0x8"),
        ([0x98, 0x83, 0x5f, 0xf8], "ldur x24, [x28, #-0x8]"),
        ([0x80, 0x86, 0x5f, 0xf8], "ldr x0, [x20], #-0x8"),
        ([0xe3, 0x6a, 0x60, 0xf8], "ldr x3, [x23, x0]"),
        ([0xc5, 0x6a, 0x7b, 0xf8], "ldr x5, [x22, x27]"),
        ([0xa0, 0x7a, 0x7c, 0xf8], "ldr x0, [x21, x28, lsl #3]"),
        ([0x08, 0xdb, 0x7c, 0xf8], "ldr x8, [x24, w28, sxtw #3]"),
        ([0x5f, 0x00, 0x00, 0xf9], "str xzr, [x2]"),
        ([0x2a, 0x6d, 0x04, 0xf9], "str x10, [x9, #0x8d8]"),
        ([0xeb, 0x1f, 0x18, 0xf9], "str x11, [sp, #0x3038]"),
        ([0xff, 0x7f, 0x18, 0xf9], "str xzr, [sp, #0x30f8]"),
        ([0xe9, 0x2f, 0x25, 0xf9], "str x9, [sp, #0x4a58]"),
        ([0xe1, 0xd7, 0x26, 0xf9], "str x1, [sp, #0x4da8]"),
        ([0x42, 0x01, 0x40, 0xf9], "ldr x2, [x10]"),
        ([0xf7, 0xef, 0x66, 0xf9], "ldr x23, [sp, #0x4dd8]"),
        ([0x01, 0x00, 0x00, 0xb0], "adrp x1, $+0x1000"),
        ([0xea, 0x6e, 0x00, 0xb0], "adrp x10, $+0xddd000"),
        ([0x13, 0x70, 0x00, 0xb0], "adrp x19, $+0xe01000"),
        ([0x20, 0x00, 0x75, 0x10], "adr x0, $+0xea004"),
        ([0x01, 0x00, 0x00, 0x90], "adrp x1, $+0x0"),
        ([0xc7, 0x6e, 0x00, 0x90], "adrp x7, $+0xdd8000"),
        ([0x01, 0x00, 0x00, 0xd0], "adrp x1, $+0x2000"),
        ([0x13, 0x70, 0x00, 0xd0], "adrp x19, $+0xe02000"),
        ([0xfc, 0xff, 0xff, 0xf0], "adrp x28, $-0x1000"),
        ([0xfd, 0x7b, 0xa1, 0xa9], "stp x29, x30, [sp, #-0x1f0]!"),
        ([0xfd, 0x7b, 0xa2, 0xa9], "stp x29, x30, [sp, #-0x1e0]!"),
        ([0xfd, 0x7b, 0xbf, 0xa9], "stp x29, x30, [sp, #-0x10]!"),
        ([0x23, 0xf0, 0x1f, 0x38], "sturb w3, [x1, #-0x1]"),
        ([0x01, 0x00, 0x00, 0x39], "strb w1, [x0]"),
        ([0x20, 0x00, 0x00, 0x39], "strb w0, [x1]"),
        ([0x80, 0x00, 0x00, 0x39], "strb w0, [x4]"),
        ([0x60, 0x02, 0x00, 0x39], "strb w0, [x19]"),
        ([0x80, 0x02, 0x00, 0x39], "strb w0, [x20]"),
        ([0xa0, 0x02, 0x00, 0x39], "strb w0, [x21]"),
        ([0xe0, 0x02, 0x00, 0x39], "strb w0, [x23]"),
        ([0x20, 0x03, 0x00, 0x39], "strb w0, [x25]"),
        ([0x40, 0x03, 0x00, 0x39], "strb w0, [x26]"),
        ([0x68, 0x13, 0x00, 0x39], "strb w8, [x27, #0x4]"),
        ([0xe0, 0x03, 0x01, 0x39], "strb w0, [sp, #0x40]"),
        ([0xe1, 0x43, 0x01, 0x39], "strb w1, [sp, #0x50]"),
        ([0xe1, 0x8b, 0x01, 0x39], "strb w1, [sp, #0x62]"),
        ([0xe2, 0xa3, 0x01, 0x39], "strb w2, [sp, #0x68]"),
        ([0xe5, 0x03, 0x0c, 0x39], "strb w5, [sp, #0x300]"),
        ([0xe0, 0xe3, 0x0d, 0x39], "strb w0, [sp, #0x378]"),
        ([0xe1, 0xe3, 0x0d, 0x39], "strb w1, [sp, #0x378]"),
        ([0xe2, 0xe3, 0x0d, 0x39], "strb w2, [sp, #0x378]"),
        ([0xe0, 0x03, 0x0e, 0x39], "strb w0, [sp, #0x380]"),
        ([0xe0, 0x23, 0x0e, 0x39], "strb w0, [sp, #0x388]"),
        ([0xe1, 0x23, 0x0e, 0x39], "strb w1, [sp, #0x388]"),
        ([0xe2, 0x23, 0x0e, 0x39], "strb w2, [sp, #0x388]"),
        ([0xe0, 0x43, 0x0e, 0x39], "strb w0, [sp, #0x390]"),
        ([0xe0, 0x23, 0x0f, 0x39], "strb w0, [sp, #0x3c8]"),
        ([0xe1, 0x23, 0x0f, 0x39], "strb w1, [sp, #0x3c8]"),
        ([0xe2, 0x23, 0x0f, 0x39], "strb w2, [sp, #0x3c8]"),
        ([0xe0, 0x43, 0x0f, 0x39], "strb w0, [sp, #0x3d0]"),
        ([0xe0, 0xa3, 0x0f, 0x39], "strb w0, [sp, #0x3e8]"),
        ([0xe1, 0xa3, 0x0f, 0x39], "strb w1, [sp, #0x3e8]"),
        ([0xe2, 0xa3, 0x0f, 0x39], "strb w2, [sp, #0x3e8]"),
        ([0xe0, 0xc3, 0x0f, 0x39], "strb w0, [sp, #0x3f0]"),
        ([0x60, 0x02, 0x32, 0x39], "strb w0, [x19, #0xc80]"),
        ([0x13, 0xb8, 0x00, 0x29], "stp w19, w14, [x0, #0x4]"),
        ([0xeb, 0x9f, 0x1f, 0x29], "stp w11, w7, [sp, #0xfc]"),
        ([0x7f, 0x7c, 0x3f, 0x29], "stp wzr, wzr, [x3, #-0x8]"),
    ];

    let errs = run_tests(TESTS);

    for err in errs.iter() {
        println!("{}", err);
    }

    assert!(errs.is_empty());
}

#[test]
fn test_openblas_data_ops() {
    const TESTS: &[([u8; 4], &'static str)] = &[
        ([0xe5, 0x03, 0x1e, 0x2a], "mov w5, w30"),
        ([0xe7, 0x03, 0x1e, 0x2a], "mov w7, w30"),
        ([0xe0, 0x03, 0x1f, 0x2a], "mov w0, wzr"),
        ([0x20, 0x0f, 0x80, 0x52], "mov w0, #0x79"),
        ([0x41, 0x0f, 0x80, 0x52], "mov w1, #0x7a"),
        ([0x01, 0x10, 0x80, 0x52], "mov w1, #0x80"),
        ([0x02, 0x10, 0x80, 0x52], "mov w2, #0x80"),
        ([0x08, 0x10, 0x80, 0x52], "mov w8, #0x80"),
        ([0x0a, 0x10, 0x80, 0x52], "mov w10, #0x80"),
        ([0x13, 0x10, 0x80, 0x52], "mov w19, #0x80"),
        ([0x15, 0x10, 0x80, 0x52], "mov w21, #0x80"),
        ([0x60, 0x10, 0x80, 0x52], "mov w0, #0x83"),
        ([0x81, 0x10, 0x80, 0x52], "mov w1, #0x84"),
        ([0x05, 0x16, 0x80, 0x52], "mov w5, #0xb0"),
        ([0x06, 0x16, 0x80, 0x52], "mov w6, #0xb0"),
        ([0x08, 0x16, 0x80, 0x52], "mov w8, #0xb0"),
        ([0x00, 0x18, 0x80, 0x52], "mov w0, #0xc0"),
        ([0x82, 0x18, 0x80, 0x52], "mov w2, #0xc4"),
        ([0x60, 0x1a, 0x80, 0x52], "mov w0, #0xd3"),
        ([0x80, 0x1a, 0x80, 0x52], "mov w0, #0xd4"),
        ([0xa0, 0x1a, 0x80, 0x52], "mov w0, #0xd5"),
        ([0xc0, 0x1a, 0x80, 0x52], "mov w0, #0xd6"),
        ([0x22, 0x1e, 0x80, 0x52], "mov w2, #0xf1"),
        ([0x00, 0x20, 0x80, 0x52], "mov w0, #0x100"),
        ([0x02, 0x20, 0x80, 0x52], "mov w2, #0x100"),
        ([0x20, 0x20, 0x80, 0x52], "mov w0, #0x101"),
        ([0x22, 0x20, 0x80, 0x52], "mov w2, #0x101"),
        ([0x80, 0x20, 0x80, 0x52], "mov w0, #0x104"),
        ([0xa0, 0x20, 0x80, 0x52], "mov w0, #0x105"),
        ([0xc2, 0x21, 0x80, 0x52], "mov w2, #0x10e"),
        ([0x42, 0x22, 0x80, 0x52], "mov w2, #0x112"),
        ([0xc6, 0x31, 0x80, 0x52], "mov w6, #0x18e"),
        ([0x25, 0x60, 0x80, 0x52], "mov w5, #0x301"),
        ([0x00, 0x82, 0x80, 0x52], "mov w0, #0x410"),
        ([0x20, 0x82, 0x80, 0x52], "mov w0, #0x411"),
        ([0x80, 0x82, 0x80, 0x52], "mov w0, #0x414"),
        ([0xa0, 0x82, 0x80, 0x52], "mov w0, #0x415"),
        ([0x60, 0x9e, 0x80, 0x52], "mov w0, #0x4f3"),
        ([0x02, 0x00, 0x81, 0x52], "mov w2, #0x800"),
        ([0x00, 0x02, 0x81, 0x52], "mov w0, #0x810"),
        ([0x20, 0x02, 0x81, 0x52], "mov w0, #0x811"),
        ([0x80, 0x02, 0x81, 0x52], "mov w0, #0x814"),
        ([0xa0, 0x02, 0x81, 0x52], "mov w0, #0x815"),
        ([0x00, 0x82, 0x81, 0x52], "mov w0, #0xc10"),
        ([0x20, 0x82, 0x81, 0x52], "mov w0, #0xc11"),
        ([0x80, 0x82, 0x81, 0x52], "mov w0, #0xc14"),
        ([0xa0, 0x82, 0x81, 0x52], "mov w0, #0xc15"),
        ([0x60, 0xfb, 0x81, 0x52], "mov w0, #0xfdb"),
        ([0x64, 0xfb, 0x81, 0x52], "mov w4, #0xfdb"),
        ([0x01, 0x00, 0x82, 0x52], "mov w1, #0x1000"),
        ([0xe2, 0x07, 0x82, 0x52], "mov w2, #0x103f"),
        ([0xe3, 0x07, 0x82, 0x52], "mov w3, #0x103f"),
        ([0xe4, 0x07, 0x82, 0x52], "mov w4, #0x103f"),
        ([0xea, 0x07, 0x82, 0x52], "mov w10, #0x103f"),
        ([0xeb, 0x07, 0x82, 0x52], "mov w11, #0x103f"),
        ([0x01, 0x08, 0x82, 0x52], "mov w1, #0x1040"),
        ([0x02, 0x08, 0x82, 0x52], "mov w2, #0x1040"),
        ([0x03, 0x08, 0x82, 0x52], "mov w3, #0x1040"),
        ([0x80, 0x46, 0x82, 0x52], "mov w0, #0x1234"),
        ([0xe0, 0x4d, 0x82, 0x52], "mov w0, #0x126f"),
        ([0xe2, 0xed, 0x82, 0x52], "mov w2, #0x176f"),
        ([0x04, 0x00, 0x84, 0x52], "mov w4, #0x2000"),
        ([0x05, 0x00, 0x84, 0x52], "mov w5, #0x2000"),
        ([0x25, 0x00, 0x84, 0x52], "mov w5, #0x2001"),
        ([0x85, 0x00, 0x84, 0x52], "mov w5, #0x2004"),
        ([0xa5, 0x00, 0x84, 0x52], "mov w5, #0x2005"),
        ([0xa0, 0x18, 0x84, 0x52], "mov w0, #0x20c5"),
        ([0xa1, 0x18, 0x84, 0x52], "mov w1, #0x20c5"),
        ([0xa2, 0x18, 0x84, 0x52], "mov w2, #0x20c5"),
        ([0x03, 0xe2, 0x84, 0x52], "mov w3, #0x2710"),
        ([0x05, 0xe2, 0x84, 0x52], "mov w5, #0x2710"),
        ([0x06, 0x00, 0x86, 0x52], "mov w6, #0x3000"),
        ([0x61, 0x66, 0x86, 0x52], "mov w1, #0x3333"),
        ([0x06, 0x00, 0x88, 0x52], "mov w6, #0x4000"),
        ([0x80, 0x29, 0x88, 0x52], "mov w0, #0x414c"),
        ([0x81, 0x29, 0x88, 0x52], "mov w1, #0x414c"),
        ([0xe0, 0x48, 0x88, 0x52], "mov w0, #0x4247"),
        ([0x00, 0x4a, 0x88, 0x52], "mov w0, #0x4250"),
        ([0xe1, 0xa8, 0x88, 0x52], "mov w1, #0x4547"),
        ([0xe2, 0xa8, 0x88, 0x52], "mov w2, #0x4547"),
        ([0x00, 0xa9, 0x88, 0x52], "mov w0, #0x4548"),
        ([0xe0, 0xe8, 0x88, 0x52], "mov w0, #0x4747"),
        ([0xc0, 0xf5, 0x88, 0x52], "mov w0, #0x47ae"),
        ([0x21, 0x8a, 0x89, 0x52], "mov w1, #0x4c51"),
        ([0xa0, 0xca, 0x89, 0x52], "mov w0, #0x4e55"),
        ([0x02, 0xea, 0x89, 0x52], "mov w2, #0x4f50"),
        ([0x81, 0x29, 0x8a, 0x52], "mov w1, #0x514c"),
        ([0x41, 0x2a, 0x8a, 0x52], "mov w1, #0x5152"),
        ([0x41, 0x48, 0x8a, 0x52], "mov w1, #0x5242"),
        ([0x01, 0x49, 0x8a, 0x52], "mov w1, #0x5248"),
        ([0xe0, 0x49, 0x8a, 0x52], "mov w0, #0x524f"),
        ([0x21, 0x4a, 0x8a, 0x52], "mov w1, #0x5251"),
        ([0x80, 0x4a, 0x8a, 0x52], "mov w0, #0x5254"),
        ([0x81, 0x4a, 0x8a, 0x52], "mov w1, #0x5254"),
        ([0x60, 0x8a, 0x8a, 0x52], "mov w0, #0x5453"),
        ([0xc0, 0xaa, 0x8a, 0x52], "mov w0, #0x5556"),
        ([0xc1, 0xaa, 0x8a, 0x52], "mov w1, #0x5556"),
        ([0xc4, 0xaa, 0x8a, 0x52], "mov w4, #0x5556"),
        ([0xc6, 0xaa, 0x8a, 0x52], "mov w6, #0x5556"),
        ([0xc8, 0xaa, 0x8a, 0x52], "mov w8, #0x5556"),
        ([0xca, 0xaa, 0x8a, 0x52], "mov w10, #0x5556"),
        ([0xd7, 0xaa, 0x8a, 0x52], "mov w23, #0x5556"),
        ([0xd8, 0xaa, 0x8a, 0x52], "mov w24, #0x5556"),
        ([0xd9, 0xaa, 0x8a, 0x52], "mov w25, #0x5556"),
        ([0x61, 0x2a, 0x8b, 0x52], "mov w1, #0x5953"),
        ([0x62, 0x2a, 0x8b, 0x52], "mov w2, #0x5953"),
        ([0xc0, 0xcc, 0x8c, 0x52], "mov w0, #0x6666"),
        ([0xc1, 0xcc, 0x8c, 0x52], "mov w1, #0x6666"),
        ([0xe0, 0xcc, 0x8c, 0x52], "mov w0, #0x6667"),
        ([0x00, 0x43, 0x8e, 0x52], "mov w0, #0x7218"),
        ([0x02, 0x43, 0x8e, 0x52], "mov w2, #0x7218"),
        ([0x40, 0xdf, 0x8f, 0x52], "mov w0, #0x7efa"),
        ([0x13, 0x00, 0x90, 0x52], "mov w19, #0x8000"),
        ([0x20, 0xc7, 0x91, 0x52], "mov w0, #0x8e39"),
        ([0x23, 0xc7, 0x91, 0x52], "mov w3, #0x8e39"),
        ([0x3b, 0xc7, 0x91, 0x52], "mov w27, #0x8e39"),
        ([0x42, 0x55, 0x95, 0x52], "mov w2, #0xaaaa"),
        ([0x64, 0x55, 0x95, 0x52], "mov w4, #0xaaab"),
        ([0xe3, 0xce, 0x97, 0x52], "mov w3, #0xbe77"),
        ([0xa2, 0x99, 0x99, 0x52], "mov w2, #0xcccd"),
        ([0x42, 0xe1, 0x9a, 0x52], "mov w2, #0xd70a"),
        ([0x62, 0x0f, 0x9e, 0x52], "mov w2, #0xf07b"),
        ([0x05, 0x70, 0xa6, 0x52], "mov w5, #0x33800000"),
        ([0x04, 0x30, 0xa7, 0x52], "mov w4, #0x39800000"),
        ([0x00, 0x50, 0xa8, 0x52], "mov w0, #0x42800000"),
        ([0x02, 0x59, 0xa8, 0x52], "mov w2, #0x42c80000"),
        ([0x40, 0x9f, 0xa8, 0x52], "mov w0, #0x44fa0000"),
        ([0x04, 0xb0, 0xa8, 0x52], "mov w4, #0x45800000"),
        ([0x05, 0x70, 0xa9, 0x52], "mov w5, #0x4b800000"),
        ([0x00, 0xf8, 0xaf, 0x52], "mov w0, #0x7fc00000"),
        ([0x41, 0x5f, 0xb8, 0x52], "mov w1, #0xc2fa0000"),
        ([0x06, 0x00, 0x80, 0x92], "mov x6, #0xffffffffffffffff"),
        ([0xfb, 0x01, 0x80, 0x92], "mov x27, #0xfffffffffffffff0"),
        ([0xe3, 0x02, 0x80, 0x92], "mov x3, #0xffffffffffffffe8"),
        ([0xf8, 0x02, 0x80, 0x92], "mov x24, #0xffffffffffffffe8"),
        ([0xfa, 0x03, 0x80, 0x92], "mov x26, #0xffffffffffffffe0"),
        ([0x6b, 0x21, 0x88, 0x92], "mov x11, #0xffffffffffffbef4"),
        ([0xe7, 0x41, 0x90, 0x92], "mov x7, #0xffffffffffff7df0"),
        ([0xe8, 0x41, 0x90, 0x92], "mov x8, #0xffffffffffff7df0"),
        ([0x00, 0x02, 0xe0, 0x92], "mov x0, #0xffefffffffffffff"),
        ([0x00, 0x02, 0xf0, 0x92], "mov x0, #0x7fefffffffffffff"),
        ([0xe0, 0x03, 0x00, 0xb2], "mov x0, #0x100000001"),
        ([0xe1, 0x03, 0x00, 0xb2], "mov x1, #0x100000001"),
        ([0xe2, 0x03, 0x00, 0xb2], "mov x2, #0x100000001"),
        ([0xe4, 0x03, 0x00, 0xb2], "mov x4, #0x100000001"),
        ([0xe1, 0xe7, 0x03, 0xb2], "mov x1, #0x6666666666666666"),
        ([0xe0, 0x1b, 0x09, 0xb2], "mov x0, #0x3f8000003f800000"),
        ([0xe0, 0x7f, 0x60, 0xb2], "mov x0, #0xffffffff00000000"),
        ([0x1b, 0x00, 0x80, 0xd2], "mov x27, #0x0"),
        ([0x8e, 0x01, 0x80, 0xd2], "mov x14, #0xc"),
        ([0x12, 0x10, 0x80, 0xd2], "mov x18, #0x80"),
        ([0x05, 0x14, 0x80, 0xd2], "mov x5, #0xa0"),
        ([0x60, 0x1d, 0x80, 0xd2], "mov x0, #0xeb"),
        ([0x02, 0x20, 0x80, 0xd2], "mov x2, #0x100"),
        ([0x15, 0x40, 0x80, 0xd2], "mov x21, #0x200"),
        ([0x14, 0xc0, 0x81, 0xd2], "mov x20, #0xe00"),
        ([0x00, 0x00, 0x82, 0xd2], "mov x0, #0x1000"),
        ([0x0c, 0x3a, 0x82, 0xd2], "mov x12, #0x11d0"),
        ([0x00, 0x4a, 0x82, 0xd2], "mov x0, #0x1250"),
        ([0x05, 0x4b, 0x82, 0xd2], "mov x5, #0x1258"),
        ([0x03, 0x4d, 0x82, 0xd2], "mov x3, #0x1268"),
        ([0x00, 0x4f, 0x82, 0xd2], "mov x0, #0x1278"),
        ([0x02, 0x4f, 0x82, 0xd2], "mov x2, #0x1278"),
        ([0x02, 0x51, 0x82, 0xd2], "mov x2, #0x1288"),
        ([0xe1, 0x7f, 0x82, 0xd2], "mov x1, #0x13ff"),
        ([0x07, 0x90, 0x82, 0xd2], "mov x7, #0x1480"),
        ([0x08, 0xdb, 0x82, 0xd2], "mov x8, #0x16d8"),
        ([0xe1, 0xff, 0x82, 0xd2], "mov x1, #0x17ff"),
        ([0x09, 0xcf, 0x83, 0xd2], "mov x9, #0x1e78"),
        ([0xe5, 0xff, 0x83, 0xd2], "mov x5, #0x1fff"),
        ([0x03, 0x00, 0x84, 0xd2], "mov x3, #0x2000"),
        ([0x0c, 0x4a, 0x84, 0xd2], "mov x12, #0x2250"),
        ([0xe2, 0x7f, 0x84, 0xd2], "mov x2, #0x23ff"),
        ([0x01, 0x80, 0x84, 0xd2], "mov x1, #0x2400"),
        ([0x05, 0xe2, 0x84, 0xd2], "mov x5, #0x2710"),
        ([0x00, 0xec, 0x84, 0xd2], "mov x0, #0x2760"),
        ([0x0c, 0xee, 0x84, 0xd2], "mov x12, #0x2770"),
        ([0x0c, 0x8a, 0x85, 0xd2], "mov x12, #0x2c50"),
        ([0x0c, 0xc6, 0x85, 0xd2], "mov x12, #0x2e30"),
        ([0x0c, 0xd6, 0x85, 0xd2], "mov x12, #0x2eb0"),
        ([0x0c, 0xd8, 0x85, 0xd2], "mov x12, #0x2ec0"),
        ([0x0c, 0xdc, 0x85, 0xd2], "mov x12, #0x2ee0"),
        ([0x0c, 0xde, 0x85, 0xd2], "mov x12, #0x2ef0"),
        ([0x0c, 0xe0, 0x85, 0xd2], "mov x12, #0x2f00"),
        ([0x0c, 0xe4, 0x85, 0xd2], "mov x12, #0x2f20"),
        ([0x0c, 0xe6, 0x85, 0xd2], "mov x12, #0x2f30"),
        ([0x0c, 0x08, 0x86, 0xd2], "mov x12, #0x3040"),
        ([0x0c, 0x20, 0x86, 0xd2], "mov x12, #0x3100"),
        ([0x02, 0x21, 0x86, 0xd2], "mov x2, #0x3108"),
        ([0x0c, 0x22, 0x86, 0xd2], "mov x12, #0x3110"),
        ([0x01, 0x29, 0x86, 0xd2], "mov x1, #0x3148"),
        ([0x0c, 0x2a, 0x86, 0xd2], "mov x12, #0x3150"),
        ([0x00, 0x2d, 0x86, 0xd2], "mov x0, #0x3168"),
        ([0x0c, 0x2e, 0x86, 0xd2], "mov x12, #0x3170"),
        ([0x0c, 0x30, 0x86, 0xd2], "mov x12, #0x3180"),
        ([0x0c, 0x42, 0x86, 0xd2], "mov x12, #0x3210"),
        ([0x0c, 0x44, 0x86, 0xd2], "mov x12, #0x3220"),
        ([0x06, 0x4d, 0x86, 0xd2], "mov x6, #0x3268"),
        ([0x0c, 0x4e, 0x86, 0xd2], "mov x12, #0x3270"),
        ([0xe1, 0xff, 0x87, 0xd2], "mov x1, #0x3fff"),
        ([0xe2, 0xff, 0x87, 0xd2], "mov x2, #0x3fff"),
        ([0x08, 0x02, 0x88, 0xd2], "mov x8, #0x4010"),
        ([0x04, 0x51, 0x88, 0xd2], "mov x4, #0x4288"),
        ([0x0c, 0x52, 0x88, 0xd2], "mov x12, #0x4290"),
        ([0x04, 0x53, 0x88, 0xd2], "mov x4, #0x4298"),
        ([0x0c, 0x54, 0x88, 0xd2], "mov x12, #0x42a0"),
        ([0x8c, 0x58, 0x88, 0xd2], "mov x12, #0x42c4"),
        ([0x02, 0x59, 0x88, 0xd2], "mov x2, #0x42c8"),
        ([0x06, 0x59, 0x88, 0xd2], "mov x6, #0x42c8"),
        ([0x08, 0x59, 0x88, 0xd2], "mov x8, #0x42c8"),
        ([0x09, 0x59, 0x88, 0xd2], "mov x9, #0x42c8"),
        ([0x0c, 0x6a, 0x88, 0xd2], "mov x12, #0x4350"),
        ([0x0c, 0x4a, 0x89, 0xd2], "mov x12, #0x4a50"),
        ([0x0c, 0x4c, 0x89, 0xd2], "mov x12, #0x4a60"),
        ([0x00, 0xb4, 0x89, 0xd2], "mov x0, #0x4da0"),
        ([0x0c, 0xb6, 0x89, 0xd2], "mov x12, #0x4db0"),
        ([0x00, 0x0a, 0x90, 0xd2], "mov x0, #0x8050"),
        ([0x01, 0x20, 0x90, 0xd2], "mov x1, #0x8100"),
        ([0x0d, 0x20, 0x90, 0xd2], "mov x13, #0x8100"),
        ([0x0a, 0x7a, 0x90, 0xd2], "mov x10, #0x83d0"),
        ([0x0c, 0x7a, 0x90, 0xd2], "mov x12, #0x83d0"),
        ([0x08, 0x7c, 0x90, 0xd2], "mov x8, #0x83e0"),
        ([0x07, 0x7d, 0x90, 0xd2], "mov x7, #0x83e8"),
        ([0x01, 0x46, 0x93, 0xd2], "mov x1, #0x9a30"),
        ([0x0c, 0x48, 0x93, 0xd2], "mov x12, #0x9a40"),
        ([0x01, 0x20, 0xa0, 0xd2], "mov x1, #0x1000000"),
        ([0x00, 0xf0, 0xa7, 0xd2], "mov x0, #0x3f800000"),
        ([0x1c, 0xf0, 0xa7, 0xd2], "mov x28, #0x3f800000"),
        ([0x00, 0xf0, 0xb7, 0xd2], "mov x0, #0xbf800000"),
        ([0x05, 0xf0, 0xb7, 0xd2], "mov x5, #0xbf800000"),
        ([0x00, 0x00, 0xc8, 0xd2], "mov x0, #0x400000000000"),
        ([0x00, 0x00, 0xd0, 0xd2], "mov x0, #0x800000000000"),
        ([0x00, 0x00, 0xdd, 0xd2], "mov x0, #0xe80000000000"),
        ([0x00, 0x02, 0xe0, 0xd2], "mov x0, #0x10000000000000"),
        ([0x01, 0x94, 0xe7, 0xd2], "mov x1, #0x3ca0000000000000"),
        ([0x01, 0x96, 0xe7, 0xd2], "mov x1, #0x3cb0000000000000"),
        ([0x02, 0xce, 0xe7, 0xd2], "mov x2, #0x3e70000000000000"),
        ([0x00, 0xe6, 0xe7, 0xd2], "mov x0, #0x3f30000000000000"),
        ([0x13, 0xf0, 0xe7, 0xd2], "mov x19, #0x3f80000000000000"),
        ([0x00, 0x0a, 0xe8, 0xd2], "mov x0, #0x4050000000000000"),
        ([0x20, 0x0b, 0xe8, 0xd2], "mov x0, #0x4059000000000000"),
        ([0x00, 0x12, 0xe8, 0xd2], "mov x0, #0x4090000000000000"),
        ([0x04, 0x16, 0xe8, 0xd2], "mov x4, #0x40b0000000000000"),
        ([0x40, 0x18, 0xe8, 0xd2], "mov x0, #0x40c2000000000000"),
        ([0x01, 0x1c, 0xe8, 0xd2], "mov x1, #0x40e0000000000000"),
        ([0x01, 0x22, 0xe8, 0xd2], "mov x1, #0x4110000000000000"),
        ([0x00, 0x2e, 0xe8, 0xd2], "mov x0, #0x4170000000000000"),
        ([0x00, 0xff, 0xef, 0xd2], "mov x0, #0x7ff8000000000000"),
        ([0x00, 0x00, 0xf0, 0xd2], "mov x0, #0x8000000000000000"),
        ([0x02, 0xfe, 0xff, 0xd2], "mov x2, #0xfff0000000000000"),
        ([0x1b, 0x00, 0x80, 0x12], "mov w27, #0xffffffff"),
        ([0x84, 0x00, 0x80, 0x12], "mov w4, #0xfffffffb"),
        ([0x00, 0x19, 0x80, 0x12], "mov w0, #0xffffff37"),
        ([0xe0, 0x07, 0x82, 0x12], "mov w0, #0xffffefc0"),
        ([0xe1, 0x07, 0x82, 0x12], "mov w1, #0xffffefc0"),
        ([0x00, 0x10, 0xb0, 0x12], "mov w0, #0x7f7fffff"),
        ([0xf9, 0x03, 0x0a, 0xaa], "mov x25, x10"),
        ([0x01, 0x20, 0xa0, 0xf2], "movk x1, #0x100, lsl #16"),
        ([0x02, 0x0e, 0xc0, 0xf2], "movk x2, #0x70, lsl #32"),
        ([0x03, 0x0e, 0xc0, 0xf2], "movk x3, #0x70, lsl #32"),
        ([0x02, 0x0f, 0xc0, 0xf2], "movk x2, #0x78, lsl #32"),
        ([0x04, 0x10, 0xc0, 0xf2], "movk x4, #0x80, lsl #32"),
        ([0x05, 0x10, 0xc0, 0xf2], "movk x5, #0x80, lsl #32"),
        ([0x04, 0x1c, 0xc0, 0xf2], "movk x4, #0xe0, lsl #32"),
        ([0x07, 0x1e, 0xc0, 0xf2], "movk x7, #0xf0, lsl #32"),
        ([0x06, 0x2c, 0xc0, 0xf2], "movk x6, #0x160, lsl #32"),
        ([0xc1, 0xfd, 0xe7, 0xf2], "movk x1, #0x3fee, lsl #48"),
        ([0x40, 0x09, 0xe8, 0xf2], "movk x0, #0x404a, lsl #48"),
        ([0xe0, 0x13, 0xe8, 0xf2], "movk x0, #0x409f, lsl #48"),
        ([0xe0, 0x11, 0xf8, 0xf2], "movk x0, #0xc08f, lsl #48"),
        ([0x42, 0x55, 0xa5, 0x72], "movk w2, #0x2aaa, lsl #16"),
        ([0x60, 0x1c, 0xa7, 0x72], "movk w0, #0x38e3, lsl #16"),
        ([0x63, 0x1c, 0xa7, 0x72], "movk w3, #0x38e3, lsl #16"),
        ([0x80, 0x99, 0xb9, 0x72], "movk w0, #0xcccc, lsl #16"),
        ([0xfa, 0x03, 0x3c, 0xaa], "mvn x26, x28"),
        ([0xfb, 0x03, 0x3c, 0xaa], "mvn x27, x28"),
        ([0xfe, 0x03, 0x3e, 0xaa], "mvn x30, x30"),
        ([0xe3, 0x03, 0x3b, 0x2a], "mvn w3, w27"),
    ];

    let errs = run_tests(TESTS);

    for err in errs.iter() {
        println!("{}", err);
    }

    assert!(errs.is_empty());
}

#[test]
fn test_openblas_cmp_ops() {
    const TESTS: &[([u8; 4], &'static str)] = &[
        ([0xff, 0x01, 0x01, 0xeb], "cmp x15, x1"),
        ([0x1f, 0x02, 0x01, 0xeb], "cmp x16, x1"),
        ([0x9f, 0x02, 0x1b, 0xeb], "cmp x20, x27"),
        ([0xbf, 0x06, 0x1b, 0xeb], "cmp x21, x27, lsl #1"),
        ([0x7f, 0xc3, 0x20, 0xeb], "cmp x27, w0, sxtw"),
        ([0xff, 0x63, 0x21, 0xeb], "cmp sp, x1"),
        ([0x1f, 0xc0, 0x21, 0xeb], "cmp x0, w1, sxtw"),
        ([0x5f, 0xc0, 0x21, 0xeb], "cmp x2, w1, sxtw"),
        ([0x5f, 0x0b, 0x00, 0xf1], "cmp x26, #0x2"),
        ([0x7f, 0x0d, 0x00, 0xf1], "cmp x11, #0x3"),
        ([0xdf, 0x0f, 0x00, 0xf1], "cmp x30, #0x3"),
        ([0x7f, 0x13, 0x00, 0xf1], "cmp x27, #0x4"),
        ([0xdf, 0x16, 0x00, 0xf1], "cmp x22, #0x5"),
        ([0x3f, 0x1b, 0x00, 0xf1], "cmp x25, #0x6"),
        ([0x7f, 0x1c, 0x00, 0xf1], "cmp x3, #0x7"),
        ([0x1f, 0x20, 0x00, 0xf1], "cmp x0, #0x8"),
        ([0x9f, 0x23, 0x00, 0xf1], "cmp x28, #0x8"),
        ([0x9f, 0x27, 0x00, 0xf1], "cmp x28, #0x9"),
        ([0x9f, 0x2b, 0x00, 0xf1], "cmp x28, #0xa"),
        ([0x1f, 0x2c, 0x00, 0xf1], "cmp x0, #0xb"),
        ([0x9f, 0x2f, 0x00, 0xf1], "cmp x28, #0xb"),
        ([0x1f, 0x30, 0x00, 0xf1], "cmp x0, #0xc"),
        ([0x9f, 0x33, 0x00, 0xf1], "cmp x28, #0xc"),
        ([0x1f, 0x34, 0x00, 0xf1], "cmp x0, #0xd"),
        ([0xbf, 0x36, 0x00, 0xf1], "cmp x21, #0xd"),
        ([0xdf, 0x36, 0x00, 0xf1], "cmp x22, #0xd"),
        ([0x1f, 0x3d, 0x00, 0xf1], "cmp x8, #0xf"),
        ([0xbf, 0x3e, 0x00, 0xf1], "cmp x21, #0xf"),
        ([0xdf, 0x3e, 0x00, 0xf1], "cmp x22, #0xf"),
        ([0x1f, 0x41, 0x00, 0xf1], "cmp x8, #0x10"),
        ([0x5f, 0x41, 0x00, 0xf1], "cmp x10, #0x10"),
        ([0xff, 0x80, 0x00, 0xf1], "cmp x7, #0x20"),
        ([0x3f, 0xfc, 0x00, 0xf1], "cmp x1, #0x3f"),
        ([0x1f, 0x01, 0x01, 0xf1], "cmp x8, #0x40"),
        ([0x3f, 0xfc, 0x07, 0xf1], "cmp x1, #0x1ff"),
        ([0x3f, 0xfc, 0x0f, 0xf1], "cmp x1, #0x3ff"),
        ([0x7f, 0xfc, 0x0f, 0xf1], "cmp x3, #0x3ff"),
        ([0x1f, 0x00, 0x10, 0xf1], "cmp x0, #0x400"),
        ([0x3f, 0x00, 0x10, 0xf1], "cmp x1, #0x400"),
        ([0x3f, 0x00, 0x24, 0xf1], "cmp x1, #0x900"),
        ([0x1f, 0xfc, 0x3f, 0xf1], "cmp x0, #0xfff"),
        ([0x3f, 0xfc, 0x3f, 0xf1], "cmp x1, #0xfff"),
        ([0x1f, 0x08, 0x40, 0xf1], "cmp x0, #0x2, lsl #12"),
        ([0x3f, 0x08, 0x40, 0xf1], "cmp x1, #0x2, lsl #12"),
        ([0x7f, 0x01, 0x00, 0xf1], "cmp x11, #0x0"),
        ([0x3f, 0x07, 0x00, 0xf1], "cmp x25, #0x1"),
        ([0x1f, 0x00, 0x00, 0x71], "cmp w0, #0x0"),
        ([0x3f, 0x04, 0x00, 0x6b], "cmp w1, w0, lsl #1"),
        ([0x5f, 0x05, 0x80, 0x6b], "cmp w10, w0, asr #1"),
        ([0x3f, 0x04, 0x00, 0x71], "cmp w1, #0x1"),
        ([0x5f, 0x24, 0x34, 0x71], "cmp w2, #0xd09"),
        ([0x7f, 0x0a, 0x40, 0x71], "cmp w19, #0x2, lsl #12"),
        ([0x1f, 0x80, 0x40, 0x71], "cmp w0, #0x20, lsl #12"),
        ([0x1f, 0x00, 0x44, 0x71], "cmp w0, #0x100, lsl #12"),
        ([0x9f, 0x02, 0x1a, 0x2b], "cmn w20, w26"),
        ([0x1f, 0x58, 0x00, 0x31], "cmn w0, #0x16"),
        ([0x1f, 0x04, 0x00, 0xb1], "cmn x0, #0x1"),
        ([0xdf, 0x38, 0x00, 0xb1], "cmn x6, #0xe"),
    ];

    let errs = run_tests(TESTS);

    for err in errs.iter() {
        println!("{}", err);
    }

    assert!(errs.is_empty());
}

#[test]
fn test_openblas_tst_ops() {
    const TESTS: &[([u8; 4], &'static str)] = &[
        ([0x9f, 0x02, 0x05, 0x6a], "tst w20, w5"),
        ([0x1f, 0x78, 0x1e, 0x72], "tst w0, #0xfffffffd"),
        ([0x3f, 0x78, 0x1e, 0x72], "tst w1, #0xfffffffd"),
        ([0x5f, 0x03, 0x13, 0xea], "tst x26, x19"),
        ([0x3f, 0x01, 0x40, 0xf2], "tst x9, #0x1"),
        ([0x5f, 0x01, 0x40, 0xf2], "tst x10, #0x1"),
        ([0x1f, 0x04, 0x40, 0xf2], "tst x0, #0x3"),
        ([0x3f, 0x09, 0x40, 0xf2], "tst x9, #0x7"),
        ([0x3f, 0x01, 0x7e, 0xf2], "tst x9, #0x4"),
        ([0x5f, 0x04, 0x7f, 0xf2], "tst x2, #0x6"),
    ];

    let errs = run_tests(TESTS);

    for err in errs.iter() {
        println!("{}", err);
    }

    assert!(errs.is_empty());
}

#[test]
fn test_openblas_conditional_ops() {
    const TESTS: &[([u8; 4], &'static str)] = &[
        ([0x82, 0xd2, 0x80, 0x1a], "csel w2, w20, w0, le"),
        ([0x94, 0xd2, 0x80, 0x1a], "csel w20, w20, w0, le"),
        ([0xe2, 0x13, 0x81, 0x1a], "csel w2, wzr, w1, ne"),
        ([0x73, 0xa2, 0x81, 0x1a], "csel w19, w19, w1, ge"),
        ([0xe1, 0xa3, 0x81, 0x1a], "csel w1, wzr, w1, ge"),
        ([0xe1, 0xa7, 0x81, 0x1a], "csinc w1, wzr, w1, ge"),
        ([0x21, 0xb3, 0x81, 0x1a], "csel w1, w25, w1, lt"),
        ([0x9c, 0xd3, 0x81, 0x1a], "csel w28, w28, w1, le"),
        ([0x00, 0x00, 0x82, 0x1a], "csel w0, w0, w2, eq"),
        ([0x00, 0x30, 0x82, 0x1a], "csel w0, w0, w2, lo"),
        ([0x42, 0x54, 0x82, 0x1a], "cinc w2, w2, mi"),
        ([0x62, 0x80, 0x82, 0x1a], "csel w2, w3, w2, hi"),
        ([0x02, 0xa0, 0x82, 0x1a], "csel w2, w0, w2, ge"),
        ([0x44, 0xa4, 0x84, 0x1a], "csinc w4, w2, w4, ge"),
        ([0x81, 0xc7, 0x9f, 0x1a], "csinc w1, w28, wzr, gt"),
        ([0x9c, 0xc7, 0x9f, 0x1a], "csinc w28, w28, wzr, gt"),
        ([0xf4, 0xc7, 0x9f, 0x1a], "cset w20, le"),
        ([0x94, 0xd2, 0x9f, 0x1a], "csel w20, w20, wzr, le"),
        ([0x00, 0xd4, 0x9f, 0x1a], "csinc w0, w0, wzr, le"),
        ([0x21, 0xd4, 0x9f, 0x1a], "csinc w1, w1, wzr, le"),
        ([0xfa, 0xd7, 0x9f, 0x1a], "cset w26, gt"),
        ([0x16, 0x04, 0x80, 0x5a], "cneg w22, w0, ne"),
        ([0x00, 0x44, 0x82, 0x5a], "csneg w0, w0, w2, mi"),
        ([0x9c, 0x17, 0x9c, 0x5a], "cneg w28, w28, eq"),
        ([0x9c, 0x03, 0x9f, 0x5a], "csinv w28, w28, wzr, eq"),
        ([0xfc, 0x03, 0x9f, 0x5a], "csetm w28, ne"),
        ([0x7b, 0x13, 0x9f, 0x5a], "csinv w27, w27, wzr, ne"),
        ([0xc6, 0xb0, 0x9f, 0x5a], "csinv w6, w6, wzr, lt"),
        ([0x63, 0xc0, 0x9f, 0x5a], "csinv w3, w3, wzr, gt"),
        ([0x84, 0x1b, 0x40, 0xfa], "ccmp x28, #0x0, #0x4, ne"),
        ([0x84, 0xca, 0x40, 0xfa], "ccmp x20, #0x0, #0x4, gt"),
        ([0x80, 0x08, 0x41, 0xfa], "ccmp x4, #0x1, #0x0, eq"),
        ([0x64, 0x12, 0x43, 0xfa], "ccmp x19, x3, #0x4, ne"),
        ([0x04, 0x10, 0x45, 0xfa], "ccmp x0, x5, #0x4, ne"),
        ([0x24, 0xd3, 0x5b, 0xfa], "ccmp x25, x27, #0x4, le"),
        ([0x00, 0x08, 0x40, 0x7a], "ccmp w0, #0x0, #0x0, eq"),
        ([0x01, 0xc0, 0x5c, 0x7a], "ccmp w0, w28, #0x1, gt"),
        ([0x04, 0x08, 0x41, 0x3a], "ccmn w0, #0x1, #0x4, eq"),
        ([0x84, 0xbb, 0x41, 0x3a], "ccmn w28, #0x1, #0x4, lt"),
        ([0x04, 0xc8, 0x41, 0x3a], "ccmn w0, #0x1, #0x4, gt"),
        ([0xa4, 0xc8, 0x41, 0x3a], "ccmn w5, #0x1, #0x4, gt"),
        ([0x84, 0xdb, 0x41, 0x3a], "ccmn w28, #0x1, #0x4, le"),
        ([0x84, 0x1b, 0x4a, 0x3a], "ccmn w28, #0xa, #0x4, ne"),
        ([0x08, 0x41, 0x80, 0x9a], "csel x8, x8, x0, mi"),
        ([0x8c, 0xb1, 0x8f, 0x9a], "csel x12, x12, x15, lt"),
        ([0xc6, 0x50, 0x97, 0x9a], "csel x6, x6, x23, pl"),
        ([0x08, 0x51, 0x97, 0x9a], "csel x8, x8, x23, pl"),
        ([0x77, 0xd0, 0x97, 0x9a], "csel x23, x3, x23, le"),
        ([0x9c, 0xa3, 0x9f, 0x9a], "csel x28, x28, xzr, ge"),
        ([0x00, 0xc0, 0x9f, 0x9a], "csel x0, x0, xzr, gt"),
        ([0x21, 0xc0, 0x9f, 0x9a], "csel x1, x1, xzr, gt"),
        ([0x33, 0xc0, 0x9f, 0x9a], "csel x19, x1, xzr, gt"),
        ([0x63, 0xc0, 0x9f, 0x9a], "csel x3, x3, xzr, gt"),
        ([0x39, 0xc3, 0x9f, 0x9a], "csel x25, x25, xzr, gt"),
        ([0x05, 0xc4, 0x9f, 0x9a], "csinc x5, x0, xzr, gt"),
        ([0x18, 0xc7, 0x9f, 0x9a], "csinc x24, x24, xzr, gt"),
        ([0x39, 0xc7, 0x9f, 0x9a], "csinc x25, x25, xzr, gt"),
        ([0xf7, 0xc7, 0x9f, 0x9a], "cset x23, le"),
        ([0x1b, 0xd0, 0x9f, 0x9a], "csel x27, x0, xzr, le"),
        ([0x00, 0xd3, 0x9f, 0x9a], "csel x0, x24, xzr, le"),
        ([0xe0, 0xd7, 0x9f, 0x9a], "cset x0, gt"),
    ];

    let errs = run_tests(TESTS);

    for err in errs.iter() {
        println!("{}", err);
    }

    assert!(errs.is_empty());
}

#[test]
fn test_openblas_muldiv_ops() {
    const TESTS: &[([u8; 4], &'static str)] = &[
        ([0x39, 0x08, 0xc0, 0x1a], "udiv w25, w1, w0"),
        ([0x3a, 0x09, 0xc1, 0x1a], "udiv w26, w9, w1"),
        ([0xa0, 0x0e, 0xc1, 0x1a], "sdiv w0, w21, w1"),
        ([0x00, 0x08, 0xc2, 0x1a], "udiv w0, w0, w2"),
        ([0x61, 0x0b, 0xc2, 0x1a], "udiv w1, w27, w2"),
        ([0x00, 0x0c, 0xc2, 0x1a], "sdiv w0, w0, w2"),
        ([0x01, 0x0d, 0xc2, 0x1a], "sdiv w1, w8, w2"),
        ([0x00, 0x08, 0xc3, 0x1a], "udiv w0, w0, w3"),
        ([0x6b, 0x0d, 0xc4, 0x1a], "sdiv w11, w11, w4"),
        ([0x73, 0x0e, 0xdc, 0x1a], "sdiv w19, w19, w28"),
        ([0xa1, 0x72, 0x00, 0x1b], "madd w1, w21, w0, w28"),
        ([0x3c, 0xa8, 0x00, 0x1b], "msub w28, w1, w0, w10"),
        ([0x80, 0x7f, 0x01, 0x1b], "mul w0, w28, w1"),
        ([0x00, 0x0c, 0xd3, 0x9a], "sdiv x0, x0, x19"),
        ([0x80, 0x0c, 0xda, 0x9a], "sdiv x0, x4, x26"),
        ([0x00, 0x0c, 0xdb, 0x9a], "sdiv x0, x0, x27"),
        ([0x00, 0x00, 0x00, 0x9b], "madd x0, x0, x0, x0"),
        ([0x9c, 0x7f, 0x1b, 0x9b], "mul x28, x28, x27"),
        ([0x00, 0xd4, 0x1b, 0x9b], "msub x0, x0, x27, x21"),
        ([0x20, 0x00, 0x1c, 0x9b], "madd x0, x1, x28, x0"),
        ([0x81, 0x7b, 0x1e, 0x9b], "madd x1, x28, x30, x30"),
        ([0xad, 0x7d, 0x1e, 0x9b], "mul x13, x13, x30"),
        ([0x02, 0x7f, 0x3a, 0x9b], "smull x2, w24, w26"),
        ([0x42, 0x64, 0x3c, 0x9b], "smaddl x2, w2, w28, x25"),
        ([0x29, 0x7c, 0x3c, 0x9b], "smull x9, w1, w28"),
        ([0x20, 0x7c, 0xa0, 0x9b], "umull x0, w1, w0"),
        ([0x42, 0x0c, 0xa4, 0x9b], "umaddl x2, w2, w4, x3"),
        ([0x21, 0x7c, 0xa4, 0x9b], "umull x1, w1, w4"),
        ([0x63, 0x98, 0xa4, 0x9b], "umsubl x3, w3, w4, x6"),
        ([0x85, 0x00, 0xa5, 0x9b], "umaddl x5, w4, w5, x0"),
        ([0x10, 0xa8, 0xb0, 0x9b], "umsubl x16, w0, w16, x10"),
    ];

    let errs = run_tests(TESTS);

    for err in errs.iter() {
        println!("{}", err);
    }

    assert!(errs.is_empty());
}

#[test]
fn test_openblas_branch_ops() {
    const TESTS: &[([u8; 4], &'static str)] = &[
        ([0x40, 0x00, 0x00, 0x34], "cbz w0, $+0x8"),
        ([0xcd, 0xff, 0xff, 0x34], "cbz w13, $-0x8"),
        ([0x40, 0x00, 0x00, 0x35], "cbnz w0, $+0x8"),
        ([0x19, 0xfd, 0xff, 0x35], "cbnz w25, $-0x60"),
        ([0xc1, 0xff, 0xff, 0x35], "cbnz w1, $-0x8"),
        ([0x61, 0x00, 0x00, 0x36], "tbz w1, #0x0, $+0xc"),
        ([0x20, 0x01, 0x00, 0x36], "tbz w0, #0x0, $+0x24"),
        ([0x73, 0x00, 0xf8, 0x36], "tbz w19, #0x1f, $+0xc"),
        ([0x27, 0xff, 0xff, 0x36], "tbz w7, #0x1f, $-0x1c"),
        ([0xf3, 0x05, 0x00, 0x37], "tbnz w19, #0x0, $+0xbc"),
        ([0xfb, 0xff, 0xff, 0x17], "b $-0x14"),
        ([0xfd, 0xff, 0xff, 0x17], "b $-0xc"),
        ([0x6d, 0x02, 0x00, 0x54], "b.le $+0x4c"),
        ([0x6d, 0xff, 0xff, 0x54], "b.le $-0x14"),
        ([0x97, 0x1e, 0x00, 0x94], "bl $+0x7a5c"),
        ([0xd2, 0xff, 0xff, 0x97], "bl $-0xb8"),
        ([0x53, 0x00, 0x00, 0xb4], "cbz x19, $+0x8"),
        ([0x7c, 0x00, 0x00, 0xb4], "cbz x28, $+0xc"),
        ([0x96, 0x00, 0x00, 0xb5], "cbnz x22, $+0x10"),
        ([0x98, 0x00, 0x00, 0xb5], "cbnz x24, $+0x10"),
        ([0xa6, 0x00, 0xf8, 0xb6], "tbz x6, #0x3f, $+0x14"),
        ([0x73, 0x03, 0xf8, 0xb7], "tbnz x19, #0x3f, $+0x6c"),
        ([0x23, 0xf8, 0xff, 0xb7], "tbnz x3, #0x3f, $-0xfc"),
        ([0x40, 0x00, 0x1f, 0xd6], "br x2"),
        ([0x00, 0x02, 0x1f, 0xd6], "br x16"),
        ([0x20, 0x02, 0x1f, 0xd6], "br x17"),
        ([0x00, 0x00, 0x3f, 0xd6], "blr x0"),
        ([0x20, 0x00, 0x3f, 0xd6], "blr x1"),
        ([0x80, 0x03, 0x3f, 0xd6], "blr x28"),
        ([0xc0, 0x03, 0x5f, 0xd6], "ret"),
    ];

    let errs = run_tests(TESTS);

    for err in errs.iter() {
        println!("{}", err);
    }

    assert!(errs.is_empty());
}

#[test]
fn test_openblas_simd_ops() {
    const TESTS: &[([u8; 4], &'static str)] = &[
        ([0x64, 0xcc, 0x21, 0x0e], "fmla v4.2s, v3.2s, v1.2s"),
        ([0x00, 0xd4, 0x21, 0x0e], "fadd v0.2s, v0.2s, v1.2s"),
        ([0x03, 0xcc, 0x22, 0x0e], "fmla v3.2s, v0.2s, v2.2s"),
        ([0x21, 0xd4, 0x23, 0x0e], "fadd v1.2s, v1.2s, v3.2s"),
        ([0x29, 0xd5, 0x35, 0x0e], "fadd v9.2s, v9.2s, v21.2s"),
        ([0x4a, 0xd5, 0x35, 0x0e], "fadd v10.2s, v10.2s, v21.2s"),
        ([0x42, 0x78, 0x61, 0x0e], "fcvtl v2.2d, v2.2s"),
        ([0x63, 0x78, 0x61, 0x0e], "fcvtl v3.2d, v3.2s"),
        ([0x00, 0xf8, 0xa0, 0x0e], "fabs v0.2s, v0.2s"),
        ([0x25, 0xcc, 0xa2, 0x0e], "fmls v5.2s, v1.2s, v2.2s"),
        ([0x10, 0x11, 0x80, 0x0f], "fmla v16.2s, v8.2s, v0.s[0]"),
        ([0x39, 0x58, 0x88, 0x0f], "fmls v25.2s, v1.2s, v8.s[2]"),
        ([0x08, 0x92, 0x8a, 0x0f], "fmul v8.2s, v16.2s, v10.s[0]"),
        ([0x01, 0x12, 0x8b, 0x0f], "fmla v1.2s, v16.2s, v11.s[0]"),
        ([0x8c, 0x12, 0x8e, 0x0f], "fmla v12.2s, v20.2s, v14.s[0]"),
        ([0xc6, 0x12, 0x8e, 0x0f], "fmla v6.2s, v22.2s, v14.s[0]"),
        ([0x3c, 0x58, 0xa9, 0x0f], "fmls v28.2s, v1.2s, v9.s[3]"),
        ([0x00, 0x06, 0x20, 0x1e], "fccmp s16, s0, #0x0, eq"),
        ([0x05, 0x08, 0x22, 0x1e], "fmul s5, s0, s2"),
        ([0x35, 0x88, 0x24, 0x1e], "fnmul s21, s1, s4"),
        ([0x42, 0x88, 0x24, 0x1e], "fnmul s2, s2, s4"),
        ([0x00, 0xdc, 0x26, 0x1e], "fcsel s0, s0, s6, le"),
        ([0x4e, 0x0c, 0x2e, 0x1e], "fcsel s14, s2, s14, eq"),
        ([0x41, 0x18, 0x2e, 0x1e], "fdiv s1, s2, s14"),
        ([0xd3, 0x5e, 0x33, 0x1e], "fcsel s19, s22, s19, pl"),
        ([0xe4, 0x3b, 0x3b, 0x1e], "fsub s4, s31, s27"),
        ([0xa1, 0x08, 0x3c, 0x1e], "fmul s1, s5, s28"),
        ([0xb8, 0x1a, 0x3c, 0x1e], "fdiv s24, s21, s28"),
        ([0x5c, 0x39, 0x3c, 0x1e], "fsub s28, s10, s28"),
        ([0x76, 0x08, 0x3d, 0x1e], "fmul s22, s3, s29"),
        ([0x59, 0x18, 0x3d, 0x1e], "fdiv s25, s2, s29"),
        ([0x26, 0x08, 0x3e, 0x1e], "fmul s6, s1, s30"),
        ([0x39, 0x3b, 0x3f, 0x1e], "fsub s25, s25, s31"),
        ([0x20, 0x04, 0x60, 0x1e], "fccmp d1, d0, #0x0, eq"),
        ([0x20, 0x06, 0x60, 0x1e], "fccmp d17, d0, #0x0, eq"),
        ([0x00, 0x08, 0x60, 0x1e], "fmul d0, d0, d0"),
        ([0xe8, 0x0b, 0x60, 0x1e], "fmul d8, d31, d0"),
        ([0xf3, 0x0b, 0x60, 0x1e], "fmul d19, d31, d0"),
        ([0x40, 0x0c, 0x60, 0x1e], "fcsel d0, d2, d0, eq"),
        ([0x08, 0x0d, 0x60, 0x1e], "fcsel d8, d8, d0, eq"),
        ([0xc4, 0x14, 0x60, 0x1e], "fccmp d6, d0, #0x4, ne"),
        ([0x72, 0x15, 0x60, 0x1e], "fccmpe d11, d0, #0x2, ne"),
        ([0xa4, 0x15, 0x60, 0x1e], "fccmp d13, d0, #0x4, ne"),
        ([0xe4, 0x15, 0x60, 0x1e], "fccmp d15, d0, #0x4, ne"),
        ([0x20, 0x18, 0x60, 0x1e], "fdiv d0, d1, d0"),
        ([0xc6, 0x1a, 0x60, 0x1e], "fdiv d6, d22, d0"),
        ([0x20, 0x1c, 0x60, 0x1e], "fcsel d0, d1, d0, ne"),
        ([0xef, 0x1d, 0x60, 0x1e], "fcsel d15, d15, d0, ne"),
        ([0x70, 0x22, 0x60, 0x1e], "fcmpe d19, d0"),
        ([0xd8, 0x22, 0x60, 0x1e], "fcmpe d22, #0.0"),
        ([0x08, 0x23, 0x60, 0x1e], "fcmp d24, #0.0"),
        ([0x80, 0x28, 0x60, 0x1e], "fadd d0, d4, d0"),
        ([0x22, 0x4c, 0x60, 0x1e], "fcsel d2, d1, d0, mi"),
        ([0x21, 0x5c, 0x60, 0x1e], "fcsel d1, d1, d0, pl"),
        ([0x20, 0x68, 0x60, 0x1e], "fmaxnm d0, d1, d0"),
        ([0xab, 0x6a, 0x60, 0x1e], "fmaxnm d11, d21, d0"),
        ([0xaf, 0x79, 0x60, 0x1e], "fminnm d15, d13, d0"),
        ([0x82, 0x8b, 0x60, 0x1e], "fnmul d2, d28, d0"),
        ([0xc0, 0x8b, 0x60, 0x1e], "fnmul d0, d30, d0"),
        ([0x22, 0x9c, 0x60, 0x1e], "fcsel d2, d1, d0, ls"),
        ([0x08, 0xad, 0x60, 0x1e], "fcsel d8, d8, d0, ge"),
        ([0x20, 0xcc, 0x60, 0x1e], "fcsel d0, d1, d0, gt"),
        ([0x6b, 0xdd, 0x60, 0x1e], "fcsel d11, d11, d0, le"),
        ([0xec, 0x0b, 0x61, 0x1e], "fmul d12, d31, d1"),
        ([0x41, 0x0c, 0x61, 0x1e], "fcsel d1, d2, d1, eq"),
        ([0x04, 0x14, 0x61, 0x1e], "fccmp d0, d1, #0x4, ne"),
        ([0xe1, 0x1b, 0x61, 0x1e], "fdiv d1, d31, d1"),
        ([0x00, 0x1c, 0x61, 0x1e], "fcsel d0, d0, d1, ne"),
        ([0xd0, 0x22, 0x61, 0x1e], "fcmpe d22, d1"),
        ([0x41, 0x2b, 0x61, 0x1e], "fadd d1, d26, d1"),
        ([0x00, 0x38, 0x61, 0x1e], "fsub d0, d0, d1"),
        ([0x81, 0x3b, 0x61, 0x1e], "fsub d1, d28, d1"),
        ([0x01, 0x4c, 0x61, 0x1e], "fcsel d1, d0, d1, mi"),
        ([0xef, 0x5d, 0x61, 0x1e], "fcsel d15, d15, d1, pl"),
        ([0x6c, 0x68, 0x61, 0x1e], "fmaxnm d12, d3, d1"),
        ([0x61, 0x78, 0x61, 0x1e], "fminnm d1, d3, d1"),
        ([0x06, 0x88, 0x61, 0x1e], "fnmul d6, d0, d1"),
        ([0x01, 0x9c, 0x61, 0x1e], "fcsel d1, d0, d1, ls"),
        ([0x02, 0xac, 0x61, 0x1e], "fcsel d2, d0, d1, ge"),
        ([0x84, 0xbc, 0x61, 0x1e], "fcsel d4, d4, d1, lt"),
        ([0x04, 0xc4, 0x61, 0x1e], "fccmp d0, d1, #0x4, gt"),
        ([0x11, 0xc4, 0x61, 0x1e], "fccmpe d0, d1, #0x1, gt"),
        ([0x50, 0xcc, 0x61, 0x1e], "fcsel d16, d2, d1, gt"),
        ([0x92, 0xcc, 0x61, 0x1e], "fcsel d18, d4, d1, gt"),
        ([0x00, 0xdc, 0x61, 0x1e], "fcsel d0, d0, d1, le"),
        ([0x20, 0x04, 0x62, 0x1e], "fccmp d1, d2, #0x0, eq"),
        ([0x00, 0x08, 0x62, 0x1e], "fmul d0, d0, d2"),
        ([0x8b, 0x09, 0x62, 0x1e], "fmul d11, d12, d2"),
        ([0x04, 0x14, 0x6a, 0x1e], "fccmp d0, d10, #0x4, ne"),
        ([0x72, 0x15, 0x6a, 0x1e], "fccmpe d11, d10, #0x2, ne"),
        ([0x00, 0x18, 0x6a, 0x1e], "fdiv d0, d0, d10"),
        ([0xf5, 0x5e, 0x75, 0x1e], "fcsel d21, d23, d21, pl"),
        ([0xa5, 0x68, 0x75, 0x1e], "fmaxnm d5, d5, d21"),
        ([0x42, 0x88, 0x75, 0x1e], "fnmul d2, d2, d21"),
        ([0x20, 0x06, 0x76, 0x1e], "fccmp d17, d22, #0x0, eq"),
        ([0x41, 0x18, 0x76, 0x1e], "fdiv d1, d2, d22"),
        ([0xa5, 0x18, 0x76, 0x1e], "fdiv d5, d5, d22"),
        ([0xf0, 0x22, 0x76, 0x1e], "fcmpe d23, d22"),
        ([0x00, 0x28, 0x76, 0x1e], "fadd d0, d0, d22"),
        ([0xf7, 0x39, 0x77, 0x1e], "fsub d23, d15, d23"),
        ([0x10, 0x3a, 0x77, 0x1e], "fsub d16, d16, d23"),
        ([0x01, 0x68, 0x77, 0x1e], "fmaxnm d1, d0, d23"),
        ([0x42, 0x88, 0x77, 0x1e], "fnmul d2, d2, d23"),
        ([0x00, 0x00, 0x78, 0x1e], "fcvtzs w0, d0"),
        ([0x85, 0x01, 0x78, 0x1e], "fcvtzs w5, d12"),
        ([0x00, 0x18, 0x7e, 0x1e], "fdiv d0, d0, d30"),
        ([0x42, 0x18, 0x7e, 0x1e], "fdiv d2, d2, d30"),
        ([0x84, 0x18, 0x7e, 0x1e], "fdiv d4, d4, d30"),
        ([0x52, 0x1a, 0x7e, 0x1e], "fdiv d18, d18, d30"),
        ([0xbd, 0x1b, 0x7e, 0x1e], "fdiv d29, d29, d30"),
        ([0x21, 0x28, 0x7e, 0x1e], "fadd d1, d1, d30"),
        ([0x63, 0x38, 0x7e, 0x1e], "fsub d3, d3, d30"),
        ([0x7c, 0x38, 0x7e, 0x1e], "fsub d28, d3, d30"),
        ([0xe7, 0x38, 0x7e, 0x1e], "fsub d7, d7, d30"),
        ([0x73, 0x3a, 0x7e, 0x1e], "fsub d19, d19, d30"),
        ([0x03, 0x08, 0x7f, 0x1e], "fmul d3, d0, d31"),
        ([0x23, 0x08, 0x7f, 0x1e], "fmul d3, d1, d31"),
        ([0x24, 0x08, 0x7f, 0x1e], "fmul d4, d1, d31"),
        ([0x29, 0x08, 0x7f, 0x1e], "fmul d9, d1, d31"),
        ([0x0c, 0xb2, 0x01, 0x1f], "fmsub s12, s16, s1, s12"),
        ([0x39, 0xe6, 0x74, 0x1f], "fnmsub d25, d17, d20, d25"),
        ([0x82, 0x0a, 0x75, 0x1f], "fnmadd d2, d20, d21, d2"),
        ([0xc6, 0xf4, 0x30, 0x4e], "fmax v6.4s, v6.4s, v16.4s"),
        ([0x64, 0xcc, 0x31, 0x4e], "fmla v4.4s, v3.4s, v17.4s"),
        ([0xc6, 0xd4, 0x3c, 0x4e], "fadd v6.4s, v6.4s, v28.4s"),
        ([0x82, 0xce, 0x3d, 0x4e], "fmla v2.4s, v20.4s, v29.4s"),
        ([0xdf, 0x7b, 0x61, 0x4e], "fcvtl2 v31.2d, v30.4s"),
        ([0x18, 0xf7, 0x79, 0x4e], "fmax v24.2d, v24.2d, v25.2d"),
        ([0xde, 0xd7, 0x7f, 0x4e], "fadd v30.2d, v30.2d, v31.2d"),
        ([0x21, 0xf8, 0xa0, 0x4e], "fabs v1.4s, v1.4s"),
        ([0x00, 0xd4, 0xa1, 0x4e], "fsub v0.4s, v0.4s, v1.4s"),
        ([0x65, 0xcc, 0xb0, 0x4e], "fmls v5.4s, v3.4s, v16.4s"),
        ([0x64, 0xcc, 0xb1, 0x4e], "fmls v4.4s, v3.4s, v17.4s"),
        ([0xff, 0xfb, 0xe0, 0x4e], "fabs v31.2d, v31.2d"),
        ([0x94, 0xd6, 0xe6, 0x4e], "fsub v20.2d, v20.2d, v6.2d"),
        ([0x30, 0xce, 0xf4, 0x4e], "fmls v16.2d, v17.2d, v20.2d"),
        ([0x4f, 0xce, 0xf4, 0x4e], "fmls v15.2d, v18.2d, v20.2d"),
        ([0x5a, 0x18, 0x88, 0x4f], "fmla v26.4s, v2.4s, v8.s[2]"),
        ([0x1d, 0x98, 0xaa, 0x4f], "fmul v29.4s, v0.4s, v10.s[3]"),
        ([0xfe, 0x18, 0xae, 0x4f], "fmla v30.4s, v7.4s, v14.s[3]"),
        ([0xfe, 0x50, 0xaf, 0x4f], "fmls v30.4s, v7.4s, v15.s[1]"),
        ([0x31, 0x50, 0xc8, 0x4f], "fmls v17.2d, v1.2d, v8.d[0]"),
        ([0xc7, 0x93, 0xcb, 0x4f], "fmul v7.2d, v30.2d, v11.d[0]"),
        ([0xb0, 0x18, 0xcc, 0x4f], "fmla v16.2d, v5.2d, v12.d[1]"),
        ([0xf7, 0x58, 0xcc, 0x4f], "fmls v23.2d, v7.2d, v12.d[1]"),
        ([0xa0, 0xb9, 0xa1, 0x5e], "fcvtzs s0, s13"),
        ([0x40, 0x10, 0x82, 0x5f], "fmla s0, s2, v2.s[0]"),
        ([0x84, 0x12, 0x8a, 0x5f], "fmla s4, s20, v10.s[0]"),
        ([0xa5, 0x12, 0x8a, 0x5f], "fmla s5, s21, v10.s[0]"),
        ([0x00, 0x13, 0x8a, 0x5f], "fmla s0, s24, v10.s[0]"),
        ([0x21, 0x13, 0x8a, 0x5f], "fmla s1, s25, v10.s[0]"),
        ([0x84, 0x13, 0x8a, 0x5f], "fmla s4, s28, v10.s[0]"),
        ([0xa5, 0x13, 0x8a, 0x5f], "fmla s5, s29, v10.s[0]"),
        ([0x00, 0x92, 0x8a, 0x5f], "fmul s0, s16, v10.s[0]"),
        ([0x84, 0x92, 0x8a, 0x5f], "fmul s4, s20, v10.s[0]"),
        ([0x00, 0x93, 0x8a, 0x5f], "fmul s0, s24, v10.s[0]"),
        ([0x84, 0x93, 0x8a, 0x5f], "fmul s4, s28, v10.s[0]"),
        ([0x01, 0x12, 0x8b, 0x5f], "fmla s1, s16, v11.s[0]"),
        ([0x85, 0x12, 0x8b, 0x5f], "fmla s5, s20, v11.s[0]"),
        ([0x01, 0x13, 0x8b, 0x5f], "fmla s1, s24, v11.s[0]"),
        ([0x85, 0x13, 0x8b, 0x5f], "fmla s5, s28, v11.s[0]"),
        ([0x20, 0x52, 0x8b, 0x5f], "fmls s0, s17, v11.s[0]"),
        ([0x20, 0x52, 0xb8, 0x5f], "fmls s0, s17, v24.s[1]"),
        ([0x40, 0x10, 0xc2, 0x5f], "fmla d0, d2, v2.d[0]"),
        ([0x03, 0x90, 0xc2, 0x5f], "fmul d3, d0, v2.d[0]"),
        ([0x20, 0x5a, 0xd8, 0x5f], "fmls d0, d17, v24.d[1]"),
        ([0x21, 0xdc, 0x60, 0x6e], "fmul v1.2d, v1.2d, v0.2d"),
        ([0x42, 0xdc, 0x60, 0x6e], "fmul v2.2d, v2.2d, v0.2d"),
        ([0x44, 0xdc, 0x60, 0x6e], "fmul v4.2d, v2.2d, v0.2d"),
        ([0x63, 0xdc, 0x60, 0x6e], "fmul v3.2d, v3.2d, v0.2d"),
        ([0x84, 0xdc, 0x60, 0x6e], "fmul v4.2d, v4.2d, v0.2d"),
        ([0x42, 0xdc, 0x61, 0x6e], "fmul v2.2d, v2.2d, v1.2d"),
        ([0x02, 0xdc, 0x62, 0x6e], "fmul v2.2d, v0.2d, v2.2d"),
        ([0x04, 0xdc, 0x62, 0x6e], "fmul v4.2d, v0.2d, v2.2d"),
        ([0x06, 0xdc, 0x62, 0x6e], "fmul v6.2d, v0.2d, v2.2d"),
        ([0x42, 0xd4, 0x63, 0x6e], "faddp v2.2d, v2.2d, v3.2d"),
        ([0x05, 0xdc, 0x63, 0x6e], "fmul v5.2d, v0.2d, v3.2d"),
        ([0x07, 0xdc, 0x63, 0x6e], "fmul v7.2d, v0.2d, v3.2d"),
        ([0x42, 0xdc, 0x63, 0x6e], "fmul v2.2d, v2.2d, v3.2d"),
        ([0x81, 0xd4, 0x64, 0x6e], "faddp v1.2d, v4.2d, v4.2d"),
        ([0x10, 0xdc, 0x64, 0x6e], "fmul v16.2d, v0.2d, v4.2d"),
        ([0x42, 0xdc, 0x64, 0x6e], "fmul v2.2d, v2.2d, v4.2d"),
        ([0x83, 0xd4, 0x65, 0x6e], "faddp v3.2d, v4.2d, v5.2d"),
        ([0x84, 0xd4, 0x65, 0x6e], "faddp v4.2d, v4.2d, v5.2d"),
        ([0x11, 0xdc, 0x65, 0x6e], "fmul v17.2d, v0.2d, v5.2d"),
        ([0x63, 0xdc, 0x65, 0x6e], "fmul v3.2d, v3.2d, v5.2d"),
        ([0x84, 0xdc, 0x65, 0x6e], "fmul v4.2d, v4.2d, v5.2d"),
        ([0xc6, 0xd4, 0x67, 0x6e], "faddp v6.2d, v6.2d, v7.2d"),
        ([0x10, 0xde, 0x68, 0x6e], "fmul v16.2d, v16.2d, v8.2d"),
        ([0x31, 0xde, 0x68, 0x6e], "fmul v17.2d, v17.2d, v8.2d"),
        ([0x52, 0xde, 0x68, 0x6e], "fmul v18.2d, v18.2d, v8.2d"),
        ([0x73, 0xde, 0x68, 0x6e], "fmul v19.2d, v19.2d, v8.2d"),
        ([0x94, 0xde, 0x68, 0x6e], "fmul v20.2d, v20.2d, v8.2d"),
        ([0xb5, 0xde, 0x68, 0x6e], "fmul v21.2d, v21.2d, v8.2d"),
        ([0xd6, 0xde, 0x68, 0x6e], "fmul v22.2d, v22.2d, v8.2d"),
        ([0xf7, 0xde, 0x68, 0x6e], "fmul v23.2d, v23.2d, v8.2d"),
        ([0x44, 0xdc, 0x70, 0x6e], "fmul v4.2d, v2.2d, v16.2d"),
        ([0x66, 0xdc, 0x70, 0x6e], "fmul v6.2d, v3.2d, v16.2d"),
        ([0x18, 0xde, 0x70, 0x6e], "fmul v24.2d, v16.2d, v16.2d"),
        ([0x54, 0xde, 0x70, 0x6e], "fmul v20.2d, v18.2d, v16.2d"),
        ([0x66, 0xde, 0x70, 0x6e], "fmul v6.2d, v19.2d, v16.2d"),
        ([0x04, 0xd6, 0x71, 0x6e], "faddp v4.2d, v16.2d, v17.2d"),
        ([0x10, 0xd6, 0x71, 0x6e], "faddp v16.2d, v16.2d, v17.2d"),
        ([0x45, 0xdc, 0x71, 0x6e], "fmul v5.2d, v2.2d, v17.2d"),
        ([0x66, 0xdc, 0x71, 0x6e], "fmul v6.2d, v3.2d, v17.2d"),
        ([0x39, 0xde, 0x71, 0x6e], "fmul v25.2d, v17.2d, v17.2d"),
        ([0x55, 0xde, 0x71, 0x6e], "fmul v21.2d, v18.2d, v17.2d"),
        ([0x66, 0xde, 0x71, 0x6e], "fmul v6.2d, v19.2d, v17.2d"),
        ([0x18, 0xf6, 0x71, 0x6e], "fmaxp v24.2d, v16.2d, v17.2d"),
        ([0x5a, 0xde, 0x72, 0x6e], "fmul v26.2d, v18.2d, v18.2d"),
        ([0x45, 0xd6, 0x73, 0x6e], "faddp v5.2d, v18.2d, v19.2d"),
        ([0x52, 0xd6, 0x73, 0x6e], "faddp v18.2d, v18.2d, v19.2d"),
        ([0x7b, 0xde, 0x73, 0x6e], "fmul v27.2d, v19.2d, v19.2d"),
        ([0x59, 0xf6, 0x73, 0x6e], "fmaxp v25.2d, v18.2d, v19.2d"),
        ([0x94, 0xd6, 0x75, 0x6e], "faddp v20.2d, v20.2d, v21.2d"),
        ([0x9a, 0xf6, 0x75, 0x6e], "fmaxp v26.2d, v20.2d, v21.2d"),
        ([0xd6, 0xd6, 0x77, 0x6e], "faddp v22.2d, v22.2d, v23.2d"),
        ([0xdb, 0xf6, 0x77, 0x6e], "fmaxp v27.2d, v22.2d, v23.2d"),
        ([0x18, 0xf7, 0x78, 0x6e], "fmaxp v24.2d, v24.2d, v24.2d"),
        ([0x18, 0xd7, 0x79, 0x6e], "faddp v24.2d, v24.2d, v25.2d"),
        ([0x18, 0xf7, 0x79, 0x6e], "fmaxp v24.2d, v24.2d, v25.2d"),
        ([0x18, 0xf7, 0x7a, 0x6e], "fmaxp v24.2d, v24.2d, v26.2d"),
        ([0x5a, 0xf7, 0x7b, 0x6e], "fmaxp v26.2d, v26.2d, v27.2d"),
        ([0x00, 0xd8, 0x30, 0x7e], "faddp s0, v0.2s"),
        ([0x21, 0xd8, 0x30, 0x7e], "faddp s1, v1.2s"),
        ([0x29, 0xd9, 0x30, 0x7e], "faddp s9, v9.2s"),
        ([0x4a, 0xd9, 0x30, 0x7e], "faddp s10, v10.2s"),
        ([0x00, 0xd8, 0x70, 0x7e], "faddp d0, v0.2d"),
        ([0x21, 0xd8, 0x70, 0x7e], "faddp d1, v1.2d"),
        ([0x29, 0xd9, 0x70, 0x7e], "faddp d9, v9.2d"),
        ([0x4a, 0xd9, 0x70, 0x7e], "faddp d10, v10.2d"),
        ([0x18, 0xdb, 0x70, 0x7e], "faddp d24, v24.2d"),
        ([0x00, 0xf8, 0x70, 0x7e], "fmaxp d0, v0.2d"),
        ([0x41, 0xf8, 0x70, 0x7e], "fmaxp d1, v2.2d"),
        ([0x20, 0xd4, 0xa0, 0x7e], "fabd s0, s1, s0"),
        ([0x21, 0xd4, 0xa0, 0x7e], "fabd s1, s1, s0"),
        ([0xd6, 0xd6, 0xf9, 0x7e], "fabd d22, d22, d25"),
        ([0x21, 0xd4, 0xfa, 0x7e], "fabd d1, d1, d26"),
        ([0x06, 0x00, 0x66, 0x9e], "fmov x6, d0"),
        ([0x11, 0x00, 0x66, 0x9e], "fmov x17, d0"),
        ([0x4b, 0x02, 0x67, 0x9e], "fmov d11, x18"),
        ([0xfc, 0x03, 0x67, 0x9e], "fmov d28, xzr"),
        ([0x06, 0x01, 0x78, 0x9e], "fcvtzs x6, d8"),
        ([0x42, 0xdc, 0x20, 0x2e], "fmul v2.2s, v2.2s, v0.2s"),
        ([0x44, 0xdc, 0x20, 0x2e], "fmul v4.2s, v2.2s, v0.2s"),
        ([0x42, 0xdc, 0x21, 0x2e], "fmul v2.2s, v2.2s, v1.2s"),
        ([0x02, 0xdc, 0x22, 0x2e], "fmul v2.2s, v0.2s, v2.2s"),
        ([0x04, 0xdc, 0x22, 0x2e], "fmul v4.2s, v0.2s, v2.2s"),
        ([0x05, 0xdc, 0x23, 0x2e], "fmul v5.2s, v0.2s, v3.2s"),
        ([0x81, 0xd4, 0x24, 0x2e], "faddp v1.2s, v4.2s, v4.2s"),
        ([0x00, 0xd4, 0x20, 0x6e], "faddp v0.4s, v0.4s, v0.4s"),
        ([0x21, 0xdc, 0x20, 0x6e], "fmul v1.4s, v1.4s, v0.4s"),
        ([0x42, 0xdc, 0x20, 0x6e], "fmul v2.4s, v2.4s, v0.4s"),
        ([0x21, 0xd4, 0x21, 0x6e], "faddp v1.4s, v1.4s, v1.4s"),
        ([0x04, 0xdc, 0x22, 0x6e], "fmul v4.4s, v0.4s, v2.4s"),
        ([0x06, 0xdc, 0x22, 0x6e], "fmul v6.4s, v0.4s, v2.4s"),
        ([0x42, 0xd4, 0x23, 0x6e], "faddp v2.4s, v2.4s, v3.4s"),
        ([0x05, 0xdc, 0x23, 0x6e], "fmul v5.4s, v0.4s, v3.4s"),
        ([0x07, 0xdc, 0x23, 0x6e], "fmul v7.4s, v0.4s, v3.4s"),
        ([0x10, 0xdc, 0x24, 0x6e], "fmul v16.4s, v0.4s, v4.4s"),
        ([0x83, 0xd4, 0x25, 0x6e], "faddp v3.4s, v4.4s, v5.4s"),
        ([0x84, 0xd4, 0x25, 0x6e], "faddp v4.4s, v4.4s, v5.4s"),
        ([0x11, 0xdc, 0x25, 0x6e], "fmul v17.4s, v0.4s, v5.4s"),
        ([0xc6, 0xd4, 0x27, 0x6e], "faddp v6.4s, v6.4s, v7.4s"),
        ([0x44, 0xdc, 0x30, 0x6e], "fmul v4.4s, v2.4s, v16.4s"),
        ([0x66, 0xdc, 0x30, 0x6e], "fmul v6.4s, v3.4s, v16.4s"),
        ([0x00, 0xf8, 0x30, 0x6e], "fmaxv s0, v0.4s"),
        ([0x21, 0xf8, 0x30, 0x6e], "fmaxv s1, v1.4s"),
        ([0x41, 0xf8, 0x30, 0x6e], "fmaxv s1, v2.4s"),
        ([0x10, 0xd6, 0x31, 0x6e], "faddp v16.4s, v16.4s, v17.4s"),
        ([0x45, 0xdc, 0x31, 0x6e], "fmul v5.4s, v2.4s, v17.4s"),
        ([0x66, 0xdc, 0x31, 0x6e], "fmul v6.4s, v3.4s, v17.4s"),
        ([0x52, 0xd6, 0x33, 0x6e], "faddp v18.4s, v18.4s, v19.4s"),
        ([0x94, 0xd6, 0x35, 0x6e], "faddp v20.4s, v20.4s, v21.4s"),
        ([0xd6, 0xd6, 0x37, 0x6e], "faddp v22.4s, v22.4s, v23.4s"),
        ([0x18, 0xd7, 0x39, 0x6e], "faddp v24.4s, v24.4s, v25.4s"),
        ([0xc7, 0x20, 0x06, 0x2e], "ext v7.8b, v6.8b, v6.8b, #0x4"),
        ([0x27, 0x1c, 0x20, 0x2e], "eor v7.8b, v1.8b, v0.8b"),
        ([0x22, 0x1c, 0x22, 0x2e], "eor v2.8b, v1.8b, v2.8b"),
        ([0x42, 0x1c, 0x32, 0x2e], "eor v2.8b, v2.8b, v18.8b"),
        ([0xc2, 0x1c, 0x70, 0x2e], "bsl v2.8b, v6.8b, v16.8b"),
        ([0xd2, 0x1d, 0xa1, 0x2e], "bit v18.8b, v14.8b, v1.8b"),
        ([0x01, 0x1c, 0xa6, 0x2e], "bit v1.8b, v0.8b, v6.8b"),
        ([0xa6, 0x1c, 0xe0, 0x2e], "bif v6.8b, v5.8b, v0.8b"),
        ([0x15, 0x1e, 0xf7, 0x2e], "bif v21.8b, v16.8b, v23.8b"),
        ([0x08, 0x85, 0xa0, 0x0e], "add v8.2s, v8.2s, v0.2s"),
        ([0x21, 0x1c, 0x21, 0x6e], "eor v1.16b, v1.16b, v1.16b"),
        ([0x42, 0x1c, 0x22, 0x6e], "eor v2.16b, v2.16b, v2.16b"),
        ([0x84, 0x1c, 0x24, 0x6e], "eor v4.16b, v4.16b, v4.16b"),
        ([0x10, 0x1e, 0x30, 0x6e], "eor v16.16b, v16.16b, v16.16b"),
        ([0x31, 0x1e, 0x31, 0x6e], "eor v17.16b, v17.16b, v17.16b"),
        ([0x73, 0x1e, 0x33, 0x6e], "eor v19.16b, v19.16b, v19.16b"),
        ([0xb5, 0x1e, 0x35, 0x6e], "eor v21.16b, v21.16b, v21.16b"),
        ([0xf7, 0x1e, 0x37, 0x6e], "eor v23.16b, v23.16b, v23.16b"),
        ([0x39, 0x1f, 0x39, 0x6e], "eor v25.16b, v25.16b, v25.16b"),
        ([0x7b, 0x1f, 0x3b, 0x6e], "eor v27.16b, v27.16b, v27.16b"),
        ([0xbd, 0x1f, 0x3d, 0x6e], "eor v29.16b, v29.16b, v29.16b"),
        ([0xff, 0x1f, 0x3f, 0x6e], "eor v31.16b, v31.16b, v31.16b"),
        ([0xa5, 0x40, 0x05, 0x6e], "ext v5.16b, v5.16b, v5.16b, #0x8"),
        ([0xc7, 0x40, 0x06, 0x6e], "ext v7.16b, v6.16b, v6.16b, #0x8"),
        ([0x02, 0x1d, 0x21, 0x0e], "and v2.8b, v8.8b, v1.8b"),
        ([0x21, 0x1c, 0x22, 0x0e], "and v1.8b, v1.8b, v2.8b"),
        ([0x21, 0x1c, 0x32, 0x0e], "and v1.8b, v1.8b, v18.8b"),
        ([0x08, 0x54, 0x22, 0x0f], "shl v8.2s, v0.2s, #0x2"),
        ([0xfe, 0x43, 0x60, 0x1e], "fmov d30, d31"),
        ([0x54, 0xc2, 0x20, 0x1e], "fabs s20, s18"),
        ([0xf7, 0xc3, 0x60, 0x1e], "fabs d23, d31"),
        ([0x23, 0x40, 0x21, 0x1e], "fneg s3, s1"),
        ([0x06, 0x40, 0x61, 0x1e], "fneg d6, d0"),
        ([0x00, 0xc0, 0x61, 0x1e], "fsqrt d0, d0"),
        ([0x02, 0xc2, 0x61, 0x1e], "fsqrt d2, d16"),
        ([0x08, 0x90, 0x26, 0x1e], "fmov s8, #20.0"),
        ([0x00, 0x10, 0x2e, 0x1e], "fmov s0, #1.0"),
        ([0x08, 0x90, 0x3b, 0x1e], "fmov s8, #-0.4375"),
        ([0x01, 0x10, 0x3c, 0x1e], "fmov s1, #-0.5"),
        ([0x00, 0x10, 0x3e, 0x1e], "fmov s0, #-1.0"),
        ([0x13, 0x10, 0x3e, 0x1e], "fmov s19, #-1.0"),
        ([0x14, 0x10, 0x60, 0x1e], "fmov d20, #2.0"),
        ([0x18, 0x10, 0x61, 0x1e], "fmov d24, #3.0"),
        ([0x13, 0x10, 0x7e, 0x1e], "fmov d19, #-1.0"),
    ];

    let errs = run_tests(TESTS);

    for err in errs.iter() {
        println!("{}", err);
    }

    assert!(errs.is_empty());
}

#[test]
fn test_openblas_simd_movs() {
    const TESTS: &[([u8; 4], &'static str)] = &[
        ([0x55, 0x04, 0x04, 0x4e], "mov v21.4s, v2.s[0]"),
        ([0x38, 0x07, 0x08, 0x4e], "mov v24.2d, v25.d[0]"),
        ([0x64, 0x64, 0x04, 0x6e], "mov v4.s[0], v3.s[3]"),
        ([0x67, 0x64, 0x04, 0x6e], "mov v7.s[0], v3.s[3]"),
        ([0x4b, 0x64, 0x14, 0x6e], "mov v11.s[2], v2.s[3]"),
        ([0x00, 0x04, 0x18, 0x6e], "mov v0.d[1], v0.d[0]"),
        ([0x21, 0x04, 0x18, 0x6e], "mov v1.d[1], v1.d[0]"),
        ([0x40, 0x04, 0x18, 0x6e], "mov v0.d[1], v2.d[0]"),
        ([0x41, 0x04, 0x18, 0x6e], "mov v1.d[1], v2.d[0]"),
        ([0x42, 0x04, 0x18, 0x6e], "mov v2.d[1], v2.d[0]"),
        ([0x48, 0x04, 0x18, 0x6e], "mov v8.d[1], v2.d[0]"),
        ([0x50, 0x04, 0x18, 0x6e], "mov v16.d[1], v2.d[0]"),
        ([0x63, 0x04, 0x18, 0x6e], "mov v3.d[1], v3.d[0]"),
        ([0x6b, 0x44, 0x18, 0x6e], "mov v11.d[1], v3.d[1]"),
        ([0xff, 0x45, 0x18, 0x6e], "mov v31.d[1], v15.d[1]"),
        ([0x5a, 0x46, 0x18, 0x6e], "mov v26.d[1], v18.d[1]"),
        ([0x00, 0x04, 0x1c, 0x6e], "mov v0.s[3], v0.s[0]"),
        ([0x21, 0x04, 0x1c, 0x6e], "mov v1.s[3], v1.s[0]"),
        ([0x68, 0x04, 0x1c, 0x6e], "mov v8.s[3], v3.s[0]"),
        ([0x69, 0x24, 0x1c, 0x6e], "mov v9.s[3], v3.s[1]"),
        ([0x6a, 0x44, 0x1c, 0x6e], "mov v10.s[3], v3.s[2]"),
        ([0x6b, 0x64, 0x1c, 0x6e], "mov v11.s[3], v3.s[3]"),
        ([0x43, 0x1c, 0xa2, 0x4e], "mov v3.16b, v2.16b"),
        ([0x1c, 0x04, 0x00, 0x0f], "movi v28.2s, #0x0"),
        ([0x28, 0x04, 0x00, 0x0f], "movi v8.2s, #0x1"),
        ([0x80, 0x66, 0x01, 0x0f], "movi v0.2s, #0x34, lsl #24"),
        ([0x17, 0x64, 0x04, 0x0f], "movi v23.2s, #0x80, lsl #24"),
        ([0x01, 0xe4, 0x00, 0x6f], "movi v1.2d, #0x0"),
        ([0x02, 0xe4, 0x00, 0x6f], "movi v2.2d, #0x0"),
    ];

    let errs = run_tests(TESTS);

    for err in errs.iter() {
        println!("{}", err);
    }

    assert!(errs.is_empty());
}

#[test]
fn test_openblas_misc_ops() {
    const TESTS: &[([u8; 4], &'static str)] = &[
        ([0x1f, 0x20, 0x03, 0xd5], "nop"),
        ([0xbf, 0x3a, 0x03, 0xd5], "dmb ishst"),
        ([0xbf, 0x3b, 0x03, 0xd5], "dmb ish"),
        ([0xdf, 0x3f, 0x03, 0xd5], "isb"),
        ([0x01, 0x00, 0x38, 0xd5], "mrs x1, midr_el1"),
        ([0x54, 0xd0, 0x3b, 0xd5], "mrs x20, tpidr_el0"),
        ([0x41, 0xe0, 0x3b, 0xd5], "mrs x1, cntvct_el0"),
        ([0x55, 0xe0, 0x3b, 0xd5], "mrs x21, cntvct_el0"),
        ([0xe2, 0x69, 0xb6, 0xf8], "prfm pldl2keep, [x15, x22]"),
        ([0x00, 0x6a, 0xb7, 0xf8], "prfm pldl1keep, [x16, x23]"),
        ([0x80, 0x00, 0x80, 0xf9], "prfm pldl1keep, [x4]"),
        ([0x81, 0x00, 0x80, 0xf9], "prfm pldl1strm, [x4]"),
        ([0x00, 0x02, 0x80, 0xf9], "prfm pldl1keep, [x16]"),
        ([0x21, 0x20, 0x80, 0xf9], "prfm pldl1strm, [x1, #0x40]"),
        ([0x00, 0x22, 0x80, 0xf9], "prfm pldl1keep, [x16, #0x40]"),
        ([0x61, 0x30, 0x80, 0xf9], "prfm pldl1strm, [x3, #0x60]"),
        ([0xa2, 0x51, 0x80, 0xf9], "prfm pldl2keep, [x13, #0xa0]"),
        ([0xc2, 0x51, 0x80, 0xf9], "prfm pldl2keep, [x14, #0xa0]"),
        ([0xe2, 0x51, 0x80, 0xf9], "prfm pldl2keep, [x15, #0xa0]"),
        ([0xa1, 0x00, 0x82, 0xf9], "prfm pldl1strm, [x5, #0x400]"),
        ([0x00, 0x82, 0x87, 0xf9], "prfm pldl1keep, [x16, #0xf00]"),
    ];

    let errs = run_tests(TESTS);

    for err in errs.iter() {
        println!("{}", err);
    }

    assert!(errs.is_empty());
}

#[test]
fn test_fmul() {
    const TESTS: &[([u8; 4], &'static str)] = &[
        ([0x00, 0x92, 0x8a, 0x5f], "fmul s0, s16, v10.s[0]"),
    ];
    let errs = run_tests(TESTS);

    for err in errs.iter() {
        println!("{}", err);
    }

    assert!(errs.is_empty());
}

#[test]
fn test_fmov_general() {
    const TESTS: &[([u8; 4], &'static str)] = &[
        ([0x4b, 0x02, 0x67, 0x9e], "fmov d11, x18"),
        ([0xfc, 0x03, 0x67, 0x9e], "fmov d28, xzr"),
        ([0x06, 0x00, 0x66, 0x9e], "fmov x6, d0"),
        ([0x11, 0x00, 0x66, 0x9e], "fmov x17, d0"),
    ];
    let errs = run_tests(TESTS);

    for err in errs.iter() {
        println!("{}", err);
    }

    assert!(errs.is_empty());
}

#[test]
fn test_cvt_general() {
    const TESTS: &[([u8; 4], &'static str)] = &[
        ([0x00, 0xd8, 0x21, 0x5e], "scvtf s0, s0"),
        ([0x82, 0xd8, 0x61, 0x5e], "scvtf d2, d4"),
        ([0x42, 0x78, 0x61, 0x0e], "fcvtl v2.2d, v2.2s"),
        ([0x63, 0x78, 0x61, 0x0e], "fcvtl v3.2d, v3.2s"),
    ];
    let errs = run_tests(TESTS);

    for err in errs.iter() {
        println!("{}", err);
    }

    assert!(errs.is_empty());
}

#[test]
fn test_fabd_general() {
    const TESTS: &[([u8; 4], &'static str)] = &[
        ([0x20, 0xd4, 0xa0, 0x7e], "fabd s0, s1, s0"),
        ([0x21, 0xd4, 0xa0, 0x7e], "fabd s1, s1, s0"),
        ([0xd6, 0xd6, 0xf9, 0x7e], "fabd d22, d22, d25"),
        ([0x21, 0xd4, 0xfa, 0x7e], "fabd d1, d1, d26"),
    ];
    let errs = run_tests(TESTS);

    for err in errs.iter() {
        println!("{}", err);
    }

    assert!(errs.is_empty());
}

#[test]
fn test_ext() {
    const TESTS: &[([u8; 4], &'static str)] = &[
        ([0xa5, 0x40, 0x05, 0x6e], "ext v5.16b, v5.16b, v5.16b, #0x8"),
        ([0xc7, 0x40, 0x06, 0x6e], "ext v7.16b, v6.16b, v6.16b, #0x8"),
        ([0xc7, 0x20, 0x06, 0x2e], "ext v7.8b, v6.8b, v6.8b, #0x4"),
    ];
    let errs = run_tests(TESTS);

    for err in errs.iter() {
        println!("{}", err);
    }

    assert!(errs.is_empty());
}

#[test]
fn test_indexed() {
    const TESTS: &[([u8; 4], &'static str)] = &[
        ([0x10, 0x11, 0x80, 0x0f], "fmla v16.2s, v8.2s, v0.s[0]"),
        ([0x39, 0x58, 0x88, 0x0f], "fmls v25.2s, v1.2s, v8.s[2]"),
        ([0x08, 0x92, 0x8a, 0x0f], "fmul v8.2s, v16.2s, v10.s[0]"),
        ([0x01, 0x12, 0x8b, 0x0f], "fmla v1.2s, v16.2s, v11.s[0]"),
        ([0x8c, 0x12, 0x8e, 0x0f], "fmla v12.2s, v20.2s, v14.s[0]"),
        ([0xc6, 0x12, 0x8e, 0x0f], "fmla v6.2s, v22.2s, v14.s[0]"),
        ([0x3c, 0x58, 0xa9, 0x0f], "fmls v28.2s, v1.2s, v9.s[3]"),
        ([0x00, 0x00, 0x50, 0x2f], "mla v0.4h, v0.4h, v0.h[1]"),
    ];
    let errs = run_tests(TESTS);

    for err in errs.iter() {
        println!("{}", err);
    }

    assert!(errs.is_empty());
}

#[test]
fn test_simd_abs() {
    const TESTS: &[([u8; 4], &'static str)] = &[
        ([0x00, 0xf8, 0xa0, 0x0e], "fabs v0.2s, v0.2s"),
        ([0x21, 0xf8, 0xa0, 0x4e], "fabs v1.4s, v1.4s"),
        ([0xf7, 0xc3, 0x60, 0x1e], "fabs d23, d31"),
    ];
    let errs = run_tests(TESTS);

    for err in errs.iter() {
        println!("{}", err);
    }

    assert!(errs.is_empty());
}

#[test]
fn test_movi() {
    const TESTS: &[([u8; 4], &'static str)] = &[
        ([0x1c, 0x04, 0x00, 0x0f], "movi v28.2s, #0x0"),
        ([0x28, 0x04, 0x00, 0x0f], "movi v8.2s, #0x1"),
        ([0x80, 0x66, 0x01, 0x0f], "movi v0.2s, #0x34, lsl #24"),
        ([0x17, 0x64, 0x04, 0x0f], "movi v23.2s, #0x80, lsl #24"),
        ([0x01, 0xe4, 0x00, 0x6f], "movi v1.2d, #0x0"),
        ([0x02, 0xe4, 0x00, 0x6f], "movi v2.2d, #0x0"),
        ([0x03, 0xe4, 0x00, 0x2f], "movi d3, #0x0"),
        ([0x23, 0xe4, 0x00, 0x2f], "movi d3, #0xff"),
        ([0xa3, 0xe4, 0x00, 0x2f], "movi d3, #0xff00ff"),
    ];
    let errs = run_tests(TESTS);

    for err in errs.iter() {
        println!("{}", err);
    }

    assert!(errs.is_empty());
}

#[test]
fn test_weird_str() {
    const TESTS: &[([u8; 4], &'static str)] = &[
        ([0x08, 0xdb, 0x7c, 0xf8], "ldr x8, [x24, w28, sxtw #3]"),
    ];
    let errs = run_tests(TESTS);

    for err in errs.iter() {
        println!("{}", err);
    }

    assert!(errs.is_empty());
}

#[test]
fn test_mismatches() {
    const TESTS: &[([u8; 4], &'static str)] = &[
        ([0x00, 0x00, 0x00, 0x08], "stxrb w0, w0, [x0]"),
    ];
    let errs = run_tests(TESTS);

    for err in errs.iter() {
        println!("{}", err);
    }

    assert!(errs.is_empty());
}

#[test]
fn test_cas() {
    const TESTS: &[([u8; 4], &'static str)] = &[
        ([0x20, 0x7c, 0x20, 0x08], "casp w0, w1, w0, w1, [x1]"),
        ([0x01, 0x7c, 0xa0, 0x08], "casb w0, w1, [x0]"),
    ];
    let errs = run_tests(TESTS);

    for err in errs.iter() {
        println!("{}", err);
    }

    assert!(errs.is_empty());
}

#[test]
fn test_stll() {
    const TESTS: &[([u8; 4], &'static str)] = &[
        ([0x00, 0x7c, 0x9f, 0x08], "stllrb w0, [x0]"),
    ];
    let errs = run_tests(TESTS);

    for err in errs.iter() {
        println!("{}", err);
    }

    assert!(errs.is_empty());
}

#[test]
fn test_ldst_structure() {
    const TESTS: &[([u8; 4], &'static str)] = &[
        ([0x00, 0xe0, 0x40, 0x0d], "ld3r {v0.8b, v1.8b, v2.8b}, [x0]"),
    ];
    let errs = run_tests(TESTS);

    for err in errs.iter() {
        println!("{}", err);
    }

    assert!(errs.is_empty());
}

#[test]
fn test_tbl() {
    const TESTS: &[([u8; 4], &'static str)] = &[
        ([0x00, 0x00, 0x00, 0x0e], "tbl v0.8b, {v0.16b}, v0.8b"),
        ([0x20, 0x00, 0x00, 0x0e], "tbl v0.8b, {v1.16b}, v0.8b"),
    ];
    let errs = run_tests(TESTS);

    for err in errs.iter() {
        println!("{}", err);
    }

    assert!(errs.is_empty());
}

#[test]
fn test_saddw() {
    const TESTS: &[([u8; 4], &'static str)] = &[
        ([0x00, 0x10, 0x20, 0x0e], "saddw v0.8h, v0.8h, v0.8b"),
        ([0x00, 0x60, 0x20, 0x0e], "subhn v0.8b, v0.8h, v0.8h"),
    ];
    let errs = run_tests(TESTS);

    for err in errs.iter() {
        println!("{}", err);
    }

    assert!(errs.is_empty());
}

#[test]
fn test_vector_cmge() {
    const TESTS: &[([u8; 4], &'static str)] = &[
        ([0x00, 0x98, 0x20, 0x0e], "cmeq v0.8b, v0.8b, #0x0"),
        ([0x00, 0xc8, 0xa0, 0x0e], "fcmgt v0.2s, v0.2s, #0.0"),
    ];
    let errs = run_tests(TESTS);

    for err in errs.iter() {
        println!("{}", err);
    }

    assert!(errs.is_empty());
}

#[test]
fn test_xtn() {
    const TESTS: &[([u8; 4], &'static str)] = &[
        ([0x00, 0x28, 0x21, 0x0e], "xtn v0.8b, v0.8h"),
    ];
    let errs = run_tests(TESTS);

    for err in errs.iter() {
        println!("{}", err);
    }

    assert!(errs.is_empty());
}

#[test]
fn test_saddlv() {
    const TESTS: &[([u8; 4], &'static str)] = &[
        ([0x00, 0x38, 0x30, 0x0e], "saddlv h0, v0.8b"),
    ];
    let errs = run_tests(TESTS);

    for err in errs.iter() {
        println!("{}", err);
    }

    assert!(errs.is_empty());
}

#[test]
fn test_simd_same_three_fp16() {
    const TESTS: &[([u8; 4], &'static str)] = &[
        ([0x00, 0x04, 0x40, 0x0e], "fmaxnm v0.4h, v0.4h, v0.4h"),
    ];
    let errs = run_tests(TESTS);

    for err in errs.iter() {
        println!("{}", err);
    }

    assert!(errs.is_empty());
}

#[test]
fn test_simd_two_register_misc_fp16() {
    const TESTS: &[([u8; 4], &'static str)] = &[
        ([0x00, 0x88, 0x79, 0x0e], "frintn v0.4h, v0.4h"),
    ];
    let errs = run_tests(TESTS);

    for err in errs.iter() {
        println!("{}", err);
    }

    assert!(errs.is_empty());
}

#[test]
fn test_simd_three_same_extra() {
    const TESTS: &[([u8; 4], &'static str)] = &[
        ([0x00, 0x94, 0x90, 0x0e], "sdot v0.2s, v0.8b, v16.8b"),
    ];
    let errs = run_tests(TESTS);

    for err in errs.iter() {
        println!("{}", err);
    }

    assert!(errs.is_empty());
}

#[test]
fn test_rev64() {
    const TESTS: &[([u8; 4], &'static str)] = &[
        ([0x00, 0x08, 0xa0, 0x0e], "rev64 v0.2s, v0.2s"),
    ];
    let errs = run_tests(TESTS);

    for err in errs.iter() {
        println!("{}", err);
    }

    assert!(errs.is_empty());
}

#[test]
fn test_adds_aliases() {
    const TESTS: &[([u8; 4], &'static str)] = &[
        ([0x1f, 0x00, 0x20, 0x2b], "cmn w0, w0, uxtb"),
        ([0x1f, 0x00, 0x40, 0x2b], "cmn w0, w0, lsr"),
    ];
    let errs = run_tests(TESTS);

    for err in errs.iter() {
        println!("{}", err);
    }

    assert!(errs.is_empty());
}

#[test]
fn test_mov_aliases() {
    const TESTS: &[([u8; 4], &'static str)] = &[
        // does not alias when lsl #12
        ([0x00, 0x00, 0x40, 0x11], "add w0, w0, #0x0, lsl #12"),
        // shift makes this not alias mov
        ([0xe0, 0x07, 0x00, 0x2a], "orr w0, wzr, w0, lsl #1"),
    ];
    let errs = run_tests(TESTS);

    for err in errs.iter() {
        println!("{}", err);
    }

    assert!(errs.is_empty());
}

#[test]
fn test_sbfm_aliases() {
    const TESTS: &[([u8; 4], &'static str)] = &[
        ([0x00, 0x7c, 0x00, 0x13], "asr w0, w0, #0x0"),
    ];
    let errs = run_tests(TESTS);

    for err in errs.iter() {
        println!("{}", err);
    }

    assert!(errs.is_empty());
}

#[test]
fn test_extr_aliases() {
    const TESTS: &[([u8; 4], &'static str)] = &[
        ([0x00, 0x40, 0x80, 0x13], "ror w0, w0, #0x10"),
    ];
    let errs = run_tests(TESTS);

    for err in errs.iter() {
        println!("{}", err);
    }

    assert!(errs.is_empty());
}

#[test]
fn test_armv8_3() {
    const TESTS: &[([u8; 4], &'static str)] = &[
        ([0x00, 0x00, 0x7e, 0x1e], "fjcvtzs w0, d0"),
    ];
    let errs = run_tests(TESTS);

    for err in errs.iter() {
        println!("{}", err);
    }

    assert!(errs.is_empty());
}

#[test]
fn test_armv8_4() {
    const TESTS: &[([u8; 4], &'static str)] = &[
        ([0x00, 0x00, 0x00, 0x19], "stlurb w0, [x0]"),
    ];
    let errs = run_tests(TESTS);

    for err in errs.iter() {
        println!("{}", err);
    }

    assert!(errs.is_empty());
}

#[test]
fn test_crc() {
    const TESTS: &[([u8; 4], &'static str)] = &[
        ([0x00, 0x44, 0xc0, 0x1a], "crc32h w0, w0, w0"),
        ([0x00, 0x50, 0xc0, 0x1a], "crc32cb w0, w0, w0"),
    ];
    let errs = run_tests(TESTS);

    for err in errs.iter() {
        println!("{}", err);
    }

    assert!(errs.is_empty());
}

#[test]
fn test_stnp() {
    const TESTS: &[([u8; 4], &'static str)] = &[
        ([0x00, 0x80, 0x00, 0x28], "stnp w0, w0, [x0, #0x4]"),
        ([0x00, 0x80, 0x00, 0x2c], "stnp s0, s0, [x0, #0x4]"),
    ];
    let errs = run_tests(TESTS);

    for err in errs.iter() {
        println!("{}", err);
    }

    assert!(errs.is_empty());
}

#[test]
fn test_preindex() {
    const TESTS: &[([u8; 4], &'static str)] = &[
        ([0x00, 0x00, 0x80, 0x2d], "stp s0, s0, [x0, #0x0]!"),
    ];
    let errs = run_tests(TESTS);

    for err in errs.iter() {
        println!("{}", err);
    }

    assert!(errs.is_empty());
}

#[test]
fn test_postindex() {
    const TESTS: &[([u8; 4], &'static str)] = &[
        ([0x20, 0x80, 0x81, 0x28], "stp w0, w0, [x1], #0xc"),
    ];
    let errs = run_tests(TESTS);

    for err in errs.iter() {
        println!("{}", err);
    }

    assert!(errs.is_empty());
}

#[test]
fn test_not_vec() {
    const TESTS: &[([u8; 4], &'static str)] = &[
        ([0x00, 0x58, 0x20, 0x2e], "mvn v0.8b, v0.8b"), // `not` instruction, aliased to mvn always
    ];
    let errs = run_tests(TESTS);

    for err in errs.iter() {
        println!("{}", err);
    }

    assert!(errs.is_empty());
}

#[test]
fn test_fcmla() {
    const TESTS: &[([u8; 4], &'static str)] = &[
        ([0x00, 0xe4, 0x40, 0x2e], "fcadd v0.4h, v0.4h, v0.4h, #0x5a"),
    ];
    let errs = run_tests(TESTS);

    for err in errs.iter() {
        println!("{}", err);
    }

    assert!(errs.is_empty());
}

#[test]
fn test_vec_shift() {
    const TESTS: &[([u8; 4], &'static str)] = &[
        ([0x00, 0xa4, 0x08, 0x0f], "sshll v0.8h, v0.8b, #0x0"),
        ([0x00, 0xa4, 0x08, 0x2f], "ushll v0.8h, v0.8b, #0x0"),
    ];
    let errs = run_tests(TESTS);

    for err in errs.iter() {
        println!("{}", err);
    }

    assert!(errs.is_empty());
}

#[test]
fn test_reserved() {
    test_err([0x00, 0x00, 0x20, 0xd5], DecodeError::InvalidOpcode);
    test_err([0x00, 0x00, 0x00, 0xd5], DecodeError::InvalidOpcode);
}

#[test]
fn test_system() {
    // capstone says `msr s0_0_c2_c0_0, xzr`, but this looks like an unallocated encoding?
    test_err([0x1f, 0x20, 0x00, 0xd5], DecodeError::InvalidOpcode);

    const TESTS: &[([u8; 4], &'static str)] = &[
        ([0x00, 0x00, 0x08, 0xd5], "sys #0x0, c0, c0, #0x0, x0"),
        ([0x00, 0x00, 0x28, 0xd5], "sysl x0, #0x0, c0, c0, #0x0"),
        ([0x10, 0x00, 0x08, 0xd5], "sys #0x0, c0, c0, #0x0, x16"),
        ([0x10, 0x00, 0x28, 0xd5], "sysl x16, #0x0, c0, c0, #0x0"),
        ([0xab, 0x03, 0x08, 0xd5], "sys #0x0, c0, c3, #0x5, x11"),
        ([0xab, 0x03, 0x28, 0xd5], "sysl x11, #0x0, c0, c3, #0x5"),
        ([0x00, 0x00, 0x30, 0xd5], "mrs x0, s2_0_c0_c0_0"),
    ];
    let errs = run_tests(TESTS);

    for err in errs.iter() {
        println!("{}", err);
    }

    assert!(errs.is_empty());
}

#[test]
fn test_pac() {
    const TESTS: &[([u8; 4], &'static str)] = &[
        ([0x00, 0x04, 0xc1, 0xda], "pacib x0, x0"),
        ([0x00, 0x04, 0xa8, 0xf8], "ldrab x0, [x0, #0x400]"),
        ([0x00, 0x04, 0xb0, 0xf8], "ldrab x0, [x0, #0x800]"),
        ([0x00, 0x04, 0xf0, 0xf8], "ldrab x0, [x0, #-0x800]"),
        ([0x00, 0x14, 0x20, 0xf8], "ldraa x0, [x0, #0x8]"),
        ([0x00, 0x04, 0xa4, 0xf8], "ldrab x0, [x0, #0x200]"),
    ];
    let errs = run_tests(TESTS);

    for err in errs.iter() {
        println!("{}", err);
    }

    assert!(errs.is_empty());
}

#[test]
fn test_uq() {
    const TESTS: &[([u8; 4], &'static str)] = &[
        ([0x00, 0x38, 0xe0, 0x4e], "suqadd v0.2d, v0.2d"),
        ([0x00, 0x88, 0xe0, 0x4e], "cmgt v0.2d, v0.2d, #0x0"),
        ([0x00, 0xb8, 0xe0, 0x4e], "abs v0.2d, v0.2d"),
        ([0x00, 0xfc, 0x20, 0x5e], "frecps s0, s0, s0"),
        ([0x00, 0xc8, 0xb0, 0x5e], "fminnmp h0, v0.2h"),
        ([0x00, 0x74, 0x08, 0x5f], "sqshl b0, b0, #0x0"),
        ([0x00, 0x94, 0x08, 0x5f], "sqshrn b0, h0, #0x8"),
    ];
    let errs = run_tests(TESTS);

    for err in errs.iter() {
        println!("{}", err);
    }

    assert!(errs.is_empty());
}

#[test]
fn test_cfi() {
    const TESTS: &[([u8; 4], &'static str)] = &[
        ([0x1f, 0x08, 0x1f, 0xd6], "braaz x0"),
        ([0x3f, 0x08, 0x1f, 0xd6], "braaz x1"),
        ([0x00, 0x08, 0x1f, 0xd7], "braa x0, x0"),
        ([0x00, 0x08, 0x3f, 0xd7], "blraa x0, x0"),
    ];
    let errs = run_tests(TESTS);

    for err in errs.iter() {
        println!("{}", err);
    }

    assert!(errs.is_empty());
}

#[test]
fn test_sha() {
    const TESTS: &[([u8; 4], &'static str)] = &[
        ([0x00, 0x00, 0x00, 0x5e], "sha1c q0, s0, v0.4s"),
    ];
    let errs = run_tests(TESTS);

    for err in errs.iter() {
        println!("{}", err);
    }

    assert!(errs.is_empty());
}

#[test]
fn test_misc() {
    const TESTS: &[([u8; 4], &'static str)] = &[
        ([0x00, 0x98, 0x00, 0x2f], "fmulx v0.4h, v0.4h, v0.h[4]"),
        ([0x00, 0xc4, 0x00, 0x2f], "mvni v0.2s, #0x0, msl #8"),
        ([0x00, 0xd4, 0x00, 0x2f], "mvni v0.2s, #0x0, msl #16"),
        ([0x00, 0xe0, 0x80, 0x2f], "udot v0.2s, v0.8b, v0.4b[0]"),
        ([0x00, 0x03, 0x00, 0x32], "orr w0, w24, #0x1"),
        ([0x00, 0x00, 0x00, 0x33], "bfxil w0, w0, #0x0, #0x1"),
//        ([0xe0, 0x03, 0x00, 0x33], "bfc w0, #0x0, #0x1"),
        ([0x00, 0x00, 0x01, 0x33], "bfi w0, w0, #0x1f, #0x1"),
        ([0x00, 0x08, 0x00, 0x38], "sttrb w0, [x0]"),
        ([0x1f, 0x00, 0x20, 0x38], "staddb w0, [x0]"),
        ([0x00, 0x58, 0x20, 0x38], "strb w0, [x0, w0, uxtw]"),
        ([0x00, 0xc0, 0xbf, 0x38], "ldaprb w0, [x0]"),
        ([0x0d, 0x08, 0x00, 0x3a], "setf8 w0"),
        ([0x00, 0x58, 0xbf, 0x3c], "str q0, [x0, wzr, uxtw #4]"),
        ([0x00, 0x04, 0x01, 0x4e], "mov v0.16b, v0.b[0]"), // "The alias is always the preferred disassembly."
        ([0x00, 0x38, 0x20, 0x4e], "suqadd v0.16b, v0.16b"),
        ([0x00, 0x88, 0x20, 0x4e], "cmgt v0.16b, v0.16b, #0x0"),
        ([0x00, 0xb8, 0x20, 0x4e], "abs v0.16b, v0.16b"),
        ([0x00, 0xe0, 0x20, 0x4e], "pmull2 v0.8h, v0.16b, v0.16b"),
        ([0x00, 0x28, 0x21, 0x4e], "xtn2 v0.16b, v0.8h"),
        ([0x00, 0x48, 0x28, 0x4e], "aese v0.16b, v0.16b"),
        ([0x00, 0x84, 0x08, 0x4f], "shrn2 v0.16b, v0.8h, #0x8"),
        ([0x00, 0x8c, 0x08, 0x4f], "rshrn2 v0.16b, v0.8h, #0x8"),
        ([0x00, 0x94, 0x08, 0x4f], "sqshrn2 v0.16b, v0.8h, #0x8"),
        ([0x00, 0x28, 0x40, 0x4f], "smlal2 v0.4s, v0.8h, v0.h[4]"),
        ([0x00, 0xc0, 0x40, 0x4f], "sqdmulh v0.8h, v0.8h, v0.h[0]"),
        ([0x00, 0xe0, 0x80, 0x4f], "sdot v0.4s, v0.16b, v0.4b[0]"),
        ([0x00, 0x24, 0x40, 0x5e], "fcmeq h0, h0, h0"),
        ([0x00, 0x88, 0xe0, 0x5e], "cmgt d0, d0, #0x0"),
        ([0x00, 0xa8, 0xf9, 0x5e], "fcvtps h0, h0"),
        ([0x00, 0x10, 0x10, 0x5f], "fmla h0, h0, v0.h[1]"),
        ([0x00, 0xd0, 0x40, 0x5f], "sqrdmulh h0, h0, v0.h[0]"),
        ([0x00, 0x10, 0xd0, 0x5f], "fmla d0, d0, v16.d[0]"),
        ([0x1f, 0x00, 0x20, 0x6b], "cmp w0, w0, uxtb"),
        ([0x00, 0x78, 0x20, 0x6e], "sqneg v0.16b, v0.16b"),
        ([0x00, 0xb8, 0x20, 0x6e], "neg v0.16b, v0.16b"),
        ([0x00, 0x28, 0x21, 0x6e], "sqxtun2 v0.16b, v0.8h"),
        ([0x1f, 0x00, 0x00, 0x72], "tst w0, #0x1"),
        ([0x00, 0x84, 0x40, 0x7e], "sqrdmlah h0, h0, h0"),
        ([0x1f, 0x00, 0x20, 0x8b], "add sp, x0, w0, uxtb"),
//        ([0x00, 0x00, 0x00, 0x00], "add x0, x0, #0x0"),
        ([0x1f, 0x00, 0x00, 0x91], "mov sp, x0"),
        ([0x1f, 0x00, 0x00, 0x92], "and sp, x0, #0x100000001"),
        ([0x00, 0x4c, 0xc0, 0x9a], "crc32x w0, w0, x0"),
        ([0x00, 0x00, 0x40, 0x9b], "smulh x0, x0, x0"),
        ([0x00, 0x08, 0x40, 0xce], "sm3ss1 v0.4s, v0.4s, v0.4s, v2.4s"),
//        ([0x00, 0x1c, 0x40, 0xd3], "ubfx x0, x0, #0x0, #0x8"),
        ([0x00, 0x08, 0x00, 0xf8], "sttr x0, [x0]"),
        ([0x00, 0x14, 0x20, 0xf8], "ldraa x0, [x0, #0x8]"),
        ([0x00, 0x00, 0x80, 0xf8], "prfum pldl1keep, [x0]"),
        ([0x06, 0x48, 0xa0, 0xf8], "prfm 0x6, [x0, w0, uxtw]"),
        ([0xe0, 0x03, 0x00, 0x0b], "add w0, wzr, w0"),
        ([0x00, 0xc0, 0x40, 0x0f], "sqdmulh v0.4h, v0.4h, v0.h[0]"),
        ([0x00, 0xd0, 0x40, 0x0f], "sqrdmulh v0.4h, v0.4h, v0.h[0]"),
        ([0x00, 0xc0, 0x40, 0x5f], "sqdmulh h0, h0, v0.h[0]"),
        ([0x00, 0x10, 0x40, 0x6f], "fcmla v0.8h, v0.8h, v0.h[0], #0x0"),
        ([0x00, 0x10, 0x80, 0x6f], "fcmla v0.4s, v0.4s, v0.s[0], #0x0"),
        ([0x00, 0x14, 0xc0, 0x7e], "fabd h0, h0, h0"),
        ([0x00, 0x24, 0xc0, 0x7e], "fcmgt h0, h0, h0"),
        ([0x00, 0xd0, 0x40, 0x7f], "sqrdmlah h0, h0, v0.h[0]"),
        ([0x00, 0x7c, 0xa0, 0x88], "cas w0, w0, [x0]"),
        ([0x00, 0x30, 0xc0, 0x9a], "pacga x0, x0, x0"),
        ([0x00, 0x00, 0xae, 0x9e], "fmov x0, v0.d[1]"),
        ([0x00, 0x00, 0xe6, 0x9e], "fmov x0, h0"),
        ([0x7f, 0x41, 0x00, 0xd5], "msr pstate.0x3, #0x1"),
        ([0x00, 0x68, 0x20, 0x38], "strb w0, [x0, x0]"),
    ];
    let errs = run_tests(TESTS);

    for err in errs.iter() {
        println!("{}", err);
    }

    assert!(errs.is_empty());
}

#[test]
fn test_bitfield() {
    const TESTS: &[([u8; 4], &'static str)] = &[
        ([0x00, 0x1c, 0x40, 0xd3], "ubfx x0, x0, #0x0, #0x8"),
        ([0x1f, 0x1c, 0x00, 0x53], "uxtb wzr, w0"),
    ];

    let errs = run_tests(TESTS);

    for err in errs.iter() {
        println!("{}", err);
    }

    assert!(errs.is_empty());
}
