/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  backend-dbib.c defines the DocBook V.3.1 bibliography output backend
  of refdbd
  markus@mhoenicka.de 2001-01-10

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, see <http://www.gnu.org/licenses/>

  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/


#include <string.h>
#include <stdio.h>
#include <dbi/dbi.h>
#include <syslog.h>

#include <assert.h>

#include "refdb.h"
#include "linklist.h"
#include "refdbd.h"
#include "backend.h"
#include "xmlhelper.h"
#include "backend-dbib.h"
#include "strfncs.h"
#include "dbfncs.h"
#include "connect.h"
#include "mset.h"

extern int n_log_level;
extern char main_db[];
extern dbi_result dbi_style_res;


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  prepare_render_dbib(): writes a header for the docbook output of a
                         query

  int prepare_render_dbib returns 0 if ok, >0 if error

  char** ptr_ref ptr to a ptr to a buffer created with malloc which will receive
        the output. The calling function must allocate the buffer
        with at least 4096 byte. This function will reallocate the
        buffer as needed. *ptr will be updated whenever a realloc is
        necessary. The calling function is responsible for freeing the
        memory again.

  size_t* ptr_ref_len pointer to variable holding current length of ref

  struct bibinfo* ptr_biblio_info ptr to a structure containing formatting
        information

  dbi_conn conn the connection to the refdb database.

  int n_ref_format requested output format: REFDOCBK, REFDOCBKX, REFTEIX

  const char* ns string containing an optional namespace prefix

 ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
int prepare_render_dbib(char** ptr_ref, size_t* ptr_ref_len, struct bibinfo* ptr_biblio_info, dbi_conn conn, int n_ref_format, const char* ns) {
  char *sql_command = NULL;
  char *item = NULL;
  char *myjournal = NULL;
  const char *drivername;
  const char* db_encoding;
  dbi_result dbires_cit;

/*    printf("start prepare_render_dbib\n"); */
  drivername = dbi_driver_get_name(dbi_conn_get_driver(conn));

  myjournal = strdup(ptr_biblio_info->format_string);

  if (!myjournal) {
    return 801;
  }

  if (dbi_conn_quote_string(conn, &myjournal) == 0) {
    free(myjournal);
    return 801;
  }

  sql_command = malloc(60+strlen(myjournal));

  if (!sql_command) {
    free(myjournal);
    return 801;
  }

  /* get the database encoding */
  if (!*(ptr_biblio_info->encoding)) {
    db_encoding = dbi_conn_get_encoding(conn);
  }
  else {
    db_encoding = (const char*)ptr_biblio_info->encoding;
  }

  if (!strcmp(my_dbi_conn_get_cap(conn, "multiple_db"), "t")) {
    sprintf(sql_command, "SELECT BIBLIOTITLE FROM %s.CITSTYLE WHERE JOURNAL=%s", main_db, myjournal);
  }
  else {
    sprintf(sql_command, "SELECT BIBLIOTITLE FROM CITSTYLE WHERE JOURNAL=%s", myjournal);
  }
  free(myjournal);

  LOG_PRINT(LOG_DEBUG, sql_command);
  dbires_cit = dbi_conn_query(conn, sql_command);
  free(sql_command);

  if (!dbires_cit || dbi_result_next_row(dbires_cit) == 0) {
    return 234;
  }
  
  item = my_dbi_result_get_string_copy_idx(dbires_cit, 1);

  if (item && !my_dbi_conn_error_flag(conn)) { /* as specified in the style */
    if (sgml_entitize(&item, n_ref_format) == NULL) {
      free(item);
      return 801;
    }
    
    /* assemble the header */
    /* - the bibliography must not have a doctype line if it is included
         as an external entity */
    /* - the initial indentation level depends on how many levels we add
         with the following strings. Adapt initialize_xmlindent() in
	 refdbdbib.c accordingly */
    if (n_ref_format == REFDOCBK) {
      sprintf(*ptr_ref, "<!-- <!DOCTYPE bibliography PUBLIC \"-//OASIS//DTD DocBook V3.1//EN\"> -->\n<bibliography role=\"refdb\">\n<title>%s</title>\n", item);
    }
    else if (n_ref_format == REFDOCBKX) {
      sprintf(*ptr_ref, "<!-- <?xml version=\"1.0\" encoding=\"%s\"?>\n<!DOCTYPE bibliography PUBLIC \"-//OASIS//DTD DocBook XML V4.1.2//EN\" \"http://www.oasis-open.org/docbook/xml/4.1.2/docbookx.dtd\"> -->\n<bibliography role=\"refdb\">\n<title>%s</title>\n", db_encoding, item);
    }
    else if (n_ref_format == REFDOCBKX5) {
      if (ns && *ns) {
	sprintf(*ptr_ref, "<?xml version=\"1.0\" encoding=\"%s\"?>\n\n<%s:bibliography role=\"refdb\" xmlns:%s=\"http://docbook.org/ns/docbook\" xmlns:xl=\"http://www.w3.org/1999/xlink\">\n<%s:title>%s</%s:title>\n", db_encoding, ns, ns, ns, item, ns);
      }
      else {
	sprintf(*ptr_ref, "<?xml version=\"1.0\" encoding=\"%s\"?>\n\n<bibliography role=\"refdb\" xmlns=\"http://docbook.org/ns/docbook\">\n<title>%s</title>\n", db_encoding, item);
      }
    }
    else if (n_ref_format == REFTEIX) {
      sprintf(*ptr_ref, "<?xml version=\"1.0\" encoding=\"%s\"?>\n<!DOCTYPE bibliography PUBLIC \"-//OASIS//DTD DocBook XML V4.1.2//EN\" \"http://www.oasis-open.org/docbook/xml/4.1.2/docbookx.dtd\">\n<bibliography role=\"refdb\">\n<title>%s</title>\n", db_encoding, item);
    }
    else if (n_ref_format == REFTEIX5) {
      if (ns && *ns) {
	sprintf(*ptr_ref, "<?xml version=\"1.0\" encoding=\"%s\"?>\n\n<%s:div type=\"chapter\" xmlns:%s=\"http://www.tei-c.org/ns/1.0\">\n<%s:head><%s:title>%s</%s:title></%s:head>\n<%s:listBibl rend=\"refdb\">\n\n", db_encoding, ns, ns, ns, ns, item, ns, ns, ns);
      }
      else {
	sprintf(*ptr_ref, "<?xml version=\"1.0\" encoding=\"%s\"?>\n\n<div type=\"chapter\" xmlns=\"http://www.tei-c.org/ns/1.0\">\n<head><title>%s</title></head>\n<listBibl rend=\"refdb\">\n", db_encoding, item);
      }
    }
  }
  else { /* leave out title as none was specified */
    if (n_ref_format == REFDOCBK) {
      strcpy(*ptr_ref, "<!-- <!DOCTYPE bibliography PUBLIC \"-//OASIS//DTD DocBook V3.1//EN\"> -->\n<bibliography role=\"refdb\">\n");
    }
    else if (n_ref_format == REFDOCBKX) {
      sprintf(*ptr_ref, "<!-- <?xml version=\"1.0\" encoding=\"%s\"?>\n<!DOCTYPE bibliography PUBLIC \"-//OASIS//DTD DocBook XML V4.1.2//EN\" \"http://www.oasis-open.org/docbook/xml/4.1.2/docbookx.dtd\"> -->\n<bibliography role=\"refdb\">\n", db_encoding);
    }
    else if (n_ref_format == REFDOCBKX5) {
      if (ns && *ns) {
	sprintf(*ptr_ref, "<?xml version=\"1.0\" encoding=\"%s\"?>\n\n<%s:bibliography role=\"refdb\" xmlns:%s=\"http://docbook.org/ns/docbook\" xmlns:xl=\"http://www.w3.org/1999/xlink\">\n", db_encoding, ns, ns);
      }
      else {
	sprintf(*ptr_ref, "<?xml version=\"1.0\" encoding=\"%s\"?>\n\n<bibliography role=\"refdb\" xmlns=\"http://docbook.org/ns/docbook\">\n", db_encoding);
      }
    }
    else if (n_ref_format == REFTEIX) {
      sprintf(*ptr_ref, "<?xml version=\"1.0\" encoding=\"%s\"?>\n<!DOCTYPE bibliography PUBLIC \"-//OASIS//DTD DocBook XML V4.1.2//EN\" \"http://www.oasis-open.org/docbook/xml/4.1.2/docbookx.dtd\">\n<bibliography role=\"refdb\">\n", db_encoding);
    }
    else if (n_ref_format == REFTEIX5) {
      if (ns && *ns) {
	sprintf(*ptr_ref, "<?xml version=\"1.0\" encoding=\"%s\"?>\n\n<%s:div type=\"chapter\" xmlns:%s=\"http://www.tei-c.org/ns/1.0\">\n<%s:listBibl rend=\"refdb\">\n", db_encoding, ns, ns, ns);
      }
      else {
	sprintf(*ptr_ref, "<?xml version=\"1.0\" encoding=\"%s\"?>\n\n<div type=\"chapter\" xmlns:%s=\"http://www.tei-c.org/ns/1.0\">\n<%s:listBibl rend=\"refdb\">\n", db_encoding, ns, ns);
      }
    }
  }

  if (item) {
    free(item);
  }

/*    printf("end prepare_render_dbib\n"); */
  return 0;
}

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  finish_render_dbib(): writes a footer for the docbook output of a query
                        This may include extra bibliomixed elements
                        for multiple citation labels

  int finish_render_dbib returns 0 successful, >0 if failed

  char** ptr_ref ptr to a ptr to a buffer created with malloc which will receive
        the output. The calling function must allocate the buffer
        with at least 4096 byte. This function will reallocate the
        buffer as needed. *ptr will be updated whenever a realloc is
        necessary. The calling function is responsible for freeing the
        memory again.

  size_t* ptr_ref_len pointer to variable holding current length of ref

  struct BIBCONNS* ptr_bibconns ptr to struct with dbi connections

  struct bibinfo* ptr_biblio_info ptr to struct with additional bibliographic
                  info

  char* table_name ptr to a string containing the name of the 
                  temporary table in refdbvar which holds the info
                  about which IDs to render

  struct CLIENT_REQUEST ptr_clrequest ptr to structure with client info

  struct xmlindent* ptr_indent ptr to indentation info

  int n_ref_format requested output format: REFDOCBK, REFDOCBKX, REFTEIX

 ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
int finish_render_dbib(char** ptr_ref, size_t* ptr_ref_len, struct BIBCONNS* ptr_bibconns, struct bibinfo* ptr_biblio_info, char* table_name, struct CLIENT_REQUEST* ptr_clrequest, struct xmlindent* ptr_indent, int n_ref_format) {
  char *new_ref;
  char *sql_command;
  char *myjournal;
  char intext_string[] = "INTEXT";
  char *item_rangesep;
  const char *item_intextseq;
  const char *item_multi_id;
  const char *item_entry_id;
  const char *drivername;
  const char *author_concat;
  char *item_preceeding;
  char *item_following;
  char *item_citsep;
  const char *item_dbname;
  int sql_command_len;
  int nis_first;
  int ndo_range = 0; /* if 1, contract ranges of consecutive numeric in-text citations */
  int nis_range = 0; /* if 1, the current row is the endpoint of a range */
  int n_skip;
  int n_xrefloop_rowpos;
  int n_consec;
  int n_multiple_db = 0; /* if 1, conn allows multiple databases */
  int n_is_subseq = 0;
  int n_currpos;
  int n_status;
  int n_prev_inline = 0;
  short int n_pubyear;
  short int title_as_author;
  unsigned long long ncurr_sorted_pos;
  unsigned long long n_sorted_pos1 = 0;
  unsigned long long n_citation_pos;
  unsigned long long n_sorted_pos;
  unsigned long long n_orig_id;
  struct lilimem sentinel;
  struct mset* ptr_mset_sentinel;

  /* we need quite a few database queries here */
  dbi_result dbires;
  dbi_result dbires1;
  dbi_result dbires2;
  dbi_result dbires3;

  /* initialize linked lists and stacks */
  sentinel.ptr_mem = NULL;
  sentinel.ptr_next = NULL;
  sentinel.varname[0] = '\0';

  if ((ptr_mset_sentinel = new_mset_stack()) == NULL) {
    return 801;
  }

  drivername = dbi_driver_get_name(dbi_conn_get_driver(ptr_bibconns->conn_refdb));

  sql_command_len = 4096;
  sql_command = malloc(sql_command_len); 
  if (!sql_command || insert_lilimem(&sentinel, (void**)&sql_command, "sql_command")) {
    clear_mset_stack(ptr_mset_sentinel);
    return 801;
  }

  myjournal = strdup(ptr_biblio_info->format_string);
  if (!myjournal || insert_lilimem(&sentinel, (void**)&myjournal, "myjournal")) {
    delete_all_lilimem(&sentinel);
    clear_mset_stack(ptr_mset_sentinel);
    return 801;
  }

  if (dbi_conn_quote_string(ptr_bibconns->conn_refdb, &myjournal) == 0) {
    delete_all_lilimem(&sentinel);
    clear_mset_stack(ptr_mset_sentinel);
    return 801;
  }

  /* get basic formatting info */
  if (!strcmp(my_dbi_conn_get_cap(ptr_bibconns->conn_refdb, "multiple_db"), "t")) {
    n_multiple_db = 1;
    sprintf(sql_command, "SELECT ID, CITSEPARATOR, FOLLOWING, PRECEEDING, RANGESEPARATOR, INTEXTSEQUENCE, STYLE from %s.CITSTYLE where JOURNAL=%s", main_db, myjournal);
  }
  else {
    sprintf(sql_command, "SELECT ID, CITSEPARATOR, FOLLOWING, PRECEEDING, RANGESEPARATOR, INTEXTSEQUENCE, STYLE from CITSTYLE where JOURNAL=%s", myjournal);
  }

  dbires = dbi_conn_query(ptr_bibconns->conn_refdb, sql_command);
  LOG_PRINT(LOG_DEBUG, sql_command);
  if (!dbires || dbi_result_next_row(dbires) == 0) {
    delete_all_lilimem(&sentinel);
    clear_mset_stack(ptr_mset_sentinel);
    return 241;
  }

  /* see whether we should contract consecutive numerical citations */
  item_rangesep = my_dbi_result_get_string_copy(dbires, "RANGESEPARATOR");
  if (item_rangesep && my_dbi_conn_error_flag(ptr_bibconns->conn_refdb)) {
    if (item_rangesep) {
      free(item_rangesep);
    }
    item_rangesep = NULL;
  }
  if (item_rangesep && insert_lilimem(&sentinel, (void**)&item_rangesep, "item_rangesep")) {
    delete_all_lilimem(&sentinel);
    clear_mset_stack(ptr_mset_sentinel);
    return 801;
  }

  if (item_rangesep && sgml_entitize(&item_rangesep, n_ref_format) == NULL) {
    delete_all_lilimem(&sentinel);
    clear_mset_stack(ptr_mset_sentinel);
    return 801;
  }
      
  if (item_rangesep && *item_rangesep) {
    ndo_range = 1;
  }

  item_intextseq = dbi_result_get_string(dbires, "INTEXTSEQUENCE");
  if (item_intextseq && my_dbi_conn_error_flag(ptr_bibconns->conn_refdb)) {
    item_intextseq = NULL;
  }

  item_preceeding = my_dbi_result_get_string_copy(dbires, "PRECEEDING");
  if (item_preceeding && my_dbi_conn_error_flag(ptr_bibconns->conn_refdb)) {
    if (item_preceeding) {
      free(item_preceeding);
    }
    item_preceeding = NULL;
  }
  if (item_preceeding && insert_lilimem(&sentinel, (void**)&item_preceeding, "item_preceeding")) {
    delete_all_lilimem(&sentinel);
    clear_mset_stack(ptr_mset_sentinel);
    return 801;
  }

  if (item_preceeding && sgml_entitize(&item_preceeding, n_ref_format) == NULL) {
    delete_all_lilimem(&sentinel);
    clear_mset_stack(ptr_mset_sentinel);
    return 801;
  }
      
  item_following = my_dbi_result_get_string_copy(dbires, "FOLLOWING");
  if (item_following && my_dbi_conn_error_flag(ptr_bibconns->conn_refdb)) {
    if (item_following) {
      free(item_following);
    }
    item_following = NULL;
  }
  if (item_following && insert_lilimem(&sentinel, (void**)&item_following, "item_following")) {
    delete_all_lilimem(&sentinel);
    clear_mset_stack(ptr_mset_sentinel);
    return 801;
  }

  if (item_following && sgml_entitize(&item_following, n_ref_format) == NULL) {
    delete_all_lilimem(&sentinel);
    clear_mset_stack(ptr_mset_sentinel);
    return 801;
  }
      
  item_citsep = my_dbi_result_get_string_copy(dbires, "CITSEPARATOR");
  if (item_citsep && my_dbi_conn_error_flag(ptr_bibconns->conn_refdb)) {
    if (item_citsep) {
      free(item_citsep);
    }
    item_citsep = NULL;
  }
  if (item_citsep && insert_lilimem(&sentinel, (void**)&item_citsep, "item_citsep")) {
    delete_all_lilimem(&sentinel);
    clear_mset_stack(ptr_mset_sentinel);
    return 801;
  }

  if (item_citsep && sgml_entitize(&item_citsep, n_ref_format) == NULL) {
    delete_all_lilimem(&sentinel);
    clear_mset_stack(ptr_mset_sentinel);
    return 801;
  }
      
  /* select all datasets in temp table with an xref_pos == 1. This gives us a list of all citation_pos with multiple citations. Walk through this list and create a bibliomset element for each citation, containing all xrefs of that citation. Use the sorted_pos column for numeric citations */
  sprintf(sql_command, "SELECT DISTINCT dbname, orig_id, author_concat, pubyear, article_title, periodical, volume, issue, startpage, entry_id, citation_pos, xref_pos, multi_id, sorted_pos, title_as_author, citekey FROM %s WHERE xref_pos=1 ORDER BY citation_pos", table_name);
  
  LOG_PRINT(LOG_DEBUG, sql_command);
  dbires1 = dbi_conn_query(ptr_bibconns->conn, sql_command);
  if (!dbires1) {
    delete_all_lilimem(&sentinel);
    dbi_result_free(dbires);
    clear_mset_stack(ptr_mset_sentinel);
    return 234;
  }

  if (n_ref_format == REFTEIX5) {
    if (print_elstart_x(ptr_ref, ptr_ref_len, "bibl", "rend", "multixref", NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 0 /* not empty */, ptr_indent, ptr_clrequest->namespace) == NULL) {
      delete_all_lilimem(&sentinel);
      dbi_result_free(dbires);
      dbi_result_free(dbires1);
      clear_mset_stack(ptr_mset_sentinel);
      return 801;
    }
  }
  else {
    if (print_elstart_x(ptr_ref, ptr_ref_len, "bibliomixed", "role", "multixref", NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 0 /* not empty */, ptr_indent, ptr_clrequest->namespace) == NULL) {
      delete_all_lilimem(&sentinel);
      dbi_result_free(dbires);
      dbi_result_free(dbires1);
      clear_mset_stack(ptr_mset_sentinel);
      return 801;
    }
  }

  /* loop over all citation_pos with more than one xref */
  while (dbi_result_next_row(dbires1)) {
    nis_first = 1; /* reset runonce flag for each cycle */
    n_citation_pos = my_dbi_result_get_idval(dbires1, "citation_pos");

    /* request all xrefs of this citation and sort them according to the required format */
    if (strcmp(item_intextseq, "ASIS") == 0) {
      sprintf(sql_command, "SELECT DISTINCT dbname, orig_id, author_concat, pubyear, article_title, periodical, volume, issue, startpage, entry_id, citation_pos, xref_pos, multi_id, sorted_pos, year_uni_suffix, title_as_author FROM %s WHERE citation_pos="ULLSPEC" ORDER BY xref_pos", table_name, (unsigned long long)n_citation_pos);
    }
    else if (strcmp(item_intextseq, "AUTHORDATE") == 0) {
      sprintf(sql_command, "SELECT DISTINCT dbname, orig_id, author_concat, pubyear, article_title, periodical, volume, issue, startpage, entry_id, citation_pos, xref_pos, multi_id, sorted_pos, year_uni_suffix, title_as_author FROM %s WHERE citation_pos="ULLSPEC" ORDER BY author_concat, pubyear, periodical, volume, issue, startpage", table_name, (unsigned long long)n_citation_pos);
    }
    else if (strcmp(item_intextseq, "CITEKEY") == 0) {
      sprintf(sql_command, "SELECT DISTINCT dbname, orig_id, author_concat, pubyear, article_title, periodical, volume, issue, startpage, entry_id, citation_pos, xref_pos, multi_id, sorted_pos, year_uni_suffix, title_as_author, citekey FROM %s WHERE citation_pos="ULLSPEC" ORDER BY citekey, author_concat, pubyear, periodical, volume, issue, startpage", table_name, (unsigned long long)n_citation_pos);
    }
    else if (strcmp(item_intextseq, "DATEASC") == 0) {
      sprintf(sql_command, "SELECT DISTINCT dbname, orig_id, author_concat, pubyear, article_title, periodical, volume, issue, startpage, entry_id, citation_pos, xref_pos, multi_id, sorted_pos, year_uni_suffix, title_as_author FROM %s WHERE citation_pos="ULLSPEC" ORDER BY pubyear, author_concat, periodical, volume, issue, startpage", table_name, (unsigned long long)n_citation_pos);
    }
    else if (strcmp(item_intextseq, "DATEDESC") == 0) {
      sprintf(sql_command, "SELECT DISTINCT dbname, orig_id, author_concat, pubyear, article_title, periodical, volume, issue, startpage, entry_id, citation_pos, xref_pos, multi_id, sorted_pos, year_uni_suffix, title_as_author FROM %s WHERE citation_pos="ULLSPEC" ORDER BY pubyear DESC, author_concat, periodical, volume, issue, startpage", table_name, (unsigned long long)n_citation_pos);
    }

    LOG_PRINT(LOG_DEBUG, sql_command);
    dbires2 = dbi_conn_query(ptr_bibconns->conn, sql_command);
    if (!dbires2) {
      delete_all_lilimem(&sentinel);
      dbi_result_free(dbires);
      dbi_result_free(dbires1);
      clear_mset_stack(ptr_mset_sentinel);
      return 234;
    }
    
    n_xrefloop_rowpos = 0;
    /* now do the real formatting by pulling all xrefs for the current citation*/
    while (dbi_result_next_row(dbires2)) {
      item_multi_id = my_dbi_result_get_string_copy(dbires2, "multi_id");
      if (item_multi_id && my_dbi_conn_error_flag(ptr_bibconns->conn)) {
	item_multi_id = NULL;
      }
      else if (item_multi_id && insert_lilimem(&sentinel, (void**)&item_multi_id, "item_multi_id")) {
	item_multi_id = NULL;
      }

      item_entry_id = my_dbi_result_get_string_copy(dbires2, "entry_id");
      if (item_entry_id && my_dbi_conn_error_flag(ptr_bibconns->conn)) {
	item_entry_id = NULL;
      }
      else if (item_entry_id && insert_lilimem(&sentinel, (void**)&item_entry_id, "item_entry_id")) {
	item_entry_id = NULL;
      }

      item_dbname = dbi_result_get_string(dbires2, "dbname");
      if (item_dbname && my_dbi_conn_error_flag(ptr_bibconns->conn)) {
	item_dbname = NULL;
      }

      if (!item_multi_id || !item_entry_id || !item_dbname) {
	delete_all_lilimem(&sentinel);
	dbi_result_free(dbires);
	dbi_result_free(dbires1);
	dbi_result_free(dbires2);
	clear_mset_stack(ptr_mset_sentinel);
	return 234;
      }

      n_sorted_pos = my_dbi_result_get_idval(dbires2, "sorted_pos");
      if (n_sorted_pos == 0 && my_dbi_conn_error_flag(ptr_bibconns->conn)) {
	delete_all_lilimem(&sentinel);
	dbi_result_free(dbires);
	dbi_result_free(dbires1);
	dbi_result_free(dbires2);
	clear_mset_stack(ptr_mset_sentinel);
	return 234;
      }

      n_orig_id = my_dbi_result_get_idval(dbires2, "orig_id");
      if (n_orig_id == 0 && my_dbi_conn_error_flag(ptr_bibconns->conn)) {
	delete_all_lilimem(&sentinel);
	dbi_result_free(dbires);
	dbi_result_free(dbires1);
	dbi_result_free(dbires2);
	clear_mset_stack(ptr_mset_sentinel);
	return 234;
      }

      ptr_biblio_info->year_unique_suffix = dbi_result_get_string(dbires2, "year_uni_suffix");


      /* start with the code that appears in front of all xrefs */
      if (nis_first) {
	nis_first = 0; /* run only once */

	/* the additional information will be encoded in a bibliomset element. relation=endtermtarget is a label to indicate the processing app that this should not be rendered directly */
	if (n_ref_format == REFDOCBKX5) {
	  if (print_elstart_x(ptr_ref, ptr_ref_len, "bibliomset", "relation", "endtermtarget", "xml:id", item_multi_id, NULL, NULL, NULL, NULL, NULL, NULL, 0 /* not empty */, ptr_indent, ptr_clrequest->namespace) == NULL) {
	    delete_all_lilimem(&sentinel);
	    dbi_result_free(dbires);
	    dbi_result_free(dbires1);
	    dbi_result_free(dbires2);
	    clear_mset_stack(ptr_mset_sentinel);
	    return 801;
	  }
	}
	else if (n_ref_format == REFTEIX5) {
	  if (print_elstart_x(ptr_ref, ptr_ref_len, "seg", "type", "endtermtarget", "xml:id", item_multi_id, NULL, NULL, NULL, NULL, NULL, NULL, 0 /* not empty */, ptr_indent, ptr_clrequest->namespace) == NULL) {
	    delete_all_lilimem(&sentinel);
	    dbi_result_free(dbires);
	    dbi_result_free(dbires1);
	    dbi_result_free(dbires2);
	    clear_mset_stack(ptr_mset_sentinel);
	    return 801;
	  }
	}
	else {
	  if (print_elstart_x(ptr_ref, ptr_ref_len, "bibliomset", "relation", "endtermtarget", "id", item_multi_id, NULL, NULL, NULL, NULL, NULL, NULL, 0 /* not empty */, ptr_indent, ptr_clrequest->namespace) == NULL) {
	    delete_all_lilimem(&sentinel);
	    dbi_result_free(dbires);
	    dbi_result_free(dbires1);
	    dbi_result_free(dbires2);
	    clear_mset_stack(ptr_mset_sentinel);
	    return 801;
	  }
	}

	n_prev_inline = inline_xmlindent(ptr_indent, 1);

	/* format the citations as the value of the bibliomset element. The in-text citation can retrieve this string with the endterm attribute */
	/* preceeding */
	if (item_preceeding) {
	  if ((new_ref = mstrcat(*ptr_ref, item_preceeding, ptr_ref_len, 0)) == NULL) {
	    delete_all_lilimem(&sentinel);
	    dbi_result_free(dbires);
	    dbi_result_free(dbires1);
	    dbi_result_free(dbires2);
	    clear_mset_stack(ptr_mset_sentinel);
	    return 234;
	  }
	  else {
	    *ptr_ref = new_ref;
	  }
	}
      }
      else { /* if not first cycle in this loop */
	if (nis_range) {
/*  	  printf("print range\n"); */
	  /* range separator */
	  if ((new_ref = mstrcat(*ptr_ref, item_rangesep, ptr_ref_len, 0)) == NULL) {
	    delete_all_lilimem(&sentinel);
	    dbi_result_free(dbires);
	    dbi_result_free(dbires1);
	    dbi_result_free(dbires2);
	    clear_mset_stack(ptr_mset_sentinel);
	    return 801;
	  }
	  else {
	    *ptr_ref = new_ref;
	  }
	  nis_range = 0; /* reset */
	}
	else if (item_citsep) {
	  /* citation separator */
	  if ((new_ref = mstrcat(*ptr_ref, item_citsep, ptr_ref_len, 0)) == NULL) {
	    delete_all_lilimem(&sentinel);
	    dbi_result_free(dbires);
	    dbi_result_free(dbires1);
	    dbi_result_free(dbires2);
	    clear_mset_stack(ptr_mset_sentinel);
	    return 801;
	  }
	  else {
	    *ptr_ref = new_ref;
	  }
	}
      }

      /* retrieve dataset */
  /* 0  refdb_id             ID  - (if numeric)
     1  refdb_type           TY  - 
     2  refdb_pubyear        PY  - (partial)
     3  refdb_startpage      SP  - 
     4  refdb_endpage        EP  - 
     5  refdb_abstract       N2  - 
     6  refdb_title          TI  - 
     7  refdb_volume         VL  - 
     8  refdb_issue          CP  - 
     9  refdb_booktitle      BT  - 
    10  refdb_city           CY  - 
    11  refdb_publisher      PB  - 
    12  refdb_title_series   T3  - 
    13  refdb_address        AD  - 
    14  refdb_issn           SN  - 
    15  refdb_periodical_id  JO  - (indirect)
    16  refdb_pyother_info   PY  - (partial)
    17  refdb_secyear        Y2  - (partial)
    18  refdb_secother_info  Y2  - (partial)
    19  refdb_user1          U1  - 
    20  refdb_user2          U2  - 
    21  refdb_user3          U3  - 
    22  refdb_user4          U4  - 
    23  refdb_user5          U5  - 
    24  refdb_typeofwork     ..  - 
    25  refdb_area           ..  - 
    26  refdb_ostype         ..  - 
    27  refdb_degree         ..  - 
    28  refdb_runningtime    ..  - 
    29  refdb_classcodeintl  ..  - 
    30  refdb_classcodeus    ..  - 
    31  refdb_senderemail    ..  - 
    32  refdb_recipientemail ..  - 
    33  refdb_mediatype      ..  - 
    34  refdb_numvolumes     ..  - 
    35  refdb_edition        ..  - 
    36  refdb_computer       ..  - 
    37  refdb_conferencelocation   ..  - 
    38  refdb_registrynum    ..  - 
    39  refdb_classification ..  - 
    40  refdb_section        ..  - 
    41  refdb_pamphletnum    ..  - 
    42  refdb_chapternum     ..  - 
    43  refdb_citekey        ID  - (if non-numeric)
*/

      if (n_multiple_db) {
	sprintf(sql_command, "SELECT %s.t_refdb.refdb_id, %s.t_refdb.refdb_type, %s.t_refdb.refdb_pubyear, %s.t_refdb.refdb_startpage, %s.t_refdb.refdb_endpage, %s.t_refdb.refdb_abstract, %s.t_refdb.refdb_title, %s.t_refdb.refdb_volume, %s.t_refdb.refdb_issue, %s.t_refdb.refdb_booktitle, %s.t_refdb.refdb_city, %s.t_refdb.refdb_publisher, %s.t_refdb.refdb_title_series, %s.t_refdb.refdb_address, %s.t_refdb.refdb_issn, %s.t_refdb.refdb_periodical_id, %s.t_refdb.refdb_pyother_info, %s.t_refdb.refdb_secyear, %s.t_refdb.refdb_secother_info, %s.t_refdb.refdb_user1, %s.t_refdb.refdb_user2, %s.t_refdb.refdb_user3, %s.t_refdb.refdb_user4, %s.t_refdb.refdb_user5, %s.t_refdb.refdb_typeofwork, %s.t_refdb.refdb_area, %s.t_refdb.refdb_ostype, %s.t_refdb.refdb_degree, %s.t_refdb.refdb_runningtime, %s.t_refdb.refdb_classcodeintl, %s.t_refdb.refdb_classcodeus, %s.t_refdb.refdb_senderemail, %s.t_refdb.refdb_recipientemail, %s.t_refdb.refdb_mediatype, %s.t_refdb.refdb_numvolumes, %s.t_refdb.refdb_edition, %s.t_refdb.refdb_computer, %s.t_refdb.refdb_conferencelocation, %s.t_refdb.refdb_registrynum, %s.t_refdb.refdb_classification, %s.t_refdb.refdb_section, %s.t_refdb.refdb_pamphletnum, %s.t_refdb.refdb_chapternum, %s.t_refdb.refdb_citekey FROM %s.t_refdb WHERE %s.t_refdb.refdb_id="ULLSPEC, item_dbname, item_dbname, item_dbname, item_dbname, item_dbname, item_dbname, item_dbname, item_dbname, item_dbname, item_dbname, item_dbname, item_dbname, item_dbname, item_dbname, item_dbname, item_dbname, item_dbname, item_dbname, item_dbname, item_dbname, item_dbname, item_dbname, item_dbname, item_dbname, item_dbname, item_dbname, item_dbname, item_dbname, item_dbname, item_dbname, item_dbname, item_dbname, item_dbname, item_dbname, item_dbname, item_dbname, item_dbname, item_dbname, item_dbname, item_dbname, item_dbname, item_dbname, item_dbname, item_dbname, item_dbname, item_dbname, (unsigned long long)n_orig_id);
	ptr_bibconns->conn_source = ptr_bibconns->conn; /* make sure */
      }
      else {
	sprintf(sql_command, "SELECT refdb_id, refdb_type, refdb_pubyear, refdb_startpage, refdb_endpage, refdb_abstract, refdb_title, refdb_volume, refdb_issue, refdb_booktitle, refdb_city, refdb_publisher, refdb_title_series, refdb_address, refdb_issn, refdb_periodical_id, refdb_pyother_info, refdb_secyear, refdb_secother_info, refdb_user1, refdb_user2, refdb_user3, refdb_user4, refdb_user5, refdb_typeofwork, refdb_area, refdb_ostype, refdb_degree, refdb_runningtime, refdb_classcodeintl, refdb_classcodeus, refdb_senderemail, refdb_recipientemail, refdb_mediatype, refdb_numvolumes, refdb_edition, refdb_computer, refdb_conferencelocation, refdb_registrynum, refdb_classification, refdb_section, refdb_pamphletnum, refdb_chapternum, refdb_citekey FROM t_refdb WHERE refdb_id="ULLSPEC, (unsigned long long)n_orig_id);
	/* need a connection to item_dbname */
	ptr_bibconns->conn_source = connect_to_db(ptr_clrequest, item_dbname, 0);
	if (!ptr_bibconns->conn_source) {
	  delete_all_lilimem(&sentinel);
	  dbi_result_free(dbires);
	  dbi_result_free(dbires1);
	  dbi_result_free(dbires2);
	  clear_mset_stack(ptr_mset_sentinel);
	  return 204;
	}
      }
	
      LOG_PRINT(LOG_DEBUG, sql_command);
      dbires3 = dbi_conn_query(ptr_bibconns->conn_source, sql_command);
      if (!dbires3 || dbi_result_next_row(dbires3) == 0) {
	delete_all_lilimem(&sentinel);
	dbi_result_free(dbires);
	dbi_result_free(dbires1);
	dbi_result_free(dbires2);
	clear_mset_stack(ptr_mset_sentinel);
	return 234;
      }
      
      /* item_entry_id checked for non-NULL above */
      ptr_biblio_info->entry_id = item_entry_id;
      ptr_biblio_info->n_refdb_id = my_dbi_result_get_idval(dbires3, "refdb_id");


      author_concat = my_dbi_result_get_string(dbires2, "author_concat");
      n_currpos = dbi_result_get_int(dbires2, "sorted_pos");
      n_pubyear = dbi_result_get_short(dbires2, "pubyear");

      n_is_subseq = check_is_subseq(author_concat, table_name, n_currpos, n_pubyear, ptr_bibconns->conn);

      if (n_is_subseq == -1) {
	delete_all_lilimem(&sentinel);
	dbi_result_free(dbires);
	dbi_result_free(dbires1);
	dbi_result_free(dbires2);
	dbi_result_free(dbires3);
	clear_mset_stack(ptr_mset_sentinel);
	return 234;
      }

      ptr_biblio_info->is_subseq = n_is_subseq;


      title_as_author = dbi_result_get_short(dbires2, "title_as_author");


      /* if we are supposed to display consecutive sequences of numerical citations as a numerical range, we have to format the first in a possible series and then look ahead if at least two more citations are consecutive (as defined by their sorted order in the bibliography) */
      if (ndo_range) {
	if ((n_status = format_entry(ptr_ref, ptr_ref_len, intext_string, ptr_bibconns, dbires, dbires3, item_dbname, ptr_clrequest->username, n_sorted_pos, ptr_mset_sentinel, ptr_biblio_info, title_as_author, ptr_clrequest->namespace, ptr_indent, n_ref_format, ptr_clrequest->pdfroot)) != 0) {
	  delete_all_lilimem(&sentinel);
	  dbi_result_free(dbires);
	  dbi_result_free(dbires1);
	  dbi_result_free(dbires2);
	  dbi_result_free(dbires3);
	  clear_mset_stack(ptr_mset_sentinel);
	  return n_status;
	}

	ncurr_sorted_pos = n_sorted_pos;
	n_skip = 0;

	/* look ahead if this is the start of consecutive numbers */
	do {
	  n_consec = dbi_result_next_row(dbires2);
	  if (n_consec != 0) {
	    n_sorted_pos1 = my_dbi_result_get_idval(dbires2, "sorted_pos");
	    if (my_dbi_conn_error_flag(ptr_bibconns->conn)) {
	      /* ToDo: drop out here? */
	      LOG_PRINT(LOG_WARNING, "temp table access error: sorted_pos unknown");
	    }
	  }
	  ncurr_sorted_pos++;
	  n_skip++;
	} while (n_consec != 0 && n_sorted_pos1 == ncurr_sorted_pos);

	/* several possibilities: */
	/* n_skip == 1: either no more entries or not a consecutive number */
	/* n_skip == 2: two consecutive numbers, third one does not
	   exist or is not consecutive */
	/* n_skip > 2: at least three consecutive numbers */

	/* either no consecutive numbers or only two. Two numbers are
	   not displayed as a range */
	if (n_skip < 3) {
	  /* no n_xrefloop_rowpos correction here, as it will be
	     incremented by one anyway at the end of the loop */
	  dbi_result_seek_row(dbires2, n_xrefloop_rowpos + 1);
	}

	/* at least three consecutive numbers. The next reference to
	   be formatted is the last one of the range. Adjust
	   n_xrefloop_rowpos accordingly */
	else { 
	  dbi_result_seek_row(dbires2, n_xrefloop_rowpos + n_skip - 1);
	  /* this will be incremented by one later */
	  n_xrefloop_rowpos += n_skip - 2;
	  nis_range = 1;
	}
      }

      /* don't use ranges. We have to format each entry */
      else {
	if ((n_status = format_entry(ptr_ref, ptr_ref_len, intext_string, ptr_bibconns, dbires, dbires3, item_dbname, ptr_clrequest->username, n_sorted_pos, ptr_mset_sentinel, ptr_biblio_info, title_as_author, ptr_clrequest->namespace, ptr_indent, n_ref_format, ptr_clrequest->pdfroot)) != 0) {
	  delete_all_lilimem(&sentinel);
	  dbi_result_free(dbires);
	  dbi_result_free(dbires1);
	  dbi_result_free(dbires2);
	  dbi_result_free(dbires3);
	  clear_mset_stack(ptr_mset_sentinel);
	  return n_status;
	}
      } /* end if use ranges */

      dbi_result_free(dbires3);
      n_xrefloop_rowpos++;

      delete_lilimem(&sentinel, "item_entry_id");
      delete_lilimem(&sentinel, "item_multi_id");

      if (!n_multiple_db) {
	dbi_conn_close(ptr_bibconns->conn_source);
	ptr_bibconns->conn_source = NULL;
      }
    } /* end loop over all xrefs of this citation */

    if ((new_ref = set_mset(ptr_mset_sentinel, "", ptr_ref, ptr_ref_len, ptr_clrequest->namespace, ptr_indent, n_ref_format)) == NULL) {
      delete_all_lilimem(&sentinel);
      dbi_result_free(dbires);
      dbi_result_free(dbires1);
      dbi_result_free(dbires2);
      clear_mset_stack(ptr_mset_sentinel);
      return 801;
    }
    else {
      *ptr_ref = new_ref;
    }

    /* following */
    if (item_following) {
      if ((new_ref = mstrcat(*ptr_ref, item_following, ptr_ref_len, 0)) == NULL) {
	delete_all_lilimem(&sentinel);
	dbi_result_free(dbires);
	dbi_result_free(dbires1);
	dbi_result_free(dbires2);
	clear_mset_stack(ptr_mset_sentinel);
	return 801;
      }
      else {
	*ptr_ref = new_ref;
      }
    }

    if (n_ref_format == REFTEIX5) {
      if (print_elend_x(ptr_ref, ptr_ref_len, "seg", ptr_indent, ptr_clrequest->namespace) == NULL) {
	delete_all_lilimem(&sentinel);
	dbi_result_free(dbires);
	dbi_result_free(dbires1);
	dbi_result_free(dbires2);
	clear_mset_stack(ptr_mset_sentinel);
	return 801;
      }
    }
    else {
      if (print_elend_x(ptr_ref, ptr_ref_len, "bibliomset", ptr_indent, ptr_clrequest->namespace) == NULL) {
	delete_all_lilimem(&sentinel);
	dbi_result_free(dbires);
	dbi_result_free(dbires1);
	dbi_result_free(dbires2);
	clear_mset_stack(ptr_mset_sentinel);
	return 801;
      }
    }

    dbi_result_free(dbires2);

    inline_xmlindent(ptr_indent, n_prev_inline);
  } /* end loop over all citations */

  dbi_result_free(dbires);
  dbi_result_free(dbires1);
  delete_all_lilimem(&sentinel);

  if (n_ref_format == REFTEIX5) {
    if (print_elend_x(ptr_ref, ptr_ref_len, "bibl", ptr_indent, ptr_clrequest->namespace) == NULL) {
      clear_mset_stack(ptr_mset_sentinel);
      return 801;
    }

    if (print_elend_x(ptr_ref, ptr_ref_len, "listBibl", ptr_indent, ptr_clrequest->namespace) == NULL) {
      clear_mset_stack(ptr_mset_sentinel);
      return 801;
    }

    if (print_elend_x(ptr_ref, ptr_ref_len, "div", ptr_indent, ptr_clrequest->namespace) == NULL) {
      clear_mset_stack(ptr_mset_sentinel);
      return 801;
    }
  }
  else {
    if (print_elend_x(ptr_ref, ptr_ref_len, "bibliomixed", ptr_indent, ptr_clrequest->namespace) == NULL) {
      clear_mset_stack(ptr_mset_sentinel);
      return 801;
    }

    if (print_elend_x(ptr_ref, ptr_ref_len, "bibliography", ptr_indent, ptr_clrequest->namespace) == NULL) {
      clear_mset_stack(ptr_mset_sentinel);
      return 801;
    }
  }

  clear_mset_stack(ptr_mset_sentinel);

  return 0;
}

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  render_dbib() renders a RIS dataset for DocBook V.3.1 bibliography

  int render_dbib returns 0 if successful, >0 if failed

  char** ptr_ref ptr to a ptr to a buffer created with malloc which will receive
        the output. The calling function must allocate the buffer
        with at least 4096 byte. This function will reallocate the
        buffer as needed. *ptr will be updated whenever a realloc is
        necessary. The calling function is responsible for freeing the
        memory again.

  size_t* ptr_ref_len pointer to a variable holding the current size
                    of ref. Will be updated accordingly.

  struct BIBCONNS* ptr_bibconns ptr to structure with database connections

  dbi_result dbires pointer to a dbi result structure holding the RIS dataset

  const char* database ptr to a string with the database name of the current
                       query

  struct bibinfo* ptr_biblio_info ptr to struct with additional bibliographic
                  info

  char* username ptr to string with the current username

  char* pdfroot ptr to string with the root path for PDF offprints

  int nref_counter counter for datasets, used for numeric citation
                  style and if the bibliography needs to be numbered

  short title_as_author if yes, the reference has no authors and the style
                  wants the title to be printed instead

  const char* ns optional namespace prefix

  struct xmlindent* ptr_indent ptr to indentation info

  int n_ref_format requested output format: REFDOCBK, REFDOCBKX, REFTEIX

  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
int render_dbib(char** ptr_ref, size_t* ptr_ref_len, struct BIBCONNS* ptr_bibconns, dbi_result dbires, const char* database, struct bibinfo* ptr_biblio_info, char* username, char* pdfroot, int nref_counter, short title_as_author, const char* ns, struct xmlindent* ptr_indent, int n_ref_format) {
  int i; /* the tried-and-true counter */
  int sql_command_len;
  int n_status;
  int n_prev_inline;
  char *new_ref; /* used for reallocating buffer ref */
  char *sql_command; /* buffer for assembling SQL queries */
  char *buffer;
  char *myjournal;
  char *item_preceeding;
  char *item_following;
  const char *drivername;
  char empty_string[1] = "";
  char bmset_buffer[256] = "";
  char id_string[7];
  char format_spec[5][12] = {"INTEXT",
			     "INTEXTS",
			     "AUTHORONLY",
			     "AUTHORONLYS",
			     "YEARONLY"};

  char bibliomset_endstring[6][5] = {"-X",
				     "-S",
				     "-A",
				     "-Q",
				     "-Y"};

  char role_string[5][13] = {"intext",
			     "intextsq",
			     "authoronly",
			     "authoronlysq",
			     "yearonly"};

  dbi_result dbires_cit;
/*   dbi_conn conn; */

  struct lilimem sentinel;
  struct mset* ptr_mset_sentinel;

  switch (n_ref_format) {
  case REFDOCBKX5:
  case REFTEIX5:
    strcpy(id_string, "xml:id");
    break;
  default:
    strcpy(id_string, "id");
    break;
  }

  sentinel.ptr_mem = NULL;
  sentinel.ptr_next = NULL;
  sentinel.varname[0] = '\0';

  if ((ptr_mset_sentinel = new_mset_stack()) == NULL) {
    return 801;
  }

/*    printf("now in render_dbib for %s\n", ptr_biblio_info->entry_id); */

  drivername = dbi_driver_get_name(dbi_conn_get_driver(ptr_bibconns->conn_refdb));

  /* get memory for sql command buffer */
  sql_command_len = 1024;
  sql_command = malloc(sql_command_len);
  if (!sql_command || insert_lilimem(&sentinel, (void**)&sql_command, "sql_command")) {
    clear_mset_stack(ptr_mset_sentinel);
    return 801;
  }

  /* ToDo: check for string overflow */

  myjournal = strdup(ptr_biblio_info->format_string);
  if (!myjournal || insert_lilimem(&sentinel, (void**)&myjournal, "myjournal")) {
    delete_all_lilimem(&sentinel);
    clear_mset_stack(ptr_mset_sentinel);
    return 801;
  }

  if (dbi_conn_quote_string(ptr_bibconns->conn_refdb, &myjournal) == 0) {
    delete_all_lilimem(&sentinel);
    clear_mset_stack(ptr_mset_sentinel);
    return 801;
  }

  /* retrieve basic information about the bibliography style from CITSTYLE into ptr_result */
  if (!strcmp(my_dbi_conn_get_cap(ptr_bibconns->conn_refdb, "multiple_db"), "t")) {
    sprintf(sql_command, "SELECT ID, CITSEPARATOR, FOLLOWING, PRECEEDING, RANGESEPARATOR, INTEXTSEQUENCE, STYLE, JANFULL, FEBFULL, MARFULL, APRFULL, MAYFULL, JUNFULL, JULFULL, AUGFULL, SEPFULL, OCTFULL, NOVFULL, DECFULL, JANABBREV, FEBABBREV, MARABBREV, APRABBREV, MAYABBREV, JUNABBREV, JULABBREV, AUGABBREV, SEPABBREV, OCTABBREV, NOVABBREV, DECABBREV, JANTHREELET, FEBTHREELET, MARTHREELET, APRTHREELET, MAYTHREELET, JUNTHREELET, JULTHREELET, AUGTHREELET, SEPTHREELET, OCTTHREELET, NOVTHREELET, DECTHREELET from %s.CITSTYLE where JOURNAL=%s", main_db, myjournal);
  }
  else {
    sprintf(sql_command, "SELECT ID, CITSEPARATOR, FOLLOWING, PRECEEDING, RANGESEPARATOR, INTEXTSEQUENCE, STYLE, JANFULL, FEBFULL, MARFULL, APRFULL, MAYFULL, JUNFULL, JULFULL, AUGFULL, SEPFULL, OCTFULL, NOVFULL, DECFULL, JANABBREV, FEBABBREV, MARABBREV, APRABBREV, MAYABBREV, JUNABBREV, JULABBREV, AUGABBREV, SEPABBREV, OCTABBREV, NOVABBREV, DECABBREV, JANTHREELET, FEBTHREELET, MARTHREELET, APRTHREELET, MAYTHREELET, JUNTHREELET, JULTHREELET, AUGTHREELET, SEPTHREELET, OCTTHREELET, NOVTHREELET, DECTHREELET from CITSTYLE where JOURNAL=%s", myjournal);
  }

  LOG_PRINT(LOG_DEBUG, sql_command);
  dbires_cit = dbi_conn_query(ptr_bibconns->conn_refdb, sql_command);
  if (!dbires_cit || dbi_result_next_row(dbires_cit) == 0) {
    delete_all_lilimem(&sentinel);
    clear_mset_stack(ptr_mset_sentinel);
    return 234;
  }
  
  item_preceeding = my_dbi_result_get_string_copy(dbires_cit, "PRECEEDING");
  if (item_preceeding && my_dbi_conn_error_flag(ptr_bibconns->conn_refdb)) {
    if (item_preceeding) {
      free(item_preceeding);
    }
    item_preceeding = NULL;
  }
  if (item_preceeding && insert_lilimem(&sentinel, (void**)&item_preceeding, "item_preceeding")) {
    delete_all_lilimem(&sentinel);
    clear_mset_stack(ptr_mset_sentinel);
    return 801;
  }

  item_following = my_dbi_result_get_string_copy(dbires_cit, "FOLLOWING");
  if (item_following && my_dbi_conn_error_flag(ptr_bibconns->conn_refdb)) {
    if (item_following) {
      free(item_following);
    }
    item_following = NULL;
  }
  if (item_following && insert_lilimem(&sentinel, (void**)&item_following, "item_following")) {
    delete_all_lilimem(&sentinel);
    clear_mset_stack(ptr_mset_sentinel);
    return 801;
  }

  /* find out reference type */
  buffer = (char*)get_refdb_type(dbires);
  if (!buffer) {
    delete_all_lilimem(&sentinel);
    dbi_result_free(dbires_cit);
    clear_mset_stack(ptr_mset_sentinel);
    return 244;
  }

  /* The id attribute of bibliomixed is used to match the bibliography
  entry with the citation(s) in the text. The role attribute of
  bibliomixed is used by the stylesheet to pick the proper formatting
  rules */
  if (n_ref_format == REFTEIX5) {
    if (print_elstart_x(ptr_ref, ptr_ref_len, "bibl", id_string, ptr_biblio_info->entry_id, "rend", buffer, NULL, NULL, NULL, NULL, NULL, NULL, 0 /* not empty */, ptr_indent, ns) == NULL) {
      delete_all_lilimem(&sentinel);
      dbi_result_free(dbires_cit);
      clear_mset_stack(ptr_mset_sentinel);
      return 244;
    }
  }
  else {
    if (print_elstart_x(ptr_ref, ptr_ref_len, "bibliomixed", id_string, ptr_biblio_info->entry_id, "role", buffer, NULL, NULL, NULL, NULL, NULL, NULL, 0 /* not empty */, ptr_indent, ns) == NULL) {
      delete_all_lilimem(&sentinel);
      dbi_result_free(dbires_cit);
      clear_mset_stack(ptr_mset_sentinel);
      return 244;
    }
  }    
  

  /* this loop creates a couple of bibliomset elements which can be targeted
     from the SGML document by link elements. They provide:
     - id="IDxyz:X": contains the standard in-text citation
     - id="IDxyz:S": contains the in-text citation for subsequent citations
     - id="IDxyz:A": contains the standard author-only in-text citation
     - id="IDxyz:Q": contains the author-only in-text citation (subsequent)
     - id="IDxyz:Y": contains the year-only in-text citation */

/*   printf("here intext contents start for %s\n", ptr_biblio_info->entry_id); */
  for (i = 0; i < 5; i++) {
    sprintf(bmset_buffer, "%s%s", ptr_biblio_info->entry_id, bibliomset_endstring[i]);

    if (n_ref_format == REFTEIX5) {
      if (print_elstart_x(ptr_ref, ptr_ref_len, "seg", "type", role_string[i], id_string, bmset_buffer, NULL, NULL, NULL, NULL, NULL, NULL, 0 /* not empty */, ptr_indent, ns) == NULL) {
	delete_all_lilimem(&sentinel);
	dbi_result_free(dbires_cit);
	clear_mset_stack(ptr_mset_sentinel);
	return 801;
      }
    }
    else {
      if (print_elstart_x(ptr_ref, ptr_ref_len, "bibliomset", "role", role_string[i], id_string, bmset_buffer, NULL, NULL, NULL, NULL, NULL, NULL, 0 /* not empty */, ptr_indent, ns) == NULL) {
	delete_all_lilimem(&sentinel);
	dbi_result_free(dbires_cit);
	clear_mset_stack(ptr_mset_sentinel);
	return 801;
      }
    }

    n_prev_inline = inline_xmlindent(ptr_indent, 1);

    if (i < 2 || i > 3) { /* all but the author-onlies */
      if (item_preceeding && *item_preceeding) { /* CITSTYLE PRECEEDING */
	if ((new_ref = mstrcat(*ptr_ref, (char*)item_preceeding, ptr_ref_len, 0)) == NULL) {
	  delete_all_lilimem(&sentinel);
	  dbi_result_free(dbires_cit);
	  clear_mset_stack(ptr_mset_sentinel);
	  return 801;
	}
	else {
	  *ptr_ref = new_ref;
	}
      }
    }

    if ((n_status = format_entry(ptr_ref, ptr_ref_len, format_spec[i], ptr_bibconns, dbires_cit, dbires, database, username, nref_counter, ptr_mset_sentinel, ptr_biblio_info, title_as_author, ns, ptr_indent, n_ref_format, pdfroot)) != 0) { 
      dbi_result_free(dbires_cit);
      delete_all_lilimem(&sentinel);
      clear_mset_stack(ptr_mset_sentinel);
      return n_status;
    }

    if ((new_ref = set_mset(ptr_mset_sentinel, empty_string, ptr_ref, ptr_ref_len, ns, ptr_indent, n_ref_format)) == NULL) {
      dbi_result_free(dbires_cit);
      delete_all_lilimem(&sentinel);
      clear_mset_stack(ptr_mset_sentinel);
      return 801;
    }
    else {
      *ptr_ref = new_ref;
    }


    if (i < 2 || i > 3) {
      if (item_following && *(item_following)) { /* CITSTYLE FOLLOWING */
	if (sgml_entitize(&item_following, n_ref_format) == NULL) {
	  delete_all_lilimem(&sentinel);
	  dbi_result_free(dbires_cit);
	  clear_mset_stack(ptr_mset_sentinel);
	  return 801;
	}
      
	if ((new_ref = mstrcat(*ptr_ref, item_following, ptr_ref_len, 0)) == NULL) {
	  dbi_result_free(dbires_cit);
	  delete_all_lilimem(&sentinel);
	  clear_mset_stack(ptr_mset_sentinel);
	  return 801;
	}
	else {
	  *ptr_ref = new_ref;
	}
      }
    }

    if (n_ref_format == REFTEIX5) {
      if (print_elend_x(ptr_ref, ptr_ref_len, "seg", ptr_indent, ns) == NULL) {
	dbi_result_free(dbires_cit);
	delete_all_lilimem(&sentinel);
	clear_mset_stack(ptr_mset_sentinel);
	return 801;
      }
    }
    else {
      if (print_elend_x(ptr_ref, ptr_ref_len, "bibliomset", ptr_indent, ns) == NULL) {
	dbi_result_free(dbires_cit);
	delete_all_lilimem(&sentinel);
	clear_mset_stack(ptr_mset_sentinel);
	return 801;
      }
    }
    inline_xmlindent(ptr_indent, n_prev_inline);

  } /* end for */
  

/*   printf("here the real bibliography contents start for %s\n", ptr_biblio_info->entry_id); */

  /* here the real bibliography contents start */
  sprintf(bmset_buffer, "%s-b", ptr_biblio_info->entry_id);
  if (n_ref_format == REFTEIX5) {
    if (print_elstart_x(ptr_ref, ptr_ref_len, "seg", "type", "bibliography", id_string, bmset_buffer, NULL, NULL, NULL, NULL, NULL, NULL, 0 /* not empty */, ptr_indent, ns) == NULL) {
      dbi_result_free(dbires_cit);
      delete_all_lilimem(&sentinel);
      clear_mset_stack(ptr_mset_sentinel);
      return 801;
    }
  }
  else {
    if (print_elstart_x(ptr_ref, ptr_ref_len, "bibliomset", "role", "bibliography", id_string, bmset_buffer, NULL, NULL, NULL, NULL, NULL, NULL, 0 /* not empty */, ptr_indent, ns) == NULL) {
      dbi_result_free(dbires_cit);
      delete_all_lilimem(&sentinel);
      clear_mset_stack(ptr_mset_sentinel);
      return 801;
    }
  }

  n_prev_inline = inline_xmlindent(ptr_indent, 1);

  if ((n_status = format_entry(ptr_ref, ptr_ref_len, buffer, ptr_bibconns, dbires_cit, dbires, database, username, nref_counter, ptr_mset_sentinel, ptr_biblio_info, title_as_author, ns, ptr_indent, n_ref_format, pdfroot)) != 0) { /* format bibliography entry body according to PUBTYPE */
    dbi_result_free(dbires_cit);
    delete_all_lilimem(&sentinel);
    clear_mset_stack(ptr_mset_sentinel);
    return n_status;
  }

  if ((new_ref = set_mset(ptr_mset_sentinel, empty_string, ptr_ref, ptr_ref_len, ns, ptr_indent, n_ref_format)) == NULL) {
    dbi_result_free(dbires_cit);
    delete_all_lilimem(&sentinel);
    clear_mset_stack(ptr_mset_sentinel);
    return 801;
  }
  else {
    *ptr_ref = new_ref;
  }

  if ((new_ref = set_mset(ptr_mset_sentinel, empty_string, ptr_ref, ptr_ref_len, ns, ptr_indent, n_ref_format)) == NULL) {
    dbi_result_free(dbires_cit);
    delete_all_lilimem(&sentinel);
    clear_mset_stack(ptr_mset_sentinel);
    return 801;
  }
  else {
    *ptr_ref = new_ref;
  }

  if (n_ref_format == REFTEIX5) {
    if (print_elend_x(ptr_ref, ptr_ref_len, "seg", ptr_indent, ns) == NULL) {
      dbi_result_free(dbires_cit);
      delete_all_lilimem(&sentinel);
      clear_mset_stack(ptr_mset_sentinel);
      return 801;
    }
  }
  else {
    if (print_elend_x(ptr_ref, ptr_ref_len, "bibliomset", ptr_indent, ns) == NULL) {
      dbi_result_free(dbires_cit);
      delete_all_lilimem(&sentinel);
      clear_mset_stack(ptr_mset_sentinel);
      return 801;
    }
  }

  inline_xmlindent(ptr_indent, n_prev_inline);

  if (n_ref_format == REFTEIX5) {
    if (print_elend_x(ptr_ref, ptr_ref_len, "bibl", ptr_indent, ns) == NULL) {
      dbi_result_free(dbires_cit);
      delete_all_lilimem(&sentinel);
      clear_mset_stack(ptr_mset_sentinel);
      return 801;
    }
  }
  else {
    if (print_elend_x(ptr_ref, ptr_ref_len, "bibliomixed", ptr_indent, ns) == NULL) {
      dbi_result_free(dbires_cit);
      delete_all_lilimem(&sentinel);
      clear_mset_stack(ptr_mset_sentinel);
      return 801;
    }
  }

  dbi_result_free(dbires_cit);
  delete_all_lilimem(&sentinel);
  clear_mset_stack(ptr_mset_sentinel);
  return 0;
}

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  format_entry(): formats a bibliography entry in DocBook cooked biblio
                  format

  int format_entry returns 0 if successful, >0 if failed

  char** ptr_ref ptr to a ptr to a buffer created with malloc which will receive
        the output. The calling function must allocate the buffer
        with at least one byte. This function will reallocate the
        buffer as needed. *ptr will be updated whenever a realloc is
        necessary. The calling function is responsible for freeing the
        memory again.

  size_t* ptr_ref_len ptr to an int holding the current length of ref.
                     Will be modified if ref is reallocated.

  char* pubtype ptr to a string containing the pubtype to format

  struct BIBCONNS* ptr_bibconns ptr to structure with database connections

  dbi_result dbires_cit ptr to a dbi result structure containing
                        the current citation style info

  dbi_result dbires ptr to a dbi result structure  containing the
                    current reference. If this is NULL, only the
                    authorlist will be formatted using the database
		    and ID info in biblio_info

  const char* database ptr to string containing the current reference database

  const char* username ptr to string containing the current user

  int nref_counter number of reference (to be displayed in bibliography)

  struct mset* ptr_mset_sentinel ptr to the bibliomset relation attribute stack

  struct bibinfo* ptr_biblio_info ptr to struct with additional bibliographic
                  info

  short title_as_author if > 0, the reference has no authors and the style
                  wants the title to be printed instead

  const char* ns optional namespace prefix

  struct xmlindent* ptr_indent  indentation information

  int n_ref_format requested output format: REFDOCBK, REFDOCBKX, REFTEIX

  const char* pdfroot optional path prefix for file:// links

  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
int format_entry(char** ptr_ref, size_t* ptr_ref_len, char* pubtype, struct BIBCONNS* ptr_bibconns, dbi_result dbires_cit, dbi_result dbires, const char* database, const char* username, int nref_counter, struct mset* ptr_mset_sentinel, struct bibinfo* ptr_biblio_info, short title_as_author, const char* ns, struct xmlindent* ptr_indent, int n_ref_format, const char* pdfroot) {
  int error = 0;
  size_t nref_buffer_len;
  int n_intext = 0; /* 0 = bibliography, 1 = intext, 2 = intext subseq */
  int n_status;
  int errflag;
  unsigned int n_id;
  unsigned int n_refstyle_id;
  unsigned int n_separator_id;
  short my_title_as_author;
  char *new_ref;
  char *ref_buffer;
  char *new_ref_buffer;
  char *sql_command;
  const char *alternatetext;
  const char *item_type;
  const char *drivername;
  char mset_string[10] = "";
  dbi_result dbires_pos;
  dbi_result dbires_ref;

/*   printf("format_entry(): title_as_author went to %d<<\n", title_as_author); */
  drivername = dbi_driver_get_name(dbi_conn_get_driver(ptr_bibconns->conn_refdb));

  /* get memory for sql command buffer */
  sql_command = malloc(8192);
  if (!sql_command) {
    return 801;
  }

  /* get memory for the  buffer */
  nref_buffer_len = 1024;
  ref_buffer = malloc(nref_buffer_len);
  if (!ref_buffer) {
    free(sql_command);
    return 801;
  }
  ref_buffer[0] = '\0'; /* terminate */

/*   printf("pubtype went to:%s\n", pubtype); */
  if (!strcmp(pubtype, "INTEXTS")) {
    pubtype[6] = '\0';
    n_intext = 2;
  }
  else if (!strcmp(pubtype, "INTEXT")) {
    n_intext = 1;
  }
  else if (!strcmp(pubtype, "AUTHORONLYS")) {
    pubtype[10] = '\0';
    n_intext = 2;
  }
  else if (!strcmp(pubtype, "AUTHORONLY")) {
    n_intext = 1;
  }
  else if (!strcmp(pubtype, "YEARONLY")) {
    n_intext = 1;
  }

  n_id = my_dbi_result_get_int_idval(dbires_cit, "ID");

  if ((dbires_ref = load_style(pubtype, n_id, ptr_bibconns->conn_refdb)) == NULL) {
    free(sql_command);
    free(ref_buffer);
    return 241;
  }

  if (title_as_author) {
    if (!strcmp(pubtype, "BOOK")) {
      my_title_as_author = 2;
    }
    else if (!strcmp(pubtype, "NEWS")) {
      alternatetext = my_dbi_result_get_string_idx(dbires_ref, AUTHORLISTALTERNATETEXT);
      if (alternatetext && !strcmp(alternatetext, "JOURNALNAME")) {
	my_title_as_author = 5;
      }
      else {
	my_title_as_author = 1;
      }
    }
    else {
      my_title_as_author = 1;
    }
  }
  else {
    my_title_as_author = 0;
  }

  /* only need authorlist (used for finding citations with same author/year) */
  if (!dbires) {
    if ((n_status = format_authorlist(ptr_ref, ptr_ref_len, ptr_bibconns, dbires, dbi_style_res, 4 /* first available */, database, 1, ptr_biblio_info, my_title_as_author, ns, ptr_indent, n_ref_format)) != 0) {
      error = 1;
    }
    else {
/*       *ptr_ref = new_ref; */
    }
    free(sql_command);
    free(ref_buffer);
    unload_style();
    return n_status;
  }


  n_refstyle_id = my_dbi_result_get_int_idval(dbi_style_res, "ID");
  if ((errflag = my_dbi_conn_error_flag(ptr_bibconns->conn_refdb)) != 0) {
    /* ToDo: jump out here? */
    LOG_PRINT(LOG_WARNING, get_status_msg(241));
    free(sql_command);
    free(ref_buffer);
    unload_style();
    return 241;
  }

  if (!strcmp(my_dbi_conn_get_cap(ptr_bibconns->conn_refdb, "multiple_db"), "t")) {
    sprintf(sql_command, "SELECT TYPE, SEPARATORID, POS FROM %s.POSITIONS WHERE REFSTYLEID=%u ORDER BY POS ASC", main_db, n_refstyle_id);
  }
  else {
    sprintf(sql_command, "SELECT TYPE, SEPARATORID, POS FROM POSITIONS WHERE REFSTYLEID=%u ORDER BY POS ASC", n_refstyle_id);
  }

  LOG_PRINT(LOG_DEBUG, sql_command);
  dbires_pos = dbi_conn_query(ptr_bibconns->conn_refdb, sql_command);
  if (!dbires_pos) {
    LOG_PRINT(LOG_WARNING, get_status_msg(241));
    dbi_result_free(dbires_pos);
    unload_style();
    free(sql_command);
    free(ref_buffer);
    return 241;
  }

  /* walk through all positions and format the elements */
  while (dbi_result_next_row(dbires_pos)) {
    item_type = dbi_result_get_string(dbires_pos, "TYPE");
    /*       n_separator_id = my_dbi_result_get_idval(dbi_style_res, "SEPARATORID"); */
    n_separator_id = dbi_result_get_uint(dbires_pos, "SEPARATORID");

    if (!item_type || !*item_type) {
      continue;
    }
/*     printf("in format_entry loop, item_type=%s; pubtype=%s<<\n", item_type, pubtype); */

    /* the following code uses a buffer to format the bits and pieces.
       this buffer will only be appended to the result string if the
       bits and pieces are not empty. */

    /*---------------------------------------------------------------*/
    if (strcmp(item_type, "REFNUMBER") == 0) {
      /*  	  if (!nformat_spec) { */
      if ((new_ref_buffer = set_mset(ptr_mset_sentinel, "", &ref_buffer, &nref_buffer_len, ns, ptr_indent, n_ref_format)) == NULL) {
	error = 801;
	break;
      }

      ref_buffer = new_ref_buffer;

      if ((new_ref_buffer = format_refnumber(&ref_buffer, &nref_buffer_len, ptr_bibconns, dbires, dbi_style_res, nref_counter, n_intext, ns, ptr_indent, n_ref_format)) == NULL) {
	error = 805;
	break;
      }

      if (ref_buffer[0]) {
	if ((new_ref = mstrcat(*ptr_ref, ref_buffer, ptr_ref_len, 0)) == NULL) {
	  error = 801;
	  break;
	}

	*ptr_ref = new_ref;
      }
      ref_buffer[0] = '\0';
    }
    /*---------------------------------------------------------------*/
    else if (strcmp(item_type, "CITEKEY") == 0) {
      /*  	  if (!nformat_spec) { */
      if ((new_ref_buffer = set_mset(ptr_mset_sentinel, "", &ref_buffer, &nref_buffer_len, ns, ptr_indent, n_ref_format)) == NULL) {
	error = 801;
	break;
      }

      ref_buffer = new_ref_buffer;

      if ((new_ref_buffer = format_citekey(&ref_buffer, &nref_buffer_len, ptr_bibconns, dbires, dbi_style_res, n_intext, ns, ptr_indent, n_ref_format)) == NULL) {
	error = 805;
	break;
      }

      if (ref_buffer[0]) {
	if ((new_ref = mstrcat(*ptr_ref, ref_buffer, ptr_ref_len, 0)) == NULL) {
	  error = 801;
	  break;
	}

	*ptr_ref = new_ref;
      }
      ref_buffer[0] = '\0';
    }
    /*---------------------------------------------------------------*/
    /* part author list */
    else if (strcmp(item_type, "AUTHORLIST") == 0) {

      if (!n_intext) {
	if (strcmp(pubtype, "BOOK") == 0) {
	  strcpy(mset_string, "book");
	}
	else if (strcmp(pubtype, "CHAP") == 0) {
	  strcpy(mset_string, "chapter");
	}
	else {
	  strcpy(mset_string, "article");
	}
      }
      else {
	/*  	      strcpy(mset_string, "intext"); */
	mset_string[0] = '\0';
      }
      if ((new_ref_buffer = set_mset(ptr_mset_sentinel, mset_string, &ref_buffer, &nref_buffer_len, ns, ptr_indent, n_ref_format)) == NULL) {
	error = 801;
	break;
      }

      ref_buffer = new_ref_buffer;

      if ((n_status = format_authorlist(&ref_buffer, &nref_buffer_len, ptr_bibconns, dbires, dbi_style_res, 1 /* part authors */, database, n_intext, ptr_biblio_info, title_as_author, ns, ptr_indent, n_ref_format)) != 0) {
	error = n_status;
	break;
      }

      if (ref_buffer[0]) {
	if ((new_ref = mstrcat(*ptr_ref, ref_buffer, ptr_ref_len, 0)) == NULL) {
	  error = 801;
	  break;
	}

	*ptr_ref = new_ref;
      }
      ref_buffer[0] = '\0';
    }
    /*---------------------------------------------------------------*/
    /* publication author list */
    else if (strcmp(item_type, "EDITORLIST") == 0) {
      if (!n_intext) {
	strcpy(mset_string, "book");
      }
      else {
	/*  	      strcpy(mset_string, "intext"); */
	mset_string[0] = '\0';
      }

      if ((new_ref_buffer = set_mset(ptr_mset_sentinel, mset_string, &ref_buffer, &nref_buffer_len, ns, ptr_indent, n_ref_format)) == NULL) {
	error = 801;
	break;
      }

      ref_buffer = new_ref_buffer;
	
      if ((n_status = format_authorlist(&ref_buffer, &nref_buffer_len, ptr_bibconns, dbires, dbi_style_res, 2 /* publication authors */, database, n_intext, ptr_biblio_info, 0 /* never use title */, ns, ptr_indent, n_ref_format)) != 0) {
	error = n_status;
	break;
      }

      if (ref_buffer[0]) {
	if ((new_ref = mstrcat(*ptr_ref, ref_buffer, ptr_ref_len, 0)) == NULL) {
	  error = 801;
	  break;
	}

	*ptr_ref = new_ref;
      }
      ref_buffer[0] = '\0';
    }
    /*---------------------------------------------------------------*/
    /* set author list */
    else if (strcmp(item_type, "SEDITORLIST") == 0) {
      if (!n_intext) {
	strcpy(mset_string, "series");
      }
      else {
	/*  	      strcpy(mset_string, "intext"); */
	mset_string[0] = '\0';
      }
      if ((new_ref_buffer = set_mset(ptr_mset_sentinel, mset_string, &ref_buffer, &nref_buffer_len, ns, ptr_indent, n_ref_format)) == NULL) {
	error = 801;
	break;
      }

      ref_buffer = new_ref_buffer;

      if ((n_status = format_authorlist(&ref_buffer, &nref_buffer_len, ptr_bibconns, dbires, dbi_style_res, 3 /* set authors */, database, n_intext, ptr_biblio_info, 0 /* never use title */, ns, ptr_indent, n_ref_format)) != 0) {
	error = n_status;
	break;
      }

      if (ref_buffer[0]) {
	if ((new_ref = mstrcat(*ptr_ref, ref_buffer, ptr_ref_len, 0)) == NULL) {
	  error = 801;
	  break;
	}

	*ptr_ref = new_ref;
      }
      ref_buffer[0] = '\0';
    }
    /*---------------------------------------------------------------*/
    else if (strcmp(item_type, "ALLALIST") == 0) {
      if (!n_intext) {
	strcpy(mset_string, "all");
      }
      else {
	/*  	      strcpy(mset_string, "intext"); */
	mset_string[0] = '\0';
      }
      if ((new_ref_buffer = set_mset(ptr_mset_sentinel, mset_string, &ref_buffer, &nref_buffer_len, ns, ptr_indent, n_ref_format)) == NULL) {
	error = 801;
	break;
      }

      ref_buffer = new_ref_buffer;

      if ((n_status = format_authorlist(&ref_buffer, &nref_buffer_len, ptr_bibconns, dbires, dbi_style_res, 4 /* first available */, database, n_intext, ptr_biblio_info, 0 /* never use title */, ns, ptr_indent, n_ref_format)) != 0) {
	error = n_status;
	break;
      }

      if (ref_buffer[0]) {
	if ((new_ref = mstrcat(*ptr_ref, ref_buffer, ptr_ref_len, 0)) == NULL) {
	  error = 801;
	  break;
	}

	*ptr_ref = new_ref;
      }
      ref_buffer[0] = '\0';
    }
    /*---------------------------------------------------------------*/
    else if (strcmp(item_type, "PUBDATE") == 0) {
      if (!n_intext) {
	if (strcmp(pubtype, "BOOK") == 0) {
	  strcpy(mset_string, "book");
	}
	else if (strcmp(pubtype, "CHAP") == 0) {
	  strcpy(mset_string, "book");
	}
	else {
	  strcpy(mset_string, "article");
	}
      }
      else {
	/* 	  strcpy(mset_string, "intext"); */
	*mset_string = '\0';
      }

      if ((new_ref_buffer = set_mset(ptr_mset_sentinel, mset_string, &ref_buffer, &nref_buffer_len, ns, ptr_indent, n_ref_format)) == NULL) {
	error = 801;
	break;
      }

      ref_buffer = new_ref_buffer;
	
      if ((new_ref_buffer = format_pubdate(&ref_buffer, &nref_buffer_len, ptr_bibconns, dbires, dbi_style_res, dbires_cit, 0, n_intext, ptr_biblio_info->year_unique_suffix, ns, ptr_indent, n_ref_format)) == NULL) {
	error = 809;
	break;
      }

      if (ref_buffer[0]) {
	if ((new_ref = mstrcat(*ptr_ref, ref_buffer, ptr_ref_len, 0)) == NULL) {
	  error = 801;
	  break;
	}

	*ptr_ref = new_ref;
      }
      ref_buffer[0] = '\0';
    }
    /*---------------------------------------------------------------*/
    else if (strcmp(item_type, "PUBDATESEC") == 0) {
      if (!n_intext) {
	if (strcmp(pubtype, "BOOK") == 0) {
	  strcpy(mset_string, "book");
	}
	else if (strcmp(pubtype, "CHAP") == 0) {
	  strcpy(mset_string, "book");
	}
	else {
	  strcpy(mset_string, "article");
	}
      }
      else {
	/* 	  strcpy(mset_string, "intext"); */
	*mset_string = '\0';
      }

      if ((new_ref_buffer = set_mset(ptr_mset_sentinel, mset_string, &ref_buffer, &nref_buffer_len, ns, ptr_indent, n_ref_format)) == NULL) {
	error = 801;
	break;
      }

      ref_buffer = new_ref_buffer;

      if ((new_ref_buffer = format_pubdate(&ref_buffer, &nref_buffer_len, ptr_bibconns, dbires, dbi_style_res, dbires_cit, 1, n_intext, NULL, ns, ptr_indent, n_ref_format)) == NULL) {
	error = 810;
	break;
      }

      if (ref_buffer[0]) {
	if ((new_ref = mstrcat(*ptr_ref, ref_buffer, ptr_ref_len, 0)) == NULL) {
	  error = 801;
	  break;
	}

	*ptr_ref = new_ref;
      }
      ref_buffer[0] = '\0';
    }
    /*---------------------------------------------------------------*/
    else if (strcmp(item_type, "TITLE") == 0
	     && my_title_as_author != 1) {
      if ((new_ref_buffer = format_title(&ref_buffer, &nref_buffer_len, ptr_bibconns, dbires, dbi_style_res, 1, n_intext, 0 /* not title as author */, ns, ptr_indent, n_ref_format)) == NULL) {
	error = 811;
	break;
      }

      if (ref_buffer[0]) {
	/* set mset only if we have some title contents */
	if (!n_intext) {
	  if (strcmp(pubtype, "CHAP") == 0
	      /* todo: is this the correct fix? */
	      || strcmp(pubtype, "CASE") == 0) {
	    strcpy(mset_string, "chapter");
	  }
	  else {
	    strcpy(mset_string, "article");
	  }
	}
	else {
	  strcpy(mset_string, "intext");
	}

	if ((new_ref = set_mset(ptr_mset_sentinel, mset_string, ptr_ref, ptr_ref_len, ns, ptr_indent, n_ref_format)) == NULL) {
	  error = 801;
	  break;
	}

	*ptr_ref = new_ref;

	if ((new_ref = mstrcat(*ptr_ref, ref_buffer, ptr_ref_len, 0)) == NULL) {
	  error = 801;
	  break;
	}

	*ptr_ref = new_ref;
      }
      ref_buffer[0] = '\0';
    }
    /*---------------------------------------------------------------*/
    else if (strcmp(item_type, "BOOKTITLE") == 0
	     && my_title_as_author != 2) {
      if ((new_ref_buffer = format_title(&ref_buffer, &nref_buffer_len, ptr_bibconns, dbires, dbi_style_res, 2, n_intext, 0 /* not title as author */, ns, ptr_indent, n_ref_format)) == NULL) {
	error = 812;
	break;
      }

      if (ref_buffer[0]) {
	/* set mset only if we have some title contents */
	if (!n_intext) {
	  strcpy(mset_string, "book");
	}
	else {
	  strcpy(mset_string, "intext");
	}
	
	if ((new_ref = set_mset(ptr_mset_sentinel, mset_string, ptr_ref, ptr_ref_len, ns, ptr_indent, n_ref_format)) == NULL) {
	  error = 801;
	  break;
	}
	
	*ptr_ref = new_ref;
	
	if ((new_ref = mstrcat(*ptr_ref, ref_buffer, ptr_ref_len, 0)) == NULL) {
	  error = 801;
	  break;
	}

	*ptr_ref = new_ref;
      }
      ref_buffer[0] = '\0';
    }
    /*---------------------------------------------------------------*/
    else if (strcmp(item_type, "SERIESTITLE") == 0
	     && my_title_as_author != 3) {
      if ((new_ref_buffer = format_title(&ref_buffer, &nref_buffer_len, ptr_bibconns, dbires, dbi_style_res, 3, n_intext, 0 /* not title as author */, ns, ptr_indent, n_ref_format)) == NULL) {
	error = 813;
	break;
      }

      if (ref_buffer[0]) {
	/* set mset only if we have some title contents */
	if (!n_intext) {
	  strcpy(mset_string, "series");
	}
	else {
	  strcpy(mset_string, "intext");
	}
	if ((new_ref = set_mset(ptr_mset_sentinel, mset_string, ptr_ref, ptr_ref_len, ns, ptr_indent, n_ref_format)) == NULL) {
	  error = 801;
	  break;
	}

	*ptr_ref = new_ref;

	if ((new_ref = mstrcat(*ptr_ref, ref_buffer, ptr_ref_len, 0)) == NULL) {
	  error = 801;
	  break;
	}

	*ptr_ref = new_ref;
      }
      ref_buffer[0] = '\0';
    }
    /*---------------------------------------------------------------*/
    else if (strcmp(item_type, "ALLTITLE") == 0) {
/*       if (!n_intext) { */
      if ((new_ref_buffer = format_title(&ref_buffer, &nref_buffer_len, ptr_bibconns, dbires, dbi_style_res, 4, n_intext, 0 /* not title as author */, ns, ptr_indent, n_ref_format)) == NULL) {
	error = 813;
	break;
      }

      ref_buffer = new_ref_buffer;
      if (ref_buffer[0]) {
	/* set mset only if we have some title contents */
	strcpy(mset_string, "all"); /* does this make sense? */

	if ((new_ref = set_mset(ptr_mset_sentinel, mset_string, ptr_ref, ptr_ref_len, ns, ptr_indent, n_ref_format)) == NULL) {
	  error = 801;
	  break;
	}

	*ptr_ref = new_ref;

	if ((new_ref = mstrcat(*ptr_ref, ref_buffer, ptr_ref_len, 0)) == NULL) {
	  error = 801;
	  break;
	}

	*ptr_ref = new_ref;
      }
      ref_buffer[0] = '\0';
    }
    /*---------------------------------------------------------------*/
    else if (strcmp(item_type, "JOURNALNAME") == 0
	     && my_title_as_author != 5) {
/*       printf("my_title_as_author went to %d\n", my_title_as_author); */
      if (!n_intext) {
	strcpy(mset_string, "journal");
      }
      else {
	strcpy(mset_string, "intext");
      }
      if ((new_ref_buffer = set_mset(ptr_mset_sentinel, mset_string, &ref_buffer, &nref_buffer_len, ns, ptr_indent, n_ref_format)) == NULL) {
	error = 801;
	break;
      }

      ref_buffer = new_ref_buffer;

      if ((new_ref_buffer = format_journalname(&ref_buffer, &nref_buffer_len, ptr_bibconns, dbires, dbi_style_res, database, n_intext, ns, ptr_indent, n_ref_format, 0 /* not title as author */)) == NULL) {
	error = 814;
	break;
      }

      if (ref_buffer[0]) {
	if ((new_ref = mstrcat(*ptr_ref, ref_buffer, ptr_ref_len, 0)) == NULL) {
	  error = 801;
	  break;
	}

	*ptr_ref = new_ref;
      }
      ref_buffer[0] = '\0';
    }
    /*---------------------------------------------------------------*/
    else if (strcmp(item_type, "VOLUME") == 0) {
      if (!n_intext) {
	strcpy(mset_string, "article");
      }
      else {
	strcpy(mset_string, "intext");
      }
      if ((new_ref_buffer = set_mset(ptr_mset_sentinel, mset_string, &ref_buffer, &nref_buffer_len, ns, ptr_indent, n_ref_format)) == NULL) {
	error = 801;
	break;
      }

      ref_buffer = new_ref_buffer;

      if ((new_ref_buffer = format_simple(&ref_buffer, &nref_buffer_len, ptr_bibconns, dbires, dbi_style_res, n_intext, REFDB_VOLUME, VOLUMEPRECEEDING, VOLUMEFOLLOWING, "volumenum", ns, ptr_indent, n_ref_format)) == NULL) {
	error = 815;
	break;
      }

      if (ref_buffer[0]) {
	if ((new_ref = mstrcat(*ptr_ref, ref_buffer, ptr_ref_len, 0)) == NULL) {
	  error = 801;
	  break;
	}

	*ptr_ref = new_ref;
      }
      ref_buffer[0] = '\0';
    }
    /*---------------------------------------------------------------*/
    else if (strcmp(item_type, "ISSUE") == 0) {
      if (!n_intext) {
	strcpy(mset_string, "article");
      }
      else {
	strcpy(mset_string, "intext");
      }
      if ((new_ref_buffer = set_mset(ptr_mset_sentinel, mset_string, &ref_buffer, &nref_buffer_len, ns, ptr_indent, n_ref_format)) == NULL) {
	error = 801;
	break;
      }

      ref_buffer = new_ref_buffer;
	
      if ((new_ref_buffer = format_simple(&ref_buffer, &nref_buffer_len, ptr_bibconns, dbires, dbi_style_res, n_intext, REFDB_ISSUE, ISSUEPRECEEDING, ISSUEFOLLOWING, "issuenum", ns, ptr_indent, n_ref_format)) == NULL) {
	error = 816;
	break;
      }
      
      if (ref_buffer[0]) {
	if ((new_ref = mstrcat(*ptr_ref, ref_buffer, ptr_ref_len, 0)) == NULL) {
	  error = 801;
	  break;
	}

	*ptr_ref = new_ref;
      }
      ref_buffer[0] = '\0';
    }
    /*---------------------------------------------------------------*/
    else if (strcmp(item_type, "PAGES") == 0) {
      if (!n_intext) {
	if (strcmp(pubtype, "CHAP") == 0) {
	  strcpy(mset_string, "chapter");
	}
	else {
	  strcpy(mset_string, "article");
	}
      }
      else {
	strcpy(mset_string, "intext");
      }
      if ((new_ref_buffer = set_mset(ptr_mset_sentinel, mset_string, &ref_buffer, &nref_buffer_len, ns, ptr_indent, n_ref_format)) == NULL) {
	error = 801;
	break;
      }

      ref_buffer = new_ref_buffer;

      if ((new_ref_buffer = format_pages(&ref_buffer, &nref_buffer_len, ptr_bibconns, dbires, dbi_style_res, n_intext, ns, ptr_indent, n_ref_format)) == NULL) {
	error = 817;
	break;
      }

      if (ref_buffer[0]) {
	if ((new_ref = mstrcat(*ptr_ref, ref_buffer, ptr_ref_len, 0)) == NULL) {
	  error = 801;
	  break;
	}

	*ptr_ref = new_ref;
      }
      ref_buffer[0] = '\0';
    }
    /*---------------------------------------------------------------*/
    else if (strcmp(item_type, "PUBLISHER") == 0) {
      if (!n_intext) {
	if (strcmp(pubtype, "CHAP") == 0 || strcmp(pubtype, "BOOK") == 0) {
	  strcpy(mset_string, "book");
	}
	else {
	  strcpy(mset_string, "journal");
	}
      }
      else {
	strcpy(mset_string, "intext");
      }
      if ((new_ref_buffer = set_mset(ptr_mset_sentinel, mset_string, &ref_buffer, &nref_buffer_len, ns, ptr_indent, n_ref_format)) == NULL) {
	error = 801;
	break;
      }

      ref_buffer = new_ref_buffer;

      if ((new_ref_buffer = format_simple(&ref_buffer, &nref_buffer_len, ptr_bibconns, dbires, dbi_style_res, n_intext, REFDB_PUBLISHER, PUBLISHERPRECEEDING, PUBLISHERFOLLOWING, "publishername", ns, ptr_indent, n_ref_format)) == NULL) {
	error = 818;
	break;
      }

      if (ref_buffer[0]) {
	if ((new_ref = mstrcat(*ptr_ref, ref_buffer, ptr_ref_len, 0)) == NULL) {
	  error = 801;
	  break;
	}

	*ptr_ref = new_ref;
      }
      ref_buffer[0] = '\0';
    }
    /*---------------------------------------------------------------*/
    else if (strcmp(item_type, "PUBPLACE") == 0) {
      if (!n_intext) {
	if (strcmp(pubtype, "CHAP") == 0 || strcmp(pubtype, "BOOK") == 0) {
	  strcpy(mset_string, "book");
	}
	else {
	  strcpy(mset_string, "journal");
	}
      }
      else {
	strcpy(mset_string, "intext");
      }
      if ((new_ref_buffer = set_mset(ptr_mset_sentinel, mset_string, &ref_buffer, &nref_buffer_len, ns, ptr_indent, n_ref_format)) == NULL) {
	error = 801;
	break;
      }

      ref_buffer = new_ref_buffer;

      if ((new_ref_buffer = format_simple(&ref_buffer, &nref_buffer_len, ptr_bibconns, dbires, dbi_style_res, n_intext, REFDB_CITY, PUBPLACEPRECEEDING, PUBPLACEFOLLOWING, "city", ns, ptr_indent, n_ref_format)) == NULL) {
	error = 819;
	break;
      }

      if (ref_buffer[0]) {
	if ((new_ref = mstrcat(*ptr_ref, ref_buffer, ptr_ref_len, 0)) == NULL) {
	  error = 801;
	  break;
	}

	*ptr_ref = new_ref;
      }
      ref_buffer[0] = '\0';
    }
    /*---------------------------------------------------------------*/
    else if (strcmp(item_type, "SERIAL") == 0) {
      char elname[16];

      if (!n_intext) {
	if (strcmp(pubtype, "CHAP") == 0 || strcmp(pubtype, "BOOK") == 0) {
	  strcpy(mset_string, "book");
	  strcpy(elname, "isbn");
	}
	else {
	  strcpy(mset_string, "journal");
	  strcpy(elname, "issn");
	}
      }
      else {
	strcpy(mset_string, "intext");
      }
      if ((new_ref_buffer = set_mset(ptr_mset_sentinel, mset_string, &ref_buffer, &nref_buffer_len, ns, ptr_indent, n_ref_format)) == NULL) {
	error = 801;
	break;
      }

      ref_buffer = new_ref_buffer;

      if ((new_ref_buffer = format_simple(&ref_buffer, &nref_buffer_len, ptr_bibconns, dbires, dbi_style_res, n_intext, REFDB_ISSN, SERIALPRECEEDING, SERIALFOLLOWING, elname, ns, ptr_indent, n_ref_format)) == NULL) {
	error = 820;
	break;
      }

      if (ref_buffer[0]) {
	if ((new_ref = mstrcat(*ptr_ref, ref_buffer, ptr_ref_len, 0)) == NULL) {
	  error = 801;
	  break;
	}

	*ptr_ref = new_ref;
      }
      ref_buffer[0] = '\0';
    }
    /*---------------------------------------------------------------*/
    else if (strcmp(item_type, "ADDRESS") == 0) {
      if (!n_intext) {
	if (strcmp(pubtype, "CHAP") == 0 || strcmp(pubtype, "BOOK") == 0) {
	  strcpy(mset_string, "book");
	}
	else {
	  strcpy(mset_string, "journal");
	}
      }
      else {
	strcpy(mset_string, "intext");
      }
      if ((new_ref_buffer = set_mset(ptr_mset_sentinel, mset_string, &ref_buffer, &nref_buffer_len, ns, ptr_indent, n_ref_format)) == NULL) {
	error = 801;
	break;
      }

      ref_buffer = new_ref_buffer;

      if ((new_ref_buffer = format_simple(&ref_buffer, &nref_buffer_len, ptr_bibconns, dbires, dbi_style_res, n_intext, REFDB_ADDRESS, ADDRESSPRECEEDING, ADDRESSFOLLOWING, "address", ns, ptr_indent, n_ref_format)) == NULL) {
	error = 821;
	break;
      }

      if (ref_buffer[0]) {
	if ((new_ref = mstrcat(*ptr_ref, ref_buffer, ptr_ref_len, 0)) == NULL) {
	  error = 801;
	  break;
	}

	*ptr_ref = new_ref;
      }
      ref_buffer[0] = '\0';
    }
    /*---------------------------------------------------------------*/
    else if (strncmp(item_type, "USERDEF", 7) == 0) {
      if (!n_intext) {
	if (strcmp(pubtype, "CHAP") == 0 || strcmp(pubtype, "BOOK") == 0) {
	  strcpy(mset_string, "book");
	}
	else {
	  strcpy(mset_string, "journal");
	}
      }
      else {
	strcpy(mset_string, "intext");
      }
      if ((new_ref_buffer = set_mset(ptr_mset_sentinel, mset_string, &ref_buffer, &nref_buffer_len, ns, ptr_indent, n_ref_format)) == NULL) {
	error = 801;
	break;
      }

      ref_buffer = new_ref_buffer;

      if (item_type[7] == '1') {
	if ((new_ref_buffer = format_simple(&ref_buffer, &nref_buffer_len, ptr_bibconns, dbires, dbi_style_res, n_intext, REFDB_USER1, USERDEF1PRECEEDING, USERDEF1FOLLOWING, "userdef1", ns, ptr_indent, n_ref_format)) == NULL) {
	  error = 823;
	  break;
	}
      }	
      else if (item_type[7] == '2') {
	if ((new_ref_buffer = format_simple(&ref_buffer, &nref_buffer_len, ptr_bibconns, dbires, dbi_style_res, n_intext, REFDB_USER2, USERDEF2PRECEEDING, USERDEF2FOLLOWING, "userdef2", ns, ptr_indent, n_ref_format)) == NULL) {
	  error = 824;
	  break;
	}
      }
      else if (item_type[7] == '3') {
	if ((new_ref_buffer = format_simple(&ref_buffer, &nref_buffer_len, ptr_bibconns, dbires, dbi_style_res, n_intext, REFDB_USER3, USERDEF3PRECEEDING, USERDEF3FOLLOWING, "userdef3", ns, ptr_indent, n_ref_format)) == NULL) {
	  error = 825;
	  break;
	}
      }
      else if (item_type[7] == '4') {
	if ((new_ref_buffer = format_simple(&ref_buffer, &nref_buffer_len, ptr_bibconns, dbires, dbi_style_res, n_intext, REFDB_USER4, USERDEF4PRECEEDING, USERDEF4FOLLOWING, "userdef4", ns, ptr_indent, n_ref_format)) == NULL) {
	  error = 826;
	  break;
	}
      }
      else if (item_type[7] == '5') {
	if ((new_ref_buffer = format_simple(&ref_buffer, &nref_buffer_len, ptr_bibconns, dbires, dbi_style_res, n_intext, REFDB_USER5, USERDEF5PRECEEDING, USERDEF5FOLLOWING, "userdef5", ns, ptr_indent, n_ref_format)) == NULL) {
	  error = 827;
	  break;
	}
      }

      if (ref_buffer[0]) {
	if ((new_ref = mstrcat(*ptr_ref, ref_buffer, ptr_ref_len, 0)) == NULL) {
	  error = 801;
	  break;
	}

	*ptr_ref = new_ref;
      }
      ref_buffer[0] = '\0';
    }
    /*---------------------------------------------------------------*/
    else if (strncmp(item_type, "LINK", 4) == 0) {
      if (!n_intext) {
	if (strcmp(pubtype, "CHAP") == 0 || strcmp(pubtype, "BOOK") == 0) {
	  strcpy(mset_string, "book");
	}
	else {
	  strcpy(mset_string, "journal");
	}
      }
      else {
	strcpy(mset_string, "intext");
      }
      if ((new_ref_buffer = set_mset(ptr_mset_sentinel, mset_string, &ref_buffer, &nref_buffer_len, ns, ptr_indent, n_ref_format)) == NULL) {
	error = 801;
	break;
      }

      ref_buffer = new_ref_buffer;

      if (item_type[4] == '0') {
	if ((new_ref_buffer = format_ulink(&ref_buffer, &nref_buffer_len, ptr_bibconns, dbires, dbi_style_res, 0 /*LINK0*/, n_intext, LINK0PRECEEDING, LINK0FOLLOWING, ns, ptr_indent, n_ref_format, pdfroot)) == NULL) {
	  error = 831;
	  break;
	}
      }
      else if (item_type[4] == '1') {
	if ((new_ref_buffer = format_ulink(&ref_buffer, &nref_buffer_len, ptr_bibconns, dbires, dbi_style_res, 1 /*LINK1*/, n_intext, LINK1PRECEEDING, LINK1FOLLOWING, ns, ptr_indent, n_ref_format, pdfroot)) == NULL) {
	  error = 831;
	  break;
	}
      }
      else if (item_type[4] == '2') {
	if ((new_ref_buffer = format_ulink(&ref_buffer, &nref_buffer_len, ptr_bibconns, dbires, dbi_style_res, 2 /*LINK2*/, n_intext, LINK2PRECEEDING, LINK2FOLLOWING, ns, ptr_indent, n_ref_format, pdfroot)) == NULL) {
	  error = 832;
	  break;
	}
      }
      else if (item_type[4] == '3') {
	if ((new_ref_buffer = format_ulink(&ref_buffer, &nref_buffer_len, ptr_bibconns, dbires, dbi_style_res, 3 /*LINK3*/, n_intext, LINK3PRECEEDING, LINK3FOLLOWING, ns, ptr_indent, n_ref_format, pdfroot)) == NULL) {
	  error = 833;
	  break;
	}
      }
      else if (item_type[4] == '4') {
	if ((new_ref_buffer = format_ulink(&ref_buffer, &nref_buffer_len, ptr_bibconns, dbires, dbi_style_res, 4 /*LINK4*/, n_intext, LINK4PRECEEDING, LINK4FOLLOWING, ns, ptr_indent, n_ref_format, pdfroot)) == NULL) {
	  error = 834;
	  break;
	}
      }

      if (ref_buffer[0]) {
	if ((new_ref = mstrcat(*ptr_ref, ref_buffer, ptr_ref_len, 0)) == NULL) {
	  error = 801;
	  break;
	}
	
	*ptr_ref = new_ref;
      }
      ref_buffer[0] = '\0';
    }
    /*---------------------------------------------------------------*/
    else if (strcmp(item_type, "ABSTRACT") == 0) {
      if (!n_intext) {
	if (strcmp(pubtype, "CHAP") == 0 || strcmp(pubtype, "BOOK") == 0) {
	  strcpy(mset_string, "book");
	}
	else {
	  strcpy(mset_string, "journal");
	}
      }
      else {
	strcpy(mset_string, "intext");
      }
      if ((new_ref_buffer = set_mset(ptr_mset_sentinel, mset_string, &ref_buffer, &nref_buffer_len, ns, ptr_indent, n_ref_format)) == NULL) {
	error = 801;
	break;
      }

      ref_buffer = new_ref_buffer;

      if ((new_ref_buffer = format_simple(&ref_buffer, &nref_buffer_len, ptr_bibconns, dbires, dbi_style_res, n_intext, REFDB_ABSTRACT, ABSTRACTPRECEEDING, ABSTRACTFOLLOWING, "abstract", ns, ptr_indent, n_ref_format)) == NULL) {
	error = 835;
	break;
      }

      if (ref_buffer[0]) {
	if ((new_ref = mstrcat(*ptr_ref, ref_buffer, ptr_ref_len, 0)) == NULL) {
	  error = 801;
	  break;
	}

	*ptr_ref = new_ref;
      }
      ref_buffer[0] = '\0';
    }


    /*---------------------------------------------------------------*/
    else if (strcmp(item_type, "TYPEOFWORK") == 0) {
      if (!n_intext) {
	if (strcmp(pubtype, "CHAP") == 0 || strcmp(pubtype, "BOOK") == 0) {
	  strcpy(mset_string, "book");
	}
	else {
	  strcpy(mset_string, "journal");
	}
      }
      else {
	strcpy(mset_string, "intext");
      }
      if ((new_ref_buffer = set_mset(ptr_mset_sentinel, mset_string, &ref_buffer, &nref_buffer_len, ns, ptr_indent, n_ref_format)) == NULL) {
	error = 801;
	break;
      }

      ref_buffer = new_ref_buffer;

      if ((new_ref_buffer = format_simple(&ref_buffer, &nref_buffer_len, ptr_bibconns, dbires, dbi_style_res, n_intext, REFDB_TYPEOFWORK, TYPEOFWORKPRECEEDING, TYPEOFWORKFOLLOWING, "typeofwork", ns, ptr_indent, n_ref_format)) == NULL) {
	error = 835;
	break;
      }

      if (ref_buffer[0]) {
	if ((new_ref = mstrcat(*ptr_ref, ref_buffer, ptr_ref_len, 0)) == NULL) {
	  error = 801;
	  break;
	}

	*ptr_ref = new_ref;
      }
      ref_buffer[0] = '\0';
    }
    /*---------------------------------------------------------------*/
    else if (strcmp(item_type, "AREA") == 0) {
      if (!n_intext) {
	if (strcmp(pubtype, "CHAP") == 0 || strcmp(pubtype, "BOOK") == 0) {
	  strcpy(mset_string, "book");
	}
	else {
	  strcpy(mset_string, "journal");
	}
      }
      else {
	strcpy(mset_string, "intext");
      }
      if ((new_ref_buffer = set_mset(ptr_mset_sentinel, mset_string, &ref_buffer, &nref_buffer_len, ns, ptr_indent, n_ref_format)) == NULL) {
	error = 801;
	break;
      }

      ref_buffer = new_ref_buffer;

      if ((new_ref_buffer = format_simple(&ref_buffer, &nref_buffer_len, ptr_bibconns, dbires, dbi_style_res, n_intext, REFDB_AREA, AREAPRECEEDING, AREAFOLLOWING, "area", ns, ptr_indent, n_ref_format)) == NULL) {
	error = 835;
	break;
      }

      if (ref_buffer[0]) {
	if ((new_ref = mstrcat(*ptr_ref, ref_buffer, ptr_ref_len, 0)) == NULL) {
	  error = 801;
	  break;
	}

	*ptr_ref = new_ref;
      }
      ref_buffer[0] = '\0';
    }
    /*---------------------------------------------------------------*/
    else if (strcmp(item_type, "OSTYPE") == 0) {
      if (!n_intext) {
	if (strcmp(pubtype, "CHAP") == 0 || strcmp(pubtype, "BOOK") == 0) {
	  strcpy(mset_string, "book");
	}
	else {
	  strcpy(mset_string, "journal");
	}
      }
      else {
	strcpy(mset_string, "intext");
      }
      if ((new_ref_buffer = set_mset(ptr_mset_sentinel, mset_string, &ref_buffer, &nref_buffer_len, ns, ptr_indent, n_ref_format)) == NULL) {
	error = 801;
	break;
      }

      ref_buffer = new_ref_buffer;

      if ((new_ref_buffer = format_simple(&ref_buffer, &nref_buffer_len, ptr_bibconns, dbires, dbi_style_res, n_intext, REFDB_OSTYPE, OSTYPEPRECEEDING, OSTYPEFOLLOWING, "ostype", ns, ptr_indent, n_ref_format)) == NULL) {
	error = 835;
	break;
      }

      if (ref_buffer[0]) {
	if ((new_ref = mstrcat(*ptr_ref, ref_buffer, ptr_ref_len, 0)) == NULL) {
	  error = 801;
	  break;
	}

	*ptr_ref = new_ref;
      }
      ref_buffer[0] = '\0';
    }
    /*---------------------------------------------------------------*/
    else if (strcmp(item_type, "DEGREE") == 0) {
      if (!n_intext) {
	if (strcmp(pubtype, "CHAP") == 0 || strcmp(pubtype, "BOOK") == 0) {
	  strcpy(mset_string, "book");
	}
	else {
	  strcpy(mset_string, "journal");
	}
      }
      else {
	strcpy(mset_string, "intext");
      }
      if ((new_ref_buffer = set_mset(ptr_mset_sentinel, mset_string, &ref_buffer, &nref_buffer_len, ns, ptr_indent, n_ref_format)) == NULL) {
	error = 801;
	break;
      }

      ref_buffer = new_ref_buffer;

      if ((new_ref_buffer = format_simple(&ref_buffer, &nref_buffer_len, ptr_bibconns, dbires, dbi_style_res, n_intext, REFDB_DEGREE, DEGREEPRECEEDING, DEGREEFOLLOWING, "degree", ns, ptr_indent, n_ref_format)) == NULL) {
	error = 835;
	break;
      }

      if (ref_buffer[0]) {
	if ((new_ref = mstrcat(*ptr_ref, ref_buffer, ptr_ref_len, 0)) == NULL) {
	  error = 801;
	  break;
	}

	*ptr_ref = new_ref;
      }
      ref_buffer[0] = '\0';
    }
    /*---------------------------------------------------------------*/
    else if (strcmp(item_type, "RUNNINGTIME") == 0) {
      if (!n_intext) {
	if (strcmp(pubtype, "CHAP") == 0 || strcmp(pubtype, "BOOK") == 0) {
	  strcpy(mset_string, "book");
	}
	else {
	  strcpy(mset_string, "journal");
	}
      }
      else {
	strcpy(mset_string, "intext");
      }
      if ((new_ref_buffer = set_mset(ptr_mset_sentinel, mset_string, &ref_buffer, &nref_buffer_len, ns, ptr_indent, n_ref_format)) == NULL) {
	error = 801;
	break;
      }

      ref_buffer = new_ref_buffer;

      if ((new_ref_buffer = format_simple(&ref_buffer, &nref_buffer_len, ptr_bibconns, dbires, dbi_style_res, n_intext, REFDB_RUNNINGTIME, RUNNINGTIMEPRECEEDING, RUNNINGTIMEFOLLOWING, "runningtime", ns, ptr_indent, n_ref_format)) == NULL) {
	error = 835;
	break;
      }

      if (ref_buffer[0]) {
	if ((new_ref = mstrcat(*ptr_ref, ref_buffer, ptr_ref_len, 0)) == NULL) {
	  error = 801;
	  break;
	}

	*ptr_ref = new_ref;
      }
      ref_buffer[0] = '\0';
    }
    /*---------------------------------------------------------------*/
    else if (strcmp(item_type, "CLASSCODEINTL") == 0) {
      if (!n_intext) {
	if (strcmp(pubtype, "CHAP") == 0 || strcmp(pubtype, "BOOK") == 0) {
	  strcpy(mset_string, "book");
	}
	else {
	  strcpy(mset_string, "journal");
	}
      }
      else {
	strcpy(mset_string, "intext");
      }
      if ((new_ref_buffer = set_mset(ptr_mset_sentinel, mset_string, &ref_buffer, &nref_buffer_len, ns, ptr_indent, n_ref_format)) == NULL) {
	error = 801;
	break;
      }

      ref_buffer = new_ref_buffer;

      if ((new_ref_buffer = format_simple(&ref_buffer, &nref_buffer_len, ptr_bibconns, dbires, dbi_style_res, n_intext, REFDB_CLASSCODEINTL, CLASSCODEINTLPRECEEDING, CLASSCODEINTLFOLLOWING, "classcodeintl", ns, ptr_indent, n_ref_format)) == NULL) {
	error = 835;
	break;
      }

      if (ref_buffer[0]) {
	if ((new_ref = mstrcat(*ptr_ref, ref_buffer, ptr_ref_len, 0)) == NULL) {
	  error = 801;
	  break;
	}

	*ptr_ref = new_ref;
      }
      ref_buffer[0] = '\0';
    }
    /*---------------------------------------------------------------*/
    else if (strcmp(item_type, "CLASSCODEUS") == 0) {
      if (!n_intext) {
	if (strcmp(pubtype, "CHAP") == 0 || strcmp(pubtype, "BOOK") == 0) {
	  strcpy(mset_string, "book");
	}
	else {
	  strcpy(mset_string, "journal");
	}
      }
      else {
	strcpy(mset_string, "intext");
      }
      if ((new_ref_buffer = set_mset(ptr_mset_sentinel, mset_string, &ref_buffer, &nref_buffer_len, ns, ptr_indent, n_ref_format)) == NULL) {
	error = 801;
	break;
      }

      ref_buffer = new_ref_buffer;

      if ((new_ref_buffer = format_simple(&ref_buffer, &nref_buffer_len, ptr_bibconns, dbires, dbi_style_res, n_intext, REFDB_CLASSCODEUS, CLASSCODEUSPRECEEDING, CLASSCODEUSFOLLOWING, "classcodeus", ns, ptr_indent, n_ref_format)) == NULL) {
	error = 835;
	break;
      }

      if (ref_buffer[0]) {
	if ((new_ref = mstrcat(*ptr_ref, ref_buffer, ptr_ref_len, 0)) == NULL) {
	  error = 801;
	  break;
	}

	*ptr_ref = new_ref;
      }
      ref_buffer[0] = '\0';
    }
    /*---------------------------------------------------------------*/
    else if (strcmp(item_type, "SENDEREMAIL") == 0) {
      if (!n_intext) {
	if (strcmp(pubtype, "CHAP") == 0 || strcmp(pubtype, "BOOK") == 0) {
	  strcpy(mset_string, "book");
	}
	else {
	  strcpy(mset_string, "journal");
	}
      }
      else {
	strcpy(mset_string, "intext");
      }
      if ((new_ref_buffer = set_mset(ptr_mset_sentinel, mset_string, &ref_buffer, &nref_buffer_len, ns, ptr_indent, n_ref_format)) == NULL) {
	error = 801;
	break;
      }

      ref_buffer = new_ref_buffer;

      if ((new_ref_buffer = format_simple(&ref_buffer, &nref_buffer_len, ptr_bibconns, dbires, dbi_style_res, n_intext, REFDB_SENDEREMAIL, SENDEREMAILPRECEEDING, SENDEREMAILFOLLOWING, "senderemail", ns, ptr_indent, n_ref_format)) == NULL) {
	error = 835;
	break;
      }

      if (ref_buffer[0]) {
	if ((new_ref = mstrcat(*ptr_ref, ref_buffer, ptr_ref_len, 0)) == NULL) {
	  error = 801;
	  break;
	}

	*ptr_ref = new_ref;
      }
      ref_buffer[0] = '\0';
    }
    /*---------------------------------------------------------------*/
    else if (strcmp(item_type, "RECIPIENTEMAIL") == 0) {
      if (!n_intext) {
	if (strcmp(pubtype, "CHAP") == 0 || strcmp(pubtype, "BOOK") == 0) {
	  strcpy(mset_string, "book");
	}
	else {
	  strcpy(mset_string, "journal");
	}
      }
      else {
	strcpy(mset_string, "intext");
      }
      if ((new_ref_buffer = set_mset(ptr_mset_sentinel, mset_string, &ref_buffer, &nref_buffer_len, ns, ptr_indent, n_ref_format)) == NULL) {
	error = 801;
	break;
      }

      ref_buffer = new_ref_buffer;

      if ((new_ref_buffer = format_simple(&ref_buffer, &nref_buffer_len, ptr_bibconns, dbires, dbi_style_res, n_intext, REFDB_RECIPIENTEMAIL, RECIPIENTEMAILPRECEEDING, RECIPIENTEMAILFOLLOWING, "recipientemail", ns, ptr_indent, n_ref_format)) == NULL) {
	error = 835;
	break;
      }

      if (ref_buffer[0]) {
	if ((new_ref = mstrcat(*ptr_ref, ref_buffer, ptr_ref_len, 0)) == NULL) {
	  error = 801;
	  break;
	}

	*ptr_ref = new_ref;
      }
      ref_buffer[0] = '\0';
    }
    /*---------------------------------------------------------------*/
    else if (strcmp(item_type, "MEDIATYPE") == 0) {
      if (!n_intext) {
	if (strcmp(pubtype, "CHAP") == 0 || strcmp(pubtype, "BOOK") == 0) {
	  strcpy(mset_string, "book");
	}
	else {
	  strcpy(mset_string, "journal");
	}
      }
      else {
	strcpy(mset_string, "intext");
      }
      if ((new_ref_buffer = set_mset(ptr_mset_sentinel, mset_string, &ref_buffer, &nref_buffer_len, ns, ptr_indent, n_ref_format)) == NULL) {
	error = 801;
	break;
      }

      ref_buffer = new_ref_buffer;

      if ((new_ref_buffer = format_simple(&ref_buffer, &nref_buffer_len, ptr_bibconns, dbires, dbi_style_res, n_intext, REFDB_MEDIATYPE, MEDIATYPEPRECEEDING, MEDIATYPEFOLLOWING, "mediatype", ns, ptr_indent, n_ref_format)) == NULL) {
	error = 835;
	break;
      }

      if (ref_buffer[0]) {
	if ((new_ref = mstrcat(*ptr_ref, ref_buffer, ptr_ref_len, 0)) == NULL) {
	  error = 801;
	  break;
	}

	*ptr_ref = new_ref;
      }
      ref_buffer[0] = '\0';
    }
    /*---------------------------------------------------------------*/
    else if (strcmp(item_type, "NUMVOLUMES") == 0) {
      if (!n_intext) {
	if (strcmp(pubtype, "CHAP") == 0 || strcmp(pubtype, "BOOK") == 0) {
	  strcpy(mset_string, "book");
	}
	else {
	  strcpy(mset_string, "journal");
	}
      }
      else {
	strcpy(mset_string, "intext");
      }
      if ((new_ref_buffer = set_mset(ptr_mset_sentinel, mset_string, &ref_buffer, &nref_buffer_len, ns, ptr_indent, n_ref_format)) == NULL) {
	error = 801;
	break;
      }

      ref_buffer = new_ref_buffer;

      if ((new_ref_buffer = format_simple(&ref_buffer, &nref_buffer_len, ptr_bibconns, dbires, dbi_style_res, n_intext, REFDB_NUMVOLUMES, NUMVOLUMESPRECEEDING, NUMVOLUMESFOLLOWING, "numvolumes", ns, ptr_indent, n_ref_format)) == NULL) {
	error = 835;
	break;
      }

      if (ref_buffer[0]) {
	if ((new_ref = mstrcat(*ptr_ref, ref_buffer, ptr_ref_len, 0)) == NULL) {
	  error = 801;
	  break;
	}

	*ptr_ref = new_ref;
      }
      ref_buffer[0] = '\0';
    }
    /*---------------------------------------------------------------*/
    else if (strcmp(item_type, "EDITION") == 0) {
      if (!n_intext) {
	if (strcmp(pubtype, "CHAP") == 0 || strcmp(pubtype, "BOOK") == 0) {
	  strcpy(mset_string, "book");
	}
	else {
	  strcpy(mset_string, "journal");
	}
      }
      else {
	strcpy(mset_string, "intext");
      }
      if ((new_ref_buffer = set_mset(ptr_mset_sentinel, mset_string, &ref_buffer, &nref_buffer_len, ns, ptr_indent, n_ref_format)) == NULL) {
	error = 801;
	break;
      }

      ref_buffer = new_ref_buffer;

      if ((new_ref_buffer = format_simple(&ref_buffer, &nref_buffer_len, ptr_bibconns, dbires, dbi_style_res, n_intext, REFDB_EDITION, EDITIONPRECEEDING, EDITIONFOLLOWING, "edition", ns, ptr_indent, n_ref_format)) == NULL) {
	error = 835;
	break;
      }

      if (ref_buffer[0]) {
	if ((new_ref = mstrcat(*ptr_ref, ref_buffer, ptr_ref_len, 0)) == NULL) {
	  error = 801;
	  break;
	}

	*ptr_ref = new_ref;
      }
      ref_buffer[0] = '\0';
    }
    /*---------------------------------------------------------------*/
    else if (strcmp(item_type, "COMPUTER") == 0) {
      if (!n_intext) {
	if (strcmp(pubtype, "CHAP") == 0 || strcmp(pubtype, "BOOK") == 0) {
	  strcpy(mset_string, "book");
	}
	else {
	  strcpy(mset_string, "journal");
	}
      }
      else {
	strcpy(mset_string, "intext");
      }
      if ((new_ref_buffer = set_mset(ptr_mset_sentinel, mset_string, &ref_buffer, &nref_buffer_len, ns, ptr_indent, n_ref_format)) == NULL) {
	error = 801;
	break;
      }

      ref_buffer = new_ref_buffer;

      if ((new_ref_buffer = format_simple(&ref_buffer, &nref_buffer_len, ptr_bibconns, dbires, dbi_style_res, n_intext, REFDB_COMPUTER, COMPUTERPRECEEDING, COMPUTERFOLLOWING, "computer", ns, ptr_indent, n_ref_format)) == NULL) {
	error = 835;
	break;
      }

      if (ref_buffer[0]) {
	if ((new_ref = mstrcat(*ptr_ref, ref_buffer, ptr_ref_len, 0)) == NULL) {
	  error = 801;
	  break;
	}

	*ptr_ref = new_ref;
      }
      ref_buffer[0] = '\0';
    }
    /*---------------------------------------------------------------*/
    else if (strcmp(item_type, "CONFERENCELOCATION") == 0) {
      if (!n_intext) {
	if (strcmp(pubtype, "CHAP") == 0 || strcmp(pubtype, "BOOK") == 0) {
	  strcpy(mset_string, "book");
	}
	else {
	  strcpy(mset_string, "journal");
	}
      }
      else {
	strcpy(mset_string, "intext");
      }
      if ((new_ref_buffer = set_mset(ptr_mset_sentinel, mset_string, &ref_buffer, &nref_buffer_len, ns, ptr_indent, n_ref_format)) == NULL) {
	error = 801;
	break;
      }

      ref_buffer = new_ref_buffer;

      if ((new_ref_buffer = format_simple(&ref_buffer, &nref_buffer_len, ptr_bibconns, dbires, dbi_style_res, n_intext, REFDB_CONFERENCELOCATION, CONFERENCELOCATIONPRECEEDING, CONFERENCELOCATIONFOLLOWING, "conferencelocation", ns, ptr_indent, n_ref_format)) == NULL) {
	error = 835;
	break;
      }

      if (ref_buffer[0]) {
	if ((new_ref = mstrcat(*ptr_ref, ref_buffer, ptr_ref_len, 0)) == NULL) {
	  error = 801;
	  break;
	}

	*ptr_ref = new_ref;
      }
      ref_buffer[0] = '\0';
    }
    /*---------------------------------------------------------------*/
    else if (strcmp(item_type, "REGISTRYNUM") == 0) {
      if (!n_intext) {
	if (strcmp(pubtype, "CHAP") == 0 || strcmp(pubtype, "BOOK") == 0) {
	  strcpy(mset_string, "book");
	}
	else {
	  strcpy(mset_string, "journal");
	}
      }
      else {
	strcpy(mset_string, "intext");
      }
      if ((new_ref_buffer = set_mset(ptr_mset_sentinel, mset_string, &ref_buffer, &nref_buffer_len, ns, ptr_indent, n_ref_format)) == NULL) {
	error = 801;
	break;
      }

      ref_buffer = new_ref_buffer;

      if ((new_ref_buffer = format_simple(&ref_buffer, &nref_buffer_len, ptr_bibconns, dbires, dbi_style_res, n_intext, REFDB_REGISTRYNUM, REGISTRYNUMPRECEEDING, REGISTRYNUMFOLLOWING, "registrynum", ns, ptr_indent, n_ref_format)) == NULL) {
	error = 835;
	break;
      }

      if (ref_buffer[0]) {
	if ((new_ref = mstrcat(*ptr_ref, ref_buffer, ptr_ref_len, 0)) == NULL) {
	  error = 801;
	  break;
	}

	*ptr_ref = new_ref;
      }
      ref_buffer[0] = '\0';
    }
    /*---------------------------------------------------------------*/
    else if (strcmp(item_type, "CLASSIFICATION") == 0) {
      if (!n_intext) {
	if (strcmp(pubtype, "CHAP") == 0 || strcmp(pubtype, "BOOK") == 0) {
	  strcpy(mset_string, "book");
	}
	else {
	  strcpy(mset_string, "journal");
	}
      }
      else {
	strcpy(mset_string, "intext");
      }
      if ((new_ref_buffer = set_mset(ptr_mset_sentinel, mset_string, &ref_buffer, &nref_buffer_len, ns, ptr_indent, n_ref_format)) == NULL) {
	error = 801;
	break;
      }

      ref_buffer = new_ref_buffer;

      if ((new_ref_buffer = format_simple(&ref_buffer, &nref_buffer_len, ptr_bibconns, dbires, dbi_style_res, n_intext, REFDB_CLASSIFICATION, CLASSIFICATIONPRECEEDING, CLASSIFICATIONFOLLOWING, "classification", ns, ptr_indent, n_ref_format)) == NULL) {
	error = 835;
	break;
      }

      if (ref_buffer[0]) {
	if ((new_ref = mstrcat(*ptr_ref, ref_buffer, ptr_ref_len, 0)) == NULL) {
	  error = 801;
	  break;
	}

	*ptr_ref = new_ref;
      }
      ref_buffer[0] = '\0';
    }
    /*---------------------------------------------------------------*/
    else if (strcmp(item_type, "SECTION") == 0) {
      if (!n_intext) {
	if (strcmp(pubtype, "CHAP") == 0 || strcmp(pubtype, "BOOK") == 0) {
	  strcpy(mset_string, "book");
	}
	else {
	  strcpy(mset_string, "journal");
	}
      }
      else {
	strcpy(mset_string, "intext");
      }
      if ((new_ref_buffer = set_mset(ptr_mset_sentinel, mset_string, &ref_buffer, &nref_buffer_len, ns, ptr_indent, n_ref_format)) == NULL) {
	error = 801;
	break;
      }

      ref_buffer = new_ref_buffer;

      if ((new_ref_buffer = format_simple(&ref_buffer, &nref_buffer_len, ptr_bibconns, dbires, dbi_style_res, n_intext, REFDB_SECTION, SECTIONPRECEEDING, SECTIONFOLLOWING, "section", ns, ptr_indent, n_ref_format)) == NULL) {
	error = 835;
	break;
      }

      if (ref_buffer[0]) {
	if ((new_ref = mstrcat(*ptr_ref, ref_buffer, ptr_ref_len, 0)) == NULL) {
	  error = 801;
	  break;
	}

	*ptr_ref = new_ref;
      }
      ref_buffer[0] = '\0';
    }
    /*---------------------------------------------------------------*/
    else if (strcmp(item_type, "PAMPHLETNUM") == 0) {
      if (!n_intext) {
	if (strcmp(pubtype, "CHAP") == 0 || strcmp(pubtype, "BOOK") == 0) {
	  strcpy(mset_string, "book");
	}
	else {
	  strcpy(mset_string, "journal");
	}
      }
      else {
	strcpy(mset_string, "intext");
      }
      if ((new_ref_buffer = set_mset(ptr_mset_sentinel, mset_string, &ref_buffer, &nref_buffer_len, ns, ptr_indent, n_ref_format)) == NULL) {
	error = 801;
	break;
      }

      ref_buffer = new_ref_buffer;

      if ((new_ref_buffer = format_simple(&ref_buffer, &nref_buffer_len, ptr_bibconns, dbires, dbi_style_res, n_intext, REFDB_PAMPHLETNUM, PAMPHLETNUMPRECEEDING, PAMPHLETNUMFOLLOWING, "pamphletnum", ns, ptr_indent, n_ref_format)) == NULL) {
	error = 835;
	break;
      }

      if (ref_buffer[0]) {
	if ((new_ref = mstrcat(*ptr_ref, ref_buffer, ptr_ref_len, 0)) == NULL) {
	  error = 801;
	  break;
	}

	*ptr_ref = new_ref;
      }
      ref_buffer[0] = '\0';
    }
    /*---------------------------------------------------------------*/
    else if (strcmp(item_type, "CHAPTERNUM") == 0) {
      if (!n_intext) {
	if (strcmp(pubtype, "CHAP") == 0 || strcmp(pubtype, "BOOK") == 0) {
	  strcpy(mset_string, "book");
	}
	else {
	  strcpy(mset_string, "journal");
	}
      }
      else {
	strcpy(mset_string, "intext");
      }
      if ((new_ref_buffer = set_mset(ptr_mset_sentinel, mset_string, &ref_buffer, &nref_buffer_len, ns, ptr_indent, n_ref_format)) == NULL) {
	error = 801;
	break;
      }

      ref_buffer = new_ref_buffer;

      if ((new_ref_buffer = format_simple(&ref_buffer, &nref_buffer_len, ptr_bibconns, dbires, dbi_style_res, n_intext, REFDB_CHAPTERNUM, CHAPTERNUMPRECEEDING, CHAPTERNUMFOLLOWING, "chapternum", ns, ptr_indent, n_ref_format)) == NULL) {
	error = 835;
	break;
      }

      if (ref_buffer[0]) {
	if ((new_ref = mstrcat(*ptr_ref, ref_buffer, ptr_ref_len, 0)) == NULL) {
	  error = 801;
	  break;
	}

	*ptr_ref = new_ref;
      }
      ref_buffer[0] = '\0';
    }



    /*---------------------------------------------------------------*/
    else if (strcmp(item_type, "NOTES") == 0) {
      if (!n_intext) {
	if (strcmp(pubtype, "CHAP") == 0) {
	  strcpy(mset_string, "chapter");
	}
	else {
	  strcpy(mset_string, "article");
	}
      }
      else {
	strcpy(mset_string, "intext");
      }

      if ((new_ref_buffer = set_mset(ptr_mset_sentinel, mset_string, &ref_buffer, &nref_buffer_len, ns, ptr_indent, n_ref_format)) == NULL) {
	error = 801;
	break;
      }

      ref_buffer = new_ref_buffer;

      if ((new_ref_buffer = format_notes(&ref_buffer, &nref_buffer_len, ptr_bibconns, dbires, dbi_style_res, username, ns, ptr_indent, n_ref_format)) == NULL) {
	error = 836;
	break;
      }

      if (ref_buffer[0]) {
	if ((new_ref = mstrcat(*ptr_ref, ref_buffer, ptr_ref_len, 0)) == NULL) {
	  error = 801;
	  break;
	}

	*ptr_ref = new_ref;
      }
      ref_buffer[0] = '\0';
    }
    /*---------------------------------------------------------------*/
    else if (strcmp(item_type, "SEPARATOR") == 0) {
      if ((new_ref_buffer = format_separator(ref_buffer, &nref_buffer_len, ptr_bibconns, n_separator_id, n_ref_format)) == NULL) {
	error = 837;
	break;
      }

      ref_buffer = new_ref_buffer;
      if (ref_buffer[0]) {
	if ((new_ref = mstrcat(*ptr_ref, ref_buffer, ptr_ref_len, 0)) == NULL) {
	  error = 801;
	  break;
	}

	*ptr_ref = new_ref;
      }
      ref_buffer[0] = '\0';
    }
  } /* end while */

  strcpy(mset_string, "");

  if ((new_ref = set_mset(ptr_mset_sentinel, mset_string, ptr_ref, ptr_ref_len, ns, ptr_indent, n_ref_format)) == NULL) {
    error = 801;
  }

  *ptr_ref = new_ref;

  dbi_result_free(dbires_pos);
  unload_style();
  free(sql_command);
  free(ref_buffer);

  return error;
}



