#!/usr/bin/perl
# generates status message listings for C, Perl, DocBook
# usage: statgen.pl c|perl|docbook > outfile
# if no argument is given, statgen.pl creates tab-separated output

## the status message array

@refstat = (
	    # ok
	    ["000", "ok"],
	    ["001",  "error"],

	    ## 1xx: protocol errors
	    ["100", "void"],
	    ["101", "incorrect scramble string"],
	    ["102", "client and server protocols do not match"],
	    ["103", "invalid client request"],
	    ["104", "incomplete client command"],
	    ["105", "missing client command"],
	    ["106", "missing client command option"],
	    ["107", "unknown client command option"],
	    ["108", "could not descramble password"],
	    ["109", "timeout while reading"],
	    ["110", "timeout while writing"],
	    ["111", "missing client command argument"],
	    ["112", "client aborted command"],

	    ## 2xx: database errors
	    ["200", "void"],
	    ["201", "main database is missing"],
	    ["202", "could not open main database"],
	    ["203", "main database is too old or corrupt"],
	    ["204", "could not open reference database"],
	    ["205", "could not connect to database server"],
	    ["206", "main database version is not supported"],
	    ["207", "could not create result from database query"],
	    ["208", "could not retrieve reference database metadata"],
	    ["209", "could not create reference database"],
	    ["210", "could not create reference database metadata"],
	    ["211", "create t_meta failed"],
	    ["212", "create t_refdb failed"],
	    ["213", "create t_author failed"],
	    ["214", "create t_keyword failed"],
	    ["215", "create t_periodical failed"],
	    ["216", "create t_note failed"],
	    ["217", "create t_user failed"],
	    ["218", "create t_xauthor failed"],
	    ["219", "create t_xkeyword failed"],
	    ["220", "create t_xuser failed"],
	    ["221", "create t_xnote failed"],
	    ["222", "could not create user group"],
	    ["223", "could not grant user permissions"],
	    ["224", "access control not supported"],
	    ["225", "not a RefDB database"],
	    ["226", "database does not exist"],
	    ["227", "begin transaction failed"],
	    ["228", "cannot lock tables"],
	    ["229", "failed to remove keyword"],
	    ["230", "failed to remove author"],
	    ["231", "failed to remove periodical"],
	    ["232", "failed to update main reference data"],
	    ["233", "inserting reference data failed"],
	    ["234", "select failed"],
	    ["235", "database successfully created"],
	    ["236", "assume localhost as host"],
	    ["237", "grant user permissions successful"],
	    ["238", "revoke user permissions successful"],
	    ["239", "could not revoke user permissions"],
	    ["240", "switched to database"],
	    ["241", "failed to access style data"],
	    ["242", "create temporary table failed"],
	    ["243", "delete temporary table failed"],
	    ["244", "incomplete reference data"],
	    ["245", "failed to remove note xlink"],
	    ["246", "failed to delete main note data"],
	    ["247", "failed to remove user"],
	    ["248", "failed to delete main reference data"],
	    ["249", "failed to delete database"],
	    ["250", "could not delete user group"],
	    ["251", "database successfully deleted"],
	    ["252", "personal interest list is empty"],
	    ["253", "failed to detach dataset from user"],
	    ["254", "sucessfully detached dataset from user"],
	    ["255", "failed to attach dataset to user"],
	    ["256", "sucessfully attached dataset to user"],
	    ["257", "create t_link failed"],
	    ["258", "create t_xlink failed"],
	    ["259", "failed to remove ulink"],
	    ["260", "failed to update journal names"],
	    ["261", "failed to create citation key"],
	    ["262", "failed to create personal list"],
	    ["263", "successfully created personal list"],
	    ["264", "failed to delete personal list"],
	    ["265", "successfully deleted personal list"],
	    ["266", "personal list not owned by current user"],
	    ["267", "personal list does not exist"],
	    ["268", "create t_temp_xdup failed"],
	    ["269", "style not found"],

	    ## 3xx: query errors
	    ["300", "void"],
	    ["301", "missing argument"],
	    ["302", "unknown output format"],

	    ## 4xx: data transmission status messages
	    ["400", "void"],
	    ["401", "no more data available"],
	    ["402", "finished transferring data"],
	    ["403", "chunk added successfully"],
	    ["404", "finished transferring dataset"],
	    ["405", "finished adding dataset"],
	    ["406", "citation key"],
	    ["407", "refused to overwrite dataset"],
	    ["408", "dataset added successfully"],
	    ["409", "numerical id ignored"],
	    ["410", "numerical id nonexistent"],
	    ["411", "citation key nonexistent"],
	    ["412", "ID and citation key missing"],
	    ["413", "dataset updated successfully"],
	    ["414", "failed to add dataset"],
	    ["415", "missing link target"],
	    ["416", "incorrect link type"],
	    ["417", "dataset not found"],
	    ["418", "link already exists"],
	    ["419", "dataset removed successfully"],
	    ["420", "failed to remove dataset"],
	    ["421", "link added successfully"],
	    ["422", "only owner can fiddle with dataset"],
	    ["423", "dataset is still in use"],
	    ["424", "dataset is already attached to user"],
	    ["425", "periodical name changed successfully"],
	    ["426", "reference type changed"],

	    # 7xx: internal server warnings
	    ["700", "void"],
	    ["701", "failed to initialize character set conversion"],
	    ["702", "character set conversion failed"],
	    ["703", "remote administration disabled"],
	    ["704", "administration is not restricted"],
	    ["705", "administration not permitted"],
	    ["706", "administration permitted"],
	    ["707", "process ID"],
	    ["708", "application server stop submitted"],
	    ["709", "set server IP submitted"],
	    ["710", "set timeout submitted"],
	    ["711", "set logfile submitted"],
	    ["712", "set logdest submitted"],
	    ["713", "set loglevel submitted"],

	    ## 8xx: internal server errors
	    ["800", "void"],
	    ["801", "out of memory"],
	    ["802", "failed to load cgi templates"],
	    ["803", "command partially processed, aborted after unrecoverable error"],
	    ["804", "suffix pool exhausted"],
	    ["805", "REFNUMBER formatting failed"],
	    ["806", "AUTHORLIST formatting failed"],
	    ["807", "EDITORLIST formatting failed"],
	    ["808", "SEDITORLIST formatting failed"],
	    ["809", "PUBDATE formatting failed"],
	    ["810", "PUBDATESEC formatting failed"],
	    ["811", "TITLE formatting failed"],
	    ["812", "BOOKTITLE formatting failed"],
	    ["813", "SERIESTITLE formatting failed"],
	    ["814", "JOURNALNAME formatting failed"],
	    ["815", "VOLUME formatting failed"],
	    ["816", "ISSUE formatting failed"],
	    ["817", "PAGES formatting failed"],
	    ["818", "PUBLISHER formatting failed"],
	    ["819", "PUBPLACE formatting failed"],
	    ["820", "SERIAL formatting failed"],
	    ["821", "ADDRESS formatting failed"],
	    ["822", "URL formatting failed"],
	    ["823", "USERDEF1 formatting failed"],
	    ["824", "USERDEF2 formatting failed"],
	    ["825", "USERDEF3 formatting failed"],
	    ["826", "USERDEF4 formatting failed"],
	    ["827", "USERDEF5 formatting failed"],
	    ["828", "MISC1 formatting failed"],
	    ["829", "MISC2 formatting failed"],
	    ["830", "MISC3 formatting failed"],
	    ["831", "LINK1 formatting failed"],
	    ["832", "LINK2 formatting failed"],
	    ["833", "LINK3 formatting failed"],
	    ["834", "LINK4 formatting failed"],
	    ["835", "ABSTRACT formatting failed"],
	    ["836", "NOTES formatting failed"],
	    ["837", "SEPARATOR formatting failed"],
	    ["838", "remote administration failed"],
	    ["839", "child->parent communication failure"],
	    ["840", "FIFO write error"],
	    ["841", "unknown command"],
	    ["842", "failed to load style"],

	    ## result summary
	    ["999", "summary"]
	    );


# check command line argument and call appropriate output sub
if ($ARGV[0] eq "c") {
    write_c();
}
elsif ($ARGV[0] eq "perl") {
    write_perl();
}
elsif ($ARGV[0] eq "docbook") {
    write_docbook();
}
elsif ($ARGV[0] eq "docbook-table") {
    write_docbook_table();
}
else {
    write_tab();
}

exit 0;

# generate output for the C language. Output is suitable as an include file
sub write_c {
    print "/* This definition is generated by statgen.pl. DO NOT modify this definition as all changes will be lost. Change statgen.pl instead */\n\n";

    print "struct refdb_status refstat[] = {\n";

    foreach my $item (@refstat) {
	printf ("{%d, \"%s\", \"%s\"},\n", @{$item}[0], @{$item}[0], @{$item}[1]);
    }
    print "};\n\n";

    print "/* end statgen.pl-generated code */\n";
}

# generate output for the Perl language.
sub write_perl {
    print "# This definition is generated by statgen.pl. DO NOT modify this definition as all changes will be lost. Change statgen.pl instead\n\n";

    print "\%{\$self->{refstat}} = (\n";

    foreach my $item (@refstat) {
	print "\"@{$item}[0]\" => \"@{$item}[1]\",\n";
    }

    print ");\n\n";
    print "# end statgen.pl-generated code\n";
}

# generate output for DocBook. The code creates a variablelist which can be included into a DocBook document as an external entity. Looks worse than a table, but the linkends work
sub write_docbook {
    print "<!-- This variablelist is generated by statgen.pl. DO NOT modify this table as all changes will be lost. Change statgen.pl instead -->\n\n";

    print "<variablelist>\n";

    foreach my $item (@refstat) {
	print "  <varlistentry id=\"cs-status-@{$item}[0]\">\n";
	print "    <term>@{$item}[0]</term>\n";
	print "    <listitem>\n";
	print "      <para>@{$item}[1]</para>\n";
	print "    </listitem>\n";
	print "  </varlistentry>\n";
    }

    print "</variablelist>\n\n";

    print "<!-- end statgen.pl-generated code -->\n";
}

# generate output for DocBook. The code creates a table with two columns which can be included into a DocBook document as an external entity. Unfortunately, the linkends within the table don't work
sub write_docbook_table {
    print "<!-- This table is generated by statgen.pl. DO NOT modify this table as all changes will be lost. Change statgen.pl instead -->\n\n";

    print "<informaltable>\n";
    print "  <tgroup cols=\"2\">\n";
    print "    <thead>\n";
    print "      <row>\n";
    print "        <entry>status code</entry>\n";
    print "        <entry>comment</entry>\n";
    print "      </row>\n";
    print "    </thead>\n";
    print "    <tbody>\n";

    foreach my $item (@refstat) {
	print "      <row id=\"cs-status-@{$item}[0]\">\n";
	print "        <entry>@{$item}[0]</entry>\n";
	print "        <entry>@{$item}[1]</entry>\n";
	print "      </row>\n";
    }

    print "    </tbody>\n";
    print "  </tgroup>\n";
    print "</informaltable>\n\n";

    print "<!-- end statgen.pl-generated code -->\n";
}

# generate tab-separated output.
sub write_tab {
    print "# This definition is generated by statgen.pl. DO NOT modify this definition as all changes will be lost. Change statgen.pl instead\n\n";

    foreach my $item (@refstat) {
	print "@{$item}[0]\t@{$item}[1]\n";
    }

    print "\n# end statgen.pl-generated code\n";
}

