/* boost random/detail/sobol_table.hpp header file
 *
 * Copyright Justinas Vygintas Daugmaudis 2010-2018
 * Distributed under the Boost Software License, Version 1.0. (See
 * accompanying file LICENSE_1_0.txt or copy at
 * http://www.boost.org/LICENSE_1_0.txt)
 */

#ifndef BOOST_RANDOM_DETAIL_SOBOL_TABLE_HPP
#define BOOST_RANDOM_DETAIL_SOBOL_TABLE_HPP

#include <boost/config.hpp>

namespace boost {
namespace random {

namespace detail {
namespace qrng_tables {

// Maximum allowed space dimension
#define BOOST_RANDOM_SOBOL_MAX_DIMENSION 3667

// Data on the primitive binary polynomials (a) and the corresponding
// starting values m, for Sobol sequences in up to 21201 dimensions,
// taken from:
//     S. Joe and F. Y. Kuo, Constructing Sobol sequences with better two-dimensional projections,
//     SIAM J. Sci. Comput. 30, 2635-2654 (2008).
// and were derived from the file new-joe-kuo-6.21201 provided on F. Kuo's website
// https://web.archive.org/web/20170817015227/http://web.maths.unsw.edu.au/~fkuo/sobol/
//
// The tables were cut-off at 3667 dimensions for practical reasons. Full 21201 dimension
// table would result in a header in 1.7Mb range. Here we cut-off the full table to the
// polynomial that still fits to 2^16-1, i.e., to unsigned short. This gives us respectable 3667
// dimensions and sizes down the header to around 230Kb.
struct sobol
{
  BOOST_STATIC_CONSTANT(unsigned int, max_dimension = BOOST_RANDOM_SOBOL_MAX_DIMENSION);
  BOOST_STATIC_CONSTANT(unsigned int, num_polynomials = max_dimension - 1);

  // log2(polynomial(num_polynomials - 1)), i.e., integer log2 of the last polynomial in the table
  BOOST_STATIC_CONSTANT(unsigned int, max_degree = 15);

  typedef unsigned short value_type;

  static value_type polynomial(std::size_t n)
  {
    // successive primitive binary-coefficient polynomials p(z)
    //   = a_0 + a_1 z + a_2 z^2 + ... a_31 z^31, where a_i is the
    //     i-th bit of sobol_a[j] for the j-th polynomial.
    static const value_type sobol_a[num_polynomials] = {
      3,7,11,13,19,25,37,41,47,55,
      59,61,67,91,97,103,109,115,131,137,
      143,145,157,167,171,185,191,193,203,211,
      213,229,239,241,247,253,285,299,301,333,
      351,355,357,361,369,391,397,425,451,463,
      487,501,529,539,545,557,563,601,607,617,
      623,631,637,647,661,675,677,687,695,701,
      719,721,731,757,761,787,789,799,803,817,
      827,847,859,865,875,877,883,895,901,911,
      949,953,967,971,973,981,985,995,1001,1019,
      1033,1051,1063,1069,1125,1135,1153,1163,1221,1239,
      1255,1267,1279,1293,1305,1315,1329,1341,1347,1367,
      1387,1413,1423,1431,1441,1479,1509,1527,1531,1555,
      1557,1573,1591,1603,1615,1627,1657,1663,1673,1717,
      1729,1747,1759,1789,1815,1821,1825,1849,1863,1869,
      1877,1881,1891,1917,1933,1939,1969,2011,2035,2041,
      2053,2071,2091,2093,2119,2147,2149,2161,2171,2189,
      2197,2207,2217,2225,2255,2257,2273,2279,2283,2293,
      2317,2323,2341,2345,2363,2365,2373,2377,2385,2395,
      2419,2421,2431,2435,2447,2475,2477,2489,2503,2521,
      2533,2551,2561,2567,2579,2581,2601,2633,2657,2669,
      2681,2687,2693,2705,2717,2727,2731,2739,2741,2773,
      2783,2793,2799,2801,2811,2819,2825,2833,2867,2879,
      2881,2891,2905,2911,2917,2927,2941,2951,2955,2963,
      2965,2991,2999,3005,3017,3035,3037,3047,3053,3083,
      3085,3097,3103,3159,3169,3179,3187,3205,3209,3223,
      3227,3229,3251,3263,3271,3277,3283,3285,3299,3305,
      3319,3331,3343,3357,3367,3373,3393,3399,3413,3417,
      3427,3439,3441,3475,3487,3497,3515,3517,3529,3543,
      3547,3553,3559,3573,3589,3613,3617,3623,3627,3635,
      3641,3655,3659,3669,3679,3697,3707,3709,3713,3731,
      3743,3747,3771,3791,3805,3827,3833,3851,3865,3889,
      3895,3933,3947,3949,3957,3971,3985,3991,3995,4007,
      4013,4021,4045,4051,4069,4073,4179,4201,4219,4221,
      4249,4305,4331,4359,4383,4387,4411,4431,4439,4449,
      4459,4485,4531,4569,4575,4621,4663,4669,4711,4723,
      4735,4793,4801,4811,4879,4893,4897,4921,4927,4941,
      4977,5017,5027,5033,5127,5169,5175,5199,5213,5223,
      5237,5287,5293,5331,5391,5405,5453,5523,5573,5591,
      5597,5611,5641,5703,5717,5721,5797,5821,5909,5913,
      5955,5957,6005,6025,6061,6067,6079,6081,6231,6237,
      6289,6295,6329,6383,6427,6453,6465,6501,6523,6539,
      6577,6589,6601,6607,6631,6683,6699,6707,6761,6795,
      6865,6881,6901,6923,6931,6943,6999,7057,7079,7103,
      7105,7123,7173,7185,7191,7207,7245,7303,7327,7333,
      7355,7365,7369,7375,7411,7431,7459,7491,7505,7515,
      7541,7557,7561,7701,7705,7727,7749,7761,7783,7795,
      7823,7907,7953,7963,7975,8049,8089,8123,8125,8137,
      8219,8231,8245,8275,8293,8303,8331,8333,8351,8357,
      8367,8379,8381,8387,8393,8417,8435,8461,8469,8489,
      8495,8507,8515,8551,8555,8569,8585,8599,8605,8639,
      8641,8647,8653,8671,8675,8689,8699,8729,8741,8759,
      8765,8771,8795,8797,8825,8831,8841,8855,8859,8883,
      8895,8909,8943,8951,8955,8965,8999,9003,9031,9045,
      9049,9071,9073,9085,9095,9101,9109,9123,9129,9137,
      9143,9147,9185,9197,9209,9227,9235,9247,9253,9257,
      9277,9297,9303,9313,9325,9343,9347,9371,9373,9397,
      9407,9409,9415,9419,9443,9481,9495,9501,9505,9517,
      9529,9555,9557,9571,9585,9591,9607,9611,9621,9625,
      9631,9647,9661,9669,9679,9687,9707,9731,9733,9745,
      9773,9791,9803,9811,9817,9833,9847,9851,9863,9875,
      9881,9905,9911,9917,9923,9963,9973,10003,10025,10043,
      10063,10071,10077,10091,10099,10105,10115,10129,10145,10169,
      10183,10187,10207,10223,10225,10247,10265,10271,10275,10289,
      10299,10301,10309,10343,10357,10373,10411,10413,10431,10445,
      10453,10463,10467,10473,10491,10505,10511,10513,10523,10539,
      10549,10559,10561,10571,10581,10615,10621,10625,10643,10655,
      10671,10679,10685,10691,10711,10739,10741,10755,10767,10781,
      10785,10803,10805,10829,10857,10863,10865,10875,10877,10917,
      10921,10929,10949,10967,10971,10987,10995,11009,11029,11043,
      11045,11055,11063,11075,11081,11117,11135,11141,11159,11163,
      11181,11187,11225,11237,11261,11279,11297,11307,11309,11327,
      11329,11341,11377,11403,11405,11413,11427,11439,11453,11461,
      11473,11479,11489,11495,11499,11533,11545,11561,11567,11575,
      11579,11589,11611,11623,11637,11657,11663,11687,11691,11701,
      11747,11761,11773,11783,11795,11797,11817,11849,11855,11867,
      11869,11873,11883,11919,11921,11927,11933,11947,11955,11961,
      11999,12027,12029,12037,12041,12049,12055,12095,12097,12107,
      12109,12121,12127,12133,12137,12181,12197,12207,12209,12239,
      12253,12263,12269,12277,12287,12295,12309,12313,12335,12361,
      12367,12391,12409,12415,12433,12449,12469,12479,12481,12499,
      12505,12517,12527,12549,12559,12597,12615,12621,12639,12643,
      12657,12667,12707,12713,12727,12741,12745,12763,12769,12779,
      12781,12787,12799,12809,12815,12829,12839,12857,12875,12883,
      12889,12901,12929,12947,12953,12959,12969,12983,12987,12995,
      13015,13019,13031,13063,13077,13103,13137,13149,13173,13207,
      13211,13227,13241,13249,13255,13269,13283,13285,13303,13307,
      13321,13339,13351,13377,13389,13407,13417,13431,13435,13447,
      13459,13465,13477,13501,13513,13531,13543,13561,13581,13599,
      13605,13617,13623,13637,13647,13661,13677,13683,13695,13725,
      13729,13753,13773,13781,13785,13795,13801,13807,13825,13835,
      13855,13861,13871,13883,13897,13905,13915,13939,13941,13969,
      13979,13981,13997,14027,14035,14037,14051,14063,14085,14095,
      14107,14113,14125,14137,14145,14151,14163,14193,14199,14219,
      14229,14233,14243,14277,14287,14289,14295,14301,14305,14323,
      14339,14341,14359,14365,14375,14387,14411,14425,14441,14449,
      14499,14513,14523,14537,14543,14561,14579,14585,14593,14599,
      14603,14611,14641,14671,14695,14701,14723,14725,14743,14753,
      14759,14765,14795,14797,14803,14831,14839,14845,14855,14889,
      14895,14909,14929,14941,14945,14951,14963,14965,14985,15033,
      15039,15053,15059,15061,15071,15077,15081,15099,15121,15147,
      15149,15157,15167,15187,15193,15203,15205,15215,15217,15223,
      15243,15257,15269,15273,15287,15291,15313,15335,15347,15359,
      15373,15379,15381,15391,15395,15397,15419,15439,15453,15469,
      15491,15503,15517,15527,15531,15545,15559,15593,15611,15613,
      15619,15639,15643,15649,15661,15667,15669,15681,15693,15717,
      15721,15741,15745,15765,15793,15799,15811,15825,15835,15847,
      15851,15865,15877,15881,15887,15899,15915,15935,15937,15955,
      15973,15977,16011,16035,16061,16069,16087,16093,16097,16121,
      16141,16153,16159,16165,16183,16189,16195,16197,16201,16209,
      16215,16225,16259,16265,16273,16299,16309,16355,16375,16381,
      16427,16441,16467,16479,16507,16553,16559,16571,16573,16591,
      16619,16627,16653,16659,16699,16707,16795,16797,16807,16813,
      16821,16853,16857,16881,16909,16983,16993,17023,17029,17053,
      17095,17099,17101,17123,17129,17135,17161,17185,17215,17277,
      17287,17301,17327,17353,17387,17389,17419,17475,17523,17619,
      17621,17631,17635,17659,17707,17721,17753,17775,17817,17823,
      17829,17847,17861,17879,17895,17907,17919,17935,17949,17959,
      17973,17991,18009,18019,18033,18043,18085,18117,18127,18139,
      18225,18255,18303,18343,18369,18405,18409,18415,18451,18457,
      18491,18499,18523,18529,18535,18559,18563,18659,18717,18733,
      18745,18793,18807,18823,18857,18895,18913,18997,19045,19067,
      19073,19079,19083,19107,19145,19165,19193,19255,19273,19291,
      19307,19309,19315,19321,19333,19351,19361,19371,19385,19403,
      19405,19413,19423,19441,19451,19465,19485,19519,19527,19531,
      19541,19581,19597,19621,19645,19653,19665,19671,19693,19743,
      19761,19781,19791,19793,19829,19855,19885,19891,19905,19963,
      19969,19989,20023,20035,20041,20049,20075,20077,20099,20123,
      20179,20197,20201,20207,20253,20309,20319,20329,20335,20383,
      20393,20407,20411,20459,20487,20511,20517,20573,20641,20693,
      20713,20781,20819,20825,20831,20861,20875,20889,20901,20913,
      20945,20955,20971,20973,20981,20991,20997,21007,21037,21093,
      21131,21145,21155,21169,21187,21189,21201,21223,21285,21289,
      21339,21403,21405,21415,21433,21439,21447,21489,21507,21519,
      21527,21557,21561,21575,21599,21627,21645,21663,21691,21725,
      21729,21785,21807,21815,21881,21887,21891,21893,21905,21933,
      21953,21971,21993,22007,22029,22037,22057,22063,22065,22171,
      22187,22189,22195,22209,22215,22221,22263,22315,22317,22335,
      22361,22371,22397,22419,22447,22461,22467,22469,22487,22531,
      22561,22579,22581,22591,22593,22677,22681,22691,22703,22749,
      22759,22763,22783,22863,22911,22927,22935,22941,22945,22951,
      22965,23007,23017,23071,23077,23099,23101,23107,23109,23113,
      23157,23221,23233,23251,23253,23257,23311,23319,23339,23353,
      23395,23401,23415,23449,23491,23493,23521,23531,23559,23563,
      23577,23601,23625,23645,23673,23683,23713,23743,23745,23755,
      23757,23781,23813,23837,23859,23919,23949,23957,23971,23977,
      23995,24015,24067,24079,24091,24109,24135,24189,24193,24217,
      24279,24283,24295,24309,24327,24345,24355,24381,24387,24389,
      24417,24427,24437,24471,24543,24589,24597,24623,24637,24679,
      24683,24713,24733,24747,24755,24761,24789,24841,24849,24877,
      24889,24897,24957,24991,24997,25007,25019,25069,25077,25139,
      25187,25199,25213,25229,25247,25253,25257,25271,25303,25307,
      25309,25331,25379,25393,25399,25435,25453,25461,25481,25489,
      25505,25535,25583,25609,25623,25665,25671,25739,25759,25831,
      25845,25857,25867,25911,25915,25925,25947,26001,26029,26041,
      26047,26069,26095,26103,26119,26125,26147,26171,26205,26219,
      26243,26263,26279,26283,26293,26329,26335,26385,26395,26443,
      26463,26473,26487,26497,26531,26577,26641,26653,26707,26743,
      26771,26783,26789,26793,26821,26879,26905,26927,26987,26995,
      27023,27037,27041,27051,27113,27137,27183,27217,27227,27239,
      27243,27245,27253,27267,27287,27315,27317,27327,27339,27369,
      27375,27395,27415,27435,27443,27449,27463,27467,27497,27517,
      27521,27533,27575,27589,27607,27617,27629,27641,27695,27709,
      27735,27763,27829,27833,27841,27847,27877,27913,27927,27947,
      27987,28003,28005,28009,28067,28081,28091,28093,28101,28169,
      28199,28205,28211,28225,28243,28283,28289,28295,28309,28335,
      28355,28379,28381,28409,28417,28437,28457,28465,28475,28495,
      28503,28561,28615,28633,28639,28649,28677,28701,28723,28797,
      28841,28859,28873,28879,28897,28947,28949,28953,28977,28983,
      28989,29035,29083,29089,29109,29151,29157,29175,29179,29215,
      29233,29243,29263,29287,29363,29377,29389,29407,29413,29425,
      29431,29443,29449,29479,29483,29581,29587,29605,29629,29649,
      29695,29715,29717,29775,29787,29803,29805,29867,29875,29901,
      29919,29929,29949,29979,29985,30071,30075,30105,30115,30141,
      30159,30161,30187,30197,30265,30279,30291,30293,30303,30307,
      30313,30367,30371,30383,30417,30443,30457,30475,30537,30551,
      30573,30579,30631,30645,30663,30675,30677,30741,30757,30769,
      30781,30829,30923,30925,30937,30959,30999,31015,31053,31065,
      31087,31089,31099,31105,31111,31153,31177,31191,31197,31235,
      31259,31275,31285,31295,31307,31317,31361,31373,31415,31419,
      31427,31457,31477,31523,31567,31569,31581,31609,31631,31649,
      31659,31673,31699,31715,31729,31749,31783,31789,31833,31883,
      31891,31893,31907,31927,31939,31953,31993,31999,32001,32021,
      32055,32069,32115,32121,32145,32151,32167,32179,32199,32205,
      32233,32251,32253,32269,32281,32303,32353,32373,32383,32413,
      32427,32467,32483,32485,32521,32545,32575,32589,32597,32625,
      32651,32653,32671,32709,32721,32743,32771,32785,32791,32813,
      32821,32863,32887,32897,32903,32915,32933,32963,32975,32989,
      32999,33013,33025,33045,33061,33111,33117,33121,33133,33157,
      33185,33191,33209,33227,33229,33247,33277,33299,33339,33349,
      33407,33417,33423,33435,33483,33497,33559,33563,33579,33587,
      33607,33613,33631,33635,33641,33649,33675,33689,33711,33725,
      33733,33745,33817,33827,33839,33841,33847,33895,33901,33913,
      33923,33943,33953,33973,34015,34039,34045,34077,34081,34087,
      34099,34119,34123,34143,34161,34171,34177,34189,34211,34225,
      34245,34249,34267,34285,34291,34313,34321,34333,34347,34389,
      34393,34405,34429,34433,34473,34479,34487,34499,34523,34559,
      34571,34573,34581,34601,34609,34667,34693,34697,34703,34731,
      34733,34739,34751,34783,34801,34817,34871,34889,34909,34913,
      34937,34947,34959,34997,35015,35033,35077,35081,35095,35111,
      35173,35225,35247,35279,35281,35293,35309,35327,35385,35413,
      35427,35429,35441,35451,35463,35467,35487,35503,35505,35549,
      35595,35597,35643,35651,35693,35699,35729,35741,35777,35787,
      35797,35813,35825,35873,35879,35911,35925,35939,35945,35975,
      35987,36003,36009,36027,36041,36065,36103,36107,36133,36163,
      36177,36187,36223,36229,36233,36251,36257,36287,36299,36301,
      36325,36329,36335,36363,36383,36411,36433,36439,36467,36469,
      36495,36503,36507,36513,36543,36545,36563,36581,36603,36623,
      36647,36651,36665,36709,36727,36733,36773,36809,36817,36833,
      36875,36889,36895,36901,36919,36925,36931,36951,36961,36973,
      36981,37001,37009,37019,37037,37125,37129,37143,37147,37149,
      37185,37197,37243,37273,37283,37289,37297,37309,37327,37345,
      37379,37393,37403,37415,37427,37439,37453,37459,37499,37511,
      37525,37539,37559,37577,37597,37621,37625,37651,37681,37701,
      37705,37719,37747,37759,37763,37789,37793,37813,37835,37855,
      37871,37873,37883,37903,37931,37941,37963,37971,38013,38035,
      38041,38053,38057,38075,38103,38107,38113,38119,38133,38143,
      38151,38165,38185,38193,38205,38213,38241,38251,38281,38299,
      38317,38367,38377,38419,38421,38449,38455,38461,38467,38503,
      38521,38551,38573,38593,38603,38623,38651,38699,38709,38733,
      38755,38805,38815,38819,38821,38825,38833,38875,38877,38899,
      38911,38921,38945,38983,39061,39065,39087,39099,39109,39127,
      39133,39179,39193,39205,39209,39215,39223,39235,39237,39277,
      39295,39305,39313,39323,39353,39359,39361,39371,39381,39395,
      39461,39503,39511,39515,39517,39521,39533,39545,39551,39575,
      39581,39595,39609,39623,39651,39675,39697,39707,39731,39745,
      39763,39775,39791,39799,39819,39827,39863,39869,39915,39935,
      39957,39983,39995,40017,40029,40039,40053,40057,40069,40127,
      40149,40165,40177,40183,40195,40277,40281,40291,40321,40331,
      40345,40351,40357,40381,40451,40465,40471,40481,40505,40519,
      40533,40537,40547,40553,40607,40611,40685,40691,40715,40723,
      40739,40751,40777,40783,40807,40859,40877,40883,40885,40909,
      40971,40995,40997,41007,41027,41051,41053,41063,41067,41069,
      41097,41105,41141,41165,41183,41211,41219,41231,41245,41255,
      41267,41269,41273,41287,41327,41339,41345,41375,41393,41413,
      41423,41441,41487,41537,41543,41571,41583,41625,41641,41669,
      41673,41687,41691,41709,41735,41739,41741,41753,41763,41797,
      41835,41843,41861,41871,41879,41889,41907,41921,41933,41941,
      41987,41989,41993,42067,42073,42113,42123,42137,42149,42159,
      42161,42173,42179,42203,42221,42241,42275,42277,42281,42319,
      42371,42407,42425,42445,42473,42481,42515,42543,42551,42577,
      42593,42611,42641,42651,42653,42695,42737,42749,42779,42795,
      42809,42847,42865,42893,42929,42939,42941,42959,42961,42973,
      42983,43051,43059,43065,43083,43085,43093,43113,43133,43137,
      43155,43167,43171,43177,43191,43215,43229,43233,43275,43283,
      43289,43337,43381,43431,43437,43497,43503,43511,43533,43541,
      43567,43569,43587,43599,43601,43617,43641,43677,43693,43705,
      43731,43779,43791,43805,43829,43833,43847,43875,43915,43929,
      43963,43973,43985,43991,43995,44019,44033,44039,44045,44073,
      44087,44123,44141,44147,44165,44175,44183,44231,44245,44255,
      44265,44285,44293,44305,44315,44321,44399,44451,44463,44489,
      44509,44519,44523,44537,44553,44573,44589,44601,44607,44639,
      44643,44657,44663,44685,44693,44707,44731,44759,44765,44775,
      44789,44801,44819,44831,44841,44847,44859,44867,44873,44903,
      44917,44943,44971,45011,45027,45053,45071,45083,45101,45109,
      45119,45141,45175,45203,45231,45251,45253,45265,45281,45299,
      45319,45323,45373,45399,45405,45429,45455,45463,45469,45473,
      45511,45535,45541,45579,45587,45589,45635,45641,45647,45695,
      45705,45723,45761,45791,45801,45807,45815,45829,45847,45863,
      45947,45953,45971,45989,46001,46007,46021,46039,46043,46045,
      46073,46081,46093,46147,46167,46189,46197,46207,46247,46271,
      46319,46327,46345,46365,46399,46407,46419,46421,46441,46455,
      46459,46485,46505,46523,46531,46551,46561,46567,46571,46585,
      46597,46601,46615,46619,46625,46631,46637,46643,46645,46667,
      46681,46715,46721,46731,46745,46767,46801,46823,46827,46879,
      46885,46917,46935,46955,46969,46979,46993,47005,47009,47027,
      47101,47183,47201,47221,47237,47241,47261,47265,47277,47295,
      47315,47317,47355,47363,47389,47399,47403,47431,47445,47449,
      47461,47479,47501,47535,47547,47555,47561,47569,47597,47603,
      47621,47633,47673,47705,47745,47751,47757,47763,47799,47831,
      47873,47893,47897,47907,47933,47939,47945,47953,47963,47975,
      47989,48017,48053,48063,48075,48077,48101,48105,48137,48179,
      48233,48251,48267,48269,48287,48297,48323,48349,48403,48421,
      48431,48487,48511,48515,48521,48569,48577,48595,48601,48613,
      48661,48675,48681,48695,48727,48731,48737,48747,48767,48771,
      48811,48881,48929,48935,48947,48949,48997,49025,49035,49059,
      49079,49091,49133,49141,49151,49153,49159,49171,49183,49189,
      49225,49267,49273,49285,49303,49309,49319,49337,49355,49365,
      49379,49403,49425,49441,49459,49471,49503,49527,49531,49533,
      49555,49573,49597,49603,49609,49627,49639,49645,49669,49673,
      49687,49709,49741,49749,49769,49817,49841,49873,49885,49909,
      49921,49933,49957,49981,50007,50011,50017,50035,50051,50075,
      50077,50093,50119,50147,50159,50173,50181,50209,50227,50247,
      50251,50271,50287,50301,50323,50335,50341,50359,50373,50397,
      50411,50425,50431,50443,50453,50479,50481,50505,50523,50549,
      50553,50569,50587,50593,50611,50613,50623,50655,50665,50671,
      50685,50737,50757,50769,50795,50805,50809,50831,50839,50859,
      50873,50881,50887,50901,50921,50947,51027,51033,51063,51097,
      51103,51139,51159,51163,51189,51203,51227,51265,51277,51295,
      51301,51305,51349,51389,51401,51445,51449,51455,51457,51469,
      51477,51487,51491,51497,51505,51511,51515,51547,51549,51587,
      51655,51659,51661,51673,51749,51753,51759,51785,51815,51855,
      51885,51903,51929,51939,51951,51991,51995,51997,52011,52031,
      52053,52081,52091,52103,52157,52199,52213,52245,52259,52285,
      52297,52333,52351,52355,52357,52379,52385,52391,52403,52417,
      52441,52471,52475,52477,52523,52531,52615,52639,52643,52677,
      52687,52695,52705,52715,52717,52729,52735,52739,52775,52789,
      52837,52849,52895,52923,52931,52933,52937,52945,52951,52979,
      52991,53005,53023,53039,53107,53113,53135,53149,53153,53191,
      53205,53225,53239,53253,53257,53311,53325,53361,53367,53383,
      53387,53389,53397,53401,53411,53425,53445,53455,53457,53463,
      53473,53497,53515,53541,53551,53565,53585,53595,53613,53659,
      53689,53695,53697,53717,53721,53743,53757,53767,53781,53795,
      53801,53815,53847,53867,53869,53903,53921,54019,54039,54055,
      54081,54111,54121,54135,54139,54145,54163,54169,54193,54217,
      54225,54237,54247,54293,54313,54331,54369,54379,54423,54429,
      54443,54451,54463,54465,54483,54501,54505,54513,54543,54567,
      54571,54599,54617,54627,54653,54711,54753,54759,54773,54789,
      54823,54827,54859,54883,54909,54953,54967,54981,54999,55009,
      55021,55075,55149,55167,55213,55219,55221,55231,55257,55287,
      55291,55309,55315,55369,55383,55405,55433,55439,55451,55463,
      55477,55489,55507,55513,55525,55561,55579,55603,55617,55623,
      55629,55647,55653,55665,55691,55705,55711,55715,55721,55729,
      55747,55767,55801,55813,55831,55847,55861,55871,55897,55933,
      55947,55955,55971,55985,56003,56029,56095,56101,56105,56123,
      56133,56143,56161,56195,56225,56263,56269,56277,56287,56291,
      56297,56303,56363,56377,56385,56431,56433,56467,56479,56495,
      56529,56535,56539,56565,56601,56617,56625,56663,56679,56691,
      56693,56733,56749,56789,56805,56823,56827,56839,56879,56893,
      56905,56911,56913,56941,56963,56965,56969,56993,57005,57035,
      57037,57043,57091,57093,57111,57117,57139,57177,57193,57199,
      57201,57275,57289,57323,57347,57359,57397,57415,57419,57439,
      57467,57485,57513,57521,57527,57541,57559,57601,57611,57649,
      57673,57679,57681,57693,57703,57727,57779,57811,57839,57863,
      57881,57891,57905,57925,57935,57959,57977,57989,58011,58013,
      58017,58027,58029,58047,58049,58069,58127,58129,58165,58201,
      58211,58213,58231,58253,58259,58271,58287,58295,58307,58331,
      58355,58367,58399,58417,58441,58459,58475,58477,58483,58501,
      58513,58519,58525,58529,58539,58571,58573,58609,58621,58627,
      58647,58651,58669,58675,58753,58773,58789,58885,58913,58937,
      58943,58951,58963,58985,59015,59067,59069,59095,59101,59125,
      59129,59137,59177,59245,59253,59267,59279,59307,59309,59317,
      59339,59347,59365,59383,59393,59439,59459,59483,59501,59513,
      59529,59537,59559,59583,59585,59595,59597,59603,59643,59651,
      59663,59681,59687,59691,59701,59737,59743,59747,59753,59761,
      59789,59831,59845,59855,59863,59913,59931,59949,59979,59993,
      60017,60029,60033,60045,60091,60099,60105,60141,60167,60171,
      60185,60201,60219,60229,60253,60263,60267,60275,60277,60311,
      60315,60333,60339,60365,60373,60387,60425,60433,60449,60469,
      60491,60517,60521,60541,60563,60569,60599,60623,60671,60679,
      60693,60707,60727,60745,60765,60769,60779,60823,60843,60871,
      60877,60889,60895,60911,60925,60929,60939,60941,60959,61045,
      61059,61085,61101,61127,61131,61145,61155,61169,61175,61199,
      61217,61229,61235,61241,61261,61303,61333,61343,61371,61415,
      61419,61427,61447,61453,61471,61481,61487,61509,61573,61591,
      61611,61631,61639,61653,61663,61681,61687,61713,61723,61749,
      61761,61771,61779,61795,61809,61837,61893,61921,61927,61939,
      61941,61981,62023,62029,62037,62041,62063,62075,62087,62111,
      62117,62171,62201,62207,62209,62219,62229,62255,62263,62341,
      62345,62353,62359,62387,62431,62437,62469,62479,62487,62497,
      62521,62547,62549,62565,62587,62603,62617,62627,62653,62671,
      62707,62709,62713,62733,62745,62757,62779,62801,62817,62829,
      62865,62877,62901,62911,62913,62919,63007,63011,63035,63045,
      63055,63109,63157,63193,63215,63227,63277,63295,63309,63315,
      63343,63367,63371,63381,63395,63409,63415,63427,63433,63451,
      63487,63491,63497,63503,63527,63551,63599,63601,63607,63635,
      63707,63725,63731,63733,63765,63779,63803,63805,63823,63825,
      63859,63865,63877,63899,63923,63929,63943,63971,63977,63991,
      64001,64007,64019,64035,64117,64131,64151,64155,64161,64193,
      64203,64217,64223,64229,64261,64271,64289,64309,64333,64351,
      64361,64385,64397,64419,64425,64439,64457,64463,64475,64481,
      64523,64525,64543,64585,64603,64615,64629,64643,64685,64723,
      64751,64783,64791,64797,64811,64813,64825,64839,64851,64881,
      64907,64921,64931,64943,64945,64989,64993,65003,65069,65075,
      65089,65101,65113,65149,65159,65177,65201,65213,65225,65259,
      65279,65299,65315,65321,65335,65359,65377,65395,65407,65425,
      65459,65479,65497,65513,65519,65533 };

    return sobol_a[n];
  }

  static value_type minit(std::size_t dim, std::size_t degree)
  {
    // starting direction #'s m[i] = sobol_minit[i][j] for i=0..d of the
    // degree-d primitive polynomial sobol_a[j].
    static const value_type sobol_minit[max_degree * num_polynomials] = {
      1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
      1,3,0,0,0,0,0,0,0,0,0,0,0,0,0,
      1,3,1,0,0,0,0,0,0,0,0,0,0,0,0,
      1,1,1,0,0,0,0,0,0,0,0,0,0,0,0,
      1,1,3,3,0,0,0,0,0,0,0,0,0,0,0,
      1,3,5,13,0,0,0,0,0,0,0,0,0,0,0,
      1,1,5,5,17,0,0,0,0,0,0,0,0,0,0,
      1,1,5,5,5,0,0,0,0,0,0,0,0,0,0,
      1,1,7,11,19,0,0,0,0,0,0,0,0,0,0,
      1,1,5,1,1,0,0,0,0,0,0,0,0,0,0,
      1,1,1,3,11,0,0,0,0,0,0,0,0,0,0,
      1,3,5,5,31,0,0,0,0,0,0,0,0,0,0,
      1,3,3,9,7,49,0,0,0,0,0,0,0,0,0,
      1,1,1,15,21,21,0,0,0,0,0,0,0,0,0,
      1,3,1,13,27,49,0,0,0,0,0,0,0,0,0,
      1,1,1,15,7,5,0,0,0,0,0,0,0,0,0,
      1,3,1,15,13,25,0,0,0,0,0,0,0,0,0,
      1,1,5,5,19,61,0,0,0,0,0,0,0,0,0,
      1,3,7,11,23,15,103,0,0,0,0,0,0,0,0,
      1,3,7,13,13,15,69,0,0,0,0,0,0,0,0,
      1,1,3,13,7,35,63,0,0,0,0,0,0,0,0,
      1,3,5,9,1,25,53,0,0,0,0,0,0,0,0,
      1,3,1,13,9,35,107,0,0,0,0,0,0,0,0,
      1,3,1,5,27,61,31,0,0,0,0,0,0,0,0,
      1,1,5,11,19,41,61,0,0,0,0,0,0,0,0,
      1,3,5,3,3,13,69,0,0,0,0,0,0,0,0,
      1,1,7,13,1,19,1,0,0,0,0,0,0,0,0,
      1,3,7,5,13,19,59,0,0,0,0,0,0,0,0,
      1,1,3,9,25,29,41,0,0,0,0,0,0,0,0,
      1,3,5,13,23,1,55,0,0,0,0,0,0,0,0,
      1,3,7,3,13,59,17,0,0,0,0,0,0,0,0,
      1,3,1,3,5,53,69,0,0,0,0,0,0,0,0,
      1,1,5,5,23,33,13,0,0,0,0,0,0,0,0,
      1,1,7,7,1,61,123,0,0,0,0,0,0,0,0,
      1,1,7,9,13,61,49,0,0,0,0,0,0,0,0,
      1,3,3,5,3,55,33,0,0,0,0,0,0,0,0,
      1,3,1,15,31,13,49,245,0,0,0,0,0,0,0,
      1,3,5,15,31,59,63,97,0,0,0,0,0,0,0,
      1,3,1,11,11,11,77,249,0,0,0,0,0,0,0,
      1,3,1,11,27,43,71,9,0,0,0,0,0,0,0,
      1,1,7,15,21,11,81,45,0,0,0,0,0,0,0,
      1,3,7,3,25,31,65,79,0,0,0,0,0,0,0,
      1,3,1,1,19,11,3,205,0,0,0,0,0,0,0,
      1,1,5,9,19,21,29,157,0,0,0,0,0,0,0,
      1,3,7,11,1,33,89,185,0,0,0,0,0,0,0,
      1,3,3,3,15,9,79,71,0,0,0,0,0,0,0,
      1,3,7,11,15,39,119,27,0,0,0,0,0,0,0,
      1,1,3,1,11,31,97,225,0,0,0,0,0,0,0,
      1,1,1,3,23,43,57,177,0,0,0,0,0,0,0,
      1,3,7,7,17,17,37,71,0,0,0,0,0,0,0,
      1,3,1,5,27,63,123,213,0,0,0,0,0,0,0,
      1,1,3,5,11,43,53,133,0,0,0,0,0,0,0,
      1,3,5,5,29,17,47,173,479,0,0,0,0,0,0,
      1,3,3,11,3,1,109,9,69,0,0,0,0,0,0,
      1,1,1,5,17,39,23,5,343,0,0,0,0,0,0,
      1,3,1,5,25,15,31,103,499,0,0,0,0,0,0,
      1,1,1,11,11,17,63,105,183,0,0,0,0,0,0,
      1,1,5,11,9,29,97,231,363,0,0,0,0,0,0,
      1,1,5,15,19,45,41,7,383,0,0,0,0,0,0,
      1,3,7,7,31,19,83,137,221,0,0,0,0,0,0,
      1,1,1,3,23,15,111,223,83,0,0,0,0,0,0,
      1,1,5,13,31,15,55,25,161,0,0,0,0,0,0,
      1,1,3,13,25,47,39,87,257,0,0,0,0,0,0,
      1,1,1,11,21,53,125,249,293,0,0,0,0,0,0,
      1,1,7,11,11,7,57,79,323,0,0,0,0,0,0,
      1,1,5,5,17,13,81,3,131,0,0,0,0,0,0,
      1,1,7,13,23,7,65,251,475,0,0,0,0,0,0,
      1,3,5,1,9,43,3,149,11,0,0,0,0,0,0,
      1,1,3,13,31,13,13,255,487,0,0,0,0,0,0,
      1,3,3,1,5,63,89,91,127,0,0,0,0,0,0,
      1,1,3,3,1,19,123,127,237,0,0,0,0,0,0,
      1,1,5,7,23,31,37,243,289,0,0,0,0,0,0,
      1,1,5,11,17,53,117,183,491,0,0,0,0,0,0,
      1,1,1,5,1,13,13,209,345,0,0,0,0,0,0,
      1,1,3,15,1,57,115,7,33,0,0,0,0,0,0,
      1,3,1,11,7,43,81,207,175,0,0,0,0,0,0,
      1,3,1,1,15,27,63,255,49,0,0,0,0,0,0,
      1,3,5,3,27,61,105,171,305,0,0,0,0,0,0,
      1,1,5,3,1,3,57,249,149,0,0,0,0,0,0,
      1,1,3,5,5,57,15,13,159,0,0,0,0,0,0,
      1,1,1,11,7,11,105,141,225,0,0,0,0,0,0,
      1,3,3,5,27,59,121,101,271,0,0,0,0,0,0,
      1,3,5,9,11,49,51,59,115,0,0,0,0,0,0,
      1,1,7,1,23,45,125,71,419,0,0,0,0,0,0,
      1,1,3,5,23,5,105,109,75,0,0,0,0,0,0,
      1,1,7,15,7,11,67,121,453,0,0,0,0,0,0,
      1,3,7,3,9,13,31,27,449,0,0,0,0,0,0,
      1,3,1,15,19,39,39,89,15,0,0,0,0,0,0,
      1,1,1,1,1,33,73,145,379,0,0,0,0,0,0,
      1,3,1,15,15,43,29,13,483,0,0,0,0,0,0,
      1,1,7,3,19,27,85,131,431,0,0,0,0,0,0,
      1,3,3,3,5,35,23,195,349,0,0,0,0,0,0,
      1,3,3,7,9,27,39,59,297,0,0,0,0,0,0,
      1,1,3,9,11,17,13,241,157,0,0,0,0,0,0,
      1,3,7,15,25,57,33,189,213,0,0,0,0,0,0,
      1,1,7,1,9,55,73,83,217,0,0,0,0,0,0,
      1,3,3,13,19,27,23,113,249,0,0,0,0,0,0,
      1,3,5,3,23,43,3,253,479,0,0,0,0,0,0,
      1,1,5,5,11,5,45,117,217,0,0,0,0,0,0,
      1,3,3,7,29,37,33,123,147,0,0,0,0,0,0,
      1,3,1,15,5,5,37,227,223,459,0,0,0,0,0,
      1,1,7,5,5,39,63,255,135,487,0,0,0,0,0,
      1,3,1,7,9,7,87,249,217,599,0,0,0,0,0,
      1,1,3,13,9,47,7,225,363,247,0,0,0,0,0,
      1,3,7,13,19,13,9,67,9,737,0,0,0,0,0,
      1,3,5,5,19,59,7,41,319,677,0,0,0,0,0,
      1,1,5,3,31,63,15,43,207,789,0,0,0,0,0,
      1,1,7,9,13,39,3,47,497,169,0,0,0,0,0,
      1,3,1,7,21,17,97,19,415,905,0,0,0,0,0,
      1,3,7,1,3,31,71,111,165,127,0,0,0,0,0,
      1,1,5,11,1,61,83,119,203,847,0,0,0,0,0,
      1,3,3,13,9,61,19,97,47,35,0,0,0,0,0,
      1,1,7,7,15,29,63,95,417,469,0,0,0,0,0,
      1,3,1,9,25,9,71,57,213,385,0,0,0,0,0,
      1,3,5,13,31,47,101,57,39,341,0,0,0,0,0,
      1,1,3,3,31,57,125,173,365,551,0,0,0,0,0,
      1,3,7,1,13,57,67,157,451,707,0,0,0,0,0,
      1,1,1,7,21,13,105,89,429,965,0,0,0,0,0,
      1,1,5,9,17,51,45,119,157,141,0,0,0,0,0,
      1,3,7,7,13,45,91,9,129,741,0,0,0,0,0,
      1,3,7,1,23,57,67,141,151,571,0,0,0,0,0,
      1,1,3,11,17,47,93,107,375,157,0,0,0,0,0,
      1,3,3,5,11,21,43,51,169,915,0,0,0,0,0,
      1,1,5,3,15,55,101,67,455,625,0,0,0,0,0,
      1,3,5,9,1,23,29,47,345,595,0,0,0,0,0,
      1,3,7,7,5,49,29,155,323,589,0,0,0,0,0,
      1,3,3,7,5,41,127,61,261,717,0,0,0,0,0,
      1,3,7,7,17,23,117,67,129,1009,0,0,0,0,0,
      1,1,3,13,11,39,21,207,123,305,0,0,0,0,0,
      1,1,3,9,29,3,95,47,231,73,0,0,0,0,0,
      1,3,1,9,1,29,117,21,441,259,0,0,0,0,0,
      1,3,1,13,21,39,125,211,439,723,0,0,0,0,0,
      1,1,7,3,17,63,115,89,49,773,0,0,0,0,0,
      1,3,7,13,11,33,101,107,63,73,0,0,0,0,0,
      1,1,5,5,13,57,63,135,437,177,0,0,0,0,0,
      1,1,3,7,27,63,93,47,417,483,0,0,0,0,0,
      1,1,3,1,23,29,1,191,49,23,0,0,0,0,0,
      1,1,3,15,25,55,9,101,219,607,0,0,0,0,0,
      1,3,1,7,7,19,51,251,393,307,0,0,0,0,0,
      1,3,3,3,25,55,17,75,337,3,0,0,0,0,0,
      1,1,1,13,25,17,65,45,479,413,0,0,0,0,0,
      1,1,7,7,27,49,99,161,213,727,0,0,0,0,0,
      1,3,5,1,23,5,43,41,251,857,0,0,0,0,0,
      1,3,3,7,11,61,39,87,383,835,0,0,0,0,0,
      1,1,3,15,13,7,29,7,505,923,0,0,0,0,0,
      1,3,7,1,5,31,47,157,445,501,0,0,0,0,0,
      1,1,3,7,1,43,9,147,115,605,0,0,0,0,0,
      1,3,3,13,5,1,119,211,455,1001,0,0,0,0,0,
      1,1,3,5,13,19,3,243,75,843,0,0,0,0,0,
      1,3,7,7,1,19,91,249,357,589,0,0,0,0,0,
      1,1,1,9,1,25,109,197,279,411,0,0,0,0,0,
      1,3,1,15,23,57,59,135,191,75,0,0,0,0,0,
      1,1,5,15,29,21,39,253,383,349,0,0,0,0,0,
      1,3,3,5,19,45,61,151,199,981,0,0,0,0,0,
      1,3,5,13,9,61,107,141,141,1,0,0,0,0,0,
      1,3,1,11,27,25,85,105,309,979,0,0,0,0,0,
      1,3,3,11,19,7,115,223,349,43,0,0,0,0,0,
      1,1,7,9,21,39,123,21,275,927,0,0,0,0,0,
      1,1,7,13,15,41,47,243,303,437,0,0,0,0,0,
      1,1,1,7,7,3,15,99,409,719,0,0,0,0,0,
      1,3,3,15,27,49,113,123,113,67,469,0,0,0,0,
      1,3,7,11,3,23,87,169,119,483,199,0,0,0,0,
      1,1,5,15,7,17,109,229,179,213,741,0,0,0,0,
      1,1,5,13,11,17,25,135,403,557,1433,0,0,0,0,
      1,3,1,1,1,61,67,215,189,945,1243,0,0,0,0,
      1,1,7,13,17,33,9,221,429,217,1679,0,0,0,0,
      1,1,3,11,27,3,15,93,93,865,1049,0,0,0,0,
      1,3,7,7,25,41,121,35,373,379,1547,0,0,0,0,
      1,3,3,9,11,35,45,205,241,9,59,0,0,0,0,
      1,3,1,7,3,51,7,177,53,975,89,0,0,0,0,
      1,1,3,5,27,1,113,231,299,759,861,0,0,0,0,
      1,3,3,15,25,29,5,255,139,891,2031,0,0,0,0,
      1,3,1,1,13,9,109,193,419,95,17,0,0,0,0,
      1,1,7,9,3,7,29,41,135,839,867,0,0,0,0,
      1,1,7,9,25,49,123,217,113,909,215,0,0,0,0,
      1,1,7,3,23,15,43,133,217,327,901,0,0,0,0,
      1,1,3,3,13,53,63,123,477,711,1387,0,0,0,0,
      1,1,3,15,7,29,75,119,181,957,247,0,0,0,0,
      1,1,1,11,27,25,109,151,267,99,1461,0,0,0,0,
      1,3,7,15,5,5,53,145,11,725,1501,0,0,0,0,
      1,3,7,1,9,43,71,229,157,607,1835,0,0,0,0,
      1,3,3,13,25,1,5,27,471,349,127,0,0,0,0,
      1,1,1,1,23,37,9,221,269,897,1685,0,0,0,0,
      1,1,3,3,31,29,51,19,311,553,1969,0,0,0,0,
      1,3,7,5,5,55,17,39,475,671,1529,0,0,0,0,
      1,1,7,1,1,35,47,27,437,395,1635,0,0,0,0,
      1,1,7,3,13,23,43,135,327,139,389,0,0,0,0,
      1,3,7,3,9,25,91,25,429,219,513,0,0,0,0,
      1,1,3,5,13,29,119,201,277,157,2043,0,0,0,0,
      1,3,5,3,29,57,13,17,167,739,1031,0,0,0,0,
      1,3,3,5,29,21,95,27,255,679,1531,0,0,0,0,
      1,3,7,15,9,5,21,71,61,961,1201,0,0,0,0,
      1,3,5,13,15,57,33,93,459,867,223,0,0,0,0,
      1,1,1,15,17,43,127,191,67,177,1073,0,0,0,0,
      1,1,1,15,23,7,21,199,75,293,1611,0,0,0,0,
      1,3,7,13,15,39,21,149,65,741,319,0,0,0,0,
      1,3,7,11,23,13,101,89,277,519,711,0,0,0,0,
      1,3,7,15,19,27,85,203,441,97,1895,0,0,0,0,
      1,3,1,3,29,25,21,155,11,191,197,0,0,0,0,
      1,1,7,5,27,11,81,101,457,675,1687,0,0,0,0,
      1,3,1,5,25,5,65,193,41,567,781,0,0,0,0,
      1,3,1,5,11,15,113,77,411,695,1111,0,0,0,0,
      1,1,3,9,11,53,119,171,55,297,509,0,0,0,0,
      1,1,1,1,11,39,113,139,165,347,595,0,0,0,0,
      1,3,7,11,9,17,101,13,81,325,1733,0,0,0,0,
      1,3,1,1,21,43,115,9,113,907,645,0,0,0,0,
      1,1,7,3,9,25,117,197,159,471,475,0,0,0,0,
      1,3,1,9,11,21,57,207,485,613,1661,0,0,0,0,
      1,1,7,7,27,55,49,223,89,85,1523,0,0,0,0,
      1,1,5,3,19,41,45,51,447,299,1355,0,0,0,0,
      1,3,1,13,1,33,117,143,313,187,1073,0,0,0,0,
      1,1,7,7,5,11,65,97,377,377,1501,0,0,0,0,
      1,3,1,1,21,35,95,65,99,23,1239,0,0,0,0,
      1,1,5,9,3,37,95,167,115,425,867,0,0,0,0,
      1,3,3,13,1,37,27,189,81,679,773,0,0,0,0,
      1,1,3,11,1,61,99,233,429,969,49,0,0,0,0,
      1,1,1,7,25,63,99,165,245,793,1143,0,0,0,0,
      1,1,5,11,11,43,55,65,71,283,273,0,0,0,0,
      1,1,5,5,9,3,101,251,355,379,1611,0,0,0,0,
      1,1,1,15,21,63,85,99,49,749,1335,0,0,0,0,
      1,1,5,13,27,9,121,43,255,715,289,0,0,0,0,
      1,3,1,5,27,19,17,223,77,571,1415,0,0,0,0,
      1,1,5,3,13,59,125,251,195,551,1737,0,0,0,0,
      1,3,3,15,13,27,49,105,389,971,755,0,0,0,0,
      1,3,5,15,23,43,35,107,447,763,253,0,0,0,0,
      1,3,5,11,21,3,17,39,497,407,611,0,0,0,0,
      1,1,7,13,15,31,113,17,23,507,1995,0,0,0,0,
      1,1,7,15,3,15,31,153,423,79,503,0,0,0,0,
      1,1,7,9,19,25,23,171,505,923,1989,0,0,0,0,
      1,1,5,9,21,27,121,223,133,87,697,0,0,0,0,
      1,1,5,5,9,19,107,99,319,765,1461,0,0,0,0,
      1,1,3,3,19,25,3,101,171,729,187,0,0,0,0,
      1,1,3,1,13,23,85,93,291,209,37,0,0,0,0,
      1,1,1,15,25,25,77,253,333,947,1073,0,0,0,0,
      1,1,3,9,17,29,55,47,255,305,2037,0,0,0,0,
      1,3,3,9,29,63,9,103,489,939,1523,0,0,0,0,
      1,3,7,15,7,31,89,175,369,339,595,0,0,0,0,
      1,3,7,13,25,5,71,207,251,367,665,0,0,0,0,
      1,3,3,3,21,25,75,35,31,321,1603,0,0,0,0,
      1,1,1,9,11,1,65,5,11,329,535,0,0,0,0,
      1,1,5,3,19,13,17,43,379,485,383,0,0,0,0,
      1,3,5,13,13,9,85,147,489,787,1133,0,0,0,0,
      1,3,1,1,5,51,37,129,195,297,1783,0,0,0,0,
      1,1,3,15,19,57,59,181,455,697,2033,0,0,0,0,
      1,3,7,1,27,9,65,145,325,189,201,0,0,0,0,
      1,3,1,15,31,23,19,5,485,581,539,0,0,0,0,
      1,1,7,13,11,15,65,83,185,847,831,0,0,0,0,
      1,3,5,7,7,55,73,15,303,511,1905,0,0,0,0,
      1,3,5,9,7,21,45,15,397,385,597,0,0,0,0,
      1,3,7,3,23,13,73,221,511,883,1265,0,0,0,0,
      1,1,3,11,1,51,73,185,33,975,1441,0,0,0,0,
      1,3,3,9,19,59,21,39,339,37,143,0,0,0,0,
      1,1,7,1,31,33,19,167,117,635,639,0,0,0,0,
      1,1,1,3,5,13,59,83,355,349,1967,0,0,0,0,
      1,1,1,5,19,3,53,133,97,863,983,0,0,0,0,
      1,3,1,13,9,41,91,105,173,97,625,0,0,0,0,
      1,1,5,3,7,49,115,133,71,231,1063,0,0,0,0,
      1,1,7,5,17,43,47,45,497,547,757,0,0,0,0,
      1,3,5,15,21,61,123,191,249,31,631,0,0,0,0,
      1,3,7,9,17,7,11,185,127,169,1951,0,0,0,0,
      1,1,5,13,11,11,9,49,29,125,791,0,0,0,0,
      1,1,1,15,31,41,13,167,273,429,57,0,0,0,0,
      1,3,5,3,27,7,35,209,65,265,1393,0,0,0,0,
      1,3,1,13,31,19,53,143,135,9,1021,0,0,0,0,
      1,1,7,13,31,5,115,153,143,957,623,0,0,0,0,
      1,1,5,11,25,19,29,31,297,943,443,0,0,0,0,
      1,3,3,5,21,11,127,81,479,25,699,0,0,0,0,
      1,1,3,11,25,31,97,19,195,781,705,0,0,0,0,
      1,1,5,5,31,11,75,207,197,885,2037,0,0,0,0,
      1,1,1,11,9,23,29,231,307,17,1497,0,0,0,0,
      1,1,5,11,11,43,111,233,307,523,1259,0,0,0,0,
      1,1,7,5,1,21,107,229,343,933,217,0,0,0,0,
      1,1,1,11,3,21,125,131,405,599,1469,0,0,0,0,
      1,3,5,5,9,39,33,81,389,151,811,0,0,0,0,
      1,1,7,7,7,1,59,223,265,529,2021,0,0,0,0,
      1,3,1,3,9,23,85,181,47,265,49,0,0,0,0,
      1,3,5,11,19,23,9,7,157,299,1983,0,0,0,0,
      1,3,1,5,15,5,21,105,29,339,1041,0,0,0,0,
      1,1,1,1,5,33,65,85,111,705,479,0,0,0,0,
      1,1,1,7,9,35,77,87,151,321,101,0,0,0,0,
      1,1,5,7,17,1,51,197,175,811,1229,0,0,0,0,
      1,3,3,15,23,37,85,185,239,543,731,0,0,0,0,
      1,3,1,7,7,55,111,109,289,439,243,0,0,0,0,
      1,1,7,11,17,53,35,217,259,853,1667,0,0,0,0,
      1,3,1,9,1,63,87,17,73,565,1091,0,0,0,0,
      1,1,3,3,11,41,1,57,295,263,1029,0,0,0,0,
      1,1,5,1,27,45,109,161,411,421,1395,0,0,0,0,
      1,3,5,11,25,35,47,191,339,417,1727,0,0,0,0,
      1,1,5,15,21,1,93,251,351,217,1767,0,0,0,0,
      1,3,3,11,3,7,75,155,313,211,491,0,0,0,0,
      1,3,3,5,11,9,101,161,453,913,1067,0,0,0,0,
      1,1,3,1,15,45,127,141,163,727,1597,0,0,0,0,
      1,3,3,7,1,33,63,73,73,341,1691,0,0,0,0,
      1,3,5,13,15,39,53,235,77,99,949,0,0,0,0,
      1,1,5,13,31,17,97,13,215,301,1927,0,0,0,0,
      1,1,7,1,1,37,91,93,441,251,1131,0,0,0,0,
      1,3,7,9,25,5,105,69,81,943,1459,0,0,0,0,
      1,3,7,11,31,43,13,209,27,1017,501,0,0,0,0,
      1,1,7,15,1,33,31,233,161,507,387,0,0,0,0,
      1,3,3,5,5,53,33,177,503,627,1927,0,0,0,0,
      1,1,7,11,7,61,119,31,457,229,1875,0,0,0,0,
      1,1,5,15,19,5,53,201,157,885,1057,0,0,0,0,
      1,3,7,9,1,35,51,113,249,425,1009,0,0,0,0,
      1,3,5,7,21,53,37,155,119,345,631,0,0,0,0,
      1,3,5,7,15,31,109,69,503,595,1879,0,0,0,0,
      1,3,3,1,25,35,65,131,403,705,503,0,0,0,0,
      1,3,7,7,19,33,11,153,45,633,499,0,0,0,0,
      1,3,3,5,11,3,29,93,487,33,703,0,0,0,0,
      1,1,3,15,21,53,107,179,387,927,1757,0,0,0,0,
      1,1,3,7,21,45,51,147,175,317,361,0,0,0,0,
      1,1,1,7,7,13,15,243,269,795,1965,0,0,0,0,
      1,1,3,5,19,33,57,115,443,537,627,0,0,0,0,
      1,3,3,9,3,39,25,61,185,717,1049,0,0,0,0,
      1,3,7,3,7,37,107,153,7,269,1581,0,0,0,0,
      1,1,7,3,7,41,91,41,145,489,1245,0,0,0,0,
      1,1,5,9,7,7,105,81,403,407,283,0,0,0,0,
      1,1,7,9,27,55,29,77,193,963,949,0,0,0,0,
      1,1,5,3,25,51,107,63,403,917,815,0,0,0,0,
      1,1,7,3,7,61,19,51,457,599,535,0,0,0,0,
      1,3,7,1,23,51,105,153,239,215,1847,0,0,0,0,
      1,1,3,5,27,23,79,49,495,45,1935,0,0,0,0,
      1,1,1,11,11,47,55,133,495,999,1461,0,0,0,0,
      1,1,3,15,27,51,93,17,355,763,1675,0,0,0,0,
      1,3,1,3,1,3,79,119,499,17,995,0,0,0,0,
      1,1,1,1,15,43,45,17,167,973,799,0,0,0,0,
      1,1,1,3,27,49,89,29,483,913,2023,0,0,0,0,
      1,1,3,3,5,11,75,7,41,851,611,0,0,0,0,
      1,3,1,3,7,57,39,123,257,283,507,0,0,0,0,
      1,3,3,11,27,23,113,229,187,299,133,0,0,0,0,
      1,1,3,13,9,63,101,77,451,169,337,0,0,0,0,
      1,3,7,3,3,59,45,195,229,415,409,0,0,0,0,
      1,3,5,3,11,19,71,93,43,857,369,0,0,0,0,
      1,3,7,9,19,33,115,19,241,703,247,0,0,0,0,
      1,3,5,11,5,35,21,155,463,1005,1073,0,0,0,0,
      1,3,7,3,25,15,109,83,93,69,1189,0,0,0,0,
      1,3,5,7,5,21,93,133,135,167,903,0,0,0,0,
      1,1,7,7,3,59,121,161,285,815,1769,3705,0,0,0,
      1,3,1,1,3,47,103,171,381,609,185,373,0,0,0,
      1,3,3,15,23,33,107,131,441,445,689,2059,0,0,0,
      1,3,3,11,7,53,101,167,435,803,1255,3781,0,0,0,
      1,1,5,11,15,59,41,19,135,835,1263,505,0,0,0,
      1,1,7,11,21,49,23,219,127,961,1065,385,0,0,0,
      1,3,5,15,7,47,117,217,45,731,1639,733,0,0,0,
      1,1,7,11,27,57,91,87,81,35,1269,1007,0,0,0,
      1,1,3,11,15,37,53,219,193,937,1899,3733,0,0,0,
      1,3,5,3,13,11,27,19,199,393,965,2195,0,0,0,
      1,3,1,3,5,1,37,173,413,1023,553,409,0,0,0,
      1,3,1,7,15,29,123,95,255,373,1799,3841,0,0,0,
      1,3,5,13,21,57,51,17,511,195,1157,1831,0,0,0,
      1,1,1,15,29,19,7,73,295,519,587,3523,0,0,0,
      1,1,5,13,13,35,115,191,123,535,717,1661,0,0,0,
      1,3,3,5,23,21,47,251,379,921,1119,297,0,0,0,
      1,3,3,9,29,53,121,201,135,193,523,2943,0,0,0,
      1,1,1,7,29,45,125,9,99,867,425,601,0,0,0,
      1,3,1,9,13,15,67,181,109,293,1305,3079,0,0,0,
      1,3,3,9,5,35,15,209,305,87,767,2795,0,0,0,
      1,3,3,11,27,57,113,123,179,643,149,523,0,0,0,
      1,1,3,15,11,17,67,223,63,657,335,3309,0,0,0,
      1,1,1,9,25,29,109,159,39,513,571,1761,0,0,0,
      1,1,3,1,5,63,75,19,455,601,123,691,0,0,0,
      1,1,1,3,21,5,45,169,377,513,1951,2565,0,0,0,
      1,1,3,11,3,33,119,69,253,907,805,1449,0,0,0,
      1,1,5,13,31,15,17,7,499,61,687,1867,0,0,0,
      1,3,7,11,17,33,73,77,299,243,641,2345,0,0,0,
      1,1,7,11,9,35,31,235,359,647,379,1161,0,0,0,
      1,3,3,15,31,25,5,67,33,45,437,4067,0,0,0,
      1,1,3,11,7,17,37,87,333,253,1517,2921,0,0,0,
      1,1,7,15,7,15,107,189,153,769,1521,3427,0,0,0,
      1,3,5,13,5,61,113,37,293,393,113,43,0,0,0,
      1,1,1,15,29,43,107,31,167,147,301,1021,0,0,0,
      1,1,1,13,3,1,35,93,195,181,2027,1491,0,0,0,
      1,3,3,3,13,33,77,199,153,221,1699,3671,0,0,0,
      1,3,5,13,7,49,123,155,495,681,819,809,0,0,0,
      1,3,5,15,27,61,117,189,183,887,617,4053,0,0,0,
      1,1,1,7,31,59,125,235,389,369,447,1039,0,0,0,
      1,3,5,1,5,39,115,89,249,377,431,3747,0,0,0,
      1,1,1,5,7,47,59,157,77,445,699,3439,0,0,0,
      1,1,3,5,11,21,19,75,11,599,1575,735,0,0,0,
      1,3,5,3,19,13,41,69,199,143,1761,3215,0,0,0,
      1,3,5,7,19,43,25,41,41,11,1647,2783,0,0,0,
      1,3,1,9,19,45,111,97,405,399,457,3219,0,0,0,
      1,1,3,1,23,15,65,121,59,985,829,2259,0,0,0,
      1,1,3,7,17,13,107,229,75,551,1299,2363,0,0,0,
      1,1,5,5,21,57,23,199,509,139,2007,3875,0,0,0,
      1,3,1,11,19,53,15,229,215,741,695,823,0,0,0,
      1,3,7,1,29,3,17,163,417,559,549,319,0,0,0,
      1,3,1,13,17,9,47,133,365,7,1937,1071,0,0,0,
      1,3,5,7,19,37,55,163,301,249,689,2327,0,0,0,
      1,3,5,13,11,23,61,205,257,377,615,1457,0,0,0,
      1,3,5,1,23,37,13,75,331,495,579,3367,0,0,0,
      1,1,1,9,1,23,49,129,475,543,883,2531,0,0,0,
      1,3,1,5,23,59,51,35,343,695,219,369,0,0,0,
      1,3,3,1,27,17,63,97,71,507,1929,613,0,0,0,
      1,1,5,1,21,31,11,109,247,409,1817,2173,0,0,0,
      1,1,3,15,23,9,7,209,301,23,147,1691,0,0,0,
      1,1,7,5,5,19,37,229,249,277,1115,2309,0,0,0,
      1,1,1,5,5,63,5,249,285,431,343,2467,0,0,0,
      1,1,1,11,7,45,35,75,505,537,29,2919,0,0,0,
      1,3,5,15,11,39,15,63,263,9,199,445,0,0,0,
      1,3,3,3,27,63,53,171,227,63,1049,827,0,0,0,
      1,1,3,13,7,11,115,183,179,937,1785,381,0,0,0,
      1,3,1,11,13,15,107,81,53,295,1785,3757,0,0,0,
      1,3,3,13,11,5,109,243,3,505,323,1373,0,0,0,
      1,3,3,11,21,51,17,177,381,937,1263,3889,0,0,0,
      1,3,5,9,27,25,85,193,143,573,1189,2995,0,0,0,
      1,3,5,11,13,9,81,21,159,953,91,1751,0,0,0,
      1,1,3,3,27,61,11,253,391,333,1105,635,0,0,0,
      1,3,3,15,9,57,95,81,419,735,251,1141,0,0,0,
      1,1,5,9,31,39,59,13,319,807,1241,2433,0,0,0,
      1,3,3,5,27,13,107,141,423,937,2027,3233,0,0,0,
      1,3,3,9,9,25,125,23,443,835,1245,847,0,0,0,
      1,1,7,15,17,17,83,107,411,285,847,1571,0,0,0,
      1,1,3,13,29,61,37,81,349,727,1453,1957,0,0,0,
      1,3,7,11,31,13,59,77,273,591,1265,1533,0,0,0,
      1,1,7,7,13,17,25,25,187,329,347,1473,0,0,0,
      1,3,7,7,5,51,37,99,221,153,503,2583,0,0,0,
      1,3,1,13,19,27,11,69,181,479,1183,3229,0,0,0,
      1,3,3,13,23,21,103,147,323,909,947,315,0,0,0,
      1,3,1,3,23,1,31,59,93,513,45,2271,0,0,0,
      1,3,5,1,7,43,109,59,231,41,1515,2385,0,0,0,
      1,3,1,5,31,57,49,223,283,1013,11,701,0,0,0,
      1,1,5,1,19,53,55,31,31,299,495,693,0,0,0,
      1,3,3,9,5,33,77,253,427,791,731,1019,0,0,0,
      1,3,7,11,1,9,119,203,53,877,1707,3499,0,0,0,
      1,1,3,7,13,39,55,159,423,113,1653,3455,0,0,0,
      1,1,3,5,21,47,51,59,55,411,931,251,0,0,0,
      1,3,7,3,31,25,81,115,405,239,741,455,0,0,0,
      1,1,5,1,31,3,101,83,479,491,1779,2225,0,0,0,
      1,3,3,3,9,37,107,161,203,503,767,3435,0,0,0,
      1,3,7,9,1,27,61,119,233,39,1375,4089,0,0,0,
      1,1,5,9,1,31,45,51,369,587,383,2813,0,0,0,
      1,3,7,5,31,7,49,119,487,591,1627,53,0,0,0,
      1,1,7,1,9,47,1,223,369,711,1603,1917,0,0,0,
      1,3,5,3,21,37,111,17,483,739,1193,2775,0,0,0,
      1,3,3,7,17,11,51,117,455,191,1493,3821,0,0,0,
      1,1,5,9,23,39,99,181,343,485,99,1931,0,0,0,
      1,3,1,7,29,49,31,71,489,527,1763,2909,0,0,0,
      1,1,5,11,5,5,73,189,321,57,1191,3685,0,0,0,
      1,1,5,15,13,45,125,207,371,415,315,983,0,0,0,
      1,3,3,5,25,59,33,31,239,919,1859,2709,0,0,0,
      1,3,5,13,27,61,23,115,61,413,1275,3559,0,0,0,
      1,3,7,15,5,59,101,81,47,967,809,3189,0,0,0,
      1,1,5,11,31,15,39,25,173,505,809,2677,0,0,0,
      1,1,5,9,19,13,95,89,511,127,1395,2935,0,0,0,
      1,1,5,5,31,45,9,57,91,303,1295,3215,0,0,0,
      1,3,3,3,19,15,113,187,217,489,1285,1803,0,0,0,
      1,1,3,1,13,29,57,139,255,197,537,2183,0,0,0,
      1,3,1,15,11,7,53,255,467,9,757,3167,0,0,0,
      1,3,3,15,21,13,9,189,359,323,49,333,0,0,0,
      1,3,7,11,7,37,21,119,401,157,1659,1069,0,0,0,
      1,1,5,7,17,33,115,229,149,151,2027,279,0,0,0,
      1,1,5,15,5,49,77,155,383,385,1985,945,0,0,0,
      1,3,7,3,7,55,85,41,357,527,1715,1619,0,0,0,
      1,1,3,1,21,45,115,21,199,967,1581,3807,0,0,0,
      1,1,3,7,21,39,117,191,169,73,413,3417,0,0,0,
      1,1,1,13,1,31,57,195,231,321,367,1027,0,0,0,
      1,3,7,3,11,29,47,161,71,419,1721,437,0,0,0,
      1,1,7,3,11,9,43,65,157,1,1851,823,0,0,0,
      1,1,1,5,21,15,31,101,293,299,127,1321,0,0,0,
      1,1,7,1,27,1,11,229,241,705,43,1475,0,0,0,
      1,3,7,1,5,15,73,183,193,55,1345,49,0,0,0,
      1,3,3,3,19,3,55,21,169,663,1675,137,0,0,0,
      1,1,1,13,7,21,69,67,373,965,1273,2279,0,0,0,
      1,1,7,7,21,23,17,43,341,845,465,3355,0,0,0,
      1,3,5,5,25,5,81,101,233,139,359,2057,0,0,0,
      1,1,3,11,15,39,55,3,471,765,1143,3941,0,0,0,
      1,1,7,15,9,57,81,79,215,433,333,3855,0,0,0,
      1,1,5,5,19,45,83,31,209,363,701,1303,0,0,0,
      1,3,7,5,1,13,55,163,435,807,287,2031,0,0,0,
      1,3,3,7,3,3,17,197,39,169,489,1769,0,0,0,
      1,1,3,5,29,43,87,161,289,339,1233,2353,0,0,0,
      1,3,3,9,21,9,77,1,453,167,1643,2227,0,0,0,
      1,1,7,1,15,7,67,33,193,241,1031,2339,0,0,0,
      1,3,1,11,1,63,45,65,265,661,849,1979,0,0,0,
      1,3,1,13,19,49,3,11,159,213,659,2839,0,0,0,
      1,3,5,11,9,29,27,227,253,449,1403,3427,0,0,0,
      1,1,3,1,7,3,77,143,277,779,1499,475,0,0,0,
      1,1,1,5,11,23,87,131,393,849,193,3189,0,0,0,
      1,3,5,11,3,3,89,9,449,243,1501,1739,0,0,0,
      1,3,1,9,29,29,113,15,65,611,135,3687,0,0,0,
      1,1,1,9,21,19,39,151,395,501,1339,959,2725,0,0,
      1,3,7,1,7,35,45,33,119,225,1631,1695,1459,0,0,
      1,1,1,3,25,55,37,79,167,907,1075,271,4059,0,0,
      1,3,5,13,5,13,53,165,437,67,1705,3177,8095,0,0,
      1,3,3,13,27,57,95,55,443,245,1945,1725,1929,0,0,
      1,3,1,9,5,33,109,35,99,827,341,2401,2411,0,0,
      1,1,5,9,7,33,43,39,87,799,635,3481,7159,0,0,
      1,3,1,1,31,15,45,27,337,113,987,2065,2529,0,0,
      1,1,5,9,5,15,105,123,479,289,1609,2177,4629,0,0,
      1,3,5,11,31,47,97,87,385,195,1041,651,3271,0,0,
      1,1,3,7,17,3,101,55,87,629,1687,1387,2745,0,0,
      1,3,5,5,7,21,9,237,313,549,1107,117,6183,0,0,
      1,1,3,9,9,5,55,201,487,851,1103,2993,4055,0,0,
      1,1,5,9,31,19,59,7,363,381,1167,2057,5715,0,0,
      1,3,3,15,23,63,19,227,387,827,487,1049,7471,0,0,
      1,3,1,5,23,25,61,245,363,863,963,3583,6475,0,0,
      1,1,5,1,5,27,81,85,275,49,235,3291,1195,0,0,
      1,1,5,7,23,53,85,107,511,779,1265,1093,7859,0,0,
      1,3,3,1,9,21,75,219,59,485,1739,3845,1109,0,0,
      1,3,5,1,13,41,19,143,293,391,2023,1791,4399,0,0,
      1,3,7,15,21,13,21,195,215,413,523,2099,2341,0,0,
      1,1,1,3,29,51,47,57,135,575,943,1673,541,0,0,
      1,3,5,1,9,13,113,175,447,115,657,4077,5973,0,0,
      1,1,1,11,17,41,37,95,297,579,911,2207,2387,0,0,
      1,3,5,3,23,11,23,231,93,667,711,1563,7961,0,0,
      1,1,7,3,17,59,13,181,141,991,1817,457,1711,0,0,
      1,3,3,5,31,59,81,205,245,537,1049,997,1815,0,0,
      1,3,7,5,17,13,9,79,17,185,5,2211,6263,0,0,
      1,3,7,13,7,53,61,145,13,285,1203,947,2933,0,0,
      1,1,7,3,31,19,69,217,47,441,1893,673,4451,0,0,
      1,1,1,1,25,9,23,225,385,629,603,3747,4241,0,0,
      1,3,1,9,5,37,31,237,431,79,1521,459,2523,0,0,
      1,3,7,3,9,43,105,179,5,225,799,1777,4893,0,0,
      1,1,3,1,29,45,29,159,267,247,455,847,3909,0,0,
      1,1,3,7,25,21,121,57,467,275,719,1521,7319,0,0,
      1,3,1,3,11,35,119,123,81,979,1187,3623,4293,0,0,
      1,1,1,7,15,25,121,235,25,487,873,1787,1977,0,0,
      1,1,1,11,3,7,17,135,345,353,383,4011,2573,0,0,
      1,3,7,15,27,13,97,123,65,675,951,1285,6559,0,0,
      1,3,7,3,7,1,71,19,325,765,337,1197,2697,0,0,
      1,3,5,1,31,37,11,71,169,283,83,3801,7083,0,0,
      1,1,3,15,17,29,83,65,275,679,1749,4007,7749,0,0,
      1,1,3,1,21,11,41,95,237,361,1819,2783,2383,0,0,
      1,3,7,11,29,57,111,187,465,145,605,1987,8109,0,0,
      1,1,3,3,19,15,55,83,357,1001,643,1517,6529,0,0,
      1,3,1,5,29,35,73,23,77,619,1523,1725,8145,0,0,
      1,1,5,5,19,23,7,197,449,337,717,2921,315,0,0,
      1,3,5,9,7,63,117,97,97,813,1925,2817,1579,0,0,
      1,1,1,11,31,7,25,235,231,133,1007,1371,1553,0,0,
      1,1,7,5,19,7,47,171,267,243,1331,567,6033,0,0,
      1,1,5,1,7,49,55,89,109,735,1455,3193,6239,0,0,
      1,1,1,7,1,61,9,103,3,929,1481,2927,2957,0,0,
      1,1,5,13,17,21,75,49,255,1019,1161,2133,1177,0,0,
      1,3,1,3,13,15,41,247,211,409,1163,523,2635,0,0,
      1,3,7,7,21,59,91,149,479,391,681,2311,6249,0,0,
      1,1,5,11,27,53,21,211,197,815,719,1605,255,0,0,
      1,1,3,3,9,33,59,3,323,1,101,1135,8105,0,0,
      1,3,3,1,29,5,17,141,51,991,841,327,3859,0,0,
      1,3,1,5,11,19,23,89,175,173,165,2881,1881,0,0,
      1,1,1,15,13,51,87,39,495,611,1341,1531,7029,0,0,
      1,1,3,11,13,55,75,185,57,61,1917,2051,5965,0,0,
      1,1,5,5,7,53,11,217,213,933,921,3607,5175,0,0,
      1,3,3,5,17,53,103,251,369,781,1319,3717,4439,0,0,
      1,3,5,13,1,39,25,235,321,773,251,3111,6397,0,0,
      1,1,7,3,31,5,25,29,325,385,1313,127,4705,0,0,
      1,1,5,15,15,27,15,85,239,243,1633,3473,2621,0,0,
      1,3,3,3,9,19,113,13,137,165,25,2957,7549,0,0,
      1,3,1,3,11,21,3,97,417,183,1205,1437,247,0,0,
      1,1,7,3,17,21,125,55,67,387,385,2323,887,0,0,
      1,3,5,5,29,11,103,223,233,641,133,415,1297,0,0,
      1,3,3,11,1,9,5,189,235,1007,1363,3985,889,0,0,
      1,3,7,9,23,19,19,183,269,403,1643,3559,5189,0,0,
      1,3,7,3,29,45,17,69,475,149,1291,2689,7625,0,0,
      1,3,7,3,27,37,41,73,253,1001,431,1111,7887,0,0,
      1,1,7,5,3,7,87,143,289,495,631,3011,6151,0,0,
      1,1,1,13,5,45,17,167,23,975,801,1975,6833,0,0,
      1,3,1,11,7,21,39,23,213,429,1301,2059,197,0,0,
      1,3,3,15,3,57,121,133,29,711,1961,2497,189,0,0,
      1,1,3,5,11,55,115,137,233,673,985,2849,5911,0,0,
      1,1,7,15,29,45,1,241,329,323,925,2821,3331,0,0,
      1,1,5,7,13,31,81,105,199,145,195,1365,5119,0,0,
      1,3,7,11,3,55,11,31,117,343,1265,1837,2451,0,0,
      1,1,3,7,29,57,61,179,429,591,177,1945,2159,0,0,
      1,3,5,11,23,49,101,137,339,323,1035,1749,7737,0,0,
      1,3,1,13,21,35,55,79,19,269,1055,2651,7083,0,0,
      1,3,3,11,9,9,95,167,437,361,1185,4083,603,0,0,
      1,1,1,7,31,61,77,65,489,657,691,2423,4147,0,0,
      1,3,5,7,21,37,87,191,311,453,2013,829,2619,0,0,
      1,1,5,9,17,47,35,101,5,813,1157,1279,7365,0,0,
      1,1,5,3,11,35,113,199,369,721,901,1471,7801,0,0,
      1,3,1,5,9,61,83,157,391,739,1957,2123,4341,0,0,
      1,3,5,11,19,19,111,225,383,219,997,717,7505,0,0,
      1,3,1,11,13,63,35,127,209,831,501,3017,3507,0,0,
      1,3,7,9,29,7,11,163,81,563,1445,3215,6377,0,0,
      1,3,7,11,25,3,39,195,491,45,839,4021,4899,0,0,
      1,3,7,15,13,5,67,143,117,505,1281,3679,5695,0,0,
      1,3,7,9,9,19,21,221,147,763,683,2211,589,0,0,
      1,1,3,5,21,47,53,109,299,807,1153,1209,7961,0,0,
      1,3,7,11,9,31,45,43,505,647,1127,2681,4917,0,0,
      1,1,5,15,31,41,63,113,399,727,673,2587,5259,0,0,
      1,1,1,13,17,53,35,99,57,243,1447,1919,2831,0,0,
      1,3,7,11,23,51,13,9,49,449,997,3073,4407,0,0,
      1,3,5,7,23,33,89,41,415,53,697,1113,1489,0,0,
      1,1,3,7,1,13,29,13,255,749,77,3463,1761,0,0,
      1,3,3,7,13,15,93,191,309,869,739,1041,3053,0,0,
      1,3,5,13,5,19,109,211,347,839,893,2947,7735,0,0,
      1,3,1,13,27,3,119,157,485,99,1703,3895,573,0,0,
      1,3,7,11,1,23,123,105,31,359,275,1775,3685,0,0,
      1,3,3,5,27,11,125,3,413,199,2043,2895,2945,0,0,
      1,3,3,3,15,49,121,159,233,543,193,4007,321,0,0,
      1,1,3,5,9,47,87,1,51,1011,1595,2239,6467,0,0,
      1,3,7,9,1,33,87,137,469,749,1413,805,6817,0,0,
      1,3,1,13,19,45,95,227,29,677,1275,3395,4451,0,0,
      1,1,7,5,7,63,33,71,443,561,1311,3069,6943,0,0,
      1,1,1,13,9,37,23,69,13,415,1479,1197,861,0,0,
      1,3,3,13,27,21,13,233,105,777,345,2443,1105,0,0,
      1,1,7,11,23,13,21,147,221,549,73,2729,6279,0,0,
      1,1,7,7,25,27,15,45,227,39,75,1191,3563,0,0,
      1,1,5,7,13,49,99,167,227,13,353,1047,8075,0,0,
      1,1,3,13,31,9,27,7,461,737,1559,3243,53,0,0,
      1,3,1,1,21,41,97,165,171,821,587,2137,2293,0,0,
      1,3,1,11,17,41,29,187,87,599,1467,1395,5931,0,0,
      1,1,1,9,9,49,89,205,409,453,61,1923,1257,0,0,
      1,3,7,3,9,43,89,143,431,83,1243,1795,3599,0,0,
      1,3,5,13,3,25,59,219,43,223,797,2651,6015,0,0,
      1,1,5,15,7,55,65,207,213,311,1287,1269,6467,0,0,
      1,3,7,11,21,57,31,183,351,857,911,1683,7155,0,0,
      1,3,5,11,27,1,21,47,387,383,1593,115,3805,0,0,
      1,3,1,1,13,23,87,173,181,619,1653,3931,6073,0,0,
      1,1,7,5,17,43,37,61,307,621,1785,55,115,0,0,
      1,3,7,15,25,61,123,15,237,671,1473,467,1907,0,0,
      1,1,7,5,29,57,75,237,85,699,159,3577,4771,0,0,
      1,1,1,11,25,19,51,1,147,31,895,2617,625,0,0,
      1,3,7,5,29,15,115,175,395,391,1141,1827,1181,0,0,
      1,3,5,7,17,7,11,193,89,243,561,3787,4551,0,0,
      1,3,1,11,7,57,7,125,403,947,1261,409,8083,0,0,
      1,1,5,13,21,63,115,233,231,921,1747,3635,2519,0,0,
      1,1,5,11,3,27,15,91,505,591,1451,3881,2997,0,0,
      1,1,3,11,21,9,109,153,317,533,593,3967,2797,0,0,
      1,3,3,13,9,57,121,245,219,867,967,791,7095,0,0,
      1,1,1,9,29,21,99,35,375,959,329,4087,7171,0,0,
      1,1,1,9,11,17,17,97,89,135,631,3809,3253,0,0,
      1,1,1,15,21,51,91,249,459,801,757,2353,2033,0,0,
      1,3,5,9,23,29,77,53,399,767,1817,2171,1629,0,0,
      1,1,3,5,29,5,43,121,17,859,1479,3785,6641,0,0,
      1,1,3,7,7,61,45,109,371,833,91,153,4553,0,0,
      1,1,3,11,7,55,81,123,389,139,1933,891,1789,0,0,
      1,3,7,15,25,17,93,165,503,717,1553,1475,1627,0,0,
      1,1,1,13,13,63,13,225,357,571,33,4073,3795,0,0,
      1,1,3,11,1,31,107,145,407,961,501,2987,103,0,0,
      1,1,7,1,23,63,49,193,173,281,25,2465,5927,0,0,
      1,1,7,1,1,1,85,77,273,693,349,1239,4503,0,0,
      1,1,5,11,7,61,9,121,25,357,1443,405,7827,0,0,
      1,1,7,13,11,53,11,207,145,211,1703,1081,2117,0,0,
      1,1,3,11,27,23,19,9,297,279,1481,2273,6387,0,0,
      1,3,3,5,15,45,3,41,305,87,1815,3461,5349,0,0,
      1,3,3,13,9,37,79,125,259,561,1087,4091,793,0,0,
      1,3,5,7,31,55,7,145,347,929,589,2783,5905,0,0,
      1,1,7,15,3,25,1,181,13,243,653,2235,7445,0,0,
      1,3,5,5,17,53,65,7,33,583,1363,1313,2319,0,0,
      1,3,3,7,27,47,97,201,187,321,63,1515,7917,0,0,
      1,1,3,5,23,9,3,165,61,19,1789,3783,3037,0,0,
      1,3,1,13,15,43,125,191,67,273,1551,2227,5253,0,0,
      1,1,1,13,25,53,107,33,299,249,1475,2233,907,0,0,
      1,3,5,1,23,37,85,17,207,643,665,2933,5199,0,0,
      1,1,7,7,25,57,59,41,15,751,751,1749,7053,0,0,
      1,3,3,1,13,25,127,93,281,613,875,2223,6345,0,0,
      1,1,5,3,29,55,79,249,43,317,533,995,1991,0,0,
      1,3,3,15,17,49,79,31,193,233,1437,2615,819,0,0,
      1,1,5,15,25,3,123,145,377,9,455,1191,3953,0,0,
      1,3,5,3,15,19,41,231,81,393,3,19,2409,0,0,
      1,1,3,1,27,43,113,179,7,853,947,2731,297,0,0,
      1,1,1,11,29,39,53,191,443,689,529,3329,7431,0,0,
      1,3,7,5,3,29,19,67,441,113,949,2769,4169,0,0,
      1,3,5,11,11,55,85,169,215,815,803,2345,3967,0,0,
      1,1,7,9,5,45,111,5,419,375,303,1725,4489,0,0,
      1,3,5,15,29,43,79,19,23,417,381,541,4923,0,0,
      1,1,3,15,3,31,117,39,117,305,1227,1223,143,0,0,
      1,1,5,9,5,47,87,239,181,353,1561,3313,1921,0,0,
      1,3,3,1,3,15,53,221,441,987,1997,2529,8059,0,0,
      1,1,7,11,15,57,111,139,137,883,1881,2823,5661,0,0,
      1,3,5,5,21,11,5,13,27,973,587,1331,1373,0,0,
      1,1,7,11,29,51,93,29,217,221,55,2477,1979,0,0,
      1,3,3,13,3,11,49,75,379,371,1441,793,7633,0,0,
      1,1,1,13,19,45,89,249,91,649,1695,915,5619,0,0,
      1,3,1,7,7,29,1,77,313,895,519,771,295,0,0,
      1,3,1,15,5,3,1,57,331,109,485,2853,6831,0,0,
      1,1,1,15,17,3,35,99,245,971,839,2509,2803,0,0,
      1,3,3,3,9,37,57,251,325,317,529,1313,6379,0,0,
      1,1,1,15,25,59,1,119,95,15,795,2375,6463,0,0,
      1,3,1,5,1,49,117,21,47,179,863,85,1669,0,0,
      1,3,7,3,9,37,19,221,455,973,571,1427,817,0,0,
      1,1,1,15,17,9,67,213,127,887,1299,2913,7451,0,0,
      1,3,1,13,27,27,41,43,171,623,691,391,4885,0,0,
      1,3,1,13,17,17,123,239,143,227,1151,519,6543,0,0,
      1,3,7,5,7,63,97,39,101,555,1057,381,7891,0,0,
      1,3,5,1,3,27,85,129,161,875,1945,3541,695,0,0,
      1,3,3,5,21,59,25,183,35,25,987,1459,181,0,0,
      1,3,5,13,1,15,127,237,349,337,1491,2383,7811,0,0,
      1,3,5,5,31,5,109,51,409,733,1395,3207,6049,0,0,
      1,1,5,7,13,35,113,25,263,389,299,2521,1783,0,0,
      1,3,7,11,15,47,97,73,55,75,113,2695,1023,0,0,
      1,3,1,1,3,13,69,211,289,483,1335,787,677,0,0,
      1,1,3,3,17,7,37,77,505,137,1113,345,2975,0,0,
      1,1,1,13,3,11,95,199,453,109,479,3725,239,0,0,
      1,1,7,15,19,53,3,145,359,863,347,3833,3043,0,0,
      1,1,7,15,25,63,127,129,125,195,155,2211,8153,0,0,
      1,1,7,13,9,49,121,115,73,119,1851,727,47,0,0,
      1,3,3,13,13,11,71,7,45,591,133,2407,5563,0,0,
      1,1,1,13,23,29,87,89,501,71,1759,1119,687,0,0,
      1,1,7,7,13,7,13,183,53,951,1877,3991,6771,0,0,
      1,3,7,11,7,1,27,47,61,21,919,961,1091,0,0,
      1,3,5,5,1,27,1,5,63,157,1297,1049,5893,0,0,
      1,3,7,9,19,33,17,133,425,797,1721,153,119,0,0,
      1,3,3,7,13,37,1,215,509,1003,61,2353,7511,0,0,
      1,1,7,1,29,19,31,79,199,555,1209,1603,6089,0,0,
      1,3,1,1,5,31,111,127,333,429,1863,3925,5411,0,0,
      1,1,7,5,5,5,123,191,47,993,269,4051,2111,0,0,
      1,1,5,15,1,9,87,5,47,463,865,1813,7357,0,0,
      1,3,1,3,23,63,123,83,511,777,63,1285,4537,0,0,
      1,3,3,7,27,25,31,65,441,529,1815,1893,323,0,0,
      1,3,7,5,11,19,7,5,397,811,755,2883,4217,0,0,
      1,3,1,13,9,21,13,7,271,539,1769,3243,5325,0,0,
      1,1,7,1,31,13,47,131,181,457,1559,2663,6653,0,0,
      1,3,3,7,29,55,25,203,419,91,437,1159,5691,0,0,
      1,1,3,13,29,19,71,217,337,329,501,939,2205,0,0,
      1,1,3,1,1,27,17,201,97,285,1269,4043,2207,0,0,
      1,1,1,1,3,41,13,199,141,129,1515,3129,5969,0,0,
      1,3,3,9,3,17,119,41,271,933,877,701,2197,0,0,
      1,1,1,7,15,47,3,195,115,821,725,843,6071,0,0,
      1,3,5,15,17,33,85,65,297,571,1123,2743,5727,0,0,
      1,1,5,11,27,15,37,235,415,293,1439,2739,4171,0,0,
      1,3,7,7,1,55,71,35,307,11,401,1881,933,0,0,
      1,3,1,11,21,37,3,177,119,339,559,3991,3437,0,0,
      1,3,3,9,17,17,97,119,301,169,157,3267,2261,0,0,
      1,3,3,9,29,3,111,101,355,869,375,2609,7377,0,0,
      1,3,5,9,7,21,123,99,343,693,1927,1605,4923,0,0,
      1,1,3,5,13,31,99,17,75,385,1539,1553,7077,0,0,
      1,3,3,5,31,35,107,11,407,1019,1317,3593,7203,0,0,
      1,3,3,13,17,33,99,245,401,957,157,1949,1571,0,0,
      1,3,1,11,27,15,11,109,429,307,1911,2701,861,0,0,
      1,1,5,13,13,35,55,255,311,957,1803,2673,5195,0,0,
      1,1,1,11,19,3,89,37,211,783,1355,3567,7135,0,0,
      1,1,5,5,21,49,79,17,509,331,183,3831,855,0,0,
      1,3,7,5,29,19,85,109,105,523,845,3385,7477,0,0,
      1,1,1,7,25,17,125,131,53,757,253,2989,2939,0,0,
      1,3,3,9,19,23,105,39,351,677,211,401,8103,0,0,
      1,3,5,1,5,11,17,3,405,469,1569,2865,3133,0,0,
      1,1,3,13,15,5,117,179,139,145,477,1137,2537,0,0,
      1,1,7,9,5,21,9,93,211,963,1207,3343,4911,0,0,
      1,1,1,9,13,43,17,53,81,793,1571,2523,3683,0,0,
      1,3,3,13,25,21,5,59,489,987,1941,171,6009,0,0,
      1,3,3,7,1,39,89,171,403,467,1767,3423,2791,0,0,
      1,1,3,9,19,49,91,125,163,1013,89,2849,6785,0,0,
      1,1,5,9,9,11,15,241,43,297,1719,1541,1821,0,0,
      1,3,7,15,29,23,103,239,191,33,1043,3649,6579,0,0,
      1,3,3,9,21,51,123,55,223,645,1463,4021,5891,0,0,
      1,1,5,7,3,41,27,235,391,303,2021,3187,7607,0,0,
      1,1,1,9,5,49,49,29,377,251,1887,1017,1301,0,0,
      1,1,3,3,13,41,27,47,223,23,517,3227,6731,0,0,
      1,1,7,1,31,25,47,9,511,623,2047,1263,1511,0,0,
      1,1,3,15,15,23,53,1,261,595,85,241,7047,0,0,
      1,3,3,11,17,5,81,73,149,781,2035,3163,4247,0,0,
      1,3,7,7,29,59,49,79,397,901,1105,2191,6277,0,0,
      1,3,3,11,13,27,25,173,107,73,1265,585,5251,0,0,
      1,1,7,15,29,23,73,229,235,887,1469,4073,2591,0,0,
      1,1,3,9,17,15,83,173,207,879,1701,1509,11,0,0,
      1,1,3,5,5,37,65,161,39,421,1153,2007,5355,0,0,
      1,1,7,11,23,37,5,11,9,499,17,157,5747,0,0,
      1,3,7,13,25,9,49,7,39,945,1349,1759,1441,0,0,
      1,1,5,3,21,15,113,81,265,837,333,3625,6133,0,0,
      1,3,1,11,13,27,73,109,297,327,299,3253,6957,0,0,
      1,1,3,13,19,39,123,73,65,5,1061,2187,5055,0,0,
      1,1,3,1,11,31,21,115,453,857,711,495,549,0,0,
      1,3,7,7,15,29,79,103,47,713,1735,3121,6321,0,0,
      1,1,5,5,29,9,97,33,471,705,329,1501,1349,0,0,
      1,3,3,1,21,9,111,209,71,47,491,2143,1797,0,0,
      1,3,3,3,11,39,21,135,445,259,607,3811,5449,0,0,
      1,1,7,9,11,25,113,251,395,317,317,91,1979,0,0,
      1,3,1,9,3,21,103,133,389,943,1235,1749,7063,0,0,
      1,1,3,7,1,11,5,15,497,477,479,3079,6969,0,0,
      1,1,3,3,15,39,105,131,475,465,181,865,3813,0,0,
      1,1,7,9,19,63,123,131,415,525,457,2471,3135,0,0,
      1,3,7,15,25,35,123,45,341,805,485,4049,7065,0,0,
      1,1,1,5,29,9,47,227,51,867,1873,1593,2271,0,0,
      1,1,7,15,31,9,71,117,285,711,837,1435,6275,0,0,
      1,3,1,1,5,19,79,25,301,415,1871,645,3251,0,0,
      1,3,1,3,17,51,99,185,447,43,523,219,429,0,0,
      1,3,1,13,29,13,51,93,7,995,757,3017,6865,0,0,
      1,1,3,15,7,25,75,17,155,981,1231,1229,1995,0,0,
      1,3,5,3,27,45,71,73,225,763,377,1139,2863,0,0,
      1,1,3,1,1,39,69,113,29,371,1051,793,3749,0,0,
      1,1,3,13,23,61,27,183,307,431,1345,2757,4031,0,0,
      1,3,7,5,5,59,117,197,303,721,877,723,1601,0,0,
      1,3,5,1,27,33,99,237,485,711,665,3077,5105,0,0,
      1,1,3,1,13,9,103,201,23,951,2029,165,2093,0,0,
      1,3,5,13,5,29,55,85,221,677,611,3613,4567,0,0,
      1,1,1,1,7,61,9,233,261,561,953,4023,2443,0,0,
      1,3,3,13,1,17,103,71,223,213,833,1747,6999,0,0,
      1,3,5,15,25,53,57,187,25,695,1207,4089,2877,0,0,
      1,1,7,1,7,31,87,129,493,519,1555,1155,4637,0,0,
      1,1,1,15,21,17,23,29,19,255,927,1791,3093,0,0,
      1,1,3,9,17,33,95,129,175,461,287,2633,2325,0,0,
      1,3,5,7,23,19,63,209,249,583,1373,2039,2225,0,0,
      1,3,3,5,5,19,79,241,459,355,1455,3313,3639,0,0,
      1,1,7,9,21,41,97,119,129,769,1541,3495,7741,0,0,
      1,1,7,11,9,29,35,255,141,937,1763,41,1393,0,0,
      1,3,7,1,13,51,61,157,177,847,1829,3539,285,0,0,
      1,1,1,15,21,13,9,55,397,19,1495,1255,7235,0,0,
      1,1,7,7,25,37,53,237,319,197,269,1205,1485,0,0,
      1,1,5,15,23,17,35,247,323,807,233,3681,4407,0,0,
      1,1,3,7,9,59,85,105,493,763,1639,391,1451,0,0,
      1,3,3,9,15,33,5,253,129,625,1527,2793,6057,0,0,
      1,3,1,1,7,47,21,161,235,83,397,3563,5953,0,0,
      1,3,7,11,3,41,25,117,375,779,1297,3715,8117,0,0,
      1,1,3,7,31,19,103,173,475,189,2035,2921,1107,0,0,
      1,1,7,3,25,7,93,255,307,113,1893,2233,6919,0,0,
      1,3,5,15,9,57,79,143,165,5,1389,193,693,0,0,
      1,3,5,1,29,45,91,49,189,461,439,1283,7835,0,0,
      1,1,3,13,11,61,41,231,373,695,395,915,5393,0,0,
      1,3,7,11,5,51,67,53,483,95,1943,247,5653,0,0,
      1,3,7,5,5,57,45,235,137,793,1069,1661,1557,0,0,
      1,3,5,3,25,55,103,177,81,861,1151,143,7655,0,0,
      1,1,3,1,21,41,67,131,253,431,1269,3181,3429,0,0,
      1,3,1,1,21,7,77,221,257,663,71,2949,2481,0,0,
      1,3,5,3,3,23,45,107,299,739,1013,3,3165,0,0,
      1,1,5,1,3,37,109,37,243,983,1221,1691,3869,0,0,
      1,1,5,5,31,7,5,193,397,867,1495,3435,7441,0,0,
      1,1,1,1,17,59,97,233,389,597,1013,1631,483,0,0,
      1,1,1,11,7,41,107,53,111,125,1513,1921,7647,0,0,
      1,3,3,3,31,29,117,3,365,971,1139,2123,5913,0,0,
      1,1,1,13,23,3,1,167,475,639,1811,3841,3081,0,0,
      1,1,5,3,5,47,65,123,275,783,95,119,7591,0,0,
      1,3,1,15,13,33,93,237,467,431,705,4013,4035,0,0,
      1,3,5,1,19,7,101,231,155,737,1381,3343,2051,0,0,
      1,1,5,9,15,49,45,163,433,765,2031,201,2589,0,0,
      1,3,7,9,19,41,31,89,93,623,105,745,4409,0,0,
      1,1,5,1,11,45,127,85,389,439,829,477,7965,0,0,
      1,3,3,15,13,41,1,207,435,585,311,1725,2737,0,0,
      1,3,3,3,13,49,21,31,197,799,1411,2959,7133,0,0,
      1,3,1,3,7,43,9,141,133,579,1059,93,957,0,0,
      1,3,7,1,15,51,23,213,381,851,699,2261,3419,0,0,
      1,3,5,9,25,35,67,141,35,409,1423,365,1645,0,0,
      1,3,3,11,15,33,27,181,93,87,1761,3511,1353,0,0,
      1,3,5,3,25,63,111,137,321,819,705,1547,7271,0,0,
      1,3,1,1,5,57,99,59,411,757,1371,3953,3695,0,0,
      1,3,5,11,11,21,25,147,239,455,709,953,7175,0,0,
      1,3,3,15,5,53,91,205,341,63,723,1565,7135,0,0,
      1,1,7,15,11,21,99,79,63,593,2007,3629,5271,0,0,
      1,3,3,1,9,21,45,175,453,435,1855,2649,6959,0,0,
      1,1,3,15,15,33,121,121,251,431,1127,3305,4199,0,0,
      1,1,1,9,31,15,71,29,345,391,1159,2809,345,0,0,
      1,3,7,1,23,29,95,151,327,727,647,1623,2971,0,0,
      1,1,7,7,9,29,79,91,127,909,1293,1315,5315,0,0,
      1,1,5,11,13,37,89,73,149,477,1909,3343,525,0,0,
      1,3,5,7,5,59,55,255,223,459,2027,237,4205,0,0,
      1,1,1,7,27,11,95,65,325,835,907,3801,3787,0,0,
      1,1,1,11,27,33,99,175,51,913,331,1851,4133,0,0,
      1,3,5,5,13,37,31,99,273,409,1827,3845,5491,0,0,
      1,1,3,7,23,19,107,85,283,523,509,451,421,0,0,
      1,3,5,7,13,9,51,81,87,619,61,2803,5271,0,0,
      1,1,1,15,9,45,35,219,401,271,953,649,6847,0,0,
      1,1,7,11,9,45,17,219,169,837,1483,1605,2901,0,0,
      1,1,7,7,21,43,37,33,291,359,71,2899,7037,0,0,
      1,3,3,13,31,53,37,15,149,949,551,3445,5455,0,0,
      1,3,1,5,19,45,81,223,193,439,2047,3879,789,0,0,
      1,1,7,3,11,63,35,61,255,563,459,2991,3359,0,0,
      1,1,5,9,13,49,47,185,239,221,1533,3635,2045,0,0,
      1,3,7,3,25,37,127,223,51,357,483,3837,6873,0,0,
      1,1,7,9,31,37,113,31,387,833,1243,1543,5535,0,0,
      1,3,1,9,23,59,119,221,73,185,2007,2885,2563,0,0,
      1,1,1,13,7,33,53,179,67,185,1541,1807,4659,0,0,
      1,3,1,11,31,37,23,215,269,357,207,645,4219,0,0,
      1,3,3,13,19,27,107,55,91,71,1695,1815,89,0,0,
      1,1,3,15,3,19,35,247,49,529,1523,3317,6151,0,0,
      1,1,7,7,23,25,107,139,483,503,1277,243,7879,0,0,
      1,3,3,13,3,15,11,197,135,839,985,275,5527,0,0,
      1,3,5,3,25,47,95,21,113,307,1001,3065,295,0,0,
      1,1,3,9,19,19,99,213,363,449,735,2851,2521,0,0,
      1,1,3,9,5,49,63,61,157,857,497,2801,6987,0,0,
      1,1,1,9,1,41,109,119,499,939,867,3675,8023,0,0,
      1,3,1,1,13,33,109,123,289,3,1271,2773,4265,0,0,
      1,3,1,11,9,57,83,221,95,43,1189,457,7133,0,0,
      1,1,7,3,11,49,33,219,229,289,685,3359,4495,0,0,
      1,3,1,3,19,43,67,193,41,771,407,81,3891,0,0,
      1,1,7,11,5,29,51,175,297,539,1,2245,6439,0,0,
      1,3,7,15,21,33,117,183,511,489,1283,3281,5979,0,0,
      1,3,7,5,9,3,125,147,359,549,369,3049,2405,0,0,
      1,3,5,7,19,5,65,97,483,377,1523,1457,2995,0,0,
      1,1,5,1,11,21,41,113,277,131,1475,1043,2367,0,0,
      1,3,3,1,15,17,101,69,443,865,817,1421,5231,0,0,
      1,1,3,3,3,55,95,99,75,195,1929,3931,5855,0,0,
      1,3,1,3,19,23,93,213,241,551,1307,585,7729,0,0,
      1,3,1,11,23,15,53,249,467,519,95,741,409,0,0,
      1,1,1,15,29,37,43,203,233,877,77,1933,2729,0,0,
      1,3,7,11,27,39,43,161,255,15,1463,833,495,0,0,
      1,1,7,11,3,53,81,67,375,823,1903,3061,395,0,0,
      1,1,1,1,15,37,93,233,247,501,1321,3275,5409,0,0,
      1,3,3,7,7,11,5,105,139,983,1239,531,3881,0,0,
      1,1,5,3,19,49,107,227,361,101,355,2649,7383,0,0,
      1,1,7,5,25,41,101,121,209,293,1937,2259,5557,0,0,
      1,1,3,7,7,1,9,13,463,1019,995,3159,107,0,0,
      1,3,5,11,5,35,127,97,261,789,807,807,6257,0,0,
      1,1,7,5,11,13,45,91,417,101,1973,3645,2107,0,0,
      1,1,3,7,5,63,57,49,203,157,115,1393,8117,0,0,
      1,3,5,5,3,43,15,155,127,489,1165,3701,4867,0,0,
      1,1,7,7,29,29,69,215,415,367,371,1901,6075,0,0,
      1,1,1,3,11,33,89,149,433,705,1437,1597,505,0,0,
      1,3,5,1,13,37,19,119,5,581,2037,1633,2099,0,0,
      1,3,7,13,5,49,103,245,215,515,133,2007,1933,0,0,
      1,3,1,9,1,3,25,197,253,387,1683,2267,221,0,0,
      1,3,5,15,21,9,73,201,405,999,437,3877,6045,0,0,
      1,1,3,1,31,55,25,83,421,395,1807,2129,7797,0,0,
      1,1,3,1,23,21,121,183,125,347,143,3685,4317,0,0,
      1,3,3,3,17,45,17,223,267,795,1815,1309,155,0,0,
      1,1,1,15,17,59,5,133,15,715,1503,153,2887,0,0,
      1,1,1,1,27,13,119,77,243,995,1851,3719,4695,0,0,
      1,3,1,5,31,49,43,165,49,609,1265,1141,505,0,0,
      1,1,7,13,11,63,21,253,229,585,1543,3719,4141,0,0,
      1,3,7,11,23,27,17,131,295,895,1493,1411,3247,0,0,
      1,1,5,9,29,7,97,15,113,445,859,1483,1121,0,0,
      1,3,1,9,13,49,99,107,323,201,681,3071,5281,0,0,
      1,1,1,15,9,19,61,161,7,87,587,2199,2811,0,0,
      1,3,3,15,15,19,95,45,299,829,981,3479,487,0,0,
      1,1,1,9,3,37,7,19,227,13,397,513,1257,0,0,
      1,1,5,15,15,13,17,111,135,929,1145,811,1801,0,0,
      1,3,1,3,27,57,31,19,279,103,693,631,3409,0,0,
      1,1,1,1,15,13,67,83,23,799,1735,2063,3363,0,0,
      1,3,3,7,3,1,61,31,41,533,2025,4067,6963,0,0,
      1,1,5,7,17,27,81,79,107,205,29,97,4883,0,0,
      1,1,1,5,19,49,91,201,283,949,651,3819,5073,0,0,
      1,1,7,9,11,13,73,197,37,219,1931,3369,6017,0,0,
      1,1,7,15,11,7,75,205,7,819,399,661,6487,0,0,
      1,3,3,3,27,37,95,41,307,165,1077,3485,563,0,0,
      1,3,5,3,21,49,57,179,109,627,1789,431,2941,0,0,
      1,1,7,5,11,19,43,137,149,679,1543,245,1381,0,0,
      1,3,5,5,15,3,69,81,135,159,1363,3401,6355,0,0,
      1,3,5,1,9,61,49,53,319,25,1647,1297,615,0,0,
      1,3,5,11,31,43,9,101,71,919,335,3147,5823,0,0,
      1,3,1,1,15,5,29,109,511,945,867,3677,6915,0,0,
      1,3,3,15,17,49,91,111,215,29,1879,97,2505,0,0,
      1,3,1,13,19,61,11,111,163,777,533,1113,5339,0,0,
      1,1,7,9,17,55,117,91,455,289,557,913,4455,0,0,
      1,3,1,7,25,19,123,37,1,277,717,2965,4469,0,0,
      1,3,7,3,19,23,87,235,209,457,2041,2893,1805,0,0,
      1,3,3,5,5,43,23,61,351,791,59,2009,2909,0,0,
      1,1,3,7,5,1,27,231,385,257,1261,2701,1807,0,0,
      1,3,1,1,27,19,87,253,131,685,1743,3983,2651,0,0,
      1,3,7,11,21,17,11,81,191,641,1821,3005,7251,0,0,
      1,3,3,5,15,31,41,213,55,931,1953,49,6037,0,0,
      1,1,7,15,7,27,65,223,113,79,1875,911,5445,0,0,
      1,3,7,7,23,55,51,167,495,25,1585,3447,799,0,0,
      1,1,3,7,27,15,95,193,337,415,975,3085,967,0,0,
      1,1,7,15,19,7,93,41,433,551,401,3169,3971,0,0,
      1,1,7,11,13,15,53,69,433,59,1117,3359,6231,0,0,
      1,1,7,3,23,5,115,201,225,109,1903,3897,6265,0,0,
      1,1,1,11,17,1,39,143,361,659,1105,23,4923,0,0,
      1,1,1,9,27,57,85,227,261,119,1881,3965,6999,0,0,
      1,3,7,7,15,7,107,17,315,49,1591,905,7789,0,0,
      1,3,1,7,29,3,47,237,157,769,839,3199,3195,0,0,
      1,1,3,15,25,39,63,15,111,857,881,1505,7671,0,0,
      1,1,7,1,3,35,41,215,99,895,1025,1483,4707,0,0,
      1,3,5,1,1,31,25,247,113,841,397,1825,6969,0,0,
      1,1,3,5,19,41,49,243,225,973,241,175,1041,0,0,
      1,1,1,7,15,15,105,141,83,75,1675,3523,5219,0,0,
      1,1,7,5,13,27,47,199,445,841,959,1157,2209,0,0,
      1,3,5,15,23,31,31,81,85,33,785,2639,7799,0,0,
      1,1,5,13,21,3,47,99,235,943,1731,2467,7891,0,0,
      1,1,1,3,17,53,85,219,73,131,1339,875,1191,0,0,
      1,1,5,7,17,63,113,7,185,557,749,3563,4973,0,0,
      1,3,3,15,15,21,43,111,155,689,345,423,3597,0,0,
      1,1,5,1,15,29,93,5,361,713,695,3937,425,0,0,
      1,3,7,7,13,41,115,175,315,937,123,2841,4457,0,0,
      1,1,3,11,25,5,103,53,423,811,657,399,7257,0,0,
      1,1,1,1,1,13,101,211,383,325,97,1703,4429,0,0,
      1,3,7,9,31,45,83,157,509,701,841,1105,3643,0,0,
      1,1,1,7,1,9,69,17,129,281,1161,2945,7693,0,0,
      1,3,7,1,11,29,51,143,77,433,1723,2317,5641,0,0,
      1,1,1,1,21,43,13,67,177,505,1629,1267,4885,0,0,
      1,1,3,11,27,63,111,47,233,781,453,1679,3209,0,0,
      1,1,3,13,29,27,119,141,493,971,461,1159,633,0,0,
      1,1,3,15,23,5,79,215,163,149,1805,2399,61,0,0,
      1,3,5,13,19,5,1,39,409,561,709,829,1357,0,0,
      1,3,3,13,19,43,9,177,449,447,73,2107,5669,0,0,
      1,3,5,1,23,13,63,109,203,593,829,4017,6881,0,0,
      1,1,5,7,3,9,53,175,391,169,1283,3793,4451,0,0,
      1,1,5,7,29,43,9,5,209,77,927,2941,8145,0,0,
      1,3,5,15,17,49,5,143,131,771,1685,925,2175,0,0,
      1,1,3,11,27,27,27,159,161,1015,1587,4049,1983,0,0,
      1,3,1,3,23,57,119,67,481,577,389,3319,5325,0,0,
      1,3,5,1,19,39,87,61,329,657,1773,31,1707,0,0,
      1,1,3,1,5,25,15,241,131,815,1751,3029,8039,0,0,
      1,3,3,13,27,13,77,87,437,57,621,1031,7891,0,0,
      1,3,1,13,23,51,117,37,331,745,605,3179,4713,0,0,
      1,1,5,5,19,17,99,167,87,721,737,789,2165,0,0,
      1,3,5,13,1,51,119,211,165,299,1327,3053,3343,0,0,
      1,1,5,15,29,45,17,129,67,345,1553,2705,7369,0,0,
      1,1,1,9,23,7,13,209,7,407,317,3077,7287,0,0,
      1,1,1,5,9,59,89,3,487,451,505,2499,7563,0,0,
      1,3,1,7,21,1,21,203,101,417,1389,2751,1397,0,0,
      1,3,7,13,7,31,3,247,349,485,1259,549,6321,0,0,
      1,1,7,7,27,33,107,197,293,729,1753,2571,103,0,0,
      1,3,5,9,25,35,5,253,137,213,2041,3387,1809,0,0,
      1,1,7,13,15,35,67,83,295,175,839,2831,839,0,0,
      1,3,3,11,3,17,55,141,247,991,117,3799,1221,0,0,
      1,1,5,1,11,37,87,233,457,653,899,2933,3105,0,0,
      1,1,3,15,3,31,67,167,437,9,651,1109,1139,0,0,
      1,1,3,1,7,63,67,17,11,883,1855,1941,4751,0,0,
      1,3,7,9,19,33,113,117,495,39,1795,2561,5519,0,0,
      1,1,7,5,1,3,103,37,201,223,1101,877,6483,0,0,
      1,1,5,9,29,49,51,33,439,917,861,1321,2135,0,0,
      1,1,3,3,1,5,17,93,217,619,613,1357,6095,0,0,
      1,3,1,11,3,21,5,41,15,175,843,2937,6849,0,0,
      1,3,3,7,9,57,55,127,79,287,445,2205,7989,0,0,
      1,1,7,13,23,17,93,129,157,135,1747,1813,4183,0,0,
      1,1,1,5,31,59,99,33,425,329,887,367,1761,0,0,
      1,1,7,9,17,53,77,139,435,387,49,3649,1773,0,0,
      1,3,3,15,21,57,45,161,331,719,273,3479,4173,0,0,
      1,1,3,9,3,3,105,201,373,877,919,1263,6649,0,0,
      1,3,1,15,13,43,13,99,73,163,353,3569,5601,0,0,
      1,3,7,3,5,9,69,177,449,47,781,1125,4245,0,0,
      1,1,1,5,3,45,1,123,409,903,205,2057,7637,0,0,
      1,3,5,9,19,47,87,135,481,799,101,3409,2241,0,0,
      1,3,1,13,3,25,15,27,181,967,669,2577,7249,0,0,
      1,1,7,3,31,5,103,53,1,911,1209,3697,6685,0,0,
      1,1,3,1,5,5,49,135,281,747,761,2973,7963,0,0,
      1,3,3,5,19,61,125,199,299,515,1365,369,7027,0,0,
      1,3,1,7,5,41,63,229,283,571,147,447,657,0,0,
      1,3,1,11,5,15,55,7,259,61,27,1429,5631,0,0,
      1,1,5,1,3,53,51,253,155,553,1293,3735,6567,0,0,
      1,3,5,9,5,41,21,159,101,785,1981,3799,7693,0,0,
      1,3,7,7,9,3,95,105,129,213,1215,1027,5699,0,0,
      1,1,3,3,29,13,9,253,449,321,341,2879,171,0,0,
      1,3,7,11,21,11,75,35,43,965,675,2217,7175,0,0,
      1,1,5,15,31,5,29,137,311,751,47,1367,5921,0,0,
      1,1,3,15,17,1,45,69,55,649,835,569,7615,0,0,
      1,3,1,13,31,7,23,15,391,145,1845,1825,1403,0,0,
      1,1,3,15,5,9,79,77,105,399,1933,2503,4781,0,0,
      1,3,1,3,17,47,19,13,107,475,759,2933,3761,0,0,
      1,1,7,11,3,7,121,209,397,877,293,847,7039,0,0,
      1,1,1,15,29,45,5,109,335,461,143,931,4045,0,0,
      1,3,1,7,11,57,73,89,201,173,803,3953,5205,0,0,
      1,1,5,11,11,33,37,29,263,1019,657,1453,7807,0,0,
      1,3,3,13,31,25,37,47,261,607,1703,2603,417,0,0,
      1,1,1,1,31,61,45,115,275,239,1989,1897,4329,0,0,
      1,3,5,3,31,3,11,173,335,579,1193,2219,7875,0,0,
      1,1,7,9,29,45,13,67,399,177,1293,3865,2225,0,0,
      1,1,7,11,11,51,121,227,469,905,929,2635,4165,0,0,
      1,3,7,9,13,39,55,167,23,147,1603,2083,4645,0,0,
      1,1,3,15,27,53,11,155,157,629,259,3009,4605,0,0,
      1,3,1,7,15,47,51,1,259,603,887,2833,6581,0,0,
      1,3,5,3,1,47,91,43,361,571,29,1453,4269,0,0,
      1,1,3,9,11,51,55,23,415,277,1423,3475,1527,0,0,
      1,1,3,11,29,49,101,75,299,709,805,4037,4389,0,0,
      1,1,7,3,23,1,37,51,379,771,1301,3717,6673,0,0,
      1,1,5,3,23,11,125,177,375,665,951,1577,2603,0,0,
      1,1,1,1,1,5,71,255,21,459,467,2083,5415,0,0,
      1,1,5,13,23,29,109,157,363,971,549,647,1177,0,0,
      1,1,3,9,7,15,101,3,365,213,745,1155,6099,0,0,
      1,3,5,15,15,19,47,179,303,521,1279,219,2415,0,0,
      1,3,3,13,27,11,83,165,369,989,261,3933,4809,0,0,
      1,1,3,11,31,59,1,185,53,703,1471,2935,1107,0,0,
      1,3,3,7,25,3,81,27,93,521,433,2859,5861,0,0,
      1,3,3,11,29,15,49,167,315,927,543,3473,4307,0,0,
      1,3,1,3,29,33,53,15,183,691,703,1311,3393,0,0,
      1,3,5,13,23,49,3,11,1,357,1407,415,7211,0,0,
      1,3,7,15,1,25,91,113,323,371,189,925,1181,0,0,
      1,3,3,3,17,59,119,199,115,223,877,2193,193,0,0,
      1,1,1,5,5,35,31,59,437,411,37,2405,3797,0,0,
      1,3,1,13,9,37,1,241,59,157,1785,1223,563,0,0,
      1,3,5,13,3,21,25,95,15,745,85,701,5361,0,0,
      1,3,7,1,31,33,111,195,35,913,2013,2951,6611,0,0,
      1,3,5,1,19,3,75,119,111,409,951,1457,4957,0,0,
      1,3,1,15,19,59,3,155,237,657,1967,3323,6235,0,0,
      1,1,5,1,3,19,45,105,377,881,167,2255,4483,0,0,
      1,1,7,7,13,13,99,89,201,279,161,2483,6001,0,0,
      1,1,7,3,13,17,97,129,137,377,1519,183,3725,0,0,
      1,1,7,9,9,5,45,135,115,181,1685,3505,4387,0,0,
      1,1,1,1,19,35,69,113,305,419,949,2969,247,0,0,
      1,1,5,13,23,61,13,139,501,811,67,1501,6493,0,0,
      1,1,3,13,15,41,27,217,293,13,145,2631,6991,0,0,
      1,3,3,13,15,37,71,123,285,49,627,1283,5993,0,0,
      1,3,3,11,9,25,11,1,203,353,1347,1999,2799,0,0,
      1,3,5,1,7,49,101,231,499,63,1977,2207,7829,0,0,
      1,1,7,1,17,15,115,139,381,943,623,4037,2971,0,0,
      1,1,3,5,13,55,23,87,139,795,1669,1375,1185,0,0,
      1,3,3,5,5,45,97,253,241,333,645,555,7867,0,0,
      1,3,5,1,1,1,89,27,407,509,1433,609,2355,0,0,
      1,3,7,1,27,29,5,157,495,811,1293,1143,827,0,0,
      1,1,3,3,25,49,127,111,191,3,845,1383,2521,0,0,
      1,1,5,7,5,51,101,155,237,461,831,3091,3851,0,0,
      1,3,7,1,29,35,105,91,285,705,131,395,6011,0,0,
      1,3,5,3,13,21,83,173,221,827,1775,1931,6727,0,0,
      1,1,3,5,3,25,95,115,205,569,1447,933,6425,0,0,
      1,1,7,9,31,3,17,175,145,447,1321,1069,6527,0,0,
      1,1,3,3,23,1,79,51,421,419,873,3939,1801,0,0,
      1,1,5,1,3,39,15,85,169,669,919,397,5579,0,0,
      1,3,5,1,21,61,87,217,251,619,1091,4009,229,0,0,
      1,1,1,11,23,55,85,121,363,867,315,447,3373,0,0,
      1,3,3,13,29,19,89,85,137,469,1873,2765,3975,0,0,
      1,3,7,13,19,63,61,77,67,361,11,1787,4703,0,0,
      1,1,3,11,7,15,127,105,179,857,1671,3647,3389,0,0,
      1,1,1,7,19,21,99,161,499,519,1287,2973,479,0,0,
      1,1,3,13,29,51,95,251,55,519,1955,2881,5951,0,0,
      1,1,3,11,23,63,121,237,175,311,701,1539,2383,0,0,
      1,1,7,5,5,45,73,97,5,153,715,2037,3353,0,0,
      1,1,1,3,13,7,67,173,425,843,1497,2729,5193,0,0,
      1,1,7,1,23,3,119,11,77,141,1905,2269,4269,0,0,
      1,1,7,15,1,23,79,251,439,603,405,2449,6383,0,0,
      1,3,7,11,29,27,47,255,47,661,1967,1007,3689,0,0,
      1,3,7,5,19,39,35,115,417,373,291,329,603,0,0,
      1,3,1,9,11,33,27,193,207,423,1311,1369,7307,0,0,
      1,1,3,11,9,29,83,17,497,493,329,3141,5935,0,0,
      1,3,1,5,31,51,29,171,51,493,1621,3501,4091,0,0,
      1,1,5,9,21,43,105,207,245,363,1191,699,1139,0,0,
      1,1,3,11,19,5,81,119,247,169,1337,45,6565,0,0,
      1,3,1,11,3,51,3,101,159,11,253,299,5043,0,0,
      1,3,1,5,11,53,85,39,57,645,2007,1039,3627,0,0,
      1,3,5,3,17,61,97,165,415,357,283,601,5505,0,0,
      1,3,7,3,9,51,49,85,3,227,137,309,243,0,0,
      1,1,5,3,11,59,11,131,409,703,455,123,6727,0,0,
      1,3,7,9,25,49,21,171,287,379,667,313,713,0,0,
      1,1,3,9,7,35,47,3,367,581,1627,1665,3905,0,0,
      1,3,1,1,29,57,35,55,255,653,823,2197,6179,0,0,
      1,3,7,15,17,15,117,83,359,163,115,2999,5373,0,0,
      1,1,5,3,21,61,35,97,71,687,207,2917,1049,0,0,
      1,1,1,15,13,15,125,81,263,661,417,3243,1669,0,0,
      1,1,7,3,3,19,111,193,443,339,659,1211,1557,0,0,
      1,3,1,3,27,3,3,173,391,213,803,3281,3207,0,0,
      1,1,5,15,19,1,7,211,157,603,403,1387,1583,0,0,
      1,3,5,13,17,53,125,13,339,723,521,413,5801,10451,0,
      1,1,3,13,29,9,99,77,141,609,1533,983,2039,51,0,
      1,1,3,11,21,55,5,51,423,309,525,3715,3025,15055,0,
      1,1,3,7,9,21,77,171,239,341,1653,1093,2273,10723,0,
      1,1,1,15,31,15,23,35,317,869,1917,1415,4313,3687,0,
      1,1,1,5,21,25,99,167,439,453,473,431,6665,4989,0,
      1,1,7,9,31,47,81,83,345,43,1363,1885,3155,3185,0,
      1,3,7,1,31,17,61,185,341,129,547,717,2071,9991,0,
      1,3,1,13,23,61,77,217,455,77,1263,1601,3501,14953,0,
      1,1,7,7,19,19,1,229,431,943,1069,1949,1289,15729,0,
      1,1,3,5,1,35,97,251,487,459,1265,1739,165,10365,0,
      1,3,5,3,11,25,79,175,383,545,187,197,4329,3363,0,
      1,1,3,3,29,9,63,55,175,277,431,2549,2629,6409,0,
      1,1,3,15,17,21,79,139,99,135,1763,1805,3471,5439,0,
      1,1,3,9,9,15,35,119,289,835,769,3843,4119,4421,0,
      1,1,1,5,19,19,67,199,307,815,1367,1231,3927,6593,0,
      1,1,3,1,29,51,121,209,431,47,1115,907,2535,9755,0,
      1,1,3,5,17,1,5,119,121,223,1719,1291,3947,15891,0,
      1,3,1,15,29,25,3,131,373,307,645,3513,1289,1987,0,
      1,3,3,11,29,45,105,179,331,465,891,1315,403,3057,0,
      1,1,5,13,17,59,77,127,485,855,1147,3093,891,9869,0,
      1,1,1,7,23,27,31,203,285,463,827,685,1349,15051,0,
      1,1,1,5,29,5,107,195,31,425,19,2865,3869,11153,0,
      1,1,7,5,7,47,1,73,307,347,393,2205,7709,15121,0,
      1,1,1,13,15,61,25,131,113,369,1995,2527,4475,1745,0,
      1,1,1,1,31,63,21,253,307,281,859,3319,6721,2891,0,
      1,1,3,11,1,17,5,183,301,979,651,1685,6343,10067,0,
      1,1,5,15,23,45,99,145,263,507,1381,3425,2215,1815,0,
      1,3,1,5,11,63,85,203,411,881,1369,1237,4657,6541,0,
      1,3,3,13,17,53,121,201,269,983,215,3187,7121,6111,0,
      1,3,5,15,15,5,13,143,3,313,1677,1093,3295,3387,0,
      1,1,3,13,3,23,73,17,257,965,239,1271,2803,7327,0,
      1,3,5,13,9,57,115,37,41,467,135,1403,3811,4741,0,
      1,3,7,15,9,33,39,203,351,367,1355,1403,3685,4757,0,
      1,3,5,11,31,3,113,123,203,421,1821,3151,2375,4419,0,
      1,1,1,7,21,63,99,23,133,79,991,1755,4989,4709,0,
      1,3,5,1,25,63,113,239,49,443,173,1261,3201,10599,0,
      1,3,3,13,3,25,101,169,23,585,327,1327,111,10059,0,
      1,3,3,5,19,1,33,89,437,213,1933,1741,2603,5625,0,
      1,3,1,3,15,15,25,139,73,335,237,2461,3101,14951,0,
      1,3,5,1,31,15,31,187,177,659,1339,3767,4975,7123,0,
      1,3,1,3,25,19,47,89,107,107,649,683,3123,11085,0,
      1,3,7,9,15,21,101,25,11,625,1555,675,3893,5805,0,
      1,1,1,5,7,49,123,21,439,369,207,535,4619,14665,0,
      1,1,5,7,1,25,103,185,99,239,1093,1561,6177,4039,0,
      1,3,7,5,29,21,43,103,343,973,1561,2975,7467,7947,0,
      1,1,7,9,19,3,13,23,461,813,1191,985,559,3317,0,
      1,3,5,5,27,31,79,15,365,901,1949,117,3619,13311,0,
      1,3,5,7,5,33,67,199,425,189,1691,3099,815,1677,0,
      1,1,7,11,13,29,73,137,265,601,445,3893,2511,8047,0,
      1,1,3,1,13,5,57,101,357,391,335,601,1359,1065,0,
      1,1,1,1,25,57,27,115,31,873,611,2125,447,13585,0,
      1,3,3,13,27,17,73,11,359,33,1153,271,4537,15141,0,
      1,3,7,3,11,63,103,61,59,629,1629,3279,3919,3177,0,
      1,1,5,15,3,63,85,193,381,165,175,3247,2501,4209,0,
      1,1,5,15,1,33,59,219,487,193,1557,703,2907,7953,0,
      1,1,7,3,9,3,105,95,389,991,21,3841,6983,285,0,
      1,1,1,1,1,31,25,137,117,67,1283,1963,6591,15541,0,
      1,3,5,11,7,15,127,89,453,777,1827,2311,7661,11833,0,
      1,1,7,13,19,29,79,165,223,453,2039,3961,6467,5481,0,
      1,3,3,7,17,41,43,157,323,3,1001,2109,4513,12127,0,
      1,1,5,9,31,57,3,217,113,271,1663,1367,6949,8165,0,
      1,1,7,15,27,35,81,235,61,205,525,311,6357,2527,0,
      1,3,1,9,19,29,71,207,321,1011,1615,1333,3459,6681,0,
      1,3,7,7,3,57,41,19,25,397,565,1837,7625,11813,0,
      1,3,3,1,27,47,31,79,441,961,1255,423,2405,913,0,
      1,3,3,13,3,29,69,227,85,201,395,3199,3869,13099,0,
      1,3,3,7,29,61,99,7,27,227,945,873,475,4363,0,
      1,3,5,13,19,21,57,149,217,443,565,453,5487,10981,0,
      1,3,3,1,9,27,47,191,35,395,1429,4079,6871,8013,0,
      1,3,5,15,5,43,9,79,279,563,1125,985,8117,4099,0,
      1,3,5,1,13,41,21,117,287,667,701,1483,8167,13283,0,
      1,3,1,3,15,15,59,5,383,509,1657,3977,7697,10941,0,
      1,3,1,1,17,29,19,23,377,45,981,1631,3557,6749,0,
      1,3,3,9,9,51,9,193,345,361,1679,3333,713,5387,0,
      1,3,5,5,17,45,97,17,385,349,105,2245,7295,14393,0,
      1,3,7,3,19,51,35,99,79,301,1563,399,5879,14675,0,
      1,1,7,15,13,53,55,203,417,161,2033,1845,6763,3585,0,
      1,1,3,3,7,23,11,43,241,309,1453,3147,2619,3163,0,
      1,1,1,11,17,1,17,137,443,465,993,3217,7879,14607,0,
      1,1,7,13,29,49,71,217,291,793,135,21,2503,11091,0,
      1,3,1,11,31,51,121,227,377,157,1457,1317,5625,6217,0,
      1,1,3,7,23,61,47,93,79,617,1805,2403,5513,16335,0,
      1,3,5,11,23,25,41,11,495,587,1223,3107,1469,15223,0,
      1,3,7,7,9,1,1,49,23,723,1761,3717,7375,10875,0,
      1,3,3,11,25,37,57,63,309,603,183,285,1663,5627,0,
      1,3,7,11,19,25,25,201,391,257,529,1645,1,15111,0,
      1,3,3,9,11,43,91,65,5,959,301,1015,6343,3453,0,
      1,3,3,11,17,17,103,37,77,973,575,439,49,3639,0,
      1,1,5,7,1,15,107,237,231,967,923,1101,6715,1713,0,
      1,3,1,15,9,33,29,211,245,601,1783,887,1209,11785,0,
      1,3,3,7,21,43,27,89,27,141,865,367,1379,4063,0,
      1,3,7,7,15,17,15,15,131,649,1955,3289,3983,10689,0,
      1,3,1,5,17,7,125,69,359,981,1345,933,5281,7113,0,
      1,1,5,9,17,7,41,207,497,1015,493,891,3563,3541,0,
      1,3,5,11,27,3,47,31,303,1007,2047,2203,6257,8369,0,
      1,1,1,15,25,15,89,51,217,357,1133,1917,213,3365,0,
      1,1,5,13,29,23,123,207,429,805,819,2357,6313,11019,0,
      1,1,3,7,19,15,41,73,279,11,1089,3107,7737,15953,0,
      1,3,5,7,7,15,41,73,493,457,1731,1139,2513,12373,0,
      1,3,5,9,17,5,55,155,173,1005,529,3175,7667,4747,0,
      1,1,7,7,5,21,105,31,205,847,1033,3167,2347,8499,0,
      1,3,5,3,11,17,59,189,179,1007,33,3287,4813,8177,0,
      1,3,3,13,27,47,47,171,413,875,1081,1259,7139,8645,0,
      1,3,5,7,25,21,51,29,361,293,51,1119,1453,5283,0,
      1,3,7,7,29,55,103,199,511,341,1957,3987,2855,1279,0,
      1,1,1,9,23,51,61,63,391,37,55,3771,6517,15913,0,
      1,1,1,9,3,19,13,147,453,855,1321,189,5043,11215,0,
      1,3,3,13,23,3,87,155,401,981,607,3413,995,6473,0,
      1,3,1,9,29,47,95,123,421,353,1867,2609,2569,14083,0,
      1,1,5,13,25,39,29,111,125,545,1493,2371,6361,6307,0,
      1,3,3,11,13,31,87,75,27,393,921,3655,3343,16349,0,
      1,1,5,9,19,19,7,129,223,715,433,1627,4463,2951,0,
      1,1,7,1,31,13,49,33,89,43,1529,725,3809,3427,0,
      1,1,7,3,1,27,45,9,309,875,659,2661,553,7069,0,
      1,1,7,15,13,37,61,19,125,683,1227,2255,1455,9339,0,
      1,3,5,7,19,7,71,21,465,645,1885,873,7405,1913,0,
      1,3,1,11,11,35,79,61,79,57,1603,3719,6323,16371,0,
      1,1,7,1,29,57,85,21,205,37,2045,683,4901,8223,0,
      1,1,5,13,31,31,65,131,259,535,967,3943,2605,2089,0,
      1,1,7,9,27,61,39,243,207,41,1909,3279,1331,4635,0,
      1,3,3,5,11,63,105,19,169,95,773,3175,1869,1797,0,
      1,3,3,15,13,33,107,197,153,795,1477,105,4965,991,0,
      1,3,7,11,11,37,23,149,197,3,1035,3857,553,1059,0,
      1,3,1,3,17,29,89,189,193,59,1477,3517,2565,7739,0,
      1,1,1,9,23,3,25,163,469,305,1791,3393,6141,8119,0,
      1,3,5,7,7,41,19,101,179,487,1071,2761,8043,5103,0,
      1,1,7,9,1,21,101,103,349,85,1841,1033,4473,3563,0,
      1,1,3,13,23,61,39,27,479,13,45,1371,7897,10637,0,
      1,1,5,9,17,61,71,55,355,99,1695,3053,839,959,0,
      1,1,3,1,7,27,87,221,327,241,461,3177,5933,8299,0,
      1,3,7,9,5,41,111,245,447,263,1363,1767,6331,3355,0,
      1,3,3,13,15,11,15,169,429,149,1965,2477,7733,2499,0,
      1,1,5,15,15,47,25,33,469,701,773,2747,1533,14633,0,
      1,3,1,5,19,57,37,75,423,11,685,2487,1779,8797,0,
      1,3,1,5,19,41,67,99,333,991,953,3221,939,4197,0,
      1,3,1,15,11,39,25,1,159,679,465,1611,5799,2537,0,
      1,1,5,11,5,37,37,7,101,703,235,23,2209,12799,0,
      1,1,7,3,11,23,71,215,45,269,1539,3625,5773,6889,0,
      1,3,5,15,27,33,105,109,205,653,821,435,1087,2495,0,
      1,1,3,5,11,39,53,213,41,385,1425,25,5553,12523,0,
      1,3,5,15,29,49,13,253,505,407,985,2569,6727,4761,0,
      1,1,1,3,29,17,69,47,25,819,1145,2479,1183,3343,0,
      1,3,1,15,25,61,43,55,279,579,361,355,6101,3143,0,
      1,3,5,11,3,59,125,101,451,495,1711,3443,3625,15579,0,
      1,3,1,11,25,61,49,219,23,795,481,3609,3691,15419,0,
      1,3,7,5,9,59,49,233,345,143,181,3587,3041,1219,0,
      1,3,7,13,9,31,39,137,261,919,1367,3145,4659,5875,0,
      1,1,3,3,27,43,95,65,301,915,31,451,7743,7277,0,
      1,3,1,5,23,37,53,31,203,453,71,1585,6011,16369,0,
      1,1,5,1,15,47,91,227,297,45,1415,3647,7811,14015,0,
      1,1,1,1,29,27,93,121,169,69,1361,2907,1867,7017,0,
      1,3,1,7,23,53,77,41,25,873,1333,3889,3239,1771,0,
      1,1,1,7,31,27,87,81,167,343,1981,2499,7749,15747,0,
      1,3,5,13,1,17,97,37,81,645,1167,3547,7769,10731,0,
      1,1,7,5,9,17,31,55,151,463,1041,2303,4015,3737,0,
      1,1,3,11,31,9,81,213,95,215,2031,2129,4299,3021,0,
      1,1,1,3,25,25,115,229,101,441,783,1729,7905,2375,0,
      1,1,5,9,3,19,73,35,379,493,1333,1647,13,197,0,
      1,1,7,9,3,55,99,43,281,9,73,2477,8183,11055,0,
      1,3,7,13,25,19,27,195,469,175,355,1861,7255,15377,0,
      1,1,3,11,15,19,115,31,413,835,697,879,6515,13465,0,
      1,3,3,15,3,61,105,201,151,739,49,3963,2573,3303,0,
      1,3,5,7,23,5,11,215,19,591,509,2887,1631,4391,0,
      1,3,3,3,25,1,109,5,363,545,1745,503,827,4677,0,
      1,1,3,15,1,45,121,141,497,745,1825,2041,2561,8153,0,
      1,3,1,11,29,7,71,241,7,39,1379,2479,7483,7195,0,
      1,1,7,11,3,27,39,97,339,217,1409,1569,4761,1567,0,
      1,1,5,15,11,53,87,213,297,923,393,717,3297,16123,0,
      1,1,1,11,27,41,121,49,225,379,1305,319,2461,5445,0,
      1,1,5,5,25,3,121,23,47,843,1679,1427,6393,4199,0,
      1,1,5,13,17,3,17,25,161,487,121,361,1375,10745,0,
      1,1,7,3,3,37,7,245,107,107,745,2415,2131,11419,0,
      1,1,5,3,3,23,67,91,281,387,465,905,883,9775,0,
      1,3,7,15,25,55,123,49,23,983,1903,2589,2073,7823,0,
      1,1,5,11,25,17,63,229,267,175,1759,1947,479,11089,0,
      1,3,7,3,11,37,83,95,415,1003,1175,2361,2117,9809,0,
      1,3,1,9,5,39,51,129,249,161,1981,2755,8057,13641,0,
      1,1,7,1,15,47,9,197,199,549,1091,2853,2331,4535,0,
      1,3,3,13,15,21,23,111,463,719,1667,377,5039,10723,0,
      1,1,3,7,23,47,39,47,307,949,1651,2525,5835,1425,0,
      1,3,3,9,23,47,111,39,251,1001,179,3985,535,15435,0,
      1,1,3,13,5,45,51,123,205,651,1583,1691,1631,11975,0,
      1,1,7,9,1,29,59,27,389,497,1459,1633,521,14037,0,
      1,1,3,3,3,23,35,247,371,729,931,681,1777,8353,0,
      1,3,3,1,19,15,17,191,495,643,319,37,5691,7049,0,
      1,3,5,11,5,31,123,243,335,573,113,209,4825,7783,0,
      1,3,7,7,29,19,25,191,89,515,55,3013,4523,12913,0,
      1,1,3,3,15,3,35,37,339,7,697,359,4553,1431,0,
      1,3,1,1,9,15,33,77,161,13,255,1187,6587,11715,0,
      1,3,7,7,25,57,61,171,231,43,1219,903,5623,4781,0,
      1,1,5,15,29,47,117,23,213,907,1423,369,4529,9651,0,
      1,1,5,7,15,55,105,249,401,37,1885,3779,3441,9057,0,
      1,1,5,3,3,27,49,89,335,561,1235,3251,2731,12711,0,
      1,1,1,15,29,49,37,173,25,743,1321,821,5899,9213,0,
      1,1,7,3,1,41,61,209,275,925,521,3029,1569,9277,0,
      1,3,5,13,17,1,11,171,441,119,1589,299,157,11439,0,
      1,1,5,9,13,33,27,77,363,939,1103,2135,1759,5429,0,
      1,3,7,1,17,39,49,201,49,803,2003,1193,7415,13847,0,
      1,1,5,5,17,49,39,19,311,801,1441,3263,7973,14181,0,
      1,1,3,9,9,27,59,89,81,473,1369,3121,7929,10905,0,
      1,3,3,5,17,35,35,239,379,431,501,3561,2059,9679,0,
      1,3,5,15,25,29,113,179,269,891,301,2017,7513,9379,0,
      1,3,1,11,17,35,49,149,135,661,1691,3169,3765,9003,0,
      1,3,7,15,5,21,53,241,475,271,683,2351,2181,6333,0,
      1,1,7,13,25,33,71,153,221,507,2017,2401,7545,8489,0,
      1,1,7,5,1,49,87,1,179,331,1597,3713,809,11109,0,
      1,3,1,5,5,61,93,39,479,977,1099,1291,7049,2797,0,
      1,3,1,13,19,41,57,77,5,117,125,115,3969,1345,0,
      1,1,1,9,15,9,57,7,219,41,767,23,5771,14175,0,
      1,3,7,9,17,61,1,59,227,349,63,189,3871,7919,0,
      1,3,5,5,9,29,33,203,413,701,1129,2103,1889,8377,0,
      1,1,3,1,9,17,69,115,123,1001,1,2893,3957,8593,0,
      1,1,3,1,31,41,83,91,113,195,1121,2665,6815,1189,0,
      1,1,1,13,3,59,13,123,95,103,1689,2809,5049,4055,0,
      1,1,1,15,21,41,11,167,375,589,207,1631,1597,8091,0,
      1,3,5,5,1,33,57,89,157,921,1353,2777,461,14567,0,
      1,1,5,1,25,5,51,247,1,577,463,3741,303,16059,0,
      1,1,7,5,13,7,17,87,51,987,835,93,5203,3973,0,
      1,1,7,7,3,27,7,1,135,171,231,3349,4459,2925,0,
      1,1,5,5,9,51,71,153,115,315,265,2207,4127,12631,0,
      1,1,3,15,23,59,35,121,425,921,1255,2123,5811,15937,0,
      1,3,7,7,11,21,45,57,269,395,555,783,6677,2889,0,
      1,3,5,7,31,19,73,35,465,349,1429,863,4707,6121,0,
      1,3,3,9,25,27,119,159,195,949,19,73,4511,15711,0,
      1,3,3,7,9,59,47,57,91,749,1579,1297,2445,5167,0,
      1,3,3,3,31,57,19,203,61,927,1477,2863,1305,11673,0,
      1,3,7,11,29,13,3,111,351,79,1863,2213,3273,7049,0,
      1,3,3,9,7,23,47,237,121,877,441,119,2723,3989,0,
      1,3,3,11,17,23,63,177,231,363,1451,33,2169,7251,0,
      1,1,5,11,31,41,93,229,39,1009,1061,433,2393,15401,0,
      1,1,5,15,31,37,25,135,135,897,33,3713,7663,8079,0,
      1,1,5,7,17,49,43,89,411,731,1431,3893,1635,7063,0,
      1,1,1,13,29,27,5,77,283,913,789,817,3309,475,0,
      1,1,3,1,19,21,67,77,423,551,5,1057,5469,7859,0,
      1,1,5,1,1,21,99,237,215,759,1505,1983,1517,8923,0,
      1,3,5,7,19,61,73,215,165,127,205,259,7755,15395,0,
      1,1,5,9,15,23,17,111,471,751,1923,775,6901,13095,0,
      1,1,7,1,25,5,63,141,461,687,1589,1559,7719,11349,0,
      1,1,1,3,11,63,11,27,253,439,297,1315,829,3765,0,
      1,3,1,1,9,47,127,179,173,809,241,35,7355,5049,0,
      1,3,1,11,19,63,93,1,205,977,303,3409,6529,10927,0,
      1,3,7,9,31,63,41,79,477,91,1801,3487,6885,13341,0,
      1,1,3,7,15,59,9,101,459,247,549,2855,5765,7785,0,
      1,1,7,3,13,59,71,123,93,517,1453,2389,4429,5053,0,
      1,1,5,3,19,21,77,53,81,879,1653,1637,3667,2623,0,
      1,1,1,15,17,57,65,53,407,765,417,497,5009,2175,0,
      1,3,3,7,31,13,5,203,263,17,119,1607,6773,11195,0,
      1,3,3,13,19,13,13,147,93,735,689,781,655,6853,0,
      1,1,1,1,1,25,63,159,493,987,71,1249,5859,11717,0,
      1,1,1,15,13,23,61,61,5,947,1853,3331,467,8081,0,
      1,1,3,9,19,61,65,189,95,309,283,1725,5683,15463,0,
      1,1,7,5,9,33,35,75,475,831,1445,1485,5047,9631,0,
      1,1,3,15,11,23,59,87,433,221,685,3113,4095,13819,0,
      1,1,7,15,25,29,67,17,349,353,1321,563,57,533,0,
      1,3,3,3,5,43,109,217,15,185,1895,1015,1831,10623,0,
      1,1,7,1,1,47,81,185,59,691,191,3709,1535,13347,0,
      1,1,5,1,23,57,83,217,457,771,1877,2789,8143,4797,0,
      1,1,3,7,23,35,79,49,227,205,1523,3873,4843,10505,0,
      1,1,1,1,17,43,121,95,205,35,189,2061,1693,13273,0,
      1,1,1,15,31,49,83,249,433,497,1949,1845,5215,5971,0,
      1,3,1,1,21,53,73,211,265,929,923,279,3621,9469,0,
      1,3,7,7,1,57,13,45,467,705,371,1345,1647,3411,0,
      1,3,1,11,27,29,117,163,143,669,489,3913,7891,9031,0,
      1,3,7,15,27,15,77,217,107,839,1517,1543,357,10365,0,
      1,1,1,5,31,17,107,245,345,939,1453,3645,6865,16173,0,
      1,3,5,5,9,61,43,97,453,917,945,2143,5473,5611,0,
      1,1,5,11,3,33,71,97,137,549,1605,3839,4883,2677,0,
      1,3,1,11,29,23,85,47,225,633,1613,1297,1415,15813,0,
      1,1,3,3,9,19,57,107,79,449,1951,753,6317,10377,0,
      1,1,1,5,21,3,39,187,299,517,1313,741,7259,4197,0,
      1,1,5,13,1,39,39,41,381,123,1257,3185,493,3723,0,
      1,3,7,7,3,37,15,161,129,169,555,3605,4287,15831,0,
      1,3,7,15,15,23,81,163,257,791,505,1903,2703,11919,0,
      1,3,7,7,27,63,17,147,111,851,1533,1365,5359,3315,0,
      1,3,7,1,15,5,61,143,385,261,1019,1705,1737,14485,0,
      1,3,5,5,25,17,49,229,431,567,1613,3431,2139,2981,0,
      1,3,5,11,17,57,71,241,31,1007,1695,2965,149,14125,0,
      1,1,3,11,7,49,39,101,5,501,1491,3061,225,12255,0,
      1,3,5,7,17,35,37,97,415,15,1349,997,2949,4511,0,
      1,3,1,5,25,35,99,183,161,59,1363,515,3767,3641,0,
      1,1,7,15,7,15,127,137,281,67,139,2315,3517,13371,0,
      1,1,5,15,23,49,19,79,425,805,1035,429,7707,14195,0,
      1,3,5,3,21,25,123,11,425,475,961,2995,7405,5449,0,
      1,1,7,1,21,1,75,231,451,595,719,2369,5907,1227,0,
      1,1,1,9,21,57,45,255,19,79,481,3363,3451,8399,0,
      1,1,7,13,31,49,95,69,483,427,37,4047,7057,9111,0,
      1,3,3,11,3,61,87,79,499,91,771,1987,2017,3381,0,
      1,3,1,7,5,57,1,121,155,225,501,477,6555,9863,0,
      1,3,7,11,27,49,83,213,61,283,1599,3205,2525,8553,0,
      1,1,1,9,9,49,3,51,141,33,301,2167,587,15067,0,
      1,1,1,11,7,55,99,81,191,553,953,3753,6731,1093,0,
      1,1,3,3,11,59,57,235,297,197,853,1411,3799,7527,0,
      1,3,5,3,7,7,5,201,393,95,91,3273,6285,10661,0,
      1,1,5,7,17,57,87,3,413,915,659,369,3593,14429,0,
      1,3,7,1,31,31,45,115,417,427,745,4087,953,1119,0,
      1,3,7,3,29,43,45,221,41,641,451,173,2999,12103,0,
      1,1,3,11,25,57,117,201,135,787,1525,3879,3247,8907,0,
      1,1,7,11,3,35,69,157,331,615,573,2169,3575,289,0,
      1,3,3,13,15,51,67,127,265,495,103,3145,2685,15919,0,
      1,3,5,11,31,27,65,57,153,465,1163,467,4103,4713,0,
      1,3,7,3,23,31,9,51,239,417,1597,229,2865,15199,0,
      1,3,5,3,11,45,123,217,31,765,1009,2001,3645,9407,0,
      1,3,3,9,5,23,117,83,237,1017,251,1187,2631,5151,0,
      1,1,1,7,23,55,97,141,501,305,467,4061,2369,15973,0,
      1,1,7,5,31,51,125,191,219,495,37,3337,813,241,0,
      1,3,1,1,11,39,93,109,285,147,1297,737,4051,7223,0,
      1,3,1,15,13,17,57,171,463,163,609,1681,7583,9231,0,
      1,3,1,1,23,5,51,5,205,415,419,989,4239,10943,0,
      1,1,3,15,3,13,65,145,387,59,395,1067,4143,5649,0,
      1,3,1,13,9,59,121,127,95,71,1541,1423,1753,8041,0,
      1,1,3,15,7,5,69,167,181,991,1189,4017,5935,6669,0,
      1,3,5,7,23,41,53,21,47,261,1231,2011,133,2247,0,
      1,1,1,5,17,47,77,19,331,609,1893,3965,3123,9093,0,
      1,3,1,3,9,39,103,231,249,75,373,107,1823,10801,0,
      1,3,3,7,1,51,35,111,137,879,1221,225,4285,2287,0,
      1,1,7,9,23,17,75,245,409,163,395,3731,7111,6845,0,
      1,1,3,13,29,47,75,153,497,621,1691,3187,2125,10533,0,
      1,1,7,7,9,7,55,159,255,417,1335,643,3843,3733,0,
      1,3,3,1,21,41,7,21,5,679,1655,95,5699,5785,0,
      1,1,1,13,19,7,85,7,195,357,1097,2893,2913,9635,0,
      1,1,5,9,25,33,41,155,39,655,1993,3117,3639,7977,0,
      1,1,1,13,3,63,121,247,151,673,609,285,2299,7663,0,
      1,3,7,11,17,13,49,253,245,21,273,993,911,863,0,
      1,1,5,5,23,1,121,95,225,9,1237,1183,6461,559,0,
      1,3,7,13,3,7,121,151,233,561,281,3583,897,1767,0,
      1,1,7,7,9,47,107,41,25,569,1697,2299,6231,12209,0,
      1,3,7,7,27,43,59,37,31,51,503,149,4043,11847,0,
      1,3,3,11,5,1,119,181,47,641,685,4017,637,16251,0,
      1,3,3,7,11,1,101,7,239,747,307,1721,5979,4367,0,
      1,3,5,7,1,63,19,151,469,333,1587,2453,897,4711,0,
      1,3,1,5,11,61,21,253,91,993,1347,1993,5607,13031,0,
      1,3,5,5,1,39,65,71,189,389,1437,1055,6439,3989,0,
      1,1,3,3,19,15,93,3,339,165,1675,3953,2145,12113,0,
      1,1,3,13,13,45,5,175,211,993,705,2761,3023,13633,0,
      1,1,3,1,19,39,121,29,287,87,281,3491,7107,13007,0,
      1,1,7,1,29,49,103,187,39,923,51,1533,3249,4399,0,
      1,1,5,5,5,43,25,107,453,955,115,57,4589,14573,0,
      1,1,3,5,21,45,103,99,183,987,1207,1697,8033,13703,0,
      1,1,5,7,11,23,9,17,261,749,1957,935,6283,8625,0,
      1,1,1,9,9,51,69,225,265,323,1161,2993,7305,2249,0,
      1,3,1,9,23,19,57,205,503,489,1499,3277,817,11931,0,
      1,3,3,5,1,7,49,1,313,123,643,2027,1469,3585,0,
      1,3,7,11,27,47,95,111,27,213,465,3693,3661,7531,0,
      1,1,7,9,3,37,115,189,31,613,1393,1229,4767,12425,0,
      1,1,3,3,25,17,99,47,161,931,959,1293,7095,8325,0,
      1,1,1,7,23,9,11,51,205,419,479,1497,2493,13921,0,
      1,3,1,9,17,29,51,79,159,435,477,413,3815,5589,0,
      1,3,7,5,7,23,99,43,169,665,403,1163,4337,1335,0,
      1,3,1,5,25,27,125,249,421,267,1259,4089,59,9377,0,
      1,3,3,1,27,37,91,17,123,597,1749,3449,6503,11043,0,
      1,3,7,7,23,41,19,245,109,569,547,1917,7943,2697,0,
      1,3,7,7,9,1,123,105,329,435,2013,2745,347,11045,0,
      1,1,1,13,29,53,51,67,105,89,1887,3543,963,8159,0,
      1,1,5,3,5,27,41,67,67,883,973,1785,901,14969,0,
      1,3,3,13,17,11,117,115,163,939,79,641,4365,2267,0,
      1,1,3,3,9,5,41,123,149,9,1533,3939,5995,12701,0,
      1,1,1,15,31,1,101,229,191,965,61,2671,4177,15779,0,
      1,1,5,15,1,25,49,185,33,697,1043,2639,7819,3171,0,
      1,3,5,13,19,9,111,49,47,847,1865,717,5287,13417,0,
      1,3,7,11,5,61,63,111,171,735,2003,73,5701,647,0,
      1,3,1,11,1,49,121,79,431,671,1241,1161,2057,263,0,
      1,3,3,1,1,23,75,15,117,641,313,1525,2041,1409,0,
      1,3,5,11,15,57,13,67,139,131,1339,2419,7945,11877,0,
      1,3,1,1,19,39,97,83,297,595,1611,5,4753,3435,0,
      1,3,1,9,7,49,125,101,383,717,63,2295,3873,13461,0,
      1,1,3,3,15,29,89,77,269,689,229,1207,7311,8663,0,
      1,1,1,1,1,61,25,255,203,233,271,987,2277,8735,0,
      1,1,5,7,21,27,63,79,337,133,1453,3633,6157,15875,0,
      1,3,1,7,7,55,31,81,203,709,1743,1677,4247,11411,0,
      1,1,3,3,29,51,37,17,487,325,1393,1433,3467,2851,0,
      1,1,7,9,3,41,99,177,241,869,739,2729,4585,14801,0,
      1,1,3,1,9,43,97,65,99,295,1693,2083,3241,4073,0,
      1,1,1,9,5,39,67,119,235,543,795,2773,3195,6273,0,
      1,1,5,5,21,41,89,1,85,81,57,2951,1531,10101,0,
      1,1,1,7,3,35,127,69,39,265,1643,2973,267,12365,0,
      1,3,1,1,21,57,99,205,119,477,1771,1989,2761,12573,0,
      1,1,3,13,1,59,93,125,279,935,1877,2061,4845,7835,0,
      1,1,7,9,7,45,69,99,273,35,1579,2137,7175,6999,0,
      1,1,7,7,29,21,127,91,9,383,787,1783,601,5047,0,
      1,1,7,13,7,29,35,219,43,581,2043,2211,6169,12173,0,
      1,3,5,13,29,29,39,63,411,645,415,2383,1989,11411,0,
      1,1,7,9,15,9,87,95,321,709,543,3831,2453,4167,0,
      1,3,1,5,31,25,5,85,239,487,1613,3937,4661,3535,0,
      1,3,5,11,27,41,3,201,39,91,1997,237,5639,14703,0,
      1,1,7,3,27,49,87,71,473,247,1007,47,475,5413,0,
      1,3,7,15,9,57,81,149,287,333,1911,3417,1081,8995,0,
      1,1,5,1,3,63,43,151,97,431,961,1019,5153,2407,0,
      1,1,5,5,27,21,127,161,507,311,129,3489,1133,3241,0,
      1,3,7,15,21,33,117,83,497,667,1399,931,1695,8171,0,
      1,1,1,13,3,39,53,27,193,993,671,1871,7579,11457,0,
      1,1,5,11,7,39,81,107,195,387,849,395,1317,6487,0,
      1,3,3,3,3,15,45,127,279,111,331,357,4637,4697,0,
      1,1,3,9,21,49,47,97,61,101,181,1867,1201,14099,0,
      1,1,5,11,25,19,51,51,101,451,545,101,7497,9141,0,
      1,1,1,3,13,53,119,81,377,245,765,251,3757,16045,0,
      1,1,1,3,5,61,65,37,331,925,1439,3219,2843,11397,0,
      1,3,5,9,23,31,95,155,83,641,1129,135,477,1623,0,
      1,1,3,9,9,61,93,11,331,585,799,1417,1533,463,0,
      1,1,7,7,21,51,61,29,467,935,11,3357,1087,12337,0,
      1,3,3,11,1,39,103,153,351,893,1823,835,2149,4203,0,
      1,1,1,9,31,13,61,235,369,359,835,2067,2697,15289,0,
      1,1,7,1,15,1,107,27,201,451,1521,313,3195,3847,0,
      1,1,5,13,1,27,63,137,355,489,2039,1015,2519,13797,0,
      1,1,7,9,29,33,23,197,49,555,1087,3447,7299,15513,0,
      1,3,5,11,7,37,55,63,443,573,1715,631,3405,6155,0,
      1,3,3,3,31,35,51,167,225,617,2007,2555,6819,12709,0,
      1,1,1,13,15,5,73,85,109,43,1067,3941,1125,10269,0,
      1,1,7,11,17,3,127,145,279,19,1007,3287,4751,12507,0,
      1,3,7,3,19,1,117,111,193,435,47,1801,529,8547,0,
      1,3,3,13,1,19,101,19,469,187,207,1243,8153,3273,0,
      1,3,1,5,11,51,69,189,453,775,241,3331,4067,14759,0,
      1,1,1,1,23,55,113,133,497,731,391,2777,3529,955,0,
      1,3,1,11,5,49,59,35,261,949,325,3595,7433,11099,0,
      1,3,5,9,13,37,103,219,329,865,1787,2497,7249,9877,0,
      1,3,7,9,11,33,19,255,191,935,1115,1901,1577,9623,0,
      1,1,5,7,29,23,77,43,283,143,1211,73,2835,10235,0,
      1,1,7,3,3,27,35,173,453,425,1225,3023,2159,8433,0,
      1,1,1,5,27,21,35,25,71,145,1545,523,4527,7655,0,
      1,1,5,3,13,49,61,157,113,775,763,1785,225,11851,0,
      1,1,3,1,31,57,97,229,291,777,213,4067,921,8203,0,
      1,1,5,1,25,13,125,123,263,207,119,3111,3841,843,0,
      1,1,7,7,25,57,81,129,31,133,1869,2949,5563,14965,0,
      1,3,3,7,3,51,33,127,281,425,1253,405,7941,8799,0,
      1,1,3,9,3,63,93,173,255,609,49,111,7785,15865,0,
      1,1,1,3,17,59,113,55,155,789,1335,177,3071,1851,0,
      1,3,7,15,15,23,35,35,131,623,47,437,1337,9891,0,
      1,3,7,5,29,57,39,31,111,271,59,1473,949,3899,0,
      1,1,3,11,17,19,41,229,259,691,1455,3023,7455,9711,0,
      1,3,5,11,29,13,9,165,499,355,1415,1395,7595,15571,0,
      1,3,1,9,5,5,25,247,185,241,1325,3133,7471,2649,0,
      1,3,3,11,17,29,57,61,51,203,993,1837,3785,15163,0,
      1,1,7,7,21,57,79,165,277,133,93,1055,7169,15685,0,
      1,1,5,3,5,17,25,177,95,323,367,1359,4915,6409,0,
      1,1,1,1,11,25,115,45,373,221,1483,591,6561,4527,0,
      1,3,5,3,5,23,69,77,313,473,1037,4045,3969,5445,0,
      1,3,1,5,1,15,73,83,439,463,203,361,6835,1061,0,
      1,1,3,11,21,5,89,233,405,253,773,3901,6085,5677,0,
      1,1,3,9,15,53,71,29,101,599,1073,705,4507,12779,0,
      1,1,3,1,3,9,27,97,207,859,417,735,2179,5071,0,
      1,1,1,3,13,63,65,125,195,611,649,2221,3143,143,0,
      1,3,3,15,17,57,99,119,243,407,1229,813,5245,1893,0,
      1,1,1,5,27,27,49,13,313,287,473,2629,3509,11371,0,
      1,1,7,7,23,3,75,59,245,689,1215,2375,3325,1593,0,
      1,3,1,5,21,51,43,107,91,611,1405,677,2087,9565,0,
      1,3,7,11,9,27,81,101,449,201,1507,2217,6767,8059,0,
      1,1,3,9,13,41,21,195,421,315,347,2621,2359,9247,0,
      1,1,5,7,31,45,77,229,455,575,1087,1147,2273,13773,0,
      1,1,1,1,9,5,87,19,207,545,1435,495,1299,4947,0,
      1,1,3,3,15,9,63,67,219,735,1911,2361,6503,11977,0,
      1,3,1,9,31,27,103,153,81,939,461,2753,697,537,0,
      1,3,3,9,21,53,49,211,415,817,321,3775,2921,9473,0,
      1,1,7,3,23,55,15,51,435,1013,73,3967,4575,13099,0,
      1,1,3,7,5,27,43,225,267,21,1261,603,6913,4421,0,
      1,1,7,13,25,31,101,109,237,91,1587,1987,2795,6991,0,
      1,1,3,13,23,51,91,89,287,39,1513,463,6135,10469,0,
      1,3,3,1,9,43,125,157,369,495,1849,785,6357,6557,0,
      1,3,1,13,5,25,107,139,367,239,1671,1239,7027,5291,0,
      1,3,5,13,11,13,35,177,45,939,251,59,333,13105,0,
      1,3,5,7,29,57,109,227,435,739,423,1941,3345,12731,0,
      1,3,3,9,23,51,19,207,69,99,955,519,7305,2415,0,
      1,1,5,13,17,1,67,201,61,403,1059,2915,2419,12773,0,
      1,3,1,11,17,19,25,27,207,299,143,1955,5669,2301,0,
      1,1,5,3,25,57,45,255,489,1011,1699,2637,5279,12211,0,
      1,3,3,15,7,47,113,33,511,907,1815,1741,2091,13857,0,
      1,3,3,5,5,27,95,3,353,253,947,393,1815,14551,0,
      1,1,5,11,29,19,63,117,293,861,2039,9,5999,6909,0,
      1,3,7,3,15,63,107,173,509,817,99,2825,131,7917,0,
      1,3,1,1,29,49,33,153,119,777,1315,3581,5675,4043,0,
      1,3,5,15,13,11,17,147,327,305,367,3237,5423,13757,0,
      1,1,5,13,1,39,35,29,25,751,1365,2243,8181,7063,0,
      1,3,7,11,25,53,11,111,289,755,1201,691,3679,3725,0,
      1,1,1,11,11,37,33,211,395,691,1817,861,6485,12077,0,
      1,3,3,11,21,3,111,171,305,561,1501,2011,7841,10931,0,
      1,3,7,9,9,59,109,113,31,915,103,1861,2779,10619,0,
      1,1,1,1,7,25,61,97,103,723,1145,3105,371,339,0,
      1,1,7,13,17,9,113,51,233,209,1117,211,6969,2347,0,
      1,1,5,9,25,43,21,217,327,735,197,1063,799,801,0,
      1,1,7,13,9,13,73,33,415,923,863,1999,5383,8119,0,
      1,3,1,5,7,33,51,185,289,967,1277,1011,767,15505,0,
      1,3,3,13,21,11,105,235,343,1021,2009,2251,3865,6923,0,
      1,3,5,9,29,11,33,17,149,155,1739,3039,7015,2401,0,
      1,3,7,7,17,13,89,177,297,267,545,3861,329,13267,0,
      1,3,5,15,27,33,1,231,181,557,447,379,7845,1295,0,
      1,1,5,13,3,63,59,33,263,877,1867,1383,641,7139,0,
      1,3,7,5,13,51,9,113,223,605,1189,4063,6925,9563,0,
      1,1,1,13,5,35,83,107,295,231,265,5,4087,6407,0,
      1,1,5,1,7,25,95,137,97,987,1753,2781,1369,6903,0,
      1,1,5,13,19,61,77,229,193,165,811,249,79,10719,0,
      1,3,7,7,27,9,119,193,459,43,1989,2959,3595,6341,0,
      1,1,5,11,5,43,35,33,25,581,897,351,4201,3971,0,
      1,1,7,11,21,29,53,45,359,197,313,3825,6717,4077,0,
      1,1,1,15,3,45,99,133,357,315,1159,241,2463,11253,0,
      1,1,7,11,9,33,111,85,443,601,447,337,6471,7029,0,
      1,3,7,9,13,33,25,31,9,729,1763,4077,7575,7877,0,
      1,3,5,13,13,37,29,103,53,229,591,1073,1323,14405,0,
      1,1,5,1,17,33,15,183,473,297,2003,93,4955,1787,0,
      1,1,5,13,5,29,113,161,267,451,1193,149,273,11809,0,
      1,1,1,9,17,39,47,233,165,373,955,2891,7523,7235,0,
      1,1,1,3,7,21,115,205,153,449,339,2073,1077,5749,0,
      1,1,7,13,9,39,117,187,37,753,227,3519,7391,5751,0,
      1,1,1,9,5,19,41,161,141,195,1719,3321,5,12877,0,
      1,3,7,11,21,13,83,55,299,75,1905,3765,4685,12297,0,
      1,1,7,3,3,23,111,243,187,297,1061,2515,977,9555,0,
      1,3,7,3,29,11,103,177,225,875,1649,1401,6383,8309,0,
      1,3,5,3,3,41,71,3,373,757,701,2825,1521,13217,0,
      1,1,5,3,11,5,103,227,209,723,1543,3895,6345,7901,0,
      1,1,5,1,9,51,77,67,359,937,557,993,3871,3577,0,
      1,3,7,1,1,15,121,239,29,113,1123,3877,6941,14129,0,
      1,1,5,1,27,61,83,113,185,601,947,3933,381,13869,0,
      1,1,5,3,5,37,97,31,81,367,747,1811,5313,14151,0,
      1,3,5,9,27,61,87,31,185,521,837,959,5001,3957,0,
      1,3,5,3,11,61,37,19,107,749,1345,3829,6701,4315,0,
      1,3,1,15,13,45,101,113,243,963,1861,3283,1419,12131,0,
      1,1,7,1,11,63,17,117,271,819,677,669,1991,12511,0,
      1,1,1,13,13,33,41,73,187,537,993,3147,1013,16063,0,
      1,3,1,1,25,21,107,81,117,917,113,349,4475,9149,0,
      1,1,1,11,21,21,29,251,125,681,141,2893,5843,14359,0,
      1,3,3,1,5,41,85,163,387,29,1593,221,2769,10809,0,
      1,3,5,11,1,17,69,127,273,449,1855,2971,7031,10583,0,
      1,1,5,7,1,61,9,211,123,563,111,1883,5801,2191,0,
      1,1,3,11,11,51,1,81,405,803,2017,161,5429,731,0,
      1,1,7,9,15,55,65,51,459,485,1539,3135,2929,7867,0,
      1,1,7,11,3,45,15,7,331,417,1813,4009,1341,10965,0,
      1,1,1,5,9,29,89,121,277,509,1989,1293,4787,16097,0,
      1,1,3,9,17,45,97,197,339,943,1377,2947,5833,7,0,
      1,1,7,9,15,61,75,233,401,705,825,2521,3787,14387,0,
      1,1,7,15,25,57,3,43,361,459,1551,1859,6787,2293,0,
      1,3,3,11,11,35,91,65,43,509,1829,1149,4801,4109,0,
      1,3,5,9,15,3,81,109,231,481,417,2505,315,6693,0,
      1,1,3,9,3,7,107,221,297,543,149,579,927,79,0,
      1,3,1,11,17,3,81,137,157,587,741,1277,2631,3953,0,
      1,1,7,5,13,43,117,19,495,185,1105,605,5249,11099,0,
      1,1,7,9,23,55,91,213,21,779,857,2047,7813,10053,0,
      1,1,1,1,27,7,39,181,63,519,1073,3147,4111,363,0,
      1,3,7,9,15,61,7,139,495,805,1545,3789,2411,3989,0,
      1,1,3,1,25,11,23,241,167,607,479,153,7787,13929,0,
      1,3,5,15,29,35,45,71,457,297,883,3021,5361,15427,0,
      1,3,1,7,29,27,93,241,427,89,1185,37,3863,14095,0,
      1,3,1,5,5,45,51,15,235,889,1649,2331,2713,10943,0,
      1,1,3,11,11,15,71,85,135,163,139,1147,1043,3195,0,
      1,3,5,13,3,43,71,131,473,933,569,2491,7751,1865,0,
      1,1,7,9,21,37,105,227,329,509,1319,307,1557,14625,0,
      1,1,3,13,15,1,25,93,335,953,769,4039,369,10727,0,
      1,3,7,5,17,21,59,89,437,679,437,1543,7663,5005,0,
      1,1,7,15,27,49,125,13,397,877,1087,2191,4711,9065,0,
      1,1,7,5,15,47,115,125,187,31,1003,2575,5397,3883,0,
      1,1,7,11,15,1,127,207,383,707,183,1053,3123,14071,0,
      1,3,3,1,31,53,15,19,477,245,777,1613,5813,7443,0,
      1,3,1,11,23,59,65,23,493,157,1389,2833,4535,3907,0,
      1,1,7,1,19,7,51,135,327,441,1841,3091,3451,14381,0,
      1,1,7,7,3,37,29,249,437,319,1693,945,7639,5923,0,
      1,3,7,15,7,61,81,127,383,99,23,3833,3973,7651,0,
      1,3,1,7,7,21,119,185,243,619,1363,2033,4835,5089,0,
      1,3,1,1,3,27,63,145,271,735,695,3981,3049,5433,0,
      1,3,3,1,3,29,79,211,279,819,501,3665,1455,10455,0,
      1,1,3,3,31,61,113,5,411,91,489,3257,5939,6715,0,
      1,1,5,1,23,11,103,89,377,441,43,967,3383,8717,0,
      1,1,5,13,29,39,97,189,197,621,1755,333,6783,9711,0,
      1,1,5,13,27,17,97,197,351,799,335,765,5329,12549,0,
      1,1,5,11,29,17,9,211,127,633,1187,3965,4145,12667,0,
      1,1,7,5,27,29,65,115,287,325,461,5,899,2027,0,
      1,1,1,5,27,17,31,13,231,627,1163,649,1693,9975,0,
      1,3,1,15,7,49,113,123,427,603,347,2785,7129,4645,0,
      1,1,3,7,1,33,113,105,411,939,205,3965,4361,4649,0,
      1,1,1,1,5,21,35,159,275,929,1193,3205,4787,3515,0,
      1,1,1,5,1,21,29,191,275,233,1239,515,4349,14989,0,
      1,1,5,11,27,43,111,83,153,577,1537,149,231,839,0,
      1,3,5,13,21,19,57,69,87,163,271,3535,1057,8517,0,
      1,3,3,13,17,49,65,45,457,241,391,2033,2507,7771,0,
      1,1,5,7,11,19,79,133,341,761,27,3905,4137,14363,0,
      1,3,3,13,19,1,11,139,249,245,1393,2151,2857,1665,0,
      1,1,3,15,11,7,127,47,385,1007,713,2235,5489,8755,0,
      1,3,5,13,19,21,21,167,405,655,1653,889,7367,4177,0,
      1,1,5,3,19,63,99,39,89,415,951,2863,6569,3797,0,
      1,1,1,13,31,29,119,35,311,839,1749,941,7487,2385,0,
      1,3,7,3,17,3,97,143,465,345,1457,2201,5329,359,0,
      1,3,7,11,1,15,3,115,335,567,1749,1811,3491,15939,0,
      1,1,3,13,3,21,7,141,149,571,1877,473,2143,9569,0,
      1,3,3,11,23,61,47,179,297,453,181,3405,2981,13409,0,
      1,3,1,13,1,43,5,201,371,1003,367,2709,7675,14973,0,
      1,3,3,15,29,17,19,241,495,317,1135,2227,6457,4783,0,
      1,3,3,7,29,9,57,95,261,531,1717,3389,7991,3793,0,
      1,1,1,5,31,43,73,119,499,589,1529,3337,4097,15641,0,
      1,1,7,9,29,43,127,91,243,979,1325,2835,2787,9445,0,
      1,1,7,5,9,3,115,199,219,901,747,1077,3197,2443,0,
      1,3,5,1,3,43,7,117,297,313,1043,1579,5099,13289,0,
      1,1,7,11,29,33,15,121,131,579,317,1871,1121,11653,0,
      1,1,5,9,25,25,43,89,355,1011,1385,2901,6387,1653,0,
      1,1,1,9,5,47,61,165,85,757,1397,1177,1657,4899,0,
      1,1,3,9,11,49,15,139,261,613,931,1299,2777,2835,0,
      1,1,1,5,3,55,83,227,125,581,1607,1171,6681,14463,0,
      1,3,5,13,5,55,3,247,493,155,1073,3743,5719,4019,0,
      1,1,7,1,11,23,13,75,399,847,499,1643,6977,3699,0,
      1,3,1,9,11,41,47,131,313,627,481,2469,3281,979,0,
      1,3,5,13,29,3,65,101,11,29,1807,153,1487,16109,0,
      1,1,5,9,13,31,83,195,351,355,467,3871,3085,4441,0,
      1,3,5,3,19,21,111,179,143,361,1619,1547,3409,6905,0,
      1,1,5,9,31,1,93,199,491,135,1627,2559,1389,14561,0,
      1,3,3,9,25,53,3,105,39,445,259,1045,1129,9153,0,
      1,1,5,9,19,63,71,9,73,435,1377,4015,1821,6453,0,
      1,3,7,13,19,13,37,247,391,23,1491,1257,6395,237,0,
      1,1,3,3,19,55,109,23,227,747,729,2221,727,2209,0,
      1,1,5,11,25,21,75,37,219,355,1005,1895,7039,5225,0,
      1,3,5,13,11,43,9,67,87,797,1077,245,4521,11845,0,
      1,3,5,3,15,29,127,237,277,373,1859,3083,587,1123,0,
      1,1,7,15,13,7,103,53,13,965,1497,775,3439,1501,0,
      1,3,3,15,17,13,97,169,67,953,189,2739,1459,10543,0,
      1,1,5,1,17,39,15,127,327,989,1471,3235,2801,15311,0,
      1,1,1,15,5,37,55,155,47,463,1851,3467,2765,9359,0,
      1,3,3,15,1,13,93,239,291,115,365,61,395,15853,0,
      1,1,5,1,19,27,61,95,105,369,1557,961,6917,3621,0,
      1,3,3,9,7,35,115,53,111,345,1145,1687,3401,12107,0,
      1,1,1,5,7,31,63,19,373,79,1369,3037,2835,4439,0,
      1,3,7,9,11,17,29,33,331,447,1981,3333,6535,6557,0,
      1,3,3,5,11,41,29,43,365,279,1919,945,179,1987,0,
      1,3,1,13,7,7,25,33,103,367,1267,763,5691,8643,0,
      1,3,1,5,11,15,3,213,511,211,1069,4047,3335,12729,0,
      1,1,3,1,5,11,27,201,361,537,679,3709,293,2997,0,
      1,1,3,1,25,15,19,185,369,577,1625,655,2363,3861,0,
      1,1,5,5,1,47,61,45,411,597,955,1007,3775,5809,0,
      1,1,5,3,27,51,101,167,429,333,1703,3541,2947,3695,0,
      1,3,5,5,1,53,17,63,141,215,1223,3129,635,15919,0,
      1,3,3,1,23,31,25,11,195,241,995,3941,573,13855,0,
      1,3,3,7,17,13,71,203,465,479,1857,1493,8067,7113,0,
      1,1,5,3,11,57,9,59,225,691,425,2423,6031,6631,0,
      1,3,7,1,29,57,103,123,401,807,471,2759,5113,15937,0,
      1,3,1,1,3,1,67,123,157,655,519,323,1853,15041,0,
      1,1,7,5,11,11,105,135,247,689,1141,2347,7113,9347,0,
      1,1,3,11,15,37,87,3,209,575,1521,3863,3893,211,0,
      1,3,1,3,29,55,115,31,19,195,985,3275,363,9801,0,
      1,1,3,9,13,31,57,251,201,275,1751,389,1463,13159,0,
      1,3,5,15,19,51,127,255,397,243,29,3007,7845,4687,0,
      1,1,7,15,9,37,39,217,509,137,1123,3361,6323,5323,0,
      1,3,7,5,25,3,93,203,345,581,261,2811,4829,6977,0,
      1,1,7,1,15,41,51,227,447,893,1209,3865,5229,4277,0,
      1,1,1,5,31,19,23,195,359,853,595,337,2503,16371,0,
      1,3,7,5,5,13,89,157,351,777,151,3565,4219,7423,0,
      1,1,1,5,7,1,9,89,175,909,1523,2295,7949,6739,0,
      1,3,5,15,27,17,11,235,19,105,457,465,3819,11335,0,
      1,3,1,13,3,41,85,221,451,613,543,2265,6831,1725,0,
      1,1,7,7,3,29,9,197,455,665,343,1811,5395,393,0,
      1,1,3,13,29,55,71,95,475,615,2029,123,413,16127,0,
      1,1,5,9,15,61,9,51,105,271,511,2801,693,11839,0,
      1,1,7,13,29,9,105,59,377,635,717,4033,6963,10541,0,
      1,1,1,13,7,13,59,17,335,355,77,3665,7003,9521,0,
      1,3,1,1,23,43,51,209,151,365,1021,2859,3937,2899,0,
      1,1,3,3,31,41,111,107,171,433,1233,505,2971,6927,0,
      1,3,7,13,17,25,127,195,257,551,1867,2145,3695,14567,0,
      1,1,5,13,13,45,39,195,55,991,1981,1043,5875,581,0,
      1,3,3,11,25,31,91,153,415,449,1301,563,7755,10671,0,
      1,1,3,5,31,63,1,157,229,949,971,137,6589,8387,0,
      1,3,7,15,25,7,89,133,73,497,1361,613,455,1005,0,
      1,3,3,1,13,5,119,93,175,511,1923,763,7573,7545,0,
      1,1,3,15,27,59,49,205,497,485,117,2523,4495,15153,0,
      1,3,7,9,15,47,111,31,363,11,475,2931,6813,1259,0,
      1,1,5,5,1,35,95,225,17,991,809,2601,6455,13803,0,
      1,1,5,5,15,1,1,171,433,887,1813,3431,2471,7803,0,
      1,3,3,15,1,15,43,179,15,949,1881,1027,6989,8955,0,
      1,3,7,13,1,3,49,183,373,175,1733,913,929,1065,0,
      1,3,5,7,15,51,107,115,323,357,167,2069,7541,9601,0,
      1,1,3,5,5,21,31,107,21,299,1937,43,3673,8155,0,
      1,3,5,11,9,55,35,113,29,99,161,1607,8141,4951,0,
      1,3,7,15,25,7,113,179,213,19,1717,1027,2021,11263,0,
      1,1,5,1,31,33,85,111,67,95,2013,2217,871,5329,0,
      1,1,1,7,7,63,67,145,495,419,1945,3437,6255,151,0,
      1,3,5,7,17,37,97,187,215,399,1603,2195,5923,769,0,
      1,1,3,9,25,1,119,193,385,861,2005,2769,675,767,0,
      1,3,1,15,19,7,5,227,173,383,289,461,579,3689,0,
      1,3,1,11,1,37,93,239,465,891,1479,921,4439,15265,0,
      1,1,1,13,27,61,99,69,279,655,1853,1593,6319,9003,0,
      1,1,1,11,5,7,19,7,387,303,321,931,5809,16029,0,
      1,1,1,15,21,55,43,107,217,687,19,3225,3419,9991,0,
      1,1,7,5,7,55,79,41,317,357,859,1205,191,9395,0,
      1,1,3,11,3,43,7,133,115,995,1205,1055,4153,10481,0,
      1,1,7,11,31,57,53,9,459,223,1969,3513,7033,8505,0,
      1,1,3,7,17,11,115,255,281,97,1685,2039,2845,11637,0,
      1,3,7,1,23,41,69,199,53,105,657,1453,4429,1101,0,
      1,3,1,5,11,33,91,131,191,73,823,117,1053,127,0,
      1,3,7,11,7,3,21,65,187,103,1393,1797,6673,1409,0,
      1,3,7,1,31,25,25,161,299,275,417,2267,6861,1255,0,
      1,3,5,13,5,11,61,155,115,1001,747,889,3235,5709,0,
      1,3,7,7,7,1,97,177,507,273,1781,3455,5123,15607,0,
      1,1,7,5,1,7,59,49,147,343,97,3517,5611,8705,0,
      1,1,5,13,21,29,13,21,503,515,1217,3905,5513,15849,0,
      1,3,1,9,9,39,65,111,385,757,583,2225,2039,2817,0,
      1,3,3,15,23,17,63,169,503,949,849,461,6799,669,0,
      1,1,1,3,1,41,63,159,251,457,521,1653,623,3287,0,
      1,1,7,3,9,1,41,37,441,921,1415,2955,5841,1451,0,
      1,1,5,11,23,29,89,185,413,357,1131,2369,3835,6233,0,
      1,1,5,15,27,35,17,73,315,911,1761,797,5349,3219,0,
      1,3,7,11,21,9,119,233,249,901,189,3625,2691,16201,0,
      1,3,3,13,29,61,105,145,187,79,609,321,4289,3933,0,
      1,3,1,15,19,63,13,185,115,219,1021,1205,4273,11521,0,
      1,1,3,3,23,31,93,153,87,947,1039,469,4047,8869,0,
      1,1,1,1,9,1,85,3,15,995,455,2769,6781,16203,0,
      1,1,3,3,13,7,55,215,185,367,765,441,4497,1521,0,
      1,1,1,5,1,31,13,95,417,735,975,3407,4871,16133,0,
      1,1,3,3,5,43,111,107,419,515,1075,3597,1187,4143,0,
      1,1,3,13,31,51,83,163,489,887,863,599,9,13861,0,
      1,3,3,3,19,27,91,115,103,969,593,3667,1867,15433,0,
      1,3,3,13,7,25,47,141,57,553,1785,1709,7453,2209,0,
      1,3,1,13,11,13,71,219,5,451,2043,1605,6439,12203,0,
      1,3,1,13,5,57,61,223,401,413,321,1365,619,12477,0,
      1,3,1,5,25,57,89,211,195,455,1165,3979,6313,5751,0,
      1,1,1,9,31,23,71,145,89,285,1593,1171,5685,15459,0,
      1,3,7,7,9,41,65,251,65,137,1577,3027,5555,2865,0,
      1,1,5,13,27,5,125,21,171,647,983,2921,6623,5695,0,
      1,1,1,13,15,9,117,197,123,953,1191,3657,5757,15957,0,
      1,1,3,7,29,13,5,175,395,127,679,255,6055,7639,0,
      1,3,7,15,15,51,77,147,319,147,1775,3983,3175,5723,0,
      1,3,3,3,7,11,119,41,43,153,975,679,3081,10359,0,
      1,1,5,13,3,7,65,67,63,399,1561,2789,2083,12289,0,
      1,1,7,3,19,53,103,67,35,865,161,93,2533,3851,0,
      1,1,1,11,31,9,29,189,199,817,1571,395,345,3777,0,
      1,3,5,11,31,3,9,67,277,735,181,2777,3009,7233,0,
      1,1,3,3,17,7,17,3,375,933,237,3919,5409,3355,0,
      1,3,3,5,9,27,19,77,221,3,1965,309,3001,15977,0,
      1,1,5,1,3,33,35,133,37,709,627,1705,2525,4307,0,
      1,1,7,3,25,21,105,55,375,681,881,1299,5879,459,0,
      1,3,7,1,13,7,113,103,313,515,1041,3683,4619,5093,0,
      1,1,3,7,19,43,83,37,39,133,1759,1171,1521,13717,0,
      1,1,7,13,7,35,15,155,293,1001,157,3883,405,1797,0,
      1,1,3,3,13,19,125,49,333,387,339,1815,4503,7359,0,
      1,1,3,13,19,19,105,225,151,27,1251,885,4815,7863,0,
      1,1,1,5,7,59,17,145,77,117,1355,1429,2301,16177,0,
      1,3,3,13,5,31,119,167,459,727,1799,2537,695,13637,0,
      1,3,3,3,27,51,107,85,267,57,1279,823,6247,3603,0,
      1,1,7,15,29,17,67,197,215,465,109,3461,5269,15287,0,
      1,1,3,5,11,15,123,53,293,797,1105,1777,6509,217,0,
      1,3,3,13,3,5,109,53,203,693,871,135,369,11149,0,
      1,3,5,15,17,43,81,235,119,817,1777,261,8049,4251,0,
      1,1,3,7,7,13,87,99,481,931,1507,651,5267,8281,0,
      1,3,1,13,27,43,77,225,341,163,933,429,4943,7781,0,
      1,1,7,1,1,49,85,211,449,479,1395,787,5653,14891,0,
      1,1,5,9,25,13,49,85,125,85,1281,3365,4305,11791,0,
      1,3,1,13,3,31,117,39,43,151,663,669,1571,5207,0,
      1,3,7,15,17,7,79,163,37,841,1799,1787,4501,3785,0,
      1,1,3,9,1,23,67,191,449,931,1521,2705,887,7037,0,
      1,1,1,1,5,13,55,161,419,577,1703,2589,2651,2873,0,
      1,3,3,3,5,19,37,169,69,1003,1755,3101,1469,8583,0,
      1,1,1,1,11,33,105,79,283,91,299,835,3193,5593,0,
      1,3,3,13,25,21,81,213,465,475,331,457,61,9511,0,
      1,1,3,11,1,11,77,95,455,949,1999,1833,1275,5631,0,
      1,1,1,1,15,25,51,137,275,451,1179,3595,5177,7105,0,
      1,3,3,3,3,59,79,143,393,583,349,3039,7079,14245,0,
      1,1,7,9,21,11,123,105,53,297,803,4025,5421,14527,0,
      1,3,7,11,21,15,103,109,311,321,1217,2777,5457,1823,0,
      1,3,5,11,19,31,79,89,295,413,817,499,3699,14411,0,
      1,1,1,5,11,3,81,13,315,841,1543,411,6883,6347,0,
      1,3,3,11,23,43,23,131,17,517,995,2687,7443,15085,0,
      1,1,1,1,11,57,73,9,123,905,1763,1789,3701,7131,0,
      1,1,3,5,9,53,99,229,43,207,625,1583,6727,15249,0,
      1,1,7,7,17,39,91,1,297,711,225,513,7391,291,0,
      1,1,7,11,7,55,111,129,423,521,1807,3015,1449,12321,0,
      1,3,7,3,13,9,125,187,11,485,647,275,3495,11989,0,
      1,1,3,11,11,25,49,33,361,105,271,3841,4837,2437,30181,
      1,3,5,1,27,15,119,35,159,273,1489,3157,5433,3337,26859,
      1,3,5,13,23,31,97,145,41,605,1455,59,5389,5527,14447,
      1,1,7,9,7,41,61,193,353,879,1805,581,5447,11177,7331,
      1,1,7,11,29,19,55,207,361,759,63,2255,2119,14671,21783,
      1,3,1,13,17,7,73,179,103,23,917,1205,4925,1691,5419,
      1,3,5,3,15,3,9,109,227,861,867,3529,1535,489,22873,
      1,3,3,9,15,15,95,193,385,997,1525,1865,1425,4079,14771,
      1,1,3,5,5,29,49,171,171,623,1167,3743,1809,12009,7043,
      1,3,7,5,23,11,87,183,299,555,1857,489,3505,9161,28763,
      1,3,5,9,19,21,85,127,337,439,1183,1891,1877,4373,10451,
      1,3,7,13,27,17,29,83,463,385,1167,3453,4523,4759,9321,
      1,1,3,7,21,59,65,83,177,763,317,2913,7527,5967,17167,
      1,1,7,15,13,27,49,35,253,101,1699,355,2181,10859,24221,
      1,1,5,1,17,17,81,91,349,655,1373,2225,945,899,31801,
      1,3,7,11,5,1,81,53,215,587,167,4045,5671,5597,13529,
      1,3,5,15,1,9,59,235,315,195,909,2237,505,10415,28145,
      1,1,1,3,9,31,41,43,275,921,25,671,5737,11241,4193,
      1,3,3,13,29,13,95,213,317,995,1489,3779,3043,8569,28823,
      1,1,7,5,9,49,125,241,87,153,1673,3849,7253,1715,11627,
      1,1,3,9,27,27,19,223,63,463,1095,1395,6643,11589,2145,
      1,1,3,15,21,17,45,23,357,11,1307,1791,2481,2123,24341,
      1,3,5,15,31,53,117,51,433,193,1239,3329,2403,12745,32219,
      1,1,5,9,7,27,9,115,417,579,83,173,4717,15665,27463,
      1,3,5,7,9,9,31,35,249,567,331,905,5101,14817,14255,
      1,3,7,3,1,61,29,129,119,421,1597,2987,3041,7629,23451,
      1,1,7,9,13,1,99,105,107,509,989,2259,1009,6827,8903,
      1,3,5,15,11,29,85,29,265,105,2035,3349,3543,13903,10213,
      1,3,1,1,25,19,53,139,467,485,491,3067,7353,13861,25819,
      1,1,5,3,3,43,41,185,45,463,351,2823,2519,6705,11395,
      1,3,7,13,11,15,87,221,427,673,1631,599,3259,10691,31283,
      1,3,5,11,9,9,15,49,275,335,1613,3587,5309,14849,26475,
      1,3,7,9,29,13,79,225,381,781,1411,2761,7157,14983,19717,
      1,1,7,11,29,25,117,183,101,651,653,3157,445,14389,23293,
      1,1,1,3,5,33,73,155,473,387,591,2045,5965,16299,31499,
      1,3,1,7,11,33,29,21,491,937,729,4075,975,2461,18991,
      1,3,7,15,29,39,105,111,173,943,69,295,8175,13037,26131,
      1,1,5,15,7,5,97,147,105,887,443,2595,5889,10753,1619,
      1,3,3,15,11,45,87,207,353,909,1847,323,2283,12885,16415,
      1,1,5,3,19,33,43,79,115,653,359,2873,4609,12439,6339,
      1,3,7,9,17,61,49,227,291,69,1753,3899,483,3187,29041,
      1,3,5,3,25,35,61,211,393,199,691,1779,6295,13371,15817,
      1,3,7,5,7,23,37,91,245,915,579,867,6193,1063,17363,
      1,3,7,7,23,51,41,63,375,3,159,1889,4419,1687,17977,
      1,1,1,7,13,11,53,43,317,325,1749,2423,4123,8595,20773,
      1,1,7,7,9,9,61,113,437,213,1407,645,4345,807,30411,
      1,3,3,11,17,39,17,113,391,385,581,2023,7449,10153,22033,
      1,1,3,5,29,31,101,215,379,377,1113,2855,7147,14377,25515,
      1,3,5,5,13,3,121,125,227,969,11,1115,5657,9209,6117,
      1,3,7,15,29,17,33,123,317,301,749,1365,5619,605,1613,
      1,3,1,15,7,53,125,249,219,655,105,2825,1649,12783,19777,
      1,1,7,1,25,53,19,53,157,373,1855,495,5065,9465,2313,
      1,3,5,13,3,57,57,161,431,415,1859,1033,6349,1577,31579,
      1,1,7,5,23,63,29,221,13,965,1997,2265,1583,10491,9551,
      1,1,3,13,31,25,23,61,285,5,2005,879,795,13299,19685,
      1,1,7,1,21,45,121,89,263,543,1333,2711,219,10823,26139,
      1,1,3,3,27,13,19,117,161,457,1541,295,4953,12125,14503,
      1,3,5,3,7,63,13,247,439,681,977,2537,6923,10323,7349,
      1,3,5,9,3,51,81,251,349,983,581,2515,2281,2849,31915,
      1,3,5,3,11,63,47,137,303,627,91,2269,7097,2145,31059,
      1,1,3,15,13,17,53,27,133,13,117,1837,4103,5843,29153,
      1,1,5,13,21,33,37,253,465,209,309,49,3209,15677,14569,
      1,1,7,15,13,21,33,203,499,141,1155,3893,1663,2115,27459,
      1,3,5,11,21,9,39,157,257,273,1257,1831,515,7969,20133,
      1,1,3,13,19,29,15,189,103,219,1395,517,7425,6585,15865,
      1,1,5,11,21,31,49,151,39,537,1783,3449,6915,223,11587,
      1,3,3,11,7,63,69,31,27,911,1903,2821,7977,12949,32257,
      1,1,7,9,25,45,23,233,511,595,1383,1721,6789,12055,21179,
      1,1,7,13,1,27,123,49,439,683,501,641,1947,6111,25423,
      1,3,3,5,1,23,57,241,243,593,2039,1617,2209,5171,9675,
      1,1,1,7,5,19,83,55,481,125,177,1021,1139,11403,23099,
      1,1,3,5,29,39,33,217,461,907,733,3795,4811,12939,27715,
      1,3,7,3,7,11,39,165,495,147,999,1827,817,603,9293,
      1,3,7,15,25,53,35,15,431,733,1213,2907,8087,3939,27363,
      1,3,7,13,13,9,33,27,485,183,455,3341,2555,4985,8793,
      1,1,1,15,25,47,75,21,205,15,1639,3067,1295,11693,16903,
      1,1,1,15,3,31,93,57,43,185,251,1899,7885,10829,3609,
      1,1,3,1,29,9,69,223,221,537,365,3411,5771,15279,5309,
      1,1,7,5,1,5,125,243,213,1003,1571,3355,3981,8781,25993,
      1,1,1,13,7,19,53,243,301,75,1183,2723,6687,13,16581,
      1,3,1,13,17,51,91,239,437,191,1065,2495,5755,3405,8299,
      1,1,5,5,11,59,21,169,299,123,1845,2199,2157,14461,10327,
      1,3,7,7,19,47,51,179,41,19,1347,2325,8063,5993,15653,
      1,1,1,9,25,27,7,133,223,533,719,353,7093,8285,10375,
      1,3,5,15,31,5,67,39,441,495,977,3699,1435,11385,14567,
      1,1,3,15,15,39,25,33,91,523,249,4035,769,5181,9691,
      1,1,3,3,3,57,83,187,423,165,161,3453,2241,981,8429,
      1,1,7,15,1,17,57,189,283,11,823,3505,7025,11879,15441,
      1,1,3,11,1,41,7,255,385,339,607,1405,1473,13697,9491,
      1,1,7,15,5,9,91,99,211,233,51,2663,1165,9283,18495,
      1,1,3,7,21,37,13,91,39,27,1021,2813,5937,6645,3403,
      1,3,1,1,29,29,5,69,399,665,1407,3921,2653,11753,18925,
      1,3,7,15,13,41,39,1,437,549,161,2315,5631,8335,22661,
      1,1,3,1,7,17,115,61,69,955,475,3763,8035,927,17893,
      1,3,1,13,21,59,81,145,463,145,1941,2777,7453,14229,11281,
      1,1,1,15,15,11,27,165,461,395,1645,3611,7463,12379,26787,
      1,1,7,9,29,19,27,123,21,149,1643,4001,7207,6769,4647,
      1,1,1,11,13,9,103,139,185,587,591,1113,2223,11667,32671,
      1,3,1,1,31,13,19,93,229,125,1471,2369,3055,10277,28563,
      1,3,7,5,7,53,99,175,161,851,617,4027,2357,11199,1931,
      1,3,5,11,3,31,111,179,237,845,539,1057,259,3417,26637,
      1,1,5,3,21,49,125,119,463,403,737,1811,3941,13015,29081,
      1,3,5,13,5,29,69,251,313,357,663,1097,3307,12845,28495,
      1,3,3,5,29,17,89,15,411,409,2013,757,4085,12521,11131,
      1,1,1,15,7,51,3,193,493,133,381,2027,227,6635,12931,
      1,1,1,15,7,23,99,203,323,1007,1465,2887,2215,1787,22069,
      1,1,5,9,29,59,77,151,509,313,415,3977,5431,8019,8571,
      1,3,1,15,19,13,57,217,87,119,25,1149,5667,3765,6959,
      1,3,7,13,19,31,119,3,457,117,905,361,1483,12405,27005,
      1,3,5,11,15,35,61,77,119,51,1753,2765,1091,10573,23595,
      1,3,3,7,1,35,17,93,197,511,1253,3031,2739,15127,15147,
      1,3,3,1,11,55,55,107,161,75,129,2195,2023,4877,25797,
      1,3,5,7,23,19,113,167,167,271,1303,125,5057,1323,5165,
      1,1,5,3,21,31,11,119,215,483,1535,407,6485,15401,30297,
      1,3,5,9,21,5,77,95,443,247,913,605,365,7465,19707,
      1,3,1,7,17,59,9,35,391,767,1493,475,4725,7529,31579,
      1,3,3,7,31,21,61,31,421,179,273,771,5745,10575,32765,
      1,3,5,15,27,13,125,55,423,1021,497,3521,6903,15111,8285,
      1,1,5,9,13,31,105,93,421,709,643,1079,1533,9149,10799,
      1,3,1,11,19,29,53,199,319,247,655,3039,6411,12267,14245,
      1,3,1,11,9,57,5,91,469,149,259,329,5433,6941,15093,
      1,3,1,5,5,51,59,25,455,367,1623,441,3155,11695,20767,
      1,3,7,7,11,49,113,95,91,389,605,1973,2051,2315,22229,
      1,3,5,3,19,11,99,135,433,781,1473,885,1105,3573,3739,
      1,3,1,11,3,25,9,227,433,723,317,139,6627,8067,28439,
      1,1,1,9,9,9,5,63,241,215,1991,2949,3943,775,31511,
      1,1,3,7,17,49,35,167,131,107,1295,2465,4577,11147,29833,
      1,1,5,1,5,25,119,129,391,743,1069,2957,349,6891,13635,
      1,3,1,7,9,31,63,253,215,51,1347,2361,3125,13049,28461,
      1,1,7,9,3,31,21,163,255,47,259,535,5461,3349,30649,
      1,3,3,13,17,33,87,47,243,709,929,3943,3107,3421,13721,
      1,3,5,11,25,61,61,173,397,735,2005,3355,8121,11593,27697,
      1,3,5,15,17,43,63,231,275,311,1277,2669,7307,2099,9755,
      1,3,5,3,25,43,71,191,9,121,1873,3747,7491,14055,24293,
      1,3,5,13,17,35,113,113,385,941,39,2705,1225,5167,1373,
      1,3,5,5,7,35,19,105,487,71,139,627,4187,3183,713,
      1,1,5,13,29,29,103,5,157,869,1675,423,6689,10697,5303,
      1,1,5,1,29,31,61,111,473,963,685,1483,2383,8109,8495,
      1,1,5,3,19,13,95,113,217,59,1353,1647,3617,3271,2321,
      1,3,5,7,25,35,59,131,309,445,415,93,1453,8789,30201,
      1,1,5,1,5,43,71,241,123,189,831,3469,8093,6187,32721,
      1,3,7,5,25,31,123,171,319,379,889,2365,4881,12225,16609,
      1,3,1,11,27,43,121,63,291,591,811,1995,4777,2083,31385,
      1,1,5,11,27,53,85,187,461,823,703,399,6925,11517,28697,
      1,1,3,5,13,11,33,121,93,717,1275,3877,4247,5845,26909,
      1,3,1,9,7,5,47,199,367,561,185,2855,5997,2699,7581,
      1,1,5,9,23,11,71,201,61,729,1011,3529,663,1413,25675,
      1,3,7,13,27,21,11,127,281,487,1217,3129,5541,3129,17783,
      1,1,5,9,1,29,85,193,213,743,1473,611,391,9405,21137,
      1,3,3,3,31,63,37,147,39,351,79,3069,2441,8901,8777,
      1,1,7,7,25,49,55,47,441,343,1267,1123,5917,14395,10579,
      1,1,7,1,13,55,55,123,103,773,125,2145,4743,13347,2589,
      1,3,7,3,9,33,25,183,469,213,291,75,6725,6847,26745,
      1,3,3,7,15,43,7,79,171,21,1767,2537,4285,12007,24039,
      1,3,7,13,9,61,125,23,227,879,215,1635,2835,883,15939,
      1,1,5,13,25,45,63,43,183,829,149,989,987,3819,12181,
      1,1,3,7,19,27,35,83,135,459,785,131,2655,3329,3009,
      1,1,7,5,11,41,9,219,475,985,1329,3787,1975,4679,8627,
      1,1,7,3,1,17,91,155,3,763,1879,233,215,2955,25993,
      1,1,1,11,25,11,23,227,453,775,1935,3833,4583,269,705,
      1,3,3,11,7,25,105,21,449,555,1275,3475,5503,15617,813,
      1,3,7,13,31,37,25,255,233,663,1155,1563,4775,7449,29949,
      1,1,3,1,23,51,51,137,63,809,349,2789,6953,10605,18959,
      1,3,3,13,21,45,15,161,393,229,437,2967,4019,3893,21305,
      1,1,3,7,5,11,15,211,287,131,1847,2569,7881,15669,31037,
      1,3,3,15,27,19,85,251,221,639,665,3729,5771,7873,28005,
      1,3,7,15,15,47,93,215,343,85,1401,1375,2949,13661,25453,
      1,1,1,9,7,51,53,217,471,389,551,1141,1767,2237,17797,
      1,1,7,9,3,29,65,29,223,591,1719,1049,7643,3853,29867,
      1,1,1,11,13,41,85,29,451,387,1783,3733,8033,4711,31643,
      1,3,1,11,11,57,75,153,7,373,2011,271,469,3267,18969,
      1,1,5,3,19,43,7,243,385,293,923,843,4895,469,8421,
      1,3,1,15,29,47,17,125,471,927,349,3859,3059,11483,14791,
      1,3,1,11,17,17,111,109,9,213,1313,3903,4411,4329,28277,
      1,3,3,15,1,55,47,69,143,789,1149,3833,5053,6949,10569,
      1,3,5,7,11,15,79,83,123,937,1115,2775,3041,11869,21167,
      1,3,7,13,9,47,45,221,139,923,1661,1379,2485,7233,6035,
      1,1,3,3,11,55,77,3,87,693,1991,1145,2783,16207,24569,
      1,1,5,11,3,35,91,9,391,927,101,1839,3755,10345,16907,
      1,3,5,3,5,49,79,91,205,443,1369,197,2537,11219,17765,
      1,1,3,15,9,7,25,25,357,247,477,421,7679,5987,30079,
      1,1,5,3,29,5,89,117,481,491,371,389,7101,2253,23617,
      1,1,5,13,29,59,17,181,511,291,1991,3499,8177,5559,30045,
      1,3,3,11,23,31,117,217,241,115,749,945,1897,12253,8473,
      1,1,7,15,25,47,31,1,165,311,635,3629,1593,8305,30033,
      1,3,5,9,3,17,101,237,379,503,49,929,1687,3865,26723,
      1,3,5,5,15,41,1,239,53,215,1733,827,579,4089,6579,
      1,3,1,15,15,21,35,21,403,257,1475,2403,4705,11553,203,
      1,3,5,11,9,53,113,9,447,511,543,3141,7389,11249,431,
      1,3,5,9,9,11,55,93,325,411,305,2573,6871,12339,6435,
      1,3,3,7,31,27,21,113,99,853,365,589,3731,10875,12767,
      1,3,1,7,15,27,31,17,275,93,1161,2619,1329,7307,587,
      1,3,5,9,17,47,49,237,27,193,1237,591,5151,5521,31583,
      1,3,5,3,13,1,27,87,43,977,305,3293,2475,14571,18321,
      1,1,5,7,15,13,101,1,291,807,1711,2277,5573,11051,13133,
      1,3,3,1,9,3,65,81,415,733,1527,2747,6069,159,7095,
      1,3,3,15,27,1,71,49,231,851,2039,613,1899,2537,14511,
      1,1,1,11,3,41,55,23,247,1011,581,2363,2745,1337,20931,
      1,1,3,11,17,61,67,255,143,357,945,3407,5817,4155,23851,
      1,3,5,3,23,1,75,247,265,413,1899,2565,6629,15655,16117,
      1,1,1,9,11,49,11,189,223,177,1457,1931,163,15905,17297,
      1,3,7,13,17,1,111,189,343,961,427,2507,2393,8653,6353,
      1,3,7,13,23,61,59,51,313,963,791,3681,5637,3965,9263,
      1,3,7,7,21,53,127,141,499,859,337,2835,3195,4351,32369,
      1,1,7,5,1,5,53,63,497,535,35,305,4395,9757,13193,
      1,1,5,13,13,31,59,229,211,745,1453,3677,3005,7703,23907,
      1,3,5,5,7,63,17,197,493,861,499,3015,6349,1815,7437,
      1,1,1,13,13,37,29,189,253,1017,321,3145,407,7547,17099,
      1,3,3,3,23,53,69,77,175,17,1831,841,3851,1295,32107,
      1,3,7,13,13,39,107,237,389,729,635,3717,3041,3169,14987,
      1,1,3,1,25,7,69,35,495,49,659,2783,6051,13875,23927,
      1,3,7,5,5,25,49,7,193,493,93,657,1515,13975,14155,
      1,3,1,1,11,15,113,45,21,595,731,3397,4117,9711,16625,
      1,3,3,9,19,19,59,7,105,579,599,2859,97,14717,15361,
      1,1,1,5,27,49,113,5,367,563,1397,2805,3021,3111,20671,
      1,3,3,15,27,51,99,167,109,365,1959,1523,6959,14405,18191,
      1,3,1,5,21,51,125,67,123,45,1657,51,4825,14081,31049,
      1,1,5,7,21,59,21,249,77,793,1687,2561,2241,4321,7477,
      1,1,1,7,15,35,71,29,267,611,1813,1823,7039,3299,9919,
      1,3,7,11,21,59,109,213,371,785,659,1687,4827,6017,19619,
      1,1,3,11,27,17,1,55,367,939,333,127,5105,2405,28139,
      1,1,7,13,5,35,59,133,509,573,625,3857,7935,5279,3727,
      1,1,1,7,11,47,127,157,19,403,151,1143,7407,8985,32521,
      1,3,1,1,5,13,105,123,63,139,1569,1983,563,7175,27705,
      1,1,3,13,9,35,105,227,145,21,1369,57,393,2921,18511,
      1,3,1,7,17,61,99,187,261,281,437,2219,5999,1857,18001,
      1,3,3,5,1,59,67,45,451,439,2005,3607,3,7167,14227,
      1,3,3,3,29,19,25,251,275,733,1749,4021,871,3227,13701,
      1,3,3,13,27,53,57,243,491,521,1921,1037,5013,5703,15261,
      1,3,1,11,13,57,1,15,123,533,785,335,1423,14269,3483,
      1,3,7,13,15,55,5,139,385,47,1981,1291,7397,12925,29445,
      1,1,7,1,23,23,59,93,117,57,63,3047,4849,11637,25311,
      1,1,7,13,19,37,25,203,477,447,1345,3485,2099,13347,11621,
      1,1,7,3,11,23,81,17,41,735,1149,3253,7665,8291,22293,
      1,1,5,3,15,9,57,167,463,493,747,1947,6471,1111,31619,
      1,1,5,15,7,15,107,205,325,167,1749,927,3589,6127,7617,
      1,1,1,13,21,25,83,147,411,399,1423,2279,3661,7591,17429,
      1,1,1,9,5,17,69,205,243,647,473,1717,1977,10725,2913,
      1,1,3,5,5,37,103,15,485,641,1761,3755,6997,10985,11773,
      1,1,5,13,9,51,87,195,97,807,1801,961,6341,4307,29105,
      1,3,1,13,9,35,83,61,387,817,951,3993,7831,8479,23941,
      1,1,7,11,19,47,75,37,91,337,953,1169,163,2259,24713,
      1,1,1,11,13,15,83,171,159,87,619,2973,2653,13725,12499,
      1,3,5,3,5,63,119,25,343,269,553,2183,959,3825,22189,
      1,1,5,15,5,37,89,109,497,1013,265,669,1859,2647,3445,
      1,3,3,9,21,21,15,245,107,649,367,1601,7279,15783,4943,
      1,3,3,15,5,41,125,113,159,161,1191,3491,3531,55,20857,
      1,3,5,9,21,57,21,195,99,193,1915,2923,6349,15085,24929,
      1,1,1,11,31,11,73,141,361,621,1021,2067,5115,12665,26845,
      1,1,1,3,29,11,43,61,209,923,1753,1937,843,205,8367,
      1,1,1,5,15,33,119,209,215,973,1775,815,6693,7957,14517,
      1,1,1,5,17,57,27,147,489,59,1439,2279,445,11791,19739,
      1,3,1,7,11,55,1,83,305,17,1909,405,2325,5293,28559,
      1,3,3,7,11,27,103,157,455,1005,2033,3145,1919,15723,25197,
      1,1,5,11,15,51,37,131,503,1007,1795,2421,1335,7413,21741,
      1,1,3,1,23,63,69,83,419,283,583,123,7725,2243,8403,
      1,1,5,5,27,45,109,17,299,65,351,947,1165,10723,2053,
      1,1,3,3,23,61,115,253,1,931,1481,3187,441,14735,27207,
      1,1,5,3,25,11,83,141,359,343,901,1629,731,12841,14357,
      1,1,3,9,7,45,97,3,299,217,663,1527,6379,4527,26147,
      1,1,7,9,11,53,9,203,337,713,1517,719,4587,11443,26905,
      1,1,7,9,11,41,125,213,237,377,361,3231,4223,3263,12655,
      1,3,7,7,7,33,99,19,117,273,985,107,3831,10135,19423,
      1,1,5,15,25,41,13,125,449,169,1149,4021,5663,3077,19163,
      1,3,5,9,25,57,47,103,269,51,1805,2503,6687,8065,12045,
      1,3,5,7,3,35,87,225,189,229,931,3293,1347,1427,3269,
      1,1,1,3,5,31,61,19,247,9,1667,343,559,2703,3763,
      1,3,5,15,31,19,57,187,109,121,1287,2269,659,16235,1273,
      1,1,1,3,5,47,59,243,255,97,1959,1723,1347,3019,26989,
      1,3,3,15,29,35,75,67,497,731,193,3307,3579,12005,7209,
      1,1,5,9,13,35,79,213,51,983,1927,1793,5037,5463,965,
      1,1,7,11,5,41,7,83,15,411,1775,3515,6755,3249,16425,
      1,3,5,1,19,61,3,19,395,819,1331,179,5225,5333,3601,
      1,1,3,9,7,5,87,15,387,609,1465,277,987,8377,903,
      1,1,1,3,15,11,123,107,355,333,285,1801,6989,1549,25791,
      1,1,7,13,27,13,73,111,481,227,1091,365,5713,5087,27217,
      1,3,3,15,1,55,95,213,377,405,139,1867,2175,4217,28813,
      1,3,5,11,21,43,109,155,181,901,1951,507,4389,10815,3141,
      1,1,1,15,17,11,43,215,501,19,259,3479,6381,6927,31247,
      1,3,5,15,19,61,75,41,391,95,865,1441,7993,13979,24663,
      1,3,1,3,21,15,115,213,1,645,777,1517,2543,11223,3633,
      1,3,5,3,9,57,39,211,407,65,1795,2805,2799,8691,1987,
      1,1,3,13,17,55,47,113,29,139,1301,3303,1129,13947,29821,
      1,1,3,13,5,35,97,151,477,409,1397,3399,4421,15929,6163,
      1,3,1,9,21,51,99,133,149,763,623,173,4311,11081,1095,
      1,3,7,15,13,3,99,3,195,907,1335,1355,7977,5773,32383,
      1,1,3,9,17,43,43,217,475,917,1373,1677,4871,9619,16657,
      1,3,3,7,31,31,55,11,73,693,25,417,1195,6225,32279,
      1,3,5,9,21,57,127,149,79,379,1609,2543,6473,16033,27191,
      1,1,5,1,13,9,81,153,297,789,1749,2819,3961,11231,24927,
      1,3,5,3,23,61,45,43,43,133,1481,1543,2991,13739,10287,
      1,1,3,9,25,43,31,177,337,193,1083,1,991,9725,8379,
      1,3,5,11,13,33,65,83,421,149,409,2443,7423,8847,29599,
      1,1,5,11,11,1,23,225,77,585,1505,2525,739,10915,25733,
      1,3,7,13,7,55,3,223,415,521,1865,2349,5663,7455,16569,
      1,1,7,13,1,45,121,49,463,99,1061,2559,5087,13389,11035,
      1,3,7,11,31,51,35,235,385,1023,1771,2013,5437,4877,22119,
      1,3,3,11,21,3,11,119,81,737,1093,2377,4055,1121,15767,
      1,1,5,13,9,3,83,217,387,249,1047,1861,4103,15367,24545,
      1,3,3,1,5,37,43,183,383,463,937,1165,1481,959,17047,
      1,1,3,5,7,43,127,243,81,1021,165,753,4711,12965,22049,
      1,1,5,5,3,61,65,53,425,89,5,1467,1395,9579,8961,
      1,3,7,13,11,35,123,21,83,689,667,1203,5959,15697,26885,
      1,1,5,13,9,49,41,101,291,339,1067,657,4453,1137,21131,
      1,3,3,3,17,61,11,213,27,805,1691,1057,6011,11941,18883,
      1,3,1,7,3,51,5,63,121,3,245,2631,3737,16121,26803,
      1,3,1,1,23,51,79,19,161,107,609,3489,3389,4035,2427,
      1,3,1,1,17,11,101,101,373,63,1641,285,1333,165,14025,
      1,1,1,5,1,51,83,137,45,1019,821,867,6055,10443,9857,
      1,3,1,5,17,23,25,181,429,495,317,3219,5963,13945,9969,
      1,3,7,3,3,15,123,191,369,177,1697,2113,3889,5201,21839,
      1,3,1,11,21,39,51,139,271,605,1007,3513,3365,3781,6799,
      1,1,7,5,13,19,47,165,249,405,255,1295,4513,14395,5587,
      1,1,3,7,5,17,99,1,393,31,621,797,6113,16003,32043,
      1,3,5,13,11,21,65,81,147,443,775,3671,7029,11749,3339,
      1,3,7,1,23,33,99,177,161,577,1729,617,3465,11787,17577,
      1,1,5,7,15,15,53,193,97,255,1223,545,5153,873,24525,
      1,3,5,1,7,57,47,121,383,835,1709,2363,4731,12163,7001,
      1,3,3,11,19,33,63,99,387,95,783,1009,6373,4021,7685,
      1,1,1,15,25,33,73,135,335,785,935,1927,5847,10501,7719,
      1,1,5,3,27,45,71,215,489,157,1189,2577,6901,10219,3025,
      1,1,7,7,21,3,97,225,101,159,293,2789,7955,14829,1209,
      1,3,1,5,23,41,83,63,361,195,1707,2081,5363,6327,179,
      1,1,3,1,21,51,59,67,175,363,825,2971,3321,8837,11805,
      1,3,7,1,19,3,15,21,429,675,1589,2615,2575,1537,7139,
      1,3,3,5,21,29,17,115,345,397,523,1699,7043,11173,3023,
      1,1,5,7,19,63,99,175,91,433,153,3749,517,13667,7423,
      1,3,7,3,25,23,53,149,65,551,1231,365,6637,15137,16319,
      1,3,7,13,5,45,11,151,323,31,1749,409,6753,10503,14991,
      1,3,7,3,5,21,29,117,321,341,1811,3619,4337,12255,8629,
      1,3,7,3,7,3,5,221,407,671,1763,3669,2353,8175,23489,
      1,1,3,7,11,55,53,185,247,35,1823,513,1379,11827,20069,
      1,3,3,5,29,51,73,191,185,961,881,2019,5651,1019,15587,
      1,3,7,13,7,55,59,5,417,829,453,2339,587,13283,797,
      1,3,7,3,11,41,75,85,65,149,1583,529,2707,11479,7109,
      1,3,7,9,13,57,37,243,91,613,665,171,1631,13737,2377,
      1,1,3,7,5,43,97,53,477,793,999,3647,2555,7371,19295,
      1,1,7,1,1,9,99,253,317,817,1559,2081,2529,14611,15997,
      1,3,3,1,5,41,57,121,387,441,709,1511,7045,8409,13297,
      1,1,1,13,29,57,63,183,327,473,1943,213,3973,16289,2739,
      1,3,7,9,25,15,75,185,335,881,1041,3339,4471,6823,21121,
      1,3,3,13,23,3,57,117,511,927,771,3229,949,15487,11963,
      1,1,3,7,27,19,55,207,331,705,1945,797,7125,10493,16585,
      1,3,1,1,29,7,91,93,459,93,1501,1927,6415,16255,9823,
      1,1,5,5,31,11,97,179,505,807,877,4003,4377,8851,4239,
      1,1,3,5,11,25,17,131,23,95,311,1429,2029,13091,23739,
      1,1,3,11,13,27,33,127,481,117,1127,1619,6493,8507,6615,
      1,3,1,13,19,27,89,101,27,235,1579,1701,4421,16037,16239,
      1,3,1,15,1,15,3,117,317,475,1691,2423,5519,1703,2969,
      1,1,3,1,13,15,19,37,237,467,1321,453,2169,13313,31499,
      1,1,3,15,29,55,31,199,85,285,967,367,3941,151,20587,
      1,3,7,15,7,13,31,35,117,543,1179,3441,3039,11225,30229,
      1,1,3,15,3,43,1,63,353,395,1775,3493,5175,13193,25343,
      1,3,3,15,17,25,57,205,411,83,1877,2093,5599,12115,8751,
      1,1,1,11,15,9,115,99,85,887,987,4015,7077,3739,21505,
      1,3,1,11,25,39,127,37,329,273,1531,3211,7115,15501,26575,
      1,1,5,13,15,3,3,101,431,645,493,723,8083,1423,14879,
      1,3,3,5,31,35,37,131,259,849,325,3403,3627,3295,30885,
      1,3,7,1,9,3,31,201,379,907,1005,3333,7457,2533,30357,
      1,3,1,9,7,7,95,103,121,157,895,2683,5839,12403,14327,
      1,3,7,3,13,5,55,233,3,855,859,1115,3883,8041,3353,
      1,1,5,9,3,55,99,79,263,831,1579,205,5673,1999,14879,
      1,3,1,5,17,25,85,19,189,141,877,667,4461,11915,23247,
      1,1,5,5,1,35,15,219,469,725,1793,3683,3661,15627,30197,
      1,1,7,5,27,3,41,153,431,487,759,1345,6735,9937,26277,
      1,1,1,11,11,13,41,121,265,465,1447,5,3407,1907,10037,
      1,3,5,9,15,63,5,7,407,83,365,3687,7721,6973,16967,
      1,1,7,7,5,41,75,155,417,565,1199,1111,2823,10703,22561,
      1,3,7,5,7,43,39,185,105,327,1977,1137,3261,10583,11661,
      1,3,7,7,19,19,103,137,169,273,1357,3413,7647,10531,32489,
      1,1,3,13,13,3,81,23,161,295,735,2031,1027,15513,20165,
      1,1,5,1,15,1,91,35,375,207,1417,1115,2237,11749,8509,
      1,3,7,3,25,51,49,219,195,417,1523,3953,5739,7499,27071,
      1,1,3,11,23,29,19,81,421,633,513,547,7545,29,11909,
      1,1,1,7,13,61,33,243,221,231,111,879,2861,1795,27531,
      1,3,7,3,19,21,1,141,159,605,969,3013,6583,2447,19919,
      1,3,7,3,31,9,91,83,29,873,929,43,2253,12539,23951,
      1,1,5,3,31,15,87,105,319,973,1489,3417,3377,15749,2357,
      1,1,3,15,7,23,3,81,383,419,713,997,6873,593,285,
      1,3,3,1,29,13,29,101,441,693,2039,2951,5921,12129,12053,
      1,1,3,15,9,29,97,117,421,433,1017,125,3607,9415,6843,
      1,3,5,9,11,13,75,155,413,75,109,1599,6161,16115,12621,
      1,3,3,3,11,13,49,225,401,599,1815,1643,7853,13305,25195,
      1,3,7,5,15,11,27,95,387,931,549,2179,3397,15883,16563,
      1,1,7,3,9,39,121,5,453,27,1747,657,2593,1289,12577,
      1,3,7,5,25,25,109,49,185,985,631,803,3865,8955,17901,
      1,1,3,13,3,59,47,49,139,275,1471,2995,5593,14011,18741,
      1,1,5,15,29,11,97,225,245,291,1873,2365,767,3419,14943,
      1,3,3,5,15,17,19,209,359,891,1375,2003,7247,5299,28841,
      1,3,7,7,9,55,105,35,77,47,1023,13,2901,847,10265,
      1,3,7,7,7,5,65,233,141,277,1333,2357,443,7257,21979,
      1,3,5,11,13,63,41,87,193,737,1085,2317,7869,10149,12163,
      1,3,1,1,7,57,75,235,461,857,155,2679,5925,2565,10881,
      1,1,7,15,13,41,63,135,433,387,1943,2249,5469,11679,28661,
      1,3,3,13,5,3,103,161,367,649,789,1179,4163,5699,16787,
      1,3,7,7,31,13,45,141,113,769,1035,457,6709,14989,27311,
      1,1,3,1,1,43,119,145,111,593,1139,417,637,4437,17285,
      1,3,5,9,9,33,19,99,201,685,1793,2621,6857,8769,5623,
      1,3,5,5,23,43,27,189,325,415,215,1253,3599,1215,10093,
      1,1,3,13,11,35,113,173,503,19,1459,503,5363,3967,13945,
      1,1,5,11,31,49,13,173,199,623,1231,2495,6581,7957,25321,
      1,3,1,9,23,3,79,149,505,937,1839,3701,1673,8589,8031,
      1,3,3,5,21,27,107,11,505,407,177,3593,4729,12773,11685,
      1,3,1,11,29,49,79,53,61,895,2035,563,5613,6065,6207,
      1,1,3,7,1,53,3,215,99,865,1749,3533,4305,1243,28463,
      1,1,1,13,31,59,115,53,403,909,847,103,4967,10623,30073,
      1,1,7,5,27,1,119,83,457,81,395,811,6221,14337,541,
      1,1,5,5,5,53,83,117,269,327,875,101,3343,715,26339,
      1,1,1,11,31,39,121,147,305,383,1211,1897,7647,11687,18907,
      1,3,3,15,23,53,17,85,395,503,61,1745,4713,4641,13787,
      1,1,7,7,27,1,105,29,287,37,959,975,4427,4705,10175,
      1,3,3,5,7,63,57,199,27,107,1095,3923,6969,713,11619,
      1,3,5,1,5,49,85,45,449,45,49,3419,1109,455,15917,
      1,1,1,5,13,15,39,27,467,85,1537,3055,1977,8829,25231,
      1,1,1,15,1,47,23,121,147,547,1865,1491,779,3515,12667,
      1,3,3,1,19,5,77,101,1,721,1149,2967,4925,11889,16655,
      1,1,1,7,1,35,95,239,127,855,1031,455,7631,6039,21983,
      1,3,7,9,23,43,75,105,335,223,1825,3217,413,7473,30005,
      1,1,5,15,29,9,43,145,223,523,511,323,5955,11141,22533,
      1,1,3,1,13,61,93,133,461,233,383,693,7347,3165,27493,
      1,3,7,1,13,45,113,207,53,1007,815,1145,2937,289,22195,
      1,3,5,5,19,17,113,89,19,1023,1625,3277,697,5187,15433,
      1,1,3,13,21,15,15,197,409,391,1993,2475,3189,4431,29585,
      1,1,5,5,31,7,111,231,187,543,45,3863,3811,4573,4437,
      1,3,3,7,19,7,123,23,79,513,189,3663,1291,13257,8949,
      1,1,5,13,3,53,109,133,157,223,651,3059,6055,14455,26903,
      1,1,7,1,23,63,59,229,17,199,643,637,7631,13647,7399,
      1,1,1,3,1,51,119,67,335,543,913,3565,4795,13405,7463,
      1,1,5,3,31,5,91,97,23,223,837,1353,1929,12043,10039,
      1,3,5,7,19,3,79,171,301,687,1545,355,4709,12965,16797,
      1,3,5,11,11,49,111,123,251,569,1605,401,5439,13519,8847,
      1,3,1,3,3,53,7,55,369,633,181,4037,2993,15815,8661,
      1,1,1,13,31,29,75,167,279,597,539,1791,8013,4387,9717,
      1,1,5,7,17,15,99,183,211,49,225,3143,4537,13141,23375,
      1,1,3,5,3,59,25,149,467,69,1939,1007,2765,4693,29815,
      1,3,1,3,17,33,119,189,447,251,879,177,5395,13487,9587,
      1,3,3,7,15,31,115,3,21,817,475,1849,6041,12541,18701,
      1,1,5,13,31,33,7,115,361,587,1919,1007,3537,7493,19357,
      1,3,7,13,23,35,15,111,123,633,805,1983,2109,14477,4985,
      1,3,3,11,25,13,11,205,97,893,927,1291,4007,13593,29693,
      1,3,5,15,9,13,121,89,215,823,1389,1581,8095,4707,16061,
      1,3,1,3,23,39,83,23,47,941,1419,2389,5699,7519,5829,
      1,3,1,9,23,43,79,237,93,203,695,225,5645,3591,16775,
      1,3,5,3,15,19,89,129,375,125,225,1323,2267,11607,17937,
      1,3,3,1,31,37,93,133,377,959,707,621,7179,15493,30287,
      1,3,7,13,5,13,15,1,37,525,1641,2829,6139,4069,19187,
      1,3,3,9,17,3,67,97,375,845,403,973,3919,2275,31627,
      1,1,3,3,25,7,91,67,271,465,481,3477,5229,241,8411,
      1,1,1,11,1,41,109,115,75,787,309,2887,179,9073,13895,
      1,3,3,15,11,31,113,91,303,907,1933,2167,7799,11821,20659,
      1,3,1,15,27,17,21,41,99,137,1397,929,5819,11977,6201,
      1,1,7,13,21,29,47,239,287,305,899,2711,1723,3315,199,
      1,1,1,3,31,21,101,149,107,761,1197,1703,4803,8411,10649,
      1,1,5,15,23,45,109,221,85,619,169,1013,3305,9451,26189,
      1,3,5,13,7,57,19,153,231,627,565,1595,6309,5037,25505,
      1,1,7,7,1,45,43,79,271,59,219,2255,1785,7919,24061,
      1,3,7,5,31,57,57,231,33,227,531,679,1141,85,19777,
      1,1,3,15,11,59,59,169,459,693,907,1191,3783,12809,6263,
      1,1,7,13,19,21,105,65,267,141,1547,781,7295,13565,17775,
      1,3,3,5,31,63,97,155,477,661,329,797,2539,4061,10537,
      1,3,3,7,11,17,119,89,71,103,1043,413,6035,12829,11559,
      1,3,1,9,5,19,53,185,103,629,2015,1257,5163,10581,13449,
      1,1,1,5,23,35,25,129,179,959,677,2249,6315,12151,3459,
      1,1,1,1,9,47,93,45,35,45,265,2065,6225,25,27135,
      1,3,1,11,21,53,127,163,311,667,597,1561,4515,23,9551,
      1,1,3,3,7,47,105,211,241,95,389,899,6001,8129,19889,
      1,1,3,15,29,45,9,27,483,799,269,1811,4493,7109,22149,
      1,1,3,3,29,5,57,205,187,615,1677,3987,4577,8799,16311,
      1,1,5,3,15,5,91,101,319,445,1261,2039,4071,8249,11611,
      1,3,7,11,19,17,1,185,153,579,1001,2031,2295,16335,24771,
      1,3,3,15,13,45,93,185,319,667,1085,93,577,11551,11355,
      1,1,7,13,3,61,45,191,51,981,1151,2715,2503,4147,4587,
      1,1,3,3,27,17,71,141,57,981,1033,333,4639,15885,1039,
      1,3,3,15,21,55,33,123,357,893,829,4045,5027,11727,13357,
      1,1,1,9,31,47,27,223,311,205,179,3411,4019,10997,28115,
      1,3,5,1,3,39,15,7,501,641,735,295,2005,12641,19779,
      1,3,3,1,15,1,75,243,329,267,1323,2285,5389,11881,15737,
      1,1,3,3,13,17,101,99,209,939,1147,3221,5159,3435,183,
      1,1,1,1,27,43,29,179,179,659,807,313,4165,963,11317,
      1,1,3,13,9,51,125,245,381,555,1383,3887,2045,12829,12029,
      1,1,1,9,29,39,55,127,235,617,1553,3133,7735,14725,16733,
      1,1,3,5,15,9,47,217,89,987,1083,1045,4745,12915,13719,
      1,3,3,7,23,3,35,79,45,435,1549,2645,2831,10359,10041,
      1,1,7,15,31,61,25,223,511,319,487,1677,739,7097,18417,
      1,1,7,5,19,21,123,237,299,367,1341,1449,2949,8629,11051,
      1,3,7,7,31,53,125,33,257,719,1297,895,5095,10237,12309,
      1,3,1,5,31,59,73,211,97,209,1289,4033,6143,14275,7997,
      1,1,5,7,31,5,75,105,389,985,9,4033,1185,7821,19083,
      1,1,1,15,11,39,73,253,275,813,25,3441,2493,5873,3739,
      1,3,7,1,31,19,119,5,109,397,1329,3347,5941,12449,2533,
      1,1,1,1,5,59,61,175,435,985,65,3781,5425,15073,16361,
      1,3,5,7,31,13,53,87,69,305,1455,273,2197,4277,24423,
      1,3,3,15,13,13,91,171,71,583,15,3599,6801,10041,26097,
      1,3,3,5,5,13,91,225,63,69,1795,341,461,5015,9471,
      1,3,7,5,21,55,109,39,459,925,229,2855,5807,2117,31739,
      1,1,3,3,1,5,17,177,401,727,1555,3097,1243,5933,14579,
      1,1,7,3,19,19,37,87,105,73,197,4067,6237,10553,9207,
      1,1,3,15,1,55,119,115,441,3,1003,1631,197,12929,25385,
      1,3,7,11,31,1,119,49,467,647,685,2771,3689,11049,26787,
      1,1,1,11,19,19,21,73,459,935,615,371,1099,14407,10375,
      1,3,5,13,15,3,107,179,259,677,1101,315,7673,14639,11241,
      1,1,7,9,15,21,93,25,349,23,1087,27,5691,12997,29301,
      1,3,3,5,7,43,1,195,69,753,1315,2629,3259,5595,19439,
      1,3,5,5,31,9,75,217,217,197,1925,2033,3585,15219,20251,
      1,1,5,11,17,31,3,209,315,49,949,2267,4611,4375,16431,
      1,1,7,9,17,35,13,115,119,553,1527,2857,3599,391,25101,
      1,3,3,15,13,59,17,177,301,719,909,1663,5033,1129,529,
      1,1,7,5,15,13,99,157,379,975,1019,2251,3807,10621,351,
      1,3,3,13,5,57,5,31,361,981,883,3723,2259,5151,11783,
      1,1,1,13,1,43,125,19,77,509,1817,3795,1863,8401,27253,
      1,1,5,7,19,41,21,151,89,189,769,1937,4497,13607,24691,
      1,1,1,9,21,9,1,195,31,907,1719,1549,809,13629,16597,
      1,1,1,3,21,61,103,219,311,849,523,21,4533,6367,3935,
      1,1,7,9,7,33,77,19,489,933,1729,1813,6741,10701,7,
      1,1,1,5,23,53,43,63,453,209,1313,2847,2641,13783,14983,
      1,3,7,7,15,45,83,241,509,659,213,221,5205,6019,18945,
      1,1,5,9,25,43,37,9,191,505,765,295,953,1045,11203,
      1,3,7,11,5,49,45,177,379,695,355,1711,7747,497,7597,
      1,1,5,13,23,47,101,145,301,207,195,2225,8093,15345,14097,
      1,3,7,13,9,9,55,223,343,921,1825,3281,2627,855,27651,
      1,1,7,1,21,1,67,149,433,111,577,3675,495,9043,23613,
      1,3,1,13,9,39,37,73,117,559,1131,2511,7599,8393,24747,
      1,3,3,7,11,15,85,229,7,21,1649,739,375,13991,27053,
      1,1,5,5,15,41,49,117,173,825,1343,377,1789,12519,30667,
      1,1,7,15,9,11,97,99,347,729,9,1703,1177,5189,9061,
      1,1,5,11,15,25,99,63,89,675,561,215,8111,3955,24635,
      1,1,1,1,7,53,99,193,233,731,733,1883,7783,14413,14003,
      1,3,5,7,31,23,45,153,337,293,443,2301,5135,7455,13123,
      1,3,1,3,23,53,23,165,53,875,1543,1035,4247,5101,28445,
      1,1,1,15,13,41,77,93,205,743,1101,1413,2371,7183,12337,
      1,1,3,15,17,63,25,101,147,149,1207,3525,2661,9539,11145,
      1,3,1,9,17,5,3,35,389,909,1017,2803,5243,13025,8851,
      1,1,7,15,19,27,69,91,71,547,1421,831,6969,5517,28233,
      1,1,3,3,17,45,55,63,263,819,1211,2739,655,13269,22281,
      1,3,1,5,23,13,81,251,83,551,491,1029,3561,357,23393,
      1,3,1,13,25,27,93,143,407,403,1395,1733,3187,1917,31453,
      1,1,7,13,3,21,85,113,483,461,1343,561,2081,10857,24253,
      1,1,1,1,11,11,53,135,25,163,1729,617,1533,10881,16041,
      1,1,5,1,3,49,125,139,77,891,815,3431,4875,12513,4595,
      1,1,1,1,27,63,111,109,421,425,345,1613,5447,1357,32413,
      1,3,5,3,17,5,37,171,259,281,1003,2901,3241,15557,21415,
      1,1,5,11,15,55,75,199,493,215,1625,2345,7873,2325,11003,
      1,3,7,1,21,33,23,5,495,941,1185,475,5799,15161,10677,
      1,1,5,9,31,37,37,29,217,389,297,3097,7319,2601,15307,
      1,3,7,5,7,45,111,167,297,275,1669,2489,1511,15753,1289,
      1,3,1,7,3,45,19,11,189,199,1227,2647,1897,9077,17189,
      1,1,1,13,15,39,19,179,147,341,283,3029,7599,8937,18761,
      1,3,3,9,3,11,41,255,365,835,921,389,919,15223,14541,
      1,1,3,3,5,37,29,203,313,271,1207,487,3711,3811,26757,
      1,3,7,9,19,53,49,139,351,537,1681,1595,5399,13839,28685,
      1,3,1,1,15,35,21,37,247,891,1855,1243,3137,10381,30379,
      1,3,7,5,9,47,91,25,479,337,781,3545,1045,9491,22853,
      1,1,5,15,19,31,81,5,117,923,565,2443,7383,1795,11685,
      1,3,3,5,17,15,21,245,489,889,2047,2737,7445,14785,13401,
      1,1,1,15,19,45,67,117,299,607,953,743,6863,12123,6701,
      1,1,3,1,1,43,19,129,345,861,209,2387,7205,7131,8235,
      1,3,5,1,1,13,75,99,333,157,23,1217,1857,15479,16031,
      1,3,3,11,7,61,119,89,491,401,227,1739,3807,16003,2875,
      1,3,7,15,13,55,3,159,405,593,975,361,2563,6061,28087,
      1,1,3,13,19,5,5,9,119,41,33,1111,4443,4663,28841,
      1,1,7,7,25,59,125,255,49,947,1673,2947,6369,2267,8813,
      1,1,5,15,25,25,111,193,217,193,821,2779,69,2957,27043,
      1,3,5,7,21,19,51,157,203,487,1745,1875,911,14071,7557,
      1,1,5,9,3,15,55,73,313,245,1061,1929,3035,607,11563,
      1,1,5,7,3,57,105,121,461,43,803,1801,4059,2157,17547,
      1,3,7,7,19,11,1,121,499,841,601,3515,2969,13697,8917,
      1,3,3,3,13,35,113,231,391,689,697,2871,7387,715,27005,
      1,1,1,13,19,5,17,43,175,291,987,1917,7635,15655,10689,
      1,1,7,15,19,37,121,243,125,623,1231,29,2325,5147,21435,
      1,3,5,15,25,27,57,187,77,401,1489,2977,5415,3381,2551,
      1,1,1,7,1,1,85,27,115,559,9,2365,711,5733,2819,
      1,3,1,15,9,29,61,113,169,349,591,1061,6041,7613,23691,
      1,1,5,1,13,45,49,227,345,563,87,3597,3961,7205,8441,
      1,1,1,5,3,21,121,183,463,83,1365,539,1485,10063,24867,
      1,3,5,5,3,61,101,237,41,147,1907,3049,7583,8283,6099,
      1,3,1,15,31,57,19,155,445,805,1793,207,1975,3357,14281,
      1,1,7,13,9,39,27,73,165,345,543,4095,133,10469,11573,
      1,1,7,15,17,57,99,81,359,367,1057,1173,4225,15127,2615,
      1,3,5,3,31,23,113,111,495,947,1625,1195,2053,1509,1347,
      1,1,5,5,9,47,25,63,455,107,771,3815,3827,16287,11615,
      1,1,7,9,17,61,51,215,63,123,1253,3927,721,9647,3283,
      1,1,5,15,11,17,83,255,473,107,681,763,7855,8043,31503,
      1,3,1,7,7,31,37,5,253,155,2017,609,1421,14927,25241,
      1,3,3,13,31,25,21,241,431,193,681,2265,5091,11479,21443,
      1,3,5,5,15,9,49,255,157,995,631,1995,3605,9085,24245,
      1,3,3,7,19,31,85,153,493,951,451,1587,6609,3681,13205,
      1,1,5,1,17,41,107,231,307,361,575,3239,3443,16159,20625,
      1,1,7,9,31,49,93,79,181,117,1241,3645,4901,12599,13247,
      1,3,3,9,7,31,127,201,11,199,1851,23,5667,8159,20951,
      1,3,3,7,3,37,29,189,65,461,769,321,6577,16223,16865,
      1,1,5,11,1,13,91,167,33,111,1445,1047,2479,12623,22893,
      1,1,3,1,3,1,47,185,329,903,1651,3005,907,1255,8303,
      1,3,5,13,19,31,5,233,265,769,1303,2503,2229,14019,20257,
      1,3,7,3,27,11,67,195,5,661,125,3761,7211,16043,7267,
      1,1,1,3,27,13,115,25,473,417,1751,2223,2099,5913,14273,
      1,3,7,15,13,53,99,115,225,737,1621,539,4131,471,31865,
      1,1,5,5,25,19,39,207,153,569,1755,2477,3065,7383,29919,
      1,3,5,11,13,59,33,3,435,273,701,3819,7291,11803,26111,
      1,1,3,9,29,19,71,59,93,1019,887,83,4675,7541,26821,
      1,3,1,3,21,53,71,73,43,321,1581,1399,4043,12995,16825,
      1,3,7,15,3,13,37,11,93,873,1193,3481,451,15869,17879,
      1,3,1,11,31,19,101,57,129,753,853,463,6757,11083,8667,
      1,3,5,15,25,41,25,197,235,609,905,993,3233,1935,24661,
      1,3,1,5,21,7,53,107,473,77,1135,1045,4933,5615,15931,
      1,3,7,11,3,9,105,183,151,527,425,975,4073,913,2793,
      1,1,7,13,19,61,81,9,413,851,1723,1113,1453,8635,3353,
      1,3,7,15,19,53,83,31,441,343,575,935,4543,1303,12567,
      1,1,1,5,29,19,119,75,3,591,845,649,1717,13695,26905,
      1,1,7,9,5,53,127,191,15,773,1433,2899,21,4977,17839,
      1,1,5,9,21,9,99,115,397,99,725,3835,973,1219,21159,
      1,3,5,3,7,39,29,93,303,913,981,3549,5225,10907,393,
      1,3,3,11,9,25,105,101,1,867,389,2241,773,14123,10015,
      1,1,5,1,1,37,117,213,391,779,1851,1485,1277,5607,819,
      1,3,7,1,3,5,43,47,483,367,749,1693,4961,15257,3775,
      1,3,3,1,27,11,21,83,437,379,1041,393,5611,2421,31739,
      1,3,5,7,19,1,79,63,53,201,1159,2501,6327,11317,9537,
      1,3,5,13,9,37,61,217,427,913,1311,3503,5473,10583,19723,
      1,1,3,9,11,29,121,175,141,515,925,837,6011,10419,32157,
      1,3,5,9,27,57,97,175,365,367,1737,3845,1257,12243,2201,
      1,3,3,9,23,1,53,123,127,333,1335,707,5747,6541,9809,
      1,3,1,9,17,37,101,41,91,61,433,979,4345,12351,10829,
      1,3,3,13,3,21,15,49,257,99,1793,2987,5233,11625,28069,
      1,1,7,11,21,13,89,11,135,153,783,2893,6815,12007,15605,
      1,3,7,13,5,61,73,5,269,699,925,2925,5919,5841,24875,
      1,3,5,5,25,45,43,93,15,927,1253,319,1173,14559,20221,
      1,1,3,3,27,45,9,103,447,627,1239,3869,2169,49,17917,
      1,3,7,7,11,9,1,1,1,527,825,3295,623,2095,10537,
      1,3,3,11,21,11,59,165,33,743,1461,1535,6393,1301,17823,
      1,1,7,3,19,43,47,245,469,551,1447,1963,169,1481,31925,
      1,1,3,1,11,21,51,7,251,199,1153,767,6417,3417,30171,
      1,3,7,1,31,5,41,103,447,263,211,2029,8021,4705,10579,
      1,1,3,5,17,25,55,75,393,107,2017,2389,1685,14021,9161,
      1,1,1,9,13,1,75,237,205,461,689,2531,2839,13925,23351,
      1,3,7,1,23,39,33,189,157,571,239,1053,1559,1685,23059,
      1,3,3,3,27,61,71,121,49,157,1341,1707,2417,11689,26507,
      1,3,7,7,19,63,47,53,95,791,1467,1273,2045,755,8555,
      1,1,3,15,27,33,21,253,317,153,1509,1765,3809,601,5907,
      1,3,5,15,11,17,97,91,165,199,1751,2135,1315,3077,29995,
      1,3,1,5,3,33,93,49,39,743,341,2549,7603,3369,30889,
      1,1,3,13,3,5,87,63,293,785,1591,675,3915,2209,18201,
      1,3,3,11,3,15,69,231,241,127,429,2201,8173,12549,25745,
      1,1,5,11,15,39,3,29,125,685,643,1385,829,7347,28793,
      1,1,7,15,27,15,59,237,299,773,1097,3875,6503,7129,28495,
      1,3,5,13,9,17,31,227,69,443,1633,525,1659,14681,15209,
      1,3,5,5,13,51,69,173,111,433,279,2145,2091,9741,24881,
      1,3,1,7,7,35,55,51,357,99,1789,333,2073,10151,14527,
      1,3,3,7,13,41,101,87,425,701,1143,2733,6473,8667,17419,
      1,1,5,5,25,29,63,31,385,537,563,607,6723,9251,6531,
      1,3,5,5,9,63,111,131,239,723,705,2805,6579,12691,17521,
      1,3,1,7,31,55,101,225,477,271,611,3179,7859,9835,2165,
      1,1,3,3,5,15,81,127,391,333,419,1091,5997,12315,31521,
      1,3,5,15,23,7,35,109,181,295,825,419,969,15753,9365,
      1,3,5,5,25,23,69,177,325,359,1577,619,6233,11753,8103,
      1,3,5,11,31,13,79,61,241,1011,1961,949,6211,497,7099,
      1,3,5,3,25,19,67,235,337,1015,1485,355,3653,12735,14503,
      1,3,5,7,31,23,35,231,147,15,263,1995,431,5941,18931,
      1,3,3,7,1,35,37,7,85,443,715,743,2189,12537,17427,
      1,1,3,1,7,41,1,209,121,929,661,3999,955,5123,31115,
      1,1,3,5,11,43,127,125,107,293,273,2071,3003,11631,7769,
      1,1,1,13,13,29,39,217,111,779,1287,1675,4201,4869,20403,
      1,1,3,15,25,53,25,135,389,925,1971,663,7545,2673,7725,
      1,1,5,13,3,59,97,91,357,45,947,3031,8095,6269,13975,
      1,1,5,15,25,31,1,171,375,939,507,3591,1089,13605,2813,
      1,1,3,7,25,21,41,131,147,737,9,1603,1859,11573,28397,
      1,3,3,9,21,9,59,27,169,875,711,1389,2899,7937,4173,
      1,1,5,9,13,29,71,39,51,337,1067,2661,1203,5967,19249,
      1,3,7,1,17,21,43,79,181,741,1901,3445,7171,2109,1589,
      1,1,3,9,23,37,105,51,227,775,1265,2987,2197,13903,28891,
      1,1,1,13,23,47,111,41,93,261,75,2155,4301,11517,16101,
      1,1,3,3,27,27,123,125,501,775,413,1065,7607,15065,26013,
      1,3,7,3,27,11,59,87,207,743,1765,2969,913,8101,11583,
      1,3,3,1,23,7,113,17,285,993,695,2399,5019,4779,28917,
      1,3,1,5,11,51,49,139,213,435,1475,2209,6695,12981,9851,
      1,3,5,7,1,63,31,151,173,767,1453,1497,6911,9597,25551,
      1,1,7,7,21,53,39,159,389,231,309,359,7701,14819,5175,
      1,1,1,1,11,47,83,29,247,89,369,2727,3103,14421,17369,
      1,3,1,5,25,25,111,245,239,755,113,1765,3583,917,403,
      1,3,3,3,5,59,85,151,463,591,743,3767,121,2927,11031,
      1,3,5,9,11,39,77,161,275,233,1991,2683,6545,2423,32113,
      1,3,5,11,5,57,13,229,329,757,1863,3959,4243,7265,15599,
      1,1,1,1,1,23,19,67,453,593,2011,1813,4695,8903,9623,
      1,3,3,7,1,29,103,255,493,647,1709,4065,4199,949,28829,
      1,1,7,9,3,55,53,33,5,223,423,3347,7647,7211,25157,
      1,3,5,13,3,43,79,255,471,573,1007,2119,6731,10047,23179,
      1,1,1,3,7,39,55,61,53,377,435,401,3307,12621,14299,
      1,3,3,7,21,31,67,17,243,425,747,2995,1389,2557,18415,
      1,3,1,3,3,39,75,11,447,249,1135,1011,1657,10767,19501,
      1,3,1,11,17,51,117,129,17,143,785,103,5049,14703,28479,
      1,3,7,5,13,17,75,255,75,661,1175,477,1811,1479,15783,
      1,3,7,9,11,57,101,77,431,247,997,3657,5117,6815,3841,
      1,1,5,1,17,21,101,183,209,69,299,1585,6381,12983,10053,
      1,1,7,3,5,13,21,63,83,857,749,1251,5363,9629,16563,
      1,3,3,9,3,59,9,45,55,489,137,2423,2661,12111,4375,
      1,1,5,9,23,9,41,177,447,671,1631,3115,4215,14435,8743,
      1,3,7,11,19,23,15,221,413,783,1247,2343,4397,3145,32043,
      1,3,3,1,31,55,31,87,333,849,1777,343,5199,1507,11621,
      1,3,7,3,17,57,63,63,111,977,631,3019,2953,14273,29209,
      1,3,1,13,9,39,87,15,397,185,701,1487,3807,13727,19883,
      1,3,7,1,17,57,57,157,119,181,899,353,3603,15041,7421,
      1,1,7,3,29,13,29,191,105,373,961,1991,5531,6793,29497,
      1,3,3,11,7,61,65,39,215,187,191,1651,2481,3951,24965,
      1,1,7,5,25,11,105,23,257,771,1359,2837,7821,12223,28033,
      1,3,5,11,3,3,23,139,407,885,1679,2979,8149,14281,12487,
      1,3,7,3,21,45,13,85,249,1015,2023,1429,965,7091,31721,
      1,1,1,13,19,5,119,47,91,285,211,2607,4287,9197,455,
      1,3,1,1,9,59,25,137,121,287,577,3325,2365,8823,5033,
      1,3,3,13,25,63,99,43,15,855,245,3189,59,5181,21299,
      1,3,5,11,7,9,41,157,359,773,1347,2049,4589,13731,32133,
      1,1,7,11,31,37,83,105,183,375,79,1821,1989,15199,22207,
      1,1,5,3,23,37,127,9,467,651,993,69,6943,4093,20871,
      1,1,3,15,31,49,123,149,211,371,1825,3011,485,1251,17343,
      1,1,1,15,11,33,127,251,89,317,1869,219,2275,14201,27063,
      1,1,5,5,19,5,81,35,233,95,9,863,725,11095,16217,
      1,1,1,15,23,47,51,43,169,637,865,57,1509,1683,7587,
      1,3,1,3,7,7,117,187,273,303,717,3091,2083,3315,647,
      1,1,5,15,13,27,23,227,145,547,1783,987,6895,7135,11023,
      1,1,5,11,21,39,57,203,477,17,985,1729,4297,7483,13263,
      1,3,7,9,3,49,71,45,143,967,39,583,2123,5165,17437,
      1,1,1,9,21,51,71,163,441,709,397,445,6167,7753,11513,
      1,1,7,7,27,35,5,181,449,53,621,3401,5263,4557,9141,
      1,1,5,7,7,37,83,111,485,881,465,3371,5603,371,29393,
      1,3,1,15,7,47,41,245,377,823,309,3929,2159,13917,13365,
      1,3,7,7,7,29,25,141,19,611,79,2689,109,12321,8345,
      1,1,1,13,3,53,113,151,381,791,137,3185,3567,211,597,
      1,1,3,9,7,53,87,89,491,861,467,3763,2025,4187,9637,
      1,1,7,1,27,33,71,41,63,1011,741,1135,175,3739,21493,
      1,3,3,5,9,19,55,175,325,55,1193,1423,2049,9633,17515,
      1,1,3,1,27,55,69,103,401,707,825,399,6799,13199,6295,
      1,3,7,3,19,63,25,151,17,159,1673,615,6317,13261,26267,
      1,3,7,9,27,1,77,129,423,647,707,2579,3525,6723,31615,
      1,3,3,7,7,31,35,241,309,369,895,3683,4795,11319,451,
      1,3,5,7,17,7,117,141,267,713,569,1915,4369,7793,30853,
      1,3,7,1,29,61,81,73,413,13,1977,3229,5853,8451,15539,
      1,3,7,1,5,45,109,21,431,487,2019,2647,927,16015,10711,
      1,3,1,3,11,19,37,183,451,377,269,3993,3229,4899,26561,
      1,3,1,11,5,19,121,55,57,117,687,83,3047,1367,17595,
      1,3,1,7,17,31,41,219,239,963,199,2895,5599,7639,17201,
      1,3,3,5,27,53,71,183,509,771,1809,1539,2229,4893,17115,
      1,1,3,9,9,9,13,49,265,643,1929,859,497,9797,27771,
      1,3,7,11,19,39,115,139,207,903,963,1849,4403,6229,10021,
      1,3,7,13,3,57,99,223,503,423,1755,807,1885,213,18723,
      1,3,7,15,11,15,111,193,243,599,593,3385,5393,15073,17777,
      1,1,5,3,19,63,121,207,99,435,1961,2747,6405,3971,23481,
      1,3,5,13,9,29,79,131,415,49,229,1003,3263,12975,15987,
      1,1,3,7,1,41,127,155,29,73,963,659,2741,3465,2595,
      1,1,3,5,23,23,93,233,113,521,427,1557,6917,12953,22441,
      1,1,5,13,5,25,85,191,387,69,955,243,4473,9813,21711,
      1,3,3,7,1,53,95,65,231,995,539,2103,5513,14087,28655,
      1,3,5,3,17,13,19,227,197,91,1437,1121,3307,6903,3297,
      1,1,5,11,31,29,109,171,257,783,861,9,4895,1859,10909,
      1,1,7,13,5,47,61,5,363,351,1525,823,2883,12435,17629,
      1,1,5,11,9,3,69,159,371,477,1223,1973,2757,413,31223,
      1,1,3,5,23,45,43,195,423,829,1673,1563,6633,14775,21097,
      1,1,3,3,13,9,107,209,49,609,1047,3691,7483,4269,7557,
      1,1,3,15,3,43,73,161,53,813,325,3439,7009,8691,11711,
      1,1,3,3,23,45,99,61,407,15,1515,1557,953,8567,13729,
      1,1,5,9,31,35,117,57,227,923,1373,1811,3405,11979,10149,
      1,1,3,9,15,53,105,209,153,67,1477,667,3077,4911,3871,
      1,1,3,3,21,53,93,101,183,1023,3,3041,5815,9043,5801,
      1,3,3,5,17,49,127,161,321,869,1369,923,3235,711,30007,
      1,1,3,3,15,17,97,229,389,159,1075,2001,7905,15191,14693,
      1,1,5,11,5,5,121,173,95,173,1883,3915,1439,9981,24375,
      1,3,3,1,31,53,29,189,37,623,217,949,3959,7189,25427,
      1,3,5,9,21,45,101,23,355,729,797,2317,2931,7433,29175,
      1,3,7,1,1,63,63,155,237,865,1169,43,7335,6445,7979,
      1,3,7,7,11,51,37,199,503,991,319,3013,7885,12837,32419,
      1,3,7,7,27,31,101,243,37,811,1909,109,6455,7903,11821,
      1,1,3,13,23,21,89,99,243,605,1017,1871,1101,12825,8227,
      1,3,3,13,19,3,51,59,501,605,385,2189,3229,7981,31407,
      1,1,1,1,25,11,127,215,295,237,1245,3657,7803,3897,655,
      1,1,7,7,5,9,63,129,143,417,795,3409,2847,5887,3093,
      1,3,3,13,7,57,67,57,5,847,1185,3349,4841,11457,8857,
      1,1,3,3,9,53,51,43,85,437,13,2543,3651,15493,767,
      1,1,7,9,1,49,97,115,133,1011,1399,2653,7765,13999,12097,
      1,1,5,1,3,27,123,107,389,401,1759,1333,1371,5277,14865,
      1,1,5,1,13,23,3,123,137,821,399,1671,3095,3121,31387,
      1,1,5,3,7,35,57,237,509,753,1783,2815,6495,13283,7091,
      1,1,7,11,5,37,77,109,7,969,1087,3705,1695,14223,28959,
      1,3,1,11,25,5,25,163,179,185,671,1031,4537,11601,9323,
      1,1,3,7,17,25,49,221,183,619,1953,343,4523,14883,6833,
      1,3,7,5,27,19,59,153,11,807,513,3019,6875,5307,8405,
      1,1,1,13,25,41,21,109,321,135,497,1235,5177,5167,18609,
      1,1,7,5,21,53,25,197,411,503,1009,1921,4305,2633,31415,
      1,3,5,1,25,45,27,227,271,903,639,3805,657,8683,29585,
      1,1,5,3,9,49,37,35,351,491,851,2983,31,5619,6919,
      1,1,5,3,11,49,33,153,393,1017,1561,2795,4435,12589,22349,
      1,1,1,15,17,29,49,245,217,359,1133,393,3317,415,16407,
      1,1,3,5,3,9,95,63,319,319,1009,19,6453,16279,6975,
      1,1,5,9,3,25,67,95,369,237,285,2409,671,5143,121,
      1,1,3,1,9,49,35,87,317,185,445,2263,7923,10183,26615,
      1,3,3,11,9,59,29,135,129,477,353,3571,1057,16329,23523,
      1,1,1,15,13,11,19,5,133,827,1799,1893,1939,1101,12147,
      1,1,3,3,15,49,33,185,511,1013,41,3499,6235,7643,16725,
      1,1,5,11,27,45,89,157,63,137,2047,1275,4995,625,6111,
      1,3,7,11,3,1,121,1,341,33,1895,3033,3929,10257,21037,
      1,3,3,11,7,11,117,5,115,287,335,3415,5397,15065,19121,
      1,3,3,13,21,25,15,125,277,125,801,3761,2623,11333,16867,
      1,3,5,11,19,33,21,71,499,747,1515,185,1759,14623,895,
      1,3,7,1,29,35,9,203,277,299,1509,2017,2897,14175,1643,
      1,3,5,11,7,47,111,197,459,941,1619,2119,2191,11049,6811,
      1,1,5,9,7,43,103,115,87,269,1235,77,5887,1611,29041,
      1,1,5,7,1,61,83,225,179,81,1145,2403,1485,8967,20607,
      1,3,3,1,25,47,27,157,359,803,1683,1995,6445,13113,17899,
      1,3,1,7,21,37,43,119,245,49,1581,2275,3311,4087,29765,
      1,1,3,13,5,33,49,191,455,105,665,3855,3207,2671,32203,
      1,3,1,1,25,63,19,217,17,353,947,1951,4097,9041,11921,
      1,3,1,7,21,31,113,97,347,993,1799,3831,3711,6193,1235,
      1,1,1,5,3,63,11,203,425,445,1361,531,1265,1755,11685,
      1,3,1,7,13,29,23,85,57,467,1835,133,7961,4175,2445,
      1,1,1,15,23,27,37,5,123,913,1293,1633,3113,5413,26407,
      1,1,5,13,27,1,121,151,303,931,375,3679,1863,12301,30907,
      1,3,1,9,31,9,49,203,177,937,1503,933,5867,12533,13621,
      1,3,3,15,1,41,23,191,191,931,837,3553,2611,4735,18105,
      1,1,5,7,27,49,51,111,435,195,1229,711,7145,14571,31707,
      1,1,7,7,3,41,59,203,291,903,1727,2757,1463,6287,31535,
      1,1,7,13,23,5,75,3,207,525,411,2133,2231,477,7155,
      1,3,5,7,13,19,111,225,489,83,1177,4077,4617,14413,7133,
      1,3,1,7,9,59,3,113,379,803,1289,3347,4127,6669,14867,
      1,3,7,3,31,37,87,79,399,749,995,1611,3137,12543,31955,
      1,1,5,7,21,59,49,45,511,639,1033,2169,3265,15001,10745,
      1,1,5,1,25,19,23,203,11,883,1031,4087,5059,11321,21675,
      1,3,7,5,11,27,33,205,163,289,501,3505,1515,1895,15889,
      1,3,1,1,23,7,39,239,29,119,1499,2071,6495,12107,5339,
      1,3,1,1,23,29,55,181,327,905,427,1033,427,3687,5367,
      1,3,3,7,21,27,115,127,393,855,1291,2121,381,9995,29757,
      1,3,5,1,25,13,15,183,269,1005,1531,3451,3975,9479,23695,
      1,3,7,7,19,31,111,97,33,179,1343,2069,977,5043,9129,
      1,3,1,5,17,57,99,129,379,829,837,1845,3613,7351,19291,
      1,3,3,5,31,23,119,229,135,389,9,705,6697,15441,5303,
      1,1,1,11,25,31,105,95,5,931,789,375,7543,9957,28627,
      1,1,7,15,21,17,19,103,389,545,1725,2867,4251,3829,6907,
      1,3,7,7,15,37,97,65,337,409,1649,2869,7929,8905,21989,
      1,3,5,3,11,15,69,29,353,207,233,411,2047,10303,31655,
      1,3,3,7,27,43,125,107,69,981,215,1955,3589,597,12703,
      1,1,7,9,25,13,109,73,227,663,1115,285,471,3359,15787,
      1,3,7,5,1,45,7,79,441,149,701,1457,6595,14829,20865,
      1,3,7,15,15,47,83,239,295,23,1085,813,1209,3573,2855,
      1,1,3,15,13,7,59,67,255,537,1841,3857,6821,15175,13997,
      1,3,1,1,9,57,59,21,21,41,1693,2805,7953,1389,14105,
      1,3,5,15,19,49,107,117,99,607,145,53,1863,9383,12029,
      1,3,3,13,1,39,5,141,503,265,281,1785,2673,6597,6333,
      1,1,5,3,3,19,3,181,169,269,955,2399,3157,11053,8563,
      1,3,3,13,11,1,95,43,179,507,443,209,3239,14239,21829,
      1,1,7,9,3,17,99,179,445,479,1897,1507,5753,4757,2135,
      1,3,3,1,9,51,29,13,295,291,927,85,5707,7447,32319,
      1,1,1,3,13,11,21,157,213,327,1071,591,2639,15405,6617,
      1,3,5,1,7,25,55,47,495,681,727,2707,2955,705,7489,
      1,1,3,9,17,3,73,67,465,367,1473,3195,7825,5299,1817,
      1,1,1,1,19,31,77,253,71,599,1601,871,2243,6699,13013,
      1,1,7,9,21,1,71,115,5,65,767,925,7901,10761,19431,
      1,3,1,7,23,31,31,15,105,391,585,2995,2635,10607,24951,
      1,3,3,1,19,25,71,211,41,197,787,225,6781,813,10117,
      1,3,3,3,17,29,3,153,231,643,1151,447,3699,9625,26677,
      1,1,5,9,1,25,71,21,395,297,557,3841,233,1877,4569,
      1,1,3,13,1,45,115,61,5,937,173,2109,2927,9599,9155,
      1,1,3,3,15,21,61,121,253,285,1083,3545,5537,6773,2629,
      1,3,3,15,13,63,33,77,49,849,1795,2771,5481,9833,603,
      1,1,7,5,1,39,113,237,225,1005,1687,2297,3213,2605,14669,
      1,1,3,1,11,1,39,23,67,441,1235,2545,3139,15901,29243,
      1,3,1,3,15,49,39,57,311,345,525,223,4923,6311,25275,
      1,1,5,7,9,13,69,11,349,423,1773,1055,1001,9359,17025,
      1,1,1,13,15,63,89,207,335,591,1223,2701,55,12471,13127,
      1,1,3,5,15,19,83,67,407,113,1961,779,5803,12417,21751,
      1,3,3,1,21,53,81,95,405,427,1047,2443,4153,5843,22511,
      1,1,7,7,7,25,115,155,453,537,741,2379,2343,16035,19587,
      1,3,3,11,27,21,111,121,503,437,803,3399,5303,10163,18199,
      1,1,5,13,19,27,7,81,259,545,965,743,4533,8813,21253,
      1,1,5,5,1,59,37,11,105,343,75,1319,6317,9593,1699,
      1,3,1,9,13,9,115,131,387,1023,253,693,5191,12777,10565,
      1,3,1,15,7,35,111,195,287,305,533,1901,3363,10085,30791,
      1,1,3,9,27,51,21,77,413,925,717,791,4147,585,5649,
      1,3,3,5,25,59,79,249,185,567,71,1997,7373,2327,18637,
      1,3,3,11,15,21,97,99,391,57,1973,29,7451,2529,25737,
      1,3,7,5,7,59,93,5,287,469,1639,3637,5465,14431,32265,
      1,1,3,11,3,1,71,75,427,299,811,3697,3529,5433,26957,
      1,3,1,9,19,59,37,255,165,1005,19,2851,4309,455,9485,
      1,1,1,5,1,55,15,233,133,47,1831,713,2601,1017,3201,
      1,1,5,5,21,55,127,69,377,41,25,2295,7595,4733,11615,
      1,1,5,3,23,5,7,181,161,775,1095,2271,6637,14489,6873,
      1,3,5,9,9,15,5,133,357,21,127,2685,6299,4363,17573,
      1,3,3,9,13,39,51,223,201,401,1839,2461,7633,6039,10445,
      1,1,5,1,9,21,19,249,227,359,255,2895,4117,2073,27687,
      1,1,5,15,5,61,113,161,95,3,877,2775,293,6655,4023,
      1,3,7,1,7,55,73,39,295,403,985,2315,1667,13525,1453,
      1,1,5,1,27,1,85,195,11,713,1841,3895,3131,2193,17607,
      1,3,5,13,25,1,119,97,239,167,1393,1753,6989,12155,12509,
      1,1,7,15,31,21,41,255,425,445,165,2097,5627,4971,13207,
      1,1,1,15,13,33,81,105,453,197,13,1547,7381,8709,15103,
      1,1,3,11,11,33,107,123,483,367,121,995,1911,8205,22577,
      1,1,1,9,9,43,71,49,273,431,1705,3313,4259,16291,14345,
      1,1,1,7,3,1,43,213,97,547,1559,1149,2791,3751,887,
      1,1,3,15,25,47,49,251,425,35,295,3767,6305,9633,5045,
      1,3,3,1,5,55,91,245,27,981,331,555,6553,11017,15289,
      1,1,3,7,1,23,23,155,223,565,1005,3211,3847,7479,3643,
      1,1,5,1,17,7,47,95,35,779,1685,2099,7505,15425,18089,
      1,3,3,7,3,63,83,151,211,147,611,1171,1681,7687,13423,
      1,3,3,1,3,27,107,117,497,537,195,3075,2753,1665,19399,
      1,1,1,7,23,5,103,209,117,845,1243,1283,4253,9723,20937,
      1,3,1,1,5,49,7,13,419,125,287,1599,8161,1275,24661,
      1,3,3,3,13,63,23,183,39,979,1301,2349,905,15805,30151,
      1,1,3,9,17,11,97,189,189,511,1779,2077,6891,11623,23949,
      1,1,7,11,13,45,15,37,11,853,915,1569,6103,10633,3137,
      1,3,3,5,15,61,91,255,131,821,1755,1501,2663,1747,941,
      1,1,3,7,19,19,65,95,499,239,2023,3185,4649,3861,3767,
      1,3,5,15,15,63,55,93,127,303,171,1763,4991,9479,9917,
      1,3,7,5,31,53,111,35,433,163,1903,3991,3585,643,21941,
      1,3,1,9,27,39,67,89,487,349,587,1723,4311,11321,25785,
      1,3,5,7,1,63,23,237,507,689,1341,441,1721,843,20335,
      1,1,3,3,31,63,83,103,25,799,1379,1817,3809,12285,16673,
      1,1,5,3,25,29,99,193,21,549,33,3109,4135,10071,32355,
      1,3,1,7,13,27,83,189,121,167,379,1503,7955,13189,313,
      1,3,5,15,25,19,83,87,257,237,709,1169,1561,7117,4785,
      1,1,1,7,9,55,21,5,439,367,403,2311,6243,8349,13127,
      1,3,7,3,5,35,51,67,453,767,29,3293,6665,11459,2799,
      1,3,3,3,5,19,59,7,367,683,783,1317,7119,6129,19525,
      1,1,5,5,5,19,61,67,381,291,875,2179,2481,9325,11253,
      1,3,5,5,7,47,107,9,141,667,1989,821,3909,1733,10187,
      1,1,7,7,31,61,1,71,477,689,1539,3617,8105,6535,3293,
      1,1,5,5,23,9,103,197,241,249,297,3607,6217,1673,30103,
      1,3,1,5,23,15,115,105,365,51,825,2687,359,16325,15083,
      1,1,3,11,29,45,65,251,169,189,1243,2345,1345,14471,25631,
      1,1,5,9,7,63,81,167,309,539,1169,3949,4193,12047,1491,
      1,3,1,9,29,33,89,167,67,73,1885,477,5745,13365,6819,
      1,3,7,9,9,49,95,13,157,997,1725,935,7543,6349,18277,
      1,1,5,5,11,59,97,17,303,469,93,2761,7395,9021,24299,
      1,1,7,3,27,63,71,99,407,139,711,2589,4715,5405,3277,
      1,3,7,3,11,15,49,57,271,493,1165,2839,8191,2609,14759,
      1,1,1,7,21,15,71,245,413,473,1321,1165,1027,6983,12867,
      1,1,5,3,15,21,19,197,401,627,2047,2761,5807,5751,28025,
      1,1,3,3,5,57,19,209,341,165,489,455,231,14385,12457,
      1,3,3,11,13,63,79,129,17,315,1881,1069,177,12013,29567,
      1,1,3,7,31,29,51,235,475,375,617,437,6379,8505,23079,
      1,1,3,7,27,3,3,137,203,959,363,371,2899,13491,22979,
      1,3,3,3,9,1,57,7,363,537,713,2417,509,7747,22135,
      1,3,3,3,13,21,79,121,487,921,113,281,2853,14855,19747,
      1,1,1,11,3,53,89,123,307,585,567,1925,505,15935,20419,
      1,1,3,3,15,45,77,197,499,683,1405,3573,981,14135,19763,
      1,1,1,11,27,31,61,191,29,601,373,2011,6193,3599,4387,
      1,3,5,9,7,13,1,193,469,603,1315,3329,3761,8355,10425,
      1,1,3,9,29,61,103,17,117,251,2029,2963,3763,16117,6627,
      1,3,1,3,7,51,91,145,497,657,871,3707,5905,10449,14901,
      1,1,3,1,3,53,23,149,461,333,1809,1315,1815,8223,13297,
      1,1,1,7,15,31,3,47,443,829,1305,893,4191,9681,32661,
      1,3,1,3,27,43,51,221,295,825,649,2953,6203,8237,20253,
      1,3,1,3,9,35,41,195,249,225,387,3789,1499,2559,28413,
      1,1,5,15,19,29,13,115,333,787,787,723,2987,6227,10865,
      1,3,5,13,5,59,5,251,79,387,11,3167,6619,13317,18979,
      1,1,7,11,31,51,43,1,189,519,1945,2129,4365,14059,3139,
      1,1,7,5,31,9,43,19,151,533,1061,3849,6871,6941,14935,
      1,3,7,5,19,57,7,129,25,353,17,1739,6513,399,28835,
      1,3,5,15,25,15,37,125,39,239,271,65,2189,10449,11815,
      1,3,7,15,19,57,47,245,509,945,385,3987,3585,14711,9655,
      1,1,3,13,21,31,13,81,9,489,1321,63,1363,2219,19541,
      1,1,5,7,3,57,25,147,23,553,889,307,6429,15807,12861,
      1,1,3,15,29,21,99,237,151,881,675,3625,1159,11759,21347,
      1,1,7,1,9,13,111,239,235,609,1569,3271,2837,13807,7301,
      1,3,1,15,7,59,27,81,129,9,647,3595,1877,1067,1859,
      1,3,7,1,3,25,119,57,145,441,1045,789,215,1265,9369,
      1,3,7,3,17,25,87,211,441,229,223,2795,7241,7007,20575,
      1,1,3,1,13,1,55,227,389,141,1097,2487,7603,4161,5025,
      1,1,3,5,15,29,29,145,233,209,891,89,8097,2897,26685,
      1,1,3,1,29,53,19,95,161,359,435,3313,4955,7965,21015,
      1,3,5,9,19,3,109,77,29,937,1663,125,2453,1069,20639,
      1,3,7,13,5,23,43,231,347,591,1963,2491,4045,16029,8149,
      1,1,5,1,13,3,75,211,419,929,901,3453,8121,799,8897,
      1,1,7,15,11,11,123,111,309,415,1071,975,2009,12945,19617,
      1,1,1,7,31,35,81,255,89,643,451,513,497,11751,24215,
      1,3,5,5,25,17,5,165,139,929,1927,1353,7427,9719,17087,
      1,3,5,1,21,55,79,85,333,847,1305,851,5057,8361,18269,
      1,3,7,15,27,17,55,125,395,223,271,781,1639,10569,11143,
      1,1,7,9,7,33,127,85,209,339,483,241,2523,14951,6855,
      1,1,3,9,5,19,9,183,435,343,1105,3139,7617,1311,267,
      1,1,5,1,15,53,11,63,113,241,855,3123,4777,3495,23345,
      1,3,1,5,19,29,119,205,167,683,289,1629,4977,8981,6867,
      1,3,1,1,31,63,95,159,267,231,863,3385,5315,7267,13757,
      1,3,5,11,19,21,53,41,125,179,533,1279,3759,7073,13905,
      1,3,5,9,17,7,27,67,97,809,1423,2743,2859,16121,329,
      1,3,1,15,1,41,59,155,509,51,1827,3739,3879,13369,30821,
      1,3,3,7,21,31,7,13,347,919,1225,497,5051,3769,20211,
      1,3,7,13,31,9,127,195,123,387,3,3593,6623,9827,29319,
      1,1,3,9,7,27,95,211,287,189,1683,1999,7641,14983,4699,
      1,1,5,3,7,21,29,189,101,423,885,3275,6569,11023,22265,
      1,3,5,3,9,33,79,75,327,975,287,3025,2157,7301,24447,
      1,3,3,15,31,27,63,1,71,119,1151,517,6131,11055,179,
      1,3,7,11,23,15,101,247,349,735,673,997,6451,229,32103,
      1,3,5,15,7,1,51,135,207,741,1831,1235,4747,11915,22009,
      1,3,1,13,9,31,19,221,465,681,627,2595,5617,14201,30355,
      1,1,3,1,13,49,55,155,11,885,1275,3591,2217,6659,30885,
      1,1,7,11,27,57,93,95,243,63,1405,2049,7689,15943,18503,
      1,1,7,7,5,11,47,189,467,631,1665,2717,4285,2087,1435,
      1,1,3,11,7,27,127,3,231,757,435,2545,3537,9127,19915,
      1,1,5,13,5,29,85,127,339,875,497,1573,6553,11983,18029,
      1,3,1,1,21,3,15,91,231,683,1529,2651,4147,13437,23861,
      1,3,1,7,27,17,19,179,243,223,1037,1501,5935,2259,25185,
      1,1,3,15,11,19,127,27,483,219,583,2555,531,3451,17875,
      1,1,1,13,31,39,89,149,363,741,1355,4067,3171,6783,1799,
      1,1,3,11,25,51,45,235,379,123,1701,725,1991,7471,9833,
      1,1,5,13,15,47,13,201,263,57,375,2963,7475,15929,13775,
      1,1,3,1,29,29,11,161,345,253,97,255,7267,2379,3933,
      1,3,1,15,3,47,11,69,347,747,795,2401,3367,2383,6125,
      1,1,7,3,1,49,101,47,71,761,1503,2619,191,8895,873,
      1,3,3,5,25,41,93,85,427,109,1675,2409,4317,9233,30283,
      1,1,3,9,11,3,67,159,425,751,887,1415,403,15977,10739,
      1,1,5,13,9,1,9,103,481,601,931,1957,5763,7095,27141,
      1,1,3,15,29,13,43,33,297,269,1041,1411,3461,12043,10045,
      1,3,5,3,3,3,5,7,185,753,133,1561,5595,13777,25795,
      1,3,5,5,1,19,29,145,163,149,619,2603,7757,10035,10189,
      1,3,7,15,27,15,111,173,135,117,157,2601,7919,12111,22795,
      1,3,1,1,29,27,65,31,101,715,289,3643,2335,6789,23397,
      1,3,1,3,11,45,71,109,321,423,1695,169,3075,12423,11391,
      1,1,3,9,13,51,35,121,203,279,433,2725,7951,2105,27333,
      1,1,1,15,23,31,25,105,501,441,1511,3133,2811,10595,21779,
      1,1,5,13,7,1,97,193,121,993,1347,1903,1883,6583,24535,
      1,1,7,9,7,29,17,41,101,447,1289,387,1891,2723,26091,
      1,1,3,3,3,53,81,81,177,165,195,3413,8177,3817,8453,
      1,3,7,15,15,31,23,31,337,439,1773,63,5351,5491,1767,
      1,3,1,11,5,15,23,75,437,553,429,2705,3625,13851,19865,
      1,3,3,9,13,15,33,235,215,415,1737,1409,2101,14623,14717,
      1,3,7,7,13,51,101,217,175,813,1639,4009,1625,4991,17525,
      1,1,5,13,23,33,29,175,39,673,557,3239,5129,11049,27227,
      1,3,7,13,1,37,33,139,493,891,1883,2525,5741,15795,5875,
      1,3,1,15,15,27,127,111,147,363,725,3077,4341,9131,24635,
      1,1,7,3,17,25,59,135,177,635,73,3455,3083,6009,13033,
      1,1,1,5,15,53,93,161,215,459,1087,179,2235,8885,15309,
      1,1,7,13,7,17,75,173,449,855,103,2739,3421,11811,18805,
      1,1,7,9,5,11,53,75,247,249,1201,953,2455,4589,6027,
      1,1,5,13,27,51,119,39,137,11,1435,3773,3889,6081,11829,
      1,1,5,5,5,35,1,197,501,185,1039,1563,6421,14373,25655,
      1,1,3,13,31,55,115,183,483,655,1351,3203,725,3299,22579,
      1,3,5,11,31,31,83,59,395,21,1881,2821,2251,11781,26265,
      1,3,7,13,21,19,103,21,403,443,1951,55,985,15983,15087,
      1,1,5,15,29,11,51,53,255,183,1475,1491,259,387,10303,
      1,3,5,7,21,37,45,39,479,637,1325,3753,3319,7403,31759,
      1,1,3,5,7,43,89,53,269,187,995,141,119,8139,29699,
      1,1,1,5,1,53,3,23,379,223,1889,4035,1437,12425,9051,
      1,3,1,13,3,31,61,43,249,449,901,1921,3495,8599,5263,
      1,1,3,5,3,25,35,133,25,597,915,3663,5147,11831,24269,
      1,1,1,9,21,27,93,93,217,299,1881,3647,4825,7989,24121,
      1,3,1,15,5,15,49,129,315,631,2037,1567,4043,15589,30905,
      1,3,3,7,25,5,123,51,47,471,1563,3947,7975,3681,9611,
      1,3,7,15,1,17,73,245,465,95,95,1159,1319,4675,8841,
      1,1,3,15,5,51,35,71,423,651,753,173,2131,15799,29601,
      1,1,1,1,3,53,83,187,445,827,1549,979,5363,1701,2149,
      1,1,7,9,3,15,65,161,37,233,771,3749,727,6857,17175,
      1,1,7,7,27,29,107,247,249,353,773,3677,7273,5419,29397,
      1,3,3,7,31,49,87,159,145,497,1715,2115,5035,6431,7245,
      1,3,3,5,7,31,51,117,101,617,557,2551,6589,13295,31975,
      1,1,3,3,15,27,125,163,169,893,1771,25,5787,10267,10297,
      1,1,1,5,9,47,85,65,289,783,1105,4035,4111,2589,24575,
      1,3,3,13,23,33,7,49,301,531,1713,2755,5543,8153,24099,
      1,1,5,9,7,39,101,67,417,923,757,1537,5553,12233,20881,
      1,1,5,1,19,7,25,123,125,183,573,3317,6867,871,17631,
      1,1,3,15,19,13,117,41,129,715,1525,2257,2179,10807,23271,
      1,3,1,5,25,53,19,169,289,569,1135,1967,7001,15883,15113,
      1,3,7,15,7,37,127,147,415,313,1541,1889,3763,16199,12681,
      1,1,3,9,1,35,95,137,237,951,899,3177,6073,10655,31687,
      1,1,5,5,29,57,45,253,297,529,1553,467,8035,15675,21691,
      1,3,7,15,25,41,59,81,87,985,1001,2369,661,7551,11829,
      1,1,7,9,27,21,7,233,309,67,701,2737,4261,2467,15691,
      1,3,7,1,19,55,47,155,333,101,517,1991,4619,10435,27241,
      1,1,7,3,23,35,7,125,157,537,933,3281,4975,8969,27581,
      1,1,3,7,19,53,81,103,461,435,777,335,5261,12249,9695,
      1,3,1,7,19,9,75,245,355,37,1855,1339,3107,7251,16543,
      1,1,1,3,5,35,39,223,113,423,1423,713,6113,349,24147,
      1,3,1,1,15,31,11,75,499,345,1253,2629,2551,7483,25395,
      1,1,3,11,25,25,3,211,185,45,1865,1805,3303,11091,529,
      1,3,1,1,9,21,7,165,107,641,1083,2805,2099,5855,18477,
      1,3,5,3,9,21,77,103,505,277,335,797,3869,2957,1979,
      1,3,5,15,31,23,77,247,303,891,1261,3233,3495,13111,13185,
      1,3,5,11,11,35,49,229,149,931,881,775,2949,3141,29157,
      1,1,3,5,19,57,23,95,347,221,195,3561,1481,2063,3979,
      1,3,5,3,13,1,23,173,431,29,421,3235,2751,4447,28283,
      1,1,5,13,23,3,1,9,327,855,1251,2997,6129,4223,11555,
      1,3,7,13,29,21,37,229,217,353,1239,3955,491,12183,14777,
      1,1,5,5,1,33,103,187,183,939,1873,2633,6143,15405,17353,
      1,1,1,9,21,27,71,129,499,279,1181,4053,2485,1961,30603,
      1,1,3,15,21,37,45,201,221,187,727,1241,6171,1383,22277,
      1,3,7,5,21,17,67,177,323,601,633,865,6131,10329,8689,
      1,3,5,9,15,45,71,43,359,651,103,403,3249,11769,6567,
      1,3,3,13,3,23,101,145,367,999,1489,3673,2959,10855,16029,
      1,3,7,3,13,43,123,87,55,1015,141,2917,6567,16025,25555,
      1,3,1,3,17,7,21,161,41,889,1315,1897,639,15451,3049,
      1,3,5,15,27,33,55,17,81,431,325,909,3547,10121,17815,
      1,1,3,1,15,37,43,137,203,191,1129,1585,435,3177,769,
      1,3,7,11,21,23,125,41,17,951,465,3691,3465,13247,13779,
      1,3,3,1,31,23,43,101,405,739,1061,2955,5643,16137,8763,
      1,1,5,1,19,33,99,109,203,65,395,2775,1373,2557,5875,
      1,3,3,3,27,51,79,63,331,365,1071,1661,4549,8561,1719,
      1,3,3,9,3,17,53,161,141,489,1325,1709,1381,5093,171,
      1,1,7,15,9,3,95,237,197,949,7,1837,729,10111,6637,
      1,1,3,3,19,31,57,173,483,861,1001,1919,3389,11777,20693,
      1,3,1,9,27,13,113,177,75,925,949,119,4759,7775,23033,
      1,1,7,15,23,15,65,61,137,653,1843,323,379,15157,29885,
      1,3,3,7,29,3,11,205,347,745,1477,3929,5749,4735,29435,
      1,3,5,9,1,11,111,15,7,69,45,3607,1099,9203,21301,
      1,3,3,3,23,3,83,173,73,485,681,1867,3839,11823,13339,
      1,1,3,11,31,43,107,127,465,389,1595,427,1571,5885,29569,
      1,1,7,9,27,25,117,27,287,391,279,3247,35,12973,5483,
      1,3,7,11,19,55,45,127,245,945,305,3907,2455,3163,31,
      1,1,7,11,15,17,65,15,37,207,1447,3027,2281,6557,16717,
      1,1,1,13,5,27,33,213,29,603,1171,3235,2255,2017,30999,
      1,3,1,5,11,1,73,233,69,125,397,297,3337,6191,31055,
      1,1,1,15,1,1,65,145,201,917,1891,2999,4069,10413,15819,
      1,3,5,13,15,51,115,167,311,375,1069,2595,3337,753,11903,
      1,1,3,1,1,23,69,125,147,915,1945,411,979,13863,30443,
      1,3,1,11,5,1,93,23,135,93,1689,23,3519,4491,24673,
      1,1,7,3,11,59,93,153,487,475,1191,1455,5963,8259,18811,
      1,1,3,1,13,15,55,71,433,33,491,1835,5695,10509,347,
      1,1,1,15,19,1,23,47,235,101,1057,901,5477,7079,30885,
      1,1,5,13,11,43,119,77,441,121,783,827,1757,12751,31593,
      1,3,7,11,19,17,37,225,329,231,515,1541,7371,6355,10905,
      1,1,5,13,7,11,35,215,345,577,147,2803,3291,4631,5329,
      1,1,3,9,21,55,113,251,25,221,1445,3385,1589,4109,29897,
      1,1,5,7,9,45,5,33,331,285,1101,3131,2713,5653,3823,
      1,3,7,7,5,39,43,167,481,629,777,1827,4653,403,4781,
      1,3,3,7,31,33,31,159,313,673,1425,663,5819,1297,26627,
      1,3,3,1,19,61,117,93,373,491,1031,757,4185,771,7265,
      1,1,7,9,3,45,65,223,437,41,1139,2733,5963,2709,25429,
      1,3,5,11,21,27,31,127,255,761,1865,1319,6583,9235,10717,
      1,1,1,5,21,1,63,43,413,557,567,2893,8017,2307,29525,
      1,1,7,3,31,1,15,235,215,395,1971,469,5275,431,5349,
      1,1,1,13,25,59,71,245,389,279,1293,89,6551,10285,14495,
      1,1,5,5,9,63,17,229,425,939,877,3689,7229,6707,30771,
      1,3,7,7,11,29,43,41,25,237,1585,3735,2617,7541,26243,
      1,1,7,9,21,5,69,231,361,39,1695,3043,2973,5487,12857,
      1,1,5,3,17,63,91,217,407,133,1373,4021,1737,10043,4561,
      1,3,7,9,31,13,101,231,175,457,89,2167,2725,8505,375,
      1,1,3,15,31,11,27,211,347,291,1881,3091,3307,5117,13341,
      1,3,5,5,13,25,5,197,237,135,635,1175,5085,14737,10807,
      1,3,3,9,7,63,107,127,147,477,1813,2619,8089,2651,26549,
      1,1,5,11,15,45,27,133,45,621,707,2679,5929,19,9599,
      1,3,7,9,21,37,41,255,69,1009,1999,367,6177,10017,3549,
      1,1,1,15,19,55,73,189,423,983,1811,2551,4765,12077,18205,
      1,1,5,7,17,13,25,225,463,471,631,1811,5797,3235,32253,
      1,3,7,1,29,7,123,187,331,735,1757,1115,2077,15725,2183,
      1,3,7,9,17,61,111,93,21,1003,1905,3719,2111,11845,6427,
      1,3,7,7,17,21,51,59,115,723,2039,2833,5969,5737,18407,
      1,3,3,13,9,47,95,233,13,281,1049,619,405,16205,20097,
      1,3,7,13,9,41,11,171,453,713,587,1669,2489,10277,18599,
      1,3,3,13,21,41,123,173,511,399,859,1515,5773,12535,26289,
      1,1,7,15,11,3,113,111,73,7,1191,2573,7713,465,27615,
      1,1,7,15,5,5,39,11,489,13,1041,1639,7879,11899,6899,
      1,1,5,9,27,31,15,237,401,795,1675,2361,7333,12507,14627,
      1,3,1,7,21,53,31,81,189,683,1283,419,7585,9207,15053,
      1,3,5,11,21,1,49,251,403,489,1235,429,4855,4081,17575,
      1,3,1,15,29,33,77,53,105,731,749,2677,3967,7967,18723,
      1,3,3,11,9,47,11,95,137,923,599,1585,1969,9625,19171,
      1,1,1,5,7,7,85,49,339,883,261,2125,3831,9797,16395,
      1,3,3,3,5,9,33,99,75,889,101,2099,6635,11511,21573,
      1,1,5,11,1,11,79,49,7,131,471,1235,3287,14777,12053,
      1,1,5,15,9,9,83,15,21,899,1785,2349,3471,6723,1405,
      1,3,5,11,1,7,121,223,509,859,1037,491,5529,481,17029,
      1,1,7,5,17,35,91,171,113,65,423,2371,5105,12827,31087,
      1,1,3,3,21,47,55,11,159,51,263,2113,661,9147,28929,
      1,1,1,9,19,7,43,223,207,787,543,2141,4247,7369,29031,
      1,1,7,11,11,51,121,9,211,905,687,889,1827,13395,3507,
      1,3,1,7,15,23,5,139,469,569,33,3477,5391,13665,17011,
      1,1,1,15,29,29,29,201,63,1019,97,1671,9,4617,19833,
      1,1,5,15,25,5,67,225,189,919,1471,1451,5017,16189,31555,
      1,3,5,5,15,51,89,221,149,863,43,2381,1767,8037,5319,
      1,3,3,1,15,17,5,77,69,27,1883,63,5987,1497,3723,
      1,3,7,11,7,5,113,229,123,709,1531,641,6655,14923,22947,
      1,3,1,13,21,15,45,175,81,499,1113,587,7573,11689,15651,
      1,3,1,1,29,43,101,37,131,757,465,743,2737,8063,23967,
      1,1,7,13,9,21,39,177,51,691,2047,1519,6137,5271,8703,
      1,1,3,3,5,55,63,21,3,317,461,527,2673,16211,6721,
      1,3,5,5,5,47,7,241,387,589,323,203,7241,14761,13287,
      1,3,5,3,23,63,55,61,231,1023,1315,1181,243,7389,25639,
      1,1,7,13,31,43,41,81,127,887,1513,4055,1361,2443,6963,
      1,1,1,5,7,43,43,33,323,911,1373,3053,6503,513,6457,
      1,1,7,11,25,61,21,149,205,349,1433,1587,149,7275,5465,
      1,3,5,5,11,9,31,217,119,511,209,3325,2023,2877,463,
      1,3,5,15,21,47,89,41,347,849,1375,3311,807,11443,27643,
      1,1,5,7,29,43,123,191,321,373,447,2145,1221,2071,12689,
      1,3,5,15,1,21,43,141,461,779,1109,2915,909,8585,19859,
      1,3,3,11,5,17,57,13,393,661,1761,2455,43,8593,20505,
      1,3,5,1,31,47,65,249,77,513,851,2381,3447,693,7729,
      1,3,5,15,31,19,83,47,369,697,1815,819,7573,9245,8013,
      1,3,5,5,11,25,27,151,107,339,299,3869,3393,5661,2947,
      1,1,3,1,1,59,85,57,175,465,239,3115,7157,7035,11463,
      1,1,7,5,31,41,53,149,121,743,189,159,5289,2945,1179,
      1,3,3,15,23,51,83,25,159,163,61,713,4529,5253,1603,
      1,3,5,11,7,29,15,177,507,695,1305,1863,7525,3063,27433,
      1,1,3,11,5,41,115,227,409,951,591,4003,7717,4369,15637,
      1,1,7,11,23,55,71,135,51,411,2003,2375,6823,1711,4443,
      1,3,1,3,31,47,31,233,243,3,313,1649,6955,13679,32327,
      1,1,3,11,29,9,1,79,247,677,685,3107,5987,9675,29523,
      1,1,1,7,25,31,39,241,483,839,1143,437,2317,2437,173,
      1,1,5,1,17,19,83,57,39,479,715,1911,1091,10937,22145,
      1,1,7,1,27,45,35,55,477,719,217,3349,7717,6853,9699,
      1,3,1,11,9,39,25,223,303,713,151,2611,4629,5855,31729,
      1,1,1,11,13,35,53,39,167,779,1673,1221,6281,15113,32027,
      1,1,5,9,19,63,89,113,199,107,1015,835,2879,9499,25597,
      1,1,7,3,19,37,15,23,449,641,1811,3407,6775,6283,31157,
      1,1,3,1,19,15,31,99,511,897,1693,2093,955,15897,26693,
      1,1,5,1,5,15,47,19,441,541,1621,3877,6407,15991,1931,
      1,3,5,9,21,61,15,77,265,351,879,3835,6555,2349,23235,
      1,1,5,11,25,37,29,181,341,641,1213,1319,6359,6231,32573,
      1,1,1,7,1,37,87,123,33,913,111,2613,4895,12595,26633,
      1,3,5,3,27,11,45,89,183,241,1355,783,3343,239,8643,
      1,3,7,7,9,35,67,187,233,577,1445,3063,6039,16233,1453,
      1,1,3,13,27,11,23,15,95,63,1931,911,8149,6833,3051,
      1,3,3,5,29,49,125,117,47,143,423,3215,3605,3677,17155,
      1,3,1,1,31,1,123,195,83,893,1947,339,2927,7183,15443,
      1,1,7,13,31,15,91,207,39,275,439,2617,3093,11041,24997,
      1,1,5,3,3,41,13,67,361,497,25,3807,3551,9681,21043,
      1,3,3,3,11,27,103,59,427,327,1705,29,8127,1641,20847,
      1,3,7,5,3,37,81,137,225,101,187,3067,2491,12687,16227,
      1,3,5,15,15,33,69,223,225,771,1917,2293,2889,12083,23971,
      1,1,3,5,11,9,121,81,203,425,1189,2011,3041,3247,739,
      1,3,1,1,13,9,39,169,437,571,1481,3355,3895,8975,31031,
      1,3,1,11,1,43,35,35,293,11,657,1415,5021,14463,17945,
      1,1,5,5,13,47,91,15,159,23,971,3575,757,13477,31757,
      1,1,7,1,5,63,69,27,71,129,123,3767,89,7865,1189,
      1,3,3,5,23,1,83,3,487,491,217,2583,3889,15009,9227,
      1,3,5,15,25,1,73,107,245,191,1449,571,1403,6953,17457,
      1,3,3,3,27,19,25,105,207,857,1161,3657,2107,7955,517,
      1,3,3,9,21,29,5,103,219,35,3,1635,4815,15797,29839,
      1,1,7,7,3,63,75,77,13,57,603,2333,7761,14397,10875,
      1,3,7,13,3,11,5,255,1,947,1695,1927,7447,7407,20797,
      1,1,5,1,1,21,105,73,429,973,1801,3943,6161,1309,3359,
      1,1,3,15,27,9,9,129,117,545,9,1983,6351,10925,27337,
      1,3,3,5,5,5,13,155,503,875,1243,2259,3445,11953,6517,
      1,1,7,3,29,21,121,147,413,423,1887,2429,2765,16335,3071,
      1,1,7,9,5,53,41,137,463,583,1627,1731,6675,3703,8177,
      1,3,5,11,31,29,67,159,425,25,1457,139,5019,701,7357,
      1,3,1,5,25,15,123,123,245,859,249,2175,2137,5765,4873,
      1,1,3,5,23,1,111,111,111,469,1473,1777,3579,13503,2569,
      1,1,7,3,17,23,51,23,499,135,713,3317,807,9589,11349,
      1,1,1,15,9,51,75,159,359,773,1521,2913,5901,3047,14649,
      1,1,3,1,13,61,117,195,49,267,57,1769,3621,9415,29443,
      1,3,7,11,3,25,33,31,315,191,359,3399,2481,13831,20205,
      1,3,3,5,31,43,35,125,291,51,1469,3857,1707,2641,32137,
      1,3,5,1,25,11,113,137,211,159,1667,939,6439,5337,32059,
      1,3,3,11,31,61,99,49,383,343,395,51,6931,16039,5901,
      1,1,3,5,9,63,63,49,405,915,1505,2141,6749,7799,17313,
      1,3,7,11,15,11,49,161,155,869,121,301,6561,4279,15233,
      1,1,5,13,19,13,103,59,503,293,701,2527,5327,13927,5025,
      1,1,7,1,1,37,55,155,485,399,855,2677,5927,9657,2795,
      1,1,1,5,19,15,121,69,385,75,1567,2649,5601,12981,15903,
      1,1,1,11,19,21,45,59,505,737,15,1383,1177,8375,15557,
      1,1,7,13,29,19,123,127,469,773,733,3289,8087,5803,27897,
      1,3,3,11,19,55,101,67,485,939,607,1521,6161,12235,16499,
      1,3,5,13,29,31,31,9,453,151,1055,3873,405,12877,29829,
      1,3,5,1,17,1,17,131,107,1003,1749,1849,6207,2153,21275,
      1,3,7,15,7,25,51,143,51,517,1841,1771,5389,4633,11123,
      1,3,7,11,23,7,89,95,403,361,835,585,2783,8091,5141,
      1,3,1,9,1,53,115,11,493,587,305,3605,1711,4169,20013,
      1,3,7,3,17,59,55,251,49,759,1227,3685,7765,1445,20385,
      1,1,5,7,29,55,19,157,129,927,893,1235,1955,8153,2865,
      1,3,1,15,21,35,81,53,175,939,1635,3597,747,14011,12867,
      1,3,7,1,27,61,91,73,405,677,603,2715,7099,941,24523,
      1,3,5,9,13,45,35,167,57,483,735,2777,7847,6257,13109,
      1,3,5,7,1,3,97,13,159,533,1791,1061,981,10795,26165,
      1,1,5,13,27,5,125,25,251,221,1909,197,6987,11537,15287,
      1,3,5,5,27,15,1,131,375,923,81,3153,6071,2515,23729,
      1,3,3,9,9,23,71,13,465,261,937,1549,5993,11325,15065,
      1,3,1,3,7,63,17,129,435,23,215,2251,1561,9703,26483,
      1,1,3,1,5,53,77,109,9,605,371,2081,6023,7145,15837,
      1,3,7,11,27,39,115,47,259,337,1857,3465,1549,7747,8525,
      1,3,7,7,29,29,75,77,29,661,899,3137,2661,15271,28093,
      1,1,1,3,3,3,11,219,39,757,1465,249,7445,7013,15187,
      1,3,7,15,15,1,39,245,427,1003,1493,1913,6435,14787,13481,
      1,1,7,3,3,37,5,97,343,833,1379,1551,5403,1843,5877,
      1,3,1,1,3,17,17,163,339,691,1707,1845,5941,4259,24531,
      1,1,1,1,27,21,85,221,71,949,1753,391,6349,15901,20811,
      1,1,1,5,31,19,45,99,469,783,1747,3807,5889,9485,13715,
      1,3,1,9,23,21,39,25,421,713,461,2857,5023,5341,6409,
      1,3,7,5,25,19,59,147,387,857,375,3103,1261,13697,25675,
      1,3,5,5,31,21,49,251,463,441,473,3487,3915,11151,17721,
      1,1,3,9,15,47,81,219,143,141,81,1705,5847,3437,30521,
      1,1,7,3,25,19,97,41,77,105,1337,695,7589,8587,7509,
      1,1,5,13,3,11,61,19,139,667,963,1567,5715,7079,15967,
      1,1,5,5,5,29,67,57,477,173,1163,727,823,15635,17705,
      1,3,7,11,13,39,57,193,73,617,535,1623,4581,4451,2589,
      1,1,5,5,9,27,75,127,325,413,1669,1749,8045,16199,12237,
      1,1,3,1,17,23,27,189,319,953,347,909,4401,12791,25077,
      1,1,3,3,17,51,37,79,301,607,885,1169,3275,3327,20013,
      1,3,5,3,21,9,99,213,387,889,575,3591,5377,2981,23989,
      1,3,3,13,11,7,23,255,279,853,453,2377,8123,15393,9669,
      1,3,1,7,11,9,109,35,405,449,1967,2943,3485,5031,14273,
      1,3,3,1,25,27,43,115,435,675,1937,1477,4831,9417,7017,
      1,1,7,13,19,45,83,241,487,215,1453,209,4061,1765,15623,
      1,1,7,7,21,31,95,9,287,1005,1933,3405,6913,7733,18975,
      1,1,1,11,13,11,25,39,283,57,255,2809,5767,6127,6705,
      1,3,1,11,1,51,73,181,261,215,385,2777,5169,12431,23563,
      1,3,3,9,9,39,123,197,501,679,109,3369,4817,8855,7997,
      1,1,5,1,29,61,15,183,453,999,1211,3217,8035,5153,19975,
      1,3,7,11,11,21,51,45,379,793,289,309,1229,7159,581,
      1,1,3,9,17,11,75,67,289,191,1083,2949,6063,6611,21595,
      1,3,7,3,27,45,59,193,485,277,27,1219,2389,15875,6273,
      1,1,5,3,31,29,65,197,115,163,9,783,5573,2833,12603,
      1,1,3,7,5,53,115,181,175,749,1335,1151,2131,12467,15811,
      1,1,1,9,27,39,11,1,443,677,777,1857,7459,3177,3875,
      1,1,7,7,17,3,23,161,105,603,1991,3845,465,11467,2077,
      1,1,3,13,5,23,39,35,399,795,265,207,1823,15069,31839,
      1,1,1,1,29,61,89,193,41,99,315,1021,6109,12507,19973,
      1,1,5,3,13,57,119,251,215,695,1521,4081,2481,657,855,
      1,1,7,3,25,5,3,133,111,385,773,1027,4327,3031,3537,
      1,3,7,5,5,27,43,117,479,83,1421,2791,6551,6231,10353,
      1,1,1,13,3,29,35,71,85,821,1671,3057,797,13683,7025,
      1,3,7,1,1,47,117,233,141,993,1381,2551,1031,11765,18429,
      1,3,1,3,13,3,77,29,35,807,1109,695,5605,5477,449,
      1,1,1,15,21,37,117,105,273,311,1287,1415,1221,1847,19487,
      1,3,1,11,21,61,107,225,335,501,1995,2399,5475,12613,18877,
      1,3,3,1,31,41,27,205,103,837,639,2007,2759,12471,1457,
      1,1,7,13,29,39,71,245,105,235,1277,1515,6129,15947,26653,
      1,1,7,5,7,13,87,251,315,1017,587,2917,5911,2919,29715,
      1,1,1,3,7,19,81,243,177,917,2023,2365,7465,4901,29841,
      1,3,5,15,1,31,15,147,285,1003,1757,47,6925,1197,19633,
      1,1,5,7,27,25,47,209,85,403,1399,2331,3663,595,13407,
      1,3,5,9,7,25,7,139,389,817,1153,1421,5735,9577,10269,
      1,1,1,9,5,61,49,117,389,541,433,1405,2575,223,7265,
      1,1,5,7,15,1,81,207,435,843,835,3797,7637,5333,31115,
      1,3,7,11,13,3,47,249,301,715,2015,3049,8155,10989,26051,
      1,1,7,7,3,33,119,113,381,575,367,41,3317,11727,4351,
      1,3,3,13,9,3,51,37,173,137,533,1827,631,10047,6267,
      1,3,3,11,17,39,61,245,13,139,1281,1319,1233,13629,32269,
      1,1,1,7,15,17,91,109,163,609,11,3251,7653,14035,31755,
      1,3,3,15,13,21,55,231,385,133,1833,2637,6935,14303,26745,
      1,1,1,7,17,41,125,141,89,823,1411,3637,6211,13323,6111,
      1,1,1,11,1,21,9,43,97,685,1223,1491,121,1793,2397,
      1,3,5,5,17,17,5,223,129,865,1839,1137,6391,4377,9233,
      1,3,7,15,21,55,5,77,341,637,1853,1435,1195,9283,21257,
      1,3,5,1,9,49,43,211,127,655,1397,1235,5279,2351,24229,
      1,3,5,3,25,29,13,229,25,675,837,2753,2125,9863,11293,
      1,3,5,7,23,43,127,1,163,237,337,3019,7747,16227,2881,
      1,3,5,5,25,9,43,171,421,521,1885,337,7873,6347,13181,
      1,3,5,5,7,47,107,173,163,191,625,3389,2833,7945,24787,
      1,1,7,3,27,57,27,209,253,815,301,1633,3945,5051,28851,
      1,3,7,9,9,51,103,213,437,189,1857,1331,5551,10641,27405,
      1,1,5,5,15,1,25,105,117,347,161,3369,3589,12903,23559,
      1,1,1,5,3,61,93,51,81,281,1383,745,4137,2005,3635,
      1,3,7,5,13,57,111,211,303,477,359,4019,6779,5129,22035,
      1,1,1,7,17,29,113,113,201,531,749,739,2879,3315,18733,
      1,3,7,13,21,55,21,183,359,75,377,2211,4281,14317,28307,
      1,3,7,1,21,1,49,213,317,75,113,1741,7963,12785,11571,
      1,3,7,9,11,31,29,101,261,141,715,2727,8187,2075,32433,
      1,3,7,3,23,9,17,143,385,211,593,241,6567,10777,6677,
      1,1,3,15,3,17,117,99,91,793,989,2421,5643,16103,9759,
      1,3,7,11,23,43,107,35,421,431,743,853,7939,12169,4199,
      1,1,1,11,21,53,17,203,123,395,59,929,255,7585,10945,
      1,3,3,15,17,57,57,133,67,71,1685,903,4079,15423,26495,
      1,1,1,15,3,47,95,39,405,407,1557,3001,6225,15187,5663,
      1,3,5,5,13,47,33,61,375,1023,1981,2773,2375,11321,17731,
      1,3,5,9,17,59,117,95,493,149,1269,2865,369,2109,24601,
      1,3,5,13,17,63,67,247,95,721,67,305,6179,15399,32559,
      1,1,5,1,3,21,41,15,453,475,2017,3193,5903,897,4237,
      1,1,5,3,15,41,1,141,441,575,155,3791,7711,11231,24611,
      1,3,7,1,17,53,27,169,31,437,963,1793,7777,1917,29311,
      1,3,3,13,9,27,77,87,329,885,749,1713,6013,6921,629,
      1,3,5,13,3,7,53,27,353,267,925,2141,439,15175,30851,
      1,3,3,13,17,57,35,101,265,901,1825,2159,6149,5967,24023,
      1,1,5,11,13,51,99,111,193,415,1541,2929,5045,3147,12587,
      1,3,7,11,15,9,33,17,511,815,299,1077,6171,10451,15039,
      1,1,1,15,25,63,51,137,449,951,1051,1101,4725,2463,7355,
      1,1,1,7,27,63,29,179,317,521,1459,827,6599,13459,15439,
      1,3,3,15,17,31,37,191,229,245,181,941,5761,1849,31599,
      1,1,1,9,27,45,67,239,481,615,1667,3751,8141,10013,2125,
      1,1,1,1,13,51,117,135,73,151,1291,2541,1411,3767,26949,
      1,3,1,9,7,11,21,187,243,857,1951,865,7273,2041,8155,
      1,1,3,3,19,33,89,115,455,137,707,1867,4221,2433,9119,
      1,1,3,11,5,3,121,1,71,951,603,3873,723,3285,19289,
      1,3,7,15,21,1,117,17,455,519,731,3003,5741,9557,29163,
      1,1,3,13,25,5,43,147,209,895,255,1231,241,487,15593,
      1,1,3,13,7,1,89,187,217,927,2029,3521,2777,8103,22819,
      1,1,7,11,7,33,3,73,5,489,227,2259,7031,6425,26135,
      1,3,3,7,31,19,97,201,455,819,945,2771,8083,8711,2835,
      1,1,1,5,15,45,43,157,245,967,877,2289,4499,9891,18827,
      1,3,1,7,21,59,123,63,231,485,1781,1211,4597,5269,1607,
      1,1,1,13,23,39,105,55,249,991,1625,3089,3825,4275,29139,
      1,3,3,1,29,29,55,169,13,895,1355,1101,6063,12935,23215,
      1,1,5,5,31,49,99,137,209,1017,1179,3931,637,14131,19285,
      1,1,1,1,3,11,119,11,215,337,243,3883,3807,7335,11901,
      1,3,7,3,7,27,71,225,219,367,1213,2739,1185,10175,21313,
      1,3,7,13,7,49,23,223,61,1011,797,1335,6711,5961,5605,
      1,3,3,11,19,37,1,149,39,661,929,2125,2299,5181,28083,
      1,3,3,13,13,9,67,21,463,279,529,523,6705,11011,31695,
      1,3,1,5,13,1,123,3,291,625,1949,2713,5917,10343,13627,
      1,1,3,9,27,41,3,207,103,265,811,549,6109,313,8889,
      1,3,3,13,23,43,99,33,279,463,955,793,4113,10615,16957,
      1,1,5,7,11,49,79,45,17,937,359,1037,1099,3003,31561,
      1,1,1,7,3,45,111,35,109,983,53,4057,7349,3599,2209,
      1,3,7,11,9,43,27,9,85,529,1497,347,759,12449,11373,
      1,1,3,9,17,1,49,31,367,813,1385,2025,773,4679,4543,
      1,1,5,15,15,9,43,97,239,995,1037,841,4167,12113,23765,
      1,3,5,9,29,53,123,49,221,113,1157,73,6087,1363,11029,
      1,3,1,13,3,15,69,199,279,919,5,161,4817,15031,121,
      1,3,1,9,3,31,117,77,393,241,645,3181,1067,15879,2037,
      1,3,3,15,3,63,57,33,117,789,941,1301,5865,12693,3523,
      1,1,5,13,3,61,51,151,175,305,95,1557,6567,7841,13903,
      1,3,3,5,15,25,127,79,245,767,645,3933,1357,12579,4067,
      1,3,5,11,21,31,13,251,127,231,1795,2627,1191,3363,23543,
      1,1,3,5,7,49,121,57,131,481,1879,525,5225,337,1957,
      1,1,5,13,9,55,27,37,211,125,119,3373,251,12357,13975,
      1,3,3,15,1,51,91,119,233,993,203,1635,1167,6327,29119,
      1,1,7,1,13,5,23,253,121,989,1105,3321,3221,6073,21185,
      1,1,3,15,13,49,121,247,247,133,485,1067,7875,411,7647,
      1,3,7,13,31,37,127,241,145,133,53,267,2029,3703,16123,
      1,3,1,15,15,9,15,89,35,367,401,61,1953,7873,17861,
      1,1,1,1,1,41,71,249,213,779,1385,1767,999,15151,16647,
      1,3,7,13,31,23,123,235,343,949,309,3777,3587,853,19779,
      1,1,3,13,29,35,5,37,63,757,303,1579,3443,243,11873,
      1,3,1,9,19,49,81,53,11,901,1857,147,3103,14019,21,
      1,3,7,13,3,39,99,99,45,91,1567,551,3129,4809,29057,
      1,3,7,3,3,27,17,231,377,381,1479,2525,2595,2799,25737,
      1,3,5,15,15,25,103,215,301,59,1417,981,7579,12731,22329,
      1,1,1,13,5,31,61,31,349,925,1301,685,435,11567,10715,
      1,1,7,9,19,57,109,1,37,377,1015,2273,6741,3191,15949,
      1,3,3,13,3,23,103,127,11,59,1847,1175,425,3423,20643,
      1,3,3,7,3,11,105,141,55,217,1427,477,667,9403,11905,
      1,3,3,5,3,27,11,187,495,907,1925,445,6639,8159,15225,
      1,3,1,5,27,31,77,213,73,343,1123,3609,2431,15329,32165,
      1,1,7,5,1,11,105,139,485,1007,709,3509,5231,11717,31433,
      1,1,3,15,23,49,95,169,399,1019,19,2013,5311,7951,22609,
      1,3,1,7,13,3,29,203,209,701,1791,2615,5351,4237,12565,
      1,3,1,15,27,11,91,31,205,205,1683,901,5129,6049,11865,
      1,1,7,7,27,59,21,3,209,79,769,4013,2041,2645,11561,
      1,3,7,11,5,45,39,243,185,871,795,1845,8043,6285,20991,
      1,1,5,7,13,7,15,165,349,179,789,1269,3787,5429,26567,
      1,3,3,13,31,23,75,41,177,735,1889,4039,3519,15003,965,
      1,3,1,7,15,57,15,139,27,469,1003,691,7893,9643,30983,
      1,3,1,13,23,27,3,237,233,875,99,883,6167,5463,6245,
      1,1,5,13,25,57,79,51,147,619,1147,279,6583,1939,477,
      1,3,5,5,31,61,125,163,213,699,583,3865,615,9707,11651,
      1,1,5,1,5,21,93,239,31,641,777,27,5247,8993,21053,
      1,3,7,9,1,13,61,57,503,453,83,3271,2845,1121,18639,
      1,1,7,5,29,53,13,219,379,441,821,3179,4877,2535,7557,
      1,1,7,13,9,53,17,183,265,393,587,2753,6453,7135,24737,
      1,1,1,13,11,23,73,109,393,1013,559,755,7291,6631,26509,
      1,3,1,5,5,15,107,103,355,307,1559,837,5413,5285,17489,
      1,1,5,7,17,21,21,23,109,709,1947,3585,3629,4669,949,
      1,3,7,1,9,33,85,147,467,259,1913,199,7399,9551,22387,
      1,3,5,11,15,53,23,41,249,515,1161,2467,1299,7449,2613,
      1,1,5,5,5,29,91,139,487,545,321,3035,4545,6747,21673,
      1,1,3,13,23,49,95,103,25,119,469,2515,2551,841,25089,
      1,1,5,7,11,31,31,197,245,715,257,4043,8099,11531,5617,
      1,1,3,3,19,7,9,179,103,995,191,179,3843,5215,27639,
      1,3,1,7,23,59,25,65,399,211,1453,3511,7203,16015,32197,
      1,3,3,5,9,35,109,67,197,449,643,519,5751,15551,11331,
      1,3,5,3,1,17,53,201,265,351,467,911,1117,7183,20371,
      1,1,7,7,27,17,93,81,227,619,1191,735,815,12615,2719,
      1,3,1,15,19,3,83,75,343,297,1019,3469,4383,13299,29755,
      1,1,5,3,13,55,119,169,85,595,299,2469,5625,2877,16117,
      1,1,3,5,15,17,61,161,47,393,143,867,5517,9495,12795,
      1,3,5,1,27,31,113,125,251,687,969,1473,2245,6355,13655,
      1,1,1,5,5,37,29,133,443,899,277,2353,7223,4459,19159,
      1,1,3,9,19,27,53,145,195,167,2045,447,1803,1895,8431,
      1,1,3,9,5,27,91,147,233,451,475,27,4629,16181,16437,
      1,3,5,3,29,17,53,167,433,689,1131,2985,1553,11697,6993,
      1,3,3,13,21,43,69,229,399,525,179,237,7017,5703,17653,
      1,1,3,15,13,39,75,163,229,875,197,477,3667,15501,15801,
      1,1,7,15,15,51,81,187,487,673,865,1915,1009,5935,8097,
      1,3,5,5,7,3,63,77,495,815,391,2321,1007,15661,30715,
      1,1,7,3,17,25,83,173,173,911,1373,2957,4549,15977,17695,
      1,1,7,13,13,23,77,147,497,1003,1687,1795,1393,1881,8479,
      1,3,7,11,27,43,97,25,61,457,203,2573,5943,15021,4003,
      1,3,3,13,9,37,37,25,219,889,1535,2791,4531,13679,12663,
      1,1,3,1,17,7,51,123,89,887,1467,1645,3767,6383,30837,
      1,3,3,1,21,47,5,151,83,257,569,2711,637,12569,16893,
      1,3,7,1,31,37,73,3,115,919,1817,2483,4811,15245,4375,
      1,1,1,5,1,39,39,231,9,733,455,3383,4777,7235,12631,
      1,1,7,9,13,25,55,25,73,59,1699,929,755,1279,5583,
      1,3,5,3,9,49,79,55,479,179,1159,4079,3503,11603,12361,
      1,1,5,9,21,45,31,163,377,817,219,147,2581,12769,30783,
      1,3,1,7,15,27,39,189,493,259,1663,1213,961,11089,16079,
      1,1,5,9,5,41,13,153,313,337,1027,1267,4249,13071,27043,
      1,3,7,3,13,11,23,255,51,527,317,3217,5037,12723,17411,
      1,1,5,1,25,57,83,97,233,513,1283,2675,4111,4111,32141,
      1,3,3,15,25,33,103,81,155,189,139,1179,2691,15119,13959,
      1,3,3,1,25,55,67,19,19,9,437,579,4273,10733,7125,
      1,1,1,7,23,41,47,5,435,749,595,199,3941,7199,4795,
      1,3,1,15,5,49,35,9,199,703,1769,3269,5689,13063,22771,
      1,1,5,5,21,55,125,55,63,149,1167,3577,1051,3921,20545,
      1,3,7,13,29,53,107,193,163,339,1335,1573,5809,5681,29487,
      1,1,1,9,17,9,91,177,211,453,1807,1881,6051,225,6021,
      1,1,1,13,15,1,29,43,181,105,1945,2313,6429,2901,6221,
      1,3,5,11,29,55,115,115,187,1013,697,1885,121,12387,32443,
      1,1,1,7,19,51,21,107,55,125,1655,2281,3293,15749,27521,
      1,1,7,9,19,9,81,93,139,401,193,73,5159,9323,6019,
      1,1,7,9,15,51,115,69,247,599,1163,2251,1211,8827,15581,
      1,1,7,9,5,39,75,185,321,911,849,843,6791,10407,10513,
      1,1,5,5,15,9,21,47,459,681,2001,1545,5939,7073,29043,
      1,3,1,11,13,25,53,97,487,797,567,3757,5597,6313,18531,
      1,1,3,3,29,55,11,219,325,591,2015,383,2595,11855,22501,
      1,1,1,5,15,57,33,125,323,749,1843,4019,2075,6673,6957,
      1,1,5,7,19,7,47,239,51,107,1081,467,5493,7617,10355,
      1,3,1,1,11,3,67,199,175,421,935,309,4449,6363,9183,
      1,1,1,7,9,33,3,219,481,513,417,1267,2863,765,18431,
      1,3,1,1,19,1,89,109,415,105,487,3241,7465,9233,16307,
      1,1,3,13,9,43,25,231,383,789,1855,691,3465,2387,11715,
      1,3,3,3,13,39,63,107,33,265,437,117,3179,5543,28179,
      1,3,3,13,21,5,31,111,321,425,253,3501,3209,15429,18383,
      1,3,5,9,1,27,117,187,433,459,1999,1069,4857,8591,26343,
      1,1,7,3,15,43,11,193,391,341,1203,1259,7265,1783,13161,
      1,1,7,1,5,15,45,143,193,985,1105,3483,2421,9687,22347,
      1,3,7,13,21,17,79,231,487,663,1101,1025,5779,14681,29181,
      1,1,3,9,15,19,55,219,27,963,1513,1017,3907,12279,32655,
      1,3,7,3,31,27,17,1,51,861,529,1225,6395,15787,5231,
      1,3,3,11,27,7,101,143,21,191,1437,2393,4097,14319,6977,
      1,3,3,3,25,35,105,141,433,269,1469,2939,5387,7373,7863,
      1,3,7,5,5,21,23,11,217,357,1847,101,1161,5297,14879,
      1,3,1,3,25,23,81,217,505,161,1667,1343,1751,2463,26431,
      1,1,3,1,17,51,125,205,385,351,731,2911,2749,2689,27031,
      1,1,5,5,5,17,31,171,477,671,167,1797,8047,10123,4325,
      1,1,7,1,11,23,123,161,99,1007,765,1965,5395,16193,17751,
      1,3,1,9,13,11,111,217,31,753,377,2267,7893,7195,24999,
      1,3,1,9,21,53,127,121,151,395,1447,1411,5179,12043,27607,
      1,1,5,3,11,37,97,139,113,835,229,3741,827,5527,5779,
      1,1,7,7,27,55,11,55,429,269,1179,233,1053,10225,16703,
      1,1,1,3,15,9,67,119,95,753,511,2507,3953,6403,27635,
      1,3,3,7,27,57,25,27,249,515,193,4043,2017,751,10949,
      1,3,1,9,31,57,67,21,177,573,1835,2015,6201,2383,31087,
      1,1,5,1,19,3,89,243,69,387,1905,3465,2775,7713,30081,
      1,1,3,3,9,59,15,89,85,605,881,263,2551,797,16541,
      1,3,7,11,25,41,59,139,405,571,1147,2963,4175,12901,6309,
      1,3,1,5,29,29,11,243,183,281,807,1,7079,10079,13865,
      1,3,7,5,5,1,89,55,423,157,675,1849,241,6467,15459,
      1,1,7,11,15,63,89,109,501,549,317,3043,1151,3895,19851,
      1,3,1,15,7,23,97,97,225,737,1117,3325,209,14169,10813,
      1,3,7,13,13,39,91,153,395,879,1041,3753,5577,1985,25247,
      1,1,1,3,17,15,113,143,101,901,1119,1819,3577,3441,31511,
      1,3,1,11,15,27,21,37,287,121,451,1353,2173,299,18791,
      1,3,3,5,23,1,49,145,315,769,99,1385,5961,9121,1465,
      1,3,3,13,21,39,39,233,271,113,615,2257,3765,5921,313,
      1,3,7,7,25,45,11,237,83,203,929,1819,2679,11583,30091,
      1,1,1,7,21,63,85,251,133,991,1515,2547,6051,7279,3569,
      1,3,7,15,11,19,87,17,313,283,1021,2743,4855,13741,17955,
      1,1,7,13,29,13,61,93,81,91,995,907,4847,2599,20041,
      1,1,3,11,13,45,103,33,243,109,2029,121,231,16179,13741,
      1,3,5,9,9,5,73,225,199,723,611,1909,2345,10257,9909,
      1,1,3,11,7,5,33,89,431,899,803,3173,6131,16097,20561,
      1,3,3,7,7,47,23,47,411,69,239,661,5591,10457,24245,
      1,1,5,15,25,35,87,23,115,939,1579,119,4001,13791,9729,
      1,3,5,11,25,45,29,195,369,237,735,155,123,4415,32255,
      1,3,3,9,13,53,15,77,313,75,529,925,5679,14585,19889,
      1,1,7,15,15,27,105,13,31,669,563,1809,4321,7797,4177,
      1,1,5,9,3,29,111,177,33,235,1951,1561,141,4803,16327,
      1,1,1,7,9,41,1,149,95,933,115,1619,771,8189,8781,
      1,1,5,3,13,41,33,159,355,159,1243,1439,6571,14397,31321,
      1,1,7,11,9,15,91,145,457,255,1449,611,1449,2521,28949,
      1,3,7,5,27,57,35,99,447,287,743,1163,4379,7361,3831,
      1,3,3,7,15,53,41,83,133,571,1739,531,2921,11527,21941,
      1,1,1,13,9,27,39,113,429,447,595,3171,5245,4095,14847,
      1,1,3,7,19,19,21,101,489,1011,265,3899,3225,11701,5193,
      1,3,7,3,15,25,103,213,441,215,1483,263,3561,7915,7969,
      1,3,3,3,11,47,97,29,489,867,1347,2155,4871,8001,18305,
      1,3,1,9,25,15,61,17,343,775,1765,3803,4577,8437,12605,
      1,1,5,3,11,39,69,23,23,65,1967,2429,1703,6671,14981,
      1,1,5,15,23,59,125,51,225,439,2019,2589,7781,13111,2911,
      1,1,1,3,1,31,37,245,203,305,821,367,5211,9791,21777,
      1,1,5,9,9,31,97,25,271,83,343,2461,1805,14383,10059,
      1,1,5,13,15,33,127,109,137,963,961,1647,7881,8133,22359,
      1,1,3,7,25,31,123,241,283,1,1781,23,971,6485,127,
      1,1,5,15,15,27,25,145,395,679,979,571,1585,14787,7465,
      1,1,5,7,13,11,7,131,511,597,379,1513,6267,16039,1503,
      1,1,1,13,15,49,73,217,353,577,1913,1127,961,11557,24993,
      1,3,3,9,7,3,105,141,377,687,1917,485,983,11149,23303,
      1,1,3,15,11,7,117,179,505,67,1817,913,5757,1981,1637,
      1,1,1,7,5,29,3,43,223,295,1895,3425,5355,5155,17197,
      1,1,7,9,21,59,121,245,73,233,1527,869,4145,7995,6473,
      1,1,5,13,17,21,89,179,495,669,453,2603,5969,6161,4743,
      1,1,7,11,25,21,103,131,391,249,1633,2603,2207,8987,15487,
      1,3,7,9,13,45,99,251,115,597,1505,2421,1231,10015,24295,
      1,1,5,5,31,49,17,67,463,813,1491,3309,7881,8109,7289,
      1,3,1,15,23,35,123,21,169,499,95,603,1829,7865,26313,
      1,1,7,1,9,29,45,65,95,97,673,3673,2969,2317,22209,
      1,1,3,7,29,33,121,17,331,487,1901,1951,5383,9375,4029,
      1,3,7,9,25,43,91,147,141,401,1647,2697,4645,7179,31857,
      1,3,5,11,9,31,127,105,39,883,1635,919,5069,2875,24519,
      1,1,5,9,1,63,73,135,95,503,385,3903,545,12635,27569,
      1,1,3,11,27,31,47,173,55,339,1255,1947,793,14133,13963,
      1,1,3,15,17,33,113,249,401,743,1307,3123,627,1253,13285,
      1,1,3,1,9,7,39,65,281,107,833,193,2987,12267,31335,
      1,1,7,3,15,21,99,211,39,179,587,1169,6455,8225,2049,
      1,3,5,13,5,13,123,1,223,273,731,2967,4793,4229,26031,
      1,1,1,1,3,17,7,23,225,757,743,1257,2047,12509,25467,
      1,1,7,15,29,3,15,113,227,675,1295,2777,2921,5485,2577,
      1,3,7,13,19,21,85,129,45,599,317,1513,4953,10383,25253,
      1,1,7,11,13,47,127,67,219,131,905,2005,851,15243,5777,
      1,1,5,3,23,57,57,189,153,37,955,2049,1295,15119,27213,
      1,3,7,11,13,61,3,241,269,789,1595,2369,4843,11347,21543,
      1,1,5,5,25,21,19,237,3,605,1343,3965,3511,7889,27759,
      1,3,1,15,21,15,123,5,345,945,283,1313,335,2085,19505,
      1,1,3,3,5,21,123,89,67,11,1247,1155,287,13455,5693,
      1,3,3,13,1,53,101,27,387,379,19,751,2445,11737,975,
      1,3,3,3,9,29,81,117,443,145,1619,1813,8125,5829,28617,
      1,1,5,15,27,15,83,83,61,715,1655,1631,3457,2727,2163,
      1,3,1,5,11,11,121,7,135,883,927,1817,6839,12361,24119,
      1,3,7,11,23,59,39,165,109,355,1303,381,5697,275,3771,
      1,3,5,11,11,5,81,157,55,435,613,127,4087,3791,21627,
      1,3,7,15,13,37,83,195,207,771,51,3685,6389,1229,11101,
      1,3,7,13,31,3,9,13,487,95,77,809,5809,12887,29933,
      1,1,3,7,25,9,13,29,353,659,1785,3825,3729,13109,12973,
      1,1,1,5,21,3,97,1,245,917,29,1429,8141,7569,32493,
      1,3,1,9,19,13,13,109,377,1007,1737,1939,1419,1145,5065,
      1,1,7,9,27,57,53,69,423,43,1629,1003,1473,10809,5659,
      1,1,1,9,1,45,11,231,351,155,663,2783,3491,5725,25207,
      1,1,1,3,15,25,77,89,231,813,657,2603,4885,1383,14499,
      1,3,5,5,9,21,101,181,449,491,737,803,659,11771,545,
      1,3,7,9,7,19,27,199,265,329,1031,1235,3191,10071,16281,
      1,1,7,11,27,55,3,127,503,1003,1041,1953,5835,4851,13485,
      1,1,7,15,5,45,97,61,221,497,1949,3163,4707,8441,1437,
      1,3,5,1,3,35,107,9,473,971,227,2225,3999,3095,18879,
      1,1,1,9,21,59,21,1,41,435,575,491,1839,1095,9727,
      1,3,5,9,13,29,123,251,465,701,1105,829,573,11503,11861,
      1,3,3,13,27,59,29,111,225,973,561,1481,835,9261,13831,
      1,1,1,7,17,3,97,211,333,315,571,3523,7305,6461,20139,
      1,3,7,11,31,21,105,247,113,863,1767,381,4623,8935,7911,
      1,1,5,7,29,45,17,155,69,17,655,1983,6385,6177,7961,
      1,3,3,15,31,15,63,81,309,115,393,3445,689,13963,18887,
      1,1,5,1,19,39,127,61,357,53,195,2745,7853,5753,3669,
      1,3,7,7,17,51,57,145,451,365,1517,909,4265,10737,9579,
      1,1,3,13,3,37,121,103,257,47,1685,2951,5753,15379,8899,
      1,1,5,7,31,63,61,197,97,773,133,1517,3093,14879,22941,
      1,1,5,1,3,9,27,53,97,663,1915,409,471,1391,24853,
      1,1,1,7,21,53,69,5,187,571,2023,997,323,12059,7071,
      1,3,3,1,7,59,55,157,101,123,1301,3709,4673,3897,28791,
      1,3,7,5,5,23,39,139,365,415,1481,3415,6323,11109,5719,
      1,3,5,3,5,11,23,143,243,229,183,3367,3187,8151,28351,
      1,3,7,9,5,37,29,23,437,827,985,2879,7611,1391,19087,
      1,3,3,5,7,9,5,143,217,757,1697,2459,453,8679,4513,
      1,3,5,5,11,33,3,143,293,921,185,2461,5547,12247,28591,
      1,3,7,5,3,53,43,179,235,417,1307,1367,3695,12809,1807,
      1,3,1,11,15,43,115,229,157,25,687,3347,271,5777,8557,
      1,3,7,5,27,37,55,135,209,47,1603,957,5785,11141,10407,
      1,1,1,15,17,17,103,29,489,493,119,1707,3463,1815,32055,
      1,3,7,11,17,13,115,145,77,515,1911,477,5997,8731,3143,
      1,3,1,13,31,41,73,91,231,1,455,2023,4691,3613,16329,
      1,1,5,15,15,39,17,117,131,657,1939,2245,2575,195,25209,
      1,3,7,15,5,51,69,141,499,931,1165,2119,1703,10867,28443,
      1,1,1,15,13,45,45,103,115,177,651,2545,1417,5349,3385,
      1,3,3,1,1,41,117,15,225,861,843,2775,4543,6275,14671,
      1,3,5,15,5,35,87,193,341,55,1131,945,6865,11271,18705,
      1,3,5,9,13,35,71,197,79,351,3,3939,1105,12455,28921,
      1,3,1,13,9,23,89,165,59,257,1369,161,6255,2997,19175,
      1,3,5,3,5,41,107,231,111,207,1865,2079,5891,2487,5863,
      1,3,7,15,3,3,105,235,263,991,367,1885,1769,7805,11909,
      1,3,3,5,15,59,67,247,77,367,1641,1959,1921,5939,17355,
      1,1,7,1,3,53,37,5,221,779,1353,1633,2769,6355,8505,
      1,1,7,13,11,13,73,227,115,523,355,3127,7545,8409,22335,
      1,1,5,11,21,15,91,115,427,683,461,2433,6313,4595,24401,
      1,3,7,5,29,21,57,215,423,717,1455,705,6835,4503,26077,
      1,1,1,15,3,33,25,227,381,477,1023,2751,2229,631,16903,
      1,3,1,11,9,17,59,73,53,671,251,1729,7593,12473,22533,
      1,3,3,1,3,35,37,173,459,143,135,3871,2689,8007,4379,
      1,3,5,9,23,19,43,45,493,509,1851,1615,5675,13793,6973,
      1,3,3,15,5,17,77,85,451,753,579,1057,4851,6017,4195,
      1,3,3,5,31,29,81,159,103,391,15,899,4623,5957,31961,
      1,1,1,7,17,57,81,17,177,633,49,2793,5229,5995,9491,
      1,1,7,15,17,19,65,57,189,239,1229,929,2681,12845,29311,
      1,3,1,11,13,47,61,203,383,875,943,139,4217,8279,1047,
      1,3,7,13,23,7,1,69,47,537,1325,3101,685,14057,19953,
      1,3,3,1,1,7,39,77,47,755,527,2985,5433,15095,27741,
      1,1,7,5,23,57,79,155,81,937,1071,3929,1655,3831,17351,
      1,3,7,1,3,41,13,235,207,487,1883,2247,1231,2751,15615,
      1,1,7,1,21,57,95,191,119,483,283,2221,5665,14819,26097,
      1,3,1,1,9,59,27,51,393,31,925,715,7705,14885,28767,
      1,1,3,3,3,61,109,131,113,249,1331,2521,2973,6375,20093,
      1,3,7,9,31,37,125,245,237,245,111,379,7495,15531,2325,
      1,3,7,13,21,21,57,21,449,969,417,2999,509,639,7797,
      1,3,7,7,7,29,11,175,55,705,891,863,3021,10071,10267,
      1,1,3,13,19,17,127,57,449,579,337,899,1235,11269,4245,
      1,1,1,11,29,61,35,75,249,683,287,45,3277,7521,2073,
      1,3,5,5,15,25,77,63,63,801,1387,1533,2185,10899,28381,
      1,3,1,1,21,49,3,249,419,575,87,3749,2523,16125,9483,
      1,1,1,11,21,43,85,211,449,439,1495,1841,4765,15253,1467,
      1,3,3,15,3,37,31,243,187,995,1103,2723,1523,15967,28649,
      1,1,5,11,9,11,17,87,335,125,1079,1657,1237,8059,29833,
      1,3,1,3,3,41,35,37,33,61,505,3203,5,101,8571,
      1,1,3,11,9,11,85,235,261,473,109,2127,5745,6389,7431,
      1,1,5,15,3,55,77,97,17,193,1267,3063,6531,9797,8639,
      1,1,5,5,25,41,79,83,485,697,149,1023,89,6115,15227,
      1,1,3,15,1,9,73,251,33,599,1017,353,4305,16033,29663,
      1,3,7,15,3,1,89,39,125,337,1445,3131,3685,9849,25829,
      1,3,7,3,19,1,63,179,349,135,185,2977,2527,15087,18133,
      1,1,3,3,23,7,91,221,325,723,345,81,8077,5501,8453,
      1,1,3,9,7,3,13,173,479,161,1989,3255,2069,6717,559,
      1,3,3,5,9,61,93,203,277,367,1141,981,4745,12625,21003,
      1,3,5,5,27,17,5,211,403,701,5,3091,4611,5615,23667,
      1,1,3,1,21,61,125,77,57,463,1499,791,2087,2805,18829,
      1,3,5,3,11,41,125,231,119,837,831,1331,7439,2381,3759,
      1,3,1,11,19,59,117,107,443,699,315,1491,2581,15871,17159,
      1,3,5,11,5,9,121,35,209,877,527,3493,4657,16093,17589,
      1,1,7,15,9,43,119,29,381,479,1443,3171,5053,9625,21161,
      1,1,3,5,15,21,31,223,83,399,1529,3605,6343,10469,10099,
      1,1,3,5,5,45,23,123,353,971,85,3069,3245,6569,13241,
      1,1,1,3,25,49,5,77,491,881,993,1195,7677,5709,10807,
      1,3,3,3,5,49,127,255,183,583,1599,987,7281,7149,28507,
      1,1,5,1,13,55,55,157,197,25,1971,3161,3903,8919,13563,
      1,3,7,9,3,37,79,193,25,103,843,2651,6341,2653,24337,
      1,1,7,3,25,49,99,139,45,211,2033,2331,7037,7177,1755,
      1,3,7,3,5,19,127,135,403,221,141,1065,3935,2745,25979,
      1,1,3,3,31,23,111,37,261,7,835,2379,7927,8181,23751,
      1,3,7,15,1,39,79,3,103,427,1917,809,5039,689,1939,
      1,1,1,15,29,37,39,243,149,353,763,3405,5751,9441,6653,
      1,3,3,11,1,57,125,151,445,423,841,2265,5017,15863,13057,
      1,3,5,13,11,49,61,159,211,917,561,1903,3985,11117,28969,
      1,3,5,13,29,5,35,51,91,291,9,3713,3341,4551,12085,
      1,3,3,1,1,39,111,141,319,179,1709,1605,5063,13279,10003,
      1,1,3,9,7,59,91,41,343,475,1669,2311,5141,12661,25847,
      1,3,5,9,9,11,49,221,1,243,791,229,503,373,19189,
      1,1,5,11,17,13,45,57,215,491,1601,2183,3713,429,22007,
      1,1,3,11,31,61,23,237,261,955,1085,1541,2601,909,7749,
      1,1,3,9,13,11,121,173,177,551,1757,2745,2265,4611,743,
      1,1,3,15,23,43,107,239,463,369,1857,1073,1247,1029,22557,
      1,1,3,11,23,35,89,93,41,941,1141,2339,1423,8007,28685,
      1,3,5,13,29,7,79,15,59,145,1237,2215,1257,12621,31101,
      1,1,3,7,13,55,57,229,205,1009,341,3901,5189,957,32587,
      1,3,7,11,1,1,41,7,365,407,1609,1423,6483,5171,32519,
      1,3,7,3,17,31,125,27,125,335,1395,2639,329,2549,14449,
      1,3,3,7,19,45,11,73,123,179,1685,3385,2379,3387,16793,
      1,3,7,5,31,25,47,153,121,453,935,3953,2081,12145,24979,
      1,1,7,13,25,11,65,3,277,237,1129,1801,4165,9065,18747,
      1,1,7,7,13,5,37,253,507,645,1355,3401,6707,6329,11237,
      1,1,3,15,17,49,3,233,407,451,69,3859,3171,12303,21031,
      1,1,3,3,9,53,119,117,401,903,1449,3639,4083,2095,22085,
      1,3,7,15,5,61,117,193,137,431,195,4019,3047,5049,14281,
      1,1,1,15,17,19,29,83,449,257,1105,1949,1749,3459,6343,
      1,1,1,15,23,39,61,219,109,365,863,1813,6673,15999,5101,
      1,1,5,5,13,11,37,151,365,719,1233,2425,1285,1721,1205,
      1,3,3,3,7,53,109,153,45,425,1741,1229,4405,8071,25155,
      1,3,1,1,1,13,39,49,413,77,1367,2553,5563,7659,3467,
      1,1,5,9,3,49,23,11,445,121,1505,877,4137,1809,2429,
      1,1,1,11,21,13,93,33,493,805,775,2939,2961,13625,31879,
      1,1,7,5,1,59,63,131,373,23,337,2107,5315,4889,22851,
      1,1,3,13,21,47,15,131,353,793,1891,1757,5793,1147,23697,
      1,3,5,13,7,59,25,135,259,109,1835,429,8153,7355,145,
      1,3,3,13,9,47,121,89,89,635,1079,2353,4803,11369,12653,
      1,3,5,9,23,39,49,231,105,603,613,2021,6073,11819,10595,
      1,3,7,7,7,19,19,155,347,387,1459,3793,619,14437,2455,
      1,1,1,15,21,35,19,185,483,425,479,3429,5403,10791,14219,
      1,1,3,11,5,51,105,63,493,677,1457,2865,5619,9321,19583,
      1,1,3,3,23,1,77,177,263,289,1567,3837,5359,3269,16023,
      1,1,7,3,13,61,79,77,51,953,1417,795,4467,2981,25131,
      1,1,5,13,23,13,29,185,337,7,149,3609,8119,9545,16579,
      1,3,1,5,23,9,123,15,99,55,1021,3709,1521,15189,22193,
      1,3,7,9,13,41,39,45,49,181,1587,3213,1037,14775,3333,
      1,1,1,7,29,55,59,31,411,601,191,283,3211,7951,7919,
      1,1,7,7,21,47,7,193,343,831,1267,3289,1015,13093,2717,
      1,3,7,1,17,9,97,19,279,827,1699,3573,3137,3535,17791,
      1,1,5,11,27,15,103,135,35,625,1575,97,7013,13353,19333,
      1,3,3,7,17,13,49,135,435,743,1799,2655,4839,2893,31153,
      1,1,5,1,3,41,1,195,53,803,1575,2939,3873,10495,5211,
      1,3,1,15,19,19,37,59,355,483,685,3899,4645,15127,3479,
      1,1,5,3,25,9,9,229,101,631,1165,4091,3723,10655,9463,
      1,3,5,15,5,13,91,61,19,469,1675,3331,3121,3435,4111,
      1,1,7,1,31,61,23,83,165,551,1097,3825,5385,4723,3635,
      1,3,7,15,9,31,11,121,503,855,561,1647,1229,1147,15997,
      1,3,7,13,21,47,41,195,197,719,1263,3609,7515,2659,30713,
      1,1,1,7,31,61,101,101,479,571,605,301,6633,15587,23665,
      1,3,7,3,25,39,35,225,135,463,53,709,5129,4135,10421,
      1,1,5,13,19,55,107,15,163,287,673,899,5197,4619,3465,
      1,3,3,5,21,49,15,105,283,877,1875,1079,3431,13053,26599,
      1,1,7,1,1,1,95,113,119,575,1159,2325,6895,12177,4369,
      1,1,1,11,25,25,83,207,301,729,1947,2321,3621,15707,11303,
      1,1,5,5,7,63,83,105,211,175,1817,2883,5385,7437,24865,
      1,3,7,5,23,39,19,211,151,295,573,223,5065,6345,23187,
      1,1,7,11,15,31,89,123,57,695,685,1799,659,9929,22933,
      1,1,7,7,19,17,27,137,117,141,1481,869,7061,3073,19671,
      1,3,3,11,9,19,123,93,39,517,883,3769,2267,8089,6617,
      1,3,1,7,9,61,51,241,319,853,1239,899,105,1677,29351,
      1,1,7,15,13,59,85,175,223,87,905,3175,3405,3489,18475,
      1,1,1,15,1,55,79,97,315,605,851,4015,3689,9371,31523,
      1,1,5,15,1,39,91,27,211,881,1375,2307,5791,10185,23093,
      1,3,1,5,3,17,59,219,105,623,21,2843,3427,4799,3793,
      1,3,3,7,21,55,17,29,397,93,1981,4047,935,5971,14589,
      1,1,3,9,5,57,63,27,373,815,167,205,367,4945,30041,
      1,1,5,9,7,3,69,35,197,309,1729,3735,1523,10427,26253,
      1,1,3,7,7,49,35,189,297,311,2025,305,3863,14393,2533,
      1,3,3,9,17,31,5,17,167,601,909,3149,2533,12123,25325,
      1,3,5,3,11,41,69,199,79,611,133,3519,5955,4609,27403,
      1,3,3,13,3,17,53,165,361,797,1447,869,6707,6541,32249,
      1,3,1,1,29,47,17,45,473,199,1595,3095,3635,6965,21859,
      1,1,3,9,1,15,59,163,91,811,1087,1707,6743,12643,29901,
      1,1,1,3,19,21,7,209,121,821,709,1085,5333,7689,28355,
      1,3,1,15,5,27,115,31,37,79,1347,155,3709,13251,32151,
      1,3,7,15,27,27,127,231,137,205,1665,1461,299,2797,879,
      1,1,1,7,13,3,127,13,253,481,1435,1895,2665,7611,17761,
      1,1,3,7,7,21,71,247,301,183,1785,331,4835,2251,4493,
      1,3,7,9,9,1,77,169,103,647,1959,1847,5803,3421,15915,
      1,3,1,7,19,17,81,45,263,549,1607,2177,1117,14427,16451,
      1,1,7,15,27,25,27,27,33,813,1667,253,2749,927,29707,
      1,1,7,3,17,29,13,67,417,303,19,3809,7225,12775,3933,
      1,1,1,11,13,41,77,217,281,659,1099,3047,1619,525,4313,
      1,3,3,9,23,47,5,33,219,531,77,2307,1893,8335,8281,
      1,3,7,3,3,35,27,249,159,495,431,3001,1475,11505,15693,
      1,1,5,9,21,49,43,159,465,959,179,993,121,11569,21027,
      1,3,1,5,1,61,9,221,231,55,191,2829,3331,8911,15109,
      1,1,7,1,7,35,67,97,159,191,935,3151,6397,10751,1835,
      1,1,1,7,15,39,127,163,437,333,829,753,8151,13239,523,
      1,1,3,13,9,25,73,155,445,239,2035,15,5243,15531,1733,
      1,3,7,15,5,25,3,55,117,57,783,1509,7043,13159,8557,
      1,3,5,1,21,55,89,119,199,79,161,1597,3263,3335,5757,
      1,3,7,5,27,23,85,113,111,211,389,1513,2759,7945,931,
      1,1,1,7,1,5,17,177,357,619,5,2583,621,2973,28845,
      1,3,7,13,11,21,47,99,421,279,1541,1305,4571,6127,20735,
      1,3,5,5,23,43,19,137,425,409,1625,2671,4385,3197,25753,
      1,1,7,5,27,17,57,15,383,181,951,2115,5237,1495,9671,
      1,3,3,11,9,1,53,127,375,499,1487,121,1465,3175,24337
    };

    return sobol_minit[max_degree * dim + degree];
  }
};

} // namespace qrng_tables
} // namespace detail

} // namespace random

} // namespace boost

#endif // BOOST_RANDOM_DETAIL_SOBOL_TABLE_HPP
