/*
 * @(#)SkewbGL.c
 *
 * Copyright 2003 - 2010  David A. Bagley, bagleyd@tux.org
 *
 * All rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear in
 * supporting documentation, and that the name of the author not be
 * used in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.
 *
 * This program is distributed in the hope that it will be "useful",
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */

/*-
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appear in all copies and that
 * both that copyright notice and this permission notice appear in
 * supporting documentation.
 *
 * This file is provided AS IS with no warranties of any kind.  The author
 * shall have no liability with respect to the infringement of copyrights,
 * trade secrets or any patents by this file or any part thereof.  In no
 *event will the author be liable for any lost revenue or profits or
 * other special, indirect and consequential damages.
 *
 * Thanks goes also to Brian Paul for making it possible and inexpensive
 * to use OpenGL at home.
 *
 * Based on rubik.c by Marcelo F. Vianna <mfvianna@centroin.com.br>
 */

#ifdef HAVE_OPENGL

/* Methods file for SkewbGL */

#include "SkewbP.h"
#include "SkewbGLP.h"

static float front_shininess[] =
{60.0};
static float front_specular[] =
{0.7, 0.7, 0.7, 1.0};
static float ambient[] =
{0.0, 0.0, 0.0, 1.0};
static float diffuse[] =
{1.0, 1.0, 1.0, 1.0};
static float position0[] =
{1.0, 1.0, 1.0, 0.0};
static float position1[] =
{-1.0, -1.0, 1.0, 0.0};
static float lmodel_ambient[] =
{0.5, 0.5, 0.5, 1.0};
static float lmodel_twoside[] =
{GL_TRUE};

static float face_material[MAX_FACES + 2][4] =
{{1.0, 1.0, 1.0, 1.0},
{1.0, 1.0, 1.0, 1.0},
{1.0, 1.0, 1.0, 1.0},
{1.0, 1.0, 1.0, 1.0},
{1.0, 1.0, 1.0, 1.0},
{1.0, 1.0, 1.0, 1.0},
{0.0, 0.0, 0.0, 1.0},
{0.0, 0.0, 0.0, 1.0}};

/*-
 * Color labels mapping:
 * =====================
 *
 *	  +------+
 *	  |3    0|
 *	  |      |
 *	  | TOP  |
 *	  | (0)  |
 *	  | RED  |
 *	  |2    1|
 * +------+------+------+
 * |3    0|3    0|3    0|
 * |      |      |      |
 * | LEFT |FRONT |RIGHT |
 * | (1)  | (2)  | (3)  |
 * | BLUE |WHITE |GREEN |
 * |2    1|2    1|2    1|
 * +------+------+------+
 *	  |3    0|
 *	  |      |
 *	  |BOTTOM|
 *	  | (4)  |
 *	  | PINK |
 *	  |2    1|
 *	  +------+	 +------+
 *	  |3    0|	 |3 /\ 0|
 *	  |      |	 | /  \ |
 *	  | BACK |	 |/xxxx\|
 *	  | (5)  |	 |\(N) /|
 *	  |YELLOW|	 | \  / |
 *	  |2    1|	 |2 \/ 1|
 *	  +------+	 +------+
 *
 *  Map to 3d
 *  FRONT  => X, Y
 *  BACK   => X, Y
 *  LEFT   => Z, Y
 *  RIGHT  => Z, Y
 *  TOP    => X, Z
 *  BOTTOM => X, Z
 */

static Boolean madeCurrent = False;

#ifdef WINVER
static HGLRC hRC = NULL;
#else
static GLXContext *glXContext = (GLXContext *) NULL;
static Boolean setValuesPuzzleGL(Widget current, Widget request, Widget renew);
static void resizePuzzleGL(SkewbGLWidget w);
static void initializePuzzleGL(Widget request, Widget renew);
static void exposePuzzleGL(Widget renew, XEvent *event, Region region);
static void movePuzzleGLTl(SkewbGLWidget w,
	XEvent *event, char **args, int nArgs);
static void movePuzzleGLTop(SkewbGLWidget w,
	XEvent *event, char **args, int nArgs);
static void movePuzzleGLTr(SkewbGLWidget w,
	XEvent *event, char **args, int nArgs);
static void movePuzzleGLLeft(SkewbGLWidget w,
	XEvent *event, char **args, int nArgs);
static void movePuzzleGLRight(SkewbGLWidget w,
	XEvent *event, char **args, int nArgs);
static void movePuzzleGLBl(SkewbGLWidget w,
	XEvent *event, char **args, int nArgs);
static void movePuzzleGLBottom(SkewbGLWidget w,
	XEvent *event, char **args, int nArgs);
static void movePuzzleGLBr(SkewbGLWidget w,
	XEvent *event, char **args, int nArgs);

static char translationsGL[] =
"<KeyPress>q: Quit()\n\
 Ctrl<KeyPress>C: Quit()\n\
 <KeyPress>osfCancel: Hide()\n\
 <KeyPress>Escape: Hide()\n\
 <KeyPress>osfEscape: Hide()\n\
 Ctrl<KeyPress>[: Hide()\n\
 <KeyPress>0x1B: Hide()\n\
 <KeyPress>0x2E: Speed()\n\
 <KeyPress>0x3E: Speed()\n\
 <KeyPress>0x3C: Slow()\n\
 <KeyPress>0x2C: Slow()\n\
 Shift<KeyPress>2: Sound()\n\
 <KeyPress>F11: MoveCcw()\n\
 <KeyPress>KP_Divide: MoveCcw()\n\
 <KeyPress>R5: MoveCcw()\n\
 <KeyPress>Home: MoveTl()\n\
 <KeyPress>KP_7: MoveTl()\n\
 <KeyPress>R7: MoveTl()\n\
 <KeyPress>Up: MoveTop()\n\
 <KeyPress>osfUp: MoveTop()\n\
 <KeyPress>KP_Up: MoveTop()\n\
 <KeyPress>KP_8: MoveTop()\n\
 <KeyPress>R8: MoveTop()\n\
 <KeyPress>Prior: MoveTr()\n\
 <KeyPress>KP_9: MoveTr()\n\
 <KeyPress>R9: MoveTr()\n\
 <KeyPress>Left: MoveLeft()\n\
 <KeyPress>osfLeft: MoveLeft()\n\
 <KeyPress>KP_Left: MoveLeft()\n\
 <KeyPress>KP_4: MoveLeft()\n\
 <KeyPress>R10: MoveLeft()\n\
 <KeyPress>F12: MoveCw()\n\
 <KeyPress>Begin: MoveCw()\n\
 <KeyPress>KP_5: MoveCw()\n\
 <KeyPress>R11: MoveCw()\n\
 <KeyPress>Right: MoveRight()\n\
 <KeyPress>osfRight: MoveRight()\n\
 <KeyPress>KP_Right: MoveRight()\n\
 <KeyPress>KP_6: MoveRight()\n\
 <KeyPress>R12: MoveRight()\n\
 <KeyPress>End: MoveBl()\n\
 <KeyPress>KP_1: MoveBl()\n\
 <KeyPress>R13: MoveBl()\n\
 <KeyPress>Down: MoveBottom()\n\
 <KeyPress>osfDown: MoveBottom()\n\
 <KeyPress>KP_Down: MoveBottom()\n\
 <KeyPress>KP_2: MoveBottom()\n\
 <KeyPress>R14: MoveBottom()\n\
 <KeyPress>Next: MoveBr()\n\
 <KeyPress>KP_3: MoveBr()\n\
 <KeyPress>R15: MoveBr()\n\
 <Btn1Down>: Select()\n\
 <Btn1Up>: Release()\n\
 <Btn2Down>: PracticeMaybe()\n\
 <Btn2Down>(2+): Practice2()\n\
 <Btn3Down>: RandomizeMaybe()\n\
 <Btn3Down>(2+): Randomize2()\n\
 <Btn4Down>: MoveTop()\n\
 <Btn5Down>: MoveBottom()\n\
 <KeyPress>r: Read()\n\
 <KeyPress>w: Write()\n\
 <KeyPress>u: Undo()\n\
 <KeyPress>r: Redo()\n\
 <KeyPress>c: Clear()\n\
 <KeyPress>z: Randomize()\n\
 <KeyPress>s: Solve()\n\
 <KeyPress>p: Practice()\n\
 <KeyPress>o: Orientize()\n\
 <KeyPress>v: View()\n\
 <EnterWindow>: Enter()\n\
 <LeaveWindow>: Leave()";

static XtActionsRec actionsListGL[] =
{
	{(char *) "Quit", (XtActionProc) quitPuzzle},
	{(char *) "Hide", (XtActionProc) hidePuzzle},
	{(char *) "MoveCcw", (XtActionProc) movePuzzleCcw},
	{(char *) "MoveTl", (XtActionProc) movePuzzleGLTl},
	{(char *) "MoveTop", (XtActionProc) movePuzzleGLTop},
	{(char *) "MoveTr", (XtActionProc) movePuzzleGLTr},
	{(char *) "MoveLeft", (XtActionProc) movePuzzleGLLeft},
	{(char *) "MoveCw", (XtActionProc) movePuzzleCw},
	{(char *) "MoveRight", (XtActionProc) movePuzzleGLRight},
	{(char *) "MoveBl", (XtActionProc) movePuzzleGLBl},
	{(char *) "MoveBottom", (XtActionProc) movePuzzleGLBottom},
	{(char *) "MoveBr", (XtActionProc) movePuzzleGLBr},
	{(char *) "Select", (XtActionProc) selectPuzzle},
	{(char *) "Release", (XtActionProc) releasePuzzle},
	{(char *) "PracticeMaybe", (XtActionProc) practicePuzzleWithQuery},
	{(char *) "Practice2", (XtActionProc) practicePuzzleWithDoubleClick},
	{(char *) "RandomizeMaybe", (XtActionProc) randomizePuzzleWithQuery},
	{(char *) "Randomize2", (XtActionProc) randomizePuzzleWithDoubleClick},
	{(char *) "Read", (XtActionProc) readPuzzle},
	{(char *) "Write", (XtActionProc) writePuzzle},
	{(char *) "Undo", (XtActionProc) undoPuzzle},
	{(char *) "Redo", (XtActionProc) redoPuzzle},
	{(char *) "Clear", (XtActionProc) clearPuzzle},
	{(char *) "Randomize", (XtActionProc) randomizePuzzle},
	{(char *) "Solve", (XtActionProc) solvePuzzle},
	{(char *) "Practice", (XtActionProc) practicePuzzle},
	{(char *) "Orientize", (XtActionProc) orientizePuzzle},
	{(char *) "View", (XtActionProc) viewPuzzle},
	{(char *) "Speed", (XtActionProc) speedUpPuzzle},
	{(char *) "Slow", (XtActionProc) slowDownPuzzle},
	{(char *) "Sound", (XtActionProc) toggleSoundPuzzle},
	{(char *) "Enter", (XtActionProc) enterPuzzle},
	{(char *) "Leave", (XtActionProc) leavePuzzle}
};

static XtResource resourcesGL[] =
{
	{XtNwidth, XtCWidth, XtRDimension, sizeof (Dimension),
	 XtOffset(SkewbWidget, core.width),
	 XtRString, (caddr_t) "250"},
	{XtNheight, XtCHeight, XtRDimension, sizeof (Dimension),
	 XtOffset(SkewbWidget, core.height),
	 XtRString, (caddr_t) "400"},
	{XtNmono, XtCMono, XtRBoolean, sizeof (Boolean),
	 XtOffset(SkewbWidget, skewb.mono),
	 XtRString, (caddr_t) "FALSE"},
	{XtNreverseVideo, XtCReverseVideo, XtRBoolean, sizeof (Boolean),
	 XtOffset(SkewbWidget, skewb.reverse),
	 XtRString, (caddr_t) "FALSE"},
	{XtNforeground, XtCForeground, XtRPixel, sizeof (Pixel),
	 XtOffset(SkewbWidget, skewb.foreground),
	 XtRString, (caddr_t) XtDefaultForeground},
	{XtNbackground, XtCBackground, XtRPixel, sizeof (Pixel),
	 XtOffset(SkewbWidget, skewb.background),
	 XtRString, (caddr_t) "#AEB2C3" /*XtDefaultBackground*/},
	{XtNframeColor, XtCColor, XtRPixel, sizeof (Pixel),
	 XtOffset(SkewbWidget, skewb.frameColor),
	 XtRString, (caddr_t) "Cyan" /*XtDefaultForeground*/},
	{XtNfaceColor0, XtCLabel, XtRString, sizeof (String),
	 XtOffset(SkewbWidget, skewb.faceName[0]),
	 XtRString, (caddr_t) "Red"},
	{XtNfaceColor1, XtCLabel, XtRString, sizeof (String),
	 XtOffset(SkewbWidget, skewb.faceName[1]),
	 XtRString, (caddr_t) "Blue"},
	{XtNfaceColor2, XtCLabel, XtRString, sizeof (String),
	 XtOffset(SkewbWidget, skewb.faceName[2]),
	 XtRString, (caddr_t) "White"},
	{XtNfaceColor3, XtCLabel, XtRString, sizeof (String),
	 XtOffset(SkewbWidget, skewb.faceName[3]),
	 XtRString, (caddr_t) "Green"},
	{XtNfaceColor4, XtCLabel, XtRString, sizeof (String),
	 XtOffset(SkewbWidget, skewb.faceName[4]),
	 XtRString, (caddr_t) "Pink"},
	{XtNfaceColor5, XtCLabel, XtRString, sizeof (String),
	 XtOffset(SkewbWidget, skewb.faceName[5]),
	 XtRString, (caddr_t) "Yellow"},
	{XtNpieceBorder, XtCColor, XtRPixel, sizeof (Pixel),
	 XtOffset(SkewbWidget, skewb.borderColor),
	 XtRString, (caddr_t) "gray25" /*XtDefaultForeground*/},
	{XtNdelay, XtCDelay, XtRInt, sizeof (int),
	 XtOffset(SkewbWidget, skewb.delay),
	 XtRString, (caddr_t) "10"},
	{XtNsound, XtCSound, XtRBoolean, sizeof (Boolean),
	 XtOffset(SkewbWidget, skewb.sound),
	 XtRString, (caddr_t) "FALSE"},
	{XtNmoveSound, XtCMoveSound, XtRString, sizeof (String),
	 XtOffset(SkewbWidget, skewb.moveSound),
	 XtRString, (caddr_t) MOVESOUND},
	{XtNfont, XtCFont, XtRString, sizeof (String),
	 XtOffset(SkewbWidget, skewb.font),
	 XtRString, (caddr_t) "9x15bold"},
	{XtNview, XtCView, XtRInt, sizeof (int),
	 XtOffset(SkewbWidget, skewb.view),
	 XtRString, (caddr_t) "1"},
	{XtNorient, XtCOrient, XtRBoolean, sizeof (Boolean),
	 XtOffset(SkewbWidget, skewb.orient),
	 XtRString, (caddr_t) "FALSE"},	/* DEFAULT_ORIENT */
	{XtNpractice, XtCPractice, XtRBoolean, sizeof (Boolean),
	 XtOffset(SkewbWidget, skewb.practice),
	 XtRString, (caddr_t) "TRUE"}, /* DEFAULT_PRACTICE */
	{XtNuserName, XtCUserName, XtRString, sizeof (String),
	 XtOffset(SkewbWidget, skewb.userName),
	 XtRString, (caddr_t) ""},
	{XtNscoreFile, XtCScoreFile, XtRString, sizeof (String),
	 XtOffset(SkewbWidget, skewb.scoreFile),
	 XtRString, (caddr_t) ""},
	{XtNscoreOnly, XtCBoolean, XtRBoolean, sizeof (Boolean),
	 XtOffset(SkewbWidget, skewb.scoreOnly),
	 XtRString, (caddr_t) "FALSE"},
	{XtNversionOnly, XtCBoolean, XtRBoolean, sizeof (Boolean),
	 XtOffset(SkewbWidget, skewb.versionOnly),
	 XtRString, (caddr_t) "FALSE"},
	{XtNmenu, XtCMenu, XtRInt, sizeof (int),
	 XtOffset(SkewbWidget, skewb.menu),
	 XtRString, (caddr_t) "999"}, /* ACTION_IGNORE */
	{XtNstart, XtCBoolean, XtRBoolean, sizeof (Boolean),
	 XtOffset(SkewbWidget, skewb.started),
	 XtRString, (caddr_t) "FALSE"},
	{XtNcheat, XtCBoolean, XtRBoolean, sizeof (Boolean),
	 XtOffset(SkewbWidget, skewb.cheat),
	 XtRString, (caddr_t) "FALSE"},
	{XtNface, XtCFace, XtRInt, sizeof (int),
	 XtOffset(SkewbWidget, skewb.currentFace),
	 XtRString, (caddr_t) "-1"},
	{XtNpos, XtCPos, XtRInt, sizeof (int),
	 XtOffset(SkewbWidget, skewb.currentPosition),
	 XtRString, (caddr_t) "-1"},
	{XtNdirection, XtCDirection, XtRInt, sizeof (int),
	 XtOffset(SkewbWidget, skewb.currentDirection),
	 XtRString, (caddr_t) "-1"},
	{XtNcontrol, XtCControl, XtRInt, sizeof (int),
	 XtOffset(SkewbWidget, skewb.currentControl),
	 XtRString, (caddr_t) "0"},
	{XtNfast, XtCFast, XtRInt, sizeof (int),
	 XtOffset(SkewbWidget, skewb.currentFast),
	 XtRString, (caddr_t) "1"},
	{XtNpixmapSize, XtCPixmapSize, XtRInt, sizeof (int),
	 XtOffset(SkewbWidget, skewb.pixmapSize),
	 XtRString, (caddr_t) "64"},
	{XtNselectCallback, XtCCallback, XtRCallback, sizeof (caddr_t),
	 XtOffset(SkewbWidget, skewb.select),
	 XtRCallback, (caddr_t) NULL}
};

SkewbGLClassRec skewbGLClassRec =
{
	{
		(WidgetClass) & skewbClassRec,	/* superclass */
		(char *) "SkewbGL",	/* class name */
		sizeof (SkewbGLRec),	/* widget size */
		NULL,		/* class initialize */
		NULL,		/* class part initialize */
		FALSE,		/* class inited */
		(XtInitProc) initializePuzzleGL,	/* initialize */
		NULL,		/* initialize hook */
		XtInheritRealize,	/* realize */
		actionsListGL,	/* actions */
		XtNumber(actionsListGL),	/* num actions */
		resourcesGL,	/* resources */
		XtNumber(resourcesGL),	/* num resources */
		NULLQUARK,	/* xrm class */
		TRUE,		/* compress motion */
		TRUE,		/* compress exposure */
		TRUE,		/* compress enterleave */
		TRUE,		/* visible interest */
		NULL,		/* destroy */
		(XtWidgetProc) resizePuzzleGL,	/* resize */
		(XtExposeProc) exposePuzzleGL,	/* expose */
		(XtSetValuesFunc) setValuesPuzzleGL,	/* set values */
		NULL,		/* set values hook */
		XtInheritSetValuesAlmost,	/* set values almost */
		NULL,		/* get values hook */
		XtInheritAcceptFocus,	/* accept focus */
		XtVersion,	/* version */
		NULL,		/* callback private */
		translationsGL,	/* tm table */
		NULL,		/* query geometry */
		NULL,		/* display accelerator */
		NULL		/* extension */
	},
	{
		0		/* ignore */
	},
	{
		0		/* ignore */
	}
};

WidgetClass skewbGLWidgetClass = (WidgetClass) & skewbGLClassRec;
#endif

#ifdef DEBUG
void
printCube(SkewbGLWidget w)
{
	int face, position;

	for (face = 0; face < MAX_FACES; face++) {
		for (position = 0; position < MAX_CUBES; position++) {
			(void) printf("%d %d  ",
				w->skewb.cubeLoc[face][position].face,
				w->skewb.cubeLoc[face][position].rotation);
		}
		(void) printf("\n");
	}
	(void) printf("\n");
}
#endif

static void
pickColor(SkewbGLWidget w, int C, int mono, float *material)
{
#ifdef WINVER
	struct tagColor {
		int red, green, blue;
	} color;
#define MAX_INTENSITY 0xFF
#else
	XColor color;
#define MAX_INTENSITY 0xFFFF
#endif

	switch (C) {
	case TOP_FACE:
	case LEFT_FACE:
	case FRONT_FACE:
	case RIGHT_FACE:
	case BOTTOM_FACE:
	case BACK_FACE:
#ifdef WINVER
		color.red = GetRValue(w->skewb.faceGC[C]);
		color.green = GetGValue(w->skewb.faceGC[C]);
		color.blue = GetBValue(w->skewb.faceGC[C]);
#else
		color.pixel = w->skewb.faceColor[C];
		XQueryColor(XtDisplay(w), DefaultColormapOfScreen(XtScreen(w)),
			&color);
#endif
		break;
	case BORDER_FACE:
#ifdef WINVER
		color.red = GetRValue(w->skewb.borderGC);
		color.green = GetGValue(w->skewb.borderGC);
		color.blue = GetBValue(w->skewb.borderGC);
#else
		color.pixel = w->skewb.borderColor;
		XQueryColor(XtDisplay(w), DefaultColormapOfScreen(XtScreen(w)),
			&color);
#endif
		break;
	case NO_FACE:
	default:
#ifdef WINVER
		color.red = GetRValue(w->skewb.inverseGC);
		color.green = GetGValue(w->skewb.inverseGC);
		color.blue = GetBValue(w->skewb.inverseGC);
#else
		color.pixel = w->skewb.background;
		XQueryColor(XtDisplay(w), DefaultColormapOfScreen(XtScreen(w)),
			&color);
#endif
		break;
	}
#ifdef DEBUG
	(void) printf("i%d %d %d\n", color.red, color.green, color.blue);
#endif
	if (mono) {
		/* really GrayScale */
		float intensity = 0.3 * color.red +
			0.59 * color.green + 0.11 * color.blue;
#ifdef DEBUG
		(void) printf("m%g\n", intensity);
#endif
		material[0] = (float) intensity / MAX_INTENSITY;
		material[1] = material[0];
		material[2] = material[0];
	} else {
		material[0] = (float) color.red / MAX_INTENSITY;
		material[1] = (float) color.green / MAX_INTENSITY;
		material[2] = (float) color.blue / MAX_INTENSITY;
	}
#ifdef DEBUG
	(void) printf("f%g %g %g\n", material[0], material[1], material[2]);
#endif
}

typedef float Matrix[3][3];
typedef float Vector[3];

static void
matrixMatrix(Matrix a, Matrix b, Matrix x)
{
	int i, j;

	for (i = 0; i < 3; i++) {
		for (j = 0; j < 3; j++) {
			x[i][j] = a[i][0] * b[0][j] +
				a[i][1] * b[1][j] +
				a[i][2] * b[2][j];
		}
	}
}

static void
matrixVector(Matrix a, Vector v, Vector x)
{
	int i;

	for (i = 0; i < 3; i++) {
		x[i] = a[i][0] * v[0] +
			a[i][1] * v[1] +
			a[i][2] * v[2];
	}
}

static void
setMatrix(float theta, float x, float y, float z, Matrix a)
{
	double s = sin(theta * M_PI / 180.0);
	double c = cos(theta * M_PI / 180.0);

	if (x != 0.0 && y == 0.0 && z == 0.0) {
		a[0][0] = 1.0, a[0][1] = 0.0; a[0][2] = 0.0;
		a[1][0] = 0.0, a[1][1] = c; a[1][2] = -s;
		a[2][0] = 0.0, a[2][1] = s; a[2][2] = c;
	} else if (y != 0.0 && x == 0.0 && z == 0.0) {
		a[0][0] = c, a[0][1] = 0.0; a[0][2] = s;
		a[1][0] = 0.0, a[1][1] = 1.0; a[1][2] = 0.0;
		a[2][0] = -s, a[2][1] = 0.0; a[2][2] = c;
	} else if (z != 0.0 && x == 0.0 && y == 0.0) {
		a[0][0] = c, a[0][1] = -s; a[0][2] = 0.0;
		a[1][0] = s, a[1][1] = c; a[1][2] = 0.0;
		a[2][0] = 0.0, a[2][1] = 0.0; a[2][2] = 1.0;
	}
}

static void
drawStickerlessCubit(void)
{
	glBegin(GL_QUADS);
	glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE,
		face_material[BORDER_FACE]);
	/* Edge of cubit */
	glNormal3f(1.0, 1.0, 0.0);
	glVertex3f(CUBE_ROUND, CUBE_LENGTH, -CUBE_ROUND);
	glVertex3f(CUBE_ROUND, CUBE_LENGTH, CUBE_ROUND);
	glVertex3f(CUBE_LENGTH, CUBE_ROUND, CUBE_ROUND);
	glVertex3f(CUBE_LENGTH, CUBE_ROUND, -CUBE_ROUND);
	glNormal3f(0.0, 1.0, 1.0);
	glVertex3f(-CUBE_ROUND, CUBE_ROUND, CUBE_LENGTH);
	glVertex3f(CUBE_ROUND, CUBE_ROUND, CUBE_LENGTH);
	glVertex3f(CUBE_ROUND, CUBE_LENGTH, CUBE_ROUND);
	glVertex3f(-CUBE_ROUND, CUBE_LENGTH, CUBE_ROUND);
	glNormal3f(1.0, 0.0, 1.0);
	glVertex3f(CUBE_LENGTH, -CUBE_ROUND, CUBE_ROUND);
	glVertex3f(CUBE_LENGTH, CUBE_ROUND, CUBE_ROUND);
	glVertex3f(CUBE_ROUND, CUBE_ROUND, CUBE_LENGTH);
	glVertex3f(CUBE_ROUND, -CUBE_ROUND, CUBE_LENGTH);
	glEnd();
	glBegin(GL_TRIANGLES);
	glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE,
		face_material[BORDER_FACE]);
	/* Put sticker here */
	glNormal3f(0.0, 0.0, 1.0);
	glVertex3f(CUBE_ROUND, -CUBE_ROUND, CUBE_LENGTH);
	glVertex3f(CUBE_ROUND, CUBE_ROUND, CUBE_LENGTH);
	glVertex3f(-CUBE_ROUND, CUBE_ROUND, CUBE_LENGTH);
	glNormal3f(1.0, 0.0, 0.0);
	glVertex3f(CUBE_LENGTH, CUBE_ROUND, -CUBE_ROUND);
	glVertex3f(CUBE_LENGTH, CUBE_ROUND, CUBE_ROUND);
	glVertex3f(CUBE_LENGTH, -CUBE_ROUND, CUBE_ROUND);
	glNormal3f(0.0, 1.0, 0.0);
	glVertex3f(-CUBE_ROUND, CUBE_LENGTH, CUBE_ROUND);
	glVertex3f(CUBE_ROUND, CUBE_LENGTH, CUBE_ROUND);
	glVertex3f(CUBE_ROUND, CUBE_LENGTH, -CUBE_ROUND);
	/* Corner of cubit */
	glNormal3f(1.0, 1.0, 1.0);
	glVertex3f(CUBE_ROUND, CUBE_ROUND, CUBE_LENGTH);
	glVertex3f(CUBE_LENGTH, CUBE_ROUND, CUBE_ROUND);
	glVertex3f(CUBE_ROUND, CUBE_LENGTH, CUBE_ROUND);
	/* Sharper corners of cubit */
	glNormal3f(-1.0, 1.0, 1.0);
	glVertex3f(-CUBE_LENGTH, CUBE_ROUND, CUBE_LENGTH);
	glVertex3f(-CUBE_LENGTH, CUBE_LENGTH, CUBE_ROUND);
	glVertex3f(-CUBE_LENGTH, CUBE_ROUND, CUBE_ROUND);
	glNormal3f(1.0, -1.0, 1.0);
	glVertex3f(CUBE_ROUND, -CUBE_LENGTH, CUBE_LENGTH);
	glVertex3f(CUBE_ROUND, -CUBE_LENGTH, CUBE_ROUND);
	glVertex3f(CUBE_LENGTH, -CUBE_LENGTH, CUBE_ROUND);
	glNormal3f(1.0, 1.0, -1.0);
	glVertex3f(CUBE_LENGTH, CUBE_ROUND, -CUBE_LENGTH);
	glVertex3f(CUBE_ROUND, CUBE_ROUND, -CUBE_LENGTH);
	glVertex3f(CUBE_ROUND, CUBE_LENGTH, -CUBE_LENGTH);
	glEnd();
	glBegin(GL_POLYGON);
	glNormal3f(-1.0, 1.0, 1.0);
	glVertex3f(-CUBE_ROUND, CUBE_ROUND, CUBE_LENGTH);
	glVertex3f(-CUBE_ROUND, CUBE_LENGTH, CUBE_ROUND);
	glVertex3f(CUBE_ROUND, CUBE_LENGTH, -CUBE_ROUND);
	glVertex3f(CUBE_LENGTH, CUBE_ROUND, -CUBE_ROUND);
	glVertex3f(CUBE_LENGTH, -CUBE_ROUND, CUBE_ROUND);
	glVertex3f(CUBE_ROUND, -CUBE_ROUND, CUBE_LENGTH);
	glEnd();
}

static void
drawStickerlessFacit(void)
{
	glBegin(GL_QUADS);
	glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE,
		face_material[BORDER_FACE]);
	/* Edge of facit */
	glNormal3f(0.0, 0.0, 1.0);
	glVertex3f(-FACEROUND, 0.0, FACELEN);
	glVertex3f(0.0, -FACEROUND, FACELEN);
	glVertex3f(FACEROUND, 0.0, FACELEN);
	glVertex3f(0.0, FACEROUND, FACELEN);
	glEnd();
	glBegin(GL_TRIANGLES);
	glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE,
		face_material[BORDER_FACE]);
	glNormal3f(0.0, 0.0, 1.0);
	glVertex3f(0.0, -FACEROUND, FACELEN);
	glVertex3f(-FACEROUND, 0.0, FACELEN);
	glVertex3f(0.0, 0.0, 0.0);
	glVertex3f(FACEROUND, 0.0, FACELEN);
	glVertex3f(0.0, -FACEROUND, FACELEN);
	glVertex3f(0.0, 0.0, 0.0);
	glVertex3f(0.0, FACEROUND, FACELEN);
	glVertex3f(FACEROUND, 0.0, FACELEN);
	glVertex3f(0.0, 0.0, 0.0);
	glVertex3f(-FACEROUND, 0.0, FACELEN);
	glVertex3f(0.0, FACEROUND, FACELEN);
	glVertex3f(0.0, 0.0, 0.0);
	glEnd();
}

static void
drawOrientLine(SkewbGLWidget w, int face, int rotation,
		float size, float x, float y, float z)
{
	int err = 0;
	char *buf1, *buf2;
	float stickerLong = STICKER_LONG * size;

	glPushMatrix();
	glTranslatef(x, y, z);
	glBegin(GL_QUADS);
	glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE,
		face_material[BORDER_FACE]);
	switch (face) {
	case 0:
		switch (rotation) {
		case 0:
			glVertex3f(-CUT, 0.0, -CUT);
			glVertex3f(CUT, 0.0, -CUT);
			glVertex3f(CUT, 0.0, -stickerLong);
			glVertex3f(-CUT, 0.0, -stickerLong);
			break;
		case 1:
			glVertex3f(CUT, 0.0, -CUT);
			glVertex3f(CUT, 0.0, CUT);
			glVertex3f(stickerLong, 0.0, CUT);
			glVertex3f(stickerLong, 0.0, -CUT);
			break;
		case 2:
			glVertex3f(CUT, 0.0, CUT);
			glVertex3f(-CUT, 0.0, CUT);
			glVertex3f(-CUT, 0.0, stickerLong);
			glVertex3f(CUT, 0.0, stickerLong);
			break;
		case 3:
			glVertex3f(-CUT, 0.0, CUT);
			glVertex3f(-CUT, 0.0, -CUT);
			glVertex3f(-stickerLong, 0.0, -CUT);
			glVertex3f(-stickerLong, 0.0, CUT);
			break;
		default:
			err = 1;
		}
		break;
	case 1:
		switch (rotation) {
		case 0:
			glVertex3f(0.0, CUT, -CUT);
			glVertex3f(0.0, CUT, CUT);
			glVertex3f(0.0, stickerLong, CUT);
			glVertex3f(0.0, stickerLong, -CUT);
			break;
		case 1:
			glVertex3f(0.0, CUT, CUT);
			glVertex3f(0.0, -CUT, CUT);
			glVertex3f(0.0, -CUT, stickerLong);
			glVertex3f(0.0, CUT, stickerLong);
			break;
		case 2:
			glVertex3f(0.0, -CUT, CUT);
			glVertex3f(0.0, -CUT, -CUT);
			glVertex3f(0.0, -stickerLong, -CUT);
			glVertex3f(0.0, -stickerLong, CUT);
			break;
		case 3:
			glVertex3f(0.0, -CUT, -CUT);
			glVertex3f(0.0, CUT, -CUT);
			glVertex3f(0.0, CUT, -stickerLong);
			glVertex3f(0.0, -CUT, -stickerLong);
			break;
		default:
			err = 1;
		}
		break;
	case 2:
		switch (rotation) {
		case 0:
			glVertex3f(-CUT, CUT, 0.0);
			glVertex3f(CUT, CUT, 0.0);
			glVertex3f(CUT, stickerLong, 0.0);
			glVertex3f(-CUT, stickerLong, 0.0);
			break;
		case 1:
			glVertex3f(CUT, CUT, 0.0);
			glVertex3f(CUT, -CUT, 0.0);
			glVertex3f(stickerLong, -CUT, 0.0);
			glVertex3f(stickerLong, CUT, 0.0);
			break;
		case 2:
			glVertex3f(CUT, -CUT, 0.0);
			glVertex3f(-CUT, -CUT, 0.0);
			glVertex3f(-CUT, -stickerLong, 0.0);
			glVertex3f(CUT, -stickerLong, 0.0);
			break;
		case 3:
			glVertex3f(-CUT, -CUT, 0.0);
			glVertex3f(-CUT, CUT, 0.0);
			glVertex3f(-stickerLong, CUT, 0.0);
			glVertex3f(-stickerLong, -CUT, 0.0);
			break;
		default:
			err = 1;
		}
		break;
	case 3:
		switch (rotation) {
		case 0:
			glVertex3f(0.0, CUT, -CUT);
			glVertex3f(0.0, stickerLong, -CUT);
			glVertex3f(0.0, stickerLong, CUT);
			glVertex3f(0.0, CUT, CUT);
			break;
		case 1:
			glVertex3f(0.0, -CUT, -CUT);
			glVertex3f(0.0, -CUT, -stickerLong);
			glVertex3f(0.0, CUT, -stickerLong);
			glVertex3f(0.0, CUT, -CUT);
			break;
		case 2:
			glVertex3f(0.0, -CUT, CUT);
			glVertex3f(0.0, -stickerLong, CUT);
			glVertex3f(0.0, -stickerLong, -CUT);
			glVertex3f(0.0, -CUT, -CUT);
			break;
		case 3:
			glVertex3f(0.0, CUT, CUT);
			glVertex3f(0.0, CUT, stickerLong);
			glVertex3f(0.0, -CUT, stickerLong);
			glVertex3f(0.0, -CUT, CUT);
			break;
		default:
			err = 1;
		}
		break;
	case 4:
		switch (rotation) {
		case 0:
			glVertex3f(CUT, 0.0, CUT);
			glVertex3f(CUT, 0.0, stickerLong);
			glVertex3f(-CUT, 0.0, stickerLong);
			glVertex3f(-CUT, 0.0, CUT);
			break;
		case 1:
			glVertex3f(CUT, 0.0, -CUT);
			glVertex3f(stickerLong, 0.0, -CUT);
			glVertex3f(stickerLong, 0.0, CUT);
			glVertex3f(CUT, 0.0, CUT);
			break;
		case 2:
			glVertex3f(-CUT, 0.0, -CUT);
			glVertex3f(-CUT, 0.0, -stickerLong);
			glVertex3f(CUT, 0.0, -stickerLong);
			glVertex3f(CUT, 0.0, -CUT);
			break;
		case 3:
			glVertex3f(-CUT, 0.0, CUT);
			glVertex3f(-stickerLong, 0.0, CUT);
			glVertex3f(-stickerLong, 0.0, -CUT);
			glVertex3f(-CUT, 0.0, -CUT);
			break;
		default:
			err = 1;
		}
		break;
	case 5:
		switch (rotation) {
		case 0:
			glVertex3f(CUT, -CUT, 0.0);
			glVertex3f(CUT, -stickerLong, 0.0);
			glVertex3f(-CUT, -stickerLong, 0.0);
			glVertex3f(-CUT, -CUT, 0.0);
			break;
		case 1:
			glVertex3f(CUT, CUT, 0.0);
			glVertex3f(stickerLong, CUT, 0.0);
			glVertex3f(stickerLong, -CUT, 0.0);
			glVertex3f(CUT, -CUT, 0.0);
			break;
		case 2:
			glVertex3f(-CUT, CUT, 0.0);
			glVertex3f(-CUT, stickerLong, 0.0);
			glVertex3f(CUT, stickerLong, 0.0);
			glVertex3f(CUT, CUT, 0.0);
			break;
		case 3:
			glVertex3f(-CUT, -CUT, 0.0);
			glVertex3f(-stickerLong, -CUT, 0.0);
			glVertex3f(-stickerLong, CUT, 0.0);
			glVertex3f(-CUT, CUT, 0.0);
			break;
		default:
			err = 1;
		}
		break;
	}
	glEnd();
	glPopMatrix();
	if (err == 1) {
		intCat(&buf1, "draw_orient_line: face ", face);
		stringCat(&buf2, buf1, ", rotation ");
		free(buf1);
		intCat(&buf1, buf2, rotation);
		free(buf2);
		DISPLAY_WARNING(buf1);
		free(buf1);
	}
}

static void
drawFacit(SkewbGLWidget w, int face)
{
	glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE,
		face_material[(int) w->skewb.cubeLoc[face][MAX_ORIENT].face]);
	switch (face) {
	case TOP_FACE:
		glBegin(GL_QUADS);
		glNormal3f(0.0, 1.0, 0.0);
		glVertex3f(0.0, FSTICKER_DEPTH, -FSTICKER_LONG);
		glVertex3f(-FSTICKER_LONG, FSTICKER_DEPTH, 0.0);
		glVertex3f(0.0, FSTICKER_DEPTH, FSTICKER_LONG);
		glVertex3f(FSTICKER_LONG, FSTICKER_DEPTH, 0.0);
		glEnd();
		if (w->skewb.currentFace == face &&
				w->skewb.currentPosition == MAX_ORIENT) {
			glBegin(GL_POLYGON);
			glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE,
				face_material[BORDER_FACE]);
			glNormal3f(0.0, 1.0, 0.0);
			glVertex3f(0.0, FSELECT_DEPTH, -FSELECT_LONG);
			glVertex3f(-FSELECT_LONG, FSELECT_DEPTH, 0.0);
			glVertex3f(0.0, FSELECT_DEPTH, FSELECT_LONG);
			glVertex3f(FSELECT_LONG, FSELECT_DEPTH, 0.0);
			glEnd();
		}
		if (w->skewb.orient) {
			drawOrientLine(w, face,
				w->skewb.cubeLoc[face][MAX_ORIENT].rotation,
				2.0,
				0.0, FCUT_DEPTH, 0.0);
		}
		break;
	case LEFT_FACE:
		glBegin(GL_QUADS);
		glNormal3f(-1.0, 0.0, 0.0);
		glVertex3f(-FSTICKER_DEPTH, 0.0, -FSTICKER_LONG);
		glVertex3f(-FSTICKER_DEPTH, -FSTICKER_LONG, 0.0);
		glVertex3f(-FSTICKER_DEPTH, 0.0, FSTICKER_LONG);
		glVertex3f(-FSTICKER_DEPTH, FSTICKER_LONG, 0.0);
		glEnd();
		if (w->skewb.currentFace == face &&
				w->skewb.currentPosition == MAX_ORIENT) {
			glBegin(GL_QUADS);
			glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE,
				face_material[BORDER_FACE]);
			glNormal3f(-1.0, 0.0, 0.0);
			glVertex3f(-FSELECT_DEPTH, 0.0, -FSELECT_LONG);
			glVertex3f(-FSELECT_DEPTH, -FSELECT_LONG, 0.0);
			glVertex3f(-FSELECT_DEPTH, 0.0, FSELECT_LONG);
			glVertex3f(-FSELECT_DEPTH, FSELECT_LONG, 0.0);
			glEnd();
		}
		if (w->skewb.orient) {
			drawOrientLine(w, face,
				w->skewb.cubeLoc[face][MAX_ORIENT].rotation,
				2.0,
				-FCUT_DEPTH, 0.0, 0.0);
		}
		break;
	case FRONT_FACE:
		glBegin(GL_QUADS);
		glNormal3f(0.0, 0.0, 1.0);
		glVertex3f(-FSTICKER_LONG, 0.0, FSTICKER_DEPTH);
		glVertex3f(0.0, -FSTICKER_LONG, FSTICKER_DEPTH);
		glVertex3f(FSTICKER_LONG, 0.0, FSTICKER_DEPTH);
		glVertex3f(0.0, FSTICKER_LONG, FSTICKER_DEPTH);
		glEnd();
		if (w->skewb.currentFace == face &&
				w->skewb.currentPosition == MAX_ORIENT) {
			glBegin(GL_QUADS);
			glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE,
				face_material[BORDER_FACE]);
			glNormal3f(0.0, 0.0, 1.0);
			glVertex3f(-FSELECT_LONG, 0.0, FSELECT_DEPTH);
			glVertex3f(0.0, -FSELECT_LONG, FSELECT_DEPTH);
			glVertex3f(FSELECT_LONG, 0.0, FSELECT_DEPTH);
			glVertex3f(0.0, FSELECT_LONG, FSELECT_DEPTH);
			glEnd();
		}
		if (w->skewb.orient) {
			drawOrientLine(w, face,
				w->skewb.cubeLoc[face][MAX_ORIENT].rotation,
				2.0,
				0.0, 0.0, FCUT_DEPTH);
		}
		break;
	case RIGHT_FACE:
		glBegin(GL_QUADS);
		glNormal3f(1.0, 0.0, 0.0);
		glVertex3f(FSTICKER_DEPTH, FSTICKER_LONG, 0.0);
		glVertex3f(FSTICKER_DEPTH, 0.0, FSTICKER_LONG);
		glVertex3f(FSTICKER_DEPTH, -FSTICKER_LONG, 0.0);
		glVertex3f(FSTICKER_DEPTH, 0.0, -FSTICKER_LONG);
		glEnd();
		if (w->skewb.currentFace == face &&
				w->skewb.currentPosition == MAX_ORIENT) {
			glBegin(GL_QUADS);
			glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE,
				face_material[BORDER_FACE]);
			glNormal3f(1.0, 0.0, 0.0);
			glVertex3f(FSELECT_DEPTH, FSELECT_LONG, 0.0);
			glVertex3f(FSELECT_DEPTH, 0.0, FSELECT_LONG);
			glVertex3f(FSELECT_DEPTH, -FSELECT_LONG, 0.0);
			glVertex3f(FSELECT_DEPTH, 0.0, -FSELECT_LONG);
			glEnd();
		}
		if (w->skewb.orient) {
			drawOrientLine(w, face,
				w->skewb.cubeLoc[face][MAX_ORIENT].rotation,
				2.0,
				FCUT_DEPTH, 0.0, 0.0);
		}
		break;
	case BOTTOM_FACE:
		glBegin(GL_QUADS);
		glNormal3f(0.0, -1.0, 0.0);
		glVertex3f(FSTICKER_LONG, -FSTICKER_DEPTH, 0.0);
		glVertex3f(0.0, -FSTICKER_DEPTH, FSTICKER_LONG);
		glVertex3f(-FSTICKER_LONG, -FSTICKER_DEPTH, 0.0);
		glVertex3f(0.0, -FSTICKER_DEPTH, -FSTICKER_LONG);
		glEnd();
		if (w->skewb.currentFace == face &&
				w->skewb.currentPosition == MAX_ORIENT) {
			glBegin(GL_QUADS);
			glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE,
				face_material[BORDER_FACE]);
			glNormal3f(0.0, -1.0, 0.0);
			glVertex3f(FSELECT_LONG, -FSELECT_DEPTH, 0.0);
			glVertex3f(0.0, -FSELECT_DEPTH, FSELECT_LONG);
			glVertex3f(-FSELECT_LONG, -FSELECT_DEPTH, 0.0);
			glVertex3f(0.0, -FSELECT_DEPTH, -FSELECT_LONG);
			glEnd();
		}
		if (w->skewb.orient) {
			drawOrientLine(w, face,
				w->skewb.cubeLoc[face][MAX_ORIENT].rotation,
				2.0,
				0.0, -FCUT_DEPTH, 0.0);
		}
		break;
	case BACK_FACE:
		glBegin(GL_QUADS);
		glNormal3f(0.0, 0.0, -1.0);
		glVertex3f(0.0, FSTICKER_LONG, -FSTICKER_DEPTH);
		glVertex3f(FSTICKER_LONG, 0.0, -FSTICKER_DEPTH);
		glVertex3f(0.0, -FSTICKER_LONG, -FSTICKER_DEPTH);
		glVertex3f(-FSTICKER_LONG, 0.0, -FSTICKER_DEPTH);
		glEnd();
		if (w->skewb.currentFace == face &&
				w->skewb.currentPosition == MAX_ORIENT) {
			glBegin(GL_QUADS);
			glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE,
				face_material[BORDER_FACE]);
			glNormal3f(0.0, 0.0, -1.0);
			glVertex3f(0.0, FSELECT_LONG, -FSELECT_DEPTH);
			glVertex3f(FSELECT_LONG, 0.0, -FSELECT_DEPTH);
			glVertex3f(0.0, -FSELECT_LONG, -FSELECT_DEPTH);
			glVertex3f(-FSELECT_LONG, 0.0, -FSELECT_DEPTH);
			glEnd();
		}
		if (w->skewb.orient) {
			drawOrientLine(w, face,
				w->skewb.cubeLoc[face][MAX_ORIENT].rotation,
				2.0,
				0.0, 0.0, -FCUT_DEPTH);
		}
		break;
	}
}

static int
positionCubit(SkewbGLWidget w, int face,
	int back, int front, int left, int right, int bottom, int top) {

	if (back != NO_FACE && face == BACK_FACE) { /* Yellow */
		if (right != NO_FACE && bottom != NO_FACE)
			return 0;
		if (right != NO_FACE && top != NO_FACE)
			return 1;
		if (left != NO_FACE && top != NO_FACE)
			return 2;
		if (left != NO_FACE && bottom != NO_FACE)
			return 3;
	}
	if (front != NO_FACE && face == FRONT_FACE) { /* White */
		if (right != NO_FACE && top != NO_FACE)
			return 0;
		if (right != NO_FACE && bottom != NO_FACE)
			return 1;
		if (left != NO_FACE && bottom != NO_FACE)
			return 2;
		if (left != NO_FACE && top != NO_FACE)
			return 3;
	}
	if (left != NO_FACE && face == LEFT_FACE) { /* Blue */
		if (front != NO_FACE && top != NO_FACE)
			return 0;
		if (front != NO_FACE && bottom != NO_FACE)
			return 1;
		if (back != NO_FACE && bottom != NO_FACE)
			return 2;
		if (back != NO_FACE && top != NO_FACE)
			return 3;
	}
	if (right != NO_FACE && face == RIGHT_FACE) { /* Green */
		if (back != NO_FACE && top != NO_FACE)
			return 0;
		if (back != NO_FACE && bottom != NO_FACE)
			return 1;
		if (front != NO_FACE && bottom != NO_FACE)
			return 2;
		if (front != NO_FACE && top != NO_FACE)
			return 3;
	}
	if (bottom != NO_FACE && face == BOTTOM_FACE) { /* Pink */
		if (right != NO_FACE && front != NO_FACE)
			return 0;
		if (right != NO_FACE && back != NO_FACE)
			return 1;
		if (left != NO_FACE && back != NO_FACE)
			return 2;
		if (left != NO_FACE && front != NO_FACE)
			return 3;
	}
	if (top != NO_FACE && face == TOP_FACE) { /* Red */
		if (right != NO_FACE && back != NO_FACE)
			return 0;
		if (right != NO_FACE && front != NO_FACE)
			return 1;
		if (left != NO_FACE && front != NO_FACE)
			return 2;
		if (left != NO_FACE && back != NO_FACE)
			return 3;
	}
	return MAX_ORIENT;
}

static void
drawCubit(SkewbGLWidget w,
		int back, int front, int left, int right, int bottom, int top)
{
	int face, position;

	if (back != NO_FACE) { /* Yellow */
		face = BACK_FACE;
		position = positionCubit(w, face, back, front, left, right, bottom, top);
		glBegin(GL_POLYGON);
		glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE,
			face_material[back]);
		glNormal3f(0.0, 0.0, -1.0);
		if (top != NO_FACE) {
			glVertex3f(-STICKER_SHORT, STICKER_LONG, -STICKER_DEPTH);
			glVertex3f(STICKER_SHORT, STICKER_LONG, -STICKER_DEPTH);
		}
		if (left != NO_FACE) {
			glVertex3f(-STICKER_LONG, -STICKER_SHORT, -STICKER_DEPTH);
			glVertex3f(-STICKER_LONG, STICKER_SHORT, -STICKER_DEPTH);
		}
		if (bottom != NO_FACE) {
			glVertex3f(STICKER_SHORT, -STICKER_LONG, -STICKER_DEPTH);
			glVertex3f(-STICKER_SHORT, -STICKER_LONG, -STICKER_DEPTH);
		}
		if (right != NO_FACE) {
			glVertex3f(STICKER_LONG, STICKER_SHORT, -STICKER_DEPTH);
			glVertex3f(STICKER_LONG, -STICKER_SHORT, -STICKER_DEPTH);
		}
		glEnd();
		if (w->skewb.currentFace == face &&
				w->skewb.currentPosition == position) {
			glBegin(GL_POLYGON);
			glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE,
				face_material[BORDER_FACE]);
			glNormal3f(0.0, 0.0, -1.0);
			if (top != NO_FACE) {
				glVertex3f(-SELECT_SHORT, SELECT_LONG, -SELECT_DEPTH);
				glVertex3f(SELECT_SHORT, SELECT_LONG, -SELECT_DEPTH);
			}
			if (left != NO_FACE) {
				glVertex3f(-SELECT_LONG, -SELECT_SHORT, -SELECT_DEPTH);
				glVertex3f(-SELECT_LONG, SELECT_SHORT, -SELECT_DEPTH);
			}
			if (bottom != NO_FACE) {
				glVertex3f(SELECT_SHORT, -SELECT_LONG, -SELECT_DEPTH);
				glVertex3f(-SELECT_SHORT, -SELECT_LONG, -SELECT_DEPTH);
			}
			if (right != NO_FACE) {
				glVertex3f(SELECT_LONG, SELECT_SHORT, -SELECT_DEPTH);
				glVertex3f(SELECT_LONG, -SELECT_SHORT, -SELECT_DEPTH);
			}
			glEnd();
		}
		if (w->skewb.orient) {
			if (right != NO_FACE && bottom != NO_FACE) {
				drawOrientLine(w, face,
					w->skewb.cubeLoc[face][0].rotation,
					0.5,
					0.1, -0.5 * STICKER_SHORT, -CUT_DEPTH);
			}
			if (right != NO_FACE && top != NO_FACE) {
				drawOrientLine(w, face,
					w->skewb.cubeLoc[face][1].rotation,
					0.5,
					0.1, 0.5 * STICKER_SHORT, -CUT_DEPTH);
			}
			if (left != NO_FACE && top != NO_FACE) {
				drawOrientLine(w, face,
					w->skewb.cubeLoc[face][2].rotation,
					0.5,
					-0.5 * STICKER_SHORT, 0.1, -CUT_DEPTH);
			}
			if (left != NO_FACE && bottom != NO_FACE) {
				drawOrientLine(w, face,
					w->skewb.cubeLoc[face][3].rotation,
					0.5,
					-0.5 * STICKER_SHORT, -0.1, -CUT_DEPTH);
			}
		}
	}
	if (front != NO_FACE) { /* White */
		face = FRONT_FACE;
		position = positionCubit(w, face, back, front, left, right, bottom, top);
		glBegin(GL_POLYGON);
		glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE,
			face_material[front]);
		glNormal3f(0.0, 0.0, 1.0);
		if (top != NO_FACE) {
			glVertex3f(STICKER_SHORT, STICKER_LONG, STICKER_DEPTH);
			glVertex3f(-STICKER_SHORT, STICKER_LONG, STICKER_DEPTH);
		}
		if (left != NO_FACE) {
			glVertex3f(-STICKER_LONG, STICKER_SHORT, STICKER_DEPTH);
			glVertex3f(-STICKER_LONG, -STICKER_SHORT, STICKER_DEPTH);
		}
		if (bottom != NO_FACE) {
			glVertex3f(-STICKER_SHORT, -STICKER_LONG, STICKER_DEPTH);
			glVertex3f(STICKER_SHORT, -STICKER_LONG, STICKER_DEPTH);
		}
		if (right != NO_FACE) {
			glVertex3f(STICKER_LONG, -STICKER_SHORT, STICKER_DEPTH);
			glVertex3f(STICKER_LONG, STICKER_SHORT, STICKER_DEPTH);
		}
		glEnd();
		if (w->skewb.currentFace == face &&
				w->skewb.currentPosition == position) {
			glBegin(GL_POLYGON);
			glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE,
				face_material[BORDER_FACE]);
			glNormal3f(0.0, 0.0, 1.0);
			if (top != NO_FACE) {
				glVertex3f(SELECT_SHORT, SELECT_LONG, SELECT_DEPTH);
				glVertex3f(-SELECT_SHORT, SELECT_LONG, SELECT_DEPTH);
			}
			if (left != NO_FACE) {
				glVertex3f(-SELECT_LONG, SELECT_SHORT, SELECT_DEPTH);
				glVertex3f(-SELECT_LONG, -SELECT_SHORT, SELECT_DEPTH);
			}
			if (bottom != NO_FACE) {
				glVertex3f(-SELECT_SHORT, -SELECT_LONG, SELECT_DEPTH);
				glVertex3f(SELECT_SHORT, -SELECT_LONG, SELECT_DEPTH);
			}
			if (right != NO_FACE) {
				glVertex3f(SELECT_LONG, -SELECT_SHORT, SELECT_DEPTH);
				glVertex3f(SELECT_LONG, SELECT_SHORT, SELECT_DEPTH);
			}
			glEnd();
		}
		if (w->skewb.orient) {
			if (right != NO_FACE && top != NO_FACE) {
				drawOrientLine(w, face,
					w->skewb.cubeLoc[face][0].rotation,
					0.5,
					0.1, 0.5 * STICKER_SHORT, CUT_DEPTH);
			}
			if (right != NO_FACE && bottom != NO_FACE) {
				drawOrientLine(w, face,
					w->skewb.cubeLoc[face][1].rotation,
					0.5,
					0.1, -0.5 * STICKER_SHORT, CUT_DEPTH);
			}
			if (left != NO_FACE && bottom != NO_FACE) {
				drawOrientLine(w, face,
					w->skewb.cubeLoc[face][2].rotation,
					0.5,
					-0.5 * STICKER_SHORT, -0.1, CUT_DEPTH);
			}
			if (left != NO_FACE && top != NO_FACE) {
				drawOrientLine(w, face,
					w->skewb.cubeLoc[face][3].rotation,
					0.5,
					-0.5 * STICKER_SHORT, 0.1, CUT_DEPTH);
			}
		}
	}
	if (left != NO_FACE) { /* Blue */
		face = LEFT_FACE;
		position = positionCubit(w, face, back, front, left, right, bottom, top);
		glBegin(GL_POLYGON);
		glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE,
			face_material[left]);
		glNormal3f(-1.0, 0.0, 0.0);
		if (front != NO_FACE) {
			glVertex3f(-STICKER_DEPTH, -STICKER_SHORT, STICKER_LONG);
			glVertex3f(-STICKER_DEPTH, STICKER_SHORT, STICKER_LONG);
		}
		if (top != NO_FACE) {
			glVertex3f(-STICKER_DEPTH, STICKER_LONG, STICKER_SHORT);
			glVertex3f(-STICKER_DEPTH, STICKER_LONG, -STICKER_SHORT);
		}
		if (back != NO_FACE) {
			glVertex3f(-STICKER_DEPTH, STICKER_SHORT, -STICKER_LONG);
			glVertex3f(-STICKER_DEPTH, -STICKER_SHORT, -STICKER_LONG);
		}
		if (bottom != NO_FACE) {
			glVertex3f(-STICKER_DEPTH, -STICKER_LONG, -STICKER_SHORT);
			glVertex3f(-STICKER_DEPTH, -STICKER_LONG, STICKER_SHORT);
		}
		glEnd();
		if (w->skewb.currentFace == face &&
				w->skewb.currentPosition == position) {
			glBegin(GL_POLYGON);
			glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE,
				face_material[BORDER_FACE]);
			glNormal3f(-1.0, 0.0, 0.0);
			if (front != NO_FACE) {
				glVertex3f(-SELECT_DEPTH, -SELECT_SHORT, SELECT_LONG);
				glVertex3f(-SELECT_DEPTH, SELECT_SHORT, SELECT_LONG);
			}
			if (top != NO_FACE) {
				glVertex3f(-SELECT_DEPTH, SELECT_LONG, SELECT_SHORT);
				glVertex3f(-SELECT_DEPTH, SELECT_LONG, -SELECT_SHORT);
			}
			if (back != NO_FACE) {
				glVertex3f(-SELECT_DEPTH, SELECT_SHORT, -SELECT_LONG);
				glVertex3f(-SELECT_DEPTH, -SELECT_SHORT, -SELECT_LONG);
			}
			if (bottom != NO_FACE) {
				glVertex3f(-SELECT_DEPTH, -SELECT_LONG, -SELECT_SHORT);
				glVertex3f(-SELECT_DEPTH, -SELECT_LONG, SELECT_SHORT);
			}
			glEnd();
		}
		if (w->skewb.orient) {
			if (front != NO_FACE && top != NO_FACE) {
				drawOrientLine(w, face,
					w->skewb.cubeLoc[face][0].rotation,
					0.5,
					-CUT_DEPTH, 0.1, 0.5 * STICKER_SHORT);
			}
			if (front != NO_FACE && bottom != NO_FACE) {
				drawOrientLine(w, face,
					w->skewb.cubeLoc[face][1].rotation,
					0.5,
					-CUT_DEPTH, -0.1, 0.5 * STICKER_SHORT);
			}
			if (back != NO_FACE && bottom != NO_FACE) {
				drawOrientLine(w, face,
					w->skewb.cubeLoc[face][2].rotation,
					0.5,
					-CUT_DEPTH, -0.5 * STICKER_SHORT, -0.1);
			}
			if (back != NO_FACE && top != NO_FACE) {
				drawOrientLine(w, face,
					w->skewb.cubeLoc[face][3].rotation,
					0.5,
					-CUT_DEPTH, 0.5 * STICKER_SHORT, -0.1);
			}
		}
	}
	if (right != NO_FACE) { /* Green */
		face = RIGHT_FACE;
		position = positionCubit(w, face, back, front, left, right, bottom, top);
		glBegin(GL_POLYGON);
		glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE,
			face_material[right]);
		glNormal3f(1.0, 0.0, 0.0);
		if (front != NO_FACE) {
			glVertex3f(STICKER_DEPTH, STICKER_SHORT, STICKER_LONG);
			glVertex3f(STICKER_DEPTH, -STICKER_SHORT, STICKER_LONG);
		}
		if (top != NO_FACE) {
			glVertex3f(STICKER_DEPTH, STICKER_LONG, -STICKER_SHORT);
			glVertex3f(STICKER_DEPTH, STICKER_LONG, STICKER_SHORT);
		}
		if (back != NO_FACE) {
			glVertex3f(STICKER_DEPTH, -STICKER_SHORT, -STICKER_LONG);
			glVertex3f(STICKER_DEPTH, STICKER_SHORT, -STICKER_LONG);
		}
		if (bottom != NO_FACE) {
			glVertex3f(STICKER_DEPTH, -STICKER_LONG, STICKER_SHORT);
			glVertex3f(STICKER_DEPTH, -STICKER_LONG, -STICKER_SHORT);
		}
		glEnd();
		if (w->skewb.currentFace == face &&
				w->skewb.currentPosition == position) {
			glBegin(GL_POLYGON);
			glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE,
				face_material[BORDER_FACE]);
			glNormal3f(1.0, 0.0, 0.0);
			if (front != NO_FACE) {
				glVertex3f(SELECT_DEPTH, SELECT_SHORT, SELECT_LONG);
				glVertex3f(SELECT_DEPTH, -SELECT_SHORT, SELECT_LONG);
			}
			if (top != NO_FACE) {
				glVertex3f(SELECT_DEPTH, SELECT_LONG, -SELECT_SHORT);
				glVertex3f(SELECT_DEPTH, SELECT_LONG, SELECT_SHORT);
			}
			if (back != NO_FACE) {
				glVertex3f(SELECT_DEPTH, -SELECT_SHORT, -SELECT_LONG);
				glVertex3f(SELECT_DEPTH, SELECT_SHORT, -SELECT_LONG);
			}
			if (bottom != NO_FACE) {
				glVertex3f(SELECT_DEPTH, -SELECT_LONG, SELECT_SHORT);
				glVertex3f(SELECT_DEPTH, -SELECT_LONG, -SELECT_SHORT);
			}
			glEnd();
		}
		if (w->skewb.orient) {
			if (back != NO_FACE && top != NO_FACE) {
				drawOrientLine(w, face,
					w->skewb.cubeLoc[face][0].rotation,
					0.5,
					CUT_DEPTH, 0.5 * STICKER_SHORT, -0.1);
			}
			if (back != NO_FACE && bottom != NO_FACE) {
				drawOrientLine(w, face,
					w->skewb.cubeLoc[face][1].rotation,
					0.5,
					CUT_DEPTH, -0.5 * STICKER_SHORT, -0.1);
			}
			if (front != NO_FACE && bottom != NO_FACE) {
				drawOrientLine(w, face,
					w->skewb.cubeLoc[face][2].rotation,
					0.5,
					CUT_DEPTH, -0.1, 0.5 * STICKER_SHORT);
			}
			if (front != NO_FACE && top != NO_FACE) {
				drawOrientLine(w, face,
					w->skewb.cubeLoc[RIGHT_FACE][3].rotation,
					0.5,
					CUT_DEPTH, 0.1, 0.5 * STICKER_SHORT);
			}
		}
	}
	if (bottom != NO_FACE) { /* Pink */
		face = BOTTOM_FACE;
		position = positionCubit(w, face, back, front, left, right, bottom, top);
		glBegin(GL_POLYGON);
		glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE,
			face_material[bottom]);
		glNormal3f(0.0, -1.0, 0.0);
		if (left != NO_FACE) {
			glVertex3f(-STICKER_LONG, -STICKER_DEPTH, STICKER_SHORT);
			glVertex3f(-STICKER_LONG, -STICKER_DEPTH, -STICKER_SHORT);
		}
		if (front != NO_FACE) {
			glVertex3f(STICKER_SHORT, -STICKER_DEPTH, STICKER_LONG);
			glVertex3f(-STICKER_SHORT, -STICKER_DEPTH, STICKER_LONG);
		}
		if (right != NO_FACE) {
			glVertex3f(STICKER_LONG, -STICKER_DEPTH, -STICKER_SHORT);
			glVertex3f(STICKER_LONG, -STICKER_DEPTH, STICKER_SHORT);
		}
		if (back != NO_FACE) {
			glVertex3f(-STICKER_SHORT, -STICKER_DEPTH, -STICKER_LONG);
			glVertex3f(STICKER_SHORT, -STICKER_DEPTH, -STICKER_LONG);
		}
		glEnd();
		if (w->skewb.currentFace == face &&
				w->skewb.currentPosition == position) {
			glBegin(GL_POLYGON);
			glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE,
				face_material[BORDER_FACE]);
			glNormal3f(0.0, -1.0, 0.0);
			if (left != NO_FACE) {
				glVertex3f(-SELECT_LONG, -SELECT_DEPTH, SELECT_SHORT);
				glVertex3f(-SELECT_LONG, -SELECT_DEPTH, -SELECT_SHORT);
			}
			if (front != NO_FACE) {
				glVertex3f(SELECT_SHORT, -SELECT_DEPTH, SELECT_LONG);
				glVertex3f(-SELECT_SHORT, -SELECT_DEPTH, SELECT_LONG);
			}
			if (right != NO_FACE) {
				glVertex3f(SELECT_LONG, -SELECT_DEPTH, -SELECT_SHORT);
				glVertex3f(SELECT_LONG, -SELECT_DEPTH, SELECT_SHORT);
			}
			if (back != NO_FACE) {
				glVertex3f(-SELECT_SHORT, -SELECT_DEPTH, -SELECT_LONG);
				glVertex3f(SELECT_SHORT, -SELECT_DEPTH, -SELECT_LONG);
			}
			glEnd();
		}
		if (w->skewb.orient) {
			if (right != NO_FACE && front != NO_FACE) {
				drawOrientLine(w, face,
					w->skewb.cubeLoc[face][0].rotation,
					0.5,
					0.5 * STICKER_SHORT, -CUT_DEPTH, 0.1);
			}
			if (right != NO_FACE && back != NO_FACE) {
				drawOrientLine(w, face,
					w->skewb.cubeLoc[face][1].rotation,
					0.5,
					0.5 * STICKER_SHORT, -CUT_DEPTH, -0.1);
			}
			if (left != NO_FACE && back != NO_FACE) {
				drawOrientLine(w, face,
					w->skewb.cubeLoc[face][2].rotation,
					0.5,
					-0.5 * STICKER_SHORT, -CUT_DEPTH, -0.1);
			}
			if (left != NO_FACE && front != NO_FACE) {
				drawOrientLine(w, face,
					w->skewb.cubeLoc[face][3].rotation,
					0.5,
					-0.5 * STICKER_SHORT, -CUT_DEPTH, 0.1);
			}
		}
	}
	if (top != NO_FACE) { /* Red */
		face = TOP_FACE;
		position = positionCubit(w, face, back, front, left, right, bottom, top);
		glBegin(GL_POLYGON);
		glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE,
			face_material[top]);
		glNormal3f(0.0, 1.0, 0.0);
		if (left != NO_FACE) {
			glVertex3f(-STICKER_LONG, STICKER_DEPTH, -STICKER_SHORT);
			glVertex3f(-STICKER_LONG, STICKER_DEPTH, STICKER_SHORT);
		}
		if (front != NO_FACE) {
			glVertex3f(-STICKER_SHORT, STICKER_DEPTH, STICKER_LONG);
			glVertex3f(STICKER_SHORT, STICKER_DEPTH, STICKER_LONG);
		}
		if (right != NO_FACE) {
			glVertex3f(STICKER_LONG, STICKER_DEPTH, STICKER_SHORT);
			glVertex3f(STICKER_LONG, STICKER_DEPTH, -STICKER_SHORT);
		}
		if (back != NO_FACE) {
			glVertex3f(STICKER_SHORT, STICKER_DEPTH, -STICKER_LONG);
			glVertex3f(-STICKER_SHORT, STICKER_DEPTH, -STICKER_LONG);
		}
		glEnd();
		if (w->skewb.currentFace == face &&
				w->skewb.currentPosition == position) {
			glBegin(GL_POLYGON);
			glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE,
				face_material[BORDER_FACE]);
			glNormal3f(0.0, 1.0, 0.0);
			if (left != NO_FACE) {
				glVertex3f(-SELECT_LONG, SELECT_DEPTH, -SELECT_SHORT);
				glVertex3f(-SELECT_LONG, SELECT_DEPTH, SELECT_SHORT);
			}
			if (front != NO_FACE) {
				glVertex3f(-SELECT_SHORT, SELECT_DEPTH, SELECT_LONG);
				glVertex3f(SELECT_SHORT, SELECT_DEPTH, SELECT_LONG);
			}
			if (right != NO_FACE) {
				glVertex3f(SELECT_LONG, SELECT_DEPTH, SELECT_SHORT);
				glVertex3f(SELECT_LONG, SELECT_DEPTH, -SELECT_SHORT);
			}
			if (back != NO_FACE) {
				glVertex3f(SELECT_SHORT, SELECT_DEPTH, -SELECT_LONG);
				glVertex3f(-SELECT_SHORT, SELECT_DEPTH, -SELECT_LONG);
			}
			glEnd();
		}
		if (w->skewb.orient) {
			if (right != NO_FACE && back != NO_FACE) {
				drawOrientLine(w, face,
					w->skewb.cubeLoc[face][0].rotation,
					0.5,
					0.5 * STICKER_SHORT, CUT_DEPTH, -0.1);
			}
			if (right != NO_FACE && front != NO_FACE) {
				drawOrientLine(w, face,
					w->skewb.cubeLoc[face][1].rotation,
					0.5,
					0.5 * STICKER_SHORT, CUT_DEPTH, 0.1);
			}
			if (left != NO_FACE && front != NO_FACE) {
				drawOrientLine(w, face,
					w->skewb.cubeLoc[face][2].rotation,
					0.5,
					-0.5 * STICKER_SHORT, CUT_DEPTH, 0.1);
			}
			if (left != NO_FACE && back != NO_FACE) {
				drawOrientLine(w, face,
					w->skewb.cubeLoc[face][3].rotation,
					0.5,
					-0.5 * STICKER_SHORT, CUT_DEPTH, -0.1);
			}
		}
	}
}

static void
drawWholeFacit(SkewbGLWidget w, int x, int y, int z)
{
	if (x < 0) {
		/* LEFT_FACE, Blue, 1 */
		glPushMatrix();
		glRotatef(-90.0, 0, 1, 0);
		drawStickerlessFacit();
		glPopMatrix();
		glPushMatrix();
		drawFacit(w, 1);
		glPopMatrix();
	} else if (x > 0) {
		 /* RIGHT_FACE, Green, 3 */
		glPushMatrix();
		glRotatef(90.0, 0, 1, 0);
		drawStickerlessFacit();
		glPopMatrix();
		glPushMatrix();
		drawFacit(w, 3);
		glPopMatrix();
	} else if (y < 0) {
		/* BOTTOM_FACE, Pink 4 */
		glPushMatrix();
		glRotatef(90.0, 1, 0, 0);
		drawStickerlessFacit();
		glPopMatrix();
		glPushMatrix();
		drawFacit(w, 4);
		glPopMatrix();
	} else if (y > 0) {
		/* TOP_FACE, Red 0 */
		glPushMatrix();
		glRotatef(-90.0, 1, 0, 0);
		drawStickerlessFacit();
		glPopMatrix();
		glPushMatrix();
		drawFacit(w, 0);
		glPopMatrix();
	} else if (z < 0) {
		/* BACK_FACE, Yellow 5 */
		glPushMatrix();
		glRotatef(180.0, 0, 1, 0);
		drawStickerlessFacit();
		glPopMatrix();
		glPushMatrix();
		drawFacit(w, 5);
		glPopMatrix();
	} else if (z > 0) {
		/* FRONT_FACE, White, 2 */
		glPushMatrix();
		drawStickerlessFacit();
		glPopMatrix();
		glPushMatrix();
		drawFacit(w, 2);
		glPopMatrix();
	}
}

static void
drawWholeCubit(SkewbGLWidget w, int x, int y, int z)
{
	const float h = 0.5;

	if (x < 0) {
		if (y < 0) {
			if (z < 0) {
				glPushMatrix();
				glTranslatef(-h, -h, -h);
				glPushMatrix();
				glRotatef(90.0, 0, 1, 0);
				glRotatef(180.0, 1, 0, 0);
				drawStickerlessCubit();
				glPopMatrix();
				drawCubit(w, /* BLL */
					w->skewb.cubeLoc[BACK_FACE][3].face,
					NO_FACE,
					w->skewb.cubeLoc[LEFT_FACE][2].face,
					NO_FACE,
					w->skewb.cubeLoc[BOTTOM_FACE][2].face,
					NO_FACE);
				glPopMatrix();
			} else if (z > 0) {
				glPushMatrix();
				glTranslatef(-h, -h, h);
				glPushMatrix();
				glRotatef(180.0, 0, 0, 1);
				drawStickerlessCubit();
				glPopMatrix();
				drawCubit(w, /* FLL */
					NO_FACE,
					w->skewb.cubeLoc[FRONT_FACE][2].face,
					w->skewb.cubeLoc[LEFT_FACE][1].face,
					NO_FACE,
					w->skewb.cubeLoc[BOTTOM_FACE][3].face,
					NO_FACE);
				glPopMatrix();
			}
		} else if (y > 0) {
			if (z < 0) {
				glPushMatrix();
				glTranslatef(-h, h, -h);
				glPushMatrix();
				glRotatef(90.0, 0, -1, 0);
				glRotatef(90.0, 0, 0, 1);
				drawStickerlessCubit();
				glPopMatrix();
				drawCubit(w, /* BUL */
					w->skewb.cubeLoc[BACK_FACE][2].face,
					NO_FACE,
					w->skewb.cubeLoc[LEFT_FACE][3].face,
					NO_FACE,
					NO_FACE,
					w->skewb.cubeLoc[TOP_FACE][3].face);
				glPopMatrix();
			} else if (z > 0) {
				glPushMatrix();
				glTranslatef(-h, h, h);
				glPushMatrix();
				glRotatef(90.0, 0, 0, 1);
				drawStickerlessCubit();
				glPopMatrix();
				drawCubit(w, /* FUL */
					NO_FACE,
					w->skewb.cubeLoc[FRONT_FACE][3].face,
					w->skewb.cubeLoc[LEFT_FACE][0].face,
					NO_FACE,
					NO_FACE,
					w->skewb.cubeLoc[TOP_FACE][2].face);
				glPopMatrix();
			}
		}
	} else if (x > 0) {
		if (y < 0) {
			if (z < 0) {
				glPushMatrix();
				glTranslatef(h, -h, -h);
				glPushMatrix();
				glRotatef(90.0, 0, 1, 0);
				glRotatef(90.0, 1, 0, 0);
				drawStickerlessCubit();
				glPopMatrix();
				drawCubit(w, /* BLR */
					w->skewb.cubeLoc[BACK_FACE][0].face,
					NO_FACE,
					NO_FACE,
					w->skewb.cubeLoc[RIGHT_FACE][1].face,
					w->skewb.cubeLoc[BOTTOM_FACE][1].face,
					NO_FACE);
				glPopMatrix();
			} else if (z > 0) {
				glPushMatrix();
				glTranslatef(h, -h, h);
				glPushMatrix();
				glRotatef(90.0, 1, 0, 0);
				drawStickerlessCubit();
				glPopMatrix();
				drawCubit(w, /* FLR */
					NO_FACE,
					w->skewb.cubeLoc[FRONT_FACE][1].face,
					NO_FACE,
					w->skewb.cubeLoc[RIGHT_FACE][2].face,
					w->skewb.cubeLoc[BOTTOM_FACE][0].face,
					NO_FACE);
				glPopMatrix();
			}
		} else if (y > 0) {
			if (z < 0) {
				glPushMatrix();
				glTranslatef(h, h, -h);
				glPushMatrix();
				glRotatef(90.0, 0, 1, 0);
				drawStickerlessCubit();
				glPopMatrix();
				drawCubit(w, /* BUR */
					w->skewb.cubeLoc[BACK_FACE][1].face,
					NO_FACE,
					NO_FACE,
					w->skewb.cubeLoc[RIGHT_FACE][0].face,
					NO_FACE,
					w->skewb.cubeLoc[TOP_FACE][0].face);
				glPopMatrix();
			} else if (z > 0) {
				glPushMatrix();
				glTranslatef(h, h, h);
				glPushMatrix();
				drawStickerlessCubit();
				glPopMatrix();
				drawCubit(w, /* FUR */
					NO_FACE,
					w->skewb.cubeLoc[FRONT_FACE][0].face,
					NO_FACE,
					w->skewb.cubeLoc[RIGHT_FACE][3].face,
					NO_FACE,
					w->skewb.cubeLoc[TOP_FACE][1].face);
				glPopMatrix();
			}
		}
	}
}

static int nextCorner[MAX_FACES][MAX_ORIENT] =
{
	{4, 5, 1, 0},
	{7, 5, 4, 6},
	{6, 4, 0, 2},
	{2, 0, 1, 3},
	{7, 6, 2, 3},
	{5, 7, 3, 1}
};

static int nextFace[MAX_FACES][MAX_ORIENT] =
{
	{-4, -1, 4, 1},
	{-1, 2, 1, -2},
	{-4, 2, 4, -2},
	{1, 2, -1, -2},
	{-4, 1, 4, -1},
	{-4, -2, 4, 2},
};

static void
convertMove(SkewbGLWidget w, SkewbMove move, SkewbSlice * slice)
{
	if ((move.position + 3) % MAX_ORIENT == move.direction) {
		(*slice).corner = nextCorner[move.face][move.position];
		(*slice).rotation = CW;
	} else if ((move.position + 1) % MAX_ORIENT == move.direction) {
		(*slice).corner = nextCorner[move.face][move.position];
		(*slice).rotation = CCW;
	} else {
		(*slice).corner = NO_CORNER;
	}
}

/*-
 * If you want to rotate a slice, you can draw that at the anglestep,
  then the without anglestep.
 * There is a sequence for drawing cubies for each axis being moved...
 */
static void
skewRotate(SkewbGLWidget w, int corner, GLfloat rotateStep)
{
#define X1 (((corner & 4) != 0) ? 1 : -1)
#define Y1 (((corner & 2) != 0) ? 1 : -1)
#define Z1 (((corner & 1) != 0) ? 1 : -1)
#define Xn (((corner & 4) != 0) ? -1 : 1)
#define Yn (((corner & 2) != 0) ? -1 : 1)
#define Zn (((corner & 1) != 0) ? -1 : 1)
	glPushMatrix();
	glRotatef(rotateStep, X1, Y1, Z1);
	drawWholeFacit(w, X1, 0, 0);
	drawWholeFacit(w, 0, Y1, 0);
	drawWholeFacit(w, 0, 0, Z1);
	drawWholeCubit(w, X1, Y1, Z1);
	drawWholeCubit(w, Xn, Y1, Z1);
	drawWholeCubit(w, X1, Yn, Z1);
	drawWholeCubit(w, X1, Y1, Zn);
	if (!w->skewbGL.movement.control)
		glPopMatrix();

	drawWholeFacit(w, Xn, 0, 0);
	drawWholeFacit(w, 0, Yn, 0);
	drawWholeFacit(w, 0, 0, Zn);
	drawWholeCubit(w, Xn, Yn, Zn);
	drawWholeCubit(w, X1, Yn, Zn);
	drawWholeCubit(w, Xn, Y1, Zn);
	drawWholeCubit(w, Xn, Yn, Z1);
	if (w->skewbGL.movement.control)
		glPopMatrix();
}

/*-
 * This rotates whole cube.
 */
static void
faceRotate(SkewbGLWidget w, int face, GLfloat rotateStep)
{
#define XP ((face == 4) ? 1 : ((face == -4) ? -1 : 0))
#define YP ((face == 2) ? 1 : ((face == -2) ? -1 : 0))
#define ZP ((face == 1) ? 1 : ((face == -1) ? -1 : 0))
	int corner = 0;

	glPushMatrix();
	glRotatef(rotateStep, XP, YP, ZP);
	drawWholeFacit(w, X1, 0, 0);
	drawWholeFacit(w, 0, Y1, 0);
	drawWholeFacit(w, 0, 0, Z1);
	drawWholeCubit(w, X1, Y1, Z1);
	drawWholeCubit(w, Xn, Y1, Z1);
	drawWholeCubit(w, X1, Yn, Z1);
	drawWholeCubit(w, X1, Y1, Zn);

	drawWholeFacit(w, Xn, 0, 0);
	drawWholeFacit(w, 0, Yn, 0);
	drawWholeFacit(w, 0, 0, Zn);
	drawWholeCubit(w, Xn, Yn, Zn);
	drawWholeCubit(w, X1, Yn, Zn);
	drawWholeCubit(w, Xn, Y1, Zn);
	drawWholeCubit(w, Xn, Yn, Z1);
	glPopMatrix();
}

static void
drawCube(SkewbGLWidget w)
{
	SkewbSlice slice;
	GLfloat rotateStep = 0.0;

	slice.corner = NO_CORNER;
	slice.rotation = CW;
	if (w->skewbGL.movement.face == NO_FACE) {
		slice.rotation = NO_ROTATION;
		skewRotate(w, slice.corner, rotateStep);
	} else if (w->skewbGL.movement.direction < MAX_ORIENT) { 
		convertMove(w, w->skewbGL.movement, &slice);
		if (slice.corner != NO_CORNER) {
			rotateStep = (slice.rotation == CCW) ?
				w->skewbGL.rotateStep : -w->skewbGL.rotateStep;
		}
		skewRotate(w, slice.corner, rotateStep);
	} else {
		int face = nextFace[w->skewbGL.movement.face][w->skewbGL.movement.direction % MAX_ORIENT];
		
		rotateStep = w->skewbGL.rotateStep;
		faceRotate(w, face, rotateStep);
	}
}

void
movePiecesGL(SkewbGLWidget w, int face, int position, int direction,
		Boolean control, int fast)
{
	int i, f;

	w->skewbGL.movement.face = face;
	w->skewbGL.movement.position = position;
	w->skewbGL.movement.direction = direction;
	w->skewbGL.movement.control = control;
	w->skewbGL.rotateStep = 0.0;
	w->skewbGL.angleStep = 0.0;
	f = (w->skewbGL.numTurnInc < fast) ? 1 : fast;
	for (i = 0; i < w->skewbGL.numTurnInc / f; i++) {
		double increment;
		if (direction < MAX_ORIENT) {	
			increment = 120.0 * f / w->skewbGL.numTurnInc;
		} else {
			increment = 90.0 * f / w->skewbGL.numTurnInc;
		}
		w->skewbGL.rotateStep += increment;
		/* rubik has 6 possibles... need 8 here */
		/*if (rotateDir[face][direction] == -1)*/
		drawAllPiecesGL(w);
		Sleep((unsigned int) ((w->skewb.delay /
			(w->skewbGL.numTurnInc * f)) * 40.0));
		w->skewb.currentFace = -1;
		w->skewb.currentPosition = -1;
	}
	w->skewbGL.angleStep = 0.0;
	w->skewbGL.movement.face = NO_FACE;
	w->skewbGL.movement.position = NO_POSITION;
	w->skewbGL.movement.direction = NO_DEPTH;
	w->skewbGL.movement.control = False;
}

void
drawAllPiecesGL(SkewbGLWidget w)
{
#ifdef WINVER
	wglMakeCurrent(w->core.hDC, hRC);
#else
	if (!glXMakeCurrent(XtDisplay(w), XtWindow(w), *glXContext)) {
		DISPLAY_WARNING("Draw All GL error");
	}
#endif
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	glPushMatrix();
	glTranslatef(0.0, 0.0, -10.0);
	if (w->core.height > w->core.width)
		glScalef(SCALE, SCALE * w->core.width / w->core.height, SCALE);
	else
		glScalef(SCALE * w->core.height / w->core.width, SCALE, SCALE);
	switch (w->skewb.view) {
	case 0: /* >-  vertical */
		glRotatef(w->skewbGL.step / 2, 0, 1, 0);
		glRotatef(w->skewbGL.step / 2, 1, 0, 0);
		break;
	case 1: /* Y  !vertical */
		glRotatef(w->skewbGL.step / 2, 1, 0, 0);
		glRotatef(w->skewbGL.step / 2, 0, 1, 0);
		break;
	case 2: /* -<  vertical */
		glRotatef(-w->skewbGL.step / 2, 0, 1, 0);
		glRotatef(-3 * w->skewbGL.step / 2, 1, 0, 0);
		break;
	case 3: /* ^  !vertical */
		glRotatef(-w->skewbGL.step / 2, 1, 0, 0);
		glRotatef(-3 * w->skewbGL.step / 2, 0, 1, 0);
		break;
	default:
		break;
	}
	drawCube(w);
	glPopMatrix();
	glFlush();
#ifdef WINVER
	SwapBuffers(w->core.hDC);
#else
	glXSwapBuffers(XtDisplay(w), XtWindow(w));
#endif
}

void
drawFrameGL(SkewbGLWidget w, Boolean focus)
{
	/* UNUSED */
}

static int
normalizePosition(float x, float y)
{
	if (x >= 0.0 && y < 0.0)
		return ((x - y > 1.0) ? 0 : MAX_ORIENT);	
	if (x >= 0.0 && y >= 0.0)
		return ((x + y > 1.0) ? 1 : MAX_ORIENT);	
	if (x < 0.0 && y >= 0.0)
		return ((-x + y > 1.0) ? 2 : MAX_ORIENT);	
	if (x < 0.0 && y < 0.0)
		return ((-x - y > 1.0) ? 3 : MAX_ORIENT);	
	return MAX_ORIENT;	
}

Boolean
selectPiecesGL(SkewbGLWidget w, int x, int y, int *face, int *position)
{
	/* Using gluUnProject By: Luke Benstead */
	/* http://nehe.gamedev.net/data/articles/article.asp?article=13 */

	GLint viewport[4];
	GLdouble modelview[16];
	GLdouble projection[16];
	GLfloat winX, winY, winZ;
	GLdouble posX, posY, posZ;
	Matrix a, b, r;
	Vector t, world, nw;

	glGetDoublev(GL_MODELVIEW_MATRIX, modelview);
	glGetDoublev(GL_PROJECTION_MATRIX, projection);
	glGetIntegerv(GL_VIEWPORT, viewport);

	winX = (float) x;
	winY = (float) viewport[3] - (float) y;
	glReadPixels(x, (int) winY, 1, 1, GL_DEPTH_COMPONENT, GL_FLOAT,
		&winZ);
	gluUnProject(winX, winY, winZ, modelview, projection, viewport,
		&(posX), &(posY), &(posZ));
	t[0] = posX;
	t[1] = posY;
	t[2] = posZ + 10;
	if (w->core.height > w->core.width)
		t[1] *= (double) w->core.height / (double) w->core.width;
	else
		t[0] *= (double) w->core.width / (double) w->core.height;
	*face = -1;
	*position = -1;
	if (winZ == 1) {
		return False;
	}
	world[0] = t[0], world[1] = t[1], world[2] = t[2];
	switch (w->skewb.view) {
	case 0: /* >-  vertical */
		setMatrix(-w->skewbGL.step / 2, 1.0, 0.0, 0.0, a);
		setMatrix(-w->skewbGL.step / 2, 0.0, 1.0, 0.0, b);
		matrixMatrix(a, b, r);
		matrixVector(r, t, world);
		nw[0] = -world[0] / 2.0;
		nw[1] = world[1] / 2.0;
		nw[2] = world[2] / 2.0;
		if (nw[1] >= nw[0] && nw[1] >= nw[2]) {
			*face = 0;
			*position = normalizePosition(world[0], world[2]);
		} else if (nw[0] >= nw[1] && nw[0] >= nw[2]) {
			*face = 1;
			*position = normalizePosition(world[2], -world[1]);
		} else if (nw[2] >= nw[0] && nw[2] >= nw[1]) {
			*face = 2;
			*position = normalizePosition(world[0], -world[1]);
		}
		break;
	case 1: /* Y  !vertical */
		setMatrix(-w->skewbGL.step / 2, 0.0, 1.0, 0.0, a);
		setMatrix(-w->skewbGL.step / 2, 1.0, 0.0, 0.0, b);
		matrixMatrix(a, b, r);
		matrixVector(r, t, world);
		nw[0] = -world[0] / 2.0;
		nw[1] = world[1] / 2.0;
		nw[2] = world[2] / 2.0;
		if (nw[1] >= nw[0] && nw[1] >= nw[2]) {
			*face = 0;
			*position = normalizePosition(world[0], world[2]);
		} else if (nw[0] >= nw[1] && nw[0] >= nw[2]) {
			*face = 1;
			*position = normalizePosition(world[2], -world[1]);
		} else if (nw[2] >= nw[0] && nw[2] >= nw[1]) {
			*face = 2;
			*position = normalizePosition(world[0], -world[1]);
		}
		break;
	case 2: /* -<  vertical */
		setMatrix(3 * w->skewbGL.step / 2, 1.0, 0.0, 0.0, a);
		setMatrix(w->skewbGL.step / 2, 0.0, 1.0, 0.0, b);
		matrixMatrix(a, b, r);
		matrixVector(r, t, world);
		nw[0] = -world[0] / 2.0;
		nw[1] = world[1] / 2.0;
		nw[2] = world[2] / 2.0;
		if (nw[0] <= nw[1] && nw[0] <= nw[2]) {
			*face = 3;
			*position = normalizePosition(-world[2], -world[1]);
		} else if (nw[1] <= nw[0] && nw[1] <= nw[2]) {
			*face = 4;
			*position = normalizePosition(world[0], -world[2]);
		} else if (nw[2] <= nw[0] && nw[2] <= nw[1]) {
			*face = 5;
			*position = normalizePosition(world[0], world[1]);
		}
		break;
	case 3: /* ^  !vertical */
		setMatrix(3 * w->skewbGL.step / 2, 0.0, 1.0, 0.0, a);
		setMatrix(w->skewbGL.step / 2, 1.0, 0.0, 0.0, b);
		matrixMatrix(a, b, r);
		matrixVector(r, t, world);
		nw[0] = -world[0] / 2.0;
		nw[1] = world[1] / 2.0;
		nw[2] = world[2] / 2.0;
		if (nw[0] <= nw[1] && nw[0] <= nw[2]) {
			*face = 3;
			*position = normalizePosition(-world[2], -world[1]);
		} else if (nw[1] <= nw[0] && nw[1] <= nw[2]) {
			*face = 4;
			*position = normalizePosition(world[0], -world[2]);
		} else if (nw[2] <= nw[0] && nw[2] <= nw[1]) {
			*face = 5;
			*position = normalizePosition(world[0], world[1]);
		}
		break;
	default:
		break;
	}
#ifdef DEBUG
	(void) printf("x %d, y %d\n", x, y);
	(void) printf(" wx %g, wy %g, wz %g\n", winX, winY, winZ);
	(void) printf("  px %g, py %g, pz %g\n", posX, posY, posZ);
	(void) printf("   tx %g, ty %g, tz %g\n", t[0], t[1], t[2]);
	(void) printf("    rx %g, ry %g, rz %g\n",
		world[0], world[1], world[2]);
	(void) printf("     face %d, pos %d\n", *face, *position);
#endif
	return True;
}

Boolean
narrowSelectionGL(SkewbGLWidget w, int *face, int *position, int *direction)
{
	switch (*direction) {
	case TR:
	case BR:
	case BL:
	case TL:
		if (w->skewb.view % 2 == 0) {
			if (*face == 1)
				*direction = *direction + 2 * MAX_ORIENT;
			else if (*face == 3)
				*direction = (*direction + 3) % MAX_ORIENT +
					2 * MAX_ORIENT;
		} else {
			if (*face == 0)
				*direction = (*direction + 3) % MAX_ORIENT +
					2 * MAX_ORIENT;
			else if (*face == 4)
				*direction = (*direction + 2) % MAX_ORIENT +
					2 * MAX_ORIENT;
			else if (*face == 5)
				*direction = (*direction + 2) % MAX_ORIENT;
		}
		break;
	case CCW:
	case CW:
		break;
	case TOP:
	case RIGHT:
	case BOTTOM:
	case LEFT:
		if (w->skewb.view % 2 == 0) {
			if (*face == 1)
				*direction = (TL + *direction) % MAX_ORIENT;
			else if (*face == 3)
				*direction = (BL + *direction) % MAX_ORIENT;
		} else {
			if (*face == 0)
				*direction = (TR + *direction) % MAX_ORIENT;
			else if (*face == 4)
				*direction = (BR + *direction) % MAX_ORIENT;
			else if (*face == 5)
				*direction = (BL + *direction) % MAX_ORIENT +
					2 * MAX_ORIENT;
		}
		break;
	default:
		return False;
	}
	if (*position != MAX_ORIENT) {
		if (*direction == CW)
			*direction = (*position + 3) % MAX_ORIENT;
		else if (*direction == CCW)
			*direction = (*position + 1) % MAX_ORIENT;
		else if (*direction < MAX_ORIENT &&
				!((*direction + *position) % 2))
			return False;
	}
	return True;
}

#ifdef WINVER
static Boolean
setupPixelFormat(SkewbGLWidget w, BYTE type, DWORD flags)
{
	PIXELFORMATDESCRIPTOR pfd;
	int pixelFormat;

	memset(&pfd, 0, sizeof(PIXELFORMATDESCRIPTOR));
	pfd.nSize = sizeof(PIXELFORMATDESCRIPTOR);
	pfd.nVersion = 1;
	pfd.dwFlags = PFD_DRAW_TO_WINDOW | PFD_SUPPORT_OPENGL | flags;
	pfd.dwLayerMask = PFD_MAIN_PLANE;
	/*pfd.cColorBits = 8;
	pfd.cDepthBits = 16;*/
	pfd.iPixelType = type;
	pfd.cColorBits = 24;
	pfd.cDepthBits = 32;

	if ((pixelFormat = ChoosePixelFormat(w->core.hDC, &pfd)) == 0) {
		DWORD err;
		char *buf1;

		err = GetLastError();
		/* 6 ERROR_INVALID_HANDLE */
		intCat(&buf1, "ChoosePixelFormat failed ", err);
		DISPLAY_WARNING(buf1);
		free(buf1);
		return FALSE;
	}

	if (SetPixelFormat(w->core.hDC, pixelFormat, &pfd) == FALSE) {
		MessageBox(NULL, "SetPixelFormat failed", "Error", MB_OK);
		return FALSE;
	}
	DescribePixelFormat(w->core.hDC, pixelFormat,
		sizeof(PIXELFORMATDESCRIPTOR), &pfd);
	return TRUE;
}

#else

static Boolean
setValuesPuzzleGL(Widget current, Widget request, Widget renew)
{
	SkewbGLWidget c = (SkewbGLWidget) current, w = (SkewbGLWidget) renew;
	Boolean redraw = False;

	if (w->skewb.view != c->skewb.view) {
		resizePuzzleGL(w);
		redraw = True;
	}
#if 0
	if (w->skewbGL.cubeSizex.x != c->skewbGL.cubeSizex.x ||
			w->skewbGL.cubeSizey.x != c->skewbGL.cubeSizey.x ||
			w->skewbGL.cubeSizez.x != c->skewbGL.cubeSizez.x ||
			w->skewbGL.cubeSizex.y != c->skewbGL.cubeSizex.y ||
			w->skewbGL.cubeSizey.y != c->skewbGL.cubeSizey.y ||
			w->skewbGL.cubeSizez.y != c->skewbGL.cubeSizez.y) {
		resizePuzzleGL(w);
		redraw = True;
	}
#endif
	if (w->skewb.delay != c->skewb.delay) {
		w->skewbGL.numTurnInc = ((w->skewb.delay < MAX_TURN_INC) ?
			w->skewb.delay + 1 : MAX_TURN_INC);
	}
	return (redraw);
}

static GLXContext *
initGL(SkewbGLWidget w) {
	XVisualInfo xviIn, *xviOut;
	int numVis;

	/*XGetWindowAttributes(XtDisplay(w), XtWindow(w), &xgwa); */
	xviIn.screen = DefaultScreen(XtDisplay(w));
	xviIn.visualid = XVisualIDFromVisual(DefaultVisual(XtDisplay(w),
		xviIn.screen));
	xviOut = XGetVisualInfo(XtDisplay(w), VisualScreenMask | VisualIDMask,
		&xviIn, &numVis);
	if (!xviOut) {
		XtWarning("Could not get XVisualInfo");
		return (GLXContext *) NULL;
	}
	if (glXContext)
		free(glXContext);
	if ((glXContext = (GLXContext *) malloc(sizeof (GLXContext))) ==
			NULL) {
		DISPLAY_ERROR("Not enough memory for glx info, exiting.");
	}
	*glXContext = glXCreateContext(XtDisplay(w), xviOut, 0, GL_TRUE);
	(void) XFree((char *) xviOut);
	if (!*glXContext) {
		XtWarning("Could not create rendering context");
		return (GLXContext *) NULL;
	}
	return glXContext;
}
#endif

#ifndef WINVER
static
#endif
void
resizePuzzleGL(SkewbGLWidget w)
{
#ifdef WINVER
	RECT rect;

	/* Determine size of client area */
	(void) GetClientRect(w->core.hWnd, &rect);
	w->core.width = rect.right;
	w->core.height = rect.bottom;
#endif
	glViewport(0, 0, (GLint) w->core.width, (GLint) w->core.height);
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	glFrustum(-1.0, 1.0, -1.0, 1.0, 5.0, 15.0);
	glMatrixMode(GL_MODELVIEW);
}

static void
makeCurrentGL(SkewbGLWidget w)
{
	GLboolean rgbaMode;

#ifdef WINVER
	wglMakeCurrent(w->core.hDC, hRC);
#else
	if (!glXMakeCurrent(XtDisplay(w), XtWindow(w), *glXContext)) {
		DISPLAY_WARNING("GL error");
	}
#endif
	madeCurrent = True;
	/* True Color junk */
	glGetBooleanv(GL_RGBA_MODE, &rgbaMode);
	if (!rgbaMode) {
#ifdef WINVER
		glClearIndex(0.0);
#else
		glIndexi(WhitePixel(XtDisplay(w),
			DefaultScreen(XtDisplay(w))));
		glClearIndex((float) BlackPixel(XtDisplay(w),
			DefaultScreen(XtDisplay(w))));
#endif
	}
	resizePuzzleGL(w);

	glDrawBuffer(GL_BACK);
	glClearDepth(1.0);
	glClearColor(face_material[NO_FACE][0], face_material[NO_FACE][1],
		face_material[NO_FACE][2], face_material[NO_FACE][3]);
	glColor3f(1.0, 1.0, 1.0);

	glLightfv(GL_LIGHT0, GL_AMBIENT, ambient);
	glLightfv(GL_LIGHT0, GL_DIFFUSE, diffuse);
	glLightfv(GL_LIGHT0, GL_POSITION, position0);
	glLightfv(GL_LIGHT1, GL_AMBIENT, ambient);
	glLightfv(GL_LIGHT1, GL_DIFFUSE, diffuse);
	glLightfv(GL_LIGHT1, GL_POSITION, position1);
	glLightModelfv(GL_LIGHT_MODEL_AMBIENT, lmodel_ambient);
	glLightModelfv(GL_LIGHT_MODEL_TWO_SIDE, lmodel_twoside);
	glEnable(GL_LIGHTING);
	glEnable(GL_LIGHT0);
	glEnable(GL_LIGHT1);
	glEnable(GL_DEPTH_TEST);
	glEnable(GL_NORMALIZE);
	glEnable(GL_CULL_FACE);
	glShadeModel(GL_FLAT);
	glMaterialfv(GL_FRONT_AND_BACK, GL_SHININESS, front_shininess);
	glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR, front_specular);

	w->skewbGL.step = 90;
	w->skewbGL.rotateStep = 0.0;
}

#ifndef WINVER
static void
initializePuzzleGL(Widget request, Widget renew)
{
	SkewbGLWidget w = (SkewbGLWidget) renew;
	int i;

	setAllColors((SkewbWidget) w);
	(void) initGL(w);
	w->skewb.dim = 4; /* 2 and 3 already taken */
	for (i = 0; i < MAX_FACES + 2; i++)
		pickColor(w, i, w->skewb.mono, face_material[i]);
	w->skewbGL.numTurnInc = ((w->skewb.delay < MAX_TURN_INC) ?
		w->skewb.delay + 1 : MAX_TURN_INC);
	resizePuzzleGL(w);
}
#endif

#ifndef WINVER
static
#endif
void
exposePuzzleGL(
#ifdef WINVER
SkewbGLWidget w
#else
Widget renew, XEvent *event, Region region
#endif
)
{
#ifdef WINVER
	if (hRC == NULL) {
		BYTE type;
		int i;

#if 1
		type = PFD_TYPE_RGBA;
#else
		type = PFD_TYPE_COLORINDEX;
#endif
		DWORD flags = PFD_DOUBLEBUFFER;
		(void) setupPixelFormat(w, type, flags);
		hRC = wglCreateContext(w->core.hDC);
		w->skewbGL.movement.face = NO_FACE;
		w->skewbGL.movement.position = NO_POSITION;
		w->skewbGL.movement.direction = NO_DEPTH;
		w->skewbGL.movement.control = False;
		w->skewbGL.rotateStep = 0.0;
		w->skewbGL.angleStep = 0.0;
		w->skewb.dim = 4; /* 2 and 3 already taken */
		for (i = 0; i < MAX_FACES + 2; i++)
			pickColor(w, i, w->skewb.mono, face_material[i]);
		w->skewbGL.numTurnInc = ((w->skewb.delay < MAX_TURN_INC) ?
			w->skewb.delay + 1 : MAX_TURN_INC);
		resizePuzzleGL(w);
	}
#else
	SkewbGLWidget w = (SkewbGLWidget) renew;

	if (!w->core.visible)
		return;
#endif
	if (!madeCurrent) {
		makeCurrentGL(w);
	}
	drawFrameGL(w, w->skewb.focus);
	drawAllPiecesGL(w);
}

#ifndef WINVER
static void
movePuzzleGLTl(SkewbGLWidget w, XEvent *event, char **args, int nArgs)
{
	movePuzzleInput((SkewbWidget) w, event->xbutton.x, event->xbutton.y,
		TL,
		(int) (event->xkey.state & ControlMask),
		(int) (event->xkey.state & (Mod1Mask | Mod3Mask | Mod4Mask | Mod5Mask)));
}

static void
movePuzzleGLTop(SkewbGLWidget w, XEvent *event, char **args, int nArgs)
{
	movePuzzleInput((SkewbWidget) w, event->xbutton.x, event->xbutton.y,
		TOP,
		(int) (event->xkey.state & ControlMask),
		(int) (event->xkey.state & (Mod1Mask | Mod3Mask | Mod4Mask | Mod5Mask)));
}

static void
movePuzzleGLTr(SkewbGLWidget w, XEvent *event, char **args, int nArgs)
{
	movePuzzleInput((SkewbWidget) w, event->xbutton.x, event->xbutton.y,
		TR,
		(int) (event->xkey.state & ControlMask),
		(int) (event->xkey.state & (Mod1Mask | Mod3Mask | Mod4Mask | Mod5Mask)));
}

static void
movePuzzleGLLeft(SkewbGLWidget w, XEvent *event, char **args, int nArgs)
{
	movePuzzleInput((SkewbWidget) w, event->xbutton.x, event->xbutton.y,
		LEFT,
		(int) (event->xkey.state & ControlMask),
		(int) (event->xkey.state & (Mod1Mask | Mod3Mask | Mod4Mask | Mod5Mask)));
}

static void
movePuzzleGLRight(SkewbGLWidget w, XEvent *event, char **args, int nArgs)
{
	movePuzzleInput((SkewbWidget) w, event->xbutton.x, event->xbutton.y,
		RIGHT,
		(int) (event->xkey.state & ControlMask),
		(int) (event->xkey.state & (Mod1Mask | Mod3Mask | Mod4Mask | Mod5Mask)));
}

static void
movePuzzleGLBl(SkewbGLWidget w, XEvent *event, char **args, int nArgs)
{
	movePuzzleInput((SkewbWidget) w, event->xbutton.x, event->xbutton.y,
		BL,
		(int) (event->xkey.state & ControlMask),
		(int) (event->xkey.state & (Mod1Mask | Mod3Mask | Mod4Mask | Mod5Mask)));
}

static void
movePuzzleGLBottom(SkewbGLWidget w, XEvent *event, char **args, int nArgs)
{
	movePuzzleInput((SkewbWidget) w, event->xbutton.x, event->xbutton.y,
		BOTTOM,
		(int) (event->xkey.state & ControlMask),
		(int) (event->xkey.state & (Mod1Mask | Mod3Mask | Mod4Mask | Mod5Mask)));
}

static void
movePuzzleGLBr(SkewbGLWidget w, XEvent *event, char **args, int nArgs)
{
	movePuzzleInput((SkewbWidget) w, event->xbutton.x, event->xbutton.y,
		BR,
		(int) (event->xkey.state & ControlMask),
		(int) (event->xkey.state & (Mod1Mask | Mod3Mask | Mod4Mask | Mod5Mask)));
}
#endif
#endif
