/**
 * This file is a part of VideoCut package.
 * ---------------------------------------------------------------------- 
 * Copyright (C) 2007-2008 troorl
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * ---------------------------------------------------------------------- 
 *
 * @author troorl <troorl@gmail.com>
 */

#include "TGraphItem.h"

TGraphItem::TGraphItem(QImage &p, int style, qreal scale, QString timeText)
{	
    this->p = &p;
	this->timeText = timeText; // timeistamp

	this->h_s = (int)(this->p->height() * scale);
	this->w_s = this->p->width() * h_s / this->p->height();

    this->style = style; //set style of paint
	
	this->textPos = 0;
	shadowSize = 7;
	tmst_offset = 4;
	
	boundRect = QRectF(0 , 0, w_s, h_s); // result rect
	realPixRect = QRectF(0, 0, p.width(), p.height()); // real size of pixmap
	
	shadowPen = QPen(QColor(10, 10, 10, 0), -1);
	timeFont = QFont();
	timeFont.setBold(true);
	
	switch(this->style)
	{
		case 0:
		{
			shadowSize = 7;
			pix_h = h_s - shadowSize;
			pix_w = this->p->width() * h_s / this->p->height() - shadowSize;
			
			pixRect = QRectF(0 ,0, pix_w, pix_h);
			break;
		}
		case 1:
		{
			pixRect = QRectF(0, 0, w_s, h_s);
			break;
		}
	}
	adjustTimestamps();
}

void TGraphItem::setStyle(int style)
{
	this->style = style;
	switch(this->style)
	{
		case 0:
		{
			//shadowSize = 7;
			pix_h = (int)(boundingRect().height() - shadowSize);
			pix_w = this->p->width() * (int)boundingRect().height() / this->p->height() - shadowSize;
			
			pixRect = QRectF(0 ,0, pix_w, pix_h);
			break;
		}
		case 1:
		{
			pixRect = QRectF(0, 0, w_s, h_s);
			break;
		}
	}
	
	adjustTimestamps();
	this->prepareGeometryChange();
}

void TGraphItem::scale(qreal scale)
{
	this->h_s = (int)(this->p->height() * scale);
	this->w_s = this->p->width() * h_s / this->p->height();
	boundRect = QRectF(0 , 0, w_s, h_s); // result rect
	
	if(style == 0)
	{
		pix_h = (int)(boundingRect().height() - shadowSize);
		pix_w = p->width() * (int)boundingRect().height() / p->height() - shadowSize;
		pixRect = QRectF(0 ,0, pix_w, pix_h);
	}
	if(style == 1)
	{
		pixRect = boundRect;
	}
	adjustTimestamps();
	this->prepareGeometryChange();
}

void TGraphItem::adjustTimestamps()
{
	switch(style)
	{
		case(0):
		{
			textRect1 = QRect(tmst_offset, tmst_offset, pix_w - tmst_offset*2, pix_h - tmst_offset*2);
			textRect2 = QRect(tmst_offset + 1, tmst_offset + 1, pix_w - tmst_offset*2, pix_h - tmst_offset*2);
			break;
		}
		case(1):
		{
			textRect1 = QRect(tmst_offset, tmst_offset, w_s - tmst_offset*2, h_s - tmst_offset*2);
			textRect2 = QRect(tmst_offset + 1, tmst_offset + 1, w_s - tmst_offset*2, h_s - tmst_offset*2);
			break;
		}
	}
}

QRectF TGraphItem::boundingRect() const
{
	return boundRect;
}

void TGraphItem::paint(QPainter *painter, const QStyleOptionGraphicsItem *,
               QWidget *)
{
    switch (this->style)
    {
    case (0):
        {
            this->paint0(painter);
            break;
        }
    case (1):
        {
            this->paint1(painter);
            break;
        }
    }
}

void TGraphItem::setItemTextPos(uint textPos)
{
	if(textPos < 4)
	{
		this->textPos = textPos;
		this->update(bound);
	}
	
}

void TGraphItem::drawTimestamps(QPainter *painter)
{
	// set timestamps
	painter->setPen(QPen(QColor(20, 20, 20, 155), 1));
	//painter->setBrush(QBrush(QColor(255, 255, 255, 230)));
	painter->setFont(timeFont);
	
	//set position of timestamps
	QTextOption tempOpts;
	switch(textPos)
	{
		case 0:
		{
			tempOpts = Qt::AlignLeft | Qt::AlignTop;
			break;
		}
		case 1:
		{
			tempOpts = Qt::AlignRight | Qt::AlignTop;
			break;
		}
		case 2:
		{
			tempOpts = Qt::AlignLeft | Qt::AlignBottom;
			break;
		}
		case 3:
		{
			tempOpts = Qt::AlignRight | Qt::AlignBottom;
			break;
		}
	}
	
	painter->drawText(textRect2, timeText, tempOpts);
	painter->setPen(QPen(QColor(250, 250, 250, 255), 6));
	painter->drawText(textRect1, timeText, tempOpts);
}

void TGraphItem::paint0(QPainter *painter) //drawing first style
{
	painter->setPen(shadowPen);
	painter->setRenderHint(QPainter::Antialiasing, true); 
	
	uint b_w = (uint)boundRect.width();
	uint b_h = (uint)boundRect.height();
	
	for(int i = shadowSize; i >= 0; i--)
	{
		painter->setBrush(QBrush(QColor(50, 50, 50, 10 + i * 15)));
		painter->drawRoundRect(0, 0, b_w - i, b_h - i, 5, 5);
	}
	//painter->drawImage(pixRect, *p, realPixRect);//, Qt::AutoColor);
	painter->drawImage(pixRect, (*p).scaledToWidth(pixRect.width(), Qt::SmoothTransformation));
	
	drawTimestamps(painter);
}

void TGraphItem::paint1(QPainter *painter) //drawing second style
{
	painter->setRenderHint(QPainter::Antialiasing, true); 
	painter->setBrush(QColor(0, 0, 0, 255));
	
	painter->drawRect(pixRect);
	
	painter->drawImage(pixRect, (*p).scaledToWidth(realPixRect.width(), Qt::SmoothTransformation));
	
	drawTimestamps(painter);
}
