
from dolfin import *

parameters["form_compiler"]["name"]= "sfc"

class BoundaryFunction(Expression): 
    def eval(self, values, x): 
        values[0] = (0.5*(x[0] + x[1]))**2     
        values[1] = (1-0.5*(x[0] + x[1]))**2  
    def value_shape(self):
        return (2,)

class DirichletBoundary(SubDomain): 
    def inside(self, x, on_boundary): 
        return bool(on_boundary)

# Geometry
N = 80  
P = 5 

mesh = UnitSquare(N,N)

# Function spaces
V = VectorFunctionSpace(mesh, "CG", 1)

# BoundaryFunction function 
U = Function(V)

# Basis functions 
v = TestFunction(V)
u = TrialFunction(V)

# Forms
L = inner(grad(U),grad(U))*dx + dot(U,U)**P*dx  
F = derivative(L, U, v)

# Start vector 
boundary_function = BoundaryFunction(element=V.ufl_element())
x0 = project(boundary_function, V)

# Solve nonlinear variational problem
dirichlet_boundary = DirichletBoundary()
bc = DirichletBC(V, boundary_function, dirichlet_boundary)

solve(F==0, U, bc)

# Write solution to file
file = File("u.pvd") 
file << U 

# Plot and wait
plot(U)
interactive()
