"""
Nothing here but dictionaries for generating LinearSegmentedColormaps,
and a dictionary of these dictionaries.
"""
from __future__ import absolute_import, division, print_function

import numpy as np
import matplotlib.colors as colors
from sunpy.extern.six.moves import range, zip

__all__ = ['aia_color_table', 'lasco_color_table', 'eit_color_table',
           'sxt_color_table', 'xrt_color_table', 'trace_color_table',
           'sot_color_table', 'hmi_mag_color_table']


# FIXME: Give me a proper name.
def _mkx(i, steps, n):
    """ Generate list according to pattern of g0 and b0. """
    x = []
    for step in steps:
        x.extend(list(range(i, step + n, n)))
        i = step + (n - 1)
    return x


def padfr(lst, len_, pad=0):
    """ Pad lst to contain at least len_ items by adding pad to the front. """
    diff = len_ - len(lst)
    diff = 0 if diff < 0 else diff
    return [pad] * diff + lst


def paden(lst, len_, pad=0):
    """ Pad lst to contain at least len_ items by adding pad to the end. """
    diff = len_ - len(lst)
    diff = 0 if diff < 0 else diff
    return lst + [pad] * diff


# The following values describe color table 3 for IDL (Red Temperature)
r0 = np.array(paden(
      [  0,   1,   2,   4,   5,   7,   8,  10,  11,  13,  14,  15,  17,
        18,  20,  21,  23,  24,  26,  27,  28,  30,  31,  33,  34,  36,
        37,  39,  40,  42,  43,  44,  46,  47,  49,  50,  52,  53,  55,
        56,  57,  59,  60,  62,  63,  65,  66,  68,  69,  70,  72,  73,
        75,  76,  78,  79,  81,  82,  84,  85,  86,  88,  89,  91,  92,
        94,  95,  97,  98,  99, 101, 102, 104, 105, 107, 108, 110, 111,
       113, 114, 115, 117, 118, 120, 121, 123, 124, 126, 127, 128, 130,
       131, 133, 134, 136, 137, 139, 140, 141, 143, 144, 146, 147, 149,
       150, 152, 153, 155, 156, 157, 159, 160, 162, 163, 165, 166, 168,
       169, 170, 172, 173, 175, 176, 178, 179, 181, 182, 184, 185, 186,
       188, 189, 191, 192, 194, 195, 197, 198, 199, 201, 202, 204, 205,
       207, 208, 210, 211, 212, 214, 215, 217, 218, 220, 221, 223, 224,
       226, 227, 228, 230, 231, 233, 234, 236, 237, 239, 240, 241, 243,
       244, 246, 247, 249, 250, 252, 253], 256, 255))

g0 = np.array(padfr(_mkx(1, range(17, 256, 17), 2), 256))
b0 = np.array(padfr(_mkx(3, range(51, 256, 51), 4), 256))

c0 = np.arange(256, dtype='f')
c1 = (np.sqrt(c0) * np.sqrt(255.0)).astype('f')
c2 = (np.arange(256) ** 2 / 255.0).astype('f')
c3 = ((c1 + c2 / 2.0) * 255.0 / (c1.max() + c2.max() / 2.0)).astype('f')

aia_wave_dict = {1600: (c3, c3, c2), 1700: (c1, c0, c0), 4500: (c0, c0, b0 / 2.0),
                 94: (c2, c3, c0), 131: (g0, r0, r0), 171: (r0, c0, b0),
                 193: (c1, c0, c2), 211: (c1, c0, c3), 304: (r0, g0, b0),
                 335: (c2, c0, c1)
                 }

def aia_color_table(wavelength):
    """Returns one of the fundamental color tables for SDO AIA images.
       Based on aia_lct.pro part of SDO/AIA on SSWIDL written by
       Karel Schrijver (2010/04/12).
    """
    try:
        r, g, b = aia_wave_dict[wavelength]
    except KeyError:
        raise ValueError(
            "Invalid AIA wavelength. Valid values are "
            "1600,1700,4500,94,131,171,193,211,304,335."
        )

    # Now create the color dictionary in the correct format
    cdict = create_cdict(r, g, b)

    return colors.LinearSegmentedColormap('SDO AIA {:s}'.format(str(wavelength)), cdict)

eit_yellow_r = np.array(
      [  0,   1,   2,   3,   5,   6,   7,   8,  10,  11,  12,  14,  15,
        16,  17,  19,  20,  21,  22,  24,  25,  26,  28,  29,  30,  31,
        33,  34,  35,  36,  38,  39,  40,  42,  43,  44,  45,  47,  48,
        49,  51,  52,  53,  54,  56,  57,  58,  59,  61,  62,  63,  65,
        66,  67,  68,  70,  71,  72,  73,  75,  76,  77,  79,  80,  81,
        82,  84,  85,  86,  87,  89,  90,  91,  93,  94,  95,  96,  98,
        99, 100, 102, 103, 104, 105, 107, 108, 109, 110, 112, 113, 114,
       116, 117, 118, 119, 121, 122, 123, 124, 126, 127, 128, 130, 131,
       132, 133, 135, 136, 137, 138, 140, 141, 142, 144, 145, 146, 147,
       149, 150, 151, 153, 154, 155, 156, 158, 159, 160, 161, 163, 164,
       165, 167, 168, 169, 170, 172, 173, 174, 175, 177, 178, 179, 181,
       182, 183, 184, 186, 187, 188, 189, 191, 192, 193, 195, 196, 197,
       198, 200, 201, 202, 204, 205, 206, 207, 209, 210, 211, 212, 214,
       215, 216, 218, 219, 220, 221, 223, 224, 225, 226, 228, 229, 230,
       232, 233, 234, 235, 237, 238, 239, 240, 242, 243, 244, 246, 247,
       248, 249, 251, 252, 253, 255, 255, 255, 255, 255, 255, 255, 255,
       255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
       255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
       255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
       255, 255, 255, 255, 255, 255, 255, 255, 255])

eit_yellow_g = np.array(
      [  0,   1,   2,   3,   4,   5,   6,   7,   8,   9,  10,  11,  12,
        13,  14,  15,  16,  17,  18,  19,  20,  21,  22,  23,  24,  25,
        26,  27,  28,  29,  30,  31,  32,  33,  34,  35,  36,  37,  38,
        39,  41,  42,  43,  44,  45,  46,  47,  48,  49,  50,  51,  52,
        53,  54,  55,  56,  57,  58,  59,  60,  61,  62,  63,  64,  65,
        66,  67,  68,  69,  70,  71,  72,  73,  74,  75,  76,  77,  78,
        79,  80,  82,  83,  84,  85,  86,  87,  88,  89,  90,  91,  92,
        93,  94,  95,  96,  97,  98,  99, 100, 101, 102, 103, 104, 105,
       106, 107, 108, 109, 110, 111, 112, 113, 114, 115, 116, 117, 118,
       119, 120, 121, 123, 124, 125, 126, 127, 128, 129, 130, 131, 132,
       133, 134, 135, 136, 137, 138, 139, 140, 141, 142, 143, 144, 145,
       146, 147, 148, 149, 150, 151, 152, 153, 154, 155, 156, 157, 158,
       159, 160, 161, 162, 164, 165, 166, 167, 168, 169, 170, 171, 172,
       173, 174, 175, 176, 177, 178, 179, 180, 181, 182, 183, 184, 185,
       186, 187, 188, 189, 190, 191, 192, 193, 194, 195, 196, 197, 198,
       199, 200, 201, 202, 203, 205, 206, 206, 207, 209, 209, 210, 210,
       212, 213, 213, 215, 216, 216, 218, 219, 219, 221, 221, 222, 224,
       224, 225, 227, 227, 228, 230, 230, 231, 231, 232, 234, 234, 235,
       237, 237, 238, 240, 240, 241, 241, 243, 244, 244, 246, 247, 247,
       249, 250, 250, 252, 252, 253, 255, 255, 255])

eit_yellow_b = np.concatenate((np.zeros(201).astype('int'),
                               np.array(
      [  7,   7,  15,  22,  22,  30,  30,  37,  45,  45,  52,  60,  60,
        67,  75,  75,  82,  82,  90,  97,  97, 105, 112, 112, 120, 127,
       127, 135, 135, 142, 150, 150, 157, 165, 165, 172, 180, 180, 187,
       187, 195, 202, 202, 210, 217, 217, 225, 232, 232, 240, 240, 247,
       255, 255, 255])))

eit_dark_blue_r = np.concatenate((np.zeros(206).astype('int'),
                                  np.array(
      [  9,  13,  21,  25,  25,  29,  33,  41,  49,  53,  57,  65,  69,
        73,  77,  77,  85,  94,  98, 102, 110, 114, 118, 122, 134, 134,
       138, 142, 146, 154, 158, 162, 166, 179, 183, 183, 187, 191, 199,
       203, 207, 215, 223, 227, 231, 231, 235, 243, 247, 255])))

eit_dark_blue_g = np.concatenate((np.zeros(128).astype('int'),
                                  np.array(
      [  2,   2,   4,   5,   7,  12,  13,  15,  17,  20,  21,  21,  23,
        25,  29,  31,  33,  34,  37,  39,  41,  41,  44,  47,  49,  50,
        52,  55,  57,  60,  61,  61,  65,  66,  68,  69,  73,  76,  77,
        79,  82,  82,  84,  85,  87,  92,  94,  95,  97, 100, 102, 103,
       103, 105, 110, 111, 113, 114, 118, 119, 121, 122, 122, 127, 129,
       130, 132, 135, 137, 138, 142, 145, 145, 146, 148, 150, 153, 154,
       158, 159, 162, 164, 164, 166, 167, 170, 174, 175, 177, 180, 182,
       183, 185, 185, 188, 191, 193, 195, 198, 199, 201, 203, 207, 207,
       209, 211, 212, 215, 217, 219, 220, 225, 227, 227, 228, 230, 233,
       235, 236, 239, 243, 244, 246, 246, 247, 251, 252, 255])))

eit_dark_blue_b = np.concatenate((np.zeros(52).astype('int'),
                                  np.array(
      [  1,   4,   5,   6,   8,   8,  10,  12,  14,  16,  18,  20,  21,
        23,  25,  25,  28,  29,  31,  33,  35,  36,  37,  42,  43,  43,
        44,  46,  48,  50,  51,  52,  56,  58,  59,  61,  61,  63,  65,
        66,  69,  71,  73,  74,  75,  78,  78,  80,  81,  84,  86,  88,
        89,  90,  93,  94,  94,  97,  99, 101, 103, 104, 105, 108, 111,
       112, 112, 113, 116, 118, 119, 120, 124, 126, 127, 128, 131, 131,
       132, 134, 135, 139, 141, 142, 143, 146, 147, 147, 149, 150, 154,
       155, 157, 158, 161, 162, 164, 164, 166, 169, 170, 172, 173, 176,
       177, 180, 181, 181, 184, 185, 187, 188, 191, 193, 195, 196, 199,
       199, 200, 202, 203, 207, 208, 210, 211, 214, 215, 217, 217, 218,
       222, 223, 225, 226, 229, 230, 231, 233, 233, 237, 238, 240, 241,
       244, 245, 246, 249, 252, 252, 253, 255, 255, 255, 255, 255, 255,
       255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
       255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
       255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
       255, 255, 255, 255, 255, 255, 255, 255, 255])))

eit_dark_green_r = np.concatenate((np.zeros(130).astype('int'),
                                   np.array(
      [  1,   3,   4,   9,  11,  12,  14,  17,  19,  19,  20,  22,  27,
        29,  30,  32,  35,  37,  38,  38,  41,  45,  46,  48,  50,  53,
        54,  58,  59,  59,  62,  64,  66,  67,  71,  74,  75,  77,  80,
        80,  82,  83,  85,  90,  91,  93,  95,  98, 100, 101, 101, 103,
       108, 109, 111, 112, 116, 117, 119, 121, 121, 125, 127, 129, 130,
       133, 135, 137, 140, 143, 143, 145, 146, 148, 151, 153, 156, 158,
       161, 163, 163, 164, 166, 169, 172, 174, 175, 179, 180, 182, 183,
       183, 187, 190, 192, 193, 196, 198, 200, 201, 206, 206, 208, 209,
       211, 214, 216, 217, 219, 224, 225, 225, 227, 229, 232, 234, 235,
       238, 242, 243, 245, 245, 246, 250, 251, 255])))

eit_dark_green_g = np.concatenate((np.zeros(52).astype('int'),
                                   np.array(
      [  1,   3,   4,   5,   6,   6,   8,   9,  11,  12,  14,  15,  16,
        17,  19,  19,  21,  22,  23,  25,  26,  27,  28,  31,  32,  32,
        33,  34,  36,  37,  38,  39,  42,  43,  44,  45,  45,  47,  48,
        49,  51,  53,  54,  55,  56,  58,  58,  59,  60,  62,  64,  65,
        66,  67,  69,  70,  70,  72,  73,  75,  76,  77,  78,  80,  82,
        83,  83,  84,  86,  87,  88,  89,  92,  93,  94,  95,  97,  97,
        98,  99, 100, 103, 104, 105, 106, 108, 109, 109, 110, 111, 114,
       115, 116, 117, 119, 120, 121, 121, 123, 125, 126, 127, 128, 130,
       131, 133, 134, 134, 136, 137, 138, 139, 141, 143, 144, 145, 147,
       147, 148, 149, 150, 153, 154, 155, 156, 158, 159, 160, 160, 161,
       164, 165, 166, 167, 169, 170, 171, 172, 172, 175, 176, 177, 178,
       180, 181, 182, 184, 186, 186, 187, 188, 189, 191, 192, 194, 195,
       197, 198, 198, 199, 200, 202, 204, 205, 206, 208, 209, 210, 211,
       211, 213, 215, 216, 217, 219, 220, 221, 222, 225, 225, 226, 227,
       228, 230, 231, 232, 233, 236, 237, 237, 238, 239, 241, 242, 243,
       245, 247, 248, 249, 249, 250, 252, 253, 255])))

eit_dark_green_b = np.concatenate((np.zeros(197).astype('int'), np.array(
      [  3,  10,  17,  17,  20,  24,  27,  34,  37,  44,  48,  55,  58,
        58,  62,  65,  72,  79,  82,  86,  93,  96,  99, 103, 103, 110,
       117, 120, 124, 130, 134, 137, 141, 151, 151, 155, 158, 161, 168,
       172, 175, 179, 189, 192, 192, 196, 199, 206, 210, 213, 220, 227,
       230, 234, 234, 237, 244, 248, 255])))

eit_dark_red_r = np.concatenate((np.zeros(52).astype('int'), np.array(
      [  1,   3,   4,   5,   6,   6,   8,   9,  11,  12,  14,  15,  16,
        17,  19,  19,  21,  22,  23,  25,  26,  27,  28,  31,  32,  32,
        33,  34,  36,  37,  38,  39,  42,  43,  44,  45,  45,  47,  48,
        49,  51,  53,  54,  55,  56,  58,  58,  59,  60,  62,  64,  65,
        66,  67,  69,  70,  70,  72,  73,  75,  76,  77,  78,  80,  82,
        83,  83,  84,  86,  87,  88,  89,  92,  93,  94,  95,  97,  97,
        98,  99, 100, 103, 104, 105, 106, 108, 109, 109, 110, 111, 114,
       115, 116, 117, 119, 120, 121, 121, 123, 125, 126, 127, 128, 130,
       131, 133, 134, 134, 136, 137, 138, 139, 141, 143, 144, 145, 147,
       147, 148, 149, 150, 153, 154, 155, 156, 158, 159, 160, 160, 161,
       164, 165, 166, 167, 169, 170, 171, 172, 172, 175, 176, 177, 178,
       180, 181, 182, 184, 186, 186, 187, 188, 189, 191, 192, 194, 195,
       197, 198, 198, 199, 200, 202, 204, 205, 206, 208, 209, 210, 211,
       211, 213, 215, 216, 217, 219, 220, 221, 222, 225, 225, 226, 227,
       228, 230, 231, 232, 233, 236, 237, 237, 238, 239, 241, 242, 243,
       245, 247, 248, 249, 249, 250, 252, 253, 255])))

eit_dark_red_g = np.concatenate((np.zeros(148).astype('int'), np.array(
      [  1,   1,   5,   9,  11,  13,  15,  18,  20,  24,  26,  26,  30,
        32,  34,  35,  39,  43,  45,  47,  51,  51,  52,  54,  56,  62,
        64,  66,  68,  71,  73,  75,  75,  77,  83,  85,  86,  88,  92,
        94,  96,  98,  98, 103, 105, 107, 109, 113, 115, 117, 120, 124,
       124, 126, 128, 130, 134, 136, 139, 141, 145, 147, 147, 149, 151,
       154, 158, 160, 162, 166, 168, 170, 171, 171, 175, 179, 181, 183,
       187, 188, 190, 192, 198, 198, 200, 202, 204, 207, 209, 211, 213,
       219, 221, 221, 222, 224, 228, 230, 232, 236, 239, 241, 243, 243,
       245, 249, 251, 255])))

eit_dark_red_b = np.concatenate((np.zeros(204).astype('int'), np.array(
      [  3,   7,  15,  19,  27,  31,  31,  35,  39,  47,  54,  58,  62,
        70,  74,  78,  82,  82,  90,  98, 102, 105, 113, 117, 121, 125,
       137, 137, 141, 145, 149, 156, 160, 164, 168, 180, 184, 184, 188,
       192, 200, 204, 207, 215, 223, 227, 231, 231, 235, 243, 247, 255])))


def eit_color_table(wavelength):
    """Returns one of the fundamental color tables for SOHO EIT images."""
    # SOHO EIT Color tables
    # EIT 171 IDL Name EIT Dark Bot Blue
    # EIT 195 IDL Name EIT Dark Bot Green
    # EIT 284 IDL Name EIT Dark Bot Yellow
    # EIT 304 IDL Name EIT Dark Bot Red
    try:
        r, g, b = {
            171: (eit_dark_blue_r, eit_dark_blue_g, eit_dark_blue_b),
            195: (eit_dark_green_r, eit_dark_green_g, eit_dark_green_b),
            284: (eit_yellow_r, eit_yellow_g, eit_yellow_b),
            304: (eit_dark_red_r, eit_dark_red_g, eit_dark_red_b)
        }[wavelength]
    except KeyError:
        raise ValueError(
            "Invalid EIT wavelength. Valid values are "
            "171, 195, 284, 304."
        )

    # Now create the color dictionary in the correct format
    cdict = create_cdict(r, g, b)

    return colors.LinearSegmentedColormap('SOHO EIT {:s}'.format(str(wavelength)), cdict)

lasco_c2_r = np.concatenate((np.array(
      [   0.,    1.,    2.,    5.,    8.,   11.,   14.,   17.,   20.,
         23.,   26.,   28.,   31.,   34.,   37.,   42.,   44.,   47.,
         50.,   55.,   57.,   60.,   65.,   68.,   70.,   75.,   78.,
         82.,   85.,   88.,   92.,   95.,   99.,  102.,  107.,  110.,
        114.,  117.,  121.,  124.,  128.,  133.,  136.,  140.,  143.,
        147.,  152.,  155.,  159.,  163.,  166.,  170.,  175.,  178.,
        182.,  186.,  189.,  194.,  198.,  201.,  205.,  210.,  214.,
        217.,  221.,  226.,  230.,  233.,  237.,  241.,  246.,  250.,  253.]),
        255 * np.ones(183)))

lasco_c2_g = np.concatenate((np.zeros(52).astype('int'), np.array(
      [   1.,    5.,   11.,   17.,   20.,   26.,   32.,   35.,   41.,
         47.,   52.,   56.,   62.,   68.,   73.,   77.,   83.,   88.,
         94.,  100.,  103.,  109.,  115.,  120.,  126.,  130.,  136.,
        141.,  147.,  153.,  158.,  164.,  168.,  173.,  179.,  185.,
        190.,  196.,  202.,  207.,  213.,  217.,  222.,  228.,  234.,
        239.,  245.,  251.]), 255 * np.ones(156)))

lasco_c2_b = np.concatenate((np.zeros(78).astype('int'), np.array(
      [   0.,    7.,   19.,   31.,   43.,   54.,   66.,   74.,   86.,
         98.,  109.,  121.,  133.,  145.,  156.,  168.,  176.,  188.,
        200.,  211.,  223.,  235.,  247.]), 255 * np.ones(156)))

lasco_c3_r = np.concatenate((np.zeros(77).astype('int'), np.array(
      [   5.,   13.,   25.,   33.,   45.,   53.,   65.,   73.,   85.,
         94.,  106.,  114.,  126.,  134.,  146.,  154.,  166.,  175.,
        187.,  195.,  207.,  215.,  227.,  235.,  247.]), 255 * np.ones(154)))

lasco_c3_g = np.concatenate((np.zeros(39).astype('int'), np.array(
      [   4.,    7.,   12.,   15.,   20.,   23.,   28.,   31.,   36.,
         39.,   44.,   47.,   52.,   55.,   60.,   63.,   68.,   71.,
         76.,   79.,   84.,   87.,   92.,   95.,  100.,  103.,  108.,
        111.,  116.,  119.,  124.,  127.,  132.,  135.,  140.,  143.,
        148.,  151.,  156.,  159.,  164.,  167.,  172.,  175.,  180.,
        183.,  188.,  191.,  196.,  199.,  204.,  207.,  212.,  215.,
        220.,  223.,  228.,  231.,  236.,  239.,  244.,  247.,  252.,
        255.,  255.]),
        255 * np.ones(154)))

lasco_c3_b = np.concatenate((np.array(
      [   0.,    4.,    6.,   10.,   13.,   17.,   20.,   24.,   27.,
         31.,   33.,   37.,   40.,   44.,   47.,   51.,   54.,   58.,
         61.,   65.,   67.,   71.,   74.,   78.,   81.,   85.,   88.,
         92.,   94.,   99.,  101.,  105.,  108.,  112.,  115.,  119.,
        122.,  126.,  128.,  132.,  135.,  139.,  142.,  146.,  149.,
        153.,  155.,  160.,  162.,  166.,  169.,  173.,  176.,  180.,
        183.,  187.,  189.,  193.,  196.,  200.,  203.,  207.,  210.,
        214.,  217.,  221.,  223.,  227.,  230.,  234.,  237.,  241.,
        244.,  248.,  250.]), 255 * np.ones(181)))


def lasco_color_table(number):
    """Returns one of the fundamental color tables for SOHO LASCO images."""
    # SOHO LASCO Color tables
    # LASCO C2 white light IDL Name
    # LASCO C3 white light IDL Name
    try:
        r, g, b = {
            2: (lasco_c2_r, lasco_c2_g, lasco_c2_b),
            3: (lasco_c3_r, lasco_c3_g, lasco_c3_b),
        }[number]
    except KeyError:
        raise ValueError(
            "Invalid LASCO number. Valid values are "
            "2, 3."
        )

    # Now create the color dictionary in the correct format
    cdict = create_cdict(r, g, b)
    return colors.LinearSegmentedColormap('SOHO LASCO C{:s}'.format(str(number)), cdict)

# Translated from the JP2Gen IDL SXT code lct_yla_gold.pro.  Might be better
# to explicitly copy the numbers from the IDL calculation.  This is a little
# more compact.
sxt_gold_r = np.concatenate((np.linspace(0, 255, num=185, endpoint=False),
                             255 * np.ones(71)))
sxt_gold_g = 255 * (np.arange(256) ** 1.25) / (255.0 ** 1.25)
sxt_gold_b = np.concatenate((np.zeros(185),
                             255.0 * np.arange(71) / 71.0))

grayscale = np.arange(256)


def sxt_color_table(sxt_filter):
    """Returns one of the fundamental color tables for Yokhoh SXT images."""
    try:
        r, g, b = {
            'al': (sxt_gold_r, sxt_gold_g, sxt_gold_b),
            'wh': (grayscale, grayscale, grayscale)
        }[sxt_filter]
    except KeyError:
        raise ValueError(
            "Invalid SXT filter type number. Valid values are "
            "'al', 'wh'."
        )

    # Now create the color dictionary in the correct format
    cdict = create_cdict(r, g, b)
    return colors.LinearSegmentedColormap('Yohkoh SXT {:s}'.format(sxt_filter.title()), cdict)


def xrt_color_table():
    """Returns the color table used for all Hinode XRT images."""
    # Now create the color dictionary in the correct format
    cdict = create_cdict(r0, g0, b0)
    return colors.LinearSegmentedColormap('Hinode XRT', cdict)

# Stereo Secchi COR colour tables
stereo_cor1_b = np.array(
      [  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
         0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
         0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
         0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
         0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
         0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
         0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
         0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
         0,   0,   0,   0,   0,   0,   3,   6,   6,  10,  13,  13,  17,
        20,  20,  24,  27,  27,  31,  31,  34,  37,  37,  41,  44,  44,
        48,  48,  51,  55,  55,  58,  58,  62,  62,  65,  68,  68,  72,
        72,  75,  75,  79,  82,  82,  86,  86,  89,  89,  93,  93,  96,
        96,  99, 103, 103, 106, 106, 110, 110, 113, 113, 117, 117, 120,
       120, 124, 124, 127, 127, 130, 130, 134, 134, 137, 137, 141, 141,
       144, 144, 148, 148, 151, 151, 155, 155, 158, 158, 161, 161, 165,
       165, 168, 168, 168, 172, 172, 175, 175, 179, 179, 182, 182, 186,
       186, 189, 189, 189, 192, 192, 196, 196, 199, 199, 203, 203, 203,
       206, 206, 210, 210, 213, 213, 213, 217, 217, 220, 220, 223, 223,
       223, 227, 227, 230, 230, 234, 234, 234, 237, 237, 241, 241, 241,
       244, 244, 248, 248, 248, 251, 251, 255, 255])

stereo_cor1_g = np.array(
      [  0,  27,  36,  43,  48,  53,  57,  60,  64,  67,  69,  72,  75,
        77,  80,  82,  84,  86,  88,  90,  92,  94,  95,  97,  99, 100,
       102, 104, 105, 107, 108, 110, 111, 112, 114, 115, 116, 118, 119,
       120, 121, 123, 124, 125, 126, 127, 128, 129, 131, 132, 133, 134,
       135, 136, 137, 138, 139, 140, 141, 142, 143, 144, 145, 146, 147,
       147, 148, 149, 150, 151, 152, 153, 154, 154, 155, 156, 157, 158,
       159, 159, 160, 161, 162, 163, 163, 164, 165, 166, 167, 167, 168,
       169, 170, 170, 171, 172, 172, 173, 174, 175, 175, 176, 177, 177,
       178, 179, 179, 180, 181, 181, 182, 183, 183, 184, 185, 185, 186,
       187, 187, 188, 189, 189, 190, 190, 191, 192, 192, 193, 194, 194,
       195, 195, 196, 197, 197, 198, 198, 199, 199, 200, 201, 201, 202,
       202, 203, 203, 204, 205, 205, 206, 206, 207, 207, 208, 208, 209,
       209, 210, 211, 211, 212, 212, 213, 213, 214, 214, 215, 215, 216,
       216, 217, 217, 218, 218, 219, 219, 220, 220, 221, 221, 222, 222,
       223, 223, 224, 224, 225, 225, 226, 226, 227, 227, 228, 228, 229,
       229, 230, 230, 230, 231, 231, 232, 232, 233, 233, 234, 234, 235,
       235, 236, 236, 236, 237, 237, 238, 238, 239, 239, 240, 240, 240,
       241, 241, 242, 242, 243, 243, 243, 244, 244, 245, 245, 246, 246,
       246, 247, 247, 248, 248, 249, 249, 249, 250, 250, 251, 251, 251,
       252, 252, 253, 253, 253, 254, 254, 255, 255])

stereo_cor1_r = np.array(
      [  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
         0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   3,   4,
         8,  11,  12,  16,  17,  20,  22,  24,  27,  29,  30,  33,  35,
        37,  38,  41,  43,  45,  46,  48,  50,  51,  54,  56,  58,  59,
        61,  62,  64,  66,  67,  69,  71,  72,  74,  75,  77,  79,  80,
        80,  82,  83,  85,  87,  88,  90,  91,  91,  93,  95,  96,  98,
       100, 100, 101, 103, 104, 106, 106, 108, 109, 111, 112, 112, 114,
       116, 117, 117, 119, 121, 121, 122, 124, 125, 125, 127, 129, 129,
       130, 132, 132, 133, 135, 135, 137, 138, 138, 140, 142, 142, 143,
       145, 145, 146, 148, 148, 150, 150, 151, 153, 153, 154, 156, 156,
       158, 158, 159, 161, 161, 163, 163, 164, 164, 166, 167, 167, 169,
       169, 171, 171, 172, 174, 174, 175, 175, 177, 177, 179, 179, 180,
       180, 182, 183, 183, 185, 185, 187, 187, 188, 188, 190, 190, 192,
       192, 193, 193, 195, 195, 196, 196, 198, 198, 200, 200, 201, 201,
       203, 203, 204, 204, 206, 206, 208, 208, 209, 209, 211, 211, 213,
       213, 214, 214, 214, 216, 216, 217, 217, 219, 219, 221, 221, 222,
       222, 224, 224, 224, 225, 225, 227, 227, 229, 229, 230, 230, 230,
       232, 232, 234, 234, 235, 235, 235, 237, 237, 238, 238, 240, 240,
       240, 242, 242, 243, 243, 245, 245, 245, 246, 246, 248, 248, 248,
       250, 250, 251, 251, 251, 253, 253, 255, 255])

stereo_cor2_b = np.array(
      [  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
         0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
         0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
         0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
         0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
         0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
         7,  19,  31,  43,  54,  66,  74,  86,  98, 109, 121, 133, 145,
       156, 168, 176, 188, 200, 211, 223, 235, 247, 255, 255, 255, 255,
       255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
       255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
       255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
       255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
       255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
       255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
       255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
       255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
       255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
       255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
       255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
       255, 255, 255, 255, 255, 255, 255, 255, 255])

stereo_cor2_g = np.array(
      [  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
         0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
         0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
         0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
         1,   5,  11,  17,  20,  26,  32,  35,  41,  47,  52,  56,  62,
        68,  73,  77,  83,  88,  94, 100, 103, 109, 115, 120, 126, 130,
       136, 141, 147, 153, 158, 164, 168, 173, 179, 185, 190, 196, 202,
       207, 213, 217, 222, 228, 234, 239, 245, 251, 255, 255, 255, 255,
       255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
       255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
       255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
       255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
       255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
       255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
       255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
       255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
       255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
       255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
       255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
       255, 255, 255, 255, 255, 255, 255, 255, 255])

stereo_cor2_r = np.array(
      [  0,   1,   2,   5,   8,  11,  14,  17,  20,  23,  26,  28,  31,
        34,  37,  42,  44,  47,  50,  55,  57,  60,  65,  68,  70,  75,
        78,  82,  85,  88,  92,  95,  99, 102, 107, 110, 114, 117, 121,
       124, 128, 133, 136, 140, 143, 147, 152, 155, 159, 163, 166, 170,
       175, 178, 182, 186, 189, 194, 198, 201, 205, 210, 214, 217, 221,
       226, 230, 233, 237, 241, 246, 250, 253, 255, 255, 255, 255, 255,
       255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
       255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
       255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
       255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
       255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
       255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
       255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
       255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
       255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
       255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
       255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
       255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
       255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
       255, 255, 255, 255, 255, 255, 255, 255, 255])


def cor_color_table(number):
    """Returns one of the fundamental color tables for STEREO coronagraph
    images."""
    # STEREO COR Color tables
    try:
        r, g, b = {
            1: (stereo_cor1_r, stereo_cor1_g, stereo_cor1_b),
            2: (stereo_cor2_r, stereo_cor2_g, stereo_cor2_b),
        }[number]
    except KeyError:
        raise ValueError(
            "Invalid COR number. Valid values are "
            "1, 2."
        )

    # Now create the color dictionary in the correct format
    cdict = create_cdict(r, g, b)

    return colors.LinearSegmentedColormap('STEREO COR{:s}'.format(str(number)), cdict)


# Standard TRACE color RGB triples, as defined by the SSWIDL program
# trace_colors.pro.
# See http://hesperia.gsfc.nasa.gov/ssw/trace/idl/util/trace_colors.pro
trace_171_r = np.array(
      [  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
         0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
         0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
         0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
         1,   3,   5,   7,   8,  10,  12,  14,  15,  17,  19,  21,  22,
        24,  26,  28,  29,  31,  33,  35,  37,  38,  40,  42,  44,  45,
        47,  49,  51,  52,  54,  56,  58,  59,  61,  63,  65,  67,  68,
        70,  72,  74,  75,  77,  79,  81,  82,  84,  86,  88,  90,  91,
        93,  95,  97,  98, 100, 102, 104, 105, 107, 109, 111, 112, 114,
       116, 118, 119, 121, 123, 125, 127, 128, 130, 132, 134, 135, 137,
       139, 141, 142, 144, 146, 148, 150, 151, 153, 155, 157, 158, 160,
       162, 164, 165, 167, 169, 171, 172, 174, 176, 178, 180, 180, 180,
       181, 181, 182, 182, 183, 183, 184, 184, 185, 185, 186, 186, 187,
       187, 188, 188, 189, 189, 190, 190, 191, 191, 192, 192, 192, 193,
       193, 194, 194, 195, 195, 196, 196, 197, 197, 198, 198, 199, 199,
       200, 200, 201, 201, 202, 202, 203, 203, 204, 204, 204, 205, 205,
       206, 206, 207, 207, 208, 208, 209, 209, 210, 210, 211, 211, 212,
       212, 213, 213, 214, 214, 215, 215, 216, 216, 217, 217, 217, 218,
       218, 219, 219, 220, 220, 221, 221, 222, 222, 223, 223, 224, 224,
       225, 225, 226, 226, 227, 227, 228, 228, 229])

trace_171_g = np.array(
      [  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
         0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
         0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   1,   3,
         4,   6,   7,   9,  10,  12,  13,  15,  16,  18,  19,  21,  23,
        24,  26,  27,  29,  30,  32,  33,  35,  36,  38,  39,  41,  43,
        44,  46,  47,  49,  50,  52,  53,  55,  56,  58,  59,  61,  63,
        64,  66,  67,  69,  70,  72,  73,  75,  76,  78,  79,  81,  83,
        84,  86,  87,  89,  90,  92,  93,  95,  96,  98, 100, 101, 103,
       104, 106, 107, 109, 110, 112, 113, 115, 116, 118, 119, 121, 123,
       124, 126, 127, 129, 130, 132, 133, 135, 136, 138, 140, 141, 143,
       144, 146, 147, 149, 150, 152, 153, 155, 156, 158, 159, 161, 163,
       164, 166, 167, 169, 170, 172, 173, 175, 176, 178, 180, 180, 180,
       181, 181, 182, 182, 183, 183, 184, 184, 185, 185, 186, 186, 187,
       187, 188, 188, 189, 189, 190, 190, 191, 191, 192, 192, 192, 193,
       193, 194, 194, 195, 195, 196, 196, 197, 197, 198, 198, 199, 199,
       200, 200, 201, 201, 202, 202, 203, 203, 204, 204, 204, 205, 205,
       206, 206, 207, 207, 208, 208, 209, 209, 210, 210, 211, 211, 212,
       212, 213, 213, 214, 214, 215, 215, 216, 216, 217, 217, 217, 218,
       218, 219, 219, 220, 220, 221, 221, 222, 222, 223, 223, 224, 224,
       225, 225, 226, 226, 227, 227, 228, 228, 229])

trace_171_b = np.array(
      [  0,   1,   2,   3,   5,   6,   7,   9,  10,  11,  13,  14,  15,
        16,  18,  19,  20,  22,  23,  24,  26,  27,  28,  30,  31,  32,
        33,  35,  36,  37,  39,  40,  41,  43,  44,  45,  47,  48,  49,
        50,  52,  53,  54,  56,  57,  58,  60,  61,  62,  64,  65,  66,
        67,  69,  70,  71,  73,  74,  75,  77,  78,  79,  81,  82,  83,
        84,  86,  87,  88,  90,  91,  92,  94,  95,  96,  98,  99, 100,
       101, 103, 104, 105, 107, 108, 109, 111, 112, 113, 115, 116, 117,
       118, 120, 121, 122, 124, 125, 126, 128, 129, 130, 132, 133, 134,
       135, 137, 138, 139, 141, 142, 143, 145, 146, 147, 149, 150, 151,
       152, 154, 155, 156, 158, 159, 160, 162, 163, 164, 166, 167, 168,
       169, 171, 172, 173, 175, 176, 177, 179, 180, 181, 183, 184, 185,
       186, 188, 189, 190, 192, 193, 194, 196, 197, 198, 200, 200, 201,
       201, 202, 202, 203, 203, 204, 204, 205, 205, 206, 206, 207, 207,
       208, 208, 209, 209, 210, 211, 211, 212, 212, 213, 213, 214, 214,
       215, 215, 216, 216, 217, 217, 218, 218, 219, 219, 220, 220, 221,
       222, 222, 223, 223, 224, 224, 225, 225, 226, 226, 227, 227, 228,
       228, 229, 229, 230, 230, 231, 231, 232, 233, 233, 234, 234, 235,
       235, 236, 236, 237, 237, 238, 238, 239, 239, 240, 240, 241, 241,
       242, 242, 243, 244, 244, 245, 245, 246, 246, 247, 247, 248, 248,
       249, 249, 250, 250, 251, 251, 252, 252, 253])

trace_195_r = np.array(
      [  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
         0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
         0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
         0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
         1,   3,   5,   7,   8,  10,  12,  14,  15,  17,  19,  21,  22,
        24,  26,  28,  29,  31,  33,  35,  37,  38,  40,  42,  44,  45,
        47,  49,  51,  52,  54,  56,  58,  59,  61,  63,  65,  67,  68,
        70,  72,  74,  75,  77,  79,  81,  82,  84,  86,  88,  90,  91,
        93,  95,  97,  98, 100, 102, 104, 105, 107, 109, 111, 112, 114,
       116, 118, 119, 121, 123, 125, 127, 128, 130, 132, 134, 135, 137,
       139, 141, 142, 144, 146, 148, 150, 151, 153, 155, 157, 158, 160,
       162, 164, 165, 167, 169, 171, 172, 174, 176, 178, 180, 180, 180,
       181, 181, 182, 182, 183, 183, 184, 184, 185, 185, 186, 186, 187,
       187, 188, 188, 189, 189, 190, 190, 191, 191, 192, 192, 192, 193,
       193, 194, 194, 195, 195, 196, 196, 197, 197, 198, 198, 199, 199,
       200, 200, 201, 201, 202, 202, 203, 203, 204, 204, 204, 205, 205,
       206, 206, 207, 207, 208, 208, 209, 209, 210, 210, 211, 211, 212,
       212, 213, 213, 214, 214, 215, 215, 216, 216, 217, 217, 217, 218,
       218, 219, 219, 220, 220, 221, 221, 222, 222, 223, 223, 224, 224,
       225, 225, 226, 226, 227, 227, 228, 228, 229])

trace_195_g = np.array(
      [  0,   1,   2,   3,   5,   6,   7,   9,  10,  11,  13,  14,  15,
        16,  18,  19,  20,  22,  23,  24,  26,  27,  28,  30,  31,  32,
        33,  35,  36,  37,  39,  40,  41,  43,  44,  45,  47,  48,  49,
        50,  52,  53,  54,  56,  57,  58,  60,  61,  62,  64,  65,  66,
        67,  69,  70,  71,  73,  74,  75,  77,  78,  79,  81,  82,  83,
        84,  86,  87,  88,  90,  91,  92,  94,  95,  96,  98,  99, 100,
       101, 103, 104, 105, 107, 108, 109, 111, 112, 113, 115, 116, 117,
       118, 120, 121, 122, 124, 125, 126, 128, 129, 130, 132, 133, 134,
       135, 137, 138, 139, 141, 142, 143, 145, 146, 147, 149, 150, 151,
       152, 154, 155, 156, 158, 159, 160, 162, 163, 164, 166, 167, 168,
       169, 171, 172, 173, 175, 176, 177, 179, 180, 181, 183, 184, 185,
       186, 188, 189, 190, 192, 193, 194, 196, 197, 198, 200, 200, 201,
       201, 202, 202, 203, 203, 204, 204, 205, 205, 206, 206, 207, 207,
       208, 208, 209, 209, 210, 211, 211, 212, 212, 213, 213, 214, 214,
       215, 215, 216, 216, 217, 217, 218, 218, 219, 219, 220, 220, 221,
       222, 222, 223, 223, 224, 224, 225, 225, 226, 226, 227, 227, 228,
       228, 229, 229, 230, 230, 231, 231, 232, 233, 233, 234, 234, 235,
       235, 236, 236, 237, 237, 238, 238, 239, 239, 240, 240, 241, 241,
       242, 242, 243, 244, 244, 245, 245, 246, 246, 247, 247, 248, 248,
       249, 249, 250, 250, 251, 251, 252, 252, 253])

trace_195_b = np.array(
      [  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
         0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
         0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   1,   3,
         4,   6,   7,   9,  10,  12,  13,  15,  16,  18,  19,  21,  23,
        24,  26,  27,  29,  30,  32,  33,  35,  36,  38,  39,  41,  43,
        44,  46,  47,  49,  50,  52,  53,  55,  56,  58,  59,  61,  63,
        64,  66,  67,  69,  70,  72,  73,  75,  76,  78,  79,  81,  83,
        84,  86,  87,  89,  90,  92,  93,  95,  96,  98, 100, 101, 103,
       104, 106, 107, 109, 110, 112, 113, 115, 116, 118, 119, 121, 123,
       124, 126, 127, 129, 130, 132, 133, 135, 136, 138, 140, 141, 143,
       144, 146, 147, 149, 150, 152, 153, 155, 156, 158, 159, 161, 163,
       164, 166, 167, 169, 170, 172, 173, 175, 176, 178, 180, 180, 180,
       181, 181, 182, 182, 183, 183, 184, 184, 185, 185, 186, 186, 187,
       187, 188, 188, 189, 189, 190, 190, 191, 191, 192, 192, 192, 193,
       193, 194, 194, 195, 195, 196, 196, 197, 197, 198, 198, 199, 199,
       200, 200, 201, 201, 202, 202, 203, 203, 204, 204, 204, 205, 205,
       206, 206, 207, 207, 208, 208, 209, 209, 210, 210, 211, 211, 212,
       212, 213, 213, 214, 214, 215, 215, 216, 216, 217, 217, 217, 218,
       218, 219, 219, 220, 220, 221, 221, 222, 222, 223, 223, 224, 224,
       225, 225, 226, 226, 227, 227, 228, 228, 229])

trace_284_r = np.array(
      [  0,   1,   2,   3,   5,   6,   7,   9,  10,  11,  13,  14,  15,
        16,  18,  19,  20,  22,  23,  24,  26,  27,  28,  30,  31,  32,
        33,  35,  36,  37,  39,  40,  41,  43,  44,  45,  47,  48,  49,
        50,  52,  53,  54,  56,  57,  58,  60,  61,  62,  64,  65,  66,
        67,  69,  70,  71,  73,  74,  75,  77,  78,  79,  81,  82,  83,
        84,  86,  87,  88,  90,  91,  92,  94,  95,  96,  98,  99, 100,
       101, 103, 104, 105, 107, 108, 109, 111, 112, 113, 115, 116, 117,
       118, 120, 121, 122, 124, 125, 126, 128, 129, 130, 132, 133, 134,
       135, 137, 138, 139, 141, 142, 143, 145, 146, 147, 149, 150, 151,
       152, 154, 155, 156, 158, 159, 160, 162, 163, 164, 166, 167, 168,
       169, 171, 172, 173, 175, 176, 177, 179, 180, 181, 183, 184, 185,
       186, 188, 189, 190, 192, 193, 194, 196, 197, 198, 200, 200, 201,
       201, 202, 202, 203, 203, 204, 204, 205, 205, 206, 206, 207, 207,
       208, 208, 209, 209, 210, 211, 211, 212, 212, 213, 213, 214, 214,
       215, 215, 216, 216, 217, 217, 218, 218, 219, 219, 220, 220, 221,
       222, 222, 223, 223, 224, 224, 225, 225, 226, 226, 227, 227, 228,
       228, 229, 229, 230, 230, 231, 231, 232, 233, 233, 234, 234, 235,
       235, 236, 236, 237, 237, 238, 238, 239, 239, 240, 240, 241, 241,
       242, 242, 243, 244, 244, 245, 245, 246, 246, 247, 247, 248, 248,
       249, 249, 250, 250, 251, 251, 252, 252, 253])

trace_284_g = np.array(
      [  0,   1,   2,   3,   5,   6,   7,   9,  10,  11,  13,  14,  15,
        16,  18,  19,  20,  22,  23,  24,  26,  27,  28,  30,  31,  32,
        33,  35,  36,  37,  39,  40,  41,  43,  44,  45,  47,  48,  49,
        50,  52,  53,  54,  56,  57,  58,  60,  61,  62,  64,  65,  66,
        67,  69,  70,  71,  73,  74,  75,  77,  78,  79,  81,  82,  83,
        84,  86,  87,  88,  90,  91,  92,  94,  95,  96,  98,  99, 100,
       101, 103, 104, 105, 107, 108, 109, 111, 112, 113, 115, 116, 117,
       118, 120, 121, 122, 124, 125, 126, 128, 129, 130, 132, 133, 134,
       135, 137, 138, 139, 141, 142, 143, 145, 146, 147, 149, 150, 151,
       152, 154, 155, 156, 158, 159, 160, 162, 163, 164, 166, 167, 168,
       169, 171, 172, 173, 175, 176, 177, 179, 180, 181, 183, 184, 185,
       186, 188, 189, 190, 192, 193, 194, 196, 197, 198, 200, 200, 201,
       201, 202, 202, 203, 203, 204, 204, 205, 205, 206, 206, 207, 207,
       208, 208, 209, 209, 210, 211, 211, 212, 212, 213, 213, 214, 214,
       215, 215, 216, 216, 217, 217, 218, 218, 219, 219, 220, 220, 221,
       222, 222, 223, 223, 224, 224, 225, 225, 226, 226, 227, 227, 228,
       228, 229, 229, 230, 230, 231, 231, 232, 233, 233, 234, 234, 235,
       235, 236, 236, 237, 237, 238, 238, 239, 239, 240, 240, 241, 241,
       242, 242, 243, 244, 244, 245, 245, 246, 246, 247, 247, 248, 248,
       249, 249, 250, 250, 251, 251, 252, 252, 253])

trace_284_b = np.array(
      [  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
         0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
         0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
         0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
         1,   3,   5,   7,   8,  10,  12,  14,  15,  17,  19,  21,  22,
        24,  26,  28,  29,  31,  33,  35,  37,  38,  40,  42,  44,  45,
        47,  49,  51,  52,  54,  56,  58,  59,  61,  63,  65,  67,  68,
        70,  72,  74,  75,  77,  79,  81,  82,  84,  86,  88,  90,  91,
        93,  95,  97,  98, 100, 102, 104, 105, 107, 109, 111, 112, 114,
       116, 118, 119, 121, 123, 125, 127, 128, 130, 132, 134, 135, 137,
       139, 141, 142, 144, 146, 148, 150, 151, 153, 155, 157, 158, 160,
       162, 164, 165, 167, 169, 171, 172, 174, 176, 178, 180, 180, 180,
       181, 181, 182, 182, 183, 183, 184, 184, 185, 185, 186, 186, 187,
       187, 188, 188, 189, 189, 190, 190, 191, 191, 192, 192, 192, 193,
       193, 194, 194, 195, 195, 196, 196, 197, 197, 198, 198, 199, 199,
       200, 200, 201, 201, 202, 202, 203, 203, 204, 204, 204, 205, 205,
       206, 206, 207, 207, 208, 208, 209, 209, 210, 210, 211, 211, 212,
       212, 213, 213, 214, 214, 215, 215, 216, 216, 217, 217, 217, 218,
       218, 219, 219, 220, 220, 221, 221, 222, 222, 223, 223, 224, 224,
       225, 225, 226, 226, 227, 227, 228, 228, 229])

trace_1216_r = np.array(
      [  0,   0,   1,   2,   3,   4,   5,   6,   7,   8,   9,  10,  11,
        12,  13,  14,  15,  16,  17,  18,  19,  20,  21,  22,  23,  24,
        25,  26,  27,  28,  29,  30,  31,  32,  33,  34,  35,  36,  37,
        38,  39,  40,  41,  42,  43,  44,  45,  46,  47,  48,  49,  50,
        51,  52,  53,  54,  55,  56,  57,  58,  59,  60,  61,  62,  63,
        64,  65,  66,  67,  68,  69,  70,  71,  72,  73,  74,  75,  76,
        77,  78,  79,  80,  81,  82,  83,  84,  85,  86,  87,  88,  89,
        90,  91,  92,  93,  94,  95,  96,  97,  98,  99, 100, 101, 102,
       103, 104, 105, 106, 107, 108, 109, 110, 111, 112, 113, 114, 115,
       116, 117, 118, 119, 120, 121, 122, 123, 124, 125, 126, 127, 128,
       129, 130, 131, 132, 133, 134, 135, 136, 137, 138, 139, 140, 141,
       142, 143, 144, 145, 146, 147, 148, 149, 150, 151, 152, 153, 154,
       155, 156, 157, 158, 159, 160, 161, 162, 163, 164, 165, 166, 167,
       168, 169, 170, 171, 172, 173, 174, 175, 176, 177, 178, 179, 180,
       181, 182, 183, 184, 185, 186, 187, 188, 189, 190, 191, 192, 193,
       194, 195, 196, 197, 198, 199, 200, 201, 202, 203, 204, 205, 206,
       207, 208, 209, 210, 211, 212, 213, 214, 215, 216, 217, 218, 219,
       220, 221, 222, 223, 224, 225, 226, 227, 228, 229, 230, 231, 232,
       233, 234, 235, 236, 237, 238, 239, 240, 241, 242, 243, 244, 245,
       246, 247, 248, 249, 250, 251, 252, 253, 254])

trace_1216_g = np.array(
      [  0,   1,   2,   4,   5,   7,   8,  10,  11,  12,  14,  15,  17,
        18,  20,  21,  23,  24,  25,  27,  28,  30,  31,  33,  34,  35,
        37,  38,  40,  41,  43,  44,  46,  47,  48,  50,  51,  53,  54,
        56,  57,  58,  60,  61,  63,  64,  66,  67,  69,  70,  71,  73,
        74,  76,  77,  79,  80,  81,  83,  84,  86,  87,  89,  90,  92,
        93,  94,  96,  97,  99, 100, 102, 103, 104, 106, 107, 109, 110,
       112, 113, 115, 116, 117, 119, 120, 122, 123, 125, 126, 127, 129,
       130, 132, 133, 135, 136, 138, 139, 140, 142, 143, 145, 146, 148,
       149, 150, 152, 153, 155, 156, 158, 159, 161, 162, 163, 165, 166,
       168, 169, 171, 172, 173, 175, 176, 178, 179, 181, 182, 184, 185,
       186, 188, 189, 191, 192, 194, 195, 196, 198, 199, 201, 202, 204,
       205, 207, 208, 209, 211, 212, 214, 215, 217, 218, 220, 220, 220,
       220, 221, 221, 221, 222, 222, 222, 223, 223, 223, 224, 224, 224,
       225, 225, 225, 226, 226, 226, 227, 227, 227, 228, 228, 228, 229,
       229, 229, 230, 230, 230, 231, 231, 231, 232, 232, 232, 233, 233,
       233, 234, 234, 234, 235, 235, 235, 236, 236, 236, 237, 237, 237,
       238, 238, 238, 239, 239, 239, 240, 240, 240, 241, 241, 241, 242,
       242, 242, 243, 243, 243, 244, 244, 244, 245, 245, 245, 246, 246,
       246, 247, 247, 247, 248, 248, 248, 249, 249, 249, 250, 250, 250,
       251, 251, 251, 252, 252, 252, 253, 253, 253])

trace_1216_b = np.array(
      [  0,   1,   2,   4,   5,   7,   8,  10,  11,  12,  14,  15,  17,
        18,  20,  21,  23,  24,  25,  27,  28,  30,  31,  33,  34,  35,
        37,  38,  40,  41,  43,  44,  46,  47,  48,  50,  51,  53,  54,
        56,  57,  58,  60,  61,  63,  64,  66,  67,  69,  70,  71,  73,
        74,  76,  77,  79,  80,  81,  83,  84,  86,  87,  89,  90,  92,
        93,  94,  96,  97,  99, 100, 102, 103, 104, 106, 107, 109, 110,
       112, 113, 115, 116, 117, 119, 120, 122, 123, 125, 126, 127, 129,
       130, 132, 133, 135, 136, 138, 139, 140, 142, 143, 145, 146, 148,
       149, 150, 152, 153, 155, 156, 158, 159, 161, 162, 163, 165, 166,
       168, 169, 171, 172, 173, 175, 176, 178, 179, 181, 182, 184, 185,
       186, 188, 189, 191, 192, 194, 195, 196, 198, 199, 201, 202, 204,
       205, 207, 208, 209, 211, 212, 214, 215, 217, 218, 220, 220, 220,
       220, 221, 221, 221, 222, 222, 222, 223, 223, 223, 224, 224, 224,
       225, 225, 225, 226, 226, 226, 227, 227, 227, 228, 228, 228, 229,
       229, 229, 230, 230, 230, 231, 231, 231, 232, 232, 232, 233, 233,
       233, 234, 234, 234, 235, 235, 235, 236, 236, 236, 237, 237, 237,
       238, 238, 238, 239, 239, 239, 240, 240, 240, 241, 241, 241, 242,
       242, 242, 243, 243, 243, 244, 244, 244, 245, 245, 245, 246, 246,
       246, 247, 247, 247, 248, 248, 248, 249, 249, 249, 250, 250, 250,
       251, 251, 251, 252, 252, 252, 253, 253, 253])

trace_1550_r = np.array(
      [ 30,  31,  32,  33,  34,  35,  36,  37,  38,  39,  40,  41,  42,
        43,  44,  45,  46,  47,  48,  49,  50,  51,  53,  54,  55,  56,
        57,  58,  59,  60,  61,  62,  63,  64,  65,  66,  67,  68,  69,
        70,  71,  72,  73,  74,  76,  77,  78,  79,  80,  81,  82,  83,
        84,  85,  86,  87,  88,  89,  90,  91,  92,  93,  94,  95,  96,
        97,  99, 100, 101, 102, 103, 104, 105, 106, 107, 108, 109, 110,
       111, 112, 113, 114, 115, 116, 117, 118, 119, 120, 122, 123, 124,
       125, 126, 127, 128, 129, 130, 131, 132, 133, 134, 135, 136, 137,
       138, 139, 140, 141, 142, 143, 145, 146, 147, 148, 149, 150, 151,
       152, 153, 154, 155, 156, 157, 158, 159, 160, 161, 162, 163, 164,
       165, 166, 168, 169, 170, 171, 172, 173, 174, 175, 176, 177, 178,
       179, 180, 181, 182, 183, 184, 185, 186, 187, 188, 190, 190, 191,
       191, 192, 193, 193, 194, 194, 195, 196, 196, 197, 198, 198, 199,
       199, 200, 201, 201, 202, 203, 203, 204, 204, 205, 206, 206, 207,
       208, 208, 209, 209, 210, 211, 211, 212, 212, 213, 214, 214, 215,
       216, 216, 217, 217, 218, 219, 219, 220, 221, 221, 222, 222, 223,
       224, 224, 225, 226, 226, 227, 227, 228, 229, 229, 230, 231, 231,
       232, 232, 233, 234, 234, 235, 235, 236, 237, 237, 238, 239, 239,
       240, 240, 241, 242, 242, 243, 244, 244, 245, 245, 246, 247, 247,
       248, 249, 249, 250, 250, 251, 252, 252, 253])

trace_1550_g = np.array(
      [  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
         0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
         0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   1,   3,
         4,   6,   7,   9,  10,  12,  13,  15,  16,  18,  20,  21,  23,
        24,  26,  27,  29,  30,  32,  33,  35,  37,  38,  40,  41,  43,
        44,  46,  47,  49,  50,  52,  53,  55,  57,  58,  60,  61,  63,
        64,  66,  67,  69,  70,  72,  74,  75,  77,  78,  80,  81,  83,
        84,  86,  87,  89,  91,  92,  94,  95,  97,  98, 100, 101, 103,
       104, 106, 107, 109, 111, 112, 114, 115, 117, 118, 120, 121, 123,
       124, 126, 128, 129, 131, 132, 134, 135, 137, 138, 140, 141, 143,
       145, 146, 148, 149, 151, 152, 154, 155, 157, 158, 160, 161, 163,
       165, 166, 168, 169, 171, 172, 174, 175, 177, 178, 180, 181, 181,
       182, 182, 183, 183, 184, 184, 185, 185, 186, 186, 187, 187, 188,
       188, 189, 189, 190, 190, 191, 191, 192, 192, 193, 193, 194, 194,
       195, 195, 196, 196, 197, 197, 198, 198, 199, 199, 200, 200, 201,
       201, 202, 202, 203, 203, 204, 204, 205, 205, 206, 206, 207, 207,
       208, 208, 209, 209, 210, 210, 211, 211, 212, 212, 213, 213, 214,
       214, 215, 216, 216, 217, 217, 218, 218, 219, 219, 220, 220, 221,
       221, 222, 222, 223, 223, 224, 224, 225, 225, 226, 226, 227, 227,
       228, 228, 229, 229, 230, 230, 231, 231, 232])

trace_1550_b = np.array(
      [  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
         0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
         0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
         0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
         0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
         2,   4,   6,   8,  10,  12,  14,  16,  18,  20,  22,  24,  26,
        28,  30,  32,  34,  36,  38,  40,  42,  44,  46,  48,  50,  52,
        54,  56,  58,  60,  62,  64,  66,  68,  70,  73,  75,  77,  79,
        81,  83,  85,  87,  89,  91,  93,  95,  97,  99, 101, 103, 105,
       107, 109, 111, 113, 115, 117, 119, 121, 123, 125, 127, 129, 131,
       133, 135, 137, 139, 141, 143, 146, 148, 150, 152, 154, 156, 158,
       160, 162, 164, 166, 168, 170, 172, 174, 176, 178, 180, 181, 181,
       182, 182, 183, 183, 184, 184, 185, 185, 186, 186, 187, 187, 188,
       188, 189, 189, 190, 190, 191, 191, 192, 192, 193, 193, 194, 194,
       195, 195, 196, 196, 197, 197, 198, 198, 199, 199, 200, 200, 201,
       201, 202, 202, 203, 203, 204, 204, 205, 205, 206, 206, 207, 207,
       208, 208, 209, 209, 210, 210, 211, 211, 212, 212, 213, 213, 214,
       214, 215, 216, 216, 217, 217, 218, 218, 219, 219, 220, 220, 221,
       221, 222, 222, 223, 223, 224, 224, 225, 225, 226, 226, 227, 227,
       228, 228, 229, 229, 230, 230, 231, 231, 232])

trace_1600_r = np.array(
      [ 30,  31,  32,  33,  34,  35,  36,  37,  38,  39,  40,  41,  42,
        43,  44,  45,  46,  47,  48,  49,  50,  51,  53,  54,  55,  56,
        57,  58,  59,  60,  61,  62,  63,  64,  65,  66,  67,  68,  69,
        70,  71,  72,  73,  74,  76,  77,  78,  79,  80,  81,  82,  83,
        84,  85,  86,  87,  88,  89,  90,  91,  92,  93,  94,  95,  96,
        97,  99, 100, 101, 102, 103, 104, 105, 106, 107, 108, 109, 110,
       111, 112, 113, 114, 115, 116, 117, 118, 119, 120, 122, 123, 124,
       125, 126, 127, 128, 129, 130, 131, 132, 133, 134, 135, 136, 137,
       138, 139, 140, 141, 142, 143, 145, 146, 147, 148, 149, 150, 151,
       152, 153, 154, 155, 156, 157, 158, 159, 160, 161, 162, 163, 164,
       165, 166, 168, 169, 170, 171, 172, 173, 174, 175, 176, 177, 178,
       179, 180, 181, 182, 183, 184, 185, 186, 187, 188, 190, 190, 191,
       191, 192, 193, 193, 194, 194, 195, 196, 196, 197, 198, 198, 199,
       199, 200, 201, 201, 202, 203, 203, 204, 204, 205, 206, 206, 207,
       208, 208, 209, 209, 210, 211, 211, 212, 212, 213, 214, 214, 215,
       216, 216, 217, 217, 218, 219, 219, 220, 221, 221, 222, 222, 223,
       224, 224, 225, 226, 226, 227, 227, 228, 229, 229, 230, 231, 231,
       232, 232, 233, 234, 234, 235, 235, 236, 237, 237, 238, 239, 239,
       240, 240, 241, 242, 242, 243, 244, 244, 245, 245, 246, 247, 247,
       248, 249, 249, 250, 250, 251, 252, 252, 253])

trace_1600_g = np.array(
      [  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
         0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
         0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
         0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
         1,   3,   5,   7,   8,  10,  12,  14,  15,  17,  19,  21,  23,
        24,  26,  28,  30,  31,  33,  35,  37,  38,  40,  42,  44,  46,
        47,  49,  51,  53,  54,  56,  58,  60,  61,  63,  65,  67,  69,
        70,  72,  74,  76,  77,  79,  81,  83,  84,  86,  88,  90,  92,
        93,  95,  97,  99, 100, 102, 104, 106, 107, 109, 111, 113, 115,
       116, 118, 120, 122, 123, 125, 127, 129, 130, 132, 134, 136, 138,
       139, 141, 143, 145, 146, 148, 150, 152, 153, 155, 157, 159, 161,
       162, 164, 166, 168, 169, 171, 173, 175, 176, 178, 180, 181, 181,
       182, 182, 183, 183, 184, 184, 185, 185, 186, 186, 187, 187, 188,
       188, 189, 189, 190, 190, 191, 191, 192, 192, 193, 193, 194, 194,
       195, 195, 196, 196, 197, 197, 198, 198, 199, 199, 200, 200, 201,
       201, 202, 202, 203, 203, 204, 204, 205, 205, 206, 206, 207, 207,
       208, 208, 209, 209, 210, 210, 211, 211, 212, 212, 213, 213, 214,
       214, 215, 216, 216, 217, 217, 218, 218, 219, 219, 220, 220, 221,
       221, 222, 222, 223, 223, 224, 224, 225, 225, 226, 226, 227, 227,
       228, 228, 229, 229, 230, 230, 231, 231, 232])

trace_1600_b = np.array(
      [  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
         0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
         0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
         0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
         0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
         0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
         0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
         0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
         0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
         0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   5,
        11,  17,  22,  28,  34,  39,  45,  51,  57,  62,  68,  74,  79,
        85,  91,  96, 102, 108, 114, 119, 125, 131, 136, 142, 142, 143,
       143, 144, 144, 144, 145, 145, 146, 146, 146, 147, 147, 148, 148,
       148, 149, 149, 150, 150, 150, 151, 151, 152, 152, 152, 153, 153,
       154, 154, 154, 155, 155, 156, 156, 156, 157, 157, 158, 158, 158,
       159, 159, 160, 160, 160, 161, 161, 162, 162, 162, 163, 163, 164,
       164, 164, 165, 165, 166, 166, 166, 167, 167, 168, 168, 168, 169,
       169, 170, 170, 170, 171, 171, 172, 172, 172, 173, 173, 174, 174,
       174, 175, 175, 176, 176, 176, 177, 177, 178, 178, 178, 179, 179,
       180, 180, 180, 181, 181, 182, 182, 182, 183])

trace_1700_r = np.array(
      [  0,   1,   2,   3,   4,   5,   6,   7,   8,  10,  11,  12,  13,
        14,  15,  16,  17,  18,  20,  21,  22,  23,  24,  25,  26,  27,
        28,  30,  31,  32,  33,  34,  35,  36,  37,  38,  40,  41,  42,
        43,  44,  45,  46,  47,  48,  50,  51,  52,  53,  54,  55,  56,
        57,  58,  60,  61,  62,  63,  64,  65,  66,  67,  68,  70,  71,
        72,  73,  74,  75,  76,  77,  78,  80,  81,  82,  83,  84,  85,
        86,  87,  88,  90,  91,  92,  93,  94,  95,  96,  97,  98, 100,
       101, 102, 103, 104, 105, 106, 107, 108, 110, 111, 112, 113, 114,
       115, 116, 117, 118, 120, 121, 122, 123, 124, 125, 126, 127, 128,
       130, 131, 132, 133, 134, 135, 136, 137, 138, 140, 141, 142, 143,
       144, 145, 146, 147, 148, 150, 151, 152, 153, 154, 155, 156, 157,
       158, 160, 161, 162, 163, 164, 165, 166, 167, 168, 170, 170, 171,
       172, 173, 174, 174, 175, 176, 177, 178, 178, 179, 180, 181, 182,
       183, 183, 184, 185, 186, 187, 187, 188, 189, 190, 191, 192, 192,
       193, 194, 195, 196, 196, 197, 198, 199, 200, 200, 201, 202, 203,
       204, 205, 205, 206, 207, 208, 209, 209, 210, 211, 212, 213, 214,
       214, 215, 216, 217, 218, 218, 219, 220, 221, 222, 223, 223, 224,
       225, 226, 227, 227, 228, 229, 230, 231, 231, 232, 233, 234, 235,
       236, 236, 237, 238, 239, 240, 240, 241, 242, 243, 244, 245, 245,
       246, 247, 248, 249, 249, 250, 251, 252, 253])

trace_1700_g = np.array(
      [  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
         0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
         0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
         0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
         1,   3,   5,   6,   8,  10,  11,  13,  15,  16,  18,  20,  21,
        23,  25,  26,  28,  30,  31,  33,  35,  36,  38,  40,  41,  43,
        45,  46,  48,  50,  51,  53,  55,  56,  58,  60,  61,  63,  65,
        66,  68,  70,  71,  73,  75,  76,  78,  80,  81,  83,  85,  86,
        88,  90,  91,  93,  95,  96,  98, 100, 101, 103, 105, 106, 108,
       110, 111, 113, 115, 116, 118, 120, 121, 123, 125, 126, 128, 130,
       131, 133, 135, 136, 138, 140, 141, 143, 145, 146, 148, 150, 151,
       153, 155, 156, 158, 160, 161, 163, 165, 166, 168, 170, 170, 171,
       171, 172, 172, 173, 173, 174, 174, 175, 175, 176, 176, 177, 178,
       178, 179, 179, 180, 180, 181, 181, 182, 182, 183, 183, 184, 184,
       185, 186, 186, 187, 187, 188, 188, 189, 189, 190, 190, 191, 191,
       192, 192, 193, 194, 194, 195, 195, 196, 196, 197, 197, 198, 198,
       199, 199, 200, 200, 201, 202, 202, 203, 203, 204, 204, 205, 205,
       206, 206, 207, 207, 208, 208, 209, 210, 210, 211, 211, 212, 212,
       213, 213, 214, 214, 215, 215, 216, 216, 217, 218, 218, 219, 219,
       220, 220, 221, 221, 222, 222, 223, 223, 224])

trace_1700_b = np.array(
      [  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
         0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
         0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
         0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
         0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
         0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
         0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
         0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
         0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
         0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   3,
         7,  11,  14,  18,  22,  26,  29,  33,  37,  41,  44,  48,  52,
        56,  59,  63,  67,  71,  74,  78,  82,  86,  89,  93,  93,  94,
        94,  94,  94,  95,  95,  95,  96,  96,  96,  97,  97,  97,  97,
        98,  98,  98,  99,  99,  99,  99, 100, 100, 100, 101, 101, 101,
       102, 102, 102, 102, 103, 103, 103, 104, 104, 104, 104, 105, 105,
       105, 106, 106, 106, 107, 107, 107, 107, 108, 108, 108, 109, 109,
       109, 109, 110, 110, 110, 111, 111, 111, 112, 112, 112, 112, 113,
       113, 113, 114, 114, 114, 114, 115, 115, 115, 116, 116, 116, 116,
       117, 117, 117, 118, 118, 118, 119, 119, 119, 119, 120, 120, 120,
       121, 121, 121, 121, 122, 122, 122, 123, 123])

def trace_color_table(measurement):
    """Returns one of the standard color tables for TRACE JP2 files."""
    # TRACE color tables
    try:
        r, g, b = {
            '171': (trace_171_r, trace_171_g, trace_171_b),
            '195': (trace_195_r, trace_195_g, trace_195_b),
            '284': (trace_284_r, trace_284_g, trace_284_b),
            '1216': (trace_1216_r, trace_1216_g, trace_1216_b),
            '1550': (trace_1550_r, trace_1550_g, trace_1550_b),
            '1600': (trace_1600_r, trace_1600_g, trace_1600_b),
            '1700': (trace_1700_r, trace_1700_g, trace_1700_b),
            'WL': (grayscale, grayscale, grayscale)
        }[measurement]
    except KeyError:
        raise ValueError(
            "Invalid TRACE filter waveband passed.  Valid values are "
            "171, 195, 284, 1216, 1550, 1600, 1700, WL"
        )

    # Now create the color dictionary in the correct format
    cdict = create_cdict(r, g, b)

    # Return the color table
    return colors.LinearSegmentedColormap('TRACE {:s}'.format(measurement), cdict)

def sot_color_table(measurement):
    """Returns one of the standard color tables for SOT files (following osdc convention).
    The relations between observation and color have been defined in hinode.py"""
    try:
        r, g, b = {
            'intensity': (r0, g0, b0),
            # :todo
            # 'stokesQUV': (),
            # 'magnetic field': (),
            # 'velocity': (),
            # 'width': (),
            }[measurement]
    except KeyError:
        raise ValueError(
            "Invalid (or not supported) SOT type. Valid values are: "
            "intensity"  # TODO, stokesQUV, magnetic field, velocity, width."
            )

    cdict = create_cdict(r, g, b)
    return colors.LinearSegmentedColormap('Hinode SOT {:s}'.format(measurement), cdict)

def iris_sji_color_table(measurement, aialike=False):
    """Return the standard color table for IRIS SJI files"""
    # base vectors for IRIS SJI color tables
    c0 = np.arange(0, 256)
    c1 = (np.sqrt(c0) * np.sqrt(255)).astype(np.uint8)
    c2 = (c0 ** 2 / 255.).astype(np.uint8)
    c3= ((c1 + c2/2.)*255./(np.max(c1) + np.max(c2)/2.)).astype(np.uint8)
    c4 = np.zeros(256).astype(np.uint8)
    c4[50:256] = (1/165. * np.arange(0, 206) ** 2).astype(np.uint8)
    c5 = ((1 + c1 + c3.astype(np.uint))/2.).astype(np.uint8)

    rr = np.ones(256, dtype=np.uint8) * 255
    rr[0:176] = np.arange(0, 176) / 175. * 255.
    gg = np.zeros(256, dtype=np.uint8)
    gg[100:256] = np.arange(0, 156) / 155. * 255.
    bb = np.zeros(256, dtype=np.uint8)
    bb[150:256] = np.arange(0, 106) / 105. * 255.
    agg = np.zeros(256, dtype=np.uint8)
    agg[120:256] = np.arange(0, 136) / 135. * 255.
    abb = np.zeros(256, dtype=np.uint8)
    abb[190:256] = np.arange(0, 66) / 65. * 255.

    if aialike:
        color_table = {
            '1330': (c1, c0, c2),
            '1400': (rr, agg, abb),
            '2796': (rr, c0, abb),
            '2832': (c3, c3, c2),
        }
    else:
        color_table = {
            '1330': (rr, gg, bb),
            '1400': (c5, c2, c4),
            '2796': (c1, c3, c2),
            '2832': (c0, c0, c2),
        }

    color_table.update({
        '1600': (c1, c0, c0),
        '5000': (c1, c1, c0),
        'FUV': (rr, gg, bb),
        'NUV': (c1, c3, c2),
        'SJI_NUV': (c0, c0, c0)
    })

    try:
        r, g, b = color_table[measurement]
    except KeyError:
        raise ValueError(
            "Invalid IRIS SJI waveband.  Valid values are \n" +
            str(list(color_table.keys()))
        )

    # Now create the color dictionary in the correct format
    cdict = create_cdict(r, g, b)
    # Return the color table
    return colors.LinearSegmentedColormap('IRIS SJI {:s}'.format(measurement), cdict)


hmi_mag_r = np.array(
      [137, 140, 142, 144, 149, 151, 154, 158, 161, 163, 167, 170, 172,
       175, 179, 182, 184, 188, 191, 193, 195, 200, 202, 205, 209, 212,
       214, 218, 221, 223, 226, 230, 232, 235, 239, 242, 244, 248, 251,
       255, 253, 253, 253, 253, 253, 253, 253, 253, 253, 253, 253, 253,
       253, 253, 253, 253, 253, 253, 253, 253, 253, 253, 253, 253, 253,
       253, 253, 253, 253, 253, 253, 253, 253, 253, 253, 253, 253, 253,
       253, 253, 254, 254, 254, 254, 254, 254, 254, 254, 254, 254, 254,
       254, 254, 254, 254, 254, 254, 254, 254, 254, 254, 254, 254, 254,
       254, 254, 254, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
       255, 255, 255, 255, 255, 255, 255, 255, 255, 199, 181, 154,  95,
         0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
         0,   0,   0,   0,   0, 200, 197, 194, 191, 189, 186, 183, 181,
       178, 175, 173, 170, 167, 165, 162, 159, 157, 154, 151, 148, 146,
       143, 140, 138, 135, 132, 130, 127, 124, 122, 119, 116, 114, 111,
       108, 105, 103, 100,  97,  95,  92,  89,  87,  84,  81,  79,  76,
        73,  71,  68,  65,  63,  60,  56,  52,  49,  45,  42,  38,  35,
        31,  28,  24,  21,  17,  14,  10,   7,   3,   0,   0,   0,   0,
         0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
         0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
         0,   0,   0,   0,   0,   0,   0,   0,   0])

hmi_mag_g = np.array(
      [  0,   1,   2,   4,   4,   7,   8,  10,  11,  14,  14,  17,  17,
        20,  20,  23,  23,  26,  28,  28,  31,  31,  33,  35,  35,  38,
        40,  42,  42,  45,  45,  47,  49,  51,  52,  54,  56,  56,  59,
        61,  63,  65,  67,  70,  72,  75,  77,  80,  82,  85,  87,  90,
        92,  95,  97,  99, 101, 104, 106, 108, 111, 113, 115, 117, 120,
       122, 124, 127, 129, 131, 134, 136, 138, 140, 143, 145, 147, 150,
       152, 154, 157, 159, 161, 163, 166, 168, 170, 173, 175, 177, 179,
       182, 184, 186, 189, 191, 193, 196, 198, 200, 202, 205, 207, 209,
       212, 214, 216, 219, 103, 111, 120, 129, 138, 147, 156, 165, 174,
       183, 192, 201, 210, 219, 228, 237, 246, 255, 200, 181, 154, 129,
       120, 127, 135, 143, 151, 159, 167, 175, 183, 191, 199, 207, 215,
       223, 231, 239, 247, 255, 200, 198, 196, 194, 192, 190, 188, 186,
       184, 183, 181, 179, 177, 175, 173, 171, 169, 168, 166, 164, 162,
       160, 158, 156, 154, 152, 151, 149, 147, 145, 143, 141, 139, 137,
       136, 134, 132, 130, 128, 126, 124, 122, 120, 119, 117, 115, 113,
       111, 109, 107, 105, 104, 101,  98,  96,  93,  90,  88,  85,  82,
        80,  77,  74,  72,  69,  66,  64,  61,  58,  56,  54,  52,  51,
        49,  47,  45,  42,  42,  40,  38,  35,  35,  33,  31,  28,  28,
        26,  23,  23,  20,  20,  17,  17,  14,  13,  11,  10,   8,   4,
         4,   2,   1,   0,   0,   0,   0,   0,   0])

hmi_mag_b = np.array(
      [ 20,  17,  17,  17,  17,  17,  16,  16,  14,  14,  14,  14,  13,
        13,  11,  11,  11,  11,  10,  10,   8,   8,   8,   8,   7,   7,
         4,   4,   4,   4,   4,   4,   2,   2,   2,   2,   1,   1,   0,
         0,   2,   4,   7,  10,  13,  16,  19,  22,  25,  28,  31,  34,
        37,  40,  42,  45,  48,  50,  53,  56,  59,  61,  64,  67,  70,
        72,  75,  78,  81,  83,  86,  89,  92,  94,  97, 100, 103, 105,
       108, 111, 114, 116, 119, 122, 124, 127, 130, 133, 135, 138, 141,
       144, 146, 149, 152, 155, 157, 160, 163, 166, 168, 171, 174, 177,
       179, 182, 185, 188,   0,   0,   0,   0,   0,   0,   0,   0,   0,
         0,   0,   0,   0,   0,   0,   0,   0,   0, 130, 166, 167, 104,
        14,  18,  23,  27,  32,  36,  41,  46,  50,  55,  59,  64,  69,
        73,  78,  82,  87,  92, 255, 254, 254, 254, 254, 254, 254, 254,
       254, 254, 254, 254, 254, 254, 254, 254, 254, 254, 253, 253, 253,
       253, 253, 253, 253, 253, 253, 253, 253, 253, 253, 253, 253, 253,
       253, 252, 252, 252, 252, 252, 252, 252, 252, 252, 252, 252, 252,
       252, 252, 252, 252, 252, 252, 251, 251, 251, 251, 251, 251, 251,
       251, 251, 251, 251, 251, 251, 251, 251, 251, 251, 246, 244, 239,
       237, 232, 228, 226, 221, 219, 214, 211, 207, 204, 200, 195, 193,
       188, 186, 182, 179, 175, 172, 168, 165, 161, 158, 154, 149, 147,
       142, 140, 135, 133, 121, 109,  96,  84,  73])

def hmi_mag_color_table():
    """
    Returns an alternate HMI Magnetogram color table; from Stanford
    University/JSOC

    Examples
    --------
    >>> # Example usage for NRT data:
    >>> import sunpy.map
    >>> import sunpy.cm
    >>> hmi = sunpy.map.Map('fblos.fits')  # doctest: +SKIP
    >>> hmi.plot_settings['cmap'] = sunpy.cm.get_cmap('hmimag')  # doctest: +SKIP
    >>> hmi.peek(vmin=-1500.0, vmax=1500.0)  # doctest: +SKIP

    >>> # OR (for a basic plot with pixel values on the axes)
    >>> import numpy as np
    >>> import matplotlib.pyplot as plt
    >>> import sunpy.map
    >>> import sunpy.cm
    >>> hmi = sunpy.map.Map('fblos.fits')  # doctest: +SKIP
    >>> plt.imshow(np.clip(hmi.data, -1500.0, 1500.0), cmap=sunpy.cm.get_cmap('hmimag'), origin='lower')  # doctest: +SKIP
    >>> plt.show()  # doctest: +SKIP

    >>> # Example usage for science (Level 1.0) data:
    >>> import numpy as np
    >>> import sunpy.map
    >>> import sunpy.cm
    >>> hmi = sunpy.map.Map('hmi.m_45s.2014.05.11_12_00_45_TAI.magnetogram.fits')  # doctest: +SKIP
    >>> hmir = hmi.rotate()  # doctest: +SKIP
    >>> hmir.plot_settings['cmap'] = sunpy.cm.get_cmap('hmimag')  # doctest: +SKIP
    >>> hmir.peek(vmin=-1500.0, vmax=1500.0)  # doctest: +SKIP

    References
    ----------
    * `Stanford Colortable (pdf) <http://jsoc.stanford.edu/data/hmi/HMI_M.ColorTable.pdf>`_
    """
    cdict = create_cdict(hmi_mag_r, hmi_mag_g, hmi_mag_b)
    return colors.LinearSegmentedColormap('SDO HMI magnetogram', cdict)

hi1_blue = np.array(
       [  0,   1,   2,   4,   5,   6,   8,   9,  10,  12,  13,  14,  16,
         17,  18,  20,  21,  23,  24,  25,  27,  28,  29,  31,  32,  33,
         35,  36,  37,  39,  40,  42,  43,  44,  46,  47,  48,  50,  51,
         52,  54,  55,  56,  58,  59,  61,  62,  63,  65,  66,  67,  69,
         70,  71,  73,  74,  75,  77,  78,  80,  81,  82,  84,  85,  86,
         88,  89,  90,  92,  93,  94,  96,  97,  99, 100, 101, 103, 104,
        105, 107, 108, 109, 111, 112, 113, 115, 116, 118, 119, 120, 122,
        123, 124, 126, 127, 128, 130, 131, 132, 134, 135, 136, 138, 139,
        141, 142, 143, 145, 146, 147, 149, 150, 151, 153, 154, 155, 157,
        158, 160, 161, 162, 164, 165, 166, 168, 169, 170, 172, 173, 174,
        176, 177, 179, 180, 181, 183, 184, 185, 187, 188, 189, 191, 192,
        193, 195, 196, 198, 199, 200, 202, 203, 204, 206, 207, 208, 210,
        211, 212, 214, 215, 217, 218, 219, 221, 222, 223, 225, 226, 227,
        229, 230, 231, 233, 234, 236, 237, 238, 240, 241, 242, 244, 245,
        246, 248, 249, 250, 252, 253, 255, 255, 255, 255, 255, 255, 255,
        255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
        255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
        255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
        255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
        255, 255, 255, 255, 255, 255, 255, 255, 255], dtype=np.uint8)

hi1_green = np.array(
       [  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
          0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
          0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
          0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
          0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
          0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
          0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
          0,   0,   0,   0,   0,   0,   2,   4,   5,   7,   9,  10,  12,
         13,  15,  17,  18,  20,  21,  23,  25,  26,  28,  29,  31,  33,
         34,  36,  37,  39,  41,  42,  44,  45,  47,  49,  50,  52,  53,
         55,  57,  58,  60,  61,  63,  65,  66,  68,  69,  71,  73,  74,
         76,  77,  79,  81,  82,  84,  85,  87,  89,  90,  92,  94,  95,
         97,  98, 100, 102, 103, 105, 106, 108, 110, 111, 113, 114, 116,
        118, 119, 121, 122, 124, 126, 127, 129, 130, 132, 134, 135, 137,
        138, 140, 142, 143, 145, 146, 148, 150, 151, 153, 154, 156, 158,
        159, 161, 162, 164, 166, 167, 169, 170, 172, 174, 175, 177, 179,
        180, 182, 183, 185, 187, 188, 190, 191, 193, 195, 196, 198, 199,
        201, 203, 204, 206, 207, 209, 211, 212, 214, 215, 217, 219, 220,
        222, 223, 225, 227, 228, 230, 231, 233, 235, 236, 238, 239, 241,
        243, 244, 246, 247, 249, 251, 252, 254, 255], dtype=np.uint8)

hi1_red = np.array(
       [  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
          0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
          0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
          0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
          0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
          0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
          0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
          0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
          0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
          0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
          0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
          0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
          0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
          0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
          0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   5,   9,
         13,  17,  21,  25,  29,  33,  37,  41,  45,  49,  53,  57,  61,
         65,  69,  73,  77,  81,  85,  90,  94,  98, 102, 106, 110, 114,
        118, 122, 126, 130, 134, 138, 142, 146, 150, 154, 158, 162, 166,
        170, 175, 179, 183, 187, 191, 195, 199, 203, 207, 211, 215, 219,
        223, 227, 231, 235, 239, 243, 247, 251, 255], dtype=np.uint8)

hi2_blue = np.array(
       [  0,   0,   0,   0,   0,   0,   0,   0,   1,   1,   1,   1,   2,
          2,   2,   2,   2,   2,   2,   2,   3,   3,   3,   3,   4,   4,
          4,   4,   4,   5,   6,   7,   9,   9,  10,  11,  12,  13,  14,
         15,  17,  17,  18,  18,  19,  20,  22,  23,  25,  25,  26,  26,
         27,  28,  29,  30,  32,  32,  33,  33,  34,  35,  36,  37,  39,
         39,  40,  40,  41,  42,  43,  44,  46,  46,  47,  48,  49,  50,
         51,  52,  54,  54,  53,  53,  52,  54,  56,  58,  60,  60,  61,
         62,  63,  64,  65,  66,  68,  69,  70,  71,  73,  74,  75,  76,
         77,  78,  79,  80,  82,  82,  83,  83,  84,  85,  86,  87,  89,
         90,  91,  92,  94,  95,  96,  97,  98,  99, 100, 101, 103, 103,
        104, 105, 106, 107, 108, 109, 110, 111, 112, 113, 115, 116, 117,
        118, 119, 120, 121, 122, 124, 124, 125, 125, 126, 127, 129, 131,
        133, 134, 136, 138, 140, 141, 143, 144, 146, 147, 149, 151, 153,
        154, 156, 157, 159, 160, 161, 162, 164, 165, 166, 167, 169, 170,
        172, 173, 175, 176, 177, 178, 180, 181, 183, 184, 186, 187, 188,
        189, 191, 192, 193, 194, 196, 197, 199, 200, 202, 203, 204, 205,
        207, 208, 209, 210, 212, 213, 214, 215, 217, 218, 219, 220, 221,
        222, 223, 224, 226, 227, 228, 229, 230, 231, 232, 233, 235, 236,
        237, 238, 239, 240, 241, 242, 243, 243, 244, 245, 246, 246, 247,
        248, 249, 249, 250, 251, 252, 252, 253, 254], dtype=np.uint8)

hi2_green = np.array(
       [  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
          0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
          0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
          0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
          0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
          1,   2,   3,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,
          4,   4,   4,   5,   6,   7,   8,   8,   8,   8,   8,   8,   8,
          8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,
          8,   8,   8,   8,   8,   9,  10,  11,  12,  12,  12,  12,  12,
         12,  12,  12,  12,  12,  12,  12,  12,  12,  12,  12,  12,  12,
         12,  12,  12,  13,  14,  15,  16,  16,  16,  16,  16,  16,  16,
         16,  16,  16,  16,  16,  16,  16,  16,  16,  16,  17,  18,  19,
         20,  20,  20,  20,  20,  21,  22,  23,  24,  24,  24,  24,  24,
         25,  26,  27,  28,  30,  32,  34,  36,  38,  40,  42,  44,  46,
         48,  50,  52,  54,  56,  58,  60,  61,  62,  63,  64,  66,  68,
         70,  72,  74,  76,  78,  80,  82,  84,  86,  88,  90,  92,  94,
         96,  98, 100, 102, 104, 107, 110, 113, 116, 119, 122, 125, 128,
        131, 134, 137, 140, 143, 146, 149, 152, 155, 158, 161, 164, 167,
        170, 173, 176, 180, 184, 188, 192, 197, 202, 207, 212, 217, 222,
        227, 232, 237, 242, 247, 252, 252, 253, 254], dtype=np.uint8)

hi2_red = np.array(
       [  0,   0,   3,   6,  10,  13,  17,  20,  24,  27,  30,  33,  36,
         39,  42,  45,  48,  51,  54,  57,  60,  63,  66,  69,  72,  75,
         78,  81,  84,  85,  86,  87,  88,  90,  92,  94,  96,  97,  98,
         99, 100, 102, 104, 106, 108, 109, 110, 111, 112, 114, 116, 118,
        120, 121, 122, 123, 124, 126, 128, 130, 132, 133, 134, 135, 136,
        138, 140, 142, 144, 145, 146, 147, 148, 150, 152, 154, 156, 157,
        158, 159, 160, 162, 164, 166, 168, 169, 170, 171, 172, 174, 176,
        178, 180, 181, 182, 183, 184, 185, 186, 187, 188, 189, 190, 191,
        192, 193, 194, 195, 196, 198, 200, 202, 204, 205, 206, 207, 208,
        209, 210, 211, 212, 213, 214, 215, 216, 217, 218, 219, 220, 222,
        224, 226, 228, 229, 230, 231, 232, 233, 234, 235, 236, 237, 238,
        239, 240, 241, 242, 243, 244, 246, 248, 250, 252, 252, 252, 252,
        252, 252, 252, 252, 252, 252, 252, 252, 252, 252, 252, 252, 252,
        252, 252, 252, 252, 252, 252, 252, 252, 252, 252, 252, 252, 252,
        252, 252, 252, 252, 252, 252, 252, 252, 252, 252, 252, 252, 252,
        252, 252, 252, 252, 252, 252, 252, 252, 252, 252, 252, 252, 252,
        252, 252, 252, 252, 252, 252, 252, 252, 252, 252, 252, 252, 252,
        252, 252, 252, 252, 252, 252, 252, 252, 252, 252, 252, 252, 252,
        252, 252, 252, 252, 252, 252, 252, 252, 252, 252, 252, 252, 252,
        252, 252, 252, 252, 252, 252, 252, 253, 254], dtype=np.uint8)


def stereo_hi_color_table(camera):
    if camera == 1:
        cdict = create_cdict(hi1_red, hi1_green, hi1_blue)
        return colors.LinearSegmentedColormap('STEREO HI1', cdict)
    elif camera == 2:
        cdict = create_cdict(hi2_red, hi2_green, hi2_blue)
        return colors.LinearSegmentedColormap('STEREO HI2', cdict)
    else:
        raise ValueError("Valid HI cameras are 1 and 2")

def create_cdict(r, g, b):
    """Create the color tuples in the correct format."""
    i = np.linspace(0, 1, r0.size)

    cdict = dict(
        (name, list(zip(i, el / 255.0, el / 255.0)))
        for el, name in [(r, 'red'), (g, 'green'), (b, 'blue')]
    )
    return cdict
