/*
 * This file is part of the Ubuntu TV Media Scanner
 * Copyright (C) 2012-2013 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact: Jim Hodapp <jim.hodapp@canonical.com>
 * Authored by: Murray Cumming <murrayc@openismus.com>
 */

#include <grilo.h>
#include <stdio.h>

#define EXAMPLE_GRL_MEDIA_SCANNER_PLUGIN_ID "grl-mediascanner"

GMainLoop* loop = NULL;

guint remaining = 1;

static void
store_cb (GrlSource *source G_GNUC_UNUSED,
          GrlMedia *media G_GNUC_UNUSED,
          GList *failed_keys G_GNUC_UNUSED,
          gpointer user_data G_GNUC_UNUSED,
          const GError *error)
{
  if (error) {
    if (error->domain == GRL_CORE_ERROR) {
        g_warning ("Grilo Error while storing media "
            "(Maybe the mediscanner-service is not running):\n"
            "  %s", error->message);
    }
    else {
        g_warning ("Error while storing media:\n"
            "  %s", error->message);
    }
  } else {
    g_debug ("Media stored");
  }

  /** Stop the main loop when we are finished.
   */
  g_main_loop_quit (loop);
}

int main(int argc, char *argv[]) {

    /*
     * These defines are set by the build system.
     * Uncomment this to use the installed grilo plugins,
     * instead of the locally-built grilo plugins.
     */
    g_setenv (GRL_PLUGIN_PATH_VAR, EXAMPLE_GRILO_PLUGIN_DIR, TRUE);
    grl_init (&argc, &argv);

    /*
     * Load the Grilo plugin:
     */
    GrlRegistry *registry = grl_registry_get_default();

    GError *error = NULL;
    gboolean plugin_loaded = grl_registry_load_plugin_by_id (
        registry, EXAMPLE_GRL_MEDIA_SCANNER_PLUGIN_ID, &error);
    g_assert (plugin_loaded);
    g_assert_no_error (error);


    /**
     * Get the Grilo Source:
     */
    GrlSource *source =
        grl_registry_lookup_source(registry, EXAMPLE_GRL_MEDIA_SCANNER_PLUGIN_ID);
    g_assert (source);


    /*
     * Store some content:
     */
    g_assert (grl_source_supported_operations (source) & GRL_OP_STORE);
    GrlCaps *caps = grl_source_get_caps (source, GRL_OP_STORE);
    g_assert (caps);

    GrlOperationOptions *options = grl_operation_options_new (caps);
    grl_operation_options_set_count (options, 5);
    grl_operation_options_set_flags (options, GRL_RESOLVE_IDLE_RELAY);

    /*
     * Add a new item.
     * The grl-mediascanner plugin will discover and set other properties
     * such as duration.
     */
    const char* url =
        "file:///home/murrayc/Music/John%20Legend/John%20Legend%20-%20Once%20Again%20%5B2006%5D/04%20-%20Show%20Me.mp3";
    GrlMedia *media = grl_media_new ();
    grl_media_set_id (media, url); /* Optional, but must match the URL. */
    grl_media_set_url (media, url);
    grl_media_set_title (media, "Test Media");

    grl_source_store (source,
                      NULL,
                      media,
                      GRL_WRITE_NORMAL,
                      store_cb,
                      NULL);

    /*
     * Start the main loop so our callback can be called:
     */
    loop = g_main_loop_new (NULL, FALSE);
    g_main_loop_run (loop);

    /*
     * Release objects:
     */
    g_object_unref (media);
    g_object_unref (caps);
    g_object_unref (options);
    g_main_loop_unref (loop);

    return 0;
}
