
#include <stdlib.h>
#include <string.h>
#include <errno.h>

#include "ufc-crypt.h"
#include "xcrypt-private.h"
#include "crypt-private.h"

#define CRYPT_OUTPUT_SIZE		(7 + 22 + 31 + 1)
#define CRYPT_GENSALT_OUTPUT_SIZE	(7 + 22 + 1)

extern char *_xcrypt_blowfish_rn (__const char *key, __const char *salt,
				  char *output, int size);
extern char *_xcrypt_gensalt_blowfish_rn (unsigned long count,
					  __const char *input, int size,
					  char *output, int output_size);

extern unsigned char _xcrypt_itoa64[];
extern char *_xcrypt_gensalt_traditional_rn (unsigned long count,
					     __const char *input, int size,
					     char *output, int output_size);
extern char *_xcrypt_gensalt_extended_rn (unsigned long count,
					  __const char *input, int size,
					  char *output, int output_size);
extern char *_xcrypt_gensalt_md5_rn (unsigned long count, __const char *input,
				     int size, char *output, int output_size);
extern char *_xcrypt_gensalt_sha_rn (unsigned long count, __const char *input,
				     int size, char *output, int output_size);

extern struct crypt_data _ufc_foobar;

static int
_xcrypt_data_alloc (void **data, int *size, int need)
{
  void *updated;

  if (*data && *size >= need)
    return 0;

  updated = realloc (*data, need);

  if (!updated)
    {
      return -1;
    }

  if (need >= sizeof (struct crypt_data))
    ((struct crypt_data *) updated)->initialized = 0;

  *data = updated;
  *size = need;

  return 0;
}

static char *
_xcrypt_retval_magic (char *retval, __const char *salt, char *output)
{
  if (retval)
    return retval;

  output[0] = '*';
  output[1] = '0';
  output[2] = '\0';

  if (salt[0] == '*' && salt[1] == '0')
    output[1] = '1';

  return output;
}

/*
 * Applications may re-use the same instance of struct crypt_data without
 * resetting the initialized field in order to let crypt_r() skip some of
 * its initialization code. Thus, it is important that our multiple hashing
 * algorithms either don't conflict with each other in their use of the
 * data area or reset the initialized field themselves whenever required.
 * Currently, the hashing algorithms simply have no conflicts: the first
 * field of struct crypt_data is the 128-byte large DES key schedule which
 * __des_crypt_r() calculates each time it is called while the two other
 * hashing algorithms use less than 128 bytes of the data area.
 */

char *
__xcrypt_rn (__const char *key, __const char *salt, void *data, int size)
{
  if (salt[0] == '$' && salt[1] == '2')
    return _xcrypt_blowfish_rn (key, salt, (char *) data, size);
  if (salt[0] == '$' && salt[1] == '1')
    return __md5_crypt_r (key, salt, (char *) data, size);
  if (salt[0] == '$' || salt[0] == '_')
    {
      __set_errno (EINVAL);
      return NULL;
    }
  if (size >= sizeof (struct crypt_data))
    {
      if (strlen (salt) > 13)
	return __bigcrypt_r (key, salt, (struct crypt_data *) data);
      else
	return __des_crypt_r (key, salt, (struct crypt_data *) data);
    }

  __set_errno (ERANGE);
  return NULL;
}

char *
__xcrypt_ra (__const char *key, __const char *salt, void **data, int *size)
{
  if (salt[0] == '$' && salt[1] == '2')
    {
      if (_xcrypt_data_alloc (data, size, CRYPT_OUTPUT_SIZE))
	return NULL;
      return _xcrypt_blowfish_rn (key, salt, (char *) *data, *size);
    }
  if (salt[0] == '$' && salt[1] == '1')
    {
      if (_xcrypt_data_alloc (data, size, CRYPT_OUTPUT_SIZE))
	return NULL;
      return __md5_crypt_r (key, salt, (char *) *data, *size);
    }
  if (salt[0] == '$' || salt[0] == '_')
    {
      __set_errno (EINVAL);
      return NULL;
    }
  if (strncmp (salt, sha_salt_prefix, strlen (sha_salt_prefix)) == 0)
    {
      if (_xcrypt_data_alloc (data, size, CRYPT_OUTPUT_SIZE))
	return NULL;
      return __sha_crypt_r (key, salt, (char *) *data, *size);
    }
  if (_xcrypt_data_alloc (data, size, sizeof (struct crypt_data)))
    return NULL;

  if (strlen (salt) > 13)
    return __bigcrypt_r (key, salt, (struct crypt_data *) *data);
  else
    return __des_crypt_r (key, salt, (struct crypt_data *) *data);
}

char *
__xcrypt_r (__const char *key, __const char *salt, struct crypt_data *data)
{
  return _xcrypt_retval_magic (__xcrypt_rn (key, salt, data, sizeof (*data)),
			      salt, (char *) data);
}

char *
__xcrypt (__const char *key, __const char *salt)
{
  return
    _xcrypt_retval_magic (__xcrypt_rn
			 (key, salt, &_ufc_foobar, sizeof (_ufc_foobar)),
			 salt, (char *) &_ufc_foobar);
}

char *
__bigcrypt (__const char *key, __const char *salt)
{
  return __bigcrypt_r (key, salt, &_ufc_foobar);
}

char *
__xcrypt_gensalt_rn (__const char *prefix, unsigned long count,
		    __const char *input, int size, char *output,
		    int output_size)
{
  char *(*use) (unsigned long count,
		__const char *input, int size, char *output, int output_size);

  /* This may be supported on some platforms in the future */
  if (!input)
    {
      __set_errno (EINVAL);
      return NULL;
    }

  if (strncmp (prefix, "$2a$", 4) == 0)
    use = _xcrypt_gensalt_blowfish_rn;
  else if (strncmp (prefix, "$1$", 3) == 0)
    use = _xcrypt_gensalt_md5_rn;
  else if (strncmp (prefix, sha_salt_prefix, strlen (sha_salt_prefix)) == 0)
    use = _xcrypt_gensalt_sha_rn;
  else if (prefix[0] == '_')
    use = _xcrypt_gensalt_extended_rn;
  else
    if (!prefix[0] ||
	(prefix[0] && prefix[1] &&
	 memchr (_xcrypt_itoa64, prefix[0], 64) &&
	 memchr (_xcrypt_itoa64, prefix[1], 64)))
    use = _xcrypt_gensalt_traditional_rn;
  else
    {
      __set_errno (EINVAL);
      return NULL;
    }

  return use (count, input, size, output, output_size);
}

char *
__xcrypt_gensalt_ra (__const char *prefix, unsigned long count,
		    __const char *input, int size)
{
  char output[CRYPT_GENSALT_OUTPUT_SIZE];
  char *retval;

  retval = __xcrypt_gensalt_rn (prefix, count,
			       input, size, output, sizeof (output));

  if (retval)
    {
      retval = strdup (retval);
    }

  return retval;
}

char *
__xcrypt_gensalt (__const char *prefix, unsigned long count,
		 __const char *input, int size)
{
  static char output[CRYPT_GENSALT_OUTPUT_SIZE];

  return __xcrypt_gensalt_rn (prefix, count,
			     input, size, output, sizeof (output));
}

weak_alias (__xcrypt_rn, crypt_rn)
weak_alias (__xcrypt_ra, crypt_ra)
weak_alias (__xcrypt_r, crypt_r)
weak_alias (__xcrypt_rn, xcrypt_rn)
weak_alias (__xcrypt_ra, xcrypt_ra)
weak_alias (__xcrypt_r, xcrypt_r)
weak_alias (__xcrypt, crypt)
weak_alias (__xcrypt, xcrypt)
weak_alias (__xcrypt, fcrypt)
weak_alias (__bigcrypt, bigcrypt)
weak_alias (__xcrypt_gensalt_rn, crypt_gensalt_rn)
weak_alias (__xcrypt_gensalt_ra, crypt_gensalt_ra)
weak_alias (__xcrypt_gensalt, crypt_gensalt)
weak_alias (__xcrypt_gensalt_rn, xcrypt_gensalt_rn)
weak_alias (__xcrypt_gensalt_ra, xcrypt_gensalt_ra)
weak_alias (__xcrypt_gensalt, xcrypt_gensalt)
