/* ***************************************************************************
 *
 * Pico Technology USB TC-08 Device Driver
 *
 *//**
 * \file      TC08Api.cpp
 * \brief     C API to the USB TC-08 driver
 **//*
 *
 * Copyright (c) 2007, Pico Technology.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *  * Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *  * Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *  * The name of Pico Technology may not be used to endorse or promote
 *    products derived from this software without specific prior written
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED BY PICO TECHNOLOGY "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL PICO TECHNOLOGY BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * Version $Id: TC08Api.cpp,v 1.11 2008/06/03 07:32:47 erik Exp $
 *
 *************************************************************************** */


#include <vector>
#include <pthread.h>
#include "TC08Device.h"
#include "TC08Enumerator.h"

// ////////////////////////////////////////////////////////////////////////////
// 
// Global variables
// 
// ////////////////////////////////////////////////////////////////////////////

// Struct type to store in the device list containing pointer to the TC08Device
// object, the last error code for that device, and a mutex to protect the 
// error code (TC08Device objects are already threadsafe)
typedef struct TC08DeviceHandle {
	TC08Device * device;
	USBTC08_ERROR lastError;
	pthread_mutex_t lastErrorMutex;
};

// TC-08 Enumerator class and related variables
// Used to list the devices currently connected and open them
TC08Enumerator * enumerator = NULL;
// Mutex to protect us from creating two enumerators
// The enumerator class itself is thread-safe
pthread_mutex_t enumeratorMutex = PTHREAD_MUTEX_INITIALIZER;

// Vector to hold pointers to the TC08Device objects representing the TC-08s to
// which we have handles
std::vector<TC08DeviceHandle *> vecDevice;
// Mutex to protect vecDevice from simultaneous accesses
pthread_mutex_t vectorMutex = PTHREAD_MUTEX_INITIALIZER;

// Last error code returned by an open_unit call (i.e. one with no specific
// handle
USBTC08_ERROR lastError = USBTC08_ERROR_OK;
// Mutex to protect this
pthread_mutex_t lastErrorMutex = PTHREAD_MUTEX_INITIALIZER;

// A thread in which to perform async open operations
pthread_t * asyncThread = NULL;
// Invalid handle value to signify operation in progress
#define INVALID_ASYNC_HANDLE (-1000)
// Handle of the device currently being opened in async thread
short asyncHandle = INVALID_ASYNC_HANDLE;
// Mutex to protect these
pthread_mutex_t asyncMutex = PTHREAD_MUTEX_INITIALIZER;

// How many seconds should we wait for the enumerator to find devices
// (only waits if it can't find all the devices initially connected)
const unsigned int enumeratorWaitTime = 30; // in seconds

// ////////////////////////////////////////////////////////////////////////////
// 
// Helper function to get the last error code in threadsafe way
// 
// ////////////////////////////////////////////////////////////////////////////
USBTC08_ERROR TC08GetError(TC08DeviceHandle * deviceHandle)
{
	USBTC08_ERROR retval;
	if (!deviceHandle){
		// Get the global error code
		pthread_mutex_lock(&lastErrorMutex);
		retval = lastError;
		pthread_mutex_unlock(&lastErrorMutex);
	} else {
		pthread_mutex_lock(&(deviceHandle->lastErrorMutex));
		retval = deviceHandle->lastError;
		pthread_mutex_unlock(&(deviceHandle->lastErrorMutex));
	}
	return retval;
}

// ////////////////////////////////////////////////////////////////////////////
// 
// Helper function to get the last error code in threadsafe way
// 
// ////////////////////////////////////////////////////////////////////////////
void TC08SetError(TC08DeviceHandle * deviceHandle, USBTC08_ERROR errCode)
{
	if (!deviceHandle){
		// Get the global error code
		pthread_mutex_lock(&lastErrorMutex);
		lastError = errCode;
		pthread_mutex_unlock(&lastErrorMutex);
	} else {
		pthread_mutex_lock(&(deviceHandle->lastErrorMutex));
		deviceHandle->lastError = errCode;
		pthread_mutex_unlock(&(deviceHandle->lastErrorMutex));
	}
}



// ////////////////////////////////////////////////////////////////////////////
// 
// Helper function to get the TC08DeviceHandle object corresponding to a handle.
// Returns 1 for success, 0 for failure.
// 
 // ///////////////////////////////////////////////////////////////////////////
static short TC08SetUnit(short handle,
                     std::vector<TC08DeviceHandle *>::iterator &currentDevice)
{
	// Lock the mutex protecting g_vecDevice
	pthread_mutex_lock(&vectorMutex);
	
	// Find the number of devices we currently have handles to
	int nDevices = (int) vecDevice.size();
	
	// Iterate through the vector trying to match the handle
	currentDevice = vecDevice.begin();
	for (int deviceIndex = 0; deviceIndex < nDevices; deviceIndex++,
	                                                  currentDevice++) {
		if ((*currentDevice)->device->GetHandle() == handle)
		{
			pthread_mutex_unlock(&vectorMutex);
			return 1;
		}
	}
	
	// The requested device cannot be found
	pthread_mutex_unlock(&vectorMutex);
	return 0;
}


///////////////////////////////////////////////////////////////////////////////
/// Open a USB TC-08 unit for access by the driver. If multiple units are 
/// connected, the function may open any unit which is not already in use: call
/// it repeatedly to open all available units. The function blocks while  
/// firmware is downloaded to the unit - for a non-blocking alternative, see 
/// usb_tc08_open_unit_async.
/// <returns> A positive short integer handle to the TC-08 if it has been 
/// successfully opened. Zero if there are no more TC-08s available for opening,
/// and -1 if an error occurs when attempting to open the unit. </returns>
///////////////////////////////////////////////////////////////////////////////
PREF0 short usb_tc08_open_unit (void)
{
	TC08Device * dev;
	short status = USBTC08_PROGRESS_PENDING;

	TC08SetError(NULL, USBTC08_ERROR_OK);

	// Check if we need to start the enumerator
	// Lock its mutex first
	pthread_mutex_lock(&enumeratorMutex);
	if (!enumerator)
	{
		enumerator=new TC08Enumerator();
		if (enumerator){
			// Wait for the enumerator to download firmware to any devices
			// already present at startup
			int timeout = enumeratorWaitTime;
			while(!(enumerator->Ready()) && timeout-- > 0){
				sleep(1);
			}
			// It doesn't really matter if the enumerator times out as long 
			// as it finds at least one device
		} else {
			// Couldn't create the enumerator, give up;
			TC08SetError(NULL, USBTC08_ERROR_FW_FAIL);
			status = USBTC08_PROGRESS_FAIL;
		}
	}
	// Done with creating the enumerator
	pthread_mutex_unlock(&enumeratorMutex);
	
	if (status == USBTC08_PROGRESS_PENDING){
		// Try to open one TC-08
		dev = enumerator->Open();
		if (dev) {  //Found a TC-08 not already in use
			// Save our device in the vector
			pthread_mutex_lock(&vectorMutex);
			TC08DeviceHandle * newDevice = new TC08DeviceHandle; 
			if (!newDevice){
				TC08SetError(NULL, USBTC08_ERROR_FW_FAIL);
				status = USBTC08_PROGRESS_FAIL;
			} else {
				pthread_mutex_init(&(newDevice->lastErrorMutex), NULL);
				TC08SetError(newDevice, USBTC08_ERROR_OK);
				newDevice->device = dev;
				vecDevice.push_back(newDevice);
				status = USBTC08_PROGRESS_COMPLETE;
			}
			pthread_mutex_unlock(&vectorMutex);
		}
	}
	
	if (status == USBTC08_PROGRESS_COMPLETE){
		// Return the handle to our device
		return dev->GetHandle();
	} else {
		// Return error code;
		return status;
	}
	
	
}

// ////////////////////////////////////////////////////////////////////////////
// 
// Helper function run in a new thread by usb_tc08_open_unit_async to do the 
// actual opening.
// 
// ////////////////////////////////////////////////////////////////////////////
void *asyncOpenThreadProc(void *args) {

	// Open the device using a temporary handle variable
	short tempHandle = usb_tc08_open_unit();
	
	// Copy the handle across under mutex protection
	pthread_mutex_lock(&asyncMutex);
	asyncHandle = tempHandle;
	pthread_mutex_unlock(&asyncMutex);
	
	return NULL;
}

///////////////////////////////////////////////////////////////////////////////
/// 
/// Open a USB TC-08 unit without locking the calling thread while firmware
/// is downloaded. This function is used instead of usb_tc08_open_unit()
/// if the calling application needs to perform operations while
/// waiting for a unit to open. The TC-08 unit is opened in the background and
/// the function returns immediately. Use usb_tc08_open_unit_progress to 
/// find out when opening has completed and get the handle to the device
/// <returns> 1 if the opening operation started successfully, or 0 if it
/// could not be started. Call usb_tc08_get_last_error in this case to get 
/// more information </returns>
/// 
///////////////////////////////////////////////////////////////////////////////
PREF0 short usb_tc08_open_unit_async (void){
	
	short status = USBTC08_PROGRESS_PENDING;

	TC08SetError(NULL, USBTC08_ERROR_OK);

	// Ensure there isn't another open operation in progress
	pthread_mutex_lock(&asyncMutex);
	if (asyncThread){
		// Another async open operation is in progress
		status = USBTC08_PROGRESS_FAIL; // Other operation in progress
		TC08SetError(NULL, USBTC08_ERROR_ENUMERATION_INCOMPLETE);
	} else {
		
		asyncHandle = INVALID_ASYNC_HANDLE;
		
		// Start a new thread to open the device
		asyncThread = new pthread_t();
		if(pthread_create(asyncThread,NULL,asyncOpenThreadProc,NULL)) {
			status = USBTC08_PROGRESS_FAIL; // Error creating the thread
			TC08SetError(NULL, USBTC08_ERROR_THREAD_FAIL);
		} else {
			status = USBTC08_PROGRESS_COMPLETE; // Open operation started
		}
	}
	pthread_mutex_unlock(&asyncMutex);
	return status;
}

///////////////////////////////////////////////////////////////////////////////
/// 
/// Determine the progress of an async opening operation. Call this function 
/// after usb_tc08_open_unit_async to determine whether that operation has 
/// completed. 
/// <param name="handle">Pass in a pointer to a location in which the handle of
/// the opened unit can be stored</param>
/// <param name ="percent_progress">Pointer to a location in which the opening
/// progress from 0 to 100% can be stored. Note that on Linux due to the nature
/// of the download process the only progress values that can be retuned are 0%
/// (not complete) and 100% (complete).</param>
/// <returns>USBTC08_PROGRESS_PENDING if a unit is still being opened, 
/// USBTC08_PROGRESS_COMPLETE if the opening attempt has completed successfully
/// or USBTC08_PROGRESS_FAIL if the opening failed or an error was made calling
/// this function (handle NULL, or no async open operation in progress). Call
/// usb_tc08_get_last_error in this case to get more information.</returns>
/// 
///////////////////////////////////////////////////////////////////////////////

PREF0 short usb_tc08_open_unit_progress (short * handle,
                                         short * percent_progress){
	short status = USBTC08_PROGRESS_PENDING;
	short progress = 0;
	// Ensure handle is not NULL
	if (!handle){
		TC08SetError(NULL, USBTC08_ERROR_INVALID_PARAMETER);
		return USBTC08_PROGRESS_FAIL;
	}
	
	// Check if open operation in progress
	pthread_mutex_lock(&asyncMutex);
	if (!asyncThread){
		TC08SetError(NULL, USBTC08_ERROR_ENUMERATION_INCOMPLETE);
		status = USBTC08_PROGRESS_FAIL; // No operation in progress
	} else {
		// An operation has been started. See whether it has completed
		if (asyncHandle == INVALID_ASYNC_HANDLE) {
			TC08SetError(NULL, USBTC08_ERROR_OK); // In progress
		} else {
			// Operation has finshed, succesfully or not
			pthread_join(*asyncThread, NULL);
			delete asyncThread;
			asyncThread = NULL;
			progress = 100;
			*handle = asyncHandle;
			TC08SetError(NULL, asyncHandle > 0 ? USBTC08_ERROR_OK 
			                                   : USBTC08_ERROR_FW_FAIL);
			status = asyncHandle > 0 ? USBTC08_PROGRESS_COMPLETE 
			                         : USBTC08_PROGRESS_FAIL;
			asyncHandle = INVALID_ASYNC_HANDLE;
		}
	}
	pthread_mutex_unlock(&asyncMutex);
	if (percent_progress)
		*percent_progress = progress;
	return status;
}


///////////////////////////////////////////////////////////////////////////////
/// 
/// Close the TC08 with the specified handle. The handle becomes invalid and 
/// may be reused when subsequent TC-08 units are opened. 
/// <param name="handle">Handle identifying the unit to be closed</param>
/// <returns>1 for success, 0 for error (handle not valid, or operation failed).
/// Call usb_tc08_get_last_error in this case to get more information.</returns>
/// 
///////////////////////////////////////////////////////////////////////////////

PREF0 short usb_tc08_close_unit (short handle)
{
	short status = true;
	std::vector<TC08DeviceHandle *>::iterator currentDevice;
	
	if (!TC08SetUnit(handle, currentDevice))
	{
		// We don't SetLastError as if the user calls usb_tc08_get_last_error
		// they will do so with the invalid handle which will always return an
		// error code of -1
		return false;
	}
	
	// We don't SetLastError as the handle is now invalid anyway...
	// Close() always returns success
	status = ((*currentDevice)->device->Close() == PICO_SUCCESS);  

	return status;
}

///////////////////////////////////////////////////////////////////////////////
/// 
/// Stop the TC08 running in streaming mode.
/// <param name="handle">Handle identifying the unit to be stopped</param>
/// <returns>1 for success, 0 for error (handle not valid, or operation failed,
/// e.g. unit was not in streaming mode).
/// Call usb_tc08_get_last_error in this case to get more information.</returns>
/// 
///////////////////////////////////////////////////////////////////////////////

PREF0 short usb_tc08_stop (short handle)
{
	short status = true;
	std::vector<TC08DeviceHandle *>::iterator currentDevice;
	
	if (!TC08SetUnit(handle, currentDevice))
	{
		return false;
	}
	
	PICO_RETURNS retVal = (*currentDevice)->device->Stop();
	// Can return PICO_DEVICE_DISCONNECTED, PICO_UNIT_STATE_ERROR, PICO_SUCCESS
	switch (retVal) {
	case PICO_SUCCESS:
		TC08SetError(*currentDevice, USBTC08_ERROR_OK);
		break;
		
	case PICO_DEVICE_DISCONNECTED:
		TC08SetError(*currentDevice, USBTC08_ERROR_COMMUNICATION);
		status = false;
		break;
		
	case PICO_UNIT_STATE_ERROR:
		TC08SetError(*currentDevice, USBTC08_ERROR_INCORRECT_MODE);
		status = false;
		break;
		
	default:
		TC08SetError(*currentDevice, USBTC08_ERROR_NOT_RESPONDING);
		status = false;
		break;
		
	}

	return status;
}


///////////////////////////////////////////////////////////////////////////////
/// 
/// Set up mains interference rejection.
/// This changes the sampling time of the ADC in the TC08 to be a whole number
/// of mains cycles, ensuring that mains-related interference is cancelled out.
/// <param name="handle">Handle identifying the unit to be set</param>
/// <param name="sixty_hertz">USBTC08_MAINS_FREQUENCY value to set the mains 
/// frequency for rejection. This will vary by country, see 
/// USBTC08_MAINS_FREQUENCY</param>
/// <returns>1 for success, 0 for error (handle not valid, or operation failed).
/// Call usb_tc08_get_last_error in this case to get more information. 
/// </returns>
/// 
///////////////////////////////////////////////////////////////////////////////

PREF0 short usb_tc08_set_mains (short handle, short sixty_hertz)
{
	short status = true;
	std::vector<TC08DeviceHandle *>::iterator currentDevice;
	
	if (!TC08SetUnit(handle, currentDevice))
	{
		return false;
	}
	
	PICO_RETURNS retVal = (*currentDevice)->device->SetMains(
	                                    (USBTC08_MAINS_FREQUENCY)sixty_hertz);
	// Can return PICO_UNIT_STATE_ERROR, PICO_ARGUMENT_ERROR, 
	// PICO_DEVICE_DISCONNECTED, PICO_SUCCESS
	switch (retVal) {
	case PICO_SUCCESS:
		TC08SetError(*currentDevice, USBTC08_ERROR_OK);
		break;
		
	case PICO_DEVICE_DISCONNECTED:
		TC08SetError(*currentDevice, USBTC08_ERROR_COMMUNICATION);
		status = false;
		break;
		
	case PICO_ARGUMENT_ERROR:
		TC08SetError(*currentDevice, USBTC08_ERROR_INVALID_PARAMETER);
		status = false;
		break;
		
	case PICO_UNIT_STATE_ERROR:
		TC08SetError(*currentDevice, USBTC08_ERROR_INCORRECT_MODE);
		status = false;
		break;
		
	default:
		TC08SetError(*currentDevice, USBTC08_ERROR_NOT_RESPONDING);
		status = false;
		break;
		
	}

	return status;
}

///////////////////////////////////////////////////////////////////////////////
/// 
/// Find the fastest possible sampling rate in streaming mode with the
/// currently enabled set of channels. Set up all channels before calling this.
/// The allowable sampling interval will be 100ms * number of channels enabled,
/// including the CJC if needed.
/// <param name="handle">Handle identifying the unit to be set</param>
/// <returns>The fastest allowable sampling rate in streaming mode with the
/// currently enabled set of channels, in milliseconds. Returns 0 if an error
/// occurs (no channels enabled), use usb_tc08_get_last_error</returns>
/// 
///////////////////////////////////////////////////////////////////////////////

PREF0 long usb_tc08_get_minimum_interval_ms (short handle)
{
	short sampleInterval = 0;
	std::vector<TC08DeviceHandle *>::iterator currentDevice;
	
	if (TC08SetUnit(handle, currentDevice))
	{
		sampleInterval = (*currentDevice)->device->GetMinimumInterval();
		if (sampleInterval == 0){
			TC08SetError(*currentDevice, USBTC08_ERROR_NO_CHANNELS_SET);
		} else {
			TC08SetError(*currentDevice, USBTC08_ERROR_OK);
		}
	} 
	
	return sampleInterval;
}



///////////////////////////////////////////////////////////////////////////////
/// 
/// Get some information about the USB TC-08 unit in the form of a USBTC08_INFO
/// structure. 
/// <param name="handle">Handle identifying the unit to get info from. If you
/// pass zero, only the driver version will be filled in and the function
/// will succeed</param>
/// <param name="info">Pointer to USBTC08_INFO structure in which the  
/// information must be written. Your program must fill in the size of the  
/// USBTC08_INFO structure in the field size like this: 
/// info.size = sizeof(USBTC08_INFO) as a check against buffer overruns</param>
/// <returns>1 for success, 0 for error (handle not valid, or operation failed).
/// Call usb_tc08_get_last_error in this case to get more information. 
/// </returns>
/// 
///////////////////////////////////////////////////////////////////////////////
PREF0 short usb_tc08_get_unit_info (short handle, USBTC08_INFO * info)
{
	short status = true;
	std::vector<TC08DeviceHandle *>::iterator currentDevice;

	// Handle 0 is allowed for this func
	if (handle != 0 && !TC08SetUnit(handle, currentDevice)) {
		TC08SetError(NULL, USBTC08_ERROR_INVALID_PARAMETER);
		return false;
	}	

	// First check size and that info isn't null
	if (!info || info->size != sizeof(USBTC08_INFO)) {
		if (handle==0) {
			TC08SetError(NULL, USBTC08_ERROR_INVALID_PARAMETER);
		} else {
			TC08SetError(*currentDevice, USBTC08_ERROR_INVALID_PARAMETER);
		}
		return false;
	}
	
	sprintf(info->DriverVersion, "%d.%d.%d OpenSource",
			                     USBTC08_DRIVER_VERSION_MAJOR,
			                     USBTC08_DRIVER_VERSION_MINOR,
			                     USBTC08_DRIVER_VERSION_REVISION);
	
	// Deal with the special case handle == 0
	if (handle == 0){
		TC08SetError(NULL, USBTC08_ERROR_OK);
		return true;
	}
	
	const USBTC08_UNIT_INFO * devInfo=(*currentDevice)->device->GetUnitInfo();
	if (!devInfo){
		TC08SetError(*currentDevice, USBTC08_ERROR_NOT_RESPONDING);
		status = false;
		
	} else {
		TC08SetError(*currentDevice, USBTC08_ERROR_OK);
		info->PicoppVersion = 0;  // No equivalent on Linux 
		                          // (maybe libusb version?)
		info->Variant = 3; 	// As returned by windows driver
		info-> HardwareVersion = devInfo->HardwareVersion;
		memcpy(info->szSerial, devInfo->Serial, USBTC08_MAX_SERIAL_CHARS);
		memcpy(info->szCalDate, devInfo->CalDate, USBTC08_MAX_DATE_CHARS);
	}
	
	return status;
	
}


///////////////////////////////////////////////////////////////////////////////
/// 
/// Get some information about the USB TC-08 unit in the form of a string
/// representing a single line of the USBTC08_INFO structure. 
/// <param name="handle">Handle identifying the unit to get info from. This
/// can be zero only if asking for the driver version</param>
/// <param name="string">Pointer to char array in which the information 
/// must be written. Your program must fill in the size of the array
/// in the string_length parameter</param>
/// <param name="string_length">The size of the array pointed to by string
/// </param>
/// <param name="line">Line number of the information from USBTC08_INFO
/// to be returned</param>
/// <returns>1 for success, 0 for error (handle not valid, or operation failed).
/// Call usb_tc08_get_last_error in this case to get more information.
/// </returns>
/// 
///////////////////////////////////////////////////////////////////////////////
PREF0 short usb_tc08_get_unit_info2 (short handle, char * string, 
                                     short string_length, short line)
{
	short status = true;
	short retVal = 0;
	std::vector<TC08DeviceHandle *>::iterator currentDevice;

	// Handle 0 is allowed for this func
	if (handle != 0 && !TC08SetUnit(handle, currentDevice))
	{
		TC08SetError(NULL, USBTC08_ERROR_INVALID_PARAMETER);
		return false;
	}
	
	// Deal with the special case handle == 0
	if (handle == 0 && line != USBTC08LINE_DRIVER_VERSION){
		TC08SetError(NULL, USBTC08_ERROR_INVALID_PARAMETER);
		return false;
	}
	
	
	// Ensure string isn't NULL and string_length is 
	// at least 2 (1 char + terminator)
	if (!string || string_length < 2){
		if(handle == 0) {
			TC08SetError(NULL, USBTC08_ERROR_INVALID_PARAMETER);
		} else {
			TC08SetError(*currentDevice, USBTC08_ERROR_INVALID_PARAMETER);
		}
		return false;
	}
	
	
	
	// Get the whole unit info
	USBTC08_INFO unitInfo;
	unitInfo.size = sizeof(unitInfo);
	retVal = usb_tc08_get_unit_info (handle, &unitInfo);
	if (!retVal){
		// Error code has already been set
		return retVal;
	}


	// Select the line we need
	switch (line){
	case USBTC08LINE_DRIVER_VERSION:
		retVal = snprintf(string, string_length, "%s",unitInfo.DriverVersion);
		break;
		
	case USBTC08LINE_KERNEL_DRIVER_VERSION:
		retVal = snprintf(string, string_length, "%d",unitInfo.PicoppVersion);
		break;
		
	case USBTC08LINE_HARDWARE_VERSION:
		retVal = snprintf(string, string_length, "%d",unitInfo.HardwareVersion);
		break;
		
	case USBTC08LINE_VARIANT_INFO:
		retVal = snprintf(string, string_length, "%d",unitInfo.Variant);
		break;
		
	case USBTC08LINE_BATCH_AND_SERIAL:
		retVal = snprintf(string, string_length, "%s",unitInfo.szSerial);
		break;

	case USBTC08LINE_CAL_DATE:
		retVal = snprintf(string, string_length, "%s",unitInfo.szCalDate);
		break;
		
	default:
		TC08SetError(*currentDevice, USBTC08_ERROR_INVALID_PARAMETER);
		snprintf(string, string_length, "");
		status = false;
		break;
	}
	
	if (status) {
		if(handle==0) {
			TC08SetError(NULL, (retVal >= string_length) 
		             ? USBTC08_ERROR_INVALID_PARAMETER 
		             : USBTC08_ERROR_OK);
		} else {
			TC08SetError(*currentDevice, (retVal >= string_length) 
		             ? USBTC08_ERROR_INVALID_PARAMETER 
		             : USBTC08_ERROR_OK);
		}
		status = (retVal < string_length);
	}
	
	return status;
	
}

///////////////////////////////////////////////////////////////////////////////
/// 
/// Get some information about the USB TC-08 unit in the form of a string
/// representing a the whole USBTC08_INFO structure. The string consists of 
/// multiple lines, each line represents a fild of USBTC08_INFO.
/// <param name="handle">Handle identifying the unit to get info from. This
/// can be zeroin which case only the driver version is valid</param>
/// <param name="unit_info">Pointer to char array in which the information 
/// must be written. Your program must fill in the size of the array
/// in the string_length parameter</param>
/// <param name="string_length">The size of the array pointed to by unit_info
/// </param>
/// <returns>1 for success, 0 for error (handle not valid, or operation failed).
/// Call usb_tc08_get_last_error in this case to get more information.
/// </returns>
/// 
///////////////////////////////////////////////////////////////////////////////

PREF0 short usb_tc08_get_formatted_info (short handle, char * unit_info,
                                         short string_length)
{
	short retVal = 0;
	std::vector<TC08DeviceHandle *>::iterator currentDevice;

	// Handle 0 is allowed for this func
	if (handle != 0 && !TC08SetUnit(handle, currentDevice))
	{
		TC08SetError(NULL, USBTC08_ERROR_INVALID_PARAMETER);
		return false;
	}
	
	
	// Ensure string isn't NULL and string_length is 
	// at least 2 (1 char + terminator)
	if (!unit_info || string_length < 2){
		if (handle == 0)
		{
			TC08SetError(NULL, USBTC08_ERROR_INVALID_PARAMETER);
			return false;
		}
		else
		{
			TC08SetError(*currentDevice, USBTC08_ERROR_INVALID_PARAMETER);
			return false;
		}
	}
	
	
	
	// Get the whole unit info
	USBTC08_INFO unitInfo;
	unitInfo.size = sizeof(unitInfo);
	retVal = usb_tc08_get_unit_info (handle, &unitInfo);
	if (!retVal){
		// Error code has already been set
		return retVal;
	}
	
	// Deal with the special case handle == 0
	if (handle == 0) {

		// Format the info
		retVal = snprintf(unit_info, string_length, "%s",
				unitInfo.DriverVersion);

		TC08SetError(NULL, (retVal >= string_length) 
	             ? USBTC08_ERROR_INVALID_PARAMETER 
	             : USBTC08_ERROR_OK);		

	} else {
	
		// Format the info
		retVal = snprintf(unit_info, string_length, 
			          "%s\n%d\n%d\n%d\n%s\n%s",
			          unitInfo.DriverVersion,
			          unitInfo.PicoppVersion,
			          unitInfo.HardwareVersion,
			          unitInfo.Variant,
			          unitInfo.szSerial,
			          unitInfo.szCalDate);

		TC08SetError(*currentDevice, (retVal >= string_length) 
	             ? USBTC08_ERROR_INVALID_PARAMETER 
	             : USBTC08_ERROR_OK);
	}
	
	return (retVal < string_length);
		
}


///////////////////////////////////////////////////////////////////////////////
/// 
/// Get a more detailed return code describing the status of the last function 
/// call. Pass 0 as the unit handle to get the last reason that 
/// usb_tc08_open_unit() failed. 
/// <param name="handle">Handle identifying the unit to be queried</param>
/// <returns>USBTC08_ERROR value indicating the result of the last function call
/// to the specified unit</returns>
/// 
///////////////////////////////////////////////////////////////////////////////
PREF0 short usb_tc08_get_last_error (short handle)
{
	std::vector<TC08DeviceHandle *>::iterator currentDevice;

	// Handle 0 is allowed for this func
	if (handle != 0 && !TC08SetUnit(handle, currentDevice))
	{
		return -1; // As per windows driver
	}
	
	if (handle == 0)
		TC08SetError(NULL, USBTC08_ERROR_INVALID_PARAMETER);
	
	if (handle > 0)
		return (short)TC08GetError(*currentDevice);
	else
		return USBTC08_ERROR_INVALID_PARAMETER;
}

///////////////////////////////////////////////////////////////////////////////
/// 
/// Set up one channel of the USB TC-08 for thermocouple (or voltage)
/// measurements. Call this function once for each channel that you want to use.
/// The cold jucntion compensation (CJC) channel is always enabled is any
/// channels are set as thermocouples, or can be manually enabled.
/// <param name="handle">Handle identifying the unit to be set</param>
/// <param name="channel">Character value identifying the channel to be set
/// up, from 0 (CJC) to 8 (Channel 8)</param>
/// <param name="tc_type">Character value identifying the thermocouple type 
/// (B,E,J,K,N,R,S,T) for the selected channel, or X for voltage or a space 
/// to disable the channel.</param>
/// <returns>1 for success, 0 for error (handle not valid, or operation failed).
/// Call usb_tc08_get_last_error in this case to get more information. 
/// </returns>
/// 
///////////////////////////////////////////////////////////////////////////////


PREF0 short usb_tc08_set_channel ( short handle, short channel, char tc_type)
{
	short status = true;
	std::vector<TC08DeviceHandle *>::iterator currentDevice;
	
	if (!TC08SetUnit(handle, currentDevice))
	{
		return false;
	}
	
	PICO_RETURNS retVal =
	         (*currentDevice)->device->SetChannelConfig(channel, tc_type);
	
	// Can return PICO_UNIT_STATE_ERROR, PICO_ARGUMENT_ERROR, PICO_SUCCESS
	switch (retVal) {
	case PICO_SUCCESS:
		TC08SetError(*currentDevice, USBTC08_ERROR_OK);
		break;
		
	case PICO_ARGUMENT_ERROR:
		TC08SetError(*currentDevice, USBTC08_ERROR_INVALID_PARAMETER);
		status = false;
		break;
		
	case PICO_UNIT_STATE_ERROR:
		TC08SetError(*currentDevice, USBTC08_ERROR_INCORRECT_MODE);
		status = false;
		break;
		
	default:
		TC08SetError(*currentDevice, USBTC08_ERROR_NOT_RESPONDING);
		status = false;
		break;
		
	}

	return status;
}

///////////////////////////////////////////////////////////////////////////////
/// 
/// Set a TC-08 unit running in streaming mode. You must set up
/// the channels to be converted using usb_tc08_set_channel() before calling
/// this function.
/// <param name="handle">Handle identifying the unit to be set running</param>
/// <param name="interval_ms"> Sample interval in milliseconds to get one sample 
/// on all enabled channels. Must be between the values returned by GetMinimumInterval() 
/// and the maximum interval of 65535ms</param>
/// <returns>1 for success, 0 for error (handle not valid, or operation failed).
/// Call usb_tc08_get_last_error in this case to get more information. 
/// </returns>
/// 
///////////////////////////////////////////////////////////////////////////////


PREF0 long usb_tc08_run (short handle, long interval_ms)
{
	short status = true;
	std::vector<TC08DeviceHandle *>::iterator currentDevice;
	
	if (!TC08SetUnit(handle, currentDevice))
	{
		return false;
	}
	
	PICO_RETURNS retVal =
	         (*currentDevice)->device->Run(interval_ms);
	
	// Can return PICO_UNIT_STATE_ERROR, PICO_NO_ENABLED_CHANNELS,
	// PICO_ARGUMENT_ERROR, PICO_SUCCESS, PICO_DEVICE_DISCONNECTED
	switch (retVal) {
	case PICO_SUCCESS:
		TC08SetError(*currentDevice, USBTC08_ERROR_OK);
		break;
		
	case PICO_ARGUMENT_ERROR:
		TC08SetError(*currentDevice, USBTC08_ERROR_INVALID_PARAMETER);
		status = false;
		break;
		
	case PICO_NO_ENABLED_CHANNELS:
		TC08SetError(*currentDevice, USBTC08_ERROR_NO_CHANNELS_SET);
		status = false;
		break;
		
	case PICO_UNIT_STATE_ERROR:
		TC08SetError(*currentDevice, USBTC08_ERROR_INCORRECT_MODE);
		status = false;
		break;
		
	case PICO_DEVICE_DISCONNECTED:
	default:
		TC08SetError(*currentDevice, USBTC08_ERROR_NOT_RESPONDING);
		status = false;
		break;
		
	}

	return status;
}

///////////////////////////////////////////////////////////////////////////////
/// 
/// Get a single reading on all enabled channels of a USB TC-08. You must set up
/// the channels to be converted using usb_tc08_set_channel() before calling
/// this function, and the unit must not be in streaming mode 
/// <param name="handle">Handle identifying the unit from which to get the 
/// reading</param>
/// <param name="temp">Pointer to an array of 9 floats where the readings will be 
/// placed. Note that the array must always be of size 9 regardless of the number 
/// of channels enabled. Each reading is placed in the array location 
/// corresponding to its channel number, and disabled channels are filled
/// with NaN values</param>
/// <param name="overflow_flags">Pointer to an unsigned short which the driver  
/// sets as a bit-field indicating whether each of the channels was 
/// over-range. The LSB represents the CJC and the next 8 bits represent the 
/// 8 thermocouple channels. </param>
/// <param name="units">USBTC08_UNITS value indicating the temperature unit in 
/// which to return the results</param>
/// <returns>1 for success, 0 for error (handle not valid, or operation failed).
/// Call usb_tc08_get_last_error in this case to get more information. 
/// </returns>
/// 
///////////////////////////////////////////////////////////////////////////////

PREF0 short usb_tc08_get_single (
  short   handle, 
  float * temp,
  short * overflow_flags,
  short   units)
{
	short status = true;
	std::vector<TC08DeviceHandle *>::iterator currentDevice;
	
	if (!TC08SetUnit(handle, currentDevice))
	{
		return false;
	}
	

	PICO_RETURNS retVal =
	         (*currentDevice)->device->GetSingle(temp, 
	         (unsigned short *) overflow_flags, (USBTC08_UNITS) units);
	
	// Can return PICO_UNIT_STATE_ERROR, PICO_NO_ENABLED_CHANNELS,
	// PICO_ARGUMENT_ERROR, PICO_SUCCESS, PICO_DEVICE_DISCONNECTED
	switch (retVal) {
	case PICO_SUCCESS:
		TC08SetError(*currentDevice, USBTC08_ERROR_OK);
		break;
		
	case PICO_ARGUMENT_ERROR:
		TC08SetError(*currentDevice, USBTC08_ERROR_INVALID_PARAMETER);
		status = false;
		break;
		
	case PICO_NO_ENABLED_CHANNELS:
		TC08SetError(*currentDevice, USBTC08_ERROR_NO_CHANNELS_SET);
		status = false;
		break;
		
	case PICO_UNIT_STATE_ERROR:
		TC08SetError(*currentDevice, USBTC08_ERROR_INCORRECT_MODE);
		status = false;
		break;
		
	case PICO_DEVICE_DISCONNECTED:
	default:
		TC08SetError(*currentDevice, USBTC08_ERROR_NOT_RESPONDING);
		status = false;
		break;
		
	}

	return status;
}

///////////////////////////////////////////////////////////////////////////////
/// 
/// Retrieve buffered readings acquired in streaming mode from the driver.
/// Must be called at least once per minute to ensure no data is lost.
/// <param name="handle">Handle identifying the unit from which to get the 
/// readings</param>
/// <param name="temp_buffer">Address of an array of floats to store the sampled 
/// temperature values</param>
/// <param name="times_ms_buffer">Address of an array of longs to store the 
/// sample times</param>
/// <param name="buffer_length">Length of the temp and times bufferss</param>
/// <param name="overflow">Pointer to a short which is set non-zero by the 
/// driver if a buffer overflow (data loss) has occurred</param>
/// <param name="channel">The number of the channel from which to get the 
/// readings. 0 = CJC, 1 = Channel 1, etc.</param>
/// <param name="units">USBTC08_UNITS value indicating the temperature scale
/// to use when returning readings</param>
/// <param name="fill_missing">If true, any missin greadings are assigned 
/// the last known good value. Otherwise, they are assigned NaN</param>
/// <returns>The number of readings copied. Zero if no readings were available
/// or -1 if an error occurs</returns>
/// 
///////////////////////////////////////////////////////////////////////////////

PREF0 long usb_tc08_get_temp ( 
  short   handle,
  float * temp_buffer,
  long  * times_ms_buffer, 
  long    buffer_length,
  short * overflow,
  short   channel,
  short   units,
  short   fill_missing)

{
	short status = true;
	unsigned int length = (unsigned int)buffer_length;
	
	std::vector<TC08DeviceHandle *>::iterator currentDevice;
	
	if (!TC08SetUnit(handle, currentDevice))
	{
		return false;
	}
	

	PICO_RETURNS retVal =
	         (*currentDevice)->device->GetSingleChannelTimesAndValues(
	                     (USBTC08_CHANNELS)channel, temp_buffer, times_ms_buffer, 
	                     &length, (unsigned short *) overflow, 
	                     (USBTC08_UNITS) units, fill_missing, false);
	
	
	// Can return PICO_UNIT_STATE_ERROR, PICO_NO_ENABLED_CHANNELS,
	// PICO_ARGUMENT_ERROR, PICO_SUCCESS, PICO_DEVICE_DISCONNECTED
	switch (retVal) {
	case PICO_SUCCESS:
		TC08SetError(*currentDevice, USBTC08_ERROR_OK);
		break;
		
	case PICO_ARGUMENT_ERROR:
		TC08SetError(*currentDevice, USBTC08_ERROR_INVALID_PARAMETER);
		status = false;
		break;
		
	case PICO_NO_ENABLED_CHANNELS:
		TC08SetError(*currentDevice, USBTC08_ERROR_NO_CHANNELS_SET);
		status = false;
		break;
		
	case PICO_UNIT_STATE_ERROR:
		TC08SetError(*currentDevice, USBTC08_ERROR_INCORRECT_MODE);
		status = false;
		break;
		
	case PICO_DEVICE_DISCONNECTED:
	default:
		TC08SetError(*currentDevice, USBTC08_ERROR_NOT_RESPONDING);
		status = false;
		break;
		
	}

	return status=true ? length : -1;
	
}


///////////////////////////////////////////////////////////////////////////////
/// 
/// Retrieve buffered readings acquired in streaming mode from the driver.
/// Must be called at least once per minute to ensure no data is lost.
/// <param name="handle">Handle identifying the unit from which to get the 
/// readings</param>
/// <param name="temp_buffer">Address of an array of floats to store the sampled 
/// temperature values</param>
/// <param name="times_ms_buffer">Address of an array of longs to store the 
/// sample times</param>
/// <param name="buffer_length">Length of the temp and times bufferss</param>
/// <param name="overflow">Pointer to a short which is set non-zero by the 
/// driver if a buffer overflow (data loss) has occurred</param>
/// <param name="channel">The number of the channel from which to get the 
/// readings. 0 = CJC, 1 = Channel 1, etc.</param>
/// <param name="units">USBTC08_UNITS value indicating the temperature scale
/// to use when returning readings</param>
/// <param name="fill_missing">If true, any missin greadings are assigned 
/// the last known good value. Otherwise, they are assigned NaN</param>
/// <returns>The number of readings copied. Zero if no readings were available
/// or -1 if an error occurs</returns>
/// 
///////////////////////////////////////////////////////////////////////////////

PREF0 long usb_tc08_get_temp_deskew ( 
  short   handle,
  float * temp_buffer,
  long  * times_ms_buffer, 
  long    buffer_length,
  short * overflow,
  short   channel,
  short   units,
  short   fill_missing)

{
	short status = true;
	unsigned int length = (unsigned int)buffer_length;
	
	std::vector<TC08DeviceHandle *>::iterator currentDevice;
	
	if (!TC08SetUnit(handle, currentDevice))
	{
		return false;
	}
	

	PICO_RETURNS retVal =
	         (*currentDevice)->device->GetSingleChannelTimesAndValues(
	                     (USBTC08_CHANNELS)channel, temp_buffer, times_ms_buffer, 
	                     &length, (unsigned short *) overflow, 
	                     (USBTC08_UNITS) units, fill_missing, true);
	
	
	// Can return PICO_UNIT_STATE_ERROR, PICO_NO_ENABLED_CHANNELS,
	// PICO_ARGUMENT_ERROR, PICO_SUCCESS, PICO_DEVICE_DISCONNECTED
	switch (retVal) {
	case PICO_SUCCESS:
		TC08SetError(*currentDevice, USBTC08_ERROR_OK);
		break;
		
	case PICO_ARGUMENT_ERROR:
		TC08SetError(*currentDevice, USBTC08_ERROR_INVALID_PARAMETER);
		status = false;
		break;
		
	case PICO_NO_ENABLED_CHANNELS:
		TC08SetError(*currentDevice, USBTC08_ERROR_NO_CHANNELS_SET);
		status = false;
		break;
		
	case PICO_UNIT_STATE_ERROR:
		TC08SetError(*currentDevice, USBTC08_ERROR_INCORRECT_MODE);
		status = false;
		break;
		
	case PICO_DEVICE_DISCONNECTED:
	default:
		TC08SetError(*currentDevice, USBTC08_ERROR_NOT_RESPONDING);
		status = false;
		break;
		
	}

	return status=true ? length : -1;
	
}




