/* blockLongEffect.t.cc
 */

#include "osl/move_classifier/blockLongEffect.h"
#include "osl/record/csaString.h"

#include <cppunit/TestCase.h>
#include <cppunit/extensions/HelperMacros.h>

#include <iostream>

class BlockLongEffectTest : public CppUnit::TestFixture 
{
  CPPUNIT_TEST_SUITE(BlockLongEffectTest);
  CPPUNIT_TEST(testBlockRookBlack);
  CPPUNIT_TEST(testBlockRookWhite);
  CPPUNIT_TEST(testBlockBishopBlack);
  CPPUNIT_TEST(testBlockBishopWhite);
  CPPUNIT_TEST(testBlockLanceBlack);
  CPPUNIT_TEST(testBlockLanceWhite);
  CPPUNIT_TEST_SUITE_END();
public:
  void testBlockRookBlack();
  void testBlockRookWhite();
  void testBlockBishopBlack();
  void testBlockBishopWhite();
  void testBlockLanceBlack();
  void testBlockLanceWhite();
};

CPPUNIT_TEST_SUITE_REGISTRATION(BlockLongEffectTest);

using namespace osl;
using namespace osl::move_classifier;

static
bool isBlockRookTest(const NumEffectState& state, Move m)
{
  assert(state.isValidMove(m,true));
  const Ptype ptype = m.ptype();
  const Square from = m.from();
  const Square to = m.to();
  if (m.player() == BLACK)
    return BlockLongEffect<BLACK,ROOK>::isMember(state, ptype, from, to);
  else
    return BlockLongEffect<WHITE,ROOK>::isMember(state, ptype, from, to);
}

void BlockLongEffectTest::testBlockRookBlack()
{
  const char *problem = 
    "P1-KY *  * -OU * +KA *  *  * \n"
    "P2 * +GI *  *  *  *  * -KY * \n"
    "P3 *  * -KE *  *  * +KI *  * \n"
    "P4-FU *  * -KE-GI-FU-FU * -FU\n"
    "P5 *  *  *  *  *  *  * -RY * \n"
    "P6+FU+FU+KI-FU-FU *  *  * +FU\n"
    "P7 *  * +KE *  *  * +FU *  * \n"
    "P8 * +OU * -GI *  *  * +FU * \n"
    "P9+KY *  *  *  *  * +KE * +KY\n"
    "P+00FU00GI\n"
    "P-00AL\n"
    "+\n";
  NumEffectState state((CsaString(problem).getInitialState()));
  // 駒を取る手は入らない
  {
    Move move(Square(3,3),Square(2,2),GOLD,LANCE,false,BLACK);
    CPPUNIT_ASSERT((!isBlockRookTest(state,move)));
  }
  // moveで利きを止める手
  {
    Move move(Square(7,6),Square(6,5),GOLD,PTYPE_EMPTY,false,BLACK);
    CPPUNIT_ASSERT(isBlockRookTest(state,move));
  }
  // dropして利きを止める手
  {
    Move move(Square(3,5),SILVER,BLACK);
    CPPUNIT_ASSERT(isBlockRookTest(state,move));
  }
  // 利きの中に入るだけではダメ
  {
    Move move(Square(3,6),SILVER,BLACK);
    CPPUNIT_ASSERT((!isBlockRookTest(state,move)));
  }
  // 利きを止めているが，盤端
  {
    Move move(Square(9,5),SILVER,BLACK);
    CPPUNIT_ASSERT((!isBlockRookTest(state,move)));
  }
  // 利きを止めた先が相手の駒でもtrue
  {
    Move move(Square(3,3),Square(2,3),GOLD,PTYPE_EMPTY,false,BLACK);
    CPPUNIT_ASSERT((isBlockRookTest(state,move)));
  }
  // 利きを止めた先が自分の駒の時はOK
  {
    Move move(Square(3,9),Square(2,7),KNIGHT,PTYPE_EMPTY,false,BLACK);
    CPPUNIT_ASSERT(isBlockRookTest(state,move));
  }
  // 元々ブロックしていた駒の場合も近づくならtrueにする
  {
    Move move(Square(2,8),Square(2,7),PAWN,PTYPE_EMPTY,false,BLACK);
    CPPUNIT_ASSERT(isBlockRookTest(state,move));
  }
}

void BlockLongEffectTest::testBlockRookWhite()
{
  const char *problem = 
    "P1-KY *  * -OU * +KA *  * -KY\n"
    "P2 * +GI *  *  *  * -KI *  * \n"
    "P3 *  * -KE *  *  *  *  *  * \n"
    "P4-FU *  * -KE-GI-FU+FU * +FU\n"
    "P5 *  *  *  *  *  * -FU *  * \n"
    "P6+FU+FU+KI-FU-FU *  *  *  * \n"
    "P7 *  * +KE *  * +RY *  * -FU \n"
    "P8 * +OU * -GI *  *  *  *  * \n"
    "P9+KY *  *  *  *  *  * +KE+KY\n"
    "P+00FU\n"
    "P-00AL\n"
    "-\n";
  NumEffectState state((CsaString(problem).getInitialState()));
  // moveで利きを止める手
  {
    Move move(Square(5,6),Square(5,7),PPAWN,PTYPE_EMPTY,true,WHITE);
    CPPUNIT_ASSERT(isBlockRookTest(state,move));
  }
  // dropして利きを止める手
  {
    Move move(Square(2,7),SILVER,WHITE);
    CPPUNIT_ASSERT(isBlockRookTest(state,move));
  }
  // 利きの中に入るだけではダメ
  {
    Move move(Square(3,8),SILVER,WHITE);
    CPPUNIT_ASSERT((!isBlockRookTest(state,move)));
  }
  // 利きを止めているが，盤端
  {
    Move move(Square(4,9),SILVER,WHITE);
    CPPUNIT_ASSERT((!isBlockRookTest(state,move)));
  }
  // 利きを止めた先が相手の駒でもtrue
  {
    Move move(Square(6,6),Square(6,7),PPAWN,PTYPE_EMPTY,true,WHITE);
    CPPUNIT_ASSERT((isBlockRookTest(state,move)));
  }
  // 利きを止めた先が自分の駒の時はOK
  {
    Move move(Square(2,7),SILVER,WHITE);
    CPPUNIT_ASSERT(isBlockRookTest(state,move));
  }
  // 駒を取る手は入らない
  {
    Move move(Square(6,8),Square(7,7),SILVER,KNIGHT,false,WHITE);
    CPPUNIT_ASSERT((!isBlockRookTest(state,move)));
  }
  // 元々ブロックしていた駒の場合も近づくならtrueにする
  {
    Move move(Square(4,4),Square(4,5),PAWN,PTYPE_EMPTY,false,WHITE);
    CPPUNIT_ASSERT(isBlockRookTest(state,move));
  }
}

static
bool isBlockBishopTest(const NumEffectState& state, Move m)
{
  assert(state.isValidMove(m,true));
  const Ptype ptype = m.ptype();
  const Square from = m.from();
  const Square to = m.to();
  if (m.player() == BLACK)
    return BlockLongEffect<BLACK,BISHOP>::isMember(state, ptype, from, to);
  else
    return BlockLongEffect<WHITE,BISHOP>::isMember(state, ptype, from, to);
}

void BlockLongEffectTest::testBlockBishopBlack()
{
  const char *problem = 
    "P1 * +GI * -OU *  *  *  *  * \n"
    "P2-KY *  *  *  *  *  * -KY * \n"
    "P3 *  * -KE *  *  * +KI *  * \n"
    "P4-FU+FU * -KE-GI-FU-FU * -FU\n"
    "P5 *  *  *  *  *  *  * -RY * \n"
    "P6+FU * +KI-FU-UM *  *  * +FU\n"
    "P7 *  * +KE *  *  * +FU *  * \n"
    "P8 * +OU+GI *  *  *  * +FU * \n"
    "P9+KY *  *  *  *  * +KE * +KY\n"
    "P+00FU00FU00GI\n"
    "P-00AL\n"
    "+\n";
  NumEffectState state((CsaString(problem).getInitialState()));
  // moveで利きを止める手
  {
    Move move(Square(7,6),Square(6,5),GOLD,PTYPE_EMPTY,false,BLACK);
    CPPUNIT_ASSERT(isBlockBishopTest(state,move));
  }
  // dropして利きを止める手
  {
    Move move(Square(7,4),SILVER,BLACK);
    CPPUNIT_ASSERT(isBlockBishopTest(state,move));
  }
  // 利きの中に入るだけではダメ
  {
    Move move(Square(5,7),SILVER,BLACK);
    CPPUNIT_ASSERT((!isBlockRookTest(state,move)));
  }
  // 利きを止めているが，盤端
  {
    Move move(Square(2,9),SILVER,BLACK);
    CPPUNIT_ASSERT((!isBlockBishopTest(state,move)));
  }
  // 利きを止めた先が相手の駒でもtrue
  {
    Move move(Square(8,3),SILVER,BLACK);
    CPPUNIT_ASSERT((isBlockBishopTest(state,move)));
  }
  // 利きを止めた先が自分の駒の時はOK
  {
    Move move(Square(6,7),SILVER,BLACK);
    CPPUNIT_ASSERT(isBlockBishopTest(state,move));
  }
  // 駒を取る手は入らない
  {
    Move move(Square(8,1),Square(9,2),PSILVER,LANCE,true,BLACK);
    CPPUNIT_ASSERT((!isBlockBishopTest(state,move)));
  }
  // 元々ブロックしていた駒の場合も近づくならtrueにする
  {
    Move move(Square(7,8),Square(6,7),SILVER,PTYPE_EMPTY,false,BLACK);
    CPPUNIT_ASSERT(isBlockBishopTest(state,move));
  }
}

void BlockLongEffectTest::testBlockBishopWhite()
{
  const char *problem = 
    "P1 *  *  * -OU *  *  *  * -KY\n"
    "P2 * -KY *  *  *  * -KI *  * \n"
    "P3 * +GI-KE *  *  *  *  *  * \n"
    "P4-FU *  * -KE-GI-FU+FU * +FU\n"
    "P5 *  *  *  *  *  * -FU *  * \n"
    "P6+FU+FU+KI-FU+UM *  *  *  * \n"
    "P7 *  * +KE *  *  *  *  * -FU \n"
    "P8 * +OU-GI * +RY *  * +KE * \n"
    "P9+KY *  *  *  *  *  *  * +KY\n"
    "P+00FU\n"
    "P-00AL\n"
    "-\n";
  NumEffectState state((CsaString(problem).getInitialState()));
  // moveで利きを止める手
  {
    Move move(Square(5,4),Square(6,5),SILVER,PTYPE_EMPTY,false,WHITE);
    CPPUNIT_ASSERT(isBlockBishopTest(state,move));
  }
  // dropして利きを止める手
  {
    Move move(Square(3,8),SILVER,WHITE);
    CPPUNIT_ASSERT(isBlockBishopTest(state,move));
  }
  // 利きの中に入るだけではダメ
  {
    Move move(Square(4,6),SILVER,WHITE);
    CPPUNIT_ASSERT((!isBlockRookTest(state,move)));
  }
  // 利きを止めているが，盤端
  {
    Move move(Square(2,9),SILVER,WHITE);
    CPPUNIT_ASSERT((!isBlockBishopTest(state,move)));
  }
  // 利きを止めた先が相手の駒でもtrue
  {
    Move move(Square(7,4),SILVER,WHITE);
    CPPUNIT_ASSERT((isBlockBishopTest(state,move)));
  }
  // 利きを止めた先が自分の駒の時はOK
  {
    Move move(Square(6,7),SILVER,WHITE);
    CPPUNIT_ASSERT(isBlockBishopTest(state,move));
  }
  // 駒を取る手は入らない
  {
    Move move(Square(8,2),Square(8,3),LANCE,SILVER,false,WHITE);
    CPPUNIT_ASSERT((!isBlockBishopTest(state,move)));
  }
  // 元々ブロックしていた駒の場合も近づくならtrueにする
  {
    Move move(Square(7,8),Square(6,7),SILVER,PTYPE_EMPTY,false,WHITE);
    CPPUNIT_ASSERT(isBlockBishopTest(state,move));
  }
}

static
bool isBlockLanceTest(const NumEffectState& state, Move m)
{
  assert(state.isValidMove(m,true));
  const Ptype ptype = m.ptype();
  const Square from = m.from();
  const Square to = m.to();
  if (m.player() == BLACK)
    return BlockLongEffect<BLACK,LANCE>::isMember(state, ptype, from, to);
  else
    return BlockLongEffect<WHITE,LANCE>::isMember(state, ptype, from, to);
}

void BlockLongEffectTest::testBlockLanceBlack()
{
  const char *problem = 
    "P1 * +GI * -OU *  *  * -KY * \n"
    "P2-KY *  *  *  *  *  *  *  * \n"
    "P3 *  * -KE *  *  * +KI *  * \n"
    "P4 * +FU * -KE-GI-FU-FU * -FU\n"
    "P5 *  *  *  *  *  *  * -RY * \n"
    "P6+FU * +KI-FU-UM *  * +FU+FU\n"
    "P7-FU * +KE *  *  * +FU * -KY\n"
    "P8 * +OU+GI *  *  *  *  *  * \n"
    "P9+KY *  *  *  *  * +KE *  * \n"
    "P+00FU00FU00GI\n"
    "P-00AL\n"
    "+\n";
  NumEffectState state((CsaString(problem).getInitialState()));
  // moveで利きを止める手
  {
    Move move(Square(3,3),Square(2,3),GOLD,PTYPE_EMPTY,false,BLACK);
    CPPUNIT_ASSERT(isBlockLanceTest(state,move));
  }
  // dropして利きを止める手
  {
    Move move(Square(1,8),SILVER,BLACK);
    CPPUNIT_ASSERT(isBlockLanceTest(state,move));
  }
  // 利きを止めているが，盤端
  {
    Move move(Square(1,9),SILVER,BLACK);
    CPPUNIT_ASSERT((!isBlockLanceTest(state,move)));
  }
  // 利きを止めた先が相手の駒でもtrue
  {
    Move move(Square(2,4),SILVER,BLACK);
    CPPUNIT_ASSERT((isBlockLanceTest(state,move)));
  }
  // 利きを止めた先が自分の駒の時はOK
  {
    Move move(Square(9,5),SILVER,BLACK);
    CPPUNIT_ASSERT(isBlockLanceTest(state,move));
  }
  // 駒を取る手は入らない
  {
    Move move(Square(2,6),Square(2,5),PAWN,PROOK,false,BLACK);
    CPPUNIT_ASSERT((!isBlockLanceTest(state,move)));
  }
  // 元々ブロックしていた駒の場合も近づくならtrueにする
  {
    Move move(Square(9,6),Square(9,5),PAWN,PTYPE_EMPTY,false,BLACK);
    CPPUNIT_ASSERT(isBlockLanceTest(state,move));
  }
}

void BlockLongEffectTest::testBlockLanceWhite()
{
  const char *problem = 
    "P1 *  *  * -OU *  *  *  *  * \n"
    "P2 * -KY *  *  *  * -KI *  * \n"
    "P3 *  * -KE *  *  *  *  *  * \n"
    "P4-FU *  * -KE-GI-FU+FU * +FU\n"
    "P5 * -GI *  *  *  * -FU * -FU\n"
    "P6+FU+FU+KI-FU+UM *  *  *  * \n"
    "P7 *  * +KE *  *  *  * +KY * \n"
    "P8 * +OU-GI * +RY *  * +KE * \n"
    "P9+KY *  *  *  *  *  *  * +KY\n"
    "P+00FU\n"
    "P-00AL\n"
    "-\n";
  NumEffectState state((CsaString(problem).getInitialState()));
  // moveで利きを止める手
  {
    Move move(Square(3,2),Square(2,2),GOLD,PTYPE_EMPTY,false,WHITE);
    CPPUNIT_ASSERT(isBlockLanceTest(state,move));
  }
  // dropして利きを止める手
  {
    Move move(Square(2,6),SILVER,WHITE);
    CPPUNIT_ASSERT(isBlockLanceTest(state,move));
  }
  // 利きを止めているが，盤端
  {
    Move move(Square(2,1),SILVER,WHITE);
    CPPUNIT_ASSERT((!isBlockLanceTest(state,move)));
  }
  // 利きを止めた先が相手の駒でもtrue
  {
    Move move(Square(9,7),SILVER,WHITE);
    CPPUNIT_ASSERT((isBlockLanceTest(state,move)));
  }
  // 利きを止めた先が自分の駒の時はOK
  {
    Move move(Square(1,6),SILVER,WHITE);
    CPPUNIT_ASSERT(isBlockLanceTest(state,move));
  }
  // 駒を取る手は入らない
  {
    Move move(Square(8,5),Square(8,6),SILVER,PAWN,false,WHITE);
    CPPUNIT_ASSERT((!isBlockLanceTest(state,move)));
  }
  // 元々ブロックしていた駒の場合も近づくならtrueにする
  {
    Move move(Square(1,5),Square(1,6),PAWN,PTYPE_EMPTY,false,WHITE);
    CPPUNIT_ASSERT(isBlockLanceTest(state,move));
  }
}

/* ------------------------------------------------------------------------- */
// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
