/*
* JBoss, Home of Professional Open Source
* Copyright 2006, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.metadata.spi.signature.javassist;

import javassist.CtConstructor;
import javassist.CtField;
import javassist.CtMember;
import javassist.CtMethod;
import javassist.NotFoundException;
import javassist.CtClass;
import javassist.CtPrimitiveType;
import org.jboss.metadata.spi.signature.Signature;

/**
 * Javassist Signature factory.
 *
 * @author <a href="ales.justin@jboss.com">Ales Justin</a>
 */
public class JavassistSignatureFactory
{
   /**
    * Get a signature for a member
    *
    * @param member the member
    * @return the result
    */
   public static Signature getSignature(CtMember member)
   {
      if (member == null)
         throw new IllegalArgumentException("Null member");

      try
      {
         if (member instanceof CtMethod)
         {
            CtMethod method = CtMethod.class.cast(member);
            return new JavassistMethodSignature(method);
         }
         if (member instanceof CtField)
         {
            CtField field = CtField.class.cast(member);
            return new JavassistFieldSignature(field);
         }
         if (member instanceof CtConstructor)
         {
            CtConstructor constructor = CtConstructor.class.cast(member);
            return new JavassistConstructorSignature(constructor);
         }
      }
      catch (NotFoundException e)
      {
         throw raiseClassNotFound("member", member.toString(), e);
      }
      throw new IllegalArgumentException("Unknown member: " + member);
   }

   /**
    * Convert ct classes to string class names.
    *
    * @param classes the classes
    * @return class names
    */
   public static String[] convertParameters(CtClass[] classes)
   {
      if (classes == null || classes.length == 0)
         return Signature.NO_PARAMETERS;

      String[] paramTypes = new String[classes.length];
      for (int i = 0; i < classes.length; ++i)
         paramTypes[i] = convertName(classes[i]);
      return paramTypes;
   }

   /**
    * Convert ct class to full classname.
    *
    * @param clazz the ct class
    * @return class name
    */
   protected static String convertName(CtClass clazz)
   {
      if (clazz == null)
         throw new IllegalArgumentException("Null CtClass");

      CtClass temp = clazz;
      if (temp.isArray())
      {
         StringBuilder buffer = new StringBuilder();
         try
         {
            while (temp.isArray())
            {
               buffer.append('[');
               temp = temp.getComponentType();
            }
            if (temp.isPrimitive())
            {
               CtPrimitiveType primitive = (CtPrimitiveType) temp;
               buffer.append(Character.toString(primitive.getDescriptor()));
            }
            else
            {
               buffer.append('L');
               buffer.append(temp.getName());
               buffer.append(';');
            }
            return buffer.toString();
         }
         catch (NotFoundException e)
         {
            throw raiseClassNotFound("class", clazz.getName(), e);
         }
      }
      return clazz.getName();
   }

   /**
    * Raise NCDFE exception.
    *
    * @param type the error type
    * @param info the info
    * @param e javassist exception
    * @return NCDFE instance
    * @throws NoClassDefFoundError transform e param into NCDFE
    */
   protected static NoClassDefFoundError raiseClassNotFound(String type, String info, NotFoundException e) throws NoClassDefFoundError
   {
      NoClassDefFoundError ex = new NoClassDefFoundError("Unable to find " + type + ": " + info);
      if (e.getCause() != null)
         ex.initCause(e.getCause()); // Hide the javassist error
      throw ex;
   }
}