#if !defined(__BUILDER_HPP)
#define __BUILDER_HPP

/*
  CoreLinux++ 
  Copyright (C) 1999,2000 CoreLinux Consortium
  
   The CoreLinux++ Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   The CoreLinux++ Library Library is distributed in the hope that it will 
   be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with the GNU C Library; see the file COPYING.LIB.  If not,
   write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  
*/   

#if   !defined(__COMMON_HPP)
#include <Common.hpp>
#endif

#if   !defined(__ABSTRACTFACTORY_HPP)
#include <AbstractFactory.hpp>
#endif

namespace corelinux
{
   /**
   Builder seperates the construction of a complex object from its 
   representation so that the same construction process can create 
   different representations. This differs from AbstractFactory in 
   that the Factory creates parts and Builder creates Products 
   (assembled parts).
   @see AbstractFactory, Allocator, AbstractAllocator
   */

   template< class ProductImpl, class UniqueId >
      class Builder
      {
      public:

         //
         // Constructors and destructor
         //

                        /**
                        Default constructor requires a abstract
                        factory for building the product parts
                        @param AbstractFactory
                        */
                        
                        Builder
                           ( 
                              AbstractFactory<UniqueId> *aAbstractFactory 
                           )
                           throw( Assertion )
                           :
                           theCurrentProduct( NULLPTR ),
                           theFactory( NULLPTR ),
                           theProductCreates( 0 ),
                           theProductDestroys( 0 )
                        {
                           REQUIRE( aAbstractFactory  != NULLPTR );
                           theFactory = aAbstractFactory ;
                        }  

                        /**
                        Copy constructor creates a new instance of the 
                        reference abstract factory
                        @param Builder const reference
                        */

                        Builder( const Builder &aBuilder )
                           throw( Assertion )
                           :
                           theCurrentProduct( NULLPTR ),
                           theFactory( aBuilder.getFactory() ),
                           theProductCreates( 0 ),
                           theProductDestroys( 0 )
                        {
                           REQUIRE( theFactory != NULLPTR );
                        }

                        /// Virtual destructor

         virtual        ~Builder( void )
                        {
                           ;  // do nothing
                        }

         //
         // Operator overloads
         //

                  /**
                  Operation assignment. Uses the same factory
                  as the reference after destroying theCurrentProduct.
                  @param Builder const reference
                  @return Builder reference
                  */

                  Builder &  operator=( const Builder &aRef )
                  {
                     if( this != &aRef )
                     {
                        this->destroy( theCurrentProduct );

                        theFactory = aRef.getFactory();

                        theCurrentProduct  = NULLPTR;
                        theProductCreates  = 0;
                        theProductDestroys = 0;
                     }
                     else
                     {
                        ;  // do nothing
                     }

                     return ( *this );
                  }

                  /**
                  Equality operator
                  @param Builder const reference
                  @return bool - true if same Builder instance
                  */

                  bool  operator==( const Builder &aRef )
                  {
                     return ( this == &aRef );
                  }

         //
         // Accessors
         //
                  /**
                  Retrieves the current product
                  @return ProductImpl pointer
                  */

         virtual  ProductImpl *  getCurrentProduct( void ) const
                  {
                     return theCurrentProduct;
                  }

                  /// Retrieve the product create counts

         virtual  CountCref      getProductCreates( void ) const
                  {
                     return theProductCreates;
                  }

                  /// Retrieve the product destroy counts

         virtual  CountCref      getProductDestroys( void ) const
                  {
                     return theProductDestroys;
                  }

                  /// Retrieve the AbstractFactory

         virtual  AbstractFactory<UniqueId> * getFactory( void ) const
                  {
                     return theFactory;
                  }
         //
         // Builder methods
         //
                  /**
                  Default create routine invokes the implementation
                  createProduct method
                  @return ProductImpl pointer
                  */

         virtual  ProductImpl *  create( void ) 
                  {
                     ProductImpl *aPtr( NULLPTR );

                     try
                     {
                        aPtr = createProduct();

                        CHECK( aPtr != NULLPTR );

                        theCurrentProduct = aPtr;

                        incrementCreates();
                     }
                     catch( ... )
                     {
                        throw;
                     }

                     return aPtr;
                  }

                  /**
                  Default destroy routine invokes the implementation
                  destroyProduct method
                  @param ProductImpl pointer
                  */

         virtual  void  destroy( ProductImpl * aPtr )
                  {
                     REQUIRE( aPtr != NULLPTR );
                     try
                     {
                        destroyProduct( aPtr );
                        incrementDestroys();
                     }
                     catch( ... )
                     {
                        throw;
                     }
                     if( aPtr == theCurrentProduct )
                     {
                        theCurrentProduct = NULLPTR;
                     }
                     else
                     {
                        ;  // do nothing
                     }
                  }

      protected:

         //
         // Constructor
         //

                        /// Default constructor not supported

                        Builder( void ) throw(Assertion)
                           :
                           theCurrentProduct( NULLPTR ),
                           theFactory( NULLPTR ),
                           theProductCreates( 0 ),
                           theProductDestroys( 0 )
                        {
                           NEVER_GET_HERE;
                        }

         //
         // Mutators
         //

                  /// Increment the creates

                  void  incrementCreates( void )
                  {
                     ++theProductCreates;
                  }

                  /// Increment the destroys

                  void  incrementDestroys( void )
                  {
                     ++theProductDestroys;
                  }

         //
         // Factory methods
         //

                  /// Pure virtual createProduct 

         virtual  ProductImpl *  createProduct( void ) const = 0;

                  /// Pure virtual destroyProduct 

         virtual  void  destroyProduct( ProductImpl * ) const = 0;

      protected:

         //
         // Data members
         //

                  /// The product that was most recently built

                  ProductImpl                *theCurrentProduct;

                  /// The factory for creating parts

                  AbstractFactory<UniqueId>  *theFactory;

                  /// The count of creates

                  Count                      theProductCreates;

                  /// The count of destroys

                  Count                      theProductDestroys;
   
      };
}

#endif // if !defined(__BUILDER_HPP)

/*
   Common rcs information do not modify
   $Author: prudhomm $
   $Revision: 1.1 $
   $Date: 2000/04/23 20:43:13 $
   $Locker:  $
*/

