/*
 *  Software MMU support
 * 
 *  Copyright (c) 2005-2009 FAUmachine Team.
 *  Copyright (c) 2003 Fabrice Bellard
 *  Modified for FAUmachine by Volkmar Sieh
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301
 * USA
 */
#define DATA_SIZE (1 << SHIFT)

#if DATA_SIZE == 8
#define SUFFIX q
#define USUFFIX q
#define DATA_TYPE uint64_t
#elif DATA_SIZE == 4
#define SUFFIX l
#define USUFFIX l
#define DATA_TYPE uint32_t
#elif DATA_SIZE == 2
#define SUFFIX w
#define USUFFIX uw
#define DATA_TYPE uint16_t
#elif DATA_SIZE == 1
#define SUFFIX b
#define USUFFIX ub
#define DATA_TYPE uint8_t
#else
#error unsupported data size
#endif


/* (just for grep) io_readb io_readub io_readw io_readl */
static inline DATA_TYPE
glue(glue(io_read, SUFFIX), MMUSUFFIX)(
	Paddr physaddr, 
	unsigned int hash
)
{
	DATA_TYPE res;

#ifdef CODE_ACCESS
#if SHIFT <= 2
	res = NAME_(io_mem_code)[hash][SHIFT](physaddr);
#else
#ifdef TARGET_WORDS_BIGENDIAN
	res = (uint64_t)NAME_(io_mem_code)[hash][2](physaddr) << 32;
	res |= NAME_(io_mem_code)[hash][2](physaddr + 4);
#else
	res = NAME_(io_mem_code)[hash][2](physaddr);
	res |= (uint64_t)NAME_(io_mem_code)[hash][2](physaddr + 4) << 32;
#endif
#endif /* SHIFT > 2 */
#else /* ! CODE_ACCESS */
#if SHIFT <= 2
	res = NAME_(io_mem_read)[hash][SHIFT](physaddr);
#else
#ifdef TARGET_WORDS_BIGENDIAN
	res = (uint64_t)NAME_(io_mem_read)[hash][2](physaddr) << 32;
	res |= NAME_(io_mem_read)[hash][2](physaddr + 4);
#else
	res = NAME_(io_mem_read)[hash][2](physaddr);
	res |= (uint64_t)NAME_(io_mem_read)[hash][2](physaddr + 4) << 32;
#endif
#endif /* SHIFT > 2 */
#endif /* ! CODE_ACCESS */
	return res;
}

/* (just for grep) slow_ldb_mmu slow_ldub_mmu slow_ldw_mmu slow_ldl_mmu */
static inline DATA_TYPE
glue(glue(slow_ld, SUFFIX), MMUSUFFIX)(
	Vaddr addr, 
	int is_user,
	void *retaddr
)
{
	DATA_TYPE res;
	int hash;
	Vaddr tlb_addr;
	Paddr physaddr;
	Haddr hostaddr;

	hash = (addr >> TARGET_PAGE_BITS) & (CPU_TLB_SIZE - 1);
redo:
#ifdef CODE_ACCESS
	tlb_addr = env->tlb_code[is_user][hash].address;
#else
	tlb_addr = env->tlb_read[is_user][hash].address;
#endif
	if ((addr & TARGET_PAGE_MASK)
			!= (tlb_addr & (TARGET_PAGE_MASK | TLB_INVALID_MASK))) {
		/* The page is not in the TLB : fill it. */
#ifdef CODE_ACCESS
		NAME_(tlb_fill)(addr, 2, is_user, retaddr);
#else
		NAME_(tlb_fill)(addr, 0, is_user, retaddr);
#endif
		goto redo;
	}

	if (tlb_addr & ~TARGET_PAGE_MASK) {
		/* I/O access. */
#ifdef CODE_ACCESS
		physaddr = addr + env->tlb_code[is_user][hash].phys_addend;
#else
		physaddr = addr + env->tlb_read[is_user][hash].phys_addend;
#endif
		res = glue(glue(io_read, SUFFIX), MMUSUFFIX)(physaddr,
			(tlb_addr >> IO_MEM_SHIFT) & (IO_MEM_NB_ENTRIES - 1));
	} else {
		/* ROM/RAM access. */
#ifdef CODE_ACCESS
		hostaddr = addr + env->tlb_code[is_user][hash].host_addend;
#else
		hostaddr = addr + env->tlb_read[is_user][hash].host_addend;
#endif
		res = glue(glue(ld, USUFFIX), _raw)(hostaddr);
	}
	return res;
}

/* Just to make this clear (and grepable):
 * Through macro magic, this will become something like
 * __ldb_mmu
 * __ldub_mmu
 * __ldw_mmu
 * __ldl_mmu        */
DATA_TYPE REGPARM(1)
glue(glue(NAME_(__ld), SUFFIX), MMUSUFFIX)(Vaddr addr, int is_user)
{
	DATA_TYPE res;

        if ((addr & (DATA_SIZE - 1)) == 0) {
		/* Aligned access. */
		res = glue(glue(slow_ld, SUFFIX), MMUSUFFIX)(addr, is_user, GETPC());
	} else {
		/* Unaligned access (might access two pages). */
		Vaddr addr1;
		Vaddr addr2;
		DATA_TYPE res1;
		DATA_TYPE res2;
		int shift;

		addr1 = addr & ~(DATA_SIZE - 1);
		addr2 = addr1 + DATA_SIZE;
		res1 = glue(glue(slow_ld, SUFFIX), MMUSUFFIX)(addr1, is_user, GETPC());
		res2 = glue(glue(slow_ld, SUFFIX), MMUSUFFIX)(addr2, is_user, GETPC());
		shift = (addr & (DATA_SIZE - 1)) * 8;
#ifdef TARGET_WORDS_BIGENDIAN
		res = (res1 << shift) | (res2 >> ((DATA_SIZE * 8) - shift));
#else
		res = (res1 >> shift) | (res2 << ((DATA_SIZE * 8) - shift));
#endif
		res = (DATA_TYPE) res;
	}

	return res;
}

#ifndef CODE_ACCESS

/* (just for grep) io_writeb io_writeub io_writew io_writel */
static inline void
glue(io_write, SUFFIX)(
	Paddr physaddr, 
	DATA_TYPE val,
	unsigned int hash
)
{
#if SHIFT <= 2
	NAME_(io_mem_write)[hash][SHIFT](physaddr, val);
#else
#ifdef TARGET_WORDS_BIGENDIAN
	NAME_(io_mem_write)[hash][2](physaddr, val >> 32);
	NAME_(io_mem_write)[hash][2](physaddr + 4, val);
#else
	NAME_(io_mem_write)[hash][2](physaddr, val);
	NAME_(io_mem_write)[hash][2](physaddr + 4, val >> 32);
#endif
#endif /* SHIFT > 2 */
}

/* (just for grep) slow_stb_mmu slow_stub_mmu slow_stw_mmu slow_stl_mmu */
static inline void
glue(glue(slow_st, SUFFIX), MMUSUFFIX)(
	Vaddr addr, 
	DATA_TYPE val,
	int is_user,
	void *retaddr
)
{
	int hash;
	Vaddr tlb_addr;
	Paddr physaddr;
	Haddr hostaddr;

	hash = (addr >> TARGET_PAGE_BITS) & (CPU_TLB_SIZE - 1);
redo:
	tlb_addr = env->tlb_write[is_user][hash].address;
	if ((addr & TARGET_PAGE_MASK)
			!= (tlb_addr & (TARGET_PAGE_MASK | TLB_INVALID_MASK))) {
		/* The page is not in the TLB : fill it. */
		NAME_(tlb_fill)(addr, 1, is_user, retaddr);
		goto redo;
	}

	if (tlb_addr & ~TARGET_PAGE_MASK) {
		/* I/O access. */
		env->mem_write_vaddr = tlb_addr;
		env->mem_write_pc = (unsigned long) retaddr;

		physaddr = addr + env->tlb_write[is_user][hash].phys_addend;
		glue(io_write, SUFFIX)(physaddr, val,
			(tlb_addr >> IO_MEM_SHIFT) & (IO_MEM_NB_ENTRIES - 1));
	} else {
		/* RAM access. */
		hostaddr = addr + env->tlb_write[is_user][hash].host_addend;
		glue(glue(st, SUFFIX), _raw)(hostaddr, val);
	}
}

/* (just for grep) __stb_mmu __stub_mmu __stw_mmu __stl_mmu */
void REGPARM(2)
glue(glue(NAME_(__st), SUFFIX), MMUSUFFIX)(
	Vaddr addr, 
	DATA_TYPE val,
	int is_user
)
{
	if ((addr & (DATA_SIZE - 1)) == 0) {
		/* Aligned access. */
		glue(glue(slow_st, SUFFIX), MMUSUFFIX)(addr, val, is_user, GETPC());
	} else {
		/* Unaligned access (might access two pages). */
		unsigned int i;

		for (i = 0; i < DATA_SIZE; i++) {
#ifdef TARGET_WORDS_BIGENDIAN
			glue(slow_stb, MMUSUFFIX)(addr + i, val >> (((DATA_SIZE - 1) * 8) - (i * 8)), is_user, GETPC());
#else
			glue(slow_stb, MMUSUFFIX)(addr + i, val >> (i * 8), is_user, GETPC());
#endif
		}
	}
}

#endif /* ! defined(CODE_ACCESS) */

#undef SHIFT
#undef DATA_TYPE
#undef SUFFIX
#undef USUFFIX
#undef DATA_SIZE
