/*
 * This file is part of din.
 *
 * din is copyright (c) 2006 - 2012 S Jagannathan <jag@dinisnoise.org>
 * For more information, please visit http://dinisnoise.org
 *
 * din is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * din is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with din.  If not, see <http://www.gnu.org/licenses/>.
 *
*/

#include "main.h"
#include "audio.h"
#include "chrono.h"
#include "command.h"
#include "console.h"
#include "tcl_interp.h"
#include "ansi_color_codes.h"
#include "delay.h"
#include "compressor.h"
#include "keyboard_keyboard.h"
#include "fader.h"
#include "ui_list.h"

#include <math.h>
#include <fstream>
#include <limits.h>
#include <stdlib.h>

#include <string>
using namespace std;

extern string dotdin; // ~/.din
extern int SAMPLE_RATE;
extern float SAMPLE_DURATION;
extern chrono clk; // din audio clock

// jack port names
extern string name, dinl, dinr;
extern const char *colon, *L, *R, *MIDI_IN;

extern audio_out aout;

extern delay left_delay, right_delay;
extern ui_list uis;

extern string APP_NAME;
extern string WIN_TITLE;

audio_out::audio_out () {

  client = 0;

  result = ams = fms = vol = gatr = 0;
  num_midi_con = 0;

  prefs_name = dotdin + "jack_prefs";

  load_prefs ();

  open ();

}

audio_out::~audio_out () {

  if (midi_rb) jack_ringbuffer_free (midi_rb);

  sample_t* ptr [] = {samples_buffers, result, ams, fms, vol, gatr};
  for (int i = 0; i < 6; ++i) if (ptr[i]) delete[] ptr[i];

  if (available) delete[] available;

  cout << FAIL << "--- destroyed audio ---" << ENDL;

}

void audio_out::alloc () {

  // alloc memory for various buffers
  //

  samples_per_buffer = num_channels * samples_per_channel;
  samples_buffer_size = sizeof (sample_t) * samples_per_buffer;
  samples_channel_size = sizeof (sample_t) * samples_per_channel;

  if (samples_buffers) delete[] samples_buffers;
  samples_buffers = new sample_t [num_samples_buffers * samples_per_buffer];

  if (available) delete[] available;
  available = new int [num_samples_buffers];

  readi = writei = 0;
  readp = writep = samples_buffers;
  for (int i = 0; i < num_samples_buffers; ++i) available[i] = 0;

  sample_t** ptr [] = {&result, &ams, &fms, &vol, &gatr};
  for (int i = 0; i < 5; ++i) {
    if (ptr[i]) delete[] *ptr[i];
    *ptr[i] = new sample_t [samples_per_channel];
  }

  if (midi_rb) jack_ringbuffer_free (midi_rb);
  midi_rb = jack_ringbuffer_create (samples_buffer_size);

}

void audio_out::set_sample_rate (int s) {

  SAMPLE_RATE = sample_rate = s;
  SAMPLE_DURATION = 1. / SAMPLE_RATE;

  // update din audio clock
  //
  clk.delta_ticks = samples_per_channel;
  clk.delta_secs = samples_per_channel * 1. / SAMPLE_RATE;

  cout << PASS << "+++ sample rate is " << SAMPLE_RATE << " Hz +++" << ENDL;

}

int audio_out::open() {

  cout << DOING << "*** connecting to JACK server ***" << ENDL;
  jack_status_t status;
  jack_options_t options = JackNullOption;

  client = jack_client_open (name.c_str(), options, &status, 0);
  if (!client) {
    cout << FAIL << "!!! couldnt connect to JACK server. !!!" << ENDL;
    exit (1);
  } else {

    name = string (jack_get_client_name (client));

    // write JACK connection name on din title bar (useful when running multiple dins)
    WIN_TITLE = APP_NAME + " [" + name + "]";

    dinl = name + colon + L;
    dinr = name + colon + R;
    cout << PASS << "+++ connected to JACK server as: " << name << " +++" << ENDL;

  }

  sample_rate = jack_get_sample_rate (client);

  samples_per_channel = jack_get_buffer_size (client);
  
  // create output audio ports
  //
  out_left = jack_port_register (client, L, JACK_DEFAULT_AUDIO_TYPE, JackPortIsOutput, 0);
  out_right = jack_port_register (client, R, JACK_DEFAULT_AUDIO_TYPE, JackPortIsOutput, 0);

  if ((out_left == 0) || (out_right == 0)) {
    cout << FAIL << "!!! couldnt create input & output ports !!!" << ENDL;
    exit (1);
  } else cout << PASS << "+++ opened input & output ports +++" << ENDL;
  lcons = rcons = 0;

  midi_in = jack_port_register (client, MIDI_IN, JACK_DEFAULT_MIDI_TYPE, JackPortIsInput, 0);
  if (!midi_in) cout << FAIL << "!!! couldn't create midi input port !!!" << ENDL; else {
    cout << PASS "+++ opened MIDI port +++" << ENDL;
  }

  alloc ();

  set_sample_rate (sample_rate);

  cout << PASS << "+++ opened JACK connection +++" << ENDL;
  return 0;

}

void audio_out::set_callbacks () {
  jack_set_sample_rate_callback (client, sample_rate_changed, (void *) this);
  jack_set_buffer_size_callback (client, buffer_size_changed, (void *) this);
  jack_set_port_connect_callback (client, connection_changed, (void *) this);
  cout << PASS << "+++ setup audio callbacks (sample rate, buffer size & midi connection) +++" << ENDL;
}

int audio_out::start () {

  if (client) {

    if (jack_set_process_callback (client, audio_wanted, (void *) this)) {
      cout << FAIL << "!!! couldnt set JACK process callback !!!" << ENDL;
      exit (1);
    };

    if (jack_activate (client)) {
      cout << FAIL << "!!! couldnt activate JACK din client !!!" << ENDL;
      exit (1);
    }; // go!

    if (AUTO_CONNECT_OUTPUTS) {

      // autoconnect din audio output to system audio output so user hears sounds when din comes up and
      // need not fiddle with connectons
      //

      if (jack_connect (client, dinl.c_str(), "system:playback_1") != 0)
        cout << FAIL << "!!! couldnt connect to system:playback_1 !!!" << ENDL;
      else
        cout << PASS "+++ connected din:L to system:playback_1 +++" << ENDL;

      if (jack_connect (client, dinr.c_str(), "system:playback_2") != 0)
        cout << FAIL "!!! couldnt connect to system:playback_2 !!!" << ENDL;
      else cout << PASS "+++ connected din:R to system:playback_2 +++" << ENDL;
      cout << PASS << "+++ " << APP_NAME << " is streaming audio now +++ " << ENDL;
    } else {
      cout << FAIL << "!!! you must connect din outputs to system audio outputs to hear audio !!!" << ENDL;
    }

  }
  return 0;
}

int audio_out::close () {

  if (client) {

    lcons = jack_port_connected (out_left);
    if (lcons) {
      con_left = jack_port_get_connections (out_left);
      for (int i = 0; i < lcons; ++i) jack_disconnect (client, dinl.c_str(), con_left[i]);
      jack_free (con_left);
      cout << FAIL << "--- disconnected " << lcons << " connections to din:L ---" << ENDL;
    }

    rcons = jack_port_connected (out_right);
    if (rcons) {
      con_right = jack_port_get_connections (out_right);
      for (int i = 0; i < rcons; ++i) jack_disconnect (client, dinl.c_str(), con_right[i]);
      jack_free (con_right);
      cout << FAIL << "--- disconnected " << rcons << " connections to din:R ---" << ENDL;
    }

    if (jack_deactivate (client) == 0) {
      cout << FAIL << "--- deactivated din from JACK ---" << ENDL;
    };

    if (jack_client_close(client) == 0) {
      cout << FAIL << "--- closed JACK connection ---" << ENDL;
    } else cout << FAIL << "!!! couldnt close JACK connection !!! " << ENDL;

  }

  return 0;

}

void audio_out::stream (sample_t* lbuf, sample_t* rbuf, sample_t* sbuf) {

  // copy samples buffer into left & right JACK bufers

  sample_t* lbuf0 = sbuf;
  sample_t* rbuf0 = sbuf + samples_per_channel;

  memcpy (lbuf, lbuf0, samples_channel_size);
  memcpy (rbuf, rbuf0, samples_channel_size);

}

void audio_out::load_prefs () {

  //
  // load jack audio preferences from disk
  //

  ifstream file (prefs_name.c_str(), ios::in);
  string ignore;

  if (!file)
    defaults ();
  else {
    string type;
    file >> ignore >> type;
    if (type != "audio_prefs") {
      cout << FAIL << "!!! bad audio prefrences file: " << prefs_name << ", will use defaults. !!!" << ENDL;
      num_samples_buffers = 2;
    } else {
      cout << LOAD << "<< loading audio prefs from: " << prefs_name;
      file >> ignore >> num_samples_buffers;
      cout << ", done. >>>" << ENDL;
    }
  }
}

void audio_out::save_prefs () {
  // save jack audio preferences to disk
  //
  ofstream file (prefs_name.c_str(), ios::out);
  if (file) {
    file << "type: audio_prefs" << eol;
    file << "num_samples_buffers: " << num_samples_buffers << eol;
  } else cout << FAIL << "!!! couldnt write audio preferences file: " << prefs_name << " !!!" << ENDL;
}

void audio_out::defaults () {
  cout << PASS << "<<< using default for JACK settings >>>" << ENDL;
  num_samples_buffers = 2;
}

int buffer_size_changed (jack_nframes_t nframes, void *arg) {
  if ((unsigned int) aout.samples_per_channel != jack_get_buffer_size (aout.client)) {
    cout << FAIL << "!!! buffer size change not supported, please restart din !!!" << ENDL;
  }
  return 0;
}

int sample_rate_changed (jack_nframes_t nframes, void *arg) {
  aout.set_sample_rate (jack_get_sample_rate (aout.client));
  return 0;
}

int audio_out::audio_wanted (jack_nframes_t nsamples, void *arg) {

  // the "process" callback of JACK
  //
  // stream audio buffer to audio card
  //
  // read midi data
  //


  if (aout.available[aout.readi]) { // samples buffer written by main thread ready for streaming

    sample_t* lbuf = (sample_t *) jack_port_get_buffer (aout.out_left, aout.samples_per_channel); // jack left buffer
    sample_t* rbuf = (sample_t *) jack_port_get_buffer (aout.out_right, aout.samples_per_channel); // jack right buffer
    aout.stream (lbuf, rbuf, aout.readp); // stream samples buffer into left & right channels

    // update samples buffer status to let main thread write
    aout.available [aout.readi] = 0;
    if (++aout.readi >= aout.num_samples_buffers) {
      aout.readi = 0;
      aout.readp = aout.samples_buffers;
    } else aout.readp += aout.samples_per_buffer;

    ++clk; // update audio clock

  }

  // read midi data
  if (aout.midi_rb) {
    void *midi_buf = jack_port_get_buffer (aout.midi_in, nsamples);
    jack_nframes_t event_count = jack_midi_get_event_count (midi_buf);

    if (event_count > 0) {

      jack_nframes_t frame_start = jack_last_frame_time (aout.client);
      jack_nframes_t event_index;
      for (event_index = 0; event_index < event_count; ++event_index) {
        jack_midi_event_t ev;
        jack_midi_event_get (&ev, midi_buf, event_index);

        // push event onto ringbuffer
        if (jack_ringbuffer_write_space (aout.midi_rb) < sizeof ev.size + sizeof ev.time + ev.size) break;

        ev.time += frame_start;
        jack_ringbuffer_write (aout.midi_rb, (const char *)&ev.size, sizeof ev.size);
        jack_ringbuffer_write (aout.midi_rb, (const char *)&ev.time, sizeof ev.time);
        jack_ringbuffer_write (aout.midi_rb, (const char *)ev.buffer, ev.size);
      }
    }
  }

  return 0;
}

void run_midi_cmd (const string& c, size_t length, jack_midi_data_t *buffer, std::vector<unsigned char>& args) {

  char buf [10];
  unsigned char *bufp = (unsigned char *)buffer;
  string cmd (c);

  args.clear ();

  for (size_t i = 0; i < length; i++) {
    args.push_back (*bufp);
    sprintf (buf, " %u", *bufp++);
    cmd += buf;
  }

  cons (cmd);

}

void midi () {

  if (!aout.midi_rb) return;

  extern double MIDI_BPM;
  extern tcl_interp interpreter;

  static double start_time = 0;
  static int num_clocks = 0;

  jack_midi_event_t ev;
  jack_midi_data_t* ebuf = 0;
  unsigned int ebuf_size = 0;

  std::vector<unsigned char> args;

  while (1) {

    size_t f = jack_ringbuffer_read_space (aout.midi_rb);
    if (f < sizeof ev.size) break;

    jack_ringbuffer_peek (aout.midi_rb, (char *) &ev.size, sizeof ev.size);
    if (f < sizeof ev.size + sizeof ev.time + ev.size) break;

    int alloc = 0;

    if (ebuf) {
      if (ev.size > ebuf_size) {
        free (ebuf);
        alloc = 1;
      }
    } else alloc = 1;

    if (alloc) {
      ebuf = (jack_midi_data_t *) malloc (ev.size);
      ebuf_size = ev.size;
    }

    ev.buffer = ebuf;
    jack_ringbuffer_read(aout.midi_rb, (char *)&ev.size, sizeof ev.size);
    jack_ringbuffer_read(aout.midi_rb, (char *)&ev.time, sizeof ev.time);
    jack_ringbuffer_read(aout.midi_rb, (char *)ev.buffer, ev.size);

    extern keyboard_keyboard keybd2;

    if ((*ev.buffer & 0xf0) == 0xb0) { // control change
      run_midi_cmd ("midi-cc", ev.size, ev.buffer, args);
    } else if ((*ev.buffer & 0xf0) == 0x90) { // note on
      run_midi_cmd ("midi-note-on", ev.size, ev.buffer, args);
      keybd2.note_on (args[1], args[2]);
    } else if ((*ev.buffer & 0xf0) == 0x80) { // note off
      run_midi_cmd ("midi-note-off", ev.size, ev.buffer, args);
      keybd2.note_off (args[1]);
    } else if ((*ev.buffer & 0xf0) == 0xc0) { // program change
      run_midi_cmd ("midi-program-change", ev.size, ev.buffer, args);
    } else if ((*ev.buffer & 0xf0) == 0xe0) { // pitch bend

      char status = ev.buffer[0];
      char lsb = ev.buffer[1];
      char msb = ev.buffer[2];

      // normalise pitchbend range to -1.0...1.0
      short value = (lsb | (msb << 7)) - 0x2000;
      float fvalue = (float)value / ((value < 0) ? 0x2000 : 0x1FFF);

      keybd2.pitch_bend (fvalue);

      ostringstream msg;
      msg << "midi-pitch-bend " << status << " " << value << " " << fvalue;
      cons (msg.str());

    } else if (*ev.buffer == 0xf8) { // midi clock

      double elapsed_time;
      if (ev.time >= start_time) {
        elapsed_time = ev.time - start_time;
      } else { // time has wrapped (~ 27 hours @ 44100Hz)
        elapsed_time = UINT_MAX - start_time + ev.time;
      }
      ++num_clocks;

      if (elapsed_time >= SAMPLE_RATE) {
        double f_num_clocks = 1. / elapsed_time * num_clocks;
        double midi_bpm = f_num_clocks * 2.5 * SAMPLE_RATE;
        MIDI_BPM = midi_bpm;
        Tcl_UpdateLinkedVar (interpreter.interp, "midibpm");
        cons ("midi-clock");
        num_clocks = 0;
        start_time = ev.time;
      }
    } else if (*ev.buffer == 0xfa) { // midi start
      cons ("midi-start");
      num_clocks = 0;
      start_time = ev.time;
    }
  }

  if (ebuf) free (ebuf);

}

void applyfx (float* out0, float* out1, int do_delay, int do_compress) {

  // apply delays
  //
	
	float fdr = uis.fdr_delay.amount;
	float* outl = out0; left_delay (outl, aout.samples_per_channel, fdr);
	float* outr = out1; right_delay (outr, aout.samples_per_channel, fdr);

  // apply compression
  //

  extern compressor coml, comr;
  if (do_compress) {
    float* outl = out0, *outr = out1;
    coml.apply (outl, aout.samples_per_channel);
    comr.apply (outr, aout.samples_per_channel);
  }
}

void update_audio () {

  // stream status
  aout.available [aout.writei] = 1;
  if (++aout.writei >= aout.num_samples_buffers) {
    aout.writep = aout.samples_buffers;
    aout.writei = 0;
  } else aout.writep += aout.samples_per_buffer;

  // store timenow as seconds on din audio clock
  extern double TIME_NOW; TIME_NOW = clk.secs;
  extern tcl_interp interpreter;
  Tcl_UpdateLinkedVar (interpreter.interp, "timenow");

}

void connection_changed (jack_port_id_t a, jack_port_id_t b, int connect, void *arg) {

  audio_out* ao = (audio_out *) arg;

  // handle MIDI connections
  int n = jack_port_connected (ao->midi_in);
  if (n) {
    if (ao->num_midi_con == 0) {
      cons << console::green << "MIDI connected" << eol;
      cout << PASS << "+++ MIDI connected +++" << ENDL;
    }
  } else {
    if (ao->num_midi_con) {
      cons << console::red << "MIDI disconnected" << eol;
      cons << FAIL << "--- MIDI disconnected ---" << ENDL;
    }
  }
  ao->num_midi_con = n;



}
