/*******************************************************************************

  Intel Data Center Bridging (DCB) Software
  Copyright(c) 2007-2009 Intel Corporation.

  This program is free software; you can redistribute it and/or modify it
  under the terms and conditions of the GNU General Public License,
  version 2, as published by the Free Software Foundation.

  This program is distributed in the hope it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along with
  this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.

  The full GNU General Public License is included in this distribution in
  the file called "COPYING".

  Contact Information:
  e1000-eedc Mailing List <e1000-eedc@lists.sourceforge.net>
  Intel Corporation, 5200 N.E. Elam Young Parkway, Hillsboro, OR 97124-6497

*******************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <syslog.h>
#include <sys/ipc.h>
#include <sys/shm.h>
#include "dcb_protocol.h"
#include "dcbd_shm.h"

/* return: 1 = success, 0 = failed */
void mark_dcbx_shm_for_removal()
{
	int shmid;
	char *shmaddr=NULL;
	struct shmid_ds shminfo;

	shmid = shmget(DCBD_SHM_KEY, DCBD_SHM_SIZE, 0);

	if (shmid < 0)
		return;

	if (shmctl(shmid, IPC_RMID, &shminfo) < 0)
		return;
}

/* return: 1 = success, 0 = failed */
int clear_dcbx_state()
{
	int shmid;
	char *shmaddr=NULL;
	struct shmid_ds shminfo;

	shmid = shmget(DCBD_SHM_KEY, DCBD_SHM_SIZE, 0);

	if (shmid < 0)
		return 0;

	if (shmctl(shmid, IPC_STAT, &shminfo) < 0)
		return 0;

	shmaddr = (char *)shmat(shmid, NULL, 0);
	if ((long) shmaddr == -1)
		return 0;

	memset(shmaddr, 0, shminfo.shm_segsz);

	shmdt(shmaddr);
	return 1;
}

/* return: 1 = success, 0 = failed */
int set_dcbx_state(const char *device_name, dcbx_state *state)
{
	int shmid;
	struct dcbd_shm_tbl *shmaddr=NULL;
	int i;
	int rval = 0;

	shmid = shmget(DCBD_SHM_KEY, DCBD_SHM_SIZE, 0);
	if (shmid < 0 && errno == ENOENT)
		shmid = shmget(DCBD_SHM_KEY, DCBD_SHM_SIZE,
			IPC_CREAT | IPC_EXCL | 0x180);

	if (shmid < 0)
		return rval;

	shmaddr = (struct dcbd_shm_tbl *)shmat(shmid, NULL, 0);
	if ((long) shmaddr == -1)
		return rval;

	/* check for invalid number of shm table entries */
	if (shmaddr->num_entries < 0 ||
		shmaddr->num_entries > MAX_DCBD_SHM_ENTRIES)
		goto done;

	/* search for existing entry */
	for (i = 0; i < shmaddr->num_entries; i++)
		if (strcmp(shmaddr->ent[i].dn, device_name) == 0)
			break;

	if (i < MAX_DCBD_SHM_ENTRIES) {
		if (i == shmaddr->num_entries) {
			shmaddr->num_entries++;
			sprintf(shmaddr->ent[i].dn, "%.*s",
				IFNAMSIZ, device_name);
		}
		memcpy((void *)&shmaddr->ent[i].st, state, sizeof(dcbx_state));
		rval = 1;
	}

done:
	shmdt(shmaddr);

	return rval;
}

/* find and return a dcbx_state for the given device_name.
 * clear the dcbx_state entry after getting it - it's only valid
 * for the first read.
 * return: 1 = success, 0 = failed */
int get_dcbx_state(const char *device_name, dcbx_state *state)
{
	int shmid;
	struct dcbd_shm_tbl *shmaddr=NULL;
	int i;
	int rval = 0;

	shmid = shmget(DCBD_SHM_KEY, DCBD_SHM_SIZE, 0);

	if (shmid < 0)
		return rval;

	shmaddr = (struct dcbd_shm_tbl *)shmat(shmid, NULL, 0);
	if ((long) shmaddr == -1)
		return rval;

	/* check for invalid number of shm table entries */
	if (shmaddr->num_entries <= 0 ||
		shmaddr->num_entries > MAX_DCBD_SHM_ENTRIES)
		goto done;

	/* search for existing entry */
	for (i = 0; i < shmaddr->num_entries; i++)
		if (strcmp(shmaddr->ent[i].dn, device_name) == 0) {
			memcpy(state, (void *)&shmaddr->ent[i].st,
				sizeof(dcbx_state));
			memset((void *)&shmaddr->ent[i].st, 0,
				sizeof(dcbx_state));
			rval = 1;
			break;
		}

done:
	shmdt(shmaddr);

	return rval;
}


#ifdef SHM_UTL
/* compile utility to print out dcbd shared memory segment as follows:
 *    gcc -o dcbd_shm -Iinclude -DSHM_UTL dcbd_shm.c
*/

int print_dcbd_shm()
{
	int shmid;
	struct dcbd_shm_tbl *shmaddr=NULL;
	int i;
	int rval = 0;

	shmid = shmget(DCBD_SHM_KEY, DCBD_SHM_SIZE, 0);

	if (shmid < 0) {
		printf("failed to shmget\n");
		return rval;
	}

	shmaddr = (struct dcbd_shm_tbl *)shmat(shmid, NULL, 0);
	if ((long) shmaddr == -1) {
		printf("failed to shmat\n");
		return rval;
	}

	printf("num_entries = %d\n", shmaddr->num_entries);

	/* check for invalid number of shm table entries */
	if (shmaddr->num_entries <= 0 ||
		shmaddr->num_entries > MAX_DCBD_SHM_ENTRIES)
		goto done;

	for (i = 0; i < shmaddr->num_entries; i++) {
		printf("ifname:     %s\n", shmaddr->ent[i].dn);
		printf("SeqNo:      %d\n", shmaddr->ent[i].st.SeqNo);
		printf("AckNo:      %d\n", shmaddr->ent[i].st.AckNo);
		printf("FCoEenable: %d\n", shmaddr->ent[i].st.FCoEenable);
	}
	rval = 1;

done:
	shmdt(shmaddr);

	return rval;
}

static void usage(void)
{
        fprintf(stderr,
                "\n"
                "usage: dcbd_shm [-r]"
                "\n"
                "options:\n"
                "   -r  dump dcbd shared memory in raw format\n");

        exit(1);
}

main()
{
	print_dcbd_shm();
}
#endif
