/*******************************************************************************

  Intel Data Center Bridging (DCB) Software
  Copyright(c) 2007-2009 Intel Corporation.

  This program is free software; you can redistribute it and/or modify it
  under the terms and conditions of the GNU General Public License,
  version 2, as published by the Free Software Foundation.

  This program is distributed in the hope it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along with
  this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.

  The full GNU General Public License is included in this distribution in
  the file called "COPYING".

  Contact Information:
  e1000-eedc Mailing List <e1000-eedc@lists.sourceforge.net>
  Intel Corporation, 5200 N.E. Elam Young Parkway, Hillsboro, OR 97124-6497

*******************************************************************************/

#include "dcb_protocol.h"
#include "dcb_driver_interface.h"
#include "dcb_osdep.h"
#include "dcb_persist_store.h"
#include "dcb_rule_chk.h"
#include "dcb_events.h"
#include "messages.h"
#include <map>
#include <string>

static void handle_opermode_true(char *device_name);
u8        gdcbx_subtype = dcbx_subtype2;

int set_configuration(char *device_name, u32 EventFlag);

/* Redefine with default specified */
dcb_result run_feature_protocol(char *device_name, u32 EventFlag,
				u32 Subtype = 0);
dcb_result run_dcb_protocol(char *device_name, u32 EventFlag, u32 Subtype = 0);

using namespace std;

typedef map <string, pg_attribs*> pg_store1;
typedef map <string, pg_attribs*>::iterator pg_it;
pg_store1 pg;
pg_store1 peer_pg;
pg_store1 oper_pg;

typedef map <string, pfc_attribs*> pfc_store;
typedef map <string, pfc_attribs*>::iterator pfc_it;
pfc_store pfc;
pfc_store peer_pfc;
pfc_store oper_pfc;

typedef map <string, pg_info*> pg_desc_store;
typedef map <string, pg_info*>::iterator pg_desc_it;
pg_desc_store pg_desc;

/* String here is adapter name appened with APP TLV subtype. */
typedef map <string, app_attribs*> app_store;
typedef map <string, app_attribs*>::iterator app_it;
app_store apptlv;
app_store peer_apptlv;
app_store oper_apptlv;

typedef map <string, llink_attribs*> llink_store;
typedef map <string, llink_attribs*>::iterator llink_it;
llink_store llink;
llink_store peer_llink;
llink_store oper_llink;

typedef map <string, control_protocol_attribs*> dcb_control_protocol;
typedef map <string, control_protocol_attribs*>::iterator control_prot_it;
dcb_control_protocol dcb_control_prot;
dcb_control_protocol dcb_peer_control_prot;

typedef map <string, feature_support*> features_store;
typedef map <string, feature_support*>::iterator features_it;
features_store feature_struc;


extern "C" int add_port(const char *device_name);
extern "C" int remove_port(const char *device_name);
extern "C" void process_somethingChangedLocal(char *ifname,
						boolean_t NewFeatureTLVs);


/* Add the store pointer to init_pg, i.e. memset store to 0,
 * then copy attribs to store
 */
void init_pg(pg_attribs *Attrib, pg_attribs *Store)
{
	memset(Store,0,sizeof(*Store));

	Attrib->protocol.Max_version = DCB_PG_MAX_VERSION;
	Attrib->protocol.State = DCB_INIT;
	Attrib->protocol.Advertise_prev = FALSE;
	Attrib->protocol.tlv_sent = FALSE;
	Attrib->protocol.dcbx_st = gdcbx_subtype;
	memcpy(Store, Attrib, sizeof (*Attrib));
}

/* pass in the pointer to attrib */
boolean_t add_pg(char *device_name, pg_attribs *Attrib)
{
	TRY_BEGIN;
	string dn(device_name);

	pg_it it = pg.find(dn);
	if (it == pg.end()) {
		/* Device not present: add */
		pg_attribs *store =
			(pg_attribs*)malloc(sizeof(*store));
		if (!store) return FALSE;

		init_pg(Attrib, store);
		pg.insert(make_pair(dn, store));
	}
	else {  /* already in data store, just update it */
		(*it).second->protocol.Advertise_prev =
				(*it).second->protocol.Advertise;
		(*it).second->protocol.Advertise = Attrib->protocol.Advertise;
		(*it).second->protocol.Enable = Attrib->protocol.Enable;
		(*it).second->protocol.Willing = Attrib->protocol.Willing;

		memcpy(&(*it).second->rx, &(Attrib->rx), sizeof(Attrib->rx));
		memcpy(&(*it).second->tx, &(Attrib->tx), sizeof(Attrib->tx));
	}
	return TRUE;

	CATCH_END(FALSE);
}

void init_oper_pg(pg_attribs *Attrib)
{
	char   sTmp[MAX_DESCRIPTION_LEN];

	memset(Attrib,0,sizeof(*Attrib));
	snprintf(sTmp, MAX_DESCRIPTION_LEN, DEF_CFG_STORE);
	pg_it itpg = pg.find(sTmp);
	if (itpg == pg.end()) {
		return;
	}
	memcpy(&(Attrib->rx), &(itpg->second->rx), sizeof(Attrib->rx));
	memcpy(&(Attrib->tx), &(itpg->second->tx), sizeof(Attrib->tx));
}

boolean_t add_oper_pg(char *device_name)
{
	TRY_BEGIN;
	string dn(device_name);

	pg_it it = oper_pg.find(dn);
	if (it == oper_pg.end()) {
		/* Device not present: add */
		pg_attribs *store =
			(pg_attribs*)malloc(sizeof(*store));

		if (!store) return FALSE;

		init_oper_pg( store);
		oper_pg.insert(make_pair(dn, store));
	}
	return TRUE;

	CATCH_END(FALSE);
}

void init_peer_pg(pg_attribs *Attrib)
{
	memset(Attrib,0,sizeof(*Attrib));
	Attrib->protocol.TLVPresent =  FALSE;
}

boolean_t add_peer_pg(char *device_name)
{
	TRY_BEGIN;
	string dn(device_name);

	pg_it it = peer_pg.find(dn);
	if (it == peer_pg.end()) {
		/* Device not present: add */
		pg_attribs *store = (pg_attribs*)malloc(sizeof(*store));

		if (!store) return FALSE;

		init_peer_pg( store);
		peer_pg.insert(make_pair(dn, store));
	}
	return TRUE;

	CATCH_END(FALSE);
}

void init_apptlv(app_attribs *Attrib, app_attribs *Store)
{
	memset(Store,0,sizeof(*Store));
	Attrib->protocol.Max_version     = DCB_APPTLV_MAX_VERSION;
	Attrib->protocol.State           = DCB_INIT;
	Attrib->protocol.Advertise_prev  = FALSE;
	Attrib->protocol.tlv_sent        = FALSE;
	Attrib->protocol.dcbx_st         = gdcbx_subtype;
	memcpy(Store, Attrib, sizeof (*Attrib));
}

boolean_t add_apptlv(char *device_name, app_attribs *Attrib, u32 Subtype,
	dcbx_state *state)
{
	full_dcb_attrib_ptrs  attr_ptr;

	/* Append Subtype to device name and then add.*/
	TRY_BEGIN;
	string dn(device_name);

	char sTemp[SHORT_STRING];
	/* safe from overflow, only called from within for loop */
	snprintf(sTemp, SHORT_STRING, "%hu", Subtype);
	/* Append device name with subtype. */
	dn.append(sTemp);

	/* If FCoE is enabled in the DCBX state record,
	 * then enable the FCoE App object and persist the change.
	*/
	if (state && Subtype == APP_FCOE_STYPE && state->FCoEenable &&
		!Attrib->protocol.Enable) {
		Attrib->protocol.Enable = TRUE;
		memset(&attr_ptr, 0, sizeof(attr_ptr));
		attr_ptr.app = Attrib;
		attr_ptr.app_subtype = (u8)Subtype;
		if (set_persistent(device_name, &attr_ptr) !=
			dcb_success) {
			printf("Set persistent failed in add_apptlv\n");
			return FALSE;
		}
	}

	app_it it = apptlv.find(dn);
	if (it == apptlv.end()) {
		/* Device not present: add */
		app_attribs *store = (app_attribs*)malloc(sizeof(*store));

		if (!store) return FALSE;

		init_apptlv(Attrib, store);
		apptlv.insert(make_pair(dn, store));
	}
	else {  /* already in data store, just update it */
		(*it).second->protocol.Advertise_prev =
			(*it).second->protocol.Advertise;
		(*it).second->protocol.Advertise = Attrib->protocol.Advertise;
		(*it).second->protocol.Enable = Attrib->protocol.Enable;
		(*it).second->protocol.Willing = Attrib->protocol.Willing;
		(*it).second->Length = Attrib->Length;
		memcpy(&(*it).second->AppData, &(Attrib->AppData),
			Attrib->Length);
	}
	return TRUE;

	CATCH_END(FALSE);
}

void init_oper_apptlv(app_attribs *Store, u32 Subtype)
{
	char   sTmp[MAX_DESCRIPTION_LEN];

	memset(Store,0,sizeof(*Store));
	snprintf(sTmp, MAX_DESCRIPTION_LEN, "%s%hu", DEF_CFG_STORE, Subtype);
	app_it itapp = apptlv.find(sTmp);
	if (itapp == apptlv.end()) {
		return;
	}
	Store->Length =itapp->second->Length;
	memcpy(&(Store->AppData), &(itapp->second->AppData), sizeof(Store->Length));
}

boolean_t add_oper_apptlv(char *device_name, u32 Subtype)
{
	/* Append Subtype to device name and then add. */
	TRY_BEGIN;
	string dn(device_name);
	char sTemp[SHORT_STRING];

	/* safe from overflow, only called from within for loop */
	snprintf(sTemp, SHORT_STRING, "%hu", Subtype);/* Localization OK */
	/* Append device name with subtype. */
	dn.append(sTemp);

	app_it it = oper_apptlv.find(dn);
	if (it == oper_apptlv.end()) {
		/* Device not present: add */
		app_attribs *store = (app_attribs*)malloc(sizeof(*store));
		if (!store) return FALSE;
		init_oper_apptlv(store, Subtype);
		oper_apptlv.insert(make_pair(dn, store));
	}
	return TRUE;

	CATCH_END(FALSE);
}

void init_peer_apptlv(app_attribs *Store)
{
	memset(Store,0,sizeof(*Store));
}

boolean_t add_peer_apptlv(char *device_name, u32 Subtype)
{
	/* Append Subtype to device name and then add. */
	TRY_BEGIN;
	string dn(device_name);
	char sTemp[SHORT_STRING];
	/* safe from overflow, only called from within for loop */
	snprintf(sTemp, SHORT_STRING, "%hu", Subtype);
	/* Append device name with subtype. */
	dn.append(sTemp);

	app_it it = peer_apptlv.find(dn);
	if (it == peer_apptlv.end()) {
		/* Device not present: add */
		app_attribs *store = (app_attribs*)malloc(sizeof(*store));
		if (!store) return FALSE;

		init_peer_apptlv(store);
		peer_apptlv.insert(make_pair(dn, store));
	}
	return TRUE;

	CATCH_END(FALSE);
}

void init_control_prot(control_protocol_attribs *Attrib, dcbx_state *state)
{
	memset(Attrib,0,sizeof(*Attrib));
	Attrib->State = DCB_INIT;
	Attrib->Max_version = DCB_MAX_VERSION;
	Attrib->SeqNo = state->SeqNo;
	Attrib->AckNo = state->AckNo;
	Attrib->MyAckNo = state->SeqNo;
}

boolean_t add_control_protocol(char *device_name, dcbx_state *state)
{
	TRY_BEGIN;
	string dn(device_name);

	control_prot_it it = dcb_control_prot.find(dn);
	if (it == dcb_control_prot.end()) {
		/* Device not present: add */
		control_protocol_attribs *store =
			(control_protocol_attribs*)malloc(sizeof(*store));

		if (!store) return FALSE;

		init_control_prot(store, state);
		dcb_control_prot.insert(make_pair(dn, store));
	}
	return TRUE;

	CATCH_END(FALSE);
}

void init_peer_control_prot(control_protocol_attribs *Attrib)
{
	memset(Attrib,0,sizeof(*Attrib));
	Attrib->Oper_version = DCB_MAX_VERSION;
	Attrib->Max_version = DCB_MAX_VERSION;
	Attrib->RxDCBTLVState = DCB_PEER_NONE;
}

boolean_t add_peer_control_protocol(char *device_name)
{
	TRY_BEGIN;
	string dn(device_name);

	control_prot_it it = dcb_peer_control_prot.find(dn);
	if (it == dcb_peer_control_prot.end()) {
		/* Device not present: add */
		control_protocol_attribs *store =
			(control_protocol_attribs*)malloc(sizeof(*store));

		if (!store) return FALSE;

		init_peer_control_prot( store);
		dcb_peer_control_prot.insert(make_pair(dn, store));
	}
	return TRUE;

	CATCH_END(FALSE);
}

void init_pfc(pfc_attribs *Attrib, pfc_attribs *Store)
{
	memset(Store,0,sizeof(*Store));
	Attrib->protocol.Max_version = DCB_PFC_MAX_VERSION;
	Attrib->protocol.State = DCB_INIT;
	Attrib->protocol.Advertise_prev = FALSE;
	Attrib->protocol.tlv_sent = FALSE;
	Attrib->protocol.dcbx_st = gdcbx_subtype;
	memcpy(Store, Attrib, sizeof(*Attrib));
}

boolean_t add_pfc(char *device_name, pfc_attribs *Attrib)
{
	TRY_BEGIN;
	string dn(device_name);

	pfc_it it = pfc.find(dn);
	if (it == pfc.end()) {
		/* Device not present: add */
		pfc_attribs *store =
			(pfc_attribs*)malloc(sizeof(*store));

		if (!store) return FALSE;

		init_pfc(Attrib, store);
		pfc.insert(make_pair(dn, store));
	} else {  /* already present in store, just update */
		(*it).second->protocol.Advertise_prev =
			(*it).second->protocol.Advertise;
		(*it).second->protocol.Advertise = Attrib->protocol.Advertise;
		(*it).second->protocol.Enable = Attrib->protocol.Enable;
		(*it).second->protocol.Willing = Attrib->protocol.Willing;

		memcpy((*it).second->admin, &Attrib->admin,
			sizeof(Attrib->admin));
	}
	return TRUE;

	CATCH_END(FALSE);
}

void init_oper_pfc(pfc_attribs *Attrib)
{
	char   sTmp[MAX_DESCRIPTION_LEN];

	memset(Attrib,0,sizeof(*Attrib));
	snprintf(sTmp, MAX_DESCRIPTION_LEN, DEF_CFG_STORE);
	pfc_it itpfc = pfc.find(sTmp);
	if (itpfc == pfc.end()) {
		return;
	}
	memcpy(&(Attrib->admin), &(itpfc->second->admin), sizeof(Attrib->admin));
}

boolean_t add_oper_pfc(char *device_name)
{
	TRY_BEGIN;
	string dn(device_name);

	pfc_it it = oper_pfc.find(dn);
	if (it == oper_pfc.end()) {
		/* Device not present: add */
		pfc_attribs *store =
			(pfc_attribs*)malloc(sizeof(*store));

		if (!store) return FALSE;

		init_oper_pfc(store);
		oper_pfc.insert(make_pair(dn, store));
	}
	return TRUE;

	CATCH_END(FALSE);
}

void init_peer_pfc(pfc_attribs *Attrib)
{
	memset(Attrib,0,sizeof(*Attrib));
	Attrib->protocol.TLVPresent =  FALSE;
}

boolean_t add_peer_pfc(char *device_name)
{
	TRY_BEGIN;
	string dn(device_name);

	pfc_it it = peer_pfc.find(dn);
	if (it == peer_pfc.end()) {
		/* Device not present: add */
		pfc_attribs *store = (pfc_attribs*)malloc(sizeof(*store));

		if (!store) return FALSE;

		init_peer_pfc(store);
		peer_pfc.insert(make_pair(dn, store));
	}
	return TRUE;

	CATCH_END(FALSE);
}

void init_bwg_desc(pg_info *Attrib, pg_info *Store)
{
	memset(Store,0,sizeof(*Store));
	memcpy(Store, Attrib, sizeof(*Attrib));
}

boolean_t add_bwg_desc(char *device_name, pg_info *Attrib)
{
	TRY_BEGIN;
	string dn(device_name);

	pg_desc_it it = pg_desc.find(dn);
	if (it == pg_desc.end()) {
		/* Device not present: add */
		pg_info *store =
			(pg_info*)malloc(sizeof(*store));

		if (!store) return FALSE;

		init_bwg_desc(Attrib, store);
		store->max_pgid_desc = 8;
		pg_desc.insert(make_pair(dn, store));
	}
	return TRUE;

	CATCH_END(FALSE);
}

/* Logical Link */
void init_llink(llink_attribs *Attrib, llink_attribs *Store)
{
	memset(Store,0,sizeof(*Store));
	Attrib->protocol.Max_version = DCB_LLINK_MAX_VERSION;
	Attrib->protocol.State = DCB_INIT;
	Attrib->protocol.Advertise_prev = FALSE;
	Attrib->protocol.tlv_sent = FALSE;
	Attrib->protocol.dcbx_st = gdcbx_subtype;
	memcpy(Store, Attrib, sizeof(*Attrib));
}

boolean_t add_llink(char *device_name, llink_attribs *Attrib, u32 subtype)
{
	TRY_BEGIN;
	string dn(device_name);

	char sTemp[SHORT_STRING];
	/* safe from overflow, only called from within for loop */
	snprintf(sTemp, SHORT_STRING, "%hu", subtype);
	/* Append device name with subtype. */
	dn.append(sTemp);

	llink_it it = llink.find(dn);
	if (it == llink.end()) {
		/* Device not present: add */
		llink_attribs *store =
			(llink_attribs*)malloc(sizeof(*store));

		if (!store) return FALSE;

		init_llink(Attrib, store);
		llink.insert(make_pair(dn, store));
	} else {  /* already present in store, just update */
		(*it).second->protocol.Advertise_prev =
			(*it).second->protocol.Advertise;
		(*it).second->protocol.Advertise = Attrib->protocol.Advertise;
		(*it).second->protocol.Enable = Attrib->protocol.Enable;
		(*it).second->protocol.Willing = Attrib->protocol.Willing;

		memcpy(&(*it).second->llink, &(Attrib->llink),
			sizeof(Attrib->llink));
	}
	return TRUE;

	CATCH_END(FALSE);
}
void init_oper_llink(llink_attribs *Attrib, u32 Subtype)
{
	char   sTmp[MAX_DESCRIPTION_LEN];

	memset(Attrib,0,sizeof(*Attrib));
	snprintf(sTmp, MAX_DESCRIPTION_LEN, "%s%hu", DEF_CFG_STORE, Subtype);
	llink_it itllink = llink.find(sTmp);
	if (itllink == llink.end()) {
		return;
	}
	memcpy(&(Attrib->llink), &(itllink->second->llink), sizeof(Attrib->llink));
}

boolean_t add_oper_llink(char *device_name, u32 subtype)
{
	TRY_BEGIN;
	string dn(device_name);
	char sTemp[SHORT_STRING];

	/* safe from overflow, only called from within for loop */
	snprintf(sTemp, SHORT_STRING, "%hu", subtype);/* Localization OK */
	/* Append device name with subtype. */
	dn.append(sTemp);

	llink_it it = oper_llink.find(dn);
	if (it == oper_llink.end()) {
		/* Device not present: add */
		llink_attribs *store =
			(llink_attribs*)malloc(sizeof(*store));

		if (!store) return 0;

		init_oper_llink(store, subtype);
		oper_llink.insert(make_pair(dn, store));
	}
	return TRUE;

	CATCH_END(FALSE);
}

void init_peer_llink(llink_attribs *Attrib)
{
	memset(Attrib,0,sizeof(*Attrib));
	Attrib->protocol.TLVPresent =  FALSE;
}

boolean_t add_peer_llink(char *device_name, u32 subtype)
{
	TRY_BEGIN;
	string dn(device_name);
	char sTemp[SHORT_STRING];

	/* safe from overflow, only called from within for loop */
	snprintf(sTemp, SHORT_STRING, "%hu", subtype);/* Localization OK */
	/* Append device name with subtype. */
	dn.append(sTemp);

	llink_it it = peer_llink.find(dn);
	if (it == peer_llink.end()) {
		/* Device not present: add */
		llink_attribs *store = (llink_attribs*)malloc(sizeof(*store));

		if (!store) return 0;

		init_peer_llink(store);
		peer_llink.insert(make_pair(dn, store));
	}
	return TRUE;

	CATCH_END(FALSE);
}

boolean_t init_dcb_support(char *device_name, full_dcb_attribs *attrib)
{
	feature_support dcb_support;

	memset(&dcb_support, 0, sizeof(feature_support));
	if (get_dcb_capabilities(device_name, &dcb_support) != 0) {
		return FALSE;
	}

	/*if (!dcb_support.pg) {
	 *    attrib->pg.protocol.Enable = FALSE;
	 *    attrib->pg.protocol.Advertise = FALSE;
	 *}
	 *if (!dcb_support.pfc) {
	 *    attrib->pfc.protocol.Enable = FALSE;
	 *    attrib->pfc.protocol.Advertise = FALSE;
	 *}
	 */

	if (get_dcb_numtcs(device_name, &attrib->pg.num_tcs,
		&attrib->pfc.num_tcs) != 0) {
		return FALSE;
	}

	TRY_BEGIN;
	string dn(device_name);

	features_it it = feature_struc.find(dn);
	if (it == feature_struc.end()) {
		/* Device not present: add */
		feature_support *store =
			(feature_support*)malloc(sizeof(*store));
		if (!store) return FALSE;

		memcpy(store, (void *)&dcb_support, sizeof(*store));
		feature_struc.insert(make_pair(dn, store));
	}
	return TRUE;

	CATCH_END(FALSE);
}

dcb_result get_dcb_support(char *device_name,  feature_support *dcb_support)
{
	TRY_BEGIN;
	string              dn(device_name);
	dcb_result          result = dcb_success;
	feature_support     features;
	full_dcb_attribs    attribs;

	memset(&attribs, 0, sizeof(attribs));
	memset(&features, 0, sizeof(feature_support));
	if (!dcb_support) return dcb_bad_params;
	features_it it = feature_struc.find(dn);
	if (it != feature_struc.end()) {
		memcpy( dcb_support, (*it).second, sizeof(*dcb_support));
	} else {
		if (get_persistent(device_name, &attribs) != dcb_success) {
			result = dcb_device_not_found;
			goto Exit;
		}
		if (!init_dcb_support(device_name, &attribs)) {
			result = dcb_device_not_found;
			goto Exit;
		}
		memset(&features, 0, sizeof(feature_support));
		features_it it = feature_struc.find(dn);
		if (it != feature_struc.end()) {
			memcpy( dcb_support, (*it).second, sizeof(*dcb_support));
		} else {
			result = dcb_device_not_found;
		}
	}
Exit:
	return result;

	CATCH_END(dcb_failed);
}

void remove_dcb_support(void)
{
	void  *itp = NULL;

	TRY_BEGIN;
	features_it itfeat = feature_struc.begin();
	while (itfeat != feature_struc.end()) {
		itp = (void *) itfeat->second;
		feature_struc.erase((char *) (itfeat++->first.c_str()));
		if (itp)
			free(itp);
		itp = NULL;
	}
	CATCH_END_VOID;
}

boolean_t add_adapter(char *device_name)
{
	u32 EventFlag = 0;
	full_dcb_attrib_ptrs  attr_ptr;
	full_dcb_attribs      attribs;
	feature_support dcb_support = {0};
	dcb_result sResult = dcb_success;
	dcbx_state state;

	printf("add_adapter (%s)\n", device_name);
	memset(&attribs, 0, sizeof(attribs));

	if ((sResult = get_persistent(device_name, &attribs)) != dcb_success) {
		printf("get_persistent returned error %d\n", sResult);
		sResult = dcb_failed;
		goto add_adapter_error;
	}

	printf("  dcbx subtype = %d\n", gdcbx_subtype);

	memset (&attr_ptr, 0, sizeof(attr_ptr));
	attr_ptr.pg = &(attribs.pg);
	if ((sResult = dcb_check_config(&attr_ptr)) != dcb_success) {
		printf("Rule checker returned error %d\n", sResult);
	} else { /* big else */
		memset(&state, 0, sizeof(state));
		get_dcbx_state(device_name, &state);

		/* Create data stores for the device. */

		if (!init_dcb_support(device_name, &attribs)) {
			sResult = dcb_failed;
			goto add_adapter_error;
		}

		if (!add_pg(device_name, &attribs.pg)) {
			log_message(MSG_ERR_RESOURCE_MEMORY, "%s",device_name);
			printf("add_pg error.\n");
			sResult = dcb_failed;
			goto add_adapter_error;
		}
		if (!add_pfc(device_name, &attribs.pfc)) {
			log_message(MSG_ERR_RESOURCE_MEMORY, "%s",device_name);
			printf("add_pfc error.\n");
			sResult = dcb_failed;
			goto add_adapter_error;
		}
		if (!add_bwg_desc(device_name, &attribs.descript)) {
			log_message(MSG_ERR_RESOURCE_MEMORY, "%s",device_name);
			printf("add_bwg_desc error.\n");
			sResult = dcb_failed;
			goto add_adapter_error;
		}
		if (!add_control_protocol(device_name, &state)) {
			log_message(MSG_ERR_RESOURCE_MEMORY, "%s",device_name);
			printf("add_control_protocol error.\n");
			sResult = dcb_failed;
			goto add_adapter_error;
		}
		if (!add_peer_pg(device_name)) {
			log_message(MSG_ERR_RESOURCE_MEMORY, "%s",device_name);
			printf("add_peer_pg error.\n");
			sResult = dcb_failed;
			goto add_adapter_error;
		}
		if (!add_peer_pfc(device_name)) {
			log_message(MSG_ERR_RESOURCE_MEMORY, "%s",device_name);
			printf("add_peer_pfc error.\n");
			sResult = dcb_failed;
			goto add_adapter_error;
		}
		if (!add_peer_control_protocol(device_name)) {
			log_message(MSG_ERR_RESOURCE_MEMORY, "%s",device_name);
			printf("add_peer_control_protocol error.\n");
			sResult = dcb_failed;
			goto add_adapter_error;
		}
		if (!add_oper_pg(device_name)) {
			log_message(MSG_ERR_RESOURCE_MEMORY, "%s",device_name);
			printf("add_oper_pg error.\n");
			sResult = dcb_failed;
			goto add_adapter_error;
		}
		if (!add_oper_pfc(device_name)) {
			log_message(MSG_ERR_RESOURCE_MEMORY, "%s",device_name);
			printf("add_oper_pfc error.\n");
			sResult = dcb_failed;
			goto add_adapter_error;
		}
		/* Add APPTLV for supported Subtypes. */
		for(int i = 0; i < DCB_MAX_APPTLV ; i++) {
			if (!add_apptlv(device_name,
				&attribs.app[i], i, &state)) {
				log_message(MSG_ERR_RESOURCE_MEMORY, "%s",
					device_name);
				printf("add_apptlv error.\n");
				sResult = dcb_failed;
				goto add_adapter_error;
			}
			if (!add_oper_apptlv(device_name, i)) {
				log_message(MSG_ERR_RESOURCE_MEMORY, "%s",
					device_name);
				printf("add_oper_apptlv error.\n");
				sResult = dcb_failed;
				goto add_adapter_error;
			}
			if (!add_peer_apptlv(device_name, i)) {
				log_message(MSG_ERR_RESOURCE_MEMORY, "%s",
					device_name);
				printf("add_peer_apptlv error.\n");
				sResult = dcb_failed;
				goto add_adapter_error;
			}
		}
		for(int i = 0; i < DCB_MAX_LLKTLV ; i++) {
			if (!add_llink(device_name, &attribs.llink[i], i)) {
				log_message(MSG_ERR_RESOURCE_MEMORY, "%s",
					device_name);
				printf("add_llink error.\n");
				sResult = dcb_failed;
				goto add_adapter_error;
			}
			if (!add_oper_llink(device_name, i)) {
				log_message(MSG_ERR_RESOURCE_MEMORY, "%s",
					device_name);
				printf("add_oper_llink error.\n");
				sResult = dcb_failed;
				goto add_adapter_error;
			}
			if (!add_peer_llink(device_name, i)) {
				log_message(MSG_ERR_RESOURCE_MEMORY, "%s",
					device_name);
				printf("add_peer_llink error.\n");
				sResult = dcb_failed;
				goto add_adapter_error;
			}
		}

		if (get_dcb_support(device_name, &dcb_support) != dcb_success) {
			sResult = dcb_failed;
			goto add_adapter_error;
		}


		/* Initialize features state machines for PG and PFC and
		 * APPTLVs. */
		if (run_feature_protocol(device_name,
			DCB_LOCAL_CHANGE_PG) !=
			dcb_success) {
			printf("run_feature_protocol error (PG)\n");
			sResult = dcb_failed;
			goto add_adapter_error;
		}
		if (run_feature_protocol(device_name,
			DCB_LOCAL_CHANGE_PFC) != dcb_success) {
			printf("run_feature_protocol error (PFC)\n");
			sResult = dcb_failed;
			goto add_adapter_error;
		}
		/* If APPTLV subtypes are supported then run the
		 * feat_prot for those supported subtypes. */
		for (int i = 0; i < DCB_MAX_APPTLV ; i++) {
			if (run_feature_protocol(device_name,
				DCB_LOCAL_CHANGE_APPTLV, i) !=
				dcb_success) {
				printf("run_feature_protocol error (APP)\n");
				sResult = dcb_failed;
				goto add_adapter_error;
			}
		}
		for (int i = 0; i < DCB_MAX_LLKTLV ; i++) {
			if (run_feature_protocol(device_name,
				DCB_LOCAL_CHANGE_LLINK, i) != dcb_success) {
				printf("run_feature_protocol error (LLINK)\n");
				sResult = dcb_failed;
				goto add_adapter_error;
			}
		}

		/* apply all feature setting to the driver:linux only */
		set_hw_all(device_name);

		EventFlag = 0;
		DCB_SET_FLAGS(EventFlag, DCB_LOCAL_CHANGE_PG |
			DCB_LOCAL_CHANGE_PFC| DCB_LOCAL_CHANGE_APPTLV);
		DCB_SET_FLAGS(EventFlag, DCB_LOCAL_CHANGE_LLINK);
		/* Initialize control state machine */
		if (run_control_protocol (device_name, EventFlag) !=
			dcb_success) {
			printf("run_control_protocol error.\n");
			sResult = dcb_failed;
			goto add_adapter_error;
		}
	}	/* big else on top */


add_adapter_error:
	if (sResult != dcb_success) {
		printf("add_adapter: Service unable to use network adapter\n");
		log_message(MSG_ERR_ADD_CARD_FAILURE, "%s", device_name);
		return FALSE;
	}

	if (add_port(device_name) < 0) {
		printf("add_adapter: add_port failed.\n");
		log_message(MSG_ERR_ADD_CARD_FAILURE, "%s", device_name);
		return FALSE;
	}
	return TRUE;
}

boolean_t remove_adapter(char *device_name)
{
	/* Set feature state machines to DCB_CLOSED first */
	/* NOTE: the parameter "device_name" is freed during this routine
	 * and must not be used after it is copied*/
	char devName[MAX_DEVICE_NAME_LEN]; /* local copy of device_name */
	feature_protocol_attribs *pFP;
	void      *itp = NULL;
	int       not_default = 1;
	boolean_t retval = TRUE;

	/* save device name for remove port */
	if (device_name == NULL) {
		printf("remove_adapter: null device\n");
		return FALSE;
	}

	not_default = memcmp(DEF_CFG_STORE, device_name,
		strlen(DEF_CFG_STORE));

	if (not_default)
		handle_opermode_true(device_name);

	strncpy (devName, device_name, MAX_DEVICE_NAME_LEN);

	if (not_default && (remove_port(devName) < 0)) {
		printf("remove_port: failed\n");
		retval = FALSE;
	}

	TRY_BEGIN;
	string dn(devName);

	features_it itfeat = feature_struc.find(dn);
	if (itfeat != feature_struc.end()) {
		itp = (void *) itfeat->second;
		feature_struc.erase(dn);
		printf("free: dcb support %p\n", itp);
		if (itp)
			free(itp);
		itp = NULL;
	} else {
		printf("remove_adapter: dcb support found\n");
	}

	pg_it itpg = pg.find(dn);
	if (itpg != pg.end()) {
		/* save the reserved memory here so we can free
		 * it since erase wipes out the it structure
		 * (but does not free the memory) */
		itp = (void *) itpg->second;

		/* Get the feature protocol for local store */
		pFP = (&((*itpg).second)->protocol);
		/* Set feature protocol state to closed. */
		pFP->State = DCB_CLOSED;
		/* this line frees the param of this function! */
		pg.erase(dn);
		if (itp) free (itp);
		itp = NULL;
	} else {
		printf("remove_adapter: pg not found\n");
	}

	pfc_it itpfc = pfc.find(dn);
	if (itpfc != pfc.end()) {
		itp = (void *) itpfc->second;
		pFP = (&((*itpfc).second)->protocol);
		pFP->State = DCB_CLOSED;
		pfc.erase(dn);
		if (itp)
			free(itp);
		itp = NULL;
	} else {
		printf("remove_adapter: pfc not found\n");
	}

	pg_desc_it itbwg = pg_desc.find(dn);
	if (itbwg != pg_desc.end()) {
		itp = (void *) itbwg->second;
		pg_desc.erase(dn);
		if (itp)
			free(itp);
		itp = NULL;
	} else if (not_default) {
		printf("remove_adapter: pgid not found\n");
	}

	control_prot_it itcp = dcb_control_prot.find(dn);
	if (itcp != dcb_control_prot.end()) {
		itp = (void *) itcp->second;
		dcb_control_prot.erase(dn);
		if (itp)
			free (itp);
		itp = NULL;
	} else if (not_default) {
		printf("remove_adapter: ctrl not found\n");
	}

	pg_it itprpg = peer_pg.find(dn);
	if (itprpg != peer_pg.end()) {
		itp = (void *) itprpg->second;
		pFP = (&((*itprpg).second)->protocol);
		pFP->State = DCB_CLOSED;
		peer_pg.erase(dn);
		if (itp) free (itp);
		itp = NULL;
	} else if (not_default) {
		printf("remove_adapter: peer pg not found\n");
	}

	pfc_it itprpfc = peer_pfc.find(dn);
	if (itprpfc != peer_pfc.end()) {
		itp = (void *) itprpfc->second;
		pFP = (&((*itprpfc).second)->protocol);
		pFP->State = DCB_CLOSED;
		peer_pfc.erase(dn);
		if (itp)
			free (itp);
		itp = NULL;
	} else if (not_default) {
		printf("remove_adapter: peer pfc not found\n");
	}

	control_prot_it itpcp = dcb_peer_control_prot.find(dn);
	if (itpcp != dcb_peer_control_prot.end()) {
		itp = (void *) itpcp->second;
		dcb_peer_control_prot.erase(dn);
		if (itp)
			free (itp);
		itp = NULL;
	} else if (not_default) {
		printf("remove_adapter: peer ctrl not found\n");
	}

	pg_it itopg = oper_pg.find(dn);
	if (itopg != oper_pg.end()) {
		itp = (void *) itopg->second;
		pFP = (&((*itopg).second)->protocol);
		pFP->State = DCB_CLOSED;
		oper_pg.erase(dn);
		if (itp)
			free(itp);
		itp = NULL;
	} else if (not_default) {
		printf("remove_adapter: oper pg not found\n");
	}

	pfc_it itopfc = oper_pfc.find(dn);
	if (itopfc != oper_pfc.end()) {
		itp = (void *) itopfc->second;
		pFP = (&((*itopfc).second)->protocol);
		pFP->State = DCB_CLOSED;
		oper_pfc.erase(dn);
		if (itp) free (itp);
		itp = NULL;
	} else if (not_default) {
		printf("remove_adapter: oper pfc not found\n");
	}

	/* Get the APP TLV  and erase. */
	for(int i = 0; i < DCB_MAX_APPTLV ; i++) {
		string dn1 = dn;
		char sTemp[SHORT_STRING];
		/* Localization OK */
		snprintf(sTemp, SHORT_STRING, "%hu", i);
		/* Append device name with subtype */
		dn1.append(sTemp);

		app_it itapp = apptlv.find(dn1);
		if (itapp != apptlv.end()) {
			itp = (void *) itapp->second;
			apptlv.erase(dn1);
			if (itp) free (itp);
			itp = NULL;
		}
		itapp = oper_apptlv.find(dn1);
		if (itapp != oper_apptlv.end()) {
			itp = (void *) itapp->second;
			oper_apptlv.erase(dn1);
			if (itp) free (itp);
			itp = NULL;
		}
		itapp = peer_apptlv.find(dn1);
		if (itapp != peer_apptlv.end()) {
			itp = (void *) itapp->second;
			peer_apptlv.erase(dn1);
			if (itp) free (itp);
			itp = NULL;
		}
	}
	for(int i = 0; i < DCB_MAX_LLKTLV ; i++) {
		string dn1 = dn;
		char sTemp[SHORT_STRING];
		/* Localization OK */
		snprintf(sTemp, SHORT_STRING, "%hu", i);
		/* Append device name with subtype */
		dn1.append(sTemp);

		/* Release LLINK data store */
		llink_it itllink = llink.find(dn1);
		if (itllink != llink.end()) {
			itp = (void *) itllink->second;
			pFP = (&((*itllink).second)->protocol);
			pFP->State = DCB_CLOSED;
			llink.erase(dn1);
			if (itp)
				free(itp);
			itp = NULL;
		} else {
			printf("remove_adapter: llink not found\n");
		}

		llink_it itprllink = peer_llink.find(dn1);
		if (itprllink != peer_llink.end()) {
			itp = (void *) itprllink->second;
			pFP = (&((*itprllink).second)->protocol);
			pFP->State = DCB_CLOSED;
			peer_llink.erase(dn1);
			if (itp)
				free (itp);
			itp = NULL;
		} else if (not_default) {
			printf("remove_adapter: peer llink not found\n");
		}

		llink_it itollink = oper_llink.find(dn1);
		if (itollink != oper_llink.end()) {
			itp = (void *) itollink->second;
			pFP = (&((*itollink).second)->protocol);
			pFP->State = DCB_CLOSED;
			oper_llink.erase(dn1);
			if (itp)
				free (itp);
			itp = NULL;
		} else if (not_default) {
			printf("remove_adapter: oper llink not found\n");
		}
	}

	CATCH
	{
		return FALSE;
	}

	return retval;
}


dcb_result save_dcbx_state(const char *device_name)
{
	TRY_BEGIN;
	string dn(device_name);
	dcbx_state state;
	app_attribs app_data;

	control_prot_it ctrl_prot = dcb_control_prot.find(dn);

	if (ctrl_prot == dcb_control_prot.end())
		return dcb_device_not_found;

	state.SeqNo = ((*ctrl_prot).second)->SeqNo;
	state.AckNo = ((*ctrl_prot).second)->AckNo;

	if (get_app((char*)device_name, 0, &app_data) == dcb_success)
		state.FCoEenable = app_data.protocol.Enable;
	else
		return dcb_bad_params;

	if (set_dcbx_state(device_name, &state))
		return dcb_success;
	else
		return dcb_failed;

	CATCH_END(dcb_failed);
}

/* Function to find all the dcb devices from store and remove them. */
void remove_all_adapters()
{
	pg_it it;
	char sTmp[MAX_DEVICE_NAME_LEN*2];

	clear_dcbx_state();

	it = pg.begin();
	while (it != pg.end()) {
		if (!memcmp(DEF_CFG_STORE, it->first.c_str(),
			strlen(DEF_CFG_STORE))) {
			it++;
			continue;
		}

		save_dcbx_state(it->first.c_str());

		/* prepare sTmp in case of error */
		snprintf(sTmp, MAX_DEVICE_NAME_LEN*2, /* Localization OK */
			"Remove_all_adapters error: Bad device name: %.*s\n",
			MAX_DEVICE_NAME_LEN, it->first.c_str());
		if (remove_adapter((char *) (it++->first.c_str())) == FALSE)
			printf(sTmp);
	}

	/* the default attributes object needs to be removed last */
	snprintf(sTmp, MAX_DEVICE_NAME_LEN*2,
		"Remove_all_adapters error: Bad device name: %.*s\n",
		MAX_DEVICE_NAME_LEN, DEF_CFG_STORE);
	if (remove_adapter(DEF_CFG_STORE) == FALSE)
		printf(sTmp);

	return;
}

/* Function to wait for all the dcb devices from store to be removed. */
/* (Currently used only by Windows) */
void wait_til_adapters_removed()
{
	int max_wait = 0;

	/* wait until all adapters are removed
	 * (removed when rx buffer is returned)
	*/
	while (pg.begin() != pg.end()) {
		if (pg.size() <= DEF_CFG_NUM)
			return;
		printf("waiting until adapters are removed %d sec \n",
			max_wait);
		Sleep(1000);
		max_wait++; 
	}
	return;
}

boolean_t add_pg_defaults()
{
	pg_attribs  pg_data;
	char        sTmp[MAX_DESCRIPTION_LEN];
	boolean_t   result = TRUE;
	int         index, portion, rmndr, temp;

	memset(&pg_data, 0, sizeof(pg_attribs));

	pg_data.protocol.Enable = 1;
	pg_data.protocol.Willing = 1;
	pg_data.protocol.Advertise = 1;

	portion = BW_PERCENT/MAX_BANDWIDTH_GROUPS;
	rmndr   = BW_PERCENT % MAX_BANDWIDTH_GROUPS;

	temp = rmndr;
	for (index=0; index < MAX_BANDWIDTH_GROUPS; index++) {
		pg_data.tx.pg_percent[index] = (u8)portion;
		if (temp >0) {
			pg_data.tx.pg_percent[index] += 1;
			temp--;
		}
	}
	for (index=0; index < MAX_USER_PRIORITIES; index++) {
		pg_data.tx.up[index].pgid = (u8)index;
		pg_data.tx.up[index].percent_of_pg_cap =
			BW_PERCENT;
		pg_data.tx.up[index].strict_priority = dcb_none;
		pg_data.tx.up[index].tcmap = (u8)(index);
	}
	temp = rmndr;
	for (index=0; index < MAX_BANDWIDTH_GROUPS; index++) {
		pg_data.rx.pg_percent[index] = (u8)portion;
		if (temp >0) {
			pg_data.rx.pg_percent[index]++;
			temp--;
		}
	}
	for (index=0; index < MAX_USER_PRIORITIES; index++) {
		pg_data.rx.up[index].pgid = (u8)index;
		pg_data.rx.up[index].percent_of_pg_cap =
			BW_PERCENT;
		pg_data.rx.up[index].strict_priority = dcb_none;
		pg_data.rx.up[index].tcmap = (u8)(index);
	}

	snprintf(sTmp, MAX_DESCRIPTION_LEN, DEF_CFG_STORE);

	/* Create pg default data store for the device. */
	if (!add_pg(sTmp, &pg_data))
		result = FALSE;

	return result;
}

boolean_t remove_pg_defaults()
{
	char   sTmp[MAX_DESCRIPTION_LEN];
	void * itp = NULL;

	snprintf(sTmp, MAX_DESCRIPTION_LEN, DEF_CFG_STORE);
	pg_it itpg = pg.find(sTmp);
	if (itpg == pg.end())
		return FALSE;
	itp = (void *) itpg->second;
	/* erase and free memory */
	pg.erase(sTmp);
	if (itp) {
		free (itp);
		itp = NULL;
	}
	return TRUE;
}

boolean_t add_pfc_defaults()
{
	pfc_attribs pfc_data;
	char        sTmp[MAX_DESCRIPTION_LEN];
	boolean_t   result = TRUE;
	int         index;

	memset (&pfc_data, 0, sizeof(pfc_attribs));

	pfc_data.protocol.Enable = 1;
	pfc_data.protocol.Willing = 1;
	pfc_data.protocol.Advertise = 1;

	for (index=0; index < MAX_TRAFFIC_CLASSES; index++) {
		pfc_data.admin[index] = pfc_disabled;
	}

	snprintf(sTmp, MAX_DESCRIPTION_LEN, DEF_CFG_STORE);
	/* Create pfc default data store for the device. */
	if (!add_pfc(sTmp, &pfc_data))
		result = FALSE;

	return result;
}

boolean_t remove_pfc_defaults()
{
	char   sTmp[MAX_DESCRIPTION_LEN];
	void * itp = NULL;

	snprintf(sTmp, MAX_DESCRIPTION_LEN, DEF_CFG_STORE);
	pfc_it itpfc = pfc.find(sTmp);
	if (itpfc == pfc.end()) {
		return FALSE;
	}
	itp = (void *) itpfc->second;
	/* erase and free memory */
	pfc.erase(sTmp);
	if (itp) {
		free (itp);
		itp = NULL;
	}
	return TRUE;
}

boolean_t add_app_defaults(u32 subtype)
{
	app_attribs app_data;
	char        sTmp[MAX_DESCRIPTION_LEN];
	boolean_t   result = TRUE;

	memset (&app_data, 0, sizeof(app_attribs));

	app_data.protocol.Enable = 0;
	app_data.protocol.Willing = 1;
	app_data.protocol.Advertise = 1;

	snprintf(sTmp, MAX_DESCRIPTION_LEN, DEF_CFG_STORE);

	switch (subtype) {
	case APP_FCOE_STYPE:  /* FCoE subtype */
		app_data.Length = 1;
		app_data.AppData[0] = 0x08; /* bit pattern: User Priority 3 */
		break;
	default:
		break;
	}

	/* Create app default data store for the device and app subtype. */
	if (!add_apptlv(sTmp, &app_data, subtype, NULL))
		result = FALSE;

	return result;
}

boolean_t remove_app_defaults(u32 subtype)
{
	char   sTmp[MAX_DESCRIPTION_LEN];
	void * itp = NULL;

	snprintf(sTmp, MAX_DESCRIPTION_LEN, "%s%hu",   /* Localization OK */
			 DEF_CFG_STORE, subtype);
	app_it it = apptlv.find(sTmp);

	if (it == apptlv.end()) {
		return FALSE;
	}
	itp = (void *) it->second;
	/* erase and free memory */
	apptlv.erase(sTmp);
	if (itp) {
		free (itp);
		itp = NULL;
	}
	return TRUE;
}

boolean_t add_llink_defaults(u32 subtype)
{
	llink_attribs llink_data;
	char          sTmp[MAX_DESCRIPTION_LEN];
	boolean_t     result = TRUE;

	memset (&llink_data, 0, sizeof(llink_attribs));

	llink_data.protocol.Enable = 1;
	llink_data.protocol.Willing = 1;
	llink_data.protocol.Advertise = 1;
	llink_data.llink.llink_status = 0;

	snprintf(sTmp, MAX_DESCRIPTION_LEN, DEF_CFG_STORE);
	/* Create llink default data store for the device. */
	if (!add_llink(sTmp, &llink_data, subtype))
		result = FALSE;

	return result;
}

boolean_t remove_llink_defaults(u32 subtype)
{
	char   sTmp[MAX_DESCRIPTION_LEN];
	void * itp = NULL;

	snprintf(sTmp, MAX_DESCRIPTION_LEN, "%s%hu",   /* Localization OK */
			 DEF_CFG_STORE, subtype);
	llink_it itllink = llink.find(sTmp);
	if (itllink == llink.end()) {
		return FALSE;
	}
	itp = (void *) itllink->second;
	/* erase and free memory */
	llink.erase(sTmp);
	if (itp) {
		free (itp);
		itp = NULL;
	}
	return TRUE;
}

dcb_result get_pg(char *device_name,  pg_attribs *pg_data)
{
	TRY_BEGIN;
	string dn(device_name);

	dcb_result result = dcb_success;
	full_dcb_attribs attribs;

	memset(&attribs, 0, sizeof(attribs));
	if (!pg_data) return dcb_bad_params;
	pg_it it = pg.find(dn);
	if (it != pg.end()) {
		memcpy( pg_data, (*it).second, sizeof(*pg_data));
	}
	else {
		result = get_persistent(device_name, &attribs);
		if (result == dcb_success)
			memcpy( pg_data, &attribs.pg, sizeof(*pg_data));
		else
			result = dcb_device_not_found;
	}
	return result;

	CATCH_END(dcb_failed);
}

dcb_result test_device_dstore(char *device_name)
{
	TRY_BEGIN;
	string dn(device_name);

	pg_it it = pg.find(dn);
	if (it != pg.end()) {
		return dcb_success;
	} else {
		return dcb_device_not_found;
	}

	CATCH_END(dcb_failed);
}

dcb_result get_oper_pg(char *device_name,  pg_attribs *pg_data)
{
	TRY_BEGIN;
	string dn(device_name);

	dcb_result result = dcb_success;

	if (!pg_data) return dcb_bad_params;
	pg_it it = oper_pg.find(dn);
	if (it != oper_pg.end()) {
		memcpy(pg_data, (*it).second, sizeof(*pg_data));
	} else {
		result = dcb_device_not_found;
	}
	return result;

	CATCH_END(dcb_failed);
}

dcb_result get_peer_pg(char *device_name,  pg_attribs *pg_data)
{
	TRY_BEGIN;
	string dn(device_name);

	dcb_result result = dcb_success;

	if (!pg_data) return dcb_bad_params;
	pg_it it = peer_pg.find(dn);
	if (it != peer_pg.end()) {
		memcpy(pg_data, (*it).second, sizeof(*pg_data));
	} else {
		result = dcb_device_not_found;
	}

	return result;

	CATCH_END(dcb_failed);
}

void mark_pg_sent(char *device_name)
{
	TRY_BEGIN;
	string dn(device_name);

	pg_it it = pg.find(dn);
	if (it != pg.end()) {
		(*it).second->protocol.tlv_sent = TRUE;
	}
	CATCH_END_VOID;
}

void mark_pfc_sent(char *device_name)
{
	TRY_BEGIN;
	string dn(device_name);

	pfc_it it = pfc.find(dn);
	if (it != pfc.end()) {
		(*it).second->protocol.tlv_sent = TRUE;
	}
	CATCH_END_VOID;
}

void mark_app_sent(char *device_name, u32 subtype)
{
	TRY_BEGIN;
	string dn(device_name);

	char stmp[SHORT_STRING];
	snprintf(stmp, SHORT_STRING, "%hu", subtype);

	/* append device name with subtype */
	dn.append(stmp);

	app_it it = apptlv.find(dn);
	if (it != apptlv.end()) {
		(*it).second->protocol.tlv_sent = TRUE;
	}
	CATCH_END_VOID;
}

void mark_llink_sent(char *device_name, u32 subtype)
{
	TRY_BEGIN;
	string dn(device_name);

	char stmp[SHORT_STRING];
	snprintf(stmp, SHORT_STRING, "%hu", subtype);

	/* append device name with subtype */
	dn.append(stmp);

	llink_it it = llink.find(dn);
	if (it != llink.end()) {
		(*it).second->protocol.tlv_sent = TRUE;
	}
	CATCH_END_VOID;
}

dcb_result put_pg(char *device_name,  pg_attribs *pg_data)
{
	TRY_BEGIN;
	full_dcb_attribs     attribs;
	full_dcb_attrib_ptrs attr_ptr;
	string               dn(device_name);
	u32                  EventFlag = 0;
	dcb_result           result = dcb_success;
	boolean_t            bChange = FALSE;

	if (!pg_data) return dcb_bad_params;

	memset(&attribs, 0, sizeof(attribs));
	pg_it it = pg.find(dn);
	if (it != pg.end()) {
		/* Lock the data first */

		/* detect no config change */
		if (memcmp((*it).second, pg_data, sizeof(*pg_data)) == 0) {
			goto Exit;
		}

		/* Check the rules */
		memset(&attr_ptr, 0, sizeof(attr_ptr));
		attr_ptr.pg = pg_data;
		if (dcb_check_config(&attr_ptr) != dcb_success) {
			printf("Rule checking failed in put_pg()\n");
			result = dcb_bad_params;
			goto Exit;
		}
		bChange = TRUE;
		if (set_persistent(device_name, &attr_ptr) !=
			dcb_success) {
			printf("Set persistent failed put_pg()\n");
			result = dcb_device_not_found;
			goto Exit;
		}
		/* Copy the writable protocol * variables */
		feature_protocol_attribs *dStore = &((*it).second->protocol);
		if (dStore->Enable && !(pg_data->protocol.Enable))
			log_message(MSG_INFO_PG_DISABLED, "%s", device_name);
		else if (!(dStore->Enable) && pg_data->protocol.Enable)
			log_message(MSG_INFO_PG_ENABLED, "%s", device_name);
		dStore->Advertise_prev  = dStore->Advertise;
		dStore->Advertise       = pg_data->protocol.Advertise;
		dStore->Enable          = pg_data->protocol.Enable;
		dStore->Willing         = pg_data->protocol.Willing;
		dStore->tlv_sent        = FALSE;

		memcpy(&(*it).second->rx, &(pg_data->rx), sizeof(pg_data->rx));
		memcpy(&(*it).second->tx, &(pg_data->tx), sizeof(pg_data->tx));
		if ((*it).second->protocol.dcbx_st == dcbx_subtype2) {
			(*it).second->num_tcs = pg_data->num_tcs;
		}

		DCB_SET_FLAGS(EventFlag, DCB_LOCAL_CHANGE_PG);

		/* Run the protocol */
		if (bChange) {
			result = run_dcb_protocol(device_name, EventFlag);
		}
	} else {
		/* Not in DCB data store, so store in persistent storage */
		if (get_persistent(device_name, &attribs) == dcb_success) {
			memset(&attr_ptr, 0, sizeof(attr_ptr));
			attr_ptr.pg = pg_data;
			attr_ptr.pgid = &attribs.descript;
			if (set_persistent(device_name, &attr_ptr) !=
					dcb_success) {
				printf("Set persistent failed in put_pg()\n");
				result = dcb_device_not_found;
			}
		} else {
			result = dcb_device_not_found;
		}
	}
Exit:
	return result;

	CATCH_END(dcb_failed);
}

dcb_result put_peer_pg(char *device_name,  pg_attribs *peer_pg_data)
{
	/* this function relies on the caller to acquire the DCB lock */
	TRY_BEGIN;
	string dn(device_name);
	dcb_result result = dcb_success;
	feature_protocol_attribs *dStore;

	if (!peer_pg_data) return dcb_bad_params;
	pg_it peer_it = peer_pg.find(dn);
	if (peer_it == peer_pg.end()) {
		printf("could not find peer_pg data for %s\n", device_name);
		result = dcb_device_not_found;
		goto Exit;
	}

	if (peer_pg_data->protocol.dcbx_st == dcbx_subtype2) {
		rebalance_uppcts(peer_pg_data);
	}

	/* detect config change */
	if (memcmp((*peer_it).second, peer_pg_data,
		sizeof(*peer_pg_data)) == 0) {
		goto Exit;
	}

	/* Copy the writable protocol variables. */
	dStore = &((*peer_it).second->protocol);
	dStore->Advertise_prev = dStore->Advertise;
	dStore->Advertise      = peer_pg_data->protocol.Advertise;
	dStore->Enable         = peer_pg_data->protocol.Enable;
	dStore->Willing        = peer_pg_data->protocol.Willing;
	dStore->Oper_version   = peer_pg_data->protocol.Oper_version;
	dStore->Max_version    = peer_pg_data->protocol.Max_version;
	dStore->TLVPresent     = peer_pg_data->protocol.TLVPresent;
	dStore->Error          = peer_pg_data->protocol.Error;
	dStore->dcbx_st        = peer_pg_data->protocol.dcbx_st;
	dStore->Error_Flag     = peer_pg_data->protocol.Error_Flag;

	memcpy(&(*peer_it).second->rx, &(peer_pg_data->rx),
		sizeof(peer_pg_data->rx));
	memcpy(&(*peer_it).second->tx, &(peer_pg_data->tx),
		sizeof(peer_pg_data->tx));
	if ((*peer_it).second->protocol.dcbx_st == dcbx_subtype2) {
		(*peer_it).second->num_tcs = peer_pg_data->num_tcs;
	}

Exit:
	return result;

	CATCH_END(dcb_failed);
}


dcb_result get_pfc(char *device_name, pfc_attribs *pfc_data)
{
	TRY_BEGIN;
	string dn(device_name);

	dcb_result result = dcb_success;
	full_dcb_attribs attribs;

	memset(&attribs, 0, sizeof(attribs));
	if (!pfc_data)
		return dcb_bad_params;
	pfc_it it = pfc.find(dn);
	if (it != pfc.end()) {
		memcpy( pfc_data, (*it).second, sizeof(*pfc_data));
	} else {
		result = get_persistent(device_name, &attribs);
		if (result == dcb_success)
			memcpy( pfc_data, &attribs.pfc, sizeof(*pfc_data));
		else
			result = dcb_device_not_found;
	}
	return result;

	CATCH_END(dcb_failed);
}

dcb_result get_oper_pfc(char *device_name, pfc_attribs *pfc_data)
{
	TRY_BEGIN;
	string dn(device_name);

	dcb_result result = dcb_success;

	if (!pfc_data) return dcb_bad_params;
	pfc_it it = oper_pfc.find(dn);
	if (it != oper_pfc.end()) {
		memcpy( pfc_data, (*it).second, sizeof(*pfc_data));
	} else {
		result = dcb_device_not_found;
	}
	return result;

	CATCH_END(dcb_failed);
}

dcb_result get_peer_pfc(char *device_name, pfc_attribs *pfc_data)
{
	TRY_BEGIN;
	string dn(device_name);

	dcb_result result = dcb_success;

	if (!pfc_data) return dcb_bad_params;
	pfc_it it = peer_pfc.find(dn);
	if (it != peer_pfc.end()) {
		memcpy( pfc_data, (*it).second, sizeof(*pfc_data));
	} else {
		result = dcb_device_not_found;
	}
	return result;

	CATCH_END(dcb_failed);
}

dcb_result put_pfc(char *device_name, pfc_attribs *pfc_data)
{
	TRY_BEGIN;
	u32              EventFlag = 0;
	dcb_result       result = dcb_success;
	full_dcb_attribs attribs;
	boolean_t        bChange = FALSE;
	string           dn(device_name);
	full_dcb_attrib_ptrs attr_ptr;

	if (!pfc_data) return dcb_bad_params;

	memset(&attribs, 0, sizeof(attribs));
	pfc_it it = pfc.find(dn);
	if (it != pfc.end()) {

		/* detect no config change */
		if (memcmp((*it).second, pfc_data, sizeof(*pfc_data)) == 0) {
			goto Exit;
		}
		bChange = TRUE;
		memset(&attr_ptr, 0, sizeof(attr_ptr));
		attr_ptr.pfc = pfc_data;
		if (set_persistent(device_name, &attr_ptr) != dcb_success) {
			printf("Set persistent failed in put_pfc()\n");
			result = dcb_device_not_found;
			goto Exit;
		}
		feature_protocol_attribs *dStore = &((*it).second->protocol);
		if (dStore->Enable && !(pfc_data->protocol.Enable))
			log_message(MSG_INFO_PFC_DISABLED, "%s", device_name);
		else if (!(dStore->Enable) && pfc_data->protocol.Enable)
			log_message(MSG_INFO_PFC_ENABLED, "%s", device_name);
		dStore->Advertise_prev  = dStore->Advertise;
		dStore->Advertise       = pfc_data->protocol.Advertise;
		dStore->Enable          = pfc_data->protocol.Enable;
		dStore->Willing         = pfc_data->protocol.Willing;
		dStore->tlv_sent        = FALSE;

		memcpy((*it).second->admin, pfc_data->admin,
			sizeof(pfc_data->admin));
		if ((*it).second->protocol.dcbx_st == dcbx_subtype2) {
			(*it).second->num_tcs = pfc_data->num_tcs;
		}

		/* Run the protocol */
		DCB_SET_FLAGS(EventFlag, DCB_LOCAL_CHANGE_PFC);
		if (bChange) {
			result = run_dcb_protocol(device_name, EventFlag);
		}
	} else {
		/* Store in persistent storage - not in DCB data store */
		if (get_persistent(device_name, &attribs) == dcb_success) {
			memset(&attr_ptr, 0, sizeof(attr_ptr));
			attr_ptr.pfc = pfc_data;
			attr_ptr.pgid = &attribs.descript;
			if (set_persistent(device_name, &attr_ptr) !=
					dcb_success) {
				result = dcb_device_not_found;
			}
		}
		else
			result = dcb_device_not_found;
	}
Exit:
	return result;

	CATCH_END(dcb_failed);
}

dcb_result put_peer_pfc(char *device_name, pfc_attribs *peer_pfc_data)
{
	/* this function relies on the caller to acquire the DCB lock */
	TRY_BEGIN;
	string dn(device_name);
	dcb_result result = dcb_success;
	feature_protocol_attribs *dStore;

	if (!peer_pfc_data) return dcb_bad_params;
	pfc_it peer_it = peer_pfc.find(dn);
	if (peer_it == peer_pfc.end()){
		printf("putting peer_pfc data - bad device name\n");
		result = dcb_device_not_found;
		goto Exit;
	}

	/* detect config change */
	if (memcmp((*peer_it).second, peer_pfc_data,
		sizeof(*peer_pfc_data)) == 0) {
		goto Exit;
	}

	dStore = &((*peer_it).second->protocol);
	dStore->Advertise_prev  = dStore->Advertise;
	dStore->Advertise       = peer_pfc_data->protocol.Advertise;
	dStore->Enable          = peer_pfc_data->protocol.Enable;
	dStore->Willing         = peer_pfc_data->protocol.Willing;
	dStore->Oper_version    = peer_pfc_data->protocol.Oper_version;
	dStore->Max_version     = peer_pfc_data->protocol.Max_version;
	dStore->TLVPresent      = peer_pfc_data->protocol.TLVPresent;
	dStore->Error           = peer_pfc_data->protocol.Error;
	dStore->dcbx_st         = peer_pfc_data->protocol.dcbx_st;
	dStore->Error_Flag      = peer_pfc_data->protocol.Error_Flag;

	memcpy((*peer_it).second->admin, &peer_pfc_data->admin,
		sizeof(peer_pfc_data->admin));
	if ((*peer_it).second->protocol.dcbx_st == dcbx_subtype2) {
		(*peer_it).second->num_tcs = peer_pfc_data->num_tcs;
	}
Exit:
	return result;

	CATCH_END(dcb_failed);
}

dcb_result get_app(char *device_name, u32 subtype, app_attribs *app_data)
{
	TRY_BEGIN;
	string dn(device_name);

	dcb_result result = dcb_success;
	full_dcb_attribs attribs;
	char stmp[SHORT_STRING];

	memset(&attribs, 0, sizeof(attribs));
	/* append device name with subtype */
	snprintf(stmp, SHORT_STRING, "%hu", subtype);	  /* Localization OK */
	dn.append(stmp);

	app_it it = apptlv.find(dn);

	if (it != apptlv.end()) {
		memcpy(app_data, (*it).second, sizeof(*app_data));
	} else {
		result = get_persistent(device_name, &attribs);
		if (result == dcb_success) {
			memcpy( app_data, &attribs.app[subtype],
			sizeof(*app_data));
		}
		else
			result = dcb_device_not_found;
	}
	return result;

	CATCH_END(dcb_failed);
}

dcb_result get_oper_app(char *device_name, u32 subtype, app_attribs *app_data)
{
	TRY_BEGIN;
	string dn(device_name);

	dcb_result result = dcb_success;
	char stmp[SHORT_STRING];
	snprintf(stmp, SHORT_STRING, "%hu", subtype);	  /* Localization OK */

	/* append device name with subtype */
	dn.append(stmp);

	app_it it = oper_apptlv.find(dn);
	if (it != oper_apptlv.end()) {
		memcpy(app_data, (*it).second, sizeof(*app_data));
	} else {
		result = dcb_device_not_found;
	}
	return result;

	CATCH_END(dcb_failed);
}

dcb_result get_peer_app(char *device_name, u32 subtype, app_attribs *app_data)
{
	TRY_BEGIN;
	string dn(device_name);
	dcb_result result = dcb_success;
	char stmp[SHORT_STRING];
	snprintf(stmp, SHORT_STRING, "%hu", subtype);	  /* Localization OK */

	/* append device name with subtype */
	dn.append(stmp);

	app_it it = peer_apptlv.find(dn);
	if (it != peer_apptlv.end()) {
		memcpy(app_data, (*it).second, sizeof(*app_data));
	} else {
		result = dcb_device_not_found;
	}
	return result;

	CATCH_END(dcb_failed);
}

dcb_result put_app(char *device_name, u32 subtype, app_attribs *app_data)
{
	TRY_BEGIN;
	dcb_result       result = dcb_success;
	u32              EventFlag = 0;
	char             stmp[SHORT_STRING];
	full_dcb_attribs attribs;
	boolean_t        bChange = FALSE;
	string           dn(device_name);
	full_dcb_attrib_ptrs attr_ptr;

	/* append device name with subtype */
	snprintf(stmp, SHORT_STRING, "%hu", subtype); /* Localization OK */
	dn.append(stmp);

	if (!app_data) return dcb_bad_params;

	memset(&attribs, 0, sizeof(attribs));
	app_it it = apptlv.find(dn);
	if (it != apptlv.end()) {
		/* detect no config change */
		if (memcmp((*it).second, app_data,
			sizeof(*app_data)) == 0) {
			goto Exit;
		}
		/* Store in persistent storage */
		bChange = TRUE;
		memset(&attr_ptr, 0, sizeof(attr_ptr));
		attr_ptr.app = app_data;
		attr_ptr.app_subtype = (u8)subtype;
		if (set_persistent(device_name, &attr_ptr) != dcb_success) {
			printf("Set persistent failed in put_app()\n");
			return dcb_device_not_found;
		}
		feature_protocol_attribs *dStore = &((*it).second->protocol);
		if (dStore->Enable && !(app_data->protocol.Enable))
			log_message(MSG_INFO_APP_DISABLED, "%s", device_name);
		else if (!(dStore->Enable) && app_data->protocol.Enable)
			log_message(MSG_INFO_APP_ENABLED, "%s", device_name);
		dStore->Advertise_prev  = dStore->Advertise;
		dStore->Advertise       = app_data->protocol.Advertise;
		dStore->Enable          = app_data->protocol.Enable;
		dStore->Willing         = app_data->protocol.Willing;
		dStore->tlv_sent        = FALSE;

		if (app_data->Length) {
			(*it).second->Length = app_data->Length;
			memcpy(&(*it).second->AppData, &(app_data->AppData),
				app_data->Length);
		}
		DCB_SET_FLAGS(EventFlag, DCB_LOCAL_CHANGE_APPTLV);

		if (bChange) {
			result = run_dcb_protocol (device_name,
				EventFlag, subtype);
		}

	} else {
		/* Not in DCB data store, store in persistent storage */
		if (get_persistent(device_name, &attribs) == dcb_success) {
			memset(&attr_ptr, 0, sizeof(attr_ptr));
			attr_ptr.app = app_data;
			attr_ptr.app_subtype = (u8)subtype;
			attr_ptr.pgid = &attribs.descript;
			if (set_persistent(device_name, &attr_ptr) !=
				dcb_success) {
				printf("Set persistent failed in put_app()\n");
				result = dcb_device_not_found;
			}
		} else {
			result = dcb_device_not_found;
		}
	}
Exit:
	return result;

	CATCH_END(dcb_failed);
}

dcb_result put_peer_app(char *device_name, u32 subtype,
			app_attribs *peer_app_data)
{
	/* this function relies on the caller to acquire the DCB lock */
	TRY_BEGIN;
	string dn(device_name);
	dcb_result result = dcb_success;
	char stmp[SHORT_STRING];
	feature_protocol_attribs *dStore;

	/* append device name with subtype */
	snprintf(stmp, SHORT_STRING, "%hu", subtype); /* Localization OK */
	dn.append(stmp);

	if (!peer_app_data) return dcb_bad_params;
	app_it peer_it = peer_apptlv.find(dn);
	if (peer_it == peer_apptlv.end()) {
		printf("putting peer_app data - bad device name\n");
		result = dcb_device_not_found;
		goto Exit;
	}
	if (memcmp((*peer_it).second, peer_app_data,
		sizeof(*peer_app_data)) == 0) {
		goto Exit;
	}

	dStore = &((*peer_it).second->protocol);
	dStore->Advertise_prev = dStore->Advertise;
	dStore->Advertise      = peer_app_data->protocol.Advertise;
	dStore->Enable         = peer_app_data->protocol.Enable;
	dStore->Willing        = peer_app_data->protocol.Willing;
	dStore->Oper_version   = peer_app_data->protocol.Oper_version;
	dStore->Max_version    = peer_app_data->protocol.Max_version;
	dStore->TLVPresent     = peer_app_data->protocol.TLVPresent;
	dStore->Error          = peer_app_data->protocol.Error;
	dStore->dcbx_st        = peer_app_data->protocol.dcbx_st;
	dStore->Error_Flag     = peer_app_data->protocol.Error_Flag;

	(*peer_it).second->Length = peer_app_data->Length;
	memcpy(&(*peer_it).second->AppData, &(peer_app_data->AppData),
		peer_app_data->Length);

Exit:
	return result;

	CATCH_END(dcb_failed);
}

dcb_result put_llink(char *device_name, u32 subtype, llink_attribs *llink_data)
{
	TRY_BEGIN;
	u32              EventFlag = 0;
	dcb_result       result = dcb_success;
	full_dcb_attribs attribs;
	boolean_t        bChange = FALSE;
	string           dn(device_name);
	char             stmp[SHORT_STRING];
	full_dcb_attrib_ptrs attr_ptr;

	/* append device name with subtype */
	snprintf(stmp, SHORT_STRING, "%hu", subtype); /* Localization OK */
	dn.append(stmp);

	if (!llink_data) return dcb_bad_params;

	memset(&attribs, 0, sizeof(attribs));
	llink_it it = llink.find(dn);
	if (it != llink.end()) {
		/* Lock the data first */

		/* detect no config change */
		if (memcmp((*it).second, llink_data,
			sizeof(*llink_data)) == 0) {
			goto Exit;
		}
		bChange = TRUE;
		memset(&attr_ptr, 0, sizeof(attr_ptr));
		attr_ptr.llink = llink_data;
		attr_ptr.llink_subtype = LLINK_FCOE_STYPE;
		if (set_persistent(device_name, &attr_ptr) != dcb_success) {
			printf("Set persistent failed in put_llink()\n");
			result = dcb_device_not_found;
			goto Exit;
		}
		feature_protocol_attribs *dStore = &((*it).second->protocol);
		if (dStore->Enable && !(llink_data->protocol.Enable))
			log_message(MSG_INFO_LLINK_DISABLED, "%s",device_name);
		else if (!(dStore->Enable) && llink_data->protocol.Enable)
			log_message(MSG_INFO_LLINK_ENABLED, "%s", device_name);
		dStore->Advertise_prev  = dStore->Advertise;
		dStore->Advertise       = llink_data->protocol.Advertise;
		dStore->Enable          = llink_data->protocol.Enable;
		dStore->Willing         = llink_data->protocol.Willing;
		dStore->tlv_sent        = FALSE;

		(*it).second->llink.llink_status =
			llink_data->llink.llink_status;

		memcpy(&(*it).second->llink, &(llink_data->llink),
			sizeof(llink_data->llink));

		/* Run the protocol */
		DCB_SET_FLAGS(EventFlag, DCB_LOCAL_CHANGE_LLINK);
		if (bChange) {
			result = run_dcb_protocol(device_name, EventFlag);
		}
	} else {
		/* Store in persistent storage - though not in DCB data store*/
		if (get_persistent(device_name, &attribs) == dcb_success) {
			memset(&attr_ptr, 0, sizeof(attr_ptr));
			attr_ptr.llink = llink_data;
			attr_ptr.llink_subtype = LLINK_FCOE_STYPE;
			attr_ptr.pgid = &attribs.descript;
			if (set_persistent(device_name, &attr_ptr)
				!= dcb_success) {
				result = dcb_device_not_found;
			}
		}
		else
			result = dcb_device_not_found;
	}
Exit:
	return result;

	CATCH_END(dcb_failed);
}

dcb_result put_peer_llink(char *device_name, u32 subtype,
			llink_attribs *peer_llink_data)
{
	/* this function relies on the caller to acquire the DCB lock */
	TRY_BEGIN;
	string dn(device_name);
	dcb_result result = dcb_success;
	feature_protocol_attribs *dStore;
	char stmp[SHORT_STRING];

	/* append device name with subtype */
	snprintf(stmp, SHORT_STRING, "%hu", subtype); /* Localization OK */
	dn.append(stmp);

	if (!peer_llink_data) return dcb_bad_params;
	llink_it peer_it = peer_llink.find(dn);
	if (peer_it == peer_llink.end()){
		printf("putting peer_llink data - bad device name\n");
		result = dcb_device_not_found;
		goto Exit;
	}
	/* detect config change */
	if (memcmp((*peer_it).second, peer_llink_data,
		sizeof(*peer_llink_data)) == 0) {
		goto Exit;
	}

	dStore = &((*peer_it).second->protocol);
	dStore->Advertise_prev  = dStore->Advertise;
	dStore->Advertise       = peer_llink_data->protocol.Advertise;
	dStore->Enable          = peer_llink_data->protocol.Enable;
	dStore->Willing         = peer_llink_data->protocol.Willing;
	dStore->Oper_version    = peer_llink_data->protocol.Oper_version;
	dStore->Max_version     = peer_llink_data->protocol.Max_version;
	dStore->TLVPresent      = peer_llink_data->protocol.TLVPresent;
	dStore->Error           = peer_llink_data->protocol.Error;
	dStore->dcbx_st         = peer_llink_data->protocol.dcbx_st;
	dStore->Error_Flag      = peer_llink_data->protocol.Error_Flag;

	memcpy(&(*peer_it).second->llink, &(peer_llink_data->llink),
		sizeof(peer_llink_data->llink));

Exit:
	return result;

	CATCH_END(dcb_failed);
}

dcb_result get_llink(char *device_name, u32 subtype, llink_attribs *llink_data)
{
	TRY_BEGIN;
	string dn(device_name);

	dcb_result result = dcb_success;
	full_dcb_attribs attribs;
	char stmp[SHORT_STRING];

	memset(&attribs, 0, sizeof(attribs));
	/* append device name with subtype */
	snprintf(stmp, SHORT_STRING, "%hu", subtype);	  /* Localization OK */
	dn.append(stmp);

	memset(&attribs, 0, sizeof(attribs));
	if (!llink_data)
		return dcb_bad_params;
	llink_it it = llink.find(dn);
	if (it != llink.end()) {
		memcpy(llink_data, (*it).second, sizeof(*llink_data));
	} else {
		result = get_persistent(device_name, &attribs);
		if (result == dcb_success)
			memcpy(llink_data, &attribs.llink[subtype],
				sizeof(*llink_data));
		else
			result = dcb_device_not_found;
	}
	return result;

	CATCH_END(dcb_failed);
}

dcb_result get_oper_llink(char *device_name, u32 subtype,
				llink_attribs *llink_data)
{
	TRY_BEGIN;
	string dn(device_name);

	dcb_result result = dcb_success;

	char stmp[SHORT_STRING];
	snprintf(stmp, SHORT_STRING, "%hu", subtype);	  /* Localization OK */

	/* append device name with subtype */
	dn.append(stmp);

	if (!llink_data) return dcb_bad_params;
	llink_it it = oper_llink.find(dn);
	if (it != oper_llink.end()) {
		memcpy(llink_data, (*it).second, sizeof(*llink_data));
	} else {
		result = dcb_device_not_found;
	}
	return result;

	CATCH_END(dcb_failed);
}

dcb_result get_peer_llink(char *device_name, u32 subtype,
				llink_attribs *llink_data)
{
	TRY_BEGIN;
	string dn(device_name);
	dcb_result result = dcb_success;
	char stmp[SHORT_STRING];
	snprintf(stmp, SHORT_STRING, "%hu", subtype);	  /* Localization OK */

	/* append device name with subtype */
	dn.append(stmp);

	if (!llink_data) return dcb_bad_params;
	llink_it it = peer_llink.find(dn);
	if (it != peer_llink.end()) {
		memcpy( llink_data, (*it).second, sizeof(*llink_data));
	} else {
		result = dcb_device_not_found;
	}
	return result;

	CATCH_END(dcb_failed);
}

dcb_result get_control(char *device_name,
			control_protocol_attribs *control_data)
{
	TRY_BEGIN;
	string dn(device_name);
	dcb_result result = dcb_success;

	if (!control_data) return dcb_bad_params;
	control_prot_it it = dcb_control_prot.find(dn);
	if (it != dcb_control_prot.end()) {
		memcpy( control_data, (*it).second, sizeof(*control_data));
	} else {
		result = dcb_device_not_found;
	}
	return result;

	CATCH_END(dcb_failed);
}

dcb_result get_peer_control(char *device_name,
				control_protocol_attribs *peer_control_data)
{
	TRY_BEGIN;
	string dn(device_name);

	dcb_result result = dcb_success;

	if (!peer_control_data) return dcb_bad_params;
	control_prot_it it = dcb_peer_control_prot.find(dn);
	if (it != dcb_peer_control_prot.end()) {
		memcpy(peer_control_data, (*it).second,
			sizeof(*peer_control_data));
	} else {
		result = dcb_device_not_found;
	}
	return result;

	CATCH_END(dcb_failed);
}


dcb_result put_peer_control(char *device_name,
				control_protocol_attribs *peer_control_data)
{
	/* this function relies on the caller to acquire the DCB lock */
	TRY_BEGIN;
	string dn(device_name);
	dcb_result result = dcb_success;
	control_protocol_attribs *dStore;

	if (!peer_control_data) return dcb_bad_params;
	control_prot_it peer_ctrl_prot = dcb_peer_control_prot.find(dn);

	dStore = (*peer_ctrl_prot).second;
	if ((peer_control_data->Error_Flag & DUP_DCBX_TLV_CTRL) ||
		(peer_control_data->Error_Flag & TOO_MANY_NGHBRS)) {
		dStore->Error_Flag      = peer_control_data->Error_Flag;
	} else if (peer_ctrl_prot != dcb_peer_control_prot.end()) {
		dStore->SeqNo           = peer_control_data->SeqNo;
		dStore->AckNo           = peer_control_data->AckNo;
		dStore->Max_version     = peer_control_data->Max_version;
		dStore->Oper_version    = peer_control_data->Oper_version;
		dStore->RxDCBTLVState   = peer_control_data->RxDCBTLVState;
		dStore->Error_Flag      = peer_control_data->Error_Flag;
	} else {
		result = dcb_device_not_found;
	}
	return result;

	CATCH_END(dcb_failed);
}

dcb_result get_bwg_descrpt(char *device_name, u8 bwgid, char **name)
{
	TRY_BEGIN;
	string dn(device_name);
	dcb_result result = dcb_success;
	full_dcb_attribs attribs;
	int size;

	if (*name != NULL)
		free(*name);

	memset(&attribs, 0, sizeof(attribs));
	pg_desc_it it = pg_desc.find(dn);

	if ((it != pg_desc.end()) &&
		(bwgid < ((*it).second)->max_pgid_desc)) {
		size = (int)strlen(((*it).second)->pgid_desc[bwgid]) +
			sizeof(char);  /* Localization OK */
		*name = (char*)malloc(size);
		if (*name != NULL) {
			strncpy(*name, ((*it).second)->pgid_desc[bwgid],
					size); /* Localization OK */
		} else {
			goto Error;
		}
	} else {
		result = get_persistent(device_name, &attribs);
		if (result == dcb_success) {
			size = (int)strlen(
				attribs.descript.pgid_desc[bwgid]) +
				sizeof(char);
			*name = (char*)malloc(size);
			if (*name != NULL) {
				strncpy(*name,
					attribs.descript.pgid_desc[bwgid],
					size); /* Localization OK */
			} else {
				goto Error;
			}
		} else {
			result = dcb_device_not_found;
		}
	}
	return result;

Error:
	printf("get_bwg_descrpt: Failed memory alloc\n");
	return dcb_failed;

	CATCH_END(dcb_failed);
}

dcb_result put_bwg_descrpt(char *device_name, u8 bwgid, char *name)
{
	TRY_BEGIN;
	string dn(device_name);
	dcb_result result = dcb_success;
	full_dcb_attribs attribs;
	unsigned int size;
	full_dcb_attrib_ptrs attr_ptr;

	if (!name) return dcb_bad_params;
	size = (unsigned int)strlen(name);  /* Localization OK */

	memset(&attribs, 0, sizeof(attribs));
	pg_desc_it it = pg_desc.find(dn);

	if ((it != pg_desc.end()) &&
		(bwgid < ((*it).second)->max_pgid_desc)) {

		/* Only take as many characters as can be held */
		if (!(size < sizeof(((*it).second)->pgid_desc[bwgid])))
			size = sizeof(((*it).second)->pgid_desc[bwgid])-1;
		memcpy(((*it).second)->pgid_desc[bwgid], name, size);
		/* Put a null at the end incase it was truncated */
		((*it).second)->pgid_desc[bwgid][size] = '\0';
		memset(&attr_ptr, 0, sizeof(attr_ptr));
		attr_ptr.pgid = ((*it).second);
		if (set_persistent(device_name, &attr_ptr) != dcb_success) {
			return dcb_device_not_found;
		}
	} else {
		/* Store in persistent storage - though not in
		 * DCB data store */
		if (get_persistent(device_name, &attribs) == dcb_success) {
			if (!(size <
				sizeof(attribs.descript.pgid_desc[bwgid])))
				size = sizeof(
				 attribs.descript.pgid_desc[bwgid]) - 1;
			memcpy(attribs.descript.pgid_desc[bwgid],
				name, size);
			/* Put a null at the end in case it was
			 * truncated */
			attribs.descript.pgid_desc[bwgid][size] = '\0';
			memset(&attr_ptr, 0, sizeof(attr_ptr));
			attr_ptr.pgid = &attribs.descript;
			if (set_persistent(device_name, &attr_ptr)
				!= dcb_success) {
				printf("Set persistent failed "
					"in put_bwg_descrpt()\n");
				result = dcb_device_not_found;
			}
		}
		else
			result = dcb_device_not_found;
	}
	return result;

	CATCH_END(dcb_failed);
}

/******************************************************************************
**
** Method:      CopyConfigToOper
**
** Description: Function to copy local or peer PG or PFC or APPTLV
**              configurations to oper configuration.
**
** Arguments: char *device_name
**            u32 SrcFlag - Tells where to copy from (local or peer)
**            u32 EventFlag
**            u32 SubType - This is valid only for DCB_LOCAL_CHANGE_APPTLV and
**                          DCB_REMOTE_CHANGE_APPTLV
**
** Returns: TRUE if successful, failure code otherwise.
**
******************************************************************************/
void CopyConfigToOper(char *device_name, u32 SrcFlag, u32 EventFlag,
			u32 Subtype)
{
	/* this function relies on the caller to acquire the DCB lock */
	TRY_BEGIN;
	string dn(device_name);
	printf("  CopyConfigToOper %s\n", device_name);
	if (DCB_TEST_FLAGS(EventFlag, DCB_LOCAL_CHANGE_PG, DCB_LOCAL_CHANGE_PG)
		|| DCB_TEST_FLAGS(EventFlag, DCB_REMOTE_CHANGE_PG,
		DCB_REMOTE_CHANGE_PG)) {

		/* Get the Local or Peer store */
		pg_it Src;
		pg_it localSrc;
		if (SrcFlag == LOCAL_STORE) {
			Src = pg.find(dn);
			if (Src == pg.end())
				return;
		} else if (SrcFlag == PEER_STORE) {
			Src = peer_pg.find(dn);
			if (Src == peer_pg.end())
				return;

			localSrc = pg.find(dn);
			if (localSrc == pg.end())
				return;
		} else {
			/* We don't support */
			return;
		}

		/* Get the Oper store */
		pg_it Oper = oper_pg.find(dn);
		if (Oper == oper_pg.end()) {
			return;
		}
		/* Copy Src to Oper. */
		for (int i = 0; i < MAX_USER_PRIORITIES; i++) {
			(*Oper).second->tx.up[i].pgid =
				(*Src).second->tx.up[i].pgid;
			(*Oper).second->rx.up[i].pgid =
				(*Src).second->rx.up[i].pgid;

			(*Oper).second->tx.up[i].strict_priority =
				(*Src).second->tx.up[i].strict_priority;
			(*Oper).second->rx.up[i].strict_priority =
				(*Src).second->rx.up[i].strict_priority;

			(*Oper).second->tx.up[i].percent_of_pg_cap =
				(*Src).second->tx.up[i].percent_of_pg_cap;
			(*Oper).second->rx.up[i].percent_of_pg_cap =
				(*Src).second->rx.up[i].percent_of_pg_cap;

			if (SrcFlag == PEER_STORE) {
				(*Oper).second->tx.up[i].tcmap =
					(*localSrc).second->tx.up[i].tcmap;
				(*Oper).second->rx.up[i].tcmap =
					(*localSrc).second->rx.up[i].tcmap;
			} else {
				(*Oper).second->tx.up[i].tcmap =
					(*Src).second->tx.up[i].tcmap;
				(*Oper).second->rx.up[i].tcmap =
					(*Src).second->rx.up[i].tcmap;
			}
		}

		for (int i = 0; i < MAX_BANDWIDTH_GROUPS; i++) {
			(*Oper).second->tx.pg_percent[i] =
				(*Src).second->tx.pg_percent[i];
			(*Oper).second->rx.pg_percent[i] =
				(*Src).second->rx.pg_percent[i];
		}
	} else if (DCB_TEST_FLAGS(EventFlag, DCB_LOCAL_CHANGE_PFC,
		DCB_LOCAL_CHANGE_PFC) ||
		DCB_TEST_FLAGS(EventFlag, DCB_REMOTE_CHANGE_PFC,
		DCB_REMOTE_CHANGE_PFC)) {

		/* Get the Local or Peer store */
		pfc_it Src;
		if (SrcFlag == LOCAL_STORE) {
			Src = pfc.find(dn);
			if (Src == pfc.end())
				return;
		}
		else if (SrcFlag == PEER_STORE) {
			Src = peer_pfc.find(dn);
			if (Src == peer_pfc.end())
				return;
		}
		else {
			/* We don't support */
			return;
		}
		/* Get Oper store */
		pfc_it Oper = oper_pfc.find(dn);
		if (Oper == oper_pfc.end()) {
			return;
		}

		/* Copy Src to Oper. */
		memcpy( &(*Oper).second->admin, &(*Src).second->admin,
			sizeof((*Src).second->admin));
	} else if (DCB_TEST_FLAGS(EventFlag, DCB_LOCAL_CHANGE_APPTLV,
			DCB_LOCAL_CHANGE_APPTLV)||
			DCB_TEST_FLAGS(EventFlag, DCB_REMOTE_CHANGE_APPTLV,
			DCB_REMOTE_CHANGE_APPTLV)) {
		/* Get the Local or Peer store */
		/* Append subtype to the device name for APPTLV. */
		string dn1(device_name);
		char sTemp[SHORT_STRING];
		snprintf(sTemp, SHORT_STRING, "%hu", Subtype);
		/* Append device name with subtype. */
		dn1.append(sTemp);

		app_it Src;
		if (SrcFlag == LOCAL_STORE) {
			Src = apptlv.find(dn1);
			if (Src == apptlv.end())
				return;
		} else if (SrcFlag == PEER_STORE) {
			Src = peer_apptlv.find(dn1);
			if (Src == peer_apptlv.end())
				return;
		} else {
			/* We don't support */
			return;
		}
		/* Get Oper store */
		app_it Oper = oper_apptlv.find(dn1);
		if (Oper != oper_apptlv.end()) {
			/* Copy Src to Oper. */
			printf("  Changing app data from %02x to %02x\n",
				(*Oper).second->AppData[0],
				(*Src).second->AppData[0]);
			(*Oper).second->Length = (*Src).second->Length;
			memcpy((*Oper).second->AppData, (*Src).second->AppData,
				(*Src).second->Length);
		}
	} else if (DCB_TEST_FLAGS(EventFlag, DCB_LOCAL_CHANGE_LLINK,
		DCB_LOCAL_CHANGE_LLINK) ||
		DCB_TEST_FLAGS(EventFlag, DCB_REMOTE_CHANGE_LLINK,
		DCB_REMOTE_CHANGE_LLINK)) {

		/* Get the Local or Peer store */
		/* Append subtype to the device name for LLINK. */
		string dn1(device_name);
		char sTemp[SHORT_STRING];
		snprintf(sTemp, SHORT_STRING, "%hu", Subtype);
		/* Append device name with subtype. */
		dn1.append(sTemp);

		/* Get the Local or Peer store */
		llink_it Src;
		if (SrcFlag == LOCAL_STORE) {
			Src = llink.find(dn1);
			if (Src == llink.end())
				return;
		}
		else if (SrcFlag == PEER_STORE) {
			Src = peer_llink.find(dn1);
			if (Src == peer_llink.end())
				return;
		}
		else {
			/* We don't support */
			return;
		}
		/* Get Oper store */
		llink_it Oper = oper_llink.find(dn1);
		if (Oper == oper_llink.end()) {
			return;
		}

		/* Copy Src to Oper. */
		memset(&(*Oper).second->llink,0,sizeof((*Oper).second->llink));
		memcpy( &(*Oper).second->llink, &(*Src).second->llink,
			sizeof((*Src).second->llink));

	}
	CATCH_END_VOID;
}

/******************************************************************************
**
** Method:      LocalPeerCompatible
**
** Description: Function to check if local and peer configurations matches.
**
** Arguments: char *device_name
**            ULONG EventFlag
**            ULONG SubType - This is valid only for DCB_LOCAL_CHANGE_APPTLV
**			and DCB_REMOTE_CHANGE_APPTLV
**
** Returns: TRUE if successful, failure code otherwise.
**
******************************************************************************/
boolean_t LocalPeerCompatible(char *device_name, u32 EventFlag, u32 Subtype)
{
	TRY_BEGIN;
	string dn(device_name);
	if (DCB_TEST_FLAGS(EventFlag, DCB_LOCAL_CHANGE_PG, DCB_LOCAL_CHANGE_PG)
		||
		DCB_TEST_FLAGS(EventFlag, DCB_REMOTE_CHANGE_PG,
		DCB_REMOTE_CHANGE_PG)) {

		/* Get the Local and Peer PG store */
		pg_it Local = pg.find(dn);
		pg_attribs *lpg;
		pg_attribs *ppg;
		boolean_t match = FALSE;

		if (Local == pg.end()) {
			goto Error;
		}
		pg_it Peer = peer_pg.find(dn);
		if (Peer == peer_pg.end()) {
			goto Error;
		}
		lpg = (*Local).second;
		ppg = (*Peer).second;

		match = TRUE;
		if (ppg->protocol.dcbx_st == dcbx_subtype1) {
			for (int i = 0; i < MAX_USER_PRIORITIES; i++) {
				if (lpg->tx.up[i].pgid !=
					ppg->tx.up[i].pgid)
					match = FALSE;
				if (lpg->tx.up[i].strict_priority !=
					ppg->tx.up[i].strict_priority)
					match = FALSE;
				if (lpg->tx.up[i].percent_of_pg_cap !=
					ppg->tx.up[i].percent_of_pg_cap)
					match = FALSE;
			}
			for (int i = 0; i < MAX_BANDWIDTH_GROUPS; i++) {
				if (lpg->tx.pg_percent[i] !=
					ppg->tx.pg_percent[i])
					match = FALSE;
			}
		}
		if (match) {
			printf("  COMPAT PG - passed\n");
			return TRUE;
		}
		printf("  COMPAT PG - failed\n");
	} else if (DCB_TEST_FLAGS(EventFlag, DCB_LOCAL_CHANGE_PFC,
			DCB_LOCAL_CHANGE_PFC)||
			DCB_TEST_FLAGS(EventFlag, DCB_REMOTE_CHANGE_PFC,
			DCB_REMOTE_CHANGE_PFC)) {

		/* Get the Local and Peer PFC store */
		pfc_it Local = pfc.find(dn);
		if (Local == pfc.end()) {
			goto Error;
		}
		pfc_it Peer = peer_pfc.find(dn);
		if (Peer == peer_pfc.end()) {
			goto Error;
		}
		if (memcmp(&(*Local).second->admin,
			&(*Peer).second->admin,
			sizeof((*Local).second->admin)) ==
			0 ) {
			printf("  COMPAT PFC - passed\n");
			return TRUE;
		}
		printf("  COMPAT PFC - failed\n");
	} else if (DCB_TEST_FLAGS(EventFlag, DCB_LOCAL_CHANGE_APPTLV,
			DCB_LOCAL_CHANGE_APPTLV)||
			DCB_TEST_FLAGS(EventFlag, DCB_REMOTE_CHANGE_APPTLV,
			DCB_REMOTE_CHANGE_APPTLV)) {

		/* Get the Local and Peer APPTLV store */
		/* Append subtype to the device name for APPTLV. */
		string dn1(device_name);
		char sTemp[SHORT_STRING];
		snprintf(sTemp, SHORT_STRING, "%hu", Subtype);
		dn1.append(sTemp);
		app_it Local = apptlv.find(dn1);
		if (Local == apptlv.end()) {
			goto Error;
		}
		app_it Peer = peer_apptlv.find(dn1);
		if (Peer == peer_apptlv.end()) {
			goto Error;
		}
		if ((*Local).second->Length == (*Peer).second->Length) {
			if (memcmp((*Local).second->AppData,
				(*Peer).second->AppData,
				(*Local).second->Length) == 0) {
				return TRUE;
			}
		}
		printf("  COMPAT APP - failed\n");
	} else if (DCB_TEST_FLAGS(EventFlag, DCB_LOCAL_CHANGE_LLINK,
			DCB_LOCAL_CHANGE_LLINK)||
			DCB_TEST_FLAGS(EventFlag, DCB_REMOTE_CHANGE_LLINK,
			DCB_REMOTE_CHANGE_LLINK)) {

		printf("  COMPAT LLINK - failed\n");
		return FALSE;
	}

	return FALSE;
Error:
	printf("  LocalPeerCompatible: device not found\n");
	return FALSE;

	CATCH_END(FALSE);
}

/* returns: 0 on success
 *          1 on failure
*/
int set_configuration(char *device_name, u32 EventFlag)
{
	TRY_BEGIN;
	string                  dn(device_name);
	dcb_result              sResult;
	full_dcb_attrib_ptrs    attr_ptr;

	if (DCB_TEST_FLAGS(EventFlag, DCB_LOCAL_CHANGE_PG, DCB_LOCAL_CHANGE_PG)
		||
		DCB_TEST_FLAGS(EventFlag, DCB_REMOTE_CHANGE_PG,
		DCB_REMOTE_CHANGE_PG)) {

		/* Get Oper store */
		pg_it Oper = oper_pg.find(dn);
		pg_it Local = pg.find(dn);
		if (Oper == oper_pg.end() || Local == pg.end()) {
			return 1;
		}
		pgroup_attribs pg_data;
		if (DCB_TEST_FLAGS(EventFlag, DCB_REMOTE_CHANGE_PG,
			DCB_REMOTE_CHANGE_PG)) {
			memset(&attr_ptr, 0, sizeof(attr_ptr));
			attr_ptr.pg = ((*Oper).second);
			if ((sResult = dcb_check_config(&attr_ptr))
				!= dcb_success) {
				printf("  PG rule check returned error %d\n",
					sResult);  /* Localization OK */
				return sResult;
			}
		}
		memcpy(&pg_data.rx, &(*Oper).second->rx, sizeof(pg_data.rx));
		memcpy(&pg_data.tx, &(*Oper).second->tx, sizeof(pg_data.tx));
		return set_hw_pg(device_name, &pg_data,
			(*Local).second->protocol.OperMode);
	} else if (DCB_TEST_FLAGS(EventFlag, DCB_LOCAL_CHANGE_PFC,
		DCB_LOCAL_CHANGE_PFC) ||
		DCB_TEST_FLAGS(EventFlag, DCB_REMOTE_CHANGE_PFC,
		DCB_REMOTE_CHANGE_PFC)) {

		/* Get Oper store */
		pfc_it Oper = oper_pfc.find(dn);
		pfc_it Local = pfc.find(dn);
		if (Oper == oper_pfc.end() || Local == pfc.end()) {
			return 1;
		}
		return set_hw_pfc(device_name, (*Oper).second->admin,
			(*Local).second->protocol.OperMode);
	} else if (DCB_TEST_FLAGS(EventFlag, DCB_LOCAL_CHANGE_LLINK,
		DCB_LOCAL_CHANGE_LLINK) ||
		DCB_TEST_FLAGS(EventFlag, DCB_REMOTE_CHANGE_LLINK,
		DCB_REMOTE_CHANGE_LLINK)) {
		return 0;
#ifdef DCB_APP_DRV_IF_SUPPORTED
	} else if (DCB_TEST_FLAGS(EventFlag, DCB_LOCAL_CHANGE_APPTLV,
		DCB_LOCAL_CHANGE_APPTLV) ||
		DCB_TEST_FLAGS(EventFlag, DCB_REMOTE_CHANGE_APPTLV,
		DCB_REMOTE_CHANGE_APPTLV)) {
		char sTemp[SHORT_STRING];
		snprintf(sTemp, SHORT_STRING, "%hu", DEFAULT_SUBTYPE);
		dn.append(sTemp);
		/* Get Oper store */
		app_it Oper = oper_apptlv.find(dn);
		if (Oper == oper_apptlv.end()) {
			return 1;
		}

		appgroup_attribs app_data;
		app_data.dcb_app_idtype = DCB_APP_IDTYPE_ETHTYPE;
		app_data.dcb_app_id = APP_FCOE_ETHTYPE;
		app_data.dcb_app_priority = (*Oper).second->AppData[0];
		return set_hw_app(device_name, &app_data);
#endif /* DCB_APP_DRV_IF_SUPPORTED */
	}
	return 0;

	CATCH_END(1);
}

/******************************************************************************
**
** Method:      handle_opermode_true
**
** Description: This routine is called by remove_adapter.
**               For any feature whose OperMode is TRUE, send an
**               event since the port is going away - indicating an
**               OperMode change.
**
** Arguments: char *device_name
**
** Returns: dcb_success if successful, failure code otherwise.
**
******************************************************************************/
static void handle_opermode_true(char *device_name)
{
	string dn(device_name);
	pg_attribs pg_data;
	pfc_attribs pfc_data;
	app_attribs app_data;
	llink_attribs llink_data;

	if (get_pg(device_name, &pg_data) == dcb_success)
		if (pg_data.protocol.OperMode)
			pg_event(device_name, EVENT_OPERMODE);

	if (get_pfc(device_name, &pfc_data) == dcb_success)
		if (pfc_data.protocol.OperMode)
			pfc_event(device_name, EVENT_OPERMODE);

	for (int i = 0; i < DCB_MAX_APPTLV ; i++)
		if (get_app(device_name, i, &app_data) == dcb_success)
			if (app_data.protocol.OperMode)
				app_event(device_name, i, EVENT_OPERMODE);

	for (int i = 0; i < DCB_MAX_LLKTLV ; i++)
		if (get_llink(device_name, i, &llink_data) == dcb_success)
			if (llink_data.protocol.OperMode)
				llink_event(device_name, i, EVENT_OPERMODE);

}

/******************************************************************************
**
** Method:      run_feature_protocol
**
** Description: This function runs feature state machine for a local or remote
** change.
** The function caller should acquire lock before calling this function.
** Caller must call this function per event.
**
** Arguments: char *device_name
**            u32 EventFlag
**            u32 SubType - This is valid only for DCB_LOCAL_CHANGE_APPTLV and
**                          DCB_REMOTE_CHANGE_APPTLV
**
** Returns: dcb_success if successful, failure code otherwise.
**
******************************************************************************/
dcb_result run_feature_protocol(char *device_name, u32 EventFlag, u32 Subtype)
{
	feature_protocol_attribs *feat_prot      = NULL;
	feature_protocol_attribs *peer_feat_prot = NULL;
	boolean_t ErrorChanged  = FALSE;
	boolean_t Err, local_change;
	boolean_t just_added = FALSE;
	pg_attribs old_pg_opcfg;
	int old_pg_opmode = 0;
	u32 pg_events = 0;
	pfc_attribs old_pfc_opcfg;
	int old_pfc_opmode = 0;
	u32 pfc_events = 0;
	app_attribs old_app_opcfg;
	int old_app_opmode = 0;
	u32 app_events = 0;
	llink_attribs old_llink_opcfg;
	int old_llink_opmode = 0;
	u32 llink_events = 0;


	memset(&old_pg_opcfg,0,sizeof(pg_attribs));
	memset(&old_pfc_opcfg,0,sizeof(pfc_attribs));
	memset(&old_app_opcfg,0,sizeof(app_attribs));
	memset(&old_llink_opcfg,0,sizeof(llink_attribs));

	TRY_BEGIN;
	string dn(device_name);
	/* Get the protocol store */
	if (DCB_TEST_FLAGS(EventFlag, DCB_EVENT_FLAGS, DCB_LOCAL_CHANGE_PG) ||
		DCB_TEST_FLAGS(EventFlag, DCB_EVENT_FLAGS,
		DCB_REMOTE_CHANGE_PG)) {

		/* Get the local feature protocol */
		pg_it it = pg.find(dn);
		if (it != pg.end()) {
			feat_prot = (&((*it).second)->protocol);
			old_pg_opmode = feat_prot->OperMode;
		} else {
			goto ErrNoDevice;
		}

		/* Get the remote feature protocol */
		pg_it it1 = peer_pg.find(dn);
		if (it1 != peer_pg.end()) {
			peer_feat_prot =(&((*it1).second)->protocol);
		} else {
			goto ErrNoDevice;
		}
		if ((peer_feat_prot->Error_Flag & DUP_DCBX_TLV_CTRL) ||
			(peer_feat_prot->Error_Flag & DUP_DCBX_TLV_PG)) {
			printf("** FLAG: MISSING PG TLV \n");
			feat_prot->Error_Flag |= FEAT_ERR_MULTI_TLV;
		} else {
			feat_prot->Error_Flag &= ~FEAT_ERR_MULTI_TLV;
		}

		pg_it Oper = oper_pg.find(dn);
		if (Oper != oper_pg.end()) {
			memcpy(&old_pg_opcfg.rx, &(*Oper).second->rx,
				sizeof(old_pg_opcfg.rx));
			memcpy(&old_pg_opcfg.tx, &(*Oper).second->tx,
				sizeof(old_pg_opcfg.tx));
		} else {
			goto ErrNoDevice;
		}
	}

	if (DCB_TEST_FLAGS(EventFlag, DCB_EVENT_FLAGS, DCB_LOCAL_CHANGE_PFC) ||
		DCB_TEST_FLAGS(EventFlag, DCB_EVENT_FLAGS,
		DCB_REMOTE_CHANGE_PFC)) {

		/* Get the local feature protocol */
		pfc_it it = pfc.find(dn);
		if (it != pfc.end()) {
			feat_prot = (&((*it).second)->protocol);
			old_pfc_opmode = feat_prot->OperMode;
		} else {
			goto ErrNoDevice;
		}

		/* Get the remote feature protocol */
		pfc_it it1 = peer_pfc.find(dn);
		if (it1 != peer_pfc.end()) {
			peer_feat_prot = (&((*it1).second)->protocol);
		} else {
			goto ErrNoDevice;
		}
		if ((peer_feat_prot->Error_Flag & DUP_DCBX_TLV_CTRL) ||
			(peer_feat_prot->Error_Flag & DUP_DCBX_TLV_PFC)) {
			printf("** FLAG: MISSING PFC TLV \n");
			feat_prot->Error_Flag |= FEAT_ERR_MULTI_TLV;
		} else {
			feat_prot->Error_Flag &= ~FEAT_ERR_MULTI_TLV;
		}

		pfc_it Oper = oper_pfc.find(dn);
		if (Oper != oper_pfc.end()) {
			memcpy(&old_pfc_opcfg.admin, &(*Oper).second->admin,
				sizeof(old_pfc_opcfg.admin));
		} else {
			goto ErrNoDevice;
		}
	}

	if (DCB_TEST_FLAGS(EventFlag, DCB_EVENT_FLAGS, DCB_LOCAL_CHANGE_APPTLV) ||
		DCB_TEST_FLAGS(EventFlag, DCB_EVENT_FLAGS,
		DCB_REMOTE_CHANGE_APPTLV)) {

		string dn1(device_name);
		char sTemp[SHORT_STRING];
		snprintf(sTemp, SHORT_STRING, "%hu", Subtype);
		dn1.append(sTemp);
		/* Get the local feature protocol */
		app_it it = apptlv.find(dn1);
		if (it != apptlv.end()) {
			feat_prot = (&((*it).second)->protocol);
			old_app_opmode = feat_prot->OperMode;
		} else {
			goto ErrNoDevice;
		}

		/* Get the remote feature protocol */
		app_it it1 = peer_apptlv.find(dn1);
		if (it1 != peer_apptlv.end()) {
			peer_feat_prot = (&((*it1).second)->protocol);
		} else {
			goto ErrNoDevice;
		}
		if ((peer_feat_prot->Error_Flag & DUP_DCBX_TLV_CTRL) ||
			(peer_feat_prot->Error_Flag & DUP_DCBX_TLV_APP)) {
			printf("** FLAG: MISSING APP TLV \n");
			feat_prot->Error_Flag |= FEAT_ERR_MULTI_TLV;
		} else {
			feat_prot->Error_Flag &= ~FEAT_ERR_MULTI_TLV;
		}

		app_it Oper = oper_apptlv.find(dn1);
		if (Oper != oper_apptlv.end()) {
			old_app_opcfg.Length = (*Oper).second->Length;
			memcpy(&old_app_opcfg.AppData[0],
				&((*Oper).second->AppData[0]),
				old_app_opcfg.Length);
		} else {
			goto ErrNoDevice;
		}
	}
	if (DCB_TEST_FLAGS(EventFlag, DCB_EVENT_FLAGS, DCB_LOCAL_CHANGE_LLINK)
		|| DCB_TEST_FLAGS(EventFlag, DCB_EVENT_FLAGS,
		DCB_REMOTE_CHANGE_LLINK)) {

		string dn1(device_name);
		char sTemp[SHORT_STRING];
		snprintf(sTemp, SHORT_STRING, "%hu", Subtype);
		dn1.append(sTemp);

		/* Get the local feature protocol */
		llink_it it = llink.find(dn1);
		if (it != llink.end()) {
			feat_prot = (&((*it).second)->protocol);
			old_llink_opmode = feat_prot->OperMode;
		} else {
			goto ErrNoDevice;
		}

		/* Get the remote feature protocol */
		llink_it it1 = peer_llink.find(dn1);
		if (it1 != peer_llink.end()) {
			peer_feat_prot = (&((*it1).second)->protocol);
		} else {
			goto ErrNoDevice;
		}
		if ((peer_feat_prot->Error_Flag & DUP_DCBX_TLV_CTRL) ||
			(peer_feat_prot->Error_Flag & DUP_DCBX_TLV_LLINK)) {
			printf("** FLAG: MISSING LLINK TLV \n");
			feat_prot->Error_Flag |= FEAT_ERR_MULTI_TLV;
		} else {
			feat_prot->Error_Flag &= ~FEAT_ERR_MULTI_TLV;
		}

		llink_it Oper = oper_llink.find(dn1);
		if (Oper != oper_llink.end()) {
			memcpy(&old_llink_opcfg.llink, &(*Oper).second->llink,
				sizeof(old_llink_opcfg.llink));
		} else {
			goto ErrNoDevice;
		}
	}

	/* Get the local control protocol variables. */
	control_prot_it ctrl_prot = dcb_control_prot.find(dn);
	if (ctrl_prot == dcb_control_prot.end()) {
		goto ErrNoDevice;
	}
	/* Get the remote control protocol variables. */
	control_prot_it peer_ctrl_prot = dcb_peer_control_prot.find(dn);
	if (peer_ctrl_prot == dcb_peer_control_prot.end()) {
		goto ErrNoDevice;
	}
	if ((feat_prot == NULL) || (peer_feat_prot == NULL)) {
		goto ErrNoDevice;
	}
	if (peer_ctrl_prot->second->Error_Flag & TOO_MANY_NGHBRS) {
		peer_feat_prot->TLVPresent = FALSE;
		printf("** Set Flag: TOO MANY NEIGHBORS \n");
		feat_prot->Error_Flag |= FEAT_ERR_MULTI_PEER;
	} else {
		feat_prot->Error_Flag &= ~FEAT_ERR_MULTI_PEER;
	}

	if (feat_prot->State == DCB_INIT) {
		feat_prot->Oper_version = feat_prot->Max_version;
		feat_prot->OperMode = FALSE;
		feat_prot->Error = FALSE;

		/* Set the parameters. */
		feat_prot->FeatureSeqNo =
			((*ctrl_prot).second)->SeqNo + 1;
		/* If Syncd false, then control state machine will
		 * TX LLDP message with local config. */
		feat_prot->Syncd = !(feat_prot->Advertise);
		printf("Set Syncd to %u [%u]\n", feat_prot->Syncd, __LINE__);
		feat_prot->State = DCB_LISTEN;

		/* Ensure PFC settings are synced up on initialization */
		if (DCB_TEST_FLAGS(EventFlag, DCB_EVENT_FLAGS,
			DCB_LOCAL_CHANGE_PFC))
			just_added = TRUE;
	}
	if (feat_prot->State == DCB_LISTEN) {
		printf("Feature state machine (flags %x)\n", EventFlag);
		local_change = FALSE;
		if (DCB_TEST_FLAGS(EventFlag, DCB_EVENT_FLAGS,
			DCB_LOCAL_CHANGE_PG)
			|| DCB_TEST_FLAGS(EventFlag, DCB_EVENT_FLAGS,
			DCB_LOCAL_CHANGE_PFC)
			|| DCB_TEST_FLAGS(EventFlag, DCB_EVENT_FLAGS,
			DCB_LOCAL_CHANGE_APPTLV)
			|| DCB_TEST_FLAGS(EventFlag, DCB_EVENT_FLAGS,
			DCB_LOCAL_CHANGE_LLINK)
			) {

			local_change = TRUE;
			printf("  Local change:");
			printf(" %s", (EventFlag & DCB_LOCAL_CHANGE_PG) ?
				"PG" : "");
			printf(" %s", (EventFlag & DCB_LOCAL_CHANGE_PFC) ?
				"PFC" : "");
			printf(" %s", (EventFlag & DCB_LOCAL_CHANGE_APPTLV) ?
				"APP" : "");
			printf(" %s", (EventFlag & DCB_LOCAL_CHANGE_LLINK) ?
				"LLINK" : "");
			printf("\n");
		}

		/* If local changed and we are already synched... */
		if (local_change && feat_prot->Syncd) {
			printf("  Local feature already synced\n");
			/* If we are not synched, we won't be able
			 * to accept new local changes until we get
			 * back remote changes for previous local
			 * change. */

			/* Set the parameters. */
			feat_prot->FeatureSeqNo =
				((*ctrl_prot).second)->SeqNo + 1;
			/* If Syncd false, then control state machine
			 * will TX LLDP message with local config. */
			if ((feat_prot->Advertise == TRUE) ||
				(feat_prot->Advertise_prev == TRUE)) {
				feat_prot->Syncd = FALSE;
				printf("  Set Syncd to %u [%u]\n",
					feat_prot->Syncd, __LINE__);
			} else {
				feat_prot->Syncd = TRUE;
				printf("  Set Syncd to %u [%u]\n",
					feat_prot->Syncd, __LINE__);
				feat_prot->tlv_sent = TRUE;
			}
		}
		/* F4 If don't advertise, then copy the local config to
		 * Oper config. */
		if (!feat_prot->Advertise) {
			printf("  F5 - Advertise mode OFF:");
			printf(" %s", (EventFlag&(DCB_LOCAL_CHANGE_PG |
				DCB_REMOTE_CHANGE_PG)) ? "PG" : "");
			printf(" %s", (EventFlag&(DCB_LOCAL_CHANGE_PFC |
				DCB_REMOTE_CHANGE_PFC)) ? "PFC" : "");
			printf(" %s", (EventFlag&(DCB_LOCAL_CHANGE_APPTLV |
				DCB_REMOTE_CHANGE_APPTLV)) ? "APP" :"");
			printf(" %s", (EventFlag&(DCB_LOCAL_CHANGE_LLINK |
				DCB_REMOTE_CHANGE_LLINK)) ? "LLINK" : "");
			printf("\n");

			/* copy the local config to Oper config. */
			CopyConfigToOper(device_name, LOCAL_STORE,
				EventFlag, Subtype);
			/* State already in Listen so don't have to
			 * change. */
			feat_prot->Error = FALSE;

			// maintain TOO_MANY_NGHBRS & FEAT_ERR_MULTI_TLV errors
			Err = feat_prot->Error_Flag;
			feat_prot->Error_Flag = FEAT_ERR_NONE;
			if (Err & FEAT_ERR_MULTI_PEER) {
				feat_prot->Error_Flag |= FEAT_ERR_MULTI_PEER;
			}
			if (Err & FEAT_ERR_MULTI_TLV) {
				feat_prot->Error_Flag |= (Err & FEAT_ERR_MULTI_TLV);
			}

			feat_prot->OperMode = feat_prot->Enable;
			set_configuration(device_name, EventFlag);
			goto OperChange;
		}

		/* On first call from add_adapter() ensure that the HW
		 * configuration is synced with the DCBX operational state.
		*/
		if (just_added) {
			CopyConfigToOper(device_name, LOCAL_STORE,
				EventFlag, Subtype);
			set_configuration(device_name, EventFlag);
		}

		/* Process remote change. */
		if (DCB_TEST_FLAGS(EventFlag, DCB_EVENT_FLAGS,
			DCB_REMOTE_CHANGE_PG)
			|| DCB_TEST_FLAGS(EventFlag, DCB_EVENT_FLAGS,
			DCB_REMOTE_CHANGE_PFC)
			|| DCB_TEST_FLAGS(EventFlag, DCB_EVENT_FLAGS,
			DCB_REMOTE_CHANGE_APPTLV)
			|| DCB_TEST_FLAGS(EventFlag, DCB_EVENT_FLAGS,
			DCB_REMOTE_CHANGE_LLINK)
		) {

			printf("  Remote change: ");
			printf(" %s", (EventFlag & DCB_REMOTE_CHANGE_PG) ?
				"PG" : "");
			printf(" %s", (EventFlag & DCB_REMOTE_CHANGE_PFC) ?
				"PFC" : "");
			printf(" %s", (EventFlag & DCB_REMOTE_CHANGE_APPTLV) ?
				"APP" : "");
			printf(" %s", (EventFlag & DCB_REMOTE_CHANGE_LLINK) ?
				"LLINK" : "");
			printf("\n");

			/* This version check is part of the Control
			 * protocol state machine that must be
			 * complete before proceeding with feature
			 * protocol state machine */
			if (!((ctrl_prot->second->Oper_version ==
				min(peer_ctrl_prot->second->Max_version,
				ctrl_prot->second->Max_version)) &&
				(ctrl_prot->second->Oper_version ==
				peer_ctrl_prot->second->Oper_version)
				)) {
				goto ErrBadVersion;
			}

			if (feat_prot->dcbx_st == dcbx_subtype2) {
				/* Handle Peer expiration */
				if (peer_ctrl_prot->second->RxDCBTLVState ==
						DCB_PEER_EXPIRED) {
					printf("  F6.2 - Peer DCBX TLV Expired\n");
					CopyConfigToOper(device_name,
						LOCAL_STORE,EventFlag,Subtype);
					feat_prot->OperMode = FALSE;
					feat_prot->Syncd = FALSE;
					feat_prot->Error = TRUE;
					feat_prot->FeatureSeqNo = 1;
					feat_prot->Error_Flag |= FEAT_ERR_NO_TLV;
					set_configuration(device_name,
						EventFlag);
					goto OperChange;
				}
			}

			/* Handle feature TLV not present */
			if (!peer_feat_prot->TLVPresent) {
				printf("  F8 - Feature not present\n");
				/* copy the local config to Oper config. */
				CopyConfigToOper(device_name, LOCAL_STORE,
					EventFlag, Subtype);
				feat_prot->OperMode = FALSE;
				feat_prot->Syncd = TRUE;
				printf("  Set Syncd to %u [%u]\n",
					feat_prot->Syncd, __LINE__);
				feat_prot->Oper_version =
					feat_prot->Max_version;
				if (feat_prot->dcbx_st == dcbx_subtype2) {
					feat_prot->Error = TRUE;
				} else {
					feat_prot->Error = FALSE;
				}
				feat_prot->Error_Flag |= FEAT_ERR_NO_TLV;
				set_configuration(device_name, EventFlag);
				goto OperChange;
			} else {
				feat_prot->Error_Flag &= ~FEAT_ERR_NO_TLV;
			}
			if (!feat_prot->Syncd &&
				(((*peer_ctrl_prot).second)->AckNo
				!= feat_prot->FeatureSeqNo)) {

				/* Wait for the Peer to synch up. */
				printf("  Wait for Peer to synch: "
					"Peer AckNo %d, FeatureSeqNo %d \n",
					((*peer_ctrl_prot).second)->AckNo,
					feat_prot->FeatureSeqNo);
				goto OperChange;
			}

			if (feat_prot->Error_Flag & FEAT_ERR_MULTI_TLV) {
				printf("  F9.1 - Rcvd Multiple DCBX TLVs\n");
				/* Copy Local config to Oper config. */
				CopyConfigToOper(device_name, LOCAL_STORE,
					EventFlag, Subtype);
				feat_prot->OperMode = FALSE;
				Err = feat_prot->Error;
				feat_prot->Error = TRUE;
				feat_prot->force_send = TRUE;
				/* Set_configuration to driver. */
				if (set_configuration(device_name, EventFlag))
					feat_prot->Error_Flag |= FEAT_ERR_CFG;
				if (Err != feat_prot->Error) {
					ErrorChanged = TRUE;
				}
				goto ErrProt;
			}

			/* Check for the Oper version */
			if (feat_prot->Oper_version !=
				min(peer_feat_prot->Max_version,
				feat_prot->Max_version)) {

				/* Update Oper version and signal LLDP send. */
				feat_prot->Oper_version =
					min(peer_feat_prot->Max_version,
					feat_prot->Max_version);
				printf("  Update feature oper version to %d "
					"and signal send\n",
					feat_prot->Oper_version);
				feat_prot->Syncd = FALSE;
				printf("  Set Syncd to %u [%u]\n",
					feat_prot->Syncd, __LINE__);
				feat_prot->FeatureSeqNo =
					((*ctrl_prot).second)->SeqNo + 1;
				goto OperChange;
			}
			feat_prot->Syncd =  TRUE;
			printf("  Set Syncd to %u [%u]\n",
				feat_prot->Syncd, __LINE__);
				/* F13/F14 */

			if (feat_prot->Oper_version !=
				peer_feat_prot->Oper_version ) {
				/* Wait for Peer to synch up with * version */
				printf("  Wait for the Peer to synch up ");
				printf("with feature version.\n");
				goto OperChange;
			}

			feat_prot->PeerWilling = peer_feat_prot->Willing;
			/* F15 If feature is disabled on any side,
			 * then make Opermode false. */
			if (!feat_prot->Enable || !peer_feat_prot->Enable) {
				printf("  F16 - Feature is disabled\n");
				/* Copy Local config to Oper config. */
				CopyConfigToOper(device_name, LOCAL_STORE,
					EventFlag, Subtype);
				feat_prot->OperMode = FALSE;
				feat_prot->Error_Flag = FEAT_ERR_NONE;
				Err = feat_prot->Error;
				/* Set_configuration to driver. */
				if (feat_prot->dcbx_st == dcbx_subtype2) {
					feat_prot->Syncd = !(feat_prot->Error);
					feat_prot->Error = FALSE;
					if (set_configuration(device_name,
						EventFlag))
						feat_prot->Error_Flag
						|= FEAT_ERR_CFG;
				} else {
					feat_prot->Error = (set_configuration(
						device_name, EventFlag) !=
						dcb_success);
					if (feat_prot->Error)
						feat_prot->Error_Flag |=
						FEAT_ERR_CFG;
				}
				if (Err != feat_prot->Error) {
					ErrorChanged = TRUE;
				}
				goto ErrProt;
			}
			/* F17 */
			if (feat_prot->Willing && !feat_prot->PeerWilling) {
				printf("  F18 - local willing,  "
					"peer NOT willing\n");

				feat_prot->Error_Flag = FEAT_ERR_NONE;
				Err = feat_prot->Error;

				if (feat_prot->dcbx_st == dcbx_subtype2) {
					feat_prot->OperMode =
						!(peer_feat_prot->Error);
					if (feat_prot->OperMode) {
						/* Copy Peer cfg to Oper cfg */
						CopyConfigToOper(device_name,
							PEER_STORE, EventFlag,
							Subtype);
					} else {
						/* Copy local cfg to Oper cfg*/
						CopyConfigToOper(device_name,
							LOCAL_STORE, EventFlag,
							Subtype);
					}
					feat_prot->Syncd = !(feat_prot->Error);
					feat_prot->Error = FALSE;
					/* Set_configuration to driver. */
					if (set_configuration(device_name,
						EventFlag))
						feat_prot->Error_Flag |=
							FEAT_ERR_CFG;
				} else {
					feat_prot->OperMode = TRUE;
					/* Copy Peer config to Oper config. */
					CopyConfigToOper(device_name,
						PEER_STORE, EventFlag,Subtype);
					/* Set_configuration to driver. */
					feat_prot->Error = (set_configuration(
						device_name, EventFlag) !=
							dcb_success);
					if (feat_prot->Error)
						feat_prot->Error_Flag |=
							FEAT_ERR_CFG;
				}
				if (Err != feat_prot->Error) {
					ErrorChanged = TRUE;
				}
				goto ErrProt;
			}
			/* F19 */
			if (!feat_prot->Willing && feat_prot->PeerWilling) {
				printf("  F20 - local NOT willing,  "
					"peer willing\n");

				/* Copy Local config to Oper config. */
				CopyConfigToOper(device_name,
					LOCAL_STORE, EventFlag, Subtype);

				feat_prot->Error_Flag = FEAT_ERR_NONE;
				Err = feat_prot->Error;

				/* Set_configuration to driver. */
				if (feat_prot->dcbx_st == dcbx_subtype2) {
					feat_prot->OperMode =
						!peer_feat_prot->Error;
					feat_prot->Syncd = !(feat_prot->Error);
					feat_prot->Error = FALSE;
					if (set_configuration(device_name,
						EventFlag))
						feat_prot->Error_Flag |=
							FEAT_ERR_CFG;
				} else {
					feat_prot->OperMode = TRUE;
					feat_prot->Error = (set_configuration(
						device_name, EventFlag) !=
						dcb_success);
					if (feat_prot->Error)
						feat_prot->Error_Flag |=
							FEAT_ERR_CFG;
				}
				if (Err != feat_prot->Error)
					ErrorChanged = TRUE;
				goto ErrProt;

			}
			/* F21 */
			if ((feat_prot->Willing == feat_prot->PeerWilling) &&
				(LocalPeerCompatible(device_name,
				EventFlag, Subtype))) {
				printf("  F22 - local willing == peer willing\n");

				/* Copy Local config to Oper config. */
				CopyConfigToOper(device_name,
					LOCAL_STORE, EventFlag, Subtype);

				feat_prot->Error_Flag = FEAT_ERR_NONE;
				Err = feat_prot->Error;
				/* Set_configuration to driver. */

				if (feat_prot->dcbx_st == dcbx_subtype2) {
					feat_prot->OperMode =
						!peer_feat_prot->Error;
					feat_prot->Syncd = !(feat_prot->Error);
					feat_prot->Error = FALSE;
					if (set_configuration(device_name,
						EventFlag))
						feat_prot->Error_Flag |=
							FEAT_ERR_CFG;
				} else {
					feat_prot->OperMode = TRUE;
					feat_prot->Error = (set_configuration(
						device_name, EventFlag) !=
							dcb_success);
					if (feat_prot->Error)
						feat_prot->Error_Flag |=
							FEAT_ERR_CFG;
				}
				if (Err != feat_prot->Error)
					ErrorChanged = TRUE;
			} else {
				printf("  F23 - Local & Peer config not"
					" compatible\n");
				/* Copy Local config to Oper config. */
				CopyConfigToOper(device_name,
					LOCAL_STORE, EventFlag, Subtype);
				feat_prot->OperMode = FALSE;
				Err = feat_prot->Error;

				/* Set default configuration */
				if (feat_prot->dcbx_st == dcbx_subtype2) {
					feat_prot->Syncd = feat_prot->Error;
					feat_prot->Error = TRUE;
					if (set_configuration(device_name,
						EventFlag))
						feat_prot->Error_Flag |=
							FEAT_ERR_CFG;
				} else {
					feat_prot->Error = TRUE;
					if (set_configuration(device_name,
						EventFlag) != dcb_success)
						feat_prot->Error_Flag |=
							FEAT_ERR_CFG;
				}
				feat_prot->Error_Flag |= FEAT_ERR_MISMATCH;
				if (Err != feat_prot->Error)
					ErrorChanged = TRUE;
			}
ErrProt:
			if (peer_feat_prot->Error)
				feat_prot->Error_Flag |= FEAT_ERR_PEER;

			if (feat_prot->dcbx_st == dcbx_subtype1) {
				if (feat_prot->Error || peer_feat_prot->Error){
					printf("  ## FEATURE ERROR: "
						"%d, %d (Error_Flag 0x%x"
						" EventFlag 0x%x)\n",
						feat_prot->Error,
						peer_feat_prot->Error,
						feat_prot->Error_Flag,
						EventFlag);
					if (feat_prot->OperMode) {
						feat_prot->OperMode = FALSE;
						/* Set default configuration */
						set_configuration(device_name,
								EventFlag);
					}
				}
			}
			if (ErrorChanged) {
				printf("  ErrorChanged \n");
				if (feat_prot->dcbx_st == dcbx_subtype1) {
					feat_prot->Syncd = FALSE;
					printf("  Set Syncd to %u [%u]\n",
						feat_prot->Syncd, __LINE__);
				}
				feat_prot->FeatureSeqNo =
					((*ctrl_prot).second)->SeqNo+ 1;
			}
		}
OperChange:
		if (DCB_TEST_FLAGS(EventFlag, DCB_EVENT_FLAGS,
			DCB_LOCAL_CHANGE_PG) ||
			DCB_TEST_FLAGS(EventFlag, DCB_EVENT_FLAGS,
			DCB_REMOTE_CHANGE_PG)) {

			pg_it Oper = oper_pg.find(dn);
			if (Oper == oper_pg.end()) {
				goto ErrNoDevice;
			}
			if (memcmp(&(old_pg_opcfg.tx), &((*Oper).second->tx),
				sizeof(old_pg_opcfg.tx)) != 0)
				pg_events = pg_events | EVENT_OPERATTR;
			if (feat_prot->OperMode != old_pg_opmode) {
				pg_events = pg_events | EVENT_OPERMODE;
				if (feat_prot->OperMode) {
					log_message(MSG_INFO_PG_OPER, "%s",
						device_name);
				} else {
					log_message(MSG_ERR_PG_NONOPER, "%s",
						device_name);
				}
			}
		}
		if (DCB_TEST_FLAGS(EventFlag, DCB_EVENT_FLAGS,
			DCB_LOCAL_CHANGE_PFC)||
			DCB_TEST_FLAGS(EventFlag, DCB_EVENT_FLAGS,
			DCB_REMOTE_CHANGE_PFC)) {

			pfc_it Oper = oper_pfc.find(dn);
			if (Oper == oper_pfc.end()) {
				goto ErrNoDevice;
			}
			if (memcmp(&(old_pfc_opcfg.admin),
				&((*Oper).second->admin),
				sizeof(old_pfc_opcfg.admin)) != 0)
				pfc_events = pfc_events | EVENT_OPERATTR;

			if (feat_prot->OperMode != old_pfc_opmode) {
				pfc_events = pfc_events | EVENT_OPERMODE;
				if (feat_prot->OperMode) {
					log_message(MSG_INFO_PFC_OPER, "%s",
						device_name);
				} else {
					log_message(MSG_ERR_PFC_NONOPER, "%s",
						device_name);
				}
			}
		}
		if (DCB_TEST_FLAGS(EventFlag, DCB_EVENT_FLAGS,
			DCB_LOCAL_CHANGE_APPTLV) ||
			DCB_TEST_FLAGS(EventFlag, DCB_EVENT_FLAGS,
			DCB_REMOTE_CHANGE_APPTLV)) {

			string dn1(device_name);
			char sTemp[SHORT_STRING];
			/* Localization OK */
			snprintf(sTemp, SHORT_STRING, "%hu", Subtype);
			dn1.append(sTemp);

			app_it Oper = oper_apptlv.find(dn1);
			if (Oper == oper_apptlv.end()) {
				goto ErrNoDevice;
			}
			if ((old_app_opcfg.Length != (*Oper).second->Length)
				|| (old_app_opcfg.Length &&
				(memcmp(old_app_opcfg.AppData,
				(*Oper).second->AppData,
				old_app_opcfg.Length) != 0))) {
				app_events = app_events|EVENT_OPERATTR;
			}

			if (feat_prot->OperMode != old_app_opmode) {
				app_events = app_events | EVENT_OPERMODE;
				if (feat_prot->OperMode) {
					log_message(MSG_INFO_APP_OPER, "%s",
						device_name);
				} else {
					log_message(MSG_ERR_APP_NONOPER, "%s",
						device_name);
				}
			}
		}
		if (DCB_TEST_FLAGS(EventFlag, DCB_EVENT_FLAGS,
			DCB_LOCAL_CHANGE_LLINK)||
			DCB_TEST_FLAGS(EventFlag, DCB_EVENT_FLAGS,
			DCB_REMOTE_CHANGE_LLINK)) {

			string dn1(device_name);
			char sTemp[SHORT_STRING];
			/* Localization OK */
			snprintf(sTemp, SHORT_STRING, "%hu", Subtype);
			dn1.append(sTemp);

			llink_it Oper = oper_llink.find(dn1);
			if (Oper == oper_llink.end()) {
				goto ErrNoDevice;
			}
			if (memcmp(&(old_llink_opcfg.llink),
				&((*Oper).second->llink),
				sizeof(old_llink_opcfg.llink)) != 0) {
				llink_events = llink_events | EVENT_OPERATTR;
				printf("llink opcfg changed \n");
			}

			if (feat_prot->OperMode != old_llink_opmode) {
				llink_events = llink_events | EVENT_OPERMODE;
				if (feat_prot->OperMode) {
					printf("llink opmode = TRUE\n");
					log_message(MSG_INFO_LLINK_OPER, "%s",
						device_name);
				} else {
					printf("llink opmode = FALSE\n");
					log_message(MSG_ERR_LLINK_NONOPER,"%s",
						device_name);
				}
			}
		}
	}
	CATCH_END(dcb_failed);

	if (pg_events) {
		pg_event(device_name, pg_events);
	}
	if (pfc_events) {
		pfc_event(device_name, pfc_events);
	}
	if (app_events) {
		app_event(device_name, Subtype, app_events);
	}
	if (llink_events) {
		llink_event(device_name, Subtype, llink_events);
	}
	return dcb_success;

ErrNoDevice:
	return dcb_device_not_found;

ErrBadVersion:
	printf("  Versions not compatible\n");
	return dcb_ctrl_vers_not_compatible;
}


dcb_result GetDCBTLVState(char *device_name, u8 *State)
{
	TRY_BEGIN;
	string dn(device_name);
	/* Get the remote control protocol variables. */
	control_prot_it peer_ctrl_prot = dcb_peer_control_prot.find(dn);
	if (peer_ctrl_prot == dcb_peer_control_prot.end()) {
		return dcb_device_not_found;
	}
	*State = (u8)peer_ctrl_prot->second->RxDCBTLVState;

	return dcb_success;

	CATCH_END(dcb_failed);
}

void SendLLDPMessage(char *device_name, boolean_t NewFeatureTLVs )
{
	/* Create DCB TLV and append stored feature TLVs to it. */
	/* send LLDP message. */
	/* lldp_send(char *device_name, void *tlv); */
	process_somethingChangedLocal(device_name, NewFeatureTLVs);
}

boolean_t FeaturesSynched(char *device_name)
{
	TRY_BEGIN;
	string dn(device_name);

	/* Get the local PG feature protocol */
	pg_it it = pg.find(dn);
	if (it == pg.end()) {
		return FALSE;
	}
	if ((&((*it).second)->protocol)->State == DCB_LISTEN) {
		if ((&((*it).second)->protocol)->Syncd == FALSE ||
			(&((*it).second)->protocol)->tlv_sent == FALSE)
			return FALSE;
	}

	/* Get the local PFC feature protocol */
	pfc_it it1 = pfc.find(dn);
	if (it1 == pfc.end()) {
		return FALSE;
	}
	if ((&((*it1).second)->protocol)->State == DCB_LISTEN) {
		if ((&((*it1).second)->protocol)->Syncd == FALSE ||
			(&((*it1).second)->protocol)->tlv_sent == FALSE)
			return FALSE;
	}

	/* Get the APP TLV feature protocol. */
	for (int i = 0; i < DCB_MAX_APPTLV ; i++) {
		string dn1(device_name);
		char sTemp[SHORT_STRING];
		 /* Localization OK */
		snprintf(sTemp, SHORT_STRING, "%hu", i);
		/* Append device name with subtype. */
		dn1.append(sTemp);
		app_it it2 = apptlv.find(dn1);
		if (it2 == apptlv.end()) {
			return FALSE;
		}
		if ((&((*it2).second)->protocol)->State == DCB_LISTEN) {
			if ((&((*it2).second)->protocol)->Syncd == FALSE ||
				(&((*it2).second)->protocol)->tlv_sent ==FALSE)
				return FALSE;
		}
	}

	for (int i = 0; i < DCB_MAX_LLKTLV ; i++) {
		string dn1(device_name);
		char sTemp[SHORT_STRING];
		 /* Localization OK */
		snprintf(sTemp, SHORT_STRING, "%hu", i);
		/* Append device name with subtype. */
		dn1.append(sTemp);

		/* Get the local LLINK feature protocol */
		llink_it it4 = llink.find(dn1);
		if (it4 == llink.end()) {
			return FALSE;
		}
		if ((&((*it4).second)->protocol)->State == DCB_LISTEN) {
			if ((&((*it4).second)->protocol)->Syncd == FALSE ||
				(&((*it4).second)->protocol)->tlv_sent == FALSE)
				return FALSE;
		}
	}

	return TRUE;

	CATCH_END(FALSE);
}

/* Set the Syncd value to TRUE for features which are not advertising.
*/
void update_feature_syncd(char *device_name)
{
	TRY_BEGIN;
	string dn(device_name);

	/* Get the local PG feature protocol */
	pg_it it = pg.find(dn);
	if (it != pg.end()) {
		if ((&((*it).second)->protocol)->Advertise == FALSE)
			(&((*it).second)->protocol)->Syncd = TRUE;
		if ((&((*it).second)->protocol)->force_send == TRUE)
			(&((*it).second)->protocol)->Syncd = TRUE;
	}
	/* Get the local PFC feature protocol */
	pfc_it it1 = pfc.find(dn);
	if (it1 != pfc.end()) {
		if ((&((*it1).second)->protocol)->Advertise == FALSE)
			(&((*it1).second)->protocol)->Syncd = TRUE;
		if ((&((*it1).second)->protocol)->force_send == TRUE)
			(&((*it1).second)->protocol)->Syncd = TRUE;
	}

	/* Get the APP TLV feature protocol. */
	for (int i = 0; i < DCB_MAX_APPTLV ; i++) {
		string dn1(device_name);
		char sTemp[SHORT_STRING];
		/* Localization OK */
		snprintf(sTemp, SHORT_STRING, "%hu", i);
		/* Append device name with subtype. */
		dn1.append(sTemp);
		app_it it2 = apptlv.find(dn1);
		if (it2 != apptlv.end()) {
			if ((&((*it2).second)->protocol)->Advertise == FALSE)
				(&((*it2).second)->protocol)->Syncd = TRUE;
			if ((&((*it2).second)->protocol)->force_send == TRUE)
				(&((*it2).second)->protocol)->Syncd = TRUE;
		}
	}

	for (int i = 0; i < DCB_MAX_LLKTLV ; i++) {
		/* Get the local LLINK feature protocol */
		string dn1(device_name);
		char sTemp[SHORT_STRING];
		/* Localization OK */
		snprintf(sTemp, SHORT_STRING, "%hu", i);
		/* Append device name with subtype. */
		dn1.append(sTemp);
		llink_it it4 = llink.find(dn1);
		if (it4 != llink.end()) {
			if ((&((*it4).second)->protocol)->Advertise == FALSE)
				(&((*it4).second)->protocol)->Syncd = TRUE;
			if ((&((*it4).second)->protocol)->force_send == TRUE)
				(&((*it4).second)->protocol)->Syncd = TRUE;
		}
	}

	CATCH_END_VOID;
}

/******************************************************************************
**
** Method:      run_control_protocol
**
** Description: This function runs control state machine for a local or
**		remote change.
** The function caller should acquire lock before calling this function.
** Caller must call this function for local or remote change but not both.
**
** Arguments: char *device_name
**            u32 EventFlag
** Returns: dcb_success if successful, failure code otherwise.
**
*******************************************************************************/
dcb_result run_control_protocol(char *device_name, u32 EventFlag)
{
	TRY_BEGIN;
	string dn(device_name);
	pg_attribs pg_dstore;

	/* Get the local control protocol variables. */
	control_prot_it ctrl_prot = dcb_control_prot.find(dn);
	if (ctrl_prot == dcb_control_prot.end()) {
		return dcb_device_not_found;
	}
	/* Get the remote control protocol variables. */
	control_prot_it peer_ctrl_prot = dcb_peer_control_prot.find(dn);
	if (peer_ctrl_prot == dcb_peer_control_prot.end()) {
		return dcb_device_not_found;
	}

	if (ctrl_prot->second->State == DCB_INIT) {
		/* Set the parameters. */
		ctrl_prot->second->Oper_version =
					ctrl_prot->second->Max_version;
		ctrl_prot->second->State = DCB_LISTEN;
	}
	if (ctrl_prot->second->State == DCB_LISTEN) {
		printf("DCB Ctrl in LISTEN \n");
		/* Process local change if any. */
		if (DCB_TEST_FLAGS(EventFlag, DCB_LOCAL_CHANGE_PG,
			DCB_LOCAL_CHANGE_PG)
			|| DCB_TEST_FLAGS(EventFlag, DCB_LOCAL_CHANGE_PFC,
			DCB_LOCAL_CHANGE_PFC)
			|| DCB_TEST_FLAGS(EventFlag, DCB_LOCAL_CHANGE_APPTLV,
			DCB_LOCAL_CHANGE_APPTLV)
			|| DCB_TEST_FLAGS(EventFlag, DCB_LOCAL_CHANGE_LLINK,
			DCB_LOCAL_CHANGE_LLINK)
			) {

			printf("  Local change detected: ");
			printf(" %s", (EventFlag & DCB_LOCAL_CHANGE_PG) ?
				"PG" : "");
			printf(" %s", (EventFlag & DCB_LOCAL_CHANGE_PFC) ?
				"PFC" : "");
			printf(" %s", (EventFlag & DCB_LOCAL_CHANGE_APPTLV) ?
				"APP" : "");
			printf(" %s", (EventFlag & DCB_LOCAL_CHANGE_LLINK) ?
				"LLINK" : "");
			printf("\n");

			if (ctrl_prot->second->SeqNo ==
				ctrl_prot->second->MyAckNo) {
				printf("  Local SeqNo == Local AckNo\n");
				if (!FeaturesSynched(device_name)) {
					update_feature_syncd(device_name);
					ctrl_prot->second->SeqNo++;

					printf("  *** Sending packet -- ");
					printf("SeqNo = %d \t AckNo =  %d \n",
						ctrl_prot->second->SeqNo,
						ctrl_prot->second->AckNo);

					/* Send new DCB ctrl & feature TLVs */
					SendLLDPMessage(device_name, TRUE);
				}
			}
			return dcb_success;
		}
		/* Process remote change if any. */
		if (DCB_TEST_FLAGS(EventFlag, DCB_REMOTE_CHANGE_PG,
			DCB_REMOTE_CHANGE_PG)
			|| DCB_TEST_FLAGS(EventFlag, DCB_REMOTE_CHANGE_PFC,
			DCB_REMOTE_CHANGE_PFC)
			|| DCB_TEST_FLAGS(EventFlag, DCB_REMOTE_CHANGE_APPTLV,
			DCB_REMOTE_CHANGE_APPTLV)
			|| DCB_TEST_FLAGS(EventFlag, DCB_REMOTE_CHANGE_LLINK,
			DCB_REMOTE_CHANGE_LLINK)
			) {

			boolean_t SendDCBTLV = FALSE;
			printf("  Remote change detected: ");
			printf(" %s", (EventFlag & DCB_REMOTE_CHANGE_PG) ?
				"PG" : "");
			printf(" %s", (EventFlag & DCB_REMOTE_CHANGE_PFC) ?
				"PFC" : "");
			printf(" %s", (EventFlag & DCB_REMOTE_CHANGE_APPTLV) ?
				"APP" : "");
			printf(" %s", (EventFlag & DCB_REMOTE_CHANGE_LLINK) ?
				"LLINK" : "");
			printf("\n");

			u8 State;
			if (GetDCBTLVState(device_name, &State) ==
				dcb_success) {
				if (State == DCB_PEER_EXPIRED) {
					ctrl_prot->second->SeqNo = 0;
					ctrl_prot->second->AckNo = 0;
					ctrl_prot->second->MyAckNo = 0;
					ctrl_prot->second->Oper_version =
					  ctrl_prot->second->Max_version;
					peer_ctrl_prot->second->RxDCBTLVState =
						DCB_PEER_RESET;

					printf(" Ctrl_prot Peer expired\n");
					if (get_pg(device_name, &pg_dstore) !=
						dcb_success) {
						printf("unable to get local pg"
						" cfg from data store\n");
						return dcb_device_not_found;
					}
					if (pg_dstore.protocol.dcbx_st ==
						dcbx_subtype2) {
						return dcb_success;
					} else {
						/* Send the updated DCB TLV */
						SendDCBTLV = TRUE;
						goto send;
					}
				}
			}

			if (peer_ctrl_prot->second->Error_Flag & DUP_DCBX_TLV_CTRL) {
				printf("** HANDLE: DUP CTRL TLVs \n");
				goto send;
			}

			if (ctrl_prot->second->Oper_version !=
				min(peer_ctrl_prot->second->Max_version,
				ctrl_prot->second->Max_version)) {

				ctrl_prot->second->Oper_version =
					min(peer_ctrl_prot->second->Max_version,
					ctrl_prot->second->Max_version);

				/* Send the updated DCB TLV */
				SendDCBTLV = TRUE;
				printf("  Change Oper Version \n");
				goto send;
			}

			if (ctrl_prot->second->Oper_version !=
				peer_ctrl_prot->second->Oper_version) {
				/* Wait for peer to synch up. */
				printf("  Wait for Peer to synch \n");
				goto send;
			}
			/* Update MyAck */
			ctrl_prot->second->MyAckNo =
				peer_ctrl_prot->second->AckNo;

			/* If received new Peer TLV, then acknowledge the
			 * Peer TLV
			 * MyAckNo == 0 means peer has started over, so
			 * also acknowledge in this case.
			*/
			if ((ctrl_prot->second->AckNo !=
				peer_ctrl_prot->second->SeqNo) ||
				(ctrl_prot->second->MyAckNo == 0)) {
				if (!(peer_ctrl_prot->second->Error_Flag
					& TOO_MANY_NGHBRS)) {
					ctrl_prot->second->AckNo =
						peer_ctrl_prot->second->SeqNo;
					SendDCBTLV = TRUE;
				}
			}

			/* If changes in feature then send message with latest
			 * DCB and FeatureTLV */
send:
			printf("  Current -- SeqNo = %d \t MyAckNo =  %d \n",
				ctrl_prot->second->SeqNo,
				ctrl_prot->second->MyAckNo);
			if ((ctrl_prot->second->SeqNo ==
				ctrl_prot->second->MyAckNo)  &&
				(!FeaturesSynched(device_name))) {
				printf("  Features not synced \n");
				update_feature_syncd(device_name);
				/* Send LLDP message. */
				ctrl_prot->second->SeqNo++;
				printf("  *** Sending Packet -- ");
				printf("SeqNo = %d \t AckNo =  %d \n",
					ctrl_prot->second->SeqNo,
					ctrl_prot->second->AckNo);
				/* Send new DCB control & feature TLVs*/
				SendLLDPMessage(device_name, TRUE);
				return dcb_success;
			}

			if (SendDCBTLV) {
				printf("  SendDCBTLV is set \n");
				/* if you didn't send LLDP message above then
				 * send one without changing feature TLVs. */
				printf("  *** Sending Packet -- ");
				printf("SeqNo = %d \t AckNo =  %d \n",
					ctrl_prot->second->SeqNo,
					ctrl_prot->second->AckNo);
				/* Send new DCB TLVs with old feature TLVs. */
				SendLLDPMessage(device_name, FALSE);
			}
		}
	}
	return dcb_success;

	CATCH_END(dcb_failed);
}

/******************************************************************************
**
** Method:      run_dcb_protocol
**
** Description: This function runs both feature and control state machines
**	for the features that are specified in the event flag. The function
**	caller should acquire lock per port before calling this function.
**	Caller can only club together local  PG*PFC*APPTLV or
**	remote PG*PFC*APPTLV eventflags and call this function.
**
** Arguments: char *device_name
**            u32 EventFlag
**            u32 SubType - This is valid for APPTLV event flags only.
**                          If >= DCB_MAX_APPTLV, then we process all Subtypes
**                          for APPTLV flags.
** Returns: dcb_success if successful, failure code otherwise.
**
*******************************************************************************/
dcb_result run_dcb_protocol(char *device_name, u32 EventFlag, u32 Subtype)
{
	dcb_result result = dcb_success;
	boolean_t LocalChange = FALSE;
	u32 i, SubTypeMin, SubTypeMax;

	printf("running DCB protocol for %s, flags:%04x\n", device_name,
		EventFlag);

	/* if valid use SubType param, otherwise process all SubTypes */
	if (Subtype < DCB_MAX_APPTLV) {
		SubTypeMin = Subtype;
		SubTypeMax = Subtype+1;
	}
	else {
		SubTypeMin = 0;
		SubTypeMax = DCB_MAX_APPTLV;
	}
	/* Run the feature state machines. */
	if (DCB_TEST_FLAGS(EventFlag, DCB_LOCAL_CHANGE_PG, DCB_LOCAL_CHANGE_PG)
		&& (result != dcb_ctrl_vers_not_compatible)) {
		result = run_feature_protocol(device_name,DCB_LOCAL_CHANGE_PG);
		LocalChange = TRUE;
	}
	if (DCB_TEST_FLAGS(EventFlag, DCB_LOCAL_CHANGE_PFC,
		DCB_LOCAL_CHANGE_PFC)
		&& (result != dcb_ctrl_vers_not_compatible)) {
		result = run_feature_protocol(device_name,
			DCB_LOCAL_CHANGE_PFC);
		LocalChange = TRUE;
	}
	if (DCB_TEST_FLAGS(EventFlag, DCB_LOCAL_CHANGE_APPTLV,
		DCB_LOCAL_CHANGE_APPTLV) &&
		(result != dcb_ctrl_vers_not_compatible)) {

		for (i = SubTypeMin; i < SubTypeMax; i++) {
			result = run_feature_protocol(device_name,
				DCB_LOCAL_CHANGE_APPTLV, i);
		}
		LocalChange = TRUE;
	}
	if (DCB_TEST_FLAGS(EventFlag, DCB_LOCAL_CHANGE_LLINK,
		DCB_LOCAL_CHANGE_LLINK)
		&& (result != dcb_ctrl_vers_not_compatible)) {

		for (i = SubTypeMin; i < SubTypeMax; i++) {
			result = run_feature_protocol(device_name,
				DCB_LOCAL_CHANGE_LLINK, i);
		}
		LocalChange = TRUE;
	}
	/* Only allow local or remote change at a time. */
	if (!LocalChange) {
		if (DCB_TEST_FLAGS(EventFlag, DCB_REMOTE_CHANGE_PG,
			DCB_REMOTE_CHANGE_PG)
			&& (result != dcb_ctrl_vers_not_compatible)) {
			result = run_feature_protocol(device_name,
				DCB_REMOTE_CHANGE_PG);
		}
		if (DCB_TEST_FLAGS(EventFlag, DCB_REMOTE_CHANGE_PFC,
			DCB_REMOTE_CHANGE_PFC)
			&& (result != dcb_ctrl_vers_not_compatible)) {
			result = run_feature_protocol(device_name,
				DCB_REMOTE_CHANGE_PFC);
		}
		if (DCB_TEST_FLAGS(EventFlag, DCB_REMOTE_CHANGE_APPTLV,
			DCB_REMOTE_CHANGE_APPTLV)
			&& (result != dcb_ctrl_vers_not_compatible)) {
			for (i = SubTypeMin; i < SubTypeMax; i++) {
				result = run_feature_protocol(device_name,
					DCB_REMOTE_CHANGE_APPTLV, i);
			}
		}
		if (DCB_TEST_FLAGS(EventFlag, DCB_REMOTE_CHANGE_LLINK,
			DCB_REMOTE_CHANGE_LLINK)
			&& (result != dcb_ctrl_vers_not_compatible)) {
			for (i = SubTypeMin; i < SubTypeMax; i++) {
				result = run_feature_protocol(device_name,
					DCB_REMOTE_CHANGE_LLINK, i);
			}
		}
	}

	/* apply all feature setting to the driver: linux only */
	set_hw_all(device_name);

	/* Run the control state machine. */
	result = run_control_protocol(device_name, EventFlag);

	return result;
}
