 !-----------------------------------------------------------------------------!
 !   CP2K: A general program to perform molecular dynamics simulations         !
 !   Copyright (C) 2000 - 2014  CP2K developers group                          !
 !-----------------------------------------------------------------------------!

 ! *****************************************************************************
 !> \brief Routines to calculate the minimax coefficients for approximating 1/x as 
 !>        1/x ~ 1/pi SUM_{i}^{K} w_i x^2/(x^2+a_i^2)^2  for x belonging to [1:Rc].
 !> \par History
 !>      06.2014 created [Mauro Del Ben]
 ! *****************************************************************************
 MODULE minimax_rpa
  USE kinds,                           ONLY: dp
  USE timings,                         ONLY: timeset,&
                                             timestop
#include "./common/cp_common_uses.f90" 

   IMPLICIT NONE

   PRIVATE

  CHARACTER(len=*), PARAMETER, PRIVATE :: moduleN = 'minimax_rpa'

   PUBLIC get_rpa_minimax_coeff

   CONTAINS

! *****************************************************************************
!> \brief The a_i and w_i coefficient are stored in aw such that the first 1:K 
!>        elements correspond to a_i and the K+1:2k correspond to w_i.   
!>        All a_i and w_i have been fitted with a 12th order polynomial as 
!>        a function of Rc for each value of K.
!> \param k ...
!> \param E_range ...
!> \param aw ...
!> \param ierr ...
!> \param error ...
!> \author Mauro Del Ben 
! *****************************************************************************
   SUBROUTINE get_rpa_minimax_coeff(k,E_range,aw,ierr,error)
    INTEGER                                  :: k
    REAL(KIND=dp)                            :: E_range
    REAL(KIND=dp), ALLOCATABLE, DIMENSION(:) :: aw
    INTEGER                                  :: ierr
    TYPE(cp_error_type), INTENT(inout)       :: error

    CHARACTER(LEN=*), PARAMETER :: routineN = 'get_rpa_minimax_coeff', &
      routineP = moduleN//':'//routineN

    CHARACTER(LEN=512)                       :: error_msg
    CHARACTER(LEN=64)                        :: Rc_cha
    INTEGER                                  :: handle, icoef, kkk
    LOGICAL                                  :: failure
    REAL(KIND=dp)                            :: L_b, R_transf, Rc, U_b
    REAL(KIND=dp), DIMENSION(13, 2*k)        :: fit_coef

     CALL timeset(routineN,handle)
     failure=.FALSE.
     Rc = E_range
     fit_coef = 0.0_dp
     ierr = 0

     SELECT CASE(k)
       CASE(1)
         CALL get_fit_coef_k01(Rc,L_b,U_b,fit_coef)
       CASE(2)
         CALL get_fit_coef_k02(Rc,L_b,U_b,fit_coef)
       CASE(3)
         CALL get_fit_coef_k03(Rc,L_b,U_b,fit_coef)
       CASE(4)
         CALL get_fit_coef_k04(Rc,L_b,U_b,fit_coef)
       CASE(5)
         CALL get_fit_coef_k05(Rc,L_b,U_b,fit_coef)
       CASE(6)
         CALL get_fit_coef_k06(Rc,L_b,U_b,fit_coef)
       CASE(7)
         CALL get_fit_coef_k07(Rc,L_b,U_b,fit_coef)
       CASE(8)
         CALL get_fit_coef_k08(Rc,L_b,U_b,fit_coef)
       CASE(9)
         CALL get_fit_coef_k09(Rc,L_b,U_b,fit_coef)
       CASE(10)
         CALL get_fit_coef_k10(Rc,L_b,U_b,fit_coef)
       CASE(11)
         CALL get_fit_coef_k11(Rc,L_b,U_b,fit_coef)
       CASE(12)
         CALL get_fit_coef_k12(Rc,L_b,U_b,fit_coef)
       CASE(13)
         CALL get_fit_coef_k13(Rc,L_b,U_b,fit_coef)
       CASE(14)
         CALL get_fit_coef_k14(Rc,L_b,U_b,fit_coef)
       CASE(15)
         CALL get_fit_coef_k15(Rc,L_b,U_b,fit_coef)
       CASE(16)
         CALL get_fit_coef_k16(Rc,L_b,U_b,fit_coef)
       CASE(17)
         CALL get_fit_coef_k17(Rc,L_b,U_b,fit_coef)
       CASE(18)
         CALL get_fit_coef_k18(Rc,L_b,U_b,fit_coef)
       CASE(19)
         CALL get_fit_coef_k19(Rc,L_b,U_b,fit_coef)
       CASE(20)
         CALL get_fit_coef_k20(Rc,L_b,U_b,fit_coef)
       CASE DEFAULT
         k=20
         CALL get_fit_coef_k20(Rc,L_b,U_b,fit_coef(1:13,1:40))
         ierr=-1
     END SELECT

     WRITE(Rc_cha,'(F16.2)') Rc
     Rc_cha = ADJUSTL(Rc_cha)
     IF( E_range < Rc ) THEN
       WRITE(error_msg,'(A,A)') " The range for the Minimax approximation is smaller than the lower bound "//& 
                  "for the required number of quadrature points. The Minimax parameters "//& 
                  "have been optimized for the lower bound of: ",TRIM(Rc_cha) 
       CALL cp_assert(.FALSE.,cp_warning_level,cp_assertion_failed,routineP,&
                      error_msg,error,only_ionode=.TRUE.)
     END IF
     IF( E_range > Rc ) THEN
       WRITE(error_msg,'(A,A)') " The range for the Minimax approximation exceeds the upper bound "//& 
                  "for the required number of quadrature points. The Minimax parameters "//& 
                  "have been optimized for the upper bound of: ",TRIM(Rc_cha) 
       CALL cp_assert(.FALSE.,cp_warning_level,cp_assertion_failed,routineP,&
                      error_msg,error,only_ionode=.TRUE.)
     END IF
     IF( ierr < 0 ) THEN
       CALL cp_assert(.FALSE.,cp_warning_level,cp_assertion_failed,routineP,&
            "The required number of quadrature point exceeds the maximum available. "//& 
            "The number of quadrature point has been reset to 20.", & 
            error,only_ionode=.TRUE.)
     END IF

     R_transf=SQRT(L_b/Rc)
     R_transf=SQRT(R_transf)*(SQRT(U_b/(U_b-L_b))-SQRT(Rc/(U_b-L_b)))
     DO kkk=1, 2*k
       aw(kkk)=fit_coef(1,kkk)
       DO icoef=1, 12
         aw(kkk)=aw(kkk)+fit_coef(icoef+1,kkk)*(R_transf**icoef)
       END DO
     END DO

     CALL timestop(handle)

   END SUBROUTINE get_rpa_minimax_coeff

! *****************************************************************************
!> \brief ...
!> \param Rc ...
!> \param L_b ...
!> \param U_b ...
!> \param fit_coef ...
! *****************************************************************************
   SUBROUTINE get_fit_coef_k01(Rc,L_b,U_b,fit_coef)
    REAL(KIND=dp)                            :: Rc, L_b, U_b
    REAL(KIND=dp), DIMENSION(13, 2)          :: fit_coef

    REAL(KIND=dp), DIMENSION(52), PARAMETER :: c01 = (/ &
      8.4569134345088148E-01_dp, -2.3864746901255809E-01_dp,&
      -1.1361819501552294E-01_dp,  6.7690505100738471E-02_dp,&
      5.3985186361341728E-03_dp, -1.9612317325166117E-02_dp,&
      7.3513074383591715E-03_dp,  1.9996243815975012E-03_dp,&
      -3.1205386442664557E-03_dp,  6.2573451848435199E-04_dp,&
      1.0159339561927395E-03_dp, -2.0336924331972007E-04_dp,&
      -8.0157986968086049E-04_dp,  8.7743290162972425E+00_dp,&
      -3.4658285658553871E+00_dp, -1.0159575820898459E+00_dp,&
      1.5316366676023252E+00_dp, -2.2374031893904067E-01_dp,&
      -4.9079952564718116E-01_dp,  2.8542459603969011E-01_dp,&
      4.0445340834152646E-02_dp, -1.1213157710848982E-01_dp,&
      2.5807944540072045E-02_dp,  3.6871815925478856E-02_dp,&
      -7.5013131351734158E-03_dp, -2.8645535700322102E-02_dp,&
      9.3922532274951809E-01_dp, -2.1747417837053222E-07_dp,&
      -4.6907902157368980E-01_dp, -2.1616033822571426E-01_dp,&
      4.1319893871687463E-01_dp,  2.9123071569339970E-01_dp,&
      -4.4281181896713878E-01_dp, -5.5512445336312510E-01_dp,&
      1.1432764546204668E+00_dp, -3.4020236095345024E-01_dp,&
      -4.4151758350979708E-01_dp,  1.8627895375666054E-01_dp,&
      1.0229706519679303E-01_dp,  1.0233271450911223E+01_dp,&
      -3.6844989012010113E-06_dp, -7.8361514387479545E+00_dp,&
      -3.6112788486477720E+00_dp,  9.5851351388967405E+00_dp,&
      7.2698001012821694E+00_dp, -1.1403856909523945E+01_dp,&
      -1.7651082087203267E+01_dp,  3.2669706643838275E+01_dp,&
      -5.4176678145626020E+00_dp, -1.4771604007512861E+01_dp,&
      1.1600808336065933E+00_dp,  6.4627594951385223E+00_dp /)
    REAL(KIND=dp), DIMENSION(13, 2, 2), PARAMETER :: &
      coefdata = RESHAPE ((/ c01 /),(/13,2,2/))

    INTEGER                                  :: irange

     IF(Rc <=  2.0000000000000000E+00_dp) Rc =   2.0000000000000000E+00_dp
     IF(Rc >=  5.8879280090332031E+00_dp) Rc =   5.8879280090332031E+00_dp
     IF(Rc >=  2.0000000000000000E+00_dp .AND. Rc <=  2.6646270991316325E+00_dp) THEN
       irange =  1
       L_b =  2.0000000000000000E+00_dp
       U_b =  2.6646270991316325E+00_dp
     END IF
     IF(Rc >=  2.6646270991316325E+00_dp .AND. Rc <=  5.8879280090332031E+00_dp) THEN
       irange =  2
       L_b =  2.6646270991316325E+00_dp
       U_b =  5.8879280090332031E+00_dp
     END IF

     fit_coef(1:13,1:2) = coefdata(1:13,1:2,irange)

   END SUBROUTINE get_fit_coef_k01

! *****************************************************************************
!> \brief ...
!> \param Rc ...
!> \param L_b ...
!> \param U_b ...
!> \param fit_coef ...
! *****************************************************************************
   SUBROUTINE get_fit_coef_k02(Rc,L_b,U_b,fit_coef)
    REAL(KIND=dp)                            :: Rc, L_b, U_b
    REAL(KIND=dp), DIMENSION(13, 4)          :: fit_coef

    REAL(KIND=dp), DIMENSION(156), PARAMETER :: c01 = (/ &
      4.9649110596357299E-01_dp, -1.8397680763708085E-01_dp,&
      -2.6764777836094898E-02_dp,  1.4685597048762128E-02_dp,&
      3.8551475599549168E-03_dp, -6.6371384965829621E-04_dp,&
      -8.2857349546675376E-04_dp,  3.7111617653054724E-05_dp,&
      1.2814660801267966E-04_dp,  6.1538340479780230E-06_dp,&
      -2.4584409073123429E-05_dp, -5.0991779586331699E-06_dp,&
      1.0669432321911992E-05_dp,  2.3206994094733933E+00_dp,&
      -1.3514563041382210E+00_dp,  3.1921448263800534E-01_dp,&
      3.5958872361872854E-02_dp, -5.3831249908297907E-02_dp,&
      1.7289732546709874E-02_dp, -2.4440065554605303E-03_dp,&
      -1.2724078521801746E-05_dp,  2.2546039620310139E-04_dp,&
      -9.7860415748580936E-05_dp, -6.5067080496765577E-05_dp,&
      2.6089206215138224E-05_dp,  5.2757851532030513E-05_dp,&
      4.3667474519629623E+00_dp, -1.8617368732105988E+00_dp,&
      -7.4376289802548262E-02_dp,  1.8267819450392606E-01_dp,&
      -3.2026231603982562E-03_dp, -1.8945310128742759E-02_dp,&
      -3.2473576905305453E-03_dp,  4.1419261548186481E-03_dp,&
      8.8569143253757507E-04_dp, -6.7268426110531814E-04_dp,&
      -3.8199070267735632E-04_dp,  6.2942305485704747E-05_dp,&
      2.1244794526782270E-04_dp,  1.3299400044019976E+01_dp,&
      -1.0102857731754517E+01_dp,  4.5167812366146727E+00_dp,&
      -7.7404161077659717E-01_dp, -3.4951153894568415E-01_dp,&
      3.1655455535081256E-01_dp, -1.2883665080828036E-01_dp,&
      2.4210630196288028E-02_dp,  6.5829796220587648E-03_dp,&
      -4.6936987882240733E-03_dp, -8.5247269973254237E-04_dp,&
      1.0535642322120099E-03_dp,  4.2267254389369188E-04_dp,&
      6.5760438758196227E-01_dp, -1.7918847011385985E-01_dp,&
      -2.2673006578030033E-01_dp,  1.0051289981916312E-01_dp,&
      -3.0133666064475630E-02_dp, -4.8262232970595134E-03_dp,&
      1.1378620206724364E-01_dp, -1.8105019665171873E-01_dp,&
      1.9273690986546238E-01_dp, -1.6962737938522135E-01_dp,&
      9.8332759521266175E-02_dp, -2.6789710748995204E-02_dp,&
      1.0387096874176875E-03_dp,  4.1246654556789606E+00_dp,&
      -3.0716392263747014E+00_dp, -2.0785962463279271E+00_dp,&
      5.2664445183974742E+00_dp, -3.0074972977105965E+00_dp,&
      -2.7503179147598340E+00_dp,  7.6392314052684709E+00_dp,&
      -8.4073849148263342E+00_dp,  4.7943508114021212E+00_dp,&
      2.6737517312394893E-01_dp, -2.9722441934551838E+00_dp,&
      2.3158844939360632E+00_dp, -6.5764037707338063E-01_dp,&
      6.1608320892349369E+00_dp, -2.1898229650048577E+00_dp,&
      -2.5422729209340225E+00_dp,  1.7935819542593845E+00_dp,&
      -3.0756618643446343E-01_dp, -3.4608361155905393E-01_dp,&
      1.6002204372920785E+00_dp, -2.6003746540622035E+00_dp,&
      2.6278080511803972E+00_dp, -2.2341338029312703E+00_dp,&
      1.4684729639545540E+00_dp, -5.4384672282947255E-01_dp,&
      7.0214867320780153E-02_dp,  3.0690007708797335E+01_dp,&
      -3.6434557745981337E+01_dp, -1.2802331409904522E+01_dp,&
      7.8237428089364300E+01_dp, -7.1628441704020958E+01_dp,&
      -3.6767672386188337E+01_dp,  1.6598068739496907E+02_dp,&
      -1.9500321193457228E+02_dp,  8.7558312074511292E+01_dp,&
      6.3428744200583431E+01_dp, -1.2995121398289083E+02_dp,&
      8.7747827655552456E+01_dp, -2.3421029630375354E+01_dp,&
      6.8832717879511052E-01_dp,  8.9994611434700178E-08_dp,&
      -1.7447802413631236E-01_dp,  1.1128373934782229E-02_dp,&
      1.0288810881396471E-01_dp, -9.6952440957114305E-02_dp,&
      -6.4905803226054201E-02_dp,  2.4883468266829842E-01_dp,&
      -2.3908071572971198E-01_dp,  4.2672150253429388E-02_dp,&
      8.4512682635884184E-02_dp, -3.6754750174471884E-02_dp,&
      -1.2870785796384022E-02_dp,  4.7101965082399921E+00_dp,&
      1.9264636868860040E-06_dp, -3.6955911120984810E+00_dp,&
      2.3579843361005315E-01_dp,  4.5049726325907349E+00_dp,&
      -2.3256735181630925E+00_dp, -5.7088231586772453E+00_dp,&
      9.2251939039827988E+00_dp, -2.1118379768244511E+00_dp,&
      -6.1276676279296476E+00_dp,  2.9985533393084327E+00_dp,&
      4.9045918358347871E+00_dp, -4.1266597706732586E+00_dp,&
      6.5430742560184436E+00_dp,  1.1412075979871461E-06_dp,&
      -2.2097196426847456E+00_dp,  1.4094456985563819E-01_dp,&
      1.5262198438277494E+00_dp, -1.2544592957128169E+00_dp,&
      -1.1128799568053420E+00_dp,  3.4976285641773011E+00_dp,&
      -3.0348111780709122E+00_dp,  2.3892287319584160E-01_dp,&
      1.1788607699039138E+00_dp, -2.2391998953368625E-01_dp,&
      -3.5101044491985517E-01_dp,  3.8067685920148641E+01_dp,&
      2.5902565425035982E-05_dp, -4.9301857791232557E+01_dp,&
      3.1465928358831281E+00_dp,  7.7404606802819202E+01_dp,&
      -3.3013072565887242E+01_dp, -1.2019494685884978E+02_dp,&
      1.5486223363142261E+02_dp,  3.2337501402758228E+01_dp,&
      -1.8488455295931200E+02_dp,  5.0234562973130487E+01_dp,&
      1.4928893567718609E+02_dp, -1.0852276420140453E+02_dp /)
    REAL(KIND=dp), DIMENSION(13, 4, 3), PARAMETER :: &
      coefdata = RESHAPE ((/ c01 /),(/13,4,3/))

    INTEGER                                  :: irange

     IF(Rc <=  2.0000000000000000E+00_dp) Rc =   2.0000000000000000E+00_dp
     IF(Rc >=  2.5266985893249512E+01_dp) Rc =   2.5266985893249512E+01_dp
     IF(Rc >=  2.0000000000000000E+00_dp .AND. Rc <=  2.8845195225564151E+00_dp) THEN
       irange =  1
       L_b =  2.0000000000000000E+00_dp
       U_b =  2.8845195225564151E+00_dp
     END IF
     IF(Rc >=  2.8845195225564151E+00_dp .AND. Rc <=  1.1473793111539274E+01_dp) THEN
       irange =  2
       L_b =  2.8845195225564151E+00_dp
       U_b =  1.1473793111539274E+01_dp
     END IF
     IF(Rc >=  1.1473793111539274E+01_dp .AND. Rc <=  2.5266985893249512E+01_dp) THEN
       irange =  3
       L_b =  1.1473793111539274E+01_dp
       U_b =  2.5266985893249512E+01_dp
     END IF

     fit_coef(1:13,1:4) = coefdata(1:13,1:4,irange)

   END SUBROUTINE get_fit_coef_k02

! *****************************************************************************
!> \brief ...
!> \param Rc ...
!> \param L_b ...
!> \param U_b ...
!> \param fit_coef ...
! *****************************************************************************
   SUBROUTINE get_fit_coef_k03(Rc,L_b,U_b,fit_coef)
    REAL(KIND=dp)                            :: Rc, L_b, U_b
    REAL(KIND=dp), DIMENSION(13, 6)          :: fit_coef

    REAL(KIND=dp), DIMENSION(234), PARAMETER :: c01 = (/ &
      4.7223379816403283E-01_dp, -2.2042504935487547E-01_dp,&
      -1.0700855642800766E-01_dp,  2.9282041797865813E-02_dp,&
      3.3026836310057026E-02_dp,  1.1386996102844459E-02_dp,&
      -1.2818345627763791E-02_dp, -4.0229827105483680E-03_dp,&
      5.2714678853526230E-03_dp, -5.7896018199156346E-03_dp,&
      2.6848950232278156E-03_dp,  1.9016456475500060E-03_dp,&
      -1.3675117740777492E-03_dp,  2.0032594434499789E+00_dp,&
      -1.6049856670510629E+00_dp, -7.5084218460525085E-02_dp,&
      6.0247881175491358E-01_dp, -1.0588835881424769E-01_dp,&
      -1.5334719856063742E-01_dp, -3.6117192085523851E-02_dp,&
      1.1475935854850117E-01_dp,  4.7173969318522357E-02_dp,&
      -1.5024338659366171E-01_dp,  9.3080665778357816E-02_dp,&
      -1.7678496124707836E-02_dp, -1.7574693551064798E-03_dp,&
      6.9852771279712096E+00_dp, -9.8678652716743986E+00_dp,&
      5.8034120673487637E+00_dp,  8.4032860298957390E-01_dp,&
      -3.9344819362623489E+00_dp,  2.9352624815927943E+00_dp,&
      -1.0295714123401773E+00_dp,  6.4840906026874964E-02_dp,&
      2.3541965450457605E-01_dp, -3.8563728580228640E-01_dp,&
      3.7881278851716366E-01_dp, -2.0274209183322051E-01_dp,&
      4.5981510089675416E-02_dp,  4.1012174536431987E+00_dp,&
      -2.2240356017273251E+00_dp, -8.3949352914134823E-01_dp,&
      4.9927067503647976E-01_dp,  2.8387636248232850E-01_dp,&
      4.7904669834267075E-04_dp, -1.7161769150318792E-01_dp,&
      -7.6990836877090770E-03_dp,  9.7592444267693049E-02_dp,&
      -9.9847132214505516E-02_dp,  9.1265071496279340E-02_dp,&
      -4.8072775002788476E-02_dp,  9.2276548198524724E-03_dp,&
      9.5513399432547033E+00_dp, -1.1675840721046807E+01_dp,&
      3.0385953635634566E+00_dp,  5.1268986462936210E+00_dp,&
      -3.3964946209348490E+00_dp, -1.3155690145638068E+00_dp,&
      9.3593509658323282E-01_dp,  1.1892612191185217E+00_dp,&
      -1.0436920428437203E-01_dp, -2.2056329972056523E+00_dp,&
      2.3677798945356323E+00_dp, -1.0414299374786686E+00_dp,&
      1.7342693495440725E-01_dp,  3.9609801714084597E+01_dp,&
      -7.3889749869401939E+01_dp,  6.9592887675897757E+01_dp,&
      -2.4811522219671406E+01_dp, -2.2841578322462372E+01_dp,&
      4.4418611403245741E+01_dp, -3.7852497542173246E+01_dp,&
      1.6302785218464713E+01_dp,  4.4126861094794840E+00_dp,&
      -1.3635441404324794E+01_dp,  1.1138794870115078E+01_dp,&
      -4.7156831629367568E+00_dp,  8.7213397592194231E-01_dp,&
      5.5875221039160261E-01_dp, -8.9468055007220743E-02_dp,&
      -1.2065995777914583E-01_dp,  4.8471007034375492E-02_dp,&
      -3.8301904720482650E-02_dp,  2.4275678600046053E-02_dp,&
      2.8136983363421911E-02_dp, -8.4794476923930245E-02_dp,&
      1.2805871374430119E-01_dp, -1.3799846162209611E-01_dp,&
      1.0242711526473426E-01_dp, -4.6555343095993788E-02_dp,&
      9.7559415471718224E-03_dp,  2.7624820993735941E+00_dp,&
      -9.4404189731579302E-01_dp, -1.0778440090092487E+00_dp,&
      1.0000508959158847E+00_dp, -4.0720645947670364E-01_dp,&
      3.2143562912951618E-02_dp,  5.5974630350026500E-01_dp,&
      -1.3059019163765901E+00_dp,  1.7808779108085686E+00_dp,&
      -1.7587309793812396E+00_dp,  1.2489411338126502E+00_dp,&
      -5.6093609251530230E-01_dp,  1.1760304859958214E-01_dp,&
      1.3544129923931036E+01_dp, -1.1316468354856699E+01_dp,&
      -7.6541274135369042E+00_dp,  2.1813286094573890E+01_dp,&
      -1.5817891923183874E+01_dp, -5.7933843562996516E+00_dp,&
      2.9001150775153093E+01_dp, -4.1936816889961740E+01_dp,&
      3.9707552859267892E+01_dp, -2.6000900774396619E+01_dp,&
      1.0755546253602940E+01_dp, -2.0944265775352795E+00_dp,&
      -1.5875486677710954E-02_dp,  5.0129963552083687E+00_dp,&
      -9.8497955022805650E-01_dp, -1.2826931706396834E+00_dp,&
      6.5452164227503740E-01_dp, -3.9738951291723318E-01_dp,&
      2.3136357394360504E-01_dp,  3.5477656525902451E-01_dp,&
      -1.0231998800842284E+00_dp,  1.5032961384142487E+00_dp,&
      -1.5914516297167578E+00_dp,  1.1742544052689099E+00_dp,&
      -5.3334314176404851E-01_dp,  1.1180896150674231E-01_dp,&
      1.5835060026431352E+01_dp, -8.7111880093588692E+00_dp,&
      -8.8826285808589898E+00_dp,  1.1712175066481947E+01_dp,&
      -4.4172083122828187E+00_dp, -1.2587072349970747E+00_dp,&
      7.1497450368592714E+00_dp, -1.4524138295210200E+01_dp,&
      1.8912025585426967E+01_dp, -1.7743638731961465E+01_dp,&
      1.2115533439508143E+01_dp, -5.3562991919747693E+00_dp,&
      1.1231498116960723E+00_dp,  1.0046853530042941E+02_dp,&
      -1.2562130523877248E+02_dp, -4.8899698987716903E+01_dp,&
      2.9033084145860795E+02_dp, -2.9969216846999444E+02_dp,&
      -5.7160567223254283E+01_dp,  5.7249435821985480E+02_dp,&
      -8.6440457177677740E+02_dp,  7.2603165823538518E+02_dp,&
      -3.1482726992615238E+02_dp, -1.7030029333405142E+01_dp,&
      9.2288380720163971E+01_dp, -3.3395722174722863E+01_dp,&
      5.7310488938023063E-01_dp,  4.7901259633936180E-08_dp,&
      -9.1651209265819295E-02_dp,  2.7732065459135504E-02_dp,&
      2.8645481150354719E-02_dp, -6.7359425012262830E-02_dp,&
      3.8584259620296449E-02_dp,  6.3446129710153454E-02_dp,&
      -1.5358151939100778E-01_dp,  1.2187793055913763E-01_dp,&
      1.5086978285231711E-02_dp, -9.6662929399273831E-02_dp,&
      5.2098208706298513E-02_dp,  2.9191177943658442E+00_dp,&
      5.4133537461405717E-07_dp, -1.0344448907244952E+00_dp,&
      3.1300849735874009E-01_dp,  5.4891616747359162E-01_dp,&
      -8.9562127361783095E-01_dp,  2.5683436290728645E-01_dp,&
      1.1981010184653673E+00_dp, -2.2140209166896678E+00_dp,&
      1.5279502299230125E+00_dp,  3.0429149987602905E-01_dp,&
      -1.2173426926379263E+00_dp,  6.0860526435749707E-01_dp,&
      1.5577506560483336E+01_dp,  7.6602384911749927E-06_dp,&
      -1.4533497939231582E+01_dp,  4.3979051512530276E+00_dp,&
      1.5541107768555671E+01_dp, -1.7236024655372692E+01_dp,&
      -8.5725482253998937E+00_dp,  4.0651917108261443E+01_dp,&
      -4.5625286541426540E+01_dp,  1.7082545781556608E+01_dp,&
      1.1810832859818145E+01_dp, -1.3783312630992485E+01_dp,&
      3.3847888256586232E+00_dp,  5.1721946579885314E+00_dp,&
      5.3542232317209938E-07_dp, -1.0242183084092691E+00_dp,&
      3.0991127039090627E-01_dp,  3.6925347779677709E-01_dp,&
      -7.8228043802145897E-01_dp,  4.0037631487810599E-01_dp,&
      8.0471691788497290E-01_dp, -1.8201969969209009E+00_dp,&
      1.4022833800509154E+00_dp,  1.9468516183415735E-01_dp,&
      -1.1133025066398865E+00_dp,  5.9138813613689389E-01_dp,&
      1.7309446051118943E+01_dp,  5.1997518715207899E-06_dp,&
      -9.9271747598821172E+00_dp,  3.0038475175646426E+00_dp,&
      6.5128609543997671E+00_dp, -9.3405057766655286E+00_dp,&
      1.2450381441251936E+00_dp,  1.4427779610011438E+01_dp,&
      -2.3898845190857141E+01_dp,  1.5265437336149422E+01_dp,&
      3.7526247706518707E+00_dp, -1.2192107238730179E+01_dp,&
      5.8009575841189420E+00_dp,  1.2420803433458813E+02_dp,&
      9.4529083484548541E-05_dp, -1.7804086701584447E+02_dp,&
      5.3879411878296139E+01_dp,  2.5019726837316304E+02_dp,&
      -2.4632876610675547E+02_dp, -2.3728096521661567E+02_dp,&
      7.2832435058031899E+02_dp, -6.3406381009820018E+02_dp,&
      7.2166814993792187E+01_dp,  2.2865645970566064E+02_dp,&
      -5.9497992164644941E+01_dp, -5.9908598748248629E+01_dp /)
    REAL(KIND=dp), DIMENSION(13, 6, 3), PARAMETER :: &
      coefdata = RESHAPE ((/ c01 /),(/13,6,3/))

    INTEGER                                  :: irange

     IF(Rc <=  2.0000000000000000E+00_dp) Rc =   2.0000000000000000E+00_dp
     IF(Rc >=  8.3026037216186523E+01_dp) Rc =   8.3026037216186523E+01_dp
     IF(Rc >=  2.0000000000000000E+00_dp .AND. Rc <=  9.5121852168680867E+00_dp) THEN
       irange =  1
       L_b =  2.0000000000000000E+00_dp
       U_b =  9.5121852168680867E+00_dp
     END IF
     IF(Rc >=  9.5121852168680867E+00_dp .AND. Rc <=  3.9527872132017364E+01_dp) THEN
       irange =  2
       L_b =  9.5121852168680867E+00_dp
       U_b =  3.9527872132017364E+01_dp
     END IF
     IF(Rc >=  3.9527872132017364E+01_dp .AND. Rc <=  8.3026037216186523E+01_dp) THEN
       irange =  3
       L_b =  3.9527872132017364E+01_dp
       U_b =  8.3026037216186523E+01_dp
     END IF

     fit_coef(1:13,1:6) = coefdata(1:13,1:6,irange)

   END SUBROUTINE get_fit_coef_k03

! *****************************************************************************
!> \brief ...
!> \param Rc ...
!> \param L_b ...
!> \param U_b ...
!> \param fit_coef ...
! *****************************************************************************
   SUBROUTINE get_fit_coef_k04(Rc,L_b,U_b,fit_coef)
    REAL(KIND=dp)                            :: Rc, L_b, U_b
    REAL(KIND=dp), DIMENSION(13, 8)          :: fit_coef

    REAL(KIND=dp), DIMENSION(312), PARAMETER :: c01 = (/ &
      4.0437679455062153E-01_dp, -1.9968645981826424E-01_dp,&
      -9.6669556285432015E-02_dp,  2.6839132012201752E-02_dp,&
      3.6190132805626725E-02_dp,  4.1166010414531336E-03_dp,&
      -1.0769456290470580E-02_dp, -8.7529874160434775E-05_dp,&
      5.7309585756926467E-03_dp, -1.0860752534826048E-02_dp,&
      4.3544925730678802E-03_dp,  3.1889806346690661E-03_dp,&
      -1.9657212784353890E-03_dp,  1.5529811167323646E+00_dp,&
      -1.1621893493531430E+00_dp, -1.7795567981281610E-01_dp,&
      3.9842720393802583E-01_dp,  4.8033159486636778E-02_dp,&
      -1.4469416247317868E-01_dp, -3.6520970201122292E-02_dp,&
      8.8008094722564351E-02_dp,  5.9318119282124994E-03_dp,&
      -7.6009276630089620E-02_dp,  6.7040879385407398E-02_dp,&
      -2.8473998439286250E-02_dp,  5.2089090903696678E-03_dp,&
      4.2289764691942366E+00_dp, -5.2559714863096954E+00_dp,&
      1.6753291726988750E+00_dp,  1.8504840635800728E+00_dp,&
      -1.6044237761904363E+00_dp, -4.1733772536083369E-01_dp,&
      7.9444215167607390E-01_dp,  3.1495572908490888E-01_dp,&
      -7.9016995059826667E-01_dp,  1.9066260373671118E-01_dp,&
      3.5297731186965675E-01_dp, -3.0243198972383212E-01_dp,&
      7.7756449578611248E-02_dp,  1.2967616994753087E+01_dp,&
      -2.4042024222254298E+01_dp,  2.0845923141347605E+01_dp,&
      -5.1419824074547673E+00_dp, -7.6388284418830947E+00_dp,&
      8.8813058337816244E+00_dp, -3.7007126133769321E+00_dp,&
      2.3382249105105274E-01_dp, -3.6597180661269557E-01_dp,&
      1.2892925753313280E+00_dp, -1.1365833385148225E+00_dp,&
      4.5209303506995352E-01_dp, -7.1102161040361975E-02_dp,&
      3.4348988639807971E+00_dp, -1.9004725480240556E+00_dp,&
      -7.5786793916739725E-01_dp,  3.8852279314766375E-01_dp,&
      3.0614786999111232E-01_dp, -4.5572384093187306E-02_dp,&
      -1.1543847219093645E-01_dp,  1.9867871621586992E-02_dp,&
      9.2786670262761134E-02_dp, -1.8405662264456446E-01_dp,&
      1.5762768904523788E-01_dp, -5.6388852934436663E-02_dp,&
      5.3681094755664528E-03_dp,  6.4373196501777841E+00_dp,&
      -7.1885129098744072E+00_dp,  8.4206205249439814E-01_dp,&
      3.2712289706274404E+00_dp, -8.7181606322780358E-01_dp,&
      -1.4986726477160628E+00_dp,  2.5136694217250172E-01_dp,&
      9.7470575226088230E-01_dp, -3.3001228959270695E-01_dp,&
      -3.8632103708308441E-01_dp,  3.1100472103485055E-01_dp,&
      -7.3188169703122946E-02_dp,  3.0493082567784697E-03_dp,&
      1.7166455411776617E+01_dp, -3.1333371912090001E+01_dp,&
      2.1039285623747343E+01_dp,  6.3380935910509359E+00_dp,&
      -1.7459609199004912E+01_dp,  4.3115360437865684E+00_dp,&
      7.8590901947406229E+00_dp, -2.1493300987092994E+00_dp,&
      -9.7183726596572662E+00_dp,  1.1703427571259667E+01_dp,&
      -5.4936928390403912E+00_dp,  7.6914546328373024E-01_dp,&
      1.3439917053669492E-01_dp,  6.8475451857998991E+01_dp,&
      -1.5591045907250185E+02_dp,  1.8464815785166243E+02_dp,&
      -1.1596150427703571E+02_dp,  7.1241317727507019E+00_dp,&
      6.6321778045233330E+01_dp, -8.0699134248162082E+01_dp,&
      5.8884469891028630E+01_dp, -2.9896108613598663E+01_dp,&
      8.3528206096836914E+00_dp,  1.7944119950838084E+00_dp,&
      -2.7711044075604732E+00_dp,  8.2189970277835223E-01_dp,&
      4.8928026378102546E-01_dp, -7.7395534889906831E-02_dp,&
      -9.4804748738418779E-02_dp,  3.4028552814568652E-02_dp,&
      -3.7372824741901886E-02_dp,  5.0155168087250734E-02_dp,&
      -2.5580726927850888E-02_dp, -1.2765405686004021E-02_dp,&
      4.7823767294427916E-02_dp, -7.0139763686717996E-02_dp,&
      6.3966662234848093E-02_dp, -3.3470391316181090E-02_dp,&
      7.6464138985924456E-03_dp,  2.1282655054474775E+00_dp,&
      -6.1070711898886576E-01_dp, -6.5624939488951939E-01_dp,&
      4.7938223840521010E-01_dp, -2.8747099836990569E-01_dp,&
      3.5208496407539869E-01_dp, -1.8503707524330915E-01_dp,&
      -2.0984499620126879E-01_dp,  5.4309316443457201E-01_dp,&
      -6.7503677537568874E-01_dp,  5.6360705631090735E-01_dp,&
      -2.8697201451927273E-01_dp,  6.5788018908140136E-02_dp,&
      7.5361243129007143E+00_dp, -4.3792129089618763E+00_dp,&
      -3.6353821872855629E+00_dp,  5.5509223987118927E+00_dp,&
      -3.0694183036306772E+00_dp,  1.6845171689522207E+00_dp,&
      -3.0607322949855914E-01_dp, -3.1489329672562265E+00_dp,&
      6.5649521591318027E+00_dp, -7.2571558002312946E+00_dp,&
      5.2325909153168650E+00_dp, -2.3623770342134143E+00_dp,&
      5.0394953774065110E-01_dp,  3.5135862296987604E+01_dp,&
      -4.1984978325667221E+01_dp, -1.2198457583034720E+01_dp,&
      8.0268759850780640E+01_dp, -9.2774677162666293E+01_dp,&
      3.2220400417767237E+01_dp,  6.9167997126742932E+01_dp,&
      -1.6719365094500833E+02_dp,  2.2313470021774773E+02_dp,&
      -2.1138538881579083E+02_dp,  1.4038632740328094E+02_dp,&
      -5.8475121575654534E+01_dp,  1.1417246912012875E+01_dp,&
      4.2741823086253019E+00_dp, -7.9536270802455533E-01_dp,&
      -9.4491706673580489E-01_dp,  4.1996484386720750E-01_dp,&
      -3.7185002444196380E-01_dp,  5.0738873025610154E-01_dp,&
      -2.6146306289710541E-01_dp, -1.6591733635017125E-01_dp,&
      5.3934898406905241E-01_dp, -7.5567887713707849E-01_dp,&
      6.7814453825724363E-01_dp, -3.5449909952088476E-01_dp,&
      8.1351842070653560E-02_dp,  1.0448175241518934E+01_dp,&
      -4.7162392903479757E+00_dp, -4.6011899797114211E+00_dp,&
      4.7226102598841244E+00_dp, -2.3542529316399179E+00_dp,&
      2.4099345435619655E+00_dp, -1.2513655298312776E+00_dp,&
      -2.1934790172813501E+00_dp,  5.1389071993822633E+00_dp,&
      -5.9476078886883359E+00_dp,  4.6727698853554340E+00_dp,&
      -2.3040973762603332E+00_dp,  5.2314098593306413E-01_dp,&
      4.0791877448869535E+01_dp, -3.5969613616045955E+01_dp,&
      -2.4171436973533854E+01_dp,  5.5569664985979848E+01_dp,&
      -3.4214469221683196E+01_dp,  9.5381575304633053E+00_dp,&
      6.8341242106838243E+00_dp, -3.7624304752009998E+01_dp,&
      7.1464033968563371E+01_dp, -7.7194048734399516E+01_dp,&
      5.2693863162066592E+01_dp, -2.2021176330374324E+01_dp,&
      4.3695027993673685E+00_dp,  2.5377039095893605E+02_dp,&
      -4.2994677187007676E+02_dp,  2.5923064187804403E+01_dp,&
      9.0781913943732923E+02_dp, -1.5171598707454250E+03_dp,&
      8.3438365693987998E+02_dp,  1.2503155206476356E+03_dp,&
      -3.6979114649792905E+03_dp,  5.0518426675233222E+03_dp,&
      -4.5767744436753173E+03_dp,  2.8251444928014480E+03_dp,&
      -1.0881627886949789E+03_dp,  1.9770830037818715E+02_dp,&
      5.0267168969343001E-01_dp, -7.4520589209002717E-09_dp,&
      -7.5842890294191626E-02_dp,  3.8582667911904310E-02_dp,&
      9.9191590198104990E-03_dp, -7.7610198141607994E-02_dp,&
      1.0186993606336588E-01_dp,  4.8879008495791680E-03_dp,&
      -2.5794860820246435E-01_dp,  5.1563319707313904E-01_dp,&
      -5.5779404977642033E-01_dp,  3.4264223722618931E-01_dp,&
      -9.4245905218378417E-02_dp,  2.2367579356890843E+00_dp,&
      -6.1826581961609364E-08_dp, -6.3088494762561431E-01_dp,&
      3.2094379025394543E-01_dp,  1.7810679428530926E-01_dp,&
      -7.4243963130673429E-01_dp,  8.2776076323167247E-01_dp,&
      3.0319271394860581E-01_dp, -2.6573611449279531E+00_dp,&
      4.8575612228055789E+00_dp, -5.0211444947565047E+00_dp,&
      2.9860321004621575E+00_dp, -8.0025459339884109E-01_dp,&
      8.3489662287757334E+00_dp, -4.8081262326306912E-07_dp,&
      -4.9360724580789306E+00_dp,  2.5110974244370783E+00_dp,&
      2.6505734507780212E+00_dp, -7.0808033521004949E+00_dp,&
      5.7735097501417201E+00_dp,  6.5854985317005275E+00_dp,&
      -2.7913575624582830E+01_dp,  4.4816273679035426E+01_dp,&
      -4.2776722427843723E+01_dp,  2.3856755737116387E+01_dp,&
      -6.0412032353147502E+00_dp,  4.3783125829708453E+01_dp,&
      -5.5075146143120555E-06_dp, -5.8188143370443861E+01_dp,&
      2.9602672801274529E+01_dp,  6.8935349052196329E+01_dp,&
      -1.2146548315255620E+02_dp,  1.2141505510027159E+01_dp,&
      2.6316188098445042E+02_dp, -5.3756800699873463E+02_dp,&
      5.8351355045248351E+02_dp, -3.6666352432161307E+02_dp,&
      1.1185697561839862E+02_dp, -6.4042001817970036E+00_dp,&
      4.4126879375879327E+00_dp, -7.7530076630345340E-08_dp,&
      -7.8950662310582564E-01_dp,  4.0163677540945897E-01_dp,&
      1.3226474718849238E-01_dp, -8.3730503258614408E-01_dp,&
      1.0575301512207400E+00_dp,  1.2535085289938416E-01_dp,&
      -2.8368981498982810E+00_dp,  5.5425059181428447E+00_dp,&
      -5.9300191913604117E+00_dp,  3.6151514581350099E+00_dp,&
      -9.8848229054852588E-01_dp,  1.1300654351102278E+01_dp,&
      -4.9316510953177751E-07_dp, -5.0417664922059764E+00_dp,&
      2.5648534509539416E+00_dp,  1.9121547495752944E+00_dp,&
      -6.4282832469653375E+00_dp,  6.4567195634727446E+00_dp,&
      3.8645778438672509E+00_dp, -2.3915493684299207E+01_dp,&
      4.1667602971700134E+01_dp, -4.1912306733231269E+01_dp,&
      2.4412186136448963E+01_dp, -6.4287151562910099E+00_dp,&
      4.7661122084367847E+01_dp, -4.1563904102160877E-06_dp,&
      -4.2866809130358867E+01_dp,  2.1807486009438179E+01_dp,&
      2.9911195911583174E+01_dp, -6.8460053271712155E+01_dp,&
      4.4618313870507741E+01_dp,  8.3146438950249362E+01_dp,&
      -2.8242592797658313E+02_dp,  4.2273976065299058E+02_dp,&
      -3.8330562005148283E+02_dp,  2.0396556823058276E+02_dp,&
      -4.9315105307208164E+01_dp,  3.4873416553466677E+02_dp,&
      -6.2812514911455712E-05_dp, -6.8223088256704716E+02_dp,&
      3.4708903230411840E+02_dp,  1.0990846081358511E+03_dp,&
      -1.7164437218305100E+03_dp, -5.1026413431245879E+02_dp,&
      4.8843366036040652E+03_dp, -7.6063698448775094E+03_dp,&
      5.5268501397810141E+03_dp, -6.4043799549226162E+02_dp,&
      -1.8963185698837842E+03_dp,  1.0098848671258462E+03_dp /)
    REAL(KIND=dp), DIMENSION(13, 8, 3), PARAMETER :: &
      coefdata = RESHAPE ((/ c01 /),(/13,8,3/))

    INTEGER                                  :: irange

     IF(Rc <=  2.0000000000000000E+00_dp) Rc =   2.0000000000000000E+00_dp
     IF(Rc >=  2.3323979377746582E+02_dp) Rc =   2.3323979377746582E+02_dp
     IF(Rc >=  2.0000000000000000E+00_dp .AND. Rc <=  1.4737051046890191E+01_dp) THEN
       irange =  1
       L_b =  2.0000000000000000E+00_dp
       U_b =  1.4737051046890191E+01_dp
     END IF
     IF(Rc >=  1.4737051046890191E+01_dp .AND. Rc <=  9.3388181952559151E+01_dp) THEN
       irange =  2
       L_b =  1.4737051046890191E+01_dp
       U_b =  9.3388181952559151E+01_dp
     END IF
     IF(Rc >=  9.3388181952559151E+01_dp .AND. Rc <=  2.3323979377746582E+02_dp) THEN
       irange =  3
       L_b =  9.3388181952559151E+01_dp
       U_b =  2.3323979377746582E+02_dp
     END IF

     fit_coef(1:13,1:8) = coefdata(1:13,1:8,irange)

   END SUBROUTINE get_fit_coef_k04

! *****************************************************************************
!> \brief ...
!> \param Rc ...
!> \param L_b ...
!> \param U_b ...
!> \param fit_coef ...
! *****************************************************************************
   SUBROUTINE get_fit_coef_k05(Rc,L_b,U_b,fit_coef)
    REAL(KIND=dp)                            :: Rc, L_b, U_b
    REAL(KIND=dp), DIMENSION(13, 10)         :: fit_coef

    REAL(KIND=dp), DIMENSION(120), PARAMETER :: c02 = (/ &
      -4.8858613834160547E-01_dp,  3.4033685149450971E-01_dp,&
      -1.0142642654121627E-01_dp,  1.8656576613307461E+00_dp,&
      -2.2611249862234847E-07_dp, -4.2139600757794465E-01_dp,&
      2.7923611356976985E-01_dp,  4.7524652113269663E-03_dp,&
      -5.0664309914469052E-01_dp,  9.4151166751519222E-01_dp,&
      -6.3706758785225281E-01_dp, -8.8091156763512435E-01_dp,&
      2.9544585972346740E+00_dp, -3.8184952299416528E+00_dp,&
      2.5987562420872985E+00_dp, -7.6383091805613368E-01_dp,&
      5.8307532704464835E+00_dp, -1.3235163063855027E-06_dp,&
      -2.4668718803223433E+00_dp,  1.6346632087317283E+00_dp,&
      4.2793395990681971E-01_dp, -3.4953188430530364E+00_dp,&
      5.6218711785562494E+00_dp, -2.5804721986537875E+00_dp,&
      -8.1487301965478487E+00_dp,  2.1579397272924947E+01_dp,&
      -2.6243755352892087E+01_dp,  1.7322843340400933E+01_dp,&
      -4.9952855649733046E+00_dp,  2.1137358448719954E+01_dp,&
      -8.8571011614827196E-06_dp, -1.6522540002420385E+01_dp,&
      1.0948652835693094E+01_dp,  7.3116158949398784E+00_dp,&
      -2.9281521088202449E+01_dp,  3.7027168556990446E+01_dp,&
      -3.0044062555390699E-01_dp, -9.2699397470543445E+01_dp,&
      1.9478454491936060E+02_dp, -2.1832575096605436E+02_dp,&
      1.3760074533870568E+02_dp, -3.8467640857581380E+01_dp,&
      1.1063436781801329E+02_dp, -9.4105195497773129E-05_dp,&
      -1.7704451196700302E+02_dp,  1.1732328617291050E+02_dp,&
      2.0294758259689306E+02_dp, -4.7724061601861166E+02_dp,&
      2.4811244866542938E+02_dp,  7.7992320599735456E+02_dp,&
      -2.3222278304245528E+03_dp,  3.3978948507388491E+03_dp,&
      -3.0865392333615719E+03_dp,  1.6634416402126349E+03_dp,&
      -4.0958960918663729E+02_dp,  3.9147839650080969E+00_dp,&
      -3.2520105963657300E-07_dp, -6.0607587797083018E-01_dp,&
      4.0161328249353928E-01_dp, -4.2547612515729416E-02_dp,&
      -6.6328981617512195E-01_dp,  1.3305448184666258E+00_dp,&
      -1.0356092213054167E+00_dp, -9.2632217066791589E-01_dp,&
      3.7416671458058000E+00_dp, -5.0171843083604752E+00_dp,&
      3.4735112435596691E+00_dp, -1.0314560294834274E+00_dp,&
      8.5066988542163120E+00_dp, -1.5956969892592431E-06_dp,&
      -2.9738314062541940E+00_dp,  1.9705962320072299E+00_dp,&
      2.6378420045856443E-01_dp, -3.8802561477551043E+00_dp,&
      6.7413347741113512E+00_dp, -3.9103035750119401E+00_dp,&
      -7.8429956431410091E+00_dp,  2.3246376149210917E+01_dp,&
      -2.9171150509747868E+01_dp,  1.9568524918391812E+01_dp,&
      -5.7008266943227328E+00_dp,  2.7615861977495285E+01_dp,&
      -9.5262930517419120E-06_dp, -1.7758829666390575E+01_dp,&
      1.1767834951864666E+01_dp,  5.0226252636691662E+00_dp,&
      -2.7730364017515328E+01_dp,  4.0708746346053992E+01_dp,&
      -1.2326936467644122E+01_dp, -7.4007384787068389E+01_dp,&
      1.7668831197880644E+02_dp, -2.0786024226383526E+02_dp,&
      1.3475739031543512E+02_dp, -3.8406427466935618E+01_dp,&
      1.2015405393414257E+02_dp, -7.2621559578354175E-05_dp,&
      -1.3559498479069612E+02_dp,  8.9852390667028018E+01_dp,&
      8.4938472949204964E+01_dp, -2.7318759226639492E+02_dp,&
      2.9372517367957340E+02_dp,  1.0805522341970379E+02_dp,&
      -9.9062857988965527E+02_dp,  1.8854424898786242E+03_dp,&
      -2.0220366613923659E+03_dp,  1.2394263774198967E+03_dp,&
      -3.3964650795183775E+02_dp,  8.8110212941579391E+02_dp,&
      -1.0631056167459761E-03_dp, -2.0215919999407249E+03_dp,&
      1.3397278899112848E+03_dp,  3.2785602402023524E+03_dp,&
      -6.7015638377241066E+03_dp,  8.2607523725594240E+02_dp,&
      1.7011468237133256E+04_dp, -3.7898634628790373E+04_dp,&
      4.6113041763967398E+04_dp, -3.4882603781213867E+04_dp,&
      1.5359465882590917E+04_dp, -2.9911038365687018E+03_dp /)
    REAL(KIND=dp), DIMENSION(400), PARAMETER :: c01 = (/ &
      2.6550926316288959E-01_dp, -1.4194808279243129E-01_dp,&
      -1.8340243943233556E-02_dp,  1.5594859644926301E-02_dp,&
      6.5213609288545414E-03_dp, -7.2580394496045918E-04_dp,&
      -1.4367869259991610E-03_dp, -5.1441935266499043E-04_dp,&
      1.1018516863846910E-04_dp,  1.3857094404802231E-04_dp,&
      2.1996788631150751E-04_dp, -1.7152748784100114E-04_dp,&
      2.0272591048711931E-05_dp,  8.8501620116612245E-01_dp,&
      -5.6405190932498062E-01_dp,  2.1373574602775065E-02_dp,&
      6.3146135511270907E-02_dp, -2.3147542901442224E-05_dp,&
      -7.8175619763017744E-03_dp, -9.0807036857844356E-04_dp,&
      5.6214855418821154E-04_dp,  2.4232956169489725E-04_dp,&
      7.6709494480488226E-05_dp,  7.0466487248201543E-05_dp,&
      -1.6311093828886886E-04_dp,  5.2007073293462518E-05_dp,&
      1.8263499497356539E+00_dp, -1.5009396517669864E+00_dp,&
      4.1870586931086229E-01_dp,  8.1727208946953692E-02_dp,&
      -7.9985490073436638E-02_dp, -2.8386179694723992E-04_dp,&
      1.3714799393655325E-02_dp, -1.2765799618254241E-03_dp,&
      -2.3532630322310429E-03_dp,  2.9144423884117456E-04_dp,&
      5.3898879161715227E-04_dp, -2.1117200296074146E-04_dp,&
      1.1890367495914655E-05_dp,  3.6068126097193955E+00_dp,&
      -3.7478336902205442E+00_dp,  1.9251356093214684E+00_dp,&
      -3.3446998446816661E-01_dp, -1.8448451339597130E-01_dp,&
      1.0688477101804973E-01_dp,  2.8699422521969551E-03_dp,&
      -1.7054194854644773E-02_dp,  2.7799971291916320E-03_dp,&
      2.1897310530225645E-03_dp, -9.6910398641202955E-04_dp,&
      -1.7014948966298835E-05_dp,  7.0764797933466401E-05_dp,&
      8.5076801150728940E+00_dp, -1.0347947968427668E+01_dp,&
      7.0922698151187697E+00_dp, -2.6797820059032738E+00_dp,&
      2.9467919296352824E-01_dp,  2.0165045409973004E-01_dp,&
      -8.9988805784438325E-02_dp,  2.4637727661109806E-03_dp,&
      8.1179838192678364E-03_dp, -1.1425709074878842E-03_dp,&
      -1.4540082710946707E-03_dp,  7.4743096963710671E-04_dp,&
      -9.5059826033180061E-05_dp,  2.1799143945696855E+00_dp,&
      -1.2215729848174013E+00_dp, -1.0378540265907392E-01_dp,&
      1.3956467575170556E-01_dp,  4.1207538207079744E-02_dp,&
      -1.1282311598005327E-02_dp, -1.0171856224520444E-02_dp,&
      -2.3373290671285817E-03_dp,  1.2653652219253339E-03_dp,&
      9.5707490422460495E-04_dp,  1.2101366427858700E-03_dp,&
      -1.2780291066208068E-03_dp,  2.5756088060468888E-04_dp,&
      2.9208363903622483E+00_dp, -2.3899201322394812E+00_dp,&
      5.8884810920058861E-01_dp,  2.2762106025416384E-01_dp,&
      -1.3362043169847870E-01_dp, -3.7501085007398918E-02_dp,&
      2.4747513399946530E-02_dp,  9.5199228229788154E-03_dp,&
      -3.2039915558658432E-03_dp, -1.9527625509728832E-03_dp,&
      -1.2995234139512991E-03_dp,  1.6838354170501096E-03_dp,&
      -3.6264295088535243E-04_dp,  4.9061852059580007E+00_dp,&
      -5.5772589654145772E+00_dp,  3.0458942185873026E+00_dp,&
      -3.1151471012072757E-01_dp, -5.0962295355233656E-01_dp,&
      1.7119353498272677E-01_dp,  7.3421207250380538E-02_dp,&
      -4.2415114572035145E-02_dp, -1.0449306726232923E-02_dp,&
      8.8592606635370576E-03_dp,  2.3422146446452656E-03_dp,&
      -3.0644537695472701E-03_dp,  7.3510771583872308E-04_dp,&
      1.0361057376012624E+01_dp, -1.3801723359447648E+01_dp,&
      1.0269637159792943E+01_dp, -3.8517372257931135E+00_dp,&
      4.8210238729628187E-02_dp,  6.4093439772823868E-01_dp,&
      -2.4036195455750892E-01_dp, -2.8318831513976336E-02_dp,&
      5.0063938591181703E-02_dp, -9.0143432978221204E-03_dp,&
      -7.6402101435511958E-03_dp,  4.7996517140455634E-03_dp,&
      -8.8688820948506499E-04_dp,  3.7821289404628558E+01_dp,&
      -5.0486826995622586E+01_dp,  3.9699236632662554E+01_dp,&
      -1.8952398193101370E+01_dp,  5.0163848584714357E+00_dp,&
      -2.0867251269953616E-01_dp, -3.6024073310994120E-01_dp,&
      1.2257744971762999E-01_dp, -2.0556888444551520E-03_dp,&
      -4.3391049691802507E-03_dp, -5.8398844655175447E-03_dp,&
      4.3389323693662314E-03_dp, -7.8818019920401015E-04_dp,&
      3.9278921854245491E-01_dp, -1.3880097398618357E-01_dp,&
      -7.8179351611203124E-02_dp,  1.1692302652801807E-02_dp,&
      2.2561926975071613E-02_dp,  2.7058132466622827E-03_dp,&
      -1.1199299265664485E-02_dp,  2.3864268449642063E-03_dp,&
      4.1154746187549900E-03_dp, -6.1786230594345682E-04_dp,&
      -3.7868033504497873E-03_dp,  2.7672863440031232E-03_dp,&
      -5.9500399325799157E-04_dp,  1.4855053290241629E+00_dp,&
      -7.8227620557414324E-01_dp, -2.7024820829841489E-01_dp,&
      2.1644799093884431E-01_dp,  9.2348870972741803E-02_dp,&
      -6.6068295446994954E-02_dp, -6.4291229844158415E-02_dp,&
      5.0728492961798738E-02_dp,  4.4472290549627404E-02_dp,&
      -7.4201442830987882E-02_dp,  3.5972879001982185E-02_dp,&
      -4.4915225522013684E-03_dp, -1.0083365147057061E-03_dp,&
      3.9016131418911306E+00_dp, -3.4134865404724506E+00_dp,&
      -1.3191399671771223E-01_dp,  1.4894435669800952E+00_dp,&
      -1.9976923375832278E-01_dp, -6.1585252828530235E-01_dp,&
      7.5074460723037759E-03_dp,  4.4772359113701282E-01_dp,&
      4.1917618732983360E-03_dp, -5.4159522447435060E-01_dp,&
      5.2585946916715653E-01_dp, -2.2522710312470215E-01_dp,&
      3.9131815966317804E-02_dp,  1.0618774133100274E+01_dp,&
      -1.4832497853376831E+01_dp,  5.4089669906270323E+00_dp,&
      5.8728749535356988E+00_dp, -5.9311544312086957E+00_dp,&
      -8.0009436905790932E-01_dp,  2.6888418432666157E+00_dp,&
      1.2359800658100402E+00_dp, -3.5501238739964611E+00_dp,&
      1.5596171190481971E+00_dp,  8.2306292720997121E-01_dp,&
      -1.0016591564251089E+00_dp,  2.8042093815884245E-01_dp,&
      3.4412759856983151E+01_dp, -7.0327105581163224E+01_dp,&
      6.3665816591943127E+01_dp, -1.3949266108790304E+01_dp,&
      -3.0883863109385448E+01_dp,  3.7009631013886363E+01_dp,&
      -1.7390259933678596E+01_dp,  2.3656018006829531E+00_dp,&
      -1.7787709312289635E+00_dp,  5.3590592455391164E+00_dp,&
      -4.8736190529851529E+00_dp,  2.0252305602012526E+00_dp,&
      -3.3791044802812337E-01_dp,  3.3248343266591256E+00_dp,&
      -1.3094675160912539E+00_dp, -6.6328990690593814E-01_dp,&
      1.8491505826004428E-01_dp,  2.0917162881033141E-01_dp,&
      -1.0803587412967225E-02_dp, -1.1437161193809799E-01_dp,&
      3.7418912128605243E-02_dp,  5.1391607560251427E-02_dp,&
      -3.0429928256627381E-02_dp, -2.3127059109442100E-02_dp,&
      2.5736365267549230E-02_dp, -6.6907054068503953E-03_dp,&
      6.0158175654245101E+00_dp, -4.7207061941701625E+00_dp,&
      -7.7455852362166566E-01_dp,  1.9487679367187645E+00_dp,&
      2.4649244194611070E-01_dp, -7.8079051035147518E-01_dp,&
      -2.9522118467223896E-01_dp,  5.0896330357227537E-01_dp,&
      2.9773062721204507E-01_dp, -8.0505532840347449E-01_dp,&
      5.8712415328747258E-01_dp, -1.9752002341908578E-01_dp,&
      2.5075590818523352E-02_dp,  1.4995996415485168E+01_dp,&
      -1.9905726450058165E+01_dp,  3.9266337752980962E+00_dp,&
      9.9560813158678823E+00_dp, -4.7405279236906184E+00_dp,&
      -4.1496599853648704E+00_dp,  2.0613032230603539E+00_dp,&
      3.2488688171619757E+00_dp, -1.7390067336108939E+00_dp,&
      -3.0683317274378470E+00_dp,  4.2625062495968713E+00_dp,&
      -2.1726002323140921E+00_dp,  4.3089670881740993E-01_dp,&
      4.4932892872124903E+01_dp, -8.8545093801744059E+01_dp,&
      6.0520129702721192E+01_dp,  2.1122643416448547E+01_dp,&
      -5.7886283556557316E+01_dp,  1.8213156609123580E+01_dp,&
      2.3077349912808526E+01_dp, -5.3877102833312573E+00_dp,&
      -3.7896180227046528E+01_dp,  4.8693135565506630E+01_dp,&
      -2.6834347784939805E+01_dp,  6.5383813741214230E+00_dp,&
      -3.5691234747562894E-01_dp,  1.8944775478692549E+02_dp,&
      -4.7251287821524852E+02_dp,  5.8393508267544064E+02_dp,&
      -3.6596986190399576E+02_dp, -2.7795511500841403E+01_dp,&
      3.2140628060441372E+02_dp, -3.8370262121429585E+02_dp,&
      2.7692519369868364E+02_dp, -1.2701896028677757E+02_dp,&
      1.8998010412055702E+01_dp,  2.1870301973287983E+01_dp,&
      -1.7120461042981180E+01_dp,  4.2347422391201750E+00_dp,&
      4.4257797087053102E-01_dp, -5.2788253001961889E-02_dp,&
      -5.2533965745094266E-02_dp,  1.4778463051880717E-02_dp,&
      -1.5137288357210552E-02_dp,  2.1322924325981523E-02_dp,&
      -1.4854865556221506E-02_dp,  5.1232024991161550E-03_dp,&
      2.8565216174117648E-03_dp, -8.8401575833623354E-03_dp,&
      9.9499800976253043E-03_dp, -5.7899840382429997E-03_dp,&
      1.4210619672786226E-03_dp,  1.7901185397017931E+00_dp,&
      -3.5083028235258551E-01_dp, -3.1680230121654573E-01_dp,&
      1.5935023709180390E-01_dp, -9.6024129986843676E-02_dp,&
      1.3660790479315130E-01_dp, -1.0337866658231480E-01_dp,&
      2.3138570375758987E-02_dp,  3.9352695535177187E-02_dp,&
      -7.4818829308294185E-02_dp,  7.5902068752193866E-02_dp,&
      -4.3371504574516717E-02_dp,  1.0713355872630637E-02_dp,&
      5.4011480843282529E+00_dp, -1.9378498606319501E+00_dp,&
      -1.4885710416432727E+00_dp,  1.3285785197655227E+00_dp,&
      -6.1573768211511359E-01_dp,  6.6799377473279897E-01_dp,&
      -5.6560044387820196E-01_dp,  4.4739939170913824E-02_dp,&
      4.1235540457510034E-01_dp, -5.8360743029848561E-01_dp,&
      5.0454137563363111E-01_dp, -2.6911125304703654E-01_dp,&
      6.5360795178081008E-02_dp,  1.8390811469886259E+01_dp,&
      -1.1816135218340939E+01_dp, -6.5672253964212350E+00_dp,&
      1.1633062919350717E+01_dp, -6.3986980710522809E+00_dp,&
      3.2939209809513676E+00_dp, -2.2272935468482853E+00_dp,&
      -1.0294340901252237E+00_dp,  4.9393479720617810E+00_dp,&
      -6.2342624050806759E+00_dp,  4.6837849201767963E+00_dp,&
      -2.1673488253493325E+00_dp,  4.7646725156583597E-01_dp,&
      8.4336157557514952E+01_dp, -1.0089617503672321E+02_dp,&
      -8.2208590050504053E+00_dp,  1.3366974137866799E+02_dp,&
      -1.5820903503567655E+02_dp,  7.9337305769534240E+01_dp,&
      3.5616012388753418E+01_dp, -1.3130781106536887E+02_dp,&
      1.8168276052903283E+02_dp, -1.7517829713272582E+02_dp,&
      1.1971260321867801E+02_dp, -5.1913306577605510E+01_dp,&
      1.0642365990915014E+01_dp,  3.8045299867771694E+00_dp,&
      -5.1963750597518243E-01_dp, -5.0493129539283388E-01_dp,&
      1.6923327934559071E-01_dp, -1.4533839571881732E-01_dp,&
      2.0900682452821240E-01_dp, -1.4833602186795575E-01_dp,&
      4.6115749291176553E-02_dp,  3.5317361550774348E-02_dp,&
      -9.2726757854478412E-02_dp,  1.0182710534344803E-01_dp,&
      -5.9085836061311556E-02_dp,  1.4532428085371502E-02_dp,&
      7.9810439938759341E+00_dp, -2.4066556149498681E+00_dp,&
      -2.0114738728086969E+00_dp,  1.3887125725182221E+00_dp,&
      -6.6411710313452954E-01_dp,  8.9805661552767102E-01_dp,&
      -7.2555023164908816E-01_dp,  1.0674885228225427E-01_dp,&
      3.7883539402094696E-01_dp, -6.0015012013448121E-01_dp,&
      5.6777546719923455E-01_dp, -3.1846433701583721E-01_dp,&
      7.8788285144172049E-02_dp,  2.4582796909232854E+01_dp,&
      -1.3409035847628800E+01_dp, -9.0550767935687659E+00_dp,&
      1.1183625388029842E+01_dp, -5.0365240125360282E+00_dp,&
      4.1413731325516521E+00_dp, -3.6960181281327560E+00_dp,&
      -1.1306341316679099E-01_dp,  3.8775217524474308E+00_dp,&
      -5.0313030646474921E+00_dp,  3.9973018771671991E+00_dp,&
      -2.0116602935864174E+00_dp,  4.7586858968669676E-01_dp,&
      9.8317340117791673E+01_dp, -9.0854955135839106E+01_dp,&
      -3.6899617942823220E+01_dp,  1.0594926320423745E+02_dp,&
      -6.9019058089648595E+01_dp,  2.3438541656567132E+01_dp,&
      -6.0730297290477884E+00_dp, -1.8851461187993809E+01_dp,&
      5.3716963536506903E+01_dp, -6.6369119452540346E+01_dp,&
      4.8465181956859681E+01_dp, -2.1165852568624782E+01_dp,&
      4.3585743760971170E+00_dp,  6.0207029114733461E+02_dp,&
      -9.8896334810595192E+02_dp,  2.3242568533798712E+02_dp,&
      1.3611103974253565E+03_dp, -2.3785984539197157E+03_dp,&
      1.6929413675263338E+03_dp,  4.9725086152063358E+02_dp,&
      -2.8910968561531386E+03_dp,  4.1515560821383906E+03_dp,&
      -3.8021899986365097E+03_dp,  2.3881906025704566E+03_dp,&
      -9.5025460048391619E+02_dp,  1.8084927413816368E+02_dp,&
      4.5372254459206868E-01_dp, -3.2706671239427972E-08_dp,&
      -6.0956447616697843E-02_dp,  4.0392497654627497E-02_dp,&
      -5.9611390145588245E-03_dp, -6.4482812112108656E-02_dp,&
      1.3285507045769324E-01_dp, -1.0786011793963135E-01_dp,&
      -8.2044143611323866E-02_dp,  3.5942368229798438E-01_dp /)
    REAL(KIND=dp), DIMENSION(13, 10, 4), PARAMETER :: &
      coefdata = RESHAPE ((/ c01, c02 /),(/13,10,4/))

    INTEGER                                  :: irange

     IF(Rc <=  2.0000000000000000E+00_dp) Rc =   2.0000000000000000E+00_dp
     IF(Rc >=  5.8933372497558594E+02_dp) Rc =   5.8933372497558594E+02_dp
     IF(Rc >=  2.0000000000000000E+00_dp .AND. Rc <=  5.0183126784059855E+00_dp) THEN
       irange =  1
       L_b =  2.0000000000000000E+00_dp
       U_b =  5.0183126784059855E+00_dp
     END IF
     IF(Rc >=  5.0183126784059855E+00_dp .AND. Rc <=  4.3973147761154188E+01_dp) THEN
       irange =  2
       L_b =  5.0183126784059855E+00_dp
       U_b =  4.3973147761154188E+01_dp
     END IF
     IF(Rc >=  4.3973147761154188E+01_dp .AND. Rc <=  2.1570029942124913E+02_dp) THEN
       irange =  3
       L_b =  4.3973147761154188E+01_dp
       U_b =  2.1570029942124913E+02_dp
     END IF
     IF(Rc >=  2.1570029942124913E+02_dp .AND. Rc <=  5.8933372497558594E+02_dp) THEN
       irange =  4
       L_b =  2.1570029942124913E+02_dp
       U_b =  5.8933372497558594E+02_dp
     END IF

     fit_coef(1:13,1:10) = coefdata(1:13,1:10,irange)

   END SUBROUTINE get_fit_coef_k05

! *****************************************************************************
!> \brief ...
!> \param Rc ...
!> \param L_b ...
!> \param U_b ...
!> \param fit_coef ...
! *****************************************************************************
   SUBROUTINE get_fit_coef_k06(Rc,L_b,U_b,fit_coef)
    REAL(KIND=dp)                            :: Rc, L_b, U_b
    REAL(KIND=dp), DIMENSION(13, 12)         :: fit_coef

    REAL(KIND=dp), DIMENSION(224), PARAMETER :: c02 = (/ &
      -2.1886099081761100E-02_dp,  1.6493591518278907E-03_dp,&
      1.1789604920345118E-03_dp,  6.4862232853117350E+00_dp,&
      -1.7754848782963699E+00_dp, -1.3982954414549702E+00_dp,&
      8.0870739791892121E-01_dp, -3.2912209721480506E-01_dp,&
      6.6541689936850468E-01_dp, -7.1706760669376735E-01_dp,&
      4.1574320605711801E-01_dp, -2.3388393290566312E-01_dp,&
      1.8531370116520027E-01_dp, -1.0473625529229623E-01_dp,&
      2.3517641650793227E-02_dp,  8.6244301390663231E-04_dp,&
      1.7041857699471105E+01_dp, -8.0569049462408699E+00_dp,&
      -5.3193377730235101E+00_dp,  5.2407625518133418E+00_dp,&
      -1.8916937701142056E+00_dp,  2.6635338859946827E+00_dp,&
      -3.4546849156995960E+00_dp,  1.9859939673472693E+00_dp,&
      -6.3042912150903241E-01_dp,  4.6651519138633141E-01_dp,&
      -4.6544250709211266E-01_dp,  2.0725559386709488E-01_dp,&
      -2.9916484811181372E-02_dp,  5.4109098148865449E+01_dp,&
      -4.0117846296551058E+01_dp, -1.9343735087203584E+01_dp,&
      3.5163214199743337E+01_dp, -1.6224366437588795E+01_dp,&
      1.0967650063875425E+01_dp, -1.6086523192511546E+01_dp,&
      1.0289145829533357E+01_dp,  7.0780220688975037E-01_dp,&
      -3.9012608853545987E+00_dp,  1.3711604453275446E+00_dp,&
      1.9280292939737056E-01_dp, -1.3952390127271364E-01_dp,&
      2.1301179294983942E+02_dp, -2.4882116172344013E+02_dp,&
      -4.4576343993020110E+01_dp,  2.8297777525326859E+02_dp,&
      -2.2475060506667668E+02_dp,  8.3513335806543239E+01_dp,&
      -4.8903037857891732E+01_dp,  3.0142938850263359E+01_dp,&
      5.6123720416754203E+01_dp, -1.2447276961108798E+02_dp,&
      1.0366026212412571E+02_dp, -4.3281169846400822E+01_dp,&
      7.6490691991820876E+00_dp,  1.2550066552494284E+03_dp,&
      -2.4322243152990673E+03_dp,  1.2301224175493214E+03_dp,&
      2.5239221182153856E+03_dp, -6.0423251531521428E+03_dp,&
      6.0570404483243292E+03_dp, -1.7810143991349457E+03_dp,&
      -4.4038002244714671E+03_dp,  8.8124430795611534E+03_dp,&
      -9.1651316571527568E+03_dp,  6.1357943020836110E+03_dp,&
      -2.5081424703717635E+03_dp,  4.7811463385949389E+02_dp,&
      4.1708346282835440E-01_dp,  5.4402119511920706E-09_dp,&
      -5.5526039243043029E-02_dp,  4.5875452433465184E-02_dp,&
      -1.9952652014140874E-02_dp, -6.2561720018590866E-02_dp,&
      1.9429518598408954E-01_dp, -2.9279948367686742E-01_dp,&
      2.3756176667253856E-01_dp, -1.4901115793698551E-02_dp,&
      -1.8840975031156562E-01_dp,  1.8774227251450593E-01_dp,&
      -6.3735493877313892E-02_dp,  1.6278649020334413E+00_dp,&
      3.2838198887433358E-08_dp, -3.3846296868812692E-01_dp,&
      2.7963563823781828E-01_dp, -9.0630995340932871E-02_dp,&
      -4.3293262954703110E-01_dp,  1.2280220838385034E+00_dp,&
      -1.7442094645170365E+00_dp,  1.2336530163787787E+00_dp,&
      2.9572964716091310E-01_dp, -1.5535951629300453E+00_dp,&
      1.3874995299222836E+00_dp, -4.5339014253773563E-01_dp,&
      4.5582381667386960E+00_dp,  1.5806304897319024E-07_dp,&
      -1.6513291934494874E+00_dp,  1.3643063680863621E+00_dp,&
      -2.2797406279030716E-01_dp, -2.4685898874588244E+00_dp,&
      6.2562167208053836E+00_dp, -8.1279599723834828E+00_dp,&
      4.3636702189236134E+00_dp,  4.3186587764282933E+00_dp,&
      -1.0540942234481506E+01_dp,  8.5267333193930419E+00_dp,&
      -2.6774459759737059E+00_dp,  1.3685608614427798E+01_dp,&
      7.8953218257276503E-07_dp, -8.3945440967525986E+00_dp,&
      6.9353993597959400E+00_dp,  3.6407979682346647E-01_dp,&
      -1.5080151073493695E+01_dp,  3.3241166265120924E+01_dp,&
      -3.7369343631286654E+01_dp,  8.3569027854846123E+00_dp,&
      4.4753024708589386E+01_dp, -7.6432229501076378E+01_dp,&
      5.6664416689962970E+01_dp, -1.7092781886452087E+01_dp,&
      4.9412897930577003E+01_dp,  4.7277509167796719E-06_dp,&
      -5.1468102651658292E+01_dp,  4.2521417495234374E+01_dp,&
      1.8361899440516524E+01_dp, -1.1920260676251814E+02_dp,&
      2.1112678563410637E+02_dp, -1.6483281126514837E+02_dp,&
      -1.3110713864031490E+02_dp,  5.5418022612192499E+02_dp,&
      -7.3738040876829530E+02_dp,  4.9946323905482791E+02_dp,&
      -1.4364581567818436E+02_dp,  2.5856094754727951E+02_dp,&
      4.9537890342657514E-05_dp, -5.1982003609948777E+02_dp,&
      4.2947182873531199E+02_dp,  6.1345329880719396E+02_dp,&
      -1.9062470900920243E+03_dp,  1.7662921114675391E+03_dp,&
      1.7231825404801061E+03_dp, -8.6457455493579109E+03_dp,&
      1.5120925743819069E+04_dp, -1.5451074708996110E+04_dp,&
      9.0782260203870264E+03_dp, -2.3814784812948374E+03_dp,&
      3.5563510586400620E+00_dp,  5.2243732431620397E-08_dp,&
      -5.3458995179895974E-01_dp,  4.4167605244136621E-01_dp,&
      -1.7944761997460976E-01_dp, -6.2338748927321674E-01_dp,&
      1.8891718769031944E+00_dp, -2.8043855484821694E+00_dp,&
      2.2030189466232226E+00_dp,  1.0320235800834847E-02_dp,&
      -1.9761291843643918E+00_dp,  1.9052121189045435E+00_dp,&
      -6.3976729377541963E-01_dp,  6.9028517698471328E+00_dp,&
      2.0949242746734285E-07_dp, -2.1756673311275518E+00_dp,&
      1.7975131975615710E+00_dp, -4.2778160177964319E-01_dp,&
      -3.0405366789532842E+00_dp,  8.1022253001293034E+00_dp,&
      -1.0982617302565613E+01_dp,  6.8131825397372605E+00_dp,&
      3.8862264333809295E+00_dp, -1.2053532500608455E+01_dp,&
      1.0154164646180618E+01_dp, -3.2431802300014412E+00_dp,&
      1.8989552788955329E+01_dp,  9.9286067106856817E-07_dp,&
      -1.0472400361079698E+01_dp,  8.6521158401880101E+00_dp,&
      -4.6627567381311841E-01_dp, -1.7284175855799315E+01_dp,&
      4.0771265171966490E+01_dp, -4.9482292403614558E+01_dp,&
      1.9569212507327620E+01_dp,  4.1160600907466055E+01_dp,&
      -8.0872171406820684E+01_dp,  6.2337361427142859E+01_dp,&
      -1.9156859827683505E+01_dp,  6.4230168958280117E+01_dp,&
      5.2702432513283289E-06_dp, -5.6724000801554709E+01_dp,&
      4.6863917998784160E+01_dp,  1.0053412110284610E+01_dp,&
      -1.1451045308948315E+02_dp,  2.3044585885631165E+02_dp,&
      -2.2906289039834832E+02_dp, -1.8606184753175402E+01_dp,&
      4.2289320906156615E+02_dp, -6.3539698307177753E+02_dp,&
      4.5150351227550135E+02_dp, -1.3329696991326111E+02_dp,&
      2.8071766112627870E+02_dp,  3.6997307117075781E-05_dp,&
      -4.0789941535711910E+02_dp,  3.3699261109303779E+02_dp,&
      2.3647894788939269E+02_dp, -1.0952495362197274E+03_dp,&
      1.6854111697437818E+03_dp, -8.5957784789276764E+02_dp,&
      -2.1979269738574790E+03_dp,  6.1001514329566426E+03_dp,&
      -7.4416162018202294E+03_dp,  4.8441672159087721E+03_dp,&
      -1.3611139403438265E+03_dp,  2.0591686087846747E+03_dp,&
      6.2055330463178533E-04_dp, -5.8307099335577632E+03_dp,&
      4.8176388108956071E+03_dp,  1.0175276953355078E+04_dp,&
      -2.6714845272387549E+04_dp,  1.3141868511851815E+04_dp,&
      5.7600064337445598E+04_dp, -1.7044732018512717E+05_dp,&
      2.5361567867653081E+05_dp, -2.3362423558935185E+05_dp,&
      1.2691185251979054E+05_dp, -3.1278542676771558E+04_dp /)
    REAL(KIND=dp), DIMENSION(400), PARAMETER :: c01 = (/ &
      2.3880318322659616E-01_dp, -1.3504254699048457E-01_dp,&
      -1.8598173825632135E-02_dp,  1.5951626606880761E-02_dp,&
      7.2684956676590529E-03_dp, -5.7141087071240480E-04_dp,&
      -1.5395949213734461E-03_dp, -7.1572363929054415E-04_dp,&
      3.6877254806601694E-05_dp,  5.4320682223059976E-06_dp,&
      5.5227348644639883E-04_dp, -3.0433080098958792E-04_dp,&
      2.0054857421109012E-05_dp,  7.7890990470111321E-01_dp,&
      -5.1037664112865433E-01_dp,  2.2683669399025620E-03_dp,&
      6.3137195324291309E-02_dp,  6.2052527538001894E-03_dp,&
      -7.4341644045277456E-03_dp, -2.1210008138554367E-03_dp,&
      -1.4297915693002166E-04_dp,  2.3675140917504573E-04_dp,&
      7.8277224401520325E-05_dp,  7.7539910614835689E-04_dp,&
      -6.6808270119695333E-04_dp,  1.4504128675696130E-04_dp,&
      1.5338400887947281E+00_dp, -1.2608929319268196E+00_dp,&
      2.7752546229559455E-01_dp,  1.1281909197533520E-01_dp,&
      -5.7715437974610692E-02_dp, -1.2667929676877377E-02_dp,&
      1.1252151548690961E-02_dp,  1.3715704256258806E-03_dp,&
      -2.1997078310386719E-03_dp, -1.9740866319126966E-04_dp,&
      6.4159705514259713E-04_dp, -2.0892816180705209E-04_dp,&
      1.2980332628359802E-05_dp,  2.7673985137655222E+00_dp,&
      -2.8693997600463321E+00_dp,  1.2857991829002497E+00_dp,&
      -5.2344086831360041E-02_dp, -2.1561376228191623E-01_dp,&
      5.8188217877604753E-02_dp,  3.2625544363151460E-02_dp,&
      -1.6261879788195335E-02_dp, -4.9150534412067178E-03_dp,&
      3.7784034995137794E-03_dp,  1.0172896355501188E-03_dp,&
      -1.3544723088620958E-03_dp,  3.3791427042144639E-04_dp,&
      5.1404992525605566E+00_dp, -6.4443570834347890E+00_dp,&
      4.1957808536165739E+00_dp, -1.2275923471635759E+00_dp,&
      -2.0396556460478615E-01_dp,  2.7559304730791950E-01_dp,&
      -4.1449669923351355E-02_dp, -3.6410768056789396E-02_dp,&
      1.4347460568477105E-02_dp,  4.4454675528836068E-03_dp,&
      -4.5378784097625498E-03_dp,  9.9961733190086135E-04_dp,&
      3.6523493717956677E-05_dp,  1.1734352150699646E+01_dp,&
      -1.6617308812671567E+01_dp,  1.3192363740983607E+01_dp,&
      -5.9719748231036816E+00_dp,  1.0980419528166845E+00_dp,&
      3.2645028974174173E-01_dp, -2.2662577335842013E-01_dp,&
      1.5473932548189148E-02_dp,  2.4554441955295301E-02_dp,&
      -3.0077324938441938E-03_dp, -6.6466692299168616E-03_dp,&
      3.3841806314929717E-03_dp, -4.5363653651646422E-04_dp,&
      1.9504561037810109E+00_dp, -1.1467433477431543E+00_dp,&
      -1.1509042021220600E-01_dp,  1.4002884228893558E-01_dp,&
      4.9229414420382964E-02_dp, -9.3425227737155385E-03_dp,&
      -1.1325413873739049E-02_dp, -4.2563268902206224E-03_dp,&
      6.7116883229536705E-04_dp,  1.5906634439756071E-04_dp,&
      3.8770231995856335E-03_dp, -2.5391926650602447E-03_dp,&
      3.3891223789822106E-04_dp,  2.4673029044535064E+00_dp,&
      -2.0268552313475112E+00_dp,  4.0046771139949433E-01_dp,&
      2.3814029693552785E-01_dp, -8.9396133850395965E-02_dp,&
      -4.4527776658762855E-02_dp,  1.4866152379554987E-02_dp,&
      9.7091069081123305E-03_dp, -1.0458116448709301E-03_dp,&
      -1.6304626515434921E-04_dp, -2.5676032129145179E-03_dp,&
      1.0030055918766197E-03_dp,  1.2679235174542378E-04_dp,&
      3.7347941735050534E+00_dp, -4.2796614963173818E+00_dp,&
      2.0984473158827268E+00_dp,  3.8759752258727312E-02_dp,&
      -4.5990959564031419E-01_dp,  5.1967372176975800E-02_dp,&
      9.7141975244401893E-02_dp, -1.3178648045339187E-02_dp,&
      -2.0209423274504174E-02_dp,  1.0105807719139734E-03_dp,&
      5.4186563920573787E-03_dp, -1.2039467701267226E-03_dp,&
      -2.3204020322593582E-04_dp,  6.5077723320734462E+00_dp,&
      -9.2352437198316721E+00_dp,  6.6750942739970016E+00_dp,&
      -1.8455859476736007E+00_dp, -6.6861629653657528E-01_dp,&
      5.9764889487189854E-01_dp, -5.0747709526665586E-03_dp,&
      -1.2913515441578532E-01_dp,  2.3272560810134339E-02_dp,&
      2.9135236898039468E-02_dp, -1.4574834005683180E-02_dp,&
      -5.2715455080038322E-04_dp,  1.3890615806649285E-03_dp,&
      1.3850913871979026E+01_dp, -2.1458713896727076E+01_dp,&
      1.8585391600120367E+01_dp, -8.8983459146558150E+00_dp,&
      1.4489567962728551E+00_dp,  9.2702628293193634E-01_dp,&
      -6.2616397632266363E-01_dp,  5.7790606569577026E-02_dp,&
      9.4342841805570538E-02_dp, -3.4918226526611050E-02_dp,&
      -1.1666274195182996E-02_dp,  1.1468596370156781E-02_dp,&
      -2.5023897917183561E-03_dp,  5.1146534060966360E+01_dp,&
      -7.7941926275268443E+01_dp,  6.8596107997848165E+01_dp,&
      -3.6890210242625017E+01_dp,  1.1357177081042030E+01_dp,&
      -9.2047899141260558E-01_dp, -7.4474117122587935E-01_dp,&
      2.5322565001955349E-01_dp,  3.2248532258476223E-02_dp,&
      -1.7095085335450899E-02_dp, -2.2598156948147496E-02_dp,&
      1.5156344013831745E-02_dp, -2.5379823676707301E-03_dp,&
      3.5751798194464240E-01_dp, -1.2600309325537651E-01_dp,&
      -6.8004084173955454E-02_dp,  1.2157899234698420E-02_dp,&
      2.1943110510575771E-02_dp, -7.9427632594316407E-04_dp,&
      -1.0361087641281600E-02_dp,  2.4821596699121824E-03_dp,&
      4.0348366680994473E-03_dp,  2.8135401244154632E-04_dp,&
      -4.6877219120401567E-03_dp,  3.1546476763206766E-03_dp,&
      -6.7677457007812901E-04_dp,  1.2968177885798766E+00_dp,&
      -6.3941693810976563E-01_dp, -2.2959973185660049E-01_dp,&
      1.6087434354085126E-01_dp,  8.6486743214774675E-02_dp,&
      -6.1614892715247518E-02_dp, -4.8028034899867122E-02_dp,&
      3.8425445859627622E-02_dp,  3.8605581267093236E-02_dp,&
      -5.6164847560257060E-02_dp,  2.0910077215322051E-02_dp,&
      1.2964252900134450E-03_dp, -1.8446256582284172E-03_dp,&
      3.1324438674766699E+00_dp, -2.4151854273598716E+00_dp,&
      -2.6047168751114452E-01_dp,  9.5966522342849570E-01_dp,&
      2.2893370477180448E-02_dp, -4.6081953292291444E-01_dp,&
      -1.8520616685596340E-02_dp,  2.7887399359474097E-01_dp,&
      7.6020644245194932E-02_dp, -4.2088644819457605E-01_dp,&
      3.6460600516358160E-01_dp, -1.4357909360196239E-01_dp,&
      2.2753321647943275E-02_dp,  7.4899427697510124E+00_dp,&
      -8.8280335952168212E+00_dp,  1.7123462104755773E+00_dp,&
      3.8707486333542227E+00_dp, -1.9815357349695950E+00_dp,&
      -1.6832510627073594E+00_dp,  1.2202327185548325E+00_dp,&
      1.1503961401223624E+00_dp, -1.1691367635695169E+00_dp,&
      -6.2515758649445019E-01_dp,  1.4260929313914967E+00_dp,&
      -8.3005376615383542E-01_dp,  1.7732045352944734E-01_dp,&
      1.9233621800739694E+01_dp, -3.2998946276611100E+01_dp,&
      1.9380091371032101E+01_dp,  8.0399253608428722E+00_dp,&
      -1.6904953829368541E+01_dp,  3.0927597143955228E+00_dp,&
      8.0994471956382856E+00_dp, -1.6782211346692346E+00_dp,&
      -1.0394704303491993E+01_dp,  1.2830229456737785E+01_dp,&
      -6.9393548763620077E+00_dp,  1.7427344551235533E+00_dp,&
      -1.2946538236454874E-01_dp,  5.8780095440425775E+01_dp,&
      -1.3609530737303956E+02_dp,  1.4571588736412139E+02_dp,&
      -6.0814926940405925E+01_dp, -4.0021974482403188E+01_dp,&
      7.1891281980526486E+01_dp, -3.7611826478981364E+01_dp,&
      3.7915945110802157E-01_dp,  3.5968397055679930E+00_dp,&
      8.9772162495814740E+00_dp, -1.2728332178915190E+01_dp,&
      6.6326513236414772E+00_dp, -1.3383759658374235E+00_dp,&
      2.9966529964603850E+00_dp, -1.1564049434132586E+00_dp,&
      -5.6912440531538755E-01_dp,  1.6403335898067262E-01_dp,&
      1.9593329346060476E-01_dp, -3.5619845228756358E-02_dp,&
      -9.7999140185112146E-02_dp,  3.4747319410711512E-02_dp,&
      4.7910901871321382E-02_dp, -2.0772008642700124E-02_dp,&
      -2.9956272785111183E-02_dp,  2.7218123458554580E-02_dp,&
      -6.6820638736595937E-03_dp,  4.9257007168397831E+00_dp,&
      -3.5217910316373517E+00_dp, -6.7964650224305845E-01_dp,&
      1.3277332421011259E+00_dp,  2.8052383481804360E-01_dp,&
      -6.1434742292954359E-01_dp, -1.8565572658205229E-01_dp,&
      3.4561046137822643E-01_dp,  2.6278957514639034E-01_dp,&
      -6.1511718055603581E-01_dp,  4.0794606897589142E-01_dp,&
      -1.1640640393281354E-01_dp,  1.0168781521691091E-02_dp,&
      1.0726490162771071E+01_dp, -1.2573726945225451E+01_dp,&
      1.2724164436038299E+00_dp,  6.0207845401403643E+00_dp,&
      -1.6036784790091536E+00_dp, -3.0437006266607063E+00_dp,&
      9.3830897885006914E-01_dp,  1.9452809069891837E+00_dp,&
      -5.0977128821033357E-01_dp, -2.3520680072775599E+00_dp,&
      2.7623531351073121E+00_dp, -1.3250450645956215E+00_dp,&
      2.5196945929743420E-01_dp,  2.6913661603602964E+01_dp,&
      -4.5878078735449954E+01_dp,  2.0881697200238932E+01_dp,&
      1.8610130455880881E+01_dp, -2.0410996419992742E+01_dp,&
      -5.3120031731806314E+00_dp,  1.2504024396654433E+01_dp,&
      4.3865502206845264E+00_dp, -1.4269112044025723E+01_dp,&
      5.9137624753890998E+00_dp,  3.7147114162202497E+00_dp,&
      -4.1086230435170954E+00_dp,  1.0998347157428636E+00_dp,&
      7.6888910996094467E+01_dp, -1.7775327311229657E+02_dp,&
      1.6411788739480767E+02_dp, -9.6173540032258611E+00_dp,&
      -1.1891208667143621E+02_dp,  8.3816879097966989E+01_dp,&
      3.2893073687014486E+01_dp, -5.7269860830743504E+01_dp,&
      -3.4020895019064923E+01_dp,  1.1350920564093998E+02_dp,&
      -1.0130944425436361E+02_dp,  4.3976474202829252E+01_dp,&
      -7.9786030819662486E+00_dp,  3.1097179840531845E+02_dp,&
      -8.4703919274622899E+02_dp,  1.1584149634487351E+03_dp,&
      -8.8086527965485436E+02_dp,  1.9720816629417556E+02_dp,&
      3.8303494310936651E+02_dp, -5.6094825875548361E+02_dp,&
      4.3778287553188028E+02_dp, -2.5506476530794743E+02_dp,&
      1.3048998666847970E+02_dp, -5.8900820248922564E+01_dp,&
      1.9052077316170529E+01_dp, -2.9873042920703918E+00_dp,&
      4.0610663028040472E-01_dp, -4.8301399895610123E-02_dp,&
      -4.4786160630258885E-02_dp,  1.0238364112541655E-02_dp,&
      -9.8866021882278981E-03_dp,  1.9164217647153457E-02_dp,&
      -1.7059481339925454E-02_dp,  1.1318694351430274E-02_dp,&
      -8.4011787568469918E-03_dp,  5.7439095201463966E-03_dp,&
      -2.2151424764168225E-03_dp,  1.2458395701843222E-04_dp,&
      1.3096563994294966E-04_dp,  1.5621404690580485E+00_dp,&
      -2.8409503600939162E-01_dp, -2.4137581977533118E-01_dp,&
      9.9141439607062593E-02_dp, -5.3835288846892100E-02_dp,&
      1.0987263584304191E-01_dp, -1.0856682700179419E-01_dp,&
      6.6251217219820219E-02_dp, -4.3024826957833544E-02_dp,&
      3.1864939035592607E-02_dp, -1.5056601946291184E-02_dp,&
      2.2374108930218049E-03_dp,  4.7878419253682188E-04_dp,&
      4.2455143126678196E+00_dp, -1.3179848214982610E+00_dp,&
      -9.7615902340430949E-01_dp,  6.9415355017497327E-01_dp,&
      -2.7067833562287136E-01_dp,  4.7087097135136075E-01_dp,&
      -5.4436814808071254E-01_dp,  3.1543314880009077E-01_dp,&
      -1.4717023929479195E-01_dp,  1.1496028816623026E-01_dp,&
      -7.8220850515749757E-02_dp,  2.4387786200859570E-02_dp,&
      -1.6180203432776962E-03_dp,  1.2149461500713047E+01_dp,&
      -6.2522971044566518E+00_dp, -3.7120029221243351E+00_dp,&
      4.5879681385727071E+00_dp, -1.8836178860735326E+00_dp,&
      1.9211532966550233E+00_dp, -2.6104498488332846E+00_dp,&
      1.5732843933959362E+00_dp, -2.5885107195900686E-01_dp,&
      -1.4630825125608142E-02_dp, -1.4387123907875027E-01_dp,&
      1.1644713286818151E-01_dp, -2.4631995705449012E-02_dp,&
      4.0517086212606905E+01_dp, -3.4146852052539010E+01_dp,&
      -1.2144458368633835E+01_dp,  3.3721864721439921E+01_dp,&
      -2.0892740591813197E+01_dp,  9.9040393481778999E+00_dp,&
      -1.0775109641610795E+01_dp,  7.1502845688098153E+00_dp,&
      3.4412821970099596E+00_dp, -9.1170213254974826E+00_dp,&
      6.5769358742518946E+00_dp, -2.2907036885136565E+00_dp,&
      3.3510039300298694E-01_dp,  1.8014990635510685E+02_dp,&
      -2.6175603836231346E+02_dp,  4.1474134976042123E+01_dp,&
      2.9935351196164999E+02_dp, -4.4486623371446325E+02_dp,&
      3.1189315679201121E+02_dp, -4.5857776124608712E+01_dp,&
      -1.9606266882144880E+02_dp,  3.3823280092634457E+02_dp,&
      -3.5026730689155517E+02_dp,  2.4561561076464852E+02_dp,&
      -1.0634490523999061E+02_dp,  2.1350371455168180E+01_dp,&
      3.4511581153937105E+00_dp, -4.6074986504725718E-01_dp,&
      -4.1793306232973060E-01_dp,  1.1447565909224683E-01_dp,&
      -9.1392964448968397E-02_dp,  1.8219557370192899E-01_dp,&
      -1.6634354182093192E-01_dp,  1.0757299759560938E-01_dp,&
      -7.7683124155527070E-02_dp,  5.4245972534987336E-02_dp /)
    REAL(KIND=dp), DIMENSION(13, 12, 4), PARAMETER :: &
      coefdata = RESHAPE ((/ c01, c02 /),(/13,12,4/))

    INTEGER                                  :: irange

     IF(Rc <=  2.0000000000000000E+00_dp) Rc =   2.0000000000000000E+00_dp
     IF(Rc >=  1.3773078918457031E+03_dp) Rc =   1.3773078918457031E+03_dp
     IF(Rc >=  2.0000000000000000E+00_dp .AND. Rc <=  6.1993729896731562E+00_dp) THEN
       irange =  1
       L_b =  2.0000000000000000E+00_dp
       U_b =  6.1993729896731562E+00_dp
     END IF
     IF(Rc >=  6.1993729896731562E+00_dp .AND. Rc <=  6.7432603912821875E+01_dp) THEN
       irange =  2
       L_b =  6.1993729896731562E+00_dp
       U_b =  6.7432603912821875E+01_dp
     END IF
     IF(Rc >=  6.7432603912821875E+01_dp .AND. Rc <=  4.2684148594526539E+02_dp) THEN
       irange =  3
       L_b =  6.7432603912821875E+01_dp
       U_b =  4.2684148594526539E+02_dp
     END IF
     IF(Rc >=  4.2684148594526539E+02_dp .AND. Rc <=  1.3773078918457031E+03_dp) THEN
       irange =  4
       L_b =  4.2684148594526539E+02_dp
       U_b =  1.3773078918457031E+03_dp
     END IF

     fit_coef(1:13,1:12) = coefdata(1:13,1:12,irange)

   END SUBROUTINE get_fit_coef_k06

! *****************************************************************************
!> \brief ...
!> \param Rc ...
!> \param L_b ...
!> \param U_b ...
!> \param fit_coef ...
! *****************************************************************************
   SUBROUTINE get_fit_coef_k07(Rc,L_b,U_b,fit_coef)
    REAL(KIND=dp)                            :: Rc, L_b, U_b
    REAL(KIND=dp), DIMENSION(13, 14)         :: fit_coef

    REAL(KIND=dp), DIMENSION(328), PARAMETER :: c02 = (/ &
      -4.1754731782680632E-02_dp,  6.9637423606893978E-03_dp,&
      9.8317123038661812E-04_dp,  9.3141247748056060E+00_dp,&
      -3.3034187378384154E+00_dp, -2.3099777024265484E+00_dp,&
      1.9795050912537626E+00_dp, -8.5251482053787397E-01_dp,&
      1.2153039011546896E+00_dp, -1.4541564892554977E+00_dp,&
      9.3955684867600964E-01_dp, -4.4317426042703595E-01_dp,&
      2.5918306441105204E-01_dp, -1.4364447330590904E-01_dp,&
      3.6774610849770517E-02_dp,  3.7984486570491994E-05_dp,&
      2.6566070447108867E+01_dp, -1.4519054730075402E+01_dp,&
      -8.2332223680110719E+00_dp,  1.1291093707590436E+01_dp,&
      -5.2875112740441601E+00_dp,  4.8057488315402921E+00_dp,&
      -6.2898759890537734E+00_dp,  4.2232285921976249E+00_dp,&
      -1.0596997995690820E+00_dp, -1.4117925460764269E-01_dp,&
      6.8927735632697190E-02_dp,  1.2553388326253440E-02_dp,&
      1.7986555726404723E-03_dp,  8.9097724809361893E+01_dp,&
      -7.5812556261643451E+01_dp, -2.6441251265412269E+01_dp,&
      7.6120519989047565E+01_dp, -5.0397684172251800E+01_dp,&
      2.5525680254727302E+01_dp, -2.5376314020646312E+01_dp,&
      1.7957148357612915E+01_dp,  3.9328335707666944E+00_dp,&
      -1.7750527732846916E+01_dp,  1.4718739531687156E+01_dp,&
      -6.0048241368590292E+00_dp,  1.0761088627700097E+00_dp,&
      4.0097466825142169E+02_dp, -5.7103639425422421E+02_dp,&
      7.4021376601505452E+01_dp,  6.6329196339694579E+02_dp,&
      -9.6191980645397257E+02_dp,  6.7184174656773678E+02_dp,&
      -1.1693581450498789E+02_dp, -3.7933371221780379E+02_dp,&
      6.7374230959220836E+02_dp, -7.0914904191998016E+02_dp,&
      5.0560984367040209E+02_dp, -2.2277301564215111E+02_dp,&
      4.5529917350575040E+01_dp,  3.2067597521812030E+00_dp,&
      -3.3444886966378912E-01_dp, -3.0363926355787030E-01_dp,&
      8.5884885141460449E-02_dp, -7.6865383844994903E-02_dp,&
      1.3715123754384689E-01_dp, -1.2901078408577635E-01_dp,&
      9.2959485070991041E-02_dp, -6.8861591314818962E-02_dp,&
      4.5033224293921377E-02_dp, -1.7988390790948459E-02_dp,&
      2.2599225511603246E-03_dp,  5.2747450486550899E-04_dp,&
      5.6016069269588744E+00_dp, -1.1356436841662141E+00_dp,&
      -9.3290577892707627E-01_dp,  4.6398361532553123E-01_dp,&
      -2.4608924526924877E-01_dp,  4.5627545506491052E-01_dp,&
      -4.7134496624771294E-01_dp,  3.1401691722393993E-01_dp,&
      -2.0538932052729497E-01_dp,  1.3700800483099954E-01_dp,&
      -5.9637767470295158E-02_dp,  8.4168325815202547E-03_dp,&
      1.8074844968448427E-03_dp,  1.3276336253382626E+01_dp,&
      -4.5209338962192493E+00_dp, -3.3197028028590223E+00_dp,&
      2.4854654336765201E+00_dp, -1.0573849001152691E+00_dp,&
      1.7194404658483133E+00_dp, -1.9805547924035773E+00_dp,&
      1.2692361653686275E+00_dp, -6.7728308285586858E-01_dp,&
      4.3924148750555142E-01_dp, -2.2613532847246806E-01_dp,&
      4.7779438733642737E-02_dp,  2.8573307488057543E-03_dp,&
      3.6866579094980089E+01_dp, -1.8921333920638745E+01_dp,&
      -1.1800048250915063E+01_dp,  1.3436395189426809E+01_dp,&
      -5.6772945221557212E+00_dp,  6.5245530820654025E+00_dp,&
      -8.4720689804565641E+00_dp,  5.4939590398921556E+00_dp,&
      -1.9352146435742505E+00_dp,  7.9538222420964921E-01_dp,&
      -5.7375841633563951E-01_dp,  2.1793383428406723E-01_dp,&
      -1.7879956499238674E-02_dp,  1.1861871394764727E+02_dp,&
      -8.9749853349748633E+01_dp, -4.1428506082732056E+01_dp,&
      8.1467781253253762E+01_dp, -4.2508182713086420E+01_dp,&
      2.7648470829113410E+01_dp, -3.6879292077457670E+01_dp,&
      2.5999309174132279E+01_dp, -1.8359836135406524E+00_dp,&
      -8.8433081142871490E+00_dp,  5.8531631602299123E+00_dp,&
      -1.6443587665309087E+00_dp,  2.0950921411586954E-01_dp,&
      4.7136803623790843E+02_dp, -5.4268938167235092E+02_dp,&
      -9.9473364877013339E+01_dp,  6.2005028905297843E+02_dp,&
      -5.1239886834398976E+02_dp,  2.1442818406738263E+02_dp,&
      -1.2285032837677923E+02_dp,  7.7219547317156128E+01_dp,&
      9.0809838346683208E+01_dp, -2.3266758546949035E+02_dp,&
      2.0624116217858705E+02_dp, -9.2082040963793673E+01_dp,&
      1.7627448485288181E+01_dp,  2.8205675067555067E+03_dp,&
      -5.3084226900396279E+03_dp,  2.4278553483206338E+03_dp,&
      5.7509013750262075E+03_dp, -1.2945226216946787E+04_dp,&
      1.2536477012596763E+04_dp, -3.5209404505394255E+03_dp,&
      -8.8941447689752422E+03_dp,  1.7502570418682619E+04_dp,&
      -1.8136983461264608E+04_dp,  1.2199553060977927E+04_dp,&
      -5.0398280865362967E+03_dp,  9.7440960962504107E+02_dp,&
      3.8829047042903236E-01_dp, -2.0369951393868561E-08_dp,&
      -4.2760469054619193E-02_dp,  3.8145842955685037E-02_dp,&
      -2.3458707726265494E-02_dp, -3.4029805428294910E-02_dp,&
      1.4008388453470724E-01_dp, -2.5114912377498655E-01_dp,&
      2.8125815091946693E-01_dp, -1.8225356246487157E-01_dp,&
      3.1654580106274942E-02_dp,  3.9862347606661845E-02_dp,&
      -2.1625916286328855E-02_dp,  1.4602532069412997E+00_dp,&
      -1.1343456594384706E-07_dp, -2.3779548855070412E-01_dp,&
      2.1213231958215728E-01_dp, -1.1477625538424836E-01_dp,&
      -2.1742813069917741E-01_dp,  8.0954389183928199E-01_dp,&
      -1.3959867402492177E+00_dp,  1.4874447805393780E+00_dp,&
      -8.5261195435033799E-01_dp, -4.8377391586045737E-03_dp,&
      3.3544213557537367E-01_dp, -1.5177913624259395E-01_dp,&
      3.7908369875064492E+00_dp, -4.8755632999722838E-07_dp,&
      -1.0201467453462310E+00_dp,  9.1004690794757048E-01_dp,&
      -3.9887236631988443E-01_dp, -1.1008475868239735E+00_dp,&
      3.6443712251015832E+00_dp, -5.9545892934730142E+00_dp,&
      5.8697667029529539E+00_dp, -2.6303342326824413E+00_dp,&
      -1.1564907871129693E+00_dp,  2.1468635303100569E+00_dp,&
      -8.4605567363145651E-01_dp,  1.0098487510998055E+01_dp,&
      -2.0928481135752870E-06_dp, -4.3686670908442755E+00_dp,&
      3.8971531298032169E+00_dp, -1.1974685442041848E+00_dp,&
      -5.6317683508763983E+00_dp,  1.6448696769067997E+01_dp,&
      -2.5044320019174183E+01_dp,  2.1867904128895628E+01_dp,&
      -5.0681337141096767E+00_dp, -1.1640616685797564E+01_dp,&
      1.3304985336172242E+01_dp, -4.7450643250946332E+00_dp,&
      3.0122997452208132E+01_dp, -9.8187954584462908E-06_dp,&
      -2.0428668438494554E+01_dp,  1.8223634263690474E+01_dp,&
      -2.1117881276751187E+00_dp, -3.2598057750235725E+01_dp,&
      8.1733300202487357E+01_dp, -1.1128759913299137E+02_dp,&
      7.5228856030747480E+01_dp,  2.4248090620667043E+01_dp,&
      -1.0465160278140920E+02_dp,  9.2524007787235988E+01_dp,&
      -3.0355195426460511E+01_dp,  1.0868874503660004E+02_dp,&
      -5.7245956587297789E-05_dp, -1.1851742957082871E+02_dp,&
      1.0572360394609667E+02_dp,  2.3152742557056847E+01_dp,&
      -2.5259994940936937E+02_dp,  5.0697463610979355E+02_dp,&
      -5.3877942879889804E+02_dp,  8.2107193792851831E+01_dp,&
      7.1782122188306062E+02_dp, -1.1851552842814824E+03_dp,&
      8.7548914212122679E+02_dp, -2.6543645341234554E+02_dp,&
      5.6870599977970653E+02_dp, -5.5564345141122099E-04_dp,&
      -1.1496133162535530E+03_dp,  1.0255182144253201E+03_dp,&
      1.1270281983242282E+03_dp, -4.0567389638844206E+03_dp,&
      4.6523831299513085E+03_dp,  9.3327881111406134E+02_dp,&
      -1.3616969369427763E+04_dp,  2.6755200078585309E+04_dp,&
      -2.9000034659790475E+04_dp,  1.7777250492541923E+04_dp,&
      -4.8299772192891851E+03_dp,  3.2824512335777354E+00_dp,&
      -1.9156078888589085E-07_dp, -4.0197908004871374E-01_dp,&
      3.5859795441503123E-01_dp, -2.1359817735657924E-01_dp,&
      -3.3236257745885156E-01_dp,  1.3306435836390915E+00_dp,&
      -2.3614269209662528E+00_dp,  2.6114918833282528E+00_dp,&
      -1.6438816802045650E+00_dp,  2.1906671723756974E-01_dp,&
      4.2427429641952058E-01_dp, -2.1705100617360901E-01_dp,&
      5.8637162691723583E+00_dp, -6.7787456845426279E-07_dp,&
      -1.4194162820318337E+00_dp,  1.2662274814594010E+00_dp,&
      -6.0649584934093481E-01_dp, -1.4391376533593900E+00_dp,&
      4.9819389045221047E+00_dp, -8.3199774717421811E+00_dp,&
      8.4774621253096996E+00_dp, -4.2614106061434578E+00_dp,&
      -9.5354015733314812E-01_dp,  2.5818521234785710E+00_dp,&
      -1.0661797920012148E+00_dp,  1.4340890738155760E+01_dp,&
      -2.8142212154235703E-06_dp, -5.8799168063088727E+00_dp,&
      5.2453046203869578E+00_dp, -1.8855227779627646E+00_dp,&
      -7.0881291395155026E+00_dp,  2.1731282085407450E+01_dp,&
      -3.4078471876229919E+01_dp,  3.1408179116155015E+01_dp,&
      -1.0416315289376390E+01_dp, -1.1856223667791461E+01_dp,&
      1.5589813689555845E+01_dp, -5.7586314676712993E+00_dp,&
      4.1438693013073269E+01_dp, -1.2431014094133345E-05_dp,&
      -2.5901135444548817E+01_dp,  2.3105496129919860E+01_dp,&
      -4.7050262856885112E+00_dp, -3.7691356222507558E+01_dp,&
      1.0121215124632137E+02_dp, -1.4560344087059264E+02_dp,&
      1.1301980711089153E+02_dp,  5.8816837789371679E-01_dp,&
      -1.0167168279243205E+02_dp,  9.8807325076452315E+01_dp,&
      -3.3542146164453939E+01_dp,  1.4116111924805350E+02_dp,&
      -6.3883119768052977E-05_dp, -1.3258808163442185E+02_dp,&
      1.1827603870174343E+02_dp,  3.8519512221529113E+00_dp,&
      -2.4308579826673557E+02_dp,  5.5187570135781425E+02_dp,&
      -6.8463116272698392E+02_dp,  3.3797300275487532E+02_dp,&
      4.1534125658831630E+02_dp, -9.4503311947133648E+02_dp,&
      7.5927098696013218E+02_dp, -2.3947738391807886E+02_dp,&
      6.1740638530300282E+02_dp, -4.4451581514866301E-04_dp,&
      -9.1747212798222540E+02_dp,  8.1842617649364138E+02_dp,&
      3.7908245119295702E+02_dp, -2.3133450738754764E+03_dp,&
      4.0499698509594637E+03_dp, -3.3872426487702342E+03_dp,&
      -1.6589887942666303E+03_dp,  9.1305340362382631E+03_dp,&
      -1.2658983382111219E+04_dp,  8.7808323210291001E+03_dp,&
      -2.5756185231754689E+03_dp,  4.5291383930690972E+03_dp,&
      -6.0796509451681840E-03_dp, -1.2729055327721569E+04_dp,&
      1.1355442435730931E+04_dp,  1.9420434975201191E+04_dp,&
      -5.7157101053784776E+04_dp,  4.1961441009360387E+04_dp,&
      8.2301170473632315E+04_dp, -2.9848869860082364E+05_dp,&
      4.7636758440468600E+05_dp, -4.6114819297381287E+05_dp,&
      2.6149542649940349E+05_dp, -6.7032773576863750E+04_dp /)
    REAL(KIND=dp), DIMENSION(400), PARAMETER :: c01 = (/ &
      2.6444089699692058E-01_dp, -1.5470504999202458E-01_dp,&
      -4.6856578935403119E-02_dp,  2.5042624726789489E-02_dp,&
      1.9131665599864848E-02_dp, -2.9696778036559586E-03_dp,&
      -3.7556543742271655E-03_dp,  1.6266657874186815E-04_dp,&
      4.6360340975112090E-03_dp, -7.8332396642386976E-03_dp,&
      1.9496480970880425E-03_dp,  3.0433369974490975E-03_dp,&
      -1.5191543419188006E-03_dp,  8.7800372636622848E-01_dp,&
      -6.2037241128343046E-01_dp, -8.1566349888960304E-02_dp,&
      1.3641008969898111E-01_dp,  3.1936419084862153E-02_dp,&
      -3.7386849280010785E-02_dp, -1.7482245191148163E-03_dp,&
      9.2077758125164444E-03_dp,  1.5212614028576979E-02_dp,&
      -4.4026339324134335E-02_dp,  3.6890098749264175E-02_dp,&
      -1.1864460175613026E-02_dp,  8.4517974895171439E-04_dp,&
      1.7847969899964811E+00_dp, -1.6779481459284062E+00_dp,&
      2.0639156665955180E-01_dp,  4.1809341533061167E-01_dp,&
      -1.2015867619368460E-01_dp, -1.3848385742735578E-01_dp,&
      7.1455146740269884E-02_dp,  4.7031832850392787E-02_dp,&
      -2.3224446381251464E-02_dp, -6.5420768433947313E-02_dp,&
      8.5668141830795880E-02_dp, -4.2757220494944660E-02_dp,&
      8.2228259261456181E-03_dp,  3.3330019221471758E+00_dp,&
      -4.2118312167575729E+00_dp,  1.6976220198254712E+00_dp,&
      7.5697277244499039E-01_dp, -8.6691818383277741E-01_dp,&
      -1.3679549434400173E-01_dp,  4.0739753850692434E-01_dp,&
      1.3289608237476967E-02_dp, -2.0557302626129631E-01_dp,&
      1.3788895525784092E-02_dp,  1.3789054262037703E-01_dp,&
      -9.7837925769756875E-02_dp,  2.2630562602574813E-02_dp,&
      6.2205176319490292E+00_dp, -1.0220190906809847E+01_dp,&
      7.1086559724074778E+00_dp, -3.5476394951841800E-01_dp,&
      -2.7983732776043846E+00_dp,  1.1308103572081498E+00_dp,&
      9.1862692160779713E-01_dp, -6.5116275075965147E-01_dp,&
      -5.4845428679228692E-01_dp,  8.5393419777208546E-01_dp,&
      -4.0098327169829756E-01_dp,  5.2146858458048319E-02_dp,&
      9.6927653738469108E-03_dp,  1.2217423515701739E+01_dp,&
      -2.4726297120084297E+01_dp,  2.4311598260589175E+01_dp,&
      -1.0091505606342285E+01_dp, -3.4422060817821842E+00_dp,&
      5.9067475855100486E+00_dp, -1.1792990447941114E+00_dp,&
      -1.9379521455346627E+00_dp,  8.8478642486008252E-01_dp,&
      1.0620423409368160E+00_dp, -1.3724468274695665E+00_dp,&
      6.3778052609253422E-01_dp, -1.1561528817453086E-01_dp,&
      2.9227809843211485E+01_dp, -6.7776067894817842E+01_dp,&
      8.1967717455976384E+01_dp, -5.5738618729830407E+01_dp,&
      1.4735306844058032E+01_dp,  8.5839415419898337E+00_dp,&
      -8.4661714956724126E+00_dp,  6.0685765490240084E-01_dp,&
      2.1941720347394860E+00_dp, -2.9568623894341339E-01_dp,&
      -1.1485475801649907E+00_dp,  8.0239930035514562E-01_dp,&
      -1.7857991241725557E-01_dp,  2.1695269471571570E+00_dp,&
      -1.3341984989631275E+00_dp, -3.4389129585042044E-01_dp,&
      2.4095334703936172E-01_dp,  1.4150794439393821E-01_dp,&
      -4.3049372222392836E-02_dp, -2.6067000342438331E-02_dp,&
      5.5273699894245716E-03_dp,  4.3837090204711077E-02_dp,&
      -8.5402788417645176E-02_dp,  4.5304897368381669E-02_dp,&
      5.5864340640867909E-03_dp, -7.5526934617299851E-03_dp,&
      2.8719411494837974E+00_dp, -2.6619899142189949E+00_dp,&
      2.1544205537663841E-01_dp,  7.3077140572600108E-01_dp,&
      -1.1791677396871071E-01_dp, -2.6552626743236452E-01_dp,&
      7.5373709525561841E-02_dp,  8.7066019221526805E-02_dp,&
      3.0676066663530050E-03_dp, -1.7129353323714722E-01_dp,&
      2.1354379443237309E-01_dp, -1.1821773827373153E-01_dp,&
      2.5969541945324099E-02_dp,  4.6027595145407618E+00_dp,&
      -6.3492168966396259E+00_dp,  2.6044413599352203E+00_dp,&
      1.5061915998878004E+00_dp, -1.3747926425853620E+00_dp,&
      -4.7941746387017842E-01_dp,  6.4441114269605160E-01_dp,&
      1.7317222818454989E-01_dp, -3.6457960453035143E-01_dp,&
      2.6412345558134549E-02_dp,  1.7751675692818278E-01_dp,&
      -1.2546750431443543E-01_dp,  3.0158242944704584E-02_dp,&
      8.2140137621645088E+00_dp, -1.5156353495817562E+01_dp,&
      1.0947240829307352E+01_dp,  4.4625153473055351E-01_dp,&
      -5.1300450604063039E+00_dp,  1.1684074353510967E+00_dp,&
      2.1518827109078962E+00_dp, -7.8884326600074284E-01_dp,&
      -1.2573516650555800E+00_dp,  1.0661112404200725E+00_dp,&
      -2.4342415056009679E-02_dp, -2.7206985827086660E-01_dp,&
      9.0849357725157689E-02_dp,  1.5901440158546173E+01_dp,&
      -3.5870927834752429E+01_dp,  3.7033971510712249E+01_dp,&
      -1.3426281591978212E+01_dp, -9.4793547969558798E+00_dp,&
      1.1009953605607825E+01_dp,  2.7169499451768997E-01_dp,&
      -5.1938101275141140E+00_dp,  5.4398901692303922E-02_dp,&
      5.2939817053415910E+00_dp, -4.9957868625718627E+00_dp,&
      2.0763539253174690E+00_dp, -3.4840813951986949E-01_dp,&
      3.5630385206267455E+01_dp, -8.9470008341766714E+01_dp,&
      1.1468454951859270E+02_dp, -7.9123579819767599E+01_dp,&
      1.4519114506197292E+01_dp,  2.4567610944439050E+01_dp,&
      -2.0756718052404963E+01_dp, -2.2859415630517427E-01_dp,&
      1.0041311285714274E+01_dp, -5.9487245509897209E+00_dp,&
      1.8350813953697698E-01_dp,  1.1110319845170011E+00_dp,&
      -3.5005274800008057E-01_dp,  1.3147420367988465E+02_dp,&
      -3.3027880006216265E+02_dp,  4.4490187241967470E+02_dp,&
      -3.6553068151822038E+02_dp,  1.7159265183702419E+02_dp,&
      -1.6947279016681499E+01_dp, -3.1006819193390971E+01_dp,&
      1.4450351194202961E+01_dp,  3.0038030298188554E+00_dp,&
      -2.7207016900807592E+00_dp, -1.8280868626869793E+00_dp,&
      2.0065403080939705E+00_dp, -5.1840780008942378E-01_dp,&
      3.4505624259329698E-01_dp, -8.8324180899744836E-02_dp,&
      -4.6110672571305959E-02_dp,  6.0246988024288736E-03_dp,&
      1.1793333275048282E-02_dp, -8.0133449953961704E-05_dp,&
      -5.3349402958927537E-03_dp,  8.7754046282992803E-04_dp,&
      1.8165729377018357E-03_dp, -1.2767142351947607E-05_dp,&
      -1.4303468914768861E-03_dp,  9.3816945542221544E-04_dp,&
      -2.0328540292083126E-04_dp,  1.2346521989036530E+00_dp,&
      -4.3285647708194658E-01_dp, -1.7305878733158908E-01_dp,&
      7.6461063018067821E-02_dp,  5.3040050433052502E-02_dp,&
      -2.1338549422926928E-02_dp, -2.7841143266162320E-02_dp,&
      1.3809276004079121E-02_dp,  1.2835193248531657E-02_dp,&
      -1.0209736277395076E-02_dp, -2.3155639359263330E-03_dp,&
      4.4602013911147409E-03_dp, -1.2940505553950435E-03_dp,&
      2.9028045455595000E+00_dp, -1.5585305439459514E+00_dp,&
      -3.5847481830556316E-01_dp,  4.6103449323803292E-01_dp,&
      1.1117795227399657E-01_dp, -1.7456928361614407E-01_dp,&
      -7.1484101037929793E-02_dp,  9.9321509433804411E-02_dp,&
      5.0234375484831069E-02_dp, -1.0773630683561888E-01_dp,&
      5.3437356912495085E-02_dp, -6.4215993294399389E-03_dp,&
      -1.6157910177500984E-03_dp,  6.6668490392315718E+00_dp,&
      -5.3944853379232089E+00_dp, -1.7333398494242408E-01_dp,&
      2.0362320110314558E+00_dp, -2.0552463518558933E-01_dp,&
      -8.4623002001901582E-01_dp,  6.6284256825966503E-02_dp,&
      4.8869806045351949E-01_dp,  9.5173683686716991E-03_dp,&
      -5.5847228905099633E-01_dp,  5.0444583260989662E-01_dp,&
      -1.9696925294729772E-01_dp,  3.0305983725112742E-02_dp,&
      1.6122832821281406E+01_dp, -1.8919243474316684E+01_dp,&
      4.0995726503647489E+00_dp,  7.3012627484186252E+00_dp,&
      -4.1313958700040532E+00_dp, -2.5095528548141255E+00_dp,&
      2.1272122489935632E+00_dp,  1.5307151716278145E+00_dp,&
      -1.6669783074449143E+00_dp, -8.7588270967422077E-01_dp,&
      1.9886177370819973E+00_dp, -1.1470721637269370E+00_dp,&
      2.4363731014453902E-01_dp,  4.2621032980490320E+01_dp,&
      -7.0376198214321079E+01_dp,  3.9171087201361452E+01_dp,&
      1.5272113738363034E+01_dp, -3.0728448719316823E+01_dp,&
      6.1171990910524778E+00_dp,  1.2221131447314844E+01_dp,&
      -3.0894759011925577E+00_dp, -1.2362510396133640E+01_dp,&
      1.3678769893431953E+01_dp, -5.7528463567135004E+00_dp,&
      5.7726933215022425E-01_dp,  1.8772765681804807E-01_dp,&
      1.3491831737342127E+02_dp, -2.9806012349757901E+02_dp,&
      2.9841543160447191E+02_dp, -1.1170387310363532E+02_dp,&
      -8.2203315408406809E+01_dp,  1.3165976851638894E+02_dp,&
      -6.6112701554090933E+01_dp,  3.9695001548314552E+00_dp,&
      2.4844454458504051E+00_dp,  1.4202168620692383E+01_dp,&
      -1.7455344455966326E+01_dp,  8.4672335324952464E+00_dp,&
      -1.6173714685616287E+00_dp,  2.8828052690184736E+00_dp,&
      -8.0317040808415729E-01_dp, -3.9338075068484524E-01_dp,&
      7.9426195162603297E-02_dp,  1.0701666183071350E-01_dp,&
      -1.0894403804546973E-02_dp, -5.0613235156913752E-02_dp,&
      1.2364103801681792E-02_dp,  1.8715420448839691E-02_dp,&
      -3.9552027806400514E-03_dp, -1.2146377166990881E-02_dp,&
      9.1425474616888842E-03_dp, -2.1144232652572591E-03_dp,&
      4.5886049198840508E+00_dp, -2.3077795883390579E+00_dp,&
      -6.5366859610516070E-01_dp,  6.2765642178924896E-01_dp,&
      2.3574184645147922E-01_dp, -2.1996460901076612E-01_dp,&
      -1.4269945382294638E-01_dp,  1.2419991538546622E-01_dp,&
      8.4407863785150652E-02_dp, -1.1735824544474278E-01_dp,&
      2.7653602309322248E-02_dp,  1.4931997736797792E-02_dp,&
      -6.8733985726516520E-03_dp,  9.5514395283650053E+00_dp,&
      -7.7961380998560772E+00_dp, -6.5864480154455607E-01_dp,&
      2.9590017002588471E+00_dp,  7.0736288843595091E-02_dp,&
      -1.2427968660363538E+00_dp, -1.3092708858258839E-01_dp,&
      6.9777177541844704E-01_dp,  1.9860466096286125E-01_dp,&
      -8.4883928953606491E-01_dp,  6.1233781394136422E-01_dp,&
      -1.8682451466359717E-01_dp,  1.8956260629113059E-02_dp,&
      2.2681504235495581E+01_dp, -2.6769556369521098E+01_dp,&
      3.8421035220956412E+00_dp,  1.1264519302530468E+01_dp,&
      -4.1146418985136082E+00_dp, -4.6910373462028936E+00_dp,&
      2.0744686686393647E+00_dp,  2.7897820660673656E+00_dp,&
      -1.2053952427752199E+00_dp, -2.8742770555552286E+00_dp,&
      3.6128431894307571E+00_dp, -1.7420891025487704E+00_dp,&
      3.2880124209784622E-01_dp,  5.9399762273351683E+01_dp,&
      -9.6606057719920145E+01_dp,  4.2613160864105112E+01_dp,&
      3.3344679979439910E+01_dp, -3.7404106629648304E+01_dp,&
      -5.9153100847618516E+00_dp,  1.9306590740934482E+01_dp,&
      4.0805389526686682E+00_dp, -1.7468356988844324E+01_dp,&
      6.0010093546906882E+00_dp,  6.3096748376840859E+00_dp,&
      -6.0062912235746344E+00_dp,  1.5620124014024448E+00_dp,&
      1.7618641024115868E+02_dp, -3.8264513931144631E+02_dp,&
      3.2569027453836890E+02_dp, -1.0595168890069804E+01_dp,&
      -2.1449524316195871E+02_dp,  1.4019517892817902E+02_dp,&
      4.6943857262423244E+01_dp, -7.6432309148664586E+01_dp,&
      -4.8967291294613943E+01_dp,  1.4017406628314595E+02_dp,&
      -1.1415196135040772E+02_dp,  4.5334643929331783E+01_dp,&
      -7.5061186786457128E+00_dp,  7.3128032527974494E+02_dp,&
      -1.8965334349732912E+03_dp,  2.4281313649625095E+03_dp,&
      -1.6963332283179839E+03_dp,  2.7274665872075781E+02_dp,&
      7.8978365574188206E+02_dp, -1.0283798449446413E+03_dp,&
      7.3549301335607697E+02_dp, -3.7290787690851232E+02_dp,&
      1.4467625159798999E+02_dp, -4.0346760405500355E+01_dp,&
      5.3828624895447126E+00_dp,  3.3309708393995346E-01_dp,&
      3.8021272284375796E-01_dp, -3.5807245218190974E-02_dp,&
      -3.3016407213619321E-02_dp,  8.2764560291011275E-03_dp,&
      -8.3775496696670056E-03_dp,  1.4694568348773565E-02_dp,&
      -1.3633406730491519E-02_dp,  9.9748664766613742E-03_dp,&
      -7.5058053308662068E-03_dp,  4.8898215686446114E-03_dp,&
      -1.9376750420256020E-03_dp,  2.4578500454910593E-04_dp,&
      5.4135419394418854E-05_dp,  1.4158828018382148E+00_dp,&
      -1.9428018520695578E-01_dp, -1.6862783721434119E-01_dp,&
      6.3605882624844068E-02_dp, -4.3227688915331915E-02_dp,&
      7.9049443390899574E-02_dp, -7.7603914841223490E-02_dp,&
      5.3822690965089216E-02_dp, -3.7801490871684225E-02_dp,&
      2.4957462270798092E-02_dp, -1.0325755989873857E-02_dp,&
      1.3457500127744759E-03_dp,  3.1573603989801897E-04_dp,&
      3.6037055399201039E+00_dp, -8.0542869559173069E-01_dp,&
      -6.3866068626924610E-01_dp,  3.6581611775536099E-01_dp,&
      -1.8032325305523136E-01_dp,  3.1726528871551424E-01_dp,&
      -3.4024223010999666E-01_dp,  2.2409139887689597E-01_dp,&
      -1.3723892898385098E-01_dp,  9.0749009387187926E-02_dp /)
    REAL(KIND=dp), DIMENSION(13, 14, 4), PARAMETER :: &
      coefdata = RESHAPE ((/ c01, c02 /),(/13,14,4/))

    INTEGER                                  :: irange

     IF(Rc <=  2.0000000000000000E+00_dp) Rc =   2.0000000000000000E+00_dp
     IF(Rc >=  3.0293891906738281E+03_dp) Rc =   3.0293891906738281E+03_dp
     IF(Rc >=  2.0000000000000000E+00_dp .AND. Rc <=  1.8737646720921180E+01_dp) THEN
       irange =  1
       L_b =  2.0000000000000000E+00_dp
       U_b =  1.8737646720921180E+01_dp
     END IF
     IF(Rc >=  1.8737646720921180E+01_dp .AND. Rc <=  1.6546209696020762E+02_dp) THEN
       irange =  2
       L_b =  1.8737646720921180E+01_dp
       U_b =  1.6546209696020762E+02_dp
     END IF
     IF(Rc >=  1.6546209696020762E+02_dp .AND. Rc <=  9.7857476979193905E+02_dp) THEN
       irange =  3
       L_b =  1.6546209696020762E+02_dp
       U_b =  9.7857476979193905E+02_dp
     END IF
     IF(Rc >=  9.7857476979193905E+02_dp .AND. Rc <=  3.0293891906738281E+03_dp) THEN
       irange =  4
       L_b =  9.7857476979193905E+02_dp
       U_b =  3.0293891906738281E+03_dp
     END IF

     fit_coef(1:13,1:14) = coefdata(1:13,1:14,irange)

   END SUBROUTINE get_fit_coef_k07

! *****************************************************************************
!> \brief ...
!> \param Rc ...
!> \param L_b ...
!> \param U_b ...
!> \param fit_coef ...
! *****************************************************************************
   SUBROUTINE get_fit_coef_k08(Rc,L_b,U_b,fit_coef)
    REAL(KIND=dp)                            :: Rc, L_b, U_b
    REAL(KIND=dp), DIMENSION(13, 16)         :: fit_coef

    REAL(KIND=dp), DIMENSION(32), PARAMETER :: c03 = (/ &
      -8.9013840273255062E+02_dp,  4.4450456221108550E+02_dp,&
      5.3174323393059228E+02_dp, -1.2592882857295644E+03_dp,&
      1.0557554278716216E+03_dp, -3.5046238641220043E+02_dp,&
      1.2930757301899466E+03_dp,  1.1516939780618657E-03_dp,&
      -1.6608759395740256E+03_dp,  1.4462154462223984E+03_dp,&
      3.8507708508937276E+02_dp, -3.0516713114026320E+03_dp,&
      5.3203711059143534E+03_dp, -4.6213330077407009E+03_dp,&
      -1.3283580700058483E+03_dp,  1.0430200674761059E+04_dp,&
      -1.5296967080846491E+04_dp,  1.1151529076955769E+04_dp,&
      -3.4549406522575855E+03_dp,  9.4857715069098995E+03_dp,&
      1.5676078832282545E-02_dp, -2.2530834004834767E+04_dp,&
      1.9619149559132464E+04_dp,  2.5759373028729122E+04_dp,&
      -7.7020575840804217E+04_dp,  6.3201727988014049E+04_dp,&
      6.8163632238120859E+04_dp, -2.9053341169350612E+05_dp,&
      4.7359098571404989E+05_dp, -4.6694609866564447E+05_dp,&
      2.7229709401159501E+05_dp, -7.2557173401722219E+04_dp /)
    REAL(KIND=dp), DIMENSION(400), PARAMETER :: c01 = (/ &
      2.7310961221813451E-01_dp, -9.5711109831144323E-02_dp,&
      -2.5960297888544426E-02_dp,  7.7258590328211849E-03_dp,&
      5.1368496261163018E-03_dp, -1.1234374052788891E-03_dp,&
      -1.0608869644587510E-03_dp,  4.0213920388401469E-04_dp,&
      3.0775150301006290E-04_dp, -5.8915574825377335E-05_dp,&
      -2.3576009852965474E-04_dp,  1.4943422000932038E-04_dp,&
      -2.7193575609055488E-05_dp,  9.1301533296989179E-01_dp,&
      -3.9165490470431008E-01_dp, -6.4063511602689208E-02_dp,&
      4.7237118624819781E-02_dp,  1.2334980874650106E-02_dp,&
      -9.9115847862352204E-03_dp, -2.3398525854494041E-03_dp,&
      3.1276613598798316E-03_dp,  6.9920456208351215E-04_dp,&
      -1.0397269514394823E-03_dp, -3.7227644159783696E-04_dp,&
      5.7251368824157297E-04_dp, -1.5641106130487642E-04_dp,&
      1.8800921994120694E+00_dp, -1.0923258017805402E+00_dp,&
      -6.4592903048945596E-03_dp,  1.7145991202300620E-01_dp,&
      -1.0258641827216298E-02_dp, -4.0650011661850526E-02_dp,&
      5.9145330731061310E-03_dp,  1.1814015431998023E-02_dp,&
      -1.7645864693727417E-03_dp, -5.1968514483175078E-03_dp,&
      2.5254767142020877E-03_dp,  1.9008839648663282E-04_dp,&
      -2.7310037630009072E-04_dp,  3.5700906437015720E+00_dp,&
      -2.8410095229290957E+00_dp,  4.7566776292824731E-01_dp,&
      4.5078537203369262E-01_dp, -1.7662327214122509E-01_dp,&
      -9.8111714854984447E-02_dp,  5.9791489912834436E-02_dp,&
      2.3988463192324526E-02_dp, -1.8060958844165663E-02_dp,&
      -1.2105234356603977E-02_dp,  1.5101056055278140E-02_dp,&
      -5.3449828416008731E-03_dp,  5.7306130912794712E-04_dp,&
      6.7581167823431345E+00_dp, -7.1699027258228432E+00_dp,&
      2.5445844349889786E+00_dp,  7.6672634398993644E-01_dp,&
      -8.3067733243108621E-01_dp, -5.7644011834428951E-02_dp,&
      2.4822641367814308E-01_dp, -1.4162606641062165E-02_dp,&
      -7.1554288045698092E-02_dp,  1.1874431773050711E-03_dp,&
      4.0456779863544762E-02_dp, -2.4380100288226139E-02_dp,&
      4.9075045448962002E-03_dp,  1.3121698258325543E+01_dp,&
      -1.7881394250826862E+01_dp,  1.0058691417480512E+01_dp,&
      -3.9782510556929929E-01_dp, -2.5566824525318088E+00_dp,&
      8.4735029392079875E-01_dp,  5.3977975375645593E-01_dp,&
      -3.5357733173281275E-01_dp, -1.1643964121149164E-01_dp,&
      1.4323415182866206E-01_dp,  2.7894270359389755E-03_dp,&
      -4.1885479536085533E-02_dp,  1.3600151683161683E-02_dp,&
      2.6983172825423299E+01_dp, -4.5136785690700066E+01_dp,&
      3.5347241038093827E+01_dp, -1.1322977696396405E+01_dp,&
      -3.6432371690369023E+00_dp,  4.5653163916195441E+00_dp,&
      -6.2527689198743497E-01_dp, -1.0316108714043186E+00_dp,&
      3.9002390962416106E-01_dp,  2.7867998432914626E-01_dp,&
      -2.7830279098452576E-01_dp,  8.6406944254303217E-02_dp,&
      -7.7988695959731675E-03_dp,  6.7020985153330088E+01_dp,&
      -1.2895350676499106E+02_dp,  1.2515988271865132E+02_dp,&
      -6.7730535979917050E+01_dp,  1.3509022726826249E+01_dp,&
      7.5870813693372865E+00_dp, -5.6551478844811518E+00_dp,&
      3.1592362611458347E-01_dp,  8.9353909446106417E-01_dp,&
      -5.1419383112248918E-02_dp, -3.4896277335001258E-01_dp,&
      1.9242088216158001E-01_dp, -3.4930665201446769E-02_dp,&
      2.2444409953889672E+00_dp, -8.2981004194755881E-01_dp,&
      -2.0149587021835375E-01_dp,  7.6862324986017819E-02_dp,&
      4.0212219357266481E-02_dp, -1.3088417493763247E-02_dp,&
      -8.3346391656963330E-03_dp,  4.4376518893953151E-03_dp,&
      2.4395035166568041E-03_dp, -9.7210852177202040E-04_dp,&
      -1.8091432752667988E-03_dp,  1.3510459080231529E-03_dp,&
      -2.7978062458459800E-04_dp,  3.0233566510549710E+00_dp,&
      -1.7229666153085565E+00_dp, -5.9154969723083692E-02_dp,&
      2.7883404191218236E-01_dp,  3.3814752466955864E-03_dp,&
      -6.7761866419578015E-02_dp,  2.6604108325230498E-03_dp,&
      2.0289299159528842E-02_dp, -6.7365152413148034E-04_dp,&
      -8.5272902970696760E-03_dp,  2.3135442605826443E-03_dp,&
      1.5922697497338322E-03_dp, -7.4271387725927010E-04_dp,&
      4.9641303680359998E+00_dp, -4.2791867262695549E+00_dp,&
      7.0701632981021489E-01_dp,  7.6974697844881379E-01_dp,&
      -2.5050637135588621E-01_dp, -1.8674240141839671E-01_dp,&
      8.5004325417316742E-02_dp,  4.9721733708245786E-02_dp,&
      -2.5721746300345582E-02_dp, -2.5577126368071773E-02_dp,&
      2.5694288079317958E-02_dp, -7.5350930197050558E-03_dp,&
      4.1016625473918824E-04_dp,  9.0428782023867402E+00_dp,&
      -1.0651609613739524E+01_dp,  3.8210533639574078E+00_dp,&
      1.4421513328672826E+00_dp, -1.2962088562797218E+00_dp,&
      -2.1856889727324386E-01_dp,  4.0769349474573302E-01_dp,&
      2.1778197894256581E-02_dp, -1.2217423363358416E-01_dp,&
      -1.8092159326321852E-02_dp,  7.8958894324477696E-02_dp,&
      -4.2323525382300678E-02_dp,  7.8988389506311936E-03_dp,&
      1.7494968543499823E+01_dp, -2.6299261098424981E+01_dp,&
      1.4981092153446502E+01_dp,  3.5566014675930263E-01_dp,&
      -4.4078347265838902E+00_dp,  9.6789361825124498E-01_dp,&
      1.1548302407139202E+00_dp, -4.8295531452813190E-01_dp,&
      -3.0638711477524511E-01_dp,  1.9053711104052087E-01_dp,&
      9.8530328249180302E-02_dp, -1.1445401656369447E-01_dp,&
      3.0271989036667647E-02_dp,  3.5912642657867430E+01_dp,&
      -6.5373617495328972E+01_dp,  5.2339797574661134E+01_dp,&
      -1.4153218765640851E+01_dp, -9.0572445032884144E+00_dp,&
      7.8281386285875207E+00_dp,  3.6341723094328376E-01_dp,&
      -2.5273798563351466E+00_dp,  3.1250363833851924E-01_dp,&
      1.0318273705347971E+00_dp, -6.6189452520406300E-01_dp,&
      1.1809550079304761E-01_dp,  1.0806151322463294E-02_dp,&
      8.3477139318733393E+01_dp, -1.7156836453569142E+02_dp,&
      1.7336798683819080E+02_dp, -9.2344665032728145E+01_dp,&
      9.9138773578404322E+00_dp,  2.1111102738858747E+01_dp,&
      -1.2703114305540435E+01_dp, -6.8491988017260363E-01_dp,&
      3.7238467452567474E+00_dp, -9.6853869036389029E-01_dp,&
      -8.2076794251227392E-01_dp,  6.5200581790559620E-01_dp,&
      -1.4892747075362087E-01_dp,  3.0901806252841146E+02_dp,&
      -6.4535118104996093E+02_dp,  7.0059421765500099E+02_dp,&
      -4.6233013645500171E+02_dp,  1.7229734254873787E+02_dp,&
      -1.0791044680098347E+01_dp, -2.2384185132904548E+01_dp,&
      8.6880396981906074E+00_dp,  4.8992454334398755E-01_dp,&
      -2.1707249656643718E-01_dp, -1.0500007845801604E+00_dp,&
      7.1848860832775263E-01_dp, -1.5485197147846710E-01_dp,&
      3.3508729161701295E-01_dp, -6.5967375280714266E-02_dp,&
      -3.6671191882035471E-02_dp,  3.7463544686884482E-03_dp,&
      8.5622563668911605E-03_dp,  6.8363383574831626E-04_dp,&
      -4.5035979562529508E-03_dp,  6.6279905673241254E-04_dp,&
      1.5024716048307220E-03_dp, -1.2576817257599221E-04_dp,&
      -1.0433416780076630E-03_dp,  7.2426679548123546E-04_dp,&
      -1.6273288963037326E-04_dp,  1.1864553104914806E+00_dp,&
      -3.1461997524782426E-01_dp, -1.4693892529281682E-01_dp,&
      4.5646442517946223E-02_dp,  4.1088276571797845E-02_dp,&
      -7.6528941123103732E-03_dp, -2.4112522507619763E-02_dp,&
      8.1014486563994372E-03_dp,  9.9635648532678794E-03_dp,&
      -5.5731398359707141E-03_dp, -3.2969694635874730E-03_dp,&
      3.7600953456707166E-03_dp, -9.9587115962167325E-04_dp,&
      2.7324626612556053E+00_dp, -1.0908886663262836E+00_dp,&
      -3.7452568962784560E-01_dp,  2.7448650188755308E-01_dp,&
      1.1947510898542109E-01_dp, -8.1512972680725215E-02_dp,&
      -8.2669082784466011E-02_dp,  5.5843294460943604E-02_dp,&
      4.4965144102735111E-02_dp, -5.5709333452207237E-02_dp,&
      1.2128490666523937E-02_dp,  7.2438183397116164E-03_dp,&
      -3.2102466121958495E-03_dp,  6.0892254495015932E+00_dp,&
      -3.6163036866884251E+00_dp, -7.2954415660089122E-01_dp,&
      1.2449867999159732E+00_dp,  1.8627074862998558E-01_dp,&
      -4.5836716242884368E-01_dp, -1.8726612737773107E-01_dp,&
      2.9412452507008080E-01_dp,  1.4464334222064024E-01_dp,&
      -3.5797443880437330E-01_dp,  2.1243449070641837E-01_dp,&
      -4.7979547413800436E-02_dp,  1.1430581477758399E-03_dp,&
      1.4139026562869708E+01_dp, -1.2041501426557486E+01_dp,&
      -3.5767048473142754E-01_dp,  4.9535087968389506E+00_dp,&
      -6.9859714731516531E-01_dp, -1.9604651106042494E+00_dp,&
      8.7491811796409316E-02_dp,  1.3013744792257471E+00_dp,&
      5.7319425555596631E-02_dp, -1.5953469275658381E+00_dp,&
      1.4745984158804955E+00_dp, -5.9602490347273818E-01_dp,&
      9.6024900648760458E-02_dp,  3.5251429422184806E+01_dp,&
      -4.1992637583344994E+01_dp,  8.2569840030412731E+00_dp,&
      1.7766017038241735E+01_dp, -9.9409207406392266E+00_dp,&
      -5.7296196204934073E+00_dp,  4.4447226214233817E+00_dp,&
      4.3695801257258298E+00_dp, -4.0443214386948814E+00_dp,&
      -2.8548551132553746E+00_dp,  5.6567089023402772E+00_dp,&
      -3.2060726305631433E+00_dp,  6.7793233648081430E-01_dp,&
      9.7332609213038637E+01_dp, -1.6055678095075575E+02_dp,&
      8.3326641397475115E+01_dp,  4.5224931903958627E+01_dp,&
      -7.6146167520984406E+01_dp,  1.3010701305137788E+01_dp,&
      2.8386463320181605E+01_dp, -1.6702334571147881E+00_dp,&
      -3.5079103558906922E+01_dp,  3.3344889931611391E+01_dp,&
      -1.1144196235639770E+01_dp, -6.2677207691225156E-01_dp,&
      9.4490594623257140E-01_dp,  3.2494740113105559E+02_dp,&
      -7.2364022403862566E+02_dp,  7.0784514785353440E+02_dp,&
      -2.1766148619497469E+02_dp, -2.8011046892812595E+02_dp,&
      3.9157870616328319E+02_dp, -2.0111436008133452E+02_dp,&
      2.6746934497255534E+01_dp, -5.1250779800289745E+00_dp,&
      4.5207247760493551E+01_dp, -4.9344510991644917E+01_dp,&
      2.2933203884729071E+01_dp, -4.2463467974507454E+00_dp,&
      2.7924787903755193E+00_dp, -5.9557486085327938E-01_dp,&
      -3.1708788737155785E-01_dp,  4.8363147353357339E-02_dp,&
      7.8353097117355233E-02_dp,  9.0057144008685636E-04_dp,&
      -4.2436219791212565E-02_dp,  8.2581734545659341E-03_dp,&
      1.5009431826959475E-02_dp, -3.1333769326353706E-03_dp,&
      -9.0848760047894626E-03_dp,  6.8972370390152915E-03_dp,&
      -1.6104367213749559E-03_dp,  4.3349564546588484E+00_dp,&
      -1.6338639472670711E+00_dp, -6.2039446459744241E-01_dp,&
      3.7176531406914165E-01_dp,  2.0468278416324942E-01_dp,&
      -9.7117001036191072E-02_dp, -1.3346018024426126E-01_dp,&
      6.9248942972530952E-02_dp,  6.5294776930919118E-02_dp,&
      -5.9636007438036251E-02_dp, -2.7144286812367025E-03_dp,&
      1.8580653270522128E-02_dp, -5.8846609276546023E-03_dp,&
      8.7127006245330918E+00_dp, -5.2814575268392803E+00_dp,&
      -1.2377871952109918E+00_dp,  1.7721827561018202E+00_dp,&
      4.1504199215701543E-01_dp, -6.2649087632354694E-01_dp,&
      -3.5476660021691170E-01_dp,  3.9492435289379613E-01_dp,&
      2.4669057876623324E-01_dp, -4.5374450788098053E-01_dp,&
      2.0534450489776798E-01_dp, -1.4337629279395947E-02_dp,&
      -9.5633384489265550E-03_dp,  1.9864553062133062E+01_dp,&
      -1.7268416697475640E+01_dp, -1.2605611184415808E+00_dp,&
      7.1361682987646899E+00_dp, -2.8155468494418773E-01_dp,&
      -2.8885437478337219E+00_dp, -3.2115843314446185E-01_dp,&
      1.8474192462862677E+00_dp,  5.0459240430275332E-01_dp,&
      -2.4256645534485304E+00_dp,  1.8935550944143869E+00_dp,&
      -6.5130637173581107E-01_dp,  8.4425479665035150E-02_dp,&
      4.9415943100591043E+01_dp, -5.8710018731217680E+01_dp,&
      7.6100350158161136E+00_dp,  2.6437526205424952E+01_dp,&
      -1.0210049310126077E+01_dp, -1.0364129503643207E+01_dp,&
      4.2703723895971040E+00_dp,  7.2550799233044732E+00_dp,&
      -2.8007257975971140E+00_dp, -7.9321546402115857E+00_dp,&
      9.8731611373346890E+00_dp, -4.8000107616726417E+00_dp,&
      9.1625956317232737E-01_dp,  1.3509570103059647E+02_dp,&
      -2.1615670162551694E+02_dp,  8.7042223042420432E+01_dp,&
      8.5074953085441251E+01_dp, -8.7680431005856249E+01_dp,&
      -1.4573782688795355E+01_dp,  4.1553388972824948E+01_dp,&
      1.5857868513717005E+01_dp, -4.3825384512915626E+01_dp,&
      1.1242194914554007E+01_dp,  1.9564845495143668E+01_dp,&
      -1.6811781491598083E+01_dp,  4.2614301447710945E+00_dp,&
      4.2182283329260298E+02_dp, -9.0232374730691106E+02_dp,&
      7.1913307941971050E+02_dp,  6.5788926678577425E+01_dp,&
      -5.6847305624230580E+02_dp,  3.2522974861279704E+02_dp,&
      1.3143394779042333E+02_dp, -1.4623977249362667E+02_dp,&
      -1.9485496537035539E+02_dp,  3.9309282560397878E+02_dp /)
    REAL(KIND=dp), DIMENSION(400), PARAMETER :: c02 = (/ &
      -2.8863592203480596E+02_dp,  1.0528430973221714E+02_dp,&
      -1.5857007223981109E+01_dp,  1.8276946029107571E+03_dp,&
      -4.8112032981737193E+03_dp,  6.1258592600600687E+03_dp,&
      -4.0381542724034271E+03_dp,  3.2440486268768652E+01_dp,&
      2.9756759864086989E+03_dp, -3.5826087401832801E+03_dp,&
      2.5220758904019117E+03_dp, -1.1725016022892619E+03_dp,&
      2.9714869827611420E+02_dp,  3.6408367716617519E+01_dp,&
      -6.1718990215158165E+01_dp,  1.6819583425152295E+01_dp,&
      3.6008296661068129E-01_dp, -2.4934495356088673E-02_dp,&
      -2.6462026390146498E-02_dp,  9.5876084101441991E-03_dp,&
      -1.1331107464939878E-02_dp,  1.6586262055557337E-02_dp,&
      -1.6106632470944200E-02_dp,  1.2751304928977305E-02_dp,&
      -8.6732664212007485E-03_dp,  4.0240760309193475E-03_dp,&
      -4.4825706457146837E-04_dp, -6.1748144054248283E-04_dp,&
      2.5357640280289075E-04_dp,  1.3098679540438654E+00_dp,&
      -1.2749258837080399E-01_dp, -1.3073323485317337E-01_dp,&
      5.8516929367271715E-02_dp, -5.6951328739513037E-02_dp,&
      8.4819766678372446E-02_dp, -8.3438172999370994E-02_dp,&
      6.3534150436398582E-02_dp, -4.0789452590645821E-02_dp,&
      1.7207516063066464E-02_dp,  2.7854405286715241E-05_dp,&
      -4.2694378230801753E-03_dp,  1.5551774834561508E-03_dp,&
      3.1817466966750025E+00_dp, -4.8685729139374834E-01_dp,&
      -4.7508372620822747E-01_dp,  2.7226827027853312E-01_dp,&
      -2.1644926109007404E-01_dp,  3.2162422597469387E-01_dp,&
      -3.2373129824840324E-01_dp,  2.3402591064647971E-01_dp,&
      -1.3552991687865093E-01_dp,  4.6569436455071500E-02_dp,&
      1.2573171658256766E-02_dp, -2.1996218996840747E-02_dp,&
      7.2391100075863378E-03_dp,  7.6667308457077841E+00_dp,&
      -1.8064423526176003E+00_dp, -1.6558509837359154E+00_dp,&
      1.2167933969012499E+00_dp, -8.2685269049351195E-01_dp,&
      1.1702629066210879E+00_dp, -1.2160740673457533E+00_dp,&
      8.3129428882777301E-01_dp, -4.0346334453079419E-01_dp,&
      7.6181979654980650E-02_dp,  1.0538509979804817E-01_dp,&
      -1.0594678252053578E-01_dp,  3.2100908588300212E-02_dp,&
      1.9789605527050799E+01_dp, -6.9317551385828775E+00_dp,&
      -5.8179930349634565E+00_dp,  5.6410261132198896E+00_dp,&
      -3.4720604618155404E+00_dp,  4.3222477406821946E+00_dp,&
      -4.6593280205438399E+00_dp,  2.9973049622920191E+00_dp,&
      -9.8458606194260101E-01_dp, -3.0033580201486476E-01_dp,&
      7.5698139555564226E-01_dp, -5.3686026095251194E-01_dp,&
      1.4785400367332710E-01_dp,  5.7071028866399622E+01_dp,&
      -2.9401869129872736E+01_dp, -2.1376956390611706E+01_dp,&
      2.9364241928690973E+01_dp, -1.8013055669371493E+01_dp,&
      1.7254148374056300E+01_dp, -1.8753111107981081E+01_dp,&
      1.1159554233110121E+01_dp, -1.2752966385306053E-01_dp,&
      -6.1325355952629250E+00_dp,  6.3822854246393721E+00_dp,&
      -3.4517625645107035E+00_dp,  8.3020380724742204E-01_dp,&
      1.9521201133501947E+02_dp, -1.5220438343077086E+02_dp,&
      -8.1912358679862876E+01_dp,  1.9161075517584456E+02_dp,&
      -1.3980358819312374E+02_dp,  8.7370038717781611E+01_dp,&
      -7.2988008432725636E+01_dp,  3.0136598201372607E+01_dp,&
      4.4492815957746984E+01_dp, -9.0138493143008020E+01_dp,&
      7.8015315638660624E+01_dp, -3.7190554397321939E+01_dp,&
      7.9761307531966610E+00_dp,  9.1278852097032075E+02_dp,&
      -1.1881974595454724E+03_dp, -1.2420992251855267E+02_dp,&
      1.8801289344962393E+03_dp, -2.4556997768600449E+03_dp,&
      1.5365426797607518E+03_dp,  1.0676304658797669E+02_dp,&
      -1.6868122887497107E+03_dp,  2.7261083335976618E+03_dp,&
      -2.8690190040817129E+03_dp,  2.0730030060640806E+03_dp,&
      -9.2805090617485291E+02_dp,  1.9285336281750696E+02_dp,&
      3.0202120791487510E+00_dp, -2.2929050737844739E-01_dp,&
      -2.4117235349996635E-01_dp,  9.2704544171716446E-02_dp,&
      -1.0360124397461544E-01_dp,  1.5260718160236281E-01_dp,&
      -1.4863649841980880E-01_dp,  1.1646276459958013E-01_dp,&
      -7.8115523264564363E-02_dp,  3.5445051921955989E-02_dp,&
      -3.0265424876823456E-03_dp, -6.2106108531044283E-03_dp,&
      2.4560375548071142E-03_dp,  4.9977115377020143E+00_dp,&
      -7.0713497336708797E-01_dp, -7.0201845826538045E-01_dp,&
      3.7205228555849740E-01_dp, -3.1239914375103917E-01_dp,&
      4.6962581162570477E-01_dp, -4.6809922294208084E-01_dp,&
      3.4405231198789693E-01_dp, -2.0774765511908164E-01_dp,&
      7.7981264100472250E-02_dp,  1.1926822036550332E-02_dp,&
      -2.9234106504804597E-02_dp,  9.9111598962051163E-03_dp,&
      1.0982352744981107E+01_dp, -2.5586757519844334E+00_dp,&
      -2.3924834863157098E+00_dp,  1.6377235454994263E+00_dp,&
      -1.1452343882258973E+00_dp,  1.6742775852013179E+00_dp,&
      -1.7208177689510733E+00_dp,  1.1933688801441626E+00_dp,&
      -6.2068748093326198E-01_dp,  1.5757464118201009E-01_dp,&
      1.2092332567847532E-01_dp, -1.3968280008702197E-01_dp,&
      4.3442636832466526E-02_dp,  2.7798834753362176E+01_dp,&
      -9.5235310673252087E+00_dp, -8.2506504296193413E+00_dp,&
      7.3158471930055118E+00_dp, -4.5365544170813825E+00_dp,&
      6.0400318657414305E+00_dp, -6.4510290250044848E+00_dp,&
      4.2170504980565306E+00_dp, -1.6565364294458416E+00_dp,&
      -6.7984032425921506E-02_dp,  8.3432875689469077E-01_dp,&
      -6.6751055992496788E-01_dp,  1.9155267749846455E-01_dp,&
      7.8875815934450927E+01_dp, -3.8326646554760494E+01_dp,&
      -2.9647322447318992E+01_dp,  3.5592030615395316E+01_dp,&
      -2.1068887318345901E+01_dp,  2.3036574856677053E+01_dp,&
      -2.5457687546678919E+01_dp,  1.5603470262769573E+01_dp,&
      -2.6161630333998538E+00_dp, -4.7974113923994901E+00_dp,&
      6.0888371971161597E+00_dp, -3.6509041918220748E+00_dp,&
      9.3827458104319261E-01_dp,  2.5820747118786790E+02_dp,&
      -1.7912518679501466E+02_dp, -1.1382161244744694E+02_dp,&
      2.0430275857235807E+02_dp, -1.3010643642425052E+02_dp,&
      1.0051327222541408E+02_dp, -1.0601266064614593E+02_dp,&
      5.8288692345255235E+01_dp,  2.1127565037978961E+01_dp,&
      -6.5273552596841654E+01_dp,  5.8416007114044838E+01_dp,&
      -2.8627550070207121E+01_dp,  6.3938023439562945E+00_dp,&
      1.0530804258036487E+03_dp, -1.0937614495281734E+03_dp,&
      -4.3098080435867519E+02_dp,  1.5667138220310976E+03_dp,&
      -1.3165673699354647E+03_dp,  6.6347634454225249E+02_dp,&
      -3.4185303502890736E+02_dp, -1.1466724937319730E+01_dp,&
      6.1723480049892203E+02_dp, -1.0180911237826692E+03_dp,&
      8.6238446090989885E+02_dp, -4.0590511772790057E+02_dp,&
      8.5233625649272710E+01_dp,  6.6217329112409152E+03_dp,&
      -1.1425351525376182E+04_dp,  2.2492305374254370E+03_dp,&
      1.8881455299767505E+04_dp, -3.4404996456886765E+04_dp,&
      2.7559397466881259E+04_dp,  2.4019070309659278E+03_dp,&
      -3.9817790294105704E+04_dp,  6.4303769223224357E+04_dp,&
      -6.3967990080210388E+04_dp,  4.2899287320689975E+04_dp,&
      -1.7914528413855718E+04_dp,  3.5182149345345074E+03_dp,&
      3.6486926164490469E-01_dp,  2.0204681876392647E-08_dp,&
      -2.9063033115982715E-02_dp,  2.5307089232711903E-02_dp,&
      -1.7387851050024779E-02_dp, -1.1298488481693316E-02_dp,&
      6.1784825098588868E-02_dp, -1.1326788530175186E-01_dp,&
      1.2736917040891757E-01_dp, -8.2068207035220206E-02_dp,&
      1.1361290138449257E-02_dp,  2.2817399150212796E-02_dp,&
      -1.2307392917240636E-02_dp,  1.3345106038416172E+00_dp,&
      1.0473100168474207E-07_dp, -1.5067373422508695E-01_dp,&
      1.3120139232313632E-01_dp, -8.3771660636503986E-02_dp,&
      -6.9708760654556362E-02_dp,  3.3279408293031759E-01_dp,&
      -5.9029754060722750E-01_dp,  6.3961521945451127E-01_dp,&
      -3.7704261134121214E-01_dp,  1.4222371809919012E-03_dp,&
      1.5638278661184743E-01_dp, -7.4976139536898731E-02_dp,&
      3.2767572317506888E+00_dp,  4.0757791931997376E-07_dp,&
      -5.8650748464803737E-01_dp,  5.1070951349780291E-01_dp,&
      -2.9175849747269827E-01_dp, -3.3130653070655985E-01_dp,&
      1.3602786476515449E+00_dp, -2.3079473886452635E+00_dp,&
      2.3670793977651505E+00_dp, -1.1932637133271387E+00_dp,&
      -3.1535910527251165E-01_dp,  8.1970673219695112E-01_dp,&
      -3.5341210513182930E-01_dp,  8.0233464032319297E+00_dp,&
      1.5469932072109146E-06_dp, -2.2267438908414117E+00_dp,&
      1.9389654319071237E+00_dp, -9.5066660364792799E-01_dp,&
      -1.5320952612124052E+00_dp,  5.4440801207413720E+00_dp,&
      -8.7625182894010312E+00_dp,  8.3459802828849448E+00_dp,&
      -3.1789409559564277E+00_dp, -2.7443202471077637E+00_dp,&
      4.1173052329751831E+00_dp, -1.6328471027246423E+00_dp,&
      2.1178994737488193E+01_dp,  6.1163444588653944E-06_dp,&
      -8.8069616938961222E+00_dp,  7.6687600580722854E+00_dp,&
      -2.9352571881725296E+00_dp, -7.4996883609505600E+00_dp,&
      2.2879137403427048E+01_dp, -3.4324990064612756E+01_dp,&
      2.9082103747106494E+01_dp, -4.8140937848072012E+00_dp,&
      -1.9508713405196168E+01_dp,  2.1824003337870533E+01_dp,&
      -8.0461535482045914E+00_dp,  6.3097771483862608E+01_dp,&
      2.7109366339854733E-05_dp, -3.9053754979821946E+01_dp,&
      3.4006411809445268E+01_dp, -7.5765986227444024E+00_dp,&
      -4.2752431653712904E+01_dp,  1.0918545502277176E+02_dp,&
      -1.4682601957283225E+02_dp,  9.7806062121677598E+01_dp,&
      3.5785765692400041E+01_dp, -1.4814419919148372E+02_dp,&
      1.3542279919125832E+02_dp, -4.6600546953699642E+01_dp,&
      2.2763840675790064E+02_dp,  1.5137505265291369E-04_dp,&
      -2.1821644080557203E+02_dp,  1.9001333141048849E+02_dp,&
      1.1018780585175488E+01_dp, -3.3195327790129170E+02_dp,&
      6.6654678166863403E+02_dp, -7.1329548357082911E+02_dp,&
      1.5569450837479587E+02_dp,  8.4955015710139378E+02_dp,&
      -1.4931577695169062E+03_dp,  1.1595826219001069E+03_dp,&
      -3.7132444207641549E+02_dp,  1.1910914431216361E+03_dp,&
      1.4263721165942622E-03_dp, -2.0551073023908939E+03_dp,&
      1.7895023674272838E+03_dp,  1.4236436615084440E+03_dp,&
      -5.4213598658487108E+03_dp,  6.4193939031260470E+03_dp,&
      -4.3652668119034223E+02_dp, -1.3307350777633279E+04_dp,&
      2.8004934478322662E+04_dp, -3.1609945290997650E+04_dp,&
      2.0246981620964787E+04_dp, -5.7903787141059511E+03_dp,&
      3.0643056444925736E+00_dp,  1.8646661584793930E-07_dp,&
      -2.6823124660733616E-01_dp,  2.3356648801342425E-01_dp,&
      -1.5749179181334688E-01_dp, -1.0949264322249078E-01_dp,&
      5.7614538690757122E-01_dp, -1.0470107757684963E+00_dp,&
      1.1661521479359405E+00_dp, -7.3514781124312867E-01_dp,&
      7.8269126791499333E-02_dp,  2.2825560761526997E-01_dp,&
      -1.1877863457204990E-01_dp,  5.1352531685543763E+00_dp,&
      5.8811945999022206E-07_dp, -8.4624001750433542E-01_dp,&
      7.3687550809193658E-01_dp, -4.3833952146723459E-01_dp,&
      -4.4767431890582660E-01_dp,  1.9312272697015882E+00_dp,&
      -3.3286199469950279E+00_dp,  3.4839435414394124E+00_dp,&
      -1.8685419680693067E+00_dp, -2.8596457697133809E-01_dp,&
      1.0725167156433020E+00_dp, -4.7794242799465891E-01_dp,&
      1.1485629169810354E+01_dp,  2.1753987536946991E-06_dp,&
      -3.1309987505895300E+00_dp,  2.7263579761484240E+00_dp,&
      -1.4087375048075199E+00_dp, -2.0285000971994687E+00_dp,&
      7.5358003620778735E+00_dp, -1.2345857753739226E+01_dp,&
      1.2071996259175435E+01_dp, -5.1411109695351618E+00_dp,&
      -3.1071844239563102E+00_dp,  5.3071010378293604E+00_dp,&
      -2.1574669369328108E+00_dp,  2.9697399912139918E+01_dp,&
      8.3135628316181069E-06_dp, -1.1969267103599444E+01_dp,&
      1.0422379107290249E+01_dp, -4.4016512622802200E+00_dp,&
      -9.4725197213611363E+00_dp,  3.0480241673650344E+01_dp,&
      -4.6979770727514214E+01_dp,  4.1763601254545264E+01_dp,&
      -1.0744664685407409E+01_dp, -2.1930155353081439E+01_dp,&
      2.6765835808358442E+01_dp, -1.0113169118847527E+01_dp,&
      8.6657911400504887E+01_dp,  3.4682726546868686E-05_dp,&
      -4.9954155938197879E+01_dp,  4.3498076605632711E+01_dp,&
      -1.2732165518503718E+01_dp, -4.9378251472346385E+01_dp,&
      1.3584067834177924E+02_dp, -1.9220755204893482E+02_dp,&
      1.4478597281785360E+02_dp,  1.1301880394535623E+01_dp,&
      -1.5313609799645960E+02_dp,  1.5075591149117258E+02_dp,&
      -5.3325987923237058E+01_dp,  2.9560124699329253E+02_dp,&
      1.7113342586626282E-04_dp, -2.4662390250801528E+02_dp,&
      2.1474962274139023E+02_dp, -2.0392071197782663E+01_dp,&
      -3.1789596824276714E+02_dp,  7.2496736168975849E+02_dp /)
    REAL(KIND=dp), DIMENSION(13, 16, 4), PARAMETER :: &
      coefdata = RESHAPE ((/ c01, c02, c03 /),(/13,16,4/))

    INTEGER                                  :: irange

     IF(Rc <=  1.0000000000000000E+01_dp) Rc =   1.0000000000000000E+01_dp
     IF(Rc >=  6.3447246551513672E+03_dp) Rc =   6.3447246551513672E+03_dp
     IF(Rc >=  1.0000000000000000E+01_dp .AND. Rc <=  4.8301780076010026E+01_dp) THEN
       irange =  1
       L_b =  1.0000000000000000E+01_dp
       U_b =  4.8301780076010026E+01_dp
     END IF
     IF(Rc >=  4.8301780076010026E+01_dp .AND. Rc <=  4.3944134912624889E+02_dp) THEN
       irange =  2
       L_b =  4.8301780076010026E+01_dp
       U_b =  4.3944134912624889E+02_dp
     END IF
     IF(Rc >=  4.3944134912624889E+02_dp .AND. Rc <=  2.4651524781467556E+03_dp) THEN
       irange =  3
       L_b =  4.3944134912624889E+02_dp
       U_b =  2.4651524781467556E+03_dp
     END IF
     IF(Rc >=  2.4651524781467556E+03_dp .AND. Rc <=  6.3447246551513672E+03_dp) THEN
       irange =  4
       L_b =  2.4651524781467556E+03_dp
       U_b =  6.3447246551513672E+03_dp
     END IF

     fit_coef(1:13,1:16) = coefdata(1:13,1:16,irange)

   END SUBROUTINE get_fit_coef_k08

! *****************************************************************************
!> \brief ...
!> \param Rc ...
!> \param L_b ...
!> \param U_b ...
!> \param fit_coef ...
! *****************************************************************************
   SUBROUTINE get_fit_coef_k09(Rc,L_b,U_b,fit_coef)
    REAL(KIND=dp)                            :: Rc, L_b, U_b
    REAL(KIND=dp), DIMENSION(13, 18)         :: fit_coef

    REAL(KIND=dp), DIMENSION(136), PARAMETER :: c03 = (/ &
      -8.4609670938814907E+02_dp,  5.2521889455815938E+02_dp,&
      -1.8158803681882837E+01_dp, -1.9195843056246903E+02_dp,&
      4.7038182235191250E+01_dp,  4.4962618821808661E+01_dp,&
      2.3952550076762327E+03_dp,  7.3919658606953450E-04_dp,&
      -3.3678854403725500E+03_dp,  2.7630103823335253E+03_dp,&
      1.5760188487094081E+03_dp, -6.2507126178305934E+03_dp,&
      7.5762009915563758E+03_dp, -3.4997817421370810E+03_dp,&
      -3.6562358282784421E+03_dp,  6.5885187381838932E+03_dp,&
      -1.0669477891854467E+03_dp, -5.1923485018358997E+03_dp,&
      3.6296377875939629E+03_dp,  2.8852081079160179E+00_dp,&
      4.0102719961273801E-08_dp, -1.7659002023480172E-01_dp,&
      1.4488106202271947E-01_dp, -1.0039965813412362E-01_dp,&
      -2.5605452610356936E-02_dp,  2.2190893738774328E-01_dp,&
      -3.9868714638912034E-01_dp,  4.1666545240789737E-01_dp,&
      -2.0838248751367475E-01_dp, -7.8702085558092494E-02_dp,&
      1.8499348107948621E-01_dp, -8.5003584077023686E-02_dp,&
      4.5954934072634712E+00_dp,  1.1523486520096986E-07_dp,&
      -5.0806402051936073E-01_dp,  4.1683398596614762E-01_dp,&
      -2.6579259191602511E-01_dp, -1.1173061386143904E-01_dp,&
      6.8157899224885998E-01_dp, -1.1725048395640862E+00_dp,&
      1.1888726238653182E+00_dp, -5.7031447220490772E-01_dp,&
      -2.3373683427653494E-01_dp,  5.0947643811010013E-01_dp,&
      -2.2753724549285903E-01_dp,  9.5744538319119368E+00_dp,&
      3.8935451948906365E-07_dp, -1.7187840838704733E+00_dp,&
      1.4101497165394266E+00_dp, -8.2424578220474254E-01_dp,&
      -5.0165771967854866E-01_dp,  2.4411624004207271E+00_dp,&
      -4.0381205898500108E+00_dp,  3.9746616048513781E+00_dp,&
      -1.8240116710035414E+00_dp, -8.1274767364034362E-01_dp,&
      1.6405984125323814E+00_dp, -7.0946369768700668E-01_dp,&
      2.2756152830064220E+01_dp,  1.3380012629757024E-06_dp,&
      -5.9153103075446083E+00_dp,  4.8531141769756569E+00_dp,&
      -2.5426282738934458E+00_dp, -2.2119329262709027E+00_dp,&
      8.9099642025165444E+00_dp, -1.4124325804320934E+01_dp,&
      1.3421000524921119E+01_dp, -5.8143931761506558E+00_dp,&
      -2.8750831570457684E+00_dp,  5.2778875665252425E+00_dp,&
      -2.1815020322245369E+00_dp,  5.9577031872248192E+01_dp,&
      4.8466710971685504E-06_dp, -2.1470062487220464E+01_dp,&
      1.7614691738721081E+01_dp, -7.8543518759744195E+00_dp,&
      -1.0297543487661295E+01_dp,  3.4589595106069012E+01_dp,&
      -5.2023617051867674E+01_dp,  4.7104796520358647E+01_dp,&
      -1.8668699591613251E+01_dp, -1.0751139468718122E+01_dp,&
      1.7201278360974712E+01_dp, -6.5744756865732210E+00_dp,&
      1.7421142487352890E+02_dp,  1.9480399941922716E-05_dp,&
      -8.6547598834693005E+01_dp,  7.1005987098306903E+01_dp,&
      -2.3983059370901209E+01_dp, -5.4191420793490103E+01_dp,&
      1.5104757590596600E+02_dp, -2.1163527408094700E+02_dp,&
      1.7786418457885577E+02_dp, -5.9642025337733756E+01_dp,&
      -4.4718701051976034E+01_dp,  5.6683066711160087E+01_dp,&
      -1.8062035214880027E+01_dp,  5.9442761921776025E+02_dp,&
      9.3312420949589195E-05_dp, -4.1653329002850757E+02_dp,&
      3.4173289748064514E+02_dp, -5.8910745159325486E+01_dp,&
      -3.5416677215144733E+02_dp,  8.0179526534346644E+02_dp,&
      -1.0066258052744280E+03_dp,  7.3393741106438824E+02_dp,&
      -1.5060845806689406E+02_dp, -2.2080860417911666E+02_dp,&
      1.6107768368211609E+02_dp, -1.4997422612735614E+01_dp,&
      2.6003372356118971E+03_dp,  6.1017236225054279E-04_dp,&
      -2.7485887342448536E+03_dp,  2.2549733577908974E+03_dp,&
      2.9333015363387631E+02_dp, -3.4638286965522780E+03_dp,&
      5.9673723572603731E+03_dp, -5.9287627550561756E+03_dp,&
      2.6547680091869743E+03_dp,  1.0828729490854735E+03_dp,&
      -1.4147281790223578E+03_dp, -6.5594324803846439E+02_dp,&
      9.5647367522013610E+02_dp,  1.9075646404982639E+04_dp,&
      7.9756724721310873E-03_dp, -3.6633955682749525E+04_dp,&
      3.0054168139170702E+04_dp,  3.0244619250697619E+04_dp,&
      -8.9553753522321902E+04_dp,  7.9875735584514681E+04_dp,&
      1.7524458945017868E+04_dp, -1.3188240705569120E+05_dp,&
      1.3592402872600380E+05_dp, -1.4644005644830598E+01_dp,&
      -1.0996321547256157E+05_dp,  6.7843413283968956E+04_dp /)
    REAL(KIND=dp), DIMENSION(400), PARAMETER :: c01 = (/ &
      2.7491556688515650E-01_dp, -1.0138397977284885E-01_dp,&
      -3.9049726894672010E-02_dp,  1.2625689412991524E-02_dp,&
      1.2491380414983764E-02_dp, -3.7765616302317911E-03_dp,&
      -5.1175283467075521E-03_dp,  1.4632298825565039E-03_dp,&
      3.6642955364173022E-03_dp, -2.0156097072982762E-03_dp,&
      -1.1372055647295432E-03_dp,  1.3123912090724280E-03_dp,&
      -3.4398637877680914E-04_dp,  9.2040539096736784E-01_dp,&
      -4.1664024852120668E-01_dp, -1.1285853825987181E-01_dp,&
      7.9706962779770921E-02_dp,  3.7420429120305111E-02_dp,&
      -3.0919600141887049E-02_dp, -1.5284758454925678E-02_dp,&
      1.2652127413759585E-02_dp,  1.7202917438349393E-02_dp,&
      -2.2424746152411607E-02_dp,  7.9553605887680342E-03_dp,&
      5.1972329893920990E-04_dp, -6.8699382298445435E-04_dp,&
      1.9006321915664388E+00_dp, -1.1695348091685061E+00_dp,&
      -1.2189417451274155E-01_dp,  3.0841663498130323E-01_dp,&
      2.8149207201224088E-02_dp, -1.3459045672526931E-01_dp,&
      -4.2172084983072775E-03_dp,  5.7470448789934472E-02_dp,&
      3.9892748401996053E-02_dp, -1.0792452192241736E-01_dp,&
      7.7031387642835639E-02_dp, -2.4799303141627441E-02_dp,&
      2.9641145262798221E-03_dp,  3.6226601304189745E+00_dp,&
      -3.0653465294668503E+00_dp,  2.3997996878917224E-01_dp,&
      9.2020850811311550E-01_dp, -2.1053920858265254E-01_dp,&
      -4.0407144020499658E-01_dp,  1.4478946512027779E-01_dp,&
      1.7808216466386911E-01_dp, -6.7382162449458046E-05_dp,&
      -2.9692981490093184E-01_dp,  3.0359529933075358E-01_dp,&
      -1.3377430156320294E-01_dp,  2.3545839240890390E-02_dp,&
      6.8829600678148175E+00_dp, -7.8109151543133537E+00_dp,&
      2.1524611550975314E+00_dp,  2.1963549588737723E+00_dp,&
      -1.4083698809615777E+00_dp, -8.1873516409798242E-01_dp,&
      8.3000971643672017E-01_dp,  3.4924147841610120E-01_dp,&
      -4.6131744612525610E-01_dp, -3.2768291484595113E-01_dp,&
      6.7712900531194742E-01_dp, -3.8698623980190255E-01_dp,&
      8.1552378085722771E-02_dp,  1.3361445315337988E+01_dp,&
      -1.9720693805354671E+01_dp,  9.8170523591388061E+00_dp,&
      3.6112843572095294E+00_dp, -5.7557505888315017E+00_dp,&
      -2.6911215231012703E-01_dp,  2.9734538015789798E+00_dp,&
      -1.1080337220770373E-01_dp, -2.2704004653913077E+00_dp,&
      1.3354167923098297E+00_dp,  2.7827788457258407E-01_dp,&
      -5.3396440146110924E-01_dp,  1.5485305521629095E-01_dp,&
      2.6868227237507195E+01_dp, -5.0102045492014518E+01_dp,&
      3.7811908718509507E+01_dp, -1.8294630056592489E+00_dp,&
      -1.7506470965502057E+01_dp,  7.8480672911596105E+00_dp,&
      6.3178773982403840E+00_dp, -4.8161399112143526E+00_dp,&
      -5.2120084586700539E+00_dp,  9.0278934674271571E+00_dp,&
      -5.4400779115770668E+00_dp,  1.4969760611556289E+00_dp,&
      -1.3708613238816467E-01_dp,  5.7398225707624746E+01_dp,&
      -1.3068240758584577E+02_dp,  1.3611896593086911E+02_dp,&
      -5.6244554349383378E+01_dp, -2.8041309641109571E+01_dp,&
      4.3346676847580149E+01_dp, -6.8148749615071447E+00_dp,&
      -1.7910192391133652E+01_dp,  7.4742481485171650E+00_dp,&
      1.1402642814244142E+01_dp, -1.4702415316240296E+01_dp,&
      7.0796474962768450E+00_dp, -1.3388574138054052E+00_dp,&
      1.4718735946959035E+02_dp, -3.8655381690361162E+02_dp,&
      5.0264423442263518E+02_dp, -3.6114141158090524E+02_dp,&
      9.0095327950470534E+01_dp,  8.0537937520274326E+01_dp,&
      -7.6164154126460289E+01_dp,  3.8131872647713339E+00_dp,&
      2.5322962717490988E+01_dp, -7.4949103011308527E+00_dp,&
      -9.1658479772166626E+00_dp,  7.7135520272691753E+00_dp,&
      -1.8523169694290555E+00_dp,  2.2601006812612785E+00_dp,&
      -8.7997390790539198E-01_dp, -3.1237293051750925E-01_dp,&
      1.2648961191291591E-01_dp,  1.0173399066050458E-01_dp,&
      -4.1985217142443479E-02_dp, -4.2043767126443553E-02_dp,&
      1.6589539959258010E-02_dp,  3.3723886270116563E-02_dp,&
      -2.5599276642292213E-02_dp, -3.3723992759883159E-03_dp,&
      9.1543230425048736E-03_dp, -2.7035579659107495E-03_dp,&
      3.0558414722769203E+00_dp, -1.8423525687001594E+00_dp,&
      -2.4806358565768091E-01_dp,  4.8522067556668447E-01_dp,&
      8.1780516282716875E-02_dp, -2.1134587830838658E-01_dp,&
      -2.9333397736968799E-02_dp,  8.8869329486145510E-02_dp,&
      7.9743471514237316E-02_dp, -1.7011594446843087E-01_dp,&
      1.0584292297514415E-01_dp, -2.7622708244182885E-02_dp,&
      1.9266826647592555E-03_dp,  5.0439918903115455E+00_dp,&
      -4.6147378247056725E+00_dp,  3.4606299922222433E-01_dp,&
      1.4907521779331261E+00_dp, -2.6458521972713844E-01_dp,&
      -6.8256595629361283E-01_dp,  1.8400335770241980E-01_dp,&
      3.0044010867457915E-01_dp,  6.7687204379828558E-02_dp,&
      -5.5503131110925941E-01_dp,  5.1852137763862582E-01_dp,&
      -2.1508674337333056E-01_dp,  3.5696005935841650E-02_dp,&
      9.2338356558127259E+00_dp, -1.1596541857331431E+01_dp,&
      3.2196780214282321E+00_dp,  3.6242503908319130E+00_dp,&
      -2.0715462391169384E+00_dp, -1.5043039760553865E+00_dp,&
      1.2499534478269310E+00_dp,  6.7566189217826012E-01_dp,&
      -6.0067926803925731E-01_dp, -8.3969188762166769E-01_dp,&
      1.3289290929674367E+00_dp, -7.1228918020345477E-01_dp,&
      1.4490578224151213E-01_dp,  1.7897430920456454E+01_dp,&
      -2.8993974640936074E+01_dp,  1.4484254763706325E+01_dp,&
      6.4853429598526233E+00_dp, -8.8659405206488398E+00_dp,&
      -1.3072959846374910E+00_dp,  4.8716122165178666E+00_dp,&
      3.7210099832799376E-01_dp, -3.7219944942361369E+00_dp,&
      1.4592913559025713E+00_dp,  1.2136291248627868E+00_dp,&
      -1.2214837337140676E+00_dp,  3.1786347913744367E-01_dp,&
      3.6268902880949476E+01_dp, -7.3109988017877853E+01_dp,&
      5.5198663558559915E+01_dp,  1.5723454940839439E+00_dp,&
      -2.9119283252532057E+01_dp,  9.2853577555148661E+00_dp,&
      1.2848162547779781E+01_dp, -6.1114123559080635E+00_dp,&
      -1.1475135076695258E+01_dp,  1.5069041357505670E+01_dp,&
      -7.1864984784676569E+00_dp,  1.1624317623595268E+00_dp,&
      8.5663539809219230E-02_dp,  7.7586632783688742E+01_dp,&
      -1.8852347266621780E+02_dp,  1.9745424592717342E+02_dp,&
      -6.6990539473087694E+01_dp, -6.4607063534478783E+01_dp,&
      7.0129963577739574E+01_dp,  5.6788736642728228E+00_dp,&
      -3.8031820871312668E+01_dp, -2.1740311275891591E+00_dp,&
      4.4736349997022785E+01_dp, -4.2964456895307912E+01_dp,&
      1.8563951012438558E+01_dp, -3.2717753104214178E+00_dp,&
      1.8607799112651591E+02_dp, -5.1602169316042023E+02_dp,&
      6.8887347424898587E+02_dp, -4.7458042855350305E+02_dp,&
      4.4622418554771478E+01_dp,  2.1616014044968463E+02_dp,&
      -1.5545535106352176E+02_dp, -3.2603087830560838E+01_dp,&
      1.1189461104665806E+02_dp, -6.2453572059122330E+01_dp,&
      1.4004003528879079E+00_dp,  1.2024016844350291E+01_dp,&
      -3.7743438683099675E+00_dp,  6.9298673553197830E+02_dp,&
      -1.9786469723055493E+03_dp,  2.8890520642257911E+03_dp,&
      -2.5529058035051989E+03_dp,  1.2587845207337734E+03_dp,&
      -7.7992454428378821E+01_dp, -3.2321033215482868E+02_dp,&
      1.6424491980317885E+02_dp,  2.4365533617277901E+01_dp,&
      -3.9605274406289432E+01_dp, -6.8896488944108363E+00_dp,&
      1.6649305464354185E+01_dp, -4.9462414102750252E+00_dp,&
      3.2469980045956409E-01_dp, -5.0884891508484555E-02_dp,&
      -3.0604197193935530E-02_dp,  2.7427469620202637E-03_dp,&
      6.5288788489622235E-03_dp,  1.4751518742325842E-03_dp,&
      -4.4569860753834612E-03_dp,  7.3984873623845994E-04_dp,&
      1.3725182324794782E-03_dp, -1.3330062499350304E-04_dp,&
      -1.0097670883910479E-03_dp,  7.2572714897585022E-04_dp,&
      -1.6788642792246261E-04_dp,  1.1375940158990296E+00_dp,&
      -2.3608340938492453E-01_dp, -1.2626283245351805E-01_dp,&
      3.0203907418656358E-02_dp,  3.1785577520873423E-02_dp,&
      6.6484062921600383E-04_dp, -2.3186399042989306E-02_dp,&
      6.3183607198872387E-03_dp,  8.6381015194322997E-03_dp,&
      -3.8842596338162639E-03_dp, -3.7122714446579746E-03_dp,&
      3.6216395853979934E-03_dp, -9.2922534715789357E-04_dp,&
      2.5662576301481543E+00_dp, -7.8767800972008517E-01_dp,&
      -3.4787162045624048E-01_dp,  1.7519834794011546E-01_dp,&
      1.0184464478563594E-01_dp, -2.9402771689865572E-02_dp,&
      -8.3714434652696970E-02_dp,  3.7221173571793567E-02_dp,&
      3.9228710671209921E-02_dp, -3.5132688186151159E-02_dp,&
      -7.2078001512086579E-04_dp,  1.0273709728764327E-02_dp,&
      -3.3154780668181900E-03_dp,  5.5500130403737691E+00_dp,&
      -2.4995868120429363E+00_dp, -8.4108231466303029E-01_dp,&
      7.8254760866878970E-01_dp,  2.5130274709663863E-01_dp,&
      -2.1094801260314211E-01_dp, -2.5468188988232654E-01_dp,&
      1.8562195286267533E-01_dp,  1.5278152349671439E-01_dp,&
      -2.2434059692085659E-01_dp,  8.4198910115564699E-02_dp,&
      3.7327357965615668E-03_dp, -7.0014770989066926E-03_dp,&
      1.2388533146386802E+01_dp, -7.9005727746800547E+00_dp,&
      -1.6768805892952201E+00_dp,  3.1303583041097780E+00_dp,&
      2.9757366586635242E-01_dp, -1.0543635092197530E+00_dp,&
      -5.9806717900324746E-01_dp,  8.4749241097756289E-01_dp,&
      4.7566246950522223E-01_dp, -1.1850653348692619E+00_dp,&
      7.6550203921832072E-01_dp, -2.1047358296363022E-01_dp,&
      1.7042274233523055E-02_dp,  2.9329613292850858E+01_dp,&
      -2.5787172597477593E+01_dp, -1.4204821244535877E+00_dp,&
      1.1964763351996828E+01_dp, -1.9597065992887659E+00_dp,&
      -4.3627229820786857E+00_dp, -3.9222745547684623E-01_dp,&
      3.6450051138022639E+00_dp,  4.9711724176984340E-01_dp,&
      -4.9839121455852364E+00_dp,  4.5843906987281251E+00_dp,&
      -1.8840025476130395E+00_dp,  3.1143090085006336E-01_dp,&
      7.5288964164161186E+01_dp, -9.0335134300893770E+01_dp,&
      1.4213295300102013E+01_dp,  4.4161535993359188E+01_dp,&
      -2.3939960756625741E+01_dp, -1.3093263270166732E+01_dp,&
      7.9641392095377705E+00_dp,  1.3766695386166811E+01_dp,&
      -9.8502878978357327E+00_dp, -1.0558330710285986E+01_dp,&
      1.7789872343339511E+01_dp, -9.7871793631893524E+00_dp,&
      2.0435013450669803E+00_dp,  2.1650557215451059E+02_dp,&
      -3.5651641429679739E+02_dp,  1.6757116494673738E+02_dp,&
      1.3519758005338315E+02_dp, -1.9366619517796931E+02_dp,&
      2.9213274253505041E+01_dp,  6.3389691171209662E+01_dp,&
      1.8187088316420692E+01_dp, -1.0836900917299248E+02_dp,&
      8.7474533360265895E+01_dp, -2.0470329785844417E+01_dp,&
      -7.7295411530191132E+00_dp,  3.9109991716175272E+00_dp,&
      7.6235912249499665E+02_dp, -1.7181530353164492E+03_dp,&
      1.6377791571835323E+03_dp, -3.3946784295902592E+02_dp,&
      -9.7252903864036193E+02_dp,  1.2354270580997018E+03_dp,&
      -6.6463719849297411E+02_dp,  1.4060533750444611E+02_dp,&
      -5.3565858987836350E+01_dp,  1.4823802021219080E+02_dp,&
      -1.4694213881725742E+02_dp,  6.5887306625564349E+01_dp,&
      -1.1862205010260631E+01_dp,  2.6990399563726419E+00_dp,&
      -4.5606541750914703E-01_dp, -2.6625736807926959E-01_dp,&
      3.3786194855342760E-02_dp,  5.9720956580049278E-02_dp,&
      1.0206299110287578E-02_dp, -4.1412493078630895E-02_dp,&
      7.9885005336768396E-03_dp,  1.3447541252729038E-02_dp,&
      -2.6017691043546186E-03_dp, -8.7910286454442373E-03_dp,&
      6.7236958305186539E-03_dp, -1.5940141945756799E-03_dp,&
      4.0846562583048689E+00_dp, -1.1927022865417531E+00_dp,&
      -5.5724233644012455E-01_dp,  2.3929021555236676E-01_dp,&
      1.6353741898513949E-01_dp, -2.9735107611800447E-02_dp,&
      -1.2793124562894490E-01_dp,  4.7930701290913279E-02_dp,&
      5.5361774728157669E-02_dp, -3.9226885480863859E-02_dp,&
      -1.1058674730255031E-02_dp,  1.8542247519973694E-02_dp,&
      -5.3076927066474976E-03_dp,  7.9215231943523268E+00_dp,&
      -3.6845503235373944E+00_dp, -1.3154610882028632E+00_dp,&
      1.1148266588452116E+00_dp,  4.2745695912670312E-01_dp,&
      -2.7683155181369429E-01_dp, -4.0167185145554668E-01_dp,&
      2.4904589781084449E-01_dp,  2.2516861354249776E-01_dp,&
      -2.7804985084751915E-01_dp,  6.9038251045187016E-02_dp,&
      2.9781052441439151E-02_dp, -1.4527696704126849E-02_dp,&
      1.7339641636383043E+01_dp, -1.1465395481325789E+01_dp,&
      -2.7302697069655961E+00_dp,  4.4629359221611402E+00_dp,&
      7.1614646223286882E-01_dp, -1.4605259953560206E+00_dp,&
      -1.0335490616197762E+00_dp,  1.1536772096440751E+00_dp,&
      7.7992282360725562E-01_dp, -1.5718730701413242E+00_dp /)
    REAL(KIND=dp), DIMENSION(400), PARAMETER :: c02 = (/ &
      8.6574572823904627E-01_dp, -1.6889080712338411E-01_dp,&
      -3.8729713566779244E-03_dp,  4.1076303571384891E+01_dp,&
      -3.6684994925539350E+01_dp, -3.4268355391161074E+00_dp,&
      1.6994010089726881E+01_dp, -1.2988970233372894E+00_dp,&
      -6.3669393876591824E+00_dp, -1.4789915209996456E+00_dp,&
      5.0459641372112225E+00_dp,  1.7604359362697857E+00_dp,&
      -7.4519441974173777E+00_dp,  6.0042280072110019E+00_dp,&
      -2.1903299806572583E+00_dp,  3.1342632696136241E-01_dp,&
      1.0537644349674036E+02_dp, -1.2495001078702305E+02_dp,&
      1.1671977638986354E+01_dp,  6.3509049743298789E+01_dp,&
      -2.4624518831017006E+01_dp, -2.2991251058742133E+01_dp,&
      6.5440301808037606E+00_dp,  2.0726271091951617E+01_dp,&
      -5.4834521038296886E+00_dp, -2.4964517024507781E+01_dp,&
      2.9635321224675756E+01_dp, -1.4308796101738487E+01_dp,&
      2.7331662205056446E+00_dp,  2.9884951418781924E+02_dp,&
      -4.7112538859165682E+02_dp,  1.6517376144736465E+02_dp,&
      2.2431316178052239E+02_dp, -2.1004912340787996E+02_dp,&
      -3.6841233998403162E+01_dp,  8.5018701750455719E+01_dp,&
      6.4270674161414831E+01_dp, -1.1762295966994319E+02_dp,&
      1.4128076172212385E+01_dp,  6.8638839101450117E+01_dp,&
      -5.2639766289471062E+01_dp,  1.2875679284434117E+01_dp,&
      9.7963199739588015E+02_dp, -2.0726596540764804E+03_dp,&
      1.5278705496087439E+03_dp,  4.2594838759546570E+02_dp,&
      -1.5473674085543355E+03_dp,  7.8543126899292076E+02_dp,&
      3.4775861155014280E+02_dp, -2.1213591402524492E+02_dp,&
      -7.6234790005893728E+02_dp,  1.1942226627680336E+03_dp,&
      -7.9341217678609769E+02_dp,  2.6356711429793290E+02_dp,&
      -3.5098730768030379E+01_dp,  4.4532080783536394E+03_dp,&
      -1.1976038810115935E+04_dp,  1.5216348304854013E+04_dp,&
      -9.1497778902350692E+03_dp, -2.5770601345297018E+03_dp,&
      1.1628742339594626E+04_dp, -1.3276634853192494E+04_dp,&
      9.1770221622615209E+03_dp, -3.7823343104978308E+03_dp,&
      2.4407451616112442E+02_dp,  8.2977752842562359E+02_dp,&
      -5.3212154229074201E+02_dp,  1.1938018424864173E+02_dp,&
      3.4262178292778683E-01_dp, -1.7261992206626605E-02_dp,&
      -2.2336602339871350E-02_dp,  1.1372975909513491E-02_dp,&
      -1.4637411764816409E-02_dp,  1.9682898826632621E-02_dp,&
      -1.9508686827707740E-02_dp,  1.5180239022615439E-02_dp,&
      -8.0494684598632436E-03_dp,  4.3996140717912250E-04_dp,&
      3.5809746378920688E-03_dp, -2.8752058938169818E-03_dp,&
      7.9428830938936665E-04_dp,  1.2227594053594877E+00_dp,&
      -8.4033698327114534E-02_dp, -1.0674419107588769E-01_dp,&
      6.0462282288982330E-02_dp, -7.0785676831264299E-02_dp,&
      9.5635549300517764E-02_dp, -9.4351933521006390E-02_dp,&
      7.0755447131140681E-02_dp, -3.4019702880768367E-02_dp,&
      -3.3728225702757239E-03_dp,  2.1611313953339161E-02_dp,&
      -1.5999188759917224E-02_dp,  4.3118753534726753E-03_dp,&
      2.8601521163296666E+00_dp, -2.9979848800794423E-01_dp,&
      -3.7094815914666179E-01_dp,  2.4059291327186513E-01_dp,&
      -2.5155247937875885E-01_dp,  3.3934844711305995E-01_dp,&
      -3.3320333083079079E-01_dp,  2.3651738850263895E-01_dp,&
      -9.4854817960181387E-02_dp, -4.0172372650363469E-02_dp,&
      9.8175845918375776E-02_dp, -6.7106282688909516E-02_dp,&
      1.7607790831527539E-02_dp,  6.5195485295712299E+00_dp,&
      -1.0269942852581080E+00_dp, -1.2316140242437659E+00_dp,&
      9.2057237167479200E-01_dp, -8.6609478324363198E-01_dp,&
      1.1499978387095577E+00_dp, -1.1250457879495033E+00_dp,&
      7.4504215559187936E-01_dp, -2.1409102804328831E-01_dp,&
      -2.5465251320434851E-01_dp,  4.2237116881553893E-01_dp,&
      -2.7025866808821075E-01_dp,  6.9222424341596048E-02_dp,&
      1.5601098721282909E+01_dp, -3.5605032547397730E+00_dp,&
      -4.1021042659113336E+00_dp,  3.5928399226527388E+00_dp,&
      -3.0663280855742157E+00_dp,  3.9096825239762416E+00_dp,&
      -3.8079613332081270E+00_dp,  2.2900409541690148E+00_dp,&
      -2.3404263306851089E-01_dp, -1.4199262179831273E+00_dp,&
      1.8507969187271571E+00_dp, -1.1142464342469358E+00_dp,&
      2.7855655064891799E-01_dp,  4.0485584978500526E+01_dp,&
      -1.3109279621523161E+01_dp, -1.4273206648108861E+01_dp,&
      1.5134358148420791E+01_dp, -1.1885791275039221E+01_dp,&
      1.3904458082083121E+01_dp, -1.3369820062766205E+01_dp,&
      6.8691260200838169E+00_dp,  1.8693426127542194E+00_dp,&
      -8.1523519343559254E+00_dp,  8.8821179988688765E+00_dp,&
      -5.0221150779769559E+00_dp,  1.2204509968663211E+00_dp,&
      1.1797058394435172E+02_dp, -5.4419195200734706E+01_dp,&
      -5.4178470328491755E+01_dp,  7.3774886159171885E+01_dp,&
      -5.5331810110639871E+01_dp,  5.4316621798868489E+01_dp,&
      -4.9351105617752985E+01_dp,  1.7430778986280817E+01_dp,&
      2.6471231206118816E+01_dp, -5.4492738860827032E+01_dp,&
      5.1545112335746424E+01_dp, -2.7216839193839707E+01_dp,&
      6.3684016980887135E+00_dp,  4.1049490910859993E+02_dp,&
      -2.8101620718403206E+02_dp, -2.3507330230842464E+02_dp,&
      4.6729944581388298E+02_dp, -3.7007836528600853E+02_dp,&
      2.5620942949620382E+02_dp, -1.6276307350822648E+02_dp,&
      -4.7413775559957635E+01_dp,  3.3696413558725703E+02_dp,&
      -5.0564162778438396E+02_dp,  4.3214949005292641E+02_dp,&
      -2.1394266212194319E+02_dp,  4.7786865055818609E+01_dp,&
      1.9870073185576307E+03_dp, -2.2734345390187736E+03_dp,&
      -1.0366644301592682E+03_dp,  4.9782231070513590E+03_dp,&
      -5.8165480519264056E+03_dp,  2.9032478426631178E+03_dp,&
      2.0625281232662687E+03_dp, -7.2292182775277870E+03_dp,&
      1.0969088485231096E+04_dp, -1.1554611433571949E+04_dp,&
      8.4527956060934266E+03_dp, -3.8415737929474558E+03_dp,&
      8.1110303330682132E+02_dp,  2.8606866433551854E+00_dp,&
      -1.5669366038116742E-01_dp, -2.0177594319939973E-01_dp,&
      1.0575962777565361E-01_dp, -1.3259082085957813E-01_dp,&
      1.7861218892252503E-01_dp, -1.7679239908469763E-01_dp,&
      1.3625980189079434E-01_dp, -7.0554930091096479E-02_dp,&
      1.3046383793103491E-03_dp,  3.4551474369077309E-02_dp,&
      -2.7081602811949396E-02_dp,  7.4285468910542333E-03_dp,&
      4.5253856130077423E+00_dp, -4.4517797470657283E-01_dp,&
      -5.5534985895292899E-01_dp,  3.4609637177398361E-01_dp,&
      -3.7311782888019995E-01_dp,  5.0533840314273726E-01_dp,&
      -4.9657643901723980E-01_dp,  3.5871272457032366E-01_dp,&
      -1.5359061407378005E-01_dp, -4.6227951687665789E-02_dp,&
      1.3588639509874001E-01_dp, -9.4986322229029627E-02_dp,&
      2.5115468494290173E-02_dp,  9.3387030569139586E+00_dp,&
      -1.4879589096614800E+00_dp, -1.7988658334718912E+00_dp,&
      1.2995109815487618E+00_dp, -1.2482340118753112E+00_dp,&
      1.6737963248029610E+00_dp, -1.6379906508005568E+00_dp,&
      1.1048426169812811E+00_dp, -3.5465330257263172E-01_dp,&
      -3.2226064337605276E-01_dp,  5.7855934320968894E-01_dp,&
      -3.7626009267967558E-01_dp,  9.6955612305782160E-02_dp,&
      2.1950328829710198E+01_dp, -5.0511260135942582E+00_dp,&
      -5.8862988319603264E+00_dp,  4.9441392015736385E+00_dp,&
      -4.2994380387089803E+00_dp,  5.5897309543939446E+00_dp,&
      -5.4527815856197339E+00_dp,  3.3734109423832814E+00_dp,&
      -5.5303193700046960E-01_dp, -1.7798059879978836E+00_dp,&
      2.4612001888686756E+00_dp, -1.5076561686602952E+00_dp,&
      3.7961828447184426E-01_dp,  5.6677731876899855E+01_dp,&
      -1.8013663018122998E+01_dp, -1.9988781444904490E+01_dp,&
      1.9975665481283382E+01_dp, -1.5915517724387783E+01_dp,&
      1.9377941803388104E+01_dp, -1.8776127280497441E+01_dp,&
      1.0211721332179954E+01_dp,  1.2166506526464049E+00_dp,&
      -9.7300470047856056E+00_dp,  1.1178017920928982E+01_dp,&
      -6.4583697384013634E+00_dp,  1.5861476220437252E+00_dp,&
      1.6266409498267876E+02_dp, -7.0874377152702721E+01_dp,&
      -7.3232748732520108E+01_dp,  9.0638778034246016E+01_dp,&
      -6.7833024061559215E+01_dp,  7.2645518074708306E+01_dp,&
      -6.8582720788466375E+01_dp,  2.9344572977161508E+01_dp,&
      2.3658760653171736E+01_dp, -5.8749244736075021E+01_dp,&
      5.8359148143262004E+01_dp, -3.1682065751004121E+01_dp,&
      7.5424009759039885E+00_dp,  5.3985696776971338E+02_dp,&
      -3.2878377292720995E+02_dp, -3.0196845089030052E+02_dp,&
      4.9836773218871861E+02_dp, -3.7106297234816765E+02_dp,&
      3.1114524824685083E+02_dp, -2.6001905721418905E+02_dp,&
      4.3657614880237190E+01_dp,  2.5997836938226095E+02_dp,&
      -4.4142314871954403E+02_dp,  3.9224023622037282E+02_dp,&
      -1.9948747668116454E+02_dp,  4.5558727242996810E+01_dp,&
      2.2518322147207505E+03_dp, -2.0304752941146858E+03_dp,&
      -1.4511666087126071E+03_dp,  3.8178668066270552E+03_dp,&
      -3.2459924653373946E+03_dp,  1.7680131097711978E+03_dp,&
      -5.2399989243414564E+02_dp, -1.3019719452938477E+03_dp,&
      3.7028205187866570E+03_dp, -5.0416739250637856E+03_dp,&
      4.1663670072848045E+03_dp, -2.0159642364314470E+03_dp,&
      4.4166941378618947E+02_dp,  1.4828081608364595E+04_dp,&
      -2.2574049136099864E+04_dp, -4.2565500168210820E+03_dp,&
      5.5084825389384707E+04_dp, -8.2670207348266471E+04_dp,&
      4.6919568554688740E+04_dp,  4.8435800070201876E+04_dp,&
      -1.5814297608851999E+05_dp,  2.2592917315340103E+05_dp,&
      -2.1798583714533239E+05_dp,  1.4577424945232089E+05_dp,&
      -6.1348593359510611E+04_dp,  1.2191827498138327E+04_dp,&
      3.4532051271897191E-01_dp,  4.4100447073726646E-09_dp,&
      -1.9415646007041806E-02_dp,  1.5929327108429592E-02_dp,&
      -1.1176571510952480E-02_dp, -2.5877640413336351E-03_dp,&
      2.4137268336731563E-02_dp, -4.3674857388440270E-02_dp,&
      4.5860088560280322E-02_dp, -2.3078520914289882E-02_dp,&
      -8.6082658270545242E-03_dp,  2.0469378895770350E-02_dp,&
      -9.4442160284978153E-03_dp,  1.2359461442291915E+00_dp,&
      2.1613211945043984E-08_dp, -9.5224148443146792E-02_dp,&
      7.8125394169053466E-02_dp, -5.2254864411876351E-02_dp,&
      -1.6917118000855980E-02_dp,  1.2319600286256702E-01_dp,&
      -2.1709123959554127E-01_dp,  2.2390587928808975E-01_dp,&
      -1.1000370261743218E-01_dp, -4.3039738152667616E-02_dp,&
      9.7915585548516884E-02_dp, -4.4457276303720650E-02_dp,&
      2.9074409526161196E+00_dp,  7.7826482682482066E-08_dp,&
      -3.4324867297490447E-01_dp,  2.8161341307550997E-01_dp,&
      -1.7555608121572958E-01_dp, -8.2110170119991194E-02_dp,&
      4.6764741429324741E-01_dp, -7.9579019571154763E-01_dp,&
      8.0041604656281196E-01_dp, -3.7948529579609275E-01_dp,&
      -1.5906346472215843E-01_dp,  3.3960719652526311E-01_dp,&
      -1.5041051454561974E-01_dp,  6.6825165744494983E+00_dp,&
      2.6947500124974605E-07_dp, -1.1899989328294034E+00_dp,&
      9.7631566011870097E-01_dp, -5.5706896221715130E-01_dp,&
      -3.6977101750415975E-01_dp,  1.7130054180455190E+00_dp,&
      -2.8047552355031407E+00_dp,  2.7378646139272180E+00_dp,&
      -1.2400284060281412E+00_dp, -5.6607288056274385E-01_dp,&
      1.1176163709452969E+00_dp, -4.7849425689590552E-01_dp,&
      1.6170325428899595E+01_dp,  9.4669250986364349E-07_dp,&
      -4.1874399518362004E+00_dp,  3.4355103239851768E+00_dp,&
      -1.7339742570487151E+00_dp, -1.6747275450356629E+00_dp,&
      6.4121308809247504E+00_dp, -1.0027369248263646E+01_dp,&
      9.4133727686313424E+00_dp, -3.9922245386706305E+00_dp,&
      -2.0492157416285433E+00_dp,  3.6364108946934444E+00_dp,&
      -1.4765386616651448E+00_dp,  4.2602647072076110E+01_dp,&
      3.5480189961398575E-06_dp, -1.5730419782630479E+01_dp,&
      1.2905699113131829E+01_dp, -5.3595105861997663E+00_dp,&
      -8.1972811001257586E+00_dp,  2.5924257535046760E+01_dp,&
      -3.8176552987451025E+01_dp,  3.3842711727305677E+01_dp,&
      -1.2840341384820070E+01_dp, -7.9417840034024891E+00_dp,&
      1.1922026639891879E+01_dp, -4.3665403821207924E+00_dp,&
      1.2689163866512382E+02_dp,  1.5118075434269657E-05_dp,&
      -6.7275555506731095E+01_dp,  5.5194557090186521E+01_dp,&
      -1.5537886798726163E+01_dp, -4.7253294284612615E+01_dp,&
      1.2145285812656974E+02_dp, -1.6368251320712150E+02_dp,&
      1.3132457381744328E+02_dp, -3.8713782602971918E+01_dp,&
      -3.5040157951924371E+01_dp,  3.7792017841024283E+01_dp,&
      -1.0016991740016113E+01_dp,  4.5777593919577163E+02_dp,&
      8.1637641969554641E-05_dp, -3.6580205008674420E+02_dp,&
      3.0011032726156765E+02_dp, -1.3953035592512641E+01_dp,&
      -3.7345160055057738E+02_dp,  7.4221682637239917E+02_dp /)
    REAL(KIND=dp), DIMENSION(13, 18, 4), PARAMETER :: &
      coefdata = RESHAPE ((/ c01, c02, c03 /),(/13,18,4/))

    INTEGER                                  :: irange

     IF(Rc <=  1.0000000000000000E+01_dp) Rc =   1.0000000000000000E+01_dp
     IF(Rc >=  1.2759075164794922E+04_dp) Rc =   1.2759075164794922E+04_dp
     IF(Rc >=  1.0000000000000000E+01_dp .AND. Rc <=  1.1566345417137983E+02_dp) THEN
       irange =  1
       L_b =  1.0000000000000000E+01_dp
       U_b =  1.1566345417137983E+02_dp
     END IF
     IF(Rc >=  1.1566345417137983E+02_dp .AND. Rc <=  1.1355688414729136E+03_dp) THEN
       irange =  2
       L_b =  1.1566345417137983E+02_dp
       U_b =  1.1355688414729136E+03_dp
     END IF
     IF(Rc >=  1.1355688414729136E+03_dp .AND. Rc <=  5.9775803550807823E+03_dp) THEN
       irange =  3
       L_b =  1.1355688414729136E+03_dp
       U_b =  5.9775803550807823E+03_dp
     END IF
     IF(Rc >=  5.9775803550807823E+03_dp .AND. Rc <=  1.2759075164794922E+04_dp) THEN
       irange =  4
       L_b =  5.9775803550807823E+03_dp
       U_b =  1.2759075164794922E+04_dp
     END IF

     fit_coef(1:13,1:18) = coefdata(1:13,1:18,irange)

   END SUBROUTINE get_fit_coef_k09

! *****************************************************************************
!> \brief ...
!> \param Rc ...
!> \param L_b ...
!> \param U_b ...
!> \param fit_coef ...
! *****************************************************************************
   SUBROUTINE get_fit_coef_k10(Rc,L_b,U_b,fit_coef)
    REAL(KIND=dp)                            :: Rc, L_b, U_b
    REAL(KIND=dp), DIMENSION(13, 20)         :: fit_coef

    REAL(KIND=dp), DIMENSION(240), PARAMETER :: c03 = (/ &
      -6.3360994610628929E-01_dp,  9.7649518951989356E-01_dp,&
      -1.0559988395317237E+00_dp,  7.8078414899977844E-01_dp,&
      -3.5500159005989329E-01_dp,  7.4673135592582374E-02_dp,&
      2.6285615412587124E+00_dp,  1.5860173767897436E-07_dp,&
      -3.6913389708324101E-01_dp,  3.7465972064134995E-01_dp,&
      -3.1561014124123743E-01_dp, -5.7643482677198092E-02_dp,&
      9.2410177205176591E-01_dp, -2.1835473996583254E+00_dp,&
      3.2970352241440182E+00_dp, -3.4960505581040451E+00_dp,&
      2.5278721705611331E+00_dp, -1.1186977080493512E+00_dp,&
      2.2749024280873054E-01_dp,  5.7493721709618049E+00_dp,&
      5.1131034673141817E-07_dp, -1.1909353988603624E+00_dp,&
      1.2087611583778604E+00_dp, -9.6018131262641071E-01_dp,&
      -3.0452567745790421E-01_dp,  3.1471609396333315E+00_dp,&
      -7.1728858417880028E+00_dp,  1.0580386353620003E+01_dp,&
      -1.0957261904298493E+01_dp,  7.7029457637581844E+00_dp,&
      -3.2879502327488281E+00_dp,  6.3689329465605082E-01_dp,&
      1.3024889596738207E+01_dp,  1.6466878135911835E-06_dp,&
      -3.8389522217497509E+00_dp,  3.8964046167903090E+00_dp,&
      -2.8718997405626689E+00_dp, -1.4373626984851318E+00_dp,&
      1.0761889985401876E+01_dp, -2.3552965075047659E+01_dp,&
      3.3769385385787992E+01_dp, -3.3923608671499672E+01_dp,&
      2.2944182267670296E+01_dp, -9.2816114732207033E+00_dp,&
      1.6586565047476383E+00_dp,  3.1432665091848261E+01_dp,&
      5.5045818221119110E-06_dp, -1.2848255060030274E+01_dp,&
      1.3040500498346995E+01_dp, -8.6561473620662710E+00_dp,&
      -6.7615560396841685E+00_dp,  3.8551980935564238E+01_dp,&
      -8.0342597911471927E+01_dp,  1.1094756895369210E+02_dp,&
      -1.0670977647200603E+02_dp,  6.7945774392356341E+01_dp,&
      -2.4996446349757960E+01_dp,  3.7523703565476070E+00_dp,&
      8.2776639766289406E+01_dp,  1.9927227577400854E-05_dp,&
      -4.6589600081344805E+01_dp,  4.7286523741856939E+01_dp,&
      -2.6646560758118390E+01_dp, -3.4199978931791321E+01_dp,&
      1.5164754664098274E+02_dp, -2.9661527406436210E+02_dp,&
      3.8780081459610972E+02_dp, -3.4748842481061382E+02_dp,&
      1.9747486810222125E+02_dp, -5.7764787565672627E+01_dp,&
      3.9741149863670704E+00_dp,  2.4653488710994122E+02_dp,&
      8.2811812634509398E-05_dp, -1.9410155849192833E+02_dp,&
      1.9700392058258890E+02_dp, -8.1422141386870905E+01_dp,&
      -2.0290030574898262E+02_dp,  6.9905487313819572E+02_dp,&
      -1.2476934937878245E+03_dp,  1.4859985026297729E+03_dp,&
      -1.1509015539974444E+03_dp,  4.7294812978393372E+02_dp,&
      -1.1374554419169920E+01_dp, -4.9602668093775350E+01_dp,&
      8.8939704304992472E+02_dp,  4.3913317689329946E-04_dp,&
      -1.0337351134189328E+03_dp,  1.0491819395923051E+03_dp,&
      -1.5674657748993846E+02_dp, -1.6456617734696722E+03_dp,&
      4.2436228771404622E+03_dp, -6.4096922420241744E+03_dp,&
      6.0719638526939207E+03_dp, -2.5708936520526772E+03_dp,&
      -1.4140287964490178E+03_dp,  2.4275532997866176E+03_dp,&
      -9.6578744659039239E+02_dp,  4.6536562243205735E+03_dp,&
      3.9438897095978847E-03_dp, -9.3476143934673928E+03_dp,&
      9.4871562777538929E+03_dp,  5.1466047555988271E+03_dp,&
      -2.8237091397710159E+04_dp,  4.3842210795647632E+04_dp,&
      -2.9527423734560154E+04_dp, -2.8125488385754445E+04_dp,&
      1.0440567050324763E+05_dp, -1.3678916485878875E+05_dp,&
      9.3760628448597679E+04_dp, -2.7703671833937438E+04_dp,&
      2.7347133607954048E+00_dp,  8.9893013343578908E-08_dp,&
      -2.0901102881611386E-01_dp,  2.1214036968796418E-01_dp,&
      -1.9242931099074201E-01_dp, -4.6222152733960808E-03_dp,&
      4.8292883261738540E-01_dp, -1.2026007686844928E+00_dp,&
      1.8735094311067497E+00_dp, -2.0464478608991055E+00_dp,&
      1.5298400325094081E+00_dp, -7.0456884498313244E-01_dp,&
      1.5049712771824222E-01_dp,  4.1788056884786338E+00_dp,&
      2.3922482564732447E-07_dp, -5.5667371811520094E-01_dp,&
      5.6500721075193983E-01_dp, -4.8270517088007758E-01_dp,&
      -7.3153070762063260E-02_dp,  1.3743457496872575E+00_dp,&
      -3.2780462608547563E+00_dp,  4.9787719700231952E+00_dp,&
      -5.3097509004990568E+00_dp,  3.8648752483084237E+00_dp,&
      -1.7244548775870321E+00_dp,  3.5436350814717738E-01_dp,&
      8.2149049298385233E+00_dp,  7.5131065115636197E-07_dp,&
      -1.7496558689886887E+00_dp,  1.7758451683120040E+00_dp,&
      -1.4284432030977785E+00_dp, -4.1105297713265365E-01_dp,&
      4.5754862006213592E+00_dp, -1.0506814920577511E+01_dp,&
      1.5577169057113929E+01_dp, -1.6217703754793195E+01_dp,&
      1.1475030104148107E+01_dp, -4.9399708582597350E+00_dp,&
      9.6830600717441639E-01_dp,  1.8261194412040087E+01_dp,&
      2.3800351273101264E-06_dp, -5.5474414243405885E+00_dp,&
      5.6304652949007590E+00_dp, -4.2224080442212388E+00_dp,&
      -1.9292280668308683E+00_dp,  1.5363529652698020E+01_dp,&
      -3.3932770509938848E+01_dp,  4.8979621527255397E+01_dp,&
      -4.9571574115119340E+01_dp,  3.3856280052461059E+01_dp,&
      -1.3889532396024331E+01_dp,  2.5377006524511256E+00_dp,&
      4.4058973569111785E+01_dp,  7.7835134661955194E-06_dp,&
      -1.8161678578197755E+01_dp,  1.8433441512633923E+01_dp,&
      -1.2590211927409817E+01_dp, -8.8344038021778299E+00_dp,&
      5.3618887193845289E+01_dp, -1.1320132345144582E+02_dp,&
      1.5796945077014172E+02_dp, -1.5386547456867649E+02_dp,&
      9.9772878898220497E+01_dp, -3.7833032575922452E+01_dp,&
      6.0353219200773358E+00_dp,  1.1568580381278532E+02_dp,&
      2.7277035891925943E-05_dp, -6.3738620396398332E+01_dp,&
      6.4692155157605299E+01_dp, -3.8542281982295471E+01_dp,&
      -4.2526459742796327E+01_dp,  2.0263872121445911E+02_dp,&
      -4.0468067321359194E+02_dp,  5.3921183956405559E+02_dp,&
      -4.9575997036083578E+02_dp,  2.9437494939739389E+02_dp,&
      -9.4973641633375749E+01_dp,  1.0052032668988744E+01_dp,&
      3.3844441481139603E+02_dp,  1.0708615847444910E-04_dp,&
      -2.5073874169991208E+02_dp,  2.5448859830947686E+02_dp,&
      -1.2076885762164136E+02_dp, -2.3028236778445304E+02_dp,&
      8.7071015385976250E+02_dp, -1.6152471653168971E+03_dp,&
      2.0059864954392992E+03_dp, -1.6663062126313935E+03_dp,&
      8.1553595526745528E+02_dp, -1.4640889696758140E+02_dp,&
      -2.6500762414317144E+01_dp,  1.1548838669601862E+03_dp,&
      5.0406126126581669E-04_dp, -1.1839335596846738E+03_dp,&
      1.2016308566170446E+03_dp, -3.4715885980007397E+02_dp,&
      -1.5427529687412427E+03_dp,  4.5828692243129517E+03_dp,&
      -7.6048801944765419E+03_dp,  8.2881674827925654E+03_dp,&
      -5.3672515338847079E+03_dp,  9.8514173264072872E+02_dp,&
      1.1594528623077892E+03_dp, -6.5416602461076945E+02_dp,&
      5.0521007201290549E+03_dp,  3.2560476887267091E-03_dp,&
      -7.6890070328778293E+03_dp,  7.8038462290402367E+03_dp,&
      3.9687198253925811E+02_dp, -1.5428524978799329E+04_dp,&
      3.4098140303215056E+04_dp, -4.5038353268521678E+04_dp,&
      3.2436968649400857E+04_dp,  3.7529823967807060E+03_dp,&
      -3.4972750744217890E+04_dp,  3.3219164891303109E+04_dp,&
      -1.1400731607075682E+04_dp,  3.7061397677567882E+04_dp,&
      4.2647739208431207E-02_dp, -1.0102666755386532E+05_dp,&
      1.0253508247597031E+05_dp,  1.0602636418265585E+05_dp,&
      -4.0728952867028292E+05_dp,  4.6976444940836332E+05_dp,&
      5.8500781101532266E+04_dp, -1.2055500949944141E+06_dp,&
      2.3598896259778733E+06_dp, -2.5585975849316372E+06_dp,&
      1.5841592037308544E+06_dp, -4.3843418788217887E+05_dp /)
    REAL(KIND=dp), DIMENSION(400), PARAMETER :: c01 = (/ &
      2.5561005851553148E-01_dp, -9.8727639373792656E-02_dp,&
      -3.5824853081321391E-02_dp,  1.3639809429163310E-02_dp,&
      1.2140312218587666E-02_dp, -4.6051578600266093E-03_dp,&
      -5.2440292404026837E-03_dp,  1.6141788048728977E-03_dp,&
      4.9068438183985687E-03_dp, -3.8643109440260576E-03_dp,&
      -1.3246866398568792E-05_dp,  9.9919760932901634E-04_dp,&
      -3.1484202326100442E-04_dp,  8.4272796404557193E-01_dp,&
      -3.8901613573165461E-01_dp, -1.0067616997785502E-01_dp,&
      7.5588609850681651E-02_dp,  3.4648448246217420E-02_dp,&
      -3.0893154344547202E-02_dp, -1.4586932713617575E-02_dp,&
      1.2131777659737109E-02_dp,  2.0819225767407759E-02_dp,&
      -2.8910339615850700E-02_dp,  1.3280887656117454E-02_dp,&
      -1.7063570583648409E-03_dp, -2.9519647286739830E-04_dp,&
      1.6891663892562130E+00_dp, -1.0264543983393131E+00_dp,&
      -1.0644178450031999E-01_dp,  2.6441701384654470E-01_dp,&
      2.6556013010702633E-02_dp, -1.1958261699563323E-01_dp,&
      -4.5401290095952633E-03_dp,  5.1008274807051057E-02_dp,&
      4.5353450819585334E-02_dp, -1.1487744057269965E-01_dp,&
      8.4790329561634001E-02_dp, -2.9250363584235169E-02_dp,&
      3.9794003608289051E-03_dp,  3.0862814349867831E+00_dp,&
      -2.5146525088601708E+00_dp,  1.6647176896057306E-01_dp,&
      7.3389153069620650E-01_dp, -1.4890295143927060E-01_dp,&
      -3.3532945050846130E-01_dp,  1.0826191575795263E-01_dp,&
      1.5317085471740924E-01_dp,  2.1245529597518711E-02_dp,&
      -2.8990367580166487E-01_dp,  2.8971901471196243E-01_dp,&
      -1.2768396771603524E-01_dp,  2.2620465138046251E-02_dp,&
      5.5612937791608781E+00_dp, -5.9652373960236220E+00_dp,&
      1.4638363568522026E+00_dp,  1.6757996697253348E+00_dp,&
      -9.4543407221718756E-01_dp, -6.8535446452406956E-01_dp,&
      5.7877541374209429E-01_dp,  3.2873788086829692E-01_dp,&
      -3.1394297986526876E-01_dp, -3.7134857531460552E-01_dp,&
      6.2436702901871433E-01_dp, -3.4134184222136776E-01_dp,&
      7.0290493767657503E-02_dp,  1.0134025998635400E+01_dp,&
      -1.3929286667259428E+01_dp,  6.1321763420892008E+00_dp,&
      2.9330166883940492E+00_dp, -3.6140733174101785E+00_dp,&
      -6.7590309312420871E-01_dp,  1.9913635070734654E+00_dp,&
      2.9670856040397131E-01_dp, -1.6412798075937387E+00_dp,&
      5.9540353669355439E-01_dp,  5.5718678791010678E-01_dp,&
      -5.3739746596150106E-01_dp,  1.3768553709528983E-01_dp,&
      1.8880739332952572E+01_dp, -3.2415164403909472E+01_dp,&
      2.1344280000859261E+01_dp,  1.8599622720347415E+00_dp,&
      -1.0841361053976007E+01_dp,  2.4783487242921880E+00_dp,&
      4.9433547282134356E+00_dp, -1.5295881914775833E+00_dp,&
      -4.7733675746929487E+00_dp,  5.4211714199806815E+00_dp,&
      -2.2746955524517021E+00_dp,  2.2802023048670481E-01_dp,&
      6.9582676550312542E-02_dp,  3.6335716382001010E+01_dp,&
      -7.6018306813772128E+01_dp,  6.8732879845752208E+01_dp,&
      -1.6316998250306515E+01_dp, -2.4243518494944528E+01_dp,&
      1.8917610164460754E+01_dp,  5.5832946785483388E+00_dp,&
      -1.0154671149564965E+01_dp, -5.3612772885590845E+00_dp,&
      1.6258259712293338E+01_dp, -1.2814243372405180E+01_dp,&
      4.8354454594839140E+00_dp, -7.5082718737346388E-01_dp,&
      7.4215448848755400E+01_dp, -1.8335462266162904E+02_dp,&
      2.1358039078323498E+02_dp, -1.1557194919902095E+02_dp,&
      -1.6853548075444962E+01_dp,  6.4520986380399108E+01_dp,&
      -2.2746737191584021E+01_dp, -2.2062101091379237E+01_dp,&
      1.7789460655504136E+01_dp,  9.0031524935346070E+00_dp,&
      -1.8542030042521016E+01_dp,  1.0164879826355794E+01_dp,&
      -2.0735114143734910E+00_dp,  1.8341446865250575E+02_dp,&
      -5.0877391178005558E+02_dp,  7.0449341000362926E+02_dp,&
      -5.5517124915990166E+02_dp,  1.8704002208932476E+02_dp,&
      8.2551818013538082E+01_dp, -1.0679714348268803E+02_dp,&
      9.0991109491737330E+00_dp,  4.1126744560482862E+01_dp,&
      -1.8803518402205185E+01_dp, -8.8575702273859580E+00_dp,&
      1.0182151829754702E+01_dp, -2.6408507910563315E+00_dp,&
      2.0934716095248742E+00_dp, -8.4746055605151593E-01_dp,&
      -2.8427750491443193E-01_dp,  1.3068018915578153E-01_dp,&
      9.7384350056846619E-02_dp, -4.7397825964349709E-02_dp,&
      -4.2114616553585213E-02_dp,  1.7316564939632989E-02_dp,&
      4.3715952641648077E-02_dp, -4.1554266092337727E-02_dp,&
      7.4054798396290897E-03_dp,  5.5977770808678454E-03_dp,&
      -2.2373612160795555E-03_dp,  2.7210036656294760E+00_dp,&
      -1.6335020259554927E+00_dp, -2.0680846939368511E-01_dp,&
      4.2349947782997177E-01_dp,  6.8122910566226311E-02_dp,&
      -1.9210163151722040E-01_dp, -2.3321895976892021E-02_dp,&
      8.0146433140384166E-02_dp,  8.7876154568316808E-02_dp,&
      -1.8834452288276832E-01_dp,  1.2728988197515606E-01_dp,&
      -3.9391516952941304E-02_dp,  4.4839495381117809E-03_dp,&
      4.2368736629375707E+00_dp, -3.7802973669397950E+00_dp,&
      2.8694988069048666E-01_dp,  1.1892835304153320E+00_dp,&
      -2.1162718986798351E-01_dp, -5.6362634575947124E-01_dp,&
      1.5250211850697887E-01_dp,  2.5537886963697748E-01_dp,&
      7.9649433013108686E-02_dp, -5.2911720038955190E-01_dp,&
      5.0012923442134083E-01_dp, -2.1252561189029046E-01_dp,&
      3.6393804438124989E-02_dp,  7.2758481879346171E+00_dp,&
      -8.8167800124863280E+00_dp,  2.3330958495202210E+00_dp,&
      2.7108042687959344E+00_dp, -1.4637274671449114E+00_dp,&
      -1.1890746322064221E+00_dp,  9.0715716740563646E-01_dp,&
      5.8170114998851308E-01_dp, -4.2989157564081804E-01_dp,&
      -7.8489539565179522E-01_dp,  1.1632291726097599E+00_dp,&
      -6.1430639748629101E-01_dp,  1.2402938836877603E-01_dp,&
      1.3170364441482098E+01_dp, -2.0369157267246411E+01_dp,&
      9.4288085210953678E+00_dp,  4.8217116031330969E+00_dp,&
      -5.6861997313549448E+00_dp, -1.3780670845149023E+00_dp,&
      3.2453891736082223E+00_dp,  6.9371123634570475E-01_dp,&
      -2.7043289246339368E+00_dp,  6.9595595974366786E-01_dp,&
      1.2333657622496881E+00_dp, -1.0391669343871834E+00_dp,&
      2.5704620438014530E-01_dp,  2.4716167001767921E+01_dp,&
      -4.7051480106065966E+01_dp,  3.2155480577188463E+01_dp,&
      3.9571391906830309E+00_dp, -1.7481873346539064E+01_dp,&
      2.9769207206580353E+00_dp,  8.6315702725343542E+00_dp,&
      -1.8390311278544129E+00_dp, -8.6328002259793184E+00_dp,&
      8.8292799810814202E+00_dp, -3.1386229669783341E+00_dp,&
      -2.3715217346491377E-02_dp,  2.0535568808319654E-01_dp,&
      4.8051665196315099E+01_dp, -1.0976439154416751E+02_dp,&
      1.0235279436257439E+02_dp, -2.0383771987790944E+01_dp,&
      -4.2573982577312300E+01_dp,  2.8489936514971767E+01_dp,&
      1.3632688101723078E+01_dp, -1.6113634285018911E+01_dp,&
      -1.5125986568044409E+01_dp,  3.3059561366308991E+01_dp,&
      -2.3793635327665104E+01_dp,  8.3081513867812440E+00_dp,&
      -1.1774786371362731E+00_dp,  9.8355243119061484E+01_dp,&
      -2.6174079961725596E+02_dp,  3.1507317101618884E+02_dp,&
      -1.6085984367937596E+02_dp, -5.3691559005483263E+01_dp,&
      1.1770786081977101E+02_dp, -2.4626973913536151E+01_dp,&
      -5.6303483179808985E+01_dp,  2.8847539985623889E+01_dp,&
      3.7907983907152932E+01_dp, -5.4916726607563149E+01_dp,&
      2.8300062799138427E+01_dp, -5.6436900258842000E+00_dp,&
      2.2780023889155092E+02_dp, -6.6928883833527561E+02_dp,&
      9.6616701087926413E+02_dp, -7.6933915858921614E+02_dp,&
      2.0967951970184987E+02_dp,  2.2235553142415941E+02_dp,&
      -2.3712254219885492E+02_dp,  8.8535187310337022E+00_dp,&
      1.3723076658510743E+02_dp, -1.0892853389708965E+02_dp,&
      3.0016643813667478E+01_dp,  3.2568204150430908E+00_dp,&
      -2.6599172689595973E+00_dp,  8.4295965555020791E+02_dp,&
      -2.5052951870900652E+03_dp,  3.8112871716634299E+03_dp,&
      -3.5307900192682346E+03_dp,  1.8732914678507132E+03_dp,&
      -2.3164482385830567E+02_dp, -3.8311716432588844E+02_dp,&
      1.7973313419763957E+02_dp,  9.3149821542615896E+01_dp,&
      -1.0608074184561853E+02_dp,  1.5210241656314690E+01_dp,&
      1.7089336905208167E+01_dp, -6.3589478647152200E+00_dp,&
      3.0592957565289286E-01_dp, -5.1110929330852831E-02_dp,&
      -2.8987289080467823E-02_dp,  3.3561895390932014E-03_dp,&
      7.5514448011207201E-03_dp,  1.5139979831753692E-04_dp,&
      -4.1708804842884705E-03_dp,  6.3660879882102028E-04_dp,&
      1.8200902121666236E-03_dp, -4.4081413042472650E-04_dp,&
      -1.0534265079120433E-03_dp,  8.4103592321804256E-04_dp,&
      -2.0186508759774654E-04_dp,  1.0525777831189882E+00_dp,&
      -2.2605017958924992E-01_dp, -1.1386949796815091E-01_dp,&
      2.9774643144139291E-02_dp,  3.3954500666697272E-02_dp,&
      -5.4949904510822227E-03_dp, -1.9959719518336631E-02_dp,&
      5.5330636693116369E-03_dp,  1.0456790729486607E-02_dp,&
      -6.1821591900185022E-03_dp, -2.5994663457028877E-03_dp,&
      3.4568072502680707E-03_dp, -9.4576165761213102E-04_dp,&
      2.2919336198838280E+00_dp, -7.0509806233015249E-01_dp,&
      -2.9188032941530057E-01_dp,  1.5316780817827591E-01_dp,&
      1.0010940584584704E-01_dp, -4.5873718932214910E-02_dp,&
      -6.4374651563716148E-02_dp,  3.0933040465657206E-02_dp,&
      4.2615946369704352E-02_dp, -4.3329755675445232E-02_dp,&
      7.3263430836030315E-03_dp,  6.6279030394463092E-03_dp,&
      -2.6348013649783435E-03_dp,  4.7111266696595502E+00_dp,&
      -2.0753802897217151E+00_dp, -6.5267974687686825E-01_dp,&
      6.1871821696202411E-01_dp,  2.3731527891030152E-01_dp,&
      -2.2970544582560379E-01_dp, -1.7157602804264291E-01_dp,&
      1.4120918754911052E-01_dp,  1.4835360673290199E-01_dp,&
      -2.2594398043496730E-01_dp,  1.0338085286845002E-01_dp,&
      -1.1064326042678516E-02_dp, -3.2650162585572047E-03_dp,&
      9.8473743889612759E+00_dp, -6.0131980554589237E+00_dp,&
      -1.2129491990772383E+00_dp,  2.2254376519254429E+00_dp,&
      3.6302894568783933E-01_dp, -9.4270878563778415E-01_dp,&
      -3.3552531531919016E-01_dp,  5.7912063800107416E-01_dp,&
      4.2064135755829263E-01_dp, -9.8733374612298730E-01_dp,&
      6.6042114139286523E-01_dp, -1.9752444965211660E-01_dp,&
      2.0598380207893886E-02_dp,  2.1453757316653739E+01_dp,&
      -1.7636195844827874E+01_dp, -1.1722260578775618E+00_dp,&
      7.5540641100286328E+00_dp, -5.1916302328323216E-01_dp,&
      -3.4197651131825317E+00_dp, -4.8633514010784806E-02_dp,&
      2.2205434162034567E+00_dp,  6.2862136700524063E-01_dp,&
      -3.5713176709083991E+00_dp,  3.1667300178926499E+00_dp,&
      -1.2773737284357061E+00_dp,  2.0804275445421447E-01_dp,&
      4.9455179001078307E+01_dp, -5.3788204979410104E+01_dp,&
      5.5826058527127156E+00_dp,  2.4646678349298128E+01_dp,&
      -9.0061724463921937E+00_dp, -1.0659622095516202E+01_dp,&
      4.4620017759335129E+00_dp,  7.7884822664291757E+00_dp,&
      -3.4529539803486271E+00_dp, -8.5214099311418341E+00_dp,&
      1.1263431473903086E+01_dp, -5.6921726426224888E+00_dp,&
      1.1214589809695603E+00_dp,  1.2269759721177081E+02_dp,&
      -1.7567244385618403E+02_dp,  5.7907863244030423E+01_dp,&
      7.4194919093471611E+01_dp, -6.3873800026211477E+01_dp,&
      -1.9928072277598243E+01_dp,  3.3660133146703842E+01_dp,&
      1.9714768039645122E+01_dp, -4.2363027488591541E+01_dp,&
      1.1603745291817669E+01_dp,  1.6417376687682371E+01_dp,&
      -1.4366009866157320E+01_dp,  3.6184944957788190E+00_dp,&
      3.3684139089809150E+02_dp, -6.3751305400246281E+02_dp,&
      4.1416908794179335E+02_dp,  1.3688234571140524E+02_dp,&
      -3.7063608979057182E+02_dp,  1.2143722822915174E+02_dp,&
      1.3731882710289389E+02_dp, -4.6346221563435897E+01_dp,&
      -2.0268131602498934E+02_dp,  2.7850969054636107E+02_dp,&
      -1.6672316424649100E+02_dp,  4.9139777093543977E+01_dp,&
      -5.4771033390485266E+00_dp,  1.1166365406065504E+03_dp,&
      -2.7547693853452047E+03_dp,  3.0256333468051180E+03_dp,&
      -1.2247747589089884E+03_dp, -1.0361905831178201E+03_dp,&
      1.8075416359519638E+03_dp, -1.0299807587967950E+03_dp,&
      1.1242398674681689E+02_dp,  5.2546835084300625E+01_dp,&
      2.0209424570528384E+02_dp, -2.8565373044965975E+02_dp,&
      1.4996446737543528E+02_dp, -3.0374236420884017E+01_dp,&
      2.5318787052609042E+00_dp, -4.5232522380132079E-01_dp,&
      -2.4894203765998704E-01_dp,  3.7844517569960621E-02_dp,&
      6.7481113092261444E-02_dp, -1.8350883302710324E-03_dp,&
      -3.7893183999051401E-02_dp,  6.9921199011088479E-03_dp,&
      1.7385860790483609E-02_dp, -5.9031381997973701E-03_dp /)
    REAL(KIND=dp), DIMENSION(400), PARAMETER :: c02 = (/ &
      -8.4867222607997398E-03_dp,  7.4160230219779905E-03_dp,&
      -1.8338341118192985E-03_dp,  3.6656246502838239E+00_dp,&
      -1.0862821007061922E+00_dp, -4.7299838594788851E-01_dp,&
      2.1791529395255307E-01_dp,  1.6235258776392880E-01_dp,&
      -5.9164629917823931E-02_dp, -1.0182204867540855E-01_dp,&
      4.1466211524979202E-02_dp,  6.2596474192031126E-02_dp,&
      -5.4084127674425816E-02_dp,  8.8525531803891741E-04_dp,&
      1.4025724547496657E-02_dp, -4.6032712047529446E-03_dp,&
      6.6769922009854454E+00_dp, -3.0970371708527678E+00_dp,&
      -1.0140989359103294E+00_dp,  9.1150511260383493E-01_dp,&
      3.9216519842874853E-01_dp, -3.2543013169962537E-01_dp,&
      -2.7778597097915597E-01_dp,  1.9772597022734270E-01_dp,&
      2.2579283394692193E-01_dp, -3.0593985439637572E-01_dp,&
      1.1622044051596585E-01_dp,  2.0410252870715667E-03_dp,&
      -8.3966296975702334E-03_dp,  1.3623320699171776E+01_dp,&
      -8.8558893274426165E+00_dp, -1.9149805228610246E+00_dp,&
      3.2774967810751803E+00_dp,  6.8478362995256370E-01_dp,&
      -1.3680993330511229E+00_dp, -5.9476667203260292E-01_dp,&
      8.2139559031258302E-01_dp,  6.9090776235235218E-01_dp,&
      -1.4058701920349088E+00_dp,  8.5413836920236308E-01_dp,&
      -2.1810741182595644E-01_dp,  1.4255082384362025E-02_dp,&
      2.9748852628521725E+01_dp, -2.5626593581102714E+01_dp,&
      -2.2281795814990457E+00_dp,  1.1083089209567575E+01_dp,&
      -1.3736928750062674E-01_dp, -5.0763678761476774E+00_dp,&
      -5.0249418564834214E-01_dp,  3.1849945099789339E+00_dp,&
      1.4726022355354449E+00_dp, -5.4992302941830431E+00_dp,&
      4.4719918101122813E+00_dp, -1.6727141406904336E+00_dp,&
      2.4977009750404219E-01_dp,  6.9037893211784962E+01_dp,&
      -7.6964135274020265E+01_dp,  5.4824348905454112E+00_dp,&
      3.6332914820361268E+01_dp, -1.0145340748522697E+01_dp,&
      -1.6794616142996990E+01_dp,  4.6404572235591202E+00_dp,&
      1.1643820898102305E+01_dp, -1.5727289341303272E+00_dp,&
      -1.6439340401781347E+01_dp,  1.8245888309513088E+01_dp,&
      -8.5306553392815161E+00_dp,  1.5879795273877024E+00_dp,&
      1.7182022390659438E+02_dp, -2.4613776736324084E+02_dp,&
      6.7186754104591998E+01_dp,  1.1416051467226143E+02_dp,&
      -7.7001090430108874E+01_dp, -4.3041023883777321E+01_dp,&
      4.2330169680693920E+01_dp,  3.6483814289573353E+01_dp,&
      -4.6424717513479678E+01_dp, -1.3561530898217379E+01_dp,&
      4.6964917734628436E+01_dp, -2.9562882553835443E+01_dp,&
      6.6027180009057211E+00_dp,  4.6773081093747555E+02_dp,&
      -8.6370782747484213E+02_dp,  4.7050209287821843E+02_dp,&
      2.9908873234382622E+02_dp, -4.7104852448475293E+02_dp,&
      1.6889908350425834E+01_dp,  2.3791881917263390E+02_dp,&
      3.6767254629124110E+01_dp, -3.4434736297874019E+02_dp,&
      2.9580663084422144E+02_dp, -8.4122378160736815E+01_dp,&
      -1.4744687754938745E+01_dp,  1.0220647890227653E+01_dp,&
      1.4510103014196029E+03_dp, -3.4619055478794439E+03_dp,&
      3.2324145408972931E+03_dp, -2.0171830139352292E+02_dp,&
      -2.4226448619655516E+03_dp,  1.8793554815989780E+03_dp,&
      3.7645024254807635E+02_dp, -1.0119465757689406E+03_dp,&
      -5.4242419196334629E+02_dp,  1.9807634322220583E+03_dp,&
      -1.7934401838025353E+03_dp,  7.8212645881853916E+02_dp,&
      -1.4202019050520397E+02_dp,  6.2496239305148747E+03_dp,&
      -1.7931776674939312E+04_dp,  2.4916823442349287E+04_dp,&
      -1.8499646284135801E+04_dp,  2.2582520021478135E+03_dp,&
      1.1773422282371406E+04_dp, -1.5812069270655764E+04_dp,&
      1.1855294782461493E+04_dp, -6.1493986813322972E+03_dp,&
      2.3758121921599381E+03_dp, -6.9588428484459973E+02_dp,&
      1.3762196426902483E+02_dp, -1.1913138218513865E+01_dp,&
      3.2470276062374315E-01_dp, -1.9003588934512133E-02_dp,&
      -1.8355145860536166E-02_dp,  6.0475341198086311E-03_dp,&
      -6.7274274564422471E-03_dp,  1.0743771705419822E-02_dp,&
      -1.1306871751006364E-02_dp,  1.0146638916666086E-02_dp,&
      -8.7364678660853939E-03_dp,  6.4093949316994117E-03_dp,&
      -3.3864501126037571E-03_dp,  1.0980797299645153E-03_dp,&
      -1.6237200712845642E-04_dp,  1.1376077378517864E+00_dp,&
      -8.8168978454148964E-02_dp, -8.2966960947689752E-02_dp,&
      3.2220461688134687E-02_dp, -3.0778141035688236E-02_dp,&
      4.9924652120375348E-02_dp, -5.3251003530151837E-02_dp,&
      4.6974990767227179E-02_dp, -3.9775722053066218E-02_dp,&
      2.9079117771650204E-02_dp, -1.5334031453898066E-02_dp,&
      4.9235548695483276E-03_dp, -7.1155987460411857E-04_dp,&
      2.5663032612728065E+00_dp, -2.9417413930700842E-01_dp,&
      -2.6658115314553854E-01_dp,  1.2655515440162460E-01_dp,&
      -1.0170953149105190E-01_dp,  1.6637618853203828E-01_dp,&
      -1.8127899261011421E-01_dp,  1.5639283645717517E-01_dp,&
      -1.2892869855711223E-01_dp,  9.3691422159345950E-02_dp,&
      -4.9367488223686778E-02_dp,  1.5676761588867189E-02_dp,&
      -2.1960515393188405E-03_dp,  5.5501559534518865E+00_dp,&
      -9.3353792174523964E-01_dp, -8.0931985604485024E-01_dp,&
      4.6774970561508383E-01_dp, -3.2499339716017250E-01_dp,&
      5.2455331020275098E-01_dp, -5.8731514467566936E-01_dp,&
      4.9583685161075564E-01_dp, -3.9426634261390675E-01_dp,&
      2.8367707651694279E-01_dp, -1.4984448875803760E-01_dp,&
      4.7282829011273904E-02_dp, -6.4299987442053754E-03_dp,&
      1.2388957662684190E+01_dp, -2.9507992375405814E+00_dp,&
      -2.4213308891809810E+00_dp,  1.7155106808462175E+00_dp,&
      -1.0620015587182443E+00_dp,  1.6352185410468572E+00_dp,&
      -1.8946239202850306E+00_dp,  1.5686621116997250E+00_dp,&
      -1.1845997162664987E+00_dp,  8.3552549739590276E-01_dp,&
      -4.4419948824934929E-01_dp,  1.4059544874631708E-01_dp,&
      -1.8723783798815802E-02_dp,  2.9330516223788635E+01_dp,&
      -9.6326512564302238E+00_dp, -7.3400314978953354E+00_dp,&
      6.5269889977849020E+00_dp, -3.7394128859062916E+00_dp,&
      5.2122480243556142E+00_dp, -6.2925500085850317E+00_dp,&
      5.1355169298916543E+00_dp, -3.5833520026413055E+00_dp,&
      2.4087579835578392E+00_dp, -1.2892152657735232E+00_dp,&
      4.1552495501679831E-01_dp, -5.5252748929712804E-02_dp,&
      7.5280704759745987E+01_dp, -3.3767849596353095E+01_dp,&
      -2.3070194711950077E+01_dp,  2.6937913499898457E+01_dp,&
      -1.5152121320132757E+01_dp,  1.7635732437963757E+01_dp,&
      -2.2164824763993362E+01_dp,  1.8037707168064706E+01_dp,&
      -1.1050554303680990E+01_dp,  6.4306484729918472E+00_dp,&
      -3.3403778828301771E+00_dp,  1.1062691086411864E+00_dp,&
      -1.4900634156067252E-01_dp,  2.1606787442647337E+02_dp,&
      -1.3382440328877573E+02_dp, -7.6047551699211525E+01_dp,&
      1.2760861971346685E+02_dp, -7.7609951567506357E+01_dp,&
      6.7526383179937113E+01_dp, -8.4589255870543653E+01_dp,&
      7.0008722016727575E+01_dp, -3.3736145353795237E+01_dp,&
      9.4396713074454439E+00_dp, -1.5637007492391390E+00_dp,&
      1.1171056998266990E-02_dp,  1.1212771079965393E-01_dp,&
      7.3384991880302323E+02_dp, -6.5417335621128620E+02_dp,&
      -2.4586250567902340E+02_dp,  7.5968666626958600E+02_dp,&
      -5.8873803425324638E+02_dp,  3.5879791502938718E+02_dp,&
      -3.3980659001725331E+02_dp,  2.7598165136284933E+02_dp,&
      -5.9140159002267268E+01_dp, -1.2294529972421002E+02_dp,&
      1.4244705471792136E+02_dp, -7.0098207002701088E+01_dp,&
      1.4534293555696086E+01_dp,  3.3844616772454051E+03_dp,&
      -4.8034622373676111E+03_dp,  2.7407731795956414E+02_dp,&
      6.4933384274191558E+03_dp, -9.3671786693748436E+03_dp,&
      6.8293651088856996E+03_dp, -1.6751505221392342E+03_dp,&
      -3.1810158919697988E+03_dp,  6.3337086240843646E+03_dp,&
      -7.0202386816691314E+03_dp,  5.1859554354248066E+03_dp,&
      -2.3452002831301538E+03_dp,  4.8871710263553763E+02_dp,&
      2.6990664833406162E+00_dp, -1.7032360609079394E-01_dp,&
      -1.6339064623599808E-01_dp,  5.6343825413842273E-02_dp,&
      -6.0034058384596536E-02_dp,  9.6355578617489959E-02_dp,&
      -1.0174675727771637E-01_dp,  9.0884582873824907E-02_dp,&
      -7.7924356474501258E-02_dp,  5.7118288518374742E-02_dp,&
      -3.0159305381508530E-02_dp,  9.7524462861410255E-03_dp,&
      -1.4333289835950528E-03_dp,  4.0847254939908550E+00_dp,&
      -4.4543680259376395E-01_dp, -4.0790852964917546E-01_dp,&
      1.8397050605060919E-01_dp, -1.5369689027094091E-01_dp,&
      2.5238408456140987E-01_dp, -2.7306185583974296E-01_dp,&
      2.3685643812599680E-01_dp, -1.9696262819285476E-01_dp,&
      1.4343603886285661E-01_dp, -7.5523339024722219E-02_dp,&
      2.4017052583640244E-02_dp, -3.3873525180420443E-03_dp,&
      7.9217355528550035E+00_dp, -1.3760861532373267E+00_dp,&
      -1.2035284285567731E+00_dp,  6.7184227370314420E-01_dp,&
      -4.7463289875497050E-01_dp,  7.7591654058981052E-01_dp,&
      -8.6294791141385430E-01_dp,  7.3053252189346307E-01_dp,&
      -5.8632946501607508E-01_dp,  4.2329064886995371E-01_dp,&
      -2.2316303933581608E-01_dp,  7.0275129511077339E-02_dp,&
      -9.5676479706661693E-03_dp,  1.7340282840591684E+01_dp,&
      -4.2821512957303280E+00_dp, -3.5551190120613683E+00_dp,&
      2.4241473256398329E+00_dp, -1.5142511846518363E+00_dp,&
      2.3855617310533646E+00_dp, -2.7426998606577166E+00_dp,&
      2.2741890274553316E+00_dp, -1.7414520515624128E+00_dp,&
      1.2378423595437182E+00_dp, -6.5653655779588638E-01_dp,&
      2.0668914871276739E-01_dp, -2.7419850897391889E-02_dp,&
      4.1078055491562758E+01_dp, -1.3702317382050255E+01_dp,&
      -1.0636229224223397E+01_dp,  8.9943416364770226E+00_dp,&
      -5.1483412011517853E+00_dp,  7.4730665242250467E+00_dp,&
      -8.9464889682750020E+00_dp,  7.2963810538971217E+00_dp,&
      -5.2128753174443299E+00_dp,  3.5795300074268930E+00_dp,&
      -1.9173641280639240E+00_dp,  6.1233155609547130E-01_dp,&
      -8.0560572449398848E-02_dp,  1.0537664003823548E+02_dp,&
      -4.6684798343138667E+01_dp, -3.2994097251570423E+01_dp,&
      3.5741451453175586E+01_dp, -1.9689433576200049E+01_dp,&
      2.4641696738638490E+01_dp, -3.0837544464503011E+01_dp,&
      2.4961159129982445E+01_dp, -1.5996745583167973E+01_dp,&
      1.0020617009760857E+01_dp, -5.3706889305591723E+00_dp,&
      1.7753999846542772E+00_dp, -2.3852515204726438E-01_dp,&
      2.9869868484411023E+02_dp, -1.7630985569199339E+02_dp,&
      -1.0788199695181154E+02_dp,  1.5896261410654549E+02_dp,&
      -9.0745960005483639E+01_dp,  8.9261769188135247E+01_dp,&
      -1.1499671293603262E+02_dp,  9.4016858776158358E+01_dp,&
      -5.0104327036931210E+01_dp,  2.2553445772362544E+01_dp,&
      -1.0392621676839312E+01_dp,  3.4909481019108575E+00_dp,&
      -4.6411039824149081E-01_dp,  9.7281571855615607E+02_dp,&
      -7.8276566598170348E+02_dp, -3.6841750220545742E+02_dp,&
      8.4221015840709913E+02_dp, -5.5710576843649176E+02_dp,&
      3.9075007223954356E+02_dp, -4.6691622115267910E+02_dp,&
      3.9364528127783149E+02_dp, -1.4353811191269111E+02_dp,&
      -3.5775860066911733E+01_dp,  6.3797892295263324E+01_dp,&
      -3.0108403820523300E+01_dp,  5.9947780639426460E+00_dp,&
      3.9315484795337375E+03_dp, -4.5545648190814945E+03_dp,&
      -1.0245924129063264E+03_dp,  5.9086864031696177E+03_dp,&
      -5.4423397520665330E+03_dp,  2.8987063024985546E+03_dp,&
      -1.8396758304894188E+03_dp,  1.2823546603176267E+03_dp,&
      3.1149245284221831E+02_dp, -1.8673961299648126E+03_dp,&
      1.9363718166711049E+03_dp, -9.6551061755211992E+02_dp,&
      2.0354900388393716E+02_dp,  2.4280135349637152E+04_dp,&
      -4.4803296384207482E+04_dp,  1.6182899528768005E+04_dp,&
      5.9523036280870147E+04_dp, -1.2394614179673967E+05_dp,&
      1.1722133038315580E+05_dp, -3.1352348768700398E+04_dp,&
      -8.6202695459030496E+04_dp,  1.6982689346809132E+05_dp,&
      -1.7890858277514763E+05_dp,  1.2286186888783661E+05_dp,&
      -5.1808342561852995E+04_dp,  1.0198816861077163E+04_dp,&
      3.2867451616851961E-01_dp,  1.0003187174814238E-08_dp,&
      -2.3255655571169480E-02_dp,  2.3603849289304341E-02_dp,&
      -2.1600208550223220E-02_dp, -1.2742608714917504E-04_dp,&
      5.3167626511118796E-02_dp, -1.3331453857752948E-01_dp,&
      2.0849621635227758E-01_dp, -2.2855126470711301E-01_dp,&
      1.7151231132777350E-01_dp, -7.9338970506037887E-02_dp,&
      1.7034794798258115E-02_dp,  1.1561315825459209E+00_dp,&
      4.6876842564306299E-08_dp, -1.0903099661435091E-01_dp,&
      1.1066333209885304E-01_dp, -9.7910572631274409E-02_dp,&
      -7.4543419120169617E-03_dp,  2.5924139627124398E-01_dp /)
    REAL(KIND=dp), DIMENSION(13, 20, 4), PARAMETER :: &
      coefdata = RESHAPE ((/ c01, c02, c03 /),(/13,20,4/))

    INTEGER                                  :: irange

     IF(Rc <=  1.0000000000000000E+01_dp) Rc =   1.0000000000000000E+01_dp
     IF(Rc >=  2.4789161682128906E+04_dp) Rc =   2.4789161682128906E+04_dp
     IF(Rc >=  1.0000000000000000E+01_dp .AND. Rc <=  1.3035677773075426E+02_dp) THEN
       irange =  1
       L_b =  1.0000000000000000E+01_dp
       U_b =  1.3035677773075426E+02_dp
     END IF
     IF(Rc >=  1.3035677773075426E+02_dp .AND. Rc <=  1.4908189080699865E+03_dp) THEN
       irange =  2
       L_b =  1.3035677773075426E+02_dp
       U_b =  1.4908189080699865E+03_dp
     END IF
     IF(Rc >=  1.4908189080699865E+03_dp .AND. Rc <=  8.7983353697963448E+03_dp) THEN
       irange =  3
       L_b =  1.4908189080699865E+03_dp
       U_b =  8.7983353697963448E+03_dp
     END IF
     IF(Rc >=  8.7983353697963448E+03_dp .AND. Rc <=  2.4789161682128906E+04_dp) THEN
       irange =  4
       L_b =  8.7983353697963448E+03_dp
       U_b =  2.4789161682128906E+04_dp
     END IF

     fit_coef(1:13,1:20) = coefdata(1:13,1:20,irange)

   END SUBROUTINE get_fit_coef_k10

! *****************************************************************************
!> \brief ...
!> \param Rc ...
!> \param L_b ...
!> \param U_b ...
!> \param fit_coef ...
! *****************************************************************************
   SUBROUTINE get_fit_coef_k11(Rc,L_b,U_b,fit_coef)
    REAL(KIND=dp)                            :: Rc, L_b, U_b
    REAL(KIND=dp), DIMENSION(13, 22)         :: fit_coef

    REAL(KIND=dp), DIMENSION(230), PARAMETER :: c04 = (/ &
      -2.5319035989384675E+00_dp, -3.1868909327109335E-01_dp,&
      7.3902114944100665E+00_dp, -1.8431274144078589E+01_dp,&
      2.9378426424405447E+01_dp, -3.3161724592145767E+01_dp,&
      2.5816413812770147E+01_dp, -1.2469330304705172E+01_dp,&
      2.8128079338690761E+00_dp,  2.4584353206019742E+01_dp,&
      -7.0898120270105360E-07_dp, -8.4777162976221128E+00_dp,&
      9.0616788124866332E+00_dp, -7.2743279493553201E+00_dp,&
      -2.0496520134147098E+00_dp,  2.4245583092874156E+01_dp,&
      -5.7939288071844402E+01_dp,  9.0054831214542489E+01_dp,&
      -9.9384665876997374E+01_dp,  7.5600795805376052E+01_dp,&
      -3.5621843185101483E+01_dp,  7.8236158425344327E+00_dp,&
      5.9289248842465618E+01_dp, -2.3090660386475871E-06_dp,&
      -2.7406069249001057E+01_dp,  2.9293762062917892E+01_dp,&
      -2.1436160745311739E+01_dp, -1.1100871619137575E+01_dp,&
      8.4659605462739805E+01_dp, -1.9223819873089482E+02_dp,&
      2.8912014235586781E+02_dp, -3.0919939021055404E+02_dp,&
      2.2729338871058800E+02_dp, -1.0300649725896395E+02_dp,&
      2.1628399735758872E+01_dp,  1.5611877204908581E+02_dp,&
      -8.2470664905436727E-06_dp, -9.6867258565390344E+01_dp,&
      1.0353887823856932E+02_dp, -6.5682095402519849E+01_dp,&
      -6.0937613047746630E+01_dp,  3.2814251019497499E+02_dp,&
      -6.9844405525445029E+02_dp,  1.0025237366016256E+03_dp,&
      -1.0211592975562098E+03_dp,  7.0836288602503657E+02_dp,&
      -2.9837334150052988E+02_dp,  5.6921862071198518E+01_dp,&
      4.6496398441680941E+02_dp, -3.4223577383871753E-05_dp,&
      -3.9563037071188347E+02_dp,  4.2287603418762706E+02_dp,&
      -2.0653649047208501E+02_dp, -3.8172667391864337E+02_dp,&
      1.5030594921729898E+03_dp, -2.9227874116449611E+03_dp,&
      3.8875752535201673E+03_dp, -3.6138894664576283E+03_dp,&
      2.2048207474371115E+03_dp, -7.5725030323513738E+02_dp,&
      9.7823912240919910E+01_dp,  1.6773973436726878E+03_dp,&
      -1.8434170056840727E-04_dp, -2.0728088439834223E+03_dp,&
      2.2155290245149013E+03_dp, -5.1394027894319083E+02_dp,&
      -3.2225108046723080E+03_dp,  9.1480623850220691E+03_dp,&
      -1.5210320266230407E+04_dp,  1.7053021607531919E+04_dp,&
      -1.1990799071970599E+04_dp,  3.6225067334732939E+03_dp,&
      1.1364153809614309E+03_dp, -9.4031209009192617E+02_dp,&
      8.7767660769754748E+03_dp, -1.7301255885247059E-03_dp,&
      -1.8469732055515993E+04_dp,  1.9741000016554910E+04_dp,&
      8.6834780160185710E+03_dp, -5.7182355776311117E+04_dp,&
      9.7112250915990357E+04_dp, -8.4229126418227272E+04_dp,&
      -1.3935593484706684E+04_dp,  1.5898931395778846E+05_dp,&
      -2.3301918179538654E+05_dp,  1.6673068363491533E+05_dp,&
      -5.0314774080942996E+04_dp,  2.6059126819246052E+00_dp,&
      -1.5184941808505334E-08_dp, -1.8508505661036193E-01_dp,&
      1.9783570907686668E-01_dp, -1.9674587968365256E-01_dp,&
      3.6851207151313237E-02_dp,  4.0611699310341248E-01_dp,&
      -1.1473620212627975E+00_dp,  1.9481749518031135E+00_dp,&
      -2.3151160483320039E+00_dp,  1.8923593661054907E+00_dp,&
      -9.5933207303263768E-01_dp,  2.2714210609760052E-01_dp,&
      3.8468115363487008E+00_dp, -3.8029440074718104E-08_dp,&
      -4.6155963237640873E-01_dp,  4.9335593943885447E-01_dp,&
      -4.6884491464814559E-01_dp,  4.5025271220622491E-02_dp,&
      1.0863412220747446E+00_dp, -2.9376032021086864E+00_dp,&
      4.8861385280514016E+00_dp, -5.7142841824577904E+00_dp,&
      4.6031605955599106E+00_dp, -2.3011975883717044E+00_dp,&
      5.3755533579904435E-01_dp,  7.2028738158550514E+00_dp,&
      -1.1279238254521241E-07_dp, -1.3633777485153236E+00_dp,&
      1.4572967288789407E+00_dp, -1.3243244997028747E+00_dp,&
      2.7095192289746449E-03_dp,  3.4096983315886717E+00_dp,&
      -8.8778329414439394E+00_dp,  1.4487046010265646E+01_dp,&
      -1.6682435020852303E+01_dp,  1.3244072546969868E+01_dp,&
      -6.5263496906408722E+00_dp,  1.5028959640528783E+00_dp,&
      1.5155338303550476E+01_dp, -3.3609978102604631E-07_dp,&
      -4.0448729535600725E+00_dp,  4.3235039861438427E+00_dp,&
      -3.7401938738456062E+00_dp, -3.9813778724854831E-01_dp,&
      1.0727618136824205E+01_dp, -2.6919155675271707E+01_dp,&
      4.3059272745573644E+01_dp, -4.8755479915707021E+01_dp,&
      3.8074633815792446E+01_dp, -1.8449947088829020E+01_dp,&
      4.1761221031219460E+00_dp,  3.4280442119778854E+01_dp,&
      -1.0209663151194477E-06_dp, -1.2222273717908696E+01_dp,&
      1.3064175401809598E+01_dp, -1.0625423963861708E+01_dp,&
      -2.6578485949092507E+00_dp,  3.4550403061250776E+01_dp,&
      -8.3242948341915039E+01_dp,  1.3004394299969496E+02_dp,&
      -1.4419493228330839E+02_dp,  1.1023246137753506E+02_dp,&
      -5.2221835584331593E+01_dp,  1.1538123809656815E+01_dp,&
      8.3023233971686949E+01_dp, -3.2566579599055860E-06_dp,&
      -3.8724653854538957E+01_dp,  4.1391988205209223E+01_dp,&
      -3.0989658913000362E+01_dp, -1.4177989050298210E+01_dp,&
      1.1765010876316009E+02_dp, -2.7042253291528965E+02_dp,&
      4.1008960465882154E+02_dp, -4.4218461562488704E+02_dp,&
      3.2805416607523796E+02_dp, -1.5027909722707574E+02_dp,&
      3.1961374875560626E+01_dp,  2.1815261062484299E+02_dp,&
      -1.1263123220565498E-05_dp, -1.3272987707048557E+02_dp,&
      1.4187169395564845E+02_dp, -9.4224265155115006E+01_dp,&
      -7.4404769563706964E+01_dp,  4.3837099398368440E+02_dp,&
      -9.5183455799001024E+02_dp,  1.3871602035765698E+03_dp,&
      -1.4365079812470631E+03_dp,  1.0170715004310093E+03_dp,&
      -4.4009662202309175E+02_dp,  8.7140147118769207E+01_dp,&
      6.3829241773452395E+02_dp, -4.4064460415062701E-05_dp,&
      -5.1269111346857051E+02_dp,  5.4799986683226075E+02_dp,&
      -2.9947328222405213E+02_dp, -4.2617722685256570E+02_dp,&
      1.8705077766963821E+03_dp, -3.7732358405587056E+03_dp,&
      5.1863433327248795E+03_dp, -5.0256458027042509E+03_dp,&
      3.2619991942285269E+03_dp, -1.2467431574611608E+03_dp,&
      2.0321857334508204E+02_dp,  2.1780996488924352E+03_dp,&
      -2.0907405555531795E-04_dp, -2.3849923698991861E+03_dp,&
      2.5492235035473764E+03_dp, -9.3326450903590683E+02_dp,&
      -2.9722230635951501E+03_dp,  9.8396661700376753E+03_dp,&
      -1.7829859393390787E+04_dp,  2.2115783909639369E+04_dp,&
      -1.8612912915580167E+04_dp,  9.4926673873045820E+03_dp,&
      -2.0581853251291723E+03_dp, -1.3506399200968644E+02_dp,&
      9.5282286909788218E+03_dp, -1.3879365845395903E-03_dp,&
      -1.5291429799079764E+04_dp,  1.6344150714985619E+04_dp,&
      -5.8506000394263890E+02_dp, -3.0670459034251217E+04_dp,&
      7.3829472150507368E+04_dp, -1.0879378872595044E+05_dp,&
      1.0183991390866101E+05_dp, -4.2693488672623760E+04_dp,&
      -2.3602726867402875E+04_dp,  4.0091252301356195E+04_dp,&
      -1.5836929022957091E+04_dp,  6.9897560151235826E+04_dp,&
      -1.8816291773517638E-02_dp, -1.9854704384469349E+05_dp,&
      2.1221227425035721E+05_dp,  1.9513181165124831E+05_dp,&
      -8.3228931356148690E+05_dp,  1.0686042433396007E+06_dp,&
      -1.6890882361242565E+05_dp, -2.0275393275062682E+06_dp,&
      4.3886071943292571E+06_dp, -4.9464312502375301E+06_dp,&
      3.1275647455199584E+06_dp, -8.7618952854779735E+05_dp /)
    REAL(KIND=dp), DIMENSION(400), PARAMETER :: c01 = (/ &
      1.9140383852290629E-01_dp, -6.8714312208041611E-02_dp,&
      -9.1979524758077930E-03_dp,  4.0542016308222793E-03_dp,&
      1.3043896673939879E-03_dp, -3.3077626826676678E-04_dp,&
      -1.1713733973944338E-04_dp,  7.6096932007647678E-05_dp,&
      1.6862094749663900E-05_dp, -1.8859482112709049E-05_dp,&
      -1.0541642966027691E-05_dp,  8.6799165647851962E-06_dp,&
      -1.1431845819938354E-06_dp,  6.0489131859947465E-01_dp,&
      -2.4024677183256804E-01_dp, -1.8798045436205172E-02_dp,&
      1.5837928504487560E-02_dp,  2.6102782284167180E-03_dp,&
      -1.5652126400000133E-03_dp, -1.3827151075313952E-04_dp,&
      3.1557964494168748E-04_dp,  2.4958120214882143E-06_dp,&
      -7.5449303381703686E-05_dp, -1.6643512580341317E-05_dp,&
      2.8571234433609825E-05_dp, -6.0597558303770187E-06_dp,&
      1.1168795367902835E+00_dp, -5.2641558523934429E-01_dp,&
      6.6463431140833772E-03_dp,  3.7711310583160011E-02_dp,&
      -1.4443909125474224E-03_dp, -4.2130528794393661E-03_dp,&
      7.2183333855785543E-04_dp,  7.3960445605806635E-04_dp,&
      -1.9348822131823822E-04_dp, -1.6874282736821310E-04_dp,&
      3.6412043398419911E-05_dp,  4.1084858267757457E-05_dp,&
      -1.5134273688639719E-05_dp,  1.8158447362882504E+00_dp,&
      -1.0478545011498810E+00_dp,  1.2341009021397938E-01_dp,&
      7.1058959620305007E-02_dp, -1.9347584416030746E-02_dp,&
      -7.7203335451403827E-03_dp,  3.8020800688892561E-03_dp,&
      1.1445156153540811E-03_dp, -7.9235221490085630E-04_dp,&
      -2.4612837134322182E-04_dp,  2.1358675908042475E-04_dp,&
      1.7784602018876602E-06_dp, -1.9703398840932572E-05_dp,&
      2.8299287437087801E+00_dp, -2.0047036100998326E+00_dp,&
      4.5453821241428582E-01_dp,  9.8865732993666661E-02_dp,&
      -6.6582584274414200E-02_dp, -7.1570894058818569E-03_dp,&
      1.1198660032441313E-02_dp,  5.0949933891323033E-04_dp,&
      -2.0685167422699120E-03_dp, -9.6150256589892975E-05_dp,&
      5.5979142435495538E-04_dp, -1.6195041286557900E-04_dp,&
      3.3998670478786237E-07_dp,  4.3574186531348724E+00_dp,&
      -3.7415946807175851E+00_dp,  1.2610537308451397E+00_dp,&
      4.7862913363269956E-02_dp, -1.6352351758172967E-01_dp,&
      1.3772664688980325E-02_dp,  2.4218199648024418E-02_dp,&
      -4.1260474719994672E-03_dp, -4.0082156154816646E-03_dp,&
      8.1922589521136123E-04_dp,  1.0199455777660676E-03_dp,&
      -5.4787322426774396E-04_dp,  7.8504581614977811E-05_dp,&
      6.7237595753310817E+00_dp, -6.8634354719280006E+00_dp,&
      3.0796744406753831E+00_dp, -3.0614743416373752E-01_dp,&
      -3.1107420468808400E-01_dp,  9.3817776800566161E-02_dp,&
      3.5729378037653900E-02_dp, -1.8761603678412703E-02_dp,&
      -4.7366169578511481E-03_dp,  3.5316646495869753E-03_dp,&
      1.0851842798406590E-03_dp, -1.2373787163121499E-03_dp,&
      2.8130651342912039E-04_dp,  1.0510156788199021E+01_dp,&
      -1.2466153777035323E+01_dp,  6.9869766995910823E+00_dp,&
      -1.5498053185363254E+00_dp, -3.9434499629744990E-01_dp,&
      2.9189754861993961E-01_dp,  9.5684893046540861E-03_dp,&
      -4.6899766144214212E-02_dp,  2.9892393239476815E-03_dp,&
      8.1317647341407617E-03_dp, -1.1496803534589862E-03_dp,&
      -1.5588269285254301E-03_dp,  5.6436279367429989E-04_dp,&
      1.6929026309404673E+01_dp, -2.2732969289472557E+01_dp,&
      1.5167967467215167E+01_dp, -5.0599918378658693E+00_dp,&
      7.1901703409584045E-02_dp,  6.0237191357615583E-01_dp,&
      -1.3337725393148273E-01_dp, -6.4361089066435215E-02_dp,&
      2.7783835281892384E-02_dp,  8.5230341553865878E-03_dp,&
      -6.6163986118154151E-03_dp,  2.1386671992928557E-04_dp,&
      4.7004720455020335E-04_dp,  2.9361444810427511E+01_dp,&
      -4.3304500814209732E+01_dp,  3.2853995310162027E+01_dp,&
      -1.4060189959048198E+01_dp,  2.4294238784534623E+00_dp,&
      7.2536802429446512E-01_dp, -4.3389039409549884E-01_dp,&
      -1.3185076785459927E-02_dp,  5.6402610165301202E-02_dp,&
      -1.1720630499261352E-03_dp, -1.0821082259544815E-02_dp,&
      3.2422174075987468E-03_dp, -5.9058577933889726E-05_dp,&
      6.4371986201665507E+01_dp, -1.0067505697246480E+02_dp,&
      8.2674796182294259E+01_dp, -4.0680037489638053E+01_dp,&
      1.0796803179141461E+01_dp, -9.8211559590887343E-02_dp,&
      -8.6947315752611443E-01_dp,  1.1519815681783646E-01_dp,&
      8.8122286786261939E-02_dp, -1.4859222473260573E-02_dp,&
      -1.7476608605390593E-02_dp,  7.7549011480211169E-03_dp,&
      -7.7298631180373564E-04_dp,  1.5512634652715025E+00_dp,&
      -5.7165382658867425E-01_dp, -6.8264917568339606E-02_dp,&
      3.4929866387098558E-02_dp,  9.6619672882347681E-03_dp,&
      -3.0449530315484925E-03_dp, -8.1140934294873649E-04_dp,&
      6.7166809665575506E-04_dp,  1.0553881582976249E-04_dp,&
      -1.6470352066950413E-04_dp, -7.5508519829998230E-05_dp,&
      7.2215264404265218E-05_dp, -1.1149724276607350E-05_dp,&
      1.8009352545445239E+00_dp, -8.5251974129151409E-01_dp,&
      7.6410708533761022E-03_dp,  6.3828560675533136E-02_dp,&
      -1.7202481171730995E-03_dp, -7.4505177561230681E-03_dp,&
      1.0572044356321228E-03_dp,  1.3218647507928171E-03_dp,&
      -3.0561817587206915E-04_dp, -3.0299445474565189E-04_dp,&
      5.5344610050092803E-05_dp,  8.1834190969111756E-05_dp,&
      -2.9680329778920826E-05_dp,  2.3527449274293573E+00_dp,&
      -1.5147537958221933E+00_dp,  2.3115619180855354E-01_dp,&
      1.1284255760843230E-01_dp, -3.6177847817668474E-02_dp,&
      -1.3539664544268073E-02_dp,  6.7019226962421333E-03_dp,&
      1.9653435924636472E-03_dp, -1.4109355286997759E-03_dp,&
      -4.1385824743472909E-04_dp,  4.1549350181307716E-04_dp,&
      -1.1959801065863547E-05_dp, -3.7255770501802784E-05_dp,&
      3.3224921863045691E+00_dp, -2.7860632861596644E+00_dp,&
      7.8052485998277177E-01_dp,  1.4483194966481014E-01_dp,&
      -1.1737504026440210E-01_dp, -1.1679042020471016E-02_dp,&
      1.9403002086844354E-02_dp,  6.6735285522260949E-04_dp,&
      -3.5780226530484115E-03_dp, -7.4687021790351436E-05_dp,&
      1.0364093806503746E-03_dp, -3.5212253142996770E-04_dp,&
      1.4459347180379926E-05_dp,  4.9188989071416884E+00_dp,&
      -5.0912381643341389E+00_dp,  2.0338384923721380E+00_dp,&
      3.4520315816372826E-02_dp, -2.7527398518237012E-01_dp,&
      2.6213290332193811E-02_dp,  4.1056443632656141E-02_dp,&
      -7.6033292189327589E-03_dp, -6.7068508750127874E-03_dp,&
      1.6134398217065211E-03_dp,  1.7099199099071764E-03_dp,&
      -1.0188154531831771E-03_dp,  1.6585315316805667E-04_dp,&
      7.5105710318956387E+00_dp, -9.1933741951079870E+00_dp,&
      4.7596295380909268E+00_dp, -5.7877693805389485E-01_dp,&
      -5.0633729223025081E-01_dp,  1.6657960364469732E-01_dp,&
      5.9754102084310180E-02_dp, -3.3655424854068201E-02_dp,&
      -7.8470267984706195E-03_dp,  6.4378442077806190E-03_dp,&
      1.7214765195746398E-03_dp, -2.1640300991494017E-03_dp,&
      5.0773164374421677E-04_dp,  1.1787767168757027E+01_dp,&
      -1.6479896456900001E+01_dp,  1.0474433181368957E+01_dp,&
      -2.6176038877413412E+00_dp, -6.1374050521456192E-01_dp,&
      5.1744487961845398E-01_dp,  1.0209694593808110E-02_dp,&
      -8.7555357959509111E-02_dp,  6.7878804533424266E-03_dp,&
      1.5788464928805319E-02_dp, -2.5231574145546760E-03_dp,&
      -3.0178852806152551E-03_dp,  1.1280928755338241E-03_dp,&
      1.9240897295596305E+01_dp, -2.9689511846963928E+01_dp,&
      2.2151809035042952E+01_dp, -8.1959918765769775E+00_dp,&
      2.3949103147685374E-01_dp,  1.0808974332068280E+00_dp,&
      -2.7708880902203492E-01_dp, -1.2364385254000407E-01_dp,&
      6.3036168861553526E-02_dp,  1.6885878085071752E-02_dp,&
      -1.6735028081352580E-02_dp,  2.0042184897993667E-03_dp,&
      7.5921274060652174E-04_dp,  3.3999306723020787E+01_dp,&
      -5.5636509752237529E+01_dp,  4.6326714382390342E+01_dp,&
      -2.1820258573048438E+01_dp,  4.3276152184902941E+00_dp,&
      1.2172727004629018E+00_dp, -9.0694960221451648E-01_dp,&
      3.1554492336115077E-02_dp,  1.2559453953809244E-01_dp,&
      -2.0064318592328701E-02_dp, -2.2560731494297201E-02_dp,&
      1.1385938701128969E-02_dp, -1.5858929811645066E-03_dp,&
      7.2814831077229968E+01_dp, -1.2072622608954870E+02_dp,&
      1.0532732283136377E+02_dp, -5.5787958513676813E+01_dp,&
      1.6603414490986122E+01_dp, -7.3755690320892586E-01_dp,&
      -1.4210730327967394E+00_dp,  3.7793417576420085E-01_dp,&
      9.1264545598547908E-02_dp, -4.8606229226741834E-02_dp,&
      -1.3188114627426912E-02_dp,  1.2197365335229388E-02_dp,&
      -2.2862081919550218E-03_dp,  2.7358153743749347E+02_dp,&
      -4.4385936209458333E+02_dp,  3.8212861114580585E+02_dp,&
      -2.0280907585143663E+02_dp,  6.3861686467341308E+01_dp,&
      -7.0005683460188894E+00_dp, -2.6528795806599486E+00_dp,&
      7.0810042010323637E-01_dp,  2.6455253738091272E-01_dp,&
      -7.9687526859254745E-02_dp, -5.7953598000465630E-02_dp,&
      3.2570226713321167E-02_dp, -4.4496456437617135E-03_dp,&
      2.5710198422843333E-01_dp, -7.6161818320978847E-02_dp,&
      -2.7674187351960450E-02_dp,  8.5898642065098691E-03_dp,&
      8.0536613015059072E-03_dp, -2.6174486149182770E-03_dp,&
      -3.2661823845164801E-03_dp,  9.2754900950433358E-04_dp,&
      2.1088398186065527E-03_dp, -1.1473435220555634E-03_dp,&
      -6.3993457458976226E-04_dp,  7.5761255017756373E-04_dp,&
      -2.0390887377095055E-04_dp,  8.4861453798468323E-01_dp,&
      -3.0105534605114320E-01_dp, -8.5111841975141406E-02_dp,&
      4.8094032800166463E-02_dp,  2.6040827909755683E-02_dp,&
      -1.7293930943123634E-02_dp, -1.0786968802803141E-02_dp,&
      6.6125719406229183E-03_dp,  9.1882171776696397E-03_dp,&
      -9.6727488215151876E-03_dp,  1.7446919122676161E-03_dp,&
      1.4418793705765515E-03_dp, -5.8840280185385851E-04_dp,&
      1.7047248048976706E+00_dp, -7.9815206934993621E-01_dp,&
      -1.2993709521910998E-01_dp,  1.7342171454714553E-01_dp,&
      3.9041789120375842E-02_dp, -6.9360101166953014E-02_dp,&
      -1.5275816598696328E-02_dp,  2.7921918147956529E-02_dp,&
      2.3894544071414819E-02_dp, -4.4643419061525462E-02_dp,&
      2.3898972121543908E-02_dp, -4.4247812420037886E-03_dp,&
      -1.6586226979854507E-04_dp,  3.1244261122013506E+00_dp,&
      -1.9657261428423933E+00_dp, -6.2195521540174485E-02_dp,&
      5.0922738991989702E-01_dp, -9.3639444916459660E-03_dp,&
      -2.1322782553698055E-01_dp,  1.5504548523492871E-02_dp,&
      8.8634058985128744E-02_dp,  3.9464172801052973E-02_dp,&
      -1.4553362414306942E-01_dp,  1.1134286669408361E-01_dp,&
      -3.8113881848227303E-02_dp,  4.9718947417133237E-03_dp,&
      5.6515577756942257E+00_dp, -4.6901233718069122E+00_dp,&
      4.9998880825294556E-01_dp,  1.2889768367120584E+00_dp,&
      -3.4014775985320350E-01_dp, -5.3198012841393982E-01_dp,&
      2.0501714262866522E-01_dp,  2.2366241606561824E-01_dp,&
      -1.2094839938125915E-02_dp, -3.4853283957718817E-01_dp,&
      3.5500068302721494E-01_dp, -1.5402651260478087E-01_dp,&
      2.6586199081616392E-02_dp,  1.0342072806880291E+01_dp,&
      -1.1025837566627148E+01_dp,  2.8320859276363368E+00_dp,&
      2.8353118375827213E+00_dp, -1.6643808637692934E+00_dp,&
      -1.0312784859484334E+00_dp,  9.2317469166909838E-01_dp,&
      4.1524786772077965E-01_dp, -4.1115535857203517E-01_dp,&
      -5.0528937194956192E-01_dp,  8.1638468056711944E-01_dp,&
      -4.3490956585830648E-01_dp,  8.7714392284929860E-02_dp,&
      1.9340679951428509E+01_dp, -2.5875245987552002E+01_dp,&
      1.1011927062253607E+01_dp,  5.0172148595146115E+00_dp,&
      -5.9922364423090562E+00_dp, -1.0100072524430268E+00_dp,&
      3.0463673833777176E+00_dp,  2.3135881603355785E-01_dp,&
      -1.9255994055152754E+00_dp,  3.7336390355963817E-01_dp,&
      1.0535281004009800E+00_dp, -8.2899219614509112E-01_dp,&
      2.0043276697370435E-01_dp,  3.7199991651764925E+01_dp,&
      -6.1293516863993275E+01_dp,  3.7969498172512104E+01_dp,&
      3.9173683745057399E+00_dp, -1.8033846800996791E+01_dp,&
      3.6542142546232963E+00_dp,  7.6853116612266872E+00_dp,&
      -2.5741014212521396E+00_dp, -5.6157937298118208E+00_dp,&
      5.6662223853193741E+00_dp, -1.5475935532793732E+00_dp,&
      -3.8649136037865522E-01_dp,  2.2567746326676860E-01_dp,&
      7.4213954026122252E+01_dp, -1.4804950267684572E+02_dp,&
      1.2495507784125873E+02_dp, -2.4776264817547396E+01_dp,&
      -4.2930419560902017E+01_dp,  2.9472395507465627E+01_dp,&
      1.0097402469676231E+01_dp, -1.5262809248340325E+01_dp,&
      -7.2300215031127406E+00_dp,  2.0357934514863608E+01_dp /)
    REAL(KIND=dp), DIMENSION(400), PARAMETER :: c02 = (/ &
      -1.4561539999459816E+01_dp,  4.8544787554718685E+00_dp,&
      -6.3387746031848102E-01_dp,  1.5731981998839461E+02_dp,&
      -3.7106120150889751E+02_dp,  4.0538316285381757E+02_dp,&
      -1.9862435563464234E+02_dp, -4.2159595434504858E+01_dp,&
      1.1123116084380190E+02_dp, -3.2616679588589356E+01_dp,&
      -3.5941786883652419E+01_dp,  2.2789754163995845E+01_dp,&
      1.8371151060137738E+01_dp, -2.9170310760936879E+01_dp,&
      1.4831275429341607E+01_dp, -2.8863895844841174E+00_dp,&
      4.0083846543407321E+02_dp, -1.0677816135164219E+03_dp,&
      1.4026777969921070E+03_dp, -1.0380548329826443E+03_dp,&
      3.1150334049658153E+02_dp,  1.6486545559165077E+02_dp,&
      -1.8516523777840163E+02_dp,  1.6518152941373941E+01_dp,&
      5.6979659267802994E+01_dp, -1.7647184173214058E+01_dp,&
      -2.1103736761399592E+01_dp,  1.7917739029353150E+01_dp,&
      -4.3165125341576296E+00_dp,  2.1062827676264901E+00_dp,&
      -6.5430642611969270E-01_dp, -2.2384079169103308E-01_dp,&
      8.2390744712270661E-02_dp,  6.6262172019689369E-02_dp,&
      -2.6633980874697070E-02_dp, -2.7111579335168608E-02_dp,&
      9.7002448728929856E-03_dp,  1.8782298233897637E-02_dp,&
      -1.2801673272427807E-02_dp, -3.2239388584569518E-03_dp,&
      5.7460187984439293E-03_dp, -1.6569702053578139E-03_dp,&
      2.7457621605201994E+00_dp, -1.2692591549164844E+00_dp,&
      -2.2947324376136316E-01_dp,  2.7323021428913846E-01_dp,&
      7.5164358453219907E-02_dp, -1.0818724938592214E-01_dp,&
      -3.1583296898807685E-02_dp,  4.3084465728622491E-02_dp,&
      4.1482878645338658E-02_dp, -6.7933308032627218E-02_dp,&
      3.1679176585648089E-02_dp, -3.3812888555787906E-03_dp,&
      -1.0360838277370485E-03_dp,  4.2942636924137831E+00_dp,&
      -2.9557754507532943E+00_dp, -7.3406311761844992E-02_dp,&
      8.0895545247383172E-01_dp, -2.2458835272879123E-03_dp,&
      -3.4387294218152459E-01_dp,  1.3473260271944267E-02_dp,&
      1.4295454214800754E-01_dp,  7.4353360154718565E-02_dp,&
      -2.3866533192162417E-01_dp,  1.7424538034935477E-01_dp,&
      -5.6402793269764662E-02_dp,  6.6940377272607246E-03_dp,&
      7.4095497593660964E+00_dp, -6.9353228276198537E+00_dp,&
      8.3586764619106058E-01_dp,  2.0316025967186220E+00_dp,&
      -5.1322687373588960E-01_dp, -8.6661941980502355E-01_dp,&
      3.0532915478203720E-01_dp,  3.6856373700427469E-01_dp,&
      1.4157957888213758E-02_dp, -6.0168493995655148E-01_dp,&
      5.8832895145574027E-01_dp, -2.4863584549454967E-01_dp,&
      4.1841159809315402E-02_dp,  1.3476526156516739E+01_dp,&
      -1.6130585523051849E+01_dp,  4.4061937443889301E+00_dp,&
      4.4430534169571274E+00_dp, -2.5570162915545356E+00_dp,&
      -1.7175195154266936E+00_dp,  1.4376957431381163E+00_dp,&
      7.1766464786951778E-01_dp, -5.9257675525589137E-01_dp,&
      -9.6205557835963396E-01_dp,  1.4268339903708875E+00_dp,&
      -7.4132394072868890E-01_dp,  1.4725124240356049E-01_dp,&
      2.5398852071685731E+01_dp, -3.7572387919172385E+01_dp,&
      1.6636186660780080E+01_dp,  7.9929423757903564E+00_dp,&
      -9.2705027979453689E+00_dp, -1.9503834582624382E+00_dp,&
      4.8813044930751017E+00_dp,  6.0504538533406726E-01_dp,&
      -3.0998271168328229E+00_dp,  2.6411993677728995E-01_dp,&
      2.0646553446087554E+00_dp, -1.5110641191317531E+00_dp,&
      3.5664182271533801E-01_dp,  4.9435136714811982E+01_dp,&
      -8.8597920159101832E+01_dp,  5.6353728135153659E+01_dp,&
      7.6467691457190172E+00_dp, -2.8439752545361475E+01_dp,&
      4.3311736788952206E+00_dp,  1.3061842905815471E+01_dp,&
      -3.3304225848462199E+00_dp, -9.9097932669193582E+00_dp,&
      8.8616730772757997E+00_dp, -1.6316122579259393E+00_dp,&
      -1.1613800601835296E+00_dp,  4.8402556303837513E-01_dp,&
      9.9762188382444535E+01_dp, -2.1339565978974167E+02_dp,&
      1.8334922150404742E+02_dp, -2.8939147749315442E+01_dp,&
      -7.2890617526936850E+01_dp,  4.2713837113284967E+01_dp,&
      2.2721643681065917E+01_dp, -2.3847907629562187E+01_dp,&
      -1.8549681716076698E+01_dp,  3.8300080834612302E+01_dp,&
      -2.4638773637155111E+01_dp,  7.2597188948475342E+00_dp,&
      -7.5984073743085179E-01_dp,  2.1197486994109249E+02_dp,&
      -5.3038671301074385E+02_dp,  5.8971315115622019E+02_dp,&
      -2.6418140422078056E+02_dp, -1.1373718823220391E+02_dp,&
      1.9211692177267807E+02_dp, -2.7169765446157413E+01_dp,&
      -8.4379798332666354E+01_dp,  2.7260492365389506E+01_dp,&
      6.7376634921239955E+01_dp, -8.0434216086053510E+01_dp,&
      3.8099371156341618E+01_dp, -7.1604086942794369E+00_dp,&
      5.0533178791676357E+02_dp, -1.4141062067950954E+03_dp,&
      1.9146864532541163E+03_dp, -1.3990795604874068E+03_dp,&
      2.9162932525768673E+02_dp,  4.3994914666230397E+02_dp,&
      -3.9085522986369466E+02_dp, -8.2879522036785254E+00_dp,&
      2.0611979984798336E+02_dp, -1.2695223934311078E+02_dp,&
      7.8185610329907984E+00_dp,  2.1623126780307519E+01_dp,&
      -7.0837826672811230E+00_dp,  1.8793380650637498E+03_dp,&
      -5.3818022034402502E+03_dp,  7.8117470886846950E+03_dp,&
      -6.8754098839868311E+03_dp,  3.4302358931127628E+03_dp,&
      -3.3780948387252249E+02_dp, -7.1180847792872282E+02_dp,&
      3.4000649096567986E+02_dp,  9.5803695633720451E+01_dp,&
      -1.0706529225064254E+02_dp, -1.7431236331464156E+01_dp,&
      4.1838749445416617E+01_dp, -1.2304788741999435E+01_dp,&
      2.9693722288060909E-01_dp, -4.0503765415150689E-02_dp,&
      -2.3374347340216213E-02_dp,  2.4649712953909672E-03_dp,&
      5.4970824224843205E-03_dp,  4.8287397465528485E-04_dp,&
      -3.3174719260196080E-03_dp,  5.7632661521211167E-04_dp,&
      1.2446956470640266E-03_dp, -2.6247595657409464E-04_dp,&
      -7.8065068838181328E-04_dp,  6.0684870308787948E-04_dp,&
      -1.4446753144174501E-04_dp,  1.0132434862618578E+00_dp,&
      -1.7523694112647539E-01_dp, -9.2556727590122526E-02_dp,&
      1.9924243846417989E-02_dp,  2.4530815588521580E-02_dp,&
      -1.2987424585305914E-03_dp, -1.5482592236816674E-02_dp,&
      4.0053098423139359E-03_dp,  6.6615628610347507E-03_dp,&
      -3.1644246597228834E-03_dp, -2.5541557205489267E-03_dp,&
      2.6188403853415528E-03_dp, -6.7853784962391733E-04_dp,&
      2.1711398209555157E+00_dp, -5.2976846220389984E-01_dp,&
      -2.4286083825272872E-01_dp,  9.7659023189940866E-02_dp,&
      7.3533850048130775E-02_dp, -1.9551255900919134E-02_dp,&
      -4.9877998593511656E-02_dp,  1.9506507612301518E-02_dp,&
      2.5578617398895529E-02_dp, -2.0627500236759959E-02_dp,&
      -1.7452808686614640E-03_dp,  6.8372280060992642E-03_dp,&
      -2.1043878950266792E-03_dp,  4.3616246933218727E+00_dp,&
      -1.5063751348282000E+00_dp, -5.7486437306662230E-01_dp,&
      3.8265267045798651E-01_dp,  1.8943507598881318E-01_dp,&
      -1.0722794884839189E-01_dp, -1.4175159806446239E-01_dp,&
      8.1999399523214028E-02_dp,  8.7843503751238977E-02_dp,&
      -1.0363055386269830E-01_dp,  2.6555214303521067E-02_dp,&
      9.8867863529191486E-03_dp, -4.9996570844378710E-03_dp,&
      8.8539699844047153E+00_dp, -4.1984202894252292E+00_dp,&
      -1.2428634026751828E+00_dp,  1.3448235920919132E+00_dp,&
      4.0461929150339243E-01_dp, -4.5733788385795898E-01_dp,&
      -3.5409248304565399E-01_dp,  3.1850290939545917E-01_dp,&
      2.7325234985046681E-01_dp, -4.5652048431894104E-01_dp,&
      2.2079297546908300E-01_dp, -2.9478620765750235E-02_dp,&
      -5.1895829058320950E-03_dp,  1.8605315387814702E+01_dp,&
      -1.1760213994880345E+01_dp, -2.3008734839719169E+00_dp,&
      4.4982007461290801E+00_dp,  5.2311515349431637E-01_dp,&
      -1.7368601503569794E+00_dp, -7.0107041944374471E-01_dp,&
      1.1883211570158956E+00_dp,  7.2483817123112193E-01_dp,&
      -1.8304322778786442E+00_dp,  1.2325954726922457E+00_dp,&
      -3.6723777123307916E-01_dp,  3.7564073010966953E-02_dp,&
      4.1004452121855905E+01_dp, -3.3883822198342756E+01_dp,&
      -2.4660978577221768E+00_dp,  1.4765625376284200E+01_dp,&
      -1.3297140620311558E+00_dp, -6.0822406363831671E+00_dp,&
      -4.2952269081750616E-01_dp,  4.3254569166267594E+00_dp,&
      1.0855910889639027E+00_dp, -6.4825677999145492E+00_dp,&
      5.6839301455511970E+00_dp, -2.2574263920344908E+00_dp,&
      3.6057185979037681E-01_dp,  9.6016519297543255E+01_dp,&
      -1.0286313276941833E+02_dp,  8.6326255820698279E+00_dp,&
      4.8051414088075639E+01_dp, -1.7261755163232731E+01_dp,&
      -1.8906364729866077E+01_dp,  6.5295315087461869E+00_dp,&
      1.5019872639436668E+01_dp, -5.3377881739836885E+00_dp,&
      -1.6841583305043358E+01_dp,  2.1034770339353262E+01_dp,&
      -1.0362819237186125E+01_dp,  2.0054323897259563E+00_dp,&
      2.4319276556516138E+02_dp, -3.3866420245722674E+02_dp,&
      9.9763909609854409E+01_dp,  1.5027385014986709E+02_dp,&
      -1.2031406869863480E+02_dp, -3.7241043779740231E+01_dp,&
      5.5370114102753035E+01_dp,  4.1642802529325209E+01_dp,&
      -7.1093015443508278E+01_dp,  6.7827155853688756E+00_dp,&
      4.1302786677291920E+01_dp, -3.0651138576665996E+01_dp,&
      7.3344210412222370E+00_dp,  6.8664225651899642E+02_dp,&
      -1.2587153573562946E+03_dp,  7.5483998218097088E+02_dp,&
      3.3826607275853559E+02_dp, -7.3529640146317138E+02_dp,&
      2.1476437158088527E+02_dp,  2.4921315487310400E+02_dp,&
      -3.9413815726828012E+01_dp, -4.0163723829203337E+02_dp,&
      4.8255471931724361E+02_dp, -2.5345967916897050E+02_dp,&
      5.9457400566630319E+01_dp, -3.2565309975817023E+00_dp,&
      2.3594394496881578E+03_dp, -5.6892144384380426E+03_dp,&
      5.9833743128811575E+03_dp, -2.0231849696975166E+03_dp,&
      -2.6090219824590527E+03_dp,  3.9842874719195775E+03_dp,&
      -2.2508330708123085E+03_dp,  3.7033823002291069E+02_dp,&
      -3.1902484361391252E+01_dp,  4.8115948604471549E+02_dp,&
      -5.8171212108552209E+02_dp,  2.8977583679044648E+02_dp,&
      -5.6846166243147294E+01_dp,  2.4525299653129071E+00_dp,&
      -3.5637069115264741E-01_dp, -2.0104566416523126E-01_dp,&
      2.6779238503295028E-02_dp,  4.8939905628270081E-02_dp,&
      2.4792825698676576E-03_dp, -2.9869803743560470E-02_dp,&
      5.8425383143963141E-03_dp,  1.1635030317581013E-02_dp,&
      -3.2996081706425048E-03_dp, -6.5241888531386676E-03_dp,&
      5.3759987970630431E-03_dp, -1.3062621653998619E-03_dp,&
      3.4788507276835041E+00_dp, -8.2208076410363606E-01_dp,&
      -3.8961998160802858E-01_dp,  1.4052212399279815E-01_dp,&
      1.1721421643645048E-01_dp, -2.4240480071210848E-02_dp,&
      -7.7735694825084065E-02_dp,  2.7075038378894713E-02_dp,&
      3.7905398760191382E-02_dp, -2.6448267280219422E-02_dp,&
      -6.6537554846141100E-03_dp,  1.1815563026552645E-02_dp,&
      -3.3909248684933454E-03_dp,  6.1543722219710162E+00_dp,&
      -2.2568137161530410E+00_dp, -8.7825522652867727E-01_dp,&
      5.6706955570340978E-01_dp,  2.9972495377368685E-01_dp,&
      -1.5163443438723695E-01_dp, -2.1990818300811518E-01_dp,&
      1.1667113151985914E-01_dp,  1.3131118147888812E-01_dp,&
      -1.4121395152578772E-01_dp,  2.5319103809441905E-02_dp,&
      2.1127868156859679E-02_dp, -8.5766267142760727E-03_dp,&
      1.2157059618437597E+01_dp, -6.2094066481490042E+00_dp,&
      -1.8855484638051927E+00_dp,  1.9867289226165976E+00_dp,&
      6.5796325888957663E-01_dp, -6.6000282590724135E-01_dp,&
      -5.5900550981956598E-01_dp,  4.5520937544335560E-01_dp,&
      4.1683712730881195E-01_dp, -6.3698977586433025E-01_dp,&
      2.7310677955566764E-01_dp, -1.5884701923367262E-02_dp,&
      -1.3177785753804044E-02_dp,  2.5597273547085955E+01_dp,&
      -1.7189605326643218E+01_dp, -3.5459040501619303E+00_dp,&
      6.5908292406530711E+00_dp,  9.9536824328795226E-01_dp,&
      -2.5268919270014778E+00_dp, -1.1778857876024893E+00_dp,&
      1.6973853342765182E+00_dp,  1.1755279384183819E+00_dp,&
      -2.6199621348432438E+00_dp,  1.6332594658854560E+00_dp,&
      -4.2930601501144439E-01_dp,  3.0668797181816410E-02_dp,&
      5.6890294980110632E+01_dp, -4.8935675805853023E+01_dp,&
      -4.3928513391337605E+00_dp,  2.1493900023941237E+01_dp,&
      -9.2561396202764701E-01_dp, -8.9832110266967078E+00_dp,&
      -1.2945023961231210E+00_dp,  6.1971050020908542E+00_dp,&
      2.4122252521897924E+00_dp, -9.7999682646599489E+00_dp,&
      7.9639626818274714E+00_dp, -2.9529703827741800E+00_dp,&
      4.3437020501980872E-01_dp,  1.3418598639315178E+02_dp,&
      -1.4639064170062176E+02_dp,  7.9688822975923612E+00_dp,&
      6.9963146941480773E+01_dp, -1.9798038249044243E+01_dp,&
      -2.9411357486525350E+01_dp,  6.3698447544865715E+00_dp /)
    REAL(KIND=dp), DIMENSION(400), PARAMETER :: c03 = (/ &
      2.2105999150585166E+01_dp, -2.2551854693657698E+00_dp,&
      -3.0252257829864192E+01_dp,  3.2690274416605568E+01_dp,&
      -1.5006475584487145E+01_dp,  2.7494470603903425E+00_dp,&
      3.4045498207568971E+02_dp, -4.7133760448357106E+02_dp,&
      1.1327780050673924E+02_dp,  2.2523014020490135E+02_dp,&
      -1.4330073759632975E+02_dp, -7.7468008856218788E+01_dp,&
      6.7539805363605566E+01_dp,  7.1533214810405553E+01_dp,&
      -7.4426510844032748E+01_dp, -4.0867003585558841E+01_dp,&
      9.5519686500466420E+01_dp, -5.6748771136081224E+01_dp,&
      1.2334798380537688E+01_dp,  9.5030306864288514E+02_dp,&
      -1.6854513316308066E+03_dp,  8.3319880141164163E+02_dp,&
      6.4915029955371108E+02_dp, -8.9512232378940689E+02_dp,&
      1.1151023474977780E+01_dp,  4.0937510009100765E+02_dp,&
      1.1637171492092472E+02_dp, -6.0214982550573529E+02_dp,&
      4.3122386361734880E+02_dp, -4.9217676673429537E+01_dp,&
      -7.4489120696598462E+01_dp,  2.7655792789203122E+01_dp,&
      3.0475256987427633E+03_dp, -7.0066488989471227E+03_dp,&
      6.0846071431580522E+03_dp,  2.4654355076157873E+02_dp,&
      -5.0986530061647118E+03_dp,  3.5130484940953365E+03_dp,&
      8.2990840282523322E+02_dp, -1.5418369657761907E+03_dp,&
      -1.6637950125880895E+03_dp,  4.0893665083154874E+03_dp,&
      -3.3614370283246321E+03_dp,  1.3720459965327138E+03_dp,&
      -2.3508756179939306E+02_dp,  1.3543997817893476E+04_dp,&
      -3.8219712441683943E+04_dp,  5.1571871314927179E+04_dp,&
      -3.5616662338932052E+04_dp, -5.5991015438729119E+02_dp,&
      3.0560634575586610E+04_dp, -3.8034115203375237E+04_dp,&
      2.7684234144350565E+04_dp, -1.3389558668848904E+04_dp,&
      3.9243885697378510E+03_dp, -2.1067618872891239E+02_dp,&
      -3.4140069782619736E+02_dp,  1.0514659620128288E+02_dp,&
      3.1070151999119766E-01_dp, -1.5418900622590423E-02_dp,&
      -1.3385402496793672E-02_dp,  4.0104520214988143E-03_dp,&
      -4.0137665511710309E-03_dp,  6.0634593372117873E-03_dp,&
      -6.0335242023330767E-03_dp,  5.1416827988300775E-03_dp,&
      -4.2479700168415501E-03_dp,  3.0278142624862105E-03_dp,&
      -1.5771186418613689E-03_dp,  5.1319284677445436E-04_dp,&
      -7.7760892249170883E-05_dp,  1.0738088654114670E+00_dp,&
      -6.9011733174083323E-02_dp, -5.8571528350665605E-02_dp,&
      2.0236912375192588E-02_dp, -1.7748843399198674E-02_dp,&
      2.7179396127280601E-02_dp, -2.7370169804265022E-02_dp,&
      2.3012442691885241E-02_dp, -1.8752163526337080E-02_dp,&
      1.3306638853040436E-02_dp, -6.8968599129474242E-03_dp,&
      2.2148106641816261E-03_dp, -3.2713412872578977E-04_dp,&
      2.3587191042167146E+00_dp, -2.1892062702350010E-01_dp,&
      -1.7981032764373062E-01_dp,  7.4266715447842788E-02_dp,&
      -5.5755162509579577E-02_dp,  8.6197238895336856E-02_dp,&
      -8.8434819428318698E-02_dp,  7.3044554544590024E-02_dp,&
      -5.8253678591248331E-02_dp,  4.1045428330803928E-02_dp,&
      -2.1141989971175491E-02_dp,  6.6663194149084379E-03_dp,&
      -9.4698866181811470E-04_dp,  4.9095569364132761E+00_dp,&
      -6.5653821881663610E-01_dp, -5.1926150788332637E-01_dp,&
      2.5545610262410418E-01_dp, -1.6752475780931578E-01_dp,&
      2.5748215779005090E-01_dp, -2.7035809285075363E-01_dp,&
      2.1945387900012747E-01_dp, -1.7026830150641939E-01_dp,&
      1.1880809008201974E-01_dp, -6.0862416134477300E-02_dp,&
      1.8834269810612231E-02_dp, -2.5543692261934688E-03_dp,&
      1.0428495867052428E+01_dp, -1.9432357381147800E+00_dp,&
      -1.4692140211872782E+00_dp,  8.6331929124290829E-01_dp,&
      -5.0585478836643105E-01_dp,  7.5530509804086554E-01_dp,&
      -8.1607591368868604E-01_dp,  6.5190588923750659E-01_dp,&
      -4.8767325783852300E-01_dp,  3.3517775887994816E-01_dp,&
      -1.7093793530670998E-01_dp,  5.1961983765136888E-02_dp,&
      -6.6742743223630954E-03_dp,  2.3180306769496532E+01_dp,&
      -5.8479292153610993E+00_dp, -4.1741626920582231E+00_dp,&
      2.9735016542960855E+00_dp, -1.5967301459661869E+00_dp,&
      2.2376916273966088E+00_dp, -2.5039185219249935E+00_dp,&
      1.9742019032983462E+00_dp, -1.4031603691615286E+00_dp,&
      9.3839465982933501E-01_dp, -4.7641240927423684E-01_dp,&
      1.4259287153642736E-01_dp, -1.7178530431716743E-02_dp,&
      5.4807444543914080E+01_dp, -1.8428736049538433E+01_dp,&
      -1.2159840401125734E+01_dp,  1.0799384839018646E+01_dp,&
      -5.5132864599838030E+00_dp,  6.8824744137130160E+00_dp,&
      -8.0181186424414683E+00_dp,  6.2811263339245347E+00_dp,&
      -4.1414130588660143E+00_dp,  2.6139308618654691E+00_dp,&
      -1.3115897057255532E+00_dp,  3.8743455254506703E-01_dp,&
      -4.3118917292752940E-02_dp,  1.4054749225956735E+02_dp,&
      -6.2916617869030020E+01_dp, -3.6937557851795980E+01_dp,&
      4.2950396779726667E+01_dp, -2.2089900157704871E+01_dp,&
      2.2746517401589692E+01_dp, -2.7468240068761069E+01_dp,&
      2.1674175015936296E+01_dp, -1.2736556725406421E+01_dp,&
      6.9130613132454206E+00_dp, -3.2447386195397336E+00_dp,&
      9.1820903948266930E-01_dp, -8.4183705014911528E-02_dp,&
      4.0295699079346474E+02_dp, -2.4412729159645014E+02_dp,&
      -1.1742697356266694E+02_dp,  1.9701773109595578E+02_dp,&
      -1.1199516755711144E+02_dp,  8.6331114706981623E+01_dp,&
      -1.0268037391817907E+02_dp,  8.3451889210861594E+01_dp,&
      -4.0566080642770373E+01_dp,  1.1849996075031711E+01_dp,&
      -1.5608524774197274E+00_dp, -6.5060728335316376E-01_dp,&
      3.8305022351595713E-01_dp,  1.3662736150558032E+03_dp,&
      -1.1715066138716820E+03_dp, -3.5576387006757238E+02_dp,&
      1.1360365394454777E+03_dp, -8.3466379882382341E+02_dp,&
      4.6643712739732109E+02_dp, -4.1192955562637843E+02_dp,&
      3.3476739896275893E+02_dp, -9.8910890189152127E+01_dp,&
      -9.7849413214247178E+01_dp,  1.3013359839770587E+02_dp,&
      -6.7589528042984526E+01_dp,  1.4836210780641425E+01_dp,&
      6.2804767862049530E+03_dp, -8.4385527636566749E+03_dp,&
      7.8564045247073238E+02_dp,  9.2835293778113901E+03_dp,&
      -1.2752206647198502E+04_dp,  8.9081514173809228E+03_dp,&
      -2.5221109786186362E+03_dp, -2.7419974694190250E+03_dp,&
      5.8446477853057622E+03_dp, -6.5356503656889390E+03_dp,&
      4.9036628540405673E+03_dp, -2.2739363291736840E+03_dp,&
      4.8962251827596492E+02_dp,  2.5741763632522208E+00_dp,&
      -1.3688666727294529E-01_dp, -1.1813071559062696E-01_dp,&
      3.6811630925133884E-02_dp, -3.5503146549161348E-02_dp,&
      5.3860501688968561E-02_dp, -5.3757184624549099E-02_dp,&
      4.5644883856042782E-02_dp, -3.7578566225404821E-02_dp,&
      2.6754376121288694E-02_dp, -1.3916750691682382E-02_dp,&
      4.5126084600130203E-03_dp, -6.7926878267826000E-04_dp,&
      3.7683048696758537E+00_dp, -3.3588966026720962E-01_dp,&
      -2.7816323679430133E-01_dp,  1.1023403695203432E-01_dp,&
      -8.5451463047366089E-02_dp,  1.3240533806228441E-01_dp,&
      -1.3509314769377362E-01_dp,  1.1202128312859926E-01_dp,&
      -8.9894231488644935E-02_dp,  6.3470572830902017E-02_dp,&
      -3.2727234081501758E-02_dp,  1.0359096690123611E-02_dp,&
      -1.4853875358901360E-03_dp,  6.9727268104466820E+00_dp,&
      -9.7719657754485867E-01_dp, -7.7724485344015970E-01_dp,&
      3.7367281005024411E-01_dp, -2.4776335609741895E-01_dp,&
      3.8410532223734450E-01_dp, -4.0146502572297815E-01_dp,&
      3.2637402854653347E-01_dp, -2.5462309800085997E-01_dp,&
      1.7806400215143009E-01_dp, -9.1212601775455024E-02_dp,&
      2.8257494209588008E-02_dp, -3.8517458799869888E-03_dp,&
      1.4477920229338737E+01_dp, -2.8534043059134824E+00_dp,&
      -2.1721992056082700E+00_dp,  1.2466640681189123E+00_dp,&
      -7.3460581190645624E-01_dp,  1.1126441446066606E+00_dp,&
      -1.1960225615602900E+00_dp,  9.5605186786242724E-01_dp,&
      -7.2049164256777010E-01_dp,  4.9716022632596263E-01_dp,&
      -2.5348072114831677E-01_dp,  7.7049718869009298E-02_dp,&
      -9.9385241865572148E-03_dp,  3.2252510027816605E+01_dp,&
      -8.4617202290005658E+00_dp, -6.1023655425242858E+00_dp,&
      4.2181327536141584E+00_dp, -2.2670906799979624E+00_dp,&
      3.2527495661814285E+00_dp, -3.6180228600037601E+00_dp,&
      2.8512301914963061E+00_dp, -2.0497082727121163E+00_dp,&
      1.3825637028872737E+00_dp, -7.0227883206852093E-01_dp,&
      2.0995045275702587E-01_dp, -2.5391882706450129E-02_dp,&
      7.6671906271617928E+01_dp, -2.6191644580623201E+01_dp,&
      -1.7589785027199280E+01_dp,  1.4956788717489948E+01_dp,&
      -7.5745756904485431E+00_dp,  9.8459676211007903E+00_dp,&
      -1.1400125550680388E+01_dp,  8.9043865524691590E+00_dp,&
      -5.9853757336841316E+00_dp,  3.8609879714924378E+00_dp,&
      -1.9491156032947727E+00_dp,  5.7551390258433999E-01_dp,&
      -6.4555191580898369E-02_dp,  1.9670731225175837E+02_dp,&
      -8.7092067535475124E+01_dp, -5.2912261467103285E+01_dp,&
      5.7371747173768014E+01_dp, -2.8714960392858028E+01_dp,&
      3.1727692420638270E+01_dp, -3.8282028198777425E+01_dp,&
      2.9960818291058100E+01_dp, -1.8238144214054639E+01_dp,&
      1.0608034141638724E+01_dp, -5.2023410679539275E+00_dp,&
      1.5132791560087548E+00_dp, -1.5181291183013737E-01_dp,&
      5.5714705001563368E+02_dp, -3.2263481480447831E+02_dp,&
      -1.6778611836804163E+02_dp,  2.4762291567849428E+02_dp,&
      -1.3146191026312459E+02_dp,  1.1356350617240891E+02_dp,&
      -1.3983007561219821E+02_dp,  1.1183958095997933E+02_dp,&
      -5.8395910760663028E+01_dp,  2.4697691761515443E+01_dp,&
      -9.5671291643381515E+00_dp,  2.2817076345157226E+00_dp,&
      -5.4636457220685601E-02_dp,  1.8121516134579620E+03_dp,&
      -1.4090173856477702E+03_dp, -5.4917522369887411E+02_dp,&
      1.2757799353011990E+03_dp, -7.9838106640274214E+02_dp,&
      5.0012033797473867E+02_dp, -5.6022159293750929E+02_dp,&
      4.6860203326723456E+02_dp, -1.8680839076443570E+02_dp,&
      -1.5501517335518674E+01_dp,  5.8637375937772674E+01_dp,&
      -3.1899570998327739E+01_dp,  7.2944841057857248E+00_dp,&
      7.3075215016852726E+03_dp, -8.0716600625892179E+03_dp,&
      -1.3185554249594411E+03_dp,  8.6801383834222961E+03_dp,&
      -7.6321624820157049E+03_dp,  3.8266089588782174E+03_dp,&
      -2.2802301436819621E+03_dp,  1.6446395857376338E+03_dp,&
      6.5040243939953628E+00_dp, -1.6194859543158686E+03_dp,&
      1.7774545762751463E+03_dp, -9.1126046751063029E+02_dp,&
      1.9853994304916299E+02_dp,  4.4936883720346559E+04_dp,&
      -7.7985520384416654E+04_dp,  2.9091936453752249E+04_dp,&
      8.2600120859143732E+04_dp, -1.6542933631024099E+05_dp,&
      1.5026229391384314E+05_dp, -4.6996941258423743E+04_dp,&
      -7.8034667778134157E+04_dp,  1.5828259652379624E+05_dp,&
      -1.6452541439182087E+05_dp,  1.1259380546309771E+05_dp,&
      -4.8049115944930432E+04_dp,  9.6994935473796631E+03_dp,&
      3.1427203235400236E-01_dp, -1.7052748574378306E-09_dp,&
      -2.0798188655925094E-02_dp,  2.2230997852694379E-02_dp,&
      -2.2254212098492477E-02_dp,  4.4543234851814424E-03_dp,&
      4.5140639377175884E-02_dp, -1.2840956067326359E-01_dp,&
      2.1871157485341305E-01_dp, -2.6051786656559134E-01_dp,&
      2.1339394919157553E-01_dp, -1.0839462535404525E-01_dp,&
      2.5713078268260792E-02_dp,  1.0898619734312922E+00_dp,&
      -7.7200614526554230E-09_dp, -9.3932730256275687E-02_dp,&
      1.0040375386032285E-01_dp, -9.8022840144582693E-02_dp,&
      1.4771149788048780E-02_dp,  2.1228972641591992E-01_dp,&
      -5.8873273993490682E-01_dp,  9.9111659249563344E-01_dp,&
      -1.1700776653422673E+00_dp,  9.5074998676885270E-01_dp,&
      -4.7927322262063310E-01_dp,  1.1286696036611708E-01_dp,&
      2.4099686080266722E+00_dp, -2.4887589540378409E-08_dp,&
      -3.0179173326782494E-01_dp,  3.2258169443715523E-01_dp,&
      -3.0369141944218259E-01_dp,  2.3279662310325418E-02_dp,&
      7.1970589930972140E-01_dp, -1.9299446920382815E+00_dp,&
      3.1967747715415227E+00_dp, -3.7261628430281388E+00_dp,&
      2.9923091861400359E+00_dp, -1.4913736558069244E+00_dp,&
      3.4734441592637294E-01_dp,  5.0643496588107419E+00_dp,&
      -7.5994415175293662E-08_dp, -9.1798921280659829E-01_dp,&
      9.8122645165553024E-01_dp, -8.8561110596829973E-01_dp,&
      -1.1263108188627524E-02_dp,  2.3148160729056904E+00_dp,&
      -5.9940875579716151E+00_dp,  9.7525037514957766E+00_dp,&
      -1.1202621964616382E+01_dp,  8.8723971586232349E+00_dp,&
      -4.3615551235670349E+00_dp,  1.0019327026030862E+00_dp,&
      1.0890417035686898E+01_dp, -2.2965810094845916E-07_dp,&
      -2.7617022482760696E+00_dp,  2.9519410935497650E+00_dp /)
    REAL(KIND=dp), DIMENSION(13, 22, 5), PARAMETER :: &
      coefdata = RESHAPE ((/ c01, c02, c03, c04 /),(/13,22,5/))

    INTEGER                                  :: irange

     IF(Rc <=  1.0000000000000000E+01_dp) Rc =   1.0000000000000000E+01_dp
     IF(Rc >=  4.6752176284790039E+04_dp) Rc =   4.6752176284790039E+04_dp
     IF(Rc >=  1.0000000000000000E+01_dp .AND. Rc <=  2.8424619208291968E+01_dp) THEN
       irange =  1
       L_b =  1.0000000000000000E+01_dp
       U_b =  2.8424619208291968E+01_dp
     END IF
     IF(Rc >=  2.8424619208291968E+01_dp .AND. Rc <=  3.0981600906727294E+02_dp) THEN
       irange =  2
       L_b =  2.8424619208291968E+01_dp
       U_b =  3.0981600906727294E+02_dp
     END IF
     IF(Rc >=  3.0981600906727294E+02_dp .AND. Rc <=  3.3622492755925045E+03_dp) THEN
       irange =  3
       L_b =  3.0981600906727294E+02_dp
       U_b =  3.3622492755925045E+03_dp
     END IF
     IF(Rc >=  3.3622492755925045E+03_dp .AND. Rc <=  1.6183902110703035E+04_dp) THEN
       irange =  4
       L_b =  3.3622492755925045E+03_dp
       U_b =  1.6183902110703035E+04_dp
     END IF
     IF(Rc >=  1.6183902110703035E+04_dp .AND. Rc <=  4.6752176284790039E+04_dp) THEN
       irange =  5
       L_b =  1.6183902110703035E+04_dp
       U_b =  4.6752176284790039E+04_dp
     END IF

     fit_coef(1:13,1:22) = coefdata(1:13,1:22,irange)

   END SUBROUTINE get_fit_coef_k11

! *****************************************************************************
!> \brief ...
!> \param Rc ...
!> \param L_b ...
!> \param U_b ...
!> \param fit_coef ...
! *****************************************************************************
   SUBROUTINE get_fit_coef_k12(Rc,L_b,U_b,fit_coef)
    REAL(KIND=dp)                            :: Rc, L_b, U_b
    REAL(KIND=dp), DIMENSION(13, 24)         :: fit_coef

    REAL(KIND=dp), DIMENSION(360), PARAMETER :: c04 = (/ &
      -3.4185637511153556E+02_dp,  3.0668182445881496E+02_dp,&
      -4.1209664021967029E+02_dp,  3.7330789502174974E+02_dp,&
      -2.3245333897278655E+02_dp,  1.3019687281058577E+02_dp,&
      -7.4166370900253412E+01_dp,  3.1434861912054572E+01_dp,&
      -6.1992398402862987E+00_dp,  3.3062345634073872E+03_dp,&
      -2.8380291130889623E+03_dp, -1.1095051926787444E+03_dp,&
      2.9595972359353800E+03_dp, -2.0418648669157474E+03_dp,&
      1.3579924383957912E+03_dp, -1.6383680580108412E+03_dp,&
      1.5632799559570371E+03_dp, -8.2188370092199273E+02_dp,&
      1.7721434488923131E+02_dp,  2.5281668531481952E+01_dp,&
      -1.8196384478159434E+01_dp,  1.9842571350266878E+00_dp,&
      1.3328564943766472E+04_dp, -1.6084133970863943E+04_dp,&
      -2.4194016592280573E+03_dp,  1.9708051080372439E+04_dp,&
      -1.9066259070011958E+04_dp,  1.0516314706873116E+04_dp,&
      -6.9116247247954307E+03_dp,  5.9899122414862086E+03_dp,&
      -1.8790330931175195E+03_dp, -3.1628969452433375E+03_dp,&
      4.4070324751659182E+03_dp, -2.3872165372019858E+03_dp,&
      5.1970220864424766E+02_dp,  8.1916821529150999E+04_dp,&
      -1.5385697042232822E+05_dp,  6.4357021299851127E+04_dp,&
      1.8222531336464582E+05_dp, -3.9926912182453059E+05_dp,&
      3.9765324348418182E+05_dp, -1.5259165941629582E+05_dp,&
      -1.9235890817676744E+05_dp,  4.4346076434029645E+05_dp,&
      -4.8289048921236163E+05_dp,  3.3598895851814025E+05_dp,&
      -1.4269220555320667E+05_dp,  2.8226493599413534E+04_dp,&
      3.0164685170479322E-01_dp, -5.5172203451047762E-10_dp,&
      -1.8506044942376530E-02_dp,  2.0535362130493114E-02_dp,&
      -2.1955161428588353E-02_dp,  7.9539755938173202E-03_dp,&
      3.5963658879027555E-02_dp, -1.1551812301088075E-01_dp,&
      2.0972249709379689E-01_dp, -2.6358369943822746E-01_dp,&
      2.2691979390776368E-01_dp, -1.2076707889951548E-01_dp,&
      2.9914909381961639E-02_dp,  1.0337362727860777E+00_dp,&
      -2.4278618133657428E-09_dp, -8.0987705054095693E-02_dp,&
      8.9868498016994175E-02_dp, -9.4244322343980405E-02_dp,&
      3.0706481031845208E-02_dp,  1.6421381213801789E-01_dp,&
      -5.1354447393972757E-01_dp,  9.2295913308558519E-01_dp,&
      -1.1521564902907171E+00_dp,  9.8650626274080910E-01_dp,&
      -5.2258730390222685E-01_dp,  1.2892814675256531E-01_dp,&
      2.2336006209022599E+00_dp, -7.5196557381381133E-09_dp,&
      -2.4886328089901064E-01_dp,  2.7615231697783232E-01_dp,&
      -2.8157691375482552E-01_dp,  7.6447427732108442E-02_dp,&
      5.3416311823157037E-01_dp, -1.6122104338954852E+00_dp,&
      2.8570238241898216E+00_dp, -3.5321996045285347E+00_dp,&
      3.0005760718496060E+00_dp, -1.5787021349008945E+00_dp,&
      3.8715834373013674E-01_dp,  4.5408396268854370E+00_dp,&
      -2.1963117911792124E-08_dp, -7.2047148070424816E-01_dp,&
      7.9947350516824989E-01_dp, -7.8952927462140421E-01_dp,&
      1.6405035377058730E-01_dp,  1.6397303078861616E+00_dp,&
      -4.7727711329957341E+00_dp,  8.3301112377347231E+00_dp,&
      -1.0188327765377178E+01_dp,  8.5774320423583141E+00_dp,&
      -4.4773330972176026E+00_dp,  1.0903113937324100E+00_dp,&
      9.3585396225448729E+00_dp, -6.3091718142079549E-08_dp,&
      -2.0488465862004168E+00_dp,  2.2735056908505160E+00_dp,&
      -2.1632353036088312E+00_dp,  2.8342599157808113E-01_dp,&
      4.9563849142841772E+00_dp, -1.3893621567301645E+01_dp,&
      2.3845002853830717E+01_dp, -2.8807589805822740E+01_dp,&
      2.3998952097393417E+01_dp, -1.2409549730494412E+01_dp,&
      2.9962231581101673E+00_dp,  2.0035661685047266E+01_dp,&
      -1.8318861950190785E-07_dp, -5.8768822305456201E+00_dp,&
      6.5212788393059284E+00_dp, -5.9260748466884259E+00_dp,&
      1.9007154556845168E-01_dp,  1.5195230767450031E+01_dp,&
      -4.0880688284867844E+01_dp,  6.8798962920532261E+01_dp,&
      -8.1885938580434171E+01_dp,  6.7326941283367191E+01_dp,&
      -3.4396112195934521E+01_dp,  8.2124819537730627E+00_dp,&
      4.5186837524564041E+01_dp, -5.5189048315790405E-07_dp,&
      -1.7435293518428235E+01_dp,  1.9347016240234929E+01_dp,&
      -1.6555028673455720E+01_dp, -1.7282546301330646E+00_dp,&
      4.8592375625498889E+01_dp, -1.2478022163971495E+02_dp,&
      2.0501118398605493E+02_dp, -2.3937993744058983E+02_dp,&
      1.9340954070002567E+02_dp, -9.7185875629100849E+01_dp,&
      2.2841035703260431E+01_dp,  1.0895632742952016E+02_dp,&
      -1.7753617695147041E-06_dp, -5.4961807655449583E+01_dp,&
      6.0987992289219527E+01_dp, -4.7993145360250452E+01_dp,&
      -1.4816878684426364E+01_dp,  1.6706696991861321E+02_dp,&
      -4.0611982341150724E+02_dp,  6.4689668450834165E+02_dp,&
      -7.3581605952141376E+02_dp,  5.7978919474213126E+02_dp,&
      -2.8419268389089797E+02_dp,  6.5166070829987390E+01_dp,&
      2.8689243638403678E+02_dp, -6.3312119940337709E-06_dp,&
      -1.9053537366222281E+02_dp,  2.1142529518359913E+02_dp,&
      -1.4641605175592150E+02_dp, -9.5966800148703925E+01_dp,&
      6.4223751431253402E+02_dp, -1.4586588921357418E+03_dp,&
      2.2256725177045328E+03_dp, -2.4338616595046051E+03_dp,&
      1.8418263857593236E+03_dp, -8.6495958991931241E+02_dp,&
      1.8950502445083771E+02_dp,  8.5444004181695163E+02_dp,&
      -2.6571362038851006E-05_dp, -7.6613502361500741E+02_dp,&
      8.5012645497370215E+02_dp, -4.6842775108287498E+02_dp,&
      -6.5473848351985168E+02_dp,  2.9351311783685087E+03_dp,&
      -6.0796556523797653E+03_dp,  8.6674356970216486E+03_dp,&
      -8.8364985676925335E+03_dp,  6.1665429803353672E+03_dp,&
      -2.6257276231829951E+03_dp,  5.0946079410937239E+02_dp,&
      3.0824642958474647E+03_dp, -1.4788465867979594E-04_dp,&
      -3.9609773254235906E+03_dp,  4.3951626916942769E+03_dp,&
      -1.3291224125020340E+03_dp, -5.8270602373543688E+03_dp,&
      1.7945918393741063E+04_dp, -3.1907297760403359E+04_dp,&
      3.9407886847883412E+04_dp, -3.3300333492099067E+04_dp,&
      1.7255106888719325E+04_dp, -3.9825476124982529E+03_dp,&
      -1.2999954606432709E+02_dp,  1.6128597938285850E+04_dp,&
      -1.5030638475001011E-03_dp, -3.4863202377279144E+04_dp,&
      3.8683719996946420E+04_dp,  1.3482962106710645E+04_dp,&
      -1.0745578386256201E+05_dp,  1.9505941245695992E+05_dp,&
      -1.9666018180780555E+05_dp,  4.4741584065591967E+04_dp,&
      2.0843454148318552E+05_dp, -3.5985775021980889E+05_dp,&
      2.7195812467571569E+05_dp, -8.4386235731227309E+04_dp,&
      2.4940301661524327E+00_dp, -4.8761432479024613E-09_dp,&
      -1.6331906973896218E-01_dp,  1.8122810750947294E-01_dp,&
      -1.9277728419454013E-01_dp,  6.8005697039744004E-02_dp,&
      3.2103938239821989E-01_dp, -1.0237726178221629E+00_dp,&
      1.8536557765917583E+00_dp, -2.3255419407411013E+00_dp,&
      1.9991982697808994E+00_dp, -1.0626828301829472E+00_dp,&
      2.6295769459071416E-01_dp,  3.5755973818818960E+00_dp,&
      -1.1603952754322735E-08_dp, -3.8475394639504729E-01_dp,&
      4.2694415860455948E-01_dp, -4.3819791361753802E-01_dp,&
      1.2459309034144038E-01_dp,  8.1545646945528116E-01_dp,&
      -2.4809209517018500E+00_dp,  4.4108062606870044E+00_dp,&
      -5.4655694478252199E+00_dp,  4.6516682189795713E+00_dp,&
      -2.4513933721396328E+00_dp,  6.0204233745356073E-01_dp,&
      6.4226158559830946E+00_dp, -3.2805352108466541E-08_dp,&
      -1.0773255604203609E+00_dp,  1.1954579885591146E+00_dp,&
      -1.1849990846141789E+00_dp,  2.5515816259290564E-01_dp,&
      2.4367480635924981E+00_dp, -7.1215213584239168E+00_dp,&
      1.2451828043400571E+01_dp, -1.5249497001548516E+01_dp,&
      1.2852721953337637E+01_dp, -6.7156694314339243E+00_dp,&
      1.6368497429599620E+00_dp,  1.2902584607163559E+01_dp,&
      -9.3010917588220549E-08_dp, -3.0241468331042811E+00_dp,&
      3.3557497057646462E+00_dp, -3.2062969899464329E+00_dp,&
      4.4807915038362744E-01_dp,  7.2715631913010199E+00_dp,&
      -2.0466348197975517E+01_dp,  3.5193113499959807E+01_dp,&
      -4.2579681063128852E+01_dp,  3.5517565733358850E+01_dp,&
      -1.8387068020037940E+01_dp,  4.4442003578290663E+00_dp,&
      2.7685125754473791E+01_dp, -2.6655333069596634E-07_dp,&
      -8.5660748337068267E+00_dp,  9.5053423418833010E+00_dp,&
      -8.6904919031127577E+00_dp,  3.9482814908508695E-01_dp,&
      2.1976974588949293E+01_dp, -5.9436677744795340E+01_dp,&
      1.0029335513740615E+02_dp, -1.1962215910808953E+02_dp,&
      9.8539996723451168E+01_dp, -5.0431243284742301E+01_dp,&
      1.2060994098735554E+01_dp,  6.2917458626872921E+01_dp,&
      -7.9134239123148668E-07_dp, -2.5068529212304448E+01_dp,&
      2.7817223140209801E+01_dp, -2.4046915225943547E+01_dp,&
      -1.9395939337241239E+00_dp,  6.9089786072269447E+01_dp,&
      -1.7876906933349807E+02_dp,  2.9493693173592038E+02_dp,&
      -3.4556771722578929E+02_dp,  2.8010388997491992E+02_dp,&
      -1.4118580950500959E+02_dp,  3.3281675996370566E+01_dp,&
      1.5253253791878910E+02_dp, -2.4954055601737901E-06_dp,&
      -7.7617749203342299E+01_dp,  8.6128067009403992E+01_dp,&
      -6.9075429768151551E+01_dp, -1.8021885618043012E+01_dp,&
      2.3192299426396963E+02_dp, -5.7052514794751085E+02_dp,&
      9.1526774152540247E+02_dp, -1.0476060088374984E+03_dp,&
      8.3055190254818183E+02_dp, -4.0964513685395752E+02_dp,&
      9.4525150515697661E+01_dp,  4.0087287412388724E+02_dp,&
      -8.6103900253895508E-06_dp, -2.6137783643603069E+02_dp,&
      2.9003519532875379E+02_dp, -2.0888205957244327E+02_dp,&
      -1.1370705124392683E+02_dp,  8.5737866354046969E+02_dp,&
      -1.9862642690863304E+03_dp,  3.0712706338441053E+03_dp,&
      -3.4013694733239226E+03_dp,  2.6087459336779320E+03_dp,&
      -1.2431758759458942E+03_dp,  2.7676845196812326E+02_dp,&
      1.1729505165460264E+03_dp, -3.3934203984688906E-05_dp,&
      -9.9532307429726222E+02_dp,  1.1044435943440794E+03_dp,&
      -6.6950542960751693E+02_dp, -7.1437477357816522E+02_dp,&
      3.6476398644962173E+03_dp, -7.8338686873921197E+03_dp,&
      1.1489560023181602E+04_dp, -1.2078261929568855E+04_dp,&
      8.7467165628783914E+03_dp, -3.9033893165920181E+03_dp,&
      8.0543786542223984E+02_dp,  4.0025763692524602E+03_dp,&
      -1.6441200621815803E-04_dp, -4.5748681371839593E+03_dp,&
      5.0763799034785297E+03_dp, -2.1894243070054299E+03_dp,&
      -5.2679217031519338E+03_dp,  1.9223347825793080E+04_dp,&
      -3.7099987002346737E+04_dp,  4.9772930678433557E+04_dp,&
      -4.7215872180717706E+04_dp,  2.9876971372619882E+04_dp,&
      -1.0991976018488713E+04_dp,  1.6682216582907936E+03_dp,&
      1.7509520393482337E+04_dp, -1.1430899598941567E-03_dp,&
      -2.9021900105895245E+04_dp,  3.2202837223613624E+04_dp,&
      -3.5725050225571181E+03_dp, -5.6472687752002603E+04_dp,&
      1.4551089741279720E+05_dp, -2.3088420890704289E+05_dp,&
      2.4762084854022125E+05_dp, -1.6041754781689952E+05_dp,&
      3.1958918982707906E+04_dp,  3.1336956460431076E+04_dp,&
      -1.8132152598903405E+04_dp,  1.2844704220123403E+05_dp,&
      -1.6871224076075653E-02_dp, -3.7306629896332178E+05_dp,&
      4.1394566251683736E+05_dp,  3.3744969046701316E+05_dp,&
      -1.5791948214027057E+06_dp,  2.1917044569001873E+06_dp,&
      -7.9115467427151953E+05_dp, -3.0601023356753197E+06_dp,&
      7.4427363873937679E+06_dp, -8.7248223821029570E+06_dp,&
      5.6367831322922539E+06_dp, -1.6014423862358064E+06_dp /)
    REAL(KIND=dp), DIMENSION(400), PARAMETER :: c01 = (/ &
      1.8031277973538481E-01_dp, -6.7333985536224536E-02_dp,&
      -8.9222590966419636E-03_dp,  4.3313443199331000E-03_dp,&
      1.3612994231258369E-03_dp, -4.2134528860866431E-04_dp,&
      -1.3958926498276367E-04_dp,  1.0459985085575114E-04_dp,&
      2.4321788489287324E-05_dp, -2.5628407616984446E-05_dp,&
      -1.6458382252127667E-05_dp,  1.3859125845187524E-05_dp,&
      -2.2063994735235424E-06_dp,  5.6642829897676783E-01_dp,&
      -2.3140933582208004E-01_dp, -1.8778663252719714E-02_dp,&
      1.6290358696371920E-02_dp,  2.8000138622044312E-03_dp,&
      -1.8197685206671524E-03_dp, -1.8239631460980203E-04_dp,&
      4.0942989349671970E-04_dp,  1.5801658807221979E-05_dp,&
      -1.0194929622221692E-04_dp, -2.8507608430196145E-05_dp,&
      4.3492764958275464E-05_dp, -9.6234726899503233E-06_dp,&
      1.0336794690647568E+00_dp, -4.9295449395081187E-01_dp,&
      2.0533773060304626E-03_dp,  3.7338083754647818E-02_dp,&
      -7.9252017933650757E-04_dp, -4.6073387291294529E-03_dp,&
      6.9737642509595181E-04_dp,  9.2284371103869235E-04_dp,&
      -2.0181932998978786E-04_dp, -2.3287683121935485E-04_dp,&
      3.9390431126406291E-05_dp,  6.1279897499314255E-05_dp,&
      -2.1659512655332614E-05_dp,  1.6524775107115341E+00_dp,&
      -9.4977382091404372E-01_dp,  9.9098600636927089E-02_dp,&
      6.9382247845005254E-02_dp, -1.6738833641680006E-02_dp,&
      -8.4516567685186013E-03_dp,  3.7770914842681614E-03_dp,&
      1.4714946234771721E-03_dp, -8.7049922044455698E-04_dp,&
      -3.7422653125675199E-04_dp,  2.7591748697712180E-04_dp,&
      1.0743072823795072E-05_dp, -2.7724542973182781E-05_dp,&
      2.5209403407090294E+00_dp, -1.7572535732534307E+00_dp,&
      3.6675784179170207E-01_dp,  1.0132030522062821E-01_dp,&
      -5.8095720513577059E-02_dp, -9.6754073273882992E-03_dp,&
      1.1051073079906482E-02_dp,  1.1782991243745834E-03_dp,&
      -2.2976290173779054E-03_dp, -3.1775371778666181E-04_dp,&
      7.5498578709957390E-04_dp, -2.0358509226504188E-04_dp,&
      -2.1487703220423853E-06_dp,  3.7842663535150787E+00_dp,&
      -3.1719815711214094E+00_dp,  9.9628607256145418E-01_dp,&
      8.4642747014531908E-02_dp, -1.4338485390045752E-01_dp,&
      3.6775926065065463E-03_dp,  2.4387343709671966E-02_dp,&
      -2.3922521075792551E-03_dp, -4.6280864130883170E-03_dp,&
      4.5453420746634649E-04_dp,  1.4427825127764225E-03_dp,&
      -6.9362473654104538E-04_dp,  9.3510079815132331E-05_dp,&
      5.6675748977110523E+00_dp, -5.6248348603552669E+00_dp,&
      2.3650242434167574E+00_dp, -1.2518277136594844E-01_dp,&
      -2.8499324383843427E-01_dp,  6.0582406722422502E-02_dp,&
      4.1263478851215951E-02_dp, -1.4469635429703774E-02_dp,&
      -6.9085971227368830E-03_dp,  2.9757772888861674E-03_dp,&
      1.9815826737071439E-03_dp, -1.5730264350839068E-03_dp,&
      3.2218540254269558E-04_dp,  8.5395933853630233E+00_dp,&
      -9.8513995851521390E+00_dp,  5.1996720804059038E+00_dp,&
      -8.9780495348559786E-01_dp, -4.3997137995311281E-01_dp,&
      2.1423762405988919E-01_dp,  4.1578026070946399E-02_dp,&
      -4.2129670371580948E-02_dp, -4.3453273897259689E-03_dp,&
      8.5341896614408736E-03_dp,  8.3854210658043302E-04_dp,&
      -2.5753940654037861E-03_dp,  7.3005171953852588E-04_dp,&
      1.3061522995672423E+01_dp, -1.7160442399057352E+01_dp,&
      1.0881660469438872E+01_dp, -3.0868825863712459E+00_dp,&
      -3.6654990016924677E-01_dp,  5.1609758519564186E-01_dp,&
      -3.5298257683150376E-02_dp, -8.1154201018169170E-02_dp,&
      1.4393228245430536E-02_dp,  1.5144340450492988E-02_dp,&
      -5.1252673274943747E-03_dp, -1.8479081064741922E-03_dp,&
      9.9116641653779008E-04_dp,  2.0632157021397099E+01_dp,&
      -3.0169058816343398E+01_dp,  2.2140092264250899E+01_dp,&
      -8.5303719478715294E+00_dp,  6.8478977718455336E-01_dp,&
      8.8233432059397909E-01_dp, -2.8211262686916566E-01_dp,&
      -8.6566980543862401E-02_dp,  5.5120484707143334E-02_dp,&
      1.2338345616801711E-02_dp, -1.4829218126644110E-02_dp,&
      2.4716302212440498E-03_dp,  4.4101212104042314E-04_dp,&
      3.5217338955590499E+01_dp, -5.5810497183858885E+01_dp,&
      4.5632291320845674E+01_dp, -2.1446416086860207E+01_dp,&
      4.5351223518918955E+00_dp,  8.5236412809032913E-01_dp,&
      -7.1131676788516363E-01_dp,  7.8756575443881364E-03_dp,&
      9.5952279114652053E-02_dp, -4.2251471621967060E-03_dp,&
      -2.2019449280769526E-02_dp,  8.2657727156656199E-03_dp,&
      -6.8583155310803173E-04_dp,  7.6395962185656373E+01_dp,&
      -1.2728686310186325E+02_dp,  1.1124646527486497E+02_dp,&
      -5.8598736421586601E+01_dp,  1.7014576293478193E+01_dp,&
      -5.7631736469304418E-01_dp, -1.3501560044152556E+00_dp,&
      2.1084835464631169E-01_dp,  1.5298196818925494E-01_dp,&
      -2.8319608074091947E-02_dp, -3.6419431099310340E-02_dp,&
      1.8351239837091964E-02_dp, -2.4603666608287310E-03_dp,&
      1.4591908177422581E+00_dp, -5.5767651166491705E-01_dp,&
      -6.6479842568471656E-02_dp,  3.6890552285677343E-02_dp,&
      1.0112353440941297E-02_dp, -3.7573056532886231E-03_dp,&
      -9.7365738212055173E-04_dp,  9.0541872635710951E-04_dp,&
      1.5916301956753239E-04_dp, -2.2301214373999525E-04_dp,&
      -1.1911948208237914E-04_dp,  1.1340817590337231E-04_dp,&
      -1.9950792471753723E-05_dp,  1.6661756196836854E+00_dp,&
      -7.9898374218448487E-01_dp,  1.7550890038460737E-03_dp,&
      6.2809001477724419E-02_dp, -9.4026377706547348E-04_dp,&
      -8.0360148249423204E-03_dp,  1.0619161940127371E-03_dp,&
      1.6209032795216753E-03_dp, -3.2460691552359526E-04_dp,&
      -4.1050813878500998E-04_dp,  6.0867379772286893E-05_dp,&
      1.1638427836478115E-04_dp, -4.1022245181328590E-05_dp,&
      2.1179439068493049E+00_dp, -1.3568029141401121E+00_dp,&
      1.9362507650379276E-01_dp,  1.0737274042258362E-01_dp,&
      -3.2523712445134516E-02_dp, -1.4173642482044387E-02_dp,&
      6.8145250252902613E-03_dp,  2.3962722609825390E-03_dp,&
      -1.5785664936154885E-03_dp, -6.1334891145104148E-04_dp,&
      5.4453171956963262E-04_dp, -1.1974429657690983E-05_dp,&
      -4.8455232546300578E-05_dp,  2.8966195759216218E+00_dp,&
      -2.3985545624839126E+00_dp,  6.4264735748769419E-01_dp,&
      1.4303859261575749E-01_dp, -1.0370087260206109E-01_dp,&
      -1.4442128737280619E-02_dp,  1.9189264767116777E-02_dp,&
      1.4633618315939528E-03_dp, -3.9872332174864444E-03_dp,&
      -3.8808657146881335E-04_dp,  1.4034332418797093E-03_dp,&
      -4.5827591441622000E-04_dp,  1.7452248924342918E-05_dp,&
      4.1458283694621647E+00_dp, -4.2276944439107638E+00_dp,&
      1.6204734519661337E+00_dp,  8.3699114369403166E-02_dp,&
      -2.4005358926470990E-01_dp,  1.2019379195118291E-02_dp,&
      4.0476206444562082E-02_dp, -5.3365562632643688E-03_dp,&
      -7.5759092434434594E-03_dp,  1.1703318240633225E-03_dp,&
      2.4154624739185747E-03_dp, -1.3300232599431343E-03_dp,&
      2.1206653667717895E-04_dp,  6.1070095083601350E+00_dp,&
      -7.3692662730790639E+00_dp,  3.6569615163999236E+00_dp,&
      -3.0543234096798377E-01_dp, -4.5325437989138290E-01_dp,&
      1.1340063624980586E-01_dp,  6.5345011752561535E-02_dp,&
      -2.6683850539911569E-02_dp, -1.0609290569814111E-02_dp,&
      5.6881166467098749E-03_dp,  2.8996179473254218E-03_dp,&
      -2.6780441022469402E-03_dp,  5.9375790281117737E-04_dp,&
      9.1967607356270840E+00_dp, -1.2728161122633493E+01_dp,&
      7.7585514743513535E+00_dp, -1.6031523579554121E+00_dp,&
      -6.5971663204666997E-01_dp,  3.7822167096008902E-01_dp,&
      5.8075256175536355E-02_dp, -7.5356462157257068E-02_dp,&
      -4.7610280346296668E-03_dp,  1.5528173541741448E-02_dp,&
      5.0291646970573997E-04_dp, -4.2910396246577829E-03_dp,&
      1.2926070332399792E-03_dp,  1.4196696181971468E+01_dp,&
      -2.1917883458855773E+01_dp,  1.5801136777519396E+01_dp,&
      -5.1163101205901276E+00_dp, -4.3929530502807101E-01_dp,&
      8.9353294761283697E-01_dp, -9.5307670162679783E-02_dp,&
      -1.4563133541061388E-01_dp,  3.3385614060572416E-02_dp,&
      2.8263201419445248E-02_dp, -1.2151984828830927E-02_dp,&
      -2.4648614361214064E-03_dp,  1.7803770851374343E-03_dp,&
      2.2826825664059452E+01_dp, -3.8162174617271774E+01_dp,&
      3.1357657732432617E+01_dp, -1.3555584080685593E+01_dp,&
      1.4927242940990013E+00_dp,  1.4849933729996307E+00_dp,&
      -5.9022811136976339E-01_dp, -1.3676085124973847E-01_dp,&
      1.2318046476580022E-01_dp,  1.6651678250228036E-02_dp,&
      -3.5419068294572351E-02_dp,  1.0103555469163566E-02_dp,&
      -2.8242465666916748E-04_dp,  3.9949146857561630E+01_dp,&
      -6.9806819497162493E+01_dp,  6.2526091490839079E+01_dp,&
      -3.2491521274710166E+01_dp,  8.0495785945780884E+00_dp,&
      1.1449352957658303E+00_dp, -1.4001449115453595E+00_dp,&
      1.4849591069311463E-01_dp,  1.8413922498441526E-01_dp,&
      -4.3848567993047871E-02_dp, -3.5796486617126269E-02_dp,&
      2.2231295011594484E-02_dp, -3.8635690919857713E-03_dp,&
      8.5422617602734633E+01_dp, -1.5007516551248531E+02_dp,&
      1.3876195898375653E+02_dp, -7.8464142326741523E+01_dp,&
      2.5573232411746961E+01_dp, -2.0124727833037532E+00_dp,&
      -1.9580157250354535E+00_dp,  5.9506449783845983E-01_dp,&
      1.4004086634287427E-01_dp, -7.8164255350495107E-02_dp,&
      -2.7687831019897382E-02_dp,  2.4943146447903092E-02_dp,&
      -4.9581832391613840E-03_dp,  3.2243620561770621E+02_dp,&
      -5.5446459400958940E+02_dp,  5.0455670468847723E+02_dp,&
      -2.8330528858502907E+02_dp,  9.4651307478009102E+01_dp,&
      -1.1293415812091316E+01_dp, -4.1837250114735332E+00_dp,&
      1.1914362296732655E+00_dp,  4.8166287147916081E-01_dp,&
      -1.4958729633043041E-01_dp, -1.2337653487937358E-01_dp,&
      7.5229238812777835E-02_dp, -1.2091269567906382E-02_dp,&
      2.4402285772918880E-01_dp, -7.3529180505381958E-02_dp,&
      -2.5723746423618070E-02_dp,  8.9632749393919135E-03_dp,&
      7.8378134914843661E-03_dp, -2.9623515386430414E-03_dp,&
      -3.3201464839523391E-03_dp,  1.0078877137589960E-03_dp,&
      2.5955945748679025E-03_dp, -1.8637178008176481E-03_dp,&
      -2.3022221614225430E-04_dp,  6.5838716850614683E-04_dp,&
      -1.9830435925879039E-04_dp,  7.9761273417323364E-01_dp,&
      -2.8289567084615880E-01_dp, -7.7983815660254863E-02_dp,&
      4.6178699662303369E-02_dp,  2.4703159881149541E-02_dp,&
      -1.7384731956342849E-02_dp, -1.0573811255673717E-02_dp,&
      6.4828830942470356E-03_dp,  1.0745058972820175E-02_dp,&
      -1.2304204780420294E-02_dp,  3.6994644715572381E-03_dp,&
      7.1771353633863510E-04_dp, -4.7696615443758658E-04_dp,&
      1.5722119399201502E+00_dp, -7.2022637684883928E-01_dp,&
      -1.1778618034469690E-01_dp,  1.5513182133862150E-01_dp,&
      3.6741006041689533E-02_dp, -6.4119193681736464E-02_dp,&
      -1.4800413983640810E-02_dp,  2.5585125964366535E-02_dp,&
      2.6791486817367951E-02_dp, -4.9141584177700877E-02_dp,&
      2.8220269783745606E-02_dp, -6.5502100323003949E-03_dp,&
      2.6599045470973711E-04_dp,  2.8051373012201362E+00_dp,&
      -1.6968205795059628E+00_dp, -6.8520667990219081E-02_dp,&
      4.3238094864773730E-01_dp,  1.3604877002541815E-03_dp,&
      -1.8671384960156767E-01_dp,  8.8883053763241949E-03_dp,&
      7.8093230403048317E-02_dp,  4.6271896457208229E-02_dp,&
      -1.4709229627835357E-01_dp,  1.1306355579348493E-01_dp,&
      -3.9830635885835595E-02_dp,  5.4972776007275979E-03_dp,&
      4.9068374696158550E+00_dp, -3.8649342494189476E+00_dp,&
      3.3961436603861528E-01_dp,  1.0516701804754258E+00_dp,&
      -2.3599107021791954E-01_dp, -4.5209301669418722E-01_dp,&
      1.4887292136073554E-01_dp,  1.9549898360847917E-01_dp,&
      1.9083808478825574E-02_dp, -3.4003659083377596E-01_dp,&
      3.3304060399936797E-01_dp, -1.4290330281954144E-01_dp,&
      2.4575713534518952E-02_dp,  8.6327262342711766E+00_dp,&
      -8.6449632701709422E+00_dp,  1.9398206133689215E+00_dp,&
      2.2633523514966289E+00_dp, -1.1404630111455665E+00_dp,&
      -8.9656935075118105E-01_dp,  6.5655059844600916E-01_dp,&
      3.9178915736248293E-01_dp, -2.5745765584870500E-01_dp,&
      -5.4365659786191212E-01_dp,  7.5377793521530634E-01_dp,&
      -3.8414425007098330E-01_dp,  7.5502828561301874E-02_dp,&
      1.5429572442140103E+01_dp, -1.9205543991334537E+01_dp,&
      7.1837956925844102E+00_dp,  4.1527008022568523E+00_dp,&
      -3.9604373888031548E+00_dp, -1.2184620720634423E+00_dp,&
      2.1268242758491254E+00_dp,  4.8977374234146870E-01_dp,&
      -1.3567457043687021E+00_dp, -1.4342868044936791E-01_dp /)
    REAL(KIND=dp), DIMENSION(400), PARAMETER :: c02 = (/ &
      1.1553671463453488E+00_dp, -7.6780781804979914E-01_dp,&
      1.7390307652239254E-01_dp,  2.8165264234356147E+01_dp,&
      -4.2781018965163859E+01_dp,  2.3179056548048951E+01_dp,&
      5.2439089413556950E+00_dp, -1.1643001503346785E+01_dp,&
      4.5770373921564939E-01_dp,  5.5851597940226441E+00_dp,&
      -5.5713676796962119E-01_dp, -4.3682925715598220E+00_dp,&
      2.9977088433118744E+00_dp,  1.1551526130399045E-01_dp,&
      -8.4503890035033269E-01_dp,  2.6553468866554275E-01_dp,&
      5.2755173634217869E+01_dp, -9.6327032783315346E+01_dp,&
      7.0566200135216633E+01_dp, -3.8183788481120411E+00_dp,&
      -2.9391413038382652E+01_dp,  1.1731796742283295E+01_dp,&
      1.0979398073072346E+01_dp, -6.6194752458984993E+00_dp,&
      -9.5048580745993743E+00_dp,  1.3808973411209031E+01_dp,&
      -7.1703659500595673E+00_dp,  1.4743357570136750E+00_dp,&
      -1.8794771258963321E-02_dp,  1.0222132432274876E+02_dp,&
      -2.2102822149401919E+02_dp,  2.0974623410290044E+02_dp,&
      -6.6317175075640407E+01_dp, -5.5810649104847528E+01_dp,&
      5.5721941193426112E+01_dp,  6.1874002936069470E+00_dp,&
      -2.6416717623490523E+01_dp, -5.5906480645235490E+00_dp,&
      3.2874933451592149E+01_dp, -2.8094658694055632E+01_dp,&
      1.1072416668088719E+01_dp, -1.7823600622119833E+00_dp,&
      2.1028466549514980E+02_dp, -5.2642609969482317E+02_dp,&
      6.2125118053900360E+02_dp, -3.5670022652999478E+02_dp,&
      -1.1979421659544816E+01_dp,  1.6071306995897604E+02_dp,&
      -6.8887503091643865E+01_dp, -4.4648249302777593E+01_dp,&
      4.3014857970719397E+01_dp,  1.6335752686095919E+01_dp,&
      -3.9503484670996066E+01_dp,  2.2138801724899430E+01_dp,&
      -4.5544474926067293E+00_dp,  5.2271868060407382E+02_dp,&
      -1.4526730604347720E+03_dp,  2.0000601499106572E+03_dp,&
      -1.5790696077296959E+03_dp,  5.6252542900631101E+02_dp,&
      1.8000459328370113E+02_dp, -2.6699351388563537E+02_dp,&
      3.0766701448493226E+01_dp,  9.2468215263256795E+01_dp,&
      -3.9890430775649456E+01_dp, -2.4228475522268266E+01_dp,&
      2.5320986601337847E+01_dp, -6.4467381198271339E+00_dp,&
      1.9943221690264703E+00_dp, -6.2720822070394933E-01_dp,&
      -2.0718877314833953E-01_dp,  8.3665992635281738E-02_dp,&
      6.3970965828668153E-02_dp, -2.8909340921094925E-02_dp,&
      -2.7244049755787644E-02_dp,  1.0154578919302151E-02_dp,&
      2.2755693177166175E-02_dp, -1.8948581830615388E-02_dp,&
      6.0818547316571962E-04_dp,  4.6474201410700100E-03_dp,&
      -1.5474535300080714E-03_dp,  2.5344309071612208E+00_dp,&
      -1.1517789229917388E+00_dp, -2.0475816431506066E-01_dp,&
      2.4704626449142972E-01_dp,  6.8510892925257152E-02_dp,&
      -1.0144220474534189E-01_dp, -2.9276180034024692E-02_dp,&
      3.9983114315973144E-02_dp,  4.6113046413331867E-02_dp,&
      -7.6975277049864677E-02_dp,  4.0552717012763861E-02_dp,&
      -7.6506455032915865E-03_dp, -1.9384086829425090E-04_dp,&
      3.8149008989961728E+00_dp, -2.5425820634363938E+00_dp,&
      -6.9974141904847256E-02_dp,  6.8655295447602072E-01_dp,&
      3.7667761955125154E-03_dp, -3.0089857114650986E-01_dp,&
      9.5441871944521653E-03_dp,  1.2570484744488805E-01_dp,&
      8.1291754180774958E-02_dp, -2.4131410671430067E-01_dp,&
      1.8097245043205007E-01_dp, -6.2022990804356130E-02_dp,&
      8.2193545283439501E-03_dp,  6.3114819275149774E+00_dp,&
      -5.6792022806645264E+00_dp,  6.2279323003148646E-01_dp,&
      1.6459331144460243E+00_dp, -3.7800717585122690E-01_dp,&
      -7.2663629659756079E-01_dp,  2.3264111653024058E-01_dp,&
      3.1646630721128349E-01_dp,  4.7657307643253675E-02_dp,&
      -5.7090704941545778E-01_dp,  5.4910822545064875E-01_dp,&
      -2.3284314940812098E-01_dp,  3.9588194044005813E-02_dp,&
      1.0984388893348632E+01_dp, -1.2553778627354628E+01_dp,&
      3.1375915699757435E+00_dp,  3.4851328166564612E+00_dp,&
      -1.8017015924479218E+00_dp, -1.4359380873215706E+00_dp,&
      1.0422397069450537E+00_dp,  6.4072298568256425E-01_dp,&
      -3.8510916515210597E-01_dp, -9.4069946857544329E-01_dp,&
      1.2691484307865526E+00_dp, -6.4142347298575852E-01_dp,&
      1.2542319982761238E-01_dp,  1.9742724664081642E+01_dp,&
      -2.7660695606300102E+01_dp,  1.1126190485661816E+01_dp,&
      6.3320842667644364E+00_dp, -6.2040316849472417E+00_dp,&
      -1.9845491148459973E+00_dp,  3.4016074403877203E+00_dp,&
      8.4883284732790321E-01_dp, -2.1910518713819105E+00_dp,&
      -3.4310026768968166E-01_dp,  2.0043227872868097E+00_dp,&
      -1.3129608772761314E+00_dp,  2.9596309142603439E-01_dp,&
      3.6446675442084000E+01_dp, -6.1312690384351988E+01_dp,&
      3.5062440706052982E+01_dp,  8.1141251417433384E+00_dp,&
      -1.8238397593931307E+01_dp,  4.2444498167686012E-01_dp,&
      9.0917913358856648E+00_dp, -6.4145530970932474E-01_dp,&
      -7.3727065993355358E+00_dp,  4.8554897125262979E+00_dp,&
      4.0288046417382256E-01_dp, -1.5272111598694897E+00_dp,&
      4.6864448241919043E-01_dp,  6.9151902111365331E+01_dp,&
      -1.3775697029732524E+02_dp,  1.0517986253443161E+02_dp,&
      -4.2845246310611023E+00_dp, -4.6935409816300506E+01_dp,&
      1.7360744079299344E+01_dp,  1.9042670125521109E+01_dp,&
      -1.0054681312184911E+01_dp, -1.7340855643857253E+01_dp,&
      2.3641147423075047E+01_dp, -1.1697618310899770E+01_dp,&
      2.1262428960669113E+00_dp,  6.0872426969359870E-02_dp,&
      1.3558768380220121E+02_dp, -3.1578118874680166E+02_dp,&
      3.0979445611347273E+02_dp, -9.1822777560522482E+01_dp,&
      -9.7277945816883886E+01_dp,  8.7519582106791688E+01_dp,&
      1.8616391208601357E+01_dp, -4.4903204542560921E+01_dp,&
      -1.8690573605516008E+01_dp,  6.6433565837523219E+01_dp,&
      -5.3403019488106459E+01_dp,  2.0231742919655424E+01_dp,&
      -3.1305074294263489E+00_dp,  2.7963732579939841E+02_dp,&
      -7.4606125632766100E+02_dp,  9.0940141987222216E+02_dp,&
      -5.0646326675575170E+02_dp, -7.8609616213890064E+01_dp,&
      2.9311367852905386E+02_dp, -9.4979204093817344E+01_dp,&
      -1.1347623458453775E+02_dp,  8.2031344824942224E+01_dp,&
      5.6241860637037483E+01_dp, -1.0224730452002980E+02_dp,&
      5.5340684759985947E+01_dp, -1.1297342601063139E+01_dp,&
      6.5120722704318109E+02_dp, -1.9033734753449851E+03_dp,&
      2.7232474210995215E+03_dp, -2.1823518641635665E+03_dp,&
      6.8086154256333919E+02_dp,  4.7729134285430774E+02_dp,&
      -5.7292260599128042E+02_dp,  6.1733473299802547E+01_dp,&
      2.6831002853289289E+02_dp, -2.0906396401698387E+02_dp,&
      4.5742535109750555E+01_dp,  1.4910878567831897E+01_dp,&
      -7.1528632268068026E+00_dp,  2.4115194635293756E+03_dp,&
      -7.1360888524468037E+03_dp,  1.0703481900930154E+04_dp,&
      -9.7696053852248097E+03_dp,  5.1324020712244110E+03_dp,&
      -6.9246367124994219E+02_dp, -9.3295400850484577E+02_dp,&
      4.2973937177886751E+02_dp,  2.2651805907394606E+02_dp,&
      -2.3513604873721363E+02_dp,  1.2713777537393858E+01_dp,&
      5.2470351852753829E+01_dp, -1.7444400068472273E+01_dp,&
      2.8440161633359956E-01_dp, -3.9823006517888213E-02_dp,&
      -2.3076160901873654E-02_dp,  2.8993315015824581E-03_dp,&
      6.3510245125292433E-03_dp, -1.8708028901690619E-05_dp,&
      -3.7414722349339334E-03_dp,  5.3858115549491718E-04_dp,&
      2.0941650894001022E-03_dp, -9.3028620337434742E-04_dp,&
      -7.2074637484255898E-04_dp,  7.4135593147601206E-04_dp,&
      -1.9162678311097998E-04_dp,  9.5981147138166634E-01_dp,&
      -1.6725087562945692E-01_dp, -8.9182688444622890E-02_dp,&
      2.0572731901361569E-02_dp,  2.7187174656436549E-02_dp,&
      -3.6082603824213978E-03_dp, -1.6627641029942530E-02_dp,&
      3.7750956656711155E-03_dp,  1.0628480501116633E-02_dp,&
      -7.1153643941278205E-03_dp, -1.2143377855200034E-03_dp,&
      2.7012120466684050E-03_dp, -7.8946988266133150E-04_dp,&
      2.0129964635290527E+00_dp, -4.8454573123042377E-01_dp,&
      -2.2604994573980364E-01_dp,  9.2677928858034012E-02_dp,&
      7.7785406032969739E-02_dp, -2.5979505885131083E-02_dp,&
      -5.0328987859669862E-02_dp,  1.7997890535406402E-02_dp,&
      3.8319048248298730E-02_dp, -3.6325104322509204E-02_dp,&
      6.7213705329095992E-03_dp,  4.8397181344369666E-03_dp,&
      -1.9887682453394975E-03_dp,  3.9222318106291012E+00_dp,&
      -1.3148639911853994E+00_dp, -5.1737762994905812E-01_dp,&
      3.4039821500135192E-01_dp,  1.9442866605932332E-01_dp,&
      -1.1862440428797107E-01_dp, -1.3471081518767758E-01_dp,&
      7.2722253898947764E-02_dp,  1.2394645141103253E-01_dp,&
      -1.5424078283263418E-01_dp,  6.0957620821992728E-02_dp,&
      -2.0727719708758920E-03_dp, -3.2589949122940306E-03_dp,&
      7.6600225072517931E+00_dp, -3.4806043928458892E+00_dp,&
      -1.0922250581494755E+00_dp,  1.1238446877190527E+00_dp,&
      4.2100547038706426E-01_dp, -4.5205119802000077E-01_dp,&
      -3.1961008012678949E-01_dp,  2.6922377501542977E-01_dp,&
      3.6631461573210744E-01_dp, -5.8790288209541941E-01_dp,&
      3.2417909023377850E-01_dp, -7.2737310780591674E-02_dp,&
      2.5717607245155992E-03_dp,  1.5357195767319828E+01_dp,&
      -9.1855216845760417E+00_dp, -2.0474502120861211E+00_dp,&
      3.5186908819973559E+00_dp,  6.9067364763413841E-01_dp,&
      -1.5707666205541846E+00_dp, -6.2320076264913815E-01_dp,&
      9.5548605981562995E-01_dp,  9.6164257386927365E-01_dp,&
      -2.0694392628725198E+00_dp,  1.4288052626518093E+00_dp,&
      -4.6175690844427097E-01_dp,  5.7161598844966313E-02_dp,&
      3.1967087548643718E+01_dp, -2.4634023685889019E+01_dp,&
      -2.8379221339115919E+00_dp,  1.0758070470373076E+01_dp,&
      9.5880676096911616E-02_dp, -5.1371158552123646E+00_dp,&
      -6.1308866074368273E-01_dp,  3.3299562667716351E+00_dp,&
      1.9037543102815111E+00_dp, -6.6312995485896993E+00_dp,&
      5.5769281199888221E+00_dp, -2.1992036324059452E+00_dp,&
      3.5304429234184187E-01_dp,  6.9739204456194557E+01_dp,&
      -6.8387481704787120E+01_dp,  1.1321639890564044E+00_dp,&
      3.2573484471282640E+01_dp, -6.9804580122418587E+00_dp,&
      -1.5736074279056968E+01_dp,  2.9654243601725896E+00_dp,&
      1.1422619493838827E+01_dp, -2.6727212260300343E-01_dp,&
      -1.7588114164311783E+01_dp,  1.8884242504372157E+01_dp,&
      -8.7648445734311817E+00_dp,  1.6290708530849365E+00_dp,&
      1.6128088895603187E+02_dp, -2.0051884945874767E+02_dp,&
      3.6821045909182011E+01_dp,  9.7251659706901165E+01_dp,&
      -5.0845173213144840E+01_dp, -4.1755777685783144E+01_dp,&
      2.8048895067605439E+01_dp,  3.6657246361996478E+01_dp,&
      -3.4007473970245094E+01_dp, -2.2004679796240055E+01_dp,&
      4.6900402309890616E+01_dp, -2.7316097181854502E+01_dp,&
      5.8577092013741536E+00_dp,  4.0194017966122044E+02_dp,&
      -6.3674119443127381E+02_dp,  2.5850957390809663E+02_dp,&
      2.6952251874150846E+02_dp, -2.9073790305138749E+02_dp,&
      -5.2677944928443630E+01_dp,  1.5741217884382442E+02_dp,&
      8.1820204921580611E+01_dp, -2.5405433230660509E+02_dp,&
      1.5943577712260654E+02_dp, -4.6996586786307697E+00_dp,&
      -3.5301765944409738E+01_dp,  1.1655632510420780E+01_dp,&
      1.1102009036442944E+03_dp, -2.2680830885994883E+03_dp,&
      1.6376815639623894E+03_dp,  4.1340967237756740E+02_dp,&
      -1.5047177979548774E+03_dp,  6.2874431977984602E+02_dp,&
      5.3667599466253898E+02_dp, -2.6715850493363729E+02_dp,&
      -9.0644414621246210E+02_dp,  1.4354167004689343E+03_dp,&
      -9.8255347076517205E+02_dp,  3.4531439529808881E+02_dp,&
      -5.0862246712184223E+01_dp,  3.7086333759961367E+03_dp,&
      -9.7206127257815970E+03_dp,  1.1363579106801713E+04_dp,&
      -5.0930367254613293E+03_dp, -3.8587254648260623E+03_dp,&
      7.5131622983303614E+03_dp, -4.5750330568023464E+03_dp,&
      4.8521172705364671E+02_dp,  4.7555952848897863E+02_dp,&
      6.7920802819038545E+02_dp, -1.2191938699136147E+03_dp,&
      6.8845273583955679E+02_dp, -1.4522061491176112E+02_dp,&
      2.3426708866702475E+00_dp, -3.4763950540505012E-01_dp,&
      -1.9719523876834619E-01_dp,  3.0016600517614064E-02_dp,&
      5.5870741584836549E-02_dp, -2.0603641124710137E-03_dp,&
      -3.3236739085782500E-02_dp,  5.4972021666434001E-03_dp,&
      1.9287416985295423E-02_dp, -9.7735471490287135E-03_dp,&
      -5.4153420961059818E-03_dp,  6.2996178645704911E-03_dp,&
      -1.6741808988248179E-03_dp,  3.2323307090443092E+00_dp,&
      -7.5868899537917556E-01_dp, -3.6403112314198777E-01_dp,&
      1.3638032105863709E-01_dp,  1.2454451915538212E-01_dp,&
      -3.5272622120219609E-02_dp, -7.9465036314149431E-02_dp /)
    REAL(KIND=dp), DIMENSION(400), PARAMETER :: c03 = (/ &
      2.5411208403973456E-02_dp,  5.7808629859203861E-02_dp,&
      -5.0196351778645114E-02_dp,  5.2149034144752344E-03_dp,&
      9.5271530622531974E-03_dp, -3.3982153227889275E-03_dp,&
      5.4948218832066447E+00_dp, -1.9768544283573861E+00_dp,&
      -7.8540931963005123E-01_dp,  5.1224166160961682E-01_dp,&
      3.0432975281369379E-01_dp, -1.7343930124124304E-01_dp,&
      -2.0939874165255690E-01_dp,  1.0529951288679236E-01_dp,&
      1.8723710201929586E-01_dp, -2.2094924684309869E-01_dp,&
      7.8605553221171381E-02_dp,  3.2239680553835400E-03_dp,&
      -6.1056020491761899E-03_dp,  1.0388070420621753E+01_dp,&
      -5.1639778369451559E+00_dp, -1.6336599184903002E+00_dp,&
      1.6828507996508695E+00_dp,  6.6282593487173314E-01_dp,&
      -6.6705890421444214E-01_dp, -5.0042252719289448E-01_dp,&
      3.9062209984428897E-01_dp,  5.5969261337300114E-01_dp,&
      -8.5374958521504130E-01_dp,  4.4514864411394023E-01_dp,&
      -8.7078324077810784E-02_dp, -6.3901198934322614E-04_dp,&
      2.0837835829959378E+01_dp, -1.3485610187672910E+01_dp,&
      -3.0562817152450337E+00_dp,  5.2210972603116508E+00_dp,&
      1.1380806549217413E+00_dp, -2.3202212441266843E+00_dp,&
      -1.0087684674158810E+00_dp,  1.3808656481997015E+00_dp,&
      1.5160097727120327E+00_dp, -3.0497799109640908E+00_dp,&
      2.0185152497063346E+00_dp, -6.1672935366387194E-01_dp,&
      6.9010721045408896E-02_dp,  4.3786529696255222E+01_dp,&
      -3.5837705535464202E+01_dp, -4.3770867029967091E+00_dp,&
      1.5847699276648193E+01_dp,  5.9230278789709845E-01_dp,&
      -7.6207423069692428E+00_dp, -1.2403077215715186E+00_dp,&
      4.7968232553191292E+00_dp,  3.3649722547708287E+00_dp,&
      -1.0082904709615054E+01_dp,  8.1126465226757194E+00_dp,&
      -3.0773616343906505E+00_dp,  4.7309562815703760E-01_dp,&
      9.6534365122504383E+01_dp, -9.8600884120947967E+01_dp,&
      2.9621706296987543E-01_dp,  4.7831849282858727E+01_dp,&
      -8.1795187324838583E+00_dp, -2.3764904478653722E+01_dp,&
      2.9262740978603858E+00_dp,  1.6552408974929406E+01_dp,&
      2.8091737862874124E+00_dp, -2.9119085060489422E+01_dp,&
      2.9028575924228921E+01_dp, -1.2928669504768751E+01_dp,&
      2.3245151888545674E+00_dp,  2.2512257878709937E+02_dp,&
      -2.8579384581407447E+02_dp,  4.4953255218681591E+01_dp,&
      1.4382256723080877E+02_dp, -6.3886295794702583E+01_dp,&
      -6.7487344867895843E+01_dp,  3.4987698625952298E+01_dp,&
      5.5556684237578004E+01_dp, -3.3788647650616660E+01_dp,&
      -5.6043553929673735E+01_dp,  8.4310762498784698E+01_dp,&
      -4.5100448710609861E+01_dp,  9.2349413542319656E+00_dp,&
      5.6281441264089858E+02_dp, -8.9092661108477409E+02_dp,&
      3.1621440639103065E+02_dp,  4.1867392990338300E+02_dp,&
      -3.6830256822695566E+02_dp, -1.3786095760420562E+02_dp,&
      2.1164276757772936E+02_dp,  1.5857658366809042E+02_dp,&
      -3.2038368617142976E+02_dp,  1.0004479022816162E+02_dp,&
      1.0938350249740985E+02_dp, -1.0143008716163932E+02_dp,&
      2.5908560637399010E+01_dp,  1.5406066303216392E+03_dp,&
      -3.0712567653270539E+03_dp,  1.9107443589577488E+03_dp,&
      9.8795055555435533E+02_dp, -1.9737417466756906E+03_dp,&
      2.8775990611293059E+02_dp,  9.9019328005226532E+02_dp,&
      7.8824363513483021E+01_dp, -1.7670587145406548E+03_dp,&
      1.9470064745111699E+03_dp, -9.6841150263156567E+02_dp,&
      2.0775921744212744E+02_dp, -6.8055324053062165E+00_dp,&
      4.8137396326482822E+03_dp, -1.2191613808338789E+04_dp,&
      1.2266285257455142E+04_dp, -1.4891625095205279E+03_dp,&
      -9.2959576828369536E+03_dp,  8.2521109655672644E+03_dp,&
      9.7025008633460880E+02_dp, -4.8188657990924448E+03_dp,&
      -1.0477423269137444E+03_dp,  7.7758077154163093E+03_dp,&
      -7.7950611646956877E+03_dp,  3.6117091083907817E+03_dp,&
      -6.8647813181077140E+02_dp,  2.0868376196912817E+04_dp,&
      -6.3171096044027225E+04_dp,  9.2357701118304758E+04_dp,&
      -7.2478877239780646E+04_dp,  1.0873910093902665E+04_dp,&
      4.7193165269176498E+04_dp, -6.6614746747051075E+04_dp,&
      5.1476389226075742E+04_dp, -2.7387825042622444E+04_dp,&
      1.1182959389903041E+04_dp, -3.8548989260871235E+03_dp,&
      1.0658643031361141E+03_dp, -1.6150497978759398E+02_dp,&
      2.9848190654960571E-01_dp, -1.4531646368209182E-02_dp,&
      -1.3349182894683412E-02_dp,  4.3341324342453481E-03_dp,&
      -4.5914461419488779E-03_dp,  7.5114003286678209E-03_dp,&
      -8.1777684041123705E-03_dp,  7.6589093791208356E-03_dp,&
      -6.9685324431716066E-03_dp,  5.4975257982100012E-03_dp,&
      -3.1992401573521976E-03_dp,  1.1765944960906449E-03_dp,&
      -2.0399784249967458E-04_dp,  1.0199389612090075E+00_dp,&
      -6.3105771054488224E-02_dp, -5.6858180138521432E-02_dp,&
      2.0836254442772213E-02_dp, -1.9744550724843178E-02_dp,&
      3.2668133118974374E-02_dp, -3.5931234657235918E-02_dp,&
      3.3317168158417933E-02_dp, -3.0032438804812003E-02_dp,&
      2.3676679695673334E-02_dp, -1.3795044313291295E-02_dp,&
      5.0675158738836823E-03_dp, -8.7502220183366809E-04_dp,&
      2.1914353100451938E+00_dp, -1.9179798443558940E-01_dp,&
      -1.6799355534241783E-01_dp,  7.1931114531331136E-02_dp,&
      -5.9484431545979891E-02_dp,  9.9336123697354881E-02_dp,&
      -1.1099688936746152E-01_dp,  1.0155478022832697E-01_dp,&
      -9.0225752064209283E-02_dp,  7.1039085577344563E-02_dp,&
      -4.1492188330594100E-02_dp,  1.5231043613680471E-02_dp,&
      -2.6165816928386950E-03_dp,  4.4195039100565916E+00_dp,&
      -5.4858126359256787E-01_dp, -4.6531366469875657E-01_dp,&
      2.3227879552470015E-01_dp, -1.7002004191404002E-01_dp,&
      2.8353543550268284E-01_dp, -3.2295870972875435E-01_dp,&
      2.9162100501061888E-01_dp, -2.5446398227605715E-01_dp,&
      1.9990573122381708E-01_dp, -1.1722136047578775E-01_dp,&
      4.3074512262631613E-02_dp, -7.3698908494786572E-03_dp,&
      9.0158149241474579E+00_dp, -1.5390032846184512E+00_dp,&
      -1.2578078408173032E+00_dp,  7.3256099290466847E-01_dp,&
      -4.8255480252021304E-01_dp,  7.9110935203918809E-01_dp,&
      -9.2223368631764924E-01_dp,  8.2261801505822885E-01_dp,&
      -7.0152375312903192E-01_dp,  5.4882264799559055E-01_dp,&
      -3.2374652428183265E-01_dp,  1.1943745886308012E-01_dp,&
      -2.0399323236584468E-02_dp,  1.9060405525354774E+01_dp,&
      -4.3442410134014242E+00_dp, -3.3923750244595929E+00_dp,&
      2.3277758289827348E+00_dp, -1.4032292783490181E+00_dp,&
      2.2111149370970846E+00_dp, -2.6514360156752161E+00_dp,&
      2.3400879330259072E+00_dp, -1.9352609112533496E+00_dp,&
      1.5013805536357132E+00_dp, -8.9331348604117622E-01_dp,&
      3.3245002695248244E-01_dp, -5.6923036199617680E-02_dp,&
      4.2321796419078780E+01_dp, -1.2635737527625851E+01_dp,&
      -9.3019892409007774E+00_dp,  7.6613108253972051E+00_dp,&
      -4.3221837921659887E+00_dp,  6.3307699962540935E+00_dp,&
      -7.8520270548486826E+00_dp,  6.8807442180215030E+00_dp,&
      -5.4525567047496342E+00_dp,  4.1550041539563418E+00_dp,&
      -2.5001930251736764E+00_dp,  9.4579959622261767E-01_dp,&
      -1.6363574334611850E-01_dp,  1.0003827134700703E+02_dp,&
      -3.8829042199894573E+01_dp, -2.6360290750789250E+01_dp,&
      2.6857452022470046E+01_dp, -1.4676880411075732E+01_dp,&
      1.9015127170909437E+01_dp, -2.4479606083190603E+01_dp,&
      2.1453660248266239E+01_dp, -1.5990524893902380E+01_dp,&
      1.1692590213258955E+01_dp, -7.1089925888310566E+00_dp,&
      2.7650175937063479E+00_dp, -4.9013357258124818E-01_dp,&
      2.5650279105853537E+02_dp, -1.3001613271097199E+02_dp,&
      -7.8215890320689127E+01_dp,  1.0374859376545058E+02_dp,&
      -5.7993318532586997E+01_dp,  6.1925129027835808E+01_dp,&
      -8.2129941440824297E+01_dp,  7.2996479221982142E+01_dp,&
      -4.9711740845864256E+01_dp,  3.2670927648676113E+01_dp,&
      -1.9601482847566309E+01_dp,  7.9400385997630769E+00_dp,&
      -1.4745807885995803E+00_dp,  7.3530003145728017E+02_dp,&
      -4.9664201976979172E+02_dp, -2.4279101607234665E+02_dp,&
      4.6398305934827602E+02_dp, -2.8940008684901966E+02_dp,&
      2.3403868192771887E+02_dp, -3.0250596676982525E+02_dp,&
      2.7884938757156840E+02_dp, -1.6597480330952183E+02_dp,&
      7.7022134764390358E+01_dp, -3.6333234939720029E+01_dp,&
      1.4834832375776335E+01_dp, -3.0059213601888550E+00_dp,&
      2.4925639619878293E+03_dp, -2.3516399736727049E+03_dp,&
      -7.0669849118397849E+02_dp,  2.6123262478824795E+03_dp,&
      -2.1096774881906767E+03_dp,  1.2752227522425612E+03_dp,&
      -1.2190189717372978E+03_dp,  1.1433571541782767E+03_dp,&
      -5.3704043024832720E+02_dp, -8.5050052992266330E+01_dp,&
      2.6432025336564107E+02_dp, -1.4590397426454322E+02_dp,&
      3.0667907896264261E+01_dp,  1.1452856000529706E+04_dp,&
      -1.6733730631591166E+04_dp,  2.0124842686447407E+03_dp,&
      2.0782792338251493E+04_dp, -3.1172495661780686E+04_dp,&
      2.3937895402534603E+04_dp, -8.4050963154502351E+03_dp,&
      -5.8783772978304578E+03_dp,  1.4993404742007511E+04_dp,&
      -1.7419881693507228E+04_dp,  1.3154166369648410E+04_dp,&
      -6.0320093070800103E+03_dp,  1.2691258478430473E+03_dp,&
      2.4661275214127114E+00_dp, -1.2798267129549132E-01_dp,&
      -1.1697137480153171E-01_dp,  3.9257803663157738E-02_dp,&
      -4.0319593020954211E-02_dp,  6.6187689447338618E-02_dp,&
      -7.2247928359382055E-02_dp,  6.7482097544372566E-02_dp,&
      -6.1252477865463867E-02_dp,  4.8314626699507028E-02_dp,&
      -2.8123924424198074E-02_dp,  1.0339419858641128E-02_dp,&
      -1.7906361701448847E-03_dp,  3.5103259814502201E+00_dp,&
      -2.9727217547633278E-01_dp, -2.6205887594376975E-01_dp,&
      1.0857238160676247E-01_dp, -9.2141383796487078E-02_dp,&
      1.5406834578313486E-01_dp, -1.7142772300076523E-01_dp,&
      1.5726488808531711E-01_dp, -1.4025477042228404E-01_dp,&
      1.1047969889471382E-01_dp, -6.4470419314857599E-02_dp,&
      2.3658250747070033E-02_dp, -4.0673029740227054E-03_dp,&
      6.2410599788925678E+00_dp, -8.2138392898720347E-01_dp,&
      -6.9906495654527301E-01_dp,  3.4410291328684828E-01_dp,&
      -2.5343941141731630E-01_dp,  4.2515725467164173E-01_dp,&
      -4.8284871607865038E-01_dp,  4.3629017443259160E-01_dp,&
      -3.8177479405146436E-01_dp,  3.0011747927293064E-01_dp,&
      -1.7581115021036545E-01_dp,  6.4536453913093872E-02_dp,&
      -1.1036630823958494E-02_dp,  1.2396354706697132E+01_dp,&
      -2.2747834976246768E+00_dp, -1.8659037974661266E+00_dp,&
      1.0729486834844844E+00_dp, -7.0823291387398257E-01_dp,&
      1.1716255191494653E+00_dp, -1.3616528360542641E+00_dp,&
      1.2146420854278106E+00_dp, -1.0394227304439734E+00_dp,&
      8.1420256203015140E-01_dp, -4.7970758970705091E-01_dp,&
      1.7666567904333222E-01_dp, -3.0134269374265207E-02_dp,&
      2.6262194331493351E+01_dp, -6.3444832902403547E+00_dp,&
      -4.9805965392317857E+00_dp,  3.3626392280343960E+00_dp,&
      -2.0268609276880918E+00_dp,  3.2376830951333413E+00_dp,&
      -3.8685734302070061E+00_dp,  3.4125011524571320E+00_dp,&
      -2.8364792541794244E+00_dp,  2.2064813341891907E+00_dp,&
      -1.3109104909016629E+00_dp,  4.8647143855153607E-01_dp,&
      -8.3081066217279925E-02_dp,  5.8791642664628654E+01_dp,&
      -1.8223978038331911E+01_dp, -1.3535672130713879E+01_dp,&
      1.0886746267759019E+01_dp, -6.1196900002518406E+00_dp,&
      9.1643774491776391E+00_dp, -1.1318457824166819E+01_dp,&
      9.9022716234880352E+00_dp, -7.9117201762867282E+00_dp,&
      6.0668367158067333E+00_dp, -3.6456805775684451E+00_dp,&
      1.3726588060126073E+00_dp, -2.3632132863188687E-01_dp,&
      1.3990455162403524E+02_dp, -5.5127425979272914E+01_dp,&
      -3.8049785729902737E+01_dp,  3.7322312094101129E+01_dp,&
      -2.0162618467243483E+01_dp,  2.7134468713776865E+01_dp,&
      -3.4792138438282116E+01_dp,  3.0368635438402379E+01_dp,&
      -2.2961340267827463E+01_dp,  1.7064301226195642E+01_dp,&
      -1.0388397417931811E+01_dp,  4.0108403964288808E+00_dp,&
      -7.0448220676099105E-01_dp,  3.5898010619414435E+02_dp,&
      -1.8010251190337905E+02_dp, -1.1207515297883917E+02_dp,&
      1.3926745663661529E+02_dp, -7.5591370594985008E+01_dp,&
      8.6197068058446064E+01_dp, -1.1457176948523994E+02_dp,&
      1.0089516510941472E+02_dp, -7.0493679332727382E+01_dp,&
      4.8625058388200841E+01_dp, -2.9732088584351676E+01_dp,&
      1.1947228503152537E+01_dp, -2.1840920197937304E+00_dp,&
      1.0166774272757106E+03_dp, -6.5774062040784861E+02_dp,&
      -3.4819173908900984E+02_dp,  5.8698390708558020E+02_dp /)
    REAL(KIND=dp), DIMENSION(13, 24, 5), PARAMETER :: &
      coefdata = RESHAPE ((/ c01, c02, c03, c04 /),(/13,24,5/))

    INTEGER                                  :: irange

     IF(Rc <=  1.0000000000000000E+01_dp) Rc =   1.0000000000000000E+01_dp
     IF(Rc >=  8.5914001464843750E+04_dp) Rc =   8.5914001464843750E+04_dp
     IF(Rc >=  1.0000000000000000E+01_dp .AND. Rc <=  3.1589054611062615E+01_dp) THEN
       irange =  1
       L_b =  1.0000000000000000E+01_dp
       U_b =  3.1589054611062615E+01_dp
     END IF
     IF(Rc >=  3.1589054611062615E+01_dp .AND. Rc <=  3.7775485225370909E+02_dp) THEN
       irange =  2
       L_b =  3.1589054611062615E+01_dp
       U_b =  3.7775485225370909E+02_dp
     END IF
     IF(Rc >=  3.7775485225370909E+02_dp .AND. Rc <=  5.0208914580369265E+03_dp) THEN
       irange =  3
       L_b =  3.7775485225370909E+02_dp
       U_b =  5.0208914580369265E+03_dp
     END IF
     IF(Rc >=  5.0208914580369265E+03_dp .AND. Rc <=  2.9478561984966764E+04_dp) THEN
       irange =  4
       L_b =  5.0208914580369265E+03_dp
       U_b =  2.9478561984966764E+04_dp
     END IF
     IF(Rc >=  2.9478561984966764E+04_dp .AND. Rc <=  8.5914001464843750E+04_dp) THEN
       irange =  5
       L_b =  2.9478561984966764E+04_dp
       U_b =  8.5914001464843750E+04_dp
     END IF

     fit_coef(1:13,1:24) = coefdata(1:13,1:24,irange)

   END SUBROUTINE get_fit_coef_k12

! *****************************************************************************
!> \brief ...
!> \param Rc ...
!> \param L_b ...
!> \param U_b ...
!> \param fit_coef ...
! *****************************************************************************
   SUBROUTINE get_fit_coef_k13(Rc,L_b,U_b,fit_coef)
    REAL(KIND=dp)                            :: Rc, L_b, U_b
    REAL(KIND=dp), DIMENSION(13, 26)         :: fit_coef

    REAL(KIND=dp), DIMENSION(400), PARAMETER :: c01 = (/ &
      1.8838771004380783E-01_dp, -7.8098504554216852E-02_dp,&
      -1.5170263296839995E-02_dp,  8.2399169935392508E-03_dp,&
      3.5286892847422469E-03_dp, -1.8215194066877041E-03_dp,&
      -9.0884305814463765E-04_dp,  6.8756727909990219E-04_dp,&
      3.7413977081317160E-04_dp, -2.6705465221031334E-04_dp,&
      -2.0802563944277461E-04_dp,  2.1290418502755695E-04_dp,&
      -5.2523675707973026E-05_dp,  5.9434702831532238E-01_dp,&
      -2.7182658534478105E-01_dp, -3.5999729360206109E-02_dp,&
      3.2340253606252901E-02_dp,  7.9358310080228263E-03_dp,&
      -7.7844468594381001E-03_dp, -1.8014703887347175E-03_dp,&
      2.8423929944335724E-03_dp,  9.1015987102848297E-04_dp,&
      -1.4560488645141763E-03_dp, -7.3222872823034983E-05_dp,&
      4.8860545790434685E-04_dp, -1.5085140620238292E-04_dp,&
      1.0937368125504157E+00_dp, -5.9132092667529690E-01_dp,&
      -1.8412335702747615E-02_dp,  7.9331829469009338E-02_dp,&
      1.0973664242275066E-03_dp, -2.0316082349166990E-02_dp,&
      1.2853862027112464E-03_dp,  7.0775762212150883E-03_dp,&
      3.4751426157424701E-04_dp, -4.5696828146617506E-03_dp,&
      2.1223196342842286E-03_dp,  1.0726013520497198E-05_dp,&
      -1.5736650409045057E-04_dp,  1.7694613840435505E+00_dp,&
      -1.1676651668520612E+00_dp,  1.0071479256453719E-01_dp,&
      1.6228303909602290E-01_dp, -3.6185203797936424E-02_dp,&
      -4.1378623218189241E-02_dp,  1.5022662586176605E-02_dp,&
      1.3484748664965476E-02_dp, -3.6199651811571775E-03_dp,&
      -1.0288246654012838E-02_dp,  8.7176610157029440E-03_dp,&
      -2.6010672222318890E-03_dp,  2.1442841577039209E-04_dp,&
      2.7395947604311148E+00_dp, -2.2168158237937692E+00_dp,&
      4.6058197011790003E-01_dp,  2.8018864797075105E-01_dp,&
      -1.4313062865562740E-01_dp, -6.4381187850727661E-02_dp,&
      5.1866744842633442E-02_dp,  1.8402911677378096E-02_dp,&
      -1.5630053920933622E-02_dp, -1.6666410769786210E-02_dp,&
      2.1958245525177594E-02_dp, -9.4695529478873997E-03_dp,&
      1.5123425133968700E-03_dp,  4.1819739103477458E+00_dp,&
      -4.1085299248060032E+00_dp,  1.3587434487195018E+00_dp,&
      3.8010926364059955E-01_dp, -3.8974887265449265E-01_dp,&
      -5.7767530618699811E-02_dp,  1.3019397541063171E-01_dp,&
      8.6445563808137237E-03_dp, -4.2595670770739558E-02_dp,&
      -1.5008742214007073E-02_dp,  4.0771284378325862E-02_dp,&
      -2.2393274717496293E-02_dp,  4.4193958726736074E-03_dp,&
      6.3740799341119825E+00_dp, -7.4856748747981650E+00_dp,&
      3.4153135374383137E+00_dp,  2.5465839179415151E-01_dp,&
      -8.7770919045013052E-01_dp,  7.1377608346000626E-02_dp,&
      2.6510553418483845E-01_dp, -4.9379331756714537E-02_dp,&
      -8.9569832352192805E-02_dp,  1.6019869840805893E-02_dp,&
      5.4400827506418850E-02_dp, -4.0239804495804533E-02_dp,&
      9.3312689304133906E-03_dp,  9.7621486867205380E+00_dp,&
      -1.3475250019084454E+01_dp,  7.8972763434980600E+00_dp,&
      -7.1565769086151843E-01_dp, -1.6844459659464686E+00_dp,&
      5.4494023769013600E-01_dp,  4.2698529938918661E-01_dp,&
      -2.2418651826182923E-01_dp, -1.4388654754391872E-01_dp,&
      1.1755325675552541E-01_dp,  2.8878081758365690E-02_dp,&
      -5.2435038016586624E-02_dp,  1.5197313495000905E-02_dp,&
      1.5092987100272357E+01_dp, -2.4074749737500255E+01_dp,&
      1.7349987294818295E+01_dp, -4.1717529748993130E+00_dp,&
      -2.5981745668440559E+00_dp,  1.7898646177351853E+00_dp,&
      4.1191953332704456E-01_dp, -6.1075785276949601E-01_dp,&
      -1.2882271191956388E-01_dp,  3.4442105482404745E-01_dp,&
      -1.1990741103975205E-01_dp, -1.9713219340545846E-02_dp,&
      1.5129094650791500E-02_dp,  2.3710574140122770E+01_dp,&
      -4.2934018545625399E+01_dp,  3.6833617889420324E+01_dp,&
      -1.4114831970803618E+01_dp, -2.2472542537243165E+00_dp,&
      4.3088109269149406E+00_dp, -4.2518679827460348E-01_dp,&
      -1.1804673921484345E+00_dp,  1.7962982669589683E-01_dp,&
      6.4586932424038102E-01_dp, -4.9112812550688612E-01_dp,&
      1.3546324224979389E-01_dp, -9.3472046563761128E-03_dp,&
      3.8416250559670239E+01_dp, -7.7380918682867062E+01_dp,&
      7.6669856574547836E+01_dp, -3.9692668605817900E+01_dp,&
      4.0800907639980108E+00_dp,  7.6431547349679390E+00_dp,&
      -3.2426728324429890E+00_dp, -1.3282905867506054E+00_dp,&
      1.0159546829067250E+00_dp,  6.3516967916479361E-01_dp,&
      -9.3377190156086576E-01_dp,  4.1042024028249102E-01_dp,&
      -6.7363978656170859E-02_dp,  6.6985406661457603E+01_dp,&
      -1.4642414370063676E+02_dp,  1.6193310131526860E+02_dp,&
      -1.0257065719645773E+02_dp,  2.8871318786796056E+01_dp,&
      7.8634102783568860E+00_dp, -8.5200231797122488E+00_dp,&
      1.2264537347714767E-01_dp,  1.9987657976971298E+00_dp,&
      7.7158069079422978E-02_dp, -1.1542897495755511E+00_dp,&
      6.7737646619243352E-01_dp, -1.3657792494227855E-01_dp,&
      1.4740538370956807E+02_dp, -3.3932989241907069E+02_dp,&
      4.0219547600147121E+02_dp, -2.8677094173034959E+02_dp,&
      1.1237202082027906E+02_dp, -4.8057836168082861E+00_dp,&
      -1.6514621066683070E+01_dp,  3.5357507348208697E+00_dp,&
      3.3035381871180327E+00_dp, -7.4033248045744904E-01_dp,&
      -1.6598699253649027E+00_dp,  1.2042118842716378E+00_dp,&
      -2.6920467176461421E-01_dp,  1.5261740947232876E+00_dp,&
      -6.4895738266931657E-01_dp, -1.1564452965859112E-01_dp,&
      7.0980947338764838E-02_dp,  2.6676015641548347E-02_dp,&
      -1.6136455032911386E-02_dp, -6.7340073746126466E-03_dp,&
      6.0342465353809367E-03_dp,  2.8819686877824442E-03_dp,&
      -2.5718142851708921E-03_dp, -1.3273365637482889E-03_dp,&
      1.6243550200292600E-03_dp, -4.2055786525236277E-04_dp,&
      1.7635041108560561E+00_dp, -9.5779430337441518E-01_dp,&
      -3.3226684026626346E-02_dp,  1.3208807728323771E-01_dp,&
      3.3225866313692422E-03_dp, -3.4496297667900466E-02_dp,&
      1.4674666962247453E-03_dp,  1.2123713242266174E-02_dp,&
      8.7892456369925113E-04_dp, -7.8123662744860019E-03_dp,&
      3.3746734942086509E-03_dp,  2.1407084187177066E-04_dp,&
      -3.1845714516548978E-04_dp,  2.2858230764507561E+00_dp,&
      -1.6828828513757415E+00_dp,  2.1342349744092762E-01_dp,&
      2.5321127952670880E-01_dp, -7.0205604577718933E-02_dp,&
      -6.7430673522987303E-02_dp,  2.7518708712689594E-02_dp,&
      2.1972840982545115E-02_dp, -6.8503456890888544E-03_dp,&
      -1.7678791877657449E-02_dp,  1.5800632995275406E-02_dp,&
      -4.9541923552455531E-03_dp,  4.5213141208361152E-04_dp,&
      3.1971135931052292E+00_dp, -3.0674576959479736E+00_dp,&
      8.2727565667300873E-01_dp,  4.1192795171230862E-01_dp,&
      -2.5340716065788732E-01_dp, -9.7677767113723460E-02_dp,&
      9.0618959728272830E-02_dp,  2.7208794713592679E-02_dp,&
      -2.7958096268066484E-02_dp, -2.7052906430807815E-02_dp,&
      3.8668773332420213E-02_dp, -1.7422012026548538E-02_dp,&
      2.8945916612542781E-03_dp,  4.6799701618791243E+00_dp,&
      -5.5646620604346912E+00_dp,  2.2397817355646192E+00_dp,&
      5.0208145461724241E-01_dp, -6.4842250541548618E-01_dp,&
      -6.7592147376930631E-02_dp,  2.1681380729701397E-01_dp,&
      4.0599420621051435E-03_dp, -7.2943200049801968E-02_dp,&
      -1.8572586454638652E-02_dp,  6.6945743501341068E-02_dp,&
      -3.8891162759268932E-02_dp,  7.9717226708572522E-03_dp,&
      7.0397079895684032E+00_dp, -9.9842598075051558E+00_dp,&
      5.3386101825655814E+00_dp,  1.7446186675610861E-01_dp,&
      -1.3948157912004475E+00_dp,  1.7791991429819645E-01_dp,&
      4.2211621944934152E-01_dp, -1.0340330103130466E-01_dp,&
      -1.4705013349847670E-01_dp,  4.5565811980008253E-02_dp,&
      7.7147756940248571E-02_dp, -6.4077566137299757E-02_dp,&
      1.5603547582033700E-02_dp,  1.0786587572187873E+01_dp,&
      -1.7760112751903939E+01_dp,  1.1918980063394129E+01_dp,&
      -1.5539236970758330E+00_dp, -2.5746531195043820E+00_dp,&
      9.9669167169678763E-01_dp,  6.4516326868847929E-01_dp,&
      -4.0607915441255105E-01_dp, -2.2302549663448712E-01_dp,&
      2.2969575799370925E-01_dp,  1.1320877228687479E-02_dp,&
      -7.2862905150103019E-02_dp,  2.3237485312351126E-02_dp,&
      1.6798695470837391E+01_dp, -3.1441464870566055E+01_dp,&
      2.5557113319742395E+01_dp, -7.2192306979436189E+00_dp,&
      -3.7887425571978977E+00_dp,  3.0784789752716946E+00_dp,&
      5.3318993850287921E-01_dp, -1.0629617159049316E+00_dp,&
      -1.6729312653851172E-01_dp,  6.2482228533775386E-01_dp,&
      -2.7075437119322843E-01_dp, -7.6526272998893416E-04_dp,&
      1.9795696185616975E-02_dp,  2.6698448272325432E+01_dp,&
      -5.5678518659905293E+01_dp,  5.3281403655988036E+01_dp,&
      -2.2925621416228282E+01_dp, -2.7448097677013457E+00_dp,&
      7.2733797930171429E+00_dp, -1.0652244860941229E+00_dp,&
      -2.0439448593750167E+00_dp,  4.3742024469413332E-01_dp,&
      1.1815987016225704E+00_dp, -1.0071997092922647E+00_dp,&
      3.2214298124885665E-01_dp, -3.3769710235736237E-02_dp,&
      4.3948016443764217E+01_dp, -9.9715998087285030E+01_dp,&
      1.0903180020753425E+02_dp, -6.2349038976881253E+01_dp,&
      8.4191726553756379E+00_dp,  1.2761370267104230E+01_dp,&
      -6.4816307926839150E+00_dp, -2.1407881050035464E+00_dp,&
      2.2505811330603973E+00_dp,  9.4541418370340546E-01_dp,&
      -1.8728571043550890E+00_dp,  9.3165721994730266E-01_dp,&
      -1.7212775126811192E-01_dp,  7.8045875406746134E+01_dp,&
      -1.8661598260984715E+02_dp,  2.2414431510122250E+02_dp,&
      -1.5518159917081857E+02_dp,  4.9918019842175177E+01_dp,&
      1.1453185884785420E+01_dp, -1.6531485621219627E+01_dp,&
      1.9110397390256724E+00_dp,  4.1466549895305009E+00_dp,&
      -1.2773383144404289E+00_dp, -1.3711697775898219E+00_dp,&
      1.1359083217574477E+00_dp, -2.6834986232570840E-01_dp,&
      1.6737659404428402E+02_dp, -4.0546870390462601E+02_dp,&
      5.0715726314925251E+02_dp, -3.8648362061791448E+02_dp,&
      1.6825446300611861E+02_dp, -1.5904401111568793E+01_dp,&
      -2.4753480319349315E+01_dp,  9.7300048676002913E+00_dp,&
      3.4706756298406813E+00_dp, -2.7684975236319453E+00_dp,&
      -7.2826041594920199E-01_dp,  1.1315837672625995E+00_dp,&
      -3.0636127893899234E-01_dp,  6.2801459623898256E+02_dp,&
      -1.4934850582877016E+03_dp,  1.8458135305428518E+03_dp,&
      -1.4058647887554894E+03_dp,  6.3659031322995588E+02_dp,&
      -1.0314180734592853E+02_dp, -5.1253805381204558E+01_dp,&
      2.0084334666874064E+01_dp,  1.0632240773130309E+01_dp,&
      -4.7733655815247635E+00_dp, -4.8730760134796665E+00_dp,&
      4.2825744412767870E+00_dp, -1.0293980933009121E+00_dp,&
      2.4470051605057291E-01_dp, -6.2837867140790746E-02_dp,&
      -2.3825186082600196E-02_dp,  7.5784912887244772E-03_dp,&
      7.5082468515253169E-03_dp, -2.6455605788751222E-03_dp,&
      -3.3362505117537893E-03_dp,  7.8091376511942739E-04_dp,&
      3.0325200376611385E-03_dp, -2.3347738852380210E-03_dp,&
      3.9936843186023246E-05_dp,  5.8244826441623864E-04_dp,&
      -1.9032202744327542E-04_dp,  8.0022162325527124E-01_dp,&
      -2.4209391447252487E-01_dp, -7.6405614359062499E-02_dp,&
      3.8989771396722744E-02_dp,  2.5419789016015160E-02_dp,&
      -1.5256813711493509E-02_dp, -1.1473513632992018E-02_dp,&
      5.2085413881328025E-03_dp,  1.2850293721253818E-02_dp,&
      -1.3920130297941820E-02_dp,  4.3891393295565915E-03_dp,&
      5.8004137460729440E-04_dp, -4.7036883093655793E-04_dp,&
      1.5788599826464651E+00_dp, -6.1762979410631058E-01_dp,&
      -1.3565786277770472E-01_dp,  1.3250818605318840E-01_dp,&
      4.7423141009731530E-02_dp, -5.6828580488514448E-02_dp,&
      -2.1224822901882736E-02_dp,  2.1560191062029284E-02_dp,&
      3.4712659716572912E-02_dp, -5.3570121520492134E-02_dp,&
      2.8798870932988822E-02_dp, -6.1083685002502114E-03_dp,&
      1.1251907951229265E-04_dp,  2.8208126201441122E+00_dp,&
      -1.4584430358075993E+00_dp, -1.6668814068976517E-01_dp,&
      3.7939412477145662E-01_dp,  5.2727438203323901E-02_dp,&
      -1.7212640736974685E-01_dp, -1.9204293272473808E-02_dp,&
      7.0402478790583201E-02_dp,  7.4962963356748721E-02_dp,&
      -1.6470750799943457E-01_dp,  1.1464911243407885E-01_dp,&
      -3.7209893008122726E-02_dp,  4.6211560443511707E-03_dp,&
      4.9425542029764467E+00_dp, -3.3299793049425701E+00_dp,&
      -1.5240471603939173E-02_dp,  9.6617116551502813E-01_dp,&
      -5.4487412666999822E-02_dp, -4.4981543663568441E-01_dp,&
      5.4235895382646740E-02_dp,  1.9616328040136954E-01_dp,&
      1.1396958212193117E-01_dp, -4.2101103801084033E-01_dp /)
    REAL(KIND=dp), DIMENSION(400), PARAMETER :: c02 = (/ &
      3.5632823078643500E-01_dp, -1.4001776162509430E-01_dp,&
      2.2286666712875530E-02_dp,  8.7125285885676789E+00_dp,&
      -7.4679426200738366E+00_dp,  8.3805209926335500E-01_dp,&
      2.2477163058559704E+00_dp, -6.0060735276290700E-01_dp,&
      -1.0331040612400506E+00_dp,  3.9698448728282515E-01_dp,&
      4.7797940178876780E-01_dp,  1.1436011046665012E-02_dp,&
      -8.7501751935782923E-01_dp,  9.1713404922379360E-01_dp,&
      -4.1604826319244603E-01_dp,  7.5505830008108218E-02_dp,&
      1.5605853014796613E+01_dp, -1.6640738573993023E+01_dp,&
      3.9973532644106071E+00_dp,  4.7869251169441966E+00_dp,&
      -2.5684017026470856E+00_dp, -2.0236077415630396E+00_dp,&
      1.5736463565912060E+00_dp,  9.8572630187217014E-01_dp,&
      -7.7868319784195095E-01_dp, -1.2477373616278120E+00_dp,&
      1.9014513769547758E+00_dp, -1.0102166550823963E+00_dp,&
      2.0453369882137556E-01_dp,  2.8549616253145054E+01_dp,&
      -3.7205075770883383E+01_dp,  1.4271765352571258E+01_dp,&
      8.9932074739321770E+00_dp, -8.6798219050700265E+00_dp,&
      -2.8631184094001214E+00_dp,  4.9827139884730887E+00_dp,&
      1.4257317765234938E+00_dp, -3.8503419132165200E+00_dp,&
      1.8060419072857684E-01_dp,  2.6148900288649957E+00_dp,&
      -1.8695443351115448E+00_dp,  4.3631072153907585E-01_dp,&
      5.3552984772777414E+01_dp, -8.4175532531260529E+01_dp,&
      4.6043669292644864E+01_dp,  1.2581369109054569E+01_dp,&
      -2.5785157300270445E+01_dp,  3.6426327688394516E-01_dp,&
      1.3334789657578380E+01_dp, -4.7226254964308417E-01_dp,&
      -1.2526270169907576E+01_dp,  9.4142206495359755E+00_dp,&
      -9.7670970926161171E-01_dp, -1.6726174022636047E+00_dp,&
      5.9874706221520879E-01_dp,  1.0346475698874463E+02_dp,&
      -1.9421400275760911E+02_dp,  1.4292733600886447E+02_dp,&
      -2.1682521535311881E+00_dp, -6.7899121747504481E+01_dp,&
      2.4906170745096762E+01_dp,  2.8087316716428784E+01_dp,&
      -1.4449093427136949E+01_dp, -2.8736513116670000E+01_dp,&
      4.0767438543005156E+01_dp, -2.2398062557021266E+01_dp,&
      5.4405001864859486E+00_dp, -3.4631065659938148E-01_dp,&
      2.0750616910138754E+02_dp, -4.6065142702781014E+02_dp,&
      4.4009500583974625E+02_dp, -1.2652217098767277E+02_dp,&
      -1.4255338662322163E+02_dp,  1.3009558694582910E+02_dp,&
      2.4337692154912165E+01_dp, -6.8052930184984419E+01_dp,&
      -2.3709608232610183E+01_dp,  9.9442268821700665E+01_dp,&
      -8.4274944171226693E+01_dp,  3.3636040311091861E+01_dp,&
      -5.5275330736437418E+00_dp,  4.4235457824221572E+02_dp,&
      -1.1417219968036247E+03_dp,  1.3690824125389506E+03_dp,&
      -7.7065096290590293E+02_dp, -8.8442075039759473E+01_dp,&
      4.2234029954766072E+02_dp, -1.6089957770769391E+02_dp,&
      -1.4258882763239106E+02_dp,  1.2603829491926012E+02_dp,&
      5.0804366296990459E+01_dp, -1.1980481260857118E+02_dp,&
      6.7459853564141966E+01_dp, -1.3960462430030432E+01_dp,&
      1.1324481119281395E+03_dp, -3.2684217630798435E+03_dp,&
      4.6328527365543323E+03_dp, -3.7266964190295712E+03_dp,&
      1.2856665138480816E+03_dp,  5.6297970920071646E+02_dp,&
      -7.5128039018902086E+02_dp,  7.0908728648373369E+01_dp,&
      2.9917130128753172E+02_dp, -1.5232184462135913E+02_dp,&
      -4.8594625636135675E+01_dp,  6.7185755709058370E+01_dp,&
      -1.8010386679280117E+01_dp,  2.0001036223672011E+00_dp,&
      -5.3620407054782981E-01_dp, -1.9434054411717244E-01_dp,&
      7.0618726869747614E-02_dp,  6.2242609610535821E-02_dp,&
      -2.5597292416828135E-02_dp, -2.7813756603712155E-02_dp,&
      7.9491914776396028E-03_dp,  2.6680695709919679E-02_dp,&
      -2.2810343453090697E-02_dp,  2.7161809486165871E-03_dp,&
      4.0718478547287487E-03_dp, -1.4880205641258518E-03_dp,&
      2.5450615398879743E+00_dp, -9.8744397592581534E-01_dp,&
      -2.2902266959137468E-01_dp,  2.0922986206151875E-01_dp,&
      8.2779560412023001E-02_dp, -8.8686545550320794E-02_dp,&
      -3.7958590200944345E-02_dp,  3.2998767331186209E-02_dp,&
      5.7441855270214746E-02_dp, -8.2952472588137552E-02_dp,&
      4.1461489832071360E-02_dp, -7.2309981798897591E-03_dp,&
      -3.4486800387515759E-04_dp,  3.8383945496778296E+00_dp,&
      -2.1858886322144140E+00_dp, -2.2638494850866908E-01_dp,&
      5.9654548742227309E-01_dp,  8.2375367991094961E-02_dp,&
      -2.7228188920384877E-01_dp, -3.2991216309614123E-02_dp,&
      1.1069144407749401E-01_dp,  1.2288856763142823E-01_dp,&
      -2.6231090706733151E-01_dp,  1.7948962970980861E-01_dp,&
      -5.6968770565566255E-02_dp,  6.8079040584616184E-03_dp,&
      6.3639884662472506E+00_dp, -4.8950253392955796E+00_dp,&
      6.7035562751566802E-02_dp,  1.4977453112460624E+00_dp,&
      -9.4857290242106540E-02_dp, -7.0786182174407719E-01_dp,&
      8.3499909648241832E-02_dp,  3.0861847626582534E-01_dp,&
      1.9124789670380135E-01_dp, -6.7776371800663660E-01_dp,&
      5.6824943668256445E-01_dp, -2.2147300860757518E-01_dp,&
      3.4925516438103342E-02_dp,  1.1100384232886654E+01_dp,&
      -1.0849365072746094E+01_dp,  1.4500169429794771E+00_dp,&
      3.4281034563810699E+00_dp, -9.5548658310972046E-01_dp,&
      -1.6125902800476346E+00_dp,  6.2195077294496282E-01_dp,&
      7.5138806761421051E-01_dp,  4.4232035492933294E-02_dp,&
      -1.4236095789326435E+00_dp,  1.4787960249515941E+00_dp,&
      -6.6792149947902735E-01_dp,  1.2078043058881406E-01_dp,&
      1.9997254830061060E+01_dp, -2.3977706500886121E+01_dp,&
      6.3262970427801459E+00_dp,  7.2118993888159375E+00_dp,&
      -4.0034174842598853E+00_dp, -3.1512209966526510E+00_dp,&
      2.4653796429483354E+00_dp,  1.5660770527154297E+00_dp,&
      -1.1858015495148018E+00_dp, -2.0863186414852812E+00_dp,&
      3.1158573924681550E+00_dp, -1.6479978556059860E+00_dp,&
      3.3293613764598917E-01_dp,  3.7002091361283142E+01_dp,&
      -5.3343789674459856E+01_dp,  2.1798301205375779E+01_dp,&
      1.3490742966183797E+01_dp, -1.3399645593923985E+01_dp,&
      -4.5399329672371129E+00_dp,  7.8331708134372793E+00_dp,&
      2.3842813657243171E+00_dp, -6.1312609096237622E+00_dp,&
      6.9373920027707620E-02_dp,  4.4439580524589513E+00_dp,&
      -3.1287274765904054E+00_dp,  7.2687886436644078E-01_dp,&
      7.0330646344558374E+01_dp, -1.2042385478592142E+02_dp,&
      6.8896945123466992E+01_dp,  1.9248160282366094E+01_dp,&
      -3.9785945322970619E+01_dp, -1.0757313037469426E-01_dp,&
      2.1300262950471573E+01_dp, -5.9398954626295727E-02_dp,&
      -2.0726405213802341E+01_dp,  1.5013319224419423E+01_dp,&
      -1.0206915169707462E+00_dp, -3.0611082438918387E+00_dp,&
      1.0504012964699687E+00_dp,  1.3771889621960588E+02_dp,&
      -2.7778114229196626E+02_dp,  2.1092710837289934E+02_dp,&
      4.9544962436490181E-01_dp, -1.0656701841455923E+02_dp,&
      3.5610982335235697E+01_dp,  4.7522077277759607E+01_dp,&
      -2.0742921049299113E+01_dp, -5.1214696900496918E+01_dp,&
      6.8525574756543946E+01_dp, -3.6081947353505832E+01_dp,&
      8.0689565180881910E+00_dp, -3.3115643672049860E-01_dp,&
      2.7951222706689759E+02_dp, -6.5906712429888637E+02_dp,&
      6.4323098194907232E+02_dp, -1.6677447485585066E+02_dp,&
      -2.4162966307021762E+02_dp,  1.9677031498186125E+02_dp,&
      6.0574620887850372E+01_dp, -1.0971429429810229E+02_dp,&
      -6.6686880639177701E+01_dp,  1.9421790532342177E+02_dp,&
      -1.5486964911422615E+02_dp,  5.9425108459636675E+01_dp,&
      -9.4080145408358931E+00_dp,  5.9731652416151724E+02_dp,&
      -1.6234750522173472E+03_dp,  1.9836471844267119E+03_dp,&
      -1.0519495956941482E+03_dp, -2.9350943708598510E+02_dp,&
      7.3559653074966434E+02_dp, -1.8444178048301910E+02_dp,&
      -3.3747019072196520E+02_dp,  2.0501261919104977E+02_dp,&
      1.9472432317933209E+02_dp, -3.1351374896534389E+02_dp,&
      1.6602020342171713E+02_dp, -3.3574624921311873E+01_dp,&
      1.4305952826928835E+03_dp, -4.3148429996300356E+03_dp,&
      6.2934312136573408E+03_dp, -5.0322189338736098E+03_dp,&
      1.3515555677503437E+03_dp,  1.4914175783605847E+03_dp,&
      -1.5590391042662025E+03_dp,  2.5540483265629923E+01_dp,&
      9.2917473249960813E+02_dp, -7.1183738950361192E+02_dp,&
      1.7736950337469372E+02_dp,  3.3996199940030365E+01_dp,&
      -2.0265055920456604E+01_dp,  5.3260400513964350E+03_dp,&
      -1.6436105385546998E+04_dp,  2.5548454078470353E+04_dp,&
      -2.4072536527098953E+04_dp,  1.2929492785729244E+04_dp,&
      -1.5539864332605835E+03_dp, -2.8010923463387767E+03_dp,&
      1.3421428470689034E+03_dp,  6.7886952374328268E+02_dp,&
      -8.2031294944055298E+02_dp,  1.5680250796613109E+02_dp,&
      1.0539258593554351E+02_dp, -4.3152598585647880E+01_dp,&
      2.7522136583648199E-01_dp, -3.2217110577238542E-02_dp,&
      -1.7033147481772627E-02_dp,  2.0210122815219334E-03_dp,&
      3.9289578029764833E-03_dp, -3.9930449290441994E-05_dp,&
      -1.9887361467012953E-03_dp,  3.6898435626645432E-04_dp,&
      7.3280730329796837E-04_dp, -1.1210128908484710E-04_dp,&
      -5.1629849116574338E-04_dp,  3.8585425500095789E-04_dp,&
      -9.0580414399608951E-05_dp,  9.2165986573140568E-01_dp,&
      -1.3249464016020585E-01_dp, -6.5239324099907584E-02_dp,&
      1.3110109010924940E-02_dp,  1.6453340473596419E-02_dp,&
      -1.8265742727726945E-03_dp, -8.6103449023340675E-03_dp,&
      2.2159504354805157E-03_dp,  3.4794700520068469E-03_dp,&
      -1.2533943941271351E-03_dp, -1.8168584053897300E-03_dp,&
      1.5997969588019021E-03_dp, -3.9784620927895906E-04_dp,&
      1.9041320449174821E+00_dp, -3.7234333772363021E-01_dp,&
      -1.6365326189221707E-01_dp,  5.5534422369305905E-02_dp,&
      4.6089398503204683E-02_dp, -1.2265328668024897E-02_dp,&
      -2.5382790394936194E-02_dp,  9.3490699632645686E-03_dp,&
      1.1622195051795290E-02_dp, -7.3570743970728667E-03_dp,&
      -3.2261443745580896E-03_dp,  4.2824229569487044E-03_dp,&
      -1.1778304868165397E-03_dp,  3.6316569669487175E+00_dp,&
      -9.7730941204074340E-01_dp, -3.7309973697493126E-01_dp,&
      1.9511449684300722E-01_dp,  1.1472304455885286E-01_dp,&
      -5.3617761770916240E-02_dp, -6.7215522012602877E-02_dp,&
      3.4468238642844179E-02_dp,  3.5251414090503870E-02_dp,&
      -3.2843194103284207E-02_dp, -6.1769442529466899E-05_dp,&
      9.2371576567999401E-03_dp, -3.0152739805040869E-03_dp,&
      6.9044315846491511E+00_dp, -2.4954078084649751E+00_dp,&
      -7.9752233983078080E-01_dp,  6.1910169828591066E-01_dp,&
      2.5700006957784816E-01_dp, -1.9664588521910206E-01_dp,&
      -1.6279316206764638E-01_dp,  1.1756639080199344E-01_dp,&
      9.9690988498802063E-02_dp, -1.2855067522072000E-01_dp,&
      3.3851769665174079E-02_dp,  1.2487678317695030E-02_dp,&
      -6.4025099233952784E-03_dp,  1.3402802913338570E+01_dp,&
      -6.3224482797045658E+00_dp, -1.5803882488352667E+00_dp,&
      1.8639565830791769E+00_dp,  4.9440377028852905E-01_dp,&
      -6.6058719060697912E-01_dp, -3.5203000086883229E-01_dp,&
      3.8416444626006885E-01_dp,  2.6219933074993468E-01_dp,&
      -4.6750014207051527E-01_dp,  2.1565685107100283E-01_dp,&
      -1.9590633219356376E-02_dp, -8.3188142246124162E-03_dp,&
      2.6847189168859529E+01_dp, -1.6164600954038654E+01_dp,&
      -2.7348299274314170E+00_dp,  5.4804038804993898E+00_dp,&
      6.3908178640514468E-01_dp, -2.1090707289755835E+00_dp,&
      -6.0397632454309369E-01_dp,  1.2258676026166442E+00_dp,&
      6.0900085656743030E-01_dp, -1.6095450133803784E+00_dp,&
      1.0185781424420182E+00_dp, -2.6410536241534338E-01_dp,&
      1.7190485044331690E-02_dp,  5.5920030994857392E+01_dp,&
      -4.2359987862833194E+01_dp, -3.0851699293052208E+00_dp,&
      1.6003551733561032E+01_dp, -7.5845896807267987E-01_dp,&
      -6.4758768912300226E+00_dp, -2.7997762367110551E-01_dp,&
      3.8438366690735388E+00_dp,  9.6178031191591695E-01_dp,&
      -5.1789423674060124E+00_dp,  4.1897184836450609E+00_dp,&
      -1.5066589469240397E+00_dp,  2.1023744160454813E-01_dp,&
      1.2214165365194877E+02_dp, -1.1560316535122182E+02_dp,&
      5.2431654538003061E+00_dp,  4.6756646171042270E+01_dp,&
      -1.1638136061220804E+01_dp, -1.8825294369662465E+01_dp,&
      4.6901099778948048E+00_dp,  1.1684043831662752E+01_dp,&
      -1.5690448976060742E+00_dp, -1.4469520844727743E+01_dp,&
      1.5229016181347497E+01_dp, -6.7413191274813640E+00_dp,&
      1.1871712862612187E+00_dp,  2.8292219949744720E+02_dp,&
      -3.3466585644002333E+02_dp,  6.5399374381441106E+01_dp,&
      1.3547954808764075E+02_dp, -7.2400177578816127E+01_dp,&
      -4.7247337505445195E+01_dp,  3.5254490079699167E+01_dp /)
    REAL(KIND=dp), DIMENSION(400), PARAMETER :: c03 = (/ &
      3.1951406154569082E+01_dp, -2.7863833003973856E+01_dp,&
      -2.4615790099798474E+01_dp,  4.4154903132889864E+01_dp,&
      -2.4321043128719936E+01_dp,  5.0336537486842516E+00_dp,&
      7.0657858026784504E+02_dp, -1.0528573743078830E+03_dp,&
      4.1625512279061735E+02_dp,  3.6427444425345516E+02_dp,&
      -3.8770748215573474E+02_dp, -4.8519737539136528E+01_dp,&
      1.8255864495792844E+02_dp,  4.9003198502854644E+01_dp,&
      -1.8896312119490256E+02_dp,  8.1337993659320531E+01_dp,&
      4.5525635099598908E+01_dp, -5.2286864309302139E+01_dp,&
      1.4107728865636403E+01_dp,  1.9572003502958112E+03_dp,&
      -3.7275124443070590E+03_dp,  2.5346776221187474E+03_dp,&
      5.1376728855467570E+02_dp, -1.9308763300790281E+03_dp,&
      7.9050799076938517E+02_dp,  5.7143225950275246E+02_dp,&
      -3.4470294413325115E+02_dp, -6.8440314851472851E+02_dp,&
      1.0471215545682312E+03_dp, -6.3503878270573148E+02_dp,&
      1.8429203829230039E+02_dp, -1.9468221079821902E+01_dp,&
      6.5615697945100492E+03_dp, -1.5941114910537892E+04_dp,&
      1.7283771807502479E+04_dp, -7.2779267165695510E+03_dp,&
      -4.8084545703952344E+03_dp,  8.8537836032320683E+03_dp,&
      -5.0434849558105634E+03_dp,  5.9362692294717317E+02_dp,&
      2.7660294111299510E+02_dp,  8.0161530341845844E+02_dp,&
      -1.1650257023506490E+03_dp,  6.0897733306934356E+02_dp,&
      -1.2237560126278971E+02_dp,  2.2627540419257364E+00_dp,&
      -2.7968575736450180E-01_dp, -1.4518668455604195E-01_dp,&
      2.0265924722043079E-02_dp,  3.4338618611475344E-02_dp,&
      -1.2556065673962331E-03_dp, -1.7531784982831655E-02_dp,&
      3.5793660343997852E-03_dp,  6.6231604787169770E-03_dp,&
      -1.3867866855376788E-03_dp, -4.3395264768239942E-03_dp,&
      3.3684735669525651E-03_dp, -8.0229330712122920E-04_dp,&
      3.0613725230519471E+00_dp, -5.8640457056345652E-01_dp,&
      -2.6339818110169105E-01_dp,  8.2855001233820474E-02_dp,&
      7.3722316481618991E-02_dp, -1.7015690354151666E-02_dp,&
      -4.0113596090714465E-02_dp,  1.3619066177511220E-02_dp,&
      1.7832370619744201E-02_dp, -1.0005418487325966E-02_dp,&
      -6.1379340236641386E-03_dp,  7.0291927722965558E-03_dp,&
      -1.8751685593083232E-03_dp,  5.0576570915087578E+00_dp,&
      -1.4710755111222258E+00_dp, -5.6312085551876734E-01_dp,&
      2.9587574582447002E-01_dp,  1.7758263331967478E-01_dp,&
      -7.9480406519325253E-02_dp, -1.0353547727129607E-01_dp,&
      5.0871576222639547E-02_dp,  5.3464895969474464E-02_dp,&
      -4.7025455703949920E-02_dp, -2.8649889086355728E-03_dp,&
      1.5163927370369992E-02_dp, -4.7605351840397164E-03_dp,&
      9.2666854098509734E+00_dp, -3.7037123145769315E+00_dp,&
      -1.1810845674556205E+00_dp,  9.3266985234712430E-01_dp,&
      3.9541518329595232E-01_dp, -2.9288519060055263E-01_dp,&
      -2.4999841635163031E-01_dp,  1.7350630646750761E-01_dp,&
      1.5126192191822729E-01_dp, -1.8602668410133402E-01_dp,&
      4.2058952933339604E-02_dp,  2.3004266980576980E-02_dp,&
      -1.0410473830431016E-02_dp,  1.7967075304979097E+01_dp,&
      -9.2886003844612315E+00_dp, -2.3166503256513042E+00_dp,&
      2.7793145895593954E+00_dp,  7.6579508186925827E-01_dp,&
      -9.8065539968708293E-01_dp, -5.4434577208491419E-01_dp,&
      5.6487894573947284E-01_dp,  4.0128002438923693E-01_dp,&
      -6.7985365844418810E-01_dp,  2.9335374977600032E-01_dp,&
      -1.3760417124864824E-02_dp, -1.5579665225528172E-02_dp,&
      3.6330014596264910E+01_dp, -2.3558077039701011E+01_dp,&
      -4.0032080208899892E+00_dp,  8.1023328627947695E+00_dp,&
      1.0572474369212368E+00_dp, -3.1246715021136469E+00_dp,&
      -9.7288399455208796E-01_dp,  1.7980101554684516E+00_dp,&
      9.6548627911437968E-01_dp, -2.3618248937200388E+00_dp,&
      1.4256316289637909E+00_dp, -3.3674236693570941E-01_dp,&
      1.3035310486665157E-02_dp,  7.6564297545359281E+01_dp,&
      -6.1327459250609188E+01_dp, -4.7038085252667656E+00_dp,&
      2.3522134846498993E+01_dp, -6.2074232510114502E-01_dp,&
      -9.6274919508419270E+00_dp, -7.4907562113744564E-01_dp,&
      5.6473524210207273E+00_dp,  1.7877391201538855E+00_dp,&
      -7.7760616092732224E+00_dp,  6.0028051651270777E+00_dp,&
      -2.0515988031519519E+00_dp,  2.6498768518488430E-01_dp,&
      1.6908758380027180E+02_dp, -1.6620256620995710E+02_dp,&
      5.8556451666173768E+00_dp,  6.8594902354899986E+01_dp,&
      -1.4646639889433427E+01_dp, -2.8503986567123022E+01_dp,&
      5.4011736632665412E+00_dp,  1.7407266610755649E+01_dp,&
      -2.1695266703893210E-01_dp, -2.3183117951883965E+01_dp,&
      2.2747354795410811E+01_dp, -9.6248972377227879E+00_dp,&
      1.6232770520249808E+00_dp,  3.9496620712272983E+02_dp,&
      -4.7630587177234480E+02_dp,  8.2634984633242880E+01_dp,&
      2.0043881475177201E+02_dp, -9.3256133931078750E+01_dp,&
      -7.6938636554802741E+01_dp,  4.5382945051577323E+01_dp,&
      5.0541291906442254E+01_dp, -3.0180631932035514E+01_dp,&
      -5.1804380534411720E+01_dp,  7.3703321624210730E+01_dp,&
      -3.7832717690973290E+01_dp,  7.4886753735631828E+00_dp,&
      9.8936735122169955E+02_dp, -1.4723135861653050E+03_dp,&
      5.1661296919801975E+02_dp,  5.6775300121587384E+02_dp,&
      -4.9823328967242446E+02_dp, -1.4291981765414513E+02_dp,&
      2.5077300933486598E+02_dp,  1.1280997635389373E+02_dp,&
      -2.4043309466984022E+02_dp,  1.8533960613943030E+01_dp,&
      1.5008777209583707E+02_dp, -1.0988215815683799E+02_dp,&
      2.6126498062203950E+01_dp,  2.7151255050497193E+03_dp,&
      -5.0457127958801984E+03_dp,  2.9824323650975698E+03_dp,&
      1.2838985854569405E+03_dp, -2.5571835909206930E+03_dp,&
      4.2544336126025911E+02_dp,  1.0994628841103713E+03_dp,&
      -1.2735938497924820E+02_dp, -1.2861687784670435E+03_dp,&
      1.2636200369734484E+03_dp, -4.3735779232907760E+02_dp,&
      -1.3792064567093755E+01_dp,  3.3103909845024816E+01_dp,&
      8.5120394945424250E+03_dp, -1.9972380653537413E+04_dp,&
      1.8712883353138033E+04_dp, -2.4296838299905257E+03_dp,&
      -1.1695432627920640E+04_dp,  9.8982067857411967E+03_dp,&
      6.8744963878254964E+02_dp, -4.5712455095728219E+03_dp,&
      -1.0807863326674956E+03_dp,  6.7875580160934505E+03_dp,&
      -6.3715584765173253E+03_dp,  2.7904520792497096E+03_dp,&
      -5.0512451195331687E+02_dp,  3.7015110168653046E+04_dp,&
      -1.0359356753096696E+05_dp,  1.3988074174530094E+05_dp,&
      -1.0150724524384555E+05_dp,  1.4669819151632983E+04_dp,&
      5.4938546232760098E+04_dp, -7.1495818225217707E+04_dp,&
      5.0437331500296590E+04_dp, -2.4139642994171907E+04_dp,&
      8.5816358667112163E+03_dp, -2.4305935666411028E+03_dp,&
      5.1492651788062108E+02_dp, -5.4029802229116505E+01_dp,&
      2.8766437095860359E-01_dp, -1.2841648915210033E-02_dp,&
      -1.1749544511224883E-02_dp,  3.9124751913915002E-03_dp,&
      -4.1340943883577163E-03_dp,  6.7857802334998418E-03_dp,&
      -7.5294031286692407E-03_dp,  7.2175306126584524E-03_dp,&
      -6.7009368542467326E-03_dp,  5.3904146360225216E-03_dp,&
      -3.2096830445151438E-03_dp,  1.2124425357336903E-03_dp,&
      -2.1637406162705776E-04_dp,  9.7356679780566036E-01_dp,&
      -5.4345431749096215E-02_dp, -4.8904874253147546E-02_dp,&
      1.8036518570911850E-02_dp, -1.7361622406643535E-02_dp,&
      2.8759362863345999E-02_dp, -3.2180437961153392E-02_dp,&
      3.0604917054349139E-02_dp, -2.8207332208422350E-02_dp,&
      2.2678944358017036E-02_dp, -1.3518527121475700E-02_dp,&
      5.1039031798528591E-03_dp, -9.0864035037277143E-04_dp,&
      2.0530666914935956E+00_dp, -1.5918196985192454E-01_dp,&
      -1.3979713465522753E-01_dp,  5.8990016258980232E-02_dp,&
      -5.0479493514827861E-02_dp,  8.4318744789823125E-02_dp,&
      -9.5580479758503561E-02_dp,  8.9926670184525562E-02_dp,&
      -8.1951748270375741E-02_dp,  6.5826636399679089E-02_dp,&
      -3.9316553314932813E-02_dp,  1.4841315675822706E-02_dp,&
      -2.6340794368276848E-03_dp,  4.0316235573187873E+00_dp,&
      -4.3717905236020804E-01_dp, -3.7355720461015557E-01_dp,&
      1.8022229937573989E-01_dp, -1.3839328046856253E-01_dp,&
      2.3139075155391939E-01_dp, -2.6640028065715393E-01_dp,&
      2.4795661325252652E-01_dp, -2.2289816786634609E-01_dp,&
      1.7876458006240628E-01_dp, -1.0708594501709379E-01_dp,&
      4.0458337362132495E-02_dp, -7.1627969268156346E-03_dp,&
      7.9515127037254940E+00_dp, -1.1725795416101994E+00_dp,&
      -9.7139505078684429E-01_dp,  5.3572393763784321E-01_dp,&
      -3.7368202650800936E-01_dp,  6.1869498945049961E-01_dp,&
      -7.2552440352875003E-01_dp,  6.6842200013424780E-01_dp,&
      -5.9088129861833538E-01_dp,  4.7264433898376779E-01_dp,&
      -2.8431635449514581E-01_dp,  1.0769203775409864E-01_dp,&
      -1.9041687452794855E-02_dp,  1.6132294510459047E+01_dp,&
      -3.1414300269804429E+00_dp, -2.5084546797336125E+00_dp,&
      1.5919506211215777E+00_dp, -1.0199649239248365E+00_dp,&
      1.6474808867343209E+00_dp, -1.9752177536142645E+00_dp,&
      1.8027216083947462E+00_dp, -1.5596608718908076E+00_dp,&
      1.2416080767881754E+00_dp, -7.5126157559420226E-01_dp,&
      2.8603278415470540E-01_dp, -5.0617454119410023E-02_dp,&
      3.4064569212641622E+01_dp, -8.5769738548301273E+00_dp,&
      -6.5423787884993834E+00_dp,  4.8406599765190608E+00_dp,&
      -2.8877282478001232E+00_dp,  4.4534784008756505E+00_dp,&
      -5.4834935225951478E+00_dp,  4.9661825471664365E+00_dp,&
      -4.1752937122300180E+00_dp,  3.2934090517963024E+00_dp,&
      -2.0084125823659398E+00_dp,  7.7180910515991674E-01_dp,&
      -1.3721280155604926E-01_dp,  7.5626882675942724E+01_dp,&
      -2.4341047240189681E+01_dp, -1.7493213476710846E+01_dp,&
      1.5398174306816493E+01_dp, -8.7275747587617456E+00_dp,&
      1.2451306283479628E+01_dp, -1.5814730127661882E+01_dp,&
      1.4263812050450625E+01_dp, -1.1531252583607163E+01_dp,&
      8.9274705109811467E+00_dp, -5.4952489571487435E+00_dp,&
      2.1450145097837918E+00_dp, -3.8566671695744043E-01_dp,&
      1.7878769023909365E+02_dp, -7.3409482660028232E+01_dp,&
      -4.8603840481754148E+01_dp,  5.2510737919075773E+01_dp,&
      -2.9190434914792892E+01_dp,  3.6789271998095046E+01_dp,&
      -4.8290125513308809E+01_dp,  4.3685642487586676E+01_dp,&
      -3.3425817156494972E+01_dp,  2.4846648059952049E+01_dp,&
      -1.5396980073755357E+01_dp,  6.1596692561473594E+00_dp,&
      -1.1327889201084458E+00_dp,  4.5853266997398288E+02_dp,&
      -2.4219834828292343E+02_dp, -1.4185912981973058E+02_dp,&
      1.9818507281291306E+02_dp, -1.1363958650629472E+02_dp,&
      1.1864500714177012E+02_dp, -1.5936064769189525E+02_dp,&
      1.4653462807503183E+02_dp, -1.0363950474324729E+02_dp,&
      6.9803513230716248E+01_dp, -4.2493410132349823E+01_dp,&
      1.7571722994223919E+01_dp, -3.3652356136199639E+00_dp,&
      1.3149209017368075E+03_dp, -9.1410171353466365E+02_dp,&
      -4.3355981239428490E+02_dp,  8.6854889246174923E+02_dp,&
      -5.5717720241772929E+02_dp,  4.4729901591974419E+02_dp,&
      -5.8023403991716850E+02_dp,  5.5363750980013492E+02_dp,&
      -3.5036288911768412E+02_dp,  1.7673630965596453E+02_dp,&
      -8.7764967631492297E+01_dp,  3.6165599936797761E+01_dp,&
      -7.4275840619886511E+00_dp,  4.4599678882045982E+03_dp,&
      -4.2852510926403547E+03_dp, -1.2345242410977692E+03_dp,&
      4.8020310925072372E+03_dp, -3.9712192527651519E+03_dp,&
      2.4421962410592055E+03_dp, -2.3444652516106917E+03_dp,&
      2.2748220642734714E+03_dp, -1.2153075874252681E+03_dp,&
      4.2618549893802005E+01_dp,  3.6957056541266746E+02_dp,&
      -2.2467629199390655E+02_dp,  4.8301366431393674E+01_dp,&
      2.0515552105599014E+04_dp, -3.0251434673870317E+04_dp,&
      3.8760590293516493E+03_dp,  3.7614063826257821E+04_dp,&
      -5.7087356760206945E+04_dp,  4.4626244651310022E+04_dp,&
      -1.6930953342940349E+04_dp, -8.6342938793903559E+03_dp,&
      2.5039453054521509E+04_dp, -2.9727441111072243E+04_dp,&
      2.2663651551108436E+04_dp, -1.0448354492148857E+04_dp,&
      2.2049629835911287E+03_dp,  2.3711821326754712E+00_dp,&
      -1.1232868750598661E-01_dp, -1.0232817697756741E-01_dp,&
      3.5035029792710649E-02_dp, -3.6080149837860856E-02_dp,&
      5.9384272779209310E-02_dp, -6.6034549944533560E-02_dp,&
      6.3165197301667780E-02_dp, -5.8533667022985264E-02_dp,&
      4.7080124186021241E-02_dp, -2.8040678094242519E-02_dp,&
      1.0590404330460124E-02_dp, -1.8887168866975497E-03_dp,&
      3.2950022547352660E+00_dp, -2.4869033985187267E-01_dp,&
      -2.1951170275277102E-01_dp,  9.0239524034429530E-02_dp /)
    REAL(KIND=dp), DIMENSION(400), PARAMETER :: c04 = (/ &
      -7.8836509708512373E-02_dp,  1.3178061660818258E-01_dp,&
      -1.4891101086305505E-01_dp,  1.4038052557472530E-01_dp,&
      -1.2827505792189706E-01_dp,  1.0306674552536969E-01_dp,&
      -6.1520533848826123E-02_dp,  2.3216931723502306E-02_dp,&
      -4.1222161991565217E-03_dp,  5.6592328962619902E+00_dp,&
      -6.5682359828028736E-01_dp, -5.6224091737511694E-01_dp,&
      2.6926868745701449E-01_dp, -2.0725768669908209E-01_dp,&
      3.4798639207117904E-01_dp, -3.9993937306967442E-01_dp,&
      3.7233564292827848E-01_dp, -3.3521912095457151E-01_dp,&
      2.6894454176131877E-01_dp, -1.6101715682600454E-01_dp,&
      6.0796203847490786E-02_dp, -1.0759696759551582E-02_dp,&
      1.0820432159449139E+01_dp, -1.7387845708330643E+00_dp,&
      -1.4425878741646769E+00_dp,  7.9167895347904427E-01_dp,&
      -5.5153935438914514E-01_dp,  9.1858113260673568E-01_dp,&
      -1.0754845180690036E+00_dp,  9.9058914214781446E-01_dp,&
      -8.7711044783346115E-01_dp,  7.0204070734569368E-01_dp,&
      -4.2204323840163149E-01_dp,  1.5970999475494435E-01_dp,&
      -2.8218316930005444E-02_dp,  2.1975384938130706E+01_dp,&
      -4.6086047370349190E+00_dp, -3.6880188824890627E+00_dp,&
      2.3253201370596082E+00_dp, -1.4865249175920201E+00_dp,&
      2.4206052762959338E+00_dp, -2.8969618325004678E+00_dp,&
      2.6424885821728354E+00_dp, -2.2914719419367309E+00_dp,&
      1.8263977484192504E+00_dp, -1.1042793052824569E+00_dp,&
      4.1984999040571597E-01_dp, -7.4204462154064413E-02_dp,&
      4.6836016437591510E+01_dp, -1.2461263924295558E+01_dp,&
      -9.5424753844301833E+00_dp,  6.9849024224173073E+00_dp,&
      -4.1532087460680200E+00_dp,  6.4834239581427680E+00_dp,&
      -7.9652050621281747E+00_dp,  7.2063417067380673E+00_dp,&
      -6.0811310415530810E+00_dp,  4.8089182798816328E+00_dp,&
      -2.9302138012528847E+00_dp,  1.1235538973792896E+00_dp,&
      -1.9930560719274457E-01_dp,  1.0501296069740016E+02_dp,&
      -3.5006611463522972E+01_dp, -2.5350753062858892E+01_dp,&
      2.1897185983038352E+01_dp, -1.2338035196247318E+01_dp,&
      1.7958095428676263E+01_dp, -2.2746802023037219E+01_dp,&
      2.0474886895879344E+01_dp, -1.6659351440344015E+01_dp,&
      1.2973198485786074E+01_dp, -7.9821240373924969E+00_dp,&
      3.1045561581164107E+00_dp, -5.5595266514842701E-01_dp,&
      2.5001854850064910E+02_dp, -1.0412293597281749E+02_dp,&
      -7.0012689271087496E+01_dp,  7.3141042075103357E+01_dp,&
      -4.0139216815255786E+01_dp,  5.2378449759973961E+01_dp,&
      -6.8602643632320721E+01_dp,  6.1798794377474557E+01_dp,&
      -4.7836268566179115E+01_dp,  3.6088120915352725E+01_dp,&
      -2.2418738636667779E+01_dp,  8.9199275986530679E+00_dp,&
      -1.6284440849442408E+00_dp,  6.4170847130306493E+02_dp,&
      -3.3564734161105105E+02_dp, -2.0319074233870467E+02_dp,&
      2.6700999172427709E+02_dp, -1.4865370577376467E+02_dp,&
      1.6483905993906203E+02_dp, -2.2243071554945487E+02_dp,&
      2.0268427674625798E+02_dp, -1.4634920891505303E+02_dp,&
      1.0281958316987021E+02_dp, -6.3797063574792020E+01_dp,&
      2.6253291146070694E+01_dp, -4.9653658173425370E+00_dp,&
      1.8179870531301624E+03_dp, -1.2124262579162116E+03_dp,&
      -6.2269753019103860E+02_dp,  1.1041853779458711E+03_dp,&
      -6.6269936764476824E+02_dp,  5.8460724515644620E+02_dp,&
      -7.9040872092006543E+02_dp,  7.4186477271070237E+02_dp,&
      -4.8589122354266311E+02_dp,  2.8409351239811036E+02_dp,&
      -1.6399352031256032E+02_dp,  7.0149332037368637E+01_dp,&
      -1.4138078938532878E+01_dp,  5.9147915732141646E+03_dp,&
      -5.1836184148231387E+03_dp, -1.9529616708993781E+03_dp,&
      5.4738511859880100E+03_dp, -3.8869620939089637E+03_dp,&
      2.5977148376502064E+03_dp, -3.1249029145093127E+03_dp,&
      3.0852665470111756E+03_dp, -1.7708800172300193E+03_dp,&
      5.3074618739108712E+02_dp, -6.6377545473118147E+01_dp,&
      9.3948507249713860E+00_dp, -4.8647165579413887E+00_dp,&
      2.3862435794753223E+04_dp, -2.9152813027206714E+04_dp,&
      -4.0489848903694369E+03_dp,  3.5895063967658702E+04_dp,&
      -3.5486880126624863E+04_dp,  2.0119320410757315E+04_dp,&
      -1.3494757960194409E+04_dp,  1.2166038486582942E+04_dp,&
      -5.1302990256297890E+03_dp, -4.1716588565092670E+03_dp,&
      7.0751423497119849E+03_dp, -4.0035381834602572E+03_dp,&
      8.8748758779155071E+02_dp,  1.4686983146318793E+05_dp,&
      -2.7739659047737991E+05_dp,  1.1760797338476461E+05_dp,&
      3.2807041818098148E+05_dp, -7.2517285349087697E+05_dp,&
      7.3150274091493548E+05_dp, -2.9882997403572657E+05_dp,&
      -3.1816573683119891E+05_dp,  7.7265175039547053E+05_dp,&
      -8.5110852845767897E+05_dp,  5.9525059737962019E+05_dp,&
      -2.5347666927566967E+05_dp,  5.0206766624547767E+04_dp,&
      2.9045854252487524E-01_dp, -9.3872029923743098E-09_dp,&
      -1.6474589315474152E-02_dp,  1.8803148185921940E-02_dp,&
      -2.1136377875972792E-02_dp,  1.0402051234732080E-02_dp,&
      2.7413508770381487E-02_dp, -1.0020420108756088E-01_dp,&
      1.9200020290840114E-01_dp, -2.5139939316453958E-01_dp,&
      2.2429428108518476E-01_dp, -1.2324671659253950E-01_dp,&
      3.1416116213041342E-02_dp,  9.8543060415902617E-01_dp,&
      -3.9997789833057460E-08_dp, -7.0180574115183969E-02_dp,&
      8.0100018518159663E-02_dp, -8.8669643824093369E-02_dp,&
      4.1168277654631887E-02_dp,  1.2222716004100626E-01_dp,&
      -4.3376557308075792E-01_dp,  8.2348524196548867E-01_dp,&
      -1.0722306945113518E+00_dp,  9.5266694074435665E-01_dp,&
      -5.2175051061240585E-01_dp,  1.3263698690739856E-01_dp,&
      2.0879815633551813E+00_dp, -1.1830858233257060E-07_dp,&
      -2.0751780374971132E-01_dp,  2.3684851551994904E-01_dp,&
      -2.5637862072869982E-01_dp,  1.0839683313493143E-01_dp,&
      3.8437385693103709E-01_dp, -1.3114065729390199E+00_dp,&
      2.4576389073052467E+00_dp, -3.1745152556387226E+00_dp,&
      2.8036316578514442E+00_dp, -1.5280660715176184E+00_dp,&
      3.8691002445303968E-01_dp,  4.1280139396328233E+00_dp,&
      -3.2843232523142131E-07_dp, -5.7587200156643770E-01_dp,&
      6.5726548338828250E-01_dp, -6.9376957208833223E-01_dp,&
      2.6019334435131042E-01_dp,  1.1358828382244106E+00_dp,&
      -3.7246497078122731E+00_dp,  6.8843611147032977E+00_dp,&
      -8.8148483202359351E+00_dp,  7.7329874627101871E+00_dp,&
      -4.1917406592895956E+00_dp,  1.0565313621508341E+00_dp,&
      8.2115292599550127E+00_dp, -8.9137561697151123E-07_dp,&
      -1.5622875225930539E+00_dp,  1.7830984621994226E+00_dp,&
      -1.8292461199409722E+00_dp,  5.8447218617253227E-01_dp,&
      3.2858522987931740E+00_dp, -1.0351403024506384E+01_dp,&
      1.8851002197235399E+01_dp, -2.3904379330116988E+01_dp,&
      2.0812769462943820E+01_dp, -1.1211467536930467E+01_dp,&
      2.8109944480167490E+00_dp,  1.6833509752553130E+01_dp,&
      -2.4208347916624588E-06_dp, -4.2408362534061510E+00_dp,&
      4.8402218801150756E+00_dp, -4.7995870275879025E+00_dp,&
      1.2056343958145297E+00_dp,  9.5508495555428166E+00_dp,&
      -2.8841634205337105E+01_dp,  5.1653143559226265E+01_dp,&
      -6.4765365903098953E+01_dp,  5.5884702982899860E+01_dp,&
      -2.9877273510186630E+01_dp,  7.4426681746137389E+00_dp,&
      3.5994294559688392E+01_dp, -6.7185205986334922E-06_dp,&
      -1.1762365188147919E+01_dp,  1.3424795806058500E+01_dp,&
      -1.2758760271073882E+01_dp,  2.0732889158938947E+00_dp,&
      2.8558432889956102E+01_dp, -8.2347703324680765E+01_dp,&
      1.4461633713546519E+02_dp, -1.7885516484206732E+02_dp,&
      1.5260695584062722E+02_dp, -8.0803840777326485E+01_dp,&
      1.9960600023375445E+01_dp,  8.1157457670966565E+01_dp,&
      -1.9458094008644022E-05_dp, -3.4039161550681698E+01_dp,&
      3.8849992038065054E+01_dp, -3.4927696505148425E+01_dp,&
      1.4181938934757687E+00_dp,  8.9934070855182398E+01_dp,&
      -2.4623095414911904E+02_dp,  4.2223594118137123E+02_dp,&
      -5.1315822696394446E+02_dp,  4.3143053246951132E+02_dp,&
      -2.2548223291042308E+02_dp,  5.5058148831168019E+01_dp,&
      1.9568047986849353E+02_dp, -6.0244262192986751E-05_dp,&
      -1.0527645970752303E+02_dp,  1.2015506150003682E+02_dp,&
      -1.0001639314265297E+02_dp, -1.4009458871747229E+01_dp,&
      3.0653332588013888E+02_dp, -7.9043502502542060E+02_dp,&
      1.3149170528475586E+03_dp, -1.5608831673196967E+03_dp,&
      1.2853217606932562E+03_dp, -6.5913017450105701E+02_dp,&
      1.5816639621647650E+02_dp,  5.1524126870993928E+02_dp,&
      -2.0601342380317144E-04_dp, -3.5945477251100289E+02_dp,&
      4.1025436478750186E+02_dp, -3.0394364960641099E+02_dp,&
      -1.3411341942097474E+02_dp,  1.1740847511426989E+03_dp,&
      -2.8153287548247690E+03_dp,  4.4935466649789141E+03_dp,&
      -5.1526402294564268E+03_dp,  4.1075902980563333E+03_dp,&
      -2.0415866612899454E+03_dp,  4.7535023510802057E+02_dp,&
      1.5345203816734511E+03_dp, -8.1996534428424410E-04_dp,&
      -1.4272114721930720E+03_dp,  1.6288998293799721E+03_dp,&
      -9.8331472929207700E+02_dp, -1.0461299234634332E+03_dp,&
      5.3699628756611428E+03_dp, -1.1700010870023980E+04_dp,&
      1.7522011300616890E+04_dp, -1.8931976535897018E+04_dp,&
      1.4192216319515732E+04_dp, -6.6082119103015066E+03_dp,&
      1.4354040244532143E+03_dp,  5.5359106840411023E+03_dp,&
      -4.2116907744044847E-03_dp, -7.2978223910327642E+03_dp,&
      8.3290217141779613E+03_dp, -3.0205311888974306E+03_dp,&
      -9.9636784453742785E+03_dp,  3.3034312432889732E+04_dp,&
      -6.1784943330522874E+04_dp,  8.1361262384593938E+04_dp,&
      -7.5874525508600782E+04_dp,  4.6962603202435283E+04_dp,&
      -1.6649687551097129E+04_dp,  2.3288381807198862E+03_dp,&
      2.8965940366189952E+04_dp, -3.7068444267143071E-02_dp,&
      -6.3565705168485889E+04_dp,  7.2545702838451150E+04_dp,&
      1.9451051149951527E+04_dp, -1.9181670386420426E+05_dp,&
      3.6675255147557601E+05_dp, -4.0868713263820746E+05_dp,&
      1.8884705302897276E+05_dp,  2.3077739263055730E+05_dp,&
      -5.1954761094616010E+05_dp,  4.2094053714259033E+05_dp,&
      -1.3519426853223852E+05_dp,  2.3956446261391355E+00_dp,&
      -8.2259922726024793E-08_dp, -1.4435805810148564E-01_dp,&
      1.6476195304653121E-01_dp, -1.8446144579171345E-01_dp,&
      8.9437025922576366E-02_dp,  2.4318075444129192E-01_dp,&
      -8.8181344063145117E-01_dp,  1.6854750516438730E+00_dp,&
      -2.2036434429093879E+00_dp,  1.9639043208955855E+00_dp,&
      -1.0782019877436793E+00_dp,  2.7464330062693504E-01_dp,&
      3.3494963759865057E+00_dp, -1.8445757660486408E-07_dp,&
      -3.2356867403047279E-01_dp,  3.6930217014151012E-01_dp,&
      -4.0166065484929164E-01_dp,  1.7339244143990060E-01_dp,&
      5.9190709926720098E-01_dp, -2.0357074336995966E+00_dp,&
      3.8253641076662022E+00_dp, -4.9496028636172511E+00_dp,&
      4.3769722375906133E+00_dp, -2.3880787149857583E+00_dp,&
      6.0519139225210616E-01_dp,  5.8040000291750982E+00_dp,&
      -4.9307010104776223E-07_dp, -8.6457441827499182E-01_dp,&
      9.8677305172000451E-01_dp, -1.0435592710758324E+00_dp,&
      3.9518781491624627E-01_dp,  1.6981221840783756E+00_dp,&
      -5.5841325524658263E+00_dp,  1.0332178192642571E+01_dp,&
      -1.3238703735787881E+01_dp,  1.1620225704849606E+01_dp,&
      -6.3016961957941975E+00_dp,  1.5889503305951129E+00_dp,&
      1.1205889768777084E+01_dp, -1.3209488146781724E-06_dp,&
      -2.3152576012558059E+00_dp,  2.6424922127194401E+00_dp,&
      -2.7155773855174825E+00_dp,  8.7695933535320614E-01_dp,&
      4.8533303358467910E+00_dp, -1.5324929900939482E+01_dp,&
      2.7934350894109556E+01_dp, -3.5445461425090642E+01_dp,&
      3.0877155066264496E+01_dp, -1.6640212268612309E+01_dp,&
      4.1736667265992375E+00_dp,  2.3003668057025042E+01_dp,&
      -3.5482604603519923E-06_dp, -6.2161341452072962E+00_dp,&
      7.0947026113909137E+00_dp, -7.0528063622407950E+00_dp,&
      1.8077921798260204E+00_dp,  1.3939173281458793E+01_dp,&
      -4.2219371371025446E+01_dp,  7.5708235249965000E+01_dp,&
      -9.5012975060705713E+01_dp,  8.2045747816253453E+01_dp,&
      -4.3891539618467377E+01_dp,  1.0939766218489742E+01_dp,&
      4.9637684295142599E+01_dp, -9.7463695738026281E-06_dp,&
      -1.7064535222921169E+01_dp,  1.9476350353952395E+01_dp,&
      -1.8590436540559089E+01_dp,  3.1925275522724674E+00_dp,&
      4.1158324783342913E+01_dp, -1.1921515605115852E+02_dp,&
      2.0979218330367425E+02_dp, -2.5985281813994447E+02_dp,&
      2.2199832068654112E+02_dp, -1.1767584219618503E+02_dp,&
      2.9097217542476354E+01_dp,  1.1295662396434658E+02_dp /)
    REAL(KIND=dp), DIMENSION(90), PARAMETER :: c05 = (/ &
      -2.7907357414271074E-05_dp, -4.8826106427338516E+01_dp,&
      5.5726828156877396E+01_dp, -5.0511477178292488E+01_dp,&
      2.9783692652581260E+00_dp,  1.2761594545085276E+02_dp,&
      -3.5194583503612256E+02_dp,  6.0567238480576850E+02_dp,&
      -7.3810346857458831E+02_dp,  6.2201716742048916E+02_dp,&
      -3.2577937709337687E+02_dp,  7.9700609240616870E+01_dp,&
      2.7392126417294304E+02_dp, -8.5014454364132483E-05_dp,&
      -1.4859738660861041E+02_dp,  1.6959859615136114E+02_dp,&
      -1.4348055742340679E+02_dp, -1.4470906151360252E+01_dp,&
      4.2504884354925576E+02_dp, -1.1092388852844888E+03_dp,&
      1.8571994555057452E+03_dp, -2.2161045374396167E+03_dp,&
      1.8334805028895823E+03_dp, -9.4436397774699310E+02_dp,&
      2.2753963301137293E+02_dp,  7.1993494851256742E+02_dp,&
      -2.8273561718275677E-04_dp, -4.9354947315699241E+02_dp,&
      5.6330058572048620E+02_dp, -4.3192291483542755E+02_dp,&
      -1.5060398376844597E+02_dp,  1.5657840098177583E+03_dp,&
      -3.8312638445092366E+03_dp,  6.1904109019097459E+03_dp,&
      -7.1744215164009311E+03_dp,  5.7783733007334085E+03_dp,&
      -2.9012807842372895E+03_dp,  6.8229724960201429E+02_dp,&
      2.1065420056611683E+03_dp, -1.0664058503153191E-03_dp,&
      -1.8579764758900715E+03_dp,  2.1205446683502560E+03_dp,&
      -1.3916730341031403E+03_dp, -1.1054018979421496E+03_dp,&
      6.6616451407058066E+03_dp, -1.5054577477071649E+04_dp,&
      2.3136946283622921E+04_dp, -2.5637844083293843E+04_dp,&
      1.9746988549792022E+04_dp, -9.4727339041338582E+03_dp,&
      2.1263470523015208E+03_dp,  7.1883726760512764E+03_dp,&
      -4.8683316646283012E-03_dp, -8.4557043884527229E+03_dp,&
      9.6505775179423381E+03_dp, -4.6968102107445948E+03_dp,&
      -8.7928604739436014E+03_dp,  3.5236438954359772E+04_dp,&
      -7.1416261140412083E+04_dp,  1.0113068401234987E+05_dp,&
      -1.0299681880727473E+05_dp,  7.2039640653129478E+04_dp,&
      -3.0819607047464066E+04_dp,  6.0214483913251506E+03_dp,&
      3.1445989346200007E+04_dp, -3.0789770328952418E-02_dp,&
      -5.3162976062730428E+04_dp,  6.0674445128441919E+04_dp,&
      -1.0676731288363026E+04_dp, -9.8618787001921461E+04_dp,&
      2.6917328710149962E+05_dp, -4.5080663875526667E+05_dp,&
      5.2615990446274763E+05_dp, -4.0824771802677517E+05_dp,&
      1.7414668875913633E+05_dp, -1.1753804603528921E+04_dp,&
      -1.4548822319029938E+04_dp,  2.3068275228418666E+05_dp,&
      -3.9689873696814088E-01_dp, -6.7753046300068102E+05_dp,&
      7.7323771401862660E+05_dp,  5.5823523395335092E+05_dp,&
      -2.8474678682125942E+06_dp,  4.1915016169727501E+06_dp,&
      -2.1435582467267131E+06_dp, -4.2583193954617819E+06_dp,&
      1.1939190162504816E+07_dp, -1.4588866431316955E+07_dp,&
      9.6445890214436613E+06_dp, -2.7844693650067835E+06_dp /)
    REAL(KIND=dp), DIMENSION(13, 26, 5), PARAMETER :: &
      coefdata = RESHAPE ((/ c01, c02, c03, c04, c05 /),(/13,26,5/))

    INTEGER                                  :: irange

     IF(Rc <=  1.0000000000000000E+01_dp) Rc =   1.0000000000000000E+01_dp
     IF(Rc >=  1.5429601669311523E+05_dp) Rc =   1.5429601669311523E+05_dp
     IF(Rc >=  1.0000000000000000E+01_dp .AND. Rc <=  6.6546340495335400E+01_dp) THEN
       irange =  1
       L_b =  1.0000000000000000E+01_dp
       U_b =  6.6546340495335400E+01_dp
     END IF
     IF(Rc >=  6.6546340495335400E+01_dp .AND. Rc <=  8.8617575097672625E+02_dp) THEN
       irange =  2
       L_b =  6.6546340495335400E+01_dp
       U_b =  8.8617575097672625E+02_dp
     END IF
     IF(Rc >=  8.8617575097672625E+02_dp .AND. Rc <=  8.9415021812443374E+03_dp) THEN
       irange =  3
       L_b =  8.8617575097672625E+02_dp
       U_b =  8.9415021812443374E+03_dp
     END IF
     IF(Rc >=  8.9415021812443374E+03_dp .AND. Rc <=  5.2961915553991363E+04_dp) THEN
       irange =  4
       L_b =  8.9415021812443374E+03_dp
       U_b =  5.2961915553991363E+04_dp
     END IF
     IF(Rc >=  5.2961915553991363E+04_dp .AND. Rc <=  1.5429601669311523E+05_dp) THEN
       irange =  5
       L_b =  5.2961915553991363E+04_dp
       U_b =  1.5429601669311523E+05_dp
     END IF

     fit_coef(1:13,1:26) = coefdata(1:13,1:26,irange)

   END SUBROUTINE get_fit_coef_k13

! *****************************************************************************
!> \brief ...
!> \param Rc ...
!> \param L_b ...
!> \param U_b ...
!> \param fit_coef ...
! *****************************************************************************
   SUBROUTINE get_fit_coef_k14(Rc,L_b,U_b,fit_coef)
    REAL(KIND=dp)                            :: Rc, L_b, U_b
    REAL(KIND=dp), DIMENSION(13, 28)         :: fit_coef

    REAL(KIND=dp), DIMENSION(220), PARAMETER :: c05 = (/ &
      -4.9474578762585689E-04_dp, -2.5939403957639643E+03_dp,&
      3.0352874416549189E+03_dp, -1.9742442806612846E+03_dp,&
      -1.5969761709828460E+03_dp,  9.5497831811419619E+03_dp,&
      -2.1791219784053799E+04_dp,  3.4007167063977518E+04_dp,&
      -3.8426789298450683E+04_dp,  3.0290020612758195E+04_dp,&
      -1.4913740382161952E+04_dp,  3.4437263625752439E+03_dp,&
      9.7404664944032957E+03_dp, -2.5387359568679966E-03_dp,&
      -1.3140569581716762E+04_dp,  1.5376194259263802E+04_dp,&
      -6.3931081330686866E+03_dp, -1.6591959902315488E+04_dp,&
      5.9198447699858742E+04_dp, -1.1572075729785630E+05_dp,&
      1.6031285635678476E+05_dp, -1.6009686057185219E+05_dp,&
      1.0949859530426569E+05_dp, -4.5488875426595805E+04_dp,&
      8.5191784067115168E+03_dp,  5.0965737544466407E+04_dp,&
      -2.2599730435201849E-02_dp, -1.1343132088409952E+05_dp,&
      1.3272584229156727E+05_dp,  2.6288738068185427E+04_dp,&
      -3.3499444278650463E+05_dp,  6.7051826395931852E+05_dp,&
      -8.0700177432248381E+05_dp,  5.0680536128272885E+05_dp,&
      1.7276049875705931E+05_dp, -7.0698870636937302E+05_dp,&
      6.3021843911543733E+05_dp, -2.1121368435571375E+05_dp,&
      2.3082323868282559E+00_dp, -2.4017040804702034E-08_dp,&
      -1.2891053118339380E-01_dp,  1.5084735253153100E-01_dp,&
      -1.7633642214925768E-01_dp,  1.0478107279403694E-01_dp,&
      1.8056901435863362E-01_dp, -7.6047719259316937E-01_dp,&
      1.5304155276743796E+00_dp, -2.0748704774691396E+00_dp,&
      1.9060146879751447E+00_dp, -1.0744161491583117E+00_dp,&
      2.8012092652976173E-01_dp,  3.1578187462227962E+00_dp,&
      -5.1567365353821572E-08_dp, -2.7650905240083334E-01_dp,&
      3.2356254508419818E-01_dp, -3.6928537030522152E-01_dp,&
      2.0370200953976045E-01_dp,  4.2500849576389860E-01_dp,&
      -1.6818132491347872E+00_dp,  3.3294488150428045E+00_dp,&
      -4.4727192170593986E+00_dp,  4.0824995398514528E+00_dp,&
      -2.2901310317535963E+00_dp,  5.9479992770208523E-01_dp,&
      5.3022457526794620E+00_dp, -1.3231464400206018E-07_dp,&
      -7.0879601306783835E-01_dp,  8.2941094524127557E-01_dp,&
      -9.2491385814921556E-01_dp,  4.7111974770571152E-01_dp,&
      1.1800589827897907E+00_dp, -4.4312033118085346E+00_dp,&
      8.6424235805572369E+00_dp, -1.1510883031319265E+01_dp,&
      1.0442741903695572E+01_dp, -5.8305757103901508E+00_dp,&
      1.5087006641831338E+00_dp,  9.8889975023413292E+00_dp,&
      -3.4049023321205348E-07_dp, -1.8221988606703043E+00_dp,&
      2.1322782325784657E+00_dp, -2.3235177347953093E+00_dp,&
      1.0834769924777288E+00_dp,  3.2578049089354004E+00_dp,&
      -1.1683733051537128E+01_dp,  2.2469887916547904E+01_dp,&
      -2.9680347503960153E+01_dp,  2.6764943785092932E+01_dp,&
      -1.4874114918252536E+01_dp,  3.8343556288753748E+00_dp,&
      1.9545871935955041E+01_dp, -8.7460941189600850E-07_dp,&
      -4.6754910887806371E+00_dp,  5.4711030020406684E+00_dp,&
      -5.8086891294068961E+00_dp,  2.4198279950235375E+00_dp,&
      8.9835555363766275E+00_dp, -3.0776859857929882E+01_dp,&
      5.8311567939739909E+01_dp, -7.6326069298093813E+01_dp,&
      6.8369012228472059E+01_dp, -3.7794300439853124E+01_dp,&
      9.7011963380606723E+00_dp,  4.0387743977509039E+01_dp,&
      -2.2802119015403580E-06_dp, -1.2173600984053655E+01_dp,&
      1.4245122575347299E+01_dp, -1.4663497964021953E+01_dp,&
      5.2167187058988729E+00_dp,  2.5244728902273341E+01_dp,&
      -8.2452555727692769E+01_dp,  1.5363036456312540E+02_dp,&
      -1.9899236224931769E+02_dp,  1.7684535970055489E+02_dp,&
      -9.7143346163942994E+01_dp,  2.4806140772967233E+01_dp,&
      8.7294816240247798E+01_dp, -6.1325232975282236E-06_dp,&
      -3.2687761451378606E+01_dp,  3.8250019320805777E+01_dp,&
      -3.7906149312736396E+01_dp,  1.0554564587180893E+01_dp,&
      7.3600534264141842E+01_dp, -2.2847415147597096E+02_dp,&
      4.1762130504843361E+02_dp, -5.3421933339350812E+02_dp,&
      4.7021823009778637E+02_dp, -2.5627821741621284E+02_dp,&
      6.5016683550652573E+01_dp,  1.9872757775908869E+02_dp,&
      -1.7288920250713497E-05_dp, -9.1967062673265403E+01_dp,&
      1.0761628002271890E+02_dp, -1.0162265582966705E+02_dp,&
      1.7864790639628893E+01_dp,  2.2661260001503362E+02_dp,&
      -6.6576659613427580E+02_dp,  1.1897709602645350E+03_dp,&
      -1.4987894458995706E+03_dp,  1.3032857841974862E+03_dp,&
      -7.0315224410172698E+02_dp,  1.7686291682282348E+02_dp,&
      4.8195697213806216E+02_dp, -5.2053770241544080E-05_dp,&
      -2.7616012508311786E+02_dp,  3.2315103970603087E+02_dp,&
      -2.8617592824287709E+02_dp,  8.9668308727996777E+00_dp,&
      7.5242665407679601E+02_dp, -2.0796726290417987E+03_dp,&
      3.6157344965745160E+03_dp, -4.4661943306924250E+03_dp,&
      3.8213658267393344E+03_dp, -2.0333395036853592E+03_dp,&
      5.0534042637321267E+02_dp,  1.2667258569739490E+03_dp,&
      -1.7160973677295130E-04_dp, -9.0710910652439338E+02_dp,&
      1.0614575664009355E+03_dp, -8.5818358894893083E+02_dp,&
      -1.6322151149271565E+02_dp,  2.7714382485293463E+03_dp,&
      -7.1430478120354646E+03_dp,  1.1989987328766261E+04_dp,&
      -1.4418456057909183E+04_dp,  1.2054815500043296E+04_dp,&
      -6.2834888773905532E+03_dp,  1.5330588576635123E+03_dp,&
      3.7064712285605369E+03_dp, -6.4340614075563080E-04_dp,&
      -3.3826536493755129E+03_dp,  3.9582066825917277E+03_dp,&
      -2.7744445291648535E+03_dp, -1.6114983670703609E+03_dp,&
      1.1820868123783597E+04_dp, -2.8002618049413450E+04_dp,&
      4.4775862324928894E+04_dp, -5.1714676070415371E+04_dp,&
      4.1654323461293316E+04_dp, -2.0958910595013618E+04_dp,&
      4.9457226648390688E+03_dp,  1.2647982284255992E+04_dp,&
      -2.9295350801209737E-03_dp, -1.5266725138724667E+04_dp,&
      1.7864187777650670E+04_dp, -9.5712259110815667E+03_dp,&
      -1.4224742236571192E+04_dp,  6.2861882562494437E+04_dp,&
      -1.3310141181211534E+05_dp,  1.9698892332270893E+05_dp,&
      -2.1146680880798530E+05_dp,  1.5786348899689523E+05_dp,&
      -7.3269193628805137E+04_dp,  1.5867217147397359E+04_dp,&
      5.5329397744117472E+04_dp, -1.8590496792721553E-02_dp,&
      -9.5252723032791575E+04_dp,  1.1145715883398660E+05_dp,&
      -2.5984748457593378E+04_dp, -1.6824051744760206E+05_dp,&
      4.8493312244212057E+05_dp, -8.5016802056288661E+05_dp,&
      1.0575469301659723E+06_dp, -9.1592531311393599E+05_dp,&
      5.0087306406000553E+05_dp, -1.3516194494539662E+05_dp,&
      4.9738319216673990E+03_dp,  4.0588762038136885E+05_dp,&
      -2.4363178639415667E-01_dp, -1.2048752475226286E+06_dp,&
      1.4098050382019044E+06_dp,  9.0383108002498816E+05_dp,&
      -5.0245780711403918E+06_dp,  7.7751430360333202E+06_dp,&
      -4.9183263330898508E+06_dp, -5.5122178990547452E+06_dp,&
      1.8712359903900910E+07_dp, -2.3939217503688838E+07_dp,&
      1.6218284186391968E+07_dp, -4.7629949218999417E+06_dp /)
    REAL(KIND=dp), DIMENSION(400), PARAMETER :: c01 = (/ &
      1.8855631887642382E-01_dp, -8.1829231046632769E-02_dp,&
      -1.8950903102790816E-02_dp,  1.1349898287774197E-02_dp,&
      5.7338935886024562E-03_dp, -3.6475885523936957E-03_dp,&
      -2.1458998142384776E-03_dp,  1.4787405505064602E-03_dp,&
      1.8498537181843471E-03_dp, -1.9614757255708645E-03_dp,&
      2.8699603397392510E-04_dp,  3.2492719703003728E-04_dp,&
      -1.2121707854025808E-04_dp,  5.9493156461017915E-01_dp,&
      -2.8489994642562694E-01_dp, -4.7541679614047191E-02_dp,&
      4.4726439188938094E-02_dp,  1.3790272162181918E-02_dp,&
      -1.5382737388282837E-02_dp, -4.7841715747880403E-03_dp,&
      6.2784351364303555E-03_dp,  5.7333760277949548E-03_dp,&
      -9.0331645457294250E-03_dp,  3.7807714485503436E-03_dp,&
      -1.8396222276796279E-04_dp, -1.8364694814447738E-04_dp,&
      1.0949959297761693E+00_dp, -6.2009062084330724E-01_dp,&
      -3.7748088011084188E-02_dp,  1.1122644744095318E-01_dp,&
      6.8260044178406754E-03_dp, -4.0370249812718174E-02_dp,&
      -5.5457356283545060E-05_dp,  1.6531192945984253E-02_dp,&
      8.5778848944558107E-03_dp, -2.5250508617166625E-02_dp,&
      1.7117124110630336E-02_dp, -5.0352617710504076E-03_dp,&
      5.0598347337982492E-04_dp,  1.7718966086100012E+00_dp,&
      -1.2253060640836630E+00_dp,  7.5765729412832300E-02_dp,&
      2.3434768509302889E-01_dp, -4.2525503260917592E-02_dp,&
      -8.5582364672439362E-02_dp,  2.5281751460349305E-02_dp,&
      3.4833638640193537E-02_dp,  3.6751693692180167E-03_dp,&
      -5.3823028208479444E-02_dp,  4.8983441632312753E-02_dp,&
      -1.9467338617706484E-02_dp,  3.0785669504690746E-03_dp,&
      2.7440213704191687E+00_dp, -2.3281411168383572E+00_dp,&
      4.3930618759633422E-01_dp,  4.2911680926061485E-01_dp,&
      -1.9302208134528706E-01_dp, -1.4788624688089952E-01_dp,&
      9.7831155740447415E-02_dp,  5.8724172544457748E-02_dp,&
      -2.7041547750709143E-02_dp, -8.6324492364363792E-02_dp,&
      1.0519866830647166E-01_dp, -4.9884486746561728E-02_dp,&
      9.2457653338246029E-03_dp,  4.1894298957234364E+00_dp,&
      -4.3191941088244246E+00_dp,  1.3686987339076067E+00_dp,&
      6.6657149222134049E-01_dp, -5.5741457317177090E-01_dp,&
      -1.9026437846512242E-01_dp,  2.6250861035098944E-01_dp,&
      6.9486254930157559E-02_dp, -1.1978905880115075E-01_dp,&
      -8.3501865329870154E-02_dp,  1.7398289631638658E-01_dp,&
      -9.8507600097967690E-02_dp,  2.0521730335798096E-02_dp,&
      6.3847973817046322E+00_dp, -7.8795893221752813E+00_dp,&
      3.5278548969550236E+00_dp,  7.7209394256709085E-01_dp,&
      -1.3260774785983964E+00_dp, -8.5419091308552686E-02_dp,&
      5.7755377384646511E-01_dp,  7.2297671316667676E-03_dp,&
      -3.2877429737301961E-01_dp,  5.4329500825053369E-02_dp,&
      1.9667270764503703E-01_dp, -1.5053976717237291E-01_dp,&
      3.5936788548594702E-02_dp,  9.7702622697290806E+00_dp,&
      -1.4206713723167212E+01_dp,  8.2873928530423839E+00_dp,&
      1.5985839391104073E-01_dp, -2.7552157435202256E+00_dp,&
      5.0282122616039127E-01_dp,  1.0695678398638087E+00_dp,&
      -2.7464467867173409E-01_dp, -6.9906236411372280E-01_dp,&
      5.2101436020119296E-01_dp,  1.9094129828846389E-02_dp,&
      -1.4958848604568709E-01_dp,  4.7222211859139149E-02_dp,&
      1.5062823541404761E+01_dp, -2.5419370366400564E+01_dp,&
      1.8439365337386121E+01_dp, -2.8259558244586622E+00_dp,&
      -4.9818385838230759E+00_dp,  2.3299538569069234E+00_dp,&
      1.5633478274790813E+00_dp, -1.0657994340246808E+00_dp,&
      -1.1646925905079819E+00_dp,  1.6169430708139660E+00_dp,&
      -6.8586900071703705E-01_dp,  5.1235251652361868E-02_dp,&
      2.8103584030951066E-02_dp,  2.3472242707636227E+01_dp,&
      -4.5307745602761770E+01_dp,  3.9608173260058393E+01_dp,&
      -1.2469876908780039E+01_dp, -7.2257373755414482E+00_dp,&
      6.7879665547641341E+00_dp,  1.1935112761475244E+00_dp,&
      -2.7643113712900522E+00_dp, -1.2334897284577950E+00_dp,&
      3.5447845919979937E+00_dp, -2.4336600487586937E+00_dp,&
      7.5137492324226218E-01_dp, -8.4754014224136878E-02_dp,&
      3.7181849843463610E+01_dp, -8.0854637759410053E+01_dp,&
      8.3057755018582043E+01_dp, -3.9158151519184301E+01_dp,&
      -5.2153320438563107E+00_dp,  1.5431055894370806E+01_dp,&
      -2.5633281395845020E+00_dp, -5.2800516403359623E+00_dp,&
      6.0861233401307058E-01_dp,  5.3386911924762455E+00_dp,&
      -5.2478418325908383E+00_dp,  2.1981670777697651E+00_dp,&
      -3.6896267338134209E-01_dp,  6.0727490472197132E+01_dp,&
      -1.4613546055640202E+02_dp,  1.7197354948450220E+02_dp,&
      -1.0678616520983826E+02_dp,  1.5368848940430027E+01_dp,&
      2.6491023605600720E+01_dp, -1.4185436270935341E+01_dp,&
      -6.0093236121070364E+00_dp,  6.0468948387099335E+00_dp,&
      3.8095072307193587E+00_dp, -7.1410151818281697E+00_dp,&
      3.7963272372520303E+00_dp, -7.5598613223750699E-01_dp,&
      1.0661517386729834E+02_dp, -2.7755039079968356E+02_dp,&
      3.6291502644599831E+02_dp, -2.7257937328142549E+02_dp,&
      9.2825581734245375E+01_dp,  2.6166156567735605E+01_dp,&
      -3.5739847667562742E+01_dp,  5.4601725492393982E-01_dp,&
      1.3032015168131986E+01_dp, -2.3134927090253932E+00_dp,&
      -6.3108826059815888E+00_dp,  4.6857674694757945E+00_dp,&
      -1.0832084080117372E+00_dp,  2.3570689411128097E+02_dp,&
      -6.4519456443598233E+02_dp,  9.0223673164973013E+02_dp,&
      -7.5922137375758791E+02_dp,  3.5264115885243655E+02_dp,&
      -2.0989321935745537E+01_dp, -6.8957744635106934E+01_dp,&
      1.6380082435687662E+01_dp,  2.3421535674813764E+01_dp,&
      -1.2505473268777655E+01_dp, -5.5686200874137679E+00_dp,&
      6.7565951949370930E+00_dp, -1.7824763136779194E+00_dp,&
      1.5275739094339187E+00_dp, -6.8001171975001351E-01_dp,&
      -1.4605823697436165E-01_dp,  9.7828030861734694E-02_dp,&
      4.3904288873512894E-02_dp, -3.2132952455475414E-02_dp,&
      -1.6222343050214385E-02_dp,  1.3057021240371581E-02_dp,&
      1.5021168482517486E-02_dp, -1.7795844885780759E-02_dp,&
      4.1587460671975805E-03_dp,  1.8886977368616257E-03_dp,&
      -8.6439907575529956E-04_dp,  1.7655488232124410E+00_dp,&
      -1.0043589160047419E+00_dp, -6.4881261437890497E-02_dp,&
      1.8408970278911196E-01_dp,  1.3505968144741386E-02_dp,&
      -6.7697647980935388E-02_dp, -1.4104212623691880E-03_dp,&
      2.7789428772147087E-02_dp,  1.5476046749690131E-02_dp,&
      -4.2895663223273178E-02_dp,  2.8323690685459327E-02_dp,&
      -7.9885591764726257E-03_dp,  7.2098905617051475E-04_dp,&
      2.2893354930889771E+00_dp, -1.7662816033235220E+00_dp,&
      1.8403075477202577E-01_dp,  3.6407316931374950E-01_dp,&
      -8.6454388639239418E-02_dp, -1.3732188470605050E-01_dp,&
      4.7671855898007404E-02_dp,  5.6196647324577562E-02_dp,&
      3.2014114180754462E-03_dp, -8.9528747999384858E-02_dp,&
      8.4420352577280325E-02_dp, -3.4350841194717596E-02_dp,&
      5.5441745297109164E-03_dp,  3.2032258575320007E+00_dp,&
      -3.2225125249560986E+00_dp,  8.1611931560454820E-01_dp,&
      6.3294842865066814E-01_dp, -3.4599308009457763E-01_dp,&
      -2.2371900126276956E-01_dp,  1.7236277609166772E-01_dp,&
      8.9265078243513069E-02_dp, -5.3947604792708909E-02_dp,&
      -1.3470741896044980E-01_dp,  1.7532582289902740E-01_dp,&
      -8.5701035571192671E-02_dp,  1.6222152583645552E-02_dp,&
      4.6899825303755200E+00_dp, -5.8524582698866139E+00_dp,&
      2.2916446118015137E+00_dp,  9.1184997824365766E-01_dp,&
      -9.3131462349314253E-01_dp, -2.5350817564214317E-01_dp,&
      4.3807172987756571E-01_dp,  9.0958165596269375E-02_dp,&
      -2.1576258659606037E-01_dp, -1.0117137327241729E-01_dp,&
      2.6846770077261650E-01_dp, -1.5994395563622057E-01_dp,&
      3.4270164544904627E-02_dp,  7.0536044987990474E+00_dp,&
      -1.0515715134514387E+01_dp,  5.5637593381103434E+00_dp,&
      8.8711884475899694E-01_dp, -2.1104915247624749E+00_dp,&
      -8.1670293239061807E-03_dp,  9.2046944679099196E-01_dp,&
      -4.1741761795501105E-02_dp, -5.6379831367556721E-01_dp,&
      1.8783938079412943E-01_dp,  2.4581866594692456E-01_dp,&
      -2.2071302879217192E-01_dp,  5.5588648839480607E-02_dp,&
      1.0795630733012548E+01_dp, -1.8740084666512331E+01_dp,&
      1.2578600627233692E+01_dp, -3.9401650322504667E-01_dp,&
      -4.2184539233060248E+00_dp,  1.0568277055345616E+00_dp,&
      1.6271512205476346E+00_dp, -5.4967161451363589E-01_dp,&
      -1.1438065064071192E+00_dp,  1.0422345804455202E+00_dp,&
      -1.6188730221930633E-01_dp, -1.6095431410441777E-01_dp,&
      6.2846743147671746E-02_dp,  1.6752881329272565E+01_dp,&
      -3.3242655345938644E+01_dp,  2.7270848317691222E+01_dp,&
      -5.5130334892731048E+00_dp, -7.3490722258440666E+00_dp,&
      4.1336683392056157E+00_dp,  2.2217585152496331E+00_dp,&
      -1.8973233340123217E+00_dp, -1.7925847714925607E+00_dp,&
      2.8980257671054690E+00_dp, -1.4704996700404205E+00_dp,&
      2.6049665793720700E-01_dp,  1.1444579602089748E-02_dp,&
      2.6366237165338223E+01_dp, -5.8894321733298867E+01_dp,&
      5.7521256562790946E+01_dp, -2.1015040091428240E+01_dp,&
      -1.0093110698669442E+01_dp,  1.1421770115513882E+01_dp,&
      1.2209454420475951E+00_dp, -4.7246394893237085E+00_dp,&
      -1.6540481482401830E+00_dp,  6.0698584679024128E+00_dp,&
      -4.5445980010104039E+00_dp,  1.5511339489005815E+00_dp,&
      -2.0629950073069225E-01_dp,  4.2291510559584431E+01_dp,&
      -1.0465649315579324E+02_dp,  1.1897915051644942E+02_dp,&
      -6.2607403829996528E+01_dp, -5.4209821188017733E+00_dp,&
      2.5469691021577223E+01_dp, -5.7934227183011000E+00_dp,&
      -8.9450737399818969E+00_dp,  2.1130287274913053E+00_dp,&
      8.8579986644448212E+00_dp, -9.6059615158346361E+00_dp,&
      4.2847833469932413E+00_dp, -7.6136171546806874E-01_dp,&
      7.0150982322458802E+01_dp, -1.8840659621861244E+02_dp,&
      2.4308030630616531E+02_dp, -1.6583105497089588E+02_dp,&
      3.0198027338055482E+01_dp,  4.3137539621256487E+01_dp,&
      -2.7521349718582130E+01_dp, -9.6205094694534150E+00_dp,&
      1.4155430281644003E+01_dp,  3.0610045263527681E+00_dp,&
      -1.1486453193732402E+01_dp,  6.8958779080000516E+00_dp,&
      -1.4735848497004862E+00_dp,  1.2521662423711092E+02_dp,&
      -3.5446949046616129E+02_dp,  5.0095178511895006E+02_dp,&
      -4.0924920293285766E+02_dp,  1.5804385724333696E+02_dp,&
      3.6770269869154461E+01_dp, -6.7438382515530122E+01_dp,&
      8.2106438781909876E+00_dp,  2.7715716832749418E+01_dp,&
      -1.6544637350100938E+01_dp, -1.8960171584647745E+00_dp,&
      5.0230826620958835E+00_dp, -1.4542149169175325E+00_dp,&
      2.6891905074142079E+02_dp, -7.7262572953372262E+02_dp,&
      1.1373217134902056E+03_dp, -1.0196332565900419E+03_dp,&
      5.2341144727679796E+02_dp, -6.1641665846344743E+01_dp,&
      -1.0165109376939833E+02_dp,  4.4685439920384923E+01_dp,&
      2.7038021992864703E+01_dp, -3.0106447966381346E+01_dp,&
      6.1843452503431493E+00_dp,  3.2384773408937977E+00_dp,&
      -1.3899494804286554E+00_dp,  1.0072868276053762E+03_dp,&
      -2.8456064913973623E+03_dp,  4.1444213836381341E+03_dp,&
      -3.7170886189465332E+03_dp,  1.9829818649289750E+03_dp,&
      -3.8268355348089460E+02_dp, -2.1461243783889245E+02_dp,&
      9.3530786144130829E+01_dp,  8.1716992654821155E+01_dp,&
      -6.6241416941144067E+01_dp, -4.3199384438482218E+00_dp,&
      1.9601634716394887E+01_dp, -5.9258959456424520E+00_dp,&
      2.3570619090301786E-01_dp, -5.5162312535829566E-02_dp,&
      -1.8870279804443474E-02_dp,  5.9631585077347333E-03_dp,&
      5.2228509160063231E-03_dp, -1.8582311804501090E-03_dp,&
      -2.0959249000252859E-03_dp,  6.5294058434205886E-04_dp,&
      1.3620688255760083E-03_dp, -8.3422748679588279E-04_dp,&
      -3.2153474478749297E-04_dp,  4.5243977001023549E-04_dp,&
      -1.2591995144422757E-04_dp,  7.6587911600589598E-01_dp,&
      -2.0875636096653921E-01_dp, -6.0166290603338807E-02_dp,&
      2.9038121823338276E-02_dp,  1.7462634992676026E-02_dp,&
      -9.9487645929256206E-03_dp, -7.1366951595613149E-03_dp,&
      3.6713354494958815E-03_dp,  5.4941565312529340E-03_dp,&
      -5.1227538205351423E-03_dp,  3.8905702168758006E-04_dp,&
      1.1382083100254657E-03_dp, -3.9779179490065093E-04_dp,&
      1.4924251913695632E+00_dp, -5.1831019334185813E-01_dp,&
      -1.0671352469554986E-01_dp,  9.3839903685068102E-02_dp,&
      3.2490232771881414E-02_dp, -3.4916436688308940E-02_dp,&
      -1.3379509563995902E-02_dp,  1.3447654845333435E-02_dp,&
      1.4076996424835847E-02_dp, -2.0087908870094155E-02_dp /)
    REAL(KIND=dp), DIMENSION(400), PARAMETER :: c02 = (/ &
      7.7725039723149673E-03_dp,  1.1145706792479703E-04_dp,&
      -5.6097890673260616E-04_dp,  2.6197138414537928E+00_dp,&
      -1.1878855591113402E+00_dp, -1.3669541461047707E-01_dp,&
      2.5880924857422194E-01_dp,  3.9410663226141771E-02_dp,&
      -1.0161291074659663E-01_dp, -1.4845376770825033E-02_dp,&
      4.0384191854304169E-02_dp,  2.9596626868951859E-02_dp,&
      -6.2931154480019291E-02_dp,  3.6745945037802222E-02_dp,&
      -8.3488837850756864E-03_dp,  2.5001869374288916E-04_dp,&
      4.4903769459069487E+00_dp, -2.6294155124375926E+00_dp,&
      -5.5431338024852818E-02_dp,  6.4004292000876550E-01_dp,&
      -1.1281945374730250E-02_dp, -2.5862693537498738E-01_dp,&
      1.5347628611049238E-02_dp,  1.0516496345674861E-01_dp,&
      4.7284909960751409E-02_dp, -1.6707094690313704E-01_dp,&
      1.2446700525343438E-01_dp, -4.1183388221565025E-02_dp,&
      5.0811070975616355E-03_dp,  7.7146703444230118E+00_dp,&
      -5.7066349635112275E+00_dp,  4.4359998413126639E-01_dp,&
      1.4564973028002151E+00_dp, -2.8264653691673164E-01_dp,&
      -5.8828533359250146E-01_dp,  1.6490349909564089E-01_dp,&
      2.4225156854407701E-01_dp,  2.9313258361025520E-02_dp,&
      -3.8043499940822334E-01_dp,  3.4922268468144424E-01_dp,&
      -1.4038340165679025E-01_dp,  2.2435164708763913E-02_dp,&
      1.3420634891703846E+01_dp, -1.2271053658859397E+01_dp,&
      2.2633203019324784E+00_dp,  3.0726848709689221E+00_dp,&
      -1.2487174695539813E+00_dp, -1.1832394997237425E+00_dp,&
      6.8072898930880155E-01_dp,  4.8388636378359617E-01_dp,&
      -1.7386926345701451E-01_dp, -7.0971295084118979E-01_dp,&
      8.3444731110830883E-01_dp, -3.9085840738204530E-01_dp,&
      7.1877924239686128E-02_dp,  2.3755596487151276E+01_dp,&
      -2.6373483711130035E+01_dp,  7.9674708316710250E+00_dp,&
      5.9098701221696377E+00_dp, -4.1849668364839099E+00_dp,&
      -1.9539106200179810E+00_dp,  2.1803405618804597E+00_dp,&
      7.4953893987111575E-01_dp, -1.0194767520566530E+00_dp,&
      -8.6460471090210544E-01_dp,  1.6344868078214525E+00_dp,&
      -9.0779850081241142E-01_dp,  1.8748507335135670E-01_dp,&
      4.2925038468224415E+01_dp, -5.7082416348496004E+01_dp,&
      2.4765245687853245E+01_dp,  9.6276167314465653E+00_dp,&
      -1.2283278522629377E+01_dp, -1.7758222239526547E+00_dp,&
      5.9861494789457375E+00_dp,  3.8161879101773316E-01_dp,&
      -3.6179634751249212E+00_dp,  5.5154525020794043E-01_dp,&
      2.1657632444483794E+00_dp, -1.6489718555108390E+00_dp,&
      3.9384533483919071E-01_dp,  7.9426971888974009E+01_dp,&
      -1.2525063212940651E+02_dp,  7.3064335204686586E+01_dp,&
      9.0748492057192287E+00_dp, -3.2688968221322831E+01_dp,&
      4.6389115697221399E+00_dp,  1.4061626099831328E+01_dp,&
      -3.4152596807585240E+00_dp, -9.7690378370180984E+00_dp,&
      7.9275061732557406E+00_dp, -6.3075146750735867E-01_dp,&
      -1.6211192224607696E+00_dp,  5.6714810234651192E-01_dp,&
      1.5111743347765383E+02_dp, -2.8035613867011256E+02_dp,&
      2.1163825913793113E+02_dp, -2.3170413390173934E+01_dp,&
      -7.7181490178355602E+01_dp,  3.6759610462492383E+01_dp,&
      2.4870389441934346E+01_dp, -1.9605319761356419E+01_dp,&
      -1.8579389164179108E+01_dp,  3.0778110404354127E+01_dp,&
      -1.6525494041464249E+01_dp,  3.4913991145457492E+00_dp,&
      -6.0347075936734364E-02_dp,  2.9794125234183332E+02_dp,&
      -6.4468281291214532E+02_dp,  6.1350680831270904E+02_dp,&
      -2.0829770735425490E+02_dp, -1.3833043913329655E+02_dp,&
      1.5148578230935769E+02_dp,  6.9315274799328979E+00_dp,&
      -6.5837068538278658E+01_dp, -5.7004307370424572E+00_dp,&
      7.0003047590575193E+01_dp, -6.1017865054790505E+01_dp,&
      2.3916329575898800E+01_dp, -3.7939335527980216E+00_dp,&
      6.2392037586903575E+02_dp, -1.5490091248195286E+03_dp,&
      1.8055806125944553E+03_dp, -1.0360644128662241E+03_dp,&
      -2.3754789025977079E+00_dp,  4.1933356341119969E+02_dp,&
      -1.8722583764205237E+02_dp, -1.0263434903918866E+02_dp,&
      1.0213234231038781E+02_dp,  3.9318968261747074E+01_dp,&
      -9.2918679848214254E+01_dp,  5.1419134926361373E+01_dp,&
      -1.0473717025421935E+01_dp,  1.5736554804419109E+03_dp,&
      -4.3215073024710609E+03_dp,  5.8432235337189013E+03_dp,&
      -4.5330421089344291E+03_dp,  1.6066632187764733E+03_dp,&
      4.6048780056295902E+02_dp, -7.0252498313679098E+02_dp,&
      9.1567126340641437E+01_dp,  2.1597646777874064E+02_dp,&
      -8.2131833864215508E+01_dp, -7.0306756274872612E+01_dp,&
      6.5598722538748618E+01_dp, -1.6248587274560606E+01_dp,&
      1.9235349518444129E+00_dp, -4.6850080403507538E-01_dp,&
      -1.5364029163795673E-01_dp,  5.4610434870493545E-02_dp,&
      4.3121661743123693E-02_dp, -1.7541190516424592E-02_dp,&
      -1.7406011867825987E-02_dp,  6.2616285228102664E-03_dp,&
      1.1811592459976310E-02_dp, -8.2397883673727759E-03_dp,&
      -1.8295153305730790E-03_dp,  3.5354495356089955E-03_dp,&
      -1.0301987802026438E-03_dp,  2.4066428993785296E+00_dp,&
      -8.3136637617833942E-01_dp, -1.7886500853362544E-01_dp,&
      1.4914120143861584E-01_dp,  5.5966341016553810E-02_dp,&
      -5.4968206983759794E-02_dp, -2.3425654868914157E-02_dp,&
      2.1010142579782284E-02_dp,  2.3289369427945999E-02_dp,&
      -3.1028342173622683E-02_dp,  1.0636269323796589E-02_dp,&
      1.1399336207735340E-03_dp, -1.0908676737398028E-03_dp,&
      3.5374997704069169E+00_dp, -1.7739971394119649E+00_dp,&
      -1.8178559222630922E-01_dp,  4.0632103157798233E-01_dp,&
      5.8516869919659278E-02_dp, -1.6056878832594537E-01_dp,&
      -2.3367297155693821E-02_dp,  6.3796767939619234E-02_dp,&
      4.7371253056304186E-02_dp, -9.9633558625783894E-02_dp,&
      5.7560660830321013E-02_dp, -1.2736642321518233E-02_dp,&
      2.7575782625087675E-04_dp,  5.7011521784732837E+00_dp,&
      -3.8423532249372361E+00_dp, -2.9642882719094453E-03_dp,&
      9.8774471298748978E-01_dp, -3.1786519630271917E-02_dp,&
      -4.0451564192974432E-01_dp,  2.7198223261336543E-02_dp,&
      1.6513101391234630E-01_dp,  7.5034065435769917E-02_dp,&
      -2.6516920000527039E-01_dp,  1.9761798053561705E-01_dp,&
      -6.5322652133406461E-02_dp,  8.0342411952895529E-03_dp,&
      9.6554344109573194E+00_dp, -8.2331879085041422E+00_dp,&
      8.3524862618218354E-01_dp,  2.2044272451218019E+00_dp,&
      -4.7078752828290560E-01_dp, -9.0706474881686883E-01_dp,&
      2.6681861233539966E-01_dp,  3.7661042003703288E-01_dp,&
      4.5852032020570602E-02_dp, -6.0242502481174587E-01_dp,&
      5.5531680066652089E-01_dp, -2.2393716090138521E-01_dp,&
      3.5887703311978866E-02_dp,  1.6859940902920791E+01_dp,&
      -1.7545862869698755E+01_dp,  3.6977782998502717E+00_dp,&
      4.5720615862129801E+00_dp, -1.9866126947412956E+00_dp,&
      -1.8011363549668156E+00_dp,  1.0806874902511809E+00_dp,&
      7.4694472223237718E-01_dp, -2.7704282161362154E-01_dp,&
      -1.1227694450891366E+00_dp,  1.3298672608779971E+00_dp,&
      -6.2600213805058025E-01_dp,  1.1557688164935315E-01_dp,&
      3.0155576587510065E+01_dp, -3.7501345657849221E+01_dp,&
      1.2398647864361692E+01_dp,  8.6755274737656904E+00_dp,&
      -6.5226597957606725E+00_dp, -2.9431221713683908E+00_dp,&
      3.4329970746320631E+00_dp,  1.1571252075338336E+00_dp,&
      -1.6281640072535604E+00_dp, -1.3658547679386668E+00_dp,&
      2.6187047264334584E+00_dp, -1.4634970806763015E+00_dp,&
      3.0358383342450590E-01_dp,  5.5192028159048192E+01_dp,&
      -8.0962165488777188E+01_dp,  3.7555961619090525E+01_dp,&
      1.3994299732088162E+01_dp, -1.8953701082797508E+01_dp,&
      -2.6423170124495257E+00_dp,  9.4281011864102400E+00_dp,&
      6.1853673917080609E-01_dp, -5.8552996587823642E+00_dp,&
      9.3429973997614690E-01_dp,  3.5009224511745121E+00_dp,&
      -2.6876808075590444E+00_dp,  6.4489189999489860E-01_dp,&
      1.0352661090439724E+02_dp, -1.7762218682439524E+02_dp,&
      1.0901655066345789E+02_dp,  1.3191430681842892E+01_dp,&
      -5.0434357437755253E+01_dp,  7.0189212422942697E+00_dp,&
      2.2479507564050024E+01_dp, -5.1663403209676666E+00_dp,&
      -1.6257258538476439E+01_dp,  1.3112384987334350E+01_dp,&
      -1.0057705657907436E+00_dp, -2.7227979411177912E+00_dp,&
      9.5157750952935138E-01_dp,  1.9966810298056458E+02_dp,&
      -3.9817155755323489E+02_dp,  3.1231785694487940E+02_dp,&
      -3.2246955189265840E+01_dp, -1.2137675854903618E+02_dp,&
      5.5648852060851773E+01_dp,  4.2338485599882645E+01_dp,&
      -3.0713631015622173E+01_dp, -3.3372363691858347E+01_dp,&
      5.2473159250251847E+01_dp, -2.7472517355576315E+01_dp,&
      5.4871815877910279E+00_dp,  5.4181088396383879E-03_dp,&
      3.9842112960611036E+02_dp, -9.1713248145130524E+02_dp,&
      8.9864978724752018E+02_dp, -2.9197223519853361E+02_dp,&
      -2.3777047005535999E+02_dp,  2.3747379514120729E+02_dp,&
      2.9189361759954174E+01_dp, -1.1235024332118923E+02_dp,&
      -2.5228006012287803E+01_dp,  1.3666058782503137E+02_dp,&
      -1.1319713359079486E+02_dp,  4.3059250023675311E+01_dp,&
      -6.6306785236143213E+00_dp,  8.3643909862800683E+02_dp,&
      -2.1911581438517564E+03_dp,  2.6233201738911630E+03_dp,&
      -1.4629002578603033E+03_dp, -1.5531076779586360E+02_dp,&
      7.5096190656559827E+02_dp, -2.6513199393656015E+02_dp,&
      -2.5306291751220505E+02_dp,  1.9176224755216879E+02_dp,&
      1.2171911094119822E+02_dp, -2.2483087333153333E+02_dp,&
      1.2085756124968174E+02_dp, -2.4483946936633131E+01_dp,&
      1.9747033956733776E+03_dp, -5.6690082004591668E+03_dp,&
      7.9229880703795752E+03_dp, -6.2070030014723598E+03_dp,&
      1.9283153836933900E+03_dp,  1.2213601402606237E+03_dp,&
      -1.4629461716889127E+03_dp,  1.7493772651497704E+02_dp,&
      6.0147021238394541E+02_dp, -4.2899362538784328E+02_dp,&
      5.2240406552156081E+01_dp,  5.8761089300646205E+01_dp,&
      -2.1062064230578645E+01_dp,  7.3293182665232616E+03_dp,&
      -2.1390250361943301E+04_dp,  3.1430512423742246E+04_dp,&
      -2.8054361932601565E+04_dp,  1.4411560239732573E+04_dp,&
      -1.9245318108120027E+03_dp, -2.4770186693515961E+03_dp,&
      1.1408945117448648E+03_dp,  5.1709534814360416E+02_dp,&
      -5.0521280755054079E+02_dp, -3.5875569643025585E+01_dp,&
      1.5800852296117296E+02_dp, -4.8036967247801527E+01_dp,&
      2.6675516114666215E-01_dp, -3.0945355721284026E-02_dp,&
      -1.7441440218854541E-02_dp,  2.3047365097519259E-03_dp,&
      4.6492436549931005E-03_dp, -1.2182731057025141E-04_dp,&
      -2.6753899877316541E-03_dp,  4.3133453711237618E-04_dp,&
      1.4760115578547502E-03_dp, -6.7275877348429937E-04_dp,&
      -5.0478172177037875E-04_dp,  5.2672478258151327E-04_dp,&
      -1.3681815219244913E-04_dp,  8.8716914786510592E-01_dp,&
      -1.2489263551188355E-01_dp, -6.6159449712017482E-02_dp,&
      1.3827016752023739E-02_dp,  1.9071901611964450E-02_dp,&
      -2.2900215117290321E-03_dp, -1.1259614209975406E-02_dp,&
      2.5040307358945432E-03_dp,  6.8228796479948492E-03_dp,&
      -4.2395551594276134E-03_dp, -1.1853357048091078E-03_dp,&
      1.9375771856216684E-03_dp, -5.4683602988019086E-04_dp,&
      1.8085261758917643E+00_dp, -3.4145203794204038E-01_dp,&
      -1.6386900603394911E-01_dp,  5.5237575273534616E-02_dp,&
      5.2221001537078073E-02_dp, -1.3801174494884877E-02_dp,&
      -3.2078301452457908E-02_dp,  1.0176963366296529E-02_dp,&
      2.2098874328449269E-02_dp, -1.8474914130739314E-02_dp,&
      1.0331691102100885E-03_dp,  4.1466751154130713E-03_dp,&
      -1.4017603905798956E-03_dp,  3.3844323807837364E+00_dp,&
      -8.6969086014254704E-01_dp, -3.7007448537416970E-01_dp,&
      1.8591067907939515E-01_dp,  1.2836223540354583E-01_dp,&
      -5.7108370411460219E-02_dp, -8.2699128673198688E-02_dp,&
      3.6214620670927643E-02_dp,  6.5374140319992780E-02_dp,&
      -6.9244074088779542E-02_dp,  1.8299747317206676E-02_dp,&
      5.5068417777118968E-03_dp, -2.9577800214382753E-03_dp,&
      6.2831509132867023E+00_dp, -2.1503252769443533E+00_dp,&
      -7.9051209248628784E-01_dp,  5.6778852490396248E-01_dp,&
      2.8990581299559376E-01_dp, -2.0042137877984503E-01_dp,&
      -1.9784689657716242E-01_dp,  1.1942781843739504E-01_dp,&
      1.8202124035828973E-01_dp, -2.3777194345333863E-01_dp,&
      9.8820369139971026E-02_dp, -6.2744671316908003E-03_dp,&
      -4.3699523495120468E-03_dp,  1.1856221271082283E+01_dp,&
      -5.2568207235502564E+00_dp, -1.5969164750173954E+00_dp,&
      1.6461069205315015E+00_dp,  5.9172061051969216E-01_dp,&
      -6.4697064642905655E-01_dp, -4.3622701630183242E-01_dp /)
    REAL(KIND=dp), DIMENSION(400), PARAMETER :: c03 = (/ &
      3.7852818456494469E-01_dp,  4.8067555347683755E-01_dp,&
      -7.7452581263261322E-01_dp,  4.1934407249887440E-01_dp,&
      -8.9234614648328783E-02_dp,  1.6775465007990663E-03_dp,&
      2.2972132228999541E+01_dp, -1.2898901071815617E+01_dp,&
      -2.9797860863789967E+00_dp,  4.6560435611069622E+00_dp,&
      1.0045232845484455E+00_dp, -1.9936051302786315E+00_dp,&
      -8.4531105176808208E-01_dp,  1.1798428768823785E+00_dp,&
      1.1850316641391561E+00_dp, -2.4272082281943557E+00_dp,&
      1.5965055566291400E+00_dp, -4.8094513367183356E-01_dp,&
      5.2209596934856856E-02_dp,  4.6003576082453456E+01_dp,&
      -3.2201217855327890E+01_dp, -4.7001617664051283E+00_dp,&
      1.3071605527953821E+01_dp,  8.5579097973198270E-01_dp,&
      -5.9644456694242445E+00_dp, -1.1661056085226225E+00_dp,&
      3.6676542174044551E+00_dp,  2.5328675562596046E+00_dp,&
      -7.2813422948153237E+00_dp,  5.6732651577216711E+00_dp,&
      -2.0778493148878012E+00_dp,  3.0597521984172754E-01_dp,&
      9.5841344366560733E+01_dp, -8.2867209315180261E+01_dp,&
      -3.6667980705527965E+00_dp,  3.6810280324243195E+01_dp,&
      -3.9027065352258030E+00_dp, -1.7352727135169747E+01_dp,&
      7.6980699945374509E-01_dp,  1.1432996249675226E+01_dp,&
      3.2078234224178548E+00_dp, -2.0167035286623779E+01_dp,&
      1.8815816418699637E+01_dp, -7.9856869761396307E+00_dp,&
      1.3743706676833676E+00_dp,  2.0942078257013398E+02_dp,&
      -2.2297406228411018E+02_dp,  1.7976991795066482E+01_dp,&
      1.0433146940440160E+02_dp, -3.3593272459503659E+01_dp,&
      -4.7934874758223437E+01_dp,  1.6761961834599632E+01_dp,&
      3.5307462464268248E+01_dp, -1.0676959688175113E+01_dp,&
      -4.5186467607051981E+01_dp,  5.5390800171275316E+01_dp,&
      -2.7302072934110832E+01_dp,  5.2930427116892105E+00_dp,&
      4.8534437062529378E+02_dp, -6.3827294390772897E+02_dp,&
      1.5269862076475343E+02_dp,  2.9397876898906549E+02_dp,&
      -1.8666196685194382E+02_dp, -1.1267861965482709E+02_dp,&
      1.0302221723226668E+02_dp,  1.0064780441992042E+02_dp,&
      -1.2667959907993337E+02_dp, -2.7955679739330272E+01_dp,&
      1.1636709243491977E+02_dp, -7.4278132336508548E+01_dp,&
      1.6627724231379677E+01_dp,  1.2129457544590350E+03_dp,&
      -1.9898851218084496E+03_dp,  8.9667454269579969E+02_dp,&
      7.6499660811561989E+02_dp, -9.4460273295149761E+02_dp,&
      -7.9982084258437382E+01_dp,  4.9143681651282088E+02_dp,&
      1.7577870976013608E+02_dp, -7.5560526889570701E+02_dp,&
      5.5420839419383049E+02_dp, -9.7827251223882456E+01_dp,&
      -6.5994619153302679E+01_dp,  2.7322580954253830E+01_dp,&
      3.3629380940608125E+03_dp, -6.9947471832857618E+03_dp,&
      5.2569129046974813E+03_dp,  9.5662372221786302E+02_dp,&
      -4.5223771656660383E+03_dp,  2.1388578525409239E+03_dp,&
      1.4458258137230293E+03_dp, -9.7995910573338404E+02_dp,&
      -2.3056338385778722E+03_dp,  3.9539197752574801E+03_dp,&
      -2.7936521089069415E+03_dp,  1.0067700692768599E+03_dp,&
      -1.5230449843591444E+02_dp,  1.1287607903620121E+04_dp,&
      -2.9770904965426005E+04_dp,  3.5105294651488788E+04_dp,&
      -1.6371982820338246E+04_dp, -1.0740533254974671E+04_dp,&
      2.2130378005569939E+04_dp, -1.3681292248100397E+04_dp,&
      1.5045907807150209E+03_dp,  1.5506170412673375E+03_dp,&
      1.7390235047408335E+03_dp, -3.3865390974169732E+03_dp,&
      1.9425096877525270E+03_dp, -4.1245955612424916E+02_dp,&
      2.1894397151835716E+00_dp, -2.6731395928049057E-01_dp,&
      -1.4827254994752054E-01_dp,  2.2501949118577419E-02_dp,&
      4.0392878417795181E-02_dp, -2.0509313301437277E-03_dp,&
      -2.3398171933515566E-02_dp,  4.1436012276036211E-03_dp,&
      1.3239233793160993E-02_dp, -6.6345121073634033E-03_dp,&
      -3.9199986704907238E-03_dp,  4.4620529455242711E-03_dp,&
      -1.1821222579064985E-03_dp,  2.9104358928220684E+00_dp,&
      -5.4034884374736947E-01_dp, -2.6382498632517509E-01_dp,&
      8.3423398368213414E-02_dp,  8.3577127933356615E-02_dp,&
      -1.9596957492619492E-02_dp, -5.0894988316946159E-02_dp,&
      1.5007727408384405E-02_dp,  3.4099429011554168E-02_dp,&
      -2.6790339407196072E-02_dp, -2.0861762641897246E-04_dp,&
      7.2003288329720860E-03_dp, -2.3042530203007294E-03_dp,&
      4.6854870375971425E+00_dp, -1.3091559374612594E+00_dp,&
      -5.5604618168351427E-01_dp,  2.8363733949175418E-01_dp,&
      1.9722299942319932E-01_dp, -8.5724412786704784E-02_dp,&
      -1.2678366277947672E-01_dp,  5.3898967563011380E-02_dp,&
      9.9143592343204467E-02_dp, -1.0221615198869149E-01_dp,&
      2.4645321523710888E-02_dp,  9.8027723202895409E-03_dp,&
      -4.7528229423811326E-03_dp,  8.3449364444476224E+00_dp,&
      -3.1882784788266245E+00_dp, -1.1623777798125512E+00_dp,&
      8.5932749444592937E-01_dp,  4.4006378157100107E-01_dp,&
      -3.0113783137236694E-01_dp, -3.0064333658838277E-01_dp,&
      1.7727771396624017E-01_dp,  2.7500235701742781E-01_dp,&
      -3.5164491355954486E-01_dp,  1.4063352553582015E-01_dp,&
      -5.3061541860887768E-03_dp, -7.3941722723575680E-03_dp,&
      1.5695844879515739E+01_dp, -7.7144784516164409E+00_dp,&
      -2.3165992414383219E+00_dp,  2.4635839948403095E+00_dp,&
      8.9286928883939432E-01_dp, -9.6611614969126347E-01_dp,&
      -6.6064108196517513E-01_dp,  5.5828529367439539E-01_dp,&
      7.2643541444293003E-01_dp, -1.1445679759723104E+00_dp,&
      6.0431143104611718E-01_dp, -1.2059693868497956E-01_dp,&
      -2.0295090300914784E-04_dp,  3.0683428112515067E+01_dp,&
      -1.8785984761642442E+01_dp, -4.2812807144192888E+00_dp,&
      6.9041136192452699E+00_dp,  1.5281967131175347E+00_dp,&
      -2.9618491125087343E+00_dp, -1.2909060055596731E+00_dp,&
      1.7294637699195108E+00_dp,  1.8115339805586135E+00_dp,&
      -3.5961083965134120E+00_dp,  2.3152629313913131E+00_dp,&
      -6.7559378765190137E-01_dp,  6.8467347739389728E-02_dp,&
      6.2200394666586682E+01_dp, -4.6645513761643670E+01_dp,&
      -6.7361062081138403E+00_dp,  1.9252946239137508E+01_dp,&
      1.4565082813699259E+00_dp, -8.8439939458906807E+00_dp,&
      -1.8841850547213637E+00_dp,  5.3507933397910890E+00_dp,&
      4.0391961741194118E+00_dp, -1.0903704455343615E+01_dp,&
      8.2986239071848669E+00_dp, -2.9674077434627351E+00_dp,&
      4.2355807369087978E-01_dp,  1.3120724805702122E+02_dp,&
      -1.1950869933893051E+02_dp, -5.5268211617968701E+00_dp,&
      5.3988318262052609E+01_dp, -4.7298665550070194E+00_dp,&
      -2.5840083994726847E+01_dp,  3.9210953422711342E-01_dp,&
      1.6659556581442590E+01_dp,  6.3342926153832790E+00_dp,&
      -3.1128246651007331E+01_dp,  2.8060315969863534E+01_dp,&
      -1.1623356406032910E+01_dp,  1.9545704467217686E+00_dp,&
      2.8992281668366485E+02_dp, -3.1986021667598641E+02_dp,&
      2.2957602830758823E+01_dp,  1.5295309506645341E+02_dp,&
      -4.3977099689006337E+01_dp, -7.2842752827744860E+01_dp,&
      2.1240405440259895E+01_dp,  5.1988276730246653E+01_dp,&
      -7.3415208227066628E+00_dp, -7.6698168738036316E+01_dp,&
      8.6864530178114521E+01_dp, -4.1342020475172049E+01_dp,&
      7.8179704154115273E+00_dp,  6.7758098148149588E+02_dp,&
      -9.0749625734977383E+02_dp,  1.9757172893678359E+02_dp,&
      4.3536816979902625E+02_dp, -2.4496931051749999E+02_dp,&
      -1.8568872790586860E+02_dp,  1.3676710116513863E+02_dp,&
      1.5660775089407684E+02_dp, -1.4802736727848816E+02_dp,&
      -1.0503595627665368E+02_dp,  2.1660252926971174E+02_dp,&
      -1.2550574533876951E+02_dp,  2.6842986324742590E+01_dp,&
      1.6983835710324086E+03_dp, -2.7822602350002267E+03_dp,&
      1.1271088712112896E+03_dp,  1.1986681939191794E+03_dp,&
      -1.2298857533356895E+03_dp, -3.0022794237993980E+02_dp,&
      6.8655519186623917E+02_dp,  3.7926097327365630E+02_dp,&
      -1.0131416464518528E+03_dp,  4.8887586578955757E+02_dp,&
      1.5852175267429436E+02_dp, -2.3155394891253087E+02_dp,&
      6.4867684171672508E+01_dp,  4.6647494493474896E+03_dp,&
      -9.4718275026936517E+03_dp,  6.2415276573638348E+03_dp,&
      2.5681659744319936E+03_dp, -6.0603657311991774E+03_dp,&
      1.3404500115938827E+03_dp,  2.8367905176183167E+03_dp,&
      -2.1312330539639728E+02_dp, -4.7786812976805413E+03_dp,&
      5.7342345176005756E+03_dp, -3.1010309026004620E+03_dp,&
      7.8357948046534830E+02_dp, -5.9923253166677775E+01_dp,&
      1.4640183187865061E+04_dp, -3.7306916780068022E+04_dp,&
      3.8220031520362521E+04_dp, -6.2809044505597049E+03_dp,&
      -2.6739415575057752E+04_dp,  2.5342729378908178E+04_dp,&
      1.2027700096180724E+03_dp, -1.4004701367972091E+04_dp,&
      -8.0202934875566336E+02_dp,  1.9450058535585955E+04_dp,&
      -2.0396420577318539E+04_dp,  9.6146216278380598E+03_dp,&
      -1.8456278447511768E+03_dp,  6.3728309588229160E+04_dp,&
      -1.9308392701878195E+05_dp,  2.8218302673358534E+05_dp,&
      -2.2238214531146569E+05_dp,  3.7703671533692228E+04_dp,&
      1.3438185722279068E+05_dp, -1.8992995882626902E+05_dp,&
      1.4401513945138350E+05_dp, -7.4358417823577242E+04_dp,&
      2.9628900862353326E+04_dp, -1.0445794075970865E+04_dp,&
      3.1174329654997350E+03_dp, -5.1375865413784766E+02_dp,&
      2.7795875304093992E-01_dp, -1.1508375559190450E-02_dp,&
      -1.0528648097849034E-02_dp,  3.5980305121916505E-03_dp,&
      -3.8046195921370984E-03_dp,  6.2928606186302520E-03_dp,&
      -7.1251215695943898E-03_dp,  6.9881088552248760E-03_dp,&
      -6.6162502135251478E-03_dp,  5.4184133074171737E-03_dp,&
      -3.2882049375220451E-03_dp,  1.2671473115865579E-03_dp,&
      -2.3060149967202290E-04_dp,  9.3295242665284350E-01_dp,&
      -4.7623070133923896E-02_dp, -4.2945598361340688E-02_dp,&
      1.6016174188036112E-02_dp, -1.5648429729476797E-02_dp,&
      2.6076911362094328E-02_dp, -2.9735563415651905E-02_dp,&
      2.8978847027043658E-02_dp, -2.7276350421693667E-02_dp,&
      2.2330036824631784E-02_dp, -1.3565168762016968E-02_dp,&
      5.2270999974457248E-03_dp, -9.4990442008638116E-04_dp,&
      1.9360104180064521E+00_dp, -1.3504439794671419E-01_dp,&
      -1.1921435653077275E-01_dp,  5.0013006045341514E-02_dp,&
      -4.4100911881113043E-02_dp,  7.4033972248669144E-02_dp,&
      -8.5347616016655026E-02_dp,  8.2444773869143534E-02_dp,&
      -7.6903614035383622E-02_dp,  6.2915522354710013E-02_dp,&
      -3.8288814369473227E-02_dp,  1.4757779755536996E-02_dp,&
      -2.6771107475512521E-03_dp,  3.7157437793497445E+00_dp,&
      -3.5796615175869162E-01_dp, -3.0857012350133511E-01_dp,&
      1.4568419777186900E-01_dp, -1.1664877750278947E-01_dp,&
      1.9623630214417229E-01_dp, -2.2916236385065755E-01_dp,&
      2.1940133178072269E-01_dp, -2.0246833223513266E-01_dp,&
      1.6546652950802851E-01_dp, -1.0094501676761625E-01_dp,&
      3.8944876117642600E-02_dp, -7.0548301311543953E-03_dp,&
      7.1202810265841023E+00_dp, -9.2376909827681186E-01_dp,&
      -7.7556752166862253E-01_dp,  4.1180402375069397E-01_dp,&
      -3.0219866001661555E-01_dp,  5.0555399088003872E-01_dp,&
      -5.9930306647530585E-01_dp,  5.6887825139019754E-01_dp,&
      -5.1830619426463687E-01_dp,  4.2285671923122187E-01_dp,&
      -2.5880981419593402E-01_dp,  1.0005085991364741E-01_dp,&
      -1.8112309200688195E-02_dp,  1.3952971079099326E+01_dp,&
      -2.3685732416421441E+00_dp, -1.9287282183697398E+00_dp,&
      1.1570319194902128E+00_dp, -7.8432675091254189E-01_dp,&
      1.2914143383187207E+00_dp, -1.5583873119306928E+00_dp,&
      1.4673017772412877E+00_dp, -1.3158217342584138E+00_dp,&
      1.0703445295466383E+00_dp, -6.5795894538224242E-01_dp,&
      2.5527153608136027E-01_dp, -4.6237736535300604E-02_dp,&
      2.8263649428018645E+01_dp, -6.1403574477253944E+00_dp,&
      -4.8190110843037894E+00_dp,  3.2973992582013865E+00_dp,&
      -2.0826346006536562E+00_dp,  3.3265966189184617E+00_dp,&
      -4.1006380665334703E+00_dp,  3.8330107038269401E+00_dp,&
      -3.3679881939021974E+00_dp,  2.7252672290525357E+00_dp,&
      -1.6848847973698691E+00_dp,  6.5763593614104732E-01_dp,&
      -1.1942745604194921E-01_dp,  5.9663070196049929E+01_dp,&
      -1.6360639408342024E+01_dp, -1.2260457072232811E+01_dp,&
      9.7081093752801308E+00_dp, -5.7831301102621087E+00_dp,&
      8.7770738779350062E+00_dp, -1.1095172356986756E+01_dp,&
      1.0314862293028543E+01_dp, -8.8233143705921826E+00_dp,&
      7.0699089843034164E+00_dp, -4.4024286308950149E+00_dp,&
      1.7350313146222107E+00_dp, -3.1699935067362639E-01_dp,&
      1.3246229877703036E+02_dp, -4.5573195710967674E+01_dp,&
      -3.2154929851246941E+01_dp,  3.0089567929013004E+01_dp /)
    REAL(KIND=dp), DIMENSION(400), PARAMETER :: c04 = (/ &
      -1.7228109793468004E+01_dp,  2.4108285826088757E+01_dp,&
      -3.1362122687167119E+01_dp,  2.9105748579791722E+01_dp,&
      -2.4011018345543842E+01_dp,  1.8874442598138845E+01_dp,&
      -1.1844908362658735E+01_dp,  4.7389221341553531E+00_dp,&
      -8.7620232886140437E-01_dp,  3.1319177678658934E+02_dp,&
      -1.3543802765633262E+02_dp, -8.7944249327211026E+01_dp,&
      1.0040343220265778E+02_dp, -5.6926025700847731E+01_dp,&
      7.0347720403350522E+01_dp, -9.4219186285633356E+01_dp,&
      8.7905354648438234E+01_dp, -6.8986573372461578E+01_dp,&
      5.2100412701990848E+01_dp, -3.2821779741733202E+01_dp,&
      1.3425508412863300E+01_dp, -2.5361926958927188E+00_dp,&
      8.0339006237151955E+02_dp, -4.4154448996371457E+02_dp,&
      -2.5318298497317585E+02_dp,  3.7186998732251453E+02_dp,&
      -2.1890233755863807E+02_dp,  2.2518951062497459E+02_dp,&
      -3.0685526869647856E+02_dp,  2.9150513350211389E+02_dp,&
      -2.1341406154345805E+02_dp,  1.4697365335701707E+02_dp,&
      -9.0644866260771636E+01_dp,  3.8104543183927014E+01_dp,&
      -7.4637665568833587E+00_dp,  2.3044953848651026E+03_dp,&
      -1.6499343359986115E+03_dp, -7.6329960501783944E+02_dp,&
      1.6025382549709414E+03_dp, -1.0575686504725684E+03_dp,&
      8.4751945418489061E+02_dp, -1.1070023776663299E+03_dp,&
      1.0916471478962596E+03_dp, -7.2797923049358553E+02_dp,&
      3.9110381271759127E+02_dp, -2.0138082123320197E+02_dp,&
      8.3652333810026974E+01_dp, -1.7347611535672375E+01_dp,&
      7.8198644325855139E+03_dp, -7.6691862797499716E+03_dp,&
      -2.1277219734434225E+03_dp,  8.7238270472921049E+03_dp,&
      -7.3961781231503164E+03_dp,  4.6349606671658858E+03_dp,&
      -4.4852525866010510E+03_dp,  4.4961185730836178E+03_dp,&
      -2.6534205822788208E+03_dp,  4.4195703445871567E+02_dp,&
      4.6808102343591264E+02_dp, -3.3365812265666904E+02_dp,&
      7.3857858542740814E+01_dp,  3.6001520527708155E+04_dp,&
      -5.3761146837784952E+04_dp,  7.3437178260156779E+03_dp,&
      6.7383560520903891E+04_dp, -1.0382851510328006E+05_dp,&
      8.2809008442700942E+04_dp, -3.3648889547776744E+04_dp,&
      -1.2152726558027885E+04_dp,  4.1767740945577811E+04_dp,&
      -5.0768326126419706E+04_dp,  3.9062230702036293E+04_dp,&
      -1.8084327244239081E+04_dp,  3.8220474286796107E+03_dp,&
      2.2865375061568898E+00_dp, -1.0007118233392230E-01_dp,&
      -9.1205849541503861E-02_dp,  3.1914700695052964E-02_dp,&
      -3.3024390409020138E-02_dp,  5.4742737190489880E-02_dp,&
      -6.2096136424361001E-02_dp,  6.0798325126475448E-02_dp,&
      -5.7475761170505700E-02_dp,  4.7065964441797066E-02_dp,&
      -2.8569616417685219E-02_dp,  1.1009159982093808E-02_dp,&
      -2.0027154539687037E-03_dp,  3.1115361446896435E+00_dp,&
      -2.1232738914527213E-01_dp, -1.8819487078015418E-01_dp,&
      7.7315071595482399E-02_dp, -6.9322784492525144E-02_dp,&
      1.1642512083091060E-01_dp, -1.3389325658449983E-01_dp,&
      1.2953231476076033E-01_dp, -1.2106215221167207E-01_dp,&
      9.9062132162017258E-02_dp, -6.0257874244903968E-02_dp,&
      2.3219784186475492E-02_dp, -4.2129071313934735E-03_dp,&
      5.1842137469273535E+00_dp, -5.3869903618846926E-01_dp,&
      -4.6467926474567495E-01_dp,  2.1884730673172473E-01_dp,&
      -1.7511335735521882E-01_dp,  2.9552329416544632E-01_dp,&
      -3.4478150622705367E-01_dp,  3.3007122270082284E-01_dp,&
      -3.0482944942907819E-01_dp,  2.4917344203144623E-01_dp,&
      -1.5196363061763693E-01_dp,  5.8604719193897523E-02_dp,&
      -1.0613235696316487E-02_dp,  9.5870634342206369E+00_dp,&
      -1.3711377799260795E+00_dp, -1.1512181070647907E+00_dp,&
      6.1170380517292722E-01_dp, -4.4707416132200822E-01_dp,&
      7.5102145460616387E-01_dp, -8.8972944046919122E-01_dp,&
      8.4414679164287010E-01_dp, -7.6958368953859069E-01_dp,&
      6.2806586528884911E-01_dp, -3.8429278159433666E-01_dp,&
      1.4848195960403690E-01_dp, -2.6866781887765284E-02_dp,&
      1.8775367525208154E+01_dp, -3.4798037363209451E+00_dp,&
      -2.8341983855308737E+00_dp,  1.7006763707506694E+00_dp,&
      -1.1475201905709496E+00_dp,  1.8990746846216189E+00_dp,&
      -2.2902563189929959E+00_dp,  2.1549148240268319E+00_dp,&
      -1.9340794515934272E+00_dp,  1.5741729298799292E+00_dp,&
      -9.6735481018495850E-01_dp,  3.7504025715761041E-01_dp,&
      -6.7883194547012568E-02_dp,  3.8394133622760471E+01_dp,&
      -8.9466981562030412E+00_dp, -7.0282409736060609E+00_dp,&
      4.7994825716760108E+00_dp, -3.0169750276153215E+00_dp,&
      4.8524390956684931E+00_dp, -5.9767609648935736E+00_dp,&
      5.5816268619870462E+00_dp, -4.9114774016645590E+00_dp,&
      3.9786679592871335E+00_dp, -2.4588990184794475E+00_dp,&
      9.5872721501601932E-01_dp, -1.7391335067678992E-01_dp,&
      8.1981262849015678E+01_dp, -2.3666893439637072E+01_dp,&
      -1.7784927738480093E+01_dp,  1.3988139300590353E+01_dp,&
      -8.2896531742092350E+00_dp,  1.2715968834351342E+01_dp,&
      -1.6055214717081412E+01_dp,  1.4906635835795530E+01_dp,&
      -1.2785323476847376E+01_dp,  1.0268923034784310E+01_dp,&
      -6.3923625597427707E+00_dp,  2.5150132422925426E+00_dp,&
      -4.5864970193217036E-01_dp,  1.8391160612598173E+02_dp,&
      -6.5390879598833081E+01_dp, -4.6443209608932030E+01_dp,&
      4.2801184197871855E+01_dp, -2.4328927514598988E+01_dp,&
      3.4666363971514954E+01_dp, -4.5025337204534765E+01_dp,&
      4.1693953701520407E+01_dp, -3.4572251788912048E+01_dp,&
      2.7325062230277453E+01_dp, -1.7152184655455500E+01_dp,&
      6.8433372267470105E+00_dp, -1.2610819268429596E+00_dp,&
      4.3796390510262017E+02_dp, -1.9195269795546488E+02_dp,&
      -1.2647230203731048E+02_dp,  1.4008459142409632E+02_dp,&
      -7.8348031326383392E+01_dp,  9.9970385018688177E+01_dp,&
      -1.3379095649086614E+02_dp,  1.2428806571516949E+02_dp,&
      -9.8477370673399108E+01_dp,  7.5392430125667914E+01_dp,&
      -4.7653420100430495E+01_dp,  1.9413552472266861E+01_dp,&
      -3.6453602426561473E+00_dp,  1.1243161739960412E+03_dp,&
      -6.1212969783621759E+02_dp, -3.6255926320818696E+02_dp,&
      5.0245861498975745E+02_dp, -2.8720395504014408E+02_dp,&
      3.1238265825952902E+02_dp, -4.2845042492346460E+02_dp,&
      4.0342871100091708E+02_dp, -3.0042323465953200E+02_dp,&
      2.1481280954412966E+02_dp, -1.3503140509729658E+02_dp,&
      5.6622775693708434E+01_dp, -1.0979355574831448E+01_dp,&
      3.1860045248229740E+03_dp, -2.1911750278244835E+03_dp,&
      -1.0979433579892768E+03_dp,  2.0455378472242323E+03_dp,&
      -1.2650045628271755E+03_dp,  1.1053428711376532E+03_dp,&
      -1.5077317283874904E+03_dp,  1.4637188657276913E+03_dp,&
      -1.0022416048330521E+03_dp,  6.0744063403024404E+02_dp,&
      -3.5491809474315596E+02_dp,  1.5302472467201832E+02_dp,&
      -3.1334099788707611E+01_dp,  1.0369219521864628E+04_dp,&
      -9.2957237022933496E+03_dp, -3.3928819922045304E+03_dp,&
      9.9967321591592136E+03_dp, -7.3074927639255448E+03_dp,&
      4.9261330266433706E+03_dp, -5.9355709483907294E+03_dp,&
      6.0546709760503109E+03_dp, -3.7342004116968578E+03_dp,&
      1.3546695864229043E+03_dp, -3.3203558186954626E+02_dp,&
      9.8087982450693005E+01_dp, -2.5243614322976278E+01_dp,&
      4.1857193674808586E+04_dp, -5.1932163797323534E+04_dp,&
      -6.6662468997263541E+03_dp,  6.4686288551278376E+04_dp,&
      -6.5456790256758810E+04_dp,  3.8148473534729375E+04_dp,&
      -2.6150038718665659E+04_dp,  2.4469246590260118E+04_dp,&
      -1.2503913145751001E+04_dp, -4.7022652434212587E+03_dp,&
      1.1183043091421301E+04_dp, -6.6810238092235504E+03_dp,&
      1.5108300867808391E+03_dp,  2.5791029976014077E+05_dp,&
      -4.9175046576096409E+05_dp,  2.1233667264455085E+05_dp,&
      5.8463884123176278E+05_dp, -1.3091052712645272E+06_dp,&
      1.3418160613037723E+06_dp, -5.8138326599118719E+05_dp,&
      -5.2549948487002426E+05_dp,  1.3542841741758550E+06_dp,&
      -1.5110583922361752E+06_dp,  1.0622068635754019E+06_dp,&
      -4.5309312113443937E+05_dp,  8.9720576460898301E+04_dp,&
      2.8045183132902857E-01_dp, -2.7574727763525926E-09_dp,&
      -1.4802663783945194E-02_dp,  1.7321648653547309E-02_dp,&
      -2.0314993734315081E-02_dp,  1.2188204632943614E-02_dp,&
      2.0453784311654996E-02_dp, -8.6946069733443987E-02_dp,&
      1.7538269591846428E-01_dp, -2.3808144401169168E-01_dp,&
      2.1890014297902782E-01_dp, -1.2347609817248510E-01_dp,&
      3.2209458319119928E-02_dp,  9.4329847413827783E-01_dp,&
      -1.1481913919029099E-08_dp, -6.1604975697409345E-02_dp,&
      7.2088324040036539E-02_dp, -8.3499507949812257E-02_dp,&
      4.8263387714568257E-02_dp,  8.9536268205010608E-02_dp,&
      -3.6778510358608535E-01_dp,  7.3540775557696592E-01_dp,&
      -9.9349209869932542E-01_dp,  9.1038678244132876E-01_dp,&
      -5.1222364328553327E-01_dp,  1.3335054733354942E-01_dp,&
      1.9654700761868769E+00_dp, -3.2854337036704686E-08_dp,&
      -1.7614103276566062E-01_dp,  2.0611488927720040E-01_dp,&
      -2.3440872269300381E-01_dp,  1.2780362813152599E-01_dp,&
      2.7418366288979301E-01_dp, -1.0758527122495292E+00_dp,&
      2.1248452756492062E+00_dp, -2.8506473734734499E+00_dp,&
      2.5994713960584761E+00_dp, -1.4571442401661396E+00_dp,&
      3.7823535850863244E-01_dp,  3.7941873168540208E+00_dp,&
      -8.7953097131811444E-08_dp, -4.7113570657516318E-01_dp,&
      5.5130825970707775E-01_dp, -6.1431256183219685E-01_dp,&
      3.1203070731979304E-01_dp,  7.8613980333408151E-01_dp,&
      -2.9472754643680608E+00_dp,  5.7453376661729898E+00_dp,&
      -7.6498729270533463E+00_dp,  6.9384231145105142E+00_dp,&
      -3.8732848550438543E+00_dp,  1.0020928498724406E+00_dp,&
      7.3237070985773096E+00_dp, -2.2941985723774246E-07_dp,&
      -1.2277519929151568E+00_dp,  1.4366756798913567E+00_dp,&
      -1.5650422648304791E+00_dp,  7.2887187219335803E-01_dp,&
      2.1962398474654630E+00_dp, -7.8721698163374203E+00_dp,&
      1.5136322777799633E+01_dp, -1.9990503168612264E+01_dp,&
      1.8024781758740165E+01_dp, -1.0015971255468424E+01_dp,&
      2.5817837059480153E+00_dp,  1.4477457547611932E+01_dp,&
      -5.9542599972157469E-07_dp, -3.1829060629238062E+00_dp,&
      3.7245298868254464E+00_dp, -3.9522312305350300E+00_dp,&
      1.6423393285449270E+00_dp,  6.1216563001120274E+00_dp,&
      -2.0953918860770138E+01_dp,  3.9686837113822115E+01_dp,&
      -5.1935176897343290E+01_dp,  4.6512173586647570E+01_dp,&
      -2.5707915801780647E+01_dp,  6.5979718743648332E+00_dp,&
      2.9632190699838180E+01_dp, -1.5657622900881051E-06_dp,&
      -8.3586532272678387E+00_dp,  9.7810033020084397E+00_dp,&
      -1.0055079501227743E+01_dp,  3.5508201891380611E+00_dp,&
      1.7377218506217933E+01_dp, -5.6648334572026592E+01_dp,&
      1.0547111094388774E+02_dp, -1.3654326425592572E+02_dp,&
      1.2129760971899201E+02_dp, -6.6608157929477457E+01_dp,&
      1.7004088480309473E+01_dp,  6.3338122170095403E+01_dp,&
      -4.2444484549751587E-06_dp, -2.2620321314472132E+01_dp,&
      2.6469466355873809E+01_dp, -2.6151213910751117E+01_dp,&
      7.1147272422339958E+00_dp,  5.1213821549050515E+01_dp,&
      -1.5836988990793796E+02_dp,  2.8901910480803514E+02_dp,&
      -3.6930520446037576E+02_dp,  3.2477605951571485E+02_dp,&
      -1.7687999522963116E+02_dp,  4.4846043091496973E+01_dp,&
      1.4279947172069063E+02_dp, -1.2080229456795278E-05_dp,&
      -6.4239399665069129E+01_dp,  7.5170425103726117E+01_dp,&
      -7.0510474822968519E+01_dp,  1.1363573169598904E+01_dp,&
      1.5996866863552859E+02_dp, -4.6665591365326327E+02_dp,&
      8.3131873091270222E+02_dp, -1.0448759235890909E+03_dp,&
      9.0690542685405194E+02_dp, -4.8852833718934801E+02_dp,&
      1.2271331760718991E+02_dp,  3.4430186167302065E+02_dp,&
      -3.6917560063034570E-05_dp, -1.9573425600940379E+02_dp,&
      2.2903991435173910E+02_dp, -1.9998078868584736E+02_dp,&
      -3.6337191265627644E-01_dp,  5.4328299769275566E+02_dp,&
      -1.4832817481049449E+03_dp,  2.5633679410315954E+03_dp,&
      -3.1519867754826223E+03_dp,  2.6865378429486905E+03_dp,&
      -1.4246759516690913E+03_dp,  3.5301663828659417E+02_dp,&
      9.0657027822430757E+02_dp, -1.2505227876874551E-04_dp,&
      -6.6016663782193780E+02_dp,  7.7249587755502171E+02_dp,&
      -6.0564642881603584E+02_dp, -1.6334533603788654E+02_dp,&
      2.0809169575290675E+03_dp, -5.2520819705711947E+03_dp,&
      8.7142803953445527E+03_dp, -1.0381449011333283E+04_dp,&
      8.6066748064158692E+03_dp, -4.4514243053148093E+03_dp,&
      1.0783194357224340E+03_dp,  2.6999974727364965E+03_dp /)
    REAL(KIND=dp), DIMENSION(13, 28, 5), PARAMETER :: &
      coefdata = RESHAPE ((/ c01, c02, c03, c04, c05 /),(/13,28,5/))

    INTEGER                                  :: irange

     IF(Rc <=  1.0000000000000000E+01_dp) Rc =   1.0000000000000000E+01_dp
     IF(Rc >=  2.7148483276367188E+05_dp) Rc =   2.7148483276367188E+05_dp
     IF(Rc >=  1.0000000000000000E+01_dp .AND. Rc <=  1.0920229970956126E+02_dp) THEN
       irange =  1
       L_b =  1.0000000000000000E+01_dp
       U_b =  1.0920229970956126E+02_dp
     END IF
     IF(Rc >=  1.0920229970956126E+02_dp .AND. Rc <=  1.1839524369543471E+03_dp) THEN
       irange =  2
       L_b =  1.0920229970956126E+02_dp
       U_b =  1.1839524369543471E+03_dp
     END IF
     IF(Rc >=  1.1839524369543471E+03_dp .AND. Rc <=  1.5414599739111178E+04_dp) THEN
       irange =  3
       L_b =  1.1839524369543471E+03_dp
       U_b =  1.5414599739111178E+04_dp
     END IF
     IF(Rc >=  1.5414599739111178E+04_dp .AND. Rc <=  9.3151526902721278E+04_dp) THEN
       irange =  4
       L_b =  1.5414599739111178E+04_dp
       U_b =  9.3151526902721278E+04_dp
     END IF
     IF(Rc >=  9.3151526902721278E+04_dp .AND. Rc <=  2.7148483276367188E+05_dp) THEN
       irange =  5
       L_b =  9.3151526902721278E+04_dp
       U_b =  2.7148483276367188E+05_dp
     END IF

     fit_coef(1:13,1:28) = coefdata(1:13,1:28,irange)

   END SUBROUTINE get_fit_coef_k14

! *****************************************************************************
!> \brief ...
!> \param Rc ...
!> \param L_b ...
!> \param U_b ...
!> \param fit_coef ...
! *****************************************************************************
   SUBROUTINE get_fit_coef_k15(Rc,L_b,U_b,fit_coef)
    REAL(KIND=dp)                            :: Rc, L_b, U_b
    REAL(KIND=dp), DIMENSION(13, 30)         :: fit_coef

    REAL(KIND=dp), DIMENSION(350), PARAMETER :: c05 = (/ &
      -3.0692468631051978E-08_dp, -3.9609383383921665E-01_dp,&
      4.7587580740675417E-01_dp, -5.5706375318965173E-01_dp,&
      3.4546263714113901E-01_dp,  5.4776062905207357E-01_dp,&
      -2.4141552540426781E+00_dp,  4.9577847041515470E+00_dp,&
      -6.8295343742172383E+00_dp,  6.3573367557181761E+00_dp,&
      -3.6226122444655293E+00_dp,  9.5268597153976176E-01_dp,&
      6.6179227443553872E+00_dp, -7.7387082815093436E-08_dp,&
      -9.9678078330036968E-01_dp,  1.1975533847961941E+00_dp,&
      -1.3762136834418899E+00_dp,  8.0743054822484472E-01_dp,&
      1.4912431514031330E+00_dp, -6.2314927241780405E+00_dp,&
      1.2629990199235216E+01_dp, -1.7275639875772516E+01_dp,&
      1.6004550790906951E+01_dp, -9.0878072804552286E+00_dp,&
      2.3834753802899566E+00_dp,  1.2682718868072019E+01_dp,&
      -1.9334328719304757E-07_dp, -2.4848694599881327E+00_dp,&
      2.9853718860501335E+00_dp, -3.3596378831067573E+00_dp,&
      1.8411087716077625E+00_dp,  4.0272695241999301E+00_dp,&
      -1.5957538805213288E+01_dp,  3.1889777329262149E+01_dp,&
      -4.3281478360785940E+01_dp,  3.9883474913142209E+01_dp,&
      -2.2556907487546944E+01_dp,  5.8978299479488649E+00_dp,&
      2.5023174807569418E+01_dp, -4.8642081259672986E-07_dp,&
      -6.2351984490271315E+00_dp,  7.4910846767260342E+00_dp,&
      -8.2248711412084816E+00_dp,  4.1238878257772091E+00_dp,&
      1.0990309451485148E+01_dp, -4.1230623601049615E+01_dp,&
      8.1118417047199273E+01_dp, -1.0912326990788030E+02_dp,&
      9.9935525762372606E+01_dp, -5.6257102890489350E+01_dp,&
      1.4655642668172396E+01_dp,  5.1192402268189625E+01_dp,&
      -1.2506631640235665E-06_dp, -1.5980438605202909E+01_dp,&
      1.9199175975593889E+01_dp, -2.0457978476944227E+01_dp,&
      9.0671788455391216E+00_dp,  3.0813008462505493E+01_dp,&
      -1.0915584619445536E+02_dp,  2.1098187077206578E+02_dp,&
      -2.8088454943931174E+02_dp,  2.5533840000208855E+02_dp,&
      -1.4292670658071711E+02_dp,  3.7067966084892596E+01_dp,&
      1.0941038451297312E+02_dp, -3.3354017068023612E-06_dp,&
      -4.2446927792656162E+01_dp,  5.0996396254497533E+01_dp,&
      -5.2338301546774389E+01_dp,  1.9247992459076787E+01_dp,&
      9.0228533336751497E+01_dp, -3.0069997073722988E+02_dp,&
      5.6936309163352882E+02_dp, -7.4856520686405372E+02_dp,&
      6.7429591378758153E+02_dp, -3.7476540313467729E+02_dp,&
      9.6644172995788892E+01_dp,  2.4666624220966949E+02_dp,&
      -9.3826543851981905E-06_dp, -1.1877946256653507E+02_dp,&
      1.4270319798697713E+02_dp, -1.3946465595615874E+02_dp,&
      3.6961694322931955E+01_dp,  2.8120044470115204E+02_dp,&
      -8.7706262371593266E+02_dp,  1.6203584176977990E+03_dp,&
      -2.0972847074266356E+03_dp,  1.8671502744648556E+03_dp,&
      -1.0280983406275520E+03_dp,  2.6312195463073755E+02_dp,&
      5.9473096717460919E+02_dp, -2.8449948791129570E-05_dp,&
      -3.5758635317581138E+02_dp,  4.2960773213012766E+02_dp,&
      -3.9251600529353743E+02_dp,  4.5183929392732445E+01_dp,&
      9.5587078733574140E+02_dp, -2.7694370693632336E+03_dp,&
      4.9627218611957405E+03_dp, -6.2929514552940627E+03_dp,&
      5.5135886103861476E+03_dp, -2.9965165662533723E+03_dp,&
      7.5862722603535246E+02_dp,  1.5659660472206674E+03_dp,&
      -9.5986179588260765E-05_dp, -1.1939116417183877E+03_dp,&
      1.4343713151150068E+03_dp, -1.1851399647021360E+03_dp,&
      -1.5231964375206397E+02_dp,  3.6741780805293706E+03_dp,&
      -9.7734189198990425E+03_dp,  1.6822530274757435E+04_dp,&
      -2.0720426077965767E+04_dp,  1.7726884301747647E+04_dp,&
      -9.4410952942756994E+03_dp,  2.3490775569298803E+03_dp,&
      4.6638457658101452E+03_dp, -3.8015552197131998E-04_dp,&
      -4.6500281550825130E+03_dp,  5.5865303975222787E+03_dp,&
      -3.8837673064102523E+03_dp, -2.3640043516048258E+03_dp,&
      1.6965406299396385E+04_dp, -4.0565840455321268E+04_dp,&
      6.5811540329868585E+04_dp, -7.7320579116722583E+04_dp,&
      6.3441897875273360E+04_dp, -3.2532774480567186E+04_dp,&
      7.8216440760836831E+03_dp,  1.6825213139968786E+04_dp,&
      -1.9715775615195739E-03_dp, -2.3368672017163331E+04_dp,&
      2.8074707632439560E+04_dp, -1.3055332264394696E+04_dp,&
      -2.7517846784628873E+04_dp,  1.0619501478660316E+05_dp,&
      -2.1693608692558491E+05_dp,  3.1502086695178028E+05_dp,&
      -3.3334547694157349E+05_dp,  2.4553599204317702E+05_dp,&
      -1.1238955180628877E+05_dp,  2.3972326899658266E+04_dp,&
      8.8035762654044956E+04_dp, -1.8243112038117435E-02_dp,&
      -2.0014249841264417E+05_dp,  2.4044019203317922E+05_dp,&
      3.2918527777915639E+04_dp, -5.8564151349304337E+05_dp,&
      1.2278214162309726E+06_dp, -1.5862044513149876E+06_dp,&
      1.2269856342422077E+06_dp, -1.6000752081890625E+05_dp,&
      -8.0881279434379749E+05_dp,  8.5894001577282173E+05_dp,&
      -3.0531479437217204E+05_dp,  2.2298855158487991E+00_dp,&
      -9.0336389369874778E-09_dp, -1.1705905877090302E-01_dp,&
      1.4063753695885436E-01_dp, -1.7131704892021460E-01_dp,&
      1.1823577935867260E-01_dp,  1.3211160731762517E-01_dp,&
      -6.7148667896732095E-01_dp,  1.4224586147700080E+00_dp,&
      -1.9911772021989478E+00_dp,  1.8732253894330964E+00_dp,&
      -1.0756592628497046E+00_dp,  2.8453654479619789E-01_dp,&
      2.9930213382361659E+00_dp, -1.8670378462002057E-08_dp,&
      -2.4143656806336988E-01_dp,  2.9006740649273083E-01_dp,&
      -3.4629089408668917E-01_dp,  2.2683680893051139E-01_dp,&
      3.0404690414859736E-01_dp, -1.4297696355269953E+00_dp,&
      2.9800638894488669E+00_dp, -4.1371932160927800E+00_dp,&
      3.8711217608603641E+00_dp, -2.2142406528000071E+00_dp,&
      5.8398891025144894E-01_dp,  4.8875160233343715E+00_dp,&
      -4.6204570149850092E-08_dp, -5.9629866259653674E-01_dp,&
      7.1640628120411509E-01_dp, -8.3868041124278481E-01_dp,&
      5.2019826135940084E-01_dp,  8.2463242915711699E-01_dp,&
      -3.6348492405505075E+00_dp,  7.4650304783387114E+00_dp,&
      -1.0283768881675757E+01_dp,  9.5730498855986266E+00_dp,&
      -5.4551689976883058E+00_dp,  1.4346476675560549E+00_dp,&
      8.8414991600481354E+00_dp, -1.1482774693269792E-07_dp,&
      -1.4789938872294259E+00_dp,  1.7768943240759401E+00_dp,&
      -2.0408571717597774E+00_dp,  1.1953236620504990E+00_dp,&
      2.2183761876611556E+00_dp, -9.2555318513362081E+00_dp,&
      1.8752221113980344E+01_dp, -2.5645072882046485E+01_dp,&
      2.3755380780125787E+01_dp, -1.3487795474893785E+01_dp,&
      3.5372434275309308E+00_dp,  1.6909293215817328E+01_dp,&
      -2.8397263995252731E-07_dp, -3.6495531995060122E+00_dp,&
      4.3846461857239438E+00_dp, -4.9313380943000595E+00_dp,&
      2.6968368321956553E+00_dp,  5.9302864628254115E+00_dp,&
      -2.3462796400928930E+01_dp,  4.6870864451652814E+01_dp,&
      -6.3602080860714842E+01_dp,  5.8601551184019506E+01_dp,&
      -3.3140382879672700E+01_dp,  8.6644568567671278E+00_dp,&
      3.3671685943037645E+01_dp, -7.0886445358394887E-07_dp,&
      -9.0866956422509908E+00_dp,  1.0916927061893476E+01_dp,&
      -1.1982122131221457E+01_dp,  5.9998216294135123E+00_dp,&
      1.6041977958350834E+01_dp, -6.0136339183264646E+01_dp,&
      1.1829300280981532E+02_dp, -1.5911935257728072E+02_dp,&
      1.4571566750741661E+02_dp, -8.2025918687439358E+01_dp,&
      2.1368285429166992E+01_dp,  6.9717468070339038E+01_dp,&
      -1.8115637114197503E-06_dp, -2.3149864597531785E+01_dp,&
      2.7812649747426072E+01_dp, -2.9648862569096533E+01_dp,&
      1.3165828947892159E+01_dp,  4.4609325588012467E+01_dp,&
      -1.5814634948605772E+02_dp,  3.0576431202144215E+02_dp,&
      -4.0715554275312178E+02_dp,  3.7018587829115569E+02_dp,&
      -2.0724045709870501E+02_dp,  5.3753492193265338E+01_dp,&
      1.5076605862295096E+02_dp, -4.8029447707686627E-06_dp,&
      -6.1142104656738780E+01_dp,  7.3457079767394291E+01_dp,&
      -7.5551348219208521E+01_dp,  2.8115955488796168E+01_dp,&
      1.2939155360574125E+02_dp, -4.3261107002125891E+02_dp,&
      8.2012582176643787E+02_dp, -1.0791090432146957E+03_dp,&
      9.7263069656922221E+02_dp, -5.4083603585758897E+02_dp,&
      1.3952472833308838E+02_dp,  3.4326171137332619E+02_dp,&
      -1.3401787740328214E-05_dp, -1.6977952594693747E+02_dp,&
      2.0397539508271359E+02_dp, -2.0046385534576143E+02_dp,&
      5.5534906744180304E+01_dp,  3.9775987919436119E+02_dp,&
      -1.2493663191105957E+03_dp,  2.3147249863985589E+03_dp,&
      -3.0017062754705989E+03_dp,  2.6762457872586815E+03_dp,&
      -1.4753536354877817E+03_dp,  3.7795655560340720E+02_dp,&
      8.3250447697650543E+02_dp, -4.0066178299457473E-05_dp,&
      -5.0433618822359546E+02_dp,  6.0591475756705029E+02_dp,&
      -5.6061550959797250E+02_dp,  8.0693121371922274E+01_dp,&
      1.3220373413009017E+03_dp, -3.8795859841778079E+03_dp,&
      6.9916807710050844E+03_dp, -8.9011916360067389E+03_dp,&
      7.8237206451432530E+03_dp, -4.2633113757326564E+03_dp,&
      1.0817488919723778E+03_dp,  2.1880792776836724E+03_dp,&
      -1.3156759027407118E-04_dp, -1.6415118517756680E+03_dp,&
      1.9721227309225790E+03_dp, -1.6756252916105827E+03_dp,&
      -9.7731609047837892E+01_dp,  4.8848208178157192E+03_dp,&
      -1.3281764469769068E+04_dp,  2.3115603902558436E+04_dp,&
      -2.8709560935485304E+04_dp,  2.4734396997303353E+04_dp,&
      -1.3253308640711919E+04_dp,  3.3150401953712826E+03_dp,&
      6.4023792745151923E+03_dp, -4.9316125127201718E-04_dp,&
      -6.0727629604647600E+03_dp,  7.2958187858552392E+03_dp,&
      -5.4291483890577892E+03_dp, -2.2232679468127371E+03_dp,&
      2.0941874763722215E+04_dp, -5.2044169864559364E+04_dp,&
      8.6393786455101945E+04_dp, -1.0345994694934737E+05_dp,&
      8.6382110934542667E+04_dp, -4.5019043803292414E+04_dp,&
      1.0986821504031903E+04_dp,  2.1847515732361062E+04_dp,&
      -2.2579708925754478E-03_dp, -2.7213243572988889E+04_dp,&
      3.2693722118754918E+04_dp, -1.9030680015810074E+04_dp,&
      -2.2781523256788889E+04_dp,  1.1219253462313319E+05_dp,&
      -2.4822897274719740E+05_dp,  3.8292601430606667E+05_dp,&
      -4.3022911687807390E+05_dp,  3.3804652466094872E+05_dp,&
      -1.6611379681486270E+05_dp,  3.8304153833710385E+04_dp,&
      9.5573339232628467E+04_dp, -1.4588462469079619E-02_dp,&
      -1.6866513691120152E+05_dp,  2.0262927744864495E+05_dp,&
      -5.7768403952849316E+04_dp, -2.8684582736809686E+05_dp,&
      8.7522813270953298E+05_dp, -1.6053947767154609E+06_dp,&
      2.1164271865422688E+06_dp, -2.0010625502698994E+06_dp,&
      1.2728583679706273E+06_dp, -4.7452866201810562E+05_dp,&
      7.3939317332975275E+04_dp,  7.0111074483186123E+05_dp,&
      -2.0104415008030188E-01_dp, -2.1194663306621597E+06_dp,&
      2.5461696440140237E+06_dp,  1.4577959403430095E+06_dp,&
      -8.8776723555507679E+06_dp,  1.4429270679797074E+07_dp,&
      -1.0764718275979346E+07_dp, -6.0081024069717359E+06_dp,&
      2.8423455083762139E+07_dp, -3.8351286309245549E+07_dp,&
      2.6572649225598279E+07_dp, -7.9000536567775914E+06_dp /)
    REAL(KIND=dp), DIMENSION(400), PARAMETER :: c01 = (/ &
      1.8728308171438685E-01_dp, -7.4958225804522882E-02_dp,&
      -1.7654418697317781E-02_dp,  1.0337296188123402E-02_dp,&
      5.3074367941912289E-03_dp, -3.4737735619981940E-03_dp,&
      -2.0980081422520588E-03_dp,  1.3332352270175691E-03_dp,&
      1.8928638178553657E-03_dp, -2.0371064972525598E-03_dp,&
      4.3322700993716177E-04_dp,  2.4269923006729229E-04_dp,&
      -1.0645391970489572E-04_dp,  5.9050213462098911E-01_dp,&
      -2.6045916825257870E-01_dp, -4.6013842770087673E-02_dp,&
      4.0578808574448226E-02_dp,  1.3431038359853705E-02_dp,&
      -1.4448196212394048E-02_dp, -5.0801158293652263E-03_dp,&
      5.6830395388698567E-03_dp,  6.0399026225143836E-03_dp,&
      -9.0362581333669131E-03_dp,  3.8977272533399529E-03_dp,&
      -2.8631051093057830E-04_dp, -1.6255826015267508E-04_dp,&
      1.0853661382142048E+00_dp, -5.6505629769884813E-01_dp,&
      -4.5259537292808079E-02_dp,  1.0107558488280066E-01_dp,&
      1.0246730819933846E-02_dp, -3.7841969991286292E-02_dp,&
      -2.4466922115694272E-03_dp,  1.5238296869265590E-02_dp,&
      1.0051703287389212E-02_dp, -2.4895630648353886E-02_dp,&
      1.6271959664872358E-02_dp, -4.6274361677158590E-03_dp,&
      4.2879818988516113E-04_dp,  1.7528845246909897E+00_dp,&
      -1.1123944492275302E+00_dp,  3.5424018217895502E-02_dp,&
      2.1579693702045874E-01_dp, -2.5857027453275973E-02_dp,&
      -8.1949081571572172E-02_dp,  1.6563668405901510E-02_dp,&
      3.3437407219596017E-02_dp,  9.0789227225405479E-03_dp,&
      -5.4452189093168082E-02_dp,  4.6324965691288536E-02_dp,&
      -1.7622865195560243E-02_dp,  2.6635992084654233E-03_dp,&
      2.7078938439530917E+00_dp, -2.1055687575566426E+00_dp,&
      3.1071744605798479E-01_dp,  4.0776524304565614E-01_dp,&
      -1.4276623438669639E-01_dp, -1.5004288372475155E-01_dp,&
      7.4789568797600153E-02_dp,  6.1204081755821510E-02_dp,&
      -1.1450125801862587E-02_dp, -9.5333496012500740E-02_dp,&
      1.0315290568119541E-01_dp, -4.6301294653322549E-02_dp,&
      8.2327674140336218E-03_dp,  4.1222585977458319E+00_dp,&
      -3.8916375985052856E+00_dp,  1.0287256818377424E+00_dp,&
      6.7824953475735794E-01_dp, -4.3616962492030631E-01_dp,&
      -2.2442617116461031E-01_dp,  2.1410714787692231E-01_dp,&
      8.9232609313346464E-02_dp, -8.4070976342451670E-02_dp,&
      -1.2224787823475017E-01_dp,  1.8617969237834020E-01_dp,&
      -9.6992308351370335E-02_dp,  1.9240741102009955E-02_dp,&
      6.2614437064378148E+00_dp, -7.0734996616320798E+00_dp,&
      2.7109877698494551E+00_dp,  9.4315201130905646E-01_dp,&
      -1.0794537771539827E+00_dp, -2.2631895031272395E-01_dp,&
      4.9923803036845726E-01_dp,  7.9249948225847855E-02_dp,&
      -2.6692610962273050E-01_dp, -5.7380279775207862E-02_dp,&
      2.6060510417597316E-01_dp, -1.6580437343809790E-01_dp,&
      3.6703525275681770E-02_dp,  9.5442495162132879E+00_dp,&
      -1.2707406352368116E+01_dp,  6.4338288503783216E+00_dp,&
      8.6207574579597068E-01_dp, -2.3483916970462482E+00_dp,&
      7.3668001058521435E-02_dp,  1.0019608211825608E+00_dp,&
      -7.4722380119410070E-02_dp, -6.3758685173908902E-01_dp,&
      2.7271888085324325E-01_dp,  2.1084047383313775E-01_dp,&
      -2.1733444967013762E-01_dp,  5.6641676594043365E-02_dp,&
      1.4643304263035175E+01_dp, -2.2655257244471311E+01_dp,&
      1.4392217701024219E+01_dp, -5.9435672642207360E-01_dp,&
      -4.5777070581320389E+00_dp,  1.2351960033513358E+00_dp,&
      1.7170546319191513E+00_dp, -6.1224168013341784E-01_dp,&
      -1.2399540344239814E+00_dp,  1.1799686690987607E+00_dp,&
      -2.3435024824910564E-01_dp, -1.4444306875541876E-01_dp,&
      6.1909340705238292E-02_dp,  2.2661556393863005E+01_dp,&
      -4.0221154146253184E+01_dp,  3.1024585967572101E+01_dp,&
      -6.1984371213472800E+00_dp, -7.8268184197828807E+00_dp,&
      4.4496347113412762E+00_dp,  2.2809196015538231E+00_dp,&
      -1.9772352681903975E+00_dp, -1.9037690088995247E+00_dp,&
      3.0942613517855539E+00_dp, -1.5987186090989016E+00_dp,&
      3.0213023980747233E-01_dp,  5.8046939919012327E-03_dp,&
      3.5468093717051559E+01_dp, -7.1336923975099211E+01_dp,&
      6.5243630185390288E+01_dp, -2.2937295080112825E+01_dp,&
      -1.0565705083043978E+01_dp,  1.1806144892783585E+01_dp,&
      1.2178868152228470E+00_dp, -4.7251305539334565E+00_dp,&
      -1.7396453277841573E+00_dp,  6.1545408517662787E+00_dp,&
      -4.5896192312693147E+00_dp,  1.5674169440757093E+00_dp,&
      -2.0955667192577354E-01_dp,  5.6442005976612165E+01_dp,&
      -1.2697432446862335E+02_dp,  1.3491229212895021E+02_dp,&
      -6.7527741358861761E+01_dp, -5.5728868452042661E+00_dp,&
      2.5434552707557021E+01_dp, -5.4812962163209651E+00_dp,&
      -8.5501736428865840E+00_dp,  1.6761122225991096E+00_dp,&
      8.5806543180294934E+00_dp, -8.9923594919231888E+00_dp,&
      3.9271082076494404E+00_dp, -6.8512753005700844E-01_dp,&
      9.2592250966591095E+01_dp, -2.2931021529690631E+02_dp,&
      2.7694141294096011E+02_dp, -1.7833095335890167E+02_dp,&
      3.0475551450022131E+01_dp,  4.2007083036843376E+01_dp,&
      -2.4692838382505418E+01_dp, -9.2334955726867634E+00_dp,&
      1.0827009524998724E+01_dp,  5.3897134051317694E+00_dp,&
      -1.1613244577817873E+01_dp,  6.3881771027172425E+00_dp,&
      -1.2985721875418899E+00_dp,  1.6317349757925933E+02_dp,&
      -4.3567621344686893E+02_dp,  5.8173974531331419E+02_dp,&
      -4.4774963325360090E+02_dp,  1.5972672378438833E+02_dp,&
      3.9090311776792390E+01_dp, -5.9596575929838380E+01_dp,&
      1.7921854401952180E+00_dp,  2.2619550601715890E+01_dp,&
      -5.3934015665785919E+00_dp, -9.7558137801544511E+00_dp,&
      7.6935514963494267E+00_dp, -1.8168321790144826E+00_dp,&
      3.6167981936470210E+02_dp, -1.0134714814220670E+03_dp,&
      1.4434041436150949E+03_dp, -1.2377699820298194E+03_dp,&
      5.8888285175980138E+02_dp, -4.1630388543112794E+01_dp,&
      -1.1428599149682097E+02_dp,  2.8274776131364867E+01_dp,&
      4.1140189506559487E+01_dp, -2.4185930418501993E+01_dp,&
      -7.4527591947965899E+00_dp,  1.0860057531580511E+01_dp,&
      -2.9572025136611222E+00_dp,  1.5169954011304849E+00_dp,&
      -6.2259099720483668E-01_dp, -1.3712328672328261E-01_dp,&
      8.8962828652861925E-02_dp,  4.1043676954109272E-02_dp,&
      -3.0443932125223352E-02_dp, -1.6102573780172595E-02_dp,&
      1.1773644941326244E-02_dp,  1.5459317774057306E-02_dp,&
      -1.8230644358955633E-02_dp,  5.1169302495753487E-03_dp,&
      1.3183518368033997E-03_dp, -7.5903468555029124E-04_dp,&
      1.7499525908457525E+00_dp, -9.1531468832248197E-01_dp,&
      -7.6236631412840317E-02_dp,  1.6660751635474580E-01_dp,&
      1.8633663668780786E-02_dp, -6.2945543301738627E-02_dp,&
      -5.1025796217109790E-03_dp,  2.5374090389752808E-02_dp,&
      1.7578523782323396E-02_dp, -4.1763776533811942E-02_dp,&
      2.6693483469204270E-02_dp, -7.3228880092403711E-03_dp,&
      6.1452305084940938E-04_dp,  2.2619483691280533E+00_dp,&
      -1.6013976329479644E+00_dp,  1.1397591566142432E-01_dp,&
      3.3354035175947067E-01_dp, -5.7655973122046261E-02_dp,&
      -1.3000506221511882E-01_dp,  3.2967403963862302E-02_dp,&
      5.3494998718352296E-02_dp,  1.1629723916519787E-02_dp,&
      -8.8697135876886060E-02_dp,  7.8248427005448054E-02_dp,&
      -3.0608523079921474E-02_dp,  4.7649824200322270E-03_dp,&
      3.1532640805277889E+00_dp, -2.9086028536368387E+00_dp,&
      6.0362531752264503E-01_dp,  6.0074812985031767E-01_dp,&
      -2.6167841326003777E-01_dp, -2.2598188022853924E-01_dp,&
      1.3384536380554243E-01_dp,  9.3175482448094510E-02_dp,&
      -2.8552228684050263E-02_dp, -1.4717381630923509E-01_dp,&
      1.6884009102232561E-01_dp, -7.8198141800493393E-02_dp,&
      1.4252294977810340E-02_dp,  4.5990362784635748E+00_dp,&
      -5.2610480580945511E+00_dp,  1.7554142769832759E+00_dp,&
      9.3954313197653416E-01_dp, -7.3530977939486497E-01_dp,&
      -3.0941341340797157E-01_dp,  3.5949602821775650E-01_dp,&
      1.2391323817308118E-01_dp, -1.5674994650740204E-01_dp,&
      -1.6497906356785436E-01_dp,  2.8695923379282917E-01_dp,&
      -1.5603347654001742E-01_dp,  3.1792664959924305E-02_dp,&
      6.8890308910671356E+00_dp, -9.4168852534476279E+00_dp,&
      4.3188327286694310E+00_dp,  1.1783565195132464E+00_dp,&
      -1.7254059330117610E+00_dp, -2.3869963236274130E-01_dp,&
      7.9885232236465131E-01_dp,  7.7283423048285574E-02_dp,&
      -4.6310391349780267E-01_dp, -1.4416187514689229E-03_dp,&
      3.5710470924999077E-01_dp, -2.4826140681908590E-01_dp,&
      5.7194043278611120E-02_dp,  1.0497250232124662E+01_dp,&
      -1.6719832965151884E+01_dp,  9.8216896813060970E+00_dp,&
      7.3970611364224259E-01_dp, -3.6059125166637358E+00_dp,&
      3.6498570473675707E-01_dp,  1.5345652988820813E+00_dp,&
      -2.2332312617279992E-01_dp, -1.0543504968226272E+00_dp,&
      6.3278478456932230E-01_dp,  1.6617016052927272E-01_dp,&
      -2.8118812153705103E-01_dp,  8.0364317707803778E-02_dp,&
      1.6202514503988390E+01_dp, -2.9552478542397893E+01_dp,&
      2.1354172370954117E+01_dp, -2.0112687983411681E+00_dp,&
      -6.7855159638167732E+00_dp,  2.3873633162830670E+00_dp,&
      2.4998632165545791E+00_dp, -1.1623801758158443E+00_dp,&
      -1.9504681292485571E+00_dp,  2.2152489068540460E+00_dp,&
      -7.3792333984291780E-01_dp, -6.4276694004470475E-02_dp,&
      6.8740941716736517E-02_dp,  2.5304655431445173E+01_dp,&
      -5.2159151005688130E+01_dp,  4.5132654192248751E+01_dp,&
      -1.1338185998416552E+01_dp, -1.1160559135910367E+01_dp,&
      7.6739991523479407E+00_dp,  3.0247629570482868E+00_dp,&
      -3.4345182182881113E+00_dp, -2.7853008863066906E+00_dp,&
      5.3467839342257841E+00_dp, -3.1694123271238630E+00_dp,&
      8.0694157345556727E-01_dp, -5.5813093165602519E-02_dp,&
      4.0044203390039350E+01_dp, -9.2178078631682354E+01_dp,&
      9.3592414174368287E+01_dp, -3.7812448013221193E+01_dp,&
      -1.4050944707063607E+01_dp,  1.9459317299329200E+01_dp,&
      6.0508974835963314E-01_dp, -7.9026281849074591E+00_dp,&
      -2.0036084707040196E+00_dp,  1.0164747665646736E+01_dp,&
      -8.2753537306586118E+00_dp,  3.0766573481774491E+00_dp,&
      -4.5919018524870359E-01_dp,  6.4548216002732048E+01_dp,&
      -1.6372767342710813E+02_dp,  1.9149111780202836E+02_dp,&
      -1.0651796580505838E+02_dp, -3.6942339641588529E+00_dp,&
      4.1116703654714840E+01_dp, -1.1688259826259763E+01_dp,&
      -1.4092944078806669E+01_dp,  4.9534063773839927E+00_dp,&
      1.3327217036380015E+01_dp, -1.5720082956234478E+01_dp,&
      7.3285690650141238E+00_dp, -1.3498478899609199E+00_dp,&
      1.0751974782069394E+02_dp, -2.9506101073517016E+02_dp,&
      3.8886438902426795E+02_dp, -2.7414973759844793E+02_dp,&
      5.7813607870059755E+01_dp,  6.6686184229942938E+01_dp,&
      -4.6907689236635093E+01_dp, -1.3942873569870745E+01_dp,&
      2.4627269489905036E+01_dp,  2.2908131761383355E+00_dp,&
      -1.7205170554994950E+01_dp,  1.0876408321603687E+01_dp,&
      -2.3799479399999801E+00_dp,  1.9247091864637630E+02_dp,&
      -5.5613733430051036E+02_dp,  7.9963641497991784E+02_dp,&
      -6.6716934500841262E+02_dp,  2.6869114067386937E+02_dp,&
      5.1976162118599937E+01_dp, -1.0977479209219821E+02_dp,&
      1.5320418459417469E+01_dp,  4.6056433428995703E+01_dp,&
      -2.9674185020400326E+01_dp, -1.1567576408411366E+00_dp,&
      7.5442392510615788E+00_dp, -2.2833978807688808E+00_dp,&
      4.1371492246319275E+02_dp, -1.2139560601068151E+03_dp,&
      1.8159618918225744E+03_dp, -1.6547693041000880E+03_dp,&
      8.6670487570533533E+02_dp, -1.1144406945065859E+02_dp,&
      -1.6484069878511906E+02_dp,  7.3842511245161845E+01_dp,&
      4.7114364272953480E+01_dp, -5.3558177553213220E+01_dp,&
      1.2533328664185559E+01_dp,  4.7084386636147091E+00_dp,&
      -2.2343883375336238E+00_dp,  1.5482456811918644E+03_dp,&
      -4.4722809407060004E+03_dp,  6.6239888100139933E+03_dp,&
      -6.0381436371333639E+03_dp,  3.2764369520591445E+03_dp,&
      -6.5019366566941835E+02_dp, -3.5765412817763291E+02_dp,&
      1.5908326595301082E+02_dp,  1.4549695520260664E+02_dp,&
      -1.2481558144644566E+02_dp,  1.0775656086159311E+00_dp,&
      3.0587066325522777E+01_dp, -9.7246696807779784E+00_dp,&
      2.3046596783109666E-01_dp, -5.0266863054748409E-02_dp,&
      -1.7179737588739360E-02_dp,  5.4654515693096711E-03_dp,&
      4.7852361622912525E-03_dp, -1.7421101769864551E-03_dp,&
      -1.9440310548800761E-03_dp,  6.0661250198343168E-04_dp,&
      1.3341594386686755E-03_dp, -8.7661120974839130E-04_dp /)
    REAL(KIND=dp), DIMENSION(400), PARAMETER :: c02 = (/ &
      -2.3785652584517211E-04_dp,  4.0415260781371566E-04_dp,&
      -1.1566827381466792E-04_dp,  7.4614843698202982E-01_dp,&
      -1.8831297390879426E-01_dp, -5.5307781300718430E-02_dp,&
      2.5755321163507486E-02_dp,  1.6130804272204945E-02_dp,&
      -8.8949071784985307E-03_dp, -6.6594057854666763E-03_dp,&
      3.2468867718668313E-03_dp,  5.3068142089810330E-03_dp,&
      -4.9387141407180018E-03_dp,  4.6635469014696219E-04_dp,&
      1.0225379001983694E-03_dp, -3.6432448373067977E-04_dp,&
      1.4438155770093133E+00_dp, -4.6034617711746817E-01_dp,&
      -1.0114569577453580E-01_dp,  8.0847736706817566E-02_dp,&
      3.1132826193078431E-02_dp, -3.0098300767058127E-02_dp,&
      -1.2990120467489992E-02_dp,  1.1474677064022006E-02_dp,&
      1.3524876701271211E-02_dp, -1.8252222127657793E-02_dp,&
      6.7665303400730129E-03_dp,  2.6588974984116941E-04_dp,&
      -5.3997537121426489E-04_dp,  2.5092558836526182E+00_dp,&
      -1.0370895440920307E+00_dp, -1.4293476604484445E-01_dp,&
      2.1892543997539182E-01_dp,  4.3916720461941325E-02_dp,&
      -8.5906691296838311E-02_dp, -1.7581973163101349E-02_dp,&
      3.3856445186467174E-02_dp,  2.9188840795829587E-02_dp,&
      -5.5487833762756317E-02_dp,  3.0644984612438222E-02_dp,&
      -6.2518601968366934E-03_dp, -1.3619319925894898E-05_dp,&
      4.2480175628654671E+00_dp, -2.2552731443674090E+00_dp,&
      -1.2020190507153349E-01_dp,  5.3601909417709703E-01_dp,&
      2.3123136649126307E-02_dp, -2.1742428493087468E-01_dp,&
      -3.1971501625727987E-03_dp,  8.7966369542184522E-02_dp,&
      5.2098492049143776E-02_dp, -1.4627449223179681E-01_dp,&
      1.0200338814556090E-01_dp, -3.1449802576644731E-02_dp,&
      3.4350745589740235E-03_dp,  7.1934846151788383E+00_dp,&
      -4.8045602148293982E+00_dp,  1.6800723431165815E-01_dp,&
      1.2178133966689220E+00_dp, -1.3794075755207405E-01_dp,&
      -4.9998735924019683E-01_dp,  8.8402497336550506E-02_dp,&
      2.0630420687207099E-01_dp,  6.3021419984179414E-02_dp,&
      -3.4072681253649273E-01_dp,  2.8686006501533551E-01_dp,&
      -1.0813824760173801E-01_dp,  1.6124430353177774E-02_dp,&
      1.2310857286275628E+01_dp, -1.0127157366490705E+01_dp,&
      1.3116248102932406E+00_dp,  2.5981695903237028E+00_dp,&
      -7.5308905349267552E-01_dp, -1.0473666363263530E+00_dp,&
      4.2542510815825957E-01_dp,  4.3632942640926314E-01_dp,&
      -2.7104988102069912E-02_dp, -6.9029606232100948E-01_dp,&
      7.0742413257606307E-01_dp, -3.0835516320907685E-01_dp,&
      5.3474117194919846E-02_dp,  2.1395537244919190E+01_dp,&
      -2.1294952582127532E+01_dp,  4.9586759348241420E+00_dp,&
      5.1888176836291438E+00_dp, -2.6758759362963236E+00_dp,&
      -1.9431216985141384E+00_dp,  1.4461702367488496E+00_dp,&
      7.9948224935758594E-01_dp, -5.1733385566911505E-01_dp,&
      -1.1192375715667364E+00_dp,  1.5165197860827433E+00_dp,&
      -7.5951327716264216E-01_dp,  1.4699498210546361E-01_dp,&
      3.7874378642397900E+01_dp, -4.4985821130051526E+01_dp,&
      1.5631115834221772E+01_dp,  9.4321652649337349E+00_dp,&
      -8.0952721883237579E+00_dp, -2.8567401038783609E+00_dp,&
      4.1805740387217281E+00_dp,  1.0764667230663318E+00_dp,&
      -2.2360065629356378E+00_dp, -9.4199681401382451E-01_dp,&
      2.6381666350602235E+00_dp, -1.5885150294156811E+00_dp,&
      3.4308353829740224E-01_dp,  6.8468294337154987E+01_dp,&
      -9.6066459756805102E+01_dp,  4.5738898453688122E+01_dp,&
      1.4111679011798346E+01_dp, -2.2267791999698446E+01_dp,&
      -1.5136616324510050E+00_dp,  1.0688893528377216E+01_dp,&
      -1.3084462945928754E-02_dp, -7.0303230583103691E+00_dp,&
      2.5138353770466910E+00_dp,  2.7198542807232915E+00_dp,&
      -2.5443022123523957E+00_dp,  6.4822382263330280E-01_dp,&
      1.2677231414292079E+02_dp, -2.0856628210561396E+02_dp,&
      1.2977414621627011E+02_dp,  9.1386695755711109E+00_dp,&
      -5.6431954234367460E+01_dp,  1.1601893649962131E+01_dp,&
      2.3601434993236484E+01_dp, -7.4033248447262663E+00_dp,&
      -1.7502732203666131E+01_dp,  1.6822593726600431E+01_dp,&
      -3.9321362151268588E+00_dp, -1.6129758489863975E+00_dp,&
      7.7708342864698754E-01_dp,  2.4137930903364659E+02_dp,&
      -4.6294363698594930E+02_dp,  3.6567746448701314E+02_dp,&
      -5.4264739702960071E+01_dp, -1.2743717778047937E+02_dp,&
      6.9426575766974068E+01_dp,  3.8441422516024915E+01_dp,&
      -3.5937690433228575E+01_dp, -3.0556184645453925E+01_dp,&
      5.6709057679208890E+01_dp, -3.3613907949953663E+01_dp,&
      8.7069360708662646E+00_dp, -6.5071547786799400E-01_dp,&
      4.7630568634676882E+02_dp, -1.0576120213436998E+03_dp,&
      1.0387559056977509E+03_dp, -3.8252814075560741E+02_dp,&
      -2.1458148531810613E+02_dp,  2.6341292507433673E+02_dp,&
      -4.9256154892799719E-01_dp, -1.1226068848553658E+02_dp,&
      -2.8240527873382217E+00_dp,  1.1790110895706198E+02_dp,&
      -1.0801679538988816E+02_dp,  4.3992074063654670E+01_dp,&
      -7.2605761932187214E+00_dp,  9.9831523554887349E+02_dp,&
      -2.5291408915064985E+03_dp,  3.0129078291719397E+03_dp,&
      -1.7907805874724606E+03_dp,  5.2655917157171132E+01_dp,&
      6.9446212730128070E+02_dp, -3.3406235416814116E+02_dp,&
      -1.6432224599188541E+02_dp,  1.8182149839084897E+02_dp,&
      5.5649378552951724E+01_dp, -1.5404182921741005E+02_dp,&
      8.7668748811935586E+01_dp, -1.8136282582373241E+01_dp,&
      2.5197956634819393E+03_dp, -7.0358606845934346E+03_dp,&
      9.6643808587164713E+03_dp, -7.6367755567523545E+03_dp,&
      2.8044634842378887E+03_dp,  7.1741892008980801E+02_dp,&
      -1.1890761393357277E+03_dp,  1.6354332870871900E+02_dp,&
      3.8067937600501506E+02_dp, -1.6046444938930748E+02_dp,&
      -1.0923720079478032E+02_dp,  1.0988436092388882E+02_dp,&
      -2.7749973961794815E+01_dp,  1.8790883577505788E+00_dp,&
      -4.2578976954083553E-01_dp, -1.4015384201338235E-01_dp,&
      4.9535242205172317E-02_dp,  3.9556319497385098E-02_dp,&
      -1.6188345328963884E-02_dp, -1.6150186727266109E-02_dp,&
      5.7212758934123784E-03_dp,  1.1515686373747173E-02_dp,&
      -8.4039177187524849E-03_dp, -1.2421200032158236E-03_dp,&
      3.1542038254185724E-03_dp, -9.4397800109121256E-04_dp,&
      2.3286072431387423E+00_dp, -7.3963153685831751E-01_dp,&
      -1.6817064802096532E-01_dp,  1.2874106425262241E-01_dp,&
      5.2793112327834887E-02_dp, -4.7503985299264515E-02_dp,&
      -2.2274548170645277E-02_dp,  1.7979044108164671E-02_dp,&
      2.2209117750315765E-02_dp, -2.8376437312440522E-02_dp,&
      9.4867238052117418E-03_dp,  1.1424442652292026E-03_dp,&
      -1.0083455255258623E-03_dp,  3.3727466736396918E+00_dp,&
      -1.5448224549513210E+00_dp, -1.9258486598122063E-01_dp,&
      3.4252847181451407E-01_dp,  6.4318683841940644E-02_dp,&
      -1.3514789671089780E-01_dp, -2.6692286257403969E-02_dp,&
      5.3252686100318150E-02_dp,  4.5847091590627692E-02_dp,&
      -8.7415775234976054E-02_dp,  4.8250758426842198E-02_dp,&
      -9.7951517189096889E-03_dp, -4.0080346635974667E-05_dp,&
      5.3476974926650058E+00_dp, -3.2821481672447801E+00_dp,&
      -1.0673693072908484E-01_dp,  8.2336777253315929E-01_dp,&
      1.9953202180531222E-02_dp, -3.3773322054580296E-01_dp,&
      -1.4581831738714061E-04_dp,  1.3707953425659358E-01_dp,&
      8.0179761639936148E-02_dp, -2.2969150949183195E-01_dp,&
      1.6148118046840720E-01_dp, -5.0234143578115772E-02_dp,&
      5.5756585628377389E-03_dp,  8.9052398487289590E+00_dp,&
      -6.8986923960309667E+00_dp,  4.0813160978388663E-01_dp,&
      1.8330154014443996E+00_dp, -2.4965908117099087E-01_dp,&
      -7.6390746125752773E-01_dp,  1.4990729029415834E-01_dp,&
      3.1721789359789093E-01_dp,  9.3176773742444513E-02_dp,&
      -5.3063124869930844E-01_dp,  4.5174324220642409E-01_dp,&
      -1.7188470254374158E-01_dp,  2.5897541226569322E-02_dp,&
      1.5277197043057445E+01_dp, -1.4403808006208923E+01_dp,&
      2.2520306793354425E+00_dp,  3.8402789683619147E+00_dp,&
      -1.2282086708774107E+00_dp, -1.5752333655294370E+00_dp,&
      6.8512948891006886E-01_dp,  6.6286709270779132E-01_dp,&
      -5.6777181683095333E-02_dp, -1.0658001277397429E+00_dp,&
      1.1088446843283359E+00_dp, -4.8809164974479147E-01_dp,&
      8.5368915942929544E-02_dp,  2.6809203330113430E+01_dp,&
      -3.0105648181028421E+01_dp,  7.8907784967325929E+00_dp,&
      7.5527940324183627E+00_dp, -4.2136321310777394E+00_dp,&
      -2.8787838332303517E+00_dp,  2.2822315773743074E+00_dp,&
      1.2025374446858241E+00_dp, -8.4310251040346118E-01_dp,&
      -1.7073511450150149E+00_dp,  2.3701711370977949E+00_dp,&
      -1.2003354391249612E+00_dp,  2.3417510086396071E-01_dp,&
      4.8050803348005907E+01_dp, -6.3414162259620326E+01_dp,&
      2.4012810181586435E+01_dp,  1.3543281743456987E+01_dp,&
      -1.2518399110121845E+01_dp, -4.1483228966192884E+00_dp,&
      6.5359361416230444E+00_dp,  1.6010875650124006E+00_dp,&
      -3.5935349636567349E+00_dp, -1.3505908589460529E+00_dp,&
      4.1102852487568118E+00_dp, -2.5128140904440519E+00_dp,&
      5.4740471934507018E-01_dp,  8.8043083290018245E+01_dp,&
      -1.3537232179695650E+02_dp,  6.8833207015861163E+01_dp,&
      1.9954488880701138E+01_dp, -3.4131409980955162E+01_dp,&
      -1.9360725759409532E+00_dp,  1.6704761193414530E+01_dp,&
      -8.8596046107263363E-02_dp, -1.1369840543223333E+01_dp,&
      4.3533647172741814E+00_dp,  4.1462824454929388E+00_dp,&
      -4.0304785375198096E+00_dp,  1.0392352576178361E+00_dp,&
      1.6528695869768484E+02_dp, -2.9435766562155197E+02_dp,&
      1.9269303927992763E+02_dp,  1.2272006016669405E+01_dp,&
      -8.6558626633572814E+01_dp,  1.8084227124270082E+01_dp,&
      3.7422286676621773E+01_dp, -1.1504422400508385E+01_dp,&
      -2.8981028446875509E+01_dp,  2.8061685767204931E+01_dp,&
      -6.8454282426917770E+00_dp, -2.5075498308103796E+00_dp,&
      1.2590902098959449E+00_dp,  3.1904415962372269E+02_dp,&
      -6.5517407839010923E+02_dp,  5.3795044575370639E+02_dp,&
      -7.8460961642246900E+01_dp, -1.9941242952975134E+02_dp,&
      1.0626216155817104E+02_dp,  6.5024974663538401E+01_dp,&
      -5.6853636598935005E+01_dp, -5.4516670856876033E+01_dp,&
      9.6886573999137951E+01_dp, -5.6786140068674428E+01_dp,&
      1.4494723569712843E+01_dp, -1.0273091986637437E+00_dp,&
      6.3717770820107091E+02_dp, -1.5006743273417519E+03_dp,&
      1.5183508248241242E+03_dp, -5.4249059143678107E+02_dp,&
      -3.6871786881056681E+02_dp,  4.1511942398217616E+02_dp,&
      2.6225823385079689E+01_dp, -1.9254671249051862E+02_dp,&
      -2.7660102820898491E+01_dp,  2.2884670976955013E+02_dp,&
      -2.0078072107272666E+02_dp,  8.0102960484258318E+01_dp,&
      -1.2994587476880236E+01_dp,  1.3388641449458219E+03_dp,&
      -3.5707279333293045E+03_dp,  4.3696883768952457E+03_dp,&
      -2.5408264232775923E+03_dp, -1.5720657008223867E+02_dp,&
      1.2433964069705148E+03_dp, -4.9064532646415466E+02_dp,&
      -4.0438252913184124E+02_dp,  3.5221285081678133E+02_dp,&
      1.6421609423683356E+02_dp, -3.5828174338405978E+02_dp,&
      1.9958566836052518E+02_dp, -4.1225182836963377E+01_dp,&
      3.1630578325136571E+03_dp, -9.2174296603112798E+03_dp,&
      1.3078767927116834E+04_dp, -1.0455636907823062E+04_dp,&
      3.4380325323762208E+03_dp,  1.9056619449914899E+03_dp,&
      -2.4483725222720182E+03_dp,  3.3598576622327710E+02_dp,&
      1.0045407494218127E+03_dp, -7.4953512509145685E+02_dp,&
      1.1389182240118748E+02_dp,  8.8631597307204402E+01_dp,&
      -3.3784175186635018E+01_dp,  1.1741583647618587E+04_dp,&
      -3.4772802927998528E+04_dp,  5.1757382963840319E+04_dp,&
      -4.6796698845286672E+04_dp,  2.4404793826910518E+04_dp,&
      -3.4240409533000625E+03_dp, -4.1525449932535876E+03_dp,&
      1.9174273170875924E+03_dp,  9.6404934848061453E+02_dp,&
      -9.6135633177130569E+02_dp, -6.2667702498202678E+00_dp,&
      2.5568713212589310E+02_dp, -8.0871228602797459E+01_dp,&
      2.5910399582486437E-01_dp, -2.8215937471986335E-02_dp,&
      -1.6057948736797313E-02_dp,  2.2232618509844711E-03_dp,&
      4.4103499063233055E-03_dp, -1.6453858095913664E-04_dp,&
      -2.6110557878466982E-03_dp,  3.9824644955139122E-04_dp,&
      1.6273475095842586E-03_dp, -8.9025885910407965E-04_dp,&
      -3.6671922956720823E-04_dp,  4.8507440460371750E-04_dp,&
      -1.3211072877791544E-04_dp,  8.5654480285554591E-01_dp,&
      -1.1200920153309295E-01_dp, -6.0375651330303620E-02_dp,&
      1.2480943363126322E-02_dp,  1.7766988818178821E-02_dp,&
      -2.1391630741560913E-03_dp, -1.0741465894792950E-02_dp /)
    REAL(KIND=dp), DIMENSION(400), PARAMETER :: c03 = (/ &
      2.2061351968797813E-03_dp,  7.2651800654771788E-03_dp,&
      -4.9666091131882828E-03_dp, -6.0509045722459322E-04_dp,&
      1.7074706793122701E-03_dp, -5.0886032182887841E-04_dp,&
      1.7258143200373688E+00_dp, -2.9884323352649256E-01_dp,&
      -1.4775030321223576E-01_dp,  4.7248271588260116E-02_dp,&
      4.7619516819167255E-02_dp, -1.1816286083326438E-02_dp,&
      -2.9752710649639229E-02_dp,  8.5535969429659311E-03_dp,&
      2.2544415185358772E-02_dp, -1.9478392039671260E-02_dp,&
      2.3867184506599064E-03_dp,  3.3938096349746282E-03_dp,&
      -1.2408011528001253E-03_dp,  3.1765354659041769E+00_dp,&
      -7.4113425888104711E-01_dp, -3.3028597049657266E-01_dp,&
      1.5279087435777911E-01_dp,  1.1534912548347100E-01_dp,&
      -4.6610003150119948E-02_dp, -7.4975530724277845E-02_dp,&
      2.9204637910627697E-02_dp,  6.4144154361418626E-02_dp,&
      -6.7383151583557066E-02_dp,  1.9495127476223385E-02_dp,&
      3.9216434341502104E-03_dp, -2.5026214876007966E-03_dp,&
      5.7762677335774812E+00_dp, -1.7814057289603056E+00_dp,&
      -7.0223258614858830E-01_dp,  4.5053468321128493E-01_dp,&
      2.6008563697492804E-01_dp, -1.5727265430895401E-01_dp,&
      -1.7715192567226051E-01_dp,  9.2605163231566937E-02_dp,&
      1.7280977666896552E-01_dp, -2.1622833321996968E-01_dp,&
      8.9907102665867045E-02_dp, -6.5342786988899322E-03_dp,&
      -3.6405862618677188E-03_dp,  1.0635866709342723E+01_dp,&
      -4.2221910963666494E+00_dp, -1.4280030736515645E+00_dp,&
      1.2620175013991994E+00_dp,  5.4430896273848306E-01_dp,&
      -4.8950416510612965E-01_dp, -3.9333543661332560E-01_dp,&
      2.8233335910502383E-01_dp,  4.4606392158013780E-01_dp,&
      -6.6340801329142474E-01_dp,  3.4698053836989273E-01_dp,&
      -7.0117046739817404E-02_dp,  3.8524475772982433E-04_dp,&
      2.0028892471519779E+01_dp, -1.0003628886391914E+01_dp,&
      -2.7543527547087376E+00_dp,  3.4459409293308343E+00_dp,&
      1.0227949830416387E+00_dp, -1.4575186303587055E+00_dp,&
      -8.0548916230565581E-01_dp,  8.4662036227712156E-01_dp,&
      1.1006704135061391E+00_dp, -1.9762595229695696E+00_dp,&
      1.2293924485257903E+00_dp, -3.4776573573238989E-01_dp,&
      3.3411960379456256E-02_dp,  3.8800864934511303E+01_dp,&
      -2.3979969508026311E+01_dp, -4.8691713652289721E+00_dp,&
      9.3268439917937549E+00_dp,  1.5091857344004103E+00_dp,&
      -4.2309476977858385E+00_dp, -1.4145743330439313E+00_dp,&
      2.5351120694533700E+00_dp,  2.5243975989462042E+00_dp,&
      -5.7315045646482972E+00_dp,  4.1375154729827681E+00_dp,&
      -1.4175776019905513E+00_dp,  1.9259934383182015E-01_dp,&
      7.7735814299357841E+01_dp, -5.8814947029987437E+01_dp,&
      -6.9572678017406089E+00_dp,  2.5308616104245754E+01_dp,&
      3.9853220364063036E-01_dp, -1.2070957719421918E+01_dp,&
      -1.4815249980025791E+00_dp,  7.6544448352112955E+00_dp,&
      4.8490644610502533E+00_dp, -1.5967127077365880E+01_dp,&
      1.3341122150477743E+01_dp, -5.2520565274679489E+00_dp,&
      8.4320063077592255E-01_dp,  1.6206206464295840E+02_dp,&
      -1.4930321627961305E+02_dp, -2.5067856011273091E+00_dp,&
      6.9341075313618006E+01_dp, -1.0743440903075324E+01_dp,&
      -3.3716158577793486E+01_dp,  3.7254112287325314E+00_dp,&
      2.3363694407054414E+01_dp,  3.9659586871550068E+00_dp,&
      -4.0721927683672071E+01_dp,  4.0561860559316933E+01_dp,&
      -1.8073377097484137E+01_dp,  3.2530077490973341E+00_dp,&
      3.5442984306993384E+02_dp, -3.9746210109659887E+02_dp,&
      4.2890577383048353E+01_dp,  1.9202721291279789E+02_dp,&
      -7.1796204519006139E+01_dp, -8.9522197012670105E+01_dp,&
      3.7812207284886419E+01_dp,  7.1003674952178457E+01_dp,&
      -3.3126726641687128E+01_dp, -7.9733890323907957E+01_dp,&
      1.0865212885707727E+02_dp, -5.6048850673878945E+01_dp,&
      1.1208474159757820E+01_dp,  8.2232427943744574E+02_dp,&
      -1.1282193717418054E+03_dp,  3.0056386653438210E+02_dp,&
      5.2970835948779984E+02_dp, -3.6912742804955758E+02_dp,&
      -1.9945931211622650E+02_dp,  2.0796916315615783E+02_dp,&
      1.9918225114845831E+02_dp, -2.9472016065289944E+02_dp,&
      7.2671923695521441E+00_dp,  1.9096380738566870E+02_dp,&
      -1.3490590492082302E+02_dp,  3.1463294283517353E+01_dp,&
      2.0580837910555242E+03_dp, -3.4946061703309515E+03_dp,&
      1.6696299583911045E+03_dp,  1.3450505077893019E+03_dp,&
      -1.7913069927563238E+03_dp, -8.6372308442249675E+01_dp,&
      9.4017698393274759E+02_dp,  3.3494319705863421E+02_dp,&
      -1.5979707059674208E+03_dp,  1.3250952509876529E+03_dp,&
      -3.9493447453139942E+02_dp, -4.2074272991575519E+01_dp,&
      3.7749026605926751E+01_dp,  5.7172884504696012E+03_dp,&
      -1.2228637615334226E+04_dp,  9.5260345584609622E+03_dp,&
      1.5243163670468698E+03_dp, -8.3375278186301748E+03_dp,&
      4.2132976943189569E+03_dp,  2.6329863532489203E+03_dp,&
      -1.9930161570578837E+03_dp, -4.4234144073523730E+03_dp,&
      7.9728267713326386E+03_dp, -5.8508031847927323E+03_dp,&
      2.1939029662054095E+03_dp, -3.4737249846121421E+02_dp,&
      1.9237695027274051E+04_dp, -5.1947738927857921E+04_dp,&
      6.2765011200887377E+04_dp, -3.0309589558643987E+04_dp,&
      -1.9264261229194974E+04_dp,  4.1446219371737265E+04_dp,&
      -2.6281354883180597E+04_dp,  2.8159263824857394E+03_dp,&
      3.6485501407647080E+03_dp,  2.6770249060636979E+03_dp,&
      -6.2669352427291251E+03_dp,  3.7308014050905122E+03_dp,&
      -8.0644597819162743E+02_dp,  2.1234916041378580E+00_dp,&
      -2.4267551942842563E-01_dp, -1.3618390064078997E-01_dp,&
      2.1235253488344166E-02_dp,  3.8121053083591247E-02_dp,&
      -2.2512374350561323E-03_dp, -2.2690956898994009E-02_dp,&
      3.7722692795086480E-03_dp,  1.4456191546859473E-02_dp,&
      -8.4462833259858951E-03_dp, -2.6996025350312465E-03_dp,&
      4.0618392843868737E-03_dp, -1.1299679877086958E-03_dp,&
      2.7792848344652787E+00_dp, -4.7477598542627597E-01_dp,&
      -2.3798512429820987E-01_dp,  7.2098824833812886E-02_dp,&
      7.6285142890366947E-02_dp, -1.7095747870246605E-02_dp,&
      -4.7341370096482475E-02_dp,  1.2775854073981439E-02_dp,&
      3.5074556718704991E-02_dp, -2.8956193783707895E-02_dp,&
      2.2732905932494967E-03_dp,  5.9324463384063473E-03_dp,&
      -2.0476473783333669E-03_dp,  4.3726378328909492E+00_dp,&
      -1.1147668218629847E+00_dp, -4.9438793138131359E-01_dp,&
      2.3411800744875491E-01_dp,  1.7621745035600736E-01_dp,&
      -7.0675331650434867E-02_dp, -1.1444595532273523E-01_dp,&
      4.3810434696352277E-02_dp,  9.7399112758829939E-02_dp,&
      -1.0075502118647087E-01_dp,  2.7820211439353416E-02_dp,&
      6.8061814563366834E-03_dp, -3.9600522823777032E-03_dp,&
      7.5940664502338420E+00_dp, -2.6359395650789450E+00_dp,&
      -1.0267420292656975E+00_dp,  6.8391874036213196E-01_dp,&
      3.9112093438927609E-01_dp, -2.3797924906131696E-01_dp,&
      -2.6702577764788687E-01_dp,  1.3830923397715175E-01_dp,&
      2.6053229138318296E-01_dp, -3.2264392621733523E-01_dp,&
      1.3155017102629712E-01_dp, -7.8585587277630424E-03_dp,&
      -5.8776989356745333E-03_dp,  1.3906829345279375E+01_dp,&
      -6.1815615927462932E+00_dp, -2.0567479439921703E+00_dp,&
      1.8929310304780897E+00_dp,  8.0939883147882108E-01_dp,&
      -7.3450777930442601E-01_dp, -5.8793748940266832E-01_dp,&
      4.1849011345367765E-01_dp,  6.6959615955201890E-01_dp,&
      -9.8583966364769504E-01_dp,  5.0895589523531948E-01_dp,&
      -9.9254205792793884E-02_dp, -6.1164209449963590E-04_dp,&
      2.6401188276808139E+01_dp, -1.4535395682302440E+01_dp,&
      -3.9190359193691893E+00_dp,  5.1181176445344025E+00_dp,&
      1.5115068531544618E+00_dp, -2.1714821317110071E+00_dp,&
      -1.2002962513354241E+00_dp,  1.2461281841515912E+00_dp,&
      1.6551300034059584E+00_dp, -2.9317809023065591E+00_dp,&
      1.8031912714428560E+00_dp, -5.0050761339182093E-01_dp,&
      4.5835377439546375E-02_dp,  5.1774174778275956E+01_dp,&
      -3.4673590093429169E+01_dp, -6.8605657769909634E+00_dp,&
      1.3753843857799250E+01_dp,  2.2488088677398350E+00_dp,&
      -6.2744810231042134E+00_dp, -2.1279575450853696E+00_dp,&
      3.7095435731655004E+00_dp,  3.8509567321319773E+00_dp,&
      -8.5280597524752579E+00_dp,  6.0785754666089655E+00_dp,&
      -2.0517020855153323E+00_dp,  2.7264394937468583E-01_dp,&
      1.0509588382613930E+02_dp, -8.4780704698196232E+01_dp,&
      -9.7392349655477037E+00_dp,  3.7137208312733918E+01_dp,&
      8.3554407985177170E-01_dp, -1.7880263854377880E+01_dp,&
      -2.4083329597635625E+00_dp,  1.1151723834747575E+01_dp,&
      7.7871979605114570E+00_dp, -2.4043267182405760E+01_dp,&
      1.9735664203306662E+01_dp, -7.6526126546828053E+00_dp,&
      1.2084415666148431E+00_dp,  2.2189324774713293E+02_dp,&
      -2.1464604053209558E+02_dp, -3.6807906910198707E+00_dp,&
      1.0146351820682573E+02_dp, -1.4155528812748599E+01_dp,&
      -5.0203644978574609E+01_dp,  4.3001718188217284E+00_dp,&
      3.4014323314955611E+01_dp,  9.1644378901327990E+00_dp,&
      -6.3442475370119638E+01_dp,  6.1193423228111932E+01_dp,&
      -2.6745055759988510E+01_dp,  4.7357733742760555E+00_dp,&
      4.9075000758815480E+02_dp, -5.6911173591276133E+02_dp,&
      5.7194456067601791E+01_dp,  2.8119762788146772E+02_dp,&
      -9.6106533145203329E+01_dp, -1.3626933155906542E+02_dp,&
      4.9851263918608808E+01_dp,  1.0456184482930480E+02_dp,&
      -3.2146041163311445E+01_dp, -1.3945447237077744E+02_dp,&
      1.7320187517099365E+02_dp, -8.6336243126137077E+01_dp,&
      1.6891573599822905E+01_dp,  1.1481520205431834E+03_dp,&
      -1.6025998192547047E+03_dp,  3.9445867091993847E+02_dp,&
      7.8459322025471704E+02_dp, -4.9045650179813879E+02_dp,&
      -3.3175436609396507E+02_dp,  2.8078846062724392E+02_dp,&
      3.1070835070821977E+02_dp, -3.6220306330311195E+02_dp,&
      -1.1681704840662950E+02_dp,  3.7714350711930985E+02_dp,&
      -2.3526844262559749E+02_dp,  5.2160651991091960E+01_dp,&
      2.8817218555073487E+03_dp, -4.8844832039242247E+03_dp,&
      2.1166301273677968E+03_dp,  2.1145167459343247E+03_dp,&
      -2.3531376839776899E+03_dp, -4.6965613633104721E+02_dp,&
      1.3278822822389677E+03_dp,  7.3531833182374783E+02_dp,&
      -2.2060454899663455E+03_dp,  1.3552254192307953E+03_dp,&
      -1.8597548287900345E+00_dp, -3.2827436822361659E+02_dp,&
      1.0578152183175671E+02_dp,  7.9287162776111199E+03_dp,&
      -1.6556637711428240E+04_dp,  1.1377040542438039E+04_dp,&
      4.3472707467271775E+03_dp, -1.1253948176882712E+04_dp,&
      2.8844083146489602E+03_dp,  5.2635488814341406E+03_dp,&
      -5.3349077678909532E+02_dp, -9.5980797679285060E+03_dp,&
      1.2237468602717436E+04_dp, -7.1752996430490848E+03_dp,&
      2.0904595579719871E+03_dp, -2.3076473049581341E+02_dp,&
      2.4941513553111414E+04_dp, -6.5057154343987058E+04_dp,&
      6.8510522134202620E+04_dp, -1.2564664624188634E+04_dp,&
      -4.8494990702839394E+04_dp,  4.8066227692409921E+04_dp,&
      1.2754284959000174E+03_dp, -2.7700527143446598E+04_dp,&
      1.3436695087102896E+03_dp,  3.5023043646265716E+04_dp,&
      -3.8543590945586038E+04_dp,  1.8589546507271829E+04_dp,&
      -3.6217344190998742E+03_dp,  1.0880372505911512E+05_dp,&
      -3.3682866668262816E+05_dp,  5.0268295205038588E+05_dp,&
      -4.0507797875013901E+05_dp,  7.2631515561032633E+04_dp,&
      2.4905603733649486E+05_dp, -3.5947552860654687E+05_dp,&
      2.7628042300900549E+05_dp, -1.4381451338213816E+05_dp,&
      5.7733437538747756E+04_dp, -2.0832981857862389E+04_dp,&
      6.4928622544881764E+03_dp, -1.1195098273879023E+03_dp,&
      2.6915793781734648E-01_dp, -1.0381942599208310E-02_dp,&
      -9.3650521014913869E-03_dp,  3.2090193795957971E-03_dp,&
      -3.3386986359229646E-03_dp,  5.5539179173563335E-03_dp,&
      -6.3480573143521249E-03_dp,  6.2871036161375651E-03_dp,&
      -6.0131949795456498E-03_dp,  4.9810441408132712E-03_dp,&
      -3.0620311237843996E-03_dp,  1.1962570988126766E-03_dp,&
      -2.2066239674921657E-04_dp,  8.9689216873395949E-01_dp,&
      -4.2122586464803081E-02_dp, -3.7513882433882192E-02_dp,&
      1.3881724731562524E-02_dp, -1.3477012568179458E-02_dp,&
      2.2562073529423785E-02_dp, -2.5951410486921640E-02_dp,&
      2.5567379426983138E-02_dp, -2.4333781051323347E-02_dp,&
      2.0151995146635144E-02_dp, -1.2400482709093779E-02_dp,&
      4.8452373650168124E-03_dp, -8.9294844328912499E-04_dp,&
      1.8352119311378547E+00_dp, -1.1605842392533414E-01_dp,&
      -1.0141001120766774E-01_dp,  4.1699067057004879E-02_dp /)
    REAL(KIND=dp), DIMENSION(400), PARAMETER :: c04 = (/ &
      -3.6931298185134437E-02_dp,  6.2239013775737956E-02_dp,&
      -7.2288174936489244E-02_dp,  7.0693666064322772E-02_dp,&
      -6.6776211112181863E-02_dp,  5.5274684972707100E-02_dp,&
      -3.4069532004586962E-02_dp,  1.3318163740503416E-02_dp,&
      -2.4516852040202333E-03_dp,  3.4526910112953169E+00_dp,&
      -2.9811951866509651E-01_dp, -2.5502175885566530E-01_dp,&
      1.1666479573438553E-01_dp, -9.4640066895441091E-02_dp,&
      1.5992007690589682E-01_dp, -1.8786362795493361E-01_dp,&
      1.8234170407841388E-01_dp, -1.7071583364783530E-01_dp,&
      1.4120711271809522E-01_dp, -8.7222157926976809E-02_dp,&
      3.4129137457941487E-02_dp, -6.2772574750481792E-03_dp,&
      6.4526874369355705E+00_dp, -7.4379137647143534E-01_dp,&
      -6.2169351557404495E-01_dp,  3.1608406508912079E-01_dp,&
      -2.3664910820793422E-01_dp,  3.9861181596578510E-01_dp,&
      -4.7437842206341091E-01_dp,  4.5710809927671725E-01_dp,&
      -4.2357791972371833E-01_dp,  3.4994703099419200E-01_dp,&
      -2.1674549913705246E-01_dp,  8.4953665492070438E-02_dp,&
      -1.5619616785382430E-02_dp,  1.2272761702136265E+01_dp,&
      -1.8365536182683235E+00_dp, -1.4952269059085537E+00_dp,&
      8.4751536974247987E-01_dp, -5.8923289605863272E-01_dp,&
      9.8181746694991578E-01_dp, -1.1862663094962580E+00_dp,&
      1.1351701430563665E+00_dp, -1.0389137932494279E+00_dp,&
      8.5664247790845927E-01_dp, -5.3242501279025167E-01_dp,&
      2.0925723568583102E-01_dp, -3.8489862677363318E-02_dp,&
      2.4000086828210332E+01_dp, -4.5581060135710860E+00_dp,&
      -3.5977120742775472E+00_dp,  2.2897011921494417E+00_dp,&
      -1.4872248187048727E+00_dp,  2.4261115525821300E+00_dp,&
      -2.9842698163078438E+00_dp,  2.8374294543564580E+00_dp,&
      -2.5569428859304861E+00_dp,  2.1013503214810827E+00_dp,&
      -1.3118947003867396E+00_dp,  5.1782664621522245E-01_dp,&
      -9.5402662711553898E-02_dp,  4.8587093531959233E+01_dp,&
      -1.1532062155028678E+01_dp, -8.7655022066070973E+00_dp,&
      6.3319337945364706E+00_dp, -3.8731156981586130E+00_dp,&
      6.0963207775734327E+00_dp, -7.6591948339186171E+00_dp,&
      7.2433544682793292E+00_dp, -6.3996585265106960E+00_dp,&
      5.2289327669288426E+00_dp, -3.2829199929814634E+00_dp,&
      1.3043816113094733E+00_dp, -2.4119202153591454E-01_dp,&
      1.0254367589701396E+02_dp, -3.0156545743278251E+01_dp,&
      -2.1861824137429267E+01_dp,  1.8198871770079609E+01_dp,&
      -1.0610092557902053E+01_dp,  1.5784369861157300E+01_dp,&
      -2.0325106722771388E+01_dp,  1.9159263068907812E+01_dp,&
      -1.6497575716105022E+01_dp,  1.3338797204809440E+01_dp,&
      -8.4308703027030507E+00_dp,  3.3832494619455296E+00_dp,&
      -6.2994976928592783E-01_dp,  2.2763518387273865E+02_dp,&
      -8.2764595584286539E+01_dp, -5.6398311753951880E+01_dp,&
      5.5286001032641892E+01_dp, -3.1300545607087930E+01_dp,&
      4.2735085101753064E+01_dp, -5.6557182696520613E+01_dp,&
      5.3353437804874766E+01_dp, -4.4397860550024987E+01_dp,&
      3.5190517825708255E+01_dp, -2.2390621695313161E+01_dp,&
      9.1181332326342606E+00_dp, -1.7187754447680910E+00_dp,&
      5.3814419453876985E+02_dp, -2.4299588074893020E+02_dp,&
      -1.5201445631333939E+02_dp,  1.8130374185702280E+02_dp,&
      -1.0262892549571430E+02_dp,  1.2338122205446149E+02_dp,&
      -1.6769634862214531E+02_dp,  1.5947931529626564E+02_dp,&
      -1.2673253607941503E+02_dp,  9.6503470517689962E+01_dp,&
      -6.1507970117755306E+01_dp,  2.5568992544284665E+01_dp,&
      -4.9210909009850994E+00_dp,  1.3801918068313626E+03_dp,&
      -7.8409403684044321E+02_dp, -4.3139798224082045E+02_dp,&
      6.6108802464127928E+02_dp, -3.9172133571873263E+02_dp,&
      3.9236375791502667E+02_dp, -5.4013676730762052E+02_dp,&
      5.2452481311060501E+02_dp, -3.9173273726869706E+02_dp,&
      2.7308075960502873E+02_dp, -1.6996111729874119E+02_dp,&
      7.2338992388678662E+01_dp, -1.4396994651161968E+01_dp,&
      3.9580600472914825E+03_dp, -2.9036032599844793E+03_dp,&
      -1.2781768816295728E+03_dp,  2.8070929268418577E+03_dp,&
      -1.8752951429725697E+03_dp,  1.4753386767800280E+03_dp,&
      -1.9323503518765785E+03_dp,  1.9527579205070861E+03_dp,&
      -1.3486577520305257E+03_dp,  7.5381327111334554E+02_dp,&
      -3.9743710790630257E+02_dp,  1.6635147838756436E+02_dp,&
      -3.4761194955563198E+01_dp,  1.3425684708567962E+04_dp,&
      -1.3383900660812744E+04_dp, -3.4320824963960499E+03_dp,&
      1.5050487619482205E+04_dp, -1.2942745398196372E+04_dp,&
      8.1171714061498142E+03_dp, -7.8406387741109365E+03_dp,&
      8.0705849662959281E+03_dp, -5.1239781721026657E+03_dp,&
      1.3200473018363366E+03_dp,  4.4071137447532215E+02_dp,&
      -4.2582815928175688E+02_dp,  9.8145031152881089E+01_dp,&
      6.1763293249195012E+04_dp, -9.3023683746086099E+04_dp,&
      1.4302377887690851E+04_dp,  1.1395909986937273E+05_dp,&
      -1.7772764092445478E+05_dp,  1.4368085671526083E+05_dp,&
      -6.1917915753482834E+04_dp, -1.3748361328957566E+04_dp,&
      6.2361092152953963E+04_dp, -7.7904069086697375E+04_dp,&
      6.0598777613532206E+04_dp, -2.8232839487469380E+04_dp,&
      5.9912034166305712E+03_dp,  2.2102099829502819E+00_dp,&
      -8.9807091694595470E-02_dp, -8.0738852355447699E-02_dp,&
      2.8245551420289328E-02_dp, -2.8838152951585126E-02_dp,&
      4.8060961814259832E-02_dp, -5.5022894223366300E-02_dp,&
      5.4417947301434594E-02_dp, -5.1980922605685237E-02_dp,&
      4.3055991607175798E-02_dp, -2.6474712012505622E-02_dp,&
      1.0343216742807535E-02_dp, -1.9074395117850485E-03_dp,&
      2.9526385249227900E+00_dp, -1.8342158808651834E-01_dp,&
      -1.6079974432000388E-01_dp,  6.4990457572363936E-02_dp,&
      -5.8357582370356993E-02_dp,  9.8375835274615853E-02_dp,&
      -1.1403666430432409E-01_dp,  1.1164874104323870E-01_dp,&
      -1.0561858908323142E-01_dp,  8.7439013178214162E-02_dp,&
      -5.3873844636419935E-02_dp,  2.1055369777301082E-02_dp,&
      -3.8763341662694521E-03_dp,  4.7882425300284677E+00_dp,&
      -4.4880027377068837E-01_dp, -3.8387620646428122E-01_dp,&
      1.7581567932182518E-01_dp, -1.4218651556165815E-01_dp,&
      2.4088533741840665E-01_dp, -2.8285896497383756E-01_dp,&
      2.7446561521278501E-01_dp, -2.5704300468522123E-01_dp,&
      2.1264001116035619E-01_dp, -1.3132372928749306E-01_dp,&
      5.1372473735998153E-02_dp, -9.4466605494810223E-03_dp,&
      8.5963629598619455E+00_dp, -1.1032812111617041E+00_dp,&
      -9.2128991958738193E-01_dp,  4.7072149697983523E-01_dp,&
      -3.5013599087368830E-01_dp,  5.9164664653692711E-01_dp,&
      -7.0411617076286537E-01_dp,  6.7802505793072532E-01_dp,&
      -6.2831000597729314E-01_dp,  5.1918399266101378E-01_dp,&
      -3.2153146818208928E-01_dp,  1.2598734950885529E-01_dp,&
      -2.3156182641453846E-02_dp,  1.6307279051808887E+01_dp,&
      -2.6964639955096659E+00_dp, -2.1929083193674388E+00_dp,&
      1.2494850984921542E+00_dp, -8.6264580098333044E-01_dp,&
      1.4424439050266553E+00_dp, -1.7431000698797154E+00_dp,&
      1.6666423180736005E+00_dp, -1.5253713037802321E+00_dp,&
      1.2581191094820894E+00_dp, -7.8188837190149219E-01_dp,&
      3.0719187328985836E-01_dp, -5.6478457126488940E-02_dp,&
      3.2180867728506904E+01_dp, -6.6411072684078274E+00_dp,&
      -5.2372172602238214E+00_dp,  3.3470888636894665E+00_dp,&
      -2.1592514567307126E+00_dp,  3.5371868143609047E+00_dp,&
      -4.3517183020335670E+00_dp,  4.1336520605961979E+00_dp,&
      -3.7258847437211742E+00_dp,  3.0636078261990956E+00_dp,&
      -1.9124981724704819E+00_dp,  7.5452097435395959E-01_dp,&
      -1.3892551807807468E-01_dp,  6.5943104829375315E+01_dp,&
      -1.6707127044375667E+01_dp, -1.2698264773736545E+01_dp,&
      9.1863412650630529E+00_dp, -5.5829483431324674E+00_dp,&
      8.8392123953951618E+00_dp, -1.1105112337604499E+01_dp,&
      1.0490324644517454E+01_dp, -9.2755649942267286E+00_dp,&
      7.5866003373259350E+00_dp, -4.7628121299140389E+00_dp,&
      1.8909156066204140E+00_dp, -3.4932019459840230E-01_dp,&
      1.4087116058808186E+02_dp, -4.3470586509878942E+01_dp,&
      -3.1572239382059106E+01_dp,  2.6185590050497311E+01_dp,&
      -1.5166880594846853E+01_dp,  2.2778700310398051E+01_dp,&
      -2.9319687356015418E+01_dp,  2.7596362755558843E+01_dp,&
      -2.3808410787503707E+01_dp,  1.9293100477047371E+01_dp,&
      -1.2194695218999417E+01_dp,  4.8872630176721907E+00_dp,&
      -9.0854459291729428E-01_dp,  3.1603181254524878E+02_dp,&
      -1.1853513661034587E+02_dp, -8.1246680173306473E+01_dp,&
      7.8648347652941197E+01_dp, -4.4153278086217064E+01_dp,&
      6.1303806049424246E+01_dp, -8.1073505393952701E+01_dp,&
      7.6296406201925507E+01_dp, -6.3752564021982288E+01_dp,&
      5.0794561685790477E+01_dp, -3.2341634414896959E+01_dp,&
      1.3141737689621349E+01_dp, -2.4701157665658586E+00_dp,&
      7.5252253575264456E+02_dp, -3.4418682220066631E+02_dp,&
      -2.1837415788962826E+02_dp,  2.5329125041502920E+02_dp,&
      -1.4128547056745032E+02_dp,  1.7507672278319296E+02_dp,&
      -2.3804626639739695E+02_dp,  2.2535235168628145E+02_dp,&
      -1.8051776327743372E+02_dp,  1.3923468265213427E+02_dp,&
      -8.9091837917185714E+01_dp,  3.6920629587167475E+01_dp,&
      -7.0690524517695410E+00_dp,  1.9315421410301064E+03_dp,&
      -1.0874625239245886E+03_dp, -6.1808932166184934E+02_dp,&
      8.9545464671544005E+02_dp, -5.1479696749349444E+02_dp,&
      5.4356197242025212E+02_dp, -7.5448078467711980E+02_dp,&
      7.2603369830764473E+02_dp, -5.4986172074879164E+02_dp,&
      3.9669672465398293E+02_dp, -2.5164163702286163E+02_dp,&
      1.0700313706124334E+02_dp, -2.1113910665526991E+01_dp,&
      5.4723123751409612E+03_dp, -3.8610955995775034E+03_dp,&
      -1.8448756713449409E+03_dp,  3.5966262696431140E+03_dp,&
      -2.2512699759682537E+03_dp,  1.9194105802260040E+03_dp,&
      -2.6322931011174383E+03_dp,  2.6185867118885690E+03_dp,&
      -1.8443677996699814E+03_dp,  1.1419121865305731E+03_dp,&
      -6.7280482181537468E+02_dp,  2.9210777012637755E+02_dp,&
      -6.0511721609154549E+01_dp,  1.7804801411705495E+04_dp,&
      -1.6259024258501489E+04_dp, -5.5718594192341498E+03_dp,&
      1.7345758837342113E+04_dp, -1.2877366253533812E+04_dp,&
      8.5915553852783196E+03_dp, -1.0315077162695961E+04_dp,&
      1.0800104463257729E+04_dp, -7.0138651831887073E+03_dp,&
      2.8554212903400171E+03_dp, -8.8722261141725619E+02_dp,&
      2.9569669149664014E+02_dp, -6.9226682688697807E+01_dp,&
      7.1835823339602095E+04_dp, -9.0186165593538084E+04_dp,&
      -9.7248478944525286E+03_dp,  1.1062096220530488E+05_dp,&
      -1.1369726450560044E+05_dp,  6.7036834170998452E+04_dp,&
      -4.6207804389370787E+04_dp,  4.4594452729411365E+04_dp,&
      -2.6072626782514293E+04_dp, -2.9846178239841133E+03_dp,&
      1.5681375871473874E+04_dp, -1.0022357789314863E+04_dp,&
      2.3179238419310946E+03_dp,  4.4219529370203713E+05_dp,&
      -8.4755186000937573E+05_dp,  3.7879188093690283E+05_dp,&
      9.7631641875634436E+05_dp, -2.2196339514296870E+06_dp,&
      2.3054796315979110E+06_dp, -1.0628907039914103E+06_dp,&
      -7.6347796690832789E+05_dp,  2.1400073212989480E+06_dp,&
      -2.4192321832669177E+06_dp,  1.7082231092388814E+06_dp,&
      -7.3012565960855584E+05_dp,  1.4473968969337529E+05_dp,&
      2.7143097167899877E-01_dp, -1.0426354655888225E-09_dp,&
      -1.3514349234655127E-02_dp,  1.6236462271473035E-02_dp,&
      -1.9832094707145951E-02_dp,  1.3779856524964107E-02_dp,&
      1.5011176255910088E-02_dp, -7.7179118599429580E-02_dp,&
      1.6386504807733129E-01_dp, -2.2964169829422010E-01_dp,&
      2.1619787855597516E-01_dp, -1.2421278261439007E-01_dp,&
      3.2870250330679252E-02_dp,  9.0613777954751162E-01_dp,&
      -4.2561353091248025E-09_dp, -5.5108681647653930E-02_dp,&
      6.6208860978777484E-02_dp, -8.0041705305285135E-02_dp,&
      5.4189321111835380E-02_dp,  6.4928039936621940E-02_dp,&
      -3.2000393170497327E-01_dp,  6.7367170849599256E-01_dp,&
      -9.4004438108949862E-01_dp,  8.8254299622584398E-01_dp,&
      -5.0603178442964947E-01_dp,  1.3370755800909928E-01_dp,&
      1.8607801318806969E+00_dp, -1.1832063710451179E-08_dp,&
      -1.5296272228579366E-01_dp,  1.8377288757765206E-01_dp,&
      -2.1880198683428151E-01_dp,  1.4228504563245828E-01_dp,&
      1.9523650997654252E-01_dp, -9.0945729598550951E-01_dp,&
      1.8916140218456958E+00_dp, -2.6232469600311217E+00_dp,&
      2.4527671312473118E+00_dp, -1.4022183672866397E+00_dp,&
      3.6967613733809346E-01_dp,  3.5186336824147015E+00_dp /)
    REAL(KIND=dp), DIMENSION(13, 30, 5), PARAMETER :: &
      coefdata = RESHAPE ((/ c01, c02, c03, c04, c05 /),(/13,30,5/))

    INTEGER                                  :: irange

     IF(Rc <=  1.5000000000000000E+01_dp) Rc =   1.5000000000000000E+01_dp
     IF(Rc >=  4.6894989013671875E+05_dp) Rc =   4.6894989013671875E+05_dp
     IF(Rc >=  1.5000000000000000E+01_dp .AND. Rc <=  1.6990973774601900E+02_dp) THEN
       irange =  1
       L_b =  1.5000000000000000E+01_dp
       U_b =  1.6990973774601900E+02_dp
     END IF
     IF(Rc >=  1.6990973774601900E+02_dp .AND. Rc <=  1.8995280520207393E+03_dp) THEN
       irange =  2
       L_b =  1.6990973774601900E+02_dp
       U_b =  1.8995280520207393E+03_dp
     END IF
     IF(Rc >=  1.8995280520207393E+03_dp .AND. Rc <=  2.6390317079362849E+04_dp) THEN
       irange =  3
       L_b =  1.8995280520207393E+03_dp
       U_b =  2.6390317079362849E+04_dp
     END IF
     IF(Rc >=  2.6390317079362849E+04_dp .AND. Rc <=  1.5948665335135348E+05_dp) THEN
       irange =  4
       L_b =  2.6390317079362849E+04_dp
       U_b =  1.5948665335135348E+05_dp
     END IF
     IF(Rc >=  1.5948665335135348E+05_dp .AND. Rc <=  4.6894989013671875E+05_dp) THEN
       irange =  5
       L_b =  1.5948665335135348E+05_dp
       U_b =  4.6894989013671875E+05_dp
     END IF

     fit_coef(1:13,1:30) = coefdata(1:13,1:30,irange)

   END SUBROUTINE get_fit_coef_k15

! *****************************************************************************
!> \brief ...
!> \param Rc ...
!> \param L_b ...
!> \param U_b ...
!> \param fit_coef ...
! *****************************************************************************
   SUBROUTINE get_fit_coef_k16(Rc,L_b,U_b,fit_coef)
    REAL(KIND=dp)                            :: Rc, L_b, U_b
    REAL(KIND=dp), DIMENSION(13, 32)         :: fit_coef

    REAL(KIND=dp), DIMENSION(400), PARAMETER :: c01 = (/ &
      1.8440983439459679E-01_dp, -7.0479249003300562E-02_dp,&
      -1.6832381157590469E-02_dp,  9.9837323526780937E-03_dp,&
      5.1989041899428937E-03_dp, -3.5423416042951205E-03_dp,&
      -2.1543663571483209E-03_dp,  1.3096125508292179E-03_dp,&
      2.2049401312140603E-03_dp, -2.5023058617921153E-03_dp,&
      7.5375163633056852E-04_dp,  1.3637466463565460E-04_dp,&
      -9.2994202830172699E-05_dp,  5.8054036572020107E-01_dp,&
      -2.4380379683949494E-01_dp, -4.4938696140059450E-02_dp,&
      3.8768465512213003E-02_dp,  1.3593778767841709E-02_dp,&
      -1.4461499926526256E-02_dp, -5.4490788053341418E-03_dp,&
      5.5553275407733936E-03_dp,  7.1422948686751375E-03_dp,&
      -1.0583913052061498E-02_dp,  4.9741685262334190E-03_dp,&
      -6.6432140641299532E-04_dp, -1.0892699705666875E-04_dp,&
      1.0638318043371691E+00_dp, -5.2505303151143268E-01_dp,&
      -4.9624782924737383E-02_dp,  9.5687743884690382E-02_dp,&
      1.2767947916143167E-02_dp, -3.7404011533309391E-02_dp,&
      -3.9949046346714071E-03_dp,  1.4966928016166042E-02_dp,&
      1.2577310120692639E-02_dp, -2.8100049351042718E-02_dp,&
      1.8433425451462267E-02_dp, -5.4088352179882731E-03_dp,&
      5.4854194494455884E-04_dp,  1.7106633703662533E+00_dp,&
      -1.0248769125439021E+00_dp,  1.0241427657345478E-02_dp,&
      2.0420458035779018E-01_dp, -1.5264072128014391E-02_dp,&
      -8.1320404223678117E-02_dp,  1.1517312423593138E-02_dp,&
      3.3558605291626020E-02_dp,  1.4544544223415962E-02_dp,&
      -6.0901876447184505E-02_dp,  5.0292199523917204E-02_dp,&
      -1.8965089106949350E-02_dp,  2.8601543886679452E-03_dp,&
      2.6282972200588932E+00_dp, -1.9229483513273367E+00_dp,&
      2.2722345074070921E-01_dp,  3.9046743014432933E-01_dp,&
      -1.1085109620769071E-01_dp, -1.5292292332946034E-01_dp,&
      6.1324562893261297E-02_dp,  6.4440760127347133E-02_dp,&
      -3.1375758896852156E-05_dp, -1.0943572242374154E-01_dp,&
      1.1119871011947056E-01_dp, -4.8686829057736594E-02_dp,&
      8.5187843378692653E-03_dp,  3.9756816802545427E+00_dp,&
      -3.5229306793001349E+00_dp,  8.0023523017838849E-01_dp,&
      6.7084504352855900E-01_dp, -3.5600097515911666E-01_dp,&
      -2.4559425187376815E-01_dp,  1.8405067150112531E-01_dp,&
      1.0450118269262665E-01_dp, -6.2242399538416641E-02_dp,&
      -1.5509553332024303E-01_dp,  2.0567739453273481E-01_dp,&
      -1.0262895944488672E-01_dp,  1.9845825902533719E-02_dp,&
      5.9958030217368909E+00_dp, -6.3467161270531429E+00_dp,&
      2.1437674998301484E+00_dp,  1.0109234785696470E+00_dp,&
      -9.0400327994454444E-01_dp, -3.1019661013539263E-01_dp,&
      4.4434854634858123E-01_dp,  1.3034314501804636E-01_dp,&
      -2.3283606473742849E-01_dp, -1.3205232328255015E-01_dp,&
      3.1131037411240431E-01_dp, -1.8221061043989620E-01_dp,&
      3.8754681380380226E-02_dp,  9.0676760085489132E+00_dp,&
      -1.1299033181149410E+01_dp,  5.1048832785744436E+00_dp,&
      1.2130368920639967E+00_dp, -2.0156334343903222E+00_dp,&
      -1.8774490141514191E-01_dp,  9.3144160276729693E-01_dp,&
      6.6775735953655421E-02_dp, -6.0860647626862330E-01_dp,&
      1.1967109798777947E-01_dp,  3.3525976868624457E-01_dp,&
      -2.6382031281912727E-01_dp,  6.3577702221923230E-02_dp,&
      1.3791772574427243E+01_dp, -1.9957844758745054E+01_dp,&
      1.1393907501217026E+01_dp,  6.3027745141587310E-01_dp,&
      -4.0736217099431213E+00_dp,  5.3209770609899276E-01_dp,&
      1.7196075448940431E+00_dp, -2.7410147020725067E-01_dp,&
      -1.2993291005460068E+00_dp,  9.1405021206328108E-01_dp,&
      3.9446714460368994E-02_dp, -2.6218217849092257E-01_dp,&
      8.2004458930670890E-02_dp,  2.1134043943396996E+01_dp,&
      -3.5091359042775800E+01_dp,  2.4438380342850234E+01_dp,&
      -2.5312397126433797E+00_dp, -7.4649061114523123E+00_dp,&
      2.7785338021255463E+00_dp,  2.7113234706643428E+00_dp,&
      -1.2881207709853775E+00_dp, -2.3173947170071818E+00_dp,&
      2.7571681079983406E+00_dp, -1.0669271742866862E+00_dp,&
      3.3286659884836270E-02_dp,  5.7463683876706147E-02_dp,&
      3.2679093868926657E+01_dp, -6.1590985372944964E+01_dp,&
      5.1071575868628280E+01_dp, -1.2865796070437154E+01_dp,&
      -1.1968112302623199E+01_dp,  8.4107024757715152E+00_dp,&
      3.1677115274179708E+00_dp, -3.6601777916288216E+00_dp,&
      -3.2433702877664414E+00_dp,  6.2231339219062374E+00_dp,&
      -3.8192853459283764E+00_dp,  1.0504543812661566E+00_dp,&
      -9.4464855653482102E-02_dp,  5.1112233923296436E+01_dp,&
      -1.0820984617012243E+02_dp,  1.0486168192581441E+02_dp,&
      -4.1564959865575254E+01_dp, -1.4578956048017877E+01_dp,&
      2.0463299891738025E+01_dp,  5.4796957878465291E-01_dp,&
      -8.1431617361849096E+00_dp, -2.3864407534639644E+00_dp,&
      1.1108185317660142E+01_dp, -9.0914165395682556E+00_dp,&
      3.4289059180469894E+00_dp, -5.2330290922697853E-01_dp,&
      8.1288207408222235E+01_dp, -1.9111334897720442E+02_dp,&
      2.1279120241913324E+02_dp, -1.1477244836095247E+02_dp,&
      -3.0743808414701879E+00_dp,  4.1497540322186111E+01_dp,&
      -1.1492293983396848E+01_dp, -1.3925871166929845E+01_dp,&
      4.2907496508986602E+00_dp,  1.3859380628329522E+01_dp,&
      -1.5876854948348997E+01_dp,  7.3152892639960951E+00_dp,&
      -1.3362731143808417E+00_dp,  1.3327942970575734E+02_dp,&
      -3.4300787025442804E+02_dp,  4.3047110358093215E+02_dp,&
      -2.9164886994241795E+02_dp,  5.9859717276872587E+01_dp,&
      6.4920747245308959E+01_dp, -4.3129286137280374E+01_dp,&
      -1.4066222386165592E+01_dp,  2.0329366289960053E+01_dp,&
      6.4662789320134308E+00_dp, -1.8713821361854901E+01_dp,&
      1.0879757257521486E+01_dp, -2.2809868996647227E+00_dp,&
      2.3476896298756850E+02_dp, -6.4871239939014094E+02_dp,&
      8.9470567514658705E+02_dp, -7.1469544614179904E+02_dp,&
      2.7168514298197266E+02_dp,  5.4663915078208277E+01_dp,&
      -9.8551569029646402E+01_dp,  4.4291571928112470E+00_dp,&
      4.0875449189810638E+01_dp, -1.3810716456952706E+01_dp,&
      -1.3765103917995090E+01_dp,  1.2346735490952792E+01_dp,&
      -3.0319784854436125E+00_dp,  5.2021357980075481E+02_dp,&
      -1.5047681469099909E+03_dp,  2.2056583940220830E+03_dp,&
      -1.9489466301285634E+03_dp,  9.6172634089814062E+02_dp,&
      -8.2178385250715877E+01_dp, -1.8689166188128539E+02_dp,&
      4.7992813706474685E+01_dp,  7.5507434893328252E+01_dp,&
      -5.0888531694752380E+01_dp, -6.1648337992718236E+00_dp,&
      1.6427444516897769E+01_dp, -4.7804380737887202E+00_dp,&
      1.4931443468447578E+00_dp, -5.8471071386162488E-01_dp,&
      -1.3137927495216153E-01_dp,  8.5629508847014146E-02_dp,&
      4.0460885624517218E-02_dp, -3.0854519142842370E-02_dp,&
      -1.6668874154937584E-02_dp,  1.1539311755993547E-02_dp,&
      1.8057976799045881E-02_dp, -2.2047699644401675E-02_dp,&
      7.7604920500705474E-03_dp,  4.2470957702766204E-04_dp,&
      -6.4150955581103778E-04_dp,  1.7150666636291758E+00_dp,&
      -8.5070446961324930E-01_dp, -8.2587877714910274E-02_dp,&
      1.5733665025528792E-01_dp,  2.2362433751069832E-02_dp,&
      -6.1920203471438347E-02_dp, -7.4462817802991854E-03_dp,&
      2.4762989163368353E-02_dp,  2.1594517266609357E-02_dp,&
      -4.6861787787818826E-02_dp,  3.0232817720782037E-02_dp,&
      -8.6536690927158984E-03_dp,  8.2816489219117028E-04_dp,&
      2.2012593013113242E+00_dp, -1.4708994286649284E+00_dp,&
      7.0779599888676187E-02_dp,  3.1369855773415140E-01_dp,&
      -4.0224055770866012E-02_dp, -1.2774489263868016E-01_dp,&
      2.4938170206917257E-02_dp,  5.3277305829681154E-02_dp,&
      1.9570090404201004E-02_dp, -9.7451375771407875E-02_dp,&
      8.3591537572802813E-02_dp, -3.2502927863455926E-02_dp,&
      5.0679445972248672E-03_dp,  3.0435610577413037E+00_dp,&
      -2.6439734331166180E+00_dp,  4.6643978960348892E-01_dp,&
      5.7255374553028604E-01_dp, -2.0967596109489331E-01_dp,&
      -2.2871315539930384E-01_dp,  1.1214157555126254E-01_dp,&
      9.7841452353856731E-02_dp, -1.1952067880384304E-02_dp,&
      -1.6611264833371459E-01_dp,  1.7882104458009906E-01_dp,&
      -8.0938254356085149E-02_dp,  1.4553336693307755E-02_dp,&
      4.4013915563626602E+00_dp, -4.7370367251945105E+00_dp,&
      1.3952141799394053E+00_dp,  9.3083326856233284E-01_dp,&
      -6.0770432754104109E-01_dp, -3.4146109652219514E-01_dp,&
      3.1148445412961517E-01_dp,  1.4799614852691056E-01_dp,&
      -1.2454577962361596E-01_dp, -2.1157820423498469E-01_dp,&
      3.1273896919979172E-01_dp, -1.6264093750898964E-01_dp,&
      3.2321056217098403E-02_dp,  6.5363325586350669E+00_dp,&
      -8.4002734007267286E+00_dp,  3.4516744838336901E+00_dp,&
      1.2958797798191262E+00_dp, -1.4530782665536190E+00_dp,&
      -3.7130371833416770E-01_dp,  7.1329068164973419E-01_dp,&
      1.5886534628143975E-01_dp, -4.1300823765828609E-01_dp,&
      -1.1297420382509935E-01_dp,  4.3095219716435623E-01_dp,&
      -2.7099924501127154E-01_dp,  5.9765906484496450E-02_dp,&
      9.8718409209594444E+00_dp, -1.4774703156334951E+01_dp,&
      7.8344307203439723E+00_dp,  1.3100128552371193E+00_dp,&
      -3.1029964454694561E+00_dp, -4.8425439664709267E-02_dp,&
      1.4305547840984860E+00_dp,  1.7043059862931740E-03_dp,&
      -1.0173113806737391E+00_dp,  3.9971961481904772E-01_dp,&
      3.5717334078499841E-01_dp, -3.5227483495704071E-01_dp,&
      9.0840950487899280E-02_dp,  1.5094285692399961E+01_dp,&
      -2.5863006570028976E+01_dp,  1.6941424618914212E+01_dp,&
      -8.2529211252986553E-02_dp, -6.0502265317186374E+00_dp,&
      1.2842402390008709E+00_dp,  2.5244804302560042E+00_dp,&
      -6.2891082730515735E-01_dp, -2.0682717362404395E+00_dp,&
      1.8199209168598900E+00_dp, -3.1696402449624006E-01_dp,&
      -2.4788872827770345E-01_dp,  1.0040053068940764E-01_dp,&
      2.3326532076830389E+01_dp, -4.5194032269559408E+01_dp,&
      3.5550320300834244E+01_dp, -5.6756451356893560E+00_dp,&
      -1.0703822653505913E+01_dp,  5.0650560255240329E+00_dp,&
      3.7433826066761373E+00_dp, -2.3546914919859425E+00_dp,&
      -3.4852289677497725E+00_dp,  4.8766148395113742E+00_dp,&
      -2.3683936704709061E+00_dp,  3.9477013548749840E-01_dp,&
      2.4217597532977045E-02_dp,  3.6438523276203654E+01_dp,&
      -7.9026710872235796E+01_dp,  7.3153148523645797E+01_dp,&
      -2.2447806103578166E+01_dp, -1.6400252521887804E+01_dp,&
      1.4117189879671018E+01_dp,  3.7704063367345246E+00_dp,&
      -6.2200433997860021E+00_dp, -4.4324532589543466E+00_dp,&
      1.0293341416331923E+01_dp, -7.0471929333047125E+00_dp,&
      2.2491704032299409E+00_dp, -2.7472784846345960E-01_dp,&
      5.7650008601931596E+01_dp, -1.3859335346138556E+02_dp,&
      1.4854728177346706E+02_dp, -6.7174062977350260E+01_dp,&
      -1.8000776577693308E+01_dp,  3.3025212703350597E+01_dp,&
      -1.6648355635515739E+00_dp, -1.3341370484527044E+01_dp,&
      -1.9519662111886593E+00_dp,  1.7476000515060889E+01_dp,&
      -1.5668489541211233E+01_dp,  6.3504551706926540E+00_dp,&
      -1.0447824464894551E+00_dp,  9.2889712584894980E+01_dp,&
      -2.4463928647750311E+02_dp,  2.9889577966035040E+02_dp,&
      -1.7875521526141426E+02_dp,  4.2730769050089581E+00_dp,&
      6.5596464491298477E+01_dp, -2.3416502194370231E+01_dp,&
      -2.2316628174048979E+01_dp,  1.1587124874044365E+01_dp,&
      1.9473620072500285E+01_dp, -2.6128292622546354E+01_dp,&
      1.2919004312807344E+01_dp, -2.4858470852747883E+00_dp,&
      1.5466343631749228E+02_dp, -4.3876623267221055E+02_dp,&
      5.9917637433678522E+02_dp, -4.4383926863394601E+02_dp,&
      1.1056185052763135E+02_dp,  9.9764791389111096E+01_dp,&
      -8.0352055369394023E+01_dp, -1.9622721204759706E+01_dp,&
      4.5089137417242156E+01_dp, -3.6046484399499579E+00_dp,&
      -2.4148729613563599E+01_dp,  1.6877244514385687E+01_dp,&
      -3.8474659319176436E+00_dp,  2.7677631153916957E+02_dp,&
      -8.2445521989176632E+02_dp,  1.2212144990999409E+03_dp,&
      -1.0555839720133740E+03_dp,  4.5253808804371664E+02_dp,&
      6.5091562231667240E+01_dp, -1.7662774539141580E+02_dp,&
      2.8862592004300790E+01_dp,  7.9025353206514183E+01_dp,&
      -5.7883045427627493E+01_dp,  5.0699996458841383E+00_dp,&
      9.8497860011119158E+00_dp, -3.3639190612129344E+00_dp,&
      5.9487274112930379E+02_dp, -1.7978611801015538E+03_dp,&
      2.7622843520701335E+03_dp, -2.5880461605405826E+03_dp,&
      1.4022414211295804E+03_dp, -2.0359258628022113E+02_dp,&
      -2.6091417556034258E+02_dp,  1.2051226872116960E+02_dp,&
      8.5298374145724267E+01_dp, -1.0115805307780569E+02_dp /)
    REAL(KIND=dp), DIMENSION(400), PARAMETER :: c02 = (/ &
      2.9539145031653501E+01_dp,  4.7715049407002184E+00_dp,&
      -3.2712144543127324E+00_dp,  2.2264348078485641E+03_dp,&
      -6.6288008445597088E+03_dp,  1.0084012629385183E+04_dp,&
      -9.4383385665132246E+03_dp,  5.2647790264708256E+03_dp,&
      -1.0875754774918837E+03_dp, -5.8837187049175338E+02_dp,&
      2.6613872996811381E+02_dp,  2.7202180766861471E+02_dp,&
      -2.5302823108588976E+02_dp,  3.0050455194523508E+01_dp,&
      4.2500745288190885E+01_dp, -1.5231116966455813E+01_dp,&
      2.2506815302769312E-01_dp, -4.6864684645862111E-02_dp,&
      -1.6076066373471819E-02_dp,  5.2617151773664064E-03_dp,&
      4.6066685262613890E-03_dp, -1.7521312997179688E-03_dp,&
      -1.9251133062711246E-03_dp,  5.9345994494066007E-04_dp,&
      1.4834834984639620E-03_dp, -1.0955734366602742E-03_dp,&
      -1.0027326064082555E-04_dp,  3.6348878322400217E-04_dp,&
      -1.1127160112187439E-04_dp,  7.2603005105721208E-01_dp,&
      -1.7378410728319532E-01_dp, -5.1996715280760691E-02_dp,&
      2.4016913773170116E-02_dp,  1.5560321159120889E-02_dp,&
      -8.5587123146005032E-03_dp, -6.5878194833641881E-03_dp,&
      3.0580493812781056E-03_dp,  5.8185879511744283E-03_dp,&
      -5.6506429717535292E-03_dp,  9.6145797073991931E-04_dp,&
      8.4659623403035588E-04_dp, -3.3812623825664234E-04_dp,&
      1.3950206706580586E+00_dp, -4.1818083799113709E-01_dp,&
      -9.6697550419933886E-02_dp,  7.3114311638722271E-02_dp,&
      3.0573982779230305E-02_dp, -2.7876982279685374E-02_dp,&
      -1.3064679941770251E-02_dp,  1.0479459233157986E-02_dp,&
      1.4673025417197233E-02_dp, -1.9452677781320156E-02_dp,&
      7.6812744538724814E-03_dp, -1.3354340974799377E-04_dp,&
      -4.6374652770920057E-04_dp,  2.4002817735423130E+00_dp,&
      -9.2576009390463998E-01_dp, -1.4413374515017688E-01_dp,&
      1.9374327535805080E-01_dp,  4.6522602576221726E-02_dp,&
      -7.7663309125627397E-02_dp, -1.9365836194771904E-02_dp,&
      3.0366121689986602E-02_dp,  3.1934293985556542E-02_dp,&
      -5.6570946869567498E-02_dp,  3.1245323124887436E-02_dp,&
      -6.6302180297778618E-03_dp,  9.0727048872160795E-05_dp,&
      4.0131529792607168E+00_dp, -1.9771947673658135E+00_dp,&
      -1.5401727009837554E-01_dp,  4.6736033429357465E-01_dp,&
      4.2297016009197330E-02_dp, -1.9390246830191787E-01_dp,&
      -1.3451643644637289E-02_dp,  7.8314725324450052E-02_dp,&
      6.0048855340030217E-02_dp, -1.4534657554682370E-01_dp,&
      9.8739620200257255E-02_dp, -3.0047651240179940E-02_dp,&
      3.2482467714997627E-03_dp,  6.6977661937400432E+00_dp,&
      -4.1338315742806850E+00_dp,  1.1682547993851978E-02_dp,&
      1.0522365350263854E+00_dp, -5.6267873028670708E-02_dp,&
      -4.4464162072743280E-01_dp,  4.6411192993530540E-02_dp,&
      1.8465630884986559E-01_dp,  8.9370042068134745E-02_dp,&
      -3.3628926971532436E-01_dp,  2.6985734900416353E-01_dp,&
      -9.8821388710520416E-02_dp,  1.4332394513831495E-02_dp,&
      1.1276246206952271E+01_dp, -8.5406849172019346E+00_dp,&
      7.5682719586455571E-01_dp,  2.2414673914939178E+00_dp,&
      -4.6842658114719654E-01_dp, -9.4428627475309324E-01_dp,&
      2.8194134886271960E-01_dp,  4.0121325571233513E-01_dp,&
      6.1942620093743794E-02_dp, -6.9757764732780769E-01_dp,&
      6.6046001816422062E-01_dp, -2.7625327157326457E-01_dp,&
      4.6339073107553620E-02_dp,  1.9243131938746824E+01_dp,&
      -1.7572180196348960E+01_dp,  3.1911493525192705E+00_dp,&
      4.5301719814922450E+00_dp, -1.7901890264940798E+00_dp,&
      -1.8378320927758121E+00_dp,  1.0148404640536322E+00_dp,&
      7.9197328474954454E-01_dp, -2.3336384070350974E-01_dp,&
      -1.2451080784453024E+00_dp,  1.4494114850308317E+00_dp,&
      -6.8378647123057212E-01_dp,  1.2708688064825907E-01_dp,&
      3.3380032277175935E+01_dp, -3.6241091255762910E+01_dp,&
      1.0285722844271358E+01_dp,  8.5886708142726427E+00_dp,&
      -5.5551083227948252E+00_dp, -3.1307221701362038E+00_dp,&
      3.0284928629647809E+00_dp,  1.3394612443539415E+00_dp,&
      -1.4247249996215281E+00_dp, -1.6410029693292962E+00_dp,&
      2.7598440670059219E+00_dp, -1.4960510679975429E+00_dp,&
      3.0535228441060097E-01_dp,  5.8990537925489349E+01_dp,&
      -7.5352687006393992E+01_dp,  2.9945186044685350E+01_dp,&
      1.4666282746247029E+01_dp, -1.5519933203166778E+01_dp,&
      -3.9261174027125954E+00_dp,  8.0505809846827407E+00_dp,&
      1.5262322012801828E+00_dp, -5.1007207081578763E+00_dp,&
      -2.3367163638289626E-01_dp,  4.0544900581412744E+00_dp,&
      -2.7636968296083531E+00_dp,  6.3238101915035738E-01_dp,&
      1.0645784500162654E+02_dp, -1.5877150896583035E+02_dp,&
      8.3344160108688257E+01_dp,  1.9521330509514783E+01_dp,&
      -4.0311527122326183E+01_dp,  3.5169704418224251E-01_dp,&
      1.9265087220786128E+01_dp, -1.1490927641348434E+00_dp,&
      -1.4483864243908888E+01_dp,  8.3828197319727593E+00_dp,&
      2.1044231861397189E+00_dp, -3.6152559380555696E+00_dp,&
      1.0333455730248924E+00_dp,  1.9673957337934047E+02_dp,&
      -3.4071102187005641E+02_dp,  2.2816253678927330E+02_dp,&
      3.5432086518209376E+00_dp, -9.7344029351124050E+01_dp,&
      2.7012016585447444E+01_dp,  3.9849386666885657E+01_dp,&
      -1.5594715004093725E+01_dp, -3.3275570936735846E+01_dp,&
      3.7857227723125447E+01_dp, -1.4266884105267037E+01_dp,&
      2.5646566664270998E-01_dp,  8.2958028001983131E-01_dp,&
      3.7380897364001447E+02_dp, -7.4843365384486981E+02_dp,&
      6.2534346604636403E+02_dp, -1.2011555987584157E+02_dp,&
      -2.0913981636077924E+02_dp,  1.3223097128254875E+02_dp,&
      5.8482694002845740E+01_dp, -6.7001391152085858E+01_dp,&
      -5.2620142808837230E+01_dp,  1.1084495171472038E+02_dp,&
      -7.3002889231094073E+01_dp,  2.2414543977585499E+01_dp,&
      -2.5763128135272004E+00_dp,  7.3590228308277256E+02_dp,&
      -1.6937719308092217E+03_dp,  1.7365973342231957E+03_dp,&
      -7.0191484123187968E+02_dp, -3.2398843711181541E+02_dp,&
      4.6233640099881148E+02_dp, -2.5325301041616040E+01_dp,&
      -1.9579955506646129E+02_dp,  9.2621990919570578E+00_dp,&
      2.0559952760345703E+02_dp, -2.0101605399136827E+02_dp,&
      8.5917578479674788E+01_dp, -1.4858279542741453E+01_dp,&
      1.5386988002229782E+03_dp, -4.0169102660006606E+03_dp,&
      4.9443246955768172E+03_dp, -3.0833947921566423E+03_dp,&
      2.0533986354945955E+02_dp,  1.1496536148736168E+03_dp,&
      -6.0713368333607593E+02_dp, -2.6680381334708136E+02_dp,&
      3.4114146837188082E+02_dp,  6.5931431069094316E+01_dp,&
      -2.5767149565086402E+02_dp,  1.5357507474959058E+02_dp,&
      -3.2545640706779942E+01_dp,  3.8760158209907208E+03_dp,&
      -1.1106072353977881E+04_dp,  1.5652437946556933E+04_dp,&
      -1.2734756310437266E+04_dp,  4.9118485439787391E+03_dp,&
      1.0865528975480108E+03_dp, -2.0311041115258663E+03_dp,&
      2.9284227544796153E+02_dp,  7.0832752282331637E+02_dp,&
      -3.4845714151078687E+02_dp, -1.5188466316875682E+02_dp,&
      1.8252641460497227E+02_dp, -4.7889305821155830E+01_dp,&
      1.8334270761363285E+00_dp, -3.9591024095020638E-01_dp,&
      -1.3126349325857345E-01_dp,  4.7222262371844306E-02_dp,&
      3.8076484409822732E-02_dp, -1.6052628355116935E-02_dp,&
      -1.5975826849070771E-02_dp,  5.5282813708112865E-03_dp,&
      1.2748899132651596E-02_dp, -1.0203081612794597E-02_dp,&
      -7.6057568131617672E-05_dp,  2.7891576352124907E-03_dp,&
      -8.9966226534393592E-04_dp,  2.2501471998076918E+00_dp,&
      -6.7293363442777321E-01_dp, -1.5991691179414610E-01_dp,&
      1.1671686002652161E-01_dp,  5.1331330187183055E-02_dp,&
      -4.4152248495071197E-02_dp, -2.2121001215930056E-02_dp,&
      1.6471795598964997E-02_dp,  2.3995999348520584E-02_dp,&
      -3.0506564606459299E-02_dp,  1.1201649086166804E-02_dp,&
      3.8654761046905383E-04_dp, -8.6532927608573725E-04_dp,&
      3.2106589696873193E+00_dp, -1.3748442557689724E+00_dp,&
      -1.9499364840046871E-01_dp,  3.0218577867948093E-01_dp,&
      6.7411675514350539E-02_dp, -1.2176826893310169E-01_dp,&
      -2.8771344876437632E-02_dp,  4.7632499769699695E-02_dp,&
      4.9529748656412943E-02_dp, -8.8817206722143702E-02_dp,&
      4.9444418187113308E-02_dp, -1.0655147393424186E-02_dp,&
      1.9439228710540853E-04_dp,  5.0066760706889388E+00_dp,&
      -2.8639206962938242E+00_dp, -1.6134501992474284E-01_dp,&
      7.1478932034735954E-01_dp,  4.8155111521357881E-02_dp,&
      -2.9955711646643696E-01_dp, -1.4766362863594271E-02_dp,&
      1.2140643553439820E-01_dp,  9.0761344780329142E-02_dp,&
      -2.2640716820860976E-01_dp,  1.5602634912722926E-01_dp,&
      -4.8310196750658556E-02_dp,  5.3965878202100323E-03_dp,&
      8.1953446443739981E+00_dp, -5.9031688012409029E+00_dp,&
      1.6445721911292407E-01_dp,  1.5757599912732250E+00_dp,&
      -1.2596820581475607E-01_dp, -6.7456338496403423E-01_dp,&
      8.6896320749570238E-02_dp,  2.8184553616038971E-01_dp,&
      1.2976825380286353E-01_dp, -5.1736963246432022E-01_dp,&
      4.2200984843870865E-01_dp, -1.5680233470759714E-01_dp,&
      2.3138933575817758E-02_dp,  1.3809992873162685E+01_dp,&
      -1.2073777985207707E+01_dp,  1.4046415603947400E+00_dp,&
      3.2935120871963131E+00_dp, -7.9491410275893204E-01_dp,&
      -1.4077655697256062E+00_dp,  4.6583962717584887E-01_dp,&
      6.0356328563664763E-01_dp,  7.2315826275806253E-02_dp,&
      -1.0589440789395266E+00_dp,  1.0229181937437832E+00_dp,&
      -4.3382313133258776E-01_dp,  7.3703880601393990E-02_dp,&
      2.3775983411163235E+01_dp, -2.4678311578916002E+01_dp,&
      5.2289598442606424E+00_dp,  6.5489084608638590E+00_dp,&
      -2.8658851271641992E+00_dp, -2.6940912874542859E+00_dp,&
      1.6164260715043517E+00_dp,  1.1759602756276122E+00_dp,&
      -4.1631339246830373E-01_dp, -1.8583417620069387E+00_dp,&
      2.2261661929860863E+00_dp, -1.0658414136189160E+00_dp,&
      2.0032087127389880E-01_dp,  4.1737161851743600E+01_dp,&
      -5.0724899954717543E+01_dp,  1.6048134070688963E+01_dp,&
      1.2233841848958997E+01_dp, -8.6515035875208675E+00_dp,&
      -4.4977564335736266E+00_dp,  4.7366008010596605E+00_dp,&
      1.9604112600157124E+00_dp, -2.3289929066425041E+00_dp,&
      -2.3525147200130276E+00_dp,  4.1996293132940137E+00_dp,&
      -2.3189605445392569E+00_dp,  4.7873798450047977E-01_dp,&
      7.4736777736519500E+01_dp, -1.0539402492295558E+02_dp,&
      4.5498177785332551E+01_dp,  2.0552859353942374E+01_dp,&
      -2.3801828295887351E+01_dp, -5.3984340277393645E+00_dp,&
      1.2478979390566309E+01_dp,  2.1568019965817968E+00_dp,&
      -8.2046138856525843E+00_dp,  1.3417335317994308E-01_dp,&
      6.0431791851769976E+00_dp, -4.2563589579645420E+00_dp,&
      9.8809406039481740E-01_dp,  1.3673546697572985E+02_dp,&
      -2.2238025265186792E+02_dp,  1.2456773646258677E+02_dp,&
      2.6565789009467544E+01_dp, -6.1369218065318094E+01_dp,&
      1.5783873793130154E+00_dp,  2.9847304045089373E+01_dp,&
      -2.0693059266805047E+00_dp, -2.3361145261027204E+01_dp,&
      1.4410244118512024E+01_dp,  2.4894867024678495E+00_dp,&
      -5.4517863752830902E+00_dp,  1.6052091637823835E+00_dp,&
      2.5623223867233128E+02_dp, -4.7860134044984517E+02_dp,&
      3.3730097004867315E+02_dp,  1.8218678998248234E+00_dp,&
      -1.4844387126254162E+02_dp,  4.2634582495319499E+01_dp,&
      6.2609487759576318E+01_dp, -2.4599352208490920E+01_dp,&
      -5.4743329555941450E+01_dp,  6.3294007287866073E+01_dp,&
      -2.4856415765702479E+01_dp,  1.1220010346374052E+00_dp,&
      1.2319782966587371E+00_dp,  4.9355841129397038E+02_dp,&
      -1.0553521276769141E+03_dp,  9.1745816655949625E+02_dp,&
      -1.7734721772935907E+02_dp, -3.2570604320747339E+02_dp,&
      2.0420589402118949E+02_dp,  9.8255664197738639E+01_dp,&
      -1.0685912026305078E+02_dp, -9.2991815283395937E+01_dp,&
      1.8938941289256240E+02_dp, -1.2451863070929338E+02_dp,&
      3.8321023072916731E+01_dp, -4.4243955972078108E+00_dp,&
      9.8341942206132705E+02_dp, -2.3964944718903166E+03_dp,&
      2.5343558839934922E+03_dp, -1.0068167115408598E+03_dp,&
      -5.5803706876145225E+02_dp,  7.3313067449087373E+02_dp,&
      -6.6657041466680189E-01_dp, -3.3788668166577065E+02_dp,&
      -1.7628434198318345E+01_dp,  3.9655688909902926E+02_dp,&
      -3.7408946589251713E+02_dp,  1.5787812785698458E+02_dp,&
      -2.7089362134634474E+01_dp,  2.0614418378112468E+03_dp,&
      -5.6578757894510300E+03_dp,  7.1620850676558530E+03_dp,&
      -4.4041308675995369E+03_dp, -5.1684129400340517E+01_dp,&
      2.0625349133742861E+03_dp, -9.2575644960768261E+02_dp /)
    REAL(KIND=dp), DIMENSION(400), PARAMETER :: c03 = (/ &
      -6.5703600920047631E+02_dp,  6.8515562434659739E+02_dp,&
      1.8173876061092901E+02_dp, -5.6990417291447488E+02_dp,&
      3.3635689037567329E+02_dp, -7.1542971828322536E+01_dp,&
      4.8609029109261082E+03_dp, -1.4519102949367327E+04_dp,&
      2.1140422991455973E+04_dp, -1.7456530345889521E+04_dp,&
      6.1840046223145300E+03_dp,  2.8967239834451047E+03_dp,&
      -4.1351692035042397E+03_dp,  6.5605620689472300E+02_dp,&
      1.7465620750833602E+03_dp, -1.4086350854964323E+03_dp,&
      2.9723222124582003E+02_dp,  1.1297881533760143E+02_dp,&
      -5.1706127610453933E+01_dp,  1.8037684068839269E+04_dp,&
      -5.4688608792211548E+04_dp,  8.3235695251795070E+04_dp,&
      -7.6979367072450463E+04_dp,  4.1185108453331864E+04_dp,&
      -6.1794736925636562E+03_dp, -6.9941711390058144E+03_dp,&
      3.2391970281109980E+03_dp,  1.9275654950388387E+03_dp,&
      -2.0014061535060721E+03_dp,  1.9033610113525435E+02_dp,&
      3.8478416754321637E+02_dp, -1.3375337573784293E+02_dp,&
      2.5233269038969647E-01_dp, -2.6125055124024122E-02_dp,&
      -1.5445007406931326E-02_dp,  2.2557733632006770E-03_dp,&
      4.5182814756085403E-03_dp, -1.7015400805549943E-04_dp,&
      -2.8956549151850828E-03_dp,  3.4556093573692947E-04_dp,&
      2.2860774424317346E-03_dp, -1.6142483329644956E-03_dp,&
      -2.0080309005498026E-05_dp,  4.1938671314358758E-04_dp,&
      -1.3211727515346973E-04_dp,  8.2985625449170086E-01_dp,&
      -1.0224002952564909E-01_dp, -5.7664608710827639E-02_dp,&
      1.1974872702710729E-02_dp,  1.7932222830025043E-02_dp,&
      -2.0242221071433827E-03_dp, -1.1683621416807838E-02_dp,&
      1.9133988199667928E-03_dp,  9.8961258386357637E-03_dp,&
      -8.0196005442698740E-03_dp,  1.0252959799708511E-03_dp,&
      1.3039819801815705E-03_dp, -4.7833675979909446E-04_dp,&
      1.6553598506640155E+00_dp, -2.6704534598816121E-01_dp,&
      -1.3976189982584905E-01_dp,  4.3195924781359557E-02_dp,&
      4.7199394458458561E-02_dp, -1.0781926573923930E-02_dp,&
      -3.1578308961048213E-02_dp,  7.3609046611193265E-03_dp,&
      2.9529666637054093E-02_dp, -2.8085086647952978E-02_dp,&
      7.5639805082039629E-03_dp,  1.8106942675531527E-03_dp,&
      -1.0454801887820322E-03_dp,  3.0038146571211781E+00_dp,&
      -6.4706656618816383E-01_dp, -3.0986077711226090E-01_dp,&
      1.3471966868340191E-01_dp,  1.1282630726578560E-01_dp,&
      -4.1339044226723645E-02_dp, -7.7965682549816187E-02_dp,&
      2.4795390028214060E-02_dp,  8.1004935022273516E-02_dp,&
      -8.8852693289714887E-02_dp,  3.3669485741906306E-02_dp,&
      -1.0427673934340815E-03_dp, -1.7583837649226093E-03_dp,&
      5.3661110332212125E+00_dp, -1.5177683390219008E+00_dp,&
      -6.5603757824139586E-01_dp,  3.8500786992205382E-01_dp,&
      2.5336092688609296E-01_dp, -1.3567978709745959E-01_dp,&
      -1.8192450564210702E-01_dp,  7.7309535643977162E-02_dp,&
      2.1128094777967329E-01_dp, -2.6436364174291122E-01_dp,&
      1.2387404838719954E-01_dp, -1.9643916427159049E-02_dp,&
      -1.4373481356760645E-03_dp,  9.6763664742460129E+00_dp,&
      -3.5034680326316079E+00_dp, -1.3384736160499082E+00_dp,&
      1.0464400018507101E+00_dp,  5.3683599645419833E-01_dp,&
      -4.1063751344235755E-01_dp, -4.0422596672493061E-01_dp,&
      2.3132613484214665E-01_dp,  5.3106145549846739E-01_dp,&
      -7.5794244173349756E-01_dp,  4.1567726800030025E-01_dp,&
      -9.8798754939939740E-02_dp,  5.6616607065354827E-03_dp,&
      1.7788408183604410E+01_dp, -8.0593752690254270E+00_dp,&
      -2.6302227053318825E+00_dp,  2.7712044719380975E+00_dp,&
      1.0604974174861457E+00_dp, -1.1889029862167542E+00_dp,&
      -8.5146872301255649E-01_dp,  6.8030468396528943E-01_dp,&
      1.2912221791895706E+00_dp, -2.1240738522644662E+00_dp,&
      1.3266071942599185E+00_dp, -3.9109485214509854E-01_dp,&
      4.2305366773465025E-02_dp,  3.3518413862389274E+01_dp,&
      -1.8679186314635402E+01_dp, -4.9162127064446723E+00_dp,&
      7.2666289813213654E+00_dp,  1.8606253476611085E+00_dp,&
      -3.3597721710782356E+00_dp, -1.6564498861825205E+00_dp,&
      1.9990348858286913E+00_dp,  3.0053373009755533E+00_dp,&
      -5.8494608250533311E+00_dp,  4.1063242486654916E+00_dp,&
      -1.3990589490178600E+00_dp,  1.9147802002843936E-01_dp,&
      6.5026247054793515E+01_dp, -4.4052475259993862E+01_dp,&
      -8.3815680918103901E+00_dp,  1.9083151743714176E+01_dp,&
      2.3780161434136025E+00_dp, -9.3704244036259556E+00_dp,&
      -2.7194194265982845E+00_dp,  5.9402535323345900E+00_dp,&
      6.4281786309828632E+00_dp, -1.5756775907653372E+01_dp,&
      1.2412062950466199E+01_dp, -4.7334254279650443E+00_dp,&
      7.4159825374866106E-01_dp,  1.3052226392818551E+02_dp,&
      -1.0674760116218118E+02_dp, -1.1161921902585187E+01_dp,&
      5.0609207068618112E+01_dp, -1.0533924589029899E+00_dp,&
      -2.5890748254175101E+01_dp, -2.2676107030737405E+00_dp,&
      1.7997385078036434E+01_dp,  1.0898308868992313E+01_dp,&
      -4.0625202454028170E+01_dp,  3.6416772710287425E+01_dp,&
      -1.5358584568725442E+01_dp,  2.6525693124594736E+00_dp,&
      2.7273422484799687E+02_dp, -2.6858858523224052E+02_dp,&
      -4.7885740801452348E-01_dp,  1.3627417330611453E+02_dp,&
      -2.6358992902829311E+01_dp, -7.0393695696082460E+01_dp,&
      9.9494808404702209E+00_dp,  5.5679648303730211E+01_dp,&
      2.3191677764627250E+00_dp, -9.3785981251887762E+01_dp,&
      1.0074347646251888E+02_dp, -4.7292967213979878E+01_dp,&
      8.8974333107109391E+00_dp,  5.9814256543256761E+02_dp,&
      -7.1057805327339554E+02_dp,  8.8601488863604075E+01_dp,&
      3.7264206640783578E+02_dp, -1.5551496835779633E+02_dp,&
      -1.8230942230852452E+02_dp,  8.3174041344709863E+01_dp,&
      1.7313461443852231E+02_dp, -1.1433889332027171E+02_dp,&
      -1.4732698175357447E+02_dp,  2.3885411412899521E+02_dp,&
      -1.3081884815432261E+02_dp,  2.7131160357794776E+01_dp,&
      1.3926535234572304E+03_dp, -2.0093649803688027E+03_dp,&
      5.7417596311169530E+02_dp,  1.0197340958636103E+03_dp,&
      -7.6503574679237079E+02_dp, -3.9341161249586122E+02_dp,&
      4.3789084981339414E+02_dp,  5.0339035381666918E+02_dp,&
      -8.3869418930766665E+02_dp,  2.3396482300537855E+02_dp,&
      2.9455554478541080E+02_dp, -2.6058293586631680E+02_dp,&
      6.5188966276420842E+01_dp,  3.5014744519247643E+03_dp,&
      -6.2163991386173120E+03_dp,  3.1151468986944483E+03_dp,&
      2.5849457928935071E+03_dp, -3.6379187148580500E+03_dp,&
      -1.0371922354864618E+02_dp,  1.9541413742804682E+03_dp,&
      9.4382928584982380E+02_dp, -4.1940808589968328E+03_dp,&
      3.9117942893214040E+03_dp, -1.6320017495377126E+03_dp,&
      2.1892754217913259E+02_dp,  2.8366828833340946E+01_dp,&
      9.7875448079300859E+03_dp, -2.1801596144554653E+04_dp,&
      1.7647392751717292E+04_dp,  3.0182497292630583E+03_dp,&
      -1.6860787344740485E+04_dp,  8.9519787903568704E+03_dp,&
      5.5338276194636665E+03_dp, -4.2835976536269254E+03_dp,&
      -1.0643234050261457E+04_dp,  1.9747629350798474E+04_dp,&
      -1.5091225398714651E+04_dp,  5.9161713721962315E+03_dp,&
      -9.8289027954559174E+02_dp,  3.3196247063191586E+04_dp,&
      -9.3274647246994995E+04_dp,  1.1696367892260675E+05_dp,&
      -5.7983849396112921E+04_dp, -4.0430038391972135E+04_dp,&
      8.8668357693324142E+04_dp, -5.8371126061495066E+04_dp,&
      6.0751284512953389E+03_dp,  1.0622268134494479E+04_dp,&
      3.2655837179587106E+03_dp, -1.2844528875114578E+04_dp,&
      8.2097522311486537E+03_dp, -1.8311415720327325E+03_dp,&
      2.0653637422040756E+00_dp, -2.2384945891375232E-01_dp,&
      -1.3073607242437763E-01_dp,  2.1162136800823701E-02_dp,&
      3.8890411553423945E-02_dp, -2.2306624268842003E-03_dp,&
      -2.5029344308511840E-02_dp,  3.2754819697756528E-03_dp,&
      2.0135897811441424E-02_dp, -1.4787158586580834E-02_dp,&
      4.3510596001332895E-04_dp,  3.4129326945082478E-03_dp,&
      -1.1123282044359531E-03_dp,  2.6671755379167270E+00_dp,&
      -4.2558997726367698E-01_dp, -2.2518643634186017E-01_dp,&
      6.6476925859854485E-02_dp,  7.5670871490107108E-02_dp,&
      -1.5785369201729928E-02_dp, -5.0359789910624432E-02_dp,&
      1.1050039154106801E-02_dp,  4.6262019696085968E-02_dp,&
      -4.2730599514122225E-02_dp,  1.0454611624572414E-02_dp,&
      3.4881697628749296E-03_dp, -1.7584705205137370E-03_dp,&
      4.1130012975486210E+00_dp, -9.7197400656395661E-01_dp,&
      -4.6237399676090191E-01_dp,  2.0698158854957260E-01_dp,&
      1.7154740439363861E-01_dp, -6.3194136479580720E-02_dp,&
      -1.1852187781079761E-01_dp,  3.7373697623651093E-02_dp,&
      1.2303837057702460E-01_dp, -1.3406730347811235E-01_dp,&
      5.0073347721850299E-02_dp, -1.0526296887983370E-03_dp,&
      -2.7741605641166435E-03_dp,  6.9879106304886687E+00_dp,&
      -2.2398959339089957E+00_dp, -9.5500233539439394E-01_dp,&
      5.8532810178543637E-01_dp,  3.7832389585154519E-01_dp,&
      -2.0649380273593768E-01_dp, -2.7235995790339301E-01_dp,&
      1.1589245735429003E-01_dp,  3.1782510260520691E-01_dp,&
      -3.9666849007623201E-01_dp,  1.8494292684281363E-01_dp,&
      -2.8775671938967632E-02_dp, -2.3253247744458075E-03_dp,&
      1.2504091216408819E+01_dp, -5.1133715337985244E+00_dp,&
      -1.9181898554326646E+00_dp,  1.5709030401577699E+00_dp,&
      7.9081643367877807E-01_dp, -6.1856497603188698E-01_dp,&
      -5.9852994860204733E-01_dp,  3.4380179574626718E-01_dp,&
      7.9330856385958726E-01_dp, -1.1300425774704193E+00_dp,&
      6.1780772258320593E-01_dp, -1.4576277626532783E-01_dp,&
      8.0559421794824965E-03_dp,  2.3150699771274905E+01_dp,&
      -1.1671570206269971E+01_dp, -3.7216455982735082E+00_dp,&
      4.1174047756097680E+00_dp,  1.5454379483839249E+00_dp,&
      -1.7754265965038893E+00_dp, -1.2509441410240485E+00_dp,&
      1.0031593326500263E+00_dp,  1.9232600826293296E+00_dp,&
      -3.1538645783335792E+00_dp,  1.9636054804026819E+00_dp,&
      -5.7580937729700632E-01_dp,  6.1579345420484835E-02_dp,&
      4.4147095393639411E+01_dp, -2.6922086577205693E+01_dp,&
      -6.8815493972125763E+00_dp,  1.0716119799573967E+01_dp,&
      2.6933147969167028E+00_dp, -4.9862251392598829E+00_dp,&
      -2.4265145944659903E+00_dp,  2.9279766729955137E+00_dp,&
      4.4935316525485280E+00_dp, -8.6797598171501242E+00_dp,&
      6.0658299107057587E+00_dp, -2.0551305190725611E+00_dp,&
      2.7895387328935622E-01_dp,  8.6795469321420356E+01_dp,&
      -6.3334977146765148E+01_dp, -1.1616855764666131E+01_dp,&
      2.7995473727190131E+01_dp,  3.4578165687230404E+00_dp,&
      -1.3855839284346782E+01_dp, -4.0148856187294424E+00_dp,&
      8.6500178158021424E+00_dp,  9.7804892449733156E+00_dp,&
      -2.3485823007059793E+01_dp,  1.8360707367435733E+01_dp,&
      -6.9529236408812043E+00_dp,  1.0806322167710474E+00_dp,&
      1.7656622530201321E+02_dp, -1.5328123200897573E+02_dp,&
      -1.5291064937327004E+01_dp,  7.3985834147322677E+01_dp,&
      -1.2152988380722913E+00_dp, -3.8270202581466002E+01_dp,&
      -3.6377050794632200E+00_dp,  2.6084885989267164E+01_dp,&
      1.7676198736610498E+01_dp, -6.1439500469950268E+01_dp,&
      5.4294556266848126E+01_dp, -2.2673187037308736E+01_dp,&
      3.8805401802065420E+00_dp,  3.7365419495324414E+02_dp,&
      -3.8515598503602843E+02_dp, -4.0361864625047167E-01_dp,&
      1.9885938187600482E+02_dp, -3.5770781805075885E+01_dp,&
      -1.0466970552404520E+02_dp,  1.2589120977813629E+01_dp,&
      8.0595793051374187E+01_dp,  1.1782181052160290E+01_dp,&
      -1.4784344384024041E+02_dp,  1.5365187663766355E+02_dp,&
      -7.0959472033717248E+01_dp,  1.3190525000857415E+01_dp,&
      8.2863322469379375E+02_dp, -1.0157657443516359E+03_dp,&
      1.1986988187949704E+02_dp,  5.4449676449347032E+02_dp,&
      -2.1036288129400012E+02_dp, -2.7738557943311429E+02_dp,&
      1.1136872225739332E+02_dp,  2.5324789577013900E+02_dp,&
      -1.2776987419497914E+02_dp, -2.7397753121367208E+02_dp,&
      3.8961772786829607E+02_dp, -2.0551276375773824E+02_dp,&
      4.1739997168171449E+01_dp,  1.9450907147920134E+03_dp,&
      -2.8508892351002860E+03_dp,  7.5747523129342949E+02_dp,&
      1.5070567428821669E+03_dp, -1.0217458181146775E+03_dp,&
      -6.5694865280062390E+02_dp,  5.9368339749825054E+02_dp,&
      7.7692645932202890E+02_dp, -1.0697937873179803E+03_dp,&
      3.8548984201540087E+01_dp,  6.7608544747530505E+02_dp,&
      -4.8167622236142665E+02_dp,  1.1263022376557032E+02_dp,&
      4.9024755081192952E+03_dp, -8.6785968466350550E+03_dp,&
      3.9572621953771327E+03_dp,  4.0485213527526844E+03_dp /)
    REAL(KIND=dp), DIMENSION(400), PARAMETER :: c04 = (/ &
      -4.7874743699806668E+03_dp, -8.5329386064550408E+02_dp,&
      2.7526945075798449E+03_dp,  1.9553980517436300E+03_dp,&
      -5.9289564226732664E+03_dp,  4.4771348003328458E+03_dp,&
      -1.0901273950898408E+03_dp, -3.0130386512273043E+02_dp,&
      1.6238769196873562E+02_dp,  1.3563319365162426E+04_dp,&
      -2.9459071556804978E+04_dp,  2.1056290292594003E+04_dp,&
      8.3902840919604496E+03_dp, -2.2673291511349471E+04_dp,&
      6.3284247944587814E+03_dp,  1.0923165055552865E+04_dp,&
      -4.1277480924433718E+02_dp, -2.4024037869185162E+04_dp,&
      3.2150151957564813E+04_dp, -2.0428697339350008E+04_dp,&
      6.7276912202797048E+03_dp, -9.1758726122739427E+02_dp,&
      4.2980689084572172E+04_dp, -1.1637801167987248E+05_dp,&
      1.2686014747328563E+05_dp, -2.3138864168402757E+04_dp,&
      -9.8492147420019101E+04_dp,  1.0071830242916004E+05_dp,&
      3.2911879605145018E+03_dp, -6.4873615309764507E+04_dp,&
      9.6403278074733826E+03_dp,  7.4571802297396542E+04_dp,&
      -8.6979131802506905E+04_dp,  4.3117461760700382E+04_dp,&
      -8.5450772808429192E+03_dp,  1.8880572840095649E+05_dp,&
      -6.0818896836644155E+05_dp,  9.4238894917301007E+05_dp,&
      -7.8416009400037490E+05_dp,  1.3237171408866072E+05_dp,&
      5.4779421601664904E+05_dp, -8.1171169360100222E+05_dp,&
      6.4638916088477266E+05_dp, -3.4673780482035381E+05_dp,&
      1.4009010473194942E+05_dp, -4.9102063709230773E+04_dp,&
      1.4876957124327950E+04_dp, -2.5652831322436327E+03_dp,&
      2.6121103454662853E-01_dp, -9.2614016920939818E-03_dp,&
      -8.3000893742049778E-03_dp,  2.9074778484552504E-03_dp,&
      -3.0144050099379580E-03_dp,  4.9532380283510919E-03_dp,&
      -5.6871174776962855E-03_dp,  5.6833144062626998E-03_dp,&
      -5.4644435872497057E-03_dp,  4.5424442498369442E-03_dp,&
      -2.8087470813445226E-03_dp,  1.1070902641492602E-03_dp,&
      -2.0646299963951532E-04_dp,  8.6492801066328462E-01_dp,&
      -3.6931303166526513E-02_dp, -3.2730494983751128E-02_dp,&
      1.2246168926080620E-02_dp, -1.1973281160824956E-02_dp,&
      1.9775557228632324E-02_dp, -2.2824439482019589E-02_dp,&
      2.2709199342063304E-02_dp, -2.1745218503398112E-02_dp,&
      1.8069693810495297E-02_dp, -1.1180161722380632E-02_dp,&
      4.4065971039600003E-03_dp, -8.2110935783146887E-04_dp,&
      1.7482557856955379E+00_dp, -9.9194016925639519E-02_dp,&
      -8.6451169814672935E-02_dp,  3.5464507882586600E-02_dp,&
      -3.2021249054978267E-02_dp,  5.3185104302619933E-02_dp,&
      -6.1882498472844098E-02_dp,  6.1184989135095555E-02_dp,&
      -5.8218328328857059E-02_dp,  4.8348235667150061E-02_dp,&
      -2.9946177814150546E-02_dp,  1.1804489751035784E-02_dp,&
      -2.1971644622055773E-03_dp,  3.2323943569806324E+00_dp,&
      -2.4780428150533954E-01_dp, -2.1198615319295999E-01_dp,&
      9.5537150881642635E-02_dp, -7.9829124976059013E-02_dp,&
      1.3295600640545427E-01_dp, -1.5615621853217190E-01_dp,&
      1.5340084396316397E-01_dp, -1.4489460259257397E-01_dp,&
      1.2023015115955925E-01_dp, -7.4572268679463607E-02_dp,&
      2.9408259186661307E-02_dp, -5.4683663834212944E-03_dp,&
      5.9110512696058004E+00_dp, -6.0023777314923710E-01_dp,&
      -5.0327168989784998E-01_dp,  2.4893207402559139E-01_dp,&
      -1.9361685019651018E-01_dp,  3.2197727902061518E-01_dp,&
      -3.8217882779932483E-01_dp,  3.7308039130993181E-01_dp,&
      -3.4946309083981375E-01_dp,  2.8964679258249543E-01_dp,&
      -1.7996777762962227E-01_dp,  7.1034404853642361E-02_dp,&
      -1.3199465528400126E-02_dp,  1.0956673224198525E+01_dp,&
      -1.4346595057416782E+00_dp, -1.1762540800120738E+00_dp,&
      6.3994506883957525E-01_dp, -4.6541568109437514E-01_dp,&
      7.6857524192355064E-01_dp, -9.2344838294545883E-01_dp,&
      8.9595278289335112E-01_dp, -8.3103399203641393E-01_dp,&
      6.8765431279190914E-01_dp, -4.2822296013721550E-01_dp,&
      1.6927920220984846E-01_dp, -3.1446728843768625E-02_dp,&
      2.0792596884134536E+01_dp, -3.4314401576726481E+00_dp,&
      -2.7415334111503054E+00_dp,  1.6495658618315001E+00_dp,&
      -1.1261669765784517E+00_dp,  1.8334764395621579E+00_dp,&
      -2.2345254604368661E+00_dp,  2.1552250426936013E+00_dp,&
      -1.9753669693004268E+00_dp,  1.6304657176257347E+00_dp,&
      -1.0182640362604070E+00_dp,  4.0351035593520562E-01_dp,&
      -7.4988906494427660E-02_dp,  4.0641614403913103E+01_dp,&
      -8.3157277744128777E+00_dp, -6.4427715366084595E+00_dp,&
      4.3229852088919545E+00_dp, -2.7837340360875835E+00_dp,&
      4.4241729410937412E+00_dp, -5.4829090349509961E+00_dp,&
      5.2597965380569276E+00_dp, -4.7499578296245764E+00_dp,&
      3.9051365477525373E+00_dp, -2.4477687477727605E+00_dp,&
      9.7366758168919942E-01_dp, -1.8121195058941905E-01_dp,&
      8.2279753709829350E+01_dp, -2.0658726903050777E+01_dp,&
      -1.5417087397668558E+01_dp,  1.1667289558374994E+01_dp,&
      -7.1366807342421215E+00_dp,  1.0920441396316519E+01_dp,&
      -1.3801231712035245E+01_dp,  1.3182065256871045E+01_dp,&
      -1.1682971218060507E+01_dp,  9.5429009419005446E+00_dp,&
      -6.0084889711222393E+00_dp,  2.4038519801319982E+00_dp,&
      -4.4882474577803322E-01_dp,  1.7368668276644541E+02_dp,&
      -5.3256212659671924E+01_dp, -3.7915111660268572E+01_dp,&
      3.2864991327369893E+01_dp, -1.9307015368211516E+01_dp,&
      2.7897233436619405E+01_dp, -3.6056588333988557E+01_dp,&
      3.4359603001317005E+01_dp, -2.9723444391223833E+01_dp,&
      2.4009950934101433E+01_dp, -1.5194650895871973E+01_dp,&
      6.1315866703523074E+00_dp, -1.1517999700175321E+00_dp,&
      3.8567284052451515E+02_dp, -1.4449356721851993E+02_dp,&
      -9.6726585251915836E+01_dp,  9.8143836449527910E+01_dp,&
      -5.6333493633577575E+01_dp,  7.4782940357757127E+01_dp,&
      -9.9046173142738638E+01_dp,  9.4524979091637363E+01_dp,&
      -7.9209350305708099E+01_dp,  6.2708234355782480E+01_dp,&
      -3.9861447795304208E+01_dp,  1.6288701950845873E+01_dp,&
      -3.0927002635391019E+00_dp,  9.1208802703023264E+02_dp,&
      -4.2025964940054922E+02_dp, -2.5838953620053741E+02_dp,&
      3.1708958164778818E+02_dp, -1.8269319770114419E+02_dp,&
      2.1443864190015620E+02_dp, -2.9053026229725748E+02_dp,&
      2.7958028873552331E+02_dp, -2.2455334922664062E+02_dp,&
      1.7097859906654173E+02_dp, -1.0856602341332952E+02_dp,&
      4.5132022374191045E+01_dp, -8.7247003764599960E+00_dp,&
      2.3403906153903949E+03_dp, -1.3455186670211949E+03_dp,&
      -7.2807901502605125E+02_dp,  1.1411003502760752E+03_dp,&
      -6.8879925865566565E+02_dp,  6.7936705271442918E+02_dp,&
      -9.2766071574488012E+02_dp,  9.1080489850452716E+02_dp,&
      -6.9147508025247521E+02_dp,  4.8478972066941219E+02_dp,&
      -2.9990358024065625E+02_dp,  1.2694128276053435E+02_dp,&
      -2.5248397806722892E+01_dp,  6.7163110511762943E+03_dp,&
      -4.9502119609359797E+03_dp, -2.1460672665525426E+03_dp,&
      4.7894312825043899E+03_dp, -3.2493183252210265E+03_dp,&
      2.5513263672070620E+03_dp, -3.3004059244053074E+03_dp,&
      3.3620235704706211E+03_dp, -2.3803057066797624E+03_dp,&
      1.3701475041710678E+03_dp, -7.2715726697134289E+02_dp,&
      3.0023731366800985E+02_dp, -6.1929348822443536E+01_dp,&
      2.2806657666318617E+04_dp, -2.2697692191739989E+04_dp,&
      -5.7569649161039997E+03_dp,  2.5430927220737100E+04_dp,&
      -2.2037671719779755E+04_dp,  1.3975577910469839E+04_dp,&
      -1.3428763146233510E+04_dp,  1.3846589383992001E+04_dp,&
      -9.1032979373949202E+03_dp,  2.8138129998584268E+03_dp,&
      3.1040518294611104E+02_dp, -5.4082375860751756E+02_dp,&
      1.3353967691905902E+02_dp,  1.0514267073816214E+05_dp,&
      -1.5728134173123082E+05_dp,  2.3393458473274026E+04_dp,&
      1.9191748448950576E+05_dp, -2.9755239107713493E+05_dp,&
      2.4052918017103928E+05_dp, -1.0595433575621128E+05_dp,&
      -1.7465087519767349E+04_dp,  9.6696437844672575E+04_dp,&
      -1.2300599526839080E+05_dp,  9.6573441236828832E+04_dp,&
      -4.5315525505061276E+04_dp,  9.6777412850281053E+03_dp,&
      2.1416242820167533E+00_dp, -7.9748902326817347E-02_dp,&
      -7.1261894234958337E-02_dp,  2.5408305090802094E-02_dp,&
      -2.5927397116631480E-02_dp,  4.2666546644005751E-02_dp,&
      -4.9054480275301224E-02_dp,  4.8964284685653690E-02_dp,&
      -4.7028426616405591E-02_dp,  3.9089916076761162E-02_dp,&
      -2.4174446086355380E-02_dp,  9.5283680688875664E-03_dp,&
      -1.7765574154452902E-03_dp,  2.8149194258528634E+00_dp,&
      -1.5742724721103563E-01_dp, -1.3756798805731643E-01_dp,&
      5.5658044121983691E-02_dp, -5.0812419090658374E-02_dp,&
      8.4411296062409852E-02_dp, -9.8069441497070670E-02_dp,&
      9.7050981626127075E-02_dp, -9.2449750838156014E-02_dp,&
      7.6786841095660932E-02_dp, -4.7549390881222578E-02_dp,&
      1.8741467289636684E-02_dp, -3.4887110096130883E-03_dp,&
      4.4566699723520529E+00_dp, -3.7284061867202412E-01_dp,&
      -3.1873352555549722E-01_dp,  1.4418917656863958E-01_dp,&
      -1.1991661862767018E-01_dp,  2.0013354404185627E-01_dp,&
      -2.3504848379348631E-01_dp,  2.3080624953499043E-01_dp,&
      -2.1800504811681376E-01_dp,  1.8090710775770399E-01_dp,&
      -1.1219870258910787E-01_dp,  4.4239450395016547E-02_dp,&
      -8.2247162512252506E-03_dp,  7.7935726447770772E+00_dp,&
      -8.8875668865490143E-01_dp, -7.4396707707618104E-01_dp,&
      3.7089303027758863E-01_dp, -2.8613997455148843E-01_dp,&
      4.7698560347380187E-01_dp, -5.6641616408552975E-01_dp,&
      5.5250474842806918E-01_dp, -5.1735291166967712E-01_dp,&
      4.2882674642307389E-01_dp, -2.6644201552357072E-01_dp,&
      1.0514781473582199E-01_dp, -1.9533141443740406E-02_dp,&
      1.4376697262131565E+01_dp, -2.1020595111518539E+00_dp,&
      -1.7200884547239541E+00_dp,  9.4395322212651767E-01_dp,&
      -6.8061166413961394E-01_dp,  1.1266400272816803E+00_dp,&
      -1.3545508846472831E+00_dp,  1.3130363890239167E+00_dp,&
      -1.2173504455983082E+00_dp,  1.0074175588607479E+00_dp,&
      -6.2736593117696249E-01_dp,  2.4795521216282801E-01_dp,&
      -4.6047734204784939E-02_dp,  2.7511316014784537E+01_dp,&
      -4.9898964170307982E+00_dp, -3.9785547201608060E+00_dp,&
      2.4140746181922275E+00_dp, -1.6342762543947815E+00_dp,&
      2.6673110732257239E+00_dp, -3.2532834256468250E+00_dp,&
      3.1347513281824200E+00_dp, -2.8718563783342388E+00_dp,&
      2.3708595750580499E+00_dp, -1.4807456214225743E+00_dp,&
      5.8665018395297608E-01_dp, -1.0898260449107275E-01_dp,&
      5.4436762909174121E+01_dp, -1.2032475553577383E+01_dp,&
      -9.3058204956370005E+00_dp,  6.2880148178096746E+00_dp,&
      -4.0175609932092797E+00_dp,  6.4042779737722988E+00_dp,&
      -7.9429695413364838E+00_dp,  7.6117413807947472E+00_dp,&
      -6.8720650528250387E+00_dp,  5.6519758743547346E+00_dp,&
      -3.5429970722615485E+00_dp,  1.4088855536325260E+00_dp,&
      -2.6208092423393642E-01_dp,  1.1164736368007399E+02_dp,&
      -2.9791384407380139E+01_dp, -2.2212377539439530E+01_dp,&
      1.6877037072718817E+01_dp, -1.0250572296995630E+01_dp,&
      1.5756255142999594E+01_dp, -1.9924053783286329E+01_dp,&
      1.9007981150929719E+01_dp, -1.6850401908340178E+01_dp,&
      1.3775589300147089E+01_dp, -8.6745222956516148E+00_dp,&
      3.4686490862588455E+00_dp, -6.4714752047628832E-01_dp,&
      2.3860418477390155E+02_dp, -7.6548665940035832E+01_dp,&
      -5.4555576502769235E+01_dp,  4.7223564660740450E+01_dp,&
      -2.7552539514865845E+01_dp,  4.0129145306232964E+01_dp,&
      -5.1876753274658711E+01_dp,  4.9362356501653707E+01_dp,&
      -4.2758141683395408E+01_dp,  3.4607643058962353E+01_dp,&
      -2.1907296486378485E+01_dp,  8.8320535955436519E+00_dp,&
      -1.6569481566503010E+00_dp,  5.3545799381479731E+02_dp,&
      -2.0661914588910187E+02_dp, -1.3901714342570125E+02_dp,&
      1.3960064752137760E+02_dp, -7.9455954526732498E+01_dp,&
      1.0705506450207533E+02_dp, -1.4178741954776285E+02_dp,&
      1.3500627788698125E+02_dp, -1.1350858118917807E+02_dp,&
      9.0286381762194210E+01_dp, -5.7449961163831816E+01_dp,&
      2.3438033636020098E+01_dp, -4.4397143591330375E+00_dp,&
      1.2754605500542539E+03_dp, -5.9491668930678827E+02_dp,&
      -3.7066724475406403E+02_dp,  4.4339421427221095E+02_dp,&
      -2.5185768559746347E+02_dp,  3.0387188886435422E+02_dp,&
      -4.1221355071839258E+02_dp,  3.9500349025307065E+02_dp,&
      -3.1942180687176204E+02_dp,  2.4606495659517455E+02_dp,&
      -1.5690263401286234E+02_dp,  6.5084151112193439E+01_dp,&
      -1.2528409779731549E+01_dp,  3.2752436692089077E+03_dp /)
    REAL(KIND=dp), DIMENSION(400), PARAMETER :: c05 = (/ &
      -1.8663851312283325E+03_dp, -1.0423380612707174E+03_dp,&
      1.5483806206375675E+03_dp, -9.0806872083263704E+02_dp,&
      9.4017350130709883E+02_dp, -1.2957349086372242E+03_dp,&
      1.2616460034642841E+03_dp, -9.6943667338306091E+02_dp,&
      7.0087846283955878E+02_dp, -4.4162263728305550E+02_dp,&
      1.8703907422642797E+02_dp, -3.6951030936082461E+01_dp,&
      9.2847122827431940E+03_dp, -6.5872616019199959E+03_dp,&
      -3.0955142203029591E+03_dp,  6.1516693509907564E+03_dp,&
      -3.9218013076513857E+03_dp,  3.3172151564545989E+03_dp,&
      -4.4931295716623717E+03_dp,  4.5137348900129173E+03_dp,&
      -3.2498919726863614E+03_dp,  2.0440411940202880E+03_dp,&
      -1.1971496951864885E+03_dp,  5.1322722240655889E+02_dp,&
      -1.0554920856973189E+02_dp,  3.0235039579578988E+04_dp,&
      -2.7601902396650934E+04_dp, -9.3215803184808592E+03_dp,&
      2.9388322193901040E+04_dp, -2.2102684178004485E+04_dp,&
      1.4848401915745148E+04_dp, -1.7579341720367716E+04_dp,&
      1.8492406134381723E+04_dp, -1.2397481070128109E+04_dp,&
      5.4498033753094114E+03_dp, -1.8981227282010670E+03_dp,&
      6.4138733136082431E+02_dp, -1.3956433688914467E+02_dp,&
      1.2216150610526593E+05_dp, -1.5253694884673052E+05_dp,&
      -1.6429729416520167E+04_dp,  1.8607958595858968E+05_dp,&
      -1.9196043087545509E+05_dp,  1.1475537459925326E+05_dp,&
      -7.9761919488329018E+04_dp,  7.6847623039708662E+04_dp,&
      -4.7002818213517261E+04_dp, -4.9407381312193525E+02_dp,&
      2.2584965140133190E+04_dp, -1.5141091047763175E+04_dp,&
      3.5667736604964957E+03_dp,  7.5406264750572247E+05_dp,&
      -1.4326299671747899E+06_dp,  6.2617574112644594E+05_dp,&
      1.6499953681879421E+06_dp, -3.7046203068710547E+06_dp,&
      3.8236409013257986E+06_dp, -1.7734757970713889E+06_dp,&
      -1.1979318870654979E+06_dp,  3.4208098012167024E+06_dp,&
      -3.8751620860478538E+06_dp,  2.7423423745086947E+06_dp,&
      -1.1761698637398595E+06_dp,  2.3420810572297117E+05_dp,&
      2.6324296748844861E-01_dp,  2.7730709344764889E-09_dp,&
      -1.2207869322742759E-02_dp,  1.4895462150693508E-02_dp,&
      -1.8697336568206407E-02_dp,  1.4407704487284264E-02_dp,&
      9.8927256082207531E-03_dp, -6.4832137218925986E-02_dp,&
      1.4426877307723651E-01_dp, -2.0783702334434193E-01_dp,&
      1.9989272270072017E-01_dp, -1.1694483602928235E-01_dp,&
      3.1444827999764245E-02_dp,  8.7304841863481009E-01_dp,&
      1.1103688365004590E-08_dp, -4.8894552708827893E-02_dp,&
      5.9658791963436070E-02_dp, -7.4239686035906957E-02_dp,&
      5.6121910416534672E-02_dp,  4.2622494290017153E-02_dp,&
      -2.6407945174133429E-01_dp,  5.8267974366545527E-01_dp,&
      -8.3609588478556940E-01_dp,  8.0215794742252378E-01_dp,&
      -4.6848952780524716E-01_dp,  1.2581178648852193E-01_dp,&
      1.7701371812793694E+00_dp,  3.0005295459047330E-08_dp,&
      -1.3217904746214579E-01_dp,  1.6127847054189673E-01_dp,&
      -1.9811745819523222E-01_dp,  1.4539987789906042E-01_dp,&
      1.2715376214627466E-01_dp, -7.3137864438796607E-01_dp,&
      1.5942745621335248E+00_dp, -2.2744691502925121E+00_dp,&
      2.1742633873086681E+00_dp, -1.2666391454859651E+00_dp,&
      3.3951831663787274E-01_dp,  3.2872518940714639E+00_dp,&
      7.5457088814963605E-08_dp, -3.3254988508622568E-01_dp,&
      4.0576105221431225E-01_dp, -4.9135955091474298E-01_dp,&
      3.4845147454343889E-01_dp,  3.5251584271462333E-01_dp,&
      -1.8875150259828974E+00_dp,  4.0622047380355637E+00_dp,&
      -5.7594642621988239E+00_dp,  5.4840978290474425E+00_dp,&
      -3.1859681965527034E+00_dp,  8.5223243994700304E-01_dp,&
      6.0444281533037323E+00_dp,  1.8405860589428224E-07_dp,&
      -8.1156394914969088E-01_dp,  9.9022995646443068E-01_dp,&
      -1.1807985506013581E+00_dp,  8.0545436367079470E-01_dp,&
      9.4407597393184040E-01_dp, -4.7270998413371172E+00_dp,&
      1.0041641519313384E+01_dp, -1.4145249627502430E+01_dp,&
      1.3413036931983747E+01_dp, -7.7692521878901104E+00_dp,&
      2.0736589380956820E+00_dp,  1.1276776635033665E+01_dp,&
      4.4330328247789032E-07_dp, -1.9557137822199457E+00_dp,&
      2.3862631597335069E+00_dp, -2.7970925554909138E+00_dp,&
      1.8223553162021626E+00_dp,  2.4945994656705284E+00_dp,&
      -1.1704415746788525E+01_dp,  2.4524089594280941E+01_dp,&
      -3.4305229650253693E+01_dp,  3.2381675186685143E+01_dp,&
      -1.8695316093989742E+01_dp,  4.9776584290105079E+00_dp,&
      2.1561789642603674E+01_dp,  1.0694691683032124E-06_dp,&
      -4.7211887146110669E+00_dp,  5.7605517287947574E+00_dp,&
      -6.6200439159302640E+00_dp,  4.0750308632033789E+00_dp,&
      6.6167943654334040E+00_dp, -2.9092299880744793E+01_dp,&
      6.0053394751935848E+01_dp, -8.3351989681794308E+01_dp,&
      7.8273643558272752E+01_dp, -4.5022143326781794E+01_dp,&
      1.1953365515829327E+01_dp,  4.2515726163437314E+01_dp,&
      2.6176175888860676E-06_dp, -1.1564453268637152E+01_dp,&
      1.4110338828481353E+01_dp, -1.5839917900098961E+01_dp,&
      9.0605780190336862E+00_dp,  1.7879972721762254E+01_dp,&
      -7.3580492861370900E+01_dp,  1.4939077315607656E+02_dp,&
      -2.0550794788511661E+02_dp,  1.9183345232179832E+02_dp,&
      -1.0985654847974544E+02_dp,  2.9069197364239841E+01_dp,&
      8.6965326344200449E+01_dp,  6.5794423109474356E-06_dp,&
      -2.9095311161257293E+01_dp,  3.5500532806834372E+01_dp,&
      -3.8732079625741207E+01_dp,  2.0049687270975561E+01_dp,&
      4.9909299757334580E+01_dp, -1.9182953635638975E+02_dp,&
      3.8224197279200575E+02_dp, -5.2037560304457224E+02_dp,&
      4.8229079686506594E+02_dp, -2.7472873500261977E+02_dp,&
      7.2399140112324929E+01_dp,  1.8585904474506876E+02_dp,&
      1.7201658225003901E-05_dp, -7.6160919574509762E+01_dp,&
      9.2927327352209844E+01_dp, -9.7830738138831876E+01_dp,&
      4.3762988977931663E+01_dp,  1.4604784584109120E+02_dp,&
      -5.2262953630197603E+02_dp,  1.0191335036472396E+03_dp,&
      -1.3702049146643706E+03_dp,  1.2588374043796580E+03_dp,&
      -7.1234187972917664E+02_dp,  1.8675582601848978E+02_dp,&
      4.1901695827903075E+02_dp,  4.7491082333765805E-05_dp,&
      -2.1060582176594269E+02_dp,  2.5696907210161868E+02_dp,&
      -2.5824967221804940E+02_dp,  9.0897718031470191E+01_dp,&
      4.5605578804061344E+02_dp, -1.5124879782856940E+03_dp,&
      2.8748868483526799E+03_dp, -3.8058921632187958E+03_dp,&
      3.4576924488814939E+03_dp, -1.9398311729774050E+03_dp,&
      5.0511116864969131E+02_dp,  1.0102801407739152E+03_dp,&
      1.4124779360070820E-04_dp, -6.2777550617602583E+02_dp,&
      7.6597355093601072E+02_dp, -7.2225725676902391E+02_dp,&
      1.5432026922197824E+02_dp,  1.5564202303382963E+03_dp,&
      -4.7511882745640760E+03_dp,  8.7516133649064413E+03_dp,&
      -1.1355288142397600E+04_dp,  1.0162253283370977E+04_dp,&
      -5.6337171128600294E+03_dp,  1.4529129986222329E+03_dp,&
      2.6601339384885455E+03_dp,  4.6606639202797416E-04_dp,&
      -2.0782638055180396E+03_dp,  2.5357620486883484E+03_dp,&
      -2.1748960778962787E+03_dp, -1.9622446905636050E+01_dp,&
      6.0167417094472839E+03_dp, -1.6721095387390953E+04_dp,&
      2.9566028222436140E+04_dp, -3.7299442151022238E+04_dp,&
      3.2650804589094801E+04_dp, -1.7775490007433858E+04_dp,&
      4.5156255093209511E+03_dp,  7.9225561991071781E+03_dp,&
      1.7918715932261867E-03_dp, -8.0336669286121869E+03_dp,&
      9.8020987286611507E+03_dp, -7.1547112917715904E+03_dp,&
      -3.1511198074212939E+03_dp,  2.7990647407511457E+04_dp,&
      -6.9404808605822764E+04_dp,  1.1568723250772001E+05_dp,&
      -1.3955704611726975E+05_dp,  1.1762161553204221E+05_dp,&
      -6.1959192901078532E+04_dp,  1.5294013166136992E+04_dp,&
      2.8581283113673122E+04_dp,  8.8478202618300621E-03_dp,&
      -4.0093340340633775E+04_dp,  4.8918432619865300E+04_dp,&
      -2.4727656895868771E+04_dp, -4.2695894630299779E+04_dp,&
      1.7698207404610471E+05_dp, -3.7294359948259842E+05_dp,&
      5.5833895825864724E+05_dp, -6.1205082918743347E+05_dp,&
      4.6999752306235145E+05_dp, -2.2582000828196839E+05_dp,&
      5.0916331737571825E+04_dp,  1.4954788596490613E+05_dp,&
      7.3099845234523828E-02_dp, -3.4101003337474016E+05_dp,&
      4.1605872519837652E+05_dp,  3.3781671415115379E+04_dp,&
      -9.6249467433489673E+05_dp,  2.0843532691271894E+06_dp,&
      -2.8167171929273452E+06_dp,  2.4303395386896660E+06_dp,&
      -8.2647776569912373E+05_dp, -8.1568634633921518E+05_dp,&
      1.1303807273279666E+06_dp, -4.3211741616069851E+05_dp,&
      2.1591311325597387E+00_dp,  2.3904566691105513E-08_dp,&
      -1.0524216061902127E-01_dp,  1.2841147400320774E-01_dp,&
      -1.6081987966006114E-01_dp,  1.2330752066629211E-01_dp,&
      8.6989463657353730E-02_dp, -5.6142224268033980E-01_dp,&
      1.2464929077820950E+00_dp, -1.7938392888269015E+00_dp,&
      1.7241491969739160E+00_dp, -1.0082371381189261E+00_dp,&
      2.7101096999794005E-01_dp,  2.8496287705980246E+00_dp,&
      4.7574479365230764E-08_dp, -2.0956063301401973E-01_dp,&
      2.5569575239259368E-01_dp, -3.1475928439865386E-01_dp,&
      2.3213358106314164E-01_dp,  1.9858313580613890E-01_dp,&
      -1.1552055679733595E+00_dp,  2.5229914218962830E+00_dp,&
      -3.6027621950380175E+00_dp,  3.4460540887413100E+00_dp,&
      -2.0083606425817289E+00_dp,  5.3849845894628512E-01_dp,&
      4.5392171215693038E+00_dp,  1.1355602415595152E-07_dp,&
      -5.0046032814876074E-01_dp,  6.1063713098642736E-01_dp,&
      -7.3915636311573130E-01_dp,  5.2365825674300659E-01_dp,&
      5.3203716081063890E-01_dp, -2.8430782160250718E+00_dp,&
      6.1165660979115399E+00_dp, -8.6707645013669357E+00_dp,&
      8.2553871441875692E+00_dp, -4.7956128548962340E+00_dp,&
      1.2827415834620399E+00_dp,  7.9911174357421331E+00_dp,&
      2.7267735672504561E-07_dp, -1.2023389067054850E+00_dp,&
      1.4670340742178860E+00_dp, -1.7474593675113077E+00_dp,&
      1.1886260371952238E+00_dp,  1.4078415190225411E+00_dp,&
      -7.0174340690018511E+00_dp,  1.4893651164758403E+01_dp,&
      -2.0970955209319627E+01_dp,  1.9879948538498052E+01_dp,&
      -1.1512860614973324E+01_dp,  3.0724084595660521E+00_dp,&
      1.4845870303287574E+01_dp,  6.4993344169887501E-07_dp,&
      -2.8673891829941871E+00_dp,  3.4986433007173297E+00_dp,&
      -4.0957052042130364E+00_dp,  2.6589353154739181E+00_dp,&
      3.6828533437700273E+00_dp, -1.7199574435158372E+01_dp,&
      3.6002283622712199E+01_dp, -5.0336430093192462E+01_dp,&
      4.7498896436592311E+01_dp, -2.7416944125665005E+01_dp,&
      7.2985761158201008E+00_dp,  2.8630233686196082E+01_dp,&
      1.5561676463892027E-06_dp, -6.8699281572824669E+00_dp,&
      8.3823328274713553E+00_dp, -9.6205476983389229E+00_dp,&
      5.8991888169639166E+00_dp,  9.6885900913409895E+00_dp,&
      -4.2426673846413735E+01_dp,  8.7497348366870099E+01_dp,&
      -1.2138586087506182E+02_dp,  1.1395537051718760E+02_dp,&
      -6.5531527032459493E+01_dp,  1.7395737984478455E+01_dp,&
      5.7149278573182144E+01_dp,  3.7895280519993520E-06_dp,&
      -1.6742161277774240E+01_dp,  2.0427905916032042E+01_dp,&
      -2.2908625009614020E+01_dp,  1.3060369342628176E+01_dp,&
      2.6002569703236279E+01_dp, -1.0671561128972567E+02_dp,&
      2.1652100588293317E+02_dp, -2.9775421501190749E+02_dp,&
      2.7788090034510896E+02_dp, -1.5910812572414045E+02_dp,&
      4.2096728824521215E+01_dp,  1.1840509257693286E+02_dp,&
      9.4900573035783242E-06_dp, -4.1965931868056849E+01_dp,&
      5.1204572200420344E+01_dp, -5.5852949973774983E+01_dp,&
      2.8888133500973176E+01_dp,  7.2090364616085481E+01_dp,&
      -2.7692734999520661E+02_dp,  5.5175378925874941E+02_dp,&
      -7.5112695001410100E+02_dp,  6.9615211100672354E+02_dp,&
      -3.9655272221580424E+02_dp,  1.0450396601481178E+02_dp,&
      2.5609655068148811E+02_dp,  2.4715305054824862E-05_dp,&
      -1.0941937632957631E+02_dp,  1.3350746185992543E+02_dp,&
      -1.4075541179078257E+02_dp,  6.3373447021950305E+01_dp,&
      2.0911980757912349E+02_dp, -7.5029290516763729E+02_dp,&
      1.4644128290940855E+03_dp, -1.9700135465819471E+03_dp,&
      1.8106713348643566E+03_dp, -1.0249522673245622E+03_dp,&
      2.6878483032699012E+02_dp,  5.8309882443897629E+02_dp,&
      6.7807520171925752E-05_dp, -3.0064117576976992E+02_dp,&
      3.6682509804999006E+02_dp, -3.7040717157728307E+02_dp,&
      1.3406424149107912E+02_dp,  6.4428398410363116E+02_dp,&
      -2.1519786183493156E+03_dp,  4.1011295900086225E+03_dp,&
      -5.4383331157833045E+03_dp,  4.9469571799236483E+03_dp /)
    REAL(KIND=dp), DIMENSION(80), PARAMETER :: c06 = (/ &
      -2.7780629763055545E+03_dp,  7.2394748688135314E+02_dp,&
      1.4141872703259592E+03_dp,  1.9924393057961005E-04_dp,&
      -8.8514298674834572E+02_dp,  1.0799982553752052E+03_dp,&
      -1.0299886964497700E+03_dp,  2.4613962885626509E+02_dp,&
      2.1496502911088132E+03_dp, -6.6513723885247055E+03_dp,&
      1.2319717518748630E+04_dp, -1.6044225972732615E+04_dp,&
      1.4399549815435170E+04_dp, -8.0011168005279378E+03_dp,&
      2.0673302306451947E+03_dp,  3.7169273499830429E+03_dp,&
      6.4174536681622018E-04_dp, -2.8588922904614119E+03_dp,&
      3.4882377284724071E+03_dp, -3.0697095638713813E+03_dp,&
      1.6426883687990320E+02_dp,  7.9840717598249521E+03_dp,&
      -2.2710590528982131E+04_dp,  4.0596610232320745E+04_dp,&
      -5.1616663987160551E+04_dp,  4.5469685184816553E+04_dp,&
      -2.4885092322884550E+04_dp,  6.3498617053543667E+03_dp,&
      1.0875832953021774E+04_dp,  2.3481599355844865E-03_dp,&
      -1.0504850853833614E+04_dp,  1.2817288640026694E+04_dp,&
      -9.9607576780561449E+03_dp, -2.6337897159272597E+03_dp,&
      3.4451262132612253E+04_dp, -8.8941818154280394E+04_dp,&
      1.5161895422610914E+05_dp, -1.8618139413710040E+05_dp,&
      1.5936755599614693E+05_dp, -8.5114171194832204E+04_dp,&
      2.1268239024390470E+04_dp,  3.7112755999414359E+04_dp,&
      1.0383653918374779E-02_dp, -4.6784795849598871E+04_dp,&
      5.7083114992325580E+04_dp, -3.5339178754296212E+04_dp,&
      -3.3888742049394437E+04_dp,  1.8605716164815886E+05_dp,&
      -4.2519544494572078E+05_dp,  6.7441504677537933E+05_dp,&
      -7.8031535165577661E+05_dp,  6.3288379982644250E+05_dp,&
      -3.2161368971558253E+05_dp,  7.6774261966426260E+04_dp,&
      1.6235209876331326E+05_dp,  6.3058311437979311E-02_dp,&
      -2.8828244204753474E+05_dp,  3.5173407478189125E+05_dp,&
      -1.1586186796904098E+05_dp, -4.5916736844352842E+05_dp,&
      1.4672512106301824E+06_dp, -2.7737577271967172E+06_dp,&
      3.7900648353620353E+06_dp, -3.7631762994996398E+06_dp,&
      2.5690955686708130E+06_dp, -1.0676056448933431E+06_dp,&
      2.0022423762086249E+05_dp,  1.1909890543833333E+06_dp,&
      7.5897543064246109E-01_dp, -3.6014432096696794E+06_dp,&
      4.3939694879051335E+06_dp,  2.2270474259830317E+06_dp,&
      -1.4748142024036435E+07_dp,  2.4772021603756029E+07_dp,&
      -2.0427694973111704E+07_dp, -5.1425641553283399E+06_dp,&
      4.1035528362137891E+07_dp, -5.8536317653180525E+07_dp,&
      4.1604269769282125E+07_dp, -1.2580153353393231E+07_dp /)
    REAL(KIND=dp), DIMENSION(13, 32, 5), PARAMETER :: &
      coefdata = RESHAPE ((/ c01, c02, c03, c04, c05, c06 /),(/13,32,5/))

    INTEGER                                  :: irange

     IF(Rc <=  2.0000000000000000E+01_dp) Rc =   2.0000000000000000E+01_dp
     IF(Rc >=  7.9661361694335938E+05_dp) Rc =   7.9661361694335938E+05_dp
     IF(Rc >=  2.0000000000000000E+01_dp .AND. Rc <=  2.4398919835450684E+02_dp) THEN
       irange =  1
       L_b =  2.0000000000000000E+01_dp
       U_b =  2.4398919835450684E+02_dp
     END IF
     IF(Rc >=  2.4398919835450684E+02_dp .AND. Rc <=  2.9062022203852043E+03_dp) THEN
       irange =  2
       L_b =  2.4398919835450684E+02_dp
       U_b =  2.9062022203852043E+03_dp
     END IF
     IF(Rc >=  2.9062022203852043E+03_dp .AND. Rc <=  4.6200345281597729E+04_dp) THEN
       irange =  3
       L_b =  2.9062022203852043E+03_dp
       U_b =  4.6200345281597729E+04_dp
     END IF
     IF(Rc >=  4.6200345281597729E+04_dp .AND. Rc <=  2.7305247284985101E+05_dp) THEN
       irange =  4
       L_b =  4.6200345281597729E+04_dp
       U_b =  2.7305247284985101E+05_dp
     END IF
     IF(Rc >=  2.7305247284985101E+05_dp .AND. Rc <=  7.9661361694335938E+05_dp) THEN
       irange =  5
       L_b =  2.7305247284985101E+05_dp
       U_b =  7.9661361694335938E+05_dp
     END IF

     fit_coef(1:13,1:32) = coefdata(1:13,1:32,irange)

   END SUBROUTINE get_fit_coef_k16

! *****************************************************************************
!> \brief ...
!> \param Rc ...
!> \param L_b ...
!> \param U_b ...
!> \param fit_coef ...
! *****************************************************************************
   SUBROUTINE get_fit_coef_k17(Rc,L_b,U_b,fit_coef)
    REAL(KIND=dp)                            :: Rc, L_b, U_b
    REAL(KIND=dp), DIMENSION(13, 34)         :: fit_coef

    REAL(KIND=dp), DIMENSION(252), PARAMETER :: c07 = (/ &
      1.8370815219029825E+05_dp, -2.2571407833669311E+05_dp,&
      1.9425009817913763E+05_dp, -1.0472915274294355E+05_dp,&
      2.6504347685948280E+04_dp,  4.7817002324082285E+04_dp,&
      2.5450661362359097E-02_dp, -6.6225824542394374E+04_dp,&
      8.1048394804074065E+04_dp, -4.3435260565217926E+04_dp,&
      -6.1476285747099682E+04_dp,  2.7171168546155246E+05_dp,&
      -5.8303836555695033E+05_dp,  8.8719630466380890E+05_dp,&
      -9.9159773972621036E+05_dp,  7.7951289244560874E+05_dp,&
      -3.8487843289358675E+05_dp,  8.9463560970755076E+04_dp,&
      2.5019631140224831E+05_dp,  2.1264619593581804E-01_dp,&
      -5.5981397809465544E+05_dp,  6.8509542672483542E+05_dp,&
      2.1284725467874094E+04_dp, -1.4753034153249892E+06_dp,&
      3.2502583758630129E+06_dp, -4.4832573914568834E+06_dp,&
      4.0445406981888828E+06_dp, -1.6710630248330070E+06_dp,&
      -9.4341987280607189E+05_dp,  1.6221639053145065E+06_dp,&
      -6.5652039505004522E+05_dp,  2.0948115699131753E+00_dp,&
      3.6258494519234841E-08_dp, -9.3232132132602369E-02_dp,&
      1.1410199411295747E-01_dp, -1.4480521511281780E-01_dp,&
      1.1917207481518551E-01_dp,  4.9241583812778933E-02_dp,&
      -4.4186107155067733E-01_dp,  1.0258906798051275E+00_dp,&
      -1.5172966271163439E+00_dp,  1.4945236952314433E+00_dp,&
      -8.9549977851517348E-01_dp,  2.4671541335739663E-01_dp,&
      2.7235696297903660E+00_dp,  6.9927670959677010E-08_dp,&
      -1.7984023158039161E-01_dp,  2.2009708683611603E-01_dp,&
      -2.7519253143108352E-01_dp,  2.1973689951913447E-01_dp,&
      1.1430694800789225E-01_dp, -8.8115350487381972E-01_dp,&
      2.0114407245057633E+00_dp, -2.9526094256716577E+00_dp,&
      2.8950743978986537E+00_dp, -1.7292996895200985E+00_dp,&
      4.7535789164424885E-01_dp,  4.2427066521500594E+00_dp,&
      1.6193093735503707E-07_dp, -4.1653266916325560E-01_dp,&
      5.0977241977686150E-01_dp, -6.2812103646832995E-01_dp,&
      4.8620027382563846E-01_dp,  3.0785581451135241E-01_dp,&
      -2.1047443629858482E+00_dp,  4.7300914686869833E+00_dp,&
      -6.8940188870393388E+00_dp,  6.7301912758278446E+00_dp,&
      -4.0080099252645738E+00_dp,  1.0993200811710275E+00_dp,&
      7.2887750366020878E+00_dp,  3.7798556209097436E-07_dp,&
      -9.7246230665416322E-01_dp,  1.1901449948154916E+00_dp,&
      -1.4463642323703936E+00_dp,  1.0857865684868881E+00_dp,&
      8.1159807200523648E-01_dp, -5.0502163134081739E+00_dp,&
      1.1193077721641433E+01_dp, -1.6208047074016658E+01_dp,&
      1.5759089654312840E+01_dp, -9.3586250334120251E+00_dp,&
      2.5616011278036979E+00_dp,  1.3195622118541367E+01_dp,&
      8.7449260204547286E-07_dp, -2.2502894227249288E+00_dp,&
      2.7540086220032598E+00_dp, -3.2983231545819116E+00_dp,&
      2.3932042656303105E+00_dp,  2.1010585628596576E+00_dp,&
      -1.2010575120676899E+01_dp,  2.6251583881954033E+01_dp,&
      -3.7760474141713388E+01_dp,  3.6560400169648275E+01_dp,&
      -2.1647605107930151E+01_dp,  5.9123811733713181E+00_dp,&
      2.4741076971735783E+01_dp,  2.0256543020120554E-06_dp,&
      -5.2136786706623308E+00_dp,  6.3807388758373085E+00_dp,&
      -7.5175090758348935E+00_dp,  5.2393521484233307E+00_dp,&
      5.4345512047312257E+00_dp, -2.8642817974261821E+01_dp,&
      6.1688125010087411E+01_dp, -8.8091580505469821E+01_dp,&
      8.4897604321769748E+01_dp, -5.0103611036029179E+01_dp,&
      1.3650920716373861E+01_dp,  4.7846812164549789E+01_dp,&
      4.7505238242518463E-06_dp, -1.2230201238224922E+01_dp,&
      1.4967872450044803E+01_dp, -1.7304693765421220E+01_dp,&
      1.1480222650031589E+01_dp,  1.4238881963448762E+01_dp,&
      -6.9310801973768122E+01_dp,  1.4691230332041425E+02_dp,&
      -2.0810906978525890E+02_dp,  1.9951727876211754E+02_dp,&
      -1.1730830010948584E+02_dp,  3.1871643718694187E+01_dp,&
      9.5584382658373627E+01_dp,  1.1389673415432144E-05_dp,&
      -2.9331774923459527E+01_dp,  3.5897526849215701E+01_dp,&
      -4.0593292635133999E+01_dp,  2.5300639865679354E+01_dp,&
      3.8217190441467515E+01_dp, -1.7193734266608891E+02_dp,&
      3.5812529431878522E+02_dp, -5.0270766323901529E+02_dp,&
      4.7906560065789068E+02_dp, -2.8044896737160479E+02_dp,&
      7.5945642333907642E+01_dp,  1.9807989409896766E+02_dp,&
      2.8179303689685462E-05_dp, -7.2597490992070050E+01_dp,&
      8.8847958009412466E+01_dp, -9.7843279644225888E+01_dp,&
      5.6165358634741970E+01_dp,  1.0621909187312551E+02_dp,&
      -4.4160242824475296E+02_dp,  9.0211884406401828E+02_dp,&
      -1.2531534970604021E+03_dp,  1.1858384963678866E+03_dp,&
      -6.9062251088863718E+02_dp,  1.8628549560366449E+02_dp,&
      4.2844728563604173E+02_dp,  7.2680619722386190E-05_dp,&
      -1.8733325328642434E+02_dp,  2.2926633331658493E+02_dp,&
      -2.4440219730170580E+02_dp,  1.2508379111590767E+02_dp,&
      3.0936543510420876E+02_dp, -1.1871960018194495E+03_dp,&
      2.3726584254496997E+03_dp, -3.2557421748838055E+03_dp,&
      3.0549203135871107E+03_dp, -1.7679853070233212E+03_dp,&
      4.7457670212220614E+02_dp,  9.7553162024702920E+02_dp,&
      1.9784461526304437E-04_dp, -5.1025103218277997E+02_dp,&
      6.2446593588851465E+02_dp, -6.3883562219794624E+02_dp,&
      2.7468433715161183E+02_dp,  9.5798846328704099E+02_dp,&
      -3.3853330597180307E+03_dp,  6.5968917599317774E+03_dp,&
      -8.9193738742063597E+03_dp,  8.2819962181539267E+03_dp,&
      -4.7550782366077383E+03_dp,  1.2684749997005017E+03_dp,&
      2.3659593721200058E+03_dp,  5.7767824463795267E-04_dp,&
      -1.4910686829640895E+03_dp,  1.8248273467436229E+03_dp,&
      -1.7679369070925552E+03_dp,  5.5958786970336223E+02_dp,&
      3.2150931773552397E+03_dp, -1.0419368658525378E+04_dp,&
      1.9704230885199639E+04_dp, -2.6153107469601153E+04_dp,&
      2.3957244240913482E+04_dp, -1.3610636015655184E+04_dp,&
      3.6004270680947861E+03_dp,  6.2184859119400480E+03_dp,&
      1.8515069283753457E-03_dp, -4.7844525772745565E+03_dp,&
      5.8553838083439105E+03_dp, -5.2557820977780702E+03_dp,&
      7.6998584833558868E+02_dp,  1.2019149643632090E+04_dp,&
      -3.5476725058178665E+04_dp,  6.4648163585761089E+04_dp,&
      -8.3736358762410033E+04_dp,  7.5284898688267742E+04_dp,&
      -4.2133475096368573E+04_dp,  1.1009804491050614E+04_dp,&
      1.8195464521428708E+04_dp,  6.7517479161416302E-03_dp,&
      -1.7477227521677822E+04_dp,  2.1389179519798992E+04_dp,&
      -1.7071068989116044E+04_dp, -2.4214225086274514E+03_dp,&
      5.2234983048949362E+04_dp, -1.3874773753806672E+05_dp,&
      2.4073094092437049E+05_dp, -3.0103520143692737E+05_dp,&
      2.6302584995883796E+05_dp, -1.4370762781981824E+05_dp,&
      3.6795383161880185E+04_dp,  6.2090310305363819E+04_dp,&
      2.9820864621100219E-02_dp, -7.7417968024254675E+04_dp,&
      9.4745982916105684E+04_dp, -6.1136770748532195E+04_dp,&
      -4.6361387820341610E+04_dp,  2.8437287677000294E+05_dp,&
      -6.6357048964616924E+05_dp,  1.0697148236834274E+06_dp,&
      -1.2605114585738920E+06_dp,  1.0444647787696669E+06_dp,&
      -5.4377630601913563E+05_dp,  1.3327154055909728E+05_dp,&
      2.7161799040869653E+05_dp,  1.8172876846851641E-01_dp,&
      -4.7458463411040365E+05_dp,  5.8080137302666670E+05_dp,&
      -2.1206348195483550E+05_dp, -6.8472287525603513E+05_dp,&
      2.2646124682806111E+06_dp, -4.3463120374242561E+06_dp,&
      6.0384830132842269E+06_dp, -6.1288439298421443E+06_dp,&
      4.3074969014170449E+06_dp, -1.8594197839029063E+06_dp,&
      3.6681152123045758E+05_dp,  1.9925461761031416E+06_dp,&
      2.2253842343468082E+00_dp, -5.8978754757524757E+06_dp,&
      7.2176798918735152E+06_dp,  3.1998615207433747E+06_dp,&
      -2.2851538821395889E+07_dp,  3.8978847055747673E+07_dp,&
      -3.3766691905692071E+07_dp, -3.6817494449235639E+06_dp,&
      5.8417659409558006E+07_dp, -8.7460434017794341E+07_dp,&
      6.4007727007001713E+07_dp, -1.9852653644652657E+07_dp /)
    REAL(KIND=dp), DIMENSION(400), PARAMETER :: c01 = (/ &
      1.3076079314075767E-01_dp, -3.1204465976027999E-02_dp,&
      -1.7122144125202067E-03_dp,  7.1244719472004776E-04_dp,&
      9.7914036643084913E-05_dp, -1.6905608998459976E-05_dp,&
      -7.4913349592830019E-05_dp,  3.9524844540922514E-04_dp,&
      -1.3207548101866621E-03_dp,  2.4893343556511901E-03_dp,&
      -1.7630401832362664E-03_dp, -1.5303430582117692E-03_dp,&
      2.4954212764800787E-03_dp,  4.0176042278607560E-01_dp,&
      -1.0052269981409805E-01_dp, -3.7752999908446687E-03_dp,&
      2.3355451575862705E-03_dp,  2.2164896522723778E-04_dp,&
      -5.9872225699415800E-05_dp, -2.3417557017388007E-04_dp,&
      1.2627576376189641E-03_dp, -4.2195733035849375E-03_dp,&
      7.9523531821684996E-03_dp, -5.6322897078267794E-03_dp,&
      -4.8887888709380819E-03_dp,  7.9720806192536044E-03_dp,&
      7.0203193009585230E-01_dp, -1.9160546782655125E-01_dp,&
      -1.2596497264163648E-03_dp,  4.4901354476166479E-03_dp,&
      1.0663595310664382E-04_dp, -1.2452948785005740E-04_dp,&
      -4.2231891451488128E-04_dp,  2.3703226650564047E-03_dp,&
      -7.9222542238009102E-03_dp,  1.4928487774690918E-02_dp,&
      -1.0573462306857859E-02_dp, -9.1774203746157711E-03_dp,&
      1.4966211175066604E-02_dp,  1.0539449282453617E+00_dp,&
      -3.2251390565619303E-01_dp,  1.0615457224673873E-02_dp,&
      7.2905132110783599E-03_dp, -4.7936555312612936E-04_dp,&
      -2.0448213231640043E-04_dp, -6.6082699844020952E-04_dp,&
      3.9081706133278242E-03_dp, -1.3071632605714272E-02_dp,&
      2.4627501045524507E-02_dp, -1.7442957122418997E-02_dp,&
      -1.5139885809291837E-02_dp,  2.4690040632825656E-02_dp,&
      1.4843975248485286E+00_dp, -5.1701819383514402E-01_dp,&
      3.9543265998243871E-02_dp,  1.0365663424557522E-02_dp,&
      -1.8681910570608855E-03_dp, -2.5533490666031401E-04_dp,&
      -9.7612040167318833E-04_dp,  6.1129491849905623E-03_dp,&
      -2.0475177295793314E-02_dp,  3.8568511251337811E-02_dp,&
      -2.7314928846016738E-02_dp, -2.3709957461900095E-02_dp,&
      3.8663702667761222E-02_dp,  2.0272052023813556E+00_dp,&
      -8.0777194692517917E-01_dp,  9.8050801990368502E-02_dp,&
      1.2353931861675717E-02_dp, -4.4992921000651621E-03_dp,&
      -1.6348775215867598E-04_dp, -1.4060441185015371E-03_dp,&
      9.2907069709225621E-03_dp, -3.1187261758263946E-02_dp,&
      5.8734750953777816E-02_dp, -4.1589834778631818E-02_dp,&
      -3.6106670683926118E-02_dp,  5.8868554322763382E-02_dp,&
      2.7262530833535337E+00_dp, -1.2406060229341269E+00_dp,&
      2.0638055327688459E-01_dp,  9.9642442503349842E-03_dp,&
      -8.8630281922776159E-03_dp,  3.0333232987031180E-04_dp,&
      -2.0137745161970229E-03_dp,  1.3848156069161642E-02_dp,&
      -4.6621551236882400E-02_dp,  8.7785867514915933E-02_dp,&
      -6.2143870434135301E-02_dp, -5.3964809521018703E-02_dp,&
      8.7958357674071022E-02_dp,  3.6399268594035759E+00_dp,&
      -1.8805467044522048E+00_dp,  3.9684620223953421E-01_dp,&
      -3.6979224560712339E-03_dp, -1.5290732711395427E-02_dp,&
      1.5650362097787600E-03_dp, -2.9164870249643996E-03_dp,&
      2.0338155085769950E-02_dp, -6.8712559932418329E-02_dp,&
      1.2936460946821179E-01_dp, -9.1545670622710518E-02_dp,&
      -7.9523365645068900E-02_dp,  1.2956570819945470E-01_dp,&
      4.8477396000212751E+00_dp, -2.8204823695808767E+00_dp,&
      7.2037192283678975E-01_dp, -4.2006232839993926E-02_dp,&
      -2.3408946961019392E-02_dp,  4.3011253069062654E-03_dp,&
      -4.3418036026321299E-03_dp,  2.9528398299157512E-02_dp,&
      -1.0014893969272984E-01_dp,  1.8854172213152740E-01_dp,&
      -1.3337119045002854E-01_dp, -1.1589897868070902E-01_dp,&
      1.8874729813192132E-01_dp,  6.4610001308210583E+00_dp,&
      -4.1941537938416324E+00_dp,  1.2562579246714793E+00_dp,&
      -1.2956693816744225E-01_dp, -3.0923800259146223E-02_dp,&
      9.4582394952995325E-03_dp, -6.7291628461492317E-03_dp,&
      4.2513667531429149E-02_dp, -1.4472119835853223E-01_dp,&
      2.7248024980715652E-01_dp, -1.9267411822422414E-01_dp,&
      -1.6749461934299184E-01_dp,  2.7264883596712303E-01_dp,&
      8.6416370834085008E+00_dp, -6.1970932928169926E+00_dp,&
      2.1269516693256731E+00_dp, -3.0975515701583661E-01_dp,&
      -3.1212680848427189E-02_dp,  1.8014526702952737E-02_dp,&
      -1.0880146572029471E-02_dp,  6.0921073275536620E-02_dp,&
      -2.0789469712183004E-01_dp,  3.9152299938109048E-01_dp,&
      -2.7675631174570009E-01_dp, -2.4066959573985081E-01_dp,&
      3.9159410702887432E-01_dp,  1.1639225306345036E+01_dp,&
      -9.1246532821522290E+00_dp,  3.5218372776759885E+00_dp,&
      -6.5629264554887390E-01_dp, -9.0889294350904488E-03_dp,&
      3.0161449462820341E-02_dp, -1.8099973853997281E-02_dp,&
      8.7313637818274087E-02_dp, -2.9790804111642638E-01_dp,&
      5.6124720781308091E-01_dp, -3.9662215564129433E-01_dp,&
      -3.4500609303341645E-01_dp,  5.6114294060164027E-01_dp,&
      1.5873255748499172E+01_dp, -1.3453102441637940E+01_dp,&
      5.7419478039571841E+00_dp, -1.2916346325142669E+00_dp,&
      6.5759417771617396E-02_dp,  4.3530054327603068E-02_dp,&
      -3.0166292171774273E-02_dp,  1.2600955452173773E-01_dp,&
      -4.2822600401379651E-01_dp,  8.0701621705779869E-01_dp,&
      -5.7019201205676040E-01_dp, -4.9611094674881584E-01_dp,&
      8.0664222793806906E-01_dp,  2.2143206347171624E+01_dp,&
      -2.0046617832223532E+01_dp,  9.3071997135744269E+00_dp,&
      -2.4230046225771531E+00_dp,  2.4836527841931291E-01_dp,&
      5.0611968862569114E-02_dp, -4.9048621468263262E-02_dp,&
      1.8493192924859195E-01_dp, -6.2385283643435641E-01_dp,&
      1.1757719724259648E+00_dp, -8.3061767393793995E-01_dp,&
      -7.2286268514831931E-01_dp,  1.1749965024749465E+00_dp,&
      3.2270336161918614E+01_dp, -3.0801175619216782E+01_dp,&
      1.5292207073605903E+01_dp, -4.4474171161113727E+00_dp,&
      6.3789574241791069E-01_dp,  3.6622362193611202E-02_dp,&
      -7.7335776114862156E-02_dp,  2.8112010508678736E-01_dp,&
      -9.4070495460370629E-01_dp,  1.7725454560696385E+00_dp,&
      -1.2520539024975985E+00_dp, -1.0898461557958572E+00_dp,&
      1.7711168752874866E+00_dp,  5.1688635070150774E+01_dp,&
      -5.1282387193841267E+01_dp,  2.6737329772837441E+01_dp,&
      -8.4076215182313732E+00_dp,  1.4596804851316547E+00_dp,&
      -2.2785940581509477E-02_dp, -1.2416451365357957E-01_dp,&
      4.6408964352239751E-01_dp, -1.5442693251973152E+00_dp,&
      2.9088148034596015E+00_dp, -2.0544486079149733E+00_dp,&
      -1.7885651298283725E+00_dp,  2.9061229744851547E+00_dp,&
      1.0762496921374742E+02_dp, -1.0933096205262035E+02_dp,&
      5.8725372851529393E+01_dp, -1.9345285433471407E+01_dp,&
      3.7179008532913183E+00_dp, -1.9659850716063998E-01_dp,&
      -2.4853838702353581E-01_dp,  9.8299055919831080E-01_dp,&
      -3.2634289123959395E+00_dp,  6.1454812603833870E+00_dp,&
      -4.3401377360820046E+00_dp, -3.7787705767862518E+00_dp,&
      6.1392696513115714E+00_dp,  1.0523497373659305E+00_dp,&
      -2.5417364415230703E-01_dp, -1.2822466959058445E-02_dp,&
      5.8356833156815681E-03_dp,  7.3675755090544322E-04_dp,&
      -1.4172040208393219E-04_dp, -6.0556073785800386E-04_dp,&
      3.2127278163324629E-03_dp, -1.0735448720595803E-02_dp,&
      2.0233569467854696E-02_dp, -1.4330270606267244E-02_dp,&
      -1.2438789926505517E-02_dp,  2.0283242462557815E-02_dp,&
      1.1287287038620462E+00_dp, -3.1013327621056686E-01_dp,&
      -1.6066847757949096E-03_dp,  7.3809503661043836E-03_dp,&
      1.4495723902261054E-04_dp, -2.1120727007587788E-04_dp,&
      -6.8141332001349158E-04_dp,  3.8348376691482336E-03_dp,&
      -1.2815205295062257E-02_dp,  2.4148412584623379E-02_dp,&
      -1.7103912448526319E-02_dp, -1.4845470277022208E-02_dp,&
      2.4209810789034943E-02_dp,  1.2882283178090226E+00_dp,&
      -4.3040597485938342E-01_dp,  2.4831881244751369E-02_dp,&
      9.9485126813913909E-03_dp, -1.2174764509962526E-03_dp,&
      -3.0299547417090487E-04_dp, -8.3788259132067935E-04_dp,&
      5.1512432549364667E-03_dp, -1.7230336261564266E-02_dp,&
      3.2458796239020947E-02_dp, -2.2990522835058941E-02_dp,&
      -1.9954198064667916E-02_dp,  3.2543079449201776E-02_dp,&
      1.5448514878021482E+00_dp, -6.3250090306882112E-01_dp,&
      7.5173375330629649E-02_dp,  1.2239074374363137E-02_dp,&
      -3.6909008476000097E-03_dp, -3.0904460311629461E-04_dp,&
      -1.0873965370004825E-03_dp,  7.3078182273191615E-03_dp,&
      -2.4497680697525372E-02_dp,  4.6136049459575182E-02_dp,&
      -3.2674527368890323E-02_dp, -2.8361888660189474E-02_dp,&
      4.6251077222106737E-02_dp,  1.9211073988689866E+00_dp,&
      -9.4502175925954501E-01_dp,  1.6441311125312935E-01_dp,&
      1.1514168833412042E-02_dp, -7.7156781634275936E-03_dp,&
      -2.5314063749536728E-05_dp, -1.4593192552218414E-03_dp,&
      1.0532414112812576E-02_dp, -3.5426284263123525E-02_dp,&
      6.6700389834680351E-02_dp, -4.7225177778801364E-02_dp,&
      -4.1002894262538177E-02_dp,  6.6845407444861935E-02_dp,&
      2.4503233903027750E+00_dp, -1.4111691869136846E+00_dp,&
      3.1685836597950173E-01_dp,  2.3051678588932364E-03_dp,&
      -1.3672390712846821E-02_dp,  9.1023388741843267E-04_dp,&
      -2.0204001124313029E-03_dp,  1.5157830646097843E-02_dp,&
      -5.1202256225836684E-02_dp,  9.6382554516795824E-02_dp,&
      -6.8210683093346311E-02_dp, -5.9248334038998998E-02_dp,&
      9.6542522830884450E-02_dp,  3.1805574548812579E+00_dp,&
      -2.0942909063464801E+00_dp,  5.7089561010061707E-01_dp,&
      -2.5833725640180539E-02_dp, -2.1491678096792352E-02_dp,&
      3.0989304847712494E-03_dp, -2.9159806953795845E-03_dp,&
      2.1665920578763782E-02_dp, -7.3546522325358535E-02_dp,&
      1.3842587042657298E-01_dp, -9.7913493025603968E-02_dp,&
      -8.5091612722190096E-02_dp,  1.3856859671520547E-01_dp,&
      4.1811646792046675E+00_dp, -3.0863241265112706E+00_dp,&
      9.8614321254810988E-01_dp, -9.2071074916824508E-02_dp,&
      -2.9756968197499723E-02_dp,  7.4456170391292899E-03_dp,&
      -4.4508265047520533E-03_dp,  3.0758812754125595E-02_dp,&
      -1.0495169546575114E-01_dp,  1.9754193374336126E-01_dp,&
      -1.3965469631124944E-01_dp, -1.2142837818243153E-01_dp,&
      1.9761867354288118E-01_dp,  5.5543701913701007E+00_dp,&
      -4.5209989365988195E+00_dp,  1.6539762241301386E+00_dp,&
      -2.3030816535912507E-01_dp, -3.3813532281998536E-02_dp,&
      1.5086566272402728E-02_dp, -7.2342195006746086E-03_dp,&
      4.3480249136807009E-02_dp, -1.4903057718459220E-01_dp,&
      2.8058221599374905E-01_dp, -1.9827032648170015E-01_dp,&
      -1.7246987861993432E-01_dp,  2.8052964857063045E-01_dp,&
      7.4574112588355606E+00_dp, -6.5963933721010299E+00_dp,&
      2.7132460533356846E+00_dp, -4.9834201048625321E-01_dp,&
      -2.2167462115990084E-02_dp,  2.6872562330083130E-02_dp,&
      -1.2387908764726825E-02_dp,  6.1461561859636082E-02_dp,&
      -2.1112442139248186E-01_dp,  3.9769622119135700E-01_dp,&
      -2.8093246365215413E-01_dp, -2.4445770977980782E-01_dp,&
      3.9743993120230359E-01_dp,  1.0149199555623431E+01_dp,&
      -9.6192305108604046E+00_dp,  4.3758772166955557E+00_dp,&
      -9.9115581475053982E-01_dp,  2.9545326239106471E-02_dp,&
      4.1823403732506717E-02_dp, -2.1690138678753676E-02_dp,&
      8.7456012212983200E-02_dp, -2.9961125255873033E-01_dp,&
      5.6476258723175166E-01_dp, -3.9887202332975563E-01_dp,&
      -3.4716515228541883E-01_dp,  5.6422730177346647E-01_dp,&
      1.4099231208017288E+01_dp, -1.4107075328742125E+01_dp,&
      6.9778612606041159E+00_dp, -1.8607045468744079E+00_dp,&
      1.6704246179217699E-01_dp,  5.3958212328329369E-02_dp,&
      -3.7277902595866740E-02_dp,  1.2645660436554337E-01_dp,&
      -4.2911347940038297E-01_dp,  8.0927294663161464E-01_dp,&
      -5.7153148233841800E-01_dp, -4.9752142302887042E-01_dp,&
      8.0839552282034810E-01_dp,  2.0271891440726208E+01_dp,&
      -2.1062627665694670E+01_dp,  1.1112095708204244E+01_dp,&
      -3.3547769325343713E+00_dp,  4.6741458495922611E-01_dp,&
      4.8143364079041427E-02_dp, -6.0571498843629903E-02_dp,&
      1.8810682220477090E-01_dp, -6.2907199074458620E-01_dp,&
      1.1862382813167087E+00_dp, -8.3776299569573820E-01_dp,&
      -7.2938082345320598E-01_dp,  1.1849370069427336E+00_dp,&
      3.0981138444088945E+01_dp, -3.2826924199015650E+01_dp,&
      1.8061271201881183E+01_dp, -5.9402359654032049E+00_dp,&
      1.0560336370860732E+00_dp, -1.3076665170325774E-03_dp,&
      -9.1952845171647243E-02_dp,  2.9356117158718775E-01_dp,&
      -9.6956307740968917E-01_dp,  1.8269216521737346E+00_dp /)
    REAL(KIND=dp), DIMENSION(400), PARAMETER :: c02 = (/ &
      -1.2901998665259486E+00_dp, -1.1234438324718303E+00_dp,&
      1.8249388295785653E+00_dp,  5.3052663877251987E+01_dp,&
      -5.6300287553126552E+01_dp,  3.1510803678504963E+01_dp,&
      -1.0839542135686512E+01_dp,  2.1894315183922908E+00_dp,&
      -1.2602122478496375E-01_dp, -1.3889457312206924E-01_dp,&
      5.0337725032938441E-01_dp, -1.6571300126908901E+00_dp,&
      3.1201665791015056E+00_dp, -2.2033768080640694E+00_dp,&
      -1.9187234757026690E+00_dp,  3.1167184219995239E+00_dp,&
      1.1438382828329563E+02_dp, -1.1990346708888428E+02_dp,&
      6.6738432612919837E+01_dp, -2.3077325719032441E+01_dp,&
      4.8314221537097728E+00_dp, -3.8128451926731016E-01_dp,&
      -2.6117870547203120E-01_dp,  1.0707535522056886E+00_dp,&
      -3.5392609127171060E+00_dp,  6.6628013349508590E+00_dp,&
      -4.7050599511008340E+00_dp, -4.0970099516280252E+00_dp,&
      6.6554064206249119E+00_dp,  4.4215573031384639E+02_dp,&
      -4.5609383814257563E+02_dp,  2.4967139254642808E+02_dp,&
      -8.4637858247865637E+01_dp,  1.7233338058664309E+01_dp,&
      -1.2625615465109370E+00_dp, -9.8044928324825564E-01_dp,&
      4.0814152411288465E+00_dp, -1.3536108639012429E+01_dp,&
      2.5486079632183817E+01_dp, -1.7998229964061135E+01_dp,&
      -1.5671083055262866E+01_dp,  2.5458811210562828E+01_dp,&
      1.8010738006583224E-01_dp, -6.2745911390591533E-02_dp,&
      -1.3976556185891661E-02_dp,  7.9986178737707029E-03_dp,&
      3.8749477893130122E-03_dp, -2.5821008682339651E-03_dp,&
      -1.4958653576095052E-03_dp,  9.6456448930898895E-04_dp,&
      1.1417500384510227E-03_dp, -1.1797959682526289E-03_dp,&
      1.5700881689688416E-04_dp,  2.1623209723117793E-04_dp,&
      -8.1228210530532070E-05_dp,  5.6570602726698116E-01_dp,&
      -2.1563012466646606E-01_dp, -3.7806458016327996E-02_dp,&
      3.0569342974261178E-02_dp,  1.0311060275225992E-02_dp,&
      -1.0309865308339399E-02_dp, -3.9036553568344260E-03_dp,&
      3.9203404473260196E-03_dp,  3.6495598931388052E-03_dp,&
      -5.0687843416395749E-03_dp,  1.7273963399649264E-03_dp,&
      1.9928519421558662E-04_dp, -1.8203789728090053E-04_dp,&
      1.0320570826693001E+00_dp, -4.5936001731287074E-01_dp,&
      -4.4411061460762145E-02_dp,  7.4144797958186112E-02_dp,&
      1.0753445855713722E-02_dp, -2.6121803125198153E-02_dp,&
      -3.5691764025816315E-03_dp,  1.0112104989405936E-02_dp,&
      6.3659041969645553E-03_dp, -1.3809991703339446E-02_dp,&
      7.7140379343898745E-03_dp, -1.5474246435163541E-03_dp,&
      -1.6652411530409433E-05_dp,  1.6490282386969555E+00_dp,&
      -8.8535365615839856E-01_dp, -4.0715433614121159E-03_dp,&
      1.5650434575864441E-01_dp, -6.3381624762084273E-03_dp,&
      -5.6308039349758643E-02_dp,  4.9139660534489723E-03_dp,&
      2.2014370541855809E-02_dp,  7.6064754785856734E-03_dp,&
      -3.1129921875843455E-02_dp,  2.3050532267550796E-02_dp,&
      -7.4383126577924879E-03_dp,  8.7131148771299547E-04_dp,&
      2.5133929798909564E+00_dp, -1.6395076438389586E+00_dp,&
      1.4912090565157865E-01_dp,  2.9864401700894339E-01_dp,&
      -6.6701724542314406E-02_dp, -1.0663176597759881E-01_dp,&
      3.3393538746138207E-02_dp,  4.1661750655118041E-02_dp,&
      1.8488948835995406E-03_dp, -5.9792963151484434E-02_dp,&
      5.4834911193647876E-02_dp, -2.1671245162956085E-02_dp,&
      3.3770605802150699E-03_dp,  3.7665004427790230E+00_dp,&
      -2.9643013129955884E+00_dp,  5.5460479257789530E-01_dp,&
      5.1853399079636375E-01_dp, -2.2256772225024538E-01_dp,&
      -1.7685996741929338E-01_dp,  1.0486353197058355E-01_dp,&
      6.7964719385209413E-02_dp, -2.4139367958340034E-02_dp,&
      -9.6856881358269989E-02_dp,  1.1068443291141189E-01_dp,&
      -5.0365313485448475E-02_dp,  8.9913262883314117E-03_dp,&
      5.6212689576437533E+00_dp, -5.2696679183943491E+00_dp,&
      1.4981304567183242E+00_dp,  8.0921229093198388E-01_dp,&
      -5.7220665206718013E-01_dp, -2.4595464439568815E-01_dp,&
      2.5924803186583678E-01_dp,  8.9778626060307662E-02_dp,&
      -9.7021696248226069E-02_dp, -1.2255486037880553E-01_dp,&
      1.9217594222861736E-01_dp, -9.9958611073620751E-02_dp,&
      1.9715217021775512E-02_dp,  8.4047702141954037E+00_dp,&
      -9.2546240707189753E+00_dp,  3.5534752138358008E+00_dp,&
      1.0782156596209165E+00_dp, -1.2868181607551561E+00_dp,&
      -2.3746783847288316E-01_dp,  5.5681719910211935E-01_dp,&
      7.0579962978307528E-02_dp, -2.6253042844784452E-01_dp,&
      -7.8903735544427714E-02_dp,  2.7820728538275824E-01_dp,&
      -1.7127967005997424E-01_dp,  3.7227062473712304E-02_dp,&
      1.2626716828773741E+01_dp, -1.6118082462763581E+01_dp,&
      7.8581631083902526E+00_dp,  9.9481739788304591E-01_dp,&
      -2.6354708004856406E+00_dp,  5.4288386772138826E-02_dp,&
      1.0677957643381861E+00_dp, -8.1556028930718116E-02_dp,&
      -5.8349972470961986E-01_dp,  1.6434754008913957E-01_dp,&
      2.9118537433730779E-01_dp, -2.4503791491071500E-01_dp,&
      6.0378888487433051E-02_dp,  1.9091662225464063E+01_dp,&
      -2.7927209435631788E+01_dp,  1.6649978899989840E+01_dp,&
      -3.7733057875206194E-01_dp, -4.9668781274772158E+00_dp,&
      1.1090269089810683E+00_dp,  1.8166148798971244E+00_dp,&
      -5.6474461880155935E-01_dp, -1.1104410579548303E+00_dp,&
      8.5108302171346073E-01_dp,  3.8997884915739786E-02_dp,&
      -2.5276025183160106E-01_dp,  7.9555438462666675E-02_dp,&
      2.9087515158702328E+01_dp, -4.8266848623079582E+01_dp,&
      3.4310610939149257E+01_dp, -5.4385586613019150E+00_dp,&
      -8.5091233691378658E+00_dp,  3.9330354951833977E+00_dp,&
      2.6016553035322700E+00_dp, -1.7480235778329487E+00_dp,&
      -1.7799379145508136E+00_dp,  2.3589742723119596E+00_dp,&
      -8.6871581926008334E-01_dp, -2.5405644527526649E-02_dp,&
      6.5524802080553057E-02_dp,  4.4716407424310830E+01_dp,&
      -8.3401654526387858E+01_dp,  6.9392101713584879E+01_dp,&
      -2.0013526749110621E+01_dp, -1.2568573670815580E+01_dp,&
      1.0394404054784644E+01_dp,  2.5120938544689957E+00_dp,&
      -4.1812179718438625E+00_dp, -2.1440743380530605E+00_dp,&
      5.0859869680028833E+00_dp, -3.0893954222950697E+00_dp,&
      7.8092848188977992E-01_dp, -5.0011270818283222E-02_dp,&
      6.9526312537902172E+01_dp, -1.4443332206413996E+02_dp,&
      1.3855136404356625E+02_dp, -5.7705992232865768E+01_dp,&
      -1.2976025146161891E+01_dp,  2.3238517986306515E+01_dp,&
      -1.1539823334646924E+00_dp, -8.2494908804502867E+00_dp,&
      -7.8117315080707395E-01_dp,  8.8027481030006598E+00_dp,&
      -7.3145712081593359E+00_dp,  2.6860366240642017E+00_dp,&
      -3.8959830551301688E-01_dp,  1.0992188203872426E+02_dp,&
      -2.5173108252917748E+02_dp,  2.7439346526745004E+02_dp,&
      -1.4922339600333521E+02_dp,  4.9511964753249949E+00_dp,&
      4.3977905438400001E+01_dp, -1.4483825851053977E+01_dp,&
      -1.2571750398213693E+01_dp,  5.2268271154719423E+00_dp,&
      1.0879079479809656E+01_dp, -1.2909711450908476E+01_dp,&
      5.8882269567429590E+00_dp, -1.0555338673912253E+00_dp,&
      1.7922144960276142E+02_dp, -4.4649909942956606E+02_dp,&
      5.4358551771857742E+02_dp, -3.6239825682407479E+02_dp,&
      8.2915134399015301E+01_dp,  6.3991587425066804E+01_dp,&
      -4.6200832193128967E+01_dp, -1.0539208929328979E+01_dp,&
      1.7867396014859761E+01_dp,  5.7571582062942861E+00_dp,&
      -1.6184489993979646E+01_dp,  9.2303743866032342E+00_dp,&
      -1.9050792144269630E+00_dp,  3.1419936221955288E+02_dp,&
      -8.3634297532370749E+02_dp,  1.1110728196113650E+03_dp,&
      -8.6002680865678747E+02_dp,  3.2634452019348345E+02_dp,&
      4.5766391820609520E+01_dp, -9.8967485930911280E+01_dp,&
      8.3582056990308278E+00_dp,  3.3401494394153403E+01_dp,&
      -8.4113932333341950E+00_dp, -1.4692641867662479E+01_dp,&
      1.1653736548115734E+01_dp, -2.7552478960951516E+00_dp,&
      6.9397291379150988E+02_dp, -1.9276242869869666E+03_dp,&
      2.7092202860876819E+03_dp, -2.2995420208229034E+03_dp,&
      1.0985958796049201E+03_dp, -1.0518020171631299E+02_dp,&
      -1.8516772374256206E+02_dp,  5.1320419469952256E+01_dp,&
      6.0714424144403772E+01_dp, -3.4538746632826253E+01_dp,&
      -1.3916031203934184E+01_dp,  1.7774507932922663E+01_dp,&
      -4.7391960892775407E+00_dp,  1.4574807068110414E+00_dp,&
      -5.1966690003049509E-01_dp, -1.0937121555649043E-01_dp,&
      6.8282130920804027E-02_dp,  3.0254852511833141E-02_dp,&
      -2.2336606510276111E-02_dp, -1.1641347572123872E-02_dp,&
      8.3890438873829229E-03_dp,  9.3122892529834168E-03_dp,&
      -1.0437466316246218E-02_dp,  2.0493938245759241E-03_dp,&
      1.4556641442013175E-03_dp, -6.1406129186868910E-04_dp,&
      1.6635774394994376E+00_dp, -7.4447036116082299E-01_dp,&
      -7.3265463407581985E-02_dp,  1.2175244735677020E-01_dp,&
      1.8417461157197813E-02_dp, -4.3129640640968338E-02_dp,&
      -6.3261088523512048E-03_dp,  1.6717304979147734E-02_dp,&
      1.0813189391350427E-02_dp, -2.2870001477521811E-02_dp,&
      1.2540227392196780E-02_dp, -2.3939994363031116E-03_dp,&
      -6.7292329081708585E-05_dp,  2.1130083057428242E+00_dp,&
      -1.2646245625758314E+00_dp,  3.9026678312747663E-02_dp,&
      2.3868043891980367E-01_dp, -2.2107148945619371E-02_dp,&
      -8.7624034141812510E-02_dp,  1.2400332443529075E-02_dp,&
      3.4503268350899216E-02_dp,  9.9312975904139698E-03_dp,&
      -4.9617985830045279E-02_dp,  3.8605161734445871E-02_dp,&
      -1.3143058004433409E-02_dp,  1.6819268858947601E-03_dp,&
      2.8861358995389326E+00_dp, -2.2379012503086972E+00_dp,&
      3.2529535767432088E-01_dp,  4.3452974907619385E-01_dp,&
      -1.3098878900070884E-01_dp, -1.5802763888684246E-01_dp,&
      6.3048141625604320E-02_dp,  6.2230049682745238E-02_dp,&
      -3.4334538372675287E-03_dp, -9.1072403941971233E-02_dp,&
      8.8902484985361505E-02_dp, -3.6762642551420852E-02_dp,&
      6.0063306694479717E-03_dp,  4.1212721248896012E+00_dp,&
      -3.9524567137450393E+00_dp,  9.8864687807522755E-01_dp,&
      7.1583307090425163E-01_dp, -3.8544667410777939E-01_dp,&
      -2.4530041815719975E-01_dp,  1.7970618099537952E-01_dp,&
      9.4646106681016307E-02_dp, -5.1054392753752079E-02_dp,&
      -1.3697099489584089E-01_dp,  1.7083844262209280E-01_dp,&
      -8.1262942364707796E-02_dp,  1.5034543179272131E-02_dp,&
      6.0427874405411135E+00_dp, -6.9114188446690799E+00_dp,&
      2.4368523205155839E+00_dp,  1.0422276743751018E+00_dp,&
      -9.2514306563957271E-01_dp, -3.0462890230255507E-01_dp,&
      4.1835014026158401E-01_dp,  1.0930727144152716E-01_dp,&
      -1.7419009015490974E-01_dp, -1.4775669359208615E-01_dp,&
      2.7804753282859701E-01_dp, -1.5293629530276798E-01_dp,&
      3.1260563924463657E-02_dp,  9.0090022431937165E+00_dp,&
      -1.1984085393120617E+01_dp,  5.4766120418255344E+00_dp,&
      1.2306964512849667E+00_dp, -1.9841323771701029E+00_dp,&
      -2.0178112750263197E-01_dp,  8.5733779402876675E-01_dp,&
      4.2784199415162032E-02_dp, -4.3931636150909686E-01_dp,&
      -2.1473925344957952E-02_dp,  3.6199744674856460E-01_dp,&
      -2.4527309632224395E-01_dp,  5.5802785639214363E-02_dp,&
      1.3591622850222656E+01_dp, -2.0671385649481678E+01_dp,&
      1.1690911823198414E+01_dp,  7.2296817333192265E-01_dp,&
      -3.9122753813128814E+00_dp,  3.9756373580542609E-01_dp,&
      1.5730792402830771E+00_dp, -2.5426387228982428E-01_dp,&
      -9.2821684388540859E-01_dp,  4.5454888767816393E-01_dp,&
      2.7835397546226770E-01_dp, -3.1429802698364062E-01_dp,&
      8.3881632785542004E-02_dp,  2.0708310558399841E+01_dp,&
      -3.5573130360984386E+01_dp,  2.4173095046726132E+01_dp,&
      -1.9624509389026361E+00_dp, -7.1194347943715250E+00_dp,&
      2.2395796302706463E+00_dp,  2.5414676107335459E+00_dp,&
      -1.0901424546362217E+00_dp, -1.6791478429090076E+00_dp,&
      1.6519357564458712E+00_dp, -2.9686578874346814E-01_dp,&
      -2.3345958133610148E-01_dp,  9.5629002374604352E-02_dp,&
      3.1849060054820505E+01_dp, -6.1216326263024023E+01_dp,&
      4.8955578605990361E+01_dp, -1.0495665728696784E+01_dp,&
      -1.1727745979077595E+01_dp,  6.8444384400614950E+00_dp,&
      3.3437289894171416E+00_dp, -3.0185515513605332E+00_dp,&
      -2.5103650595979259E+00_dp,  4.0981205243496870E+00_dp,&
      -1.9744886540859123E+00_dp,  2.7900551212540414E-01_dp,&
      3.9609099066905094E-02_dp,  4.9480572244963817E+01_dp,&
      -1.0554295952164991E+02_dp,  9.7784802913758284E+01_dp,&
      -3.3609900633466118E+01_dp, -1.6293722769181993E+01_dp,&
      1.6978324437699740E+01_dp,  2.3989997683487170E+00_dp /)
    REAL(KIND=dp), DIMENSION(400), PARAMETER :: c03 = (/ &
      -6.8292295875978279E+00_dp, -2.5859581635928466E+00_dp,&
      8.2643295812249669E+00_dp, -5.7320634779870945E+00_dp,&
      1.7650956249120218E+00_dp, -1.9567508222089966E-01_dp,&
      7.7834138139689216E+01_dp, -1.8267266020330240E+02_dp,&
      1.9349073152056008E+02_dp, -9.1481588588459843E+01_dp,&
      -1.3822783835670613E+01_dp,  3.6637703347062448E+01_dp,&
      -4.7917952106854855E+00_dp, -1.2983441330102631E+01_dp,&
      5.6026423699990802E-01_dp,  1.3524777870542334E+01_dp,&
      -1.2577242006230376E+01_dp,  5.0578704679411794E+00_dp,&
      -8.1335121215823181E-01_dp,  1.2469972524515586E+02_dp,&
      -3.1853576272145068E+02_dp,  3.8049814102352741E+02_dp,&
      -2.2932721454772252E+02_dp,  1.9316804878393324E+01_dp,&
      6.7651019536813379E+01_dp, -2.8556483274297605E+01_dp,&
      -1.8915160882598556E+01_dp,  1.2394440997926818E+01_dp,&
      1.4599065487288630E+01_dp, -2.0997926004529241E+01_dp,&
      1.0415399006452484E+01_dp, -1.9915861476472365E+00_dp,&
      2.0658312239862570E+02_dp, -5.6523320243902458E+02_dp,&
      7.4772242646605821E+02_dp, -5.4520823589717588E+02_dp,&
      1.4978849258832480E+02_dp,  9.3899472952695064E+01_dp,&
      -8.4549171246107946E+01_dp, -1.1373667866556911E+01_dp,&
      3.7590687286378447E+01_dp, -2.2520800129803975E+00_dp,&
      -2.1045686666495470E+01_dp,  1.4379152932515209E+01_dp,&
      -3.2398302906841523E+00_dp,  3.6839922338686353E+02_dp,&
      -1.0537707025969034E+03_dp,  1.5006891025872155E+03_dp,&
      -1.2562081672322104E+03_dp,  5.3825286433010865E+02_dp,&
      4.2498035313542289E+01_dp, -1.7156738426552047E+02_dp,&
      3.5936393290874996E+01_dp,  6.0177434607070502E+01_dp,&
      -4.0248403369709855E+01_dp, -2.7568019702721092E+00_dp,&
      1.1202607958032809E+01_dp, -3.3375095387496079E+00_dp,&
      7.9098125896269880E+02_dp, -2.2904299594238068E+03_dp,&
      3.3675571212100285E+03_dp, -3.0249252348258237E+03_dp,&
      1.5837024358091369E+03_dp, -2.4417286558439139E+02_dp,&
      -2.4707398238911915E+02_dp,  1.1661864596469552E+02_dp,&
      6.5275940183027629E+01_dp, -7.1150510177242779E+01_dp,&
      1.0564083514283231E+01_dp,  1.0686772402576475E+01_dp,&
      -4.0141425428278463E+00_dp,  2.9638192147333571E+03_dp,&
      -8.4573529628979268E+03_dp,  1.2304466526150441E+04_dp,&
      -1.1014363127026381E+04_dp,  5.8847019588549192E+03_dp,&
      -1.1791468203519848E+03_dp, -5.8911861079676396E+02_dp,&
      2.7195710510299079E+02_dp,  2.1685800896742452E+02_dp,&
      -1.7746143301747480E+02_dp, -1.4574518198891317E+01_dp,&
      5.4805869945752612E+01_dp, -1.6433623764473392E+01_dp,&
      2.2036130805159376E-01_dp, -4.5076804566774459E-02_dp,&
      -1.6076370477688957E-02_dp,  5.5534315752250891E-03_dp,&
      5.0461774412725036E-03_dp, -2.0604744482374273E-03_dp,&
      -2.2689084798617320E-03_dp,  5.9099274703236049E-04_dp,&
      2.3874131512182165E-03_dp, -2.1618256480298356E-03_dp,&
      4.3684618125611787E-04_dp,  2.5191107660970112E-04_dp,&
      -1.0861832710107730E-04_dp,  7.0865218236726824E-01_dp,&
      -1.6570352438040939E-01_dp, -5.2249512547067715E-02_dp,&
      2.4669238373278793E-02_dp,  1.7087628445085885E-02_dp,&
      -9.6906052918900448E-03_dp, -7.7468377444847780E-03_dp,&
      3.0933546649085024E-03_dp,  9.2517865757837520E-03_dp,&
      -1.0056570203112576E-02_dp,  3.4720383247476838E-03_dp,&
      1.6952014914339465E-04_dp, -2.7523734696923005E-04_dp,&
      1.3534864020651076E+00_dp, -3.9336467243616868E-01_dp,&
      -9.8656745599151094E-02_dp,  7.3087777433389595E-02_dp,&
      3.4093865889400041E-02_dp, -3.0490786739633617E-02_dp,&
      -1.5507854335106205E-02_dp,  1.0740733786632572E-02_dp,&
      2.3081322466090199E-02_dp, -3.1460955605477889E-02_dp,&
      1.5456737064545917E-02_dp, -2.6763084092928935E-03_dp,&
      -1.2381650776694105E-04_dp,  2.3090234088808299E+00_dp,&
      -8.5776891469241912E-01_dp, -1.5338027586293296E-01_dp,&
      1.8991620137293749E-01_dp,  5.4828916209970144E-02_dp,&
      -8.3025281965828188E-02_dp, -2.4296812195096610E-02_dp,&
      3.1469074113093229E-02_dp,  5.0330137251257137E-02_dp,&
      -8.5515406097693397E-02_dp,  5.1866160160934108E-02_dp,&
      -1.4179706064901558E-02_dp,  1.2560395955065946E-03_dp,&
      3.8197475422582392E+00_dp, -1.8037294425434220E+00_dp,&
      -1.8899434601612383E-01_dp,  4.5191973881055397E-01_dp,&
      6.3713987378258216E-02_dp, -2.0444755855365859E-01_dp,&
      -2.4088171877681655E-02_dp,  8.2291692084297763E-02_dp,&
      9.7362650813698964E-02_dp, -2.0934041478609497E-01_dp,&
      1.4766839822240871E-01_dp, -4.9288591476610653E-02_dp,&
      6.4499622528296279E-03_dp,  6.2966160150403754E+00_dp,&
      -3.7109866191524525E+00_dp, -1.0522184486152265E-01_dp,&
      1.0085149071581117E+00_dp,  2.5323534878071412E-03_dp,&
      -4.6630138967976670E-01_dp,  2.1076711993510598E-02_dp,&
      1.9828760927368860E-01_dp,  1.5998299484831749E-01_dp,&
      -4.6825207004727926E-01_dp,  3.7645280036684625E-01_dp,&
      -1.4285165419296325E-01_dp,  2.2001423131429464E-02_dp,&
      1.0454409969612419E+01_dp, -7.5375644876744872E+00_dp,&
      3.9449043473913742E-01_dp,  2.1412348960549363E+00_dp,&
      -3.0355677655830265E-01_dp, -9.9615781067345155E-01_dp,&
      2.1711098138944293E-01_dp,  4.4692476224298294E-01_dp,&
      1.8591520945768222E-01_dp, -9.5485093882726246E-01_dp,&
      8.7779557412188514E-01_dp, -3.6914472091627620E-01_dp,&
      6.2977146118752103E-02_dp,  1.7567505895441929E+01_dp,&
      -1.5225622537868910E+01_dp,  2.1193499070085169E+00_dp,&
      4.3510998858589849E+00_dp, -1.3318083138416941E+00_dp,&
      -1.9898827710887639E+00_dp,  8.4640106489365297E-01_dp,&
      9.4249431258822869E-01_dp, -3.7448636838545483E-02_dp,&
      -1.7262268981146505E+00_dp,  1.8723774805378806E+00_dp,&
      -8.6854820742115435E-01_dp,  1.6062602638927947E-01_dp,&
      2.9958154145281728E+01_dp, -3.0775664710589034E+01_dp,&
      7.2009528080278491E+00_dp,  8.4351100129247225E+00_dp,&
      -4.3138466133479865E+00_dp, -3.6417064107767114E+00_dp,&
      2.6101763292430844E+00_dp,  1.8247368979816314E+00_dp,&
      -1.1799262417500072E+00_dp, -2.5186933824400533E+00_dp,&
      3.5685918319046439E+00_dp, -1.8540412040242573E+00_dp,&
      3.7039085515950843E-01_dp,  5.1951978548581039E+01_dp,&
      -6.2581108665881274E+01_dp,  2.1182571267411070E+01_dp,&
      1.5325960406620894E+01_dp, -1.2296597851170151E+01_dp,&
      -5.7213940765686004E+00_dp,  7.1318501498457465E+00_dp,&
      3.0496412877399690E+00_dp, -5.0553615396661806E+00_dp,&
      -1.8070209122459882E+00_dp,  5.6345736738491370E+00_dp,&
      -3.4763373861946438E+00_dp,  7.6167100749572603E-01_dp,&
      9.1798276836285979E+01_dp, -1.2863475356597820E+02_dp,&
      5.8496101669050397E+01_dp,  2.4744077629967428E+01_dp,&
      -3.2497120098098861E+01_dp, -5.8945347267566541E+00_dp,&
      1.7820196379746736E+01_dp,  3.4558058060427674E+00_dp,&
      -1.5893088806805018E+01_dp,  5.7568757129573056E+00_dp,&
      5.3329494587249027E+00_dp, -5.1537631328962918E+00_dp,&
      1.3188423737811730E+00_dp,  1.6564728585867815E+02_dp,&
      -2.6848193454407402E+02_dp,  1.5714914105375516E+02_dp,&
      2.9015793106076185E+01_dp, -8.0936336478366655E+01_dp,&
      6.0337388562561021E+00_dp,  4.0541279455914804E+01_dp,&
      -2.5358711523755098E+00_dp, -4.1489451794467044E+01_dp,&
      3.4994248985293758E+01_dp, -7.6772498820742348E+00_dp,&
      -3.3389802476741770E+00_dp,  1.5472908363224536E+00_dp,&
      3.0609437432716891E+02_dp, -5.7157248282740568E+02_dp,&
      4.1925052593925301E+02_dp, -1.1600417484219872E+01_dp,&
      -1.8849689345974764E+02_dp,  6.6110045063594768E+01_dp,&
      7.9669860475699736E+01_dp, -3.4687343159162722E+01_dp,&
      -8.8625160627562337E+01_dp,  1.1831559439724820E+02_dp,&
      -6.2643850352276246E+01_dp,  1.4313449464928395E+01_dp,&
      -6.9462104331569086E-01_dp,  5.8152372234088853E+02_dp,&
      -1.2471092561355499E+03_dp,  1.1264512659705742E+03_dp,&
      -2.6350048877660970E+02_dp, -3.8978177911431425E+02_dp,&
      2.8606115420791843E+02_dp,  1.0671832025528735E+02_dp,&
      -1.4850079552469927E+02_dp, -1.2702723207166953E+02_dp,&
      2.9628275973114745E+02_dp, -2.1971106597050107E+02_dp,&
      7.9255756055034652E+01_dp, -1.1738659870534550E+01_dp,&
      1.1446896056961609E+03_dp, -2.8062684434259190E+03_dp,&
      3.0792684302402877E+03_dp, -1.3773330948044859E+03_dp,&
      -5.6315721236341506E+02_dp,  9.4384383423094675E+02_dp,&
      -9.1298241779040950E+01_dp, -4.3161000322227324E+02_dp,&
      5.6869256603055426E+01_dp,  4.6537109542580185E+02_dp,&
      -4.9778545245192629E+02_dp,  2.2723211423906713E+02_dp,&
      -4.1682116814902678E+01_dp,  2.3931486245110582E+03_dp,&
      -6.6248847444164976E+03_dp,  8.6607172498776090E+03_dp,&
      -5.7963496001557987E+03_dp,  5.6074777366093997E+02_dp,&
      2.2520445132559971E+03_dp, -1.3103161337837334E+03_dp,&
      -5.8712209332349289E+02_dp,  8.9204271290751376E+02_dp,&
      -1.1639397991700832E+01_dp, -5.0558768097343659E+02_dp,&
      3.3540358542052877E+02_dp, -7.4587683513078233E+01_dp,&
      6.0278043205707454E+03_dp, -1.8260441253553501E+04_dp,&
      2.7198231026846435E+04_dp, -2.3440178398044740E+04_dp,&
      9.7072987485955946E+03_dp,  1.9816258432428326E+03_dp,&
      -4.2204293880177993E+03_dp,  5.7832304639916708E+02_dp,&
      1.8692380816063537E+03_dp, -1.2109634326915912E+03_dp,&
      -5.4026291769502137E+01_dp,  3.1528828654208075E+02_dp,&
      -9.4786678043392129E+01_dp,  1.7937094180089550E+00_dp,&
      -3.7993808012512725E-01_dp, -1.3139337881541585E-01_dp,&
      4.9430879424174734E-02_dp,  4.1715801584312942E-02_dp,&
      -1.8655007611371892E-02_dp, -1.8806529335258913E-02_dp,&
      5.5337190143686437E-03_dp,  2.0442542773902748E-02_dp,&
      -1.9495932303552625E-02_dp,  4.7817782309535614E-03_dp,&
      1.6859794985952033E-03_dp, -8.4516904829219403E-04_dp,&
      2.1832690683719891E+00_dp, -6.3377544246855000E-01_dp,&
      -1.6248271996603472E-01_dp,  1.1690489216647633E-01_dp,&
      5.6807251626275680E-02_dp, -4.8432873661308128E-02_dp,&
      -2.6022849299146757E-02_dp,  1.6853696869085871E-02_dp,&
      3.7648457313444655E-02_dp, -4.9878575440930099E-02_dp,&
      2.3642088947003155E-02_dp, -3.6336297159725440E-03_dp,&
      -3.3755205819095051E-04_dp,  3.0753249432146119E+00_dp,&
      -1.2701520583639132E+00_dp, -2.0849770294749378E-01_dp,&
      2.9526772321314387E-01_dp,  7.8995542205080829E-02_dp,&
      -1.2972869147593624E-01_dp, -3.5582090248904033E-02_dp,&
      4.9305013374453963E-02_dp,  7.7353411801215979E-02_dp,&
      -1.3353482456470839E-01_dp,  8.1872538186610708E-02_dp,&
      -2.2734499256472550E-02_dp,  2.0934547665217729E-03_dp,&
      4.7271509364185604E+00_dp, -2.6008689221752657E+00_dp,&
      -2.1378302448991618E-01_dp,  6.8829890166551655E-01_dp,&
      7.8155780076278006E-02_dp, -3.1427392410807997E-01_dp,&
      -2.9055243870744554E-02_dp,  1.2728338790291072E-01_dp,&
      1.4542960480058301E-01_dp, -3.2288152522122560E-01_dp,&
      2.3145832634114052E-01_dp, -7.8570015248536254E-02_dp,&
      1.0527188402678012E-02_dp,  7.6239589870130429E+00_dp,&
      -5.2716113961903401E+00_dp, -1.1543925130729318E-02_dp,&
      1.5034494543003840E+00_dp, -4.1512319049193738E-02_dp,&
      -7.0326534010639863E-01_dp,  5.2072123155519456E-02_dp,&
      3.0170745289904521E-01_dp,  2.3013280605622619E-01_dp,&
      -7.1093920590192394E-01_dp,  5.8207150943308372E-01_dp,&
      -2.2424295809781689E-01_dp,  3.5093648865483100E-02_dp,&
      1.2651454564011702E+01_dp, -1.0594016000348530E+01_dp,&
      8.6079526700003395E-01_dp,  3.1309762712299110E+00_dp,&
      -5.5343393928245810E-01_dp, -1.4752776243845915E+00_dp,&
      3.7341588237894630E-01_dp,  6.7003976934246967E-01_dp,&
      2.4249290952305499E-01_dp, -1.4249882020825018E+00_dp,&
      1.3395429928449589E+00_dp, -5.7182949222827995E-01_dp,&
      9.8841365839720668E-02_dp,  2.1429965966478342E+01_dp,&
      -2.1247628542515503E+01_dp,  3.6287863654310013E+00_dp,&
      6.2581819127944680E+00_dp, -2.1863113529437315E+00_dp,&
      -2.8967923413102912E+00_dp,  1.3681886836307373E+00_dp,&
      1.3955776856915574E+00_dp, -1.5897239290899864E-01_dp,&
      -2.5212216600023445E+00_dp,  2.8219668706971945E+00_dp,&
      -1.3307700093912902E+00_dp,  2.4913514170866757E-01_dp,&
      3.6963550418790938E+01_dp, -4.2781735704421529E+01_dp,&
      1.1468857291085394E+01_dp,  1.1956021064279158E+01_dp /)
    REAL(KIND=dp), DIMENSION(400), PARAMETER :: c04 = (/ &
      -6.7963494820353745E+00_dp, -5.2037548606327730E+00_dp,&
      4.1015955423288375E+00_dp,  2.6719939603737344E+00_dp,&
      -2.0314435192987417E+00_dp, -3.5279015440942261E+00_dp,&
      5.3011832033957269E+00_dp, -2.8122613024625087E+00_dp,&
      5.6915484145040629E-01_dp,  6.4926664043803811E+01_dp,&
      -8.6899342041810002E+01_dp,  3.2560767520934299E+01_dp,&
      2.1401514791066290E+01_dp, -1.8945123297568252E+01_dp,&
      -7.9472372366379886E+00_dp,  1.1031467887006638E+01_dp,&
      4.4055773998180028E+00_dp, -8.2307667741735777E+00_dp,&
      -1.9496849372131122E+00_dp,  8.1532056320068413E+00_dp,&
      -5.2079110157651822E+00_dp,  1.1601051300795284E+00_dp,&
      1.1627932827440712E+02_dp, -1.7882190134095538E+02_dp,&
      8.8065313863488811E+01_dp,  3.3852987626345111E+01_dp,&
      -4.9425190938476227E+01_dp, -7.4515204120012459E+00_dp,&
      2.7343411149027549E+01_dp,  4.8027447886962538E+00_dp,&
      -2.5480067206341694E+01_dp,  1.0841781549031017E+01_dp,&
      6.8639785168072986E+00_dp, -7.5117713591262296E+00_dp,&
      1.9816437521619110E+00_dp,  2.1273862203414356E+02_dp,&
      -3.7428369024165295E+02_dp,  2.3346238297745839E+02_dp,&
      3.7654110424279338E+01_dp, -1.2236868407636074E+02_dp,&
      1.1842889589859949E+01_dp,  6.2188087891764269E+01_dp,&
      -4.6949238597179814E+00_dp, -6.6545313811818971E+01_dp,&
      5.8913671022159285E+01_dp, -1.5425914307453965E+01_dp,&
      -3.9040075343478216E+00_dp,  2.2156462706329005E+00_dp,&
      3.9862923384302263E+02_dp, -8.0001729806986441E+02_dp,&
      6.1729015727077262E+02_dp, -2.5059892452212821E+01_dp,&
      -2.8573151086745020E+02_dp,  1.0461889229880697E+02_dp,&
      1.2389826235882019E+02_dp, -5.4939060293644125E+01_dp,&
      -1.4442389426530306E+02_dp,  1.9642382267159812E+02_dp,&
      -1.0695650576544364E+02_dp,  2.5875876655613105E+01_dp,&
      -1.6417786429176835E+00_dp,  7.6777159359322945E+02_dp,&
      -1.7537133335705055E+03_dp,  1.6482226768525409E+03_dp,&
      -3.9211708766427637E+02_dp, -6.0392305987307441E+02_dp,&
      4.4333559405067615E+02_dp,  1.7787450181484434E+02_dp,&
      -2.3734798751752581E+02_dp, -2.2095920452471174E+02_dp,&
      5.0350326325990295E+02_dp, -3.7436592829883779E+02_dp,&
      1.3589060804004382E+02_dp, -2.0292510387655746E+01_dp,&
      1.5296195207071578E+03_dp, -3.9622350559145198E+03_dp,&
      4.4857786928390806E+03_dp, -1.9870420059955213E+03_dp,&
      -9.7134936492482530E+02_dp,  1.4998589385742246E+03_dp,&
      -7.2892049877523178E+01_dp, -7.4631312910496581E+02_dp,&
      3.4005793563591304E+01_dp,  8.9042700432665424E+02_dp,&
      -9.2292552288497313E+02_dp,  4.1790581952714399E+02_dp,&
      -7.6433006503481721E+01_dp,  3.2060087628137712E+03_dp,&
      -9.3159464375811276E+03_dp,  1.2527209661291899E+04_dp,&
      -8.3053185228384245E+03_dp,  2.0211803182403949E+02_dp,&
      4.0355629264948511E+03_dp, -2.0304106473113000E+03_dp,&
      -1.4364497108817213E+03_dp,  1.8398623005265499E+03_dp,&
      -9.3541599607360126E+00_dp, -1.0484935445809742E+03_dp,&
      6.9858483574260197E+02_dp, -1.5661215432484158E+02_dp,&
      7.5591031206278803E+03_dp, -2.3841921827888185E+04_dp,&
      3.6673405643627026E+04_dp, -3.2125806874645870E+04_dp,&
      1.2397751752493754E+04_dp,  5.3198673575983330E+03_dp,&
      -8.5053962353377374E+03_dp,  1.3907593831382503E+03_dp,&
      4.2903518293564521E+03_dp, -3.9548845908239964E+03_dp,&
      1.3227466776399026E+03_dp, -1.3766911144115962E+01_dp,&
      -7.0037204280174294E+01_dp,  2.8049596976249963E+04_dp,&
      -8.9761481908365226E+04_dp,  1.4403023637372538E+05_dp,&
      -1.4044772416853864E+05_dp,  7.9387433338889328E+04_dp,&
      -1.2949273069561426E+04_dp, -1.4404242828698498E+04_dp,&
      6.6705867587399225E+03_dp,  5.5214430686198393E+03_dp,&
      -6.3725355581415643E+03_dp,  1.8293089995376904E+03_dp,&
      3.3442777562420838E+02_dp, -2.1590463856083068E+02_dp,&
      2.4597046908671585E-01_dp, -2.4131515313491576E-02_dp,&
      -1.4593663353954249E-02_dp,  2.2311800236706780E-03_dp,&
      4.4466949112589912E-03_dp, -1.8404928039251618E-04_dp,&
      -2.9927966501996396E-03_dp,  2.6735816123596879E-04_dp,&
      2.7889490611109772E-03_dp, -2.2452342695937332E-03_dp,&
      3.6231203373800745E-04_dp,  3.0171331853223634E-04_dp,&
      -1.1748533887842431E-04_dp,  8.0512062220512370E-01_dp,&
      -9.3210652061862007E-02_dp, -5.4083956239739717E-02_dp,&
      1.1275869730545569E-02_dp,  1.7399565786205721E-02_dp,&
      -1.9029409691100781E-03_dp, -1.1869164318760249E-02_dp,&
      1.5167816775997040E-03_dp,  1.1742529707275380E-02_dp,&
      -1.0427559162066466E-02_dp,  2.5937524919136981E-03_dp,&
      7.7071430704681043E-04_dp, -4.0179192653180969E-04_dp,&
      1.5913824833072012E+00_dp, -2.3872493245015622E-01_dp,&
      -1.2969157416835858E-01_dp,  3.8869865775083785E-02_dp,&
      4.4967623155562533E-02_dp, -9.6244127552489699E-03_dp,&
      -3.1355874505723864E-02_dp,  5.8983064535336914E-03_dp,&
      3.3784953332417815E-02_dp, -3.3869634002932601E-02_dp,&
      1.1722170608515301E-02_dp,  2.1904742245513310E-04_dp,&
      -7.8368264635060559E-04_dp,  2.8504450774327998E+00_dp,&
      -5.6612512571131524E-01_dp, -2.8459684321944051E-01_dp,&
      1.1706520518696334E-01_dp,  1.0585372455591220E-01_dp,&
      -3.5695445721860848E-02_dp, -7.5839023807993372E-02_dp,&
      1.9812098408427733E-02_dp,  8.9483211008635308E-02_dp,&
      -1.0040209061808522E-01_dp,  4.2905149773221109E-02_dp,&
      -5.0125013089657635E-03_dp, -1.0293435042649617E-03_dp,&
      5.0103582234362856E+00_dp, -1.2983685620824574E+00_dp,&
      -5.9781702803632175E-01_dp,  3.2458841568543156E-01_dp,&
      2.3547324163176134E-01_dp, -1.1375382180367667E-01_dp,&
      -1.7422262057421442E-01_dp,  6.1111170753431865E-02_dp,&
      2.2598196223392233E-01_dp, -2.8189868001107560E-01_dp,&
      1.4003280395960366E-01_dp, -2.7702266029124158E-02_dp,&
      2.3088321236861272E-04_dp,  8.8649193387428031E+00_dp,&
      -2.9256588227359037E+00_dp, -1.2152264078580886E+00_dp,&
      8.5696828796979208E-01_dp,  4.9899734214418356E-01_dp,&
      -3.3446674272664295E-01_dp, -3.8401558036591399E-01_dp,&
      1.7999675719079797E-01_dp,  5.5182828585602572E-01_dp,&
      -7.6573695808554709E-01_dp,  4.2908723819998862E-01_dp,&
      -1.0932416618525242E-01_dp,  8.4551173907571678E-03_dp,&
      1.5946104667560101E+01_dp, -6.5535743239849209E+00_dp,&
      -2.3980297467455691E+00_dp,  2.2035136645166129E+00_dp,&
      1.0045777679055572E+00_dp, -9.4065923178633748E-01_dp,&
      -8.1356526611789204E-01_dp,  5.1914828261288715E-01_dp,&
      1.3116012772845194E+00_dp, -2.0399962005155539E+00_dp,&
      1.2690404031225635E+00_dp, -3.7979574497355501E-01_dp,&
      4.2950086794989274E-02_dp,  2.9311091290898574E+01_dp,&
      -1.4740506211940973E+01_dp, -4.5764593211909546E+00_dp,&
      5.6031566250743383E+00_dp,  1.8826906137057735E+00_dp,&
      -2.5823838485985520E+00_dp, -1.6411360147721636E+00_dp,&
      1.4915672640388642E+00_dp,  3.0268734350794677E+00_dp,&
      -5.3670886083417155E+00_dp,  3.6770733976275096E+00_dp,&
      -1.2384352707729609E+00_dp,  1.6844098822094386E-01_dp,&
      5.5269891112628450E+01_dp, -3.3588337863064204E+01_dp,&
      -8.3060983403295587E+00_dp,  1.4245858269068547E+01_dp,&
      3.0679193808833980E+00_dp, -7.0049866947254937E+00_dp,&
      -3.0485764419964387E+00_dp,  4.3234572865357226E+00_dp,&
      6.6611087254448593E+00_dp, -1.3946268257843466E+01_dp,&
      1.0509462585713232E+01_dp, -3.8962221995669863E+00_dp,&
      5.9561183401098095E-01_dp,  1.0733973915516557E+02_dp,&
      -7.8192038336418292E+01_dp, -1.3621080023967423E+01_dp,&
      3.6519885331470277E+01_dp,  3.1798382551855648E+00_dp,&
      -1.8903100012045829E+01_dp, -4.6635840130611124E+00_dp,&
      1.2767153185864498E+01_dp,  1.3205696710402899E+01_dp,&
      -3.5455541355133271E+01_dp,  2.9604035476104617E+01_dp,&
      -1.1959941279367301E+01_dp,  1.9943808572689838E+00_dp,&
      2.1574110229087182E+02_dp, -1.8759091378514898E+02_dp,&
      -1.6548194406177625E+01_dp,  9.5013977871018469E+01_dp,&
      -5.4491937574046752E+00_dp, -5.0801726334298664E+01_dp,&
      -2.6993423034851483E+00_dp,  3.8656939223630474E+01_dp,&
      1.8853421819247718E+01_dp, -8.5648508663829901E+01_dp,&
      8.1175335514105896E+01_dp, -3.5754750625577735E+01_dp,&
      6.4231041798051232E+00_dp,  4.5152529264002453E+02_dp,&
      -4.6839884015400759E+02_dp,  7.0294758026416391E+00_dp,&
      2.5195876228718237E+02_dp, -5.8200098802286071E+01_dp,&
      -1.3476486579792476E+02_dp,  2.3084889406857080E+01_dp,&
      1.1999084006699820E+02_dp, -1.4733025794216744E+01_dp,&
      -1.7989156147155072E+02_dp,  2.0847512854987923E+02_dp,&
      -1.0186724973687066E+02_dp,  1.9730960727082149E+01_dp,&
      9.9218910784116417E+02_dp, -1.2321348321492649E+03_dp,&
      1.7515742609839145E+02_dp,  6.8070921799959092E+02_dp,&
      -3.1209102904765376E+02_dp, -3.4057274892355639E+02_dp,&
      1.6687726273187675E+02_dp,  3.7518950074597132E+02_dp,&
      -3.1475910486719397E+02_dp, -2.1096647744491858E+02_dp,&
      4.4206246642368058E+02_dp, -2.5724080612716710E+02_dp,&
      5.5041848263909372E+01_dp,  2.3157777551412210E+03_dp,&
      -3.4705465948827878E+03_dp,  1.0560938859361700E+03_dp,&
      1.8457238342746532E+03_dp, -1.4757403839796784E+03_dp,&
      -7.0935408588748635E+02_dp,  8.4504740826609702E+02_dp,&
      1.0995279604639743E+03_dp, -1.9951407194066446E+03_dp,&
      8.8275263167062758E+02_dp,  3.0984268568964131E+02_dp,&
      -4.2142428256752106E+02_dp,  1.1510935171638715E+02_dp,&
      5.8410998191536091E+03_dp, -1.0715057679305270E+04_dp,&
      5.5881348394814086E+03_dp,  4.6493931663950134E+03_dp,&
      -6.8728966621051050E+03_dp, -4.1924139258334478E+01_dp,&
      3.7206102777900010E+03_dp,  2.1202646040436580E+03_dp,&
      -9.2875013030481878E+03_dp,  9.2985656387185409E+03_dp,&
      -4.4824184492676550E+03_dp,  9.6577982916343615E+02_dp,&
      -4.0485333749672158E+01_dp,  1.6398745921063954E+04_dp,&
      -3.7596389625372154E+04_dp,  3.1336122401268269E+04_dp,&
      5.4048022371223969E+03_dp, -3.1578107170577830E+04_dp,&
      1.7491375187499234E+04_dp,  1.0558791507837142E+04_dp,&
      -8.6578643507329816E+03_dp, -2.1657103563023124E+04_dp,&
      4.1477022148521210E+04_dp, -3.2576961817922802E+04_dp,&
      1.3109718009568427E+04_dp, -2.2341346231179591E+03_dp,&
      5.5933770441125977E+04_dp, -1.6151312582203400E+05_dp,&
      2.0784369795231955E+05_dp, -1.0524097468765127E+05_dp,&
      -7.7042400583628056E+04_dp,  1.7217692234932515E+05_dp,&
      -1.1629573180003346E+05_dp,  1.1490194503876031E+04_dp,&
      2.5783214909699680E+04_dp,  1.0864388170919431E+03_dp,&
      -2.2701386651906592E+04_dp,  1.5634205854882433E+04_dp,&
      -3.5877583180293955E+03_dp,  2.0109439642887108E+00_dp,&
      -2.0605705561447749E-01_dp, -1.2328473266793621E-01_dp,&
      2.0611707730536010E-02_dp,  3.8124292134014297E-02_dp,&
      -2.2565406289175886E-03_dp, -2.5747130192002296E-02_dp,&
      2.5568925787927566E-03_dp,  2.4379620109946373E-02_dp,&
      -2.0171664468884437E-02_dp,  3.7637966228315930E-03_dp,&
      2.3586802910798275E-03_dp, -9.7522517747690776E-04_dp,&
      2.5650771305620559E+00_dp, -3.8147293599496823E-01_dp,&
      -2.0907944912054602E-01_dp,  6.0270406124987005E-02_dp,&
      7.2175105576574167E-02_dp, -1.4263061186970644E-02_dp,&
      -5.0116196827266875E-02_dp,  8.8786858753379574E-03_dp,&
      5.3249935798219614E-02_dp, -5.2294415587463297E-02_dp,&
      1.7261637124157323E-02_dp,  9.2009373639893323E-04_dp,&
      -1.3430266626453476E-03_dp,  3.8828170221494824E+00_dp,&
      -8.4885642696270247E-01_dp, -4.2339961499366990E-01_dp,&
      1.8016120959522738E-01_dp,  1.6028857054099307E-01_dp,&
      -5.4950371377315503E-02_dp, -1.1487630174987833E-01_dp,&
      3.0013157516531445E-02_dp,  1.3585245394835330E-01_dp,&
      -1.5217460713811157E-01_dp,  6.4788928372283699E-02_dp,&
      -7.4203095016135562E-03_dp, -1.6019171078434956E-03_dp,&
      6.4636980648849729E+00_dp, -1.9099631989520829E+00_dp,&
      -8.6651521700026635E-01_dp,  4.9374292598778025E-01_dp,&
      3.4952603809371302E-01_dp, -1.7396151407323862E-01_dp,&
      -2.5934913691469813E-01_dp,  9.2007238929733004E-02_dp,&
      3.3903391371774966E-01_dp, -4.2381412446029582E-01_dp,&
      2.1088118199107522E-01_dp, -4.1855791726591811E-02_dp,&
      3.8884987212805058E-04_dp,  1.1321884964418704E+01_dp /)
    REAL(KIND=dp), DIMENSION(400), PARAMETER :: c05 = (/ &
      -4.2539604263036663E+00_dp, -1.7332279212896191E+00_dp,&
      1.2863899425408518E+00_dp,  7.2978346101421798E-01_dp,&
      -5.0542068060802536E-01_dp, -5.6453129220842824E-01_dp,&
      2.6844891959416117E-01_dp,  8.2065620557106000E-01_dp,&
      -1.1421990003730462E+00_dp,  6.4148861901481036E-01_dp,&
      -1.6396257011823606E-01_dp,  1.2796070085478475E-02_dp,&
      2.0487853209932744E+01_dp, -9.4517537769870792E+00_dp,&
      -3.3759733637452087E+00_dp,  3.2722131653337412E+00_dp,&
      1.4507872941308217E+00_dp, -1.4073698425446566E+00_dp,&
      -1.1842661757288730E+00_dp,  7.6778103478918880E-01_dp,&
      1.9398286469535142E+00_dp, -3.0258530426324368E+00_dp,&
      1.8858507556008846E+00_dp, -5.6545804339658101E-01_dp,&
      6.4128521008321809E-02_dp,  3.8095174335565567E+01_dp,&
      -2.1154495441708484E+01_dp, -6.3728233078180931E+00_dp,&
      8.2563638968778950E+00_dp,  2.6905728467335490E+00_dp,&
      -3.8356557488916927E+00_dp, -2.3727163256891193E+00_dp,&
      2.1906341809472898E+00_dp,  4.4719906295754370E+00_dp,&
      -7.9395530803829812E+00_dp,  5.4435322355460913E+00_dp,&
      -1.8339025028763316E+00_dp,  2.4941000766842886E-01_dp,&
      7.2794607863908539E+01_dp, -4.8093277265190174E+01_dp,&
      -1.1453054881238076E+01_dp,  2.0880285157258903E+01_dp,&
      4.3451760510124844E+00_dp, -1.0353879135082220E+01_dp,&
      -4.3940844971702182E+00_dp,  6.3120734481245231E+00_dp,&
      9.8973417965680408E+00_dp, -2.0647117445046018E+01_dp,&
      1.5538811111774422E+01_dp, -5.7518348230309915E+00_dp,&
      8.7742951648656331E-01_dp,  1.4332617622926281E+02_dp,&
      -1.1190083363446227E+02_dp, -1.8573930061634336E+01_dp,&
      5.3338145508390340E+01_dp,  4.4631414700445591E+00_dp,&
      -2.7868449098144676E+01_dp, -6.7549296888413908E+00_dp,&
      1.8541177811553482E+01_dp,  2.0031367391249091E+01_dp,&
      -5.2810064519241095E+01_dp,  4.3877834330533581E+01_dp,&
      -1.7658169541203652E+01_dp,  2.9331138916030355E+00_dp,&
      2.9197759925863352E+02_dp, -2.6852040701030040E+02_dp,&
      -2.2082783044921865E+01_dp,  1.3846051549368181E+02_dp,&
      -7.6189827475657532E+00_dp, -7.4936271383495338E+01_dp,&
      -4.2799690014967950E+00_dp,  5.5901437494488015E+01_dp,&
      3.1119328948155172E+01_dp, -1.2977676359921708E+02_dp,&
      1.2145063411930138E+02_dp, -5.3095774315419469E+01_dp,&
      9.4800311502936196E+00_dp,  6.1887006052817526E+02_dp,&
      -6.7028990777509409E+02_dp,  1.1201345750961481E+01_dp,&
      3.6684552949179283E+02_dp, -8.0419700058574776E+01_dp,&
      -2.0013407508924359E+02_dp,  3.0658023480604339E+01_dp,&
      1.7336606540154875E+02_dp, -4.2234575269977981E+00_dp,&
      -2.8652377100140529E+02_dp,  3.2016960404528265E+02_dp,&
      -1.5405177965842552E+02_dp,  2.9536421581524074E+01_dp,&
      1.3750163796936240E+03_dp, -1.7589799109221406E+03_dp,&
      2.3961277028254179E+02_dp,  9.9301157485970555E+02_dp,&
      -4.2575252374052224E+02_dp, -5.1817603461276519E+02_dp,&
      2.2602380506510636E+02_dp,  5.4799637021657588E+02_dp,&
      -3.8063255724265480E+02_dp, -4.3209967552717637E+02_dp,&
      7.3643375490972539E+02_dp, -4.1008578171950961E+02_dp,&
      8.5854890766784024E+01_dp,  3.2350942724992897E+03_dp,&
      -4.9196762475376900E+03_dp,  1.4005845015490424E+03_dp,&
      2.7240803683332629E+03_dp, -1.9809371044851687E+03_dp,&
      -1.1912780067707383E+03_dp,  1.1497315070977018E+03_dp,&
      1.6931690976040784E+03_dp, -2.6191658279561266E+03_dp,&
      6.8165720237798473E+02_dp,  9.6374997813179459E+02_dp,&
      -8.3276968242760586E+02_dp,  2.0676706863037907E+02_dp,&
      8.1777664155924858E+03_dp, -1.4947471572678814E+04_dp,&
      7.1193687656003467E+03_dp,  7.2704699942482303E+03_dp,&
      -9.0701497685694721E+03_dp, -1.3829479450049473E+03_dp,&
      5.2343291775954631E+03_dp,  4.3188817279079030E+03_dp,&
      -1.3395512259314450E+04_dp,  1.1338333668387049E+04_dp,&
      -4.0650141614603253E+03_dp,  1.8500545674922793E+02_dp,&
      1.8404119432961565E+02_dp,  2.2712516525332092E+04_dp,&
      -5.0737374807702967E+04_dp,  3.7414818324359119E+04_dp,&
      1.5002435412785790E+04_dp, -4.2430746564604517E+04_dp,&
      1.2806412701410625E+04_dp,  2.0720578130718106E+04_dp,&
      -2.6832946122836898E+02_dp, -5.0834266091037163E+04_dp,&
      7.0480937773566868E+04_dp, -4.6917539041131182E+04_dp,&
      1.6400456910054389E+04_dp, -2.4239470406997216E+03_dp,&
      7.2346780322228064E+04_dp, -2.0101001503791616E+05_dp,&
      2.2465714313474763E+05_dp, -4.1692320974158160E+04_dp,&
      -1.8428799273171904E+05_dp,  1.9351303403408095E+05_dp,&
      6.5115811053828320E+03_dp, -1.3528658618327085E+05_dp,&
      3.3663152446055996E+04_dp,  1.3655504570022557E+05_dp,&
      -1.6923190168772679E+05_dp,  8.5874640094980365E+04_dp,&
      -1.7241422711442276E+04_dp,  3.1939792849337222E+05_dp,&
      -1.0568561779921057E+06_dp,  1.6796432468799651E+06_dp,&
      -1.4288114404238255E+06_dp,  2.3272192404290100E+05_dp,&
      1.0799323551139913E+06_dp, -1.6297117460196065E+06_dp,&
      1.3266412491764352E+06_dp, -7.2330143312487204E+05_dp,&
      2.9143884697513317E+05_dp, -9.9073557291649820E+04_dp,&
      2.9147576845580217E+04_dp, -5.0077315833356834E+03_dp,&
      2.5399771849532937E-01_dp, -8.3033887225757580E-03_dp,&
      -7.6240573657738115E-03_dp,  2.8522290409030179E-03_dp,&
      -3.0540795652697220E-03_dp,  4.9663641512514419E-03_dp,&
      -5.8165865884663133E-03_dp,  5.9728067889719752E-03_dp,&
      -5.8369679736555158E-03_dp,  4.8909245070782462E-03_dp,&
      -3.0462113393202300E-03_dp,  1.2108511933957031E-03_dp,&
      -2.2780712761746828E-04_dp,  8.3638364305768520E-01_dp,&
      -3.2608326200435286E-02_dp, -2.9657073560502185E-02_dp,&
      1.1717032907011333E-02_dp, -1.1961865091520774E-02_dp,&
      1.9527087573284384E-02_dp, -2.2960112994842900E-02_dp,&
      2.3490832187062813E-02_dp, -2.2880044526118807E-02_dp,&
      1.9162416281042115E-02_dp, -1.1938349942081816E-02_dp,&
      4.7445678978824480E-03_dp, -8.9197277469357805E-04_dp,&
      1.6724533608911527E+00_dp, -8.5614488407303832E-02_dp,&
      -7.6755331134380211E-02_dp,  3.2773035396635355E-02_dp,&
      -3.1314839841457949E-02_dp,  5.1345159212853306E-02_dp,&
      -6.0742187908847688E-02_dp,  6.1817450219547841E-02_dp,&
      -5.9900400959394209E-02_dp,  5.0128559376177953E-02_dp,&
      -3.1246531279548323E-02_dp,  1.2416023551359569E-02_dp,&
      -2.3317856969346015E-03_dp,  3.0453512199679000E+00_dp,&
      -2.0863090315180247E-01_dp, -1.8408153868430499E-01_dp,&
      8.5169636305938420E-02_dp, -7.6204738289630397E-02_dp,&
      1.2525692165315666E-01_dp, -1.4923544481462697E-01_dp,&
      1.5103590459034816E-01_dp, -1.4548513388214768E-01_dp,&
      1.2163468207365730E-01_dp, -7.5871468061049080E-02_dp,&
      3.0148268152665792E-02_dp, -5.6563574453512922E-03_dp,&
      5.4638310929832095E+00_dp, -4.9229245214837142E-01_dp,&
      -4.2703130683166335E-01_dp,  2.1395029978012914E-01_dp,&
      -1.7998803117127329E-01_dp,  2.9569827835085399E-01_dp,&
      -3.5510720815802771E-01_dp,  3.5742646671849382E-01_dp,&
      -3.4201443477799331E-01_dp,  2.8560882144847355E-01_dp,&
      -1.7831621218447033E-01_dp,  7.0874969102503266E-02_dp,&
      -1.3286288180972624E-02_dp,  9.9026613863259954E+00_dp,&
      -1.1436820473123184E+00_dp, -9.7368133633730680E-01_dp,&
      5.2913145605271328E-01_dp, -4.1979778183307109E-01_dp,&
      6.8678629087967658E-01_dp, -8.3225011865141296E-01_dp,&
      8.3313008194203841E-01_dp, -7.9113948287700431E-01_dp,&
      6.5965573759522067E-01_dp, -4.1234076860417629E-01_dp,&
      1.6399646715620800E-01_dp, -3.0723929620273597E-02_dp,&
      1.8310843645113124E+01_dp, -2.6497670237209929E+00_dp,&
      -2.2084953089663300E+00_dp,  1.3073414834056065E+00_dp,&
      -9.8030423668319455E-01_dp,  1.5894351154630064E+00_dp,&
      -1.9463691136255397E+00_dp,  1.9378432641209999E+00_dp,&
      -1.8235505263636096E+00_dp,  1.5173210666977606E+00_dp,&
      -9.4993178976001713E-01_dp,  3.7824140655577221E-01_dp,&
      -7.0840946919823303E-02_dp,  3.4734198615946212E+01_dp,&
      -6.1913475701008718E+00_dp, -5.0342585633204626E+00_dp,&
      3.2672480067080283E+00_dp, -2.3210081478694335E+00_dp,&
      3.7057412387846540E+00_dp, -4.5938530888721134E+00_dp,&
      4.5493414807353512E+00_dp, -4.2340607625465676E+00_dp,&
      3.5125800180349342E+00_dp, -2.2034834637331495E+00_dp,&
      8.7903170473612890E-01_dp, -1.6467198958207585E-01_dp,&
      6.7907060498973522E+01_dp, -1.4739076394337873E+01_dp,&
      -1.1636844538005446E+01_dp,  8.3520518383431011E+00_dp,&
      -5.6406452024251346E+00_dp,  8.7907314207810714E+00_dp,&
      -1.1054622145984020E+01_dp,  1.0892905149402702E+01_dp,&
      -1.0000415655277743E+01_dp,  8.2597416232575309E+00_dp,&
      -5.1944403144978049E+00_dp,  2.0783350249549093E+00_dp,&
      -3.8976488880608956E-01_dp,  1.3754189884920987E+02_dp,&
      -3.6112261566861719E+01_dp, -2.7506089296448486E+01_dp,&
      2.2081510409742627E+01_dp, -1.4257561271282713E+01_dp,&
      2.1425371349192375E+01_dp, -2.7393711620526801E+01_dp,&
      2.6880120365914127E+01_dp, -2.4256007023950794E+01_dp,&
      1.9896595191742847E+01_dp, -1.2550072865137675E+01_dp,&
      5.0440291354648172E+00_dp, -9.4830475587057206E-01_dp,&
      2.9051478597825161E+02_dp, -9.2077480881216914E+01_dp,&
      -6.7042751616157403E+01_dp,  6.1122066501813805E+01_dp,&
      -3.8074088991537131E+01_dp,  5.4229813974137763E+01_dp,&
      -7.0650491998597872E+01_dp,  6.9148772922131428E+01_dp,&
      -6.1030422251934006E+01_dp,  4.9501109185729597E+01_dp,&
      -3.1320759235557357E+01_dp,  1.2672891600799508E+01_dp,&
      -2.3940255865041160E+00_dp,  6.4557191457606052E+02_dp,&
      -2.4763182690496433E+02_dp, -1.7000441338399474E+02_dp,&
      1.7979797134338500E+02_dp, -1.0963259713335488E+02_dp,&
      1.4445301204007444E+02_dp, -1.9199563810236253E+02_dp,&
      1.8801003542743408E+02_dp, -1.6118263890219302E+02_dp,&
      1.2820143562173234E+02_dp, -8.1268684993159169E+01_dp,&
      3.3201648029135029E+01_dp, -6.3257064933095632E+00_dp,&
      1.5281770790560115E+03_dp, -7.1515428321482318E+02_dp,&
      -4.5276182324024262E+02_dp,  5.7338834147225236E+02_dp,&
      -3.5031930447070647E+02_dp,  4.1265596366351411E+02_dp,&
      -5.5821259522651553E+02_dp,  5.4994852484809564E+02_dp,&
      -4.5353406400816010E+02_dp,  3.4777962951959489E+02_dp,&
      -2.1955613483579080E+02_dp,  9.0846938995140135E+01_dp,&
      -1.7557396032513779E+01_dp,  3.9261873180851890E+03_dp,&
      -2.2769021329796587E+03_dp, -1.2770843901758556E+03_dp,&
      2.0407998981381138E+03_dp, -1.2978099365872522E+03_dp,&
      1.3049821930118717E+03_dp, -1.7706647534018759E+03_dp,&
      1.7719691434495785E+03_dp, -1.3872679059180307E+03_dp,&
      9.8676308339813556E+02_dp, -6.0572190633134835E+02_dp,&
      2.5343498559405921E+02_dp, -5.0067017168943025E+01_dp,&
      1.1287275974519674E+04_dp, -8.3427028800217486E+03_dp,&
      -3.7990573642673103E+03_dp,  8.4927259072795641E+03_dp,&
      -5.9962833139676068E+03_dp,  4.8878274555726575E+03_dp,&
      -6.2818683353242614E+03_dp,  6.4697978795118643E+03_dp,&
      -4.7394997944678944E+03_dp,  2.8301056804072027E+03_dp,&
      -1.5087936765466591E+03_dp,  6.0963560380239039E+02_dp,&
      -1.2304467591412092E+02_dp,  3.8437627273177568E+04_dp,&
      -3.8176312227787312E+04_dp, -1.0634568511977122E+04_dp,&
      4.4919022886395134E+04_dp, -3.9765187408990765E+04_dp,&
      2.6349805711576602E+04_dp, -2.5672293572137780E+04_dp,&
      2.6451194932949369E+04_dp, -1.7901001830063731E+04_dp,&
      6.2663747959682069E+03_dp, -7.9046224464493122E+01_dp,&
      -7.7755945708786203E+02_dp,  2.1450377554813323E+02_dp,&
      1.7818869100699757E+05_dp, -2.6537447971430060E+05_dp,&
      3.1712593310074910E+04_dp,  3.4422146844366990E+05_dp,&
      -5.3053517524802696E+05_dp,  4.3145350047972775E+05_dp,&
      -1.9228471756741623E+05_dp, -3.1796392782362986E+04_dp,&
      1.7950338120613602E+05_dp, -2.3038705409991340E+05_dp,&
      1.8179053082015848E+05_dp, -8.5539393856822993E+04_dp,&
      1.8297112165331673E+04_dp,  2.0796368651156318E+00_dp,&
      -7.1211773323306998E-02_dp, -6.5222594833814318E-02_dp,&
      2.4758880150403027E-02_dp, -2.6172471480240402E-02_dp,&
      4.2607134478414486E-02_dp, -4.9952200375915141E-02_dp,&
      5.1244059328384543E-02_dp, -5.0035111663177034E-02_dp,&
      4.1920311429239167E-02_dp, -2.6111046961287777E-02_dp /)
    REAL(KIND=dp), DIMENSION(400), PARAMETER :: c06 = (/ &
      1.0378429926308540E-02_dp, -1.9521878957809889E-03_dp,&
      2.6944073713370376E+00_dp, -1.3634250253884062E-01_dp,&
      -1.2248962009310110E-01_dp,  5.1739894797576326E-02_dp,&
      -4.9861795300424228E-02_dp,  8.1764747263490686E-02_dp,&
      -9.6628896510658951E-02_dp,  9.8409054684497427E-02_dp,&
      -9.5437204809848047E-02_dp,  7.9879375951563511E-02_dp,&
      -4.9785180785078605E-02_dp,  1.9781814649150051E-02_dp,&
      -3.7155106897152532E-03_dp,  4.1754065857664679E+00_dp,&
      -3.1351013649643816E-01_dp, -2.7633289430167707E-01_dp,&
      1.2854478131208438E-01_dp, -1.1437860765449445E-01_dp,&
      1.8829614057148475E-01_dp, -2.2438400638247086E-01_dp,&
      2.2698373182776574E-01_dp, -2.1859649315655161E-01_dp,&
      1.8276068899158232E-01_dp, -1.1399571200820550E-01_dp,&
      4.5292156705966799E-02_dp, -8.4963179977451710E-03_dp,&
      7.1321772449454741E+00_dp, -7.2704365835075646E-01_dp,&
      -6.2939140508259084E-01_dp,  3.1836503791119075E-01_dp,&
      -2.6546680650849463E-01_dp,  4.3689678639928536E-01_dp,&
      -5.2499672256707541E-01_dp,  5.2799497456180577E-01_dp,&
      -5.0495196946337140E-01_dp,  4.2165902676363498E-01_dp,&
      -2.6325905625353568E-01_dp,  1.0462371284509191E-01_dp,&
      -1.9608344465334434E-02_dp,  1.2834515530424907E+01_dp,&
      -1.6706472138204553E+00_dp, -1.4188832578032204E+00_dp,&
      7.7932291344618254E-01_dp, -6.1248631370805595E-01_dp,&
      1.0036315751279516E+00_dp, -1.2172307293345568E+00_dp,&
      1.2173794784926901E+00_dp, -1.1552141520797918E+00_dp,&
      9.6318085874165005E-01_dp, -6.0209450103939621E-01_dp,&
      2.3943637699340395E-01_dp, -4.4845635855033761E-02_dp,&
      2.3907644544268969E+01_dp, -3.8409238287608631E+00_dp,&
      -3.1927866471055242E+00_dp,  1.9107508410085583E+00_dp,&
      -1.4194444132695128E+00_dp,  2.3047039849449251E+00_dp,&
      -2.8251319104705459E+00_dp,  2.8099411726428549E+00_dp,&
      -2.6421180276849054E+00_dp,  2.1983609044304298E+00_dp,&
      -1.3764093323588191E+00_dp,  5.4799175204302053E-01_dp,&
      -1.0260464712972778E-01_dp,  4.5900219693354039E+01_dp,&
      -8.9317693484926668E+00_dp, -7.2428742526117622E+00_dp,&
      4.7498551429452860E+00_dp, -3.3446060263422406E+00_dp,&
      5.3474540805623176E+00_dp, -6.6363730788695472E+00_dp,&
      6.5652622867554413E+00_dp, -6.1054269491608801E+00_dp,&
      5.0652251409978781E+00_dp, -3.1778332100528144E+00_dp,&
      1.2675663829725419E+00_dp, -2.3738207936101491E-01_dp,&
      9.0939009801418251E+01_dp, -2.1207951305576458E+01_dp,&
      -1.6703836717599909E+01_dp,  1.2094601423920158E+01_dp,&
      -8.1037239824558487E+00_dp,  1.2652193762144261E+01_dp,&
      -1.5927629386120495E+01_dp,  1.5678136007593519E+01_dp,&
      -1.4384696049040816E+01_dp,  1.1883309669508408E+01_dp,&
      -7.4743470332662731E+00_dp,  2.9899963557394957E+00_dp,&
      -5.6051343972032064E-01_dp,  1.8664874611056891E+02_dp,&
      -5.1882278871112675E+01_dp, -3.9458004229447603E+01_dp,&
      3.1854721012338533E+01_dp, -2.0425886295138511E+01_dp,&
      3.0791578686657544E+01_dp, -3.9403681473957676E+01_dp,&
      3.8622643042566317E+01_dp, -3.4847520475636244E+01_dp,&
      2.8602543521658834E+01_dp, -1.8045198909363659E+01_dp,&
      7.2502140242936921E+00_dp, -1.3622957283347905E+00_dp,&
      3.9915188049530610E+02_dp, -1.3203707855054867E+02_dp,&
      -9.6178119362902649E+01_dp,  8.7705958866288142E+01_dp,&
      -5.4284439736638681E+01_dp,  7.7802647990103949E+01_dp,&
      -1.0141935559157001E+02_dp,  9.9136132884069752E+01_dp,&
      -8.7572824663943678E+01_dp,  7.1146305460401948E+01_dp,&
      -4.5034774737857425E+01_dp,  1.8210581017182808E+01_dp,&
      -3.4368084704922492E+00_dp,  8.9640126039779705E+02_dp,&
      -3.5362542578945090E+02_dp, -2.4381286270301160E+02_dp,&
      2.5567043117350039E+02_dp, -1.5473035694623934E+02_dp,&
      2.0643065091772985E+02_dp, -2.7449876024024695E+02_dp,&
      2.6829509790813256E+02_dp, -2.3063853177772037E+02_dp,&
      1.8419775073300784E+02_dp, -1.1690662622678340E+02_dp,&
      4.7711080741790596E+01_dp, -9.0740124097056771E+00_dp,&
      2.1371273099507980E+03_dp, -1.0117484690800054E+03_dp,&
      -6.4833923675694632E+02_dp,  8.0221902155722398E+02_dp,&
      -4.8409799486432814E+02_dp,  5.8407650925627354E+02_dp,&
      -7.9151735905004318E+02_dp,  7.7715699922847898E+02_dp,&
      -6.4473901677909225E+02_dp,  4.9950084061013206E+02_dp,&
      -3.1670516768758597E+02_dp,  1.3088948582201553E+02_dp,&
      -2.5216009702956097E+01_dp,  5.4941634656312544E+03_dp,&
      -3.1579566223942602E+03_dp, -1.8240594402316519E+03_dp,&
      2.7722798032040118E+03_dp, -1.7188010021120501E+03_dp,&
      1.8049036287011472E+03_dp, -2.4720525322843669E+03_dp,&
      2.4575593777441154E+03_dp, -1.9454017485166187E+03_dp,&
      1.4216353652861419E+03_dp, -8.8794262472737205E+02_dp,&
      3.7236649124083812E+02_dp, -7.3224838489909345E+01_dp,&
      1.5598907311734893E+04_dp, -1.1102735305658800E+04_dp,&
      -5.4496895758397304E+03_dp,  1.0920783488778621E+04_dp,&
      -7.2896930341331999E+03_dp,  6.3648779067823270E+03_dp,&
      -8.5400815400530537E+03_dp,  8.7029682626190970E+03_dp,&
      -6.4829758274370470E+03_dp,  4.1818738901113929E+03_dp,&
      -2.4318300865338133E+03_dp,  1.0217539466953230E+03_dp,&
      -2.0696183440979297E+02_dp,  5.0910784596133592E+04_dp,&
      -4.6410694622357332E+04_dp, -1.6759575861048310E+04_dp,&
      5.1878639177724821E+04_dp, -4.0256109414878149E+04_dp,&
      2.8334810650844160E+04_dp, -3.3461954000327562E+04_dp,&
      3.5317418178144384E+04_dp, -2.4510772031663375E+04_dp,&
      1.1577080835902063E+04_dp, -4.3532742678888289E+03_dp,&
      1.4492552333627673E+03_dp, -2.9351788438751623E+02_dp,&
      2.0646310805017679E+05_dp, -2.5649451003031401E+05_dp,&
      -3.4599088204528663E+04_dp,  3.2876093443253404E+05_dp,&
      -3.4295275058681634E+05_dp,  2.1298567642959324E+05_dp,&
      -1.5333540067239694E+05_dp,  1.4673765999310571E+05_dp,&
      -9.1327695628929680E+04_dp,  4.0267201625102462E+03_dp,&
      3.8179083987894133E+04_dp, -2.6616546576670629E+04_dp,&
      6.3775166535376393E+03_dp,  1.2836671172308552E+06_dp,&
      -2.4267891820420851E+06_dp,  9.8426137787763157E+05_dp,&
      3.0244705295962645E+06_dp, -6.6314140027439278E+06_dp,&
      6.7915841815625075E+06_dp, -3.0584748882667138E+06_dp,&
      -2.3579722151722158E+06_dp,  6.4360152608277015E+06_dp,&
      -7.2732554452784210E+06_dp,  5.1574036293220147E+06_dp,&
      -2.2174478505159747E+06_dp,  4.4243860364206659E+05_dp,&
      2.5576624471415554E-01_dp,  4.2237570088002974E-09_dp,&
      -1.0860357249158544E-02_dp,  1.3291431359068883E-02_dp,&
      -1.6900731238607920E-02_dp,  1.3962484092300874E-02_dp,&
      5.5824330603980879E-03_dp, -5.1241539388943998E-02_dp,&
      1.1924290064807870E-01_dp, -1.7653776055574968E-01_dp,&
      1.7399289115806521E-01_dp, -1.0429706396163735E-01_dp,&
      2.8742956296286825E-02_dp,  8.4334174286158137E-01_dp,&
      1.6647331740500862E-08_dp, -4.2808548044690892E-02_dp,&
      5.2391164460175478E-02_dp, -6.6127942116811278E-02_dp,&
      5.3832866343907902E-02_dp,  2.4299402435477839E-02_dp,&
      -2.0540730598820642E-01_dp,  4.7389926254432846E-01_dp,&
      -6.9894838310754248E-01_dp,  6.8730283012545457E-01_dp,&
      -4.1135207214867037E-01_dp,  1.1323604650004115E-01_dp,&
      1.6907728029821778E+00_dp,  4.3944714380629980E-08_dp,&
      -1.1301966435617677E-01_dp,  1.3831887050265168E-01_dp,&
      -1.7265894786567948E-01_dp,  1.3739412392853392E-01_dp,&
      7.3149394658010375E-02_dp, -5.5568362275266503E-01_dp,&
      1.2661948729096475E+00_dp, -1.8571387585078984E+00_dp,&
      1.8200395995789005E+00_dp, -1.0867803636088169E+00_dp,&
      2.9866422519059560E-01_dp,  3.0901288057755520E+00_dp,&
      1.0772079547945405E-07_dp, -2.7708675910432151E-01_dp,&
      3.3911191201867580E-01_dp, -4.1813678737718835E-01_dp,&
      3.2415947918917831E-01_dp,  2.0334817882080883E-01_dp,&
      -1.3978592144512130E+00_dp,  3.1438040492554027E+00_dp,&
      -4.5835634903417697E+00_dp,  4.4755448980546646E+00_dp,&
      -2.6656753625759242E+00_dp,  7.3121673386587283E-01_dp,&
      5.5698272110483344E+00_dp,  2.5576027456667277E-07_dp,&
      -6.5799686641669253E-01_dp,  8.0528746329636880E-01_dp,&
      -9.8004848950174905E-01_dp,  7.3810118982122419E-01_dp,&
      5.4249771450196238E-01_dp, -3.4069578044619200E+00_dp,&
      7.5616388404833437E+00_dp, -1.0956710664782413E+01_dp,&
      1.0657500411874638E+01_dp, -6.3307799811259393E+00_dp,&
      1.7331860358171081E+00_dp,  1.0149759614132440E+01_dp,&
      5.9816188704636412E-07_dp, -1.5391916143113304E+00_dp,&
      1.8837342032506716E+00_dp, -2.2598790657906993E+00_dp,&
      1.6463608426933884E+00_dp,  1.4189272312199954E+00_dp,&
      -8.1875793783310566E+00_dp,  1.7923938935972270E+01_dp,&
      -2.5801297646966532E+01_dp,  2.4993035113196413E+01_dp,&
      -1.4803360412194611E+01_dp,  4.0440613015781999E+00_dp,&
      1.8885544797422838E+01_dp,  1.3962193082829181E-06_dp,&
      -3.5935492300793954E+00_dp,  4.3979504208452127E+00_dp,&
      -5.1910280549710484E+00_dp,  3.6347018516401288E+00_dp,&
      3.7005632026129085E+00_dp, -1.9673683832970518E+01_dp,&
      4.2439595400855616E+01_dp, -6.0651898779119989E+01_dp,&
      5.8481834431875996E+01_dp, -3.4526009414347030E+01_dp,&
      9.4091800758801245E+00_dp,  3.6082927628028706E+01_dp,&
      3.2901193014272859E-06_dp, -8.4702295022993539E+00_dp,&
      1.0366249728117493E+01_dp, -1.2006521585567233E+01_dp,&
      8.0045054262700059E+00_dp,  9.7574715974163553E+00_dp,&
      -4.7844176520379150E+01_dp,  1.0156232558264223E+02_dp,&
      -1.4397490593201323E+02_dp,  1.3809637723282293E+02_dp,&
      -8.1222582326131445E+01_dp,  2.2073010705828167E+01_dp,&
      7.1134245645863814E+01_dp,  7.9097869358485311E-06_dp,&
      -2.0369747780417786E+01_dp,  2.4929401403798764E+01_dp,&
      -2.8233117758625095E+01_dp,  1.7675076772578425E+01_dp,&
      2.6333062643496763E+01_dp, -1.1907948950314673E+02_dp,&
      2.4831111955679535E+02_dp, -3.4875954493874565E+02_dp,&
      3.3248082048627452E+02_dp, -1.9468856176014137E+02_dp,&
      5.2732162646628915E+01_dp,  1.4549672148303156E+02_dp,&
      1.9606098644651494E-05_dp, -5.0510545608226444E+01_dp,&
      6.1816996598134779E+01_dp, -6.8124325068817015E+01_dp,&
      3.9197095706813819E+01_dp,  7.3632728580914488E+01_dp,&
      -3.0676194479178452E+02_dp,  6.2695742728703999E+02_dp,&
      -8.7111493883908997E+02_dp,  8.2443456649865914E+02_dp,&
      -4.8018933005153883E+02_dp,  1.2953318523991635E+02_dp,&
      3.1094648618254405E+02_dp,  5.0696414767541625E-05_dp,&
      -1.3067261929794532E+02_dp,  1.5992265246911447E+02_dp,&
      -1.7032412326959059E+02_dp,  8.6865456810714306E+01_dp,&
      2.1627551271719494E+02_dp, -8.2833324696946715E+02_dp,&
      1.6543740153928604E+03_dp, -2.2691680798433426E+03_dp,&
      2.1285371190344981E+03_dp, -1.2315575770352384E+03_dp,&
      3.3052109092285252E+02_dp,  7.0102332426653004E+02_dp,&
      1.3873488339462509E-04_dp, -3.5783165787516657E+02_dp,&
      4.3792884674714333E+02_dp, -4.4617993522197833E+02_dp,&
      1.8813865534569888E+02_dp,  6.7882008059907184E+02_dp,&
      -2.3814302002445670E+03_dp,  4.6290899430085301E+03_dp,&
      -6.2490819879456340E+03_dp,  5.7958888003535612E+03_dp,&
      -3.3247298296427966E+03_dp,  8.8628594048774914E+02_dp,&
      1.6902170532998441E+03_dp,  4.0974181820914717E-04_dp,&
      -1.0577892962329620E+03_dp,  1.2945628573020617E+03_dp,&
      -1.2412789684761844E+03_dp,  3.6518353070382403E+02_dp,&
      2.3309880593651797E+03_dp, -7.4457710895261789E+03_dp,&
      1.4002929494342166E+04_dp, -1.8518696747336475E+04_dp,&
      1.6917288236393451E+04_dp, -9.5900986554442443E+03_dp,&
      2.5323928944806203E+03_dp,  4.4504521254867541E+03_dp,&
      1.3448249990557044E-03_dp, -3.4764938517534570E+03_dp,&
      4.2546538860367436E+03_dp, -3.7288910130907971E+03_dp,&
      3.3785610612894226E+02_dp,  9.0749096864554558E+03_dp,&
      -2.6126306878210624E+04_dp,  4.7085181240615682E+04_dp,&
      -6.0517025375512349E+04_dp,  5.4074356432769353E+04_dp,&
      -3.0109150792826324E+04_dp,  7.8344472467430223E+03_dp,&
      1.3254579477277694E+04_dp,  5.1532479271565908E-03_dp,&
      -1.3351228791461199E+04_dp,  1.6339623735533780E+04_dp,&
      -1.2301930953556473E+04_dp, -3.6684681819111252E+03_dp,&
      4.2559142880279360E+04_dp, -1.0831862123749210E+05_dp /)
    REAL(KIND=dp), DIMENSION(13, 34, 6), PARAMETER :: &
      coefdata = RESHAPE ((/ c01, c02, c03, c04, c05, c06, c07 /),(/13,34,6/))

    INTEGER                                  :: irange

     IF(Rc <=  2.0000000000000000E+01_dp) Rc =   2.0000000000000000E+01_dp
     IF(Rc >=  1.3327491760253906E+06_dp) Rc =   1.3327491760253906E+06_dp
     IF(Rc >=  2.0000000000000000E+01_dp .AND. Rc <=  3.0823724363668457E+01_dp) THEN
       irange =  1
       L_b =  2.0000000000000000E+01_dp
       U_b =  3.0823724363668457E+01_dp
     END IF
     IF(Rc >=  3.0823724363668457E+01_dp .AND. Rc <=  3.1985602006226213E+02_dp) THEN
       irange =  2
       L_b =  3.0823724363668457E+01_dp
       U_b =  3.1985602006226213E+02_dp
     END IF
     IF(Rc >=  3.1985602006226213E+02_dp .AND. Rc <=  4.5183964268225072E+03_dp) THEN
       irange =  3
       L_b =  3.1985602006226213E+02_dp
       U_b =  4.5183964268225072E+03_dp
     END IF
     IF(Rc >=  4.5183964268225072E+03_dp .AND. Rc <=  7.8644668751788398E+04_dp) THEN
       irange =  4
       L_b =  4.5183964268225072E+03_dp
       U_b =  7.8644668751788398E+04_dp
     END IF
     IF(Rc >=  7.8644668751788398E+04_dp .AND. Rc <=  4.6977420089882286E+05_dp) THEN
       irange =  5
       L_b =  7.8644668751788398E+04_dp
       U_b =  4.6977420089882286E+05_dp
     END IF
     IF(Rc >=  4.6977420089882286E+05_dp .AND. Rc <=  1.3327491760253906E+06_dp) THEN
       irange =  6
       L_b =  4.6977420089882286E+05_dp
       U_b =  1.3327491760253906E+06_dp
     END IF

     fit_coef(1:13,1:34) = coefdata(1:13,1:34,irange)

   END SUBROUTINE get_fit_coef_k17

! *****************************************************************************
!> \brief ...
!> \param Rc ...
!> \param L_b ...
!> \param U_b ...
!> \param fit_coef ...
! *****************************************************************************
   SUBROUTINE get_fit_coef_k18(Rc,L_b,U_b,fit_coef)
    REAL(KIND=dp)                            :: Rc, L_b, U_b
    REAL(KIND=dp), DIMENSION(13, 36)         :: fit_coef

    REAL(KIND=dp), DIMENSION(340), PARAMETER :: c06 = (/ &
      -4.2209902072425757E+02_dp,  1.1367036683435998E+02_dp,&
      2.4004100563047302E+02_dp,  2.6442809338541683E-05_dp,&
      -9.2920139462935452E+01_dp,  1.1857226657184384E+02_dp,&
      -1.3712334751086243E+02_dp,  8.9614212066663356E+01_dp,&
      1.2954531399327851E+02_dp, -6.1204117014894439E+02_dp,&
      1.3038237052684744E+03_dp, -1.8532736683780777E+03_dp,&
      1.7733881888322414E+03_dp, -1.0349286327234870E+03_dp,&
      2.7775933648182263E+02_dp,  5.1299854071485697E+02_dp,&
      6.7687074050697422E-05_dp, -2.3807455011160289E+02_dp,&
      3.0379853851555936E+02_dp, -3.3999113773695723E+02_dp,&
      2.0053502227186451E+02_dp,  3.8602011516394481E+02_dp,&
      -1.6462886216839220E+03_dp,  3.4237001138452265E+03_dp,&
      -4.8064857217217213E+03_dp,  4.5625744054824872E+03_dp,&
      -2.6475930973206077E+03_dp,  7.0759015679511151E+02_dp,&
      1.1565450996747870E+03_dp,  1.8361085330211829E-04_dp,&
      -6.4662037045130342E+02_dp,  8.2512803376189629E+02_dp,&
      -8.8496728903978203E+02_dp,  4.4602324539501097E+02_dp,&
      1.2288632621141262E+03_dp, -4.7252668279211357E+03_dp,&
      9.5564883176047988E+03_dp, -1.3215309577113516E+04_dp,&
      1.2419557109971771E+04_dp, -7.1549681548517901E+03_dp,&
      1.9018665194857740E+03_dp,  2.7885119751535453E+03_dp,&
      5.3798578681814367E-04_dp, -1.8979726842190619E+03_dp,&
      2.4219260114581057E+03_dp, -2.4509451315420783E+03_dp,&
      9.3302092536482598E+02_dp,  4.2792080086216884E+03_dp,&
      -1.4781335864606486E+04_dp,  2.8910881818664639E+04_dp,&
      -3.9221458601856801E+04_dp,  3.6377452574481533E+04_dp,&
      -2.0754440568654329E+04_dp,  5.4759314189430270E+03_dp,&
      7.3423343437531503E+03_dp,  1.7523478403478510E-03_dp,&
      -6.1987624682842961E+03_dp,  7.9099609072407902E+03_dp,&
      -7.3469720853921908E+03_dp,  1.3590388991922907E+03_dp,&
      1.6895282676913615E+04_dp, -5.2015173713595323E+04_dp,&
      9.7530203120827064E+04_dp, -1.2891927678944364E+05_dp,&
      1.1735562908487483E+05_dp, -6.6007676931770868E+04_dp,&
      1.7222837048140107E+04_dp,  2.1867340894031433E+04_dp,&
      6.6610965479193367E-03_dp, -2.3670256838196394E+04_dp,&
      3.0204363587247230E+04_dp, -2.4289431365742868E+04_dp,&
      -4.4793087397477648E+03_dp,  8.0399556072960419E+04_dp,&
      -2.1698457697832960E+05_dp,  3.8372266226749582E+05_dp,&
      -4.8749358651122235E+05_dp,  4.3044669176928164E+05_dp,&
      -2.3628550514088655E+05_dp,  6.0447814633299982E+04_dp,&
      7.8888258338212341E+04_dp,  3.2559081234041241E-02_dp,&
      -1.1677721904936734E+05_dp,  1.4901143678641456E+05_dp,&
      -8.7205284381827863E+04_dp, -1.0593928818725457E+05_dp,&
      5.2180063081218430E+05_dp, -1.1828235545327321E+06_dp,&
      1.8923972467739715E+06_dp, -2.2220643842051183E+06_dp,&
      1.8334827911676331E+06_dp, -9.4845074792786303E+05_dp,&
      2.3035540725488932E+05_dp,  4.1277266013211373E+05_dp,&
      2.6625034408035703E-01_dp, -9.8168542137299629E+05_dp,&
      1.2526189136341743E+06_dp, -1.6524351056354993E+04_dp,&
      -2.7322480720883626E+06_dp,  6.3977913782139570E+06_dp,&
      -9.5302632217170522E+06_dp,  9.9340828649581820E+06_dp,&
      -6.5415577947778786E+06_dp,  1.6425385275771332E+06_dp,&
      9.1574731282429548E+05_dp, -6.1255945160736761E+05_dp,&
      2.0360025664444423E+00_dp,  2.5713335383125262E-08_dp,&
      -9.0033864102026753E-02_dp,  1.1488977052985240E-01_dp,&
      -1.5339668231823797E-01_dp,  1.3943382095570961E-01_dp,&
      2.3403104409030068E-02_dp, -4.3679119831297458E-01_dp,&
      1.0806320687760871E+00_dp, -1.6418760026934989E+00_dp,&
      1.6349608211334608E+00_dp, -9.8011211272180954E-01_dp,&
      2.6815864103304071E-01_dp,  2.6117528386499291E+00_dp,&
      4.8175407514969741E-08_dp, -1.6873111063388158E-01_dp,&
      2.1531310300017245E-01_dp, -2.8376201294612502E-01_dp,&
      2.5179501088112777E-01_dp,  6.2880810206506932E-02_dp,&
      -8.4876332472530991E-01_dp,  2.0624430496348749E+00_dp,&
      -3.1112002808161887E+00_dp,  3.0855832382580046E+00_dp,&
      -1.8448645916831607E+00_dp,  5.0382953298570421E-01_dp,&
      3.9873063048304820E+00_dp,  1.0846158670211902E-07_dp,&
      -3.7998757133131345E-01_dp,  4.8489142926013917E-01_dp,&
      -6.3086126110902885E-01_dp,  5.4610493299372398E-01_dp,&
      1.8329858289416012E-01_dp, -1.9772478991069387E+00_dp,&
      4.7253027814453725E+00_dp, -7.0796876549675778E+00_dp,&
      6.9941004208554265E+00_dp, -4.1711166061111760E+00_dp,&
      1.1370891467073994E+00_dp,  6.7001043314976272E+00_dp,&
      2.4667185517548435E-07_dp, -8.6443414945752739E-01_dp,&
      1.1030796184932681E+00_dp, -1.4180463633457838E+00_dp,&
      1.1985397640306839E+00_dp,  5.0364134607866440E-01_dp,&
      -4.6338532066208620E+00_dp,  1.0914232611086101E+01_dp,&
      -1.6252611187619642E+01_dp,  1.5999475073408924E+01_dp,&
      -9.5194949221022522E+00_dp,  2.5908465591882055E+00_dp,&
      1.1851418633552694E+01_dp,  5.5547702724000489E-07_dp,&
      -1.9471816828933401E+00_dp,  2.4847416347076288E+00_dp,&
      -3.1545013399132991E+00_dp,  2.5980360723724814E+00_dp,&
      1.3345505038407754E+00_dp, -1.0749199950188043E+01_dp,&
      2.4957927423377040E+01_dp, -3.6937125272225181E+01_dp,&
      3.6230950319225336E+01_dp, -2.1505397766830995E+01_dp,&
      5.8430179763106009E+00_dp,  2.1671718972763752E+01_dp,&
      1.2490628723124012E-06_dp, -4.3799641771907298E+00_dp,&
      5.5891415983185286E+00_dp, -6.9981143458634438E+00_dp,&
      5.5940275426254855E+00_dp,  3.4904237077772149E+00_dp,&
      -2.4933029773709567E+01_dp,  5.7033160660992166E+01_dp,&
      -8.3854458025895269E+01_dp,  8.1931375876771739E+01_dp,&
      -4.8504788239908393E+01_dp,  1.3154247835903428E+01_dp,&
      4.0763243915536528E+01_dp,  2.8331268185654416E-06_dp,&
      -9.9385399289344427E+00_dp,  1.2682267211216036E+01_dp,&
      -1.5631827819043766E+01_dp,  1.2059187258319630E+01_dp,&
      9.1510546139662399E+00_dp, -5.8458649242606107E+01_dp,&
      1.3161495724281883E+02_dp, -1.9212585630556700E+02_dp,&
      1.8691186329645851E+02_dp, -1.1033319945556754E+02_dp,&
      2.9859331950975829E+01_dp,  7.8908010778509308E+01_dp,&
      6.5386761281513677E-06_dp, -2.2948202615132029E+01_dp,&
      2.9283481126861815E+01_dp, -3.5445800826485623E+01_dp,&
      2.6183727841564750E+01_dp,  2.4329279868365493E+01_dp,&
      -1.3982752010927283E+02_dp,  3.0947642362503592E+02_dp,&
      -4.4818522192425121E+02_dp,  4.3391508597732701E+02_dp,&
      -2.5529342544325627E+02_dp,  6.8925136413432071E+01_dp,&
      1.5767917857157383E+02_dp,  1.5478496535958464E-05_dp,&
      -5.4354086424776057E+01_dp,  6.9359488995831569E+01_dp,&
      -8.2189060026724775E+01_dp,  5.7491106121444517E+01_dp,&
      6.6265828876160398E+01_dp, -3.4412046626243608E+02_dp,&
      7.4760950475972925E+02_dp, -1.0730888794956911E+03_dp,&
      1.0331995461163826E+03_dp, -6.0557217116128845E+02_dp,&
      1.6304262960925644E+02_dp,  3.2678336205766686E+02_dp,&
      3.7894877275203884E-05_dp, -1.3316371456228885E+02_dp,&
      1.6992573781710709E+02_dp, -1.9629816380606513E+02_dp,&
      1.2788120327692410E+02_dp,  1.8683976508725286E+02_dp,&
      -8.7919648882012120E+02_dp,  1.8714058585605610E+03_dp,&
      -2.6590278379766187E+03_dp,  2.5438261029985429E+03_dp,&
      -1.4843168619653752E+03_dp,  3.9832397479258418E+02_dp,&
      7.0684637540150163E+02_dp,  9.6867269795911109E-05_dp,&
      -3.4069427410525071E+02_dp,  4.3474797507118456E+02_dp,&
      -4.8678867720092836E+02_dp,  2.8761368885618197E+02_dp,&
      5.5179921872736327E+02_dp, -2.3562414577310824E+03_dp,&
      4.9020823980878231E+03_dp, -6.8836792573968105E+03_dp,&
      6.5355383068211950E+03_dp, -3.7929919281300995E+03_dp,&
      1.0138137129943455E+03_dp,  1.6094255057648668E+03_dp,&
      2.6161705713683781E-04_dp, -9.2119908813007453E+02_dp,&
      1.1755078805571836E+03_dp, -1.2652473969070488E+03_dp,&
      6.4695642698297149E+02_dp,  1.7319053641298342E+03_dp,&
      -6.7103673916556281E+03_dp,  1.3602270272451808E+04_dp,&
      -1.8834904642552094E+04_dp,  1.7716918035172537E+04_dp,&
      -1.0213676976914645E+04_dp,  2.7162972544969493E+03_dp,&
      3.9033474711962631E+03_dp,  7.5844834317149079E-04_dp,&
      -2.6748123810493139E+03_dp,  3.4132212597088978E+03_dp,&
      -3.4872403633089002E+03_dp,  1.3999647101781925E+03_dp,&
      5.8893036531862190E+03_dp, -2.0663289718638232E+04_dp,&
      4.0631504446060877E+04_dp, -5.5298982600424795E+04_dp,&
      5.1405733453152017E+04_dp, -2.9378611655370314E+04_dp,&
      7.7615915923671664E+03_dp,  1.0259227531224709E+04_dp,&
      2.4144621632299105E-03_dp, -8.5342291700014030E+03_dp,&
      1.0890155897749342E+04_dp, -1.0343731878628403E+04_dp,&
      2.4584152986258155E+03_dp,  2.2303664961658811E+04_dp,&
      -7.0523457767913642E+04_dp,  1.3364816601587992E+05_dp,&
      -1.7787359000140551E+05_dp,  1.6273980884683644E+05_dp,&
      -9.1893860960891921E+04_dp,  2.4051527420140064E+04_dp,&
      3.0018788916766829E+04_dp,  8.7440843910547086E-03_dp,&
      -3.1015062922638364E+04_dp,  3.9576778234732534E+04_dp,&
      -3.3617796506662242E+04_dp, -1.2664803729988839E+03_dp,&
      9.8208694230122404E+04_dp, -2.7712338104205776E+05_dp,&
      5.0070250342457759E+05_dp, -6.4583598645926779E+05_dp,&
      5.7713900782644108E+05_dp, -3.1991818541374581E+05_dp,&
      8.2502449933280441E+04_dp,  1.0243629251738689E+05_dp,&
      3.8315196281821000E-02_dp, -1.3673145392947458E+05_dp,&
      1.7447480748440127E+05_dp, -1.2128767346456964E+05_dp,&
      -7.4731473660592732E+04_dp,  5.4209346541303862E+05_dp,&
      -1.3366186725741250E+06_dp,  2.2534054543823483E+06_dp,&
      -2.7624422501378348E+06_dp,  2.3684409505399591E+06_dp,&
      -1.2681911269942287E+06_dp,  3.1767018404991477E+05_dp,&
      4.4811404216243100E+05_dp,  2.3079117700841423E-01_dp,&
      -8.3433208879112056E+05_dp,  1.0646234386996420E+06_dp,&
      -4.3900763949985575E+05_dp, -1.2299926310603216E+06_dp,&
      4.3873424472257951E+06_dp, -8.9070735680865049E+06_dp,&
      1.3145595299621349E+07_dp, -1.4320366639666997E+07_dp,&
      1.0965291282036081E+07_dp, -5.2620189925113497E+06_dp,&
      1.1859559777996268E+06_dp,  3.2872930098168775E+06_dp,&
      2.7482124839381648E+00_dp, -1.0319720350104814E+07_dp,&
      1.3167559540232308E+07_dp,  5.3141307787938649E+06_dp,&
      -4.2806172437992170E+07_dp,  7.7700737994820118E+07_dp,&
      -7.6880123309893116E+07_dp,  1.7343391064140473E+07_dp,&
      7.7860350009209916E+07_dp, -1.3364460712376091E+08_dp,&
      1.0093529309506236E+08_dp, -3.1433866759847842E+07_dp /)
    REAL(KIND=dp), DIMENSION(400), PARAMETER :: c01 = (/ &
      1.8104907156518354E-01_dp, -6.1817005353024562E-02_dp,&
      -1.5365071642354823E-02_dp,  9.1489217818269322E-03_dp,&
      4.9967787273344161E-03_dp, -3.5367914652488903E-03_dp,&
      -2.2075007488833458E-03_dp,  1.2029276142200300E-03_dp,&
      2.8328851353903797E-03_dp, -3.4438772536686739E-03_dp,&
      1.4299272644572190E-03_dp, -1.0931267316481758E-04_dp,&
      -5.5894388646958038E-05_dp,  5.6894444769190411E-01_dp,&
      -2.1274164519782207E-01_dp, -4.2890815190342713E-02_dp,&
      3.5035480705044961E-02_dp,  1.3874240699660001E-02_dp,&
      -1.4090755641772217E-02_dp, -6.0016188181493280E-03_dp,&
      5.1060704779823706E-03_dp,  9.3683628268561563E-03_dp,&
      -1.3681594752909918E-02_dp,  7.1674311571180219E-03_dp,&
      -1.4730236769969446E-03_dp,  1.7117880701502458E-05_dp,&
      1.0389637522125577E+00_dp, -4.5428176175934221E-01_dp,&
      -5.6266159555088222E-02_dp,  8.5555486145475090E-02_dp,&
      1.7149370136037647E-02_dp, -3.5874852805917547E-02_dp,&
      -6.6722317760714526E-03_dp,  1.3972200727155170E-02_dp,&
      1.7715212621408963E-02_dp, -3.4443652948503337E-02_dp,&
      2.2673023691549964E-02_dp, -6.9481802096620863E-03_dp,&
      7.9021807437504202E-04_dp,  1.6623569960343112E+00_dp,&
      -8.7799922392501561E-01_dp, -2.8938611876543164E-02_dp,&
      1.8305597014563521E-01_dp,  2.6823924227795206E-03_dp,&
      -7.8638677468377882E-02_dp,  2.5052055477927018E-03_dp,&
      3.2539080971677445E-02_dp,  2.5797503204750667E-02_dp,&
      -7.3577309739427957E-02_dp,  5.7896954096048170E-02_dp,&
      -2.1479855679151314E-02_dp,  3.2218345849532076E-03_dp,&
      2.5381059435843234E+00_dp, -1.6305859324151268E+00_dp,&
      9.9940175243888832E-02_dp,  3.5719858476252214E-01_dp,&
      -5.7992644536333599E-02_dp, -1.5378842508458723E-01_dp,&
      3.6992487976869874E-02_dp,  6.6983136752400169E-02_dp,&
      2.4029970044661862E-02_dp, -1.3686421359551432E-01_dp,&
      1.2628872883212300E-01_dp, -5.3004210025772776E-02_dp,&
      9.0124008618107931E-03_dp,  3.8112298597695000E+00_dp,&
      -2.9567738386099500E+00_dp,  4.6159700647669799E-01_dp,&
      6.4257759684283600E-01_dp, -2.2510024931676006E-01_dp,&
      -2.6991449769951259E-01_dp,  1.2839677612061992E-01_dp,&
      1.2324005868213329E-01_dp, -1.3633470926998539E-02_dp,&
      -2.1853429296999061E-01_dp,  2.4123664332249928E-01_dp,&
      -1.1237127932737824E-01_dp,  2.0806220268101396E-02_dp,&
      5.7008390024972444E+00_dp, -5.2719032477064589E+00_dp,&
      1.3286136063792067E+00_dp,  1.0639787628820851E+00_dp,&
      -6.1724692798358538E-01_dp, -4.1836007817403786E-01_dp,&
      3.3558355186522965E-01_dp,  2.0049791059094066E-01_dp,&
      -1.4522348760099926E-01_dp, -2.7848113288997439E-01_dp,&
      4.0294851159988648E-01_dp, -2.0991973963891600E-01_dp,&
      4.1918373279036081E-02_dp,  8.5444952115836337E+00_dp,&
      -9.2870882366649212E+00_dp,  3.2542494456110536E+00_dp,&
      1.5891401650173187E+00_dp, -1.4538307368581864E+00_dp,&
      -5.3611578876987864E-01_dp,  7.5799867206976324E-01_dp,&
      2.7324277500873972E-01_dp, -4.8297349611741452E-01_dp,&
      -1.9621208011655733E-01_dp,  5.6521268371561728E-01_dp,&
      -3.4379015311921068E-01_dp,  7.4666191775106056E-02_dp,&
      1.2869608889008939E+01_dp, -1.6227381735998069E+01_dp,&
      7.3457273329354766E+00_dp,  2.0007984970256341E+00_dp,&
      -3.1135650279596425E+00_dp, -4.2035573694616968E-01_dp,&
      1.5423312586717872E+00_dp,  2.4895859945054985E-01_dp,&
      -1.2143416136828173E+00_dp,  3.0888829641492249E-01_dp,&
      5.6592145994268328E-01_dp, -4.7133862783050268E-01_dp,&
      1.1553402263192626E-01_dp,  1.9510047195341826E+01_dp,&
      -2.8215377333929258E+01_dp,  1.5800988109124468E+01_dp,&
      1.5753381956046189E+00_dp, -6.1888970404653927E+00_dp,&
      4.5340289216053564E-01_dp,  2.8529274310442476E+00_dp,&
      -1.1985631183237680E-01_dp, -2.5919936322336810E+00_dp,&
      1.7904459124895560E+00_dp,  3.3185572647898222E-03_dp,&
      -4.5120198605812339E-01_dp,  1.4430781237161322E-01_dp,&
      2.9799686613683338E+01_dp, -4.8951476312595261E+01_dp,&
      3.2965439950410698E+01_dp, -1.6925499201780276E+00_dp,&
      -1.1444787017566560E+01_dp,  3.2893773175079954E+00_dp,&
      4.7232798353674390E+00_dp, -1.3946455243162874E+00_dp,&
      -4.8138784821254603E+00_dp,  5.1806730186148213E+00_dp,&
      -1.9484855421973464E+00_dp,  5.7565415788330868E-02_dp,&
      1.0328584787941356E-01_dp,  4.5900987586785305E+01_dp,&
      -8.4931873768978448E+01_dp,  6.7417416166242447E+01_dp,&
      -1.2917213048507392E+01_dp, -1.9369123838067868E+01_dp,&
      1.0603449957874252E+01_dp,  6.5793074414015802E+00_dp,&
      -4.6781732888897354E+00_dp, -7.6130671417336284E+00_dp,&
      1.1733674325796130E+01_dp, -6.7380063796773628E+00_dp,&
      1.7431827284203307E+00_dp, -1.3742674245542177E-01_dp,&
      7.1384655908206980E+01_dp, -1.4766110646876876E+02_dp,&
      1.3608901365331167E+02_dp, -4.4554212981906190E+01_dp,&
      -2.8355600002395651E+01_dp,  2.7091425069539174E+01_dp,&
      5.9567085220792704E+00_dp, -1.1750053684710588E+01_dp,&
      -9.1674840428949729E+00_dp,  2.2293423124322853E+01_dp,&
      -1.6371253102977199E+01_dp,  5.7669097930761808E+00_dp,&
      -8.2353886063174842E-01_dp,  1.1233183308389430E+02_dp,&
      -2.5780369487207054E+02_dp,  2.7242671117115714E+02_dp,&
      -1.2599753109481574E+02_dp, -2.8803649715248678E+01_dp,&
      5.9808941846251201E+01_dp, -4.4333363973394899E+00_dp,&
      -2.4294323397425206E+01_dp, -3.7425366988330224E+00_dp,&
      3.4394335271359864E+01_dp, -3.1907344026729994E+01_dp,&
      1.3411678831415276E+01_dp, -2.2957107908605567E+00_dp,&
      1.7976027905757553E+02_dp, -4.5369809560786518E+02_dp,&
      5.4292666408037110E+02_dp, -3.2469460257856400E+02_dp,&
      1.3642493310471760E+01_dp,  1.1322144991645266E+02_dp,&
      -4.1415851597078642E+01_dp, -3.9344925288346268E+01_dp,&
      2.0505996019668014E+01_dp,  3.5640047716368947E+01_dp,&
      -4.8066784685431486E+01_dp,  2.3931298245668948E+01_dp,&
      -4.6302204728291798E+00_dp,  2.9648400099998616E+02_dp,&
      -8.1318193308728723E+02_dp,  1.0852146967185377E+03_dp,&
      -7.9253616514248370E+02_dp,  1.9916533921620839E+02_dp,&
      1.6646952551534545E+02_dp, -1.3078063936770988E+02_dp,&
      -3.7561085470724848E+01_dp,  7.2748490941620716E+01_dp,&
      3.1770585593328282E+00_dp, -4.8379526363826820E+01_dp,&
      3.1452688247501925E+01_dp, -6.9488664317550564E+00_dp,&
      5.2490296655177804E+02_dp, -1.5381034281001564E+03_dp,&
      2.2397394630871509E+03_dp, -1.8980625603236201E+03_dp,&
      7.8793326721990013E+02_dp,  1.2072768890951619E+02_dp,&
      -2.8333018924346351E+02_dp,  1.5827025468773952E+01_dp,&
      1.4107764823365741E+02_dp, -7.1508335180567997E+01_dp,&
      -2.2202983634603370E+01_dp,  3.1298832279951391E+01_dp,&
      -8.4329929352037851E+00_dp,  1.1671482535247758E+03_dp,&
      -3.5701355790217854E+03_dp,  5.5029528129983019E+03_dp,&
      -5.1174808457482495E+03_dp,  2.6785762412568697E+03_dp,&
      -2.8427948185688939E+02_dp, -5.3279071516229101E+02_dp,&
      1.4157049520139412E+02_dp,  2.6643897464789472E+02_dp,&
      -2.1826868704262932E+02_dp,  2.6638738596435580E+01_dp,&
      3.3936069509129055E+01_dp, -1.2188642274117770E+01_dp,&
      1.4652812785816598E+00_dp, -5.1216317098725084E-01_dp,&
      -1.2107111132320940E-01_dp,  7.8124136614970738E-02_dp,&
      3.9373299417261073E-02_dp, -3.0559387326754693E-02_dp,&
      -1.7326558360561344E-02_dp,  1.0589249771069641E-02_dp,&
      2.3291147261995698E-02_dp, -2.9747108410639177E-02_dp,&
      1.3279817871973330E-02_dp, -1.5925954187249202E-03_dp,&
      -3.3358537008160058E-04_dp,  1.6747706949071501E+00_dp,&
      -7.3619993787034099E-01_dp, -9.2524849392815814E-02_dp,&
      1.4009317461191056E-01_dp,  2.8993455579017200E-02_dp,&
      -5.8947753721227604E-02_dp, -1.1544746505501786E-02_dp,&
      2.2888806638087296E-02_dp,  2.9752756422560003E-02_dp,&
      -5.6904268755167908E-02_dp,  3.7075509291121668E-02_dp,&
      -1.1205940796047410E-02_dp,  1.2414097457956487E-03_dp,&
      2.1320569710835793E+00_dp, -1.2554495472557166E+00_dp,&
      2.0663593584744656E-03_dp,  2.7880698885400201E-01_dp,&
      -1.0341136500847944E-02_dp, -1.2182591843005953E-01_dp,&
      1.0375885305318683E-02_dp,  5.1177746698586865E-02_dp,&
      3.5915727679690139E-02_dp, -1.1457196488446013E-01_dp,&
      9.3552441080291246E-02_dp, -3.5818521010418942E-02_dp,&
      5.5618136531600925E-03_dp,  2.9198954720291050E+00_dp,&
      -2.2293473658819969E+00_dp,  2.5486825622381931E-01_dp,&
      5.2106454908534805E-01_dp, -1.2338568730458786E-01_dp,&
      -2.2813706694695601E-01_dp,  7.2867460607848183E-02_dp,&
      1.0156141288573106E-01_dp,  2.2927036394505256E-02_dp,&
      -2.0292387455972360E-01_dp,  1.9739334607294629E-01_dp,&
      -8.5711263409840824E-02_dp,  1.5008093390084035E-02_dp,&
      4.1809650084217482E+00_dp, -3.9498692026171232E+00_dp,&
      8.5920971095646503E-01_dp,  8.9595233084619708E-01_dp,&
      -4.0043826925842790E-01_dp, -3.7963806599838434E-01_dp,&
      2.2335390257611457E-01_dp,  1.7832326489249364E-01_dp,&
      -5.3406140763388878E-02_dp, -3.0094516220647582E-01_dp,&
      3.5944195478704649E-01_dp, -1.7390038059962756E-01_dp,&
      3.3089261580483392E-02_dp,  6.1472397966948646E+00_dp,&
      -6.9285464394150278E+00_dp,  2.2070567132326251E+00_dp,&
      1.4063446953296506E+00_dp, -1.0126899665970956E+00_dp,&
      -5.4584355985905086E-01_dp,  5.4616991585738983E-01_dp,&
      2.7226973838426027E-01_dp, -2.8505286023998866E-01_dp,&
      -3.2863288737826130E-01_dp,  5.6233816955748084E-01_dp,&
      -3.0862618834553968E-01_dp,  6.3566187976697816E-02_dp,&
      9.1900790489272914E+00_dp, -1.2052670995254264E+01_dp,&
      5.0808380779589832E+00_dp,  1.9509309699776836E+00_dp,&
      -2.2667200167260235E+00_dp, -6.0497438189785557E-01_dp,&
      1.1769408070141436E+00_dp,  3.3139818277041716E-01_dp,&
      -8.4084448230590603E-01_dp, -7.5893210001079320E-02_dp,&
      7.0499033302583869E-01_dp, -4.7230545858400730E-01_dp,&
      1.0717094028717249E-01_dp,  1.3903288184276960E+01_dp,&
      -2.0861113079586804E+01_dp,  1.1028873449856791E+01_dp,&
      2.1247273818735861E+00_dp, -4.6741212415573994E+00_dp,&
      -2.1321596221060898E-01_dp,  2.3014636907442734E+00_dp,&
      1.9447225381512184E-01_dp, -1.9861181112351922E+00_dp,&
      9.1330548244123566E-01_dp,  4.9319113247404922E-01_dp,&
      -5.7394999803708169E-01_dp,  1.5315431429978804E-01_dp,&
      2.1241141277279649E+01_dp, -3.6033094867817468E+01_dp,&
      2.3102278552533658E+01_dp,  7.3272373739498387E-01_dp,&
      -8.9990510035402540E+00_dp,  1.4661536441210716E+00_dp,&
      4.0908947410985483E+00_dp, -5.4327633403126974E-01_dp,&
      -4.0397081958320689E+00_dp,  3.4883517046190322E+00_dp,&
      -7.4279189342756524E-01_dp, -3.5883086711361650E-01_dp,&
      1.6165467197230585E-01_dp,  3.2750700659675516E+01_dp,&
      -6.2260617506502861E+01_dp,  4.7306599319039115E+01_dp,&
      -5.3630667640010694E+00_dp, -1.6135965596523331E+01_dp,&
      6.2850804710731145E+00_dp,  6.4437698302620037E+00_dp,&
      -2.7654299843452557E+00_dp, -7.1374063007838009E+00_dp,&
      8.9855168947369108E+00_dp, -4.2936389770830674E+00_dp,&
      7.3708343499722073E-01_dp,  3.2368466859189433E-02_dp,&
      5.0971587949712941E+01_dp, -1.0781940838147709E+02_dp,&
      9.5470719828705825E+01_dp, -2.3933268645557146E+01_dp,&
      -2.6335385696434315E+01_dp,  1.8016780861843714E+01_dp,&
      8.2132156983356950E+00_dp, -8.1534450671776550E+00_dp,&
      -1.0525394951164396E+01_dp,  1.9008340533796282E+01_dp,&
      -1.2282071103729981E+01_dp,  3.7805441921538856E+00_dp,&
      -4.4383200779161114E-01_dp,  8.0151692226233365E+01_dp,&
      -1.8743506352274528E+02_dp,  1.9092478332081694E+02_dp,&
      -7.3582255682872955E+01_dp, -3.6322792299277531E+01_dp,&
      4.3581813355095242E+01_dp,  5.1785945753389013E+00_dp,&
      -1.9329067501593595E+01_dp, -1.0809469621978822E+01_dp,&
      3.4142519766130000E+01_dp, -2.7441723274242250E+01_dp,&
      1.0488104966880156E+01_dp, -1.6444555466331974E+00_dp,&
      1.2763045722861104E+02_dp, -3.2766564765272250E+02_dp,&
      3.7973217630740623E+02_dp, -1.9781100978811261E+02_dp,&
      -3.0157621574998899E+01_dp,  9.3237621383503154E+01_dp,&
      -1.4664555123007290E+01_dp, -3.8610014643549675E+01_dp,&
      1.9561581150219733E+00_dp,  4.9178714877907694E+01_dp /)
    REAL(KIND=dp), DIMENSION(400), PARAMETER :: c02 = (/ &
      -5.0683586321199940E+01_dp,  2.2685614985361322E+01_dp,&
      -4.0911306259362732E+00_dp,  2.0692715444769155E+02_dp,&
      -5.7784708851700361E+02_dp,  7.5313473849594516E+02_dp,&
      -4.9591356082585395E+02_dp,  4.6934746915158954E+01_dp,&
      1.7278593891248715E+02_dp, -7.9461883488471031E+01_dp,&
      -6.0640581635323905E+01_dp,  4.9709605032605261E+01_dp,&
      3.9785417005473363E+01_dp, -7.0707070950857386E+01_dp,&
      3.8301537914504102E+01_dp, -7.8036753533627845E+00_dp,&
      3.4643294122869480E+02_dp, -1.0373289229669426E+03_dp,&
      1.4964894908260794E+03_dp, -1.1883576064664164E+03_dp,&
      3.5395000506872071E+02_dp,  2.4438341252020561E+02_dp,&
      -2.3525196154593584E+02_dp, -4.7569852552505999E+01_dp,&
      1.5264487252042383E+02_dp, -4.7249655989039340E+01_dp,&
      -4.6426435042560392E+01_dp,  4.1614986793760508E+01_dp,&
      -1.0251258037382264E+01_dp,  6.2235104125450675E+02_dp,&
      -1.9530000742095463E+03_dp,  3.0381848576161515E+03_dp,&
      -2.7721007214385299E+03_dp,  1.2886472933015646E+03_dp,&
      1.1997495108070251E+02_dp, -4.8923220063164257E+02_dp,&
      8.9320317305675530E+01_dp,  2.5417223540252016E+02_dp,&
      -2.1855818229530024E+02_dp,  5.3225449020928423E+01_dp,&
      1.4035211359660455E+01_dp, -7.4385089727473837E+00_dp,&
      1.3392562526989720E+03_dp, -4.2661156874133503E+03_dp,&
      6.8704505045145133E+03_dp, -6.7486562942964165E+03_dp,&
      3.8557980607995291E+03_dp, -6.4041197681422841E+02_dp,&
      -7.2028812629191964E+02_dp,  3.4296722416927707E+02_dp,&
      2.9920668043243813E+02_dp, -3.7866211734835207E+02_dp,&
      1.4679850489782487E+02_dp, -8.2562501236068311E+00_dp,&
      -6.1258993537203876E+00_dp,  5.0065694781245784E+03_dp,&
      -1.5735707066352348E+04_dp,  2.5115152335838251E+04_dp,&
      -2.4641268459826806E+04_dp,  1.4432198334563878E+04_dp,&
      -3.1899549141849984E+03_dp, -1.6897674776327710E+03_dp,&
      7.8118394579369010E+02_dp,  9.8213799629630739E+02_dp,&
      -1.0349003003258824E+03_dp,  2.8781829158913928E+02_dp,&
      5.6520242044190624E+01_dp, -3.5018492673502557E+01_dp,&
      2.1672646531042281E-01_dp, -4.0280327089000083E-02_dp,&
      -1.4141841987473180E-02_dp,  4.7576924044397707E-03_dp,&
      4.2680527641714202E-03_dp, -1.7121426116300659E-03_dp,&
      -1.8727973098150161E-03_dp,  5.2727512023623743E-04_dp,&
      1.7921888843793570E-03_dp, -1.5480544238380691E-03_dp,&
      2.0854567230273012E-04_dp,  2.5776529992692170E-04_dp,&
      -9.6383744843509111E-05_dp,  6.9533459997442570E-01_dp,&
      -1.4709564320905877E-01_dp, -4.6296062125088105E-02_dp,&
      2.0685418524643968E-02_dp,  1.4532396018071178E-02_dp,&
      -7.8206235021459156E-03_dp, -6.4330680110342292E-03_dp,&
      2.5911543805967269E-03_dp,  6.8719471260085367E-03_dp,&
      -7.0817523244719764E-03_dp,  1.9939578385323588E-03_dp,&
      4.5553991899127580E-04_dp, -2.7486672885741699E-04_dp,&
      1.3220348620246702E+00_dp, -3.4559558249226729E-01_dp,&
      -8.9054370747977230E-02_dp,  5.9965845713972753E-02_dp,&
      2.9540216889318294E-02_dp, -2.3945083153648589E-02_dp,&
      -1.3172332990522255E-02_dp,  8.5295161742936637E-03_dp,&
      1.6992003498685241E-02_dp, -2.1830534111603173E-02_dp,&
      9.4603169609586910E-03_dp, -9.0624437091804298E-04_dp,&
      -3.1639345757305219E-04_dp,  2.2408348572564285E+00_dp,&
      -7.4493621182222391E-01_dp, -1.4437935708506164E-01_dp,&
      1.5345216731463060E-01_dp,  4.9981931944418634E-02_dp,&
      -6.4068353634840702E-02_dp, -2.2064095582909744E-02_dp,&
      2.4136625260774338E-02_dp,  3.7163716242542298E-02_dp,&
      -5.8961636814152951E-02_dp,  3.2540613635375810E-02_dp,&
      -7.3389754359831682E-03_dp,  2.7235853348761487E-04_dp,&
      3.6772058442244018E+00_dp, -1.5479451960257449E+00_dp,&
      -1.9841574174749130E-01_dp,  3.6154130846694421E-01_dp,&
      6.8141128228278253E-02_dp, -1.5626923127364656E-01_dp,&
      -2.8071652682116390E-02_dp,  6.1659614693382665E-02_dp,&
      7.3720675498160068E-02_dp, -1.4480983948301773E-01_dp,&
      9.4136312025964453E-02_dp, -2.8018590746203921E-02_dp,&
      2.9795911304120759E-03_dp,  6.0051316400388179E+00_dp,&
      -3.1459525542112772E+00_dp, -1.9087960860761674E-01_dp,&
      8.0233225074828274E-01_dp,  5.0436194677495369E-02_dp,&
      -3.5562444186561837E-01_dp, -1.0808342335055487E-02_dp,&
      1.4631431396534988E-01_dp,  1.3021643058978571E-01_dp,&
      -3.2912594390655925E-01_dp,  2.4429333403753731E-01_dp,&
      -8.5093523653718778E-02_dp,  1.1740285611601968E-02_dp,&
      9.8661407780096138E+00_dp, -6.3079683609443906E+00_dp,&
      6.4668417524656907E-02_dp,  1.7019207709152095E+00_dp,&
      -1.0864690051990379E-01_dp, -7.6490539642302580E-01_dp,&
      9.2463045587798220E-02_dp,  3.2754329325028214E-01_dp,&
      1.9036075703407310E-01_dp, -6.9593654311936159E-01_dp,&
      5.8500589456574437E-01_dp, -2.2833809379472947E-01_dp,&
      3.6085090534356240E-02_dp,  1.6387345232640328E+01_dp,&
      -1.2566581816514574E+01_dp,  1.0833488063785175E+00_dp,&
      3.4780856044371822E+00_dp, -7.0298676205420429E-01_dp,&
      -1.5603437365117936E+00_dp,  4.5358830961481994E-01_dp,&
      6.9487899892425453E-01_dp,  1.6306574054540915E-01_dp,&
      -1.3565280066889112E+00_dp,  1.3044257170639009E+00_dp,&
      -5.6107520030086444E-01_dp,  9.7339437137894266E-02_dp,&
      2.7590362528083165E+01_dp, -2.5020953292207636E+01_dp,&
      4.2199543975733604E+00_dp,  6.8574653755300341E+00_dp,&
      -2.5008013151358859E+00_dp, -2.9959805517111318E+00_dp,&
      1.5109398081007834E+00_dp,  1.3866679059684508E+00_dp,&
      -2.7320241923625971E-01_dp, -2.3552778799554877E+00_dp,&
      2.6919927683133067E+00_dp, -1.2750517732227757E+00_dp,&
      2.3894973963551455E-01_dp,  4.7176080794132197E+01_dp,&
      -5.0043699620567423E+01_dp,  1.2963098635058795E+01_dp,&
      1.2956187817919837E+01_dp, -7.4293132600153990E+00_dp,&
      -5.2715667610509955E+00_dp,  4.3201347820454323E+00_dp,&
      2.5318631031919860E+00_dp, -2.0681119464958941E+00_dp,&
      -3.2740269844269108E+00_dp,  5.0056055519591984E+00_dp,&
      -2.6579602424397017E+00_dp,  5.3739683583410036E-01_dp,&
      8.2067870773483293E+01_dp, -1.0099618717254077E+02_dp,&
      3.6251613177931354E+01_dp,  2.2955720667473543E+01_dp,&
      -2.0162813094031463E+01_dp, -7.8515331287445758E+00_dp,&
      1.1267353610368847E+01_dp,  3.8948541826633067E+00_dp,&
      -7.6972479076468243E+00_dp, -1.9606037339217177E+00_dp,&
      7.7570215809451124E+00_dp, -4.9132929567084096E+00_dp,&
      1.0893358659958350E+00_dp,  1.4552872970382410E+02_dp,&
      -2.0653403368911415E+02_dp,  9.7133525385800795E+01_dp,&
      3.5870243090002134E+01_dp, -5.1612920970104334E+01_dp,&
      -6.8814440565183297E+00_dp,  2.7253921030056905E+01_dp,&
      3.4513432377643651E+00_dp, -2.2786048871961697E+01_dp,&
      8.9085416689751842E+00_dp,  7.2036115050483476E+00_dp,&
      -7.2654992116157242E+00_dp,  1.8811183601144852E+00_dp,&
      2.6365252354702568E+02_dp, -4.2974049946773454E+02_dp,&
      2.5594897338735430E+02_dp,  3.8712804569966629E+01_dp,&
      -1.2572180308400272E+02_dp,  1.3481924026494765E+01_dp,&
      6.0461082538064041E+01_dp, -7.3930191264914962E+00_dp,&
      -5.7463183771426053E+01_dp,  4.9457275105138656E+01_dp,&
      -1.0700147395230934E+01_dp, -4.9418175760713634E+00_dp,&
      2.2591130172972433E+00_dp,  4.8938924956022072E+02_dp,&
      -9.1372186196753569E+02_dp,  6.7434387995902102E+02_dp,&
      -3.2155447353298307E+01_dp, -2.8807904077131656E+02_dp,&
      1.0787587156933184E+02_dp,  1.1604641801031272E+02_dp,&
      -5.7495945231578801E+01_dp, -1.1982979280830179E+02_dp,&
      1.6386009398044138E+02_dp, -8.5958117255417733E+01_dp,&
      1.8875664993395461E+01_dp, -6.7677172652916551E-01_dp,&
      9.3447004994210636E+02_dp, -1.9946219423819334E+03_dp,&
      1.7982346755590097E+03_dp, -4.3838272051099790E+02_dp,&
      -5.8789690476235876E+02_dp,  4.4360653113106764E+02_dp,&
      1.4921155606077679E+02_dp, -2.2511093876438125E+02_dp,&
      -1.6695730504683976E+02_dp,  4.0917420561017377E+02_dp,&
      -3.0271931512428114E+02_dp,  1.0796446015104219E+02_dp,&
      -1.5703656639435989E+01_dp,  1.8498047274145415E+03_dp,&
      -4.4984616584856849E+03_dp,  4.8992366073257654E+03_dp,&
      -2.1977425810406944E+03_dp, -8.3345216169575804E+02_dp,&
      1.4355964803086322E+03_dp, -1.5812943023377832E+02_dp,&
      -6.2781654249010364E+02_dp,  9.2380286057259525E+01_dp,&
      6.5527187346706739E+02_dp, -6.9983355536916895E+02_dp,&
      3.1751326162224495E+02_dp, -5.7861253194568050E+01_dp,&
      3.8898521960899338E+03_dp, -1.0660258370556912E+04_dp,&
      1.3786469354695366E+04_dp, -9.1465237275417621E+03_dp,&
      9.3013441723421602E+02_dp,  3.4076602842055540E+03_dp,&
      -1.9886757703666833E+03_dp, -8.2892218311343163E+02_dp,&
      1.2569071836213366E+03_dp,  3.2509800881158377E+01_dp,&
      -7.6144423093941646E+02_dp,  4.9335766114991003E+02_dp,&
      -1.0865110989329460E+02_dp,  9.8449165454890644E+03_dp,&
      -2.9503930183220888E+04_dp,  4.3421888480353453E+04_dp,&
      -3.6978110090308204E+04_dp,  1.5168356908680847E+04_dp,&
      2.9788779106407983E+03_dp, -6.3846589497297919E+03_dp,&
      9.3105734344115615E+02_dp,  2.6308310447041681E+03_dp,&
      -1.6146136611487138E+03_dp, -1.9731348014190954E+02_dp,&
      5.0643537932149997E+02_dp, -1.4599209745780860E+02_dp,&
      1.7630990396200976E+00_dp, -3.3892327099449571E-01_dp,&
      -1.1576929041053773E-01_dp,  4.2074501755300822E-02_dp,&
      3.5318613391807509E-02_dp, -1.5361815626986417E-02_dp,&
      -1.5538770819816210E-02_dp,  4.8377403154006675E-03_dp,&
      1.5296287290468806E-02_dp, -1.3888302637667885E-02_dp,&
      2.4825409810328397E-03_dp,  1.8872230081976539E-03_dp,&
      -7.6714693939084367E-04_dp,  2.1325733506956639E+00_dp,&
      -5.5729186875233505E-01_dp, -1.4615706136468917E-01_dp,&
      9.6026856885913170E-02_dp,  4.8904598893406155E-02_dp,&
      -3.8100135476357294E-02_dp, -2.1911777538135460E-02_dp,&
      1.3454311097079341E-02_dp,  2.7637903087209887E-02_dp,&
      -3.4616620258338109E-02_dp,  1.4428685212043010E-02_dp,&
      -1.0336299096049299E-03_dp, -5.9505869733514696E-04_dp,&
      2.9744753512390276E+00_dp, -1.1003780471193960E+00_dp,&
      -1.9785159455719278E-01_dp,  2.3780637253194109E-01_dp,&
      7.1963817509901273E-02_dp, -9.9748145826735912E-02_dp,&
      -3.2190554128659045E-02_dp,  3.7695550143958513E-02_dp,&
      5.6651162671219071E-02_dp, -9.1715891141988914E-02_dp,&
      5.1468581747088488E-02_dp, -1.1995276658616574E-02_dp,&
      5.5439593134832155E-04_dp,  4.5219935606964325E+00_dp,&
      -2.2236721397856853E+00_dp, -2.3730761168654535E-01_dp,&
      5.4856278124972280E-01_dp,  8.7445847277008756E-02_dp,&
      -2.3912431815003291E-01_dp, -3.6191282630300727E-02_dp,&
      9.4910314828061157E-02_dp,  1.0914636445746341E-01_dp,&
      -2.2219329966642776E-01_dp,  1.4753788499623779E-01_dp,&
      -4.5129424300298711E-02_dp,  5.0589183084530339E-03_dp,&
      7.2107757897887188E+00_dp, -4.4495581221107656E+00_dp,&
      -1.5949892284174227E-01_dp,  1.1916501833218205E+00_dp,&
      3.8159808588412052E-02_dp, -5.3381719311232834E-01_dp,&
      1.1987078987062662E-04_dp,  2.2134348100162349E-01_dp,&
      1.8661057319208169E-01_dp, -4.9690591490122710E-01_dp,&
      3.7695345860164003E-01_dp, -1.3414080875310089E-01_dp,&
      1.9025830154460060E-02_dp,  1.1826586706724369E+01_dp,&
      -8.8233400141049501E+00_dp,  3.3211087191020516E-01_dp,&
      2.4798971011020616E+00_dp, -2.4321974910475175E-01_dp,&
      -1.1276491372635893E+00_dp,  1.7580577072785533E-01_dp,&
      4.8773057223976912E-01_dp,  2.5762633845103161E-01_dp,&
      -1.0335790498546602E+00_dp,  8.8984162729180138E-01_dp,&
      -3.5392500454239423E-01_dp,  5.7015366649628613E-02_dp,&
      1.9787246843226512E+01_dp, -1.7445132201772360E+01_dp,&
      2.0255129628444672E+00_dp,  4.9871197966879164E+00_dp,&
      -1.2082606849268420E+00_dp, -2.2629643535078277E+00_dp,&
      7.5536562784156702E-01_dp,  1.0207918987429019E+00_dp,&
      1.7192575597177209E-01_dp, -1.9796532433429683E+00_dp,&
      1.9593380309152029E+00_dp, -8.5836023264597161E-01_dp,&
      1.5125817632192159E-01_dp,  3.3681131542986975E+01_dp,&
      -3.4585438137019239E+01_dp,  6.9509291872553263E+00_dp,&
      9.6987345107099117E+00_dp, -4.0173741038954836E+00_dp,&
      -4.2756064482503922E+00_dp,  2.4041920018559479E+00_dp /)
    REAL(KIND=dp), DIMENSION(400), PARAMETER :: c03 = (/ &
      2.0123022827718544E+00_dp, -5.8064769221536761E-01_dp,&
      -3.3561931133425400E+00_dp,  3.9944119179735149E+00_dp,&
      -1.9296809249289084E+00_dp,  3.6684710023030970E-01_dp,&
      5.8314392583364203E+01_dp, -6.9071915064420764E+01_dp,&
      2.0273316780786242E+01_dp,  1.8091276263426014E+01_dp,&
      -1.1553296198966734E+01_dp, -7.3776447143848456E+00_dp,&
      6.7127519171282239E+00_dp,  3.6256900346252308E+00_dp,&
      -3.4861066477531804E+00_dp, -4.4280915350121814E+00_dp,&
      7.3155197483498444E+00_dp, -3.9817222244987160E+00_dp,&
      8.1716033590013382E-01_dp,  1.0279442872714777E+02_dp,&
      -1.3951572323665187E+02_dp,  5.5126922655550800E+01_dp,&
      3.1593961331540505E+01_dp, -3.0773981811737219E+01_dp,&
      -1.0609400500428917E+01_dp,  1.7269150027597533E+01_dp,&
      5.4550950252540860E+00_dp, -1.2412007314711813E+01_dp,&
      -1.6944363104250255E+00_dp,  1.1010026315963469E+01_dp,&
      -7.2647707908744747E+00_dp,  1.6412253503340308E+00_dp,&
      1.8478392611717035E+02_dp, -2.8606063621708472E+02_dp,&
      1.4520791993720482E+02_dp,  4.8248284045813477E+01_dp,&
      -7.7922693756595990E+01_dp, -7.9408226077719561E+00_dp,&
      4.1485197987820960E+01_dp,  4.3678621514692084E+00_dp,&
      -3.6233004314355206E+01_dp,  1.6591239288982003E+01_dp,&
      8.9301404752582219E+00_dp, -1.0434258526682950E+01_dp,&
      2.7944372114336824E+00_dp,  3.3944758158812471E+02_dp,&
      -5.9758838258126650E+02_dp,  3.7841395922782914E+02_dp,&
      4.8417305572292761E+01_dp, -1.8893041807173518E+02_dp,&
      2.4741002322332704E+01_dp,  9.2060942372558330E+01_dp,&
      -1.2828891284744461E+01_dp, -9.1422655242511411E+01_dp,&
      8.2905747183313039E+01_dp, -2.1697703734039433E+01_dp,&
      -5.6962675534492826E+00_dp,  3.1991345493662133E+00_dp,&
      6.3893928008600767E+02_dp, -1.2768305025665254E+03_dp,&
      9.8951899589348307E+02_dp, -6.0443695658991189E+01_dp,&
      -4.3440590534894858E+02_dp,  1.7014936207019227E+02_dp,&
      1.7915396372204998E+02_dp, -9.1142430248866432E+01_dp,&
      -1.9369374933733687E+02_dp,  2.7062993560444278E+02_dp,&
      -1.4648606822230460E+02_dp,  3.4401173794203238E+01_dp,&
      -1.8643660577461350E+00_dp,  1.2368631912166727E+03_dp,&
      -2.8021015677999771E+03_dp,  2.6245851326980301E+03_dp,&
      -6.5184739072448338E+02_dp, -9.0662683327423667E+02_dp,&
      6.8595985546862971E+02_dp,  2.4708292671840775E+02_dp,&
      -3.5944943741598553E+02_dp, -2.8802980788436065E+02_dp,&
      6.9079812496750924E+02_dp, -5.1324954557812202E+02_dp,&
      1.8452547451235932E+02_dp, -2.7127058028799510E+01_dp,&
      2.4780267312523883E+03_dp, -6.3481274679156131E+03_dp,&
      7.1223078050965232E+03_dp, -3.1670682000868255E+03_dp,&
      -1.4343302633933163E+03_dp,  2.2735595623482527E+03_dp,&
      -1.4893036405924593E+02_dp, -1.0812056752629833E+03_dp,&
      7.5011532640876709E+01_dp,  1.2375394471103587E+03_dp,&
      -1.2835762825052959E+03_dp,  5.7825279537734082E+02_dp,&
      -1.0513964454975101E+02_dp,  5.2240008736035516E+03_dp,&
      -1.4989583733809592E+04_dp,  1.9904817992583929E+04_dp,&
      -1.3071959006342418E+04_dp,  4.1890770123089112E+02_dp,&
      6.0736481096366579E+03_dp, -3.0780323603297416E+03_dp,&
      -2.0133593320735015E+03_dp,  2.5522083062383117E+03_dp,&
      1.0175091050765498E+02_dp, -1.5715434454560022E+03_dp,&
      1.0200855278610385E+03_dp, -2.2627481009306229E+02_dp,&
      1.2373847610812090E+04_dp, -3.8548285445828238E+04_dp,&
      5.8497379094576674E+04_dp, -5.0546288117095348E+04_dp,&
      1.9279080238692830E+04_dp,  8.0456785161874832E+03_dp,&
      -1.2762585183952415E+04_dp,  2.1391906252619819E+03_dp,&
      6.0395885681160671E+03_dp, -5.3701230424802343E+03_dp,&
      1.5974805021564043E+03_dp,  1.1957935329310017E+02_dp,&
      -1.2701661891975539E+02_dp,  4.5956408995458441E+04_dp,&
      -1.4543165723892135E+05_dp,  2.3046137165679206E+05_dp,&
      -2.2184270858179423E+05_dp,  1.2378962647413493E+05_dp,&
      -1.9979906545203474E+04_dp, -2.1878488592470068E+04_dp,&
      1.0209913111179272E+04_dp,  7.6876868768320492E+03_dp,&
      -8.6795300997867835E+03_dp,  2.1149514187852687E+03_dp,&
      7.3353058897293010E+02_dp, -3.6026759783745257E+02_dp,&
      2.3973743190763880E-01_dp, -2.1972173943175737E-02_dp,&
      -1.2933433013553911E-02_dp,  2.0263255241757100E-03_dp,&
      3.8558136859502359E-03_dp, -2.3496377608067925E-04_dp,&
      -2.4948639401902385E-03_dp,  2.7317726766600036E-04_dp,&
      2.1977795539286837E-03_dp, -1.7085734424693412E-03_dp,&
      1.8807577698138860E-04_dp,  2.9444043588833577E-04_dp,&
      -1.0495679299232483E-04_dp,  7.8119548885908119E-01_dp,&
      -8.3814148577581293E-02_dp, -4.7508135810116980E-02_dp,&
      9.8019626915272322E-03_dp,  1.4868280045836485E-02_dp,&
      -1.8006665222561487E-03_dp, -9.7377554307787259E-03_dp,&
      1.4125595007739784E-03_dp,  9.0509059488131458E-03_dp,&
      -7.7385054530521970E-03_dp,  1.5388691452156726E-03_dp,&
      8.5747629581810734E-04_dp, -3.6661944358525332E-04_dp,&
      1.5306841908146380E+00_dp, -2.1063022274943391E-01_dp,&
      -1.1241196620289133E-01_dp,  3.2360998322751140E-02_dp,&
      3.7665015340093901E-02_dp, -8.0772948523086009E-03_dp,&
      -2.5159527309634066E-02_dp,  5.0554039417750034E-03_dp,&
      2.5273349130359068E-02_dp, -2.4353394331749184E-02_dp,&
      7.2955857496937882E-03_dp,  9.9758288171452984E-04_dp,&
      -7.6511170564210666E-04_dp,  2.7079852668931923E+00_dp,&
      -4.8920830337568294E-01_dp, -2.4324920879045647E-01_dp,&
      9.4157225722568336E-02_dp,  8.7017036286837601E-02_dp,&
      -2.8177328136367519E-02_dp, -5.9539436788281913E-02_dp,&
      1.5947059865743421E-02_dp,  6.4998406125438488E-02_dp,&
      -7.0010003351382771E-02_dp,  2.6870981588887691E-02_dp,&
      -1.2287304809945091E-03_dp, -1.2626866804900177E-03_dp,&
      4.6871272746842072E+00_dp, -1.0979373448364005E+00_dp,&
      -5.0439064698554803E-01_dp,  2.5329109255504134E-01_dp,&
      1.9065417291741427E-01_dp, -8.5926530886646244E-02_dp,&
      -1.3414673664499402E-01_dp,  4.6615086648998535E-02_dp,&
      1.5969458749606510E-01_dp, -1.9077663190094568E-01_dp,&
      8.6799804720832860E-02_dp, -1.2802583451222354E-02_dp,&
      -1.2752933836137432E-03_dp,  8.1448100654677873E+00_dp,&
      -2.4178207414003650E+00_dp, -1.0140790036261034E+00_dp,&
      6.4948091684454845E-01_dp,  4.0002662216023965E-01_dp,&
      -2.4319032508794547E-01_dp, -2.9094633344678594E-01_dp,&
      1.3045357990505776E-01_dp,  3.8023332842632163E-01_dp,&
      -5.0306877230005320E-01_dp,  2.6113769839458034E-01_dp,&
      -5.6298924203416423E-02_dp,  1.8123949457853315E-03_dp,&
      1.4352898973567878E+01_dp, -5.2818184817482203E+00_dp,&
      -1.9859061847519863E+00_dp,  1.6209923485665547E+00_dp,&
      8.0480146245262718E-01_dp, -6.5980134024898218E-01_dp,&
      -6.0983822335215754E-01_dp,  3.5728072539684819E-01_dp,&
      8.8464717874409349E-01_dp, -1.3018445886435062E+00_dp,&
      7.5507980073580228E-01_dp, -2.0136154790359589E-01_dp,&
      1.7515223022048523E-02_dp,  2.5776640136003774E+01_dp,&
      -1.1552325240946562E+01_dp, -3.7871289474355541E+00_dp,&
      3.9950291812053065E+00_dp,  1.5381543034175054E+00_dp,&
      -1.7488517416194553E+00_dp, -1.2307547949552229E+00_dp,&
      9.7227430479047705E-01_dp,  2.0152195969554660E+00_dp,&
      -3.3340448599009007E+00_dp,  2.1353902718112705E+00_dp,&
      -6.5792855059375088E-01_dp,  7.7757862605825775E-02_dp,&
      4.7340489049606639E+01_dp, -2.5502622340548470E+01_dp,&
      -6.9771561423823139E+00_dp,  9.8241770037760148E+00_dp,&
      2.7041503874198094E+00_dp, -4.5827496067732367E+00_dp,&
      -2.3499942979626960E+00_dp,  2.6589465299988495E+00_dp,&
      4.4638845876459596E+00_dp, -8.4790434572470250E+00_dp,&
      5.9637608044446919E+00_dp, -2.0511810835588626E+00_dp,&
      2.8556951006602344E-01_dp,  8.9206434643405373E+01_dp,&
      -5.7246392101459577E+01_dp, -1.2118388129984879E+01_dp,&
      2.4298560398108567E+01_dp,  3.9314821788160956E+00_dp,&
      -1.1961915849435366E+01_dp, -4.0288255301413383E+00_dp,&
      7.3737596068968578E+00_dp,  9.3863903576693257E+00_dp,&
      -2.1365482846607684E+01_dp,  1.6512440427129999E+01_dp,&
      -6.2254495487803139E+00_dp,  9.6619197544078494E-01_dp,&
      1.7312884123481155E+02_dp, -1.3162884253179351E+02_dp,&
      -1.8435423873064877E+01_dp,  6.0837883751267668E+01_dp,&
      2.3531394283228084E+00_dp, -3.1220104773374494E+01_dp,&
      -5.0963636597563147E+00_dp,  2.0884887034584892E+01_dp,&
      1.7434242907928009E+01_dp, -5.2737432022531095E+01_dp,&
      4.5207653351126943E+01_dp, -1.8537109806799002E+01_dp,&
      3.1268716041690032E+00_dp,  3.4769947397156164E+02_dp,&
      -3.1249298768889707E+02_dp, -1.7437461781643293E+01_dp,&
      1.5498716759905759E+02_dp, -1.5841772368812352E+01_dp,&
      -8.1377538168402268E+01_dp,  1.8724669219067327E+00_dp,&
      6.0675444135269096E+01_dp,  2.0816898730371967E+01_dp,&
      -1.2351369514321331E+02_dp,  1.2071107009758312E+02_dp,&
      -5.3951445844698021E+01_dp,  9.7892956954004831E+00_dp,&
      7.2703354843362490E+02_dp, -7.7313108411402254E+02_dp,&
      3.5023735034956978E+01_dp,  4.0297017473577085E+02_dp,&
      -1.1042256365889763E+02_dp, -2.0941813845985146E+02_dp,&
      5.2087715690791505E+01_dp,  1.8039722228227356E+02_dp,&
      -4.1352516140890167E+01_dp, -2.4923882189703463E+02_dp,&
      3.0196465579696888E+02_dp, -1.5000260743356392E+02_dp,&
      2.9343587506438965E+01_dp,  1.5958129013599184E+03_dp,&
      -2.0168957177610125E+03_dp,  3.4240220658253907E+02_dp,&
      1.0672373281250943E+03_dp, -5.3613234405799597E+02_dp,&
      -5.1084300344864096E+02_dp,  2.9925364586857182E+02_dp,&
      5.3763268220233249E+02_dp, -5.0331545545250009E+02_dp,&
      -2.5847420072564825E+02_dp,  6.2069474368006570E+02_dp,&
      -3.7002167042097642E+02_dp,  8.0101106126116946E+01_dp,&
      3.7194250317395058E+03_dp, -5.6366551640950183E+03_dp,&
      1.8508257063062345E+03_dp,  2.8281524590337158E+03_dp,&
      -2.4073828332691851E+03_dp, -9.9894429778883796E+02_dp,&
      1.3896422496078037E+03_dp,  1.4795882623602579E+03_dp,&
      -2.9286651253849855E+03_dp,  1.3827022620856706E+03_dp,&
      3.7841886473089392E+02_dp, -5.8826257543824522E+02_dp,&
      1.6416552679796825E+02_dp,  9.3643659386802919E+03_dp,&
      -1.7268153319694040E+04_dp,  9.3340833529031097E+03_dp,&
      6.8721045784191338E+03_dp, -1.0801964919124486E+04_dp,&
      3.3862132734645331E+02_dp,  5.7555001034296838E+03_dp,&
      2.4166845618580087E+03_dp, -1.2988122107330906E+04_dp,&
      1.3288937622953321E+04_dp, -6.4646546543504810E+03_dp,&
      1.4032801485599530E+03_dp, -6.0214798988559934E+01_dp,&
      2.6224691656659063E+04_dp, -6.0092717495861041E+04_dp,&
      5.0748946885645579E+04_dp,  6.4296047875928125E+03_dp,&
      -4.7887862033455087E+04_dp,  2.7735324023937988E+04_dp,&
      1.4938668072319664E+04_dp, -1.4316737379758091E+04_dp,&
      -2.8451024275428434E+04_dp,  5.7244647432622391E+04_dp,&
      -4.5386091585022587E+04_dp,  1.8311548691603668E+04_dp,&
      -3.1224051036545457E+03_dp,  8.9159916001930978E+04_dp,&
      -2.5586332776355915E+05_dp,  3.2843439673832123E+05_dp,&
      -1.7020742611488042E+05_dp, -1.0846427488360634E+05_dp,&
      2.5316054359400918E+05_dp, -1.7089677062997792E+05_dp,&
      1.6823614758843752E+04_dp,  3.7175896672429459E+04_dp,&
      2.2014007110624079E+03_dp, -3.3289475790468583E+04_dp,&
      2.2732719096251702E+04_dp, -5.1983853159329319E+03_dp,&
      1.9578084287802890E+00_dp, -1.8700206095269101E-01_dp,&
      -1.0900328443936919E-01_dp,  1.8470854213375257E-02_dp,&
      3.2926833694674286E-02_dp, -2.5244548830637811E-03_dp,&
      -2.1371119257102094E-02_dp,  2.5373572035113992E-03_dp,&
      1.9096936556220606E-02_dp, -1.5244081816584576E-02_dp,&
      2.0668203447746073E-03_dp,  2.3576566574104695E-03_dp,&
      -8.7491071381053489E-04_dp,  2.4679674747272893E+00_dp,&
      -3.3737601133348555E-01_dp, -1.8139040419117963E-01_dp,&
      5.0523868752123523E-02_dp,  6.0550371224777057E-02_dp,&
      -1.2172815962913154E-02_dp, -4.0308427219634048E-02_dp,&
      7.7287143244938123E-03_dp,  4.0019771312533585E-02_dp,&
      -3.7854821497454338E-02_dp,  1.0769335283667015E-02_dp,&
      1.9476127553671263E-03_dp, -1.2801087884499606E-03_dp,&
      3.6694426304025538E+00_dp, -7.3185799640057625E-01_dp,&
      -3.6064251381359935E-01_dp,  1.4504547183962732E-01_dp /)
    REAL(KIND=dp), DIMENSION(400), PARAMETER :: c04 = (/ &
      1.3124621600522313E-01_dp, -4.3574928379419176E-02_dp,&
      -8.9899230702256708E-02_dp,  2.4290637640954200E-02_dp,&
      9.8569915866746516E-02_dp, -1.0636170473106374E-01_dp,&
      4.0918764154109298E-02_dp, -1.9272580657675622E-03_dp,&
      -1.9052149763963443E-03_dp,  5.9890800326505680E+00_dp,&
      -1.6089545544303487E+00_dp, -7.2735892893642817E-01_dp,&
      3.8519972111009043E-01_dp,  2.8134695163873363E-01_dp,&
      -1.3173769525033738E-01_dp, -1.9866704055549164E-01_dp,&
      7.0436060618907739E-02_dp,  2.3886114608119013E-01_dp,&
      -2.8708303282688091E-01_dp,  1.3162016111650496E-01_dp,&
      -1.9942386857965708E-02_dp, -1.7671512955273788E-03_dp,&
      1.0277071420779189E+01_dp, -3.4998322063470488E+00_dp,&
      -1.4378747336383482E+00_dp,  9.7407812913651848E-01_dp,&
      5.8100968550036769E-01_dp, -3.6798347669688403E-01_dp,&
      -4.2502364982377133E-01_dp,  1.9512289769610702E-01_dp,&
      5.6300271102111910E-01_dp, -7.5027827357326204E-01_dp,&
      3.9235942713735511E-01_dp, -8.5921250073101446E-02_dp,&
      3.1413061827254385E-03_dp,  1.8195484597436025E+01_dp,&
      -7.5799465416797727E+00_dp, -2.7779407333989208E+00_dp,&
      2.4038088840560472E+00_dp,  1.1529366402453889E+00_dp,&
      -9.8759392678972879E-01_dp, -8.8098967842317599E-01_dp,&
      5.2980884220561242E-01_dp,  1.3004888148740070E+00_dp,&
      -1.9285239537624044E+00_dp,  1.1258532118551621E+00_dp,&
      -3.0321009022168460E-01_dp,  2.7043746144518259E-02_dp,&
      3.3035404761027252E+01_dp, -1.6491595898542446E+01_dp,&
      -5.2375830412033784E+00_dp,  5.8765409929385362E+00_dp,&
      2.1768388716426843E+00_dp, -2.5969780543056746E+00_dp,&
      -1.7626117368164964E+00_dp,  1.4322289011391323E+00_dp,&
      2.9516064099497408E+00_dp, -4.9190419937796754E+00_dp,&
      3.1669116909973813E+00_dp, -9.8168653418504981E-01_dp,&
      1.1715471750362567E-01_dp,  6.1469293824180966E+01_dp,&
      -3.6319806298406590E+01_dp, -9.5514353824431790E+00_dp,&
      1.4372704901961265E+01_dp,  3.7807422307232739E+00_dp,&
      -6.7679914605420413E+00_dp, -3.3434548753675188E+00_dp,&
      3.8965635434521699E+00_dp,  6.5408575540338090E+00_dp,&
      -1.2492746836072200E+01_dp,  8.8166589002534259E+00_dp,&
      -3.0419114467219512E+00_dp,  4.2503541134711348E-01_dp,&
      1.1743554310779896E+02_dp, -8.1509386397962075E+01_dp,&
      -1.6406506874595095E+01_dp,  3.5425877135044423E+01_dp,&
      5.4008009960033583E+00_dp, -1.7605685937044811E+01_dp,&
      -5.7037271668928087E+00_dp,  1.0759910584842750E+01_dp,&
      1.3847290180791390E+01_dp, -3.1533337618401688E+01_dp,&
      2.4393105499495121E+01_dp, -9.2014396622544474E+00_dp,&
      1.4283532789355629E+00_dp,  2.3109385131857380E+02_dp,&
      -1.8761966730640114E+02_dp, -2.4520412603580802E+01_dp,&
      8.8516278234992583E+01_dp,  2.9328252339946874E+00_dp,&
      -4.5889710850574829E+01_dp, -7.1950183014523059E+00_dp,&
      3.0366027067610979E+01_dp,  2.6306482065369778E+01_dp,&
      -7.8357372730095506E+01_dp,  6.6967780172219548E+01_dp,&
      -2.7398393744283439E+01_dp,  4.6109478575715883E+00_dp,&
      4.7043109624338825E+02_dp, -4.4606475685262001E+02_dp,&
      -2.1735952206843098E+01_dp,  2.2527950930573971E+02_dp,&
      -2.3012422758538580E+01_dp, -1.1981796817643516E+02_dp,&
      2.5291315036908233E+00_dp,  8.8003829705726233E+01_dp,&
      3.4937617704092141E+01_dp, -1.8678178196032340E+02_dp,&
      1.8052624320492154E+02_dp, -8.0190538925395487E+01_dp,&
      1.4478678662148862E+01_dp,  9.9623504829206684E+02_dp,&
      -1.1043779912072609E+03_dp,  5.3154531274160959E+01_dp,&
      5.8589592555521381E+02_dp, -1.5504952430522471E+02_dp,&
      -3.1084069476317563E+02_dp,  7.2155851070132712E+01_dp,&
      2.6196369615057944E+02_dp, -3.7901934092893228E+01_dp,&
      -3.9677065726243325E+02_dp,  4.6351722925927106E+02_dp,&
      -2.2696455653345569E+02_dp,  4.3990267926862309E+01_dp,&
      2.2110877219599788E+03_dp, -2.8764768741892444E+03_dp,&
      4.7492320131449105E+02_dp,  1.5567264701241727E+03_dp,&
      -7.3938540087289493E+02_dp, -7.7880598566919457E+02_dp,&
      4.1299391414202665E+02_dp,  7.9154965016367521E+02_dp,&
      -6.3312008083422745E+02_dp, -5.4669713260373055E+02_dp,&
      1.0339049127853618E+03_dp, -5.8970336298410371E+02_dp,&
      1.2498590069609844E+02_dp,  5.1953268832294525E+03_dp,&
      -7.9884476380606466E+03_dp,  2.4793733613308304E+03_dp,&
      4.1824523715100868E+03_dp, -3.2616952824520654E+03_dp,&
      -1.7015636062823082E+03_dp,  1.9191005411391400E+03_dp,&
      2.3134019608791955E+03_dp, -3.9142318595195161E+03_dp,&
      1.2048151664984891E+03_dp,  1.2558303204971614E+03_dp,&
      -1.1624542002305350E+03_dp,  2.9431974485633702E+02_dp,&
      1.3110918396829140E+04_dp, -2.4102687398992093E+04_dp,&
      1.1995982410614766E+04_dp,  1.0841615122259704E+04_dp,&
      -1.4393293492492468E+04_dp, -1.5198734249430386E+03_dp,&
      8.2366314984664277E+03_dp,  5.3369330859444490E+03_dp,&
      -1.8989222784588852E+04_dp,  1.6516142437368548E+04_dp,&
      -6.0985673599833599E+03_dp,  3.7054895141125257E+02_dp,&
      2.4857740696458779E+02_dp,  3.6333087513022270E+04_dp,&
      -8.1221783017847876E+04_dp,  6.1132709382630717E+04_dp,&
      2.0704857862231747E+04_dp, -6.5197332745595239E+04_dp,&
      2.1778211045947377E+04_dp,  3.0647530299908398E+04_dp,&
      -4.3442782810474673E+03_dp, -6.8761517136658484E+04_dp,&
      9.8370328388803697E+04_dp, -6.6087566887000561E+04_dp,&
      2.3191079659114679E+04_dp, -3.4339593811428126E+03_dp,&
      1.1539025686929584E+05_dp, -3.1920426799906261E+05_dp,&
      3.5835520091848128E+05_dp, -7.7332737818564172E+04_dp,&
      -2.7179397309597122E+05_dp,  2.9504892028661666E+05_dp,&
      -1.5850909591480361E+03_dp, -1.9513271077423554E+05_dp,&
      5.8739957732688774E+04_dp,  1.8218684831437774E+05_dp,&
      -2.3253340265484373E+05_dp,  1.1891512847722569E+05_dp,&
      -2.3965471466063856E+04_dp,  5.0796510014340468E+05_dp,&
      -1.6654594573956954E+06_dp,  2.6259920107028279E+06_dp,&
      -2.2320118568638815E+06_dp,  4.1945039078323619E+05_dp,&
      1.5259082255208537E+06_dp, -2.3070026158079393E+06_dp,&
      1.8438958389471469E+06_dp, -9.8021455412542529E+05_dp,&
      3.8852172824498301E+05_dp, -1.3583505516405738E+05_dp,&
      4.2733204548142872E+04_dp, -7.7320302758653297E+03_dp,&
      2.4714592967992341E-01_dp, -7.8085959775806109E-03_dp,&
      -6.7886161955341851E-03_dp,  2.3575304344995184E-03_dp,&
      -2.3517034282114465E-03_dp,  3.9158802999507076E-03_dp,&
      -4.5461153332725527E-03_dp,  4.5805194409910457E-03_dp,&
      -4.4524512104898387E-03_dp,  3.7555627758705215E-03_dp,&
      -2.3599409622613479E-03_dp,  9.4509400621173940E-04_dp,&
      -1.7884563401109535E-04_dp,  8.0966640464626982E-01_dp,&
      -3.0233867484354637E-02_dp, -2.6043673616899996E-02_dp,&
      9.5433966009009930E-03_dp, -9.0783414688469580E-03_dp,&
      1.5175856706915760E-02_dp, -1.7697460473983996E-02_dp,&
      1.7774158182275648E-02_dp, -1.7223209104464284E-02_dp,&
      1.4524811645378713E-02_dp, -9.1332381344281059E-03_dp,&
      3.6583813732935489E-03_dp, -6.9204130195118878E-04_dp,&
      1.6030458569493182E+00_dp, -7.7712739846721682E-02_dp,&
      -6.6010044423433833E-02_dp,  2.6127869801634368E-02_dp,&
      -2.3253711186275654E-02_dp,  3.9050839842992117E-02_dp,&
      -4.5857572874862529E-02_dp,  4.5840961396754240E-02_dp,&
      -4.4204734538243924E-02_dp,  3.7266920335001827E-02_dp,&
      -2.3458701939590761E-02_dp,  9.4004046670134924E-03_dp,&
      -1.7773797446858689E-03_dp,  2.8781590421606613E+00_dp,&
      -1.8502266072689696E-01_dp, -1.5472615939338874E-01_dp,&
      6.6339708849260826E-02_dp, -5.5249687600741691E-02_dp,&
      9.3041185647943017E-02_dp, -1.1013517526810855E-01_dp,&
      1.0955921820654839E-01_dp, -1.0506048446608686E-01_dp,&
      8.8531090619630326E-02_dp, -5.5801054481577961E-02_dp,&
      2.2374634069922589E-02_dp, -4.2288141306295558E-03_dp,&
      5.0740496965044271E+00_dp, -4.2609069373861314E-01_dp,&
      -3.5047238596729913E-01_dp,  1.6258723345136517E-01_dp,&
      -1.2725913237484188E-01_dp,  2.1429433697276487E-01_dp,&
      -2.5589943506352225E-01_dp,  2.5335777197170445E-01_dp,&
      -2.4146172104155522E-01_dp,  2.0334399963063937E-01_dp,&
      -1.2836270975711614E-01_dp,  5.1515644012971963E-02_dp,&
      -9.7343687583029415E-03_dp,  9.0087543486586252E+00_dp,&
      -9.6436054026559159E-01_dp, -7.7907974479593267E-01_dp,&
      3.9137647622410721E-01_dp, -2.8888988562914647E-01_dp,&
      4.8474201985199905E-01_dp, -5.8455659582923736E-01_dp,&
      5.7609137874519123E-01_dp, -5.4522036473334778E-01_dp,&
      4.5874071108557374E-01_dp, -2.9010852464938575E-01_dp,&
      1.1657483507620653E-01_dp, -2.2028508267688093E-02_dp,&
      1.6269081610823477E+01_dp, -2.1706703701271688E+00_dp,&
      -1.7186386235096978E+00_dp,  9.3797983745884228E-01_dp,&
      -6.5457017839938392E-01_dp,  1.0896280805830605E+00_dp,&
      -1.3287688787666494E+00_dp,  1.3036771339867574E+00_dp,&
      -1.2238033857287740E+00_dp,  1.0283519414815741E+00_dp,&
      -6.5175578160501146E-01_dp,  2.6235561770753085E-01_dp,&
      -4.9594225091627114E-02_dp,  3.0039520360403436E+01_dp,&
      -4.9089049295402107E+00_dp, -3.7977674798271108E+00_dp,&
      2.2636667351052835E+00_dp, -1.4972289723671712E+00_dp,&
      2.4584044262615730E+00_dp, -3.0368285262935673E+00_dp,&
      2.9668326974499704E+00_dp, -2.7581511636752905E+00_dp,&
      2.3131661843896207E+00_dp, -1.4699607563494359E+00_dp,&
      5.9316114483746940E-01_dp, -1.1222595041176153E-01_dp,&
      5.6934882930425353E+01_dp, -1.1255095485895385E+01_dp,&
      -8.4747322577007207E+00_dp,  5.5559279273831361E+00_dp,&
      -3.4946282450544621E+00_dp,  5.6164991647309872E+00_dp,&
      -7.0421238497195029E+00_dp,  6.8536265224353468E+00_dp,&
      -6.2970979175561093E+00_dp,  5.2656213569535302E+00_dp,&
      -3.3569884616944177E+00_dp,  1.3592370978249693E+00_dp,&
      -2.5758842009927652E-01_dp,  1.1122973850857802E+02_dp,&
      -2.6389905796723561E+01_dp, -1.9237652263540834E+01_dp,&
      1.3999317461544457E+01_dp, -8.4157473265769305E+00_dp,&
      1.3101786091490547E+01_dp, -1.6712238563942623E+01_dp,&
      1.6216483953281678E+01_dp, -1.4686105782654973E+01_dp,&
      1.2224250268447630E+01_dp, -7.8233760607390384E+00_dp,&
      3.1827221380884012E+00_dp, -6.0484510597143726E-01_dp,&
      2.2511331110403108E+02_dp, -6.3849510824888611E+01_dp,&
      -4.4736822972493066E+01_dp,  3.6570941293529650E+01_dp,&
      -2.1172368843978809E+01_dp,  3.1477570716288106E+01_dp,&
      -4.0948178433469003E+01_dp,  3.9674042582809108E+01_dp,&
      -3.5289493665272310E+01_dp,  2.9158345876804262E+01_dp,&
      -1.8743398238805774E+01_dp,  7.6757245316723974E+00_dp,&
      -1.4653860927918039E+00_dp,  4.7504666597161093E+02_dp,&
      -1.6105638525337199E+02_dp, -1.0734835568336860E+02_dp,&
      1.0017665095009950E+02_dp, -5.6502935380445450E+01_dp,&
      7.8671378853738148E+01_dp, -1.0460354615827933E+02_dp,&
      1.0145728089695098E+02_dp, -8.8217580108401904E+01_dp,&
      7.2003186291951550E+01_dp, -4.6494111524940820E+01_dp,&
      1.9214944281734663E+01_dp, -3.6952935742615893E+00_dp,&
      1.0544524739814613E+03_dp, -4.2902221397650425E+02_dp,&
      -2.6782200683344763E+02_dp,  2.9185470703014238E+02_dp,&
      -1.6321979346586420E+02_dp,  2.0720494992345783E+02_dp,&
      -2.8184954114105011E+02_dp,  2.7486750102605271E+02_dp,&
      -2.3215822832786790E+02_dp,  1.8548517561875852E+02_dp,&
      -1.2013476439901228E+02_dp,  5.0268802195171816E+01_dp,&
      -9.7793374602533572E+00_dp,  2.4925376021791531E+03_dp,&
      -1.2280929973471482E+03_dp, -7.0003357512153480E+02_dp,&
      9.2189747375624108E+02_dp, -5.2511408044203222E+02_dp,&
      5.8621847685149055E+02_dp, -8.1277799265203862E+02_dp,&
      8.0292827818442333E+02_dp, -6.5338665830524042E+02_dp,&
      5.0183079090547267E+02_dp, -3.2380554523618252E+02_dp,&
      1.3763419521849823E+02_dp, -2.7255890559133967E+01_dp,&
      6.3918219717306083E+03_dp, -3.8763187937858406E+03_dp,&
      -1.9257015318780568E+03_dp,  3.2472036360073544E+03_dp,&
      -1.9642544031850207E+03_dp,  1.8422466462758632E+03_dp,&
      -2.5561629910519391E+03_dp,  2.5895289445906710E+03_dp,&
      -2.0116564542023402E+03_dp,  1.4283611462124322E+03_dp,&
      -8.9489172870925711E+02_dp,  3.8577334167655448E+02_dp,&
      -7.8417479541745152E+01_dp,  1.8326761523831301E+04_dp /)
    REAL(KIND=dp), DIMENSION(400), PARAMETER :: c05 = (/ &
      -1.4073380941871266E+04_dp, -5.4884212128021491E+03_dp,&
      1.3338114839397269E+04_dp, -9.1704389994968824E+03_dp,&
      6.9234192947715492E+03_dp, -8.9848495511906458E+03_dp,&
      9.4904838722939803E+03_dp, -7.0131996973248588E+03_dp,&
      4.2053129160131894E+03_dp, -2.2892321189981762E+03_dp,&
      9.5853233917332932E+02_dp, -2.0059595633199862E+02_dp,&
      6.2145484992675003E+04_dp, -6.3683464018306629E+04_dp,&
      -1.3478172242723740E+04_dp,  6.9122642519095578E+04_dp,&
      -6.1149663704367988E+04_dp,  3.8450079402543517E+04_dp,&
      -3.6623205602609836E+04_dp,  3.9290615378684073E+04_dp,&
      -2.8219652444004834E+04_dp,  1.1412807988657049E+04_dp,&
      -1.7224522927570106E+03_dp, -3.9040817007118409E+02_dp,&
      1.4646631734514426E+02_dp,  2.8572736221717932E+05_dp,&
      -4.3463112005624449E+05_dp,  8.0601652340817658E+04_dp,&
      5.0202883058267244E+05_dp, -7.9735137443422247E+05_dp,&
      6.5856450151112222E+05_dp, -3.1469388927102822E+05_dp,&
      4.5380852232874913E+03_dp,  1.9064981445481826E+05_dp,&
      -2.6045344183859305E+05_dp,  2.0964678182384724E+05_dp,&
      -9.9723937310119960E+04_dp,  2.1478645199727333E+04_dp,&
      2.0209842869144579E+00_dp, -6.6718986697254859E-02_dp,&
      -5.7864043748775733E-02_dp,  2.0385124539857375E-02_dp,&
      -2.0076648172407022E-02_dp,  3.3467236798597394E-02_dp,&
      -3.8898880867761311E-02_dp,  3.9159779450315686E-02_dp,&
      -3.8033899789202824E-02_dp,  3.2079462322166769E-02_dp,&
      -2.0161695424411413E-02_dp,  8.0746211326208697E-03_dp,&
      -1.5278524818652483E-03_dp,  2.5837103332261946E+00_dp,&
      -1.2412305755869253E-01_dp, -1.0562856977936072E-01_dp,&
      4.1401929220466462E-02_dp, -3.7137069265504780E-02_dp,&
      6.2373047952612670E-02_dp, -7.3167139494141137E-02_dp,&
      7.3181444229383899E-02_dp, -7.0620695217909493E-02_dp,&
      5.9541226705652497E-02_dp, -3.7473058713811576E-02_dp,&
      1.5014631941726080E-02_dp, -2.8389529827758256E-03_dp,&
      3.9243795810678530E+00_dp, -2.7752925964967601E-01_dp,&
      -2.3174959372548332E-01_dp,  1.0011305293692697E-01_dp,&
      -8.2767548665314883E-02_dp,  1.3960986500361180E-01_dp,&
      -1.6533811213529379E-01_dp,  1.6437972410323695E-01_dp,&
      -1.5757476076598648E-01_dp,  1.3278585216426170E-01_dp,&
      -8.3698567928797013E-02_dp,  3.3559074454691487E-02_dp,&
      -6.3419102382915853E-03_dp,  6.5574218511326112E+00_dp,&
      -6.2720528041932788E-01_dp, -5.1457373180315114E-01_dp,&
      2.4165147820461136E-01_dp, -1.8706272866170806E-01_dp,&
      3.1556412614357066E-01_dp, -3.7722243153075635E-01_dp,&
      3.7313797192010750E-01_dp, -3.5535802476053696E-01_dp,&
      2.9926050182479369E-01_dp, -1.8893716237645972E-01_dp,&
      7.5825057518831993E-02_dp, -1.4325630615601024E-02_dp,&
      1.1531103735581528E+01_dp, -1.4032618163838810E+00_dp,&
      -1.1301837639020289E+00_dp,  5.7553236097795246E-01_dp,&
      -4.1986809658151530E-01_dp,  7.0558155272988798E-01_dp,&
      -8.5201620306659231E-01_dp,  8.3882578576766720E-01_dp,&
      -7.9314185990164077E-01_dp,  6.6733090862986277E-01_dp,&
      -4.2210637535148637E-01_dp,  1.6962027057736975E-01_dp,&
      -3.2047172429002457E-02_dp,  2.0953856325861402E+01_dp,&
      -3.1332175704610203E+00_dp, -2.4722201318657344E+00_dp,&
      1.3685689515151018E+00_dp, -9.4397783930672019E-01_dp,&
      1.5731408557496862E+00_dp, -1.9214695312017522E+00_dp,&
      1.8831652333939095E+00_dp, -1.7658826005426529E+00_dp,&
      1.4838298143011921E+00_dp, -9.4067732071431376E-01_dp,&
      3.7868226521725007E-01_dp, -7.1573121884394186E-02_dp,&
      3.9141027382473368E+01_dp, -7.0510935911389439E+00_dp,&
      -5.4349441219378498E+00_dp,  3.2859057843378880E+00_dp,&
      -2.1492051697234911E+00_dp,  3.5316064256642075E+00_dp,&
      -4.3703582897803956E+00_dp,  4.2649336499090476E+00_dp,&
      -3.9602705121144273E+00_dp,  3.3213171066858487E+00_dp,&
      -2.1112848586815800E+00_dp,  8.5203244277433265E-01_dp,&
      -1.6118013209548254E-01_dp,  7.5179666844985803E+01_dp,&
      -1.6129525112712493E+01_dp, -1.2099880210005864E+01_dp,&
      8.0393543887536687E+00_dp, -5.0050419349925148E+00_dp,&
      8.0491356578510747E+00_dp, -1.0111118000567620E+01_dp,&
      9.8296164480187027E+00_dp, -9.0208256786749921E+00_dp,&
      7.5435906846169685E+00_dp, -4.8109961052478889E+00_dp,&
      1.9481547611953693E+00_dp, -3.6913036892062134E-01_dp,&
      1.4889534986689205E+02_dp, -3.7795695700544215E+01_dp,&
      -2.7459767911560213E+01_dp,  2.0211149927742266E+01_dp,&
      -1.2040077089570843E+01_dp,  1.8764352859283580E+01_dp,&
      -2.3978110155581604E+01_dp,  2.3240884657405484E+01_dp,&
      -2.1026891338026122E+01_dp,  1.7506149076342183E+01_dp,&
      -1.1208140964030401E+01_dp,  4.5598936248731166E+00_dp,&
      -8.6634263812127699E-01_dp,  3.0539686947459830E+02_dp,&
      -9.1444333087771867E+01_dp, -6.3921240045689750E+01_dp,&
      5.2663651852895761E+01_dp, -3.0245493927589340E+01_dp,&
      4.5084324511456899E+01_dp, -5.8738721586867534E+01_dp,&
      5.6842137146892654E+01_dp, -5.0535367379859217E+01_dp,&
      4.1783876072414230E+01_dp, -2.6871281987162011E+01_dp,&
      1.1002744908733408E+01_dp, -2.0995744007601536E+00_dp,&
      6.5253194924819445E+02_dp, -2.3049888975569243E+02_dp,&
      -1.5360350190567564E+02_dp,  1.4363449886237967E+02_dp,&
      -8.0388980829080381E+01_dp,  1.1261707068019670E+02_dp,&
      -1.4991417030352827E+02_dp,  1.4517931236170054E+02_dp,&
      -1.2629435322089957E+02_dp,  1.0326348811780197E+02_dp,&
      -6.6723417286080917E+01_dp,  2.7562125203340528E+01_dp,&
      -5.2956364086681749E+00_dp,  1.4638759455072827E+03_dp,&
      -6.1204832725053041E+02_dp, -3.8364710618768743E+02_dp,&
      4.1505445999875218E+02_dp, -2.2991842627563597E+02_dp,&
      2.9570852999172786E+02_dp, -4.0267828686987724E+02_dp,&
      3.9173441007461383E+02_dp, -3.3158823108198732E+02_dp,&
      2.6608267952167319E+02_dp, -1.7259790833879237E+02_dp,&
      7.2150618982427929E+01_dp, -1.4010375050526267E+01_dp,&
      3.4854439155843625E+03_dp, -1.7372008876781651E+03_dp,&
      -1.0030747630399879E+03_dp,  1.2911405329593324E+03_dp,&
      -7.2402985904807974E+02_dp,  8.2901556502561914E+02_dp,&
      -1.1526573479976369E+03_dp,  1.1334817444713410E+03_dp,&
      -9.2688170377254664E+02_dp,  7.1957041790704716E+02_dp,&
      -4.6656972587699590E+02_dp,  1.9808433106431835E+02_dp,&
      -3.9093212258748324E+01_dp,  8.9452333593947151E+03_dp,&
      -5.3804654916003874E+03_dp, -2.7615678829835888E+03_dp,&
      4.4211939369662241E+03_dp, -2.5941258974348689E+03_dp,&
      2.5441305071386205E+03_dp, -3.5725792116391317E+03_dp,&
      3.5873032039055993E+03_dp, -2.8097510721061076E+03_dp,&
      2.0498805737947519E+03_dp, -1.3081644460504003E+03_dp,&
      5.6519369679006809E+02_dp, -1.1429316751443122E+02_dp,&
      2.5338916917680017E+04_dp, -1.8766091152234465E+04_dp,&
      -7.9810318917773557E+03_dp,  1.7229548905712414E+04_dp,&
      -1.1116122770842045E+04_dp,  8.9620856529374523E+03_dp,&
      -1.2236743640136612E+04_dp,  1.2740974975183684E+04_dp,&
      -9.4927044237527389E+03_dp,  6.1054857729098449E+03_dp,&
      -3.6163779068096483E+03_dp,  1.5701523669074877E+03_dp,&
      -3.2867702174385209E+02_dp,  8.2423613014789240E+04_dp,&
      -7.7738621638431970E+04_dp, -2.2856223996885383E+04_dp,&
      8.0654743982346874E+04_dp, -6.1910908781176040E+04_dp,&
      4.0486392582576533E+04_dp, -4.7535009010394941E+04_dp,&
      5.2027077505429726E+04_dp, -3.7183064561553911E+04_dp,&
      1.8156573347610247E+04_dp, -7.2562244437422933E+03_dp,&
      2.6129376328153694E+03_dp, -5.6069917781247318E+02_dp,&
      3.3241938028525532E+05_dp, -4.2449719865456008E+05_dp,&
      -2.7991543060281649E+04_dp,  4.9815416292546922E+05_dp,&
      -5.2713017647717579E+05_dp,  3.1848201456787082E+05_dp,&
      -2.2093992761096239E+05_dp,  2.2247276428230319E+05_dp,&
      -1.5785444477482370E+05_dp,  3.4768551468795849E+04_dp,&
      3.5161092722404210E+04_dp, -2.9940407867245645E+04_dp,&
      7.4882756263552019E+03_dp,  2.0447139661067696E+06_dp,&
      -3.9283854089172743E+06_dp,  1.8502764473921761E+06_dp,&
      4.1951868191134324E+06_dp, -9.7529644361083508E+06_dp,&
      1.0318888468459859E+07_dp, -5.2802722255622474E+06_dp,&
      -2.1595666128842793E+06_dp,  7.7832591679149270E+06_dp,&
      -9.0678311244019158E+06_dp,  6.4686899047139660E+06_dp,&
      -2.7819922020553104E+06_dp,  5.5456672891937534E+05_dp,&
      2.4890279420896633E-01_dp,  3.0066734866592875E-09_dp,&
      -1.0527327234770156E-02_dp,  1.3433637283418252E-02_dp,&
      -1.7966097650019941E-02_dp,  1.6380268724846176E-02_dp,&
      2.5827143469974796E-03_dp, -5.0828042512973688E-02_dp,&
      1.2605197871020793E-01_dp, -1.9169993841817892E-01_dp,&
      1.9099346055836872E-01_dp, -1.1453423015330329E-01_dp,&
      3.1344018718015468E-02_dp,  8.1648099483442071E-01_dp,&
      1.1681804630596778E-08_dp, -4.0907368999085904E-02_dp,&
      5.2200775677796407E-02_dp, -6.9370421329701656E-02_dp,&
      6.2517319547751971E-02_dp,  1.2303423022201899E-02_dp,&
      -2.0110939199019731E-01_dp,  4.9426482807255301E-01_dp,&
      -7.4900279581754514E-01_dp,  7.4475006092198848E-01_dp,&
      -4.4603037474417395E-01_dp,  1.2195270515337567E-01_dp,&
      1.6206094508793791E+00_dp,  3.0183277900631536E-08_dp,&
      -1.0571814695626443E-01_dp,  1.3490399708668319E-01_dp,&
      -1.7755777128909234E-01_dp,  1.5716554526027085E-01_dp,&
      4.0575662816921231E-02_dp, -5.3363368225929542E-01_dp,&
      1.2944475547431196E+00_dp, -1.9512977152300366E+00_dp,&
      1.9344516123948587E+00_dp, -1.1563005205015051E+00_dp,&
      3.1572545247324441E-01_dp,  2.9200997436150145E+00_dp,&
      7.2272211306585797E-08_dp, -2.5319659711553028E-01_dp,&
      3.2309704851649657E-01_dp, -4.2074358141218299E-01_dp,&
      3.6486472342550103E-01_dp,  1.2013651265592906E-01_dp,&
      -1.3142416201487426E+00_dp,  3.1444431701789064E+00_dp,&
      -4.7133821392535511E+00_dp,  4.6576470048302268E+00_dp,&
      -2.7781948677092143E+00_dp,  7.5745674162454923E-01_dp,&
      5.1709355253819069E+00_dp,  1.6742699148040768E-07_dp,&
      -5.8671080264769337E-01_dp,  7.4868485784133132E-01_dp,&
      -9.6403362067884502E-01_dp,  8.1750470049367896E-01_dp,&
      3.3369738745733207E-01_dp, -3.1320371788601724E+00_dp,&
      7.3913387513511770E+00_dp, -1.1015588989961810E+01_dp,&
      1.0849123684781043E+01_dp, -6.4570948721058814E+00_dp,&
      1.7577598775352674E+00_dp,  9.2287628542971909E+00_dp,&
      3.8134106077365812E-07_dp, -1.3367097614968169E+00_dp,&
      1.7057363546008353E+00_dp, -2.1697290395664353E+00_dp,&
      1.7942979800129226E+00_dp,  8.9447410621817325E-01_dp,&
      -7.3445546549349077E+00_dp,  1.7090148060084527E+01_dp,&
      -2.5316713230788089E+01_dp,  2.4846267033206001E+01_dp,&
      -1.4753191637652103E+01_dp,  4.0094693816808542E+00_dp,&
      1.6766111681930639E+01_dp,  8.6435256783724784E-07_dp,&
      -3.0308060554179863E+00_dp,  3.8675214211329658E+00_dp,&
      -4.8529498283031085E+00_dp,  3.8976870882182326E+00_dp,&
      2.3616553134743259E+00_dp, -1.7167702511583300E+01_dp,&
      3.9359851775548030E+01_dp, -5.7927748597033606E+01_dp,&
      5.6632670017808159E+01_dp, -3.3540723665316413E+01_dp,&
      9.0986198218588719E+00_dp,  3.1168178536681012E+01_dp,&
      1.9700422170976357E-06_dp, -6.9105394933144373E+00_dp,&
      8.8183289442136772E+00_dp, -1.0894028309812372E+01_dp,&
      8.4485589750840440E+00_dp,  6.2361739723108975E+00_dp,&
      -4.0446807666182103E+01_dp,  9.1267594151432164E+01_dp,&
      -1.3336330401014530E+02_dp,  1.2982241038880991E+02_dp,&
      -7.6664694454963950E+01_dp,  2.0753697286099698E+01_dp,&
      5.9534897429557844E+01_dp,  4.5573631978173512E-06_dp,&
      -1.5993863131481948E+01_dp,  2.0409268076898947E+01_dp,&
      -2.4759174458477368E+01_dp,  1.8389829887940540E+01_dp,&
      1.6674262282079525E+01_dp, -9.7004865265531805E+01_dp,&
      2.1513624544240966E+02_dp, -3.1185418300656380E+02_dp,&
      3.0209632244299593E+02_dp, -1.7780645081741756E+02_dp,&
      4.8018190809835360E+01_dp,  1.1735957000719010E+02_dp,&
      1.0796734300526574E-05_dp, -3.7912420090540877E+01_dp,&
      4.8378813528148328E+01_dp, -5.7435863659353501E+01_dp,&
      4.0377808851989506E+01_dp,  4.5658311276489918E+01_dp,&
      -2.3911739347016501E+02_dp,  5.2031388028459696E+02_dp,&
      -7.4739624739763394E+02_dp,  7.1994201689992678E+02_dp /)
    REAL(KIND=dp), DIMENSION(13, 36, 5), PARAMETER :: &
      coefdata = RESHAPE ((/ c01, c02, c03, c04, c05, c06 /),(/13,36,5/))

    INTEGER                                  :: irange

     IF(Rc <=  4.0000000000000000E+01_dp) Rc =   4.0000000000000000E+01_dp
     IF(Rc >=  2.1987628936767578E+06_dp) Rc =   2.1987628936767578E+06_dp
     IF(Rc >=  4.0000000000000000E+01_dp .AND. Rc <=  5.5746054090099528E+02_dp) THEN
       irange =  1
       L_b =  4.0000000000000000E+01_dp
       U_b =  5.5746054090099528E+02_dp
     END IF
     IF(Rc >=  5.5746054090099528E+02_dp .AND. Rc <=  7.4758824984654611E+03_dp) THEN
       irange =  2
       L_b =  5.5746054090099528E+02_dp
       U_b =  7.4758824984654611E+03_dp
     END IF
     IF(Rc >=  7.4758824984654611E+03_dp .AND. Rc <=  1.2462901325470702E+05_dp) THEN
       irange =  3
       L_b =  7.4758824984654611E+03_dp
       U_b =  1.2462901325470702E+05_dp
     END IF
     IF(Rc >=  1.2462901325470702E+05_dp .AND. Rc <=  7.3667319606433529E+05_dp) THEN
       irange =  4
       L_b =  1.2462901325470702E+05_dp
       U_b =  7.3667319606433529E+05_dp
     END IF
     IF(Rc >=  7.3667319606433529E+05_dp .AND. Rc <=  2.1987628936767578E+06_dp) THEN
       irange =  5
       L_b =  7.3667319606433529E+05_dp
       U_b =  2.1987628936767578E+06_dp
     END IF

     fit_coef(1:13,1:36) = coefdata(1:13,1:36,irange)

   END SUBROUTINE get_fit_coef_k18

! *****************************************************************************
!> \brief ...
!> \param Rc ...
!> \param L_b ...
!> \param U_b ...
!> \param fit_coef ...
! *****************************************************************************
   SUBROUTINE get_fit_coef_k19(Rc,L_b,U_b,fit_coef)
    REAL(KIND=dp)                            :: Rc, L_b, U_b
    REAL(KIND=dp), DIMENSION(13, 38)         :: fit_coef

    REAL(KIND=dp), DIMENSION(164), PARAMETER :: c08 = (/ &
      1.0499223216920694E+01_dp,  4.4951728164305678E+00_dp,&
      -4.0844623976889828E+01_dp,  9.6469089733765301E+01_dp,&
      -1.4457308237433509E+02_dp,  1.4358591897299843E+02_dp,&
      -8.6337055270254567E+01_dp,  2.3772804326078404E+01_dp,&
      6.6359118115924133E+01_dp, -2.8591296818375736E-06_dp,&
      -1.7227311155696889E+01_dp,  2.2124688216575461E+01_dp,&
      -2.7524752734790738E+01_dp,  2.2372786649592609E+01_dp,&
      1.2155184152551417E+01_dp, -9.4518163584077797E+01_dp,&
      2.1941577176458151E+02_dp, -3.2637649842310663E+02_dp,&
      3.2272807635497901E+02_dp, -1.9348838342099501E+02_dp,&
      5.3166737579708254E+01_dp,  1.2849926566575485E+02_dp,&
      -6.5250582816014808E-06_dp, -3.9291091128744178E+01_dp,&
      5.0460729666257137E+01_dp, -6.1689440682005809E+01_dp,&
      4.8223722297260849E+01_dp,  3.3157059718603954E+01_dp,&
      -2.2390910129266675E+02_dp,  5.1024271893998286E+02_dp,&
      -7.5272887345661684E+02_dp,  7.4065729983418430E+02_dp,&
      -4.4258825906333180E+02_dp,  1.2132797157588969E+02_dp,&
      2.5680058641412631E+02_dp, -1.5312146095985592E-05_dp,&
      -9.2132612013296807E+01_dp,  1.1832391986938538E+02_dp,&
      -1.4171780637542841E+02_dp,  1.0550987736400363E+02_dp,&
      9.2295690191958684E+01_dp, -5.4710074862993338E+02_dp,&
      1.2219574601299419E+03_dp, -1.7860867470382068E+03_dp,&
      1.7476196856227098E+03_dp, -1.0403464625313584E+03_dp,&
      2.8441474799760618E+02_dp,  5.3222213783565780E+02_dp,&
      -3.7234716544473444E-05_dp, -2.2382764453118818E+02_dp,&
      2.8745677810311867E+02_dp, -3.3594463439207505E+02_dp,&
      2.3480947417458006E+02_dp,  2.6515448809283430E+02_dp,&
      -1.3903458609430929E+03_dp,  3.0378333769430460E+03_dp,&
      -4.3939128957463727E+03_dp,  4.2714720839997299E+03_dp,&
      -2.5314697435154421E+03_dp,  6.8983232947349563E+02_dp,&
      1.1512267982072294E+03_dp, -9.4695299828165009E-05_dp,&
      -5.6855359361921421E+02_dp,  7.3017979817497314E+02_dp,&
      -8.2805783277340424E+02_dp,  5.3123615799294919E+02_dp,&
      7.9604494681759957E+02_dp, -3.7117311299625480E+03_dp,&
      7.9142883179915798E+03_dp, -1.1309056810379891E+04_dp,&
      1.0909850675069327E+04_dp, -6.4311777917450936E+03_dp,&
      1.7456613494233954E+03_dp,  2.6212437093759077E+03_dp,&
      -2.5485209034150442E-04_dp, -1.5277300945628297E+03_dp,&
      1.9620247585698864E+03_dp, -2.1419889374981735E+03_dp,&
      1.2132646910388833E+03_dp,  2.5352427487043710E+03_dp,&
      -1.0542926029232349E+04_dp,  2.1867497472080187E+04_dp,&
      -3.0801081479878208E+04_dp,  2.9437600466280288E+04_dp,&
      -1.7238435953682736E+04_dp,  4.6562418705978034E+03_dp,&
      6.3573171344602770E+03_dp, -7.3750251193115297E-04_dp,&
      -4.4115279609245035E+03_dp,  5.6656029053998782E+03_dp,&
      -5.8829581406359675E+03_dp,  2.7234456726279182E+03_dp,&
      8.7354783549759413E+03_dp, -3.2415576280063124E+04_dp,&
      6.5114222442079190E+04_dp, -9.0113853476169999E+04_dp,&
      8.5113563712506788E+04_dp, -4.9417241881668429E+04_dp,&
      1.3262343922822860E+04_dp,  1.6709034360467489E+04_dp,&
      -2.3486683702591328E-03_dp, -1.4005755867923166E+04_dp,&
      1.7987158768947684E+04_dp, -1.7415019499264930E+04_dp,&
      5.3886108848086124E+03_dp,  3.3486192491280985E+04_dp,&
      -1.1058426237334734E+05_dp,  2.1373071702116015E+05_dp,&
      -2.8916673365143739E+05_dp,  2.6883819423513982E+05_dp,&
      -1.5426221397195625E+05_dp,  4.1027854475280248E+04_dp,&
      4.8891105989388227E+04_dp, -8.5375882415901910E-03_dp,&
      -5.0669203017368134E+04_dp,  6.5072639338247180E+04_dp,&
      -5.6622801394688104E+04_dp,  3.0202589453576870E+03_dp,&
      1.4914012489017315E+05_dp, -4.3485803392159345E+05_dp,&
      8.0005087015935709E+05_dp, -1.0489157105679847E+06_dp,&
      9.5284382085401053E+05_dp, -5.3704348597552557E+05_dp,&
      1.4082751009772695E+05_dp,  1.6683629894264581E+05_dp,&
      -3.7790306832555944E-02_dp, -2.2242916225572405E+05_dp,&
      2.8565589535384666E+05_dp, -2.0552031775314044E+05_dp,&
      -9.7947481898987331E+04_dp,  8.3244217354800832E+05_dp,&
      -2.1022382927533309E+06_dp,  3.6045398713238072E+06_dp,&
      -4.4935831417357307E+06_dp,  3.9211888536126167E+06_dp,&
      -2.1383095567867076E+06_dp,  5.4562133604672970E+05_dp,&
      7.2983594458675524E+05_dp, -2.3366549317086566E-01_dp,&
      -1.3516445238670940E+06_dp,  1.7358328797716717E+06_dp,&
      -7.6941638889201207E+05_dp, -1.8347737779277351E+06_dp,&
      6.8172234608326862E+06_dp, -1.4087562537947292E+07_dp,&
      2.1161535706049107E+07_dp, -2.3534962968572598E+07_dp,&
      1.8463007196192324E+07_dp, -9.1061534880313724E+06_dp,&
      2.1147690424670577E+06_dp,  5.3539598700785628E+06_dp,&
      -3.0211150365300039E+00_dp, -1.6646735851000162E+07_dp,&
      2.1377543596431356E+07_dp,  7.5558560062402077E+06_dp,&
      -6.6595246625963897E+07_dp,  1.2300183044218574E+08_dp,&
      -1.2663664724774376E+08_dp,  4.1031313200280927E+07_dp,&
      1.0283971430652012E+08_dp, -1.9310895904024291E+08_dp,&
      1.5058167673706385E+08_dp, -4.7897275226966202E+07_dp /)
    REAL(KIND=dp), DIMENSION(400), PARAMETER :: c01 = (/ &
      1.4159553890012480E-01_dp, -3.2212259532342614E-02_dp,&
      -2.5211569112270739E-03_dp,  1.0259263161798598E-03_dp,&
      1.7009451604343799E-04_dp, -7.4233588848268952E-05_dp,&
      -1.4091849072407304E-05_dp,  8.9792205912272354E-06_dp,&
      -6.1068721080120308E-07_dp,  2.3673956901460579E-06_dp,&
      -1.4083255944590247E-06_dp, -2.7195755501531476E-06_dp,&
      2.8146140048823581E-06_dp,  4.3687738412731247E-01_dp,&
      -1.0507959602935127E-01_dp, -6.2038156589891806E-03_dp,&
      3.4867260532288986E-03_dp,  4.0214827530637130E-04_dp,&
      -2.5825399367695534E-04_dp, -3.1252892528233610E-05_dp,&
      3.0947573806607806E-05_dp, -3.6098038277409914E-06_dp,&
      7.3259195158793407E-06_dp, -4.3056106890235391E-06_dp,&
      -8.7173329450119566E-06_dp,  9.0229802552072765E-06_dp,&
      7.6969894835349451E-01_dp, -2.0480550688388621E-01_dp,&
      -5.1614525784059537E-03_dp,  7.1434595006984259E-03_dp,&
      2.5699998302359232E-04_dp, -5.4217957031654817E-04_dp,&
      -9.9497238896661655E-06_dp,  6.3976878971682059E-05_dp,&
      -1.2652209576345682E-05_dp,  1.3188831511557239E-05_dp,&
      -7.3852575250832500E-06_dp, -1.6514325039434995E-05_dp,&
      1.7056531547933975E-05_dp,  1.1694161663883196E+00_dp,&
      -3.5457725458566364E-01_dp,  6.0405099172997122E-03_dp,&
      1.2652283618377483E-02_dp, -6.9573612869031239E-04_dp,&
      -9.6570008981377035E-04_dp,  9.3185669048836430E-05_dp,&
      1.1170291259476769E-04_dp, -3.4059568676163194E-05_dp,&
      2.1411436640986164E-05_dp, -1.0587098957813508E-05_dp,&
      -2.7739876771009426E-05_dp,  2.8436369456750092E-05_dp,&
      1.6722923778351919E+00_dp, -5.8623250288584683E-01_dp,&
      3.6803481489419941E-02_dp,  2.0292829632816360E-02_dp,&
      -3.1641052891675505E-03_dp, -1.5095879003062503E-03_dp,&
      3.4669256907157714E-04_dp,  1.7005900081597947E-04_dp,&
      -7.7249793311931302E-05_dp,  3.5028575965622117E-05_dp,&
      -1.3652892102255319E-05_dp, -4.4739744474324642E-05_dp,&
      4.5186940571270951E-05_dp,  2.3252215053670944E+00_dp,&
      -9.4543684120515115E-01_dp,  1.0338931425768211E-01_dp,&
      2.9396745206708937E-02_dp, -8.2573760376551154E-03_dp,&
      -2.0274975592751354E-03_dp,  8.4994675565299837E-04_dp,&
      2.1942491207429306E-04_dp, -1.5506894551206067E-04_dp,&
      5.9943198545642097E-05_dp, -1.6238288004320363E-05_dp,&
      -7.1013836471124102E-05_dp,  7.0143453024049232E-05_dp,&
      3.1906616395439409E+00_dp, -1.4990569225058550E+00_dp,&
      2.3356101549045380E-01_dp,  3.7152742084143882E-02_dp,&
      -1.7592764802963623E-02_dp, -2.1213171272414589E-03_dp,&
      1.7296120802156852E-03_dp,  2.1165538052484334E-04_dp,&
      -2.8264733608776007E-04_dp,  1.0684070365301925E-04_dp,&
      -1.8316063933529629E-05_dp, -1.1178382801896222E-04_dp,&
      1.0737911151933133E-04_dp,  4.3534382015127369E+00_dp,&
      -2.3459506946039794E+00_dp,  4.7408783987239850E-01_dp,&
      3.6262330606119872E-02_dp, -3.3299942574138243E-02_dp,&
      -9.3140059419598656E-04_dp,  3.1137054827187964E-03_dp,&
      5.1022668227572255E-05_dp, -4.7327734643169223E-04_dp,&
      1.9371495091259679E-04_dp, -2.1116509746927200E-05_dp,&
      -1.7465725270621350E-04_dp,  1.6279100510407628E-04_dp,&
      5.9303714588445304E+00_dp, -3.6329281082192590E+00_dp,&
      9.0285285883188005E-01_dp,  1.0499413585049452E-02_dp,&
      -5.7706685341736653E-02_dp,  3.1914219035582910E-03_dp,&
      5.0564950350439003E-03_dp, -4.3044903889887784E-04_dp,&
      -7.2872478490099949E-04_dp,  3.4876195691509751E-04_dp,&
      -2.8964892295533430E-05_dp, -2.7029244144607034E-04_dp,&
      2.4488000058885411E-04_dp,  8.0842352986683856E+00_dp,&
      -5.5785127765670346E+00_dp,  1.6482906476540169E+00_dp,&
      -7.3501953025711345E-02_dp, -9.2213493322770707E-02_dp,&
      1.3150366867854405E-02_dp,  7.3635168477600224E-03_dp,&
      -1.4982003516811509E-03_dp, -1.0196873457808851E-03_dp,&
      6.1292284623671504E-04_dp, -5.2523833942698411E-05_dp,&
      -4.1290691731029868E-04_dp,  3.6576881274268374E-04_dp,&
      1.1044653770852007E+01_dp, -8.5086264786359092E+00_dp,&
      2.9204021062989258E+00_dp, -2.8111787188864740E-01_dp,&
      -1.3429818303779237E-01_dp,  3.3638242999846295E-02_dp,&
      9.2262993367846335E-03_dp, -3.5217929571347388E-03_dp,&
      -1.2493756075931993E-03_dp,  1.0399803214705392E-03_dp,&
      -1.1423406576364230E-04_dp, -6.2036982403473006E-04_dp,&
      5.4255571555972932E-04_dp,  1.5140946973553808E+01_dp,&
      -1.2911167463604777E+01_dp,  5.0599558722610176E+00_dp,&
      -7.3604696739389575E-01_dp, -1.7049477007591834E-01_dp,&
      7.1395167777044383E-02_dp,  8.5304346040664409E-03_dp,&
      -6.9043372440724740E-03_dp, -1.1917442674394089E-03_dp,&
      1.6877417649437795E-03_dp, -2.5692597772479920E-04_dp,&
      -9.1320979479733819E-04_dp,  7.9917704737221692E-04_dp,&
      2.0857767729149380E+01_dp, -1.9522972409838573E+01_dp,&
      8.6166217738442601E+00_dp, -1.6655977945022149E+00_dp,&
      -1.6130946336101359E-01_dp,  1.3407695333391192E-01_dp,&
      7.3462104420434426E-04_dp, -1.1786497079900466E-02_dp,&
      -4.1197943246183656E-04_dp,  2.5837162514269450E-03_dp,&
      -5.5402838065590862E-04_dp, -1.3110273196903820E-03_dp,&
      1.1688682117688078E-03_dp,  2.8939950737247685E+01_dp,&
      -2.9480526610440588E+01_dp,  1.4476451649973361E+01_dp,&
      -3.4764436718654159E+00_dp, -1.1492003298758746E-02_dp,&
      2.2502856445645558E-01_dp, -2.2365897846246961E-02_dp,&
      -1.7268108576283563E-02_dp,  1.7427539085000711E-03_dp,&
      3.6430042980382660E-03_dp, -1.1065725030894821E-03_dp,&
      -1.8275133932960331E-03_dp,  1.6976814016934469E-03_dp,&
      4.0618956502082611E+01_dp, -4.4622059577162375E+01_dp,&
      2.4090578510499956E+01_dp, -6.8810902494300450E+00_dp,&
      4.8297850963216071E-01_dp,  3.2924474886243160E-01_dp,&
      -7.2369776779368955E-02_dp, -2.0078222605003318E-02_dp,&
      5.8293068973593737E-03_dp,  4.5736167580590990E-03_dp,&
      -1.9885526061121971E-03_dp, -2.4869733482698786E-03_dp,&
      2.4561874555545063E-03_dp,  5.8186666034640020E+01_dp,&
      -6.8227495131103126E+01_dp,  3.9981099201342317E+01_dp,&
      -1.3134066085088833E+01_dp,  1.7219507581901394E+00_dp,&
      3.8681398325887517E-01_dp, -1.5929705043037320E-01_dp,&
      -1.3820689038096395E-02_dp,  1.1500886478842712E-02_dp,&
      4.9890084140602840E-03_dp, -3.1430775711513877E-03_dp,&
      -3.4228763531890033E-03_dp,  3.5888038606415973E-03_dp,&
      8.6758629460574809E+01_dp, -1.0730358374322792E+02_dp,&
      6.7254446024704066E+01_dp, -2.4650559229093147E+01_dp,&
      4.4657471444515995E+00_dp,  2.5973390816722725E-01_dp,&
      -2.8281531420486583E-01_dp,  8.6395325668061733E-03_dp,&
      1.7190940344631079E-02_dp,  4.9683014527230921E-03_dp,&
      -4.4797399994558479E-03_dp, -5.0527887500770325E-03_dp,&
      5.4485658862832098E-03_dp,  1.4144570865336470E+02_dp,&
      -1.8203037827531273E+02_dp,  1.1996729501753330E+02_dp,&
      -4.7597068467016200E+01_dp,  1.0433251046275085E+01_dp,&
      -3.0715161860465295E-01_dp, -4.4333987528632623E-01_dp,&
      5.2266426831911433E-02_dp,  2.2622601468742540E-02_dp,&
      5.6448052547070247E-03_dp, -6.6314050067125879E-03_dp,&
      -8.2807273014170093E-03_dp,  8.9883473255153619E-03_dp,&
      2.9783846082084170E+02_dp, -3.9281745649836722E+02_dp,&
      2.6705245994554446E+02_dp, -1.1116159567569434E+02_dp,&
      2.7022970008284052E+01_dp, -2.0393625814260337E+00_dp,&
      -7.8644885792012886E-01_dp,  1.4664747765129174E-01_dp,&
      3.6698696189955908E-02_dp,  1.0255959134649445E-02_dp,&
      -1.3496519985266115E-02_dp, -1.7057146077249077E-02_dp,&
      1.8839418118163746E-02_dp,  1.1407412355136231E+00_dp,&
      -2.6322614975892222E-01_dp, -1.9304807485994203E-02_dp,&
      8.4818056715191947E-03_dp,  1.2921553214948484E-03_dp,&
      -6.1798626362584322E-04_dp, -1.0574734927292684E-04_dp,&
      7.4567513561016663E-05_dp, -6.0228382320527032E-06_dp,&
      1.9073396980006737E-05_dp, -1.1324448161278565E-05_dp,&
      -2.2120940011110670E-05_dp,  2.2897251630575986E-05_dp,&
      1.2383058680166454E+00_dp, -3.3177779766973436E-01_dp,&
      -8.0046278756808614E-03_dp,  1.1734934753880745E-02_dp,&
      3.9208403634974115E-04_dp, -8.9816173297452740E-04_dp,&
      -1.3958158920641618E-05_dp,  1.0603060144415561E-04_dp,&
      -2.0721841907929155E-05_dp,  2.1014251840363297E-05_dp,&
      -1.1928943188164544E-05_dp, -2.6670935778900919E-05_dp,&
      2.7588164189909763E-05_dp,  1.4436406826714989E+00_dp,&
      -4.8123537796251287E-01_dp,  1.9845018196607334E-02_dp,&
      1.7971084322191626E-02_dp, -1.8324606541230578E-03_dp,&
      -1.4018145539285401E-03_dp,  2.1266040253605433E-04_dp,&
      1.6089571777858162E-04_dp, -5.5758950174576915E-05_dp,&
      2.6807942924299252E-05_dp, -1.2671269591892968E-05_dp,&
      -3.6769890090760815E-05_dp,  3.7679474165084254E-05_dp,&
      1.7780289941084260E+00_dp, -7.3782695706333956E-01_dp,&
      7.5852316169277137E-02_dp,  2.6321369018930191E-02_dp,&
      -6.2530501547849882E-03_dp, -1.9778677729210049E-03_dp,&
      6.5769145784193284E-04_dp,  2.1870962072471271E-04_dp,&
      -1.2174982968187595E-04_dp,  4.0911670934797237E-05_dp,&
      -1.2765102734804405E-05_dp, -5.4569409911678445E-05_dp,&
      5.4688757320726091E-05_dp,  2.2759204975194458E+00_dp,&
      -1.1453785133959720E+00_dp,  1.8079354115664725E-01_dp,&
      3.4432382914558510E-02_dp, -1.4225585470058071E-02_dp,&
      -2.2837089659632549E-03_dp,  1.4367339690524530E-03_dp,&
      2.3591365549879067E-04_dp, -2.3239837931551176E-04_dp,&
      7.1833055128279899E-05_dp, -1.1507142004923511E-05_dp,&
      -8.3724655681505198E-05_dp,  8.1177843733305105E-05_dp,&
      2.9887064285877973E+00_dp, -1.7721307937649995E+00_dp,&
      3.6991743513447239E-01_dp,  3.6672484709771344E-02_dp,&
      -2.7596569461930998E-02_dp, -1.6078064447974708E-03_dp,&
      2.6752315429307439E-03_dp,  1.2818773343005159E-04_dp,&
      -4.0049270094921513E-04_dp,  1.3443710602048975E-04_dp,&
      -9.0985942281952549E-06_dp, -1.2990401189688037E-04_dp,&
      1.2115099317095869E-04_dp,  3.9907366836701370E+00_dp,&
      -2.7216952914669283E+00_dp,  7.0185367490954009E-01_dp,&
      2.0790650087634404E-02_dp, -4.8523023243476252E-02_dp,&
      1.4087441403410760E-03_dp,  4.4519866778520399E-03_dp,&
      -2.5248221642837002E-04_dp, -6.2960311984174694E-04_dp,&
      2.5254010380090493E-04_dp, -8.2259066803737854E-06_dp,&
      -2.0128177724252732E-04_dp,  1.8061401811982056E-04_dp,&
      5.3884484703298714E+00_dp, -4.1497815192118628E+00_dp,&
      1.2730263450206805E+00_dp, -3.8210239808827151E-02_dp,&
      -7.8752008993259112E-02_dp,  9.1787131489447105E-03_dp,&
      6.6677701366350964E-03_dp, -1.1427359524673077E-03_dp,&
      -8.9785110256968275E-04_dp,  4.6106575711624083E-04_dp,&
      -1.6852240096092439E-05_dp, -3.0877518141384864E-04_dp,&
      2.6828132709743447E-04_dp,  7.3341836390666773E+00_dp,&
      -6.2894029533210078E+00_dp,  2.2407098704133954E+00_dp,&
      -1.8918988940658871E-01_dp, -1.1761608329525802E-01_dp,&
      2.5701025480113431E-02_dp,  8.7678899455993949E-03_dp,&
      -2.8850805066351374E-03_dp, -1.1285234289110727E-03_dp,&
      8.0654726904078294E-04_dp, -5.2692171698939168E-05_dp,&
      -4.6577350369995772E-04_dp,  3.9642409742979582E-04_dp,&
      1.0047710850872544E+01_dp, -9.4890315198232518E+00_dp,&
      3.8596089366441526E+00_dp, -5.2472117726711742E-01_dp,&
      -1.5718795036158192E-01_dp,  5.7072740328761015E-02_dp,&
      9.1939643128644485E-03_dp, -5.9066984136172977E-03_dp,&
      -1.1387477711487443E-03_dp,  1.3430321326247108E-03_dp,&
      -1.5016650487631499E-04_dp, -6.8710530153629065E-04_dp,&
      5.8196249923196924E-04_dp,  1.3851688904176260E+01_dp,&
      -1.4271598955417318E+01_dp,  6.5394232734786746E+00_dp,&
      -1.2157582321122917E+00_dp, -1.7145472518196578E-01_dp,&
      1.1137485470268504E-01_dp,  4.3748739759179806E-03_dp,&
      -1.0567793142941557E-02_dp, -5.5398723777692134E-04_dp,&
      2.1139656566692543E-03_dp, -3.7128217874996786E-04_dp,&
      -9.8650715039269197E-04_dp,  8.4812423349403076E-04_dp,&
      1.9235334619586016E+01_dp, -2.1430578544580889E+01_dp,&
      1.0935348023770239E+01_dp, -2.5703290931021043E+00_dp,&
      -9.3684439339661404E-02_dp,  1.9608483863631485E-01_dp,&
      -1.2873315513389102E-02_dp, -1.6635857332202259E-02_dp,&
      1.3017736724714114E-03_dp,  3.0919884927816301E-03_dp /)
    REAL(KIND=dp), DIMENSION(400), PARAMETER :: c02 = (/ &
      -8.1899012855183456E-04_dp, -1.3697632531141069E-03_dp,&
      1.2268039489488590E-03_dp,  2.6982367118180953E+01_dp,&
      -3.2203467268313922E+01_dp,  1.8094671738936693E+01_dp,&
      -5.1291575498824864E+00_dp,  2.2721945369117036E-01_dp,&
      3.0848319451415068E-01_dp, -5.4944705448018556E-02_dp,&
      -2.1918290310463735E-02_dp,  5.3889292561536137E-03_dp,&
      4.0275986677558995E-03_dp, -1.6284311851828951E-03_dp,&
      -1.8215037039926844E-03_dp,  1.7609141110641021E-03_dp,&
      3.8463536907902750E+01_dp, -4.8640580195082968E+01_dp,&
      2.9722457220673043E+01_dp, -9.8185297472519668E+00_dp,&
      1.1017236982453140E+00_dp,  4.1094138704569305E-01_dp,&
      -1.3790144274071689E-01_dp, -1.9836806896771739E-02_dp,&
      1.2281130435731527E-02_dp,  4.2485407574071928E-03_dp,&
      -2.8502932676645093E-03_dp, -2.3204621866155586E-03_dp,&
      2.5139038804018183E-03_dp,  5.6403019544607950E+01_dp,&
      -7.4558074137380629E+01_dp,  4.8792909062352564E+01_dp,&
      -1.8222521073735450E+01_dp,  3.1113259029642006E+00_dp,&
      3.8525550418227494E-01_dp, -2.6956980951719761E-01_dp,&
      1.7730572492532272E-03_dp,  2.0231218118677733E-02_dp,&
      2.8971793876300693E-03_dp, -4.1670706906750512E-03_dp,&
      -3.0299779823530571E-03_dp,  3.6417755428282965E-03_dp,&
      8.7215443808268901E+01_dp, -1.1851941798879672E+02_dp,&
      8.1443910354759353E+01_dp, -3.3294997428270143E+01_dp,&
      7.2813751191796143E+00_dp, -1.4949444862504643E-02_dp,&
      -4.2555970602879750E-01_dp,  5.3440375091183941E-02_dp,&
      2.3936805411273999E-02_dp,  4.2943352588307808E-04_dp,&
      -4.9592740606350010E-03_dp, -4.7297595630543931E-03_dp,&
      5.6368552675931643E-03_dp,  1.4975251635328343E+02_dp,&
      -2.0531240455284888E+02_dp,  1.4459250832690373E+02_dp,&
      -6.2316167155448824E+01_dp,  1.5665274878658106E+01_dp,&
      -1.1636674097249506E+00_dp, -5.5714790450689045E-01_dp,&
      1.2874387847404120E-01_dp,  2.0981878079326013E-02_dp,&
      1.6097121510941226E-04_dp, -6.0483121208980773E-03_dp,&
      -8.8514602159330009E-03_dp,  9.7828665118702071E-03_dp,&
      3.2129985348974549E+02_dp, -4.3705037011365653E+02_dp,&
      3.0768674866967689E+02_dp, -1.3421668175563934E+02_dp,&
      3.5382166323770129E+01_dp, -3.7150156401778789E+00_dp,&
      -7.8066308454448430E-01_dp,  2.3019139497858526E-01_dp,&
      2.6698155626115754E-02_dp,  6.9031498421899722E-03_dp,&
      -1.3061502264041481E-02_dp, -1.8618105437648663E-02_dp,&
      2.0549011501193169E-02_dp,  1.2327094475164540E+03_dp,&
      -1.6517971287587309E+03_dp,  1.1453145166666175E+03_dp,&
      -4.9100832192840704E+02_dp,  1.2656644760117726E+02_dp,&
      -1.2735766360571926E+01_dp, -2.6884467290906815E+00_dp,&
      6.7843516874228083E-01_dp,  1.2057077781690270E-01_dp,&
      4.0211801204278225E-02_dp, -5.5885010022905059E-02_dp,&
      -6.8430520456983487E-02_dp,  7.7231660087382012E-02_dp,&
      1.8496027008239549E-01_dp, -5.2635957315967263E-02_dp,&
      -1.3363208633747327E-02_dp,  7.0753080700979715E-03_dp,&
      4.0250101520871492E-03_dp, -2.5572951671070160E-03_dp,&
      -1.7101514222507485E-03_dp,  9.0274083105137438E-04_dp,&
      1.7429230871490228E-03_dp, -1.9224938737565124E-03_dp,&
      5.7596057684833031E-04_dp,  1.0659543229708461E-04_dp,&
      -7.1924049427382215E-05_dp,  5.8244506483690739E-01_dp,&
      -1.8223607569173905E-01_dp, -3.8829020775716601E-02_dp,&
      2.7386770439359200E-02_dp,  1.1779696794311099E-02_dp,&
      -1.0280482019437921E-02_dp, -4.9672977515278658E-03_dp,&
      3.7715941587838780E-03_dp,  5.9254776501040194E-03_dp,&
      -7.8754309889097502E-03_dp,  3.3370240188511616E-03_dp,&
      -2.4573980952193470E-04_dp, -1.3766621817983240E-04_dp,&
      1.0679356036687564E+00_dp, -3.9301786697228080E-01_dp,&
      -5.7337229831602214E-02_dp,  6.8138158271713281E-02_dp,&
      1.7189370118560782E-02_dp, -2.6680191557440031E-02_dp,&
      -6.9610330874151343E-03_dp,  1.0234093194331958E-02_dp,&
      1.1928832504988295E-02_dp, -2.0825809621203371E-02_dp,&
      1.1845502328236113E-02_dp, -2.7625162573559322E-03_dp,&
      1.2266527208606984E-04_dp,  1.7186762769732131E+00_dp,&
      -7.6842283829929103E-01_dp, -5.2094219929163819E-02_dp,&
      1.4952064221850928E-01_dp,  1.3105518526079931E-02_dp,&
      -6.0277036362458583E-02_dp, -3.8993134189913609E-03_dp,&
      2.3998100221899753E-02_dp,  1.9722360015163132E-02_dp,&
      -4.7449318260552320E-02_dp,  3.3022861705973044E-02_dp,&
      -1.0525079839838058E-02_dp,  1.2565018126329791E-03_dp,&
      2.6433292578569403E+00_dp, -1.4442699325289519E+00_dp,&
      1.8749612571812301E-02_dp,  3.0152805079239342E-01_dp,&
      -1.9072651296996059E-02_dp, -1.2313703162394884E-01_dp,&
      1.3982139796035626E-02_dp,  5.0529807099352154E-02_dp,&
      2.5915913834361832E-02_dp, -9.6301299125299111E-02_dp,&
      7.8387553232621393E-02_dp, -2.9347212002760113E-02_dp,&
      4.3984494251308407E-03_dp,  4.0031890214864516E+00_dp,&
      -2.6506777836645301E+00_dp,  2.5002709371286003E-01_dp,&
      5.6678418963927524E-01_dp, -1.2166589054182736E-01_dp,&
      -2.3025504915310566E-01_dp,  6.8512851967640287E-02_dp,&
      9.6897025962713448E-02_dp,  1.8655348484778574E-02_dp,&
      -1.7509091764540302E-01_dp,  1.6512362797291477E-01_dp,&
      -6.9241563143609311E-02_dp,  1.1677077362884182E-02_dp,&
      6.0451957328475263E+00_dp, -4.7842190085775247E+00_dp,&
      8.4607600090330592E-01_dp,  9.9838014493270644E-01_dp,&
      -3.8386031304018225E-01_dp, -3.9385951905586181E-01_dp,&
      2.0432223719801115E-01_dp,  1.6924599740654714E-01_dp,&
      -3.2820891989357739E-02_dp, -2.8096972471170906E-01_dp,&
      3.1382191127514508E-01_dp, -1.4536535595852404E-01_dp,&
      2.6668413841726423E-02_dp,  9.1549135569876245E+00_dp,&
      -8.5345664784207340E+00_dp,  2.2346205651622792E+00_dp,&
      1.6411516718069554E+00_dp, -9.8335661502361238E-01_dp,&
      -6.0422667740241987E-01_dp,  5.0601416326899085E-01_dp,&
      2.6377746002222652E-01_dp, -1.9726347500816235E-01_dp,&
      -3.7523198569027921E-01_dp,  5.3625497856328741E-01_dp,&
      -2.7599511569361151E-01_dp,  5.4528659506659791E-02_dp,&
      1.3943035992310987E+01_dp, -1.5109323197529408E+01_dp,&
      5.2959149276944357E+00_dp,  2.4661972731333157E+00_dp,&
      -2.2567333201120254E+00_dp, -7.7787392178326420E-01_dp,&
      1.1214179222198557E+00_dp,  3.4190618100067971E-01_dp,&
      -6.1152850879396203E-01_dp, -3.2726172380330121E-01_dp,&
      7.9909066319447652E-01_dp, -4.7152537410281026E-01_dp,&
      1.0076309501925237E-01_dp,  2.1388611942642935E+01_dp,&
      -2.6637185765365800E+01_dp,  1.1824571495957908E+01_dp,&
      3.1834095908537434E+00_dp, -4.8063853776563112E+00_dp,&
      -6.3222648705256546E-01_dp,  2.2812373088061939E+00_dp,&
      2.7053699574642753E-01_dp, -1.5164506840508636E+00_dp,&
      1.9390080320187825E-01_dp,  9.3774412092831516E-01_dp,&
      -7.0197405735123353E-01_dp,  1.6622614771280861E-01_dp,&
      3.3079428604239290E+01_dp, -4.6898194866003884E+01_dp,&
      2.5464018502546349E+01_dp,  2.7484480884475584E+00_dp,&
      -9.6279273060536603E+00_dp,  5.9552677798513798E-01_dp,&
      4.2759743597723183E+00_dp, -3.0699576070505968E-01_dp,&
      -3.2669097676275891E+00_dp,  1.8987807572689366E+00_dp,&
      4.8132579922105273E-01_dp, -8.2065702179074362E-01_dp,&
      2.3362433782091457E-01_dp,  5.1621626508739872E+01_dp,&
      -8.2660353947017612E+01_dp,  5.3607937311050435E+01_dp,&
      -1.8797644714387658E+00_dp, -1.8143027819974449E+01_dp,&
      4.7369680178784570E+00_dp,  7.2692132436006371E+00_dp,&
      -2.2184482777750349E+00_dp, -6.2154142523989133E+00_dp,&
      6.1059357543242241E+00_dp, -1.6522745114655870E+00_dp,&
      -4.1115789631675914E-01_dp,  2.3756928000969607E-01_dp,&
      8.1351582550594543E+01_dp, -1.4615422101218340E+02_dp,&
      1.1129430956108159E+02_dp, -1.8800774059904494E+01_dp,&
      -3.1684177587873151E+01_dp,  1.5820010337807913E+01_dp,&
      1.0616280037682005E+01_dp, -7.1547281508105547E+00_dp,&
      -1.0210813394710581E+01_dp,  1.4851459426239472E+01_dp,&
      -7.5823159284462474E+00_dp,  1.4755007442866093E+00_dp,&
      7.1704854962532146E-03_dp,  1.2961216352300849E+02_dp,&
      -2.5972094896967849E+02_dp,  2.2922833292335898E+02_dp,&
      -6.8618818175144867E+01_dp, -4.8903263939216217E+01_dp,&
      4.1885218071518977E+01_dp,  1.0862853016310472E+01_dp,&
      -1.8022963902540312E+01_dp, -1.3119830070208033E+01_dp,&
      3.0238247725930801E+01_dp, -2.0742505936307470E+01_dp,&
      6.6612181005611406E+00_dp, -8.2499200913490378E-01_dp,&
      2.0918002109462682E+02_dp, -4.6477061154654319E+02_dp,&
      4.7040392223308885E+02_dp, -2.0231664523073340E+02_dp,&
      -5.6353025630719443E+01_dp,  9.6488871901532022E+01_dp,&
      -3.3153170467196271E+00_dp, -3.8112379830274591E+01_dp,&
      -7.2773894428920212E+00_dp,  5.0927199428565473E+01_dp,&
      -4.4589076648414782E+01_dp,  1.7786229263712446E+01_dp,&
      -2.8840133398763834E+00_dp,  3.4348422405496660E+02_dp,&
      -8.4023075962910389E+02_dp,  9.6509249025561269E+02_dp,&
      -5.4322732013402708E+02_dp,  1.1614085203846807E+00_dp,&
      1.9306947976062503E+02_dp, -6.0953011969071319E+01_dp,&
      -6.4473641869529004E+01_dp,  2.5599475521189774E+01_dp,&
      6.1661793378629682E+01_dp, -7.5299742803616596E+01_dp,&
      3.5784746893679127E+01_dp, -6.6920352740423770E+00_dp,&
      5.8082030750243996E+02_dp, -1.5482290606357024E+03_dp,&
      1.9913534204232064E+03_dp, -1.3842027389980281E+03_dp,&
      3.0176915978714999E+02_dp,  3.0701230954874183E+02_dp,&
      -2.1433469154899214E+02_dp, -6.6515928662840182E+01_dp,&
      1.0587060354314622E+02_dp,  2.4158616076495992E+01_dp,&
      -8.8819422206168397E+01_dp,  5.3284774844267872E+01_dp,&
      -1.1352258425217249E+01_dp,  1.0507824282448173E+03_dp,&
      -3.0034485650910306E+03_dp,  4.2378849656165157E+03_dp,&
      -3.4559382913401400E+03_dp,  1.3449288846420820E+03_dp,&
      2.6444208061663255E+02_dp, -4.9877900967018985E+02_dp,&
      2.5127991468850134E+01_dp,  2.1767335090020083E+02_dp,&
      -8.6072216194219422E+01_dp, -6.0325796108173265E+01_dp,&
      6.0083116119914528E+01_dp, -1.5156462785073193E+01_dp,&
      2.3717539683229165E+03_dp, -7.0993078298523542E+03_dp,&
      1.0652966496878691E+04_dp, -9.6140748259170232E+03_dp,&
      4.8504224379382586E+03_dp, -4.4395885474091079E+02_dp,&
      -9.6288919502292026E+02_dp,  2.5917249174048044E+02_dp,&
      4.0854133722363883E+02_dp, -3.0143744269409973E+02_dp,&
      -4.5418714874973256E+00_dp,  7.5623186578469742E+01_dp,&
      -2.3436275595093466E+01_dp,  1.4977112766698923E+00_dp,&
      -4.3677613112273472E-01_dp, -1.0626472632002160E-01_dp,&
      6.0579861136460085E-02_dp,  3.2094769702270376E-02_dp,&
      -2.2140138450254837E-02_dp, -1.3621095148298569E-02_dp,&
      7.9037816732341545E-03_dp,  1.4423141126330896E-02_dp,&
      -1.6738731357106910E-02_dp,  5.6195012107057513E-03_dp,&
      5.0623354758645299E-04_dp, -5.2923485963463817E-04_dp,&
      1.7217160143503936E+00_dp, -6.3675005873485835E-01_dp,&
      -9.4151747966956217E-02_dp,  1.1116729873281835E-01_dp,&
      2.8800708837475177E-02_dp, -4.3580539400664406E-02_dp,&
      -1.1821195607030708E-02_dp,  1.6681071628927815E-02_dp,&
      1.9836760036767896E-02_dp, -3.4063478523860956E-02_dp,&
      1.9103471265017376E-02_dp, -4.3249027191962740E-03_dp,&
      1.5532200025133099E-04_dp,  2.2127628736646106E+00_dp,&
      -1.1034935942153976E+00_dp, -4.3577466272544045E-02_dp,&
      2.2803871592728966E-01_dp,  1.0104708726025797E-02_dp,&
      -9.3215357617339387E-02_dp, -1.8053387009684143E-03_dp,&
      3.7448006485105621E-02_dp,  2.8302008106949928E-02_dp,&
      -7.3968859165537221E-02_dp,  5.3453226207458779E-02_dp,&
      -1.7780936779581476E-02_dp,  2.2736357196637734E-03_dp,&
      3.0642372558529560E+00_dp, -1.9876242013174268E+00_dp,&
      1.1168757098085445E-01_dp,  4.4279774497587876E-01_dp,&
      -5.4403602466477574E-02_dp, -1.8361663846171278E-01_dp,&
      3.2420048824757658E-02_dp,  7.6305200794048844E-02_dp,&
      3.2103644959620231E-02_dp, -1.4486824002301485E-01_dp,&
      1.2355203927413878E-01_dp, -4.8092052541477776E-02_dp,&
      7.5259001655546970E-03_dp,  4.4383894095260246E+00_dp,&
      -3.5679718788094252E+00_dp,  5.1476908985660208E-01_dp,&
      8.0172648142794001E-01_dp, -2.2976447620038551E-01_dp,&
      -3.2956771002295171E-01_dp,  1.2448866609398915E-01_dp /)
    REAL(KIND=dp), DIMENSION(400), PARAMETER :: c03 = (/ &
      1.4079316176694362E-01_dp,  1.0204817722398823E-02_dp,&
      -2.5165636141352937E-01_dp,  2.5091425850523480E-01_dp,&
      -1.0907652615992168E-01_dp,  1.8995561214566335E-02_dp,&
      6.6017064494743369E+00_dp, -6.3396206574849705E+00_dp,&
      1.4645915581906761E+00_dp,  1.3567138102239846E+00_dp,&
      -6.4776462885481734E-01_dp, -5.3664210086520570E-01_dp,&
      3.4037092684481507E-01_dp,  2.3476223625427250E-01_dp,&
      -8.7556632295100814E-02_dp, -3.7830556673362864E-01_dp,&
      4.5738992125084094E-01_dp, -2.2026577794379970E-01_dp,&
      4.1599788241118318E-02_dp,  9.9858326732371161E+00_dp,&
      -1.1174218555192047E+01_dp,  3.5747006721239503E+00_dp,&
      2.1296809807914152E+00_dp, -1.5636677908078949E+00_dp,&
      -7.6880577851663401E-01_dp,  8.0089159003032162E-01_dp,&
      3.4298150442274511E-01_dp, -3.6805510673665137E-01_dp,&
      -4.4644114872399454E-01_dp,  7.4222316810942002E-01_dp,&
      -4.0152913150399111E-01_dp,  8.1829151439933356E-02_dp,&
      1.5289854701740817E+01_dp, -1.9608078427701756E+01_dp,&
      8.0917990845543581E+00_dp,  3.0041832124620726E+00_dp,&
      -3.4494990735255384E+00_dp, -8.6609625198945162E-01_dp,&
      1.7095858524603544E+00_dp,  3.9120562018981797E-01_dp,&
      -1.0360932314069153E+00_dp, -2.2758467416281059E-01_dp,&
      1.0203793428499848E+00_dp, -6.5239517137454839E-01_dp,&
      1.4501295283759075E-01_dp,  2.3652443598918335E+01_dp,&
      -3.4362906201199991E+01_dp,  1.7536976813856338E+01_dp,&
      3.4319229466110013E+00_dp, -7.1298251478464278E+00_dp,&
      -3.5204255806543472E-01_dp,  3.3680128935599196E+00_dp,&
      1.5319402163742263E-01_dp, -2.4370830597624149E+00_dp,&
      8.1997373400602092E-01_dp,  9.8380055990672011E-01_dp,&
      -8.9662392912364008E-01_dp,  2.2625667920011164E-01_dp,&
      3.6944557452608692E+01_dp, -6.0295621236170625E+01_dp,&
      3.7003398215145729E+01_dp,  1.6834081679861252E+00_dp,&
      -1.3925568490337929E+01_dp,  1.9888475074288903E+00_dp,&
      6.1139267076108847E+00_dp, -9.4588148977566311E-01_dp,&
      -5.0444398561656572E+00_dp,  3.8065719195626566E+00_dp,&
      -1.6831147750489744E-01_dp, -8.5900446249491913E-01_dp,&
      2.8780543981072360E-01_dp,  5.8274202677422771E+01_dp,&
      -1.0614856307111749E+02_dp,  7.6800301489740932E+01_dp,&
      -7.0079594145933406E+00_dp, -2.5617180005330020E+01_dp,&
      8.9815644644679367E+00_dp,  9.9914118876838458E+00_dp,&
      -4.1991995337660075E+00_dp, -9.2225161080668876E+00_dp,&
      1.0699436467194234E+01_dp, -4.1744895657292922E+00_dp,&
      1.6968874745604984E-01_dp,  2.1150225597430636E-01_dp,&
      9.2874008697647312E+01_dp, -1.8781074240763044E+02_dp,&
      1.5788030861616468E+02_dp, -3.4972432415418027E+01_dp,&
      -4.3527351698049202E+01_dp,  2.6710127846672595E+01_dp,&
      1.3641876922340176E+01_dp, -1.2188594976459228E+01_dp,&
      -1.4352260173596875E+01_dp,  2.4342038573890058E+01_dp,&
      -1.4279132767898336E+01_dp,  3.7137073675422725E+00_dp,&
      -2.8757236845263162E-01_dp,  1.4971128490394287E+02_dp,&
      -3.3447762227306481E+02_dp,  3.2305460093335603E+02_dp,&
      -1.1305851039826165E+02_dp, -6.4420176838297635E+01_dp,&
      6.7169772276610786E+01_dp,  1.1142459865038420E+01_dp,&
      -2.9260375212110183E+01_dp, -1.6413218989655853E+01_dp,&
      4.7209519021715842E+01_dp, -3.5494808693208000E+01_dp,&
      1.2580735551183873E+01_dp, -1.7935944649833950E+00_dp,&
      2.4458399494694996E+02_dp, -6.0055061871622843E+02_dp,&
      6.6027498087928825E+02_dp, -3.1693333009014430E+02_dp,&
      -6.6059668231172040E+01_dp,  1.5055616810459574E+02_dp,&
      -1.6035040303862690E+01_dp, -6.0205525165442332E+01_dp,&
      -2.1832622908315549E+00_dp,  7.5674463602179827E+01_dp,&
      -7.2506243056554936E+01_dp,  3.0788078901881761E+01_dp,&
      -5.2942708588931886E+00_dp,  4.0681467176552570E+02_dp,&
      -1.0899373555775655E+03_dp,  1.3513251866930286E+03_dp,&
      -8.2903784544199414E+02_dp,  3.8380965916357390E+01_dp,&
      2.9721645411143447E+02_dp, -1.1594319332969438E+02_dp,&
      -9.9972427583413506E+01_dp,  6.1208430774721975E+01_dp,&
      8.0856974870358002E+01_dp, -1.1693883280777918E+02_dp,&
      5.9389649528398166E+01_dp, -1.1628989395474973E+01_dp,&
      6.9722251479747388E+02_dp, -2.0140066190150276E+03_dp,&
      2.7790283348654434E+03_dp, -2.0796098972388722E+03_dp,&
      5.2869394312949373E+02_dp,  4.6652987556767943E+02_dp,&
      -3.8221920854190296E+02_dp, -9.2752399207456079E+01_dp,&
      2.1910131922818687E+02_dp, -2.2923675949907071E+01_dp,&
      -1.1207458200684393E+02_dp,  7.9761888545321156E+01_dp,&
      -1.8298506671237661E+01_dp,  1.2744662066142864E+03_dp,&
      -3.8866078972090563E+03_dp,  5.8310452947172871E+03_dp,&
      -5.0818244125139472E+03_dp,  2.1788811500495763E+03_dp,&
      3.4081840675497295E+02_dp, -8.7556575958971519E+02_dp,&
      1.2965937398586175E+02_dp,  4.1054055060278523E+02_dp,&
      -2.9796372200111415E+02_dp,  2.6343327262803022E+01_dp,&
      5.0424032533223901E+01_dp, -1.7235143944934833E+01_dp,&
      2.7604064480154507E+03_dp, -8.6019432779733779E+03_dp,&
      1.3474832660141481E+04_dp, -1.2816761276941676E+04_dp,&
      7.0149847866566979E+03_dp, -9.9103658229985535E+02_dp,&
      -1.3749290812180745E+03_dp,  6.2574740788577901E+02_dp,&
      4.7802482876747689E+02_dp, -5.7305069900934677E+02_dp,&
      1.8314783781950712E+02_dp,  1.5464936084215722E+01_dp,&
      -1.5810901025213084E+01_dp,  1.0274044897416868E+04_dp,&
      -3.1659430536269701E+04_dp,  4.9322572161259443E+04_dp,&
      -4.7163561469793902E+04_dp,  2.6891304291869375E+04_dp,&
      -5.7571317404894771E+03_dp, -3.0444779122812329E+03_dp,&
      1.4450586928263419E+03_dp,  1.4829738315237773E+03_dp,&
      -1.4849083905506739E+03_dp,  2.8382264810325569E+02_dp,&
      1.7516211991683696E+02_dp, -7.2468749687854157E+01_dp,&
      2.1318222830584657E-01_dp, -3.3684034054268112E-02_dp,&
      -1.0832627512700675E-02_dp,  3.3145187729886286E-03_dp,&
      2.7275281503664904E-03_dp, -9.9837098618575626E-04_dp,&
      -1.0357840754560805E-03_dp,  3.5875531553344507E-04_dp,&
      6.0259734691266913E-04_dp, -3.5423446604539031E-04_dp,&
      -1.7556949097338197E-04_dp,  2.2034591354691384E-04_dp,&
      -6.0116378539179255E-05_dp,  6.8243285676064791E-01_dp,&
      -1.2222900203139513E-01_dp, -3.5650354344508521E-02_dp,&
      1.4124232653500343E-02_dp,  9.3169904587289860E-03_dp,&
      -4.4428123313210782E-03_dp, -3.5817403682234456E-03_dp,&
      1.6213661243337648E-03_dp,  2.2617072347383576E-03_dp,&
      -1.7251617279990248E-03_dp, -3.0117239462936477E-04_dp,&
      6.9071851016345640E-04_dp, -2.0557421403978617E-04_dp,&
      1.2918736815826171E+00_dp, -2.8431849514217356E-01_dp,&
      -6.9503682880585616E-02_dp,  4.0079778234935019E-02_dp,&
      1.9169899730441080E-02_dp, -1.3254501345108947E-02_dp,&
      -7.4879325459152992E-03_dp,  4.9273958248561584E-03_dp,&
      5.4431528418503244E-03_dp, -5.6527461842056205E-03_dp,&
      6.4400303030163011E-04_dp,  1.1358340960384338E-03_dp,&
      -4.1629994324220604E-04_dp,  2.1761841600359864E+00_dp,&
      -6.0602677049954834E-01_dp, -1.1600614333211007E-01_dp,&
      1.0094374685942331E-01_dp,  3.3524435971905575E-02_dp,&
      -3.4830286515735560E-02_dp, -1.3217120228077264E-02_dp,&
      1.3167612202744752E-02_dp,  1.1658344857093769E-02_dp,&
      -1.6002826381700426E-02_dp,  4.9726471567697389E-03_dp,&
      1.0322574937840596E-03_dp, -6.7937956240013083E-04_dp,&
      3.5436318724159870E+00_dp, -1.2448897785654798E+00_dp,&
      -1.7047825329078076E-01_dp,  2.3516821229813373E-01_dp,&
      4.9997768154929345E-02_dp, -8.3991468599014865E-02_dp,&
      -1.9461645231788477E-02_dp,  3.2227076132211548E-02_dp,&
      2.2927327872461805E-02_dp, -4.0981673652954999E-02_dp,&
      1.8645143724385239E-02_dp, -1.4710909163262332E-03_dp,&
      -7.9628203284441701E-04_dp,  5.7352535342850013E+00_dp,&
      -2.5003246419581067E+00_dp, -2.0277450208455863E-01_dp,&
      5.1788148421079194E-01_dp,  5.4494531427097280E-02_dp,&
      -1.9004943455969087E-01_dp, -1.9106874087512889E-02_dp,&
      7.3855738319935568E-02_dp,  4.1005914394026836E-02_dp,&
      -9.7462681711003807E-02_dp,  5.6025894780165515E-02_dp,&
      -1.1671638371992546E-02_dp, -1.2317531086698035E-05_dp,&
      9.3283234643961990E+00_dp, -4.9517090926784189E+00_dp,&
      -1.1237619085161876E-01_dp,  1.0938292009097146E+00_dp,&
      -3.6578939270913327E-04_dp, -4.0903466934184857E-01_dp,&
      1.1571092173393985E-02_dp,  1.6054688245457299E-01_dp,&
      6.3796913867728455E-02_dp, -2.1826491336856391E-01_dp,&
      1.4979809357033552E-01_dp, -4.3231775436116136E-02_dp,&
      3.9227329925919244E-03_dp,  1.5322921886299406E+01_dp,&
      -9.7350236118409814E+00_dp,  3.8501066580685789E-01_dp,&
      2.2354405913551587E+00_dp, -2.5000970304965098E-01_dp,&
      -8.4251562237350353E-01_dp,  1.4017068692381385E-01_dp,&
      3.3257684318281500E-01_dp,  7.3319727667607221E-02_dp,&
      -4.6190571472728487E-01_dp,  3.7010532372901361E-01_dp,&
      -1.2889503527555926E-01_dp,  1.6973488867294845E-02_dp,&
      2.5485983352834001E+01_dp, -1.9107468063951504E+01_dp,&
      2.0402311845689449E+00_dp,  4.4371579701624126E+00_dp,&
      -1.0529422595840883E+00_dp, -1.6585347909210049E+00_dp,&
      5.4410632305326356E-01_dp,  6.5376504315884909E-01_dp,&
      4.1661500351891439E-04_dp, -9.1692979997264157E-01_dp,&
      8.5690261662255818E-01_dp, -3.4197738694486834E-01_dp,&
      5.3687965659661094E-02_dp,  4.2999540010684285E+01_dp,&
      -3.7622665990809367E+01_dp,  6.7752865496365642E+00_dp,&
      8.5406611669389410E+00_dp, -3.3166201661466737E+00_dp,&
      -3.0842874201654951E+00_dp,  1.6635414194652112E+00_dp,&
      1.1980859949014355E+00_dp, -3.7016765925282619E-01_dp,&
      -1.6659214344665423E+00_dp,  1.8593601546633085E+00_dp,&
      -8.3381793576913044E-01_dp,  1.4684476667934723E-01_dp,&
      7.3709904847928598E+01_dp, -7.4630438320318333E+01_dp,&
      1.9452643501026422E+01_dp,  1.5793696701925368E+01_dp,&
      -9.2675373118737916E+00_dp, -5.2467842188148071E+00_dp,&
      4.5372916469923439E+00_dp,  1.9496492078416041E+00_dp,&
      -1.6349099319243841E+00_dp, -2.5906847912536093E+00_dp,&
      3.7265500877505593E+00_dp, -1.8799923390423963E+00_dp,&
      3.6327549283233479E-01_dp,  1.2859438624321660E+02_dp,&
      -1.4972520267351325E+02_dp,  5.2383826750269051E+01_dp,&
      2.7367488598061911E+01_dp, -2.4219671211959398E+01_dp,&
      -7.4290230400283290E+00_dp,  1.1475384287478459E+01_dp,&
      2.3922423306082199E+00_dp, -5.2921831798037999E+00_dp,&
      -2.6730364381610836E+00_dp,  6.6169572702353063E+00_dp,&
      -3.8645306885918953E+00_dp,  8.1818488592343996E-01_dp,&
      2.2876608607133824E+02_dp, -3.0495185259837291E+02_dp,&
      1.3706237077883176E+02_dp,  4.1439813867536394E+01_dp,&
      -6.0428830848804978E+01_dp, -5.3399496761646414E+00_dp,&
      2.7145380315073801E+01_dp,  1.0963663429032429E-01_dp,&
      -1.4667285308366607E+01_dp,  2.0716234645629714E+00_dp,&
      9.1721609250763034E+00_dp, -6.9025941825134423E+00_dp,&
      1.6398955739095382E+00_dp,  4.1594691714391070E+02_dp,&
      -6.3301093600335355E+02_dp,  3.5553694173797044E+02_dp,&
      4.0753399602504416E+01_dp, -1.4452338056841705E+02_dp,&
      1.7908795314096562E+01_dp,  5.9142120003589028E+01_dp,&
      -1.3457064524764537E+01_dp, -3.5600526788255479E+01_dp,&
      2.3692561370282714E+01_dp,  4.1369265076592301E+00_dp,&
      -9.1363336250763520E+00_dp,  2.7024750645260252E+00_dp,&
      7.7521438841071472E+02_dp, -1.3447541987663474E+03_dp,&
      9.2710345217552867E+02_dp, -5.5662691197356949E+01_dp,&
      -3.2666994854804346E+02_dp,  1.1949426674679283E+02_dp,&
      1.1148985958051878E+02_dp, -6.4189392858052045E+01_dp,&
      -7.2080786102618873E+01_dp,  9.1605160930916881E+01_dp,&
      -3.3190298346214078E+01_dp, -1.1968423393425880E+00_dp,&
      2.5799094822024733E+00_dp,  1.4870387159310956E+03_dp,&
      -2.9373961910368489E+03_dp,  2.4568455127217608E+03_dp,&
      -5.7572036132261246E+02_dp, -6.5915764219002381E+02_dp,&
      4.7301818737714882E+02_dp,  1.3863635256023798E+02_dp,&
      -2.1631236718932931E+02_dp, -9.4803921196274686E+01_dp,&
      2.5426811844807858E+02_dp, -1.6576110763951490E+02_dp,&
      4.7407189814529652E+01_dp, -4.5099294020295337E+00_dp,&
      2.9586882462973167E+03_dp, -6.6390589153338169E+03_dp,&
      6.6755402509804217E+03_dp, -2.7894432451376142E+03_dp /)
    REAL(KIND=dp), DIMENSION(400), PARAMETER :: c04 = (/ &
      -9.1855299569510419E+02_dp,  1.5086598025400613E+03_dp,&
      -1.6982251398122094E+02_dp, -5.5006675710728985E+02_dp,&
      8.6766882152744344E+01_dp,  4.7256010495026828E+02_dp,&
      -4.6889366551390941E+02_dp,  1.9581559031410015E+02_dp,&
      -3.2716661828879559E+01_dp,  6.2548032409521911E+03_dp,&
      -1.5788950870448605E+04_dp,  1.8797183431156966E+04_dp,&
      -1.1500723521848280E+04_dp,  1.1311068823604562E+03_dp,&
      3.5696653498968658E+03_dp, -1.9548307992473563E+03_dp,&
      -6.5409377594636578E+02_dp,  8.8415097294428301E+02_dp,&
      2.0828579656796524E+02_dp, -6.8559158021726762E+02_dp,&
      3.9487798673853291E+02_dp, -8.1813494801799735E+01_dp,&
      1.5900659171374931E+04_dp, -4.3865264886054203E+04_dp,&
      5.9368981473015963E+04_dp, -4.6496642597520760E+04_dp,&
      1.7576030674673242E+04_dp,  3.1090094533818497E+03_dp,&
      -6.2551252886095626E+03_dp,  1.0254795734419797E+03_dp,&
      1.7700143703154242E+03_dp, -6.9300411007170703E+02_dp,&
      -5.9600061313823699E+02_dp,  5.5645136364522648E+02_dp,&
      -1.3789134413973542E+02_dp,  1.7333022186632598E+00_dp,&
      -2.8295145016009599E-01_dp, -8.8783345881969303E-02_dp,&
      2.9137501520617400E-02_dp,  2.2582343344786884E-02_dp,&
      -8.8870496740718532E-03_dp, -8.6050069441200982E-03_dp,&
      3.2075742114612876E-03_dp,  5.1124562607267323E-03_dp,&
      -3.2396586093912825E-03_dp, -1.2782830098815570E-03_dp,&
      1.7892266110778137E-03_dp, -4.9828059483079956E-04_dp,&
      2.0839177842079417E+00_dp, -4.5883342975975994E-01_dp,&
      -1.1377653794136394E-01_dp,  6.4262118792829045E-02_dp,&
      3.1589922005670561E-02_dp, -2.1128047919275923E-02_dp,&
      -1.2370808453459683E-02_dp,  7.8307499965700952E-03_dp,&
      8.8525819512241723E-03_dp, -8.9036430575215944E-03_dp,&
      7.7278653207137722E-04_dp,  1.9574113869564288E-03_dp,&
      -6.9432538273257499E-04_dp,  2.8790965126363099E+00_dp,&
      -8.9291494000038507E-01_dp, -1.5980217852948253E-01_dp,&
      1.5593871145279911E-01_dp,  4.8212590796327996E-02_dp,&
      -5.4049164283410382E-02_dp, -1.9262868395269681E-02_dp,&
      2.0444212857397715E-02_dp,  1.7597567867215835E-02_dp,&
      -2.4946097804552481E-02_dp,  8.1983007050586126E-03_dp,&
      1.3052693431584528E-03_dp, -9.9046233632061080E-04_dp,&
      4.3304768482591722E+00_dp, -1.7813853808128957E+00_dp,&
      -2.1010494972641455E-01_dp,  3.5547178388532347E-01_dp,&
      6.5506905553492881E-02_dp, -1.2798724844398165E-01_dp,&
      -2.5933808745545982E-02_dp,  4.9217194247932812E-02_dp,&
      3.3506489973277181E-02_dp, -6.3047933152743540E-02_dp,&
      3.0110932740404765E-02_dp, -3.2402195006371740E-03_dp,&
      -1.0036785562597503E-03_dp,  6.8299081160337032E+00_dp,&
      -3.5205397689128302E+00_dp, -2.0822929848963964E-01_dp,&
      7.6631150098260470E-01_dp,  5.8155387220312556E-02_dp,&
      -2.8404749249030603E-01_dp, -1.9917085295170896E-02_dp,&
      1.1077158299302442E-01_dp,  5.8009365155884812E-02_dp,&
      -1.4754924485395102E-01_dp,  8.8224148397498189E-02_dp,&
      -2.0017588821696984E-02_dp,  5.1524515103809545E-04_dp,&
      1.1076132539365005E+01_dp, -6.8920385138882194E+00_dp,&
      1.0149059418049949E-02_dp,  1.5881453532787797E+00_dp,&
      -5.0514917369298917E-02_dp, -6.0049701223304608E-01_dp,&
      3.6944486690867563E-02_dp,  2.3676450668964813E-01_dp,&
      8.5900161567404182E-02_dp, -3.2547794450289452E-01_dp,&
      2.3138356568902774E-01_dp, -7.0029009427246630E-02_dp,&
      7.1213088371686710E-03_dp,  1.8313497271878418E+01_dp,&
      -1.3441485775194154E+01_dp,  8.8580011501243916E-01_dp,&
      3.1948031252200022E+00_dp, -4.7058826034590084E-01_dp,&
      -1.2174961672232065E+00_dp,  2.4957240968045155E-01_dp,&
      4.8310590222314870E-01_dp,  8.6483956963456024E-02_dp,&
      -6.7940646376194236E-01_dp,  5.6338924998429407E-01_dp,&
      -2.0291123162723659E-01_dp,  2.8032368883630622E-02_dp,&
      3.0780617013122821E+01_dp, -2.6257569697681561E+01_dp,&
      3.5521591798072638E+00_dp,  6.2587792976548826E+00_dp,&
      -1.7460143870796279E+00_dp, -2.3619614892636438E+00_dp,&
      8.8818001701109950E-01_dp,  9.3606848206238136E-01_dp,&
      -5.0794860926712801E-02_dp, -1.3302240326510510E+00_dp,&
      1.2893794094134079E+00_dp, -5.2876901741570692E-01_dp,&
      8.5451865781826092E-02_dp,  5.2567287396157639E+01_dp,&
      -5.1604816743560406E+01_dp,  1.0866611400666184E+01_dp,&
      1.1908868775950642E+01_dp, -5.2359341711279104E+00_dp,&
      -4.3250063052196532E+00_dp,  2.6159067633220250E+00_dp,&
      1.6876932949921251E+00_dp, -6.6168203286073823E-01_dp,&
      -2.3751835992262973E+00_dp,  2.7700021611620960E+00_dp,&
      -1.2732445592684205E+00_dp,  2.2902755062897792E-01_dp,&
      9.1285593514466271E+01_dp, -1.0241866626620062E+02_dp,&
      2.9991030801919234E+01_dp,  2.1769589255916912E+01_dp,&
      -1.4249979282246676E+01_dp, -7.2044673782332227E+00_dp,&
      6.9889285697343118E+00_dp,  2.6782377021603785E+00_dp,&
      -2.6677863139577163E+00_dp, -3.5770528758936435E+00_dp,&
      5.4957103661007007E+00_dp, -2.8440183689455196E+00_dp,&
      5.5949615778311945E-01_dp,  1.6140675350929016E+02_dp,&
      -2.0596934838864317E+02_dp,  7.8952124827283598E+01_dp,&
      3.7178752267348052E+01_dp, -3.6659944916518448E+01_dp,&
      -9.7656735399369428E+00_dp,  1.7464710569932944E+01_dp,&
      3.0679469902932719E+00_dp, -8.3851627852850044E+00_dp,&
      -3.2636862826796440E+00_dp,  9.6185791182519313E+00_dp,&
      -5.7987749350325295E+00_dp,  1.2497007872223715E+00_dp,&
      2.9110313668489431E+02_dp, -4.2112906631689452E+02_dp,&
      2.0368167006668779E+02_dp,  5.4837106594561327E+01_dp,&
      -9.0638707836013396E+01_dp, -5.3325509510817621E+00_dp,&
      4.1073827539931052E+01_dp, -8.9364940761646350E-01_dp,&
      -2.3013798898787297E+01_dp,  4.9430590435737045E+00_dp,&
      1.2851186457902051E+01_dp, -1.0246075994503601E+01_dp,&
      2.4884401830064262E+00_dp,  5.3670606902267980E+02_dp,&
      -8.7847920003528930E+02_dp,  5.2349235503272166E+02_dp,&
      4.8749806265367148E+01_dp, -2.1602127708232794E+02_dp,&
      3.1729321388441790E+01_dp,  8.9567868142842315E+01_dp,&
      -2.2390677578649012E+01_dp, -5.6017285780368070E+01_dp,&
      4.0387845593738454E+01_dp,  3.4667266516341293E+00_dp,&
      -1.3141255438243856E+01_dp,  4.0592288247989385E+00_dp,&
      1.0143591938132444E+03_dp, -1.8766993856001438E+03_dp,&
      1.3564571841587124E+03_dp, -9.8696563163335753E+01_dp,&
      -4.9033312510266632E+02_dp,  1.8780383859095875E+02_dp,&
      1.7118665610355563E+02_dp, -1.0185617481240935E+02_dp,&
      -1.1558713439161205E+02_dp,  1.5151598909811219E+02_dp,&
      -5.8664397974976858E+01_dp,  6.3267833198642698E-01_dp,&
      3.6766915824367761E+00_dp,  1.9726076865000123E+03_dp,&
      -4.1232818836429924E+03_dp,  3.5781197436471589E+03_dp,&
      -8.5509074775503893E+02_dp, -1.0123738678655812E+03_dp,&
      7.2969707150714839E+02_dp,  2.2828683121334075E+02_dp,&
      -3.4563856543415034E+02_dp, -1.6400405779415590E+02_dp,&
      4.2769450363111764E+02_dp, -2.8094753019966493E+02_dp,&
      8.1695817906598137E+01_dp, -8.0867822790318300E+00_dp,&
      3.9722127523258755E+03_dp, -9.3649615551361039E+03_dp,&
      9.6872702091042847E+03_dp, -4.0152449268719452E+03_dp,&
      -1.5781648063622556E+03_dp,  2.3817881583693224E+03_dp,&
      -1.7512622509953385E+02_dp, -9.4466208919082828E+02_dp,&
      8.8832120896167638E+01_dp,  8.8054897643088452E+02_dp,&
      -8.4980315712561492E+02_dp,  3.5239533036204887E+02_dp,&
      -5.8700991234149221E+01_dp,  8.4183972987919678E+03_dp,&
      -2.2200192543210582E+04_dp,  2.7095512964836042E+04_dp,&
      -1.6398496436394067E+04_dp,  5.8588432027345868E+02_dp,&
      6.3339571905285420E+03_dp, -3.0333212281281785E+03_dp,&
      -1.5835275583468535E+03_dp,  1.7348972889932504E+03_dp,&
      5.0425398982402010E+02_dp, -1.4466434445749767E+03_dp,&
      8.3093046218870779E+02_dp, -1.7343507171721367E+02_dp,&
      2.0025459522044373E+04_dp, -5.7346919835612738E+04_dp,&
      7.9917525300310852E+04_dp, -6.3405260767018226E+04_dp,&
      2.2238895649329628E+04_dp,  8.4397777042427315E+03_dp,&
      -1.2412923298499947E+04_dp,  2.2353310629429734E+03_dp,&
      4.1643117321199643E+03_dp, -2.9557407553812504E+03_dp,&
      1.6207931060663054E+02_dp,  5.4977314747273965E+02_dp,&
      -1.7928307380491745E+02_dp,  7.4440121917214812E+04_dp,&
      -2.1677815388585156E+05_dp,  3.1577001931789052E+05_dp,&
      -2.7930028207177343E+05_dp,  1.4320805176019639E+05_dp,&
      -2.1245008317147029E+04_dp, -2.1554991554281445E+04_dp,&
      9.9376537143063269E+03_dp,  4.6744532080188510E+03_dp,&
      -4.2905179066964693E+03_dp, -5.3149376874680104E+02_dp,&
      1.4881154495908686E+03_dp, -4.3984467711458336E+02_dp,&
      2.3404090378356895E-01_dp, -2.0097322397469076E-02_dp,&
      -1.1609180582537834E-02_dp,  1.8601938852069603E-03_dp,&
      3.4063516337523563E-03_dp, -2.5328555239247899E-04_dp,&
      -2.1526388476503741E-03_dp,  2.6541196853022917E-04_dp,&
      1.8337856370389378E-03_dp, -1.4007221924521144E-03_dp,&
      1.1370777613700699E-04_dp,  2.7147984801578659E-04_dp,&
      -9.3281631667276296E-05_dp,  7.5958720353542519E-01_dp,&
      -7.5810202725780582E-02_dp, -4.2313502138352527E-02_dp,&
      8.6657522638846456E-03_dp,  1.2968885710289797E-02_dp,&
      -1.6608526759463512E-03_dp, -8.2851212755105860E-03_dp,&
      1.2914133195331631E-03_dp,  7.4052964778980055E-03_dp,&
      -6.1849568325369995E-03_dp,  1.0315421017034565E-03_dp,&
      8.3158039993379697E-04_dp, -3.2797209696527126E-04_dp,&
      1.4768419304919898E+00_dp, -1.8729413362838598E-01_dp,&
      -9.8945335892159453E-02_dp,  2.7515284560837457E-02_dp,&
      3.2275791153780403E-02_dp, -6.8395663023995482E-03_dp,&
      -2.0986896600458896E-02_dp,  4.3486105991930634E-03_dp,&
      2.0127441003510580E-02_dp, -1.8849674488696535E-02_dp,&
      5.0191511478448469E-03_dp,  1.2331926287606307E-03_dp,&
      -7.0207519810811295E-04_dp,  2.5840964671288975E+00_dp,&
      -4.2690659349976090E-01_dp, -2.1145308910161936E-01_dp,&
      7.7555603458720743E-02_dp,  7.3310657354651587E-02_dp,&
      -2.2709496661448086E-02_dp, -4.8700826546977563E-02_dp,&
      1.3059501608799757E-02_dp,  5.0389115213805714E-02_dp,&
      -5.2544419268517009E-02_dp,  1.8412947257736761E-02_dp,&
      3.6589716526811415E-04_dp, -1.2543984993995988E-03_dp,&
      4.4117567910870852E+00_dp, -9.3963835482824520E-01_dp,&
      -4.3338088445410117E-01_dp,  2.0292403047377275E-01_dp,&
      1.5835719985334940E-01_dp, -6.6721718153669785E-02_dp,&
      -1.0781515206665805E-01_dp,  3.6564985747604667E-02_dp,&
      1.2068898656418255E-01_dp, -1.3900826230358432E-01_dp,&
      5.8565521670857705E-02_dp, -5.8908104279455217E-03_dp,&
      -1.7476722084679196E-03_dp,  7.5445309724776202E+00_dp,&
      -2.0272400898868317E+00_dp, -8.6251312311759643E-01_dp,&
      5.0671870987293466E-01_dp,  3.2885263809384618E-01_dp,&
      -1.8271408651872414E-01_dp, -2.3038824323149212E-01_dp,&
      9.8194271811996164E-02_dp,  2.8055474423540522E-01_dp,&
      -3.5598861752761291E-01_dp,  1.7250528507673124E-01_dp,&
      -3.0809943938599049E-02_dp, -8.6747384125205782E-04_dp,&
      1.3055861531649315E+01_dp, -4.3314881266793979E+00_dp,&
      -1.6761083589215517E+00_dp,  1.2312731236949950E+00_dp,&
      6.5894439998792775E-01_dp, -4.8036396219622884E-01_dp,&
      -4.7776938854359963E-01_dp,  2.5796491007239708E-01_dp,&
      6.3870537313146425E-01_dp, -8.9484473757249905E-01_dp,&
      4.8676186523295428E-01_dp, -1.1467909217180164E-01_dp,&
      6.3856877573706490E-03_dp,  2.2973813103234260E+01_dp,&
      -9.2447576119387715E+00_dp, -3.1862366340950352E+00_dp,&
      2.9511614190933089E+00_dp,  1.2698033917738343E+00_dp,&
      -1.2342653176010880E+00_dp, -9.6141838186275863E-01_dp,&
      6.7220233082876679E-01_dp,  1.4302140919443622E+00_dp,&
      -2.2279767372545760E+00_dp,  1.3410180260314670E+00_dp,&
      -3.7636626450613619E-01_dp,  3.6869243870961514E-02_dp,&
      4.1235933833121898E+01_dp, -1.9856535292982361E+01_dp,&
      -5.9070868285783193E+00_dp,  7.0462234223485760E+00_dp,&
      2.3163307108458544E+00_dp, -3.1353286429629565E+00_dp,&
      -1.8613396596490337E+00_dp,  1.7553943424387961E+00_dp,&
      3.1455571650317942E+00_dp, -5.5222826187397756E+00_dp,&
      3.6478129213769028E+00_dp, -1.1619081276403027E+00_dp,&
      1.4446834803157857E-01_dp,  7.5710886366993023E+01_dp /)
    REAL(KIND=dp), DIMENSION(400), PARAMETER :: c05 = (/ &
      -4.3207068963149325E+01_dp, -1.0554104925448858E+01_dp,&
      1.6886407464491334E+01_dp,  3.8180534827516577E+00_dp,&
      -7.9351781128404237E+00_dp, -3.3784899123822600E+00_dp,&
      4.6319521704833901E+00_dp,  6.7239240450100590E+00_dp,&
      -1.3642372104076109E+01_dp,  9.8589344031369794E+00_dp,&
      -3.4720965735924705E+00_dp,  4.9693345251046339E-01_dp,&
      1.4263572197170834E+02_dp, -9.5857301698247568E+01_dp,&
      -1.7588925878417403E+01_dp,  4.0872193751092674E+01_dp,&
      4.8184722668567055E+00_dp, -2.0108942400948376E+01_dp,&
      -5.3282163675227920E+00_dp,  1.2435599975018286E+01_dp,&
      1.3556226217519358E+01_dp, -3.3461490386803618E+01_dp,&
      2.6523447839347572E+01_dp, -1.0184357103374717E+01_dp,&
      1.6085526053328969E+00_dp,  2.7676317157922375E+02_dp,&
      -2.1826938584162323E+02_dp, -2.4695849258246266E+01_dp,&
      1.0043798763217026E+02_dp, -1.3947867483132029E-01_dp,&
      -5.1127927754529111E+01_dp, -5.1560469151207808E+00_dp,&
      3.4156690568543311E+01_dp,  2.3554036850825067E+01_dp,&
      -8.0434522638194011E+01_dp,  7.0744011691971110E+01_dp,&
      -2.9459728084185446E+01_dp,  5.0323051191546622E+00_dp,&
      5.5569710557487417E+02_dp, -5.1390329950924320E+02_dp,&
      -1.5624537990859787E+01_dp,  2.5163004909295555E+02_dp,&
      -3.5139741515483045E+01_dp, -1.3006473245475482E+02_dp,&
      1.0550946938596450E+01_dp,  9.6255888915734857E+01_dp,&
      2.2051927486829232E+01_dp, -1.8305163287186298E+02_dp,&
      1.8429047593569263E+02_dp, -8.3575794890910672E+01_dp,&
      1.5320836610304321E+01_dp,  1.1616226247615350E+03_dp,&
      -1.2623095596855940E+03_dp,  8.6933886681923610E+01_dp,&
      6.4414090263090793E+02_dp, -1.9978982244793221E+02_dp,&
      -3.2660522885185867E+02_dp,  1.0098963434615148E+02_dp,&
      2.7721445435633876E+02_dp, -8.9111509314974313E+01_dp,&
      -3.5498277231331673E+02_dp,  4.4957236371547003E+02_dp,&
      -2.2695337686830618E+02_dp,  4.4827771222570036E+01_dp,&
      2.5488397812481312E+03_dp, -3.2720196575275063E+03_dp,&
      6.2514938447284021E+02_dp,  1.6797717015255266E+03_dp,&
      -9.0609391756137052E+02_dp, -7.7376478784489177E+02_dp,&
      5.1365112699708175E+02_dp,  7.9705181518637903E+02_dp,&
      -8.1808800039951734E+02_dp, -3.1602106846125673E+02_dp,&
      8.9474824450035931E+02_dp, -5.4669103099058145E+02_dp,&
      1.1970060288606828E+02_dp,  5.9381087032667047E+03_dp,&
      -9.0911589870974240E+03_dp,  3.1519440108030135E+03_dp,&
      4.3742938194787512E+03_dp, -3.9151712191612141E+03_dp,&
      -1.4306765495221396E+03_dp,  2.2531425467709741E+03_dp,&
      2.0894645666101583E+03_dp, -4.4544178967310481E+03_dp,&
      2.2383591529298183E+03_dp,  4.4379030282863710E+02_dp,&
      -8.3686062012151353E+02_dp,  2.3941035244133383E+02_dp,&
      1.4941870061839112E+04_dp, -2.7698725637069860E+04_dp,&
      1.5368590597048256E+04_dp,  1.0343886022762168E+04_dp,&
      -1.7085228275250280E+04_dp,  1.0290154787655192E+03_dp,&
      8.9420986845818061E+03_dp,  2.9316545272252147E+03_dp,&
      -1.8997312206692230E+04_dp,  1.9823842525145723E+04_dp,&
      -9.7693564885756969E+03_dp,  2.1647867710169985E+03_dp,&
      -1.0378839433368972E+02_dp,  4.1812142330903938E+04_dp,&
      -9.5876465655889318E+04_dp,  8.1756920297039134E+04_dp,&
      7.9039564977741147E+03_dp, -7.3842379280105539E+04_dp,&
      4.4204923551467022E+04_dp,  2.1730397459397547E+04_dp,&
      -2.3137741549322684E+04_dp, -3.9615691029120862E+04_dp,&
      8.2843892021041815E+04_dp, -6.6187145198822764E+04_dp,&
      2.6773721003469775E+04_dp, -4.5704387359470866E+03_dp,&
      1.4201349500169407E+05_dp, -4.0614829221312096E+05_dp,&
      5.2070095300974668E+05_dp, -2.7373429328130849E+05_dp,&
      -1.5922589262630625E+05_dp,  3.8367646454422048E+05_dp,&
      -2.5932110490762870E+05_dp,  2.5826860555147938E+04_dp,&
      5.5653603356457446E+04_dp,  3.5258106303212062E+03_dp,&
      -4.9963042898806314E+04_dp,  3.4015110532782375E+04_dp,&
      -7.7664796273009124E+03_dp,  1.9093974668075968E+00_dp,&
      -1.7054400904209266E-01_dp, -9.7641580671686004E-02_dp,&
      1.6766255796770199E-02_dp,  2.8988444707523806E-02_dp,&
      -2.5620769652305881E-03_dp, -1.8370205087387178E-02_dp,&
      2.4203819467354571E-03_dp,  1.5850000696904308E-02_dp,&
      -1.2409755539837933E-02_dp,  1.3104701010835478E-03_dp,&
      2.1950558162222062E-03_dp, -7.7840696413045292E-04_dp,&
      2.3816405893090655E+00_dp, -3.0058399408021230E-01_dp,&
      -1.5977869438048026E-01_dp,  4.3205664883079593E-02_dp,&
      5.1954260872828326E-02_dp, -1.0435767802951716E-02_dp,&
      -3.3687805923760396E-02_dp,  6.7210749032354392E-03_dp,&
      3.1994491176302167E-02_dp, -2.9480232130482186E-02_dp,&
      7.4474480162902649E-03_dp,  2.2084644917421768E-03_dp,&
      -1.1620522957490494E-03_dp,  3.4843292108184070E+00_dp,&
      -6.3704889535412579E-01_dp, -3.1247856056891354E-01_dp,&
      1.1946809479052665E-01_dp,  1.1015354223955937E-01_dp,&
      -3.5224341675936091E-02_dp, -7.3289581156592246E-02_dp,&
      1.9950208126348418E-02_dp,  7.6285287328668447E-02_dp,&
      -7.9932304695661163E-02_dp,  2.8268292772225596E-02_dp,&
      3.8090423507330300E-04_dp, -1.8684578732096576E-03_dp,&
      5.5863369323075549E+00_dp, -1.3713721433368866E+00_dp,&
      -6.2196190605453838E-01_dp,  3.0824039077130033E-01_dp,&
      2.3240679161688174E-01_dp, -1.0242926268918769E-01_dp,&
      -1.5885664856881426E-01_dp,  5.5322292014197520E-02_dp,&
      1.7989426377403095E-01_dp, -2.0915679370975637E-01_dp,&
      8.9364744914267866E-02_dp, -9.7174197412381834E-03_dp,&
      -2.4358156829036412E-03_dp,  9.4101499417614978E+00_dp,&
      -2.9205066473747960E+00_dp, -1.2163674262839712E+00_dp,&
      7.5863769149748450E-01_dp,  4.7467102033224490E-01_dp,&
      -2.7655989190772234E-01_dp, -3.3454490509347323E-01_dp,&
      1.4696126355412018E-01_dp,  4.1354189925528473E-01_dp,&
      -5.3037498985122311E-01_dp,  2.6031262682300332E-01_dp,&
      -4.8168408431760540E-02_dp, -7.7871669409946837E-04_dp,&
      1.6338815605587858E+01_dp, -6.1833514746412481E+00_dp,&
      -2.3311621003398191E+00_dp,  1.8217844419225089E+00_dp,&
      9.3758726271025805E-01_dp, -7.1865224719975085E-01_dp,&
      -6.8550844653228826E-01_dp,  3.8262747601717012E-01_dp,&
      9.3366571673349752E-01_dp, -1.3231064225590690E+00_dp,&
      7.2780109182395636E-01_dp, -1.7510717849303473E-01_dp,&
      1.0697404777480228E-02_dp,  2.9045107955015176E+01_dp,&
      -1.3122115286638619E+01_dp, -4.3806637776150632E+00_dp,&
      4.3296889561695737E+00_dp,  1.7839258348899563E+00_dp,&
      -1.8308198205695498E+00_dp, -1.3662114443942674E+00_dp,&
      9.9043960784398033E-01_dp,  2.0798949625159446E+00_dp,&
      -3.2780668652253730E+00_dp,  1.9918330621946101E+00_dp,&
      -5.6663166075695159E-01_dp,  5.7158452667943484E-02_dp,&
      5.2800048392041845E+01_dp, -2.8109720042733336E+01_dp,&
      -8.0405851706326441E+00_dp,  1.0279577923775973E+01_dp,&
      3.2138452521312746E+00_dp, -4.6231746304589763E+00_dp,&
      -2.6241582510606816E+00_dp,  2.5740075419555373E+00_dp,&
      4.5650980871610862E+00_dp, -8.1046297603881445E+00_dp,&
      5.3943552024246850E+00_dp, -1.7331884894412846E+00_dp,&
      2.1832820004259532E-01_dp,  9.8274043380510179E+01_dp,&
      -6.1150381159704402E+01_dp, -1.4222299310017810E+01_dp,&
      2.4549816525007763E+01_dp,  5.2132243676002723E+00_dp,&
      -1.1655504278147887E+01_dp, -4.7278213600031389E+00_dp,&
      6.7676550797734309E+00_dp,  9.7732086339422466E+00_dp,&
      -2.0017061889583658E+01_dp,  1.4544370076050541E+01_dp,&
      -5.1483846630015107E+00_dp,  7.4121547100618823E-01_dp,&
      1.8774247624315791E+02_dp, -1.3585965292035331E+02_dp,&
      -2.3384417210458281E+01_dp,  5.9307322699677300E+01_dp,&
      6.3425555998420622E+00_dp, -2.9474617795317286E+01_dp,&
      -7.3842765889098940E+00_dp,  1.8118043634237178E+01_dp,&
      1.9851466755132208E+01_dp, -4.9221535131083812E+01_dp,&
      3.9121944733752876E+01_dp, -1.5053605862917172E+01_dp,&
      2.3820642460959474E+00_dp,  3.6938617681120809E+02_dp,&
      -3.1008390641542189E+02_dp, -3.1908899493735571E+01_dp,&
      1.4562271786149682E+02_dp, -1.2559633861124211E+00_dp,&
      -7.4923876238492468E+01_dp, -6.9384384355100055E+00_dp,&
      4.9653981256260799E+01_dp,  3.5331302879921083E+01_dp,&
      -1.1918108482817887E+02_dp,  1.0467883559485558E+02_dp,&
      -4.3551657522660385E+01_dp,  7.4318254787716498E+00_dp,&
      7.5178053722265770E+02_dp, -7.3185923065526492E+02_dp,&
      -1.6464637400435404E+01_dp,  3.6485923276725026E+02_dp,&
      -5.1587363386350894E+01_dp, -1.9111170221147160E+02_dp,&
      1.5490715798392426E+01_dp,  1.3978292003195943E+02_dp,&
      3.8110337744866719E+01_dp, -2.7620328681571948E+02_dp,&
      2.7537298325765300E+02_dp, -1.2425623890169337E+02_dp,&
      2.2689284891309541E+01_dp,  1.5916154447623735E+03_dp,&
      -1.8003650102566801E+03_dp,  1.3061885225064071E+02_dp,&
      9.3513854431018569E+02_dp, -2.8305798212942278E+02_dp,&
      -4.8432743152230262E+02_dp,  1.4241629292606842E+02_dp,&
      4.0370510754094880E+02_dp, -9.9685940038875515E+01_dp,&
      -5.6492023569870310E+02_dp,  6.8962159878288605E+02_dp,&
      -3.4344985934760621E+02_dp,  6.7268170706916536E+01_dp,&
      3.5313329066420033E+03_dp, -4.6623582136066398E+03_dp,&
      8.7352686043406538E+02_dp,  2.4489787138449915E+03_dp,&
      -1.2584736353041660E+03_dp, -1.1810855855446016E+03_dp,&
      7.1672145189107039E+02_dp,  1.1795141993550299E+03_dp,&
      -1.0569593213292389E+03_dp, -7.0128668015997494E+02_dp,&
      1.4914938794172158E+03_dp, -8.7124256006302119E+02_dp,&
      1.8684842614746950E+02_dp,  8.2940863893933129E+03_dp,&
      -1.2880093011243731E+04_dp,  4.2495122306124758E+03_dp,&
      6.4758081698510341E+03_dp, -5.3383669126949280E+03_dp,&
      -2.4685864914850517E+03_dp,  3.1413524828014874E+03_dp,&
      3.3033312228152809E+03_dp, -6.0335109923486816E+03_dp,&
      2.1284718544868892E+03_dp,  1.6537769702474902E+03_dp,&
      -1.6613883929668332E+03_dp,  4.2939992860686181E+02_dp,&
      2.0920103579440787E+04_dp, -3.8669758427570443E+04_dp,&
      1.9866909734865130E+04_dp,  1.6420139899251604E+04_dp,&
      -2.2915184173335136E+04_dp, -1.5997192354548479E+03_dp,&
      1.2941589682976841E+04_dp,  7.0023617449621897E+03_dp,&
      -2.8057100680185933E+04_dp,  2.5033445993726258E+04_dp,&
      -9.5663441270494550E+03_dp,  7.6757608917678635E+02_dp,&
      3.3218276502429876E+02_dp,  5.7934335499806577E+04_dp,&
      -1.2968557309336646E+05_dp,  9.9056954372674198E+04_dp,&
      2.9476068192368246E+04_dp, -1.0138652618742791E+05_dp,&
      3.6368515099655910E+04_dp,  4.6073637136932237E+04_dp,&
      -1.0492843328428793E+04_dp, -9.7848110148587089E+04_dp,&
      1.4355967431747538E+05_dp, -9.7281253297314208E+04_dp,&
      3.4312801274826343E+04_dp, -5.1027940556969679E+03_dp,&
      1.8382562965656386E+05_dp, -5.0724133133864752E+05_dp,&
      5.7124379095245234E+05_dp, -1.3430281607756464E+05_dp,&
      -4.1104787472683209E+05_dp,  4.5698302097076108E+05_dp,&
      -1.4772171067732785E+04_dp, -2.9013984608709166E+05_dp,&
      9.7841936518310904E+04_dp,  2.5630153474370242E+05_dp,&
      -3.3461736758467171E+05_dp,  1.7210001134766926E+05_dp,&
      -3.4775431355638953E+04_dp,  8.0851544458545872E+05_dp,&
      -2.6363345449983794E+06_dp,  4.1361998354174164E+06_dp,&
      -3.5121162276064917E+06_dp,  7.0893150895957707E+05_dp,&
      2.2570973383430163E+06_dp, -3.4154205506155374E+06_dp,&
      2.6954736607133211E+06_dp, -1.4066197350891286E+06_dp,&
      5.4951201324373030E+05_dp, -1.9512919132011509E+05_dp,&
      6.3990065228808700E+04_dp, -1.1942735238243258E+04_dp,&
      2.4099562266117897E-01_dp, -7.2049876247263943E-03_dp,&
      -6.4116087584877362E-03_dp,  2.3477247701907569E-03_dp,&
      -2.4131694227638371E-03_dp,  4.0371201990794785E-03_dp,&
      -4.8026052577681737E-03_dp,  4.9757874425969597E-03_dp,&
      -4.9284265776128131E-03_dp,  4.2028583295808742E-03_dp,&
      -2.6617139437737393E-03_dp,  1.0721512574701931E-03_dp,&
      -2.0360724244160395E-04_dp,  7.8600093766184087E-01_dp,&
      -2.7552722482330382E-02_dp, -2.4320818015224920E-02_dp,&
      9.3274629486534248E-03_dp, -9.2079352354389778E-03_dp,&
      1.5452692636155842E-02_dp, -1.8450575101340298E-02_dp,&
      1.9063757888167280E-02_dp, -1.8832830863784338E-02_dp,&
      1.6057085510439863E-02_dp, -1.0174585816282621E-02_dp /)
    REAL(KIND=dp), DIMENSION(400), PARAMETER :: c06 = (/ &
      4.0992188628857576E-03_dp, -7.7827564716362676E-04_dp,&
      1.5427835617899610E+00_dp, -6.9505394993586406E-02_dp,&
      -6.0594885888198051E-02_dp,  2.4860739271249380E-02_dp,&
      -2.3168221093078572E-02_dp,  3.9027511015698728E-02_dp,&
      -4.6867833756562442E-02_dp,  4.8230445510777237E-02_dp,&
      -4.7451819606639183E-02_dp,  4.0444324071817472E-02_dp,&
      -2.5649820333915784E-02_dp,  1.0337965959194499E-02_dp,&
      -1.9621474197091537E-03_dp,  2.7361460521635825E+00_dp,&
      -1.6210915733697953E-01_dp, -1.3938372870854468E-01_dp,&
      6.1370009081324846E-02_dp, -5.3953321352611480E-02_dp,&
      9.1109307724040633E-02_dp, -1.1013676364888772E-01_dp,&
      1.1285815532268084E-01_dp, -1.1051732784079558E-01_dp,&
      9.4154657670159986E-02_dp, -5.9775221985383778E-02_dp,&
      2.4105091177407285E-02_dp, -4.5740232709998250E-03_dp,&
      4.7504664477072316E+00_dp, -3.6540640732652457E-01_dp,&
      -3.0963304480188580E-01_dp,  1.4617823006458483E-01_dp,&
      -1.2163472991350244E-01_dp,  2.0546897879778972E-01_dp,&
      -2.5017072872400081E-01_dp,  2.5528811465811574E-01_dp,&
      -2.4871800987649034E-01_dp,  2.1177224606033893E-01_dp,&
      -1.3460697362966942E-01_dp,  5.4321881683681905E-02_dp,&
      -1.0306557157615706E-02_dp,  8.2846134626133292E+00_dp,&
      -8.0838138618414457E-01_dp, -6.7434278259781932E-01_dp,&
      3.4154350399965178E-01_dp, -2.6969855520786901E-01_dp,&
      4.5455382056357474E-01_dp, -5.5786575129632443E-01_dp,&
      5.6695864745421198E-01_dp, -5.4922218190823124E-01_dp,&
      4.6727981986625922E-01_dp, -2.9742716810172054E-01_dp,&
      1.2014838410171294E-01_dp, -2.2797188357298853E-02_dp,&
      1.4659069700977295E+01_dp, -1.7747757205152632E+00_dp,&
      -1.4550424144280802E+00_dp,  7.9267096661767655E-01_dp,&
      -5.9501413092507949E-01_dp,  9.9738596204312624E-01_dp,&
      -1.2351218666597548E+00_dp,  1.2502002535211376E+00_dp,&
      -1.2031718276241572E+00_dp,  1.0225830615341298E+00_dp,&
      -6.5195914949791578E-01_dp,  2.6371662619257846E-01_dp,&
      -5.0052760660977261E-02_dp,  2.6448156436321245E+01_dp,&
      -3.9033379057570827E+00_dp, -3.1379124640819920E+00_dp,&
      1.8465185787105332E+00_dp, -1.3195424388936023E+00_dp,&
      2.1908275051266819E+00_dp, -2.7410453141653983E+00_dp,&
      2.7635561809966092E+00_dp, -2.6392193814105300E+00_dp,&
      2.2397534106652977E+00_dp, -1.4308096005721775E+00_dp,&
      5.7980482655024335E-01_dp, -1.1011473830562242E-01_dp,&
      4.8827566212215686E+01_dp, -8.6709145648362504E+00_dp,&
      -6.8145806531242146E+00_dp,  4.3563149107877930E+00_dp,&
      -2.9691106736363246E+00_dp,  4.8564608241711680E+00_dp,&
      -6.1483109774475455E+00_dp,  6.1755130247165688E+00_dp,&
      -5.8438669005291510E+00_dp,  4.9487151155432789E+00_dp,&
      -3.1688647001286903E+00_dp,  1.2872479155148069E+00_dp,&
      -2.4474367721377402E-01_dp,  9.2556502815334994E+01_dp,&
      -1.9603282099708711E+01_dp, -1.5002883382810225E+01_dp,&
      1.0494569374376898E+01_dp, -6.8410742350968299E+00_dp,&
      1.0944746242954064E+01_dp, -1.4045586508307052E+01_dp,&
      1.4060381579443410E+01_dp, -1.3158840677577247E+01_dp,&
      1.1107899993760929E+01_dp, -7.1328849625134936E+00_dp,&
      2.9070868330198718E+00_dp, -5.5374097833048719E-01_dp,&
      1.8086387815603692E+02_dp, -4.5449136805946708E+01_dp,&
      -3.3697127799416755E+01_dp,  2.6030360696858182E+01_dp,&
      -1.6300016117567488E+01_dp,  2.5261110581327873E+01_dp,&
      -3.2926409290629678E+01_dp,  3.2875420065927933E+01_dp,&
      -3.0353354761618480E+01_dp,  2.5499432007902808E+01_dp,&
      -1.6427830450896202E+01_dp,  6.7254398764218735E+00_dp,&
      -1.2848018257579334E+00_dp,  3.6615307455742288E+02_dp,&
      -1.0895580944560962E+02_dp, -7.7707894068596588E+01_dp,&
      6.7083059154405120E+01_dp, -4.0625930911416084E+01_dp,&
      6.0191592289485307E+01_dp, -7.9850740811585609E+01_dp,&
      7.9626930467390395E+01_dp, -7.2291475212265667E+01_dp,&
      6.0275016793375315E+01_dp, -3.8970639832754756E+01_dp,&
      1.6051798367917286E+01_dp, -3.0804040922174489E+00_dp,&
      7.7295926933662383E+02_dp, -2.7274630136859054E+02_dp,&
      -1.8524424755998632E+02_dp,  1.8158011767619479E+02_dp,&
      -1.0746016288132108E+02_dp,  1.4949540595499082E+02_dp,&
      -2.0220074697334576E+02_dp,  2.0184753824607512E+02_dp,&
      -1.7940902459359080E+02_dp,  1.4776377024332152E+02_dp,&
      -9.5855229939181896E+01_dp,  3.9808239583015897E+01_dp,&
      -7.6927802615479832E+00_dp,  1.7164950125258283E+03_dp,&
      -7.2191533002872904E+02_dp, -4.5988985220715472E+02_dp,&
      5.2344948407025970E+02_dp, -3.0757874260869397E+02_dp,&
      3.9198001571233146E+02_dp, -5.4073857326588131E+02_dp,&
      5.4249811741022972E+02_dp, -4.6932848633916359E+02_dp,&
      3.7853944596752183E+02_dp, -2.4590148935655793E+02_dp,&
      1.0323206520266561E+02_dp, -2.0162118950386379E+01_dp,&
      4.0597984203925230E+03_dp, -2.0554408966225374E+03_dp,&
      -1.1981956551284402E+03_dp,  1.6379438988785680E+03_dp,&
      -9.7944271492137625E+02_dp,  1.1058705957255131E+03_dp,&
      -1.5495244988136308E+03_dp,  1.5727925045092773E+03_dp,&
      -1.3143543851594472E+03_dp,  1.0206697674392115E+03_dp,&
      -6.5915665050050154E+02_dp,  2.8033098204930724E+02_dp,&
      -5.5627786985458201E+01_dp,  1.0418737555296313E+04_dp,&
      -6.4587705877741246E+03_dp, -3.2930962522435352E+03_dp,&
      5.7217121047356122E+03_dp, -3.6200039792272896E+03_dp,&
      3.4703091404762904E+03_dp, -4.8499754392515288E+03_dp,&
      5.0352943316006931E+03_dp, -4.0300238041629141E+03_dp,&
      2.9069303818723929E+03_dp, -1.8196316384987094E+03_dp,&
      7.8120202146243457E+02_dp, -1.5851065777637234E+02_dp,&
      2.9904840276131377E+04_dp, -2.3365682875309405E+04_dp,&
      -9.4213557234108248E+03_dp,  2.3341037180846441E+04_dp,&
      -1.6664545324000497E+04_dp,  1.3018178357102739E+04_dp,&
      -1.7013313819623374E+04_dp,  1.8330955289497884E+04_dp,&
      -1.4004140465670787E+04_dp,  8.6573316238168354E+03_dp,&
      -4.7417789173697429E+03_dp,  1.9635793580241952E+03_dp,&
      -4.0643170846761330E+02_dp,  1.0157948762160608E+05_dp,&
      -1.0548383390007413E+05_dp, -2.3722704012505274E+04_dp,&
      1.2044737338200401E+05_dp, -1.0942907243642975E+05_dp,&
      7.1576985837329717E+04_dp, -6.9634226489907858E+04_dp,&
      7.5769005808596703E+04_dp, -5.6409114203720761E+04_dp,&
      2.4902335273688092E+04_dp, -5.3550087892323263E+03_dp,&
      2.8390167115008719E+01_dp,  1.4643329837669990E+02_dp,&
      4.6857889664810052E+05_dp, -7.2038307415608899E+05_dp,&
      1.2509311324741252E+05_dp,  8.8091622462875850E+05_dp,&
      -1.4127842818682543E+06_dp,  1.1880475416018902E+06_dp,&
      -5.8502506049625343E+05_dp,  2.4743808750507604E+04_dp,&
      3.3699929860959132E+05_dp, -4.7117288930957159E+05_dp,&
      3.8186109459051176E+05_dp, -1.8185745287369000E+05_dp,&
      3.9096271590969533E+04_dp,  1.9685202318288957E+00_dp,&
      -6.1360979308742736E-02_dp, -5.4488336108174731E-02_dp,&
      2.0199240738539220E-02_dp, -2.0538828062587423E-02_dp,&
      3.4390780667720153E-02_dp, -4.0950900871404537E-02_dp,&
      4.2396877890840033E-02_dp, -4.1964658244183287E-02_dp,&
      3.5784807205071087E-02_dp, -2.2666021560142281E-02_dp,&
      9.1304409682086869E-03_dp, -1.7338039826641876E-03_dp,&
      2.4873431607352861E+00_dp, -1.1127896744503744E-01_dp,&
      -9.7160647648538795E-02_dp,  3.9549196995538598E-02_dp,&
      -3.7088461256170678E-02_dp,  6.2482162326251998E-02_dp,&
      -7.4973797450662821E-02_dp,  7.7187912043400322E-02_dp,&
      -7.5984338726898373E-02_dp,  6.4767341178840990E-02_dp,&
      -4.1070002288843978E-02_dp,  1.6551536662038395E-02_dp,&
      -3.1415083350615496E-03_dp,  3.7115819235040099E+00_dp,&
      -2.4263735069592671E-01_dp, -2.0828276881828411E-01_dp,&
      9.2474300956586888E-02_dp, -8.0673281317597170E-02_dp,&
      1.3641415406027710E-01_dp, -1.6499696868142674E-01_dp,&
      1.6897498706163439E-01_dp, -1.6540000447304345E-01_dp,&
      1.4091102128378122E-01_dp, -8.9465194191119327E-02_dp,&
      3.6077418740267916E-02_dp, -6.8451878020381164E-03_dp,&
      6.0819380402782102E+00_dp, -5.3592127980283966E-01_dp,&
      -4.5287536801814943E-01_dp,  2.1661663264454648E-01_dp,&
      -1.7820374811642009E-01_dp,  3.0146509927127713E-01_dp,&
      -3.6745474931004735E-01_dp,  3.7463188233201217E-01_dp,&
      -3.6470166760065986E-01_dp,  3.1051645875911826E-01_dp,&
      -1.9740158338751423E-01_dp,  7.9665597214015602E-02_dp,&
      -1.5113366050427108E-02_dp,  1.0479537652915743E+01_dp,&
      -1.1712760954555530E+00_dp, -9.7387580140040209E-01_dp,&
      5.0050202014332057E-01_dp, -3.9044301463179548E-01_dp,&
      6.5882177079888460E-01_dp, -8.0968653097802212E-01_dp,&
      8.2205031893424119E-01_dp, -7.9554850265578514E-01_dp,&
      6.7681690340488077E-01_dp, -4.3089093066787693E-01_dp,&
      1.7407475175520742E-01_dp, -3.3025834490739599E-02_dp,&
      1.8635108288599536E+01_dp, -2.5497121657871862E+00_dp,&
      -2.0826834920345156E+00_dp,  1.1522254517634021E+00_dp,&
      -8.5439272544641709E-01_dp,  1.4332103583869653E+00_dp,&
      -1.7777542118465444E+00_dp,  1.7975307886658498E+00_dp,&
      -1.7279183793446284E+00_dp,  1.4684523252875816E+00_dp,&
      -9.3648068065724421E-01_dp,  3.7885018835420770E-01_dp,&
      -7.1898509086654641E-02_dp,  3.3994590239686019E+01_dp,&
      -5.5786535866163760E+00_dp, -4.4666506461735587E+00_dp,&
      2.6703461173069210E+00_dp, -1.8856235312128826E+00_dp,&
      3.1315158571185848E+00_dp, -3.9253072081456857E+00_dp,&
      3.9531937219561639E+00_dp, -3.7704367525106743E+00_dp,&
      3.1994378130940819E+00_dp, -2.0445460353306122E+00_dp,&
      8.2864620464328365E-01_dp, -1.5736316384893229E-01_dp,&
      6.3588405510579335E+01_dp, -1.2363461132070416E+01_dp,&
      -9.6753997605292525E+00_dp,  6.2824458697670735E+00_dp,&
      -4.2341320945537095E+00_dp,  6.9247378274318594E+00_dp,&
      -8.7844588705872138E+00_dp,  8.8136469170395682E+00_dp,&
      -8.3287909317576716E+00_dp,  7.0522992389582715E+00_dp,&
      -4.5175728840847738E+00_dp,  1.8354907072519964E+00_dp,&
      -3.4896177050152971E-01_dp,  1.2220829809176205E+02_dp,&
      -2.7944214858134188E+01_dp, -2.1296201547453109E+01_dp,&
      1.5115164278434872E+01_dp, -9.7538219914620825E+00_dp,&
      1.5600604221804334E+01_dp, -2.0061710148318788E+01_dp,&
      2.0061333896341090E+01_dp, -1.8749429601270812E+01_dp,&
      1.5826299380047390E+01_dp, -1.0167002373267730E+01_dp,&
      4.1445597664016303E+00_dp, -7.8940481877510260E-01_dp,&
      2.4212494294883325E+02_dp, -6.4848567095130747E+01_dp,&
      -4.7896931247182728E+01_dp,  3.7461397357650824E+01_dp,&
      -2.3251790808727218E+01_dp,  3.6039980420712112E+01_dp,&
      -4.7067767611689661E+01_dp,  4.6945415655437806E+01_dp,&
      -4.3293284411662249E+01_dp,  3.6374015775739899E+01_dp,&
      -2.3444250992847859E+01_dp,  9.5995302583925408E+00_dp,&
      -1.8336252417158323E+00_dp,  4.9678277300087296E+02_dp,&
      -1.5564477617682064E+02_dp, -1.1069764568693903E+02_dp,&
      9.6407239682845059E+01_dp, -5.7938205328089253E+01_dp,&
      8.5979370733647627E+01_dp, -1.1425483537959983E+02_dp,&
      1.1380794672668260E+02_dp, -1.0325159247360516E+02_dp,&
      8.6134415628011325E+01_dp, -5.5718825689552844E+01_dp,&
      2.2951002054178900E+01_dp, -4.4030493121892027E+00_dp,&
      1.0618466971760956E+03_dp, -3.8969710352881111E+02_dp,&
      -2.6450171371001460E+02_dp,  2.6007018186143932E+02_dp,&
      -1.5279188655132614E+02_dp,  2.1360891249135918E+02_dp,&
      -2.8931621709083004E+02_dp,  2.8840853173138242E+02_dp,&
      -2.5640283554040690E+02_dp,  2.1150347347655622E+02_dp,&
      -1.3730820428083371E+02_dp,  5.7010139200952608E+01_dp,&
      -1.1009287497802793E+01_dp,  2.3831525849788013E+03_dp,&
      -1.0288984124938795E+03_dp, -6.5778574392811515E+02_dp,&
      7.4418907067655823E+02_dp, -4.3344091595675718E+02_dp,&
      5.5871797886676870E+02_dp, -7.7182037553705447E+02_dp,&
      7.7263905710645179E+02_dp, -6.6963947277791476E+02_dp,&
      5.4224742270029606E+02_dp, -3.5283793388597093E+02_dp,&
      1.4803941533425319E+02_dp, -2.8871200173717554E+01_dp,&
      5.6772385530114016E+03_dp, -2.9062895360890998E+03_dp,&
      -1.7148517487548181E+03_dp,  2.2947989851698503E+03_dp,&
      -1.3525758366401831E+03_dp,  1.5626038948903565E+03_dp,&
      -2.1964155478564353E+03_dp,  2.2203872701695277E+03_dp /)
    REAL(KIND=dp), DIMENSION(400), PARAMETER :: c07 = (/ &
      -1.8635835707916260E+03_dp,  1.4615094004676805E+03_dp,&
      -9.4860776301801513E+02_dp,  4.0324630910185732E+02_dp,&
      -7.9802696400978931E+01_dp,  1.4580340019039370E+04_dp,&
      -8.9648221759975659E+03_dp, -4.7160531127075310E+03_dp,&
      7.7970756952852253E+03_dp, -4.7953125668139064E+03_dp,&
      4.7904213388018679E+03_dp, -6.7759341874801794E+03_dp,&
      6.9800960085149118E+03_dp, -5.6283983143506020E+03_dp,&
      4.1614628365758181E+03_dp, -2.6520659708846315E+03_dp,&
      1.1425936294797418E+03_dp, -2.3097558189199052E+02_dp,&
      4.1339464053298638E+04_dp, -3.1163959542726632E+04_dp,&
      -1.3657950683331510E+04_dp,  3.0183606937467408E+04_dp,&
      -2.0297598820224881E+04_dp,  1.6867253952668740E+04_dp,&
      -2.3145329533718246E+04_dp,  2.4629319790813181E+04_dp,&
      -1.8956801723961318E+04_dp,  1.2473061205766879E+04_dp,&
      -7.3797593844846533E+03_dp,  3.1721822281891500E+03_dp,&
      -6.5940325840740331E+02_dp,  1.3465183476229242E+05_dp,&
      -1.2878154349397100E+05_dp, -3.9551202896785959E+04_dp,&
      1.4058730137702916E+05_dp, -1.1150513454150937E+05_dp,&
      7.5934850042442980E+04_dp, -9.0027245525794060E+04_dp,&
      1.0007380677391199E+05_dp, -7.4106008002716495E+04_dp,&
      3.8168389198024262E+04_dp, -1.5926680674690388E+04_dp,&
      5.6819996226797621E+03_dp, -1.1787084557648736E+03_dp,&
      5.4426445660263638E+05_dp, -7.0252376902961778E+05_dp,&
      -5.5472023615715923E+04_dp,  8.6719603863591724E+05_dp,&
      -9.3672058996953757E+05_dp,  5.8718722970563383E+05_dp,&
      -4.2246478920076904E+05_dp,  4.3113805307864543E+05_dp,&
      -3.1790388144140469E+05_dp,  8.9294940174578951E+04_dp,&
      4.9835062579173107E+04_dp, -4.9965645162383262E+04_dp,&
      1.2952090357486064E+04_dp,  3.3621760396656832E+06_dp,&
      -6.5229123935961705E+06_dp,  3.0062160391263529E+06_dp,&
      7.4531686564585082E+06_dp, -1.7301829485568821E+07_dp,&
      1.8484963345212858E+07_dp, -9.5516135467152409E+06_dp,&
      -4.0021370551100709E+06_dp,  1.4467922484597271E+07_dp,&
      -1.6960571811769929E+07_dp,  1.2138850561498789E+07_dp,&
      -5.2211586557498844E+06_dp,  1.0381744850822145E+06_dp,&
      2.4257259348117230E-01_dp, -1.5851429704634029E-09_dp,&
      -9.5774069407972163E-03_dp,  1.2300098639931586E-02_dp,&
      -1.6677557521014314E-02_dp,  1.5981292332277006E-02_dp,&
      -2.8119788057016178E-04_dp, -4.1411633666622608E-02_dp,&
      1.0826576167260124E-01_dp, -1.6888673263610837E-01_dp,&
      1.7154817499141597E-01_dp, -1.0466126191680286E-01_dp,&
      2.9111302266020433E-02_dp,  7.9204097116662908E-01_dp,&
      -6.0818994537216391E-09_dp, -3.6740838833795079E-02_dp,&
      4.7185619635050413E-02_dp, -6.3625273088436185E-02_dp,&
      6.0397868728266732E-02_dp,  7.5718647217901276E-04_dp,&
      -1.6182361088743230E-01_dp,  4.1907586547824699E-01_dp,&
      -6.5145319082734277E-01_dp,  6.6046883780165377E-01_dp,&
      -4.0246846324937580E-01_dp,  1.1185374791833239E-01_dp,&
      1.5580568229128680E+00_dp, -1.5419744030696444E-08_dp,&
      -9.3127809560014999E-02_dp,  1.1960239960607019E-01_dp,&
      -1.5991695786356333E-01_dp,  1.4960080232162254E-01_dp,&
      8.9510077773447185E-03_dp, -4.2148836269473983E-01_dp,&
      1.0765070072706033E+00_dp, -1.6647789307570040E+00_dp,&
      1.6830424466049698E+00_dp, -1.0237444784742269E+00_dp,&
      2.8416557439803364E-01_dp,  2.7718621061429261E+00_dp,&
      -3.6163513981536213E-08_dp, -2.1834886480726926E-01_dp,&
      2.8042152161200956E-01_dp, -3.7145151138366161E-01_dp,&
      3.4176124001058317E-01_dp,  3.9044747975845936E-02_dp,&
      -1.0171603426069817E+00_dp,  2.5602863098988093E+00_dp,&
      -3.9374011756654790E+00_dp,  3.9683779293500430E+00_dp,&
      -2.4091012786988979E+00_dp,  6.6779737409364437E-01_dp,&
      4.8311936966109696E+00_dp, -8.1986505697328631E-08_dp,&
      -4.9486997262600380E-01_dp,  6.3555244877383477E-01_dp,&
      -8.3363638388937433E-01_dp,  7.5337822797117326E-01_dp,&
      1.3086191484801463E-01_dp, -2.3728506879221536E+00_dp,&
      5.8867883672658534E+00_dp, -9.0020738404552869E+00_dp,&
      9.0443088573663957E+00_dp, -5.4794215122691758E+00_dp,&
      1.5167430446672299E+00_dp,  8.4637235711253460E+00_dp,&
      -1.8249091149929743E-07_dp, -1.1011450973952845E+00_dp,&
      1.4141801024460730E+00_dp, -1.8355159334262476E+00_dp,&
      1.6263138799610943E+00_dp,  3.9073092232731099E-01_dp,&
      -5.4376260018676819E+00_dp,  1.3293600447123115E+01_dp,&
      -2.0209696632944365E+01_dp,  2.0237473247415195E+01_dp,&
      -1.2234446749358080E+01_dp,  3.3815255705843770E+00_dp,&
      1.5053546138154489E+01_dp, -4.0334165130598051E-07_dp,&
      -2.4328249179107155E+00_dp,  3.1244306626860210E+00_dp,&
      -4.0084799014945292E+00_dp,  3.4724453638380903E+00_dp,&
      1.1020158511749709E+00_dp, -1.2389534810286600E+01_dp,&
      2.9830068384681105E+01_dp, -4.5066858817381963E+01_dp,&
      4.4968188905675269E+01_dp, -2.7122037857043569E+01_dp,&
      7.4841528688448022E+00_dp,  2.7318806501465819E+01_dp,&
      -8.9373103139548194E-07_dp, -5.3883015584663889E+00_dp,&
      6.9200904347821890E+00_dp, -8.7625163748632833E+00_dp,&
      7.3930201736202728E+00_dp,  3.0266739418528634E+00_dp,&
      -2.8356377784336452E+01_dp,  6.7176527817712639E+01_dp,&
      -1.0080221398959092E+02_dp,  1.0018754691787824E+02_dp,&
      -6.0271077807630199E+01_dp,  1.6601208530464657E+01_dp,&
      5.0769359622143128E+01_dp, -2.0022110266338276E-06_dp,&
      -1.2064973865474135E+01_dp,  1.5494803546470303E+01_dp,&
      -1.9327179050582775E+01_dp,  1.5798630009622276E+01_dp,&
      8.2524723662168071E+00_dp, -6.5779344719277219E+01_dp,&
      1.5314732669567439E+02_dp, -2.2809101477129849E+02_dp,&
      2.2570758430280716E+02_dp, -1.3538681440746080E+02_dp,&
      3.7214435740581607E+01_dp,  9.6966640947802489E+01_dp,&
      -4.5708702824465076E-06_dp, -2.7525782848920322E+01_dp,&
      3.5350791499997797E+01_dp, -4.3326681349790739E+01_dp,&
      3.4065698277940633E+01_dp,  2.2663529783272740E+01_dp,&
      -1.5595875787184744E+02_dp,  3.5632753228509131E+02_dp,&
      -5.2627723474899005E+02_dp,  5.1819280410896465E+02_dp,&
      -3.0979403001144362E+02_dp,  8.4952402372718240E+01_dp,&
      1.9114310962046079E+02_dp, -1.0718381310198472E-05_dp,&
      -6.4495689652753413E+01_dp,  8.2830422426384970E+01_dp,&
      -9.9422424907189324E+01_dp,  7.4415368027743341E+01_dp,&
      6.3487036604073083E+01_dp, -3.8117146994730530E+02_dp,&
      8.5310336400357869E+02_dp, -1.2481132204175772E+03_dp,&
      1.2219161469975245E+03_dp, -7.2767253907573388E+02_dp,&
      1.9898804514956225E+02_dp,  3.9095152547016971E+02_dp,&
      -2.6044693436334060E-05_dp, -1.5656455100212540E+02_dp,&
      2.0107231298299899E+02_dp, -2.3531086163800461E+02_dp,&
      1.6507461742953816E+02_dp,  1.8371250260285103E+02_dp,&
      -9.6951867862351389E+02_dp,  2.1208296289805753E+03_dp,&
      -3.0692048830008334E+03_dp,  2.9846327006006663E+03_dp,&
      -1.7692097125381856E+03_dp,  4.8218836520790217E+02_dp,&
      8.3551256508341919E+02_dp, -6.6276418699394831E-05_dp,&
      -3.9790489467747528E+02_dp,  5.1101973017952952E+02_dp,&
      -5.7941288348416379E+02_dp,  3.7150601573538091E+02_dp,&
      5.5698245764046999E+02_dp, -2.5960834505545481E+03_dp,&
      5.5345357608728345E+03_dp, -7.9075149042235771E+03_dp,&
      7.6276140487209941E+03_dp, -4.4959994932460531E+03_dp,&
      1.2203067212487733E+03_dp,  1.8836460432374965E+03_dp,&
      -1.7906558419193089E-04_dp, -1.0732156319450071E+03_dp,&
      1.3783032272117819E+03_dp, -1.5000340075540309E+03_dp,&
      8.4018168997033501E+02_dp,  1.8007044978323806E+03_dp,&
      -7.4288132647801831E+03_dp,  1.5374600294490150E+04_dp,&
      -2.1628986973636649E+04_dp,  2.0654257790969808E+04_dp,&
      -1.2087613800492580E+04_dp,  3.2634643687020621E+03_dp,&
      4.5416034797674138E+03_dp, -5.2365813191155002E-04_dp,&
      -3.1308942252975776E+03_dp,  4.0209190324650995E+03_dp,&
      -4.1383392506453802E+03_dp,  1.8377028026049759E+03_dp,&
      6.3590629378903286E+03_dp, -2.3198652550819283E+04_dp,&
      4.6349523517001166E+04_dp, -6.3943547760604444E+04_dp,&
      6.0263738399936170E+04_dp, -3.4932837998398638E+04_dp,&
      9.3635357641943356E+03_dp,  1.1958338781224160E+04_dp,&
      -1.7071768156226256E-03_dp, -1.0169488624467269E+04_dp,&
      1.3060348340403734E+04_dp, -1.2381213086186415E+04_dp,&
      3.2314378644640055E+03_dp,  2.5435849905159415E+04_dp,&
      -8.1602901155135565E+04_dp,  1.5601679506781770E+05_dp,&
      -2.0965853398512429E+05_dp,  1.9396469663853475E+05_dp,&
      -1.1088271281624625E+05_dp,  2.9404441290840648E+04_dp,&
      3.5614977210684003E+04_dp, -6.5265362044827011E-03_dp,&
      -3.8636947008429313E+04_dp,  4.9619945449366554E+04_dp,&
      -4.0999782846154798E+04_dp, -3.3222504922255262E+03_dp,&
      1.2256002817686876E+05_dp, -3.4080284163623268E+05_dp,&
      6.1352219860694092E+05_dp, -7.9241303014367342E+05_dp,&
      7.1153559384913277E+05_dp, -3.9732963315665961E+05_dp,&
      1.0341010372176219E+05_dp,  1.2848400434943594E+05_dp,&
      -3.2448901589420744E-02_dp, -1.8969566877398198E+05_dp,&
      2.4361640221589044E+05_dp, -1.4921343554708638E+05_dp,&
      -1.5091238856037860E+05_dp,  8.0563254617528804E+05_dp,&
      -1.8647433709473095E+06_dp,  3.0346692605618830E+06_dp,&
      -3.6281356744333007E+06_dp,  3.0528225777004999E+06_dp,&
      -1.6122300376973741E+06_dp,  3.9997604947801877E+05_dp,&
      6.7227601896939008E+05_dp, -2.8143215138966765E-01_dp,&
      -1.5867302113995703E+06_dp,  2.0376973639408911E+06_dp,&
      -1.1025901634165533E+05_dp, -4.2036295031807078E+06_dp,&
      1.0044743271165637E+07_dp, -1.5282870940461934E+07_dp,&
      1.6518963178353507E+07_dp, -1.1769749475844450E+07_dp,&
      4.0871559754661410E+06_dp,  5.6596099490722292E+05_dp,&
      -7.4109536787231569E+05_dp,  1.9819560025056868E+00_dp,&
      -1.3511569339814371E-08_dp, -8.1633253443258433E-02_dp,&
      1.0484017653035560E-01_dp, -1.4194524209112128E-01_dp,&
      1.3568557553371516E-01_dp, -1.3234983778788343E-03_dp,&
      -3.5470515598290253E-01_dp,  9.2500068772695143E-01_dp,&
      -1.4416049296037063E+00_dp,  1.4635924511712637E+00_dp,&
      -8.9265360989689835E-01_dp,  2.4823595537612544E-01_dp,&
      2.5117886684929931E+00_dp, -2.4671854792314193E-08_dp,&
      -1.4901109047023370E-01_dp,  1.9137231477123434E-01_dp,&
      -2.5614818172008030E-01_dp,  2.4006670513485362E-01_dp,&
      1.2935867778898439E-02_dp, -6.7220614776726906E-01_dp,&
      1.7197502043268926E+00_dp, -2.6612289097071167E+00_dp,&
      2.6913697392321345E+00_dp, -1.6374477528376692E+00_dp,&
      4.5458459181050526E-01_dp,  3.7650561907510078E+00_dp,&
      -5.4161605376903836E-08_dp, -3.2700972896614366E-01_dp,&
      4.1997270677737458E-01_dp, -5.5572559086377915E-01_dp,&
      5.1034976841239854E-01_dp,  6.1519345356843586E-02_dp,&
      -1.5283321353925809E+00_dp,  3.8408489596559470E+00_dp,&
      -5.9031588254427607E+00_dp,  5.9476108822547875E+00_dp,&
      -3.6098724362817247E+00_dp,  1.0005011402155195E+00_dp,&
      6.2003958313268743E+00_dp, -1.2037059248503575E-07_dp,&
      -7.2652194441561535E-01_dp,  9.3305879954457793E-01_dp,&
      -1.2217147867466474E+00_dp,  1.1004967117720030E+00_dp,&
      2.0336664708568603E-01_dp, -3.5018460450989166E+00_dp,&
      8.6657256927685911E+00_dp, -1.3238463235718530E+01_dp,&
      1.3293210387322198E+01_dp, -8.0507159019795775E+00_dp,&
      2.2279456762186776E+00_dp,  1.0739206251612034E+01_dp,&
      -2.6473797876852818E-07_dp, -1.5973297281490579E+00_dp,&
      2.0514206663572310E+00_dp, -2.6570672559948862E+00_dp,&
      2.3448587301741366E+00_dp,  5.9565428178472757E-01_dp,&
      -7.9344395190937238E+00_dp,  1.9342826383683839E+01_dp,&
      -2.9372682545317804E+01_dp,  2.9394222808923349E+01_dp,&
      -1.7762737595349108E+01_dp,  4.9080935520946882E+00_dp,&
      1.9202199285653183E+01_dp, -5.8024376773466123E-07_dp,&
      -3.4996092621466643E+00_dp,  4.4944812099998321E+00_dp,&
      -5.7527459450885452E+00_dp,  4.9604902525102608E+00_dp,&
      1.6548760224475922E+00_dp, -1.7934154017551453E+01_dp,&
      4.3051248399276048E+01_dp, -6.4961623842017133E+01_dp,&
      6.4774151275958900E+01_dp, -3.9049982184732009E+01_dp,&
      1.0772159604979572E+01_dp,  3.5239797671962883E+01_dp,&
      -1.2791831843911305E-06_dp, -7.7116293148741368E+00_dp,&
      9.9038942846037248E+00_dp, -1.2509090324776968E+01_dp /)
    REAL(KIND=dp), DIMENSION(13, 38, 6), PARAMETER :: coefdata = RESHAPE ((/ &
      c01, c02, c03, c04, c05, c06, c07, c08 /),(/13,38,6/))

    INTEGER                                  :: irange

     IF(Rc <=  5.0000000000000000E+01_dp) Rc =   5.0000000000000000E+01_dp
     IF(Rc >=  3.5810871124267578E+06_dp) Rc =   3.5810871124267578E+06_dp
     IF(Rc >=  5.0000000000000000E+01_dp .AND. Rc <=  9.6916963080248536E+01_dp) THEN
       irange =  1
       L_b =  5.0000000000000000E+01_dp
       U_b =  9.6916963080248536E+01_dp
     END IF
     IF(Rc >=  9.6916963080248536E+01_dp .AND. Rc <=  1.2185306936104239E+03_dp) THEN
       irange =  2
       L_b =  9.6916963080248536E+01_dp
       U_b =  1.2185306936104239E+03_dp
     END IF
     IF(Rc >=  1.2185306936104239E+03_dp .AND. Rc <=  1.2217104533148158E+04_dp) THEN
       irange =  3
       L_b =  1.2185306936104239E+03_dp
       U_b =  1.2217104533148158E+04_dp
     END IF
     IF(Rc >=  1.2217104533148158E+04_dp .AND. Rc <=  1.9815072618369377E+05_dp) THEN
       irange =  4
       L_b =  1.2217104533148158E+04_dp
       U_b =  1.9815072618369377E+05_dp
     END IF
     IF(Rc >=  1.9815072618369377E+05_dp .AND. Rc <=  1.2188439903425989E+06_dp) THEN
       irange =  5
       L_b =  1.9815072618369377E+05_dp
       U_b =  1.2188439903425989E+06_dp
     END IF
     IF(Rc >=  1.2188439903425989E+06_dp .AND. Rc <=  3.5810871124267578E+06_dp) THEN
       irange =  6
       L_b =  1.2188439903425989E+06_dp
       U_b =  3.5810871124267578E+06_dp
     END IF

     fit_coef(1:13,1:38) = coefdata(1:13,1:38,irange)

   END SUBROUTINE get_fit_coef_k19

! *****************************************************************************
!> \brief ...
!> \param Rc ...
!> \param L_b ...
!> \param U_b ...
!> \param fit_coef ...
! *****************************************************************************
   SUBROUTINE get_fit_coef_k20(Rc,L_b,U_b,fit_coef)
    REAL(KIND=dp)                            :: Rc, L_b, U_b
    REAL(KIND=dp), DIMENSION(13, 40)         :: fit_coef

    REAL(KIND=dp), DIMENSION(320), PARAMETER :: c08 = (/ &
      3.4482542694423209E+03_dp,  9.3380638606834109E+03_dp,&
      -3.6141948437832449E+04_dp,  7.3811576243227406E+04_dp,&
      -1.0360614380245237E+05_dp,  9.9273882530972987E+04_dp,&
      -5.8503038150581720E+04_dp,  1.5941507970166829E+04_dp,&
      1.9245609411100908E+04_dp,  1.5758311389454111E-03_dp,&
      -1.6493149055229915E+04_dp,  2.1319315251857988E+04_dp,&
      -2.0594905686666327E+04_dp,  6.7450368572538291E+03_dp,&
      3.7943888089438202E+04_dp, -1.2716219726681517E+05_dp,&
      2.4802524381258758E+05_dp, -3.3897801223706041E+05_dp,&
      3.1885864066745085E+05_dp, -1.8535737515320248E+05_dp,&
      4.9984902189495566E+04_dp,  5.7318324293103229E+04_dp,&
      5.9071432526945937E-03_dp, -6.2380841746085105E+04_dp,&
      8.0634158042088748E+04_dp, -6.8284731483765951E+04_dp,&
      5.4681930309783763E+02_dp,  1.8545057497723409E+05_dp,&
      -5.3191922255590232E+05_dp,  9.7505194465025037E+05_dp,&
      -1.2805757822087191E+06_dp,  1.1696153509061884E+06_dp,&
      -6.6458955725748790E+05_dp,  1.7602465115753218E+05_dp,&
      2.0678064129765515E+05_dp,  2.8344038829274502E-02_dp,&
      -3.0494093965445616E+05_dp,  3.9416660234818154E+05_dp,&
      -2.5130146580176344E+05_dp, -2.1177938413183455E+05_dp,&
      1.2359116116152727E+06_dp, -2.9217324689694936E+06_dp,&
      4.8357925748930285E+06_dp, -5.8846345861200411E+06_dp,&
      5.0469092229344351E+06_dp, -2.7194600229616752E+06_dp,&
      6.8869342689229129E+05_dp,  1.0819530962975370E+06_dp,&
      2.2283944744663364E-01_dp, -2.5391699671295690E+06_dp,&
      3.2820491767507303E+06_dp, -2.9879597797710146E+05_dp,&
      -6.4273657969231484E+06_dp,  1.5669734066822972E+07_dp,&
      -2.4314998224766523E+07_dp,  2.7124297775591992E+07_dp,&
      -2.0552349360272344E+07_dp,  8.5713366254611593E+06_dp,&
      -3.7935610363443429E+05_dp, -8.3928846510169783E+05_dp,&
      1.9320589271290354E+00_dp,  7.2621425609846251E-09_dp,&
      -7.4528731965208886E-02_dp,  9.6338004250833206E-02_dp,&
      -1.3214657018893700E-01_dp,  1.3186456446235820E-01_dp,&
      -2.0166775874314861E-02_dp, -2.8991359670853600E-01_dp,&
      8.0125762227548336E-01_dp, -1.2825636229353268E+00_dp,&
      1.3287373087728476E+00_dp, -8.2519003619160525E-01_dp,&
      2.3343121150717866E-01_dp,  2.4217999169457665E+00_dp,&
      1.2934929155472219E-08_dp, -1.3279748613131781E-01_dp,&
      1.7165785375157250E-01_dp, -2.3307585350609009E-01_dp,&
      2.2877594664050899E-01_dp, -2.3346318222297352E-02_dp,&
      -5.3715564779172731E-01_dp,  1.4543022329299131E+00_dp,&
      -2.3114184688427013E+00_dp,  2.3856880646661343E+00_dp,&
      -1.4781477166371391E+00_dp,  4.1748286135655011E-01_dp,&
      3.5699082894477758E+00_dp,  2.7703387762280234E-08_dp,&
      -2.8453205801269615E-01_dp,  3.6779425246710412E-01_dp,&
      -4.9428962102081125E-01_dp,  4.7696326068383171E-01_dp,&
      -2.3216522248151596E-02_dp, -1.1945691723631664E+00_dp,&
      3.1721266716571264E+00_dp, -5.0071605495889964E+00_dp,&
      5.1491452709951524E+00_dp, -3.1830440507795661E+00_dp,&
      8.9760588696477739E-01_dp,  5.7714745756709691E+00_dp,&
      6.0203678580378609E-08_dp, -6.1856466138001853E-01_dp,&
      7.9957416304825757E-01_dp, -1.0645544695648401E+00_dp,&
      1.0109461180627741E+00_dp,  1.9498075311132036E-03_dp,&
      -2.6818798561564088E+00_dp,  7.0044906403410883E+00_dp,&
      -1.0989948493463517E+01_dp,  1.1264809560038588E+01_dp,&
      -6.9492617232208316E+00_dp,  1.9569143430774056E+00_dp,&
      9.8063851627247747E+00_dp,  1.2943227446405601E-07_dp,&
      -1.3303860307558404E+00_dp,  1.7196942244173261E+00_dp,&
      -2.2679112076886052E+00_dp,  2.1180849214493009E+00_dp,&
      1.1718272413659171E-01_dp, -5.9500812130151042E+00_dp,&
      1.5295418487216114E+01_dp, -2.3856483405279011E+01_dp,&
      2.4374146939476983E+01_dp, -1.5005553650474871E+01_dp,&
      4.2196153451488474E+00_dp,  1.7182196524894046E+01_dp,&
      2.7681980610640121E-07_dp, -2.8465978348393266E+00_dp,&
      3.6795911913729618E+00_dp, -4.8029054852258346E+00_dp,&
      4.4032195652656236E+00_dp,  5.0834247955654877E-01_dp,&
      -1.3143792363416638E+01_dp,  3.3245320187678388E+01_dp,&
      -5.1533317361899485E+01_dp,  5.2471959863900395E+01_dp,&
      -3.2233041672826495E+01_dp,  9.0504077773870328E+00_dp,&
      3.0845352294831098E+01_dp,  5.9398312369643793E-07_dp,&
      -6.1111889591236563E+00_dp,  7.8994899897307205E+00_dp,&
      -1.0193982404223158E+01_dp,  9.1495116492406492E+00_dp,&
      1.6952456171928449E+00_dp, -2.9178719666384300E+01_dp,&
      7.2568642728771039E+01_dp, -1.1174619343091653E+02_dp,&
      1.1336163692278953E+02_dp, -6.9471395810743132E+01_dp,&
      1.9474058838425890E+01_dp,  5.6681482844036843E+01_dp,&
      1.2880602601267124E-06_dp, -1.3260114690148326E+01_dp,&
      1.7140381644716403E+01_dp, -2.1836576138010962E+01_dp,&
      1.9120563971615155E+01_dp,  5.1270149635423383E+00_dp,&
      -6.5601700286248231E+01_dp,  1.6028123406894895E+02_dp,&
      -2.4505157526179127E+02_dp,  2.4758883306634189E+02_dp,&
      -1.5133127650620855E+02_dp,  4.2343189674877024E+01_dp,&
      1.0677930992732443E+02_dp,  2.8406374165609642E-06_dp,&
      -2.9263985515033859E+01_dp,  3.7827402760261627E+01_dp,&
      -4.7491127745716220E+01_dp,  4.0381636711719807E+01_dp,&
      1.4885135448607762E+01_dp, -1.5037491992215962E+02_dp,&
      3.6054446889308713E+02_dp, -5.4693897575387132E+02_dp,&
      5.5012746515130448E+02_dp, -3.3526230511591672E+02_dp,&
      9.3615003920193601E+01_dp,  2.0679533410485035E+02_dp,&
      6.4094168244319851E-06_dp, -6.6085152661798745E+01_dp,&
      8.5423381867854914E+01_dp, -1.0544823391402035E+02_dp,&
      8.6528860573033953E+01_dp,  4.2714834499989976E+01_dp,&
      -3.5371775303988545E+02_dp,  8.3117344347077687E+02_dp,&
      -1.2500496388367815E+03_dp,  1.2510302711576599E+03_dp,&
      -7.5988036054840700E+02_dp,  2.1168378030342967E+02_dp,&
      4.1328664733661651E+02_dp,  1.4889806008040278E-05_dp,&
      -1.5368210432639367E+02_dp,  1.9865336911012668E+02_dp,&
      -2.4040314537499890E+02_dp,  1.8872866671662570E+02_dp,&
      1.2351690805087296E+02_dp, -8.5972088745757685E+02_dp,&
      1.9767740484365552E+03_dp, -2.9445141512794567E+03_dp,&
      2.9300403857222186E+03_dp, -1.7729344307768993E+03_dp,&
      4.9255725456854503E+02_dp,  8.5654938775191738E+02_dp,&
      3.5872303016542066E-05_dp, -3.7072601294454631E+02_dp,&
      4.7920949648375620E+02_dp, -5.6631511928892780E+02_dp,&
      4.1997532374726688E+02_dp,  3.6557905857252490E+02_dp,&
      -2.1763659954041159E+03_dp,  4.8870520139000928E+03_dp,&
      -7.2006664507013447E+03_dp,  7.1181616235454258E+03_dp,&
      -4.2879275292267075E+03_dp,  1.1874668147376524E+03_dp,&
      1.8527695319692295E+03_dp,  9.0414525740239963E-05_dp,&
      -9.3593883032510507E+02_dp,  1.2098163913708674E+03_dp,&
      -1.3887279432851076E+03_dp,  9.5392042649204359E+02_dp,&
      1.1242551714786039E+03_dp, -5.7944326329618743E+03_dp,&
      1.2675523603724201E+04_dp, -1.8443335807050775E+04_dp,&
      1.8090734173380570E+04_dp, -1.0839637669115842E+04_dp,&
      2.9902567983122644E+03_dp,  4.2185980993912917E+03_dp,&
      2.4111494993262110E-04_dp, -2.5013688864927140E+03_dp,&
      3.2333243651908087E+03_dp, -3.5775439041628538E+03_dp,&
      2.2018907529049170E+03_dp,  3.6526563185232840E+03_dp,&
      -1.6430818396518105E+04_dp,  3.4901684323665315E+04_dp,&
      -5.0036642596126949E+04_dp,  4.8619455802354423E+04_dp,&
      -2.8940491721336319E+04_dp,  7.9451453468075997E+03_dp,&
      1.0231390945985855E+04_dp,  6.9083287944081352E-04_dp,&
      -7.1882842790158111E+03_dp,  9.2917205293700135E+03_dp,&
      -9.7955483893616456E+03_dp,  5.0678948479689298E+03_dp,&
      1.2800733529560235E+04_dp, -5.0479717751341195E+04_dp,&
      1.0365862765556478E+05_dp, -1.4595280309876031E+05_dp,&
      1.4014598216542669E+05_dp, -8.2717100120144009E+04_dp,&
      2.2565914851566984E+04_dp,  2.6891322723431167E+04_dp,&
      2.1742181990913696E-03_dp, -2.2721782303665794E+04_dp,&
      2.9370569957731357E+04_dp, -2.8945217516976965E+04_dp,&
      1.0780164053029053E+04_dp,  4.9798530718129528E+04_dp,&
      -1.7223393818390579E+05_dp,  3.3968227753935917E+05_dp,&
      -4.6738665105989081E+05_dp,  4.4176278346948413E+05_dp,&
      -2.5773351553389218E+05_dp,  6.9696495544733130E+04_dp,&
      7.8684769327617381E+04_dp,  7.7807434887690492E-03_dp,&
      -8.1870048706807342E+04_dp,  1.0582629970733885E+05_dp,&
      -9.4133820403246747E+04_dp,  1.2453504120756526E+04_dp,&
      2.2473369976015511E+05_dp, -6.7807284436999192E+05_dp,&
      1.2707645472222809E+06_dp, -1.6939265958734713E+06_dp,&
      1.5647390178375142E+06_dp, -8.9707114806702349E+05_dp,&
      2.3930069996788309E+05_dp,  2.6850437188439124E+05_dp,&
      3.3617187911264466E-02_dp, -3.5802500077804463E+05_dp,&
      4.6278519403414475E+05_dp, -3.4336642578337889E+05_dp,&
      -1.2299914820280354E+05_dp,  1.2698214439994770E+06_dp,&
      -3.2863510817135861E+06_dp,  5.7313706772967186E+06_dp,&
      -7.2659207395268613E+06_dp,  6.4530123779051183E+06_dp,&
      -3.5837813575776960E+06_dp,  9.3152895714921399E+05_dp,&
      1.1745893617395842E+06_dp,  1.9823345837793316E-01_dp,&
      -2.1674806769945836E+06_dp,  2.8016646898503760E+06_dp,&
      -1.3208943556501460E+06_dp, -2.7152441097245929E+06_dp,&
      1.0527862934716102E+07_dp, -2.2141959028414574E+07_dp,&
      3.3833815704067335E+07_dp, -3.8375451355696611E+07_dp,&
      3.0793030373920619E+07_dp, -1.5573075367459131E+07_dp,&
      3.7151288981291451E+06_dp,  8.6165998717888258E+06_dp,&
      2.2428828403966108E+00_dp, -2.6590263782914072E+07_dp,&
      3.4369174208174527E+07_dp,  1.0603207649981629E+07_dp,&
      -1.0297545596222048E+08_dp,  1.9337680822524059E+08_dp,&
      -2.0606965563558266E+08_dp,  8.3351497204227954E+07_dp,&
      1.3400532965008634E+08_dp, -2.7949474181137478E+08_dp,&
      2.2551932538988832E+08_dp, -7.3358489823807165E+07_dp /)
    REAL(KIND=dp), DIMENSION(400), PARAMETER :: c01 = (/ &
      1.4818427186393188E-01_dp, -3.9684125490741781E-02_dp,&
      -4.5544691512272280E-03_dp,  2.2868531356116861E-03_dp,&
      5.4988095348367525E-04_dp, -3.2090841260734175E-04_dp,&
      -9.2010045248866118E-05_dp,  6.1214638366653809E-05_dp,&
      2.2665438652455234E-05_dp, -1.8900573386763588E-05_dp,&
      1.9737341940349761E-06_dp, -4.0098588379752778E-07_dp,&
      7.9423337565922097E-07_dp,  4.5845629764106710E-01_dp,&
      -1.3050008497162252E-01_dp, -1.1746015834640975E-02_dp,&
      7.9238498027192957E-03_dp,  1.3726359502348783E-03_dp,&
      -1.1375480080520228E-03_dp, -2.2424819941956490E-04_dp,&
      2.1867699685858426E-04_dp,  5.8049724664724386E-05_dp,&
      -6.6759583496292301E-05_dp,  1.1220242809507542E-05_dp,&
      -1.3505737470233449E-06_dp,  2.1298269803731967E-06_dp,&
      8.1205262169756842E-01_dp, -2.5796327574778904E-01_dp,&
      -1.2091050279258431E-02_dp,  1.6773647500350573E-02_dp,&
      1.1878661490072853E-03_dp, -2.4697243937931864E-03_dp,&
      -1.6605322411568808E-04_dp,  4.7805723848042757E-04_dp,&
      5.6067445750992246E-05_dp, -1.4470898201670074E-04_dp,&
      3.9438442939321917E-05_dp, -4.1919645541969900E-06_dp,&
      2.9194474822921137E-06_dp,  1.2433837102757057E+00_dp,&
      -4.5449390505874582E-01_dp,  2.8772391163010618E-03_dp,&
      3.0979924684101916E-02_dp, -1.2867784281123959E-03_dp,&
      -4.6126614533263735E-03_dp,  3.2856131571824614E-04_dp,&
      8.9223222108096990E-04_dp, -3.3421577643852616E-05_dp,&
      -2.7142959004058064E-04_dp,  1.0953992802467019E-04_dp,&
      -1.5370427764270796E-05_dp,  3.6238473019299857E-06_dp,&
      1.7957394158840545E+00_dp, -7.6574403746434661E-01_dp,&
      4.8537842200104014E-02_dp,  5.2360822807135658E-02_dp,&
      -8.3350755683544234E-03_dp, -7.7088356658967239E-03_dp,&
      1.6907326728181175E-03_dp,  1.4727004194089355E-03_dp,&
      -2.9848376058600709E-04_dp, -4.5890070262808070E-04_dp,&
      2.5968046486014553E-04_dp, -5.0492382323740779E-05_dp,&
      6.4133621780081155E-06_dp,  2.5262136301711431E+00_dp,&
      -1.2588607298899273E+00_dp,  1.5253497398909188E-01_dp,&
      8.1421469717775821E-02_dp, -2.3821436496503780E-02_dp,&
      -1.1489040350698325E-02_dp,  4.6191262169961445E-03_dp,&
      2.1284229023209771E-03_dp, -8.7863297220430183E-04_dp,&
      -6.9885410634119751E-04_dp,  5.4823184687904317E-04_dp,&
      -1.4084932910802381E-04_dp,  1.6899366991871437E-05_dp,&
      3.5123659329872625E+00_dp, -2.0346082761913720E+00_dp,&
      3.6366813808794807E-01_dp,  1.1503446620296379E-01_dp,&
      -5.3909505121735528E-02_dp, -1.4743484990619836E-02_dp,&
      1.0146826347631005E-02_dp,  2.5497745431026582E-03_dp,&
      -1.9698501663441875E-03_dp, -9.3634077876968234E-04_dp,&
      1.0547263092030720E-03_dp, -3.4197789086921887E-04_dp,&
      4.6897245033077326E-05_dp,  4.8615658543578890E+00_dp,&
      -3.2456179761655868E+00_dp,  7.6674797762756552E-01_dp,&
      1.4154002387869766E-01_dp, -1.0779042624053684E-01_dp,&
      -1.4342190696884314E-02_dp,  1.9628242867656083E-02_dp,&
      2.0123316603197143E-03_dp, -3.8036526812309686E-03_dp,&
      -1.0255285079106839E-03_dp,  1.8698313419945043E-03_dp,&
      -7.4237378807489164E-04_dp,  1.1820859561659198E-04_dp,&
      6.7243640488374226E+00_dp, -5.1241379505231510E+00_dp,&
      1.5073836478498175E+00_dp,  1.3090312459656980E-01_dp,&
      -1.9797012483116577E-01_dp, -3.5368923851968148E-03_dp,&
      3.4498661615170442E-02_dp, -9.2463996990214416E-04_dp,&
      -6.5658463176524523E-03_dp, -6.6485773777846281E-04_dp,&
      3.0624077548521331E-03_dp, -1.4675840447194452E-03_dp,&
      2.6687950020770981E-04_dp,  9.3139083034785379E+00_dp,&
      -8.0244497096949736E+00_dp,  2.8332477118834278E+00_dp,&
      1.5522414229484492E-02_dp, -3.3893468747140731E-01_dp,&
      3.0800758429289306E-02_dp,  5.5506097333631294E-02_dp,&
      -8.8311029352697995E-03_dp, -1.0194631369398816E-02_dp,&
      6.8342519451028135E-04_dp,  4.6062825095427749E-03_dp,&
      -2.6705893223239008E-03_dp,  5.4643420350779194E-04_dp,&
      1.2934585921249521E+01_dp, -1.2487983193387047E+01_dp,&
      5.1625830662902139E+00_dp, -3.4641867453590114E-01_dp,&
      -5.4137725689650251E-01_dp,  1.1230929110201276E-01_dp,&
      8.0834531173369778E-02_dp, -2.5867633057602857E-02_dp,&
      -1.3957958298784179E-02_dp,  3.8652333980646594E-03_dp,&
      6.2435692573387415E-03_dp, -4.4931052938362573E-03_dp,&
      1.0271118708257464E-03_dp,  1.8025229894947888E+01_dp,&
      -1.9343860235887981E+01_dp,  9.1975266340592636E+00_dp,&
      -1.2397685603690685E+00_dp, -7.9561029845592168E-01_dp,&
      2.8043792713585208E-01_dp,  1.0205012717274836E-01_dp,&
      -5.8031621983963111E-02_dp, -1.5652637896035132E-02_dp,&
      1.0030483755240904E-02_dp,  7.2529698214355435E-03_dp,&
      -6.9701747701425868E-03_dp,  1.7852314445813721E-03_dp,&
      2.5226686181538120E+01_dp, -2.9865961999886299E+01_dp,&
      1.6110802529841504E+01_dp, -3.2194475911241494E+00_dp,&
      -1.0303513955388184E+00_dp,  5.9474554354538078E-01_dp,&
      9.6101297590719212E-02_dp, -1.1237368378646433E-01_dp,&
      -1.0220072288871951E-02_dp,  2.0372906608218531E-02_dp,&
      6.0853027356999037E-03_dp, -9.8194206242380334E-03_dp,&
      2.8679427689706485E-03_dp,  3.5494031265018705E+01_dp,&
      -4.6025018877931025E+01_dp,  2.7852880135496353E+01_dp,&
      -7.3401774215925366E+00_dp, -1.0186566668613926E+00_dp,&
      1.1300029991052669E+00_dp,  1.1295976109331713E-02_dp,&
      -1.9301508698030923E-01_dp,  1.2241843214785244E-02_dp,&
      3.5142635468207804E-02_dp, -8.2517129158308055E-05_dp,&
      -1.2012456289332549E-02_dp,  4.1969205275104700E-03_dp,&
      5.0301713674828171E+01_dp, -7.0918826017007646E+01_dp,&
      4.7664290809984450E+01_dp, -1.5561816383882222E+01_dp,&
      -1.7868631517805517E-01_dp,  1.9389162015454402E+00_dp,&
      -2.5170989059236282E-01_dp, -2.8945054924244118E-01_dp,&
      6.7800129944182816E-02_dp,  5.1001095146721069E-02_dp,&
      -1.5261130863393309E-02_dp, -1.1165054631480001E-02_dp,&
      5.3713423174483309E-03_dp,  7.2073804790616393E+01_dp,&
      -1.0960262316704710E+02_dp,  8.0985249028988193E+01_dp,&
      -3.1454847999518755E+01_dp,  2.8131045955063794E+00_dp,&
      2.9325467224256498E+00_dp, -8.4832494632348165E-01_dp,&
      -3.5274801045453463E-01_dp,  1.7425095910591704E-01_dp,&
      5.6535696476405536E-02_dp, -4.2150084007274102E-02_dp,&
      -3.6504731904105233E-03_dp,  5.5123410746420135E-03_dp,&
      1.0525713658114117E+02_dp, -1.7104517278891666E+02_dp,&
      1.3737253723605420E+02_dp, -6.1507403720685481E+01_dp,&
      1.0708265591623199E+01_dp,  3.6132806701224829E+00_dp,&
      -1.9428164726167740E+00_dp, -2.7177296350258656E-01_dp,&
      3.3358187899034614E-01_dp,  3.1602468053086578E-02_dp,&
      -7.6640928780817658E-02_dp,  1.2767086056539591E-02_dp,&
      3.7015373482892546E-03_dp,  1.5961638373875255E+02_dp,&
      -2.7407208180371384E+02_dp,  2.3597941902719086E+02_dp,&
      -1.1831815835540151E+02_dp,  2.8965862566108601E+01_dp,&
      2.6695849787510491E+00_dp, -3.5799821757574688E+00_dp,&
      1.0012437250743024E-01_dp,  5.1714245220836208E-01_dp,&
      -3.8887168121094023E-02_dp, -1.0903222378111407E-01_dp,&
      3.6184421001653103E-02_dp,  7.4184612058038470E-06_dp,&
      2.6370713876922872E+02_dp, -4.7201989715682282E+02_dp,&
      4.2842096437204810E+02_dp, -2.3341869455214848E+02_dp,&
      6.9905383809584009E+01_dp, -2.5086723240417781E+00_dp,&
      -5.7498093848956318E+00_dp,  8.7370241169426954E-01_dp,&
      7.2690318270483367E-01_dp, -1.5862760095796691E-01_dp,&
      -1.4433107481771534E-01_dp,  6.7767335244992954E-02_dp,&
      -5.0600724217742732E-03_dp,  5.6003009143569716E+02_dp,&
      -1.0287297849968459E+03_dp,  9.6498150903106421E+02_dp,&
      -5.5317904766641254E+02_dp,  1.8492063710688871E+02_dp,&
      -1.9101040737585297E+01_dp, -1.0231770142026468E+01_dp,&
      2.5465845134577312E+00_dp,  1.2703101988094840E+00_dp,&
      -4.1883844922802427E-01_dp, -2.4642046076164414E-01_dp,&
      1.4229171344177202E-01_dp, -1.4875308764204570E-02_dp,&
      1.1946368463753128E+00_dp, -3.2495379783849715E-01_dp,&
      -3.5228366362451968E-02_dp,  1.9001587589605372E-02_dp,&
      4.2259462352662765E-03_dp, -2.6842657875635143E-03_dp,&
      -7.0387680924777728E-04_dp,  5.1324943831075869E-04_dp,&
      1.7519190242729584E-04_dp, -1.5790677102250255E-04_dp,&
      1.9118266393283089E-05_dp, -3.2127263681394069E-06_dp,&
      6.1720484771177976E-06_dp,  1.3069315340948260E+00_dp,&
      -4.1805297500996297E-01_dp, -1.9174651660622710E-02_dp,&
      2.7527354291961389E-02_dp,  1.8936163303805468E-03_dp,&
      -4.0779517864187736E-03_dp, -2.6637654979470501E-04_dp,&
      7.9164825616038234E-04_dp,  9.0616104454619150E-05_dp,&
      -2.3837680235491997E-04_dp,  6.3556401416228639E-05_dp,&
      -5.3380803695105839E-06_dp,  4.2840530293578485E-06_dp,&
      1.5445250149914536E+00_dp, -6.2287628679630180E-01_dp,&
      2.1843824089781916E-02_dp,  4.4718022389577831E-02_dp,&
      -4.1973146173572883E-03_dp, -6.7826557553283739E-03_dp,&
      8.9075466785271920E-04_dp,  1.3203001424771435E-03_dp,&
      -1.3887733251644589E-04_dp, -3.9798804282131669E-04_dp,&
      1.7705748242976088E-04_dp, -2.1402551771088614E-05_dp,&
      2.2801890600343344E-06_dp,  1.9346618851392008E+00_dp,&
      -9.7926756464428466E-01_dp,  1.0781255156226323E-01_dp,&
      7.0204267153656671E-02_dp, -1.7083979272462264E-02_dp,&
      -1.0472010647795030E-02_dp,  3.3479446665571036E-03_dp,&
      2.0046454310945101E-03_dp, -6.3478895071837540E-04_dp,&
      -6.2462188170554333E-04_dp,  4.0883505029748853E-04_dp,&
      -7.7995690482165342E-05_dp,  4.6890795156480489E-06_dp,&
      2.5218110054658807E+00_dp, -1.5547109430512907E+00_dp,&
      2.7533116442749239E-01_dp,  1.0132320908146478E-01_dp,&
      -4.1829776586315127E-02_dp, -1.4151497493277881E-02_dp,&
      8.0101836422415695E-03_dp,  2.5858991445510790E-03_dp,&
      -1.5782258678436166E-03_dp, -8.7134756801638584E-04_dp,&
      8.3107610724402121E-04_dp, -2.2325382352150771E-04_dp,&
      2.1184031900921113E-05_dp,  3.3730785337688096E+00_dp,&
      -2.4562132009879658E+00_dp,  5.8822468241512738E-01_dp,&
      1.2938256628900288E-01_dp, -8.6117944085599613E-02_dp,&
      -1.5346369501680956E-02_dp,  1.6105261132836165E-02_dp,&
      2.4662787067982008E-03_dp, -3.1947037676917276E-03_dp,&
      -1.0145572939541253E-03_dp,  1.5309117164874644E-03_dp,&
      -5.3600219225621295E-04_dp,  7.0324364856116483E-05_dp,&
      4.5868466733976998E+00_dp, -3.8497189820072570E+00_dp,&
      1.1557884102835358E+00_dp,  1.3236494031740839E-01_dp,&
      -1.6068493389320054E-01_dp, -8.7123214871505979E-03_dp,&
      2.9032643032267314E-02_dp,  4.4156030667138075E-04_dp,&
      -5.6942421453560228E-03_dp, -7.9246900794011370E-04_dp,&
      2.5835195275365408E-03_dp, -1.1316665352679273E-03_dp,&
      1.8389558466057045E-04_dp,  6.3058543134695348E+00_dp,&
      -5.9904367486606711E+00_dp,  2.1632706388868730E+00_dp,&
      6.0862754524593270E-02_dp, -2.7876465363888409E-01_dp,&
      1.6240870724100132E-02_dp,  4.7821555177730193E-02_dp,&
      -5.6736395491679926E-03_dp, -9.1125860875598368E-03_dp,&
      2.7444438602033391E-04_dp,  3.9878923020532997E-03_dp,&
      -2.1556997302423510E-03_dp,  4.1038336500161027E-04_dp,&
      8.7367679966561802E+00_dp, -9.2696858881557223E+00_dp,&
      3.9225237552290722E+00_dp, -1.8871498473285156E-01_dp,&
      -4.5266022613648998E-01_dp,  7.8678436144091410E-02_dp,&
      7.1755861676809696E-02_dp, -1.9502646657472384E-02_dp,&
      -1.2970460818387342E-02_dp,  2.9640688478987454E-03_dp,&
      5.5440927395923583E-03_dp, -3.7503139256693141E-03_dp,&
      8.1466890564677789E-04_dp,  1.2179715928272094E+01_dp,&
      -1.4287247815361980E+01_dp,  6.9559238252243123E+00_dp,&
      -8.2452423618630410E-01_dp, -6.8317025405101606E-01_dp,&
      2.1141333388888769E-01_dp,  9.5346291787819212E-02_dp,&
      -4.6499824144226799E-02_dp, -1.5623028966346148E-02_dp,&
      8.3849673127276043E-03_dp,  6.6500033547118049E-03_dp,&
      -5.9740661993351847E-03_dp,  1.4691884865455198E-03_dp,&
      1.7073926981017319E+01_dp, -2.1963630226832553E+01_dp,&
      1.2134829552891802E+01_dp, -2.2528424944623944E+00_dp,&
      -9.3240764730384440E-01_dp,  4.6660358949448766E-01_dp,&
      1.0216844176487623E-01_dp, -9.3824494827588173E-02_dp,&
      -1.3105510915164605E-02_dp,  1.7849498315239853E-02_dp /)
    REAL(KIND=dp), DIMENSION(400), PARAMETER :: c02 = (/ &
      5.9915911031709962E-03_dp, -8.6423694074923528E-03_dp,&
      2.4291300518560965E-03_dp,  2.4071366084770816E+01_dp,&
      -3.3716515170389293E+01_dp,  2.0906788438349480E+01_dp,&
      -5.2492585742829574E+00_dp, -1.0597221453010373E+00_dp,&
      9.1826115756590143E-01_dp,  5.3116568030487622E-02_dp,&
      -1.6840980447490020E-01_dp,  2.7804548003092637E-03_dp,&
      3.2324581009718446E-02_dp,  1.0845459715807160E-03_dp,&
      -1.1013651706859648E-02_dp,  3.6730611525692776E-03_dp,&
      3.4164354226130186E+01_dp, -5.1747145255461746E+01_dp,&
      3.5667082252749204E+01_dp, -1.1265379545809436E+01_dp,&
      -6.7939439898041476E-01_dp,  1.6480605577661318E+00_dp,&
      -1.3378965635839782E-01_dp, -2.6979010160400613E-01_dp,&
      4.7149105264278728E-02_dp,  5.0747602507635034E-02_dp,&
      -1.2294628821884937E-02_dp, -1.1142091315825215E-02_dp,&
      4.9501418663369838E-03_dp,  4.8931770063798176E+01_dp,&
      -7.9540390842429701E+01_dp,  6.0373542926316304E+01_dp,&
      -2.2962155467014412E+01_dp,  1.1360885415171147E+00_dp,&
      2.6763499892965861E+00_dp, -6.0987396392114657E-01_dp,&
      -3.6997629107051999E-01_dp,  1.4345000248927353E-01_dp,&
      6.5524578572070433E-02_dp, -3.9862967761504962E-02_dp,&
      -4.8299270547189094E-03_dp,  5.4310464886229816E-03_dp,&
      7.1084878218521894E+01_dp, -1.2286891263137933E+02_dp,&
      1.0163325401682647E+02_dp, -4.5111853024962613E+01_dp,&
      6.4218229411147210E+00_dp,  3.7532636523243239E+00_dp,&
      -1.6002854623816907E+00_dp, -3.7101035717591935E-01_dp,&
      3.1409505327231513E-01_dp,  5.3941115043966030E-02_dp,&
      -8.3818143472302709E-02_dp,  1.4207263645641415E-02_dp,&
      3.3546858581714827E-03_dp,  1.0588222696611157E+02_dp,&
      -1.9226833008452053E+02_dp,  1.7101798076439422E+02_dp,&
      -8.6186553476897572E+01_dp,  1.9271662529722853E+01_dp,&
      3.8930409361722291E+00_dp, -3.2922928695756104E+00_dp,&
      -6.0113623291277216E-02_dp,  5.4062487480709565E-01_dp,&
      -2.4107047161548600E-02_dp, -1.2869189718624949E-01_dp,&
      4.7717461953990542E-02_dp, -2.9342027944556193E-03_dp,&
      1.6546877079051265E+02_dp, -3.1068544904103140E+02_dp,&
      2.9180626201261043E+02_dp, -1.6205766726181861E+02_dp,&
      4.7344515303703460E+01_dp,  7.0564709632446621E-01_dp,&
      -5.4990736658749366E+00_dp,  8.1374078900166369E-01_dp,&
      7.1175171802245729E-01_dp, -1.8680907638713953E-01_dp,&
      -1.3919953678136415E-01_dp,  8.1913606331528166E-02_dp,&
      -1.1490899293904754E-02_dp,  2.8527479104831519E+02_dp,&
      -5.4324338868812106E+02_dp,  5.2597081274478194E+02_dp,&
      -3.1032555466251750E+02_dp,  1.0592128649569207E+02_dp,&
      -9.9230478811039369E+00_dp, -7.5217801155503921E+00_dp,&
      2.2260389199008581E+00_dp,  7.4389817085116938E-01_dp,&
      -3.7487661915364073E-01_dp, -1.2135071368815714E-01_dp,&
      1.0472900042735607E-01_dp, -1.7467242484406273E-02_dp,&
      6.1069908841010852E+02_dp, -1.1574002440882412E+03_dp,&
      1.1246410459943154E+03_dp, -6.7559373392291536E+02_dp,&
      2.4469690351149762E+02_dp, -3.4981011467122990E+01_dp,&
      -1.0447270088817676E+01_dp,  4.1199824733302801E+00_dp,&
      1.0753683950263284E+00_dp, -6.1667707322635523E-01_dp,&
      -1.9231863825085113E-01_dp,  1.6351050609684276E-01_dp,&
      -2.4438171008690827E-02_dp,  2.3309249136020708E+03_dp,&
      -4.3538180199326071E+03_dp,  4.1700553851275899E+03_dp,&
      -2.4659990924985104E+03_dp,  8.7681545314646689E+02_dp,&
      -1.2252110351183714E+02_dp, -3.4744515550283161E+01_dp,&
      1.1919976509846366E+01_dp,  4.5071725742760815E+00_dp,&
      -1.9152099076735201E+00_dp, -8.5859628246475261E-01_dp,&
      5.7717184691703094E-01_dp, -7.0410149927279939E-02_dp,&
      1.8682839453664454E-01_dp, -4.6619677822158430E-02_dp,&
      -1.2244969271828127E-02_dp,  6.0589256164018451E-03_dp,&
      3.6233383923308837E-03_dp, -2.1697547440741212E-03_dp,&
      -1.5324832573249004E-03_dp,  7.6307014457020446E-04_dp,&
      1.5009945716157327E-03_dp, -1.5948622920885888E-03_dp,&
      4.2721023924721354E-04_dp,  1.2452965510300625E-04_dp,&
      -6.8101344859438604E-05_dp,  5.8892228459182272E-01_dp,&
      -1.6187708429798231E-01_dp, -3.6609865098592521E-02_dp,&
      2.3522804102595687E-02_dp,  1.0989418080689319E-02_dp,&
      -8.7225886606387588E-03_dp, -4.6358129182236165E-03_dp,&
      3.1750312025791852E-03_dp,  5.1844812097367743E-03_dp,&
      -6.5275901023297469E-03_dp,  2.5192599245746250E-03_dp,&
      -2.7790386311357066E-05_dp, -1.5519525312227047E-04_dp,&
      1.0819380779186476E+00_dp, -3.5078545793786275E-01_dp,&
      -5.8317591891871842E-02_dp,  5.8961787807382718E-02_dp,&
      1.7698109526329605E-02_dp, -2.2770460521331585E-02_dp,&
      -7.3205917206429702E-03_dp,  8.6317273038198361E-03_dp,&
      1.0821379193248763E-02_dp, -1.7385782357341700E-02_dp,&
      9.1208897862528268E-03_dp, -1.7582601800537039E-03_dp,&
      -2.5708035752802444E-05_dp,  1.7461298655116124E+00_dp,&
      -6.8967176721527312E-01_dp, -6.6751042045674530E-02_dp,&
      1.3103208070134215E-01_dp,  1.9338401160822482E-02_dp,&
      -5.2191210709296623E-02_dp, -7.2199928465625264E-03_dp,&
      2.0478760930027633E-02_dp,  1.9174065370320473E-02_dp,&
      -4.0384174885238051E-02_dp,  2.6020899384681384E-02_dp,&
      -7.4705034438871248E-03_dp,  7.2143802459460730E-04_dp,&
      2.6950773766538831E+00_dp, -1.3037225406473845E+00_dp,&
      -3.2614496759563683E-02_dp,  2.6916418726799823E-01_dp,&
      3.0053439263977664E-03_dp, -1.0921441669530184E-01_dp,&
      2.4831672286212188E-03_dp,  4.4091697655829862E-02_dp,&
      2.9150030585017250E-02_dp, -8.4774815208886814E-02_dp,&
      6.3626601084289261E-02_dp, -2.2005245386043593E-02_dp,&
      2.9767021718693147E-03_dp,  4.0984375451052717E+00_dp,&
      -2.4066178073051301E+00_dp,  1.1383731173835311E-01_dp,&
      5.1909474912068099E-01_dp, -6.2603108434964416E-02_dp,&
      -2.1175654622555612E-01_dp,  3.8133384598550361E-02_dp,&
      8.7587360901696096E-02_dp,  3.4278741909182162E-02_dp,&
      -1.6260127734933916E-01_dp,  1.3944397087807067E-01_dp,&
      -5.4447820679856637E-02_dp,  8.5453383708596811E-03_dp,&
      6.2176136529295158E+00_dp, -4.3693211280877762E+00_dp,&
      5.2774370702314355E-01_dp,  9.4773880309914649E-01_dp,&
      -2.4690257552432496E-01_dp, -3.8251497933289635E-01_dp,&
      1.3533123197212421E-01_dp,  1.6155301558102667E-01_dp,&
      1.4718947044978946E-02_dp, -2.8450075717598156E-01_dp,&
      2.7982684165522875E-01_dp, -1.2045734281774570E-01_dp,&
      2.0790000977242391E-02_dp,  9.4634111053439103E+00_dp,&
      -7.8418332358592870E+00_dp,  1.5417436172364343E+00_dp,&
      1.6414265351909745E+00_dp, -6.9529406365081492E-01_dp,&
      -6.4050441706217742E-01_dp,  3.6647261823447669E-01_dp,&
      2.7522127699205079E-01_dp, -7.9228003624491242E-02_dp,&
      -4.4498374478349312E-01_dp,  5.1708841341964662E-01_dp,&
      -2.4407600743956154E-01_dp,  4.5405289677376504E-02_dp,&
      1.4490868628033503E+01_dp, -1.3971233939072121E+01_dp,&
      3.8542271282507108E+00_dp,  2.6797445232398029E+00_dp,&
      -1.6991502505639382E+00_dp, -9.7162873904940783E-01_dp,&
      8.6994103205685502E-01_dp,  4.2251973253482306E-01_dp,&
      -3.5811394605060659E-01_dp, -5.8327449682364263E-01_dp,&
      8.7181325262147957E-01_dp, -4.5547158946665350E-01_dp,&
      9.0828400398500778E-02_dp,  2.2357425070187674E+01_dp,&
      -2.4796353116963004E+01_dp,  8.9144869553283570E+00_dp,&
      4.0371985902019292E+00_dp, -3.8186248869629074E+00_dp,&
      -1.2467737038604281E+00_dp,  1.8925278595991524E+00_dp,&
      5.4265414205624996E-01_dp, -1.0448438269132569E+00_dp,&
      -4.9352647807166389E-01_dp,  1.2974717296843923E+00_dp,&
      -7.7527513958093197E-01_dp,  1.6672848898454873E-01_dp,&
      3.4790250541342665E+01_dp, -4.3969980740698468E+01_dp,&
      1.9713761658579021E+01_dp,  5.2896392037293767E+00_dp,&
      -8.0806472982724422E+00_dp, -1.0170835399174816E+00_dp,&
      3.8335251168155606E+00_dp,  4.2219424033780178E-01_dp,&
      -2.5430562807078081E+00_dp,  3.5748238853390918E-01_dp,&
      1.5419198749245833E+00_dp, -1.1644678241284623E+00_dp,&
      2.7666437341984224E-01_dp,  5.4644357897813208E+01_dp,&
      -7.8095758505940708E+01_dp,  4.2420187894560037E+01_dp,&
      4.8151968808568224E+00_dp, -1.6245875469915777E+01_dp,&
      9.6243602774289128E-01_dp,  7.2326780768599983E+00_dp,&
      -5.2544186970366702E-01_dp, -5.4789383956688651E+00_dp,&
      3.1531099403716132E+00_dp,  8.4886474149751467E-01_dp,&
      -1.3986356059167204E+00_dp,  3.9650600036767097E-01_dp,&
      8.6699804225947645E+01_dp, -1.3923361555998270E+02_dp,&
      8.9786173246194181E+01_dp, -2.2858823629011713E+00_dp,&
      -3.0992469263910380E+01_dp,  7.7741399253657599E+00_dp,&
      1.2505917232015936E+01_dp, -3.7068949594152851E+00_dp,&
      -1.0555346369807513E+01_dp,  1.0188356709429819E+01_dp,&
      -2.6001240535680141E+00_dp, -7.9853252277010678E-01_dp,&
      4.2303845984206989E-01_dp,  1.3907554680969875E+02_dp,&
      -2.4965842431321846E+02_dp,  1.8831846945865320E+02_dp,&
      -2.9489704359881337E+01_dp, -5.5316316864452965E+01_dp,&
      2.6447389036265232E+01_dp,  1.8854798651484575E+01_dp,&
      -1.2120609962757287E+01_dp, -1.7763715122527280E+01_dp,&
      2.5231831140577533E+01_dp, -1.2544891115185020E+01_dp,&
      2.2668677068936747E+00_dp,  6.8708716177258419E-02_dp,&
      2.2580720743716594E+02_dp, -4.5104097133606439E+02_dp,&
      3.9351587358059925E+02_dp, -1.1220803805233957E+02_dp,&
      -8.8584998126696050E+01_dp,  7.1690786222667725E+01_dp,&
      2.0820379520370651E+01_dp, -3.1251600385405137E+01_dp,&
      -2.3842474793325504E+01_dp,  5.2733349756860250E+01_dp,&
      -3.5442716338070284E+01_dp,  1.1104138822595329E+01_dp,&
      -1.3193636495537726E+00_dp,  3.7178643267228017E+02_dp,&
      -8.2258367035709261E+02_dp,  8.2255509467989646E+02_dp,&
      -3.4122567397482152E+02_dp, -1.1069056818136373E+02_dp,&
      1.7023549623079217E+02_dp, -1.0915097896851977E+00_dp,&
      -6.8273494641212551E+01_dp, -1.5708419656731760E+01_dp,&
      9.2205658507516148E+01_dp, -7.8945038472936645E+01_dp,&
      3.0974630589084953E+01_dp, -4.9401496535210629E+00_dp,&
      6.2326881452254656E+02_dp, -1.5188270356173509E+03_dp,&
      1.7254522561739329E+03_dp, -9.4519869568904846E+02_dp,&
      -2.7999063550782221E+01_dp,  3.5481857290613175E+02_dp,&
      -1.0099523826322665E+02_dp, -1.2123045796995562E+02_dp,&
      4.0970797552648818E+01_dp,  1.1891202216538902E+02_dp,&
      -1.3972808947295965E+02_dp,  6.5322452115099665E+01_dp,&
      -1.2075994070187011E+01_dp,  1.0756923535067231E+03_dp,&
      -2.8616598027309929E+03_dp,  3.6507795405336346E+03_dp,&
      -2.4909049195091211E+03_dp,  4.8798405235788942E+02_dp,&
      5.9835790021628566E+02_dp, -3.8831159871198366E+02_dp,&
      -1.3665662485667869E+02_dp,  1.9320859447294200E+02_dp,&
      5.7805579962611311E+01_dp, -1.7452945217439253E+02_dp,&
      1.0228317193023724E+02_dp, -2.1552120617229455E+01_dp,&
      1.9817023447702200E+03_dp, -5.6689826598340678E+03_dp,&
      7.9668997146417360E+03_dp, -6.4333249134176367E+03_dp,&
      2.4219830207338719E+03_dp,  5.7967220853445599E+02_dp,&
      -9.5984621210825696E+02_dp,  3.2978769612793030E+01_dp,&
      4.1917841670563894E+02_dp, -1.5534544314105773E+02_dp,&
      -1.2421868756049768E+02_dp,  1.1843742614679046E+02_dp,&
      -2.9556703864728348E+01_dp,  4.5306956901767944E+03_dp,&
      -1.3608434324291762E+04_dp,  2.0416922578512735E+04_dp,&
      -1.8375713453892236E+04_dp,  9.1917215327749964E+03_dp,&
      -7.4781042033255528E+02_dp, -1.8977895694601118E+03_dp,&
      5.0289039209206737E+02_dp,  7.8898968358442164E+02_dp,&
      -5.7658145419403400E+02_dp, -1.4194226893666439E+01_dp,&
      1.4827723853173043E+02_dp, -4.5635215304885165E+01_dp,&
      1.5132189223523540E+00_dp, -3.8714502724407229E-01_dp,&
      -9.8018889420702443E-02_dp,  5.1908250545652360E-02_dp,&
      2.9129428048283938E-02_dp, -1.8776985333903432E-02_dp,&
      -1.2318593850205199E-02_dp,  6.6695589588984475E-03_dp,&
      1.2466570600707088E-02_dp, -1.3874197741980808E-02_dp,&
      4.1907856070478042E-03_dp,  7.5271368574731106E-04_dp,&
      -5.1647877466666774E-04_dp,  1.7444005611942730E+00_dp,&
      -5.6824316669549102E-01_dp, -9.5567270387908632E-02_dp,&
      9.5953975937291464E-02_dp,  2.9440932840795256E-02_dp,&
      -3.7050946327260699E-02_dp, -1.2287347330040912E-02_dp /)
    REAL(KIND=dp), DIMENSION(400), PARAMETER :: c03 = (/ &
      1.4012464467116794E-02_dp,  1.7870937744543961E-02_dp,&
      -2.8284778305657239E-02_dp,  1.4614940139686226E-02_dp,&
      -2.7014027690474133E-03_dp, -7.8952988099918739E-05_dp,&
      2.2522270551584125E+00_dp, -9.9237513069804251E-01_dp,&
      -7.1665511461607434E-02_dp,  1.9968203248776958E-01_dp,&
      2.1586264982050893E-02_dp, -8.0464217361784704E-02_dp,&
      -7.7695595383537727E-03_dp,  3.1814347737576659E-02_dp,&
      2.7811012184441854E-02_dp, -6.2649965060119633E-02_dp,&
      4.1918544103320216E-02_dp, -1.2666876155687441E-02_dp,&
      1.3660025331692276E-03_dp,  3.1355625943697762E+00_dp,&
      -1.7996536940652996E+00_dp,  2.2381691245553063E-02_dp,&
      3.9612212867481145E-01_dp, -1.6449727468430150E-02_dp,&
      -1.6295147202484259E-01_dp,  1.2800103481059741E-02_dp,&
      6.6491589187543318E-02_dp,  3.8496472917503681E-02_dp,&
      -1.2749901018687695E-01_dp,  1.0004618939892632E-01_dp,&
      -3.6137844427782950E-02_dp,  5.1770195187856929E-03_dp,&
      4.5668263189960809E+00_dp, -3.2508325919683241E+00_dp,&
      2.9442091272203441E-01_dp,  7.3905390144916405E-01_dp,&
      -1.3384430189810520E-01_dp, -3.0519473426292121E-01_dp,&
      7.5103144196923433E-02_dp,  1.2781992372799392E-01_dp,&
      3.7492879106017292E-02_dp, -2.3587781768150057E-01_dp,&
      2.1232133279926324E-01_dp, -8.6033979418852449E-02_dp,&
      1.4025330806284918E-02_dp,  6.8306176002396661E+00_dp,&
      -5.8119086384988803E+00_dp,  9.7490754567894877E-01_dp,&
      1.3049740304463888E+00_dp, -4.3490563945641791E-01_dp,&
      -5.3072489778072540E-01_dp,  2.3270303823013319E-01_dp,&
      2.2740302928499284E-01_dp, -8.6507435588130746E-03_dp,&
      -3.9468254647236639E-01_dp,  4.1189269031114867E-01_dp,&
      -1.8379654833529319E-01_dp,  3.2710121302165010E-02_dp,&
      1.0390575731969639E+01_dp, -1.0309231722090752E+01_dp,&
      2.5464315850199029E+00_dp,  2.1830827795271524E+00_dp,&
      -1.1319594654536267E+00_dp, -8.4971117936736751E-01_dp,&
      5.9122626817509949E-01_dp,  3.7127494548695916E-01_dp,&
      -1.7896260036582676E-01_dp, -5.7922303833023392E-01_dp,&
      7.3368980026983677E-01_dp, -3.6037859396806582E-01_dp,&
      6.9003335230075069E-02_dp,  1.6002364677374977E+01_dp,&
      -1.8210615137406258E+01_dp,  6.0073025151814532E+00_dp,&
      3.4250138514203066E+00_dp, -2.6414806725325550E+00_dp,&
      -1.2096574762879975E+00_dp,  1.3472659206289432E+00_dp,&
      5.3635873100939280E-01_dp, -6.4288697341708967E-01_dp,&
      -6.6970318883021285E-01_dp,  1.1814039303496457E+00_dp,&
      -6.4954054414694495E-01_dp,  1.3361067762784279E-01_dp,&
      2.4905165004773185E+01_dp, -3.2138208303483374E+01_dp,&
      1.3409824399537735E+01_dp,  4.8815305907327238E+00_dp,&
      -5.7496874849540909E+00_dp, -1.3655975920609458E+00_dp,&
      2.8416424417161545E+00_dp,  6.0744001443999063E-01_dp,&
      -1.7319413268413426E+00_dp, -3.1102100539977884E-01_dp,&
      1.6338737133746211E+00_dp, -1.0583411311910322E+00_dp,&
      2.3663443320973138E-01_dp,  3.9149516015116362E+01_dp,&
      -5.6813706285333311E+01_dp,  2.8965448354605932E+01_dp,&
      5.7421686338785047E+00_dp, -1.1871980500371624E+01_dp,&
      -5.6543446935291142E-01_dp,  5.6045107680728776E+00_dp,&
      2.2503023424962715E-01_dp, -4.0269224337355114E+00_dp,&
      1.3631950103938133E+00_dp,  1.6221634173962662E+00_dp,&
      -1.4818038878243160E+00_dp,  3.7423258125912723E-01_dp,&
      6.2165735856370389E+01_dp, -1.0082025924844557E+02_dp,&
      6.1330857086424636E+01_dp,  3.3187227725079316E+00_dp,&
      -2.3380689687069282E+01_dp,  3.1683638576480098E+00_dp,&
      1.0296270093353964E+01_dp, -1.5512204348401999E+00_dp,&
      -8.3821659603468390E+00_dp,  6.2014658906591276E+00_dp,&
      -1.3435796755846094E-01_dp, -1.4991994694273796E+00_dp,&
      4.9235900829999746E-01_dp,  9.9764309320929868E+01_dp,&
      -1.7992761720012572E+02_dp,  1.2838124909734995E+02_dp,&
      -1.0067710243253384E+01_dp, -4.3747416798516454E+01_dp,&
      1.4588791418335694E+01_dp,  1.7234209303970911E+01_dp,&
      -6.9157222786756467E+00_dp, -1.5616463216416106E+01_dp,&
      1.7661695341240645E+01_dp, -6.5632441414580995E+00_dp,&
      4.6107861550632338E-02_dp,  4.0429578336215916E-01_dp,&
      1.6192996213509178E+02_dp, -3.2345183259818674E+02_dp,&
      2.6728260309780984E+02_dp, -5.4934721948077964E+01_dp,&
      -7.6420235817495239E+01_dp,  4.4341723962570370E+01_dp,&
      2.4617318327247116E+01_dp, -2.0464011397274241E+01_dp,&
      -2.5136972926424946E+01_dp,  4.1157382314364590E+01_dp,&
      -2.3454249403793014E+01_dp,  5.7905373012948962E+00_dp,&
      -3.7151924625073762E-01_dp,  2.6612804981637356E+02_dp,&
      -5.8660374248587323E+02_dp,  5.5597284125538818E+02_dp,&
      -1.8432038838199645E+02_dp, -1.1859595383908474E+02_dp,&
      1.1449618036240184E+02_dp,  2.3096079841726318E+01_dp,&
      -5.0496124411414627E+01_dp, -3.0760587855042580E+01_dp,&
      8.2441989484414663E+01_dp, -6.0440740405265736E+01_dp,&
      2.0892064007476247E+01_dp, -2.8821707975965407E+00_dp,&
      4.4369669215955679E+02_dp, -1.0749724494514057E+03_dp,&
      1.1593969816222766E+03_dp, -5.3302317687626135E+02_dp,&
      -1.3715111027236267E+02_dp,  2.6529860970775599E+02_dp,&
      -1.9004859388517797E+01_dp, -1.0782865542385048E+02_dp,&
      -1.0024949426649870E+01_dp,  1.3842513763413518E+02_dp,&
      -1.2847895610869600E+02_dp,  5.3470240396583165E+01_dp,&
      -9.0328668602524544E+00_dp,  7.5346534185182429E+02_dp,&
      -1.9952392505299072E+03_dp,  2.4299447215024820E+03_dp,&
      -1.4396109992918582E+03_dp,  1.3298234472702070E+01_dp,&
      5.4802453254041689E+02_dp, -1.9023333858956633E+02_dp,&
      -1.8964738445274472E+02_dp,  9.7587567615069020E+01_dp,&
      1.6354143452048581E+02_dp, -2.1971374673088104E+02_dp,&
      1.0879392628260086E+02_dp, -2.0963583502054494E+01_dp,&
      1.3170419393603040E+03_dp, -3.7739560335424503E+03_dp,&
      5.1335488235209250E+03_dp, -3.7408311327837364E+03_dp,&
      8.4211641182365474E+02_dp,  9.2551814404559434E+02_dp,&
      -6.8464014949108071E+02_dp, -2.0411488709587312E+02_dp,&
      3.9392227255553178E+02_dp, -2.7256439657068738E+00_dp,&
      -2.3813864878020763E+02_dp,  1.5893698689400233E+02_dp,&
      -3.5584326778431219E+01_dp,  2.4452352059405862E+03_dp,&
      -7.4377745780056084E+03_dp,  1.1067488576469721E+04_dp,&
      -9.4896153343244987E+03_dp,  3.8782719555563267E+03_dp,&
      8.3846279328892012E+02_dp, -1.6963477989104424E+03_dp,&
      1.9083731740985843E+02_dp,  8.1093174916219630E+02_dp,&
      -5.4226175427579835E+02_dp,  1.2518886616972225E+01_dp,&
      1.1490785535678236E+02_dp, -3.6491764381788030E+01_dp,&
      5.3330085765150261E+03_dp, -1.6666877738151037E+04_dp,&
      2.6076020406118962E+04_dp, -2.4677010132749703E+04_dp,&
      1.3307416777568624E+04_dp, -1.6319434420381626E+03_dp,&
      -2.8258156451600034E+03_dp,  1.2347677417808713E+03_dp,&
      9.6224967002132507E+02_dp, -1.1349195336895932E+03_dp,&
      3.5716055241575810E+02_dp,  3.3514646317404001E+01_dp,&
      -3.1835363222344533E+01_dp,  1.9791355268743882E+04_dp,&
      -6.1293055686968582E+04_dp,  9.5680928609287061E+04_dp,&
      -9.1568459060692185E+04_dp,  5.2201017929389076E+04_dp,&
      -1.1121707130041303E+04_dp, -5.9896108180577212E+03_dp,&
      2.8711813866255552E+03_dp,  2.8442192738211143E+03_dp,&
      -2.8748668019934485E+03_dp,  5.5449597276475947E+02_dp,&
      3.3697546690239642E+02_dp, -1.4001963907445077E+02_dp,&
      2.1435457880133899E-01_dp, -3.0445050547489378E-02_dp,&
      -1.1502220181206861E-02_dp,  3.3927624439129855E-03_dp,&
      3.4189557542295174E-03_dp, -1.2218199554707497E-03_dp,&
      -1.5066279978736984E-03_dp,  3.5867793931900605E-04_dp,&
      1.4303870829534720E-03_dp, -1.1751047917745799E-03_dp,&
      1.1448519522410556E-04_dp,  2.2484723844012693E-04_dp,&
      -7.9625602391486381E-05_dp,  6.8669139798068002E-01_dp,&
      -1.1070670760471460E-01_dp, -3.8815146888622191E-02_dp,&
      1.4421740546154914E-02_dp,  1.1981770590339011E-02_dp,&
      -5.3869891046450912E-03_dp, -5.3259041256285061E-03_dp,&
      1.6932851375693670E-03_dp,  5.4587177464697767E-03_dp,&
      -5.1318888796206240E-03_dp,  1.1113119925963892E-03_dp,&
      5.5707042656183924E-04_dp, -2.5031832857251932E-04_dp,&
      1.3017958498163928E+00_dp, -2.5836549390329550E-01_dp,&
      -7.9522740248279142E-02_dp,  4.0966970905658096E-02_dp,&
      2.5961846371615496E-02_dp, -1.5996708083253323E-02_dp,&
      -1.1668689158977025E-02_dp,  5.3987213548460644E-03_dp,&
      1.3553939593849051E-02_dp, -1.5161416199164019E-02_dp,&
      5.2920214407258222E-03_dp,  2.4789363134447497E-04_dp,&
      -4.1875707155075752E-04_dp,  2.1973724796967060E+00_dp,&
      -5.5274701142067917E-01_dp, -1.4357245126583923E-01_dp,&
      1.0384774390949460E-01_dp,  4.9473349286753690E-02_dp,&
      -4.2206845894195689E-02_dp, -2.2382617295611081E-02_dp,&
      1.5059572073191307E-02_dp,  3.0398772810786283E-02_dp,&
      -4.0085034231927559E-02_dp,  1.8281218393724739E-02_dp,&
      -2.3368359760019852E-03_dp, -4.1946031605447358E-04_dp,&
      3.5872393736741892E+00_dp, -1.1397626678673476E+00_dp,&
      -2.3978729107103397E-01_dp,  2.4474181628521541E-01_dp,&
      8.5876014179872409E-02_dp, -1.0302284902670406E-01_dp,&
      -3.8644600170405448E-02_dp,  3.8463608204633855E-02_dp,&
      6.3915061285770847E-02_dp, -9.8422842593201207E-02_dp,&
      5.3458921537296901E-02_dp, -1.1737485346159822E-02_dp,&
      3.5223316761622068E-04_dp,  5.8230089184481608E+00_dp,&
      -2.2981115043123683E+00_dp, -3.6708206680875766E-01_dp,&
      5.4793630683732597E-01_dp,  1.3260167763700370E-01_dp,&
      -2.3785074235442757E-01_dp, -5.7719416353132290E-02_dp,&
      9.2450227914151389E-02_dp,  1.2669173511185230E-01_dp,&
      -2.2861142653418301E-01_dp,  1.4176511400260272E-01_dp,&
      -3.9625027891297522E-02_dp,  3.6743327870223182E-03_dp,&
      9.5024717953790585E+00_dp, -4.5698286444549199E+00_dp,&
      -4.8749963554224052E-01_dp,  1.1830931580696589E+00_dp,&
      1.6497571972527433E-01_dp, -5.2711345036646928E-01_dp,&
      -6.2998011226482770E-02_dp,  2.1284167669799156E-01_dp,&
      2.3538540644563008E-01_dp, -5.0792488101412725E-01_dp,&
      3.5245039897125219E-01_dp, -1.1446896115953678E-01_dp,&
      1.4289249240853091E-02_dp,  1.5666042754500245E+01_dp,&
      -9.0233915532177953E+00_dp, -4.5159576117067335E-01_dp,&
      2.4887534553074904E+00_dp,  9.0670470640627859E-02_dp,&
      -1.1312846755872035E+00_dp,  3.8111343009444320E-03_dp,&
      4.7424674847483844E-01_dp,  4.0054202834589053E-01_dp,&
      -1.0842444122478243E+00_dp,  8.3507352445861216E-01_dp,&
      -3.0284416515089485E-01_dp,  4.4096366950424772E-02_dp,&
      2.6161032398783654E+01_dp, -1.7794137065441063E+01_dp,&
      1.9957922278720683E-01_dp,  5.1325134511234731E+00_dp,&
      -3.7238280322014222E-01_dp, -2.3605015078492113E+00_dp,&
      3.1646959944127984E-01_dp,  1.0276213558669789E+00_dp,&
      5.8276774951493282E-01_dp, -2.2183031836814724E+00_dp,&
      1.8984835446720232E+00_dp, -7.5410521451439838E-01_dp,&
      1.2153634803586906E-01_dp,  4.4332125750992894E+01_dp,&
      -3.5216911762790389E+01_dp,  2.7515312485019741E+00_dp,&
      1.0407651938533359E+01_dp, -2.0146926217048411E+00_dp,&
      -4.7839201365532205E+00_dp,  1.3535664467773134E+00_dp,&
      2.1644804230399943E+00_dp,  5.4450764825895215E-01_dp,&
      -4.2990170260660685E+00_dp,  4.1392834856622303E+00_dp,&
      -1.7868794706520530E+00_dp,  3.1134654318639488E-01_dp,&
      7.6360718814403441E+01_dp, -7.0253763630464462E+01_dp,&
      1.0664558219640512E+01_dp,  2.0743007038958023E+01_dp,&
      -6.9544924163722843E+00_dp, -9.3467664173592819E+00_dp,&
      4.3712655485208352E+00_dp,  4.3994628592890503E+00_dp,&
      -5.9790313958115127E-01_dp, -7.6609316918298695E+00_dp,&
      8.5784839688301471E+00_dp, -4.0319809725090563E+00_dp,&
      7.5205840776261002E-01_dp,  1.3392877071998257E+02_dp,&
      -1.4183042758875911E+02_dp,  3.3121435990401395E+01_dp,&
      4.0426295721862452E+01_dp, -2.0721850212906318E+01_dp,&
      -1.7263628579210469E+01_dp,  1.2547672408466081E+01_dp,&
      8.4591865266308197E+00_dp, -5.6249151706448863E+00_dp,&
      -1.1592672327907211E+01_dp,  1.6508808139297699E+01_dp,&
      -8.5771693846067460E+00_dp,  1.7125060439946929E+00_dp,&
      2.3966601282404417E+02_dp, -2.9090772769614887E+02_dp,&
      9.4528375241536281E+01_dp,  7.5979855023555004E+01_dp /)
    REAL(KIND=dp), DIMENSION(400), PARAMETER :: c04 = (/ &
      -5.7499561390798192E+01_dp, -2.8667295698939792E+01_dp,&
      3.3623712254987076E+01_dp,  1.4613232172192909E+01_dp,&
      -2.2425748408516260E+01_dp, -1.0625183610714041E+01_dp,&
      2.7811282349984850E+01_dp, -1.6721270864181005E+01_dp,&
      3.6190172602275243E+00_dp,  4.3864077011236401E+02_dp,&
      -6.0869407672507202E+02_dp,  2.6061415383600962E+02_dp,&
      1.3295484378240570E+02_dp, -1.5302514848145449E+02_dp,&
      -3.6108018483517967E+01_dp,  8.5434219515898235E+01_dp,&
      1.9050431653711076E+01_dp, -7.1236044045457149E+01_dp,&
      1.7168053422465423E+01_dp,  3.3523780444104382E+01_dp,&
      -2.7691407702644518E+01_dp,  6.7871470608408817E+00_dp,&
      8.2347485996135970E+02_dp, -1.3050923635822317E+03_dp,&
      7.1254991634726696E+02_dp,  1.9443051128113493E+02_dp,&
      -3.9477087573253618E+02_dp,  1.1049102653040705E+00_dp,&
      2.0471622937020155E+02_dp, -1.0305127225062771E+00_dp,&
      -1.9658691426799172E+02_dp,  1.4177328838717565E+02_dp,&
      -9.4534452677625325E+00_dp, -2.8803213147124975E+01_dp,&
      9.8371070114488415E+00_dp,  1.5915710687424632E+03_dp,&
      -2.8820355770147166E+03_dp,  1.9663633983704333E+03_dp,&
      1.1799337397837300E+02_dp, -9.8270228323130664E+02_dp,&
      2.4974930541290666E+02_dp,  4.4552578036203141E+02_dp,&
      -1.4025777789091055E+02_dp, -4.6557102028422628E+02_dp,&
      5.4786103049593976E+02_dp, -2.4864894143143812E+02_dp,&
      3.6996297729905450E+01_dp,  3.6444195240281498E+00_dp,&
      3.1826214065789259E+03_dp, -6.5947551051531327E+03_dp,&
      5.5553678841919664E+03_dp, -8.2212981706636367E+02_dp,&
      -2.2837692981734585E+03_dp,  1.3151448072342987E+03_dp,&
      7.6228584365461666E+02_dp, -7.2065304049405756E+02_dp,&
      -8.2516821940278430E+02_dp,  1.5841227049850456E+03_dp,&
      -1.0731502300722150E+03_dp,  3.5302215207929873E+02_dp,&
      -4.6365877199563009E+01_dp,  6.6383831975153153E+03_dp,&
      -1.5758014247231216E+04_dp,  1.6270823641939674E+04_dp,&
      -5.9638112817090259E+03_dp, -4.3121794406394965E+03_dp,&
      5.1144301182847603E+03_dp,  1.8462464529373887E+02_dp,&
      -2.5218095815008292E+03_dp, -2.3054431334728457E+02_dp,&
      3.1514734149831415E+03_dp, -2.9921625738998023E+03_dp,&
      1.2837985283984451E+03_dp, -2.2471322113598779E+02_dp,&
      1.4761997601240251E+04_dp, -3.9939749414678743E+04_dp,&
      5.0012213870608291E+04_dp, -3.0242466715096940E+04_dp,&
      -1.1894217165033751E+03_dp,  1.5195501545453046E+04_dp,&
      -6.8048688127790801E+03_dp, -4.7801527082331741E+03_dp,&
      5.1277990028047161E+03_dp,  1.1223380413403229E+03_dp,&
      -4.0040636605906966E+03_dp,  2.3860787033124807E+03_dp,&
      -5.0750360903198020E+02_dp,  3.9204413377125406E+04_dp,&
      -1.1769916409401350E+05_dp,  1.7194986056957813E+05_dp,&
      -1.4277204840178991E+05_dp,  5.2345034138577081E+04_dp,&
      1.9880789878096351E+04_dp, -2.9425779677239327E+04_dp,&
      3.4033464159393416E+03_dp,  1.2155031462654326E+04_dp,&
      -7.1247822519758038E+03_dp, -1.0526634908337862E+03_dp,&
      2.3358638291647071E+03_dp, -6.6372768936816408E+02_dp,&
      1.7431528419766495E+00_dp, -2.5588307542896549E-01_dp,&
      -9.4854900124262329E-02_dp,  2.9794665014952194E-02_dp,&
      2.8483247338140741E-02_dp, -1.0841558810136384E-02_dp,&
      -1.2582020468356882E-02_dp,  3.2484447101857397E-03_dp,&
      1.2184683287913497E-02_dp, -1.0393043078939668E-02_dp,&
      1.3734683830726878E-03_dp,  1.7378837582184665E-03_dp,&
      -6.4675618806164781E-04_dp,  2.0999282096752392E+00_dp,&
      -4.1684702634050547E-01_dp, -1.2966447121561134E-01_dp,&
      6.5531871892109725E-02_dp,  4.2460205357563126E-02_dp,&
      -2.5433368014272466E-02_dp, -1.9110254921504966E-02_dp,&
      8.5170411027574670E-03_dp,  2.1895399268815847E-02_dp,&
      -2.4046927930312594E-02_dp,  8.0713367078222315E-03_dp,&
      6.1894677827445367E-04_dp, -7.1620971319254038E-04_dp,&
      2.9103286794502852E+00_dp, -8.1511239504650834E-01_dp,&
      -2.0272420121372503E-01_dp,  1.6000857299250551E-01_dp,&
      7.2180276628084461E-02_dp, -6.5213326048744386E-02_dp,&
      -3.2904145344168502E-02_dp,  2.3370721007330255E-02_dp,&
      4.5890422026197968E-02_dp, -6.1738612510888358E-02_dp,&
      2.8848028203159046E-02_dp, -4.0746891743398553E-03_dp,&
      -5.3779869066474179E-04_dp,  4.3929184784296806E+00_dp,&
      -1.6330912689205386E+00_dp, -3.1604791770547475E-01_dp,&
      3.6946636122180865E-01_dp,  1.1843930520825167E-01_dp,&
      -1.5647910975653245E-01_dp, -5.3772992985622166E-02_dp,&
      5.8807839980255634E-02_dp,  9.4040122713274651E-02_dp,&
      -1.4935257927475140E-01_dp,  8.3382587423705509E-02_dp,&
      -1.9374603363277509E-02_dp,  8.9154656100587327E-04_dp,&
      6.9535647507228511E+00_dp, -3.2407207810334469E+00_dp,&
      -4.5450216314505965E-01_dp,  8.1120311488282126E-01_dp,&
      1.7199729289635310E-01_dp, -3.5502288370822022E-01_dp,&
      -7.5071517391571155E-02_dp,  1.3903228386390781E-01_dp,&
      1.8246528140825818E-01_dp, -3.4175970207501360E-01_dp,&
      2.1737747326196696E-01_dp, -6.3021723009680411E-02_dp,&
      6.3596358141532021E-03_dp,  1.1318724501442688E+01_dp,&
      -6.3711340630689817E+00_dp, -5.4290267282805349E-01_dp,&
      1.7222212774009904E+00_dp,  1.8800546629110781E-01_dp,&
      -7.7473483399344423E-01_dp, -6.8074344065584824E-02_dp,&
      3.1549770235302227E-01_dp,  3.3134230367045064E-01_dp,&
      -7.4915846108706863E-01_dp,  5.3259600474652258E-01_dp,&
      -1.7774557823373618E-01_dp,  2.3124115670092340E-02_dp,&
      1.8787690401309099E+01_dp, -1.2481591332976498E+01_dp,&
      -3.3240738329162378E-01_dp,  3.5746152138023852E+00_dp,&
      1.6582305790403686E-02_dp, -1.6418374158804947E+00_dp,&
      6.0369727099797245E-02_dp,  6.9497178729859110E-01_dp,&
      5.4759326848085665E-01_dp, -1.5808873877659888E+00_dp,&
      1.2475010854139670E+00_dp, -4.6259424760260659E-01_dp,&
      6.9150781216906557E-02_dp,  3.1709193194004314E+01_dp,&
      -2.4500936384364149E+01_dp,  8.9530424805676101E-01_dp,&
      7.2967750917633403E+00_dp, -7.8033873418639665E-01_dp,&
      -3.3908784872316935E+00_dp,  5.7873892445767738E-01_dp,&
      1.4926423753313225E+00_dp,  7.5568258108913688E-01_dp,&
      -3.2012379368822348E+00_dp,  2.8106680717115542E+00_dp,&
      -1.1384202117706752E+00_dp,  1.8702710303308939E-01_dp,&
      5.4397061089613771E+01_dp, -4.8406686443798705E+01_dp,&
      5.0922674916099249E+00_dp,  1.4682021374582865E+01_dp,&
      -3.4028187575347335E+00_dp, -6.8129880929714179E+00_dp,&
      2.2134139481608028E+00_dp,  3.1225310594685567E+00_dp,&
      5.6939022073493750E-01_dp, -6.1380258185628369E+00_dp,&
      6.0832724690905824E+00_dp, -2.6743583995551066E+00_dp,&
      4.7321157108309642E-01_dp,  9.4927520965517601E+01_dp,&
      -9.6626388107481489E+01_dp,  1.7423709956011194E+01_dp,&
      2.9087093576015224E+01_dp, -1.1024713061645821E+01_dp,&
      -1.3204431435260902E+01_dp,  6.8519114491064999E+00_dp,&
      6.3123437593929230E+00_dp, -1.3914514684572890E+00_dp,&
      -1.0781645944177450E+01_dp,  1.2524886415280703E+01_dp,&
      -5.9961542854634038E+00_dp,  1.1336014805592638E+00_dp,&
      1.6875369326167805E+02_dp, -1.9555844833588057E+02_dp,&
      5.1452978374839745E+01_dp,  5.6408183065757186E+01_dp,&
      -3.1869524561997707E+01_dp, -2.4169756910093785E+01_dp,&
      1.9246480032110174E+01_dp,  1.2080073322081908E+01_dp,&
      -9.4081964481552394E+00_dp, -1.5842846845631168E+01_dp,&
      2.3929916792690261E+01_dp, -1.2694153793065942E+01_dp,&
      2.5676754426641781E+00_dp,  3.0617464355184120E+02_dp,&
      -4.0267737873989802E+02_dp,  1.4297706304699409E+02_dp,&
      1.0552060360349593E+02_dp, -8.6951847266308548E+01_dp,&
      -3.9583498741809187E+01_dp,  5.0971829663593091E+01_dp,&
      2.0756769842847500E+01_dp, -3.5608237890535712E+01_dp,&
      -1.2635454822619382E+01_dp,  3.9827896269018439E+01_dp,&
      -2.4641903351275278E+01_dp,  5.4102958418032170E+00_dp,&
      5.6824683386744607E+02_dp, -8.4670140808710323E+02_dp,&
      3.8789802089036772E+02_dp,  1.8361314535314401E+02_dp,&
      -2.2924223071701195E+02_dp, -4.8028389813997478E+01_dp,&
      1.2888310185706845E+02_dp,  2.6716683644805315E+01_dp,&
      -1.1141263626421915E+02_dp,  3.3081570439399378E+01_dp,&
      4.6048288731971617E+01_dp, -4.0522118414987965E+01_dp,&
      1.0132031238850658E+01_dp,  1.0818700689018240E+03_dp,&
      -1.8254063981441693E+03_dp,  1.0493135940933003E+03_dp,&
      2.6594497727926023E+02_dp, -5.8924539721155190E+02_dp,&
      1.0644611347412503E+01_dp,  3.0960803248045971E+02_dp,&
      -3.5086367758831214E+00_dp, -3.0809815313509745E+02_dp,&
      2.3143304305545996E+02_dp, -2.4814271428416337E+01_dp,&
      -4.0649889869082884E+01_dp,  1.4627459402254191E+01_dp,&
      2.1203027222125434E+03_dp, -4.0538804318582847E+03_dp,&
      2.8727141030336588E+03_dp,  1.5342090598361241E+02_dp,&
      -1.4719332795253706E+03_dp,  3.8093938541663044E+02_dp,&
      6.8480755414744362E+02_dp, -2.1085331534342177E+02_dp,&
      -7.4477742346268985E+02_dp,  8.8210214793824537E+02_dp,&
      -4.0743642895084110E+02_dp,  6.4513827918347346E+01_dp,&
      4.7653632712549898E+00_dp,  4.2971394915069723E+03_dp,&
      -9.3250590824070587E+03_dp,  8.0591228301709589E+03_dp,&
      -1.1497631929121603E+03_dp, -3.4872799439592382E+03_dp,&
      1.9571861069180222E+03_dp,  1.2474018707737353E+03_dp,&
      -1.0953820294698519E+03_dp, -1.4138330704595035E+03_dp,&
      2.6149313818024643E+03_dp, -1.7568818980540248E+03_dp,&
      5.7462043816387722E+02_dp, -7.4876139865788033E+01_dp,&
      9.0670446226651657E+03_dp, -2.2366825283029859E+04_dp,&
      2.3435539043591565E+04_dp, -8.1542691185909571E+03_dp,&
      -7.1386507741248670E+03_dp,  7.7340914601856848E+03_dp,&
      8.4604251076613593E+02_dp, -4.0914375105218251E+03_dp,&
      -9.5634197023942659E+02_dp,  5.8324252454260395E+03_dp,&
      -5.3006063534238547E+03_dp,  2.2296251828676996E+03_dp,&
      -3.8479712431583698E+02_dp,  2.0208180308777472E+04_dp,&
      -5.6497936633236641E+04_dp,  7.1357135395308476E+04_dp,&
      -4.1006680278242755E+04_dp, -6.7704662049699600E+03_dp,&
      2.5578407436596837E+04_dp, -8.5907976304829099E+03_dp,&
      -1.0696042935228554E+04_dp,  8.5873059932908618E+03_dp,&
      4.3094498446251810E+03_dp, -9.2350938585649274E+03_dp,&
      5.1924066610847412E+03_dp, -1.0815797042462589E+03_dp,&
      5.0205247084381823E+04_dp, -1.5546532286454976E+05_dp,&
      2.3094943138771786E+05_dp, -1.8852660057995931E+05_dp,&
      5.4423083001147803E+04_dp,  5.2087747303112206E+04_dp,&
      -5.7021055651190843E+04_dp,  1.6354776707089343E+03_dp,&
      3.3477410681770663E+04_dp, -2.5610268529159377E+04_dp,&
      6.0980258016920243E+03_dp,  1.4484722062152532E+03_dp,&
      -7.8447165419517830E+02_dp,  1.8887929346820957E+05_dp,&
      -6.0391416820596519E+05_dp,  9.6236000817517866E+05_dp,&
      -9.2625010870439373E+05_dp,  5.0796682925588451E+05_dp,&
      -6.3746047988236256E+04_dp, -1.1255310501935899E+05_dp,&
      5.5203234544571715E+04_dp,  2.9119699905471261E+04_dp,&
      -3.7170133622062072E+04_dp,  9.5002199317095183E+03_dp,&
      3.0563342511347387E+03_dp, -1.5546850581139452E+03_dp,&
      2.2978154445120763E-01_dp, -1.5786385884921532E-02_dp,&
      -8.6068171899530838E-03_dp,  1.2219346454486037E-03_dp,&
      1.9728968192239591E-03_dp,  4.8787622431389017E-06_dp,&
      -1.2138585236970429E-03_dp,  2.8402149565212066E-04_dp,&
      5.0364218620146115E-04_dp, -2.0104688190957889E-04_dp,&
      -2.3690088629292757E-04_dp,  2.2016840122659345E-04_dp,&
      -5.5777402901406545E-05_dp,  7.4358607798286502E-01_dp,&
      -5.9062732873971499E-02_dp, -3.1312168545837522E-02_dp,&
      5.5131899950392558E-03_dp,  7.4627215090000507E-03_dp,&
      -2.9791391683538964E-04_dp, -4.6367875641200578E-03_dp,&
      1.2011301801606153E-03_dp,  2.0057385312576845E-03_dp,&
      -9.6341439171540230E-04_dp, -7.8097734459690806E-04_dp,&
      8.0779389096505641E-04_dp, -2.1048482828717736E-04_dp,&
      1.4375579606459543E+00_dp, -1.4409331357953084E-01_dp,&
      -7.3049415971812165E-02_dp,  1.6928103093679515E-02_dp,&
      1.8417725387954376E-02_dp, -1.9453592218161570E-03_dp,&
      -1.1634328023640639E-02_dp,  3.4714019819061617E-03_dp,&
      5.3490931351907831E-03_dp, -3.1907950343243647E-03_dp,&
      -1.4616713023815479E-03_dp,  1.8903207684751867E-03_dp,&
      -5.1679973398017858E-04_dp,  2.4951739894697100E+00_dp /)
    REAL(KIND=dp), DIMENSION(400), PARAMETER :: c05 = (/ &
      -3.2389785098025670E-01_dp, -1.5588421416935677E-01_dp,&
      4.6469218783312088E-02_dp,  4.1557942308222309E-02_dp,&
      -7.5142092188104671E-03_dp, -2.6806031107589168E-02_dp,&
      9.2275566818059377E-03_dp,  1.3149892067901462E-02_dp,&
      -9.4462212865469193E-03_dp, -1.9880182010796601E-03_dp,&
      3.9637132356526448E-03_dp, -1.1550825018293159E-03_dp,&
      4.2174292864101881E+00_dp, -7.0273735438846152E-01_dp,&
      -3.1958858729380285E-01_dp,  1.1897201461891156E-01_dp,&
      8.9479712944561557E-02_dp, -2.3599626911379866E-02_dp,&
      -5.9172184220296763E-02_dp,  2.3418908433583512E-02_dp,&
      3.0998875121130358E-02_dp, -2.6100693280547826E-02_dp,&
      -8.2991968088997443E-04_dp,  7.6967034888079256E-03_dp,&
      -2.4440953982349913E-03_dp,  7.1283946095005790E+00_dp,&
      -1.4935062082301014E+00_dp, -6.3792079328025064E-01_dp,&
      2.9134212583550634E-01_dp,  1.8609503658057067E-01_dp,&
      -6.6849972955691214E-02_dp, -1.2679209365908473E-01_dp,&
      5.7788861383825232E-02_dp,  7.1125670220970105E-02_dp,&
      -6.9155756511572325E-02_dp,  7.4772007009126021E-03_dp,&
      1.3632029714336589E-02_dp, -4.9228934078391299E-03_dp,&
      1.2173813507842725E+01_dp, -3.1400013912848532E+00_dp,&
      -1.2483074964973158E+00_dp,  6.9485765409820177E-01_dp,&
      3.7611257997832859E-01_dp, -1.7890856047138773E-01_dp,&
      -2.6578443159351284E-01_dp,  1.4054639374393529E-01_dp,&
      1.6047068051743860E-01_dp, -1.7899907540313789E-01_dp,&
      4.0285268988641564E-02_dp,  2.0634397349215801E-02_dp,&
      -9.3741744134483111E-03_dp,  2.1107642864276819E+01_dp,&
      -6.5843431773724026E+00_dp, -2.4050262723680573E+00_dp,&
      1.6350096374661267E+00_dp,  7.3996123655732526E-01_dp,&
      -4.6373337696649547E-01_dp, -5.4747966879238541E-01_dp,&
      3.4034950017735888E-01_dp,  3.5845721258746249E-01_dp,&
      -4.5829008373764241E-01_dp,  1.5001000994476824E-01_dp,&
      1.9940954618215254E-02_dp, -1.6426829440793874E-02_dp,&
      3.7266334989026802E+01_dp, -1.3867340387230591E+01_dp,&
      -4.5696148656214710E+00_dp,  3.8316970122330818E+00_dp,&
      1.4108582397700304E+00_dp, -1.1818903323968282E+00_dp,&
      -1.1088915296439605E+00_dp,  8.2704736824941671E-01_dp,&
      7.9552904564683069E-01_dp, -1.1705005968235462E+00_dp,&
      4.9161298485989080E-01_dp, -2.5476053684134296E-02_dp,&
      -2.4313596900675315E-02_dp,  6.7166643382165645E+01_dp,&
      -2.9516187500335366E+01_dp, -8.5451854440662061E+00_dp,&
      9.0099953442390479E+00_dp,  2.5665490024398552E+00_dp,&
      -2.9920833686411239E+00_dp, -2.1979231493373184E+00_dp,&
      2.0291778674850649E+00_dp,  1.7530383607299391E+00_dp,&
      -2.9992079584428746E+00_dp,  1.5201611647974216E+00_dp,&
      -2.4963049256105524E-01_dp, -1.9576317627699253E-02_dp,&
      1.2391224737051206E+02_dp, -6.3860904547818556E+01_dp,&
      -1.5596243868028813E+01_dp,  2.1390445119414000E+01_dp,&
      4.2665327922213292E+00_dp, -7.5789334284051302E+00_dp,&
      -4.2009119095859688E+00_dp,  5.0530481463321744E+00_dp,&
      3.8091681592822395E+00_dp, -7.7351382657783132E+00_dp,&
      4.5739670847276521E+00_dp, -1.1088234533884338E+00_dp,&
      5.7788156102331803E-02_dp,  2.3472890639646755E+02_dp,&
      -1.4127311447474941E+02_dp, -2.7175909935628077E+01_dp,&
      5.1557286383939626E+01_dp,  5.6020406213249556E+00_dp,&
      -1.9305968542309145E+01_dp, -7.4481250666061314E+00_dp,&
      1.2828955385228877E+01_dp,  7.9993419405969064E+00_dp,&
      -2.0087543738369188E+01_dp,  1.3607203316826732E+01_dp,&
      -4.1171412084629671E+00_dp,  4.3982127296754747E-01_dp,&
      4.5834886708487102E+02_dp, -3.2158236377162405E+02_dp,&
      -4.2500822021650286E+01_dp,  1.2681749970623152E+02_dp,&
      1.0331207737661403E+00_dp, -4.9592638546541615E+01_dp,&
      -1.0824736389255063E+01_dp,  3.3331519274645522E+01_dp,&
      1.5343959985063165E+01_dp, -5.2295332834515250E+01_dp,&
      4.0323068230649206E+01_dp, -1.4240819192072191E+01_dp,&
      1.9740981942447056E+00_dp,  9.2713486252741257E+02_dp,&
      -7.5881923406724286E+02_dp, -4.6362583755929869E+01_dp,&
      3.1985382205090053E+02_dp, -3.4656066891479590E+01_dp,&
      -1.2831744385936224E+02_dp, -4.9055399255105359E+00_dp,&
      8.8808607339626235E+01_dp,  2.1618208770971705E+01_dp,&
      -1.3449947060170553E+02_dp,  1.1892412424943839E+02_dp,&
      -4.7580164322585055E+01_dp,  7.6650405494235141E+00_dp,&
      1.9552554903261487E+03_dp, -1.8730960487320954E+03_dp,&
      4.5978532346317706E+01_dp,  8.3029128979817312E+02_dp,&
      -2.0862888435830430E+02_dp, -3.3108436117182572E+02_dp,&
      5.8912046909874640E+01_dp,  2.4205987690357750E+02_dp,&
      -1.5528194709333421E+01_dp, -3.2781660557594995E+02_dp,&
      3.4387197703428257E+02_dp, -1.5465208352189103E+02_dp,&
      2.7818133740736446E+01_dp,  4.3367077238709990E+03_dp,&
      -4.8949990956050187E+03_dp,  6.3916888657632228E+02_dp,&
      2.2193372246794902E+03_dp, -9.7830000114740096E+02_dp,&
      -8.2164251683291820E+02_dp,  4.0343060366377574E+02_dp,&
      6.6365758988347898E+02_dp, -3.5763798810336135E+02_dp,&
      -6.5703543744238846E+02_dp,  9.2616781594453050E+02_dp,&
      -4.7750440193060797E+02_dp,  9.5070584811891436E+01_dp,&
      1.0241393178730643E+04_dp, -1.3770619099685142E+04_dp,&
      3.6724807315668318E+03_dp,  6.0414514428452330E+03_dp,&
      -4.3826177232463688E+03_dp, -1.6912184393750047E+03_dp,&
      2.0017186887168280E+03_dp,  1.7046426163395738E+03_dp,&
      -2.4055027646909889E+03_dp, -2.6196488354718417E+02_dp,&
      1.9094004144765859E+03_dp, -1.2635639019999310E+03_dp,&
      2.8729634415947623E+02_dp,  2.6233288034113684E+04_dp,&
      -4.2749978247629486E+04_dp,  1.9270216857049596E+04_dp,&
      1.5749822825859681E+04_dp, -2.0118002433411439E+04_dp,&
      4.6708746343642822E+01_dp,  8.6670798148225003E+03_dp,&
      2.7018890777277093E+03_dp, -1.2539246090174649E+04_dp,&
      8.7032187871660572E+03_dp, -9.0580632943659941E+02_dp,&
      -1.5024640379499738E+03_dp,  5.4315272376617247E+02_dp,&
      7.5229570283632551E+04_dp, -1.5238226013483954E+05_dp,&
      1.0963939977708698E+05_dp,  2.3699789741127177E+04_dp,&
      -9.5127308763506648E+04_dp,  4.5638161082392988E+04_dp,&
      2.3668280641175519E+04_dp, -1.5344112755035001E+04_dp,&
      -4.0975894536356587E+04_dp,  6.5391139391724340E+04_dp,&
      -4.3370233536422988E+04_dp,  1.4481627041640608E+04_dp,&
      -1.9779208943771914E+03_dp,  2.6391498082474648E+05_dp,&
      -6.7578257857516059E+05_dp,  7.6143463474363345E+05_dp,&
      -3.1336546372684737E+05_dp, -2.7916162801665330E+05_dp,&
      5.0127439350212057E+05_dp, -3.0745998913516488E+05_dp,&
      5.4150294767096661E+04_dp,  9.3778431953998916E+03_dp,&
      4.7079371166969926E+04_dp, -6.9604664413008839E+04_dp,&
      3.7411996446493475E+04_dp, -7.6881721943026469E+03_dp,&
      1.8732918832849885E+00_dp, -1.3367410054138293E-01_dp,&
      -7.2352034213483338E-02_dp,  1.0909107202525322E-02_dp,&
      1.6759377488280897E-02_dp, -1.4463247410901762E-04_dp,&
      -1.0337062028004253E-02_dp,  2.4860738197468890E-03_dp,&
      4.3367529883595983E-03_dp, -1.8252291072459487E-03_dp,&
      -1.9460343870739156E-03_dp,  1.8559495947100581E-03_dp,&
      -4.7355776784216511E-04_dp,  2.3185521937218758E+00_dp,&
      -2.3156055207094098E-01_dp, -1.1794102738288226E-01_dp,&
      2.6687995574725933E-02_dp,  2.9649726476853933E-02_dp,&
      -2.8985963429409112E-03_dp, -1.8676999303934575E-02_dp,&
      5.4709623980301811E-03_dp,  8.5219863451357503E-03_dp,&
      -4.9468255033507554E-03_dp, -2.4659023720406152E-03_dp,&
      3.0700450686352012E-03_dp, -8.3320990292621406E-04_dp,&
      3.3517744635877706E+00_dp, -4.8229570440427993E-01_dp,&
      -2.3009121135863886E-01_dp,  7.1478533606108507E-02_dp,&
      6.2277049326382221E-02_dp, -1.1852516550801632E-02_dp,&
      -4.0220971604340308E-02_dp,  1.4012449854287748E-02_dp,&
      1.9865467670200296E-02_dp, -1.4442483293443300E-02_dp,&
      -2.8348238293451422E-03_dp,  5.9174029177587939E-03_dp,&
      -1.7331773541936861E-03_dp,  5.3031945680151331E+00_dp,&
      -1.0221259715172619E+00_dp, -4.5807387470277838E-01_dp,&
      1.8030806698751617E-01_dp,  1.3089607103055093E-01_dp,&
      -3.6736851851184617E-02_dp, -8.6877938204299371E-02_dp,&
      3.5099225799060947E-02_dp,  4.6044781064172931E-02_dp,&
      -3.9504731274502158E-02_dp, -5.0833388917190258E-04_dp,&
      1.1128529395928691E-02_dp, -3.5796831935575950E-03_dp,&
      8.8118115386268983E+00_dp, -2.1430470190320698E+00_dp,&
      -8.9884915867193405E-01_dp,  4.3510992099274409E-01_dp,&
      2.6780537007744093E-01_dp, -1.0216018744057363E-01_dp,&
      -1.8352430459115163E-01_dp,  8.5686435047266335E-02_dp,&
      1.0440791497056170E-01_dp, -1.0353084282964660E-01_dp,&
      1.2964730094686697E-02_dp,  1.9176877419226003E-02_dp,&
      -7.0877110142053322E-03_dp,  1.5082358533124399E+01_dp,&
      -4.4627150879444883E+00_dp, -1.7360210374491152E+00_dp,&
      1.0253983925629231E+00_dp,  5.3391870561552945E-01_dp,&
      -2.6940028777485425E-01_dp, -3.8042443552829347E-01_dp,&
      2.0666296997581385E-01_dp,  2.3347998166148740E-01_dp,&
      -2.6561413847670828E-01_dp,  6.3743999442031934E-02_dp,&
      2.7867301306668354E-02_dp, -1.3280682087156985E-02_dp,&
      2.6402399708460496E+01_dp, -9.3010729536877754E+00_dp,&
      -3.3106893280192309E+00_dp,  2.3920970871922971E+00_dp,&
      1.0385304890158198E+00_dp, -6.9082350726466202E-01_dp,&
      -7.7708115592966898E-01_dp,  4.9760389999262650E-01_dp,&
      5.1852022401451869E-01_dp, -6.7581533521215909E-01_dp,&
      2.2983801183532374E-01_dp,  2.3431816929989063E-02_dp,&
      -2.2860171257873246E-02_dp,  4.7194342446443912E+01_dp,&
      -1.9531042894379219E+01_dp, -6.2415251534557656E+00_dp,&
      5.5741252723118899E+00_dp,  1.9607368455071121E+00_dp,&
      -1.7473248563445485E+00_dp, -1.5646513259774506E+00_dp,&
      1.2048716833535995E+00_dp,  1.1473081165267458E+00_dp,&
      -1.7190501090563362E+00_dp,  7.4009478335832446E-01_dp,&
      -4.9906011575939872E-02_dp, -3.2866315872047365E-02_dp,&
      8.6211948052474540E+01_dp, -4.1554142631369068E+01_dp,&
      -1.1600185792684515E+01_dp,  1.3061955480753966E+01_dp,&
      3.5320613718765403E+00_dp, -4.4004006892425798E+00_dp,&
      -3.0887855024650599E+00_dp,  2.9501535623742106E+00_dp,&
      2.5271027814898610E+00_dp, -4.3943167718995682E+00_dp,&
      2.2632129132314391E+00_dp, -3.9043244143139921E-01_dp,&
      -2.3099495422322626E-02_dp,  1.6127098938108156E+02_dp,&
      -9.0039525643435795E+01_dp, -2.1053710089218743E+01_dp,&
      3.0952950172066647E+01_dp,  5.7949336008223513E+00_dp,&
      -1.1107924269366810E+01_dp, -5.8875195813116301E+00_dp,&
      7.3386204896080276E+00_dp,  5.5043800120782960E+00_dp,&
      -1.1322988619866953E+01_dp,  6.7597230603340916E+00_dp,&
      -1.6667162278158267E+00_dp,  9.4315470904943952E-02_dp,&
      3.0980553856001376E+02_dp, -1.9972543431441864E+02_dp,&
      -3.6433629844988346E+01_dp,  7.4549189020146230E+01_dp,&
      7.3841116630708612E+00_dp, -2.8240512722579421E+01_dp,&
      -1.0420298428282244E+01_dp,  1.8622263811434856E+01_dp,&
      1.1642216142881098E+01_dp, -2.9427949723634473E+01_dp,&
      2.0016125436219472E+01_dp, -6.0848036620178672E+00_dp,&
      6.5557403628564404E-01_dp,  6.1339991529169913E+02_dp,&
      -4.5611357812319022E+02_dp, -5.6273315070163605E+01_dp,&
      1.8335187300879113E+02_dp,  3.6753316352523047E-01_dp,&
      -7.2514546876872799E+01_dp, -1.5147050715967145E+01_dp,&
      4.8376562193982153E+01_dp,  2.2769214359032059E+01_dp,&
      -7.6885524107612937E+01_dp,  5.9199566480206009E+01_dp,&
      -2.0866616660745976E+01_dp,  2.8826441383546944E+00_dp,&
      1.2575364403629890E+03_dp, -1.0794714024934246E+03_dp,&
      -5.8881576395916468E+01_dp,  4.6255041181187573E+02_dp,&
      -5.0617974041393261E+01_dp, -1.8799904813345231E+02_dp,&
      -7.1004252575506328E+00_dp,  1.2896162702553036E+02_dp,&
      3.4617522397077572E+01_dp, -1.9966691145147081E+02_dp,&
      1.7499140865274785E+02_dp, -6.9518624039918890E+01_dp,&
      1.1110268324191310E+01_dp,  2.6853251507125306E+03_dp,&
      -2.6690269182691513E+03_dp,  7.2463711925762794E+01_dp,&
      1.2014230806845137E+03_dp, -2.9322356189139873E+02_dp,&
      -4.8871486180307096E+02_dp,  8.0189444854946785E+01_dp,&
      3.5253998447998373E+02_dp, -4.4344999858279763E+00_dp,&
      -4.9961491943629647E+02_dp,  5.1183627581614923E+02_dp /)
    REAL(KIND=dp), DIMENSION(400), PARAMETER :: c06 = (/ &
      -2.2712994716763473E+02_dp,  4.0385181038065113E+01_dp,&
      6.0197819306524152E+03_dp, -6.9664015206915119E+03_dp,&
      8.8000828312450119E+02_dp,  3.2181044508507798E+03_dp,&
      -1.3451011011345620E+03_dp, -1.2427951210535916E+03_dp,&
      5.4994829125785282E+02_dp,  9.7855377199584927E+02_dp,&
      -4.2361871901332773E+02_dp, -1.0938340303256634E+03_dp,&
      1.4307426913415468E+03_dp, -7.1802169806496249E+02_dp,&
      1.4043890455439342E+02_dp,  1.4319961817475929E+04_dp,&
      -1.9465824088680063E+04_dp,  4.8714430805874626E+03_dp,&
      8.8417682243147774E+03_dp, -5.8918891377734217E+03_dp,&
      -2.8286954277399191E+03_dp,  2.7351780669919335E+03_dp,&
      2.6465333742965945E+03_dp, -3.0382947594497841E+03_dp,&
      -1.2233930010289332E+03_dp,  3.3943700985724181E+03_dp,&
      -2.0626667671078540E+03_dp,  4.5139108215934982E+02_dp,&
      3.6710729399870332E+04_dp, -5.9406878052428168E+04_dp,&
      2.4439529663225516E+04_dp,  2.4144745080947698E+04_dp,&
      -2.6373382875748357E+04_dp, -3.2022704306036744E+03_dp,&
      1.2266276717430037E+04_dp,  5.7681700935135077E+03_dp,&
      -1.6920132638314262E+04_dp,  8.0315452344738505E+03_dp,&
      2.9576472354183184E+03_dp, -4.0703120694405825E+03_dp,&
      1.1327630799673143E+03_dp,  1.0388253490761654E+05_dp,&
      -2.0409751637993401E+05_dp,  1.2925285017231829E+05_dp,&
      5.4968049843609188E+04_dp, -1.2447761109417022E+05_dp,&
      3.2483521887329320E+04_dp,  4.7616175662189016E+04_dp,&
      -4.6528610805370972E+03_dp, -7.6633080388216375E+04_dp,&
      8.7935387930962621E+04_dp, -4.3615780121465235E+04_dp,&
      8.9761933428718621E+03_dp, -1.5583757123341093E+02_dp,&
      3.3941315862276591E+05_dp, -8.2872883215502277E+05_dp,&
      7.9822211386825750E+05_dp, -8.4074217002807636E+04_dp,&
      -5.7957777085555380E+05_dp,  5.1317015480880765E+05_dp,&
      5.0080177118401307E+03_dp, -2.1554743060333602E+05_dp,&
      -6.8710719687172372E+04_dp,  3.7597911959563685E+05_dp,&
      -3.5898885567466140E+05_dp,  1.6051728031093234E+05_dp,&
      -2.9622503438194970E+04_dp,  1.5375178283066794E+06_dp,&
      -4.5263227540075248E+06_dp,  6.3451073165447786E+06_dp,&
      -4.6240857945008520E+06_dp,  2.4500245785436130E+05_dp,&
      3.5323273923941231E+06_dp, -4.5233461773433862E+06_dp,&
      3.2676821863851110E+06_dp, -1.5689703219501805E+06_dp,&
      5.1341607784453162E+05_dp, -1.1143590110389760E+05_dp,&
      1.3969999513488921E+04_dp, -4.2927092431353054E+02_dp,&
      2.3528218884138855E-01_dp, -6.1622809221375703E-03_dp,&
      -5.1621212301375691E-03_dp,  1.8256971283183122E-03_dp,&
      -1.7757001927316855E-03_dp,  2.7587868056723900E-03_dp,&
      -3.1055628572827037E-03_dp,  3.0788733937437561E-03_dp,&
      -2.9393177388430942E-03_dp,  2.4422551665436885E-03_dp,&
      -1.5309508783932135E-03_dp,  6.2070307138328850E-04_dp,&
      -1.2037481424199890E-04_dp,  7.6427517492838903E-01_dp,&
      -2.3301267122622669E-02_dp, -1.9379395197215672E-02_dp,&
      7.1363033962990527E-03_dp, -6.7034010276187488E-03_dp,&
      1.0440998275629443E-02_dp, -1.1789250952983365E-02_dp,&
      1.1659409239080437E-02_dp, -1.1104556236899659E-02_dp,&
      9.2224304313982004E-03_dp, -5.7812574625441581E-03_dp,&
      2.3431890007339553E-03_dp, -4.5412494246756713E-04_dp,&
      1.4884452945163158E+00_dp, -5.7780418368486818E-02_dp,&
      -4.7524525258133422E-02_dp,  1.8574891035076632E-02_dp,&
      -1.6590444169109660E-02_dp,  2.5921469668594304E-02_dp,&
      -2.9408901004723566E-02_dp,  2.8978898590561707E-02_dp,&
      -2.7497730106109036E-02_dp,  2.2820362291516471E-02_dp,&
      -1.4306212773108158E-02_dp,  5.7958440272391429E-03_dp,&
      -1.1221744170445015E-03_dp,  2.6105984893027929E+00_dp,&
      -1.3223929038081408E-01_dp, -1.0742680517105528E-01_dp,&
      4.4715325992489605E-02_dp, -3.7927286855369426E-02_dp,&
      5.9386145153437431E-02_dp, -6.7746283118929551E-02_dp,&
      6.6496265207841132E-02_dp, -6.2831201303096690E-02_dp,&
      5.2098699546617035E-02_dp, -3.2665012485418560E-02_dp,&
      1.3227984141057456E-02_dp, -2.5585133273128336E-03_dp,&
      4.4702121379027533E+00_dp, -2.9229427267092511E-01_dp,&
      -2.3438486623605742E-01_dp,  1.0383070807863753E-01_dp,&
      -8.3847740273577026E-02_dp,  1.3135552153343474E-01_dp,&
      -1.5073934324268146E-01_dp,  1.4739044102412518E-01_dp,&
      -1.3863284292466410E-01_dp,  1.1484099496527858E-01_dp,&
      -7.2018264561787296E-02_dp,  2.9154836504191747E-02_dp,&
      -5.6333601427675465E-03_dp,  7.6709266037709876E+00_dp,&
      -6.3341405923166583E-01_dp, -5.0094369630952285E-01_dp,&
      2.3625700783289558E-01_dp, -1.8202632717122910E-01_dp,&
      2.8474545391588596E-01_dp, -3.2890009162789791E-01_dp,&
      3.2037234643104090E-01_dp, -2.9982983600410501E-01_dp,&
      2.4809503450350234E-01_dp, -1.5563160367252610E-01_dp,&
      6.2990071438569953E-02_dp, -1.2159286810559670E-02_dp,&
      1.3326570559389127E+01_dp, -1.3598533563961679E+00_dp,&
      -1.0592963255701000E+00_dp,  5.3297178906709941E-01_dp,&
      -3.9225796386832035E-01_dp,  6.1125294976131017E-01_dp,&
      -7.1120845081937767E-01_dp,  6.9013373426681579E-01_dp,&
      -6.4223671050719877E-01_dp,  5.3070213765084473E-01_dp,&
      -3.3305766295054873E-01_dp,  1.3479152751847270E-01_dp,&
      -2.5995336263354377E-02_dp,  2.3553113968366869E+01_dp,&
      -2.9176276962014072E+00_dp, -2.2345934256000408E+00_dp,&
      1.2036034488905603E+00_dp, -8.4690962758329502E-01_dp,&
      1.3107060305534961E+00_dp, -1.5377199381631459E+00_dp,&
      1.4864614399378571E+00_dp, -1.3743037289490871E+00_dp,&
      1.1337221588786441E+00_dp, -7.1192149066697252E-01_dp,&
      2.8815685634273214E-01_dp, -5.5524774908207841E-02_dp,&
      4.2483858960737933E+01_dp, -6.3034996022484222E+00_dp,&
      -4.7356656620562889E+00_dp,  2.7433815095122966E+00_dp,&
      -1.8473371186027441E+00_dp,  2.8284164031479886E+00_dp,&
      -3.3498320968923769E+00_dp,  3.2259356440170177E+00_dp,&
      -2.9598546271753077E+00_dp,  2.4364391557222511E+00_dp,&
      -1.5311624837953171E+00_dp,  6.1999991359852169E-01_dp,&
      -1.1937851432712819E-01_dp,  7.8435361033567304E+01_dp,&
      -1.3807243700243477E+01_dp, -1.0144752030644700E+01_dp,&
      6.3577896872198973E+00_dp, -4.1032215968515038E+00_dp,&
      6.1834653423448023E+00_dp, -7.4031369163085019E+00_dp,&
      7.1034525244070581E+00_dp, -6.4588685672011907E+00_dp,&
      5.3015382264941371E+00_dp, -3.3350883832865144E+00_dp,&
      1.3515222561641431E+00_dp, -2.6009258886737091E-01_dp,&
      1.4870314684342992E+02_dp, -3.0864834118508210E+01_dp,&
      -2.2093258811557757E+01_dp,  1.5088268587806473E+01_dp,&
      -9.3558068710238516E+00_dp,  1.3783237313514368E+01_dp,&
      -1.6707764004413356E+01_dp,  1.5978749580565010E+01_dp,&
      -1.4371991456513806E+01_dp,  1.1750956854689353E+01_dp,&
      -7.4016805238109020E+00_dp,  3.0035875232982088E+00_dp,&
      -5.7793452739336704E-01_dp,  2.9063992566159970E+02_dp,&
      -7.0899303612086015E+01_dp, -4.9191236361965750E+01_dp,&
      3.6944732038741407E+01_dp, -2.2096224350928072E+01_dp,&
      3.1535478662551061E+01_dp, -3.8769926784154620E+01_dp,&
      3.6981135439762390E+01_dp, -3.2826571132059875E+01_dp,&
      2.6693078989105775E+01_dp, -1.6838972567744609E+01_dp,&
      6.8482490585591780E+00_dp, -1.3183464225945833E+00_dp,&
      5.8853959957153813E+02_dp, -1.6866367548444850E+02_dp,&
      -1.1264217046922096E+02_dp,  9.4139630285255294E+01_dp,&
      -5.4641451607748081E+01_dp,  7.4621772797404674E+01_dp,&
      -9.3205667243426078E+01_dp,  8.8778688875908884E+01_dp,&
      -7.7534194668658188E+01_dp,  6.2541347341539755E+01_dp,&
      -3.9518334321274587E+01_dp,  1.6126307270781510E+01_dp,&
      -3.1093032562817622E+00_dp,  1.2428008833245567E+03_dp,&
      -4.1947785521914795E+02_dp, -2.6701285426652703E+02_dp,&
      2.5226385512723408E+02_dp, -1.4345141862564316E+02_dp,&
      1.8433585054421314E+02_dp, -2.3423907181798836E+02_dp,&
      2.2325867412280667E+02_dp, -1.9105874160160232E+02_dp,&
      1.5221467577400151E+02_dp, -9.6307754715699645E+01_dp,&
      3.9500236966449748E+01_dp, -7.6415388902361059E+00_dp,&
      2.7608915592989856E+03_dp, -1.1041691712029990E+03_dp,&
      -6.6001010192912725E+02_dp,  7.2065717805192821E+02_dp,&
      -4.0742001889818755E+02_dp,  4.8140472788779994E+02_dp,&
      -6.2229993000320292E+02_dp,  5.9562070307754016E+02_dp,&
      -4.9676841427948847E+02_dp,  3.8786350248878023E+02_dp,&
      -2.4530498580103026E+02_dp,  1.0134423863055014E+02_dp,&
      -1.9728796542023890E+01_dp,  6.5330539597289735E+03_dp,&
      -3.1289896695941670E+03_dp, -1.7144513670222454E+03_dp,&
      2.2366769584555950E+03_dp, -1.2862337799451041E+03_dp,&
      1.3545526987521687E+03_dp, -1.7732225521374326E+03_dp,&
      1.7145546967607834E+03_dp, -1.3838754531006168E+03_dp,&
      1.0431149756407765E+03_dp, -6.5542908071128045E+02_dp,&
      2.7332077381383078E+02_dp, -5.3786830655175223E+01_dp,&
      1.6776395337771821E+04_dp, -9.7928996284343357E+03_dp,&
      -4.7065652179644330E+03_dp,  7.7563596005434492E+03_dp,&
      -4.7062831481779740E+03_dp,  4.2440444961091507E+03_dp,&
      -5.5256490153285331E+03_dp,  5.4499874871954971E+03_dp,&
      -4.2216100644922999E+03_dp,  2.9760556872278821E+03_dp,&
      -1.8186260564964466E+03_dp,  7.6369653195481897E+02_dp,&
      -1.5282174308169036E+02_dp,  4.8196026704304764E+04_dp,&
      -3.5311796147047455E+04_dp, -1.3499860063436423E+04_dp,&
      3.1444987874104710E+04_dp, -2.1409313414419648E+04_dp,&
      1.5890405287246669E+04_dp, -1.9335940318034241E+04_dp,&
      1.9685996849087067E+04_dp, -1.4546076258636267E+04_dp,&
      8.9035324606561226E+03_dp, -4.8468802353721112E+03_dp,&
      1.9729740359294485E+03_dp, -4.0033508034806044E+02_dp,&
      1.6394238215154631E+05_dp, -1.5905119636599772E+05_dp,&
      -3.4636012557649119E+04_dp,  1.6159468155671351E+05_dp,&
      -1.3873213467843784E+05_dp,  8.6609937728528355E+04_dp,&
      -7.9189862325619106E+04_dp,  8.0498770341005089E+04_dp,&
      -5.6800098872495153E+04_dp,  2.4318200214846263E+04_dp,&
      -5.3486374635222237E+03_dp,  1.4978633693132076E+02_dp,&
      1.3126224230101201E+02_dp,  7.5834075124608912E+05_dp,&
      -1.0864193654712576E+06_dp,  1.6418235171236837E+05_dp,&
      1.1876482137492767E+06_dp, -1.7741895129516888E+06_dp,&
      1.3985011610223595E+06_dp, -6.4697226138982608E+05_dp,&
      1.5340670486444154E+04_dp,  3.7060552578940440E+05_dp,&
      -5.1246119196262851E+05_dp,  4.1905388782645378E+05_dp,&
      -2.0351458872316248E+05_dp,  4.4969428447952065E+04_dp,&
      1.9199342325621056E+00_dp, -5.2325699613614447E-02_dp,&
      -4.3750513777635607E-02_dp,  1.5640015963402101E-02_dp,&
      -1.5070969781007941E-02_dp,  2.3431356008326149E-02_dp,&
      -2.6397531067032395E-02_dp,  2.6153766119785420E-02_dp,&
      -2.4952874087669753E-02_dp,  2.0730661937501563E-02_dp,&
      -1.2995238603525089E-02_dp,  5.2682907391888040E-03_dp,&
      -1.0215189419958690E-03_dp,  2.4002587623301448E+00_dp,&
      -9.2692544030490695E-02_dp, -7.6334957658037619E-02_dp,&
      2.9644232823144580E-02_dp, -2.6611713402345558E-02_dp,&
      4.1582159829453708E-02_dp, -4.7147451242685809E-02_dp,&
      4.6475506392680971E-02_dp, -4.4120374420174135E-02_dp,&
      3.6619045036844551E-02_dp, -2.2956089487110191E-02_dp,&
      9.3003902294016821E-03_dp, -1.8008841146529171E-03_dp,&
      3.5238883296645747E+00_dp, -1.9744854381747892E-01_dp,&
      -1.6011934282987575E-01_dp,  6.7242589745269768E-02_dp,&
      -5.6584064388948051E-02_dp,  8.8699180368368491E-02_dp,&
      -1.0124759733146146E-01_dp,  9.9318595794023787E-02_dp,&
      -9.3797388513328375E-02_dp,  7.7768501343562266E-02_dp,&
      -4.8758543054448447E-02_dp,  1.9743071622430480E-02_dp,&
      -3.8179581523259494E-03_dp,  5.6716908861136890E+00_dp,&
      -4.2699382481264164E-01_dp, -3.4141921609801612E-01_dp,&
      1.5331722645320270E-01_dp, -1.2238630321990675E-01_dp,&
      1.9196242245137163E-01_dp, -2.2053214264272086E-01_dp,&
      2.1543129874600328E-01_dp, -2.0245400581809397E-01_dp,&
      1.6768202331832444E-01_dp, -1.0515504828494937E-01_dp,&
      4.2563447502959498E-02_dp, -8.2219895159869709E-03_dp,&
      9.5923398264069970E+00_dp, -9.1351432511338171E-01_dp,&
      -7.2002060302993121E-01_dp,  3.4478262562537942E-01_dp,&
      -2.6236445928893765E-01_dp,  4.1080033835374796E-01_dp,&
      -4.7515478577796189E-01_dp,  4.6235013115468138E-01_dp /)
    REAL(KIND=dp), DIMENSION(400), PARAMETER :: c07 = (/ &
      -4.3224032563729953E-01_dp,  3.5758483946236974E-01_dp,&
      -2.2431923396808751E-01_dp,  9.0777959192790053E-02_dp,&
      -1.7518061809443625E-02_dp,  1.6725498665042689E+01_dp,&
      -1.9435939058523524E+00_dp, -1.5082483802542699E+00_dp,&
      7.7123067496772879E-01_dp, -5.6050187357065773E-01_dp,&
      8.7386773329950440E-01_dp, -1.0184147160321777E+00_dp,&
      9.8713775000143567E-01_dp, -9.1748347756413584E-01_dp,&
      7.5796037251628001E-01_dp, -4.7570197410379467E-01_dp,&
      1.9249696312506864E-01_dp, -3.7112410202830466E-02_dp,&
      2.9867921112026917E+01_dp, -4.1467710457417182E+00_dp,&
      -3.1626313649547511E+00_dp,  1.7323863084109623E+00_dp,&
      -1.2038693567355865E+00_dp,  1.8631804826928859E+00_dp,&
      -2.1899213857816511E+00_dp,  2.1144847654909018E+00_dp,&
      -1.9521830909580684E+00_dp,  1.6099731981821668E+00_dp,&
      -1.0110621403038136E+00_dp,  4.0919444918433101E-01_dp,&
      -7.8821746413100843E-02_dp,  5.4567905169406032E+01_dp,&
      -8.9359125078559316E+00_dp, -6.6829857757469942E+00_dp,&
      3.9380155360006719E+00_dp, -2.6201981403151646E+00_dp,&
      4.0098204947674123E+00_dp, -4.7586752532202130E+00_dp,&
      4.5773565964791283E+00_dp, -4.1933229746486287E+00_dp,&
      3.4506609624332976E+00_dp, -2.1687844901543256E+00_dp,&
      8.7811842578345611E-01_dp, -1.6902289559128800E-01_dp,&
      1.0213272364646357E+02_dp, -1.9568894490980483E+01_dp,&
      -1.4310663259886253E+01_dp,  9.1189354843768466E+00_dp,&
      -5.8202843298748856E+00_dp,  8.7633901669854701E+00_dp,&
      -1.0514412487755664E+01_dp,  1.0077333974037595E+01_dp,&
      -9.1480969346309582E+00_dp,  7.5063522206702977E+00_dp,&
      -4.7227755759865069E+00_dp,  1.9137734403969708E+00_dp,&
      -3.6817562821028288E-01_dp,  1.9634789478808003E+02_dp,&
      -4.3805003142660127E+01_dp, -3.1210983480765918E+01_dp,&
      2.1646060102378659E+01_dp, -1.3290007437531367E+01_dp,&
      1.9559470489534739E+01_dp, -2.3760530475729208E+01_dp,&
      2.2699107507081777E+01_dp, -2.0384285092804305E+01_dp,&
      1.6661734343002301E+01_dp, -1.0496738624690892E+01_dp,&
      4.2593912604244579E+00_dp, -8.1930517662403424E-01_dp,&
      3.8911243929907391E+02_dp, -1.0084567929859232E+02_dp,&
      -6.9675590237727150E+01_dp,  5.3023392551490424E+01_dp,&
      -3.1443103504058708E+01_dp,  4.4848086974289387E+01_dp,&
      -5.5248419076016638E+01_dp,  5.2644425241400143E+01_dp,&
      -4.6665529615040874E+01_dp,  3.7940330501057126E+01_dp,&
      -2.3939320376490915E+01_dp,  9.7353325974390632E+00_dp,&
      -1.8734799945633591E+00_dp,  7.9857425247754088E+02_dp,&
      -2.4041614411913994E+02_dp, -1.6005711637448368E+02_dp,&
      1.3505138321984933E+02_dp, -7.7813606689547342E+01_dp,&
      1.0635214708532304E+02_dp, -1.3307542579001566E+02_dp,&
      1.2661963840999799E+02_dp, -1.1047991689937190E+02_dp,&
      8.9136639272186287E+01_dp, -5.6339722379846961E+01_dp,&
      2.2987396443453235E+01_dp, -4.4301925174203562E+00_dp,&
      1.7074190243146495E+03_dp, -5.9849403848656630E+02_dp,&
      -3.8056898159220981E+02_dp,  3.6095620533881140E+02_dp,&
      -2.0384611935344842E+02_dp,  2.6298454814944631E+02_dp,&
      -3.3468311722540773E+02_dp,  3.1858350725758493E+02_dp,&
      -2.7261989113275348E+02_dp,  2.1744286850398822E+02_dp,&
      -1.3765043100313622E+02_dp,  5.6438287133666776E+01_dp,&
      -1.0910502440622219E+01_dp,  3.8334094340274683E+03_dp,&
      -1.5723908956163691E+03_dp, -9.4281020491913853E+02_dp,&
      1.0242611119269077E+03_dp, -5.7428751500833221E+02_dp,&
      6.8547311891561162E+02_dp, -8.8749541005121341E+02_dp,&
      8.4777812676065867E+02_dp, -7.0810279313566184E+02_dp,&
      5.5469907294877623E+02_dp, -3.5128294755457512E+02_dp,&
      1.4504011186174179E+02_dp, -2.8198572213336920E+01_dp,&
      9.1361304192165644E+03_dp, -4.4226274015114341E+03_dp,&
      -2.4512781079839870E+03_dp,  3.1345626862048653E+03_dp,&
      -1.7784413126977358E+03_dp,  1.9127074406180236E+03_dp,&
      -2.5124489551522956E+03_dp,  2.4206332112005493E+03_dp,&
      -1.9613261391666301E+03_dp,  1.4910365526546971E+03_dp,&
      -9.4077115625989279E+02_dp,  3.9209237872283859E+02_dp,&
      -7.6978949238131889E+01_dp,  2.3476804836368778E+04_dp,&
      -1.3592633254062974E+04_dp, -6.7328872117040401E+03_dp,&
      1.0577880817867974E+04_dp, -6.2497295482203444E+03_dp,&
      5.8566563032174154E+03_dp, -7.7179494949154041E+03_dp,&
      7.5607028303424113E+03_dp, -5.8984755977119921E+03_dp,&
      4.2500229634280931E+03_dp, -2.6366823005050137E+03_dp,&
      1.1101454768091173E+03_dp, -2.2141619218110822E+02_dp,&
      6.6614484110205842E+04_dp, -4.7108388287965288E+04_dp,&
      -1.9523320398634307E+04_dp,  4.0705057477236631E+04_dp,&
      -2.6182350826008271E+04_dp,  2.0606177082035400E+04_dp,&
      -2.6289727061071782E+04_dp,  2.6497961110492906E+04_dp,&
      -1.9762650337886316E+04_dp,  1.2821176945806279E+04_dp,&
      -7.4626721995415519E+03_dp,  3.1351727273388092E+03_dp,&
      -6.3906389016555295E+02_dp,  2.1722047847281824E+05_dp,&
      -1.9421988001893414E+05_dp, -5.7005765806413925E+04_dp,&
      1.8870977976178579E+05_dp, -1.4215955667570615E+05_dp,&
      9.2469073856111092E+04_dp, -1.0223677196613803E+05_dp,&
      1.0679192105089303E+05_dp, -7.6172821818506985E+04_dp,&
      3.9251745674129270E+04_dp, -1.6935502287943222E+04_dp,&
      6.1072726038873288E+03_dp, -1.2312049579639786E+03_dp,&
      8.7962812129087350E+05_dp, -1.0583094001579208E+06_dp,&
      -8.7726089363737672E+04_dp,  1.1620532119345395E+06_dp,&
      -1.1801685233582184E+06_dp,  7.0461861958712188E+05_dp,&
      -4.8091128165205766E+05_dp,  4.5337411670429324E+05_dp,&
      -3.0573011401678680E+05_dp,  6.8451145563412196E+04_dp,&
      6.0365054336862682E+04_dp, -5.2641113555194272E+04_dp,&
      1.3429156007824822E+04_dp,  5.4534124986841530E+06_dp,&
      -9.8505910169624574E+06_dp,  4.1125439885084690E+06_dp,&
      1.0145049015186040E+07_dp, -2.1737139793601342E+07_dp,&
      2.1610941564617176E+07_dp, -1.0288896330046380E+07_dp,&
      -4.5951658076810353E+06_dp,  1.5022207247350259E+07_dp,&
      -1.7099728394099496E+07_dp,  1.2171622980394198E+07_dp,&
      -5.3020153298938526E+06_dp,  1.0818379069385820E+06_dp,&
      2.3670956436095472E-01_dp,  8.5466042782728561E-10_dp,&
      -8.7706516429572629E-03_dp,  1.1337199418484344E-02_dp,&
      -1.5571021209192253E-02_dp,  1.5569362340987348E-02_dp,&
      -2.4778844274855861E-03_dp, -3.3946179190323028E-02_dp,&
      9.4071406905596136E-02_dp, -1.5071574728552392E-01_dp,&
      1.5621592696596148E-01_dp, -9.7043862738537434E-02_dp,&
      2.7457399409936200E-02_dp,  7.6968000393308511E-01_dp,&
      3.2401450648662975E-09_dp, -3.3256925970225171E-02_dp,&
      4.2988865647330857E-02_dp, -5.8756741104987492E-02_dp,&
      5.8295248976357440E-02_dp, -7.8860907725801094E-03_dp,&
      -1.3118813251172537E-01_dp,  3.5989835704601603E-01_dp,&
      -5.7462950608239283E-01_dp,  5.9452625957445671E-01_dp,&
      -3.6891620144798964E-01_dp,  1.0430150010072287E-01_dp,&
      1.5018763179292813E+00_dp,  8.0666558227795613E-09_dp,&
      -8.2819934409362694E-02_dp,  1.0705542955448639E-01_dp,&
      -1.4523496967030083E-01_dp,  1.4235570567914205E-01_dp,&
      -1.3911957597914196E-02_dp, -3.3604701889620353E-01_dp,&
      9.0831435978670716E-01_dp, -1.4428049024652592E+00_dp,&
      1.4887036137719374E+00_dp, -9.2220669302792391E-01_dp,&
      2.6043051480110496E-01_dp,  2.6414098653587632E+00_dp,&
      1.8542770560249732E-08_dp, -1.9043939699928683E-01_dp,&
      2.4616740044234692E-01_dp, -3.3120274807858702E-01_dp,&
      3.2019629965984886E-01_dp, -1.7514299065979757E-02_dp,&
      -7.9626913862920790E-01_dp,  2.1188458241158257E+00_dp,&
      -3.3470422670498117E+00_dp,  3.4433083070863675E+00_dp,&
      -2.1290723817341952E+00_dp,  6.0049052563642336E-01_dp,&
      4.5384820955717666E+00_dp,  4.1171481244663121E-08_dp,&
      -4.2299027025076608E-01_dp,  5.4676919672569546E-01_dp,&
      -7.2930405536067833E-01_dp,  6.9476844166586404E-01_dp,&
      -5.7289627094700328E-03_dp, -1.8223453736507966E+00_dp,&
      4.7747854172686690E+00_dp, -7.5003110665766188E+00_dp,&
      7.6927313673657078E+00_dp, -4.7475264948136440E+00_dp,&
      1.3372664279873743E+00_dp,  7.8192504156388125E+00_dp,&
      8.9643714677238562E-08_dp, -9.2134105247858755E-01_dp,&
      1.1909512854185862E+00_dp, -1.5740081463467599E+00_dp,&
      1.4756520617606019E+00_dp,  6.3250121089543448E-02_dp,&
      -4.0913969501850129E+00_dp,  1.0554863467153654E+01_dp,&
      -1.6484455241456860E+01_dp,  1.6854353882166951E+01_dp,&
      -1.0380878084152998E+01_dp,  2.9200587692297555E+00_dp,&
      1.3645885165943300E+01_dp,  1.9343550021859286E-07_dp,&
      -1.9889558975248320E+00_dp,  2.5709795975545706E+00_dp,&
      -3.3639874881302920E+00_dp,  3.0976636513324340E+00_dp,&
      3.1247903773887803E-01_dp, -9.1142796291772967E+00_dp,&
      2.3139806188864029E+01_dp, -3.5920374960036447E+01_dp,&
      3.6603598484231796E+01_dp, -2.2496614716254765E+01_dp,&
      6.3188068542226405E+00_dp,  2.4240315177351366E+01_dp,&
      4.1735160476847653E-07_dp, -4.2934724986403312E+00_dp,&
      5.5498602800946042E+00_dp, -7.1808885190927310E+00_dp,&
      6.4773412225047196E+00_dp,  1.0915405525625834E+00_dp,&
      -2.0338678533491709E+01_dp,  5.0778253166199057E+01_dp,&
      -7.8310233286780999E+01_dp,  7.9509375168412404E+01_dp,&
      -4.8752154654454642E+01_dp,  1.3671218890716160E+01_dp,&
      4.3973636879745392E+01_dp,  9.0731441379826837E-07_dp,&
      -9.3394244687918899E+00_dp,  1.2072392500013727E+01_dp,&
      -1.5423438922958889E+01_dp,  1.3579561953094331E+01_dp,&
      3.3846975946055915E+00_dp, -4.5839585399730254E+01_dp,&
      1.1242639699367048E+02_dp, -1.7215228837760810E+02_dp,&
      1.7408628097597264E+02_dp, -1.0646507180414103E+02_dp,&
      2.9801105395511023E+01_dp,  8.1711262931542294E+01_dp,&
      2.0013821500952097E-06_dp, -2.0615673482163860E+01_dp,&
      2.6648367171543072E+01_dp, -3.3552503990308622E+01_dp,&
      2.8697356550258888E+01_dp,  9.9848598970918747E+00_dp,&
      -1.0512807173143871E+02_dp,  2.5297323747845883E+02_dp,&
      -3.8433225027457678E+02_dp,  3.8690514140568553E+02_dp,&
      -2.3592263278951384E+02_dp,  6.5902326877061824E+01_dp,&
      1.5605879855050961E+02_dp,  4.5098443290657727E-06_dp,&
      -4.6494277087601013E+01_dp,  6.0099711541289679E+01_dp,&
      -7.4393330371570187E+01_dp,  6.1409065971332915E+01_dp,&
      2.8984093656381923E+01_dp, -2.4713866438037331E+02_dp,&
      5.8259718635376578E+02_dp, -8.7740192761657318E+02_dp,&
      8.7878792290311787E+02_dp, -5.3405825659980758E+02_dp,&
      1.4883009303091876E+02_dp,  3.0762435042193624E+02_dp,&
      1.0455746769612150E-05_dp, -1.0790746717317136E+02_dp,&
      1.3948392350653936E+02_dp, -1.6920163788653028E+02_dp,&
      1.3356024260004625E+02_dp,  8.4612560681060771E+01_dp,&
      -6.0021504339910359E+02_dp,  1.3835888832168876E+03_dp,&
      -2.0632260865604612E+03_dp,  2.0544267400630752E+03_dp,&
      -1.2436499579901790E+03_dp,  3.4561729478378913E+02_dp,&
      6.2919318362091553E+02_dp,  2.5144905278024844E-05_dp,&
      -2.5985287556874499E+02_dp,  3.3589217889390994E+02_dp,&
      -3.9757888190288548E+02_dp,  2.9600943596616952E+02_dp,&
      2.5281006909384925E+02_dp, -1.5196535530174046E+03_dp,&
      3.4176113708246221E+03_dp, -5.0389952265142629E+03_dp,&
      4.9832566201452173E+03_dp, -3.0026770436385773E+03_dp,&
      8.3169610915648207E+02_dp,  1.3446642485369698E+03_dp,&
      6.3355693696567805E-05_dp, -6.5587608695957738E+02_dp,&
      8.4780072445181315E+02_dp, -9.7324930571061304E+02_dp,&
      6.6865816098059202E+02_dp,  7.8642740762815720E+02_dp,&
      -4.0561879015463091E+03_dp,  8.8735395997333635E+03_dp,&
      -1.2911085013308702E+04_dp,  1.2663863439439963E+04_dp,&
      -7.5877330082628787E+03_dp,  2.0931234638857791E+03_dp,&
      3.0315178012421479E+03_dp,  1.6945412072655205E-04_dp,&
      -1.7583989533457666E+03_dp,  2.2729448301394605E+03_dp,&
      -2.5079766439284267E+03_dp,  1.5298178702512521E+03_dp,&
      2.5970516090366850E+03_dp, -1.1583854023864891E+04_dp,&
      2.4553873694398699E+04_dp, -3.5160886963347439E+04_dp,&
      3.4138635891960170E+04_dp, -2.0309554368209294E+04_dp,&
      5.5733463618645619E+03_dp,  7.3092031064017010E+03_dp,&
      4.9005183646288716E-04_dp, -5.1024325120903641E+03_dp,&
      6.5955046072210243E+03_dp, -6.8957569912961944E+03_dp /)
    REAL(KIND=dp), DIMENSION(13, 40, 6), PARAMETER :: coefdata = RESHAPE ((/ &
      c01, c02, c03, c04, c05, c06, c07, c08 /),(/13,40,6/))

    INTEGER                                  :: irange

     IF(Rc <=  6.0000000000000000E+01_dp) Rc =   6.0000000000000000E+01_dp
     IF(Rc >=  5.7633609771728516E+06_dp) Rc =   5.7633609771728516E+06_dp
     IF(Rc >=  6.0000000000000000E+01_dp .AND. Rc <=  1.9739889204664721E+02_dp) THEN
       irange =  1
       L_b =  6.0000000000000000E+01_dp
       U_b =  1.9739889204664721E+02_dp
     END IF
     IF(Rc >=  1.9739889204664721E+02_dp .AND. Rc <=  2.4626386425352434E+03_dp) THEN
       irange =  2
       L_b =  1.9739889204664721E+02_dp
       U_b =  2.4626386425352434E+03_dp
     END IF
     IF(Rc >=  2.4626386425352434E+03_dp .AND. Rc <=  3.3579125411394947E+04_dp) THEN
       irange =  3
       L_b =  2.4626386425352434E+03_dp
       U_b =  3.3579125411394947E+04_dp
     END IF
     IF(Rc >=  3.3579125411394947E+04_dp .AND. Rc <=  3.9047249374088546E+05_dp) THEN
       irange =  4
       L_b =  3.3579125411394947E+04_dp
       U_b =  3.9047249374088546E+05_dp
     END IF
     IF(Rc >=  3.9047249374088546E+05_dp .AND. Rc <=  1.9873406038870332E+06_dp) THEN
       irange =  5
       L_b =  3.9047249374088546E+05_dp
       U_b =  1.9873406038870332E+06_dp
     END IF
     IF(Rc >=  1.9873406038870332E+06_dp .AND. Rc <=  5.7633609771728516E+06_dp) THEN
       irange =  6
       L_b =  1.9873406038870332E+06_dp
       U_b =  5.7633609771728516E+06_dp
     END IF

     fit_coef(1:13,1:40) = coefdata(1:13,1:40,irange)

   END SUBROUTINE get_fit_coef_k20

 END MODULE minimax_rpa
