package charactermanaj.model;

import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Map;

public class CustomLayerOrderKey implements Comparable<CustomLayerOrderKey> {

	private final String id;

	private final String displayName;

	private final Map<String, String> localizedNames;

	public static final String DEFAULT_NAME_KEY = "";

	public CustomLayerOrderKey(String id, String name, Map<String, String> localizedNames) {
		if (id == null || name == null) {
			throw new NullPointerException("id and name is required.");
		}
		this.id = id;
		this.displayName = name;
		if (localizedNames == null) {
			localizedNames = Collections.emptyMap();
		}
		this.localizedNames = Collections.unmodifiableMap(new HashMap<String, String>(localizedNames));
	}

	public String getId() {
		return id;
	}

	public String getDisplayName() {
		return displayName;
	}

	public Map<String, String> getLocalizedNames() {
		return localizedNames;
	}

	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + ((id == null) ? 0 : id.hashCode());
		return result;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		CustomLayerOrderKey other = (CustomLayerOrderKey) obj;
		if (id == null) {
			if (other.id != null)
				return false;
		} else if (!id.equals(other.id))
			return false;
		return true;
	}

	/**
	 * 表示名によるソート順
	 */
	public static final Comparator<CustomLayerOrderKey> COMPARATOR = new Comparator<CustomLayerOrderKey>() {
		@Override
		public int compare(CustomLayerOrderKey o1, CustomLayerOrderKey o2) {
			// 表示名でソートする
			// ※ ソート順でマップ化した場合、表示名が異なるがIDが同一のものができうるため
			// ツリーマップ化などを使う場合はロジックでID重複を防ぐ必要がある。
			int ret = o1.displayName.compareTo(o2.displayName);
			if (ret == 0) {
				ret = o1.id.compareTo(o2.id);
			}
			return ret;
		}
	};

	@Override
	public int compareTo(CustomLayerOrderKey o) {
		return COMPARATOR.compare(this, o);
	}

	@Override
	public String toString() {
		return displayName;
	}
}
