/*
 * Asterisk -- An open source telephony toolkit.
 *
 * Copyright (C) 1999 - 2008, Digium, Inc.
 *
 * Matthew Nicholson <mnicholson@digium.com>
 *
 * See http://www.asterisk.org for more information about
 * the Asterisk project. Please do not directly contact
 * any of the maintainers of this project for assistance;
 * the project provides a web site, mailing lists and IRC
 * channels for your use.
 *
 * This program is free software, distributed under the terms of
 * the GNU General Public License Version 2. See the LICENSE file
 * at the top of the source tree.
 */

#include <errno.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>

struct mkstring_options {
	char *infile;
	char *outfile;
	char *name;
};

int mkstring(struct mkstring_options *opts) {
	FILE *infile;
	FILE *outfile;
	char c;
	int count;

	if (!(infile = fopen(opts->infile, "r"))) {
		fprintf(stderr, "Error opening file '%s' for reading: %s\n", opts->infile, strerror(errno));
		return 1;
	}
	
	if (!(outfile = fopen(opts->outfile, "w"))) {
		fprintf(stderr, "Error opening file '%s' for wriring: %s\n", opts->outfile, strerror(errno));
		fclose(infile);
		return 1;
	}

	fprintf(outfile,
		"/* This file is automatically generated.  DO NOT EDIT. */\n"
		"\n"
		"const char %s[] = {\n", opts->name);

	count = 0;
	while (fread(&c, sizeof(char), 1, infile)) {
		count += fprintf(outfile, "%3i, ", c);
		if (count > 72) {
			count = 0;
			fprintf(outfile, "\n");
		}
	}

	fprintf(outfile, "\n};\n");

	fclose(infile);
	fclose(outfile);
	return 0;
}

int parse_cmdline(int argc, char *argv[], struct mkstring_options *opts) {
	char c;
	int i;
	while ((c = getopt(argc, argv, "n:o:")) != -1) {
		switch (c) {
		case 'n':
			opts->name = optarg;
			break;
		case 'o':
			opts->outfile = optarg;
			break;
		case '?':
			return 1;
			break;
		}
	}

	for (i = optind; i < argc; i++) {
		opts->infile = argv[i];
	}

	if (!opts->infile
	|| !opts->outfile
	|| !opts->name) {
		fprintf(stderr, "Missing arguments\n");
		return 1;
	}

	return 0;
}

void usage(const char *prog_name) {
	fprintf(stderr, "Usage:\n");
	fprintf(stderr, "  %s -n <string_name> -o <out.h> <in.txt>\n", prog_name);
}

int main(int argc, char *argv[]) {
	struct mkstring_options opts;
	memset(&opts, 0, sizeof(struct mkstring_options));

	if (parse_cmdline(argc, argv, &opts)) {
		usage(argv[0]);
		return 1;
	}

	return mkstring(&opts);
}
