// ---------------------------------------------------------------------------
// - Rco.cpp                                                                 -
// - afnix:csm service - registered cloud object class implementation        -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2015 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Rco.hpp"
#include "Csmsid.hxx"
#include "Vector.hpp"
#include "Boolean.hpp"
#include "Runnable.hpp"
#include "QuarkZone.hpp"
#include "Exception.hpp"

namespace afnix {

  // -------------------------------------------------------------------------
  // - private section                                                       -
  // -------------------------------------------------------------------------

  // this procedure returns a new rco for deserialization
  static Serial* mksob (void) {
    return new Rco;
  }
  // register this rco serial id
  static const t_byte SERIAL_ID = Serial::setsid (SERIAL_RCO_ID, mksob);

  // -------------------------------------------------------------------------
  // - class section                                                         -
  // -------------------------------------------------------------------------

  // create a nil object

  Rco::Rco (void) {
    d_kid = 0LL;
    p_rco = nilp;
  }

  // create an anonymous object

  Rco::Rco (Object* rco) {
    d_kid = 0LL;
    Object::iref (p_rco = rco);
  }

  // create a registered object by name

  Rco::Rco (Object* rco, const String& name) {
    d_kid  = 0LL;
    d_name = name;
    Object::iref (p_rco = rco);
  }

  // create a registered object by name and info

  Rco::Rco (Object* rco, const String& name, const String& info) {
    d_kid  = 0LL;
    d_name = name;
    d_info = info;
    Object::iref (p_rco = rco);
  }

  // create a registered object by rid, name and info

  Rco::Rco (const String& rid,
	    Object* rco, const String& name, const String& info) {
    d_kid  = 0LL;
    d_rid  = rid;
    d_name = name;
    d_info = info;
    Object::iref (p_rco = rco);
  }

  // copy construct this registered object

  Rco::Rco (const Rco& that) {
    that.rdlock ();
    try {
      d_kid  = that.d_kid;
      d_rid  = that.d_rid;
      d_name = that.d_name;
      d_info = that.d_info;
      Object::iref (p_rco = that.p_rco);
      that.unlock ();
    } catch (...) {
      that.unlock ();
      throw;
    }
  }

  // destroy this registered object

  Rco::~Rco (void) {
    Object::dref (p_rco);
  }

  // return the rco class name

  String Rco::repr (void) const {
    return "Rco";
  }

  // return a clone of this object

  Object* Rco::clone (void) const {
    return new Rco (*this);
  }

  // return the rco serial code

  t_byte Rco::serialid (void) const {
    return SERIAL_RCO_ID;
  }

  // serialize this rco

  void Rco::wrstream (OutputStream& os) const {
    rdlock ();
    try {
      csm_wrlong (d_kid, os);
      d_rid.wrstream  (os);
      d_name.wrstream (os);
      d_info.wrstream (os);
      if (p_rco == nilp) {
	Serial::wrnilid (os);
      } else {
	Serial* sobj = dynamic_cast <Serial*> (p_rco);
	if (sobj == nilp) {
	  throw Exception ("serial-error", "cannot serialize object", 
			   p_rco->repr ());
	}
	sobj->serialize (os);
      }
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // deserialize this rco

  void Rco::rdstream (InputStream& is) {
    wrlock ();
    try {
      d_kid = csm_rdlong (is);
      d_rid.rdstream  (is);
      d_name.rdstream (is);
      d_info.rdstream (is);
      Object::iref (p_rco = Serial::deserialize (is));
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // get the registration name

  String Rco::getname (void) const {
    rdlock ();
    try {
      String result = d_name;
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // assign a registered object to this one

  Rco& Rco::operator = (const Rco& that) {
    // check for self assignation
    if (this == &that) return *this;
    // lock and assign
    wrlock ();
    that.rdlock ();
    try {
      // protect the copy
      Object::iref (that.p_rco);
      // remove old object
      Object::dref (p_rco);
      // set local data
      d_kid  = that.d_kid;
      d_rid  = that.d_rid;
      d_name = that.d_name;
      d_info = that.d_info;
      p_rco  = that.p_rco;
      unlock ();
      that.unlock ();
      return *this;
    } catch (...) {
      unlock ();
      that.unlock ();
      throw;
    }
  }

  // validate a rco key id

  bool Rco::iskid (const t_long kid) const {
    rdlock ();
    try {
      bool result = (d_kid == kid);
      unlock ();
      return result;
    } catch  (...) {
      unlock ();
      throw;
    }
  }

  // set the key id

  void Rco::setkid (const t_long kid) {
    wrlock ();
    try {
      d_kid = kid;
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }
  
  // get the key id

  t_long Rco::getkid (void) const {
    rdlock ();
    try {
      t_long result = d_kid;
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // validate a rco registration id

  bool Rco::isrid (const String& rid) const {
    rdlock ();
    try {
      bool result = (d_rid == rid);
      unlock ();
      return result;
    } catch  (...) {
      unlock ();
      throw;
    }
  }

  // set the registration id

  void Rco::setrid (const String& rid) {
    wrlock ();
    try {
      d_rid = rid;
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }
  
  // get the registration id

  String Rco::getrid (void) const {
    rdlock ();
    try {
      String result = d_rid;
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // get the registration info

  String Rco::getinfo (void) const {
    rdlock ();
    try {
      String result = d_info;
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // get the registrered object

  Object* Rco::getobj (void) const {
    rdlock ();
    try {
      Object* result = p_rco;
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }
  
  // -------------------------------------------------------------------------
  // - object section                                                        -
  // -------------------------------------------------------------------------

  // the quark zone
  static const long QUARK_ZONE_LENGTH = 8;
  static QuarkZone  zone (QUARK_ZONE_LENGTH);

  // the object supported quarks
  static const long QUARK_ISKIDP  = zone.intern ("kid-p");
  static const long QUARK_ISRIDP  = zone.intern ("rid-p");
  static const long QUARK_SETRID  = zone.intern ("set-rid");
  static const long QUARK_GETKID  = zone.intern ("get-kid");
  static const long QUARK_SETKID  = zone.intern ("set-kid");
  static const long QUARK_GETRID  = zone.intern ("get-rid");
  static const long QUARK_GETOBJ  = zone.intern ("get-object");
  static const long QUARK_GETINFO = zone.intern ("get-info");

  // create a new object in a generic way

  Object* Rco::mknew (Vector* argv) {
    long argc = (argv == nilp) ? 0 : argv->length ();
    // create a default registered object
    if (argc == 0) return new Rco;
    // check for 1 argument
    if (argc == 1) {
      Object* rco = argv->get (0);
      return new Rco (rco);
    }
    // check for 2 arguments
    if (argc == 2) {
      Object*  rco = argv->get (0);
      String  name = argv->getstring (1);
      return new Rco (rco, name);
    }
    // check for 3 arguments
    if (argc == 3) {
      Object*  rco = argv->get (0);
      String  name = argv->getstring (1);
      String  info = argv->getstring (2);
      return new Rco (rco, name, info);
    }
    // check for 4 arguments
    if (argc == 4) {
      String   rid = argv->getstring (0);
      Object*  rco = argv->get (1);
      String  name = argv->getstring (2);
      String  info = argv->getstring (3);
      return new Rco (rid, rco, name, info);
    }
    throw Exception ("argument-error",
                     "too many argument with rco constructor");
  }

  // return true if the given quark is defined

  bool Rco::isquark (const long quark, const bool hflg) const {
    rdlock ();
    if (zone.exists (quark) == true) {
      unlock ();
      return true;
    }
    bool result = hflg ? Nameable::isquark (quark, hflg) : false;
    if (result == false) {
      result = hflg ? Serial::isquark (quark, hflg) : false;
    }
    unlock ();
    return result;
  }
  
  // apply this object with a set of arguments and a quark
  
  Object* Rco::apply (Runnable* robj, Nameset* nset, const long quark,
		      Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();
    
    // check for 0 argument
    if (argc == 0) {
      if (quark == QUARK_GETKID)  return new Integer (getkid  ());
      if (quark == QUARK_GETRID)  return new String  (getrid  ());
      if (quark == QUARK_GETINFO) return new String  (getinfo ());
      if (quark == QUARK_GETOBJ) {
	rdlock ();
	try {
	  Object* result = getobj ();
	  robj->post (result);
	  unlock ();
	  return result;
	} catch (...) {
	  unlock ();
	  throw;
	}
      }
    }
    // check for 1 argument
    if (argc == 1) {
      if (quark == QUARK_ISKIDP) {
	t_long kid = argv->getlong (0);
	return new Boolean (iskid (kid));
      }
      if (quark == QUARK_ISRIDP) {
	String rid = argv->getstring (0);
	return new Boolean (isrid (rid));
      }
      if (quark == QUARK_SETKID) {
	t_long kid = argv->getlong (0);
	setkid (kid);
	return nilp;
      }
      if (quark == QUARK_SETRID) {
	String rid = argv->getstring (0);
	setrid (rid);
	return nilp;
      }
    }
    // check the nameable method
    if (Nameable::isquark (quark, true) == true) {
      return Nameable::apply (robj, nset, quark, argv);
    }
    // call the serial method
    return Serial::apply (robj, nset, quark, argv);
  }
}
