/* playback.c - high-level audio playback implementation
 *
 * Copyright 2010 Petteri Hintsanen <petterih@iki.fi>
 *
 * This file is part of abx.
 *
 * abx is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * abx is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with abx.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "playback.h"
#include "player.h"
#include <assert.h>
#include <stdio.h>
#include <glib.h>

static Player *player_a;        /* Player for the first sample. */
static Player *player_b;        /* Player for the second sample. */
static Player *current_player;  /* Current player, if playback is in
                                 * progress. */
static sem_t semaphore;         /* Semaphore for synchronizing with
                                 * the players. */

/*
 * Initialize playback using sample files a and b.  Samples must have
 * equal duration.
 *
 * Return
 *   0 on success,
 *   1 if playback has been already initialized,
 *   2 if player for a could not be initialized, 
 *   3 if player for b could not be initialized, or
 *   4 if the samples have different duration.
 */
int 
init_playback(const char *a, const char *b, PaDeviceIndex outdev)
{
    Metadata ma;
    Metadata mb;
    Player *pa;
    Player *pb;

    if (player_a) {
        assert(player_b);
        return 1;
    }

    if (!(pa = init_player(a, outdev))) {
        g_warning("can't initialize player for '%s'", a);
        return 2;
    }

    if (!(pb = init_player(b, outdev))) {
        g_warning("can't initialize player for '%s'", b);
        close_player(pa);
        return 3;
    }

    ma = get_player_metadata(pa);
    mb = get_player_metadata(pb);
    if (ma.duration != mb.duration) {
        g_warning("samples '%s' and '%s' have different duration", a, b);
        close_player(pa);
        close_player(pb);
        return 4;
    }
   
    player_a = pa;
    player_b = pb;
    sem_init(&semaphore, 0, 0);
    return 0;
}

/*
 * Close playback and release resources.
 */
void
close_playback(void)
{
    close_player(player_a);
    close_player(player_b);
    player_a = player_b = current_player = NULL;
    sem_destroy(&semaphore);
}

/*
 * Fill in given metadata structures from initialized players.
 */
void
get_metadatas(Metadata *a, Metadata *b)
{
    assert(a && b);
    *a = get_player_metadata(player_a);
    *b = get_player_metadata(player_b);
}

/*
 * Get the current playback state and store it to *state.  Return the
 * current sample id, or -1 if no playback is in progress.
 */
int
get_playback_state(Player_state *state)
{
    assert(state);
    if (!current_player) {
        return -1;
    } else if (current_player == player_a) {
        *state = get_player_state(player_a);
        return 0;
    } else {
        *state = get_player_state(player_b);
        return 1;
    }
}

/*
 * Play sample from the given location (in seconds).  This function
 * blocks until the playback has been started.
 */
void
start_playback(int sample, double location)
{
    Player *pl;
    /* g_debug("starting playback of sample %d from %f", sample, location); */
    if (sample == 0) pl = player_a;
    else pl = player_b;
    stop_playback();
    seek_player(pl, location, SEEK_SET, NULL);
    start_player(pl, &semaphore);
    current_player = pl;
    sem_wait(&semaphore);
}

/*
 * Stop playback.  This function blocks until the playback has been
 * stopped.
 */
void
stop_playback(void)
{
    if (current_player == NULL) return;
    stop_player(current_player, &semaphore);
    sem_wait(&semaphore);
    current_player = NULL;
}

/*
 * Pause or resume playback.  This function blocks until the playback
 * request has been serviced by the controller thread.
 *
 * Return 0 if the playback was paused, 1 if the playback was resumed,
 * or 2 if there is no current player or playback has been stopped.
 */
int
pause_or_resume_playback(void)
{
    Player_state state;
    if (current_player == NULL) return 2;
    pause_or_resume_player(current_player, &semaphore);
    sem_wait(&semaphore);
    get_playback_state(&state);
    switch (state.playback) {
    case PAUSED:
        return 0;
    case PLAYING:
        return 1;
    case STOPPED:
        return 2;
    };
    return 2;
}

/*
 * Reposition current player to location (in seconds, measured from
 * the beginning).  This function blocks until the seek has been
 * completed.
 */
void
seek_playback(double offset)
{
    if (current_player == NULL) return;
    seek_player(current_player, offset, SEEK_SET, &semaphore);
    sem_wait(&semaphore);
}
