/*
 *  KCemu -- The emulator for the KC85 homecomputer series and much more.
 *  Copyright (C) 1997-2010 Torsten Paul
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifndef __kc_keyb0k_h
#define __kc_keyb0k_h

/*
 *  ===============================================================================
 *  multiplexer
 *  value:  0        1        2        3        4        5        6        7      - (high nibble)
 *  ===============================================================================
 *  shift level 0
 *
 *          @        A        B        C        D        E        F        G      - 0x1e (1)
 *          H        I        J        K        L        M        N        O      - 0x1d (2)
 *          P        Q        R        S        T        U        V        W      - 0x1b (4)
 *        SHIFT1   SHIFT2   SHIFT3   SHIFT4    <--     SPACE     -->     ENTER    - 0x17 (8)
 *
 *  -------------------------------------------------------------------------------
 *  shift level 1
 *
 *          X        Y        Z        [        \        ]        ^        _      - 0x1e (1)
 *          0        1        2        3        4        5        6        7      - 0x1d (2)
 *          8        9        :        ;        <        =        >        ?      - 0x1b (4)
 *        SHIFT1   SHIFT2   SHIFT3   SHIFT4    <--     SPACE     -->     ENTER    - 0x17 (8)
 *
 *  -------------------------------------------------------------------------------
 *  shift level 2
 *
 *          x        y        z        {        |        }        ~       DEL?    - 0x1e (1)
 *        SPACE?     !        "        #        $        %        &        '      - 0x1d (2)
 *          (        )        *        +        ,        -        .        /      - 0x1b (4)
 *        SHIFT1   SHIFT2   SHIFT3   SHIFT4    <--     SPACE     -->     ENTER    - 0x17 (8)
 *
 *  -------------------------------------------------------------------------------
 *  shift level 3
 *
 *          `        a        b        c        d        e        f        g      - 0x1e (1)
 *          h        i        j        k        l        m        n        o      - 0x1d (2)
 *          p        q        r        s        t        u        v        w      - 0x1b (4)
 *        SHIFT1   SHIFT2   SHIFT3   SHIFT4    <--     SPACE     -->     ENTER    - 0x17 (8)
 *  ===============================================================================
 */
static unsigned int __keys[] = {
  /* 0x00 [.] */ 0x0000,
  /* 0x01 [.] */ 0x0000,
  /* 0x02 [.] */ 0x0000,
  /* 0x03 [.] */ 0x0000,
  /* 0x04 [.] */ 0x0000,
  /* 0x05 [.] */ 0x0000,
  /* 0x06 [.] */ 0x0000,
  /* 0x07 [.] */ 0x0000,
  /* 0x08 [.] */ 0x0000,
  /* 0x09 [.] */ 0x0000,
  /* 0x0a [.] */ 0x0000,
  /* 0x0b [.] */ 0x0000,
  /* 0x0c [.] */ 0x0000,
  /* 0x0d [.] */ 0x0078, /* ENTER */
  /* 0x0e [.] */ 0x0000,
  /* 0x0f [.] */ 0x0000,
  /* 0x10 [.] */ 0x0000,
  /* 0x11 [.] */ 0x0000,
  /* 0x12 [.] */ 0x0000,
  /* 0x13 [.] */ 0x0000,
  /* 0x14 [.] */ 0x0000,
  /* 0x15 [.] */ 0x0000,
  /* 0x16 [.] */ 0x0000,
  /* 0x17 [.] */ 0x0000,
  /* 0x18 [.] */ 0x0000,
  /* 0x19 [.] */ 0x0000,
  /* 0x1a [.] */ 0x0000,
  /* 0x1b [.] */ 0x0000,
  /* 0x1c [.] */ 0x0000,
  /* 0x1d [.] */ 0x0000,
  /* 0x1e [.] */ 0x0000,
  /* 0x1f [.] */ 0x0000,
  /* 0x20 [ ] */ 0x0058, /* SPACE */
  /* 0x21 [!] */ 0x0212,
  /* 0x22 ["] */ 0x0222,
  /* 0x23 [#] */ 0x0232,
  /* 0x24 [$] */ 0x0242,
  /* 0x25 [%] */ 0x0252,
  /* 0x26 [&] */ 0x0262,
  /* 0x27 ['] */ 0x0272,
  /* 0x28 [(] */ 0x0204,
  /* 0x29 [)] */ 0x0214,
  /* 0x2a [*] */ 0x0224,
  /* 0x2b [+] */ 0x0234,
  /* 0x2c [,] */ 0x0244,
  /* 0x2d [-] */ 0x0254,
  /* 0x2e [.] */ 0x0264,
  /* 0x2f [/] */ 0x0274,
  /* 0x30 [0] */ 0x0102,
  /* 0x31 [1] */ 0x0112,
  /* 0x32 [2] */ 0x0122,
  /* 0x33 [3] */ 0x0132,
  /* 0x34 [4] */ 0x0142,
  /* 0x35 [5] */ 0x0152,
  /* 0x36 [6] */ 0x0162,
  /* 0x37 [7] */ 0x0172,
  /* 0x38 [8] */ 0x0104,
  /* 0x39 [9] */ 0x0114,
  /* 0x3a [:] */ 0x0124,
  /* 0x3b [;] */ 0x0134,
  /* 0x3c [<] */ 0x0144,
  /* 0x3d [=] */ 0x0154,
  /* 0x3e [>] */ 0x0164,
  /* 0x3f [?] */ 0x0174,
  /* 0x40 [@] */ 0x0001,
  /* 0x41 [A] */ 0x0311,
  /* 0x42 [B] */ 0x0321,
  /* 0x43 [C] */ 0x0331,
  /* 0x44 [D] */ 0x0341,
  /* 0x45 [E] */ 0x0351,
  /* 0x46 [F] */ 0x0361,
  /* 0x47 [G] */ 0x0371,
  /* 0x48 [H] */ 0x0302,
  /* 0x49 [I] */ 0x0312,
  /* 0x4a [J] */ 0x0322,
  /* 0x4b [K] */ 0x0332,
  /* 0x4c [L] */ 0x0342,
  /* 0x4d [M] */ 0x0352,
  /* 0x4e [N] */ 0x0362,
  /* 0x4f [O] */ 0x0372,
  /* 0x50 [P] */ 0x0304,
  /* 0x51 [Q] */ 0x0314,
  /* 0x52 [R] */ 0x0324,
  /* 0x53 [S] */ 0x0334,
  /* 0x54 [T] */ 0x0344,
  /* 0x55 [U] */ 0x0354,
  /* 0x56 [V] */ 0x0364,
  /* 0x57 [W] */ 0x0374,
  /* 0x58 [X] */ 0x0201,
  /* 0x59 [Y] */ 0x0211,
  /* 0x5a [Z] */ 0x0221,
  /* 0x5b [[] */ 0x0131,
  /* 0x5c [\] */ 0x0141,
  /* 0x5d []] */ 0x0151,
  /* 0x5e [^] */ 0x0161,
  /* 0x5f [_] */ 0x0171,
  /* 0x60 [`] */ 0x0301,
  /* 0x61 [a] */ 0x0011,
  /* 0x62 [b] */ 0x0021,
  /* 0x63 [c] */ 0x0031,
  /* 0x64 [d] */ 0x0041,
  /* 0x65 [e] */ 0x0051,
  /* 0x66 [f] */ 0x0061,
  /* 0x67 [g] */ 0x0071,
  /* 0x68 [h] */ 0x0002,
  /* 0x69 [i] */ 0x0012,
  /* 0x6a [j] */ 0x0022,
  /* 0x6b [k] */ 0x0032,
  /* 0x6c [l] */ 0x0042,
  /* 0x6d [m] */ 0x0052,
  /* 0x6e [n] */ 0x0062,
  /* 0x6f [o] */ 0x0072,
  /* 0x70 [p] */ 0x0004,
  /* 0x71 [q] */ 0x0014,
  /* 0x72 [r] */ 0x0024,
  /* 0x73 [s] */ 0x0034,
  /* 0x74 [t] */ 0x0044,
  /* 0x75 [u] */ 0x0054,
  /* 0x76 [v] */ 0x0064,
  /* 0x77 [w] */ 0x0074,
  /* 0x78 [x] */ 0x0101,
  /* 0x79 [y] */ 0x0111,
  /* 0x7a [z] */ 0x0121,
  /* 0x7b [{] */ 0x0231,
  /* 0x7c [|] */ 0x0241,
  /* 0x7d [}] */ 0x0251,
  /* 0x7e [~] */ 0x0261,
  /* 0x7f [.] */ 0x0271,
  /* 0x80 [.] */ 0x0000,
  /* 0x81 [.] */ 0x0000,
  /* 0x82 [.] */ 0x0000,
  /* 0x83 [.] */ 0x0000,
  /* 0x84 [.] */ 0x0000,
  /* 0x85 [.] */ 0x0000,
  /* 0x86 [.] */ 0x0000,
  /* 0x87 [.] */ 0x0000,
  /* 0x88 [.] */ 0x0000,
  /* 0x89 [.] */ 0x0000,
  /* 0x8a [.] */ 0x0000,
  /* 0x8b [.] */ 0x0000,
  /* 0x8c [.] */ 0x0000,
  /* 0x8d [.] */ 0x0000,
  /* 0x8e [.] */ 0x0000,
  /* 0x8f [.] */ 0x0000,
  /* 0x90 [.] */ 0x0000,
  /* 0x91 [.] */ 0x0000,
  /* 0x92 [.] */ 0x0000,
  /* 0x93 [.] */ 0x0000,
  /* 0x94 [.] */ 0x0000,
  /* 0x95 [.] */ 0x0000,
  /* 0x96 [.] */ 0x0000,
  /* 0x97 [.] */ 0x0000,
  /* 0x98 [.] */ 0x0000,
  /* 0x99 [.] */ 0x0000,
  /* 0x9a [.] */ 0x0000,
  /* 0x9b [.] */ 0x0000,
  /* 0x9c [.] */ 0x0000,
  /* 0x9d [.] */ 0x0000,
  /* 0x9e [.] */ 0x0000,
  /* 0x9f [.] */ 0x0000,
  /* 0xa0 [.] */ 0x0000,
  /* 0xa1 [.] */ 0x0000,
  /* 0xa2 [.] */ 0x0000,
  /* 0xa3 [.] */ 0x0000,
  /* 0xa4 [.] */ 0x0000,
  /* 0xa5 [.] */ 0x0000,
  /* 0xa6 [.] */ 0x0000,
  /* 0xa7 [.] */ 0x0000,
  /* 0xa8 [.] */ 0x0000,
  /* 0xa9 [.] */ 0x0000,
  /* 0xaa [.] */ 0x0000,
  /* 0xab [.] */ 0x0000,
  /* 0xac [.] */ 0x0000,
  /* 0xad [.] */ 0x0000,
  /* 0xae [.] */ 0x0000,
  /* 0xaf [.] */ 0x0000,
  /* 0xb0 [.] */ 0x0000,
  /* 0xb1 [.] */ 0x0000,
  /* 0xb2 [.] */ 0x0000,
  /* 0xb3 [.] */ 0x0000,
  /* 0xb4 [.] */ 0x0000,
  /* 0xb5 [.] */ 0x0000,
  /* 0xb6 [.] */ 0x0000,
  /* 0xb7 [.] */ 0x0000,
  /* 0xb8 [.] */ 0x0000,
  /* 0xb9 [.] */ 0x0000,
  /* 0xba [.] */ 0x0000,
  /* 0xbb [.] */ 0x0000,
  /* 0xbc [.] */ 0x0000,
  /* 0xbd [.] */ 0x0000,
  /* 0xbe [.] */ 0x0000,
  /* 0xbf [.] */ 0x0000,
  /* 0xc0 [.] */ 0x0000,
  /* 0xc1 [.] */ 0x0000,
  /* 0xc2 [.] */ 0x0000,
  /* 0xc3 [.] */ 0x0000,
  /* 0xc4 [.] */ 0x0000,
  /* 0xc5 [.] */ 0x0000,
  /* 0xc6 [.] */ 0x0000,
  /* 0xc7 [.] */ 0x0000,
  /* 0xc8 [.] */ 0x0000,
  /* 0xc9 [.] */ 0x0000,
  /* 0xca [.] */ 0x0000,
  /* 0xcb [.] */ 0x0000,
  /* 0xcc [.] */ 0x0000,
  /* 0xcd [.] */ 0x0000,
  /* 0xce [.] */ 0x0000,
  /* 0xcf [.] */ 0x0000,
  /* 0xd0 [.] */ 0x0000,
  /* 0xd1 [.] */ 0x0000,
  /* 0xd2 [.] */ 0x0000,
  /* 0xd3 [.] */ 0x0000,
  /* 0xd4 [.] */ 0x0000,
  /* 0xd5 [.] */ 0x0000,
  /* 0xd6 [.] */ 0x0000,
  /* 0xd7 [.] */ 0x0000,
  /* 0xd8 [.] */ 0x0000,
  /* 0xd9 [.] */ 0x0000,
  /* 0xda [.] */ 0x0000,
  /* 0xdb [.] */ 0x0000,
  /* 0xdc [.] */ 0x0000,
  /* 0xdd [.] */ 0x0000,
  /* 0xde [.] */ 0x0000,
  /* 0xdf [.] */ 0x0000,
  /* 0xe0 [.] */ 0x0000,
  /* 0xe1 [.] */ 0x0000,
  /* 0xe2 [.] */ 0x0000,
  /* 0xe3 [.] */ 0x0000,
  /* 0xe4 [.] */ 0x0000,
  /* 0xe5 [.] */ 0x0000,
  /* 0xe6 [.] */ 0x0000,
  /* 0xe7 [.] */ 0x0000,
  /* 0xe8 [.] */ 0x0000,
  /* 0xe9 [.] */ 0x0000,
  /* 0xea [.] */ 0x0000,
  /* 0xeb [.] */ 0x0000,
  /* 0xec [.] */ 0x0000,
  /* 0xed [.] */ 0x0000,
  /* 0xee [.] */ 0x0000,
  /* 0xef [.] */ 0x0000,
  /* 0xf0 [.] */ 0x0000,
  /* 0xf1 [.] */ 0x0000,
  /* 0xf2 [.] */ 0x0000,
  /* 0xf3 [.] */ 0x0000,
  /* 0xf4 [.] */ 0x0000,
  /* 0xf5 [.] */ 0x0000,
  /* 0xf6 [.] */ 0x0000,
  /* 0xf7 [.] */ 0x0000,
  /* 0xf8 [.] */ 0x0000,
  /* 0xf9 [.] */ 0x0000,
  /* 0xfa [.] */ 0x0000,
  /* 0xfb [.] */ 0x0000,
  /* 0xfc [.] */ 0x0000,
  /* 0xfd [.] */ 0x0000,
  /* 0xfe [.] */ 0x0000,
  /* 0xff [.] */ 0x0000,

  /* 0x100 */ 0x0000,
  /* 0x101 */ 0x0000,
  /* 0x102 */ 0x0000,
  /* 0x103 */ 0x0000,
  /* 0x104 */ 0x0000,
  /* 0x105 */ 0x0000,
  /* 0x106 */ 0x0000,
  /* 0x107 */ 0x0000,
  /* 0x108 */ 0x0000,
  /* 0x109 */ 0x0000,
  /* 0x10a */ 0x0000,
  /* 0x10b */ 0x0000,
  /* 0x10c */ 0x0000,
  /* 0x10d */ 0x0000,
  /* 0x10e */ 0x0000,
  /* 0x10f */ 0x0000,
  /* 0x110 */ 0x0000,   /* SHIFT */
  /* 0x111 */ 0x0000,   /* CONTROL */
  /* 0x112 */ 0x0000,   /* ALT */
  /* 0x113 */ 0x0000,   /* ALT_GR */
  /* 0x114 */ 0x0048,   /* LEFT */
  /* 0x115 */ 0x0068,   /* RIGHT */
  /* 0x116 */ 0x0000,   /* DOWN */
  /* 0x117 */ 0x0000,   /* UP */
  /* 0x118 */ 0x0000,   /* HOME */
  /* 0x119 */ 0x0000,   /* END */
  /* 0x11a */ 0x0000,   /* DEL */
  /* 0x11b */ 0x0000,   /* INSERT */
  /* 0x11c */ 0x0000,
  /* 0x11d */ 0x0000,
  /* 0x11e */ 0x0000,   /* PAUSE */
  /* 0x11f */ 0x0000,
  /* 0x120 */ 0x0000,   /* COLOR */
  /* 0x121 */ 0x0000,   /* GRAPHIC */
  /* 0x122 */ 0x0000,   /* LIST */
  /* 0x123 */ 0x0000,   /* RUN */
  /* 0x124 */ 0x0000,   /* STOP */
  /* 0x125 */ 0x0000,   /* SHIFT LOCK */

  /* 0x126 */ 0x0000,
  /* 0x127 */ 0x0000,
  /* 0x128 */ 0x0000,
  /* 0x129 */ 0x0000,
  /* 0x12a */ 0x0000,
  /* 0x12b */ 0x0000,
  /* 0x12c */ 0x0000,
  /* 0x12d */ 0x0000,
  /* 0x12e */ 0x0000,
  /* 0x12f */ 0x0000,
};

/*
 *
 */
static unsigned int __keys_a2[] = {
  /* 0x00 [.] */ 0x0000,
  /* 0x01 [.] */ 0x0000,
  /* 0x02 [.] */ 0x0000,
  /* 0x03 [.] */ 0x0000,
  /* 0x04 [.] */ 0x0000,
  /* 0x05 [.] */ 0x0000,
  /* 0x06 [.] */ 0x0000,
  /* 0x07 [.] */ 0x0000,
  /* 0x08 [.] */ 0x0000,
  /* 0x09 [.] */ 0x0000,
  /* 0x0a [.] */ 0x0000,
  /* 0x0b [.] */ 0x0000,
  /* 0x0c [.] */ 0x0000,
  /* 0x0d [.] */ 0x0017, /* ENTER */
  /* 0x0e [.] */ 0x0000,
  /* 0x0f [.] */ 0x0000,
  /* 0x10 [.] */ 0x0000,
  /* 0x11 [.] */ 0x0000,
  /* 0x12 [.] */ 0x0000,
  /* 0x13 [.] */ 0x0000,
  /* 0x14 [.] */ 0x0000,
  /* 0x15 [.] */ 0x0000,
  /* 0x16 [.] */ 0x0000,
  /* 0x17 [.] */ 0x0000,
  /* 0x18 [.] */ 0x0000,
  /* 0x19 [.] */ 0x0000,
  /* 0x1a [.] */ 0x0000,
  /* 0x1b [.] */ 0x0000,
  /* 0x1c [.] */ 0x0000,
  /* 0x1d [.] */ 0x0000,
  /* 0x1e [.] */ 0x0000,
  /* 0x1f [.] */ 0x0000,
  /* 0x20 [ ] */ 0x0047, /* SPACE */
  /* 0x21 [!] */ 0x6801,
  /* 0x22 ["] */ 0x6841,
  /* 0x23 [#] */ 0x6802,
  /* 0x24 [$] */ 0x6842,
  /* 0x25 [%] */ 0x6803,
  /* 0x26 [&] */ 0x6843,
  /* 0x27 ['] */ 0x6804,
  /* 0x28 [(] */ 0x6844,
  /* 0x29 [)] */ 0x6805,
  /* 0x2a [*] */ 0x0026,
  /* 0x2b [+] */ 0x0065,
  /* 0x2c [,] */ 0x0074,
  /* 0x2d [-] */ 0x0006,
  /* 0x2e [.] */ 0x0035,
  /* 0x2f [/] */ 0x0075,
  /* 0x30 [0] */ 0x0045,
  /* 0x31 [1] */ 0x0001,
  /* 0x32 [2] */ 0x0041,
  /* 0x33 [3] */ 0x0002,
  /* 0x34 [4] */ 0x0042,
  /* 0x35 [5] */ 0x0003,
  /* 0x36 [6] */ 0x0043,
  /* 0x37 [7] */ 0x0004,
  /* 0x38 [8] */ 0x0044,
  /* 0x39 [9] */ 0x0005,
  /* 0x3a [:] */ 0x6826,
  /* 0x3b [;] */ 0x6865,
  /* 0x3c [<] */ 0x6874,
  /* 0x3d [=] */ 0x6806,
  /* 0x3e [>] */ 0x6835,
  /* 0x3f [?] */ 0x6875,
  /* 0x40 [@] */ 0x0016,
  /* 0x41 [A] */ 0x6821,
  /* 0x42 [B] */ 0x6833,
  /* 0x43 [C] */ 0x6832,
  /* 0x44 [D] */ 0x6822,
  /* 0x45 [E] */ 0x6812,
  /* 0x46 [F] */ 0x6862,
  /* 0x47 [G] */ 0x6823,
  /* 0x48 [H] */ 0x6863,
  /* 0x49 [I] */ 0x6854,
  /* 0x4a [J] */ 0x6824,
  /* 0x4b [K] */ 0x6864,
  /* 0x4c [L] */ 0x6825,
  /* 0x4d [M] */ 0x6834,
  /* 0x4e [N] */ 0x6873,
  /* 0x4f [O] */ 0x6815,
  /* 0x50 [P] */ 0x6855,
  /* 0x51 [Q] */ 0x6811,
  /* 0x52 [R] */ 0x6852,
  /* 0x53 [S] */ 0x6861,
  /* 0x54 [T] */ 0x6813,
  /* 0x55 [U] */ 0x6814,
  /* 0x56 [V] */ 0x6872,
  /* 0x57 [W] */ 0x6851,
  /* 0x58 [X] */ 0x6871,
  /* 0x59 [Y] */ 0x6831,
  /* 0x5a [Z] */ 0x6853,
  /* 0x5b [[] */ 0x0046,
  /* 0x5c [\] */ 0x0066,
  /* 0x5d []] */ 0x0056,
  /* 0x5e [^] */ 0x0036,
  /* 0x5f [_] */ 0x0076,
  /* 0x60 [`] */ 0x6816,
  /* 0x61 [a] */ 0x0021,
  /* 0x62 [b] */ 0x0033,
  /* 0x63 [c] */ 0x0032,
  /* 0x64 [d] */ 0x0022,
  /* 0x65 [e] */ 0x0012,
  /* 0x66 [f] */ 0x0062,
  /* 0x67 [g] */ 0x0023,
  /* 0x68 [h] */ 0x0063,
  /* 0x69 [i] */ 0x0054,
  /* 0x6a [j] */ 0x0024,
  /* 0x6b [k] */ 0x0064,
  /* 0x6c [l] */ 0x0025,
  /* 0x6d [m] */ 0x0034,
  /* 0x6e [n] */ 0x0073,
  /* 0x6f [o] */ 0x0015,
  /* 0x70 [p] */ 0x0055,
  /* 0x71 [q] */ 0x0011,
  /* 0x72 [r] */ 0x0052,
  /* 0x73 [s] */ 0x0061,
  /* 0x74 [t] */ 0x0013,
  /* 0x75 [u] */ 0x0014,
  /* 0x76 [v] */ 0x0072,
  /* 0x77 [w] */ 0x0051,
  /* 0x78 [x] */ 0x0071,
  /* 0x79 [y] */ 0x0031,
  /* 0x7a [z] */ 0x0053,
  /* 0x7b [{] */ 0x6846,
  /* 0x7c [|] */ 0x6866,
  /* 0x7d [}] */ 0x6856,
  /* 0x7e [~] */ 0x6836,
  /* 0x7f [.] */ 0x0000,
  /* 0x80 [.] */ 0x0000,
  /* 0x81 [.] */ 0x0000,
  /* 0x82 [.] */ 0x0000,
  /* 0x83 [.] */ 0x0000,
  /* 0x84 [.] */ 0x0000,
  /* 0x85 [.] */ 0x0000,
  /* 0x86 [.] */ 0x0000,
  /* 0x87 [.] */ 0x0000,
  /* 0x88 [.] */ 0x0000,
  /* 0x89 [.] */ 0x0000,
  /* 0x8a [.] */ 0x0000,
  /* 0x8b [.] */ 0x0000,
  /* 0x8c [.] */ 0x0000,
  /* 0x8d [.] */ 0x0000,
  /* 0x8e [.] */ 0x0000,
  /* 0x8f [.] */ 0x0000,
  /* 0x90 [.] */ 0x0000,
  /* 0x91 [.] */ 0x0000,
  /* 0x92 [.] */ 0x0000,
  /* 0x93 [.] */ 0x0000,
  /* 0x94 [.] */ 0x0000,
  /* 0x95 [.] */ 0x0000,
  /* 0x96 [.] */ 0x0000,
  /* 0x97 [.] */ 0x0000,
  /* 0x98 [.] */ 0x0000,
  /* 0x99 [.] */ 0x0000,
  /* 0x9a [.] */ 0x0000,
  /* 0x9b [.] */ 0x0000,
  /* 0x9c [.] */ 0x0000,
  /* 0x9d [.] */ 0x0000,
  /* 0x9e [.] */ 0x0000,
  /* 0x9f [.] */ 0x0000,
  /* 0xa0 [.] */ 0x0000,
  /* 0xa1 [.] */ 0x0000,
  /* 0xa2 [.] */ 0x0000,
  /* 0xa3 [.] */ 0x0000,
  /* 0xa4 [.] */ 0x0000,
  /* 0xa5 [.] */ 0x0000,
  /* 0xa6 [.] */ 0x0000,
  /* 0xa7 [.] */ 0x0000,
  /* 0xa8 [.] */ 0x0000,
  /* 0xa9 [.] */ 0x0000,
  /* 0xaa [.] */ 0x0000,
  /* 0xab [.] */ 0x0000,
  /* 0xac [.] */ 0x0000,
  /* 0xad [.] */ 0x0000,
  /* 0xae [.] */ 0x0000,
  /* 0xaf [.] */ 0x0000,
  /* 0xb0 [.] */ 0x0000,
  /* 0xb1 [.] */ 0x0000,
  /* 0xb2 [.] */ 0x0000,
  /* 0xb3 [.] */ 0x0000,
  /* 0xb4 [.] */ 0x0000,
  /* 0xb5 [.] */ 0x0000,
  /* 0xb6 [.] */ 0x0000,
  /* 0xb7 [.] */ 0x0000,
  /* 0xb8 [.] */ 0x0000,
  /* 0xb9 [.] */ 0x0000,
  /* 0xba [.] */ 0x0000,
  /* 0xbb [.] */ 0x0000,
  /* 0xbc [.] */ 0x0000,
  /* 0xbd [.] */ 0x0000,
  /* 0xbe [.] */ 0x0000,
  /* 0xbf [.] */ 0x0000,
  /* 0xc0 [.] */ 0x0000,
  /* 0xc1 [.] */ 0x0000,
  /* 0xc2 [.] */ 0x0000,
  /* 0xc3 [.] */ 0x0000,
  /* 0xc4 [.] */ 0x0000,
  /* 0xc5 [.] */ 0x0000,
  /* 0xc6 [.] */ 0x0000,
  /* 0xc7 [.] */ 0x0000,
  /* 0xc8 [.] */ 0x0000,
  /* 0xc9 [.] */ 0x0000,
  /* 0xca [.] */ 0x0000,
  /* 0xcb [.] */ 0x0000,
  /* 0xcc [.] */ 0x0000,
  /* 0xcd [.] */ 0x0000,
  /* 0xce [.] */ 0x0000,
  /* 0xcf [.] */ 0x0000,
  /* 0xd0 [.] */ 0x0000,
  /* 0xd1 [.] */ 0x0000,
  /* 0xd2 [.] */ 0x0000,
  /* 0xd3 [.] */ 0x0000,
  /* 0xd4 [.] */ 0x0000,
  /* 0xd5 [.] */ 0x0000,
  /* 0xd6 [.] */ 0x0000,
  /* 0xd7 [.] */ 0x0000,
  /* 0xd8 [.] */ 0x0000,
  /* 0xd9 [.] */ 0x0000,
  /* 0xda [.] */ 0x0000,
  /* 0xdb [.] */ 0x0000,
  /* 0xdc [.] */ 0x0000,
  /* 0xdd [.] */ 0x0000,
  /* 0xde [.] */ 0x0000,
  /* 0xdf [.] */ 0x0000,
  /* 0xe0 [.] */ 0x0000,
  /* 0xe1 [.] */ 0x0000,
  /* 0xe2 [.] */ 0x0000,
  /* 0xe3 [.] */ 0x0000,
  /* 0xe4 [.] */ 0x0000,
  /* 0xe5 [.] */ 0x0000,
  /* 0xe6 [.] */ 0x0000,
  /* 0xe7 [.] */ 0x0000,
  /* 0xe8 [.] */ 0x0000,
  /* 0xe9 [.] */ 0x0000,
  /* 0xea [.] */ 0x0000,
  /* 0xeb [.] */ 0x0000,
  /* 0xec [.] */ 0x0000,
  /* 0xed [.] */ 0x0000,
  /* 0xee [.] */ 0x0000,
  /* 0xef [.] */ 0x0000,
  /* 0xf0 [.] */ 0x0000,
  /* 0xf1 [.] */ 0x0000,
  /* 0xf2 [.] */ 0x0000,
  /* 0xf3 [.] */ 0x0000,
  /* 0xf4 [.] */ 0x0000,
  /* 0xf5 [.] */ 0x0000,
  /* 0xf6 [.] */ 0x0000,
  /* 0xf7 [.] */ 0x0000,
  /* 0xf8 [.] */ 0x0000,
  /* 0xf9 [.] */ 0x0000,
  /* 0xfa [.] */ 0x0000,
  /* 0xfb [.] */ 0x0000,
  /* 0xfc [.] */ 0x0000,
  /* 0xfd [.] */ 0x0000,
  /* 0xfe [.] */ 0x0000,
  /* 0xff [.] */ 0x0000,

  /* 0x100 */ 0x0000,
  /* 0x101 */ 0x000c,
  /* 0x102 */ 0x001c,
  /* 0x103 */ 0x002c,
  /* 0x104 */ 0x003c,
  /* 0x105 */ 0x004c,
  /* 0x106 */ 0x005c,
  /* 0x107 */ 0x006c,
  /* 0x108 */ 0x007c,
  /* 0x109 */ 0x008c,
  /* 0x10a */ 0x009c,
  /* 0x10b */ 0x00ac,
  /* 0x10c */ 0x0032,   /*  */
  /* 0x10d */ 0x0000,
  /* 0x10e */ 0x0000,
  /* 0x10f */ 0x0000,
  /* 0x110 */ 0x0068,   /* SHIFT */
  /* 0x111 */ 0x0057,   /* CONTROL */
  /* 0x112 */ 0x0000,   /* ALT */
  /* 0x113 */ 0x0000,   /* ALT_GR */
  /* 0x114 */ 0x0027,   /* LEFT */
  /* 0x115 */ 0x0037,   /* RIGHT */
  /* 0x116 */ 0x0077,   /* DOWN */
  /* 0x117 */ 0x0067,   /* UP */
  /* 0x118 */ 0x0000,   /* HOME */
  /* 0x119 */ 0x0000,   /* END */
  /* 0x11a */ 0x0000,   /* DEL */
  /* 0x11b */ 0x0000,   /* INSERT */
  /* 0x11c */ 0x0000,
  /* 0x11d */ 0x0000,
  /* 0x11e */ 0x0000,   /* PAUSE */
  /* 0x11f */ 0x0000,
  /* 0x120 */ 0x0000,   /* COLOR */
  /* 0x121 */ 0x0000,   /* GRAPHIC */
  /* 0x122 */ 0x0000,   /* LIST */
  /* 0x123 */ 0x0000,   /* RUN */
  /* 0x124 */ 0x0000,   /* STOP */
  /* 0x125 */ 0x0078,   /* SHIFT LOCK */

  /* 0x126 */ 0x0000,
  /* 0x127 */ 0x0000,
  /* 0x128 */ 0x0000,
  /* 0x129 */ 0x0000,
  /* 0x12a */ 0x0000,
  /* 0x12b */ 0x0000,
  /* 0x12c */ 0x0000,
  /* 0x12d */ 0x0000,
  /* 0x12e */ 0x0000,
  /* 0x12f */ 0x0000,
};


/*
 *  Brosig monitor 'Z1013+K7659/2.028 RB'
 *
 *  ohne Shift:
 *
 *           0       1       2       3       4       5       6       7
 *   +-----------------------------------------------------------------------+
 *  0:       1       Q       A       Y       2       W       S       X       | 1
 *  1:       3       E       D       C       4       R       F       V       | 2
 *  2:       5       T       G       B       6       Z       H       N       | 3
 *  3:       7       U       J       M       8       I       K       ,       | 4
 *  4:       9       O       L       .       0       P       \(�)    -       | 5
 *  5:       ~(�)    ](�)    [(�)    <       +       #       ^       @       | 6
 *  6:       >       ENTER   UP      LEFT    SPACE   SHLOCK  DOWN    RIGHT   | 7
 *  7:                                                       SHIFT   CTRL    | 8
 *  8:               DEL     TAB     MONITOR ESC     {       }       2MHz    | 9
 *  9:       4MHz    32/64   ZG-SW   CTRL-U  CTRL-Y  CTRL-R  CTRL-P  CTRL-C  | a
 * 10:       CTRL-F          GRAFIK                  BEEP            CTRL-S  | b
 * 11:       CTRL-]                          WIPPE+          WIPPE-          | c
 *
 *
 *
 *  mit Shift:
 *
 *           0       1       2       3       4       5       6       7
 *   +-----------------------------------------------------------------------+
 *  0:       !       q       a       y       "       w       s       x       | 1
 *  1:       @       e       d       c       $       r       f       v       | 2
 *  2:       %       t       g       b       &       z       h       n       | 3
 *  3:       /       u       j       m       (       i       k       ;       | 4
 *  4:       )       o       l       :       =       p       �       _       | 5
 *  5:       ?       }       {       [       *       '       |       \       | 6
 *  6:       ]       ENTER   UP      LEFT    SPACE   SM-TAST DOWN    RIGHT   | 7
 *  7:                                                       SHIFT   CTRL    | 8
 *  8:               CTRL-G  CTRL-S  MONITOR ESC     `               JP 100  | 9
 *  9:                               CTRL-O  CTRL-X  CTRL-Q          CTRL-C  | a
 * 10:       NEWLINE         GRAFIK                  BEEP                    | b
 * 11:                                       CTRL-Y          CTRL-X          | c
 */
static unsigned int __keys_rb_k7659[] = {
  /* 0x00 [.] */ 0x0000,
  /* 0x01 [.] */ 0x0000, /* CTRL-A */
  /* 0x02 [.] */ 0x0000, /* CTRL-B */
  /* 0x03 [.] */ 0x007a, /* CTRL-C */
  /* 0x04 [.] */ 0x0000, /* CTRL-D */
  /* 0x05 [.] */ 0x0000, /* CTRL-E */
  /* 0x06 [.] */ 0x000b, /* CTRL-F */
  /* 0x07 [.] */ 0x6819, /* CTRL-G */
  /* 0x08 [.] */ 0x0000, /* CTRL-H */
  /* 0x09 [.] */ 0x0000, /* CTRL-I */
  /* 0x0a [.] */ 0x680b, /* CTRL-J, NEWLINE */
  /* 0x0b [.] */ 0x0000, /* CTRL-K */
  /* 0x0c [.] */ 0x0000, /* CTRL-L */
  /* 0x0d [.] */ 0x0017, /* CTRL-M, ENTER */
  /* 0x0e [.] */ 0x0000, /* CTRL-N */
  /* 0x0f [.] */ 0x683a, /* CTRL-O */
  /* 0x10 [.] */ 0x006a, /* CTRL-P */
  /* 0x11 [.] */ 0x0000, /* CTRL-Q */
  /* 0x12 [.] */ 0x0000, /* CTRL-R */
  /* 0x13 [.] */ 0x007b, /* CTRL-S */
  /* 0x14 [.] */ 0x0000, /* CTRL-T */
  /* 0x15 [.] */ 0x003a, /* CTRL-U */
  /* 0x16 [.] */ 0x005a, /* CTRL-V */
  /* 0x17 [.] */ 0x0000, /* CTRL-W */
  /* 0x18 [.] */ 0x684a, /* CTRL-X */
  /* 0x19 [.] */ 0x004a, /* CTRL-Y */
  /* 0x1a [.] */ 0x0000, /* CTRL-Z */
  /* 0x1b [.] */ 0x0049, /* ESC */
  /* 0x1c [.] */ 0x0000,
  /* 0x1d [.] */ 0x0000,
  /* 0x1e [.] */ 0x0000,
  /* 0x1f [.] */ 0x0000,
  /* 0x20 [ ] */ 0x0047, /* SPACE */
  /* 0x21 [!] */ 0x6801,
  /* 0x22 ["] */ 0x6841,
  /* 0x23 [#] */ 0x0056,
  /* 0x24 [$] */ 0x6842,
  /* 0x25 [%] */ 0x6803,
  /* 0x26 [&] */ 0x6843,
  /* 0x27 ['] */ 0x6856,
  /* 0x28 [(] */ 0x6844,
  /* 0x29 [)] */ 0x6805,
  /* 0x2a [*] */ 0x6846,
  /* 0x2b [+] */ 0x0046,
  /* 0x2c [,] */ 0x0074,
  /* 0x2d [-] */ 0x0075,
  /* 0x2e [.] */ 0x0035,
  /* 0x2f [/] */ 0x6804,
  /* 0x30 [0] */ 0x0045,
  /* 0x31 [1] */ 0x0001,
  /* 0x32 [2] */ 0x0041,
  /* 0x33 [3] */ 0x0002,
  /* 0x34 [4] */ 0x0042,
  /* 0x35 [5] */ 0x0003,
  /* 0x36 [6] */ 0x0043,
  /* 0x37 [7] */ 0x0004,
  /* 0x38 [8] */ 0x0044,
  /* 0x39 [9] */ 0x0005,
  /* 0x3a [:] */ 0x6835,
  /* 0x3b [;] */ 0x6874,
  /* 0x3c [<] */ 0x0036,
  /* 0x3d [=] */ 0x6845,
  /* 0x3e [>] */ 0x0007,
  /* 0x3f [?] */ 0x6806,
  /* 0x40 [@] */ 0x6802,
  /* 0x41 [A] */ 0x6821,
  /* 0x42 [B] */ 0x6833,
  /* 0x43 [C] */ 0x6832,
  /* 0x44 [D] */ 0x6822,
  /* 0x45 [E] */ 0x6812,
  /* 0x46 [F] */ 0x6862,
  /* 0x47 [G] */ 0x6823,
  /* 0x48 [H] */ 0x6863,
  /* 0x49 [I] */ 0x6854,
  /* 0x4a [J] */ 0x6824,
  /* 0x4b [K] */ 0x6864,
  /* 0x4c [L] */ 0x6825,
  /* 0x4d [M] */ 0x6834,
  /* 0x4e [N] */ 0x6873,
  /* 0x4f [O] */ 0x6815,
  /* 0x50 [P] */ 0x6855,
  /* 0x51 [Q] */ 0x6811,
  /* 0x52 [R] */ 0x6852,
  /* 0x53 [S] */ 0x6861,
  /* 0x54 [T] */ 0x6813,
  /* 0x55 [U] */ 0x6814,
  /* 0x56 [V] */ 0x6872,
  /* 0x57 [W] */ 0x6851,
  /* 0x58 [X] */ 0x6871,
  /* 0x59 [Y] */ 0x6831,
  /* 0x5a [Z] */ 0x6853,
  /* 0x5b [[] */ 0x6836,
  /* 0x5c [\] */ 0x6876,
  /* 0x5d []] */ 0x6807,
  /* 0x5e [^] */ 0x0066,
  /* 0x5f [_] */ 0x6875,
  /* 0x60 [`] */ 0x6859,
  /* 0x61 [a] */ 0x0021,
  /* 0x62 [b] */ 0x0033,
  /* 0x63 [c] */ 0x0032,
  /* 0x64 [d] */ 0x0022,
  /* 0x65 [e] */ 0x0012,
  /* 0x66 [f] */ 0x0062,
  /* 0x67 [g] */ 0x0023,
  /* 0x68 [h] */ 0x0063,
  /* 0x69 [i] */ 0x0054,
  /* 0x6a [j] */ 0x0024,
  /* 0x6b [k] */ 0x0064,
  /* 0x6c [l] */ 0x0025,
  /* 0x6d [m] */ 0x0034,
  /* 0x6e [n] */ 0x0073,
  /* 0x6f [o] */ 0x0015,
  /* 0x70 [p] */ 0x0055,
  /* 0x71 [q] */ 0x0011,
  /* 0x72 [r] */ 0x0052,
  /* 0x73 [s] */ 0x0061,
  /* 0x74 [t] */ 0x0013,
  /* 0x75 [u] */ 0x0014,
  /* 0x76 [v] */ 0x0072,
  /* 0x77 [w] */ 0x0051,
  /* 0x78 [x] */ 0x0071,
  /* 0x79 [y] */ 0x0031,
  /* 0x7a [z] */ 0x0053,
  /* 0x7b [{] */ 0x6826,
  /* 0x7c [|] */ 0x6866,
  /* 0x7d [}] */ 0x6816,
  /* 0x7e [~] */ 0x0006,
  /* 0x7f [.] */ 0x0000,
  /* 0x80 [.] */ 0x0000,
  /* 0x81 [.] */ 0x0000,
  /* 0x82 [.] */ 0x0000,
  /* 0x83 [.] */ 0x0000,
  /* 0x84 [.] */ 0x0000,
  /* 0x85 [.] */ 0x0000,
  /* 0x86 [.] */ 0x0000,
  /* 0x87 [.] */ 0x0000,
  /* 0x88 [.] */ 0x0000,
  /* 0x89 [.] */ 0x0000,
  /* 0x8a [.] */ 0x0000,
  /* 0x8b [.] */ 0x0000,
  /* 0x8c [.] */ 0x0000,
  /* 0x8d [.] */ 0x0000,
  /* 0x8e [.] */ 0x0000,
  /* 0x8f [.] */ 0x0000,
  /* 0x90 [.] */ 0x0000,
  /* 0x91 [.] */ 0x0000,
  /* 0x92 [.] */ 0x0000,
  /* 0x93 [.] */ 0x0000,
  /* 0x94 [.] */ 0x0000,
  /* 0x95 [.] */ 0x0000,
  /* 0x96 [.] */ 0x0000,
  /* 0x97 [.] */ 0x0000,
  /* 0x98 [.] */ 0x0000,
  /* 0x99 [.] */ 0x0000,
  /* 0x9a [.] */ 0x0000,
  /* 0x9b [.] */ 0x0000,
  /* 0x9c [.] */ 0x0000,
  /* 0x9d [.] */ 0x0000,
  /* 0x9e [.] */ 0x0000,
  /* 0x9f [.] */ 0x0000,
  /* 0xa0 [.] */ 0x0000,
  /* 0xa1 [.] */ 0x0000,
  /* 0xa2 [.] */ 0x0000,
  /* 0xa3 [.] */ 0x0000,
  /* 0xa4 [.] */ 0x0000,
  /* 0xa5 [.] */ 0x0000,
  /* 0xa6 [.] */ 0x0000,
  /* 0xa7 [.] */ 0x0000,
  /* 0xa8 [.] */ 0x0000,
  /* 0xa9 [.] */ 0x0000,
  /* 0xaa [.] */ 0x0000,
  /* 0xab [.] */ 0x0000,
  /* 0xac [.] */ 0x0000,
  /* 0xad [.] */ 0x0000,
  /* 0xae [.] */ 0x0000,
  /* 0xaf [.] */ 0x0000,
  /* 0xb0 [.] */ 0x0000,
  /* 0xb1 [.] */ 0x0000,
  /* 0xb2 [.] */ 0x0000,
  /* 0xb3 [.] */ 0x0000,
  /* 0xb4 [.] */ 0x0000,
  /* 0xb5 [.] */ 0x0000,
  /* 0xb6 [.] */ 0x0000,
  /* 0xb7 [.] */ 0x0000,
  /* 0xb8 [.] */ 0x0000,
  /* 0xb9 [.] */ 0x0000,
  /* 0xba [.] */ 0x0000,
  /* 0xbb [.] */ 0x0000,
  /* 0xbc [.] */ 0x0000,
  /* 0xbd [.] */ 0x0000,
  /* 0xbe [.] */ 0x0000,
  /* 0xbf [.] */ 0x0000,
  /* 0xc0 [.] */ 0x0000,
  /* 0xc1 [.] */ 0x0000,
  /* 0xc2 [.] */ 0x0000,
  /* 0xc3 [.] */ 0x0000,
  /* 0xc4 [.] */ 0x0000,
  /* 0xc5 [.] */ 0x0000,
  /* 0xc6 [.] */ 0x0000,
  /* 0xc7 [.] */ 0x0000,
  /* 0xc8 [.] */ 0x0000,
  /* 0xc9 [.] */ 0x0000,
  /* 0xca [.] */ 0x0000,
  /* 0xcb [.] */ 0x0000,
  /* 0xcc [.] */ 0x0000,
  /* 0xcd [.] */ 0x0000,
  /* 0xce [.] */ 0x0000,
  /* 0xcf [.] */ 0x0000,
  /* 0xd0 [.] */ 0x0000,
  /* 0xd1 [.] */ 0x0000,
  /* 0xd2 [.] */ 0x0000,
  /* 0xd3 [.] */ 0x0000,
  /* 0xd4 [.] */ 0x0000,
  /* 0xd5 [.] */ 0x0000,
  /* 0xd6 [.] */ 0x0000,
  /* 0xd7 [.] */ 0x0000,
  /* 0xd8 [.] */ 0x0000,
  /* 0xd9 [.] */ 0x0000,
  /* 0xda [.] */ 0x0000,
  /* 0xdb [.] */ 0x0000,
  /* 0xdc [.] */ 0x0000,
  /* 0xdd [.] */ 0x0000,
  /* 0xde [.] */ 0x0000,
  /* 0xdf [.] */ 0x0000,
  /* 0xe0 [.] */ 0x0000,
  /* 0xe1 [.] */ 0x0000,
  /* 0xe2 [.] */ 0x0000,
  /* 0xe3 [.] */ 0x0000,
  /* 0xe4 [.] */ 0x0000,
  /* 0xe5 [.] */ 0x0000,
  /* 0xe6 [.] */ 0x0000,
  /* 0xe7 [.] */ 0x0000,
  /* 0xe8 [.] */ 0x0000,
  /* 0xe9 [.] */ 0x0000,
  /* 0xea [.] */ 0x0000,
  /* 0xeb [.] */ 0x0000,
  /* 0xec [.] */ 0x0000,
  /* 0xed [.] */ 0x0000,
  /* 0xee [.] */ 0x0000,
  /* 0xef [.] */ 0x0000,
  /* 0xf0 [.] */ 0x0000,
  /* 0xf1 [.] */ 0x0000,
  /* 0xf2 [.] */ 0x0000,
  /* 0xf3 [.] */ 0x0000,
  /* 0xf4 [.] */ 0x0000,
  /* 0xf5 [.] */ 0x0000,
  /* 0xf6 [.] */ 0x0000,
  /* 0xf7 [.] */ 0x0000,
  /* 0xf8 [.] */ 0x0000,
  /* 0xf9 [.] */ 0x0000,
  /* 0xfa [.] */ 0x0000,
  /* 0xfb [.] */ 0x0000,
  /* 0xfc [.] */ 0x0000,
  /* 0xfd [.] */ 0x0000,
  /* 0xfe [.] */ 0x0000,
  /* 0xff [.] */ 0x0000,

  /* 0x100 */ 0x0000,
  /* 0x101 */ 0x0000,
  /* 0x102 */ 0x0000,
  /* 0x103 */ 0x0000,
  /* 0x104 */ 0x0000,
  /* 0x105 */ 0x0000,
  /* 0x106 */ 0x0000,
  /* 0x107 */ 0x0000,
  /* 0x108 */ 0x0000,
  /* 0x109 */ 0x0000,
  /* 0x10a */ 0x0000,
  /* 0x10b */ 0x0000,
  /* 0x10c */ 0x007a,
  /* 0x10d */ 0x0000,
  /* 0x10e */ 0x0000,
  /* 0x10f */ 0x0000,
  /* 0x110 */ 0x0081,   /* SHIFT */
  /* 0x111 */ 0x0000,   /* CONTROL */
  /* 0x112 */ 0x0000,   /* ALT */
  /* 0x113 */ 0x0000,   /* ALT_GR */
  /* 0x114 */ 0x0037,   /* LEFT */
  /* 0x115 */ 0x0077,   /* RIGHT */
  /* 0x116 */ 0x0067,   /* DOWN */
  /* 0x117 */ 0x0027,   /* UP */
  /* 0x118 */ 0x0000,   /* HOME */
  /* 0x119 */ 0x0000,   /* END */
  /* 0x11a */ 0x0000,   /* DEL */
  /* 0x11b */ 0x0000,   /* INSERT */
  /* 0x11c */ 0x0000,
  /* 0x11d */ 0x0000,
  /* 0x11e */ 0x0000,   /* PAUSE */
  /* 0x11f */ 0x0000,
  /* 0x120 */ 0x0000,   /* COLOR */
  /* 0x121 */ 0x0000,   /* GRAPHIC */
  /* 0x122 */ 0x0000,   /* LIST */
  /* 0x123 */ 0x0000,   /* RUN */
  /* 0x124 */ 0x0000,   /* STOP */
  /* 0x125 */ 0x0057,   /* SHIFT LOCK */

  /* 0x126 */ 0x0000,
  /* 0x127 */ 0x0000,
  /* 0x128 */ 0x0000,
  /* 0x129 */ 0x0000,
  /* 0x12a */ 0x0000,
  /* 0x12b */ 0x0000,
  /* 0x12c */ 0x0000,
  /* 0x12d */ 0x0000,
  /* 0x12e */ 0x0000,
  /* 0x12f */ 0x0000,
};

#if 0
static unsigned int __keys_rb_k7669[] = {
  /* 0x00 [.] */ 0x0000,
  /* 0x01 [.] */ 0x0000,
  /* 0x02 [.] */ 0x0000,
  /* 0x03 [.] */ 0x0000,
  /* 0x04 [.] */ 0x0000,
  /* 0x05 [.] */ 0x0000,
  /* 0x06 [.] */ 0x0000,
  /* 0x07 [.] */ 0x0000,
  /* 0x08 [.] */ 0x0000,
  /* 0x09 [.] */ 0x0000,
  /* 0x0a [.] */ 0x0000,
  /* 0x0b [.] */ 0x0000,
  /* 0x0c [.] */ 0x0000,
  /* 0x0d [.] */ 0x0017, /* ENTER */
  /* 0x0e [.] */ 0x0000,
  /* 0x0f [.] */ 0x0000,
  /* 0x10 [.] */ 0x0000,
  /* 0x11 [.] */ 0x0000,
  /* 0x12 [.] */ 0x0000,
  /* 0x13 [.] */ 0x0000,
  /* 0x14 [.] */ 0x0000,
  /* 0x15 [.] */ 0x0000,
  /* 0x16 [.] */ 0x0000,
  /* 0x17 [.] */ 0x0000,
  /* 0x18 [.] */ 0x0000,
  /* 0x19 [.] */ 0x0000,
  /* 0x1a [.] */ 0x0000,
  /* 0x1b [.] */ 0x0000,
  /* 0x1c [.] */ 0x0000,
  /* 0x1d [.] */ 0x0000,
  /* 0x1e [.] */ 0x0000,
  /* 0x1f [.] */ 0x0000,
  /* 0x20 [ ] */ 0x0047, /* SPACE */
  /* 0x21 [!] */ 0x0000,
  /* 0x22 ["] */ 0x0000,
  /* 0x23 [#] */ 0x0000,
  /* 0x24 [$] */ 0x0000,
  /* 0x25 [%] */ 0x0000,
  /* 0x26 [&] */ 0x0000,
  /* 0x27 ['] */ 0x0000,
  /* 0x28 [(] */ 0x0000,
  /* 0x29 [)] */ 0x0000,
  /* 0x2a [*] */ 0x0000,
  /* 0x2b [+] */ 0x0000,
  /* 0x2c [,] */ 0x0000,
  /* 0x2d [-] */ 0x0000,
  /* 0x2e [.] */ 0x0000,
  /* 0x2f [/] */ 0x0000,
  /* 0x30 [0] */ 0x0045,
  /* 0x31 [1] */ 0x0001,
  /* 0x32 [2] */ 0x0041,
  /* 0x33 [3] */ 0x0002,
  /* 0x34 [4] */ 0x0042,
  /* 0x35 [5] */ 0x0003,
  /* 0x36 [6] */ 0x0043,
  /* 0x37 [7] */ 0x0004,
  /* 0x38 [8] */ 0x0044,
  /* 0x39 [9] */ 0x0005,
  /* 0x3a [:] */ 0x0065,
  /* 0x3b [;] */ 0x0075,
  /* 0x3c [<] */ 0x0000,
  /* 0x3d [=] */ 0x0000,
  /* 0x3e [>] */ 0x0000,
  /* 0x3f [?] */ 0x0000,
  /* 0x40 [@] */ 0x0000,
  /* 0x41 [A] */ 0x0000,
  /* 0x42 [B] */ 0x0000,
  /* 0x43 [C] */ 0x0000,
  /* 0x44 [D] */ 0x0000,
  /* 0x45 [E] */ 0x0000,
  /* 0x46 [F] */ 0x0000,
  /* 0x47 [G] */ 0x0000,
  /* 0x48 [H] */ 0x0000,
  /* 0x49 [I] */ 0x0000,
  /* 0x4a [J] */ 0x0000,
  /* 0x4b [K] */ 0x0000,
  /* 0x4c [L] */ 0x0000,
  /* 0x4d [M] */ 0x0000,
  /* 0x4e [N] */ 0x0000,
  /* 0x4f [O] */ 0x0000,
  /* 0x50 [P] */ 0x0000,
  /* 0x51 [Q] */ 0x0000,
  /* 0x52 [R] */ 0x0000,
  /* 0x53 [S] */ 0x0000,
  /* 0x54 [T] */ 0x0000,
  /* 0x55 [U] */ 0x0000,
  /* 0x56 [V] */ 0x0000,
  /* 0x57 [W] */ 0x0000,
  /* 0x58 [X] */ 0x0000,
  /* 0x59 [Y] */ 0x0000,
  /* 0x5a [Z] */ 0x0000,
  /* 0x5b [[] */ 0x0000,
  /* 0x5c [\] */ 0x0000,
  /* 0x5d []] */ 0x0000,
  /* 0x5e [^] */ 0x0000,
  /* 0x5f [_] */ 0x0000,
  /* 0x60 [`] */ 0x0000,
  /* 0x61 [a] */ 0x0021,
  /* 0x62 [b] */ 0x0033,
  /* 0x63 [c] */ 0x0032,
  /* 0x64 [d] */ 0x0022,
  /* 0x65 [e] */ 0x0012,
  /* 0x66 [f] */ 0x0062,
  /* 0x67 [g] */ 0x0023,
  /* 0x68 [h] */ 0x0063,
  /* 0x69 [i] */ 0x0054,
  /* 0x6a [j] */ 0x0024,
  /* 0x6b [k] */ 0x0064,
  /* 0x6c [l] */ 0x0025,
  /* 0x6d [m] */ 0x0034,
  /* 0x6e [n] */ 0x0073,
  /* 0x6f [o] */ 0x0015,
  /* 0x70 [p] */ 0x0055,
  /* 0x71 [q] */ 0x0011,
  /* 0x72 [r] */ 0x0052,
  /* 0x73 [s] */ 0x0061,
  /* 0x74 [t] */ 0x0013,
  /* 0x75 [u] */ 0x0014,
  /* 0x76 [v] */ 0x0072,
  /* 0x77 [w] */ 0x0051,
  /* 0x78 [x] */ 0x0071,
  /* 0x79 [y] */ 0x0031,
  /* 0x7a [z] */ 0x0053,
  /* 0x7b [{] */ 0x0000,
  /* 0x7c [|] */ 0x0000,
  /* 0x7d [}] */ 0x0000,
  /* 0x7e [~] */ 0x0000,
  /* 0x7f [.] */ 0x0000,
  /* 0x80 [.] */ 0x0000,
  /* 0x81 [.] */ 0x0000,
  /* 0x82 [.] */ 0x0000,
  /* 0x83 [.] */ 0x0000,
  /* 0x84 [.] */ 0x0000,
  /* 0x85 [.] */ 0x0000,
  /* 0x86 [.] */ 0x0000,
  /* 0x87 [.] */ 0x0000,
  /* 0x88 [.] */ 0x0000,
  /* 0x89 [.] */ 0x0000,
  /* 0x8a [.] */ 0x0000,
  /* 0x8b [.] */ 0x0000,
  /* 0x8c [.] */ 0x0000,
  /* 0x8d [.] */ 0x0000,
  /* 0x8e [.] */ 0x0000,
  /* 0x8f [.] */ 0x0000,
  /* 0x90 [.] */ 0x0000,
  /* 0x91 [.] */ 0x0000,
  /* 0x92 [.] */ 0x0000,
  /* 0x93 [.] */ 0x0000,
  /* 0x94 [.] */ 0x0000,
  /* 0x95 [.] */ 0x0000,
  /* 0x96 [.] */ 0x0000,
  /* 0x97 [.] */ 0x0000,
  /* 0x98 [.] */ 0x0000,
  /* 0x99 [.] */ 0x0000,
  /* 0x9a [.] */ 0x0000,
  /* 0x9b [.] */ 0x0000,
  /* 0x9c [.] */ 0x0000,
  /* 0x9d [.] */ 0x0000,
  /* 0x9e [.] */ 0x0000,
  /* 0x9f [.] */ 0x0000,
  /* 0xa0 [.] */ 0x0000,
  /* 0xa1 [.] */ 0x0000,
  /* 0xa2 [.] */ 0x0000,
  /* 0xa3 [.] */ 0x0000,
  /* 0xa4 [.] */ 0x0000,
  /* 0xa5 [.] */ 0x0000,
  /* 0xa6 [.] */ 0x0000,
  /* 0xa7 [.] */ 0x0000,
  /* 0xa8 [.] */ 0x0000,
  /* 0xa9 [.] */ 0x0000,
  /* 0xaa [.] */ 0x0000,
  /* 0xab [.] */ 0x0000,
  /* 0xac [.] */ 0x0000,
  /* 0xad [.] */ 0x0000,
  /* 0xae [.] */ 0x0000,
  /* 0xaf [.] */ 0x0000,
  /* 0xb0 [.] */ 0x0000,
  /* 0xb1 [.] */ 0x0000,
  /* 0xb2 [.] */ 0x0000,
  /* 0xb3 [.] */ 0x0000,
  /* 0xb4 [.] */ 0x0000,
  /* 0xb5 [.] */ 0x0000,
  /* 0xb6 [.] */ 0x0000,
  /* 0xb7 [.] */ 0x0000,
  /* 0xb8 [.] */ 0x0000,
  /* 0xb9 [.] */ 0x0000,
  /* 0xba [.] */ 0x0000,
  /* 0xbb [.] */ 0x0000,
  /* 0xbc [.] */ 0x0000,
  /* 0xbd [.] */ 0x0000,
  /* 0xbe [.] */ 0x0000,
  /* 0xbf [.] */ 0x0000,
  /* 0xc0 [.] */ 0x0000,
  /* 0xc1 [.] */ 0x0000,
  /* 0xc2 [.] */ 0x0000,
  /* 0xc3 [.] */ 0x0000,
  /* 0xc4 [.] */ 0x0000,
  /* 0xc5 [.] */ 0x0000,
  /* 0xc6 [.] */ 0x0000,
  /* 0xc7 [.] */ 0x0000,
  /* 0xc8 [.] */ 0x0000,
  /* 0xc9 [.] */ 0x0000,
  /* 0xca [.] */ 0x0000,
  /* 0xcb [.] */ 0x0000,
  /* 0xcc [.] */ 0x0000,
  /* 0xcd [.] */ 0x0000,
  /* 0xce [.] */ 0x0000,
  /* 0xcf [.] */ 0x0000,
  /* 0xd0 [.] */ 0x0000,
  /* 0xd1 [.] */ 0x0000,
  /* 0xd2 [.] */ 0x0000,
  /* 0xd3 [.] */ 0x0000,
  /* 0xd4 [.] */ 0x0000,
  /* 0xd5 [.] */ 0x0000,
  /* 0xd6 [.] */ 0x0000,
  /* 0xd7 [.] */ 0x0000,
  /* 0xd8 [.] */ 0x0000,
  /* 0xd9 [.] */ 0x0000,
  /* 0xda [.] */ 0x0000,
  /* 0xdb [.] */ 0x0000,
  /* 0xdc [.] */ 0x0000,
  /* 0xdd [.] */ 0x0000,
  /* 0xde [.] */ 0x0000,
  /* 0xdf [.] */ 0x0000,
  /* 0xe0 [.] */ 0x0000,
  /* 0xe1 [.] */ 0x0000,
  /* 0xe2 [.] */ 0x0000,
  /* 0xe3 [.] */ 0x0000,
  /* 0xe4 [.] */ 0x0000,
  /* 0xe5 [.] */ 0x0000,
  /* 0xe6 [.] */ 0x0000,
  /* 0xe7 [.] */ 0x0000,
  /* 0xe8 [.] */ 0x0000,
  /* 0xe9 [.] */ 0x0000,
  /* 0xea [.] */ 0x0000,
  /* 0xeb [.] */ 0x0000,
  /* 0xec [.] */ 0x0000,
  /* 0xed [.] */ 0x0000,
  /* 0xee [.] */ 0x0000,
  /* 0xef [.] */ 0x0000,
  /* 0xf0 [.] */ 0x0000,
  /* 0xf1 [.] */ 0x0000,
  /* 0xf2 [.] */ 0x0000,
  /* 0xf3 [.] */ 0x0000,
  /* 0xf4 [.] */ 0x0000,
  /* 0xf5 [.] */ 0x0000,
  /* 0xf6 [.] */ 0x0000,
  /* 0xf7 [.] */ 0x0000,
  /* 0xf8 [.] */ 0x0000,
  /* 0xf9 [.] */ 0x0000,
  /* 0xfa [.] */ 0x0000,
  /* 0xfb [.] */ 0x0000,
  /* 0xfc [.] */ 0x0000,
  /* 0xfd [.] */ 0x0000,
  /* 0xfe [.] */ 0x0000,
  /* 0xff [.] */ 0x0000,

  /* 0x100 */ 0x0000,
  /* 0x101 */ 0x0079,
  /* 0x102 */ 0x000a,
  /* 0x103 */ 0x001a,
  /* 0x104 */ 0x0000,
  /* 0x105 */ 0x0000,
  /* 0x106 */ 0x0000,
  /* 0x107 */ 0x0000,
  /* 0x108 */ 0x0000,
  /* 0x109 */ 0x0000,
  /* 0x10a */ 0x0000,
  /* 0x10b */ 0x0000,
  /* 0x10c */ 0x0000,
  /* 0x10d */ 0x0000,
  /* 0x10e */ 0x0000,
  /* 0x10f */ 0x0000,
  /* 0x110 */ 0x0081,   /* SHIFT */
  /* 0x111 */ 0x0000,   /* CONTROL */
  /* 0x112 */ 0x0000,   /* ALT */
  /* 0x113 */ 0x0000,   /* ALT_GR */
  /* 0x114 */ 0x0037,   /* LEFT */
  /* 0x115 */ 0x0077,   /* RIGHT */
  /* 0x116 */ 0x0067,   /* DOWN */
  /* 0x117 */ 0x0027,   /* UP */
  /* 0x118 */ 0x0000,   /* HOME */
  /* 0x119 */ 0x0000,   /* END */
  /* 0x11a */ 0x0000,   /* DEL */
  /* 0x11b */ 0x0000,   /* INSERT */
  /* 0x11c */ 0x0000,
  /* 0x11d */ 0x0000,
  /* 0x11e */ 0x0000,   /* PAUSE */
  /* 0x11f */ 0x0000,
  /* 0x120 */ 0x0000,   /* COLOR */
  /* 0x121 */ 0x0000,   /* GRAPHIC */
  /* 0x122 */ 0x0000,   /* LIST */
  /* 0x123 */ 0x0000,   /* RUN */
  /* 0x124 */ 0x0000,   /* STOP */
  /* 0x125 */ 0x0057,   /* SHIFT LOCK */

  /* 0x126 */ 0x0000,
  /* 0x127 */ 0x0000,
  /* 0x128 */ 0x0000,
  /* 0x129 */ 0x0000,
  /* 0x12a */ 0x0000,
  /* 0x12b */ 0x0000,
  /* 0x12c */ 0x0000,
  /* 0x12d */ 0x0000,
  /* 0x12e */ 0x0000,
  /* 0x12f */ 0x0000,
};
#endif

#endif /* __kc_keyb0k_h */
