/*
 * Copyright (C) 2025 Michael Brown <mbrown@fensystems.co.uk>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 *
 * You can also choose to distribute this program under the terms of
 * the Unmodified Binary Distribution Licence (as given in the file
 * COPYING.UBDL), provided that you have satisfied its requirements.
 */

FILE_LICENCE ( GPL2_OR_LATER_OR_UBDL );

/** @file
 *
 * NIST P-384 elliptic curve self-tests
 *
 */

/* Forcibly enable assertions */
#undef NDEBUG

#include <ipxe/p384.h>
#include <ipxe/test.h>
#include "elliptic_test.h"

/* http://point-at-infinity.org/ecc/nisttv k=1 */
ELLIPTIC_TEST ( poi_1, &p384_curve, BASE_GENERATOR,
		SCALAR ( 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01 ),
		EXPECTED ( 0xaa, 0x87, 0xca, 0x22, 0xbe, 0x8b, 0x05, 0x37,
			   0x8e, 0xb1, 0xc7, 0x1e, 0xf3, 0x20, 0xad, 0x74,
			   0x6e, 0x1d, 0x3b, 0x62, 0x8b, 0xa7, 0x9b, 0x98,
			   0x59, 0xf7, 0x41, 0xe0, 0x82, 0x54, 0x2a, 0x38,
			   0x55, 0x02, 0xf2, 0x5d, 0xbf, 0x55, 0x29, 0x6c,
			   0x3a, 0x54, 0x5e, 0x38, 0x72, 0x76, 0x0a, 0xb7,
			   0x36, 0x17, 0xde, 0x4a, 0x96, 0x26, 0x2c, 0x6f,
			   0x5d, 0x9e, 0x98, 0xbf, 0x92, 0x92, 0xdc, 0x29,
			   0xf8, 0xf4, 0x1d, 0xbd, 0x28, 0x9a, 0x14, 0x7c,
			   0xe9, 0xda, 0x31, 0x13, 0xb5, 0xf0, 0xb8, 0xc0,
			   0x0a, 0x60, 0xb1, 0xce, 0x1d, 0x7e, 0x81, 0x9d,
			   0x7a, 0x43, 0x1d, 0x7c, 0x90, 0xea, 0x0e, 0x5f ) );

/* http://point-at-infinity.org/ecc/nisttv k=2 */
ELLIPTIC_TEST ( poi_2, &p384_curve, BASE_GENERATOR,
		SCALAR ( 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02 ),
		EXPECTED ( 0x08, 0xd9, 0x99, 0x05, 0x7b, 0xa3, 0xd2, 0xd9,
			   0x69, 0x26, 0x00, 0x45, 0xc5, 0x5b, 0x97, 0xf0,
			   0x89, 0x02, 0x59, 0x59, 0xa6, 0xf4, 0x34, 0xd6,
			   0x51, 0xd2, 0x07, 0xd1, 0x9f, 0xb9, 0x6e, 0x9e,
			   0x4f, 0xe0, 0xe8, 0x6e, 0xbe, 0x0e, 0x64, 0xf8,
			   0x5b, 0x96, 0xa9, 0xc7, 0x52, 0x95, 0xdf, 0x61,
			   0x8e, 0x80, 0xf1, 0xfa, 0x5b, 0x1b, 0x3c, 0xed,
			   0xb7, 0xbf, 0xe8, 0xdf, 0xfd, 0x6d, 0xba, 0x74,
			   0xb2, 0x75, 0xd8, 0x75, 0xbc, 0x6c, 0xc4, 0x3e,
			   0x90, 0x4e, 0x50, 0x5f, 0x25, 0x6a, 0xb4, 0x25,
			   0x5f, 0xfd, 0x43, 0xe9, 0x4d, 0x39, 0xe2, 0x2d,
			   0x61, 0x50, 0x1e, 0x70, 0x0a, 0x94, 0x0e, 0x80 ) );

/* http://point-at-infinity.org/ecc/nisttv k=2 (as base) to k=20 */
ELLIPTIC_TEST ( poi_2_20, &p384_curve,
		BASE ( 0x08, 0xd9, 0x99, 0x05, 0x7b, 0xa3, 0xd2, 0xd9,
		       0x69, 0x26, 0x00, 0x45, 0xc5, 0x5b, 0x97, 0xf0,
		       0x89, 0x02, 0x59, 0x59, 0xa6, 0xf4, 0x34, 0xd6,
		       0x51, 0xd2, 0x07, 0xd1, 0x9f, 0xb9, 0x6e, 0x9e,
		       0x4f, 0xe0, 0xe8, 0x6e, 0xbe, 0x0e, 0x64, 0xf8,
		       0x5b, 0x96, 0xa9, 0xc7, 0x52, 0x95, 0xdf, 0x61,
		       0x8e, 0x80, 0xf1, 0xfa, 0x5b, 0x1b, 0x3c, 0xed,
		       0xb7, 0xbf, 0xe8, 0xdf, 0xfd, 0x6d, 0xba, 0x74,
		       0xb2, 0x75, 0xd8, 0x75, 0xbc, 0x6c, 0xc4, 0x3e,
		       0x90, 0x4e, 0x50, 0x5f, 0x25, 0x6a, 0xb4, 0x25,
		       0x5f, 0xfd, 0x43, 0xe9, 0x4d, 0x39, 0xe2, 0x2d,
		       0x61, 0x50, 0x1e, 0x70, 0x0a, 0x94, 0x0e, 0x80 ),
		SCALAR ( 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0a ),
		EXPECTED ( 0x60, 0x55, 0x08, 0xec, 0x02, 0xc5, 0x34, 0xbc,
			   0xee, 0xe9, 0x48, 0x4c, 0x86, 0x08, 0x6d, 0x21,
			   0x39, 0x84, 0x9e, 0x2b, 0x11, 0xc1, 0xa9, 0xca,
			   0x1e, 0x28, 0x08, 0xde, 0xc2, 0xea, 0xf1, 0x61,
			   0xac, 0x8a, 0x10, 0x5d, 0x70, 0xd4, 0xf8, 0x5c,
			   0x50, 0x59, 0x9b, 0xe5, 0x80, 0x0a, 0x62, 0x3f,
			   0x51, 0x58, 0xee, 0x87, 0x96, 0x2a, 0xc6, 0xb8,
			   0x1f, 0x00, 0xa1, 0x03, 0xb8, 0x54, 0x3a, 0x07,
			   0x38, 0x1b, 0x76, 0x39, 0xa3, 0xa6, 0x5f, 0x13,
			   0x53, 0xae, 0xf1, 0x1b, 0x73, 0x31, 0x06, 0xdd,
			   0xe9, 0x2e, 0x99, 0xb7, 0x8d, 0xe3, 0x67, 0xb4,
			   0x8e, 0x23, 0x8c, 0x38, 0xda, 0xd8, 0xee, 0xdd ) );

/* http://point-at-infinity.org/ecc/nisttv k=112233445566778899 */
ELLIPTIC_TEST ( poi_mid, &p384_curve, BASE_GENERATOR,
		SCALAR ( 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			 0x01, 0x8e, 0xbb, 0xb9, 0x5e, 0xed, 0x0e, 0x13 ),
		EXPECTED ( 0xa4, 0x99, 0xef, 0xe4, 0x88, 0x39, 0xbc, 0x3a,
			   0xbc, 0xd1, 0xc5, 0xce, 0xdb, 0xdd, 0x51, 0x90,
			   0x4f, 0x95, 0x14, 0xdb, 0x44, 0xf4, 0x68, 0x6d,
			   0xb9, 0x18, 0x98, 0x3b, 0x0c, 0x9d, 0xc3, 0xae,
			   0xe0, 0x5a, 0x88, 0xb7, 0x24, 0x33, 0xe9, 0x51,
			   0x5f, 0x91, 0xa3, 0x29, 0xf5, 0xf4, 0xfa, 0x60,
			   0x3b, 0x7c, 0xa2, 0x8e, 0xf3, 0x1f, 0x80, 0x9c,
			   0x2f, 0x1b, 0xa2, 0x4a, 0xae, 0xd8, 0x47, 0xd0,
			   0xf8, 0xb4, 0x06, 0xa4, 0xb8, 0x96, 0x85, 0x42,
			   0xde, 0x13, 0x9d, 0xb5, 0x82, 0x8c, 0xa4, 0x10,
			   0xe6, 0x15, 0xd1, 0x18, 0x2e, 0x25, 0xb9, 0x1b,
			   0x11, 0x31, 0xe2, 0x30, 0xb7, 0x27, 0xd3, 0x6a ) );

/* http://point-at-infinity.org/ecc/nisttv k=<largest> */
ELLIPTIC_TEST ( poi_large, &p384_curve, BASE_GENERATOR,
		SCALAR ( 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
			 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
			 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
			 0xc7, 0x63, 0x4d, 0x81, 0xf4, 0x37, 0x2d, 0xdf,
			 0x58, 0x1a, 0x0d, 0xb2, 0x48, 0xb0, 0xa7, 0x7a,
			 0xec, 0xec, 0x19, 0x6a, 0xcc, 0xc5, 0x29, 0x72 ),
		EXPECTED ( 0xaa, 0x87, 0xca, 0x22, 0xbe, 0x8b, 0x05, 0x37,
			   0x8e, 0xb1, 0xc7, 0x1e, 0xf3, 0x20, 0xad, 0x74,
			   0x6e, 0x1d, 0x3b, 0x62, 0x8b, 0xa7, 0x9b, 0x98,
			   0x59, 0xf7, 0x41, 0xe0, 0x82, 0x54, 0x2a, 0x38,
			   0x55, 0x02, 0xf2, 0x5d, 0xbf, 0x55, 0x29, 0x6c,
			   0x3a, 0x54, 0x5e, 0x38, 0x72, 0x76, 0x0a, 0xb7,
			   0xc9, 0xe8, 0x21, 0xb5, 0x69, 0xd9, 0xd3, 0x90,
			   0xa2, 0x61, 0x67, 0x40, 0x6d, 0x6d, 0x23, 0xd6,
			   0x07, 0x0b, 0xe2, 0x42, 0xd7, 0x65, 0xeb, 0x83,
			   0x16, 0x25, 0xce, 0xec, 0x4a, 0x0f, 0x47, 0x3e,
			   0xf5, 0x9f, 0x4e, 0x30, 0xe2, 0x81, 0x7e, 0x62,
			   0x85, 0xbc, 0xe2, 0x84, 0x6f, 0x15, 0xf1, 0xa0 ) );

/* Invalid curve point zero */
ELLIPTIC_TEST ( invalid_zero, &p384_curve,
		BASE ( 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		       0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		       0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		       0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		       0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		       0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		       0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		       0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		       0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		       0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		       0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		       0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 ),
		SCALAR ( 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01 ),
		EXPECTED_FAIL );

/* Invalid curve point (base_x, base_y - 1) */
ELLIPTIC_TEST ( invalid_one, &p384_curve,
		BASE ( 	0xaa, 0x87, 0xca, 0x22, 0xbe, 0x8b, 0x05, 0x37,
			0x8e, 0xb1, 0xc7, 0x1e, 0xf3, 0x20, 0xad, 0x74,
			0x6e, 0x1d, 0x3b, 0x62, 0x8b, 0xa7, 0x9b, 0x98,
			0x59, 0xf7, 0x41, 0xe0, 0x82, 0x54, 0x2a, 0x38,
			0x55, 0x02, 0xf2, 0x5d, 0xbf, 0x55, 0x29, 0x6c,
			0x3a, 0x54, 0x5e, 0x38, 0x72, 0x76, 0x0a, 0xb7,
			0x36, 0x17, 0xde, 0x4a, 0x96, 0x26, 0x2c, 0x6f,
			0x5d, 0x9e, 0x98, 0xbf, 0x92, 0x92, 0xdc, 0x29,
			0xf8, 0xf4, 0x1d, 0xbd, 0x28, 0x9a, 0x14, 0x7c,
			0xe9, 0xda, 0x31, 0x13, 0xb5, 0xf0, 0xb8, 0xc0,
			0x0a, 0x60, 0xb1, 0xce, 0x1d, 0x7e, 0x81, 0x9d,
			0x7a, 0x43, 0x1d, 0x7c, 0x90, 0xea, 0x0e, 0x5e ),
		SCALAR ( 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01 ),
		EXPECTED_FAIL );

/**
 * Perform P-384 self-test
 *
 */
static void p384_test_exec ( void ) {

	/* Tests from http://point-at-infinity.org/ecc/nisttv */
	elliptic_ok ( &poi_1 );
	elliptic_ok ( &poi_2 );
	elliptic_ok ( &poi_2_20 );
	elliptic_ok ( &poi_mid );
	elliptic_ok ( &poi_large );

	/* Invalid point tests */
	elliptic_ok ( &invalid_zero );
	elliptic_ok ( &invalid_one );
}

/** P-384 self-test */
struct self_test p384_test __self_test = {
	.name = "p384",
	.exec = p384_test_exec,
};
