/*
 * Copyright (C) 2000-2024 the xine project
 *
 * This file is part of xine, a unix video player.
 *
 * xine is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * xine is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110, USA
 *
 *
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>

#include "common.h"
#include "actions.h"
#include "event.h"
#include "kbindings.h"
#include "control.h"
#include "skins.h"
#include "menus.h"
#include "videowin.h"
#include "errors.h"
#include "xine-toolkit/backend.h"
#include "xine-toolkit/slider.h"
#include "xine-toolkit/label.h"
#include "xine-toolkit/labelbutton.h"
#include "xine-toolkit/browser.h"
#include <xine/video_out.h>


#define CONTROL_MIN     0
#define CONTROL_MAX     65535
#define CONTROL_STEP    565	/* approx. 1 pixel slider step */
#define STEP_SIZE       256     /* action event step */

#define TEST_VO_VALUE(val)  (val < CONTROL_MAX/3 || val > CONTROL_MAX*2/3) ? (CONTROL_MAX - val) : 0

#define NUM_SLIDERS 7
#define NUM_SL_8 ((NUM_SLIDERS + 7) & ~7)

#ifndef XINE_PARAM_VO_GAMMA
#  define XINE_PARAM_VO_GAMMA 0
#endif
#ifndef XINE_PARAM_VO_SHARPNESS
#  define XINE_PARAM_VO_SHARPNESS 0
#endif
#ifndef XINE_PARAM_VO_NOISE_REDUCTION
#  define XINE_PARAM_VO_NOISE_REDUCTION 0
#endif

static const int _vctrl_xine_props[NUM_SLIDERS] = {
  XINE_PARAM_VO_HUE,
  XINE_PARAM_VO_SATURATION,
  XINE_PARAM_VO_BRIGHTNESS,
  XINE_PARAM_VO_CONTRAST,
  XINE_PARAM_VO_GAMMA,
  XINE_PARAM_VO_SHARPNESS,
  XINE_PARAM_VO_NOISE_REDUCTION
};

static const char * const _vctrl_cfg[NUM_SLIDERS] = {
  "gui.vo_hue", "gui.vo_saturation", "gui.vo_brightness", "gui.vo_contrast",
  "gui.vo_gamma", "gui.vo_sharpness", "gui.vo_noise_reduction"
};

typedef struct {
  const char    *name;
  int            enable;
  int            value, default_value, saved_value;
} vctrl_item_t;

struct xui_vctrl_st {
  gui_new_window_t      nw;

  uint8_t               helipad[NUM_SL_8];
  xitk_widget_t        *sl[NUM_SLIDERS];
  vctrl_item_t          items[NUM_SLIDERS];

  xitk_widget_t        *list_title_w, *skinlist;
  const char           *skins[64];
  int                   skins_num;

  /* TJ. Since 2020, Petri wants this window to tripple toward the
   * bottom right corner of the screen on each show. This may be annoying, too.
   * Lets stop trippling for the current session when user has
   * moved the window manually. */
  struct {
    int                 yes, value;
  }                     tripple;

  xitk_register_key_t   widget_key;

  xitk_widget_t         *combo;

  char                   list_title_s[64];
};

static void _control_set (xui_vctrl_t *vctrl, int i, int value) {
  vctrl_item_t *item = vctrl->items + i;
  int real_value;

  if (!item->enable)
    return;

  if (value < CONTROL_MIN)
    value = CONTROL_MIN;
  if (value > CONTROL_MAX)
    value = CONTROL_MAX;
  if (value == item->value)
    return;

  xine_set_param (vctrl->nw.gui->stream, _vctrl_xine_props[i], value);
  real_value = xine_get_param (vctrl->nw.gui->stream, _vctrl_xine_props[i]);
  /* xine video out may support just a few coarse steps, and snap us back.
   * escape this black hole while we still can. */
  if (value > item->value) {
    if (real_value > item->value)
      value = real_value;
  } else {
    if (real_value < item->value)
      value = real_value;
  }
  item->value = value;
  osd_draw_bar (vctrl->nw.gui, item->name, CONTROL_MIN, CONTROL_MAX, value, OSD_BAR_STEPPER);
  xitk_slider_set_pos (vctrl->sl[i], value);
}

static void control_select_new_skin (xitk_widget_t *w, void *data, int selected, int modifier) {
  xui_vctrl_t *vctrl = data;

  (void)w;
  (void)modifier;
  if (vctrl && (selected >= 0)) {
    xitk_browser_set_select (vctrl->skinlist, -1);
    skin_select (vctrl->nw.gui, selected);
  }
}

static void control_set_value (xitk_widget_t *w, void *data, int value) {
  uint8_t *land = (uint8_t *)data;
  int i = *land;
  xui_vctrl_t *vctrl;

  (void)w;
  xitk_container (vctrl, land - i, helipad[0]);
  _control_set (vctrl, i, value);
}

static void control_changes_cb (void *data, xine_cfg_entry_t *cfg) {
  vctrl_item_t *item = data;

  item->value = (cfg->num_value < 0) ? item->default_value : cfg->num_value;
}

static void _control_close (xui_vctrl_t *vctrl) {
  if (vctrl->nw.xwin) {
    unsigned int u;

    for (u = 0; u < NUM_SLIDERS; u++)
      vctrl->sl[u] = NULL;
    gui_save_window_pos (vctrl->nw.gui, "control", vctrl->widget_key);
    xitk_unregister_event_handler (vctrl->nw.gui->xitk, &vctrl->widget_key);
    xitk_window_destroy_window (vctrl->nw.xwin);
    vctrl->nw.xwin = NULL;
    vctrl->tripple.yes = 5;
  }
}

static void _control_end (xui_vctrl_t *vctrl) {
  unsigned int u;

  for (u = 0; u < NUM_SLIDERS; u++) {
    vctrl_item_t *item = vctrl->items + u;
    if (item->enable) {
      int value = (item->value != item->default_value) ? item->value : -1;
      if (value != item->saved_value)
        config_update_num (vctrl->nw.gui->xine, _vctrl_cfg[u], value);
    }
  }

  video_window_set_input_focus (vctrl->nw.gui->vwin);
#ifdef HAVE_XINE_CONFIG_UNREGISTER_CALLBACKS
  xine_config_unregister_callbacks (vctrl->nw.gui->xine, NULL, NULL, vctrl, sizeof (*vctrl));
#endif
  vctrl->nw.gui->vctrl = NULL;
  free (vctrl);
}

/*
 * Handle X events here.
 */

static int control_event (void *data, const xitk_be_event_t *e) {
  xui_vctrl_t *vctrl = data;

  switch (e->type) {
    case XITK_EV_DEL_WIN:
      _control_close (vctrl);
      return 1;
    case XITK_EV_SHOW:
      xitk_label_change_label (vctrl->list_title_w, vctrl->list_title_s);
      return 1;
    case XITK_EV_HIDE:
      /* stop possible time consuming animation. */
      xitk_label_change_label (vctrl->list_title_w, "");
      return 1;
    case XITK_EV_POS_SIZE:
      if (vctrl->tripple.yes) {
        if (--vctrl->tripple.yes == 0)
          vctrl->tripple.value = 0;
      }
      return 1;
    case XITK_EV_BUTTON_DOWN:
      if (e->code == 3) {
        control_menu (vctrl->nw.gui, vctrl->nw.wl, e->w, e->h);
        return 1;
      }
      break;
    case XITK_EV_KEY_DOWN:
      if (e->utf8[0] == XITK_CTRL_KEY_PREFIX) {
        switch (e->utf8[1]) {
          case XITK_KEY_ESCAPE:
            _control_close (vctrl);
            return 1;
          case XITK_KEY_MENU:
            control_menu (vctrl->nw.gui, vctrl->nw.wl, e->w, e->h);
            return 1;
          case XITK_KEY_UP:
          case XITK_KEY_LEFT:
          case XITK_KEY_DOWN:
          case XITK_KEY_RIGHT:
            /* we dont get keys handled by sliders already here. */
            return xitk_widgets_cycle (vctrl->sl, NUM_SLIDERS, e->utf8[1]);
          default: ;
        }
      }
      break;
    default: ;
  }
  return gui_handle_be_event (vctrl->nw.gui, e);
}

static void _control_w_close (xitk_widget_t *w, void *data, int state) {
  xui_vctrl_t *vctrl = data;

  (void)w;
  (void)state;
  _control_close (vctrl);
}

static void _vctrl_adjust (gui_new_window_t *nw) {
  xui_vctrl_t *vctrl = nw->gui->vctrl;

  vctrl->nw.wr.x += vctrl->tripple.value;
  vctrl->nw.wr.y += vctrl->tripple.value;
}

/*
 * Create control panel window
 */
void control_main (xitk_widget_t *mode, void *data) {
  gGui_t *gui = data;
  xui_vctrl_t *vctrl;

  if (!gui)
    return;

  vctrl = gui->vctrl;
  if (mode == XUI_W_OFF) {
    if (!vctrl)
      return;
    _control_close (vctrl);
    return;
  } else if (mode == XUI_W_ON) {
    if (vctrl && vctrl->nw.xwin) {
      gui_raise_window (vctrl->nw.gui, vctrl->nw.xwin, 1, 0);
      xitk_window_set_input_focus (vctrl->nw.xwin);
      return;
    }
  } else if (mode == XUI_W_INIT) {
    if (vctrl)
      return;
  } else if (mode == XUI_W_DEINIT) {
    if (!vctrl)
        return;
    _control_close (vctrl);
    _control_end (vctrl);
    return;
  } else { /* toggle */
    if (vctrl && vctrl->nw.xwin) {
      _control_close (vctrl);
      return;
    }
  }

  if (!vctrl) {
    unsigned int u;

    vctrl = (xui_vctrl_t *)calloc (1, sizeof (*vctrl));
    if (!vctrl)
      return;
    vctrl->nw.gui = gui;
    vctrl->nw.id = "control";
    vctrl->nw.skin = "CtlBG";
    vctrl->nw.wfskin = "CtlWF";
    vctrl->nw.adjust = _vctrl_adjust;
    vctrl->nw.wr.x = 200;
    vctrl->nw.wr.y = 100;
    vctrl->nw.xwin = NULL;

    gui->vctrl = vctrl;

    vctrl->tripple.yes = 5;
    vctrl->tripple.value = 100;

    for (u = 0; u < NUM_SL_8; u++)
      vctrl->helipad[u] = u;

    vctrl->items[0].name  = _("Hue");
    vctrl->items[1].name  = _("Saturation");
    vctrl->items[2].name  = _("Brightness");
    vctrl->items[3].name  = _("Contrast");
    vctrl->items[4].name  = _("Gamma");
    vctrl->items[5].name  = _("Sharpness");
    vctrl->items[6].name  = _("Noise");

#ifdef VO_CAP_HUE
    /* xine-lib-1.2 */
    {
      uint32_t cap = vctrl->nw.gui->vo_port->get_capabilities (vctrl->nw.gui->vo_port);

      vctrl->items[0].enable = !!(cap & VO_CAP_HUE);
      vctrl->items[1].enable = !!(cap & VO_CAP_SATURATION);
      vctrl->items[2].enable = !!(cap & VO_CAP_BRIGHTNESS);
      vctrl->items[3].enable = !!(cap & VO_CAP_CONTRAST);
#ifdef VO_CAP_GAMMA
      vctrl->items[4].enable = !!(cap & VO_CAP_GAMMA);
#endif
#ifdef VO_CAP_SHARPNESS
      vctrl->items[5].enable = !!(cap & VO_CAP_SHARPNESS);
#endif
#ifdef VO_CAP_NOISE_REDUCTION
      vctrl->items[6].enable = !!(cap & VO_CAP_NOISE_REDUCTION);
#endif
    }
#else
    /* xine-lib-1.1 */
    for (u = 0; u < NUM_SLIDERS; u++) {
      vctrl_item_t *item = &vctrl->items[u];
      int old_value = xine_get_param (vctrl->nw.gui->stream, _vctrl_xine_props[u]);

      xine_set_param (vctrl->nw.gui->stream, _vctrl_xine_props[u], TEST_VO_VALUE (old_value));
      if (xine_get_param (vctrl->nw.gui->stream, _vctrl_xine_props[u]) != old_value) {
        xine_set_param (vctrl->nw.gui->stream, _vctrl_xine_props[u], old_value);
        item->enable = 1;
      }
    }
#endif

    {
      const char * const desc[] = {  N_("hue value"), N_("saturation value"), N_("brightness value"),
        N_("contrast value"), N_("gamma value"), N_("sharpness value"), N_("noise reduction value") };
      const char * const help[] = { N_("Hue value."), N_("Saturation value."), N_("Brightness value."),
        N_("Contrast value."), N_("Gamma value."), N_("Sharpness value."), N_("Noise reduction value.") };

      for (u = 0; u < NUM_SLIDERS; u++) {
        vctrl_item_t *item = &vctrl->items[u];
        if (!item->enable)
          continue;
        item->saved_value = item->value = xine_config_register_range (vctrl->nw.gui->xine,
          _vctrl_cfg[u], -1, CONTROL_MIN, CONTROL_MAX,
          gettext (desc[u]), gettext (help[u]), CONFIG_LEVEL_DEB, control_changes_cb, item);
        item->default_value = xine_get_param (vctrl->nw.gui->stream, _vctrl_xine_props[u]);
        if (item->value < 0) {
          item->value = item->default_value;
        } else {
          xine_set_param (vctrl->nw.gui->stream, _vctrl_xine_props[u], item->value);
          item->value = xine_get_param (vctrl->nw.gui->stream, _vctrl_xine_props[u]);
        }
      }
    }
  }

  if (mode == XUI_W_INIT)
    return;

  vctrl->nw.title = _("xine Control Window");
  vctrl->nw.wr.width = 0;
  vctrl->nw.wr.height = 0;
  {
    int r = gui_window_new (&vctrl->nw);
    if (r < 0) {
      gui_msg (vctrl->nw.gui, XUI_MSG_ERROR, _("control: couldn't find image for background\n"));
      free (vctrl);
      return;
    }
  }

  {
    /* attempt to save some relocation table entries. */
    static const char skins[][16] = {
        "SliderCtlHue",    "CtlHueLbl",
        "SliderCtlSat",    "CtlSatLbl",
        "SliderCtlBright", "CtlBrightLbl",
        "SliderCtlCont",   "CtlContLbl",
        "SliderCtlGamma",  "CtlGammaLbl",
        "SliderCtlSharp",  "CtlSharpLbl",
        "SliderCtlNoise",  "CtlNoiseLbl",
        "",                "CtlSkLbl"
    };
    static const uint32_t add_state[2] = {
      XITK_WIDGET_STATE_VISIBLE,
      XITK_WIDGET_STATE_VISIBLE | XITK_WIDGET_STATE_ENABLE
    };
    xitk_slider_widget_t sl = {
      .nw = {
        .wl = vctrl->nw.wl,
        .add_state = add_state[vctrl->items[0].enable],
        .skin_element_name = skins[0],
        .userdata = vctrl->helipad + 0,
        .tips = _("Control HUE value")
      },
      .min    = CONTROL_MIN,
      .max    = CONTROL_MAX,
      .step   = CONTROL_STEP,
      .value  = vctrl->items[0].value,
      .type   = XITK_WRAP_VSLIDER,
      .callback = control_set_value,
      .motion_callback = control_set_value
    };
    xitk_label_widget_t lbl = {
      .nw = {
        .wl = vctrl->nw.wl,
        .skin_element_name = sl.nw.skin_element_name + sizeof (skins[0]),
        .add_state = XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE,
        .userdata = vctrl
      }
    };
    /* gcc will unroll these loops anyway, so what :-) */
    vctrl->sl[0] = xitk_slider_create (&sl, vctrl->nw.gui->skin_config);
    sl.nw.skin_element_name += 2 * sizeof (skins[0]);
    /* TRANSLATORS: only ASCII characters (skin). */
    lbl.label = pgettext ("skin", "Hue");
    xitk_label_create (&lbl, vctrl->nw.gui->skin_config);
    lbl.nw.skin_element_name += 2 * sizeof (skins[0]);

    sl.type = XITK_VSLIDER;

    sl.nw.userdata = vctrl->helipad + 1;
    sl.nw.tips = _("Control SATURATION value");
    sl.nw.add_state = add_state[vctrl->items[1].enable];
    sl.value = vctrl->items[1].value;
    vctrl->sl[1] = xitk_slider_create (&sl, vctrl->nw.gui->skin_config);
    sl.nw.skin_element_name += 2 * sizeof (skins[0]);
    /* TRANSLATORS: only ASCII characters (skin). */
    lbl.label = pgettext ("skin", "Sat");
    xitk_label_create (&lbl, vctrl->nw.gui->skin_config);
    lbl.nw.skin_element_name += 2 * sizeof (skins[0]);

    sl.nw.userdata = vctrl->helipad + 2;
    sl.nw.tips = _("Control BRIGHTNESS value");
    sl.nw.add_state = add_state[vctrl->items[2].enable];
    sl.value = vctrl->items[2].value;
    vctrl->sl[2] = xitk_slider_create (&sl, vctrl->nw.gui->skin_config);
    sl.nw.skin_element_name += 2 * sizeof (skins[0]);
    /* TRANSLATORS: only ASCII characters (skin). */
    lbl.label = pgettext ("skin", "Brt");
    xitk_label_create (&lbl, vctrl->nw.gui->skin_config);
    lbl.nw.skin_element_name += 2 * sizeof (skins[0]);

    sl.nw.userdata = vctrl->helipad + 3;
    sl.nw.tips = _("Control CONTRAST value");
    sl.nw.add_state = add_state[vctrl->items[3].enable];
    sl.value = vctrl->items[3].value;
    vctrl->sl[3] = xitk_slider_create (&sl, vctrl->nw.gui->skin_config);
    sl.nw.skin_element_name += 2 * sizeof (skins[0]);
    /* TRANSLATORS: only ASCII characters (skin). */
    lbl.label = pgettext ("skin", "Ctr");
    xitk_label_create (&lbl, vctrl->nw.gui->skin_config);
    lbl.nw.skin_element_name += 2 * sizeof (skins[0]);

    sl.nw.userdata = vctrl->helipad + 4;
    sl.nw.tips = _("Control GAMMA value");
    sl.nw.add_state = add_state[vctrl->items[4].enable];
    sl.value = vctrl->items[4].value;
    vctrl->sl[4] = xitk_slider_create (&sl, vctrl->nw.gui->skin_config);
    sl.nw.skin_element_name += 2 * sizeof (skins[0]);
    /* TRANSLATORS: only ASCII characters (skin). */
    lbl.label = pgettext ("skin", "Gam");
    xitk_label_create (&lbl, vctrl->nw.gui->skin_config);
    lbl.nw.skin_element_name += 2 * sizeof (skins[0]);

    sl.nw.userdata = vctrl->helipad + 5;
    sl.nw.tips = _("Control SHARPNESS value");
    sl.nw.add_state = add_state[vctrl->items[5].enable];
    sl.value = vctrl->items[5].value;
    vctrl->sl[5] = xitk_slider_create (&sl, vctrl->nw.gui->skin_config);
    sl.nw.skin_element_name += 2 * sizeof (skins[0]);
    /* TRANSLATORS: only ASCII characters (skin). */
    lbl.label = pgettext ("skin", "Sha");
    xitk_label_create (&lbl, vctrl->nw.gui->skin_config);
    lbl.nw.skin_element_name += 2 * sizeof (skins[0]);

    sl.nw.userdata = vctrl->helipad + 6;
    sl.nw.tips = _("Control NOISE REDUCTION value");
    sl.nw.add_state = add_state[vctrl->items[6].enable];
    sl.value = vctrl->items[6].value;
    vctrl->sl[6] = xitk_slider_create (&sl, vctrl->nw.gui->skin_config);
    /* sl.nw.skin_element_name += 2 * sizeof (skins[0]); */
    /* TRANSLATORS: only ASCII characters (skin). */
    lbl.label = pgettext ("skin", "Noi");
    xitk_label_create (&lbl, vctrl->nw.gui->skin_config);
    lbl.nw.skin_element_name += 2 * sizeof (skins[0]);

    /* TRANSLATORS: only ASCII characters (skin) */
    strlcpy (vctrl->list_title_s, pgettext ("skin", "Choose a Skin"), sizeof (vctrl->list_title_s));
    lbl.label = vctrl->list_title_s;
    vctrl->list_title_w = xitk_label_create (&lbl, vctrl->nw.gui->skin_config);
  }

  vctrl->skins_num = skin_get_names (vctrl->nw.gui, vctrl->skins, sizeof (vctrl->skins) / sizeof (vctrl->skins[0]));

  {
    xitk_browser_widget_t br = {
      .nw = {
        .wl = vctrl->nw.wl,
        .skin_element_name = "CtlSkItemBtn",
        .userdata = vctrl,
        .add_state = XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE
      },
      .arrow_up = { .skin_element_name = "CtlSkUp" },
      .slider = { .skin_element_name = "SliderCtlSk" },
      .arrow_dn = { .skin_element_name = "CtlSkDn" },
      .arrow_left = { .skin_element_name = "CtlSkLeft" },
      .slider_h = { .skin_element_name = "SliderHCtlSk" },
      .arrow_right = { .skin_element_name = "CtlSkRight" },
      .browser = {
        .num_entries       = vctrl->skins_num,
        .entries           = vctrl->skins
      },
      .callback = control_select_new_skin,
      .dbl_click_callback = control_select_new_skin
    };
    vctrl->skinlist = xitk_browser_create (&br, vctrl->nw.gui->skin_config);
  }

  {
    xitk_labelbutton_widget_t lb = {
      .nw = {
        .wl = vctrl->nw.wl,
        .skin_element_name = "CtlDismiss",
        .userdata = vctrl,
        .tips = _("Close control window"),
        .add_state = XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE
      },
      .button_type    = CLICK_BUTTON,
      .align          = ALIGN_DEFAULT,
      .label          = _("Dismiss"),
      .callback       = _control_w_close
    };
    xitk_labelbutton_create (&lb, vctrl->nw.gui->skin_config);
  }

  vctrl->widget_key = xitk_be_register_event_handler ("control", vctrl->nw.xwin, control_event, vctrl, NULL, NULL);
  gui_raise_window (vctrl->nw.gui, vctrl->nw.xwin, 1, 0);
  xitk_window_set_input_focus (vctrl->nw.xwin);
}

void control_action (xui_vctrl_t *vctrl, action_id_t action) {
  if (vctrl && (action >= ACTID_HUECONTROLp)) {
    const int step[2] = { STEP_SIZE, -STEP_SIZE };
    unsigned int u = action - ACTID_HUECONTROLp;

    if (u < 2 * NUM_SLIDERS)
      _control_set (vctrl, u >> 1, vctrl->items[u >> 1].value + step[u & 1]);
  }
}

void control_reset (xui_vctrl_t *vctrl) {
  if (vctrl) {
    unsigned int u;

    for (u = 0; u < NUM_SLIDERS; u++) {
      vctrl_item_t *item = &vctrl->items[u];

      if (item->enable) {
        item->value = item->default_value;
        xine_set_param (vctrl->nw.gui->stream, _vctrl_xine_props[u], item->default_value);
        config_update_num (vctrl->nw.gui->xine, _vctrl_cfg[u], -1);
        xitk_slider_set_pos (vctrl->sl[u], item->default_value);
      }
    }
  }
}

/*
 * Raise control->window
 */
void control_raise_window (xui_vctrl_t *vctrl) {
  if (vctrl)
    gui_raise_window (vctrl->nw.gui, vctrl->nw.xwin, 1, 0);
}

/*
 * Change the current skin.
 */
void control_change_skins (xui_vctrl_t *vctrl, int synthetic) {
  (void)synthetic;
  if (vctrl) {
    xitk_widget_t *on[NUM_SLIDERS], *off[NUM_SLIDERS], **p[2] = { off, on };
    unsigned int u;

    if (xitk_window_change_skin (vctrl->nw.xwin, vctrl->nw.gui->skin_config, "CtlBG") == 2) {
      gui_msg (vctrl->nw.gui, XUI_MSG_ERROR, _("%s(): couldn't find image for background\n"), __XINE_FUNCTION__);
      exit(-1);
    }
    for (u = 0; u < NUM_SLIDERS; u++)
      *p[vctrl->items[u].enable]++ = vctrl->sl[u];
    xitk_widgets_state (on, p[1] - on, XITK_WIDGET_STATE_ENABLE, ~0u);
    xitk_widgets_state (off, p[0] - off, XITK_WIDGET_STATE_ENABLE, 0);
  }
}
