use ui;
use core:geometry;
use core:io;
use progvis:net;

/**
 * Action to take when the ProblemsDlg is closed.
 *
 * Allows creating a suitable behavior for the main Ui.
 */
class Action on Ui {
	// The problem to solve.
	Problem problem;

	// Code for the implementation on disk.
	Url implCode;

	// Code for the test program on disk.
	Url testCode;

	// Create.
	init(progvis:Settings settings, Problem problem) {
		init {
			problem = problem;
			implCode = saveCode(settings, problem, "", problem.impl);
			testCode = saveCode(settings, problem, "_test", problem.test);
		}
	}

	// Create a behavior.
	progvis:Behavior behavior(progvis:MainWin window, Client client) : abstract;

	// Return a list of files to open.
	Url[] files() {
		[testCode, implCode];
	}

	// Save source code to disk, and retrieve an URL for it.
	private Url saveCode(progvis:Settings settings, Problem problem, Str suffix, Code code) : static {
		Str title = "${problem.id,f06}";
		if (problem.implId >= 0)
			title += "_p${problem.implId,f03}";
		if (problem.testId >= 0)
			title += "_t${problem.testId,f03}";

		Url file = settings.downloadFile("${title}${suffix}.${code.language}");
		Utf8Output out(file.write());
		out.autoFlush = false;
		out.write(code.src);
		out.flush();
		out.close();
		return file;
	}
}


/**
 * Action to debug a problem.
 */
class DebugAction extends Action {
	init(progvis:Settings settings, Problem problem) {
		init(settings, problem) {}
	}

	Url[] files() : override {
		// To prevent modifications to the test code, we put it in an in-memory store.
		MemoryProtocol mem;
		Url test = problem.test.put("test", mem);
		[implCode, test];
	}

	progvis:Behavior behavior(progvis:MainWin window, Client client) : override {
		return progvis:DebugProblemBehavior(window, client, this);
	}
}


/**
 * Action to submit a new problem, skips the "find error" part.
 */
class ImplAction extends Action {
	init(progvis:Settings settings, Problem problem) {
		init(settings, problem) {}
	}

	Url[] files() : override {
		// To prevent modifications to the test code, we put it in an in-memory store.
		MemoryProtocol mem;
		Url test = problem.test.put("test", mem);
		[implCode, test];
	}

	progvis:Behavior behavior(progvis:MainWin window, Client client) : override {
		return progvis:ImproveBehavior(window, client, problem, testCode, implCode, "<none, admin action>");
	}
}


/**
 * Test a problem (change the main function).
 */
class TestAction extends Action {
	init(progvis:Settings settings, Problem problem) {
		init(settings, problem) {}
	}

	Url[] files() : override {
		// To prevent modifications to the implementation, we put it in an in-memory store.
		MemoryProtocol mem;
		Url impl = problem.impl.put("impl", mem);
		[testCode, impl];
	}

	progvis:Behavior behavior(progvis:MainWin window, Client client) : override {
		return progvis:TestBehavior(window, client, problem, testCode, implCode);
	}
}


/**
 * Show a solution.
 */
// class ShowSolutionAction extends Action {
// 	Str solution;

// 	init(progvis:Settings settings, Problem problem, Solution solution) {
// 		init(settings, problem) {
// 			solution = solution.steps;
// 		}
// 	}

// 	progvis:Behavior behavior(progvis:MainWin window, Client client) : override {
// 		return progvis:ReplayBehavior(window, solution);
// 	}
// }
