/*
 * QuteCom, a voice over Internet phone
 * Copyright (C) 2010 Mbdsys
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "QtQuteComConfigDialog.h"

#include "ui_QuteComConfigDialog.h"

#include "QtNotificationSettings.h"
#include "QtGeneralSettings.h"
#include "QtAccountSettings.h"
#include "QtPrivacySettings.h"
#include "QtAppearanceSettings.h"
#include "QtAudioSettings.h"
#include "QtVideoSettings.h"
#include "QtAdvancedSettings.h"
#include "QtCallForwardSettings.h"
#include "QtLanguagesSettings.h"
#include "QtVoicemailSettings.h"
#include "QtSecuritySettings.h"

#include <control/CQuteCom.h>
#include <control/profile/CUserProfileHandler.h>

#include <util/Logger.h>

#include <qtutil/WidgetUtils.h>

#include <QtGui/QtGui>

#include <QtCore/QPointer>

QtQuteComConfigDialog::QtQuteComConfigDialog(CQuteCom & cQuteCom)
	: QDialog(0) {

	_ui = new Ui::QuteComConfigDialog();
	_ui->setupUi(this);
	_ui->retranslateUi(this);

	// Set bold ourself: if we do it from Designer it alters the font name (at
	// least with Qt 4.1)
	QFont font(_ui->titleLabel->font());
	font.setBold(true);
	_ui->titleLabel->setFont(font);

	_settingsList += new QtGeneralSettings(cQuteCom, 0);
	if (cQuteCom.getCUserProfileHandler().getCUserProfile()) {
		_settingsList += new QtAccountSettings(cQuteCom, 0);
	}
	_settingsList += new QtLanguagesSettings(0);
	_settingsList += new QtAppearanceSettings(0);
	_settingsList += new QtAudioSettings(cQuteCom, 0);
	_settingsList += new QtVideoSettings(cQuteCom, 0);
#ifndef DISABLE_CALL_FORWARD
	_settingsList += new QtCallForwardSettings(cQuteCom, 0);
#endif
	_settingsList += new QtNotificationSettings(0);
#ifndef DISABLE_VOICE_MAIL
	_settingsList += new QtVoicemailSettings(cQuteCom, 0);
#endif
#if 0
	_settingsList += new QtSecuritySettings(cQuteCom, 0);
#endif
	_settingsList += new QtAdvancedSettings(0);

	//stackedWidget
	Q_FOREACH(QtISettings* settings, _settingsList) {
		QString iconName = QString(":pics/config/%1.png").arg(settings->getIconName());
		QIcon icon = QPixmap(iconName);
		new QListWidgetItem(icon, settings->getName(), _ui->listWidget);
		QWidget* widget = settings->getWidget();
		widget->layout()->setMargin(0);
		_ui->stackedWidget->addWidget(widget);
	}
	_ui->listWidget->setCurrentRow(0);
	showSettingPage(0);

	int minWidth = WidgetUtils::computeListViewMinimumWidth(_ui->listWidget);
	_ui->listWidget->setFixedWidth(minWidth);

	//listWidget
	connect(_ui->listWidget, SIGNAL(currentRowChanged(int)), SLOT(showSettingPage(int)));

	//saveButton
	connect(_ui->saveButton, SIGNAL(clicked()), SLOT(save()));
}

QtQuteComConfigDialog::~QtQuteComConfigDialog() {
	delete _ui;
}

void QtQuteComConfigDialog::showSettingPage(int row) {
	if (row<0 || row >= _settingsList.size()) {
		return;
	}
	QtISettings* settings = _settingsList.at(row);
	_ui->titleLabel->setText(settings->getTitle());
	QWidget* widget = settings->getWidget();
	_ui->stackedWidget->setCurrentWidget(widget);
	settings->postInitialize();
}

void QtQuteComConfigDialog::save() {
	Q_FOREACH(QtISettings* settings, _settingsList) {
		settings->saveConfig();
	}
}

void QtQuteComConfigDialog::showPage(const QString & pageName) {
	int count = _settingsList.count();
	for (int row = 0; row < count; ++row) {
		if (_settingsList[row]->getName() == pageName) {
			_ui->listWidget->setCurrentRow(row);
			return;
		}
	}
	LOG_WARN("unknown page name=" + pageName.toUtf8());
}

void QtQuteComConfigDialog::showGeneralPage() {
	showPage(tr("General"));
}

void QtQuteComConfigDialog::showLanguagePage() {
	showPage(tr("Language"));
}

void QtQuteComConfigDialog::showAccountsPage() {
	showPage(tr("Accounts"));
}

void QtQuteComConfigDialog::showPrivacyPage() {
	showPage(tr("Privacy"));
}

void QtQuteComConfigDialog::showAudioPage() {
	showPage(tr("Audio"));
}

void QtQuteComConfigDialog::showVideoPage() {
	showPage(tr("Video"));
}

void QtQuteComConfigDialog::showNotificationsPage() {
	showPage(tr("Notifications"));
}

void QtQuteComConfigDialog::showCallForwardPage() {
	showPage(tr("Call Forward"));
}

void QtQuteComConfigDialog::showAdvancedPage() {
	showPage(tr("Advanced"));
}

void QtQuteComConfigDialog::showVoicemailPage() {
	showPage(tr("Voicemail"));
}

void QtQuteComConfigDialog::showSecurityPage() {
	showPage(tr("security"));
}

QtQuteComConfigDialog* QtQuteComConfigDialog::showInstance(CQuteCom& cQuteCom) {
	static QPointer<QtQuteComConfigDialog> instance;
	if (!instance) {
		instance = new QtQuteComConfigDialog(cQuteCom);
		instance->setAttribute(Qt::WA_DeleteOnClose);
		instance->show();
	} else {
		instance->raise();
	}
	return instance;
}
