///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#ifndef __POVRAY_EXPORT_INTERFACE_H
#define __POVRAY_EXPORT_INTERFACE_H

#include <core/Core.h>
#include <core/scene/animation/TimeInterval.h>
#include <core/viewport/Viewport.h>
#include "../POVRay.h"

namespace Core {

class ObjectNode;		// defined in ObjectNode.h
class SceneObject;		// defined in SceneObject.h
class DataSet;			// defined in DataSet.h

};	// End of namespace Core

namespace POVRay {

class POVRayWriter; 	// defined below
class POVRayRenderer;	// defined in POVRayRenderer.h

/******************************************************************************
* Plugins can implement this abstract interface class to export
* their custom object types to the POV-Ray format.
******************************************************************************/
class POVRAY_DLLEXPORT POVRayExportInterface : public PluginClass
{
public:
	/// The default constructor.
	POVRayExportInterface() : PluginClass() {}

	/// Exports the given scene object in the POV-Ray format and returns true.
	/// Return false if the scene object type is not supported by this interface class.
	/// Throws an exception if an error occurred.
	virtual bool exportSceneObject(SceneObject* sceneObj, POVRayWriter& writer, ObjectNode* contextNode, const AffineTransformation& objToWorldTM) = 0;

private:
	Q_OBJECT
	DECLARE_ABSTRACT_PLUGIN_CLASS(POVRayExportInterface)
};

/******************************************************************************
* This is a wrapper for a text output stream that is used to write
* an output file in the POV-Ray format.
* It swaps the Y and Z components of all vectors written to the stream
* because POV-Ray uses another coordinate system than this application.
******************************************************************************/
class POVRAY_DLLEXPORT POVRayWriter
{
public:
	/// Constructor.
	POVRayWriter(QTextStream& _stream, DataSet* dataset, TimeTicks time, const CameraViewDescription& view, POVRayRenderer* renderer = NULL) :
		stream(_stream), _dataset(dataset), _time(time), _view(view), _renderer(renderer) {
	}

	/// The viewport that is being used to specify the camera in the POVRay file.
	const CameraViewDescription& view() const { return _view; }

	/// Returns the scene that is being exported.
	DataSet* dataSet() const { return _dataset; }

	/// Returns the scene time that is being exported.
	TimeTicks time() const { return _time; }

	/// If this export is part of a rendering operation then
	/// this return the plugin renderer; otherwise returns NULL.
	POVRayRenderer* renderer() const { return _renderer; }

private:

	/// The internal output stream.
	QTextStream& stream;

	/// Specifies the viewing parameters and camera positions.
	CameraViewDescription _view;

	/// The scene that is being exported.
	DataSet* _dataset;

	/// The scene time that is being exported.
	TimeTicks _time;

	/// If this export is part of a rendering operation then
	/// this field contains the plugin renderer.
	POVRayRenderer* _renderer;

public:

	///////////////////////// Output functions //////////////////////////////////////

	POVRayWriter& operator<<(const char* string) { stream << string; return *this; }
    POVRayWriter& operator<<(const QString& string) { stream << string; return *this; }
    POVRayWriter& operator<<(double f) { stream << f; return *this; }
    POVRayWriter& operator<<(float f) { stream << f; return *this; }
    POVRayWriter& operator<<(qint8 c) { stream << c; return *this; }
    POVRayWriter& operator<<(quint8 c) { stream << c; return *this; }
    POVRayWriter& operator<<(qint16 c) { stream << c; return *this; }
    POVRayWriter& operator<<(quint16 c) { stream << c; return *this; }
    POVRayWriter& operator<<(qint32 c) { stream << c; return *this; }
    POVRayWriter& operator<<(quint32 c) { stream << c; return *this; }
    POVRayWriter& operator<<(qint64 c) { stream << c; return *this; }
    POVRayWriter& operator<<(quint64 c) { stream << c; return *this; }

    /// Writes a 3d vector to the output stream in POV-Ray format.
	POVRayWriter& operator<<(const Vector3& v) {
		stream << "<" << v.X << ", " << v.Z << ", " << v.Y << ">";
		return *this;
	}

    /// Writes a 3d point to the output stream in POV-Ray format.
	POVRayWriter& operator<<(const Point3& p) {
		stream << "<" << p.X << ", " << p.Z << ", " << p.Y << ">";
		return *this;
	}

    /// Writes a color to the output stream in POV-Ray format.
	POVRayWriter& operator<<(const Color& c) {
		stream << "rgb <" << c.r << ", " << c.g << ", " << c.b << ">";
		return *this;
	}

    /// Writes a color with alpha channel to the output stream in POV-Ray format.
	POVRayWriter& operator<<(const ColorA& c) {
		stream << "rgbf <" << c.r << ", " << c.g << ", " << c.b << ", " << c.a << ">";
		return *this;
	}

    /// Writes a matrix to the output stream in POV-Ray format.
	POVRayWriter& operator<<(const AffineTransformation& m) {
		stream << "<";
		stream << m(0,0) << ", ";
		stream << m(2,0) << ", ";
		stream << m(1,0) << ", ";
		stream << m(0,2) << ", ";
		stream << m(2,2) << ", ";
		stream << m(1,2) << ", ";
		stream << m(0,1) << ", ";
		stream << m(2,1) << ", ";
		stream << m(1,1) << ", ";
		stream << m(0,3) << ", ";
		stream << m(2,3) << ", ";
		stream << m(1,3);
		stream << ">";
		return *this;
	}

	POVRayWriter& operator<<(QTextStreamFunction f) { stream << f; return *this; }
	POVRayWriter& operator<<(QTextStreamManipulator m) { stream << m; return *this; }
};


};

#endif // __POVRAY_EXPORT_INTERFACE_H
