///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

/**
 * \file BezierPolygon.h
 * \brief Contains definition of Core::BezierPolygon class.
 */

#ifndef __OVITO_BEZIER_POLYGON_H
#define __OVITO_BEZIER_POLYGON_H

#include <core/Core.h>
#include "PolygonVertex.h"

namespace Core {

class BezierCurve;		// defined in BezierCurve.h

/**
 * \brief A bezier polygon is an approximation of a BezierCurve.
 *
 * \author Alexander Stukowski
 * \sa PolygonVertex
 */
class CORE_DLLEXPORT BezierPolygon
{
public:

	/// \brief Creates an empty bezier polygon with no vertices.
	BezierPolygon() : _isClosed(false) {}

	/// \brief Creates a bezier polygon from a bezier curve.
	/// \param curve The bezier curve that should be approximated by the polygon.
	/// \param subdivisionSteps The number of subdivision steps to use for the curve segments. This parameter must be non-negative.
	BezierPolygon(const BezierCurve& curve, int subdivisionSteps) { setCurve(curve, subdivisionSteps); }

	/// \brief Copy constructor.
	BezierPolygon(const BezierPolygon& polygon) : _vertices(polygon._vertices), _isClosed(polygon._isClosed) {}

	/// \brief Creates a bezier polygon from a bezier curve.
	/// \param curve The bezier curve that should be approximated by the polygon.
	/// \param subdivisionSteps The number of subdivision steps to use for the curve segments. This parameter must be non-negative.
	void setCurve(const BezierCurve& curve, int subdivisionSteps);

	/// \brief Returns the sequence of vertices for this polygon.
	/// \return A read-only list of vertices.
	const QVector<PolygonVertex>& vertices() const { return _vertices; }

	/// \brief Returns the vector of vertices for this polygon.
	/// \return A mutable list of vertices.
	QVector<PolygonVertex>& vertices() { return _vertices; }

	/// \brief Returns the number of vertices in this polygon.
	/// \return The number of vertices. Can be any non-negative number.
	/// \sa setVertexCount()
	int vertexCount() const { return _vertices.size(); }

	/// \brief Sets the number of vertices in this polygon.
	/// \param n The new number of vertices. Can be any non-negative number.
	///
	/// Existing vertices are preserved by this method.
	///
	/// \sa vertexCount()
	void setVertexCount(int n) { OVITO_ASSERT(n >= 0); _vertices.resize(n); }

	/// \brief Returns the number of line segments in this polygon.
	/// \return The number of segments. This depends on the number vertices and whether the polygon is closed or open.
	/// \sa vertexCount()
	int segmentCount() const { return isClosed() ? vertexCount() : max(0,vertexCount()-1); }

	/// \brief Indicates whether the polygon is closed or not.
	/// \return \c true if the polygon curve is closed; \c false if it is open.
	/// \sa setClosed()
	bool isClosed() const { return _isClosed; }

	/// \brief Sets whether the polygon is closed.
	/// \param closed The closed state flag.
	/// \sa isClosed()
	void setClosed(bool closed) { _isClosed = closed; }

	/// \brief Saves the polygon to the given output stream.
    /// \param stream The destination stream.
	/// \sa loadFromStream()
	void saveToStream(SaveStream& stream) const;

	/// \brief Loads the polygon from the given input stream.
    /// \param stream The source stream.
	/// \sa saveToStream()
	void loadFromStream(LoadStream& stream);

private:

	/// The list of vertices that make up the polygon.
	QVector<PolygonVertex> _vertices;

	/// Indicates whether this polygon is closed.
	bool _isClosed;
};


};	// End of namespace Core

#endif // __OVITO_BEZIER_POLYGON_H
