///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/scene/animation/controller/TransformationController.h>
#include <core/data/units/ParameterUnit.h>
#include <base/linalg/AffineDecomposition.h>

namespace Core {

IMPLEMENT_ABSTRACT_PLUGIN_CLASS(TransformationController, Controller)
IMPLEMENT_SERIALIZABLE_PLUGIN_CLASS(PRSTransformationController, TransformationController)
DEFINE_REFERENCE_FIELD(PRSTransformationController, PositionController, "Position", position)
DEFINE_REFERENCE_FIELD(PRSTransformationController, RotationController, "Rotation", rotation)
DEFINE_REFERENCE_FIELD(PRSTransformationController, ScalingController, "Scaling", scaling)
SET_PROPERTY_FIELD_LABEL(PRSTransformationController, position, "Position")
SET_PROPERTY_FIELD_LABEL(PRSTransformationController, rotation, "Rotation")
SET_PROPERTY_FIELD_LABEL(PRSTransformationController, scaling, "Scaling")
SET_PROPERTY_FIELD_UNITS(PRSTransformationController, position, WorldParameterUnit)
SET_PROPERTY_FIELD_UNITS(PRSTransformationController, rotation, AngleParameterUnit)
SET_PROPERTY_FIELD_UNITS(PRSTransformationController, scaling, PercentParameterUnit)

/******************************************************************************
* Default constructor.
******************************************************************************/
PRSTransformationController::PRSTransformationController(bool isLoading) 
	: TransformationController(isLoading)
{
	INIT_PROPERTY_FIELD(PRSTransformationController, position);
	INIT_PROPERTY_FIELD(PRSTransformationController, rotation);
	INIT_PROPERTY_FIELD(PRSTransformationController, scaling);
	if(!isLoading) {
		position = CONTROLLER_MANAGER.createDefaultController<PositionController>();
		rotation = CONTROLLER_MANAGER.createDefaultController<RotationController>();
		scaling = CONTROLLER_MANAGER.createDefaultController<ScalingController>();
	}
}

/******************************************************************************
* Let the controller apply its value at a certain time to the input value.
******************************************************************************/
void PRSTransformationController::applyValue(TimeTicks time, AffineTransformation& result, TimeInterval& validityInterval)
{
	positionController()->applyValue(time, result, validityInterval);
	rotationController()->applyValue(time, result, validityInterval);
	scalingController()->applyValue(time, result, validityInterval);
}

/******************************************************************************
* Sets the controller's value at the specified time.
******************************************************************************/
void PRSTransformationController::setValue(TimeTicks time, const AffineTransformation& newValue, bool isAbsoluteValue)
{
	AffineDecomposition decomp(newValue);
	positionController()->setValue(time, decomp.translation, isAbsoluteValue);
	rotationController()->setValue(time, Rotation(decomp.rotation), isAbsoluteValue);
	OVITO_ASSERT_MSG(abs(DotProduct(decomp.scaling.Q,decomp.scaling.Q) - 1.0) <= FLOATTYPE_EPSILON, "SetValue", "Quaternion must be normalized.");	
	scalingController()->setValue(time, decomp.scaling, isAbsoluteValue);
}

/******************************************************************************
* This asks the controller to adjust its value after a scene node has got a new
* parent node.
*		oldParentTM - The transformation of the old parent node
*		newParentTM - The transformation of the new parent node
*		contextNode - The node to which this cotnroller is assigned to
******************************************************************************/
void PRSTransformationController::changeParent(TimeTicks time, const AffineTransformation& oldParentTM, const AffineTransformation& newParentTM, SceneNode* contextNode)
{
	positionController()->changeParent(time, oldParentTM, newParentTM, contextNode);
	rotationController()->changeParent(time, oldParentTM, newParentTM, contextNode);
	scalingController()->changeParent(time, oldParentTM, newParentTM, contextNode);
}

/******************************************************************************
* Computes the largest time interval containing the given time during which the
* controller's value is constant.
******************************************************************************/
void PRSTransformationController::validityInterval(TimeTicks time, TimeInterval& validityInterval)
{
	positionController()->validityInterval(time, validityInterval);
	rotationController()->validityInterval(time, validityInterval);
	scalingController()->validityInterval(time, validityInterval);
}


};
