/*
* @license Apache-2.0
*
* Copyright (c) 2021 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

// TypeScript Version: 2.0

/* tslint:disable:max-line-length */
/* tslint:disable:max-file-line-count */

import expandContractions = require( './../../expand-contractions' );
import lda = require( './../../lda' );
import porterStemmer = require( './../../porter-stemmer' );
import tokenize = require( './../../tokenize' );

/**
* Interface describing the `nlp` namespace.
*/
interface Namespace {
	/**
	* Expands all contractions to their formal equivalents.
	*
	* @param str - string to convert
	* @returns string with expanded contractions
	*
	* @example
	* var str = 'I won\'t be able to get y\'all out of this one.';
	* var out = ns.expandContractions( str );
	* // returns 'I will not be able to get you all out of this one.'
	*
	* @example
	* var str = 'It oughtn\'t to be my fault, because, you know, I didn\'t know';
	* var out = ns.expandContractions( str );
	* // returns 'It ought not to be my fault, because, you know, I did not know'
	*/
	expandContractions: typeof expandContractions;

	/**
	* Latent Dirichlet Allocation via collapsed Gibbs sampling.
	*
	* @param documents - document corpus
	* @param K - number of topics
	* @param options - options object
	* @param options.alpha - Dirichlet hyper-parameter of topic vector theta:
	* @param options.beta - Dirichlet hyper-parameter for word vector phi
	* @throws second argument must be a positive integer
	* @throws must provide valid options
	* @returns model object
	*/
	lda: typeof lda;

	/**
	* Extracts the stem of a given word.
	*
	* @param word - input word
	* @returns word stem
	*
	* @example
	* var out = ns.porterStemmer( 'walking' );
	* // returns 'walk'
	*
	* @example
	* var out = ns.porterStemmer( 'walked' );
	* // returns 'walk'
	*
	* @example
	* var out = ns.porterStemmer( 'walks' );
	* // returns 'walk'
	*
	* @example
	* var out = ns.porterStemmer( 'worldwide' );
	* // returns 'worldwid'
	*
	* @example
	* var out = ns.porterStemmer( '' );
	* // returns ''
	*/
	porterStemmer: typeof porterStemmer;

	/**
	* Tokenize a string.
	*
	* @param str - input string
	* @param keepWhitespace - boolean indicating whether whitespace characters should be returned as part of the token array (default: false)
	* @returns array of tokens
	*
	* @example
	* var str = 'Hello World!';
	* var out = ns.tokenize( str );
	* // returns [ 'Hello', 'World', '!' ]
	*
	* @example
	* var str = '';
	* var out = ns.tokenize( str );
	* // returns []
	*
	* @example
	* var str = 'Hello Mrs. Maple, could you call me back?';
	* var out = ns.tokenize( str );
	* // returns [ 'Hello', 'Mrs.', 'Maple', ',', 'could', 'you', 'call', 'me', 'back', '?' ]
	*/
	tokenize: typeof tokenize;
}

/**
* Standard library natural language processing.
*/
declare var ns: Namespace;


// EXPORTS //

export = ns;
