/*
 * This file is part of the KDE Milou Project
 * Copyright (C) 2013  Vishesh Handa <me@vhanda.in>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) version 3, or any
 * later version accepted by the membership of KDE e.V. (or its
 * successor approved by the membership of KDE e.V.), which shall
 * act as a proxy defined in Section 6 of version 3 of the license.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#ifndef MATCH_H
#define MATCH_H

#include <QString>
#include <QVariant>

namespace Milou {

class AbstractSource;
class MatchType;

/**
 * @class Match match.h
 *
 * @short A match returned by an Source
 *
 * The Match should always contain the type and the text. It is also useful
 * to pass the icon name. Additional preview information can also be
 * set in order to use the correct preview plugin to display a preview
 * of the match.
 */
class Match
{
public:
    explicit Match(AbstractSource* source = 0);
    ~Match();

    AbstractSource* source() const;

    /**
     * Return a type which categorizes the result. This type
     * should ideally be registered in the AbstractSource
     */
    MatchType* type() const;
    void setType(MatchType* type);

    /**
     * Returns the text which will be displayed to the user
     */
    QString text() const;
    void setText(const QString& text);

    QString icon() const;
    void setIcon(const QString& iconName);

    QVariant data() const;
    void setData(const QVariant& data);

    QString previewType() const;
    void setPreviewType(const QString& type);

    QString previewUrl() const;
    void setPreviewUrl(const QString& url);

    /**
     * This different from the normal text. It is shown
     * when a preview is being display, and can therefore
     * be much longer.
     */
    QString previewLabel();
    void setPreviewLabel(const QString& label);
private:
    AbstractSource* m_source;
    MatchType* m_type;

    QString m_text;
    QString m_icon;
    QVariant m_data;

    QString m_previewType;
    QString m_previewUrl;
    QString m_previewLabel;
};


/**
 * @class MatchType
 *
 * A MatchType is an abstract way of categorizing the Match. It contains
 * a user visible name and icon.
 *
 * Each Source should create their own instances of MatchType, register them
 * via the AbstractSource::setTypes function and then use instances
 * fo those types in the matches they provide
 */
class MatchType {
public:
    MatchType();
    MatchType(const QString& name, const QString& icon);

    QString name();
    void setName(const QString& n);

    QString icon();
    void setIcon(const QString& icon);

private:
    QString m_name;
    QString m_icon;
};

}

#endif // MATCH_H
