/* dir.hh - directory access
 * Copyright 2006 Bas Wijnen <wijnen@debian.org>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef SHEVEK_DIR_HH
#define SHEVEK_DIR_HH

#include <string>
#include <sys/types.h>
#include <set>
#include <glibmm.h>

namespace shevek
{
  /// Get information about files in a directory.
  class dir
  {
  public:
    /// Information about a single file in a directory.
    struct file
    {
      /// The name of the file.
      std::string name;
      /// Whether it is a directory.
      bool is_dir;
      /// The user-ID of the file.
      uid_t uid;
      /// The group-ID of the file.
      gid_t gid;
      /// The size of the file.
      off_t size;
      /// Sorting operator, needed for putting them in a std::set.
      bool operator< (file const &that) const;
    };
    /// Storage of the files.
    typedef std::set <file> store;
    /// Iterator for looping over the files.
    typedef store::const_iterator const_iterator;
    /// Create an empty directory object.
    dir ();
    /// Create a directory object and load content into it.
    dir (std::string const &path);
    /// Load new content into an existing directory object.
    void load (std::string const &path);
    /// Loop over the files.
    const_iterator begin () const;
    /// Loop over the files.
    const_iterator end () const;
    /// Number of files in the directory.
    unsigned size () const;
  private:
    store data;
  };
}

#endif
