package org.apache.lucene.analysis.miscellaneous;

/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import java.io.IOException;

import org.apache.lucene.util.AttributeImpl;
import org.apache.lucene.analysis.Token;
import org.apache.lucene.analysis.TokenStream;
import org.apache.lucene.analysis.tokenattributes.TermAttribute;

/**
 * A {@link TokenStream} containing a single token.
 */
public class SingleTokenTokenStream extends TokenStream {

  private boolean exhausted = false;
  
  // The token needs to be immutable, so work with clones!
  private Token singleToken;
  private final AttributeImpl tokenAtt;

  private static final AttributeFactory TOKEN_ATTRIBUTE_FACTORY = new AttributeFactory() {
    public AttributeImpl createAttributeInstance(Class attClass) {
      return attClass.isAssignableFrom(Token.class)
        ? new Token() : DEFAULT_ATTRIBUTE_FACTORY.createAttributeInstance(attClass);
    }
  };

  public SingleTokenTokenStream(Token token) {
    super(TOKEN_ATTRIBUTE_FACTORY);
    
    assert token != null;
    this.singleToken = (Token) token.clone();
    
    tokenAtt = (AttributeImpl) addAttribute(TermAttribute.class);
    assert (tokenAtt instanceof Token || tokenAtt.getClass().getName().equals("org.apache.lucene.analysis.TokenWrapper"));
  }

  public final boolean incrementToken() throws IOException {
    if (exhausted) {
      return false;
    } else {
      clearAttributes();
      singleToken.copyTo(tokenAtt);
      exhausted = true;
      return true;
    }
  }
  
  /** @deprecated Will be removed in Lucene 3.0. This method is final, as it should
   * not be overridden. Delegates to the backwards compatibility layer. */
  public final Token next(final Token reusableToken) throws java.io.IOException {
    return super.next(reusableToken);
  }

  /** @deprecated Will be removed in Lucene 3.0. This method is final, as it should
   * not be overridden. Delegates to the backwards compatibility layer. */
  public final Token next() throws java.io.IOException {
    return super.next();
  }

  public void reset() throws IOException {
    exhausted = false;
  }

  public Token getToken() {
    return (Token) singleToken.clone();
  }

  public void setToken(Token token) {
    this.singleToken = (Token) token.clone();
  }
}
