/*
 * File    : ZipFileSystem.java
 * Created : 24-sep-2001 19:48
 * By      : fbusquets
 *
 * JClic - Authoring and playing system for educational activities
 *
 * Copyright (C) 2000 - 2005 Francesc Busquets & Departament
 * d'Educacio de la Generalitat de Catalunya                                        
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details (see the LICENSE file).
 */

package edu.xtec.jclic.fileSystem;

import java.util.zip.*;
import java.util.Vector;
import java.io.IOException;
import java.io.InputStream;
import java.awt.Image;
import java.awt.Toolkit;
import edu.xtec.util.ResourceBridge;

/**
 *
 * @author Francesc Busquets (fbusquets@xtec.net)
 * @version 1.0
 */
public abstract class ZipFileSystem extends FileSystem {
    
    String zipName;
    protected boolean loaded;
    ExtendedZipEntry[] entries;

    public ZipFileSystem(ResourceBridge rb) {
        super("", rb);
        entries=null;
        zipName=null;
        loaded=false;
    }
    
    public ZipFileSystem(String rootPath, String zipFileName, ResourceBridge rb) {
        super(rootPath, rb);
        zipName = getCanonicalNameOf(zipFileName);
        entries=null;
        loaded=false;
    }
    
    public final static ZipFileSystem createZipFileSystem(String rootPath, String zipFileName, ResourceBridge rb) throws Exception {
        if (isStrUrl(rootPath))
            return new UrlZip(rootPath, zipFileName, rb);
        else
            return new FileZip(rootPath, zipFileName, rb);
    }
    
    public String getFullRoot(){
        return root+zipName;
    }
    
    public abstract class ExtendedZipEntry extends ZipEntry{
        public boolean ignore;
        ExtendedZipEntry(ZipEntry entry){
            super(entry);
        }
        public abstract byte[] getBytes() throws IOException;
        public abstract InputStream getInputStream() throws IOException;
        public String getName(){
            return stdFn(super.getName());
        }
    }
    
    public ExtendedZipEntry getEntry(String fName){
        if(fName==null || fName.length()==0 || entries==null) return null;
        String name=getCanonicalNameOf(fName);
        for(int i=0; i<entries.length; i++)
            if(!entries[i].ignore && entries[i].getName().equals(name))
                return entries[i];
        return null;
    }    
            
    public byte[] getBytes(String fileName) throws IOException {
        //System.out.println("requesting "+fileName);
        ExtendedZipEntry entry;
        if((entry=getEntry(fileName))==null)
            return super.getBytes(fileName);        
        else return entry.getBytes();
    }
    
    public boolean fileExists(String fName){
        return super.fileExists(fName);
    }
    
    public long getFileLength(String fileName) throws IOException{
        ExtendedZipEntry entry;
        if((entry=getEntry(fileName))==null)
            return super.getFileLength(fileName);        
        else return entry.getSize();
    }
    
    public Image getImageFile(String fName) throws Exception {        
        ExtendedZipEntry entry;    
        if((entry=getEntry(fName))==null)
            return super.getImageFile(fName);
        
        return Toolkit.getDefaultToolkit().createImage(entry.getBytes());
    }
    
    public InputStream getInputStream(String fName) throws IOException{
        ExtendedZipEntry entry;        
        if((entry=getEntry(fName))==null)
            return super.getInputStream(fName);
        
        return entry.getInputStream();
    }
    
    public Object getMediaDataSource(String fName) throws Exception{
        ExtendedZipEntry entry;        
        if((entry=getEntry(fName))==null)
            return super.getMediaDataSource(fName);
        
        return new edu.xtec.util.ExtendedByteArrayInputStream(entry.getBytes(), fName);
    }
    
    public String[] getEntries(String ext) throws Exception{
        String[] result=null;
        if(entries!=null && entries.length>0){
            Vector v=new Vector();
            for(int i=0; i<entries.length; i++){
                String entryName=entries[i].getName();
                if(ext==null || entryName.endsWith(ext))
                    v.add(entryName);
            }
            if(!v.isEmpty())
                result=(String[])v.toArray(new String[v.size()]);
        }
        return result;
    }
    
    public void close(){
        entries=null;
        super.close();
    }    
}
