/* IMSpector - Instant Messenger Transparent Proxy Service
 * http://www.imspector.org/
 * (c) Lawrence Manning <lawrence@aslak.net>, 2006
 *          
 * Released under the GPL v2. */

#include "imspector.h"

#define PLUGIN_NAME "MSN IMSpector protocol plugin"
#define PROTOCOL_NAME "MSN"
#define PROTOCOL_PORT 1863

extern "C"
{
	bool initprotocolplugin(struct protocolplugininfo &pprotocolplugininfo,
		class Options &options, bool debugmode);
	void closeprotocolplugin(void);
	int processpacket(bool outgoing, class Socket &incomingsock, char *replybuffer, 
		int *replybufferlength, std::vector<struct imevent> &imevents, std::string &clientaddress);
	int generatemessagepacket(struct response &response, char *replybuffer, int *replybufferlength);
};

#pragma pack(2)

struct p2pheader
{
	uint32_t sessionid;
	uint32_t id;
	uint64_t offset;
	uint64_t datasize;
	uint32_t messagesize;
	uint32_t flags;
	uint32_t ackid;
	uint32_t ackuid;
	uint64_t acksize;
};

struct context
{
	uint32_t headerlength;
	uint32_t version;
	uint64_t filesize;
	uint32_t type;
	uint16_t filename[260];
};

#pragma pack()

void setlocalid(std::string id);
void setremoteid(std::string id);
bool processmessage(bool outgoing, std::string id, int headerlength, char *msg,
	std::vector<struct imevent> &imevents, std::string clientaddress);
char *getheadervalues(char *buffer, std::map<std::string, std::string> &headers);
char *getstring(char *buffer, std::string &str);

#define MSGTYPE_NULL 0
#define MSGTYPE_PLAIN 1
#define MSGTYPE_P2P 2
#define MSGTYPE_CONTROL 3

std::string localid = "Unknown";
std::string remoteid = "Unknown";
bool groupchat = false;
bool gotremoteid = false;
int packetcount = 0;
bool tracing = false;
std::map<uint32_t, std::string> filetransfers;
bool localdebugmode = false;

bool initprotocolplugin(struct protocolplugininfo &protocolplugininfo,
	class Options &options, bool debugmode)
{
	if (options["msn_protocol"] != "on") return false;

	localdebugmode = debugmode;
	
	protocolplugininfo.pluginname = PLUGIN_NAME;
	protocolplugininfo.protocolname = PROTOCOL_NAME;
	protocolplugininfo.port = htons(PROTOCOL_PORT);
	
	if (options["msn_trace"] == "on") tracing = true;
	
	return true;
}

void closeprotocolplugin(void)
{
	return;
}

/* The main plugin function. See protocolplugin.cpp. */
int processpacket(bool outgoing, class Socket &incomingsock, char *replybuffer, 
	int *replybufferlength, std::vector<struct imevent> &imevents, std::string &clientaddress)
{
	char string[STRING_SIZE];
	memset(string, 0, STRING_SIZE);
	
	int headerlength;
	if ((headerlength = incomingsock.recvline(string, STRING_SIZE)) < 0) return 1;
	if (headerlength < 0) return 1;
	
	debugprint(localdebugmode, PROTOCOL_NAME ": Got %d bytes of header\n", headerlength);
	
	memcpy(replybuffer, string, headerlength);
	*replybufferlength = headerlength;
	
	std::string command;
	std::vector<std::string> args; int argc;

	char *s;
	s = chopline(string, command, args, argc);
	
	debugprint(localdebugmode, PROTOCOL_NAME ": Command: %s\n", command.c_str());
	
	/* These are all ways of getting the party (ID) information. */
	if (outgoing)
	{
		/* The local user is logging in. */
		if (command == "ANS" && argc > 1) 
			setlocalid(args[1]);
	}
	else
	{
		/* The local user completed the login. */
		if (command == "USR")
		{
			if (args[1] == "OK" && argc > 2)
				setlocalid(args[2]);
		}
		/* A remote user joined the chat. */
		if (command == "JOI" && argc > 0)
			setremoteid(args[0]);
		/* Could be multiple of these - group chats */
		if (command == "IRO" && argc > 3)
			setremoteid(args[3]);
	}
	
	if (command == "MSG" && argc > 2)
	{
		/* msgbuffer holds the actual message, ie. all data after the first line
		 * packet header. */
		char msgbuffer[BUFFER_SIZE];
		
		memset(msgbuffer, 0, BUFFER_SIZE);
		
		/* arg3 is the number of bytes following the end of the first line. */
		int lengthint = atol(args[2].c_str());

		if (!(incomingsock.recvalldata(msgbuffer, lengthint))) return 1;
		
		/* We get messages from "hotmail"  that we don't need to care about. */
		if (args[0] != "Hotmail")
			processmessage(outgoing, args[0], headerlength, msgbuffer, imevents, clientaddress);
		
		/* Now we copy the msgbuffer back into the replybuffer, starting at the end
		 * of the first line. This gives us a chance to modify msgbuffer when we do
		 * some content filtering. */
		memcpy(replybuffer + headerlength, msgbuffer, lengthint);
		*replybufferlength += lengthint;
	}
	
	/* These commands all have following data.  We need to pull it down, but
	 * we presently don't do anything with it. */
	if (
		((command == "ADL" || command == "RML" || command == "UUN" ||
		command == "UBN" || command == "GCF" ||
		command == "UUX" || command == "UBX" ||
		command == "QRY" || command == "PAG" || command == "NOT")
		&& argc > 1) ||
		((command == "NOT")
		&& argc)
	)
	{
		int lengthint = 0;
		
		lengthint = atol(args[(argc - 1)].c_str());
			
		debugprint(localdebugmode, PROTOCOL_NAME ": %d bytes of %s data\n",
			lengthint, command.c_str());

		char databuffer[BUFFER_SIZE];
			
		memset(databuffer, 0, BUFFER_SIZE);
	
		if (!(incomingsock.recvalldata(databuffer, lengthint))) return 1;
					
		memcpy(replybuffer + headerlength, databuffer, lengthint);
		*replybufferlength += lengthint;
	}
	
	/* Write out trace packets if enabled. */
	if (tracing) tracepacket("msn", packetcount, replybuffer, *replybufferlength);
	
	packetcount++;

	return 0;
}

int generatemessagepacket(struct response &response, char *replybuffer, int *replybufferlength)
{
	if (groupchat || localid.empty() || remoteid.empty()) return 1;

	std::string body = stringprintf(
		"MIME-Version: 1.0\r\n" \
		"Content-Type: text/plain; charset=UTF-8\r\n" \
		"\r\n" \
		"%s", response.text.c_str());		
	
	if (response.outgoing)
	{
		snprintf(replybuffer, BUFFER_SIZE - 1,
			"MSG 1 U %d\r\n" \
			"%s", body.length(), body.c_str());
	}
	else
	{
		snprintf(replybuffer, BUFFER_SIZE - 1,
			"MSG %s %s %d\r\n" \
			"%s", remoteid.c_str(), remoteid.c_str(), body.length(), body.c_str());
	}
	
	*replybufferlength = strlen(replybuffer);
	
	if (tracing) tracepacket("msn-out", packetcount, replybuffer, *replybufferlength);
	
	packetcount++;
	
	return 0;
}

/* Sets the localid.  ID may have a UUID, if its a 2009 client. */
void setlocalid(std::string id)
{
	localid = id;

	size_t n = localid.find_last_of(";");
	
	if (n != std::string::npos)
		localid = localid.substr(0, n);
}

/* Sets the remoteid, depending on wether or not this is the first remote id
 * spotted. */
void setremoteid(std::string id)
{
	/* ID may have a UUID, if its a 2009 client. */
	std::string idcopy = id;
	
	size_t n = idcopy.find_last_of(";");
	
	if (n != std::string::npos)
		idcopy = idcopy.substr(0, n);
	
	/* Sometimes we can be called with the same ID, ignore those. */
	if (idcopy == remoteid) return;
	
	/* MSN 2009 beta appears to "CALL" itself, thus resulting in a JOI to the
	 * local user. Ignore those too. */
	if (idcopy == localid) return;

	if (!gotremoteid)
	{
		remoteid = idcopy;
		gotremoteid = true;
	}
	else if (!groupchat)
	{
		remoteid = "groupchat-" + stringprintf("%d", (int) time(NULL));
		debugprint(localdebugmode, PROTOCOL_NAME ": Group chat, %s\n", remoteid.c_str());
		groupchat = true;
	}
}

/* This is the MSG processing.  char *msg will point at the start of the block.
 * A message can be: text, typing, file, or stuff we don't know anything about
 * yet. */
bool processmessage(bool outgoing, std::string id, int headerlength, char *msg,
	std::vector<struct imevent> &imevents, std::string clientaddress)
{
	std::map<std::string, std::string> headers;

	int msgtype = MSGTYPE_NULL;
		
	char *start = msg;
	start = getheadervalues(start, headers);
	
	const char *contenttype = headers["Content-Type"].c_str();
			
	if (strncmp(contenttype, "text/plain;", 11) == 0)
			msgtype = MSGTYPE_PLAIN;
	if (strcmp(contenttype, "application/x-msnmsgrp2p") == 0)
			msgtype = MSGTYPE_P2P;
	if (strcmp(contenttype, "text/x-msmsgscontrol") == 0)
			msgtype = MSGTYPE_CONTROL;
			
	if (msgtype != MSGTYPE_NULL)
	{
		struct imevent imevent;
		
		imevent.timestamp = time(NULL);
		imevent.clientaddress = clientaddress;
		imevent.protocolname = PROTOCOL_NAME;
		imevent.outgoing = outgoing;
		imevent.localid = localid;
		imevent.remoteid = remoteid;
		/* We are not yet commited to creating an event. */
		imevent.type = TYPE_NULL;
		imevent.filtered = false;
		imevent.messageextent.start = 0;
		imevent.messageextent.length = 0;
		
		if (msgtype == MSGTYPE_PLAIN)
		{
			imevent.type = TYPE_MSG;
			
			if (outgoing)
				imevent.eventdata = start;
			else if (!groupchat)
				imevent.eventdata = start;
			else
				imevent.eventdata = id + ": " + start;

			imevent.messageextent.start = start - msg + headerlength;
			imevent.messageextent.length = -1; /* NULL terminated. */
		}
		if (msgtype == MSGTYPE_P2P)
		{
			debugprint(localdebugmode, PROTOCOL_NAME ": P2P");
			
			struct p2pheader p2pheader;
			
			memcpy(&p2pheader, start, sizeof(struct p2pheader));
		
			debugprint(localdebugmode, PROTOCOL_NAME ": sessionid: %u id: %u offset: %llu datasize: %llu messagesize: %u",
				p2pheader.sessionid, p2pheader.id, p2pheader.offset, p2pheader.datasize,
				p2pheader.messagesize);

			start += sizeof(struct p2pheader);
			
			/* if Session ID is 0 then it is a "start transfer" message. */
			if (!p2pheader.sessionid)
			{
				std::string invite;
				
				start = getstring(start, invite);
							
				if (strncmp(invite.c_str(), "INVITE ", 7) == 0)
				{
					debugprint(localdebugmode, PROTOCOL_NAME ": now onto header level two");
					
					std::map<std::string, std::string> headersleveltwo;
					start = getheadervalues(start, headersleveltwo);
					
					debugprint(localdebugmode, PROTOCOL_NAME ": now onto header level three");
					
					std::map<std::string, std::string> headerslevelthree;
					start = getheadervalues(start, headerslevelthree);
					
					/* AppID is 2 for normal file transfers. We will ignore buddy icons. */
					if (headerslevelthree["AppID"] == "2")
					{
						struct context context;
						
						memset(&context, 0, sizeof(struct context));
						
						decodebase64(headerslevelthree["Context"], (uint8_t *) &context, sizeof(struct context));
			
						debugprint(localdebugmode, PROTOCOL_NAME ": headerlength: %u version: %u filesize: %llu type: %u",
							context.headerlength, context.version, context.filesize, context.type);
						
						std::string filename;
						
						for (int c = 0; context.filename[c]; c++)
							filename += context.filename[c];					
						
						std::string sessionid = headerslevelthree["SessionID"];
						
						if (!sessionid.empty())
						{
							debugprint(localdebugmode, PROTOCOL_NAME ": FT sessionid: %s filename: %s",
								sessionid.c_str(), filename.c_str());
							filetransfers[atol(sessionid.c_str())] = filename;
						}

						imevent.type = TYPE_FILE;		
						imevent.eventdata = stringprintf("%s %llu bytes", filename.c_str(), 
							(long long unsigned int)context.filesize);
					}
				}
			}
		}
		if (msgtype == MSGTYPE_CONTROL)
		{
			if (!headers["TypingUser"].empty())
			{
				imevent.type = TYPE_TYPING;
				imevent.eventdata = "";
			}
		}
		
		if (imevent.type != TYPE_NULL)
		{
			std::transform(imevent.localid.begin(), imevent.localid.end(), imevent.localid.begin(), tolower);
			std::transform(imevent.remoteid.begin(), imevent.remoteid.end(), imevent.remoteid.begin(), tolower);
			imevents.push_back(imevent);
			
			return true;
		}
	}
	
	return false;
}

/* Chomps down the headers, filling out a map. Returns a pointer to the end of
 * the headers. */
char *getheadervalues(char *buffer, std::map<std::string, std::string> &headers)
{
	char *s = buffer;
	
	while (*s && *s != '\r')
	{
		std::string header, value;
		
		/* Get the header key upto the colon. */
		while (*s && *s != ':')
		{
			header += *s;
			s++;
		}
		/* Past the colon */
		s++;
		/* Past any number of spaces. */
		while (*s && *s == ' ') s++;
		/* Finally, chomp down on the value until the \r. */
		while (*s && *s != '\r')
		{
			value += *s;
			s++;
		}
		
		headers[header] = value;
		
		debugprint(localdebugmode, PROTOCOL_NAME ": header: %s value: %s", header.c_str(),
			value.c_str());
		
		/* If we are at the end of the buffer, hop out. */
		if (!*s) break;
		
		/* Now onto the start of the next line. */
		s += 2;
		
		/* If we have a \r at the start of this line, then it is a blank one.
		 * In that case its the end of the header block so jump out. */
		if (*s && *s == '\r') break;
	}

	/* Past the blank line, and onto the first line of "non header" data. */
	s += 2;
	
	return s;
}

/* Get a string out of the buffer, upto the first \r and advance the buffer
 * pointer to the start of the following line (basic a getline-type funct). */
char *getstring(char *buffer, std::string &str)
{
	char *s = buffer;

	while (*s && *s != '\r')
	{
		str += *s;
		s++;
	}
	s += 2;
	
	return s;
}
