package opt_test

import (
	"strings"
	"testing"

	"git.sr.ht/~rjarry/go-opt"
	"github.com/stretchr/testify/assert"
)

type CompleteStruct struct {
	Name    string   `opt:"-n,--name" required:"true" complete:"CompleteName"`
	Delay   float64  `opt:"--delay"`
	Zero    bool     `opt:"-z"`
	Backoff bool     `opt:"-B,--backoff"`
	Tags    []string `opt:"..." complete:"CompleteTag"`
}

func (c *CompleteStruct) CompleteName(arg string) []string {
	var names []string
	for _, n := range []string{"leonardo", "michelangelo", "rafaelo", "donatello"} {
		if strings.HasPrefix(n, arg) {
			names = append(names, n)
		}
	}
	return names
}

func (c *CompleteStruct) CompleteTag(arg string) []string {
	var results []string
	prefix := ""
	if strings.HasPrefix(arg, "-") {
		prefix = "-"
	} else if strings.HasPrefix(arg, "+") {
		prefix = "+"
	}
	tags := []string{"unread", "sent", "important", "inbox", "trash"}
	for _, t := range tags {
		t = prefix + t
		if strings.HasPrefix(t, arg) {
			results = append(results, t)
		}
	}
	return results
}

func TestComplete(t *testing.T) {
	vectors := []struct {
		cmdline     string
		completions []string
		prefix      string
	}{
		{
			"foo --delay 33..33.3 -n",
			[]string{"-n "},
			"foo --delay 33..33.3 ",
		},
		{
			"foo --delay 33..33.3 -n ",
			[]string{"leonardo", "michelangelo", "rafaelo", "donatello"},
			"foo --delay 33..33.3 -n ",
		},
		{
			"foo --delay 33..33.3 -n don",
			[]string{"donatello"},
			"foo --delay 33..33.3 -n ",
		},
		{
			"foo --delay 33..33.3 --name=",
			[]string{"leonardo", "michelangelo", "rafaelo", "donatello"},
			"foo --delay 33..33.3 --name=",
		},
		{
			"foo --delay 33..33.3 --name=leo",
			[]string{"leonardo"},
			"foo --delay 33..33.3 --name=",
		},
		{
			"foo --nam",
			[]string{
				"--name ",
			},
			"foo ",
		},
		{
			"foo --delay 33..33.3 --backoff",
			[]string{
				"--backoff ",
			},
			"foo --delay 33..33.3 ",
		},
		{
			"foo --delay 33..33.3 -",
			[]string{
				"-unread",
				"-sent",
				"-important",
				"-inbox",
				"-trash",
				"-n ",
				"--name ",
				"-z ",
				"-B ",
				"--backoff ",
			},
			"foo --delay 33..33.3 ",
		},
		{
			"foo --delay 33..33.3 ",
			[]string{
				"unread",
				"sent",
				"important",
				"inbox",
				"trash",
				"-n ",
				"--name ",
				"-z ",
				"-B ",
				"--backoff ",
			},
			"foo --delay 33..33.3 ",
		},
		{
			"foo --delay 33..33.3 -n leonardo i",
			[]string{"important", "inbox"},
			"foo --delay 33..33.3 -n leonardo ",
		},
		{
			"foo +",
			[]string{"+unread", "+sent", "+important", "+inbox", "+trash"},
			"foo ",
		},
		{
			"foo -i",
			[]string{"-important", "-inbox"},
			"foo ",
		},
	}

	for _, v := range vectors {
		t.Run(v.cmdline, func(t *testing.T) {
			args := opt.LexArgs(v.cmdline)
			spec := opt.NewCmdSpec(args.Arg(0), new(CompleteStruct))
			completions, prefix := spec.GetCompletions(args)
			assert.Equal(t, v.completions, completions)
			assert.Equal(t, v.prefix, prefix)
		})
	}
}
