/*
 * $Id: elevator.c,v 1.24 2012-02-22 09:27:20 siflkres Exp $
 *
 * Copyright (C) 2003-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

/*
 * Mapping of dio48 signals to elements of elevator:
 *
 * port	pin#	dir	function
 *
 * 0A	00	in	7-segment-display a
 * 	01	in	7-segment-display b
 * 	02	in	7-segment-display c
 * 	03	in	7-segment-display d
 * 	04	in	7-segment-display e
 * 	05	in	7-segment-display f
 * 	06	in	7-segment-display g
 * 	07	in	big down arrow
 *
 * 0B	08	in	big up arrow
 * 	09	in	led of floor button 'E' (= 0)
 * 	10	in	led of floor button '1'
 * 	11	in	led of floor button '2'
 * 	12	in	up arrow of call button on floor 0
 * 	13	in	down arrow of call button on floor 1
 * 	14	in	up arrow of call button on floor 1
 * 	15	in	down arrow of call button on floor 2
 *
 * 0C	16	in	motor on/off
 * 	17	in	direction down/up
 * 	18	in	open/close control of door on floor 0	
 * 	19	in	open/close control of door on floor 1	
 * 	20	in	open/close control of door on floor 2	
 *	21		(unused)
 *	22		(unused)
 *	23		(unused)
 *
 * 1A	24	out	floor button 'E' (= 0)
 * 	25	out	floor button '1'
 * 	26	out	floor button '2'
 * 	27	out	up call button on floor 0
 * 	28	out	down call button on floor 1
 * 	29	out	up call button on floor 1
 * 	30	out	down call button on floor 2
 * 	31	out	cabin position sensor floor 0
 *
 * 1B	32	out	cabin position sensor floor 1
 * 	33	out	cabin position sensor floor 2
 * 	34	out	door open sensor floor 0
 * 	35	out	door open sensor floor 1
 * 	36	out	door open sensor floor 2
 * 	37		(unused)
 * 	38		(unused)
 * 	39		(unused)
 *
 */

#include <assert.h>
#include <stdbool.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "glue.h"

#include "elevator.h"

struct cpssp {
	struct sig_boolean *port_power;
	struct sig_dio48 *port_dio48;

	struct opt_led {
		struct cpssp * parent;
		int state;
		struct sig_boolean *port_opt_led;
	} opt_leds[16];

	struct button {
		struct cpssp * parent;
		int state;
		struct sig_std_logic *sig;
	} buttons[7];

	struct door_state {
		int opening;
		int state;
		struct sig_std_logic *sensor;
		struct sig_integer *gui_position;
	} doors[3];

	int motor_on;
	int direction;
	int cabin_position;
	struct sig_integer *gui_cabin_position;
	struct sig_std_logic *floor_sensor[3];

	bool power_state;
};

static void
elevator_power_set(void *_cpssp, unsigned int val)
{
	int i;
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

        cpssp->power_state = val;

	if (val) {
		for (i = 0; i < 16; i++) {
			sig_boolean_set(cpssp->opt_leds[i].port_opt_led, cpssp, cpssp->opt_leds[i].state);
		}
		for (i = 0; i < 7; i++) {
			sig_std_logic_set(cpssp->buttons[i].sig, cpssp, cpssp->buttons[i].state);
		}
	} else {
		for (i = 0; i < 16; i++) {
			sig_boolean_set(cpssp->opt_leds[i].port_opt_led, cpssp, 0);
		}
		for (i = 0; i < 7; i++) {
			sig_std_logic_set(cpssp->buttons[i].sig, cpssp, SIG_STD_LOGIC_Z);
		}
		for (i = 0; i < 3; i++) {
			sig_std_logic_set(cpssp->floor_sensor[i], cpssp, SIG_STD_LOGIC_Z);
		}
		for (i = 0; i < 3; i++) {
			sig_std_logic_set(cpssp->doors[i].sensor, cpssp, SIG_STD_LOGIC_Z);
		}
	}
}

static void
elevator_led_set(void *_cpssp, unsigned int val)
{
	struct opt_led * cpssp = (struct opt_led *) _cpssp;

	if (val == SIG_STD_LOGIC_1) {
		cpssp->state = 1;
	} else {
		cpssp->state = 0;
	}

	if (cpssp->parent->power_state) {
		sig_boolean_set(cpssp->port_opt_led, cpssp->parent, cpssp->state);
	}
}

static void
elevator_button_set(void *_cpssp, unsigned int val)
{
	struct button * cpssp = (struct button *) _cpssp;

	if (val) {
		cpssp->state = SIG_STD_LOGIC_1;
	} else {
		cpssp->state = SIG_STD_LOGIC_0;
	}

	if (cpssp->parent->power_state) {
		sig_std_logic_set(cpssp->sig, cpssp->parent, cpssp->state);
	}
}

static void
elevator_motor_set(void *_cpssp, unsigned int val)
{
	struct cpssp * cpssp = (struct cpssp *) _cpssp;

	if (val == SIG_STD_LOGIC_0) {
		cpssp->motor_on = 0;
	} else if (val == SIG_STD_LOGIC_1) {
		cpssp->motor_on = 1;
	}
}

static void
elevator_direction_set(void *_cpssp, unsigned int val)
{
	struct cpssp * cpssp = (struct cpssp *) _cpssp;

	if (val == SIG_STD_LOGIC_0) {
		cpssp->direction = 0;
	} else if (val == SIG_STD_LOGIC_1) {
		cpssp->direction = 1;
	}
}

static void
elevator_door_set(void *_cpssp, unsigned int val)
{
	struct door_state * cpssp = (struct door_state *) _cpssp;

	if (val == SIG_STD_LOGIC_0) {
		cpssp->opening = 0;
	} else if (val == SIG_STD_LOGIC_1) {
		cpssp->opening = 1;
	}
}

static void
do_elevator(void * _cpssp)
{
	int i;
	struct cpssp * cpssp = (struct cpssp *) _cpssp;

	if (cpssp->power_state) {

		/* motor control */

		if (cpssp->motor_on) {
			if (cpssp->direction == 1) {
				if(cpssp->cabin_position < 200) {
					cpssp->cabin_position += 5;
				}
			} else if (cpssp->direction == 0) {
				if(cpssp->cabin_position > 0) {
					cpssp->cabin_position -= 5;
				}
			}
		}

		sig_integer_set(cpssp->gui_cabin_position, cpssp, cpssp->cabin_position);

		/* cabin position sensors */

		if (cpssp->cabin_position == 0) {
			sig_std_logic_set(cpssp->floor_sensor[0], cpssp, SIG_STD_LOGIC_1);
			sig_std_logic_set(cpssp->floor_sensor[1], cpssp, SIG_STD_LOGIC_0);
			sig_std_logic_set(cpssp->floor_sensor[2], cpssp, SIG_STD_LOGIC_0);
		} else if(cpssp->cabin_position == 100) {
			sig_std_logic_set(cpssp->floor_sensor[0], cpssp, SIG_STD_LOGIC_0);
			sig_std_logic_set(cpssp->floor_sensor[1], cpssp, SIG_STD_LOGIC_1);
			sig_std_logic_set(cpssp->floor_sensor[2], cpssp, SIG_STD_LOGIC_0);
		} else if(cpssp->cabin_position == 200) {
			sig_std_logic_set(cpssp->floor_sensor[0], cpssp, SIG_STD_LOGIC_0);
			sig_std_logic_set(cpssp->floor_sensor[1], cpssp, SIG_STD_LOGIC_0);
			sig_std_logic_set(cpssp->floor_sensor[2], cpssp, SIG_STD_LOGIC_1);
		} else {
			sig_std_logic_set(cpssp->floor_sensor[0], cpssp, SIG_STD_LOGIC_0);
			sig_std_logic_set(cpssp->floor_sensor[1], cpssp, SIG_STD_LOGIC_0);
			sig_std_logic_set(cpssp->floor_sensor[2], cpssp, SIG_STD_LOGIC_0);
		}

		/* door control */

		for (i = 0; i < 3; i++) {
			if (cpssp->doors[i].opening) {
				if(cpssp->doors[i].state < 100) {
					cpssp->doors[i].state += 10;
				}
			} else {
				if(cpssp->doors[i].state > 0) {
					cpssp->doors[i].state -= 10;
				}
			}

			sig_integer_set(cpssp->doors[i].gui_position, cpssp, cpssp->doors[i].state);

			if (cpssp->doors[i].state == 0) {
				sig_std_logic_set(cpssp->doors[i].sensor, cpssp, SIG_STD_LOGIC_0);
			} else {
				sig_std_logic_set(cpssp->doors[i].sensor, cpssp, SIG_STD_LOGIC_1);
			}
		}
	}

	time_call_after(TIME_HZ / 8, do_elevator, cpssp);
}

void *
elevator_create(
	const char *name,
	struct sig_manage *port_manage,
	struct sig_boolean *port_mech_power_switch,
	struct sig_dio48 *port_dio48,
	struct sig_boolean *port_opt_led_a,
	struct sig_boolean *port_opt_led_b,
	struct sig_boolean *port_opt_led_c,
	struct sig_boolean *port_opt_led_d,
	struct sig_boolean *port_opt_led_e,
	struct sig_boolean *port_opt_led_f,
	struct sig_boolean *port_opt_led_g,
	struct sig_boolean *port_opt_led_arrow_down,
	struct sig_boolean *port_opt_led_arrow_up,
	struct sig_boolean *port_opt_led_panel_0,
	struct sig_boolean *port_opt_led_panel_1,
	struct sig_boolean *port_opt_led_panel_2,
	struct sig_boolean *port_opt_led_0_up,
	struct sig_boolean *port_opt_led_1_down,
	struct sig_boolean *port_opt_led_1_up,
	struct sig_boolean *port_opt_led_2_down,
	struct sig_boolean *port_mech_button_panel_0,
	struct sig_boolean *port_mech_button_panel_1,
	struct sig_boolean *port_mech_button_panel_2,
	struct sig_boolean *port_mech_button_0_up,
	struct sig_boolean *port_mech_button_1_down,
	struct sig_boolean *port_mech_button_1_up,
	struct sig_boolean *port_mech_button_2_down,
	struct sig_integer *port_mech_door_0,
	struct sig_integer *port_mech_door_1,
	struct sig_integer *port_mech_door_2,
	struct sig_integer *port_mech_cabin
)
{
	static struct sig_boolean_funcs epf = {
		.set = elevator_power_set
	};
	static struct sig_std_logic_funcs led_funcs = {
		.set = elevator_led_set
	};
	static struct sig_boolean_funcs button_funcs = {
		.set = elevator_button_set
	};
	static struct sig_std_logic_funcs motor_funcs = {
		.set = elevator_motor_set
	};
	static struct sig_std_logic_funcs direction_funcs = {
		.set = elevator_direction_set
	};
	static struct sig_std_logic_funcs door_funcs = {
		.set = elevator_door_set
	};
	struct cpssp *cpssp;
	int i;

	cpssp = shm_alloc(sizeof(*cpssp));
	assert(cpssp);
	memset(cpssp, 0, sizeof(*cpssp)); /* FIXME */

	cpssp->port_dio48 = port_dio48;

	/* connect leds */

	cpssp->opt_leds[0].port_opt_led = port_opt_led_a;
	cpssp->opt_leds[1].port_opt_led = port_opt_led_b;
	cpssp->opt_leds[2].port_opt_led = port_opt_led_c;
	cpssp->opt_leds[3].port_opt_led = port_opt_led_d;
	cpssp->opt_leds[4].port_opt_led = port_opt_led_e;
	cpssp->opt_leds[5].port_opt_led = port_opt_led_f;
	cpssp->opt_leds[6].port_opt_led = port_opt_led_g;
	cpssp->opt_leds[7].port_opt_led = port_opt_led_arrow_down;
	cpssp->opt_leds[8].port_opt_led = port_opt_led_arrow_up;
	cpssp->opt_leds[9].port_opt_led = port_opt_led_panel_0;
	cpssp->opt_leds[10].port_opt_led = port_opt_led_panel_1;
	cpssp->opt_leds[11].port_opt_led = port_opt_led_panel_2;
	cpssp->opt_leds[12].port_opt_led = port_opt_led_0_up;
	cpssp->opt_leds[13].port_opt_led = port_opt_led_1_down;
	cpssp->opt_leds[14].port_opt_led = port_opt_led_1_up;
	cpssp->opt_leds[15].port_opt_led = port_opt_led_2_down;

	for (i = 0; i < 16; i++) {
		cpssp->opt_leds[i].parent = cpssp;
		sig_boolean_connect_out(cpssp->opt_leds[i].port_opt_led, cpssp, 0);
		sig_std_logic_connect_in(port_dio48->sig[i], &cpssp->opt_leds[i], &led_funcs);
	}

	/* connect motor control and sensors */

	cpssp->gui_cabin_position = port_mech_cabin;
	cpssp->motor_on = 0;
	cpssp->direction = 0;
	cpssp->cabin_position = 0;
	for (i = 0; i < 3; i++) {
		cpssp->floor_sensor[i] = port_dio48->sig[31 + i];
		sig_std_logic_connect_out(cpssp->floor_sensor[i], cpssp, SIG_STD_LOGIC_0);
	}
	sig_integer_set(port_mech_cabin, cpssp, 0);
	sig_std_logic_connect_in(port_dio48->sig[16], cpssp, &motor_funcs);
	sig_std_logic_connect_in(port_dio48->sig[17], cpssp, &direction_funcs);

	/* connect door control and sensors */

	cpssp->doors[0].gui_position = port_mech_door_0;
	cpssp->doors[1].gui_position = port_mech_door_1;
	cpssp->doors[2].gui_position = port_mech_door_2;
	for (i = 0; i < 3; i++) {
		cpssp->doors[i].state = 0;
		cpssp->doors[i].opening = 0;
		cpssp->doors[i].sensor = port_dio48->sig[34 + i];
		sig_integer_set(cpssp->doors[i].gui_position, cpssp, 0);
		sig_std_logic_connect_out(cpssp->doors[i].sensor, cpssp, SIG_STD_LOGIC_0);
		sig_std_logic_connect_in(port_dio48->sig[18 + i], &cpssp->doors[i], &door_funcs);
	}

	/* connect buttons */

	for (i = 0; i < 7; i++) {
		cpssp->buttons[i].parent = cpssp;
		cpssp->buttons[i].sig = port_dio48->sig[24 + i];
		sig_std_logic_connect_out(cpssp->buttons[i].sig, cpssp, SIG_STD_LOGIC_0);
	}

	sig_boolean_connect_in(port_mech_button_panel_0, &cpssp->buttons[0], &button_funcs);
	sig_boolean_connect_in(port_mech_button_panel_1, &cpssp->buttons[1], &button_funcs);
	sig_boolean_connect_in(port_mech_button_panel_2, &cpssp->buttons[2], &button_funcs);
	sig_boolean_connect_in(port_mech_button_0_up, &cpssp->buttons[3], &button_funcs);
	sig_boolean_connect_in(port_mech_button_1_down, &cpssp->buttons[4], &button_funcs);
	sig_boolean_connect_in(port_mech_button_1_up, &cpssp->buttons[5], &button_funcs);
	sig_boolean_connect_in(port_mech_button_2_down, &cpssp->buttons[6], &button_funcs);

	/* connect power switch */

	cpssp->port_power = port_mech_power_switch;
	sig_boolean_connect_in(port_mech_power_switch, cpssp, &epf);

	time_call_after(TIME_HZ / 8, do_elevator, cpssp);

	return cpssp;
}

void
elevator_destroy(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

	shm_free(cpssp);
}

void
elevator_suspend(void *_cpssp, FILE *fComp)
{
	struct cpssp *cpssp = _cpssp;
	
	generic_suspend(cpssp, sizeof(*cpssp), fComp);
}

void
elevator_resume(void *_cpssp, FILE *fComp)
{
	struct cpssp *cpssp = _cpssp;
	
	generic_resume(cpssp, sizeof(*cpssp), fComp);
}
