#  Copyright (c) 2023, Manfred Moitzi
#  License: MIT License

import pytest

from ezdxf.lldxf.tags import Tags
from ezdxf.proxygraphic import ProxyGraphic
from ezdxf.query import EntityQuery

@pytest.fixture(scope="module")
def proxy_graphic() -> bytes:
    data = Tags.from_text(TAGS)
    return b"".join(tag.value for tag in data)


def test_loaded_data(proxy_graphic: bytes):
    assert len(proxy_graphic) == 4768


def test_circle_with_inverted_normal(proxy_graphic: bytes):
    """ The circle has a normal vector but center is in WCS.

    It's not clear if all coordinates in proxy graphics are WCS coordinates even for OCS
    entities - the ODA documentation about proxy graphic has no information about that
    and I have not enough test data with normals which are not aligned to the WCS
    Z-axis for a final conclusion
    """
    pg = ProxyGraphic(proxy_graphic)
    entities = EntityQuery(pg.virtual_entities())
    assert len(entities) == 22

    circle = entities.query("CIRCLE").first
    assert circle.dxftype() == "CIRCLE"
    assert circle.dxf.extrusion.isclose((0, 0, -1))
    assert circle.dxf.center.isclose((-131.9091007888657, 309.8491975967545))


TAGS = """310
A0120000FB0000000C00000010000000020000000C00000016000000000000C00C00000012000000FE7F00000C00000017000000FFFFFFFF1000000018000000000000000000F03F0C0000001400000002000000100000001A00000000000000000000000C00000010000000000000000C00000012000000000000000C0000
310
0017000000FFFFFFFF1000000018000000000000000000F03F0C00000016000000070000C35400000020000000020000006BF74832109E6040E0AB9694713F73400000000000000000FC0D013154A6604097A03A954F3B7340000000000000000000000000000000000000000000000000000000000000F0BF0C0000001000
310
0000020000000C00000016000000000000C00C00000012000000FE7F00000C00000017000000000000001000000018000000000000000000F03F0C00000010000000000000000C00000012000000000000000C00000017000000FFFFFFFF1000000018000000000000000000F03F0C00000016000000070000C35400000020
310
0000000200000089F3881987956040B2B1B6393F4773400000000000000000B80652F99FAA60401B28D2C9B23C7340000000000000000000000000000000000000000000000000000000000000F0BF0C00000010000000020000000C00000016000000000000C00C00000012000000FE7F00000C0000001700000000000000
310
1000000018000000000000000000F03F0C00000010000000000000000C00000012000000000000000C00000017000000FFFFFFFF1000000018000000000000000000F03F0C00000016000000070000C3540000002000000002000000A8EFC800FE8C604084B7D6DE0C4F7340000000000000000007B5C649B0AE6040D4D457
310
BA333E7340000000000000000000000000000000000000000000000000000000000000F0BF0C00000010000000020000000C00000016000000000000C00C00000012000000FE7F00000C00000017000000000000001000000018000000000000000000F03F0C00000010000000000000000C00000012000000000000000C00
310
000017000000FFFFFFFF1000000018000000000000000000F03F0C00000016000000070000C3540000002000000002000000C7EB08E87484604056BDF683DA567340000000000000000089C476CD86B26040F5D03F91D13F7340000000000000000000000000000000000000000000000000000000000000F0BF0C00000010
310
000000020000000C00000016000000000000C00C00000012000000FE7F00000C00000017000000000000001000000018000000000000000000F03F0C00000010000000000000000C00000012000000000000000C00000017000000FFFFFFFF1000000018000000000000000000F03F0C00000016000000070000C354000000
310
2000000002000000E5E748CFEB7B604028C31629A85E73400000000000000000C965B57124B660403684E0D78B417340000000000000000000000000000000000000000000000000000000000000F0BF0C00000010000000020000000C00000016000000000000C00C00000012000000FE7F00000C00000017000000000000
310
001000000018000000000000000000F03F0C00000010000000000000000C00000012000000000000000C00000017000000FFFFFFFF1000000018000000000000000000F03F0C00000016000000070000C354000000200000000200000004E488B662736040FAC836CE756673400000000000000000894CA27489B96040B814
310
2A6F62437340000000000000000000000000000000000000000000000000000000000000F0BF0C00000010000000020000000C00000016000000000000C00C00000012000000FE7F00000C00000017000000000000001000000018000000000000000000F03F0C00000010000000000000000C00000012000000000000000C
310
00000017000000FFFFFFFF1000000018000000000000000000F03F0C00000016000000070000C354000000200000000200000023E0C89DD96A6040CCCE5673436E734000000000000000001264E368B5BC6040D48CC98D55457340000000000000000000000000000000000000000000000000000000000000F0BF0C000000
310
10000000020000000C00000016000000000000C00C00000012000000FE7F00000C00000017000000000000001000000018000000000000000000F03F0C00000010000000000000000C00000012000000000000000C00000017000000FFFFFFFF1000000018000000000000000000F03F0C00000016000000070000C3540000
310
00200000000200000041DC0885506260409ED47618117673400000000000000000A7DE0F2FA7BF60406B5373C365477340000000000000000000000000000000000000000000000000000000000000F0BF0C00000010000000020000000C00000016000000000000C00C00000012000000FE7F00000C000000170000000000
310
00001000000018000000000000000000F03F0C00000010000000000000000C00000012000000000000000C00000017000000FFFFFFFF1000000018000000000000000000F03F0C00000016000000070000C354000000200000000200000060D8486CC759604070DA96BDDE7D73400000000000000000D0EBD9E35CC26040B8
310
50CE0194497340000000000000000000000000000000000000000000000000000000000000F0BF0C00000010000000020000000C00000016000000000000C00C00000012000000FE7F00000C00000017000000000000001000000018000000000000000000F03F0C00000010000000000000000C0000001200000000000000
310
0C00000017000000FFFFFFFF1000000018000000000000000000F03F0C00000016000000070000C35400000020000000020000000D12C812AE5960407B4117837481734000000000000000005BE447C0D3C46040545857ACE14B7340000000000000000000000000000000000000000000000000000000000000F0BF0C0000
310
0010000000020000000C00000016000000000000C00C00000012000000FE7F00000C00000017000000000000001000000018000000000000000000F03F0C00000010000000000000000C00000012000000000000000C00000017000000FFFFFFFF1000000018000000000000000000F03F0C00000016000000070000C35400
310
00002000000002000000292CC21D8C5E60404E385A968E827340000000000000000002D2CDE607C760406265D4B1504E7340000000000000000000000000000000000000000000000000000000000000F0BF0C00000010000000020000000C00000016000000000000C00C00000012000000FE7F00000C0000001700000000
310
0000001000000018000000000000000000F03F0C00000010000000000000000C00000012000000000000000C00000017000000FFFFFFFF1000000018000000000000000000F03F0C00000016000000070000C3540000002000000002000000EC261A22B2636040CE3EEEAC84837340000000000000000002DFF513F4C86040
310
C36200B4E3507340000000000000000000000000000000000000000000000000000000000000F0BF0C00000010000000020000000C00000016000000000000C00C00000012000000FE7F00000C00000017000000000000001000000018000000000000000000F03F0C00000010000000000000000C00000012000000000000
310
000C00000017000000FFFFFFFF1000000018000000000000000000F03F0C00000016000000070000C354000000200000000200000076CDCE45276960406AEFD3335384734000000000000000003A40C12191CA604008B6DA459E537340000000000000000000000000000000000000000000000000000000000000F0BF0C00
310
000010000000020000000C00000016000000000000C00C00000012000000FE7F00000C00000017000000000000001000000018000000000000000000F03F0C00000010000000000000000C00000012000000000000000C00000017000000FFFFFFFF1000000018000000000000000000F03F0C00000016000000070000C354
310
00000020000000020000008810F35AF56E6040C3D10142F58473400000000000000000EB041D3ED5CB604091D76C5085567340000000000000000000000000000000000000000000000000000000000000F0BF0C00000010000000020000000C00000016000000000000C00C00000012000000FE7F00000C00000017000000
310
000000001000000018000000000000000000F03F0C00000010000000000000000C00000012000000000000000C00000017000000FFFFFFFF1000000018000000000000000000F03F0C00000016000000070000C354000000200000000200000082126D352A756040A7D484ED638573400000000000000000B30A2395B2CC60
310
408FD8A9BD9F597340000000000000000000000000000000000000000000000000000000000000F0BF0C00000010000000020000000C00000016000000000000C00C00000012000000FE7F00000C00000017000000000000001000000018000000000000000000F03F0C00000010000000000000000C000000120000000000
310
00000C00000017000000FFFFFFFF1000000018000000000000000000F03F0C00000016000000070000C35400000020000000020000003AA0AB16DA7B6040AC91A515958573400000000000000000BE8464E514CD60406B1F49AEF75C7340000000000000000000000000000000000000000000000000000000000000F0BF0C
310
00000010000000020000000C00000016000000000000C00C00000012000000FE7F00000C00000017000000000000001000000018000000000000000000F03F0C00000010000000000000000C00000012000000000000000C00000017000000FFFFFFFF1000000018000000000000000000F03F0C00000016000000070000C3
310
540000002000000002000000F5179A7F24836040B159EEF9788573400000000000000000C614F6ADDCCC6040485BC0E29C607340000000000000000000000000000000000000000000000000000000000000F0BF0C00000010000000020000000C00000016000000000000C00C00000012000000FE7F00000C000000170000
310
00000000001000000018000000000000000000F03F0C00000010000000000000000C00000012000000000000000C00000017000000FFFFFFFF1000000018000000000000000000F03F0C00000016000000070000C354000000200000000200000026BCB42C3F8B6040790B21BCF4847340000000000000000057785B32D4CB
310
604061AD4D39AA647340000000000000000000000000000000000000000000000000000000000000F0BF0C00000010000000020000000C00000016000000000000C00C00000012000000FE7F00000C00000017000000000000001000000018000000000000000000F03F0C00000010000000000000000C0000001200000000
310
0000000C00000017000000FFFFFFFF1000000018000000000000000000F03F0C00000016000000070000C35400000020000000020000004108EE06959460404D69C4E7D28373400000000000000000FF33A28990C960406E536A2655697340000000000000000000000000000000000000000000000000000000000000F0BF
310
0C00000010000000020000000C00000016000000000000C00C00000012000000FE7F00000C00000017000000000000001000000018000000000000000000F03F0C00000010000000000000000C00000012000000000000000C00000017000000FFFFFFFF1000000018000000000000000000F03F0C00000016000000070000
310
C3540000002000000002000000A631C74F48A060407CD8175C8281734000000000000000005D124972EFC4604020E8D6CA2E6F7340000000000000000000000000000000000000000000000000000000000000F0BF0C00000010000000020000000C00000016000000000000C00C00000012000000FE7F00000C0000001700
310
0000000000001000000018000000000000000000F03F0C00000010000000000000000C00000012000000FF7F00000C00000017000000FFFFFFFF1000000018000000000000000000F03F0C00000016000000070000C30C00000014000000020000004000000002000000459E895A177D6040701E3850965D73400000000000
310
000000000000000000044000000000000000000000000000000000000000000000F0BF0C00000014000000010000000C00000010000000020000000C00000016000000000000C00C00000012000000FE7F00000C00000017000000000000001000000018000000000000000000F03F0C00000010000000000000000C000000
310
12000000FF7F00000C00000017000000FFFFFFFF1000000018000000000000000000F03F0C00000016000000070000C3540000002000000002000000C7382E33B456604039B73A6EAE8073400000000000000000C303E5817AA36040A78535327E3A7340000000000000000000000000000000000000000000000000000000
310
000000F0BF0C00000010000000020000000C00000016000000000000C00C00000012000000FE7F00000C00000017000000000000001000000018000000000000000000F03F
"""


if __name__ == '__main__':
    pytest.main([__file__])

