//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/View/Projection/ProjectionsToolbar.cpp
//! @brief     Implements class ProjectionsToolbar
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "GUI/View/Projection/ProjectionsToolbar.h"
#include "GUI/View/Projection/ProjectionsEditorActions.h"
#include "GUI/View/Tool/mainwindow_constants.h"
#include <QButtonGroup>
#include <QLabel>
#include <QToolButton>

namespace {

const QString pan_zoom_tooltip = "Pan/zoom mode (space)\n"
                                 "Drag axes with the mouse, use mouse wheel to zoom in/out";

const QString reset_view_tooltip = "Reset view\nx,y,z axes range will be set to default";

const QString selection_mode_tooltip =
    "Selection mode\nYou can select existing projections and move them around";

const QString horizontal_mode_tooltip =
    "Horizontal projections mode\nCreate projection along x-axis by clicking on color map";

const QString vertical_mode_tooltip =
    "Vertical projections mode\nCreate projection along y-axis by clicking on color map";

} // namespace

ProjectionsToolbar::ProjectionsToolbar(ProjectionsEditorActions* editorActions, QWidget* parent)
    : QToolBar(parent)
    , m_editorActions(editorActions)
    , m_activityButtonGroup(new QButtonGroup(this))
{
    setIconSize(QSize(GUI::Constants::toolbar_icon_size, GUI::Constants::toolbar_icon_size));
    setProperty("_q_custom_style_disabled", QVariant(true));

    setup_selection_group();
    setup_shapes_group();
    setup_extratools_group();

    connect(m_activityButtonGroup, &QButtonGroup::idClicked, this,
            &ProjectionsToolbar::onActivityGroupChange);

    m_previousActivity = currentActivity();
}

void ProjectionsToolbar::onChangeActivityRequest(MaskEditorFlags::Activity value)
{
    if (value == MaskEditorFlags::PREVIOUS_MODE)
        setCurrentActivity(m_previousActivity);
    else {
        m_previousActivity = currentActivity();
        setCurrentActivity(value);
    }
    emit activityModeChanged(currentActivity());
}

//! Change activity only if current activity is one of drawing mode (horizontal, vertical
//! projections drawing).
void ProjectionsToolbar::onProjectionTabChange(MaskEditorFlags::Activity value)
{
    if (currentActivity() == MaskEditorFlags::HORIZONTAL_LINE_MODE
        || currentActivity() == MaskEditorFlags::VERTICAL_LINE_MODE)
        onChangeActivityRequest(value);
}

void ProjectionsToolbar::onActivityGroupChange(int)
{
    emit activityModeChanged(currentActivity());
}

void ProjectionsToolbar::setup_selection_group()
{
    auto* panButton = new QToolButton(this);
    panButton->setIcon(QIcon(":/images/hand-right.svg"));
    panButton->setToolTip(pan_zoom_tooltip);
    panButton->setCheckable(true);
    panButton->setChecked(true);
    addWidget(panButton);

    auto* resetViewButton = new QToolButton(this);
    resetViewButton->setIcon(QIcon(":/images/camera-metering-center.svg"));
    resetViewButton->setToolTip(reset_view_tooltip);
    addWidget(resetViewButton);
    connect(resetViewButton, &QToolButton::clicked, m_editorActions,
            &ProjectionsEditorActions::resetViewRequest);

    add_separator();

    auto* selectionButton = new QToolButton(this);
    selectionButton->setIcon(QIcon(":/Mask/images/maskeditor_arrow.svg"));
    selectionButton->setToolTip(selection_mode_tooltip);
    selectionButton->setCheckable(true);
    addWidget(selectionButton);

    m_activityButtonGroup->addButton(panButton, MaskEditorFlags::PAN_ZOOM_MODE);
    m_activityButtonGroup->addButton(selectionButton, MaskEditorFlags::SELECTION_MODE);
}

void ProjectionsToolbar::setup_shapes_group()
{
    auto* horizontalLineButton = new QToolButton(this);
    horizontalLineButton->setIcon(QIcon(":/Mask/images/maskeditor_horizontalline.svg"));
    horizontalLineButton->setToolTip(horizontal_mode_tooltip);
    horizontalLineButton->setCheckable(true);
    addWidget(horizontalLineButton);

    auto* verticalLineButton = new QToolButton(this);
    verticalLineButton->setIcon(QIcon(":/Mask/images/maskeditor_verticalline.svg"));
    verticalLineButton->setToolTip(vertical_mode_tooltip);
    verticalLineButton->setCheckable(true);
    addWidget(verticalLineButton);

    m_activityButtonGroup->addButton(verticalLineButton, MaskEditorFlags::VERTICAL_LINE_MODE);
    m_activityButtonGroup->addButton(horizontalLineButton, MaskEditorFlags::HORIZONTAL_LINE_MODE);

    add_separator();
}

void ProjectionsToolbar::setup_extratools_group()
{
    auto* saveButton = new QToolButton(this);
    saveButton->setIcon(QIcon(":/Mask/images/maskeditor_save.svg"));
    saveButton->setToolTip("Save created projections in multi-column ASCII file.");
    addWidget(saveButton);
    connect(saveButton, &QToolButton::clicked, m_editorActions,
            &ProjectionsEditorActions::onSaveAction);
}

void ProjectionsToolbar::add_separator()
{
    addWidget(new QLabel(" "));
    addSeparator();
    addWidget(new QLabel(" "));
}

MaskEditorFlags::Activity ProjectionsToolbar::currentActivity() const
{
    return MaskEditorFlags::EActivityType(m_activityButtonGroup->checkedId());
}

void ProjectionsToolbar::setCurrentActivity(MaskEditorFlags::Activity value)
{
    int button_index = static_cast<int>(value);
    m_activityButtonGroup->button(button_index)->setChecked(true);
}
