//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/View/Numeric/DoubleSpinBox.cpp
//! @brief     Implements class DoubleSpinBox
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2021
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "GUI/View/Numeric/DoubleSpinBox.h"
#include "GUI/View/Numeric/NumberUtil.h"
#include <QWheelEvent>

DoubleSpinBox::DoubleSpinBox(DoubleProperty& d, bool easyScrollable, QWidget* parent)
    : QDoubleSpinBox(parent)
    , m_valueProperty(d)
    , m_easyScrollable(easyScrollable)
{
    setFocusPolicy(Qt::StrongFocus);
    GUI::View::NumberUtil::configSpinbox(this, d.decimals(), d.limits());
    setToolTip(d.tooltip());
    setSizePolicy(QSizePolicy::Minimum, QSizePolicy::Fixed);

    if (std::holds_alternative<QString>(m_valueProperty.unit()))
        setDisplayUnit(Unit::other);
    else
        setDisplayUnit(std::get<Unit>(m_valueProperty.unit()));

    QObject::connect(this, QOverload<double>::of(&QDoubleSpinBox::valueChanged), this,
                     &DoubleSpinBox::onDisplayValueChanged);
    setSingleStep(m_valueProperty.step());
}
void DoubleSpinBox::setDisplayUnit(Unit displayUnit)
{
    m_displayUnit = displayUnit;

    if (m_showUnitAsSuffix) {
        const QString suffix = displayUnitAsString();
        if (suffix.isEmpty())
            setSuffix("");
        else
            setSuffix(" " + suffix);
    }

    QSignalBlocker b(this);
    setValue(toDisplayValue(m_valueProperty.value()));
}

double DoubleSpinBox::toDisplayValue(double baseValue) const
{
    return convert(baseValue, baseUnit(), m_displayUnit);
}

double DoubleSpinBox::toBaseValue(double displayValue) const
{
    return convert(displayValue, m_displayUnit, baseUnit());
}

QString DoubleSpinBox::displayUnitAsString() const
{
    if (std::holds_alternative<QString>(m_valueProperty.unit()))
        return std::get<QString>(m_valueProperty.unit());

    return unitAsString(m_displayUnit);
}

// This method is only used in undo/redo functionality in SampleEditorController
QString DoubleSpinBox::uid() const
{
    return m_valueProperty.uid();
}

// This method is only used in undo/redo functionality in SampleEditorController
// On its removal the input arguments of DoubleSpinbox and GUI::Util::createDoubleSpinBoxRow
// can be changed to "const DoubleProperty&"
void DoubleSpinBox::setPropertyValue(double v)
{
    m_valueProperty.setValue(v);
}

void DoubleSpinBox::setBaseValue(double baseValue)
{
    setValue(toDisplayValue(baseValue));
}

void DoubleSpinBox::wheelEvent(QWheelEvent* event)
{
    if (hasFocus() || m_easyScrollable)
        QDoubleSpinBox::wheelEvent(event);
    else
        event->ignore();
}

void DoubleSpinBox::onDisplayValueChanged(double newDisplayValue)
{
    emit baseValueChanged(toBaseValue(newDisplayValue));
}

Unit DoubleSpinBox::baseUnit() const
{
    if (std::holds_alternative<QString>(m_valueProperty.unit()))
        return Unit::other;

    return std::get<Unit>(m_valueProperty.unit());
}

void DoubleSpinBox::updateValue()
{
    QSignalBlocker b(this);
    setBaseValue(m_valueProperty.value());
}
