//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/Model/Sample/SampleValidator.cpp
//! @brief     Implements class SampleValidator
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "GUI/Model/Sample/SampleValidator.h"
#include "Base/Util/Assert.h"
#include "GUI/Model/Sample/CompoundItem.h"
#include "GUI/Model/Sample/CoreAndShellItem.h"
#include "GUI/Model/Sample/InterferenceItems.h"
#include "GUI/Model/Sample/LayerItem.h"
#include "GUI/Model/Sample/ParticleLayoutItem.h"
#include "GUI/Model/Sample/SampleItem.h"

void SampleValidator::validateItem(const ItemWithParticles* item)
{
    if (const auto* p = dynamic_cast<const CoreAndShellItem*>(item)) {
        if (!p->coreItem())
            addMessage("Sim/shell particle doesn't have core defined.");
        if (!p->shellItem())
            addMessage("Sim/shell particle doesn't have shell defined.");
    } else if (const auto* p = dynamic_cast<const CompoundItem*>(item))
        if (p->itemsWithParticles().isEmpty())
            addMessage("Particle composition doesn't have any particles.");
}

void SampleValidator::addMessage(const QString& m)
{
    m_messages += QString("* ") + m + "\n";
}

bool SampleValidator::isValidSample(const SampleItem* sample)
{
    m_messages.clear();

    QVector<LayerItem*> layers = sample->layerItems();

    if (layers.isEmpty())
        addMessage("MultiLayer should contain at least one layer.");
    if (layers.size() == 1)
        if (layers.front()->layoutItems().isEmpty())
            addMessage("The single layer in your MultiLayer should contain a particle layout.");

    for (const auto* layer : layers) {
        for (const auto* layout : layer->layoutItems()) {
            if (layout->itemsWithParticles().isEmpty())
                addMessage("Particle layout doesn't contain any particles.");
            for (const auto* particle : layout->containedItemsWithParticles()) {
                validateItem(particle);
            }
        }
    }
    if (!m_messages.isEmpty())
        m_messages = "Cannot setup DWBA simulation for given MultiLayer.\n" + m_messages;

    return m_messages.isEmpty();
}
