# -*- Autoconf -*-
#
# Copyright (c) 2005-2007 The ABINIT Group (Yann Pouillon)
# All rights reserved.
#
# This file is part of the ABINIT software package. For license information,
# please see the COPYING file in the top-level directory of the ABINIT source
# distribution.
#

#
# MPI support for ABINIT
#



# _ABI_MPI_CHECK_GENERIC(PREFIX)
# ------------------------------
#
# Looks for a generic implementation of MPI, using the provided prefix.
#
AC_DEFUN([_ABI_MPI_CHECK_GENERIC],
[dnl Set default values
 mpi_generic_prefix="$1"
 mpi_generic_usable="no"
 mpi_generic_cc=""
 mpi_generic_cxx=""
 mpi_generic_fc=""
 mpi_generic_run=""
 mpi_generic_cppflags=""
 mpi_generic_cflags=""
 mpi_generic_cxxflags=""
 mpi_generic_fcflags=""
 mpi_generic_ldflags=""

 AC_MSG_CHECKING([for a usable generic implementation])

 dnl Check whether generic files are available
 if test "${mpi_generic_prefix}" != ""; then
  if test -x "${mpi_generic_prefix}/bin/mpicc"; then
   mpi_generic_cc="${mpi_generic_prefix}/bin/mpicc"
  fi
  if test -x "${mpi_generic_prefix}/bin/mpic++"; then
   mpi_generic_cxx="${mpi_generic_prefix}/bin/mpic++"
  fi
  if test -x "${mpi_generic_prefix}/bin/mpif90"; then
   mpi_generic_fc="${mpi_generic_prefix}/bin/mpif90"
  fi
  if test -x "${mpi_generic_prefix}/bin/mpirun"; then
   mpi_generic_run="${mpi_generic_prefix}/bin/mpirun"
  fi
  if test -s "${mpi_generic_prefix}/include/mpif.h"; then
   mpi_generic_cppflags="-I${mpi_generic_prefix}/include"
  fi
  if test -s "${mpi_generic_prefix}/lib/libmpi.a"; then
   mpi_generic_ldflags="-L${mpi_generic_prefix}/lib"
  fi
 fi

 dnl Decide whether generic MPI implementation can be used or not
 if test "${mpi_generic_fc}" != "" -a "${mpi_generic_run}" != ""; then
  mpi_generic_usable="yes"
 fi
 if test "${mpi_generic_cppflags}" != "" -a "${mpi_generic_ldflags}" != ""; then
  mpi_generic_usable="yes"
 fi

 dnl BEGIN DEBUG
 dnl AC_SUBST(mpi_generic_cc)
 dnl AC_SUBST(mpi_generic_cxx)
 dnl AC_SUBST(mpi_generic_fc)
 dnl AC_SUBST(mpi_generic_run)
 dnl AC_SUBST(mpi_generic_cppflags)
 dnl AC_SUBST(mpi_generic_cflags)
 dnl AC_SUBST(mpi_generic_cxxflags)
 dnl AC_SUBST(mpi_generic_fcflags)
 dnl AC_SUBST(mpi_generic_ldflags)
 dnl END DEBUG

 AC_MSG_RESULT([${mpi_generic_usable}])
]) # _ABI_MPI_CHECK_GENERIC



# _ABI_MPI_CHECK_LAM(PREFIX)
# --------------------------
#
# Looks for a LAM implementation of MPI, using the provided prefix.
#
AC_DEFUN([_ABI_MPI_CHECK_LAM],
[dnl Set default values
 mpi_lam_prefix="$1"
 mpi_lam_usable="no"
 mpi_lam_cc=""
 mpi_lam_cxx=""
 mpi_lam_fc=""
 mpi_lam_run=""
 mpi_lam_cppflags=""
 mpi_lam_cflags=""
 mpi_lam_cxxflags=""
 mpi_lam_fcflags=""
 mpi_lam_ldflags=""

 AC_MSG_CHECKING([for a usable LAM implementation])

 dnl Check whether LAM files are available
 if test "${mpi_lam_prefix}" != ""; then
  if test -s "${mpi_lam_prefix}/lib/liblam.a"; then
   if test -x "${mpi_lam_prefix}/bin/mpicc"; then
    mpi_lam_cc="${mpi_lam_prefix}/bin/mpicc"
   fi
   if test -x "${mpi_lam_prefix}/bin/mpic++"; then
    mpi_lam_cxx="${mpi_lam_prefix}/bin/mpic++"
   fi
   if test -x "${mpi_lam_prefix}/bin/mpif90"; then
    mpi_lam_fc="${mpi_lam_prefix}/bin/mpif90"
   fi
   if test -x "${mpi_lam_prefix}/bin/mpirun"; then
    mpi_lam_run="${mpi_lam_prefix}/bin/mpirun"
   fi
   if test -s "${mpi_lam_prefix}/include/mpif.h"; then
    mpi_lam_cppflags="-I${mpi_lam_prefix}/include"
   fi
  fi
 fi

 dnl Decide whether LAM can be used or not
 if test "${mpi_lam_fc}" != "" -a "${mpi_lam_run}" != ""; then
  mpi_lam_usable="yes"
 fi

 dnl BEGIN DEBUG
 dnl AC_SUBST(mpi_lam_cc)
 dnl AC_SUBST(mpi_lam_cxx)
 dnl AC_SUBST(mpi_lam_fc)
 dnl AC_SUBST(mpi_lam_run)
 dnl AC_SUBST(mpi_lam_cppflags)
 dnl AC_SUBST(mpi_lam_cflags)
 dnl AC_SUBST(mpi_lam_cxxflags)
 dnl AC_SUBST(mpi_lam_fcflags)
 dnl AC_SUBST(mpi_lam_ldflags)
 dnl END DEBUG

 AC_MSG_RESULT([${mpi_lam_usable}])
]) # _ABI_MPI_CHECK_LAM



# _ABI_MPI_CHECK_MPICH(PREFIX)
# ----------------------------
#
# Looks for a MPICH implementation of MPI, using the provided prefix.
#
AC_DEFUN([_ABI_MPI_CHECK_MPICH],
[dnl Set default values
 mpi_mpich_prefix="$1"
 mpi_mpich_usable="no"
 mpi_mpich_cc=""
 mpi_mpich_cxx=""
 mpi_mpich_fc=""
 mpi_mpich_run=""
 mpi_mpich_cppflags=""
 mpi_mpich_cflags=""
 mpi_mpich_cxxflags=""
 mpi_mpich_fcflags=""
 mpi_mpich_ldflags=""

 AC_MSG_CHECKING([for a usable MPICH implementation])

 dnl Check whether MPICH files are available
 if test "${mpi_mpich_prefix}" != ""; then
  if test -s "${mpi_mpich_prefix}/lib/libmpich.a" -a \
          -s "${mpi_mpich_prefix}/lib/libfmpich.a"; then
   if test -x "${mpi_mpich_prefix}/bin/mpicc"; then
    mpi_mpich_cc="${mpi_mpich_prefix}/bin/mpicc"
   fi
   if test -x "${mpi_mpich_prefix}/bin/mpicxx"; then
    mpi_mpich_cxx="${mpi_mpich_prefix}/bin/mpicxx"
   fi
   if test -x "${mpi_mpich_prefix}/bin/mpif90"; then
    mpi_mpich_fc="${mpi_mpich_prefix}/bin/mpif90"
   fi
   if test -x "${mpi_mpich_prefix}/bin/mpirun"; then
    mpi_mpich_run="${mpi_mpich_prefix}/bin/mpirun"
   fi
   if test -s "${mpi_mpich_prefix}/include/mpif.h"; then
    mpi_mpich_cppflags="-I${mpi_mpich_prefix}/include"
   fi
  fi
 fi

 dnl Decide whether MPICH can be used or not
 if test "${mpi_mpich_fc}" != "" -a "${mpi_mpich_run}" != ""; then
  mpi_mpich_usable="yes"
 fi

 dnl BEGIN DEBUG
 dnl AC_SUBST(mpi_mpich_cc)
 dnl AC_SUBST(mpi_mpich_cxx)
 dnl AC_SUBST(mpi_mpich_fc)
 dnl AC_SUBST(mpi_mpich_run)
 dnl AC_SUBST(mpi_mpich_cppflags)
 dnl AC_SUBST(mpi_mpich_cflags)
 dnl AC_SUBST(mpi_mpich_cxxflags)
 dnl AC_SUBST(mpi_mpich_fcflags)
 dnl AC_SUBST(mpi_mpich_ldflags)
 dnl END DEBUG

 AC_MSG_RESULT([${mpi_mpich_usable}])
]) # _ABI_MPI_CHECK_MPICH



# _ABI_MPI_CHECK_NATIVE()
# -----------------------
#
# Looks for a native implementation of MPI, i.e. checks if the Fortran
# compiler is able to produce MPI binaries.
#
AC_DEFUN([_ABI_MPI_CHECK_NATIVE],
[dnl Set default values
 mpi_native_usable="no"
 mpi_native_cc="/bin/false"
 mpi_native_cxx="/bin/false"
 mpi_native_fc="${FC}"
 mpi_native_run=""
 mpi_native_cppflags=""
 mpi_native_cflags=""
 mpi_native_cxxflags=""
 mpi_native_fcflags=""
 mpi_native_ldflags=""

 AC_MSG_CHECKING([for a native MPI support])

 dnl Try to compile a MPI program
 AC_LANG_PUSH([Fortran])
 AC_LINK_IFELSE([AC_LANG_PROGRAM([],
  [[#include "mpif.h"
      integer :: ierr
      call mpi_init(ierr)
      call mpi_finalize(ierr)
  ]])], [mpi_native_usable="yes"])
 AC_LANG_POP()

 dnl Look for mpirun
 if test "${mpi_native_usable}" = "yes"; then
  AC_PATH_PROG(mpi_native_run,mpirun)
 fi

 dnl AC_SUBST(mpi_native_cc)
 dnl AC_SUBST(mpi_native_cxx)
 dnl AC_SUBST(mpi_native_fc)
 dnl AC_SUBST(mpi_native_run)
 dnl AC_SUBST(mpi_native_cppflags)
 dnl AC_SUBST(mpi_native_cflags)
 dnl AC_SUBST(mpi_native_cxxflags)
 dnl AC_SUBST(mpi_native_fcflags)
 dnl AC_SUBST(mpi_native_ldflags)

 AC_MSG_RESULT([${mpi_native_usable}])
]) # _ABI_MPI_CHECK_NATIVE



# _ABI_MPI_CHECK_OMPI(PREFIX)
# ---------------------------
#
# Looks for a Open-MPI implementation of MPI, using the provided prefix.
#
AC_DEFUN([_ABI_MPI_CHECK_OMPI],
[dnl Set default values
 mpi_ompi_prefix="$1"
 mpi_ompi_usable="no"
 mpi_ompi_cc=""
 mpi_ompi_cxx=""
 mpi_ompi_fc=""
 mpi_ompi_run=""
 mpi_ompi_cppflags=""
 mpi_ompi_cflags=""
 mpi_ompi_cxxflags=""
 mpi_ompi_fcflags=""
 mpi_ompi_ldflags=""

 AC_MSG_CHECKING([for a usable Open-MPI implementation])

 dnl Check whether Open-MPI files are available
 if test "${mpi_ompi_prefix}" != ""; then
  if test -x "${mpi_ompi_prefix}/bin/ompi_info" -a \
          -x "${mpi_ompi_prefix}/bin/opal_wrapper" -a \
          -x "${mpi_ompi_prefix}/bin/orterun"; then
   if test -x "${mpi_ompi_prefix}/bin/mpicc"; then
    mpi_ompi_cc="${mpi_ompi_prefix}/bin/mpicc"
   fi
   if test -x "${mpi_ompi_prefix}/bin/mpic++"; then
    mpi_ompi_cxx="${mpi_ompi_prefix}/bin/mpic++"
   fi
   if test -x "${mpi_ompi_prefix}/bin/mpif90"; then
    mpi_ompi_fc="${mpi_ompi_prefix}/bin/mpif90"
   fi
   if test -x "${mpi_ompi_prefix}/bin/mpirun"; then
    mpi_ompi_run="${mpi_ompi_prefix}/bin/mpirun"
   fi
   if test -s "${mpi_ompi_prefix}/include/mpif.h"; then
    mpi_ompi_cppflags="-I${mpi_ompi_prefix}/include"
   fi
  fi
 fi

 dnl Decide whether MPICH can be used or not
 if test "${mpi_ompi_fc}" != "" -a "${mpi_ompi_run}" != ""; then
  mpi_ompi_usable="yes"
 fi

 dnl BEGIN DEBUG
 dnl AC_SUBST(mpi_ompi_cc)
 dnl AC_SUBST(mpi_ompi_cxx)
 dnl AC_SUBST(mpi_ompi_fc)
 dnl AC_SUBST(mpi_ompi_run)
 dnl AC_SUBST(mpi_ompi_cppflags)
 dnl AC_SUBST(mpi_ompi_cflags)
 dnl AC_SUBST(mpi_ompi_cxxflags)
 dnl AC_SUBST(mpi_ompi_fcflags)
 dnl AC_SUBST(mpi_ompi_ldflags)
 dnl END DEBUG

 AC_MSG_RESULT([${mpi_ompi_usable}])
]) # _ABI_MPI_CHECK_OMPI



# ABI_MPI_CHECK()
# ---------------
#
# Tries first to determine which kind of MPI implementation is available,
# then applies tricks and workarounds. The order of search is important,
# as some implementations are currently better supported than others.
#
# Note: ABI_MPI_INIT() should be called first.
#
AC_DEFUN([ABI_MPI_CHECK],
[dnl Check whether MPI support has been forced on or off
 if test "${enable_mpi}" = "yes"; then
  dnl Look for a native implementation
  if test "${mpi_usable}" = "no"; then
   _ABI_MPI_CHECK_NATIVE
   if test "${mpi_native_usable}" = "yes"; then
    mpi_usable="yes"
    mpi_type="native"
    MPI_FC="${mpi_native_fc}"
    MPI_RUN="${mpi_native_run}"
   fi
  fi

  dnl Check whether a MPI prefix has been specified
  if test "${mpi_usable}" = "no"; then
   if test "${with_mpi_prefix}" != ""; then
    AC_MSG_NOTICE([using MPI prefix ${with_mpi_prefix}])
   fi
  fi

  dnl Look for an Open-MPI implementation
  if test "${mpi_usable}" = "no" -a "${with_mpi_prefix}" != ""; then
   _ABI_MPI_CHECK_OMPI(${with_mpi_prefix})
   if test "${mpi_ompi_usable}" = "yes"; then
    mpi_usable="yes"
    mpi_type="open-mpi"
    MPI_FC="${mpi_ompi_fc}"
    MPI_RUN="${mpi_ompi_run}"
   fi
  fi

  dnl Look for a MPICH implementation
  if test "${mpi_usable}" = "no" -a "${with_mpi_prefix}" != ""; then
   _ABI_MPI_CHECK_MPICH(${with_mpi_prefix})
   if test "${mpi_mpich_usable}" = "yes"; then
    mpi_usable="yes"
    mpi_type="mpich"
    MPI_FC="${mpi_mpich_fc}"
    MPI_RUN="${mpi_mpich_run}"
   fi
  fi

  dnl Look for a LAM implementation
  if test "${mpi_usable}" = "no" -a "${with_mpi_prefix}" != ""; then
   _ABI_MPI_CHECK_LAM(${with_mpi_prefix})
   if test "${mpi_lam_usable}" = "yes"; then
    mpi_usable="yes"
    mpi_type="lam"
    MPI_FC="${mpi_lam_fc}"
    MPI_RUN="${mpi_lam_run}"
   fi
  fi

  dnl Look for a generic implementation (fallback mechanism)
  if test "${mpi_usable}" = "no" -a "${with_mpi_prefix}" != ""; then
   _ABI_MPI_CHECK_GENERIC(${with_mpi_prefix})
   if test "${mpi_generic_usable}" = "yes"; then
    mpi_usable="yes"
    mpi_type="generic"
    MPI_FC="${mpi_generic_fc}"
    MPI_RUN="${mpi_generic_run}"
   fi
  fi
 
  dnl Enable MPI support if something usable has been found
  if test "${mpi_usable}" = "yes"; then
   enable_mpi="yes"
  else
   enable_mpi="no"
  fi
 else
  if test "${enable_mpi}" = "manual"; then
   AC_MSG_NOTICE([using manual settings for MPI configuration])
   enable_mpi="yes"
   mpi_usable="unknown"
   mpi_type="unknown"
  else
   enable_mpi="no"
   MPI_CPP="/bin/false"
   MPI_CC="/bin/false"
   MPI_CXX="/bin/false"
   MPI_FC="/bin/false"
   MPI_RUN="/bin/false"
   MPI_CPPFLAGS=""
   MPI_CFLAGS=""
   MPI_CXXFLAGS=""
   MPI_FCFLAGS=""
   MPI_FC_LDFLAGS=""
  fi
 fi
]) # ABI_MPI_CHECK



# ABI_MPI_INIT()
# --------------
#
# Sets MPI default values.
#
AC_DEFUN([ABI_MPI_INIT],
[dnl Set default values
 mpi_usable="no"
 mpi_type="unknown"

 dnl Set-up environment
 SEQ_FC="${FC}"
 
 if test "${with_mpi_cpp}" = "yes" -o "${with_mpi_cpp}" = "no"; then
  MPI_CPP="/bin/false"
 else
  MPI_CPP="${with_mpi_cpp}"
 fi
 if test "${with_mpi_cppflags}" = "yes" -o "${with_mpi_cppflags}" = "no"; then
  MPI_CPPFLAGS=""
 else
  MPI_CPPFLAGS="${with_mpi_cppflags}"
 fi

 if test "${with_mpi_cc}" = "yes" -o "${with_mpi_cc}" = "no"; then
  MPI_CC="/bin/false"
 else
  MPI_CC="${with_mpi_cc}"
 fi
 if test "${with_mpi_cflags}" = "yes" -o "${with_mpi_cflags}" = "no"; then
  MPI_CFLAGS=""
 else
  MPI_CFLAGS="${with_mpi_cflags}"
 fi
 if test "${with_mpi_c_ldflags}" = "yes" -o "${with_mpi_c_ldflags}" = "no"; then
  MPI_CC_LDFLAGS=""
 else
  MPI_CC_LDFLAGS="${with_mpi_c_ldflags}"
 fi

 if test "${with_mpi_cxx}" = "yes" -o "${with_mpi_cxx}" = "no"; then
  MPI_CXX="/bin/false"
 else
  MPI_CXX="${with_mpi_cxx}"
 fi
 if test "${with_mpi_cxxflags}" = "yes" -o "${with_mpi_cxxflags}" = "no"; then
  MPI_CXXFLAGS=""
 else
  MPI_CXXFLAGS="${with_mpi_cxxflags}"
 fi
 if test "${with_mpi_cxx_ldflags}" = "yes" -o \
         "${with_mpi_cxx_ldflags}" = "no"; then
  MPI_CXX_LDFLAGS=""
 else
  MPI_CXX_LDFLAGS="${with_mpi_cxx_ldflags}"
 fi

 if test "${with_mpi_fc}" = "yes" -o "${with_mpi_fc}" = "no"; then
  MPI_FC="/bin/false"
 else
  MPI_FC="${with_mpi_fc}"
 fi
 if test "${with_mpi_fcflags}" = "yes" -o "${with_mpi_fcflags}" = "no"; then
  MPI_FCFLAGS=""
 else
  MPI_FCFLAGS="${with_mpi_fcflags}"
 fi
 if test "${with_mpi_fc_ldflags}" = "yes" -o \
         "${with_mpi_fc_ldflags}" = "no"; then
  MPI_FC_LDFLAGS=""
 else
  MPI_FC_LDFLAGS="${with_mpi_fc_ldflags}"
 fi

 if test "${with_mpi_run}" = "yes" -o "${with_mpi_run}" = "no"; then
  MPI_RUN="/bin/false"
 else
  MPI_RUN="${with_mpi_run}"
 fi

 dnl Enable substitution
 AC_SUBST(mpi_usable)
 AC_SUBST(mpi_type)
 AC_SUBST(MPI_CPP)
 AC_SUBST(MPI_CPPFLAGS)
 AC_SUBST(MPI_CC)
 AC_SUBST(MPI_CFLAGS)
 AC_SUBST(MPI_CC_LDFLAGS)
 AC_SUBST(MPI_CXX)
 AC_SUBST(MPI_CXXFLAGS)
 AC_SUBST(MPI_CXX_LDFLAGS)
 AC_SUBST(SEQ_FC)
 AC_SUBST(MPI_FC)
 AC_SUBST(MPI_FCFLAGS)
 AC_SUBST(MPI_FC_LDFLAGS)
 AC_SUBST(MPI_RUN)
]) # ABI_MPI_INIT



# ABI_MPI_REPORT()
# ----------------
#
# Shows all relevant MPI-related variables.
#
AC_DEFUN([ABI_MPI_REPORT],
[dnl Status
 AC_MSG_CHECKING([whether to build parallel code])
 AC_MSG_RESULT([${enable_mpi}])

 dnl Build parameters
 if test "${enable_mpi}" = "yes"; then
  AC_MSG_CHECKING([if MPI is usable])
  AC_MSG_RESULT([${mpi_usable}])
  AC_MSG_CHECKING([for MPI type])
  AC_MSG_RESULT([${mpi_type}])
  AC_MSG_CHECKING([for MPI Fortran compiler])
  if test "${MPI_FC}" = ""; then
   AC_MSG_RESULT([none])
  else
   AC_MSG_RESULT([${MPI_FC}])
  fi
  AC_MSG_CHECKING([for MPI CPPFLAGS])
  if test "${MPI_CPPFLAGS}" = ""; then
   AC_MSG_RESULT([none])
  else
   AC_MSG_RESULT([${MPI_CPPFLAGS}])
  fi
  AC_MSG_CHECKING([for MPI FCFLAGS])
  if test "${MPI_FCFLAGS}" = ""; then
   AC_MSG_RESULT([none])
  else
   AC_MSG_RESULT([${MPI_FCFLAGS}])
  fi
  AC_MSG_CHECKING([for MPI Fortran LDFLAGS])
  if test "${MPI_FC_LDFLAGS}" = ""; then
   AC_MSG_RESULT([none])
  else
   AC_MSG_RESULT([${MPI_FC_LDFLAGS}])
  fi
  AC_MSG_CHECKING([for MPI run script])
  if test "${MPI_RUN}" = ""; then
   AC_MSG_RESULT([none])
  else
   AC_MSG_RESULT([${MPI_RUN}])
  fi
 fi
]) # ABI_MPI_REPORT
